/*************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/*
//
//  Purpose:
//     Cryptography Primitive.
//     EC over Prime Finite Field (P224r1 precomputed)
//
//
*/
#include "owncp.h"
#include "gfpec/pcpgfpecstuff.h"


#define OPERAND_BITSIZE (224)
#define LEN_P224        (BITS_BNU_CHUNK(OPERAND_BITSIZE))

/* P192 affine point */
typedef struct {
    BNU_CHUNK_T X[LEN_P224];
    BNU_CHUNK_T Y[LEN_P224];
} P224_POINT_AFFINE;

extern const __ALIGN64 P224_POINT_AFFINE ec_p224r1_precomputed[33][64];


#if defined(_IPP_DATA)

#if !defined(_DISABLE_ECP_224R1_HARDCODED_BP_TBL_)
/* see ippcp_baseptbl.cpp test for generation details */

#if ((_IPP_ARCH == _IPP_ARCH_EM64T) || (_IPP_ARCH == _IPP_ARCH_LP64))
const __ALIGN64 P224_POINT_AFFINE ec_p224r1_precomputed[33][64] = {
    /* digit=0 base_pwr=2^0 */
    /* clang-format off */
{
   {{LL(0x6d0a4aea,0xbc905226),LL(0x6018bfaa,0x85259736),LL(0xf96bec05,0x6dd3af9b),L_(0xa21b5e60)}, {LL(0xeff3ede8,0x2edca1e5),LL(0x05335a6b,0xf8cd672b),LL(0x03dfe878,0xaea9c5ae),L_(0x614786f1)}},
   {{LL(0xe722ff54,0x4ca9a1ec),LL(0x5650df9b,0xd704f84f),LL(0x2a0f1689,0x10c911f7),L_(0xcafb50f5)}, {LL(0x95a24e5e,0x02aac79b),LL(0xc90ae186,0x518c11b1),LL(0x76cc101a,0x0c84ced0),L_(0x5cabb880)}},
   {{LL(0x86614a9a,0xe37710d7),LL(0xe0b02223,0x24c17ffc),LL(0xc3cf1c5b,0x00505bb6),L_(0xaf8b6496)}, {LL(0x73e4e25d,0x9a73b173),LL(0x04144e62,0x70501632),LL(0x357940ac,0xafe10c35),L_(0x6dfa492c)}},
   {{LL(0x840f10e9,0x0a72faf6),LL(0x0860856c,0x3c2dbf31),LL(0xcb021706,0x4256e88a),L_(0x89f0c6a0)}, {LL(0x8c1f2b39,0x6743e273),LL(0x98933269,0xd8933d1c),LL(0x2058f0ba,0x7ee104d3),L_(0x8c87db62)}},
   {{LL(0xd42fac58,0x0b10cc50),LL(0xf8dfc283,0xd80cdd0a),LL(0x975a451e,0x39211b77),L_(0xf7f4fe95)}, {LL(0x64fade9e,0x2b552e73),LL(0x61a9e695,0x5e5610f3),LL(0x11bcf038,0x42b54f79),L_(0xc858da82)}},
   {{LL(0xde4901db,0xde577c0d),LL(0x1a8aec73,0x09e4f789),LL(0xe0cd01d4,0x1addb3a1),L_(0xf25fe17c)}, {LL(0xdc7c211c,0x5fae6c32),LL(0x0f5c7709,0x5603672c),LL(0xf2d2c41c,0x9b6731a6),L_(0x162f9e1c)}},
   {{LL(0x899abac2,0x90462820),LL(0x68646e35,0xc6adf683),LL(0x1e2526ba,0xc81df812),L_(0x7ed50814)}, {LL(0x5dd6407c,0x610f2185),LL(0x1fb32889,0x3f3879aa),LL(0xed8ceeee,0xce5eb25e),L_(0x0e5952e3)}},
   {{LL(0xde1707ec,0xb4c9d865),LL(0x7bd696a2,0x3fcec855),LL(0xa3bebfe9,0x8e4a5712),L_(0x1b59038c)}, {LL(0x29a5f205,0x1bc1ff36),LL(0x7558734a,0x0efc11fd),LL(0x09762590,0xc4e19066),L_(0x2199c364)}},
   {{LL(0x7b94d3c7,0x1c1ecc77),LL(0x743eda4a,0x23767b5f),LL(0x2f993a50,0x94f5e9ba),L_(0x79d4095a)}, {LL(0x8f776409,0xe07e55b6),LL(0x909bf3f1,0x2b57fa61),LL(0x77e1db6d,0xe733d627),L_(0x5114ac9d)}},
   {{LL(0xa53cb542,0xb808145c),LL(0xb4f2672c,0x8d51557e),LL(0x3e8abf6a,0xc1f86e2c),L_(0x9d0cdd53)}, {LL(0x8dd67e37,0xccddceff),LL(0x02883b6f,0x94892c1d),LL(0x873d7e4c,0xdbc0f8ee),L_(0x2a687506)}},
   {{LL(0x6c6825d9,0xac358ca8),LL(0xe1bdd5ca,0x71f50674),LL(0xd0dee306,0x0f060758),L_(0xd6d89093)}, {LL(0xf1b7227e,0x649ea142),LL(0x08181c2c,0x87d166a2),LL(0x6c63713a,0xc1182bf0),L_(0x45e48d79)}},
   {{LL(0x8f86d9da,0xf6495da1),LL(0xa37b7441,0x5dc634c9),LL(0x2b0b8e1a,0x571d38d9),L_(0x517af941)}, {LL(0x92dd72b5,0x1305b612),LL(0x8cf21835,0x8db28ab1),LL(0x5f59deb1,0xbc243af0),L_(0xa71404ae)}},
   {{LL(0x3a1f6a38,0x489c8db2),LL(0xffbe563f,0x4943a5ae),LL(0xbc99fb0b,0x68ade63c),L_(0x398eacf0)}, {LL(0xed549cc4,0x83fdacb5),LL(0x800df824,0xb6ab4a8b),LL(0x442ed616,0xb4dc6bf8),L_(0x3f8060b4)}},
   {{LL(0xb83ee7d6,0xf4109221),LL(0xcdbd6eec,0x565559e4),LL(0x1ef814eb,0xf4550d2e),L_(0xb3a9b5e4)}, {LL(0x6d218d7b,0x1544e90e),LL(0x3f51f9f9,0x31b0bc68),LL(0x825b5795,0x6516bbc3),L_(0x3d1419d2)}},
   {{LL(0xe977f339,0x55bfd519),LL(0xb16ed1af,0x5cd307d1),LL(0xd91345bb,0xc5e26f62),L_(0x64a35d22)}, {LL(0xa243f1ee,0x05015842),LL(0x70203aa0,0x6fa10db3),LL(0x05a83a63,0xdaf3972b),L_(0x9422f10e)}},
   {{LL(0x73b437e2,0x6a4bd1f9),LL(0xb3f66de6,0xbb0f749a),LL(0xfdf9a0e9,0x471a2a7f),L_(0xb9c51b5f)}, {LL(0x44574243,0xfa29681a),LL(0x9c409bd0,0x9522774b),LL(0x169405c3,0x48be80cb),L_(0xf22d3fc0)}},
   {{LL(0xf845831a,0x86ade48b),LL(0x79cec3fb,0x473e361f),LL(0xb9622253,0x5523244b),L_(0x538d543e)}, {LL(0xf8570c03,0x12bb38bf),LL(0xb4e20b8c,0x784e1b25),LL(0xe6873448,0x2a2e7758),L_(0x2954321a)}},
   {{LL(0x47b54bc5,0x3591891e),LL(0x27a72a86,0x3ca319e3),LL(0x482460f6,0x70114363),L_(0x3b204e52)}, {LL(0xcca0ee1f,0xd5f98053),LL(0xc4a1a96b,0xd84149c4),LL(0x0c0e97bb,0x6bc4b6e9),L_(0xfb212c8c)}},
   {{LL(0x1bfad739,0x716b6add),LL(0xea007122,0x0e68b721),LL(0x08a75056,0x4e35e1f6),L_(0xe28f8fa3)}, {LL(0x93c21468,0x7393ad16),LL(0x4e27959e,0x4533e928),LL(0x433c708d,0xe5b33144),L_(0x4c003b64)}},
   {{LL(0x32804a04,0xc57bd47a),LL(0x238addb6,0x5ae81313),LL(0x703c82d5,0xa0691b17),L_(0x14a5d58e)}, {LL(0x825037c4,0xf9cefcb5),LL(0xfd4764c3,0x38e801b7),LL(0x7b23c9fe,0x23cf8d21),L_(0x1d7b0c07)}},
   {{LL(0x8738f5d4,0x869c61a1),LL(0x8d8b998b,0xe83f1dc1),LL(0x5cc8116d,0x1f271989),L_(0xa8992ade)}, {LL(0x88f6e037,0xb2334f8e),LL(0x1fadf19a,0x9fcaafe1),LL(0x71d5904e,0x939bd9ec),L_(0x534ec549)}},
   {{LL(0xa2cbcd92,0x22be0a44),LL(0xb12549d7,0x275cd50c),LL(0xb64feadd,0xfa13b698),L_(0xd61f5f42)}, {LL(0xedfc9326,0x7ea2fed9),LL(0xd797283d,0xdbd2171d),LL(0x5e1f2529,0x113c4da8),L_(0x4d22be7a)}},
   {{LL(0xd7a234b5,0x746daf5f),LL(0x210782c4,0x8d13c8a1),LL(0x358ace9d,0x1008fd0f),L_(0xaf231c63)}, {LL(0xeedf77aa,0x17203da0),LL(0x3534d554,0x0974abbe),LL(0x30962d48,0xf7552ee7),L_(0xa1ab0080)}},
   {{LL(0x77890e5b,0x7708ee68),LL(0x981a94f1,0x624e8fe3),LL(0x50822536,0x2632b017),L_(0x21031b41)}, {LL(0x0b603325,0xa43cc164),LL(0xfb24577f,0xc20ef5b2),LL(0x67a5fdbd,0x583ccbb4),L_(0xdf198e9a)}},
   {{LL(0x2390b30d,0xec729f5b),LL(0x347bddd0,0x5467eb45),LL(0x4576657b,0x79a42c59),L_(0x3a4d5c62)}, {LL(0x871cf39c,0x72672c8d),LL(0x34edf6de,0x22af9ec1),LL(0xc615c176,0x264b7c52),L_(0x6b6c91f7)}},
   {{LL(0x3dc0652c,0x3c93deb4),LL(0x454fe5e1,0x83476556),LL(0xdaedb455,0x895933d6),L_(0xc3d92afc)}, {LL(0xbabeca3c,0x62afc582),LL(0xb1ad4196,0xd877f8d1),LL(0x122626f8,0x56df6aa3),L_(0x88c5db33)}},
   {{LL(0xc8d94c12,0xb5862dd8),LL(0xfb7f7ec7,0x3488de74),LL(0xf9ca08fd,0xf3fb356b),L_(0xa3379ec2)}, {LL(0x27d39af3,0x5680ab94),LL(0x1f4b780d,0x4aa3a013),LL(0x181ad0f0,0x9c77acf2),L_(0xc5ed6c19)}},
   {{LL(0xa57812ca,0x57cdf956),LL(0x2e235ce3,0xec26e698),LL(0x3b66736f,0x02924c24),L_(0x9611c7f6)}, {LL(0x20dcabee,0x2cd39c40),LL(0x0ca01257,0x6c0fb396),LL(0x36cea89c,0x576c9d6d),L_(0xc0386b9b)}},
   {{LL(0x69c717f0,0x3ff1d620),LL(0xb26ad028,0x975088a3),LL(0x3395a449,0xbb9b7453),L_(0xac8575b5)}, {LL(0x0cb11f18,0x19b52294),LL(0x61b7e04c,0x84cd1038),LL(0xea595a3a,0xb7e04864),L_(0xf641867a)}},
   {{LL(0x68c289a6,0xf9c67812),LL(0x0d5cd555,0x64da8983),LL(0x15609314,0x071a1814),L_(0xcec729a1)}, {LL(0x2725b13c,0xd82b0a63),LL(0xe9fd6113,0xba49569c),LL(0xf4a523e8,0x095fdd3b),L_(0x601388e8)}},
   {{LL(0x382f2331,0x5d150588),LL(0x6e0c12f8,0xabac1d43),LL(0xcc4b0247,0xe25d613a),L_(0x98b62cd3)}, {LL(0xc09181c3,0xca8e4ca9),LL(0x1f7fb23f,0xae6dcc27),LL(0x41da9f53,0x44527d49),L_(0x9e1f47c6)}},
   {{LL(0xbf5c291a,0xb17f74ec),LL(0xe460275b,0x7fe47107),LL(0x56dc0e69,0xaef13524),L_(0x055bebff)}, {LL(0x85eeba4b,0x5a0a6115),LL(0x52aae002,0xd72b6016),LL(0x65c91bf7,0x55229ca9),L_(0x837c7497)}},
   {{LL(0x05073659,0x724287d6),LL(0x86eb5493,0x266f1738),LL(0x70786420,0x72b46e9e),L_(0x0bd8afe8)}, {LL(0x2e7a6b37,0xeea543b6),LL(0x7400cd52,0x1221cd6e),LL(0x0a6c2e1c,0x9604a54b),L_(0xe2c2c60a)}},
   {{LL(0xaacc2f1d,0x771d897e),LL(0x80cd8b3c,0xa4304d46),LL(0x3b961927,0x56c6e625),L_(0x8d0c1d5a)}, {LL(0xf0576bd1,0x4b4c8802),LL(0xee49b988,0xd528b6a1),LL(0x617aabd8,0xb63d84cc),L_(0xb309e31a)}},
   {{LL(0x21a4d862,0x6ef3ab05),LL(0xfbca1b97,0x7fe6fa59),LL(0xd8911f8e,0x639e011c),L_(0xb731d607)}, {LL(0x654578a1,0x0706cd08),LL(0x8e5b6703,0x5d08a7fc),LL(0x4d443d17,0xdc2ea468),L_(0xccc5dbca)}},
   {{LL(0x2986a64c,0xf86e4ee6),LL(0xb38b3cb9,0xb2a50e4f),LL(0xd001bff6,0xad920b0c),L_(0x5f833048)}, {LL(0xceab268e,0x3c687112),LL(0x9cac7077,0xb6207b74),LL(0x2bf74e82,0x92b4f4b1),L_(0xfcfe6f73)}},
   {{LL(0x4a1a48c2,0x9afdccc1),LL(0xf2645ccf,0x9befe15b),LL(0x170bc3de,0xc06a289b),L_(0x8b3174cc)}, {LL(0xcb55af1c,0xc289c4c5),LL(0xfcbc66cf,0x73ec1671),LL(0xea32e7fb,0xfdc7bcec),L_(0x2d89d32d)}},
   {{LL(0xc3672949,0x31801b58),LL(0xcf102dd0,0x2e36c7e0),LL(0x28f8286a,0x1600629e),L_(0x6d3edc02)}, {LL(0x16920812,0xf21cabe6),LL(0x31851ac8,0xf3657a9a),LL(0x037f5693,0x3f33781d),L_(0xb0a0ce6c)}},
   {{LL(0x15c4b334,0x4821f125),LL(0x43b3629d,0x3a678868),LL(0x33190540,0xa3b32151),L_(0x4038845d)}, {LL(0xcf5706d9,0xedfaa06c),LL(0x00044014,0x08de20a4),LL(0xd5eb20d6,0x1a8a108b),L_(0x38cb0a6b)}},
   {{LL(0x15f2ae4d,0xb6e363d4),LL(0xa45c3df2,0x80d28480),LL(0x00c26ab1,0x85098966),L_(0x6d03e325)}, {LL(0x36056e89,0x8eec84d1),LL(0x9134e03a,0xce2d0f6b),LL(0xbbc8d92d,0x4ffdb217),L_(0x6e97a4bb)}},
   {{LL(0xb60e66c5,0x231d20c5),LL(0xa36c8848,0x89bcc285),LL(0xcb7e8b54,0xc4b2c9f8),L_(0x2f8b6de0)}, {LL(0x243f2d3f,0xb8860525),LL(0xfdcf530d,0x945673bf),LL(0x49da0f6b,0xefc627b6),L_(0x4e9fb807)}},
   {{LL(0x21bc7f20,0x64eae113),LL(0x23074bd5,0x2718ea4a),LL(0xb246633c,0xefdb4759),L_(0xc6af484b)}, {LL(0xcdbec413,0xe5997f28),LL(0x8979b2ec,0x82bcb2f5),LL(0x530f4963,0x55ee3298),L_(0xccf127a1)}},
   {{LL(0xba2c7ded,0x687018fc),LL(0x7c723f5d,0xcc9d7b65),LL(0x2221f721,0x93828714),L_(0x13d720d3)}, {LL(0xe6d02ad4,0xe61c9b05),LL(0xc198db55,0x72b115de),LL(0xbe38af8f,0x621f284e),L_(0x18f09bf6)}},
   {{LL(0x7c8299c2,0x65cd0db6),LL(0xbb33fff7,0x7d479cee),LL(0x44ac0a56,0x7a9b66be),L_(0xc9308cae)}, {LL(0x43b7cc76,0x1b4048cb),LL(0x5b7ecc1b,0x0b2ccca8),LL(0x7fa30aae,0x7e09ad23),L_(0xff1f76e3)}},
   {{LL(0x13e6bef2,0xa178a33a),LL(0xdd724842,0x7d66c450),LL(0xa5d7a87c,0x30befc70),L_(0x69874db8)}, {LL(0x3ef58d85,0xcfcdbfce),LL(0xe3d6346a,0x9fa94e38),LL(0x84c65476,0xde39cef5),L_(0xe4efb3c1)}},
   {{LL(0x5b582473,0x493ae0ba),LL(0xbc282cc0,0x8c93a914),LL(0xa024a0ea,0xfa3a15cd),L_(0x240e6a89)}, {LL(0x572dde87,0xbba5d02d),LL(0x95b6f1e0,0x2f1b39cd),LL(0x15b8172f,0x158b8956),L_(0x5eb7385c)}},
   {{LL(0x4b30aee1,0x92f507cd),LL(0x9137f80a,0xa95f0372),LL(0xcb924116,0x2be56c53),L_(0x0703cdda)}, {LL(0xbf64ed3a,0x0b8faa7f),LL(0xb73eae3f,0x9943cbed),LL(0xf227ad25,0xdbca2fc4),L_(0x06231927)}},
   {{LL(0x92d256df,0x5983f1a7),LL(0xee7ac222,0x640d13e7),LL(0x48c69cc0,0xa7dc8e21),L_(0xbd701252)}, {LL(0xaa2ef84d,0x483083a1),LL(0xc0d30797,0x560bf32f),LL(0xd4ec2e63,0xbda6fdc0),L_(0x15074c14)}},
   {{LL(0xc0aebb53,0x29e4a335),LL(0x600e1a17,0x155e8020),LL(0xa542f638,0x556905e7),L_(0xc7209080)}, {LL(0xf57b447b,0x0b55b0fd),LL(0xa6cf1f95,0xc37cde3e),LL(0x9d44e115,0x1acfe880),L_(0xe820af09)}},
   {{LL(0x08d29dee,0xc0bba382),LL(0x02857752,0x40ac12bb),LL(0x8b30f556,0xf120846d),L_(0xac6401e1)}, {LL(0x41706845,0xcb20c290),LL(0x84fbf362,0x6005431c),LL(0xecddc9db,0xe1839421),L_(0x63952a13)}},
   {{LL(0x93d07eac,0x24f4b135),LL(0xf0524e07,0xfc097ef4),LL(0x138ee6ed,0x1770de3d),L_(0xedc6985e)}, {LL(0x34e15a3c,0x997ae14f),LL(0xb5a50098,0xa168adf0),LL(0x05ed2b31,0xe84702e1),L_(0x4701e013)}},
   {{LL(0x991691d6,0xa2d01aa3),LL(0x5c0291c5,0xe3a7de4e),LL(0x87f294c6,0xdc575065),L_(0xf7c66733)}, {LL(0x1ae8eb0b,0x236da942),LL(0xb2bb31c0,0xd46c809a),LL(0xefda656f,0x0d4ba22a),L_(0xd90a6c4a)}},
   {{LL(0x009f6ea1,0xd7b84536),LL(0x6ae52eb4,0x52644093),LL(0x4989ea92,0x58155fb2),L_(0x51865588)}, {LL(0xd29d86e0,0xb2f24a3e),LL(0x4a65a557,0x6e88be4d),LL(0xc5c558ab,0x39d391fe),L_(0x71637238)}},
   {{LL(0x551e6187,0xc6f1655b),LL(0xda9557a2,0x49fef895),LL(0xad9724f5,0x02bd8dec),L_(0x286ae065)}, {LL(0x519ec0a6,0x66f88628),LL(0x409bbff9,0xb11907a8),LL(0xa035e3ce,0xfa9ce7d1),L_(0xb11a05b6)}},
   {{LL(0x835832b0,0x7d3144d7),LL(0xeeb7e587,0xca5729ee),LL(0x63105e87,0x24b31279),L_(0x358ad95d)}, {LL(0x44854e57,0x8b96f12f),LL(0xa1f8699a,0x43d92d9e),LL(0x36720d2f,0x19da3cf7),L_(0x0c3e259d)}},
   {{LL(0x4f9b1862,0x3ebef321),LL(0x18a10476,0x898e24fc),LL(0x93fc7b18,0x2dd5dfa6),L_(0x453f9b19)}, {LL(0x9fe968bd,0x2e375ef1),LL(0xde52e14c,0xc56e3c8e),LL(0xb86d2dac,0xcdf1057b),L_(0x8978f37a)}},
   {{LL(0xa47a47e4,0x01ae0872),LL(0xf209f351,0x1ffda591),LL(0x1c3d4f64,0xfd2dee37),L_(0x23082359)}, {LL(0x92135e7e,0x226bc4e4),LL(0xc1dc2c14,0x68af001c),LL(0x4a88888d,0xd11ebfc0),L_(0x5bb60dfa)}},
   {{LL(0xbdac7377,0x6d7442e6),LL(0x1a37fe3d,0x111de286),LL(0x64b6847f,0x54c71408),L_(0x9321abab)}, {LL(0x605934d9,0x8f6064c1),LL(0x0fe08518,0xab526e1f),LL(0x48846076,0x18d88b42),L_(0x5027824f)}},
   {{LL(0x58d7da2d,0x2fefa18a),LL(0x0a5f6d1a,0x28fd1689),LL(0xaeecb1e4,0xe2cc44ed),L_(0x1a3c357a)}, {LL(0x0093ad3c,0xe129986c),LL(0x78fb4e9b,0x8691e9c2),LL(0x6b166a07,0x1f456692),L_(0x55649b1b)}},
   {{LL(0x9465a494,0xbb9f7822),LL(0x0c7e73f7,0x720482c7),LL(0x4c6140bb,0x1d71b6bb),L_(0xc0d4cee9)}, {LL(0x846e827c,0x8f080a16),LL(0xf403aa8c,0xa7eed738),LL(0xb53cebd4,0x45561f5f),L_(0x176891fe)}},
   {{LL(0xd85db49c,0xffa27852),LL(0x4d6d2499,0xebaf8c05),LL(0xb5248259,0x5a3e6570),L_(0xb7125d0c)}, {LL(0xf6311485,0x37979bfc),LL(0x785afdfc,0x732a179e),LL(0x1a4c7732,0xe1b6b02f),L_(0xa84e1866)}},
   {{LL(0x70e07038,0x11dbcdcd),LL(0x4243fd62,0xf7e91887),LL(0x9892ddcf,0xdc0c4d01),L_(0x5638c3c1)}, {LL(0x71bade10,0xa42313ce),LL(0xfcf531d5,0xe28bc38a),LL(0x1417aa25,0x7e0b8323),L_(0x41b2315e)}},
   {{LL(0xac4962cb,0xd6c61ac1),LL(0x9d056208,0xb7765b3a),LL(0xa70e6aa2,0x566c60a7),L_(0x483b996a)}, {LL(0x05f2cab3,0xa38d32bc),LL(0xc38fefeb,0x21c5e288),LL(0x57f15c48,0x689ae7bd),L_(0x3e0d8010)}},
   {{LL(0xbb871aa2,0x2587df14),LL(0x904f01e9,0xc5d6ae2a),LL(0xbce7f04f,0x88aa4d08),L_(0xd7f1a354)}, {LL(0x75d9b4b6,0x5bfae677),LL(0x299925d3,0xf0375024),LL(0x6a708cd7,0x74d5a757),L_(0x97968c25)}},
},
/* digit=1 base_pwr=2^7 */
{
   {{LL(0x44a6cdeb,0xd1a6df98),LL(0x084a7e0f,0x89ef410c),LL(0x0b68659d,0xdf5b6590),L_(0xe1cee8dc)}, {LL(0xb48fcec5,0xf5210071),LL(0xccbda01d,0x01031fdf),LL(0x4c7a3392,0x01bb7a1c),L_(0xa7484df7)}},
   {{LL(0x698ab732,0xcb74232b),LL(0xb62e25de,0x4f40ee59),LL(0xc5c8bed1,0xfb8ee6c4),L_(0x4617192d)}, {LL(0x626b516c,0xeaf8b0a3),LL(0xdc2453a5,0x0f8cc094),LL(0x4d7f2e28,0xf43e697c),L_(0x2e81523e)}},
   {{LL(0x212c04e5,0x37eb6d84),LL(0x29543b70,0xa63887c8),LL(0xe67d4662,0xbf221e31),L_(0xde581875)}, {LL(0x5a447606,0xc3db23ff),LL(0x82bba001,0x0b2a5765),LL(0x60a68af2,0x5e5ccc27),L_(0x619f6149)}},
   {{LL(0x008cfd4a,0xe80619b5),LL(0x25f0799b,0xd4f88882),LL(0xeb6dfe0d,0x9bf5f7d0),L_(0x63a366a0)}, {LL(0x3703627a,0x927212be),LL(0x02c160de,0xbd5509cb),LL(0x09ea0af6,0xfe05e98d),L_(0xbae9ad71)}},
   {{LL(0xb26352aa,0x65b1d33d),LL(0x7f5ed157,0x84979794),LL(0x1ed1728d,0x39babd58),L_(0x0137cf58)}, {LL(0x836ccdc1,0x2546cd9b),LL(0xd0dfb090,0xbfa9bb59),LL(0xb59a51ab,0x6476def4),L_(0xc5dce06b)}},
   {{LL(0xceaa7c3c,0x23138239),LL(0x80c0cfc1,0xa694e0b5),LL(0x27734f88,0x587454f5),L_(0xb527a9af)}, {LL(0xee6f2afb,0x049823f2),LL(0x25a1f7dc,0xe574bc64),LL(0xb3ea257d,0x1d11fd76),L_(0xafd0ae51)}},
   {{LL(0x9b56a4c2,0xc1bc485c),LL(0x407824fd,0xbe617f5c),LL(0xa0e87d81,0x71256aba),L_(0x76217f94)}, {LL(0x3e2877fa,0x8b3e508b),LL(0xe501c9b4,0xbda0977d),LL(0x08a19d4e,0xfb512fa2),L_(0x59ab4cb8)}},
   {{LL(0x71544a12,0xe4756696),LL(0xaf2eb7f6,0x93d2ed81),LL(0x5c0e9df7,0xc52db668),L_(0x2b7064bd)}, {LL(0xded1b829,0x43184720),LL(0xd4b9fc59,0xad215e80),LL(0x1343040b,0x88ef261a),L_(0xb9864942)}},
   {{LL(0x8eee1fca,0x0f601216),LL(0xc3ce93c3,0x8e99383b),LL(0x40d758a5,0xc0aa59c7),L_(0x54ae6327)}, {LL(0x9df0859e,0x111b9e3c),LL(0x09cd3a96,0x2e522f76),LL(0x6bcd40ef,0xb04993c8),L_(0xbcda6d48)}},
   {{LL(0x0deb6f15,0xbabee791),LL(0x030c3495,0x66179f3c),LL(0xd426f7b5,0x457bca6d),L_(0x81319526)}, {LL(0xd35cd40e,0x2498c158),LL(0x0b2bd04a,0xaec4ff20),LL(0x12985a92,0xcff74849),L_(0x837d8e0e)}},
   {{LL(0xf9b841f9,0x139de2db),LL(0xf41adf6c,0xf5c32cc1),LL(0xb7dc7dc5,0xede2ea87),L_(0x6e2033f0)}, {LL(0x706a5efb,0xcd213864),LL(0x287f4ece,0xdccf34ae),LL(0x50f400e9,0x0fc6614f),L_(0x89db6d9e)}},
   {{LL(0x4782e689,0xc42a6b83),LL(0x9e9a2ddd,0xa767b1e1),LL(0xf3f5ab2e,0xabdd54a4),L_(0xc69f9a88)}, {LL(0x5eeb634c,0x8affd662),LL(0xb4602516,0x434e3998),LL(0xc75f7a3f,0xfbd56bb7),L_(0xe4aa58aa)}},
   {{LL(0xfdddeaed,0xbb083ae3),LL(0x96aff3db,0x8b900a50),LL(0x2d9b4c87,0x77208838),L_(0x2b1d390f)}, {LL(0xeb40774a,0x408ee7f4),LL(0xc9a98c1e,0x491fbb55),LL(0xb6789b9e,0xf0a9cf2d),L_(0xcc0aecb1)}},
   {{LL(0x4d8759b7,0xf99e1bc1),LL(0xf9d97ac1,0xe489d308),LL(0x31242e73,0xe520dfcf),L_(0xba519709)}, {LL(0x502d066a,0xaada88da),LL(0xdf453d98,0x2db805ff),LL(0x48d83f43,0xa35584c3),L_(0x7efe383e)}},
   {{LL(0x1ce90c15,0x18351405),LL(0x987fc6a6,0x16ded7d8),LL(0xf2b213ba,0x17bc0ecf),L_(0xd5c3d1eb)}, {LL(0x4a42fef2,0xc603801c),LL(0x037728cd,0xbe4db3d6),LL(0xc5cf397d,0xde6243a1),L_(0xaf004bfd)}},
   {{LL(0xee10b9c9,0x81c21906),LL(0x140e4fb4,0x5c74da45),LL(0xb07d29b4,0x65c368fc),L_(0x6829786b)}, {LL(0x7aba9db1,0x7c01e8b7),LL(0x5c2ad4ba,0x597dbeb5),LL(0x8b692603,0xca528c39),L_(0x09175739)}},
   {{LL(0xd1bd09d7,0x3e778b35),LL(0x332aa21b,0x97ef919b),LL(0x7761d7c9,0xf6af5223),L_(0xded7e546)}, {LL(0xe8470307,0xde909622),LL(0x254a8b2a,0xe9be14c8),LL(0x6fa060cc,0x7cccaea0),L_(0x723e3222)}},
   {{LL(0x6fa0e5d8,0x00e54d7e),LL(0x80274626,0x53d70b13),LL(0xf63104d3,0x448cb4f0),L_(0x86f7822c)}, {LL(0x567fec1a,0x635d1d19),LL(0x74c9d852,0xad5841d3),LL(0x2f27a18d,0xa7638f56),L_(0x7df0f814)}},
   {{LL(0x2f7d4c38,0xe3919fa1),LL(0x11b8e6be,0x426c3ac2),LL(0x2038bc39,0x1a7c24cb),L_(0x547f1d30)}, {LL(0xad339016,0x4937319b),LL(0x14498e9f,0xe9acdbc9),LL(0xd9e8ae36,0x71231dd7),L_(0x099cc29c)}},
   {{LL(0x0f0c5086,0xbfa21d69),LL(0x9934b257,0x0d29b38c),LL(0x2552b462,0xfad28b1a),L_(0x87244f2d)}, {LL(0x042aa2fb,0xe091c63b),LL(0x224579d6,0x86f0f8da),LL(0x7927774b,0xa4177fae),L_(0x39e5970f)}},
   {{LL(0x8ecd5fa1,0xf8a953db),LL(0x00e1559f,0x8ef3b9f7),LL(0xa70e63b9,0xf8851e82),L_(0x48191bdc)}, {LL(0xefc3b637,0x6d398f41),LL(0xc96682f5,0x40655248),LL(0xc2927516,0xd7d600fe),L_(0x6435359a)}},
   {{LL(0x4a966ff2,0x43c60bf9),LL(0xbe98cc32,0x94a73ea6),LL(0xe899fd7f,0x6f6a67d4),L_(0x4805ca93)}, {LL(0x49a69773,0x532ddf8d),LL(0xb5f6725b,0x4785c103),LL(0x9818fcf0,0xa6151612),L_(0xa9419c51)}},
   {{LL(0x351340ab,0x544bdd84),LL(0xf568cb9c,0x1359eeae),LL(0xe070e7d3,0xa9680c77),L_(0xf00849de)}, {LL(0x3f049e65,0xc159fff4),LL(0x9f8b9614,0xc00e514e),LL(0x0503644f,0xf325f0bc),L_(0x58ef4f26)}},
   {{LL(0x0afb5680,0xbb5a090b),LL(0x2a905098,0x29e6e124),LL(0xeb89d752,0xb46f0674),L_(0xc23b6e08)}, {LL(0x7d46e8c8,0x84d934a5),LL(0x7446341f,0x814623ef),LL(0x2d200178,0x428fe6ef),L_(0x663531ec)}},
   {{LL(0xc36811cc,0xf0622226),LL(0xe7d3d99c,0x26dbef18),LL(0x87323666,0x0421f1d4),L_(0xb81113de)}, {LL(0x2f1ac158,0x121ae440),LL(0x14b9bfe5,0x7f1d1f81),LL(0xab8ce457,0xfb0b5381),L_(0xeebf8b75)}},
   {{LL(0x1ed0f16e,0x4ab996f9),LL(0x3bca93de,0xf3972d52),LL(0x9c224825,0x51f0a250),L_(0x4390fa94)}, {LL(0x52d41c04,0xa17f9bcf),LL(0x87ca94cd,0x1da254f1),LL(0x8eb876e4,0x8c10d532),L_(0x6ab8e456)}},
   {{LL(0x251d18df,0x00191031),LL(0xbe58c480,0xfb757330),LL(0xf0df6b45,0x51d6cd1a),L_(0x3a2b0d2e)}, {LL(0x72d910fc,0x6e0dcc32),LL(0x17ad59a0,0xd56ef620),LL(0x24ff8cf7,0xb1342fd7),L_(0x514eda6b)}},
   {{LL(0x3d2d0473,0xaede35a8),LL(0x43fadb36,0xc3bc4ec3),LL(0xda078971,0x89ff847e),L_(0x757ad40d)}, {LL(0x3367d69b,0x063bed04),LL(0xfbec9b9f,0x96916bd1),LL(0xa2938401,0xed5c65dd),L_(0xd3c116f5)}},
   {{LL(0x4ab81e63,0xbf6d76ab),LL(0x4a94b590,0x92ace598),LL(0x43e3ebf0,0x6d7848f4),L_(0xe0fe0161)}, {LL(0xd3008b3b,0x503ad71b),LL(0x14c43727,0x1bf34193),LL(0xafbcb091,0xbb5f8c82),L_(0x3bf8d893)}},
   {{LL(0x9e3f46b2,0xa2d0d277),LL(0xd8be2204,0x6e8eb285),LL(0xc9fa15ec,0xac17e25a),L_(0x91b10a2f)}, {LL(0x897ae118,0x12c8ef7f),LL(0x087fa01d,0x919f4309),LL(0xb6f728bd,0xd198666a),L_(0xcef80aee)}},
   {{LL(0xf8d94d59,0x6112903c),LL(0x89add916,0xda5061e7),LL(0x1c13960f,0xf15aa481),L_(0x48b52034)}, {LL(0x16623c8b,0x2d1bf700),LL(0xb5d87e2f,0x9a5f3c4f),LL(0xc7e5ba8d,0x6e4a90fb),L_(0xfd04fbda)}},
   {{LL(0x16d9fe00,0xce2f6f2e),LL(0xae50cdc8,0x205e3401),LL(0x5c73b50f,0x59a0da00),L_(0x6b833239)}, {LL(0xb1b01bc6,0x54c575fa),LL(0xfdd6dfac,0x38a3a688),LL(0x761a3a3e,0xb3095407),L_(0xca8c1a6e)}},
   {{LL(0x2aa45623,0x56e12b9e),LL(0xa3fc4119,0xd0bffaa7),LL(0xf5bdb564,0xe34b58ec),L_(0x178622a7)}, {LL(0xcda19c33,0x9e9ffc05),LL(0xfbbaa19e,0x51b7b9c6),LL(0x42f09c2b,0xb2d6157b),L_(0x0c83351e)}},
   {{LL(0x80fda4ab,0x623598c1),LL(0xae4a61b7,0x488f724a),LL(0x47014b1c,0xddd6a618),L_(0xc1476c47)}, {LL(0xbc2cc2d5,0x7eb38a43),LL(0x31a26c96,0x7b20f4d7),LL(0xc5ea2b2a,0xa5a136e8),L_(0x13b6043f)}},
   {{LL(0x3199bbf6,0x07cb2293),LL(0x210180c0,0xe5a1b306),LL(0x782987ea,0xab40f28e),L_(0x4a4f4f9e)}, {LL(0x95fa6103,0x591b5ef8),LL(0xf6832896,0xb357ced7),LL(0x38f56fa2,0x5abb2a56),L_(0x5a65a98b)}},
   {{LL(0xf1c770e6,0xca04bf7b),LL(0x022f4557,0x88cfe334),LL(0x657a19e3,0xc772f98c),L_(0xa19ab123)}, {LL(0x3ab78500,0x82db0eac),LL(0xd6b0ec00,0xf668d4b5),LL(0xdeadd9cc,0x9dab24ce),L_(0x1c2e2051)}},
   {{LL(0x0ac55884,0x7647407a),LL(0x489801b8,0xfa51e70a),LL(0x56034023,0x26d5313c),L_(0x29b7a999)}, {LL(0x834f99e6,0x6f0fc77c),LL(0xfdc7c2f1,0x647839fb),LL(0x006982c9,0xcb5af1f9),L_(0x9323d960)}},
   {{LL(0x91a088be,0x76954497),LL(0xa4de751e,0x393750b7),LL(0x6dc8702a,0x805eb5f7),L_(0x1cfb5ce3)}, {LL(0xaa971870,0x94bbb1b2),LL(0x43b64965,0x939fd8c5),LL(0xaf8d3158,0xc4fd0925),L_(0xeb438d66)}},
   {{LL(0x0e5bf508,0xc7a170c1),LL(0xb2973b56,0x43a0c49d),LL(0x9e4dbaca,0x2cb87666),L_(0xcf3bacaf)}, {LL(0xef7d14b1,0xa2effdce),LL(0x20a60ee8,0x586540be),LL(0x60509402,0x676121bd),L_(0x745c0ff2)}},
   {{LL(0x6d713b1e,0x3407acc5),LL(0xb6d85b13,0x4730880c),LL(0x7e9049e1,0x9fc877ce),L_(0x64c1005f)}, {LL(0xe9cfd783,0xb606ab7b),LL(0xe272e2ce,0xc2eb602b),LL(0x97ac2d38,0x954f3f1f),L_(0xa0e40e6d)}},
   {{LL(0x479a41f2,0xdf2474e7),LL(0x1f05663e,0x3574feb4),LL(0x234012a9,0x89989069),L_(0xc2481dbc)}, {LL(0x24c35b2a,0xf495b125),LL(0x452948bf,0x0325369d),LL(0xa43ea9bc,0x280bca9e),L_(0x8a34a29a)}},
   {{LL(0x2326ccfd,0xae198471),LL(0x848b5387,0x840ea47e),LL(0xcce778e1,0x4288a820),L_(0x34cf4dad)}, {LL(0x53f54930,0x9e011fed),LL(0x30c9f97b,0x29fd0d6f),LL(0xcba3ae4e,0xdb99670d),L_(0x1db11c4c)}},
   {{LL(0x59d694e8,0x9d825f38),LL(0x2980d39c,0xc504447d),LL(0x5a3ddebc,0x9c96ad62),L_(0x7f86e997)}, {LL(0x516005e5,0xac938f78),LL(0x7a80b2ac,0x3bbc6276),LL(0xe8b1101b,0x055eb004),L_(0x6c8daec0)}},
   {{LL(0x50fa66d9,0xcaf9f5b8),LL(0x88ad0652,0x4681401f),LL(0x35f6f9f8,0xbced09e3),L_(0x51bdbe8d)}, {LL(0x7b25a089,0xd40e66f7),LL(0xa071947a,0x63257022),LL(0x2ee8b520,0x9757d231),L_(0x4f80de84)}},
   {{LL(0x822c5f31,0xdf7efaf0),LL(0x3121c652,0x39218972),LL(0xd7749654,0xb988c422),L_(0x493a2d8b)}, {LL(0x76360207,0x98e08450),LL(0x2760de2f,0x8f841286),LL(0xcf07c50e,0xe5f81246),L_(0x1a00e240)}},
   {{LL(0x9a2b0329,0x9d26d17c),LL(0x89b4fc25,0x9bc053f8),LL(0xc765d35e,0x45f96752),L_(0xbfefae3b)}, {LL(0x6da731c5,0xf9a4089b),LL(0x22ecc154,0xcdbfce36),LL(0x5053964b,0xdcd4205f),L_(0x1e216001)}},
   {{LL(0x30c80608,0x1db56abf),LL(0x7ae4875c,0xacce82ac),LL(0x4a976988,0x4fa88f83),L_(0xd67fc048)}, {LL(0xd5a17ca7,0x49b6eedb),LL(0xedcaacaf,0x635d8bdf),LL(0xd26b609b,0x29f6f1cc),L_(0xa9ceab78)}},
   {{LL(0x8a159d18,0xf6f8a0a1),LL(0x95c6ea4c,0x766b602d),LL(0x61916ad2,0x8497eb60),L_(0xb0ce6cae)}, {LL(0x1a0eedff,0x50c80948),LL(0xa31d0581,0x811b1e1b),LL(0x4cec302c,0x0bfcd85b),L_(0x36484967)}},
   {{LL(0x37db6bc4,0x6e5e0c25),LL(0x6df2fa39,0x21b92740),LL(0x248a89e5,0xdcc52026),L_(0x67b816f2)}, {LL(0xbd55bfd0,0xee7ad3eb),LL(0x5bb4e55c,0x3cdb6fe7),LL(0xdaa70d9f,0xacc63cd9),L_(0x9155d520)}},
   {{LL(0x868a4cc8,0x6abedf02),LL(0x22a83e65,0xa18aad01),LL(0xb90ac85d,0x088ae4b8),L_(0x65e3e716)}, {LL(0x60fc849a,0xf19b0bfb),LL(0xfbebb9d9,0x9c3ff4ac),LL(0x7355b4ab,0xef331b93),L_(0xfcb3b647)}},
   {{LL(0xaf2dc94a,0x4e49fdb7),LL(0x76e1b0d3,0xf8a1df9c),LL(0x94293ef4,0x588501b4),L_(0xd8359bfc)}, {LL(0x1ef75655,0x3cd45f3a),LL(0xee63deea,0x2a08f333),LL(0x98af1b2f,0xd1f2806f),L_(0x12a331f8)}},
   {{LL(0xd2c7e263,0x3577e68b),LL(0x7e2770f6,0x355f7336),LL(0xe8913f4b,0x77ead765),L_(0xe027fdcb)}, {LL(0xac403db5,0x9a6f6902),LL(0xf459aa5e,0x8c922198),LL(0x24bf5d49,0x130316d1),L_(0xd808c825)}},
   {{LL(0x672ccabd,0xe048c7d8),LL(0x33437c46,0x674f7855),LL(0x2ce661a8,0xe77ba9cf),L_(0x46b5165f)}, {LL(0x9cd0b0e7,0xaa89fa66),LL(0xd3057aed,0xe440af7a),LL(0x142332ba,0xb60e8644),L_(0x3eb16842)}},
   {{LL(0xa0b4b4f4,0x35fc5d9b),LL(0x9f0177bd,0xc346b37c),LL(0x63dcb419,0x4fc638bc),L_(0x0d4a9edd)}, {LL(0xe8dc39c5,0xd3a1ffca),LL(0x2be34520,0x6520ec71),LL(0x6ee7420c,0x449c7ffa),L_(0xca54fb79)}},
   {{LL(0x391c4163,0x447c4950),LL(0x07a8fa48,0x212fb092),LL(0xfe67f50a,0x0d70f3ac),L_(0xdbc48919)}, {LL(0xec4b9c9f,0xc5e2455d),LL(0xc9b41ae6,0x306c9f8e),LL(0x1f329372,0xaab1f6eb),L_(0xa76cbb84)}},
   {{LL(0x50bb236e,0x7f11366c),LL(0xd38d4421,0x865c47f2),LL(0x9815fc32,0x1e6e5fa0),L_(0x42bf5236)}, {LL(0x980283c7,0x15c5a008),LL(0x03ac6a2a,0x13c27e86),LL(0x8f3eaacf,0x9eb61a43),L_(0x1a7ff3ad)}},
   {{LL(0x0c067d55,0xf6ee6117),LL(0x4446a48b,0xc9f90069),LL(0x08776dbb,0x62dd1551),L_(0x856288fd)}, {LL(0x2c6743db,0xb1d08622),LL(0x59e29beb,0x3e4f8e95),LL(0xca32b4fd,0x1744a1d3),L_(0xad3745b8)}},
   {{LL(0x4ef78890,0x3831190a),LL(0xaaeece81,0x44c9a76b),LL(0xd465548a,0x6dc6a194),L_(0xdbb627e1)}, {LL(0x18f3c7f1,0x91571bcb),LL(0xf5fb9008,0x31e00308),LL(0xf421b41f,0xeaac6d91),L_(0x01604108)}},
   {{LL(0x72bdb3aa,0xa2f0a871),LL(0x276d7a20,0x680da72c),LL(0xa823fb58,0x507ee8a9),L_(0xe6fef46d)}, {LL(0x9c541554,0xa385d033),LL(0x2785c811,0xdf53b551),LL(0xbcf3686a,0x8ac8344a),L_(0x3bf9ea41)}},
   {{LL(0x450260e1,0x8075a440),LL(0xfea1ac4e,0xe6e29d49),LL(0x0fedea77,0xb3c70edf),L_(0x2398f701)}, {LL(0xa3d35062,0xbbbf63ab),LL(0x9c1eb75a,0x62e94dcf),LL(0x9c3a2d57,0xee68eb75),L_(0x4d73d0da)}},
   {{LL(0xbcef4afe,0xe663f1d7),LL(0x31f69d33,0x6b81992e),LL(0x46cf68c3,0xc7fa33cc),L_(0xb4e3665a)}, {LL(0x0a54ba22,0x48f183b6),LL(0x154b0c79,0x17fddba7),LL(0x32607f61,0x1e044a43),L_(0xc1030470)}},
   {{LL(0x5565823c,0x25aa9de0),LL(0x88786097,0x3798fe3f),LL(0x045b969a,0x2e64a0a5),L_(0x921c1980)}, {LL(0x89334dcd,0x8b058ded),LL(0xbe46c84c,0xbe78b8bd),LL(0x5b271860,0xfe16e851),L_(0x48f0acc8)}},
   {{LL(0x11ee6162,0x31bafd7e),LL(0x7c0c6e0d,0xc8efb7fd),LL(0xe2d4fff9,0x2b1da468),L_(0x95ee5ad6)}, {LL(0x82ecd1e6,0xfebeb584),LL(0x307f35dd,0x3c68f532),LL(0x406d6f3a,0xe2245982),L_(0xd6be6d55)}},
   {{LL(0x979df4a2,0x002c4e27),LL(0x4f02beba,0xf600aa2e),LL(0xc26d73f0,0x21de614f),L_(0x38b705d3)}, {LL(0x13b06b78,0x23429db6),LL(0xf4967923,0xa3e9a4f2),LL(0x198f9eed,0x9eb6af93),L_(0x6bb259cc)}},
},
/* digit=2 base_pwr=2^14 */
{
   {{LL(0x6f5b4a5a,0xdcabdf3a),LL(0xd382ee31,0x1e2db542),LL(0xeeccf680,0xe97abee6),L_(0xc6941824)}, {LL(0xc6749744,0x47a3192f),LL(0x7f45fc06,0xc0f578ba),LL(0x7b1af995,0x4d9341e8),L_(0x248e916d)}},
   {{LL(0xdde84f14,0x082378c2),LL(0x4bf3bcae,0xaf15229f),LL(0x82c65899,0xb15f6e52),L_(0x5919e8e7)}, {LL(0x6798e12c,0x6c251212),LL(0xce97fdbe,0x696c3757),LL(0xf8d74875,0x1738524d),L_(0x44e69e4f)}},
   {{LL(0x9c3b9c4f,0x54e9eb41),LL(0x1bc6447d,0x2ee02fd1),LL(0x65482175,0x63694975),L_(0x72869bff)}, {LL(0xb7191008,0xf6aea0b9),LL(0xa70d16cd,0x5bd0a826),LL(0xa212780d,0x14730725),L_(0xdc25b909)}},
   {{LL(0x09a50ad3,0x8d673442),LL(0x01c6af2b,0xc4f517cc),LL(0x24e6fbcf,0xc6a67d3a),L_(0xbbe5e1ed)}, {LL(0x8492568c,0x964e75d1),LL(0xb97b0d10,0x09dfd2ce),LL(0x09007531,0x67a932fe),L_(0xc85f611d)}},
   {{LL(0x7d086ac1,0xfaef942c),LL(0xfc2eb46d,0x4a2d4ce4),LL(0x51096873,0x75df38bf),L_(0x75ab062b)}, {LL(0x6825bc8a,0x1f5151d2),LL(0x587ef818,0x0a14774b),LL(0xf3f93877,0x0eae8822),L_(0x5b5cb708)}},
   {{LL(0x2f159690,0x36982470),LL(0x165fe970,0x2b470b0b),LL(0x2ff27509,0x89cd99ad),L_(0x98c66330)}, {LL(0xc33eb1a9,0x0397edc7),LL(0xd2a0cfc6,0x868918fb),LL(0x7e36813d,0x722feffe),L_(0x829c6b89)}},
   {{LL(0x5c8de701,0xe76a6990),LL(0x394ea602,0x568ac304),LL(0x1348f9f4,0x34e40071),L_(0x3b2e0279)}, {LL(0x6e909cab,0x5da0385c),LL(0x0b77484a,0xd9ed956a),LL(0xbbb8ff2e,0x7e2ac222),L_(0xf54f007f)}},
   {{LL(0xe5f3820b,0x306867f0),LL(0x271f814c,0x5fe6163a),LL(0x455790c2,0x038f4fe6),L_(0x58008770)}, {LL(0x2a2eebb4,0xa3ffa255),LL(0x2ae5134e,0x4eecbe86),LL(0xfa061ce0,0x2bb3a978),L_(0xa4fe1b53)}},
   {{LL(0xcfa6c306,0x1840dd3f),LL(0x24b14e0e,0xbd1f6ada),LL(0xc72a3b9b,0x5ca5c5d6),L_(0x8625f308)}, {LL(0xdf5f8793,0x6afba384),LL(0x5e8c195d,0xd59557a2),LL(0x6bdd2d33,0x0975350a),L_(0xeff53d99)}},
   {{LL(0x35527ee1,0x3fb95972),LL(0xc84c9bda,0xaea90e2e),LL(0x172afe3a,0xd069e505),L_(0x2d143bfb)}, {LL(0x7231a957,0xaf5715f4),LL(0x9c2db5f2,0x7e8468f0),LL(0x6e7b80b4,0x71ec23d9),L_(0x4f3f9687)}},
   {{LL(0x1034850b,0xf5d591b7),LL(0x0c9c04f2,0x3e07210b),LL(0x6d31163d,0x530f24db),L_(0x66c42b68)}, {LL(0x7457ed53,0x1378c67e),LL(0x777097e8,0x8164d6d2),LL(0x3f1062b9,0x9ee5175c),L_(0xb09a9492)}},
   {{LL(0x20bad002,0x12b71603),LL(0x0ee3063d,0xd5045d6c),LL(0x2261149a,0x853772f7),L_(0x8f349671)}, {LL(0x9ec1ef24,0xdd3b39e1),LL(0x3acb7c05,0x02d7c1cd),LL(0x34c3e767,0x9381a0b0),L_(0x74081fec)}},
   {{LL(0xfcd9c832,0x49667bc0),LL(0xd57108d0,0x9658b33b),LL(0x6834cfcd,0xf7fb9009),L_(0x9b0baa9c)}, {LL(0x352f5036,0xd797b7f8),LL(0x13c1353f,0x3152ce4a),LL(0x6f581f80,0x48a60c42),L_(0x7f9e960b)}},
   {{LL(0x00259aa6,0x474d3703),LL(0xcfd42398,0xd3db7f63),LL(0x1c1a54ad,0x3c4fd36b),L_(0xd3eac930)}, {LL(0x79efe57a,0x7c4dc3a3),LL(0xdce7096a,0xfbb39dec),LL(0x732dfde0,0xa1787513),L_(0xe804b476)}},
   {{LL(0x0535a13e,0x926b6b5c),LL(0xe5f713d5,0x96e21c9c),LL(0x3d019dde,0xd52c15ec),L_(0x3e7f7e62)}, {LL(0x883a0c0e,0x6609394e),LL(0x694f2696,0x9ff80d9a),LL(0x69aa76bd,0x07cee438),L_(0x234e3949)}},
   {{LL(0xae230adf,0x90a49b52),LL(0xf501aeb8,0xdfc8b644),LL(0xc50fe003,0x8b89e79b),L_(0xa04a6f7a)}, {LL(0xc225c287,0xff3922dd),LL(0xc4ed1911,0x0cd4b29f),LL(0x0ce76929,0x8af23a3e),L_(0x918974b5)}},
   {{LL(0x3d52ecd2,0xdfd72a7f),LL(0xdea21ead,0x7da62126),LL(0x8e3f7ba7,0x1f678c3b),L_(0x6df4e74e)}, {LL(0x22dc8168,0xc5a010e6),LL(0xd7eea56a,0x54e30f4f),LL(0x389d8453,0xc6ed7b9d),L_(0x2e489cfc)}},
   {{LL(0x80f6e62d,0x914f8bfd),LL(0xbcee9591,0xa0be9a1f),LL(0x7fab5561,0xec5af129),L_(0xd4d265a9)}, {LL(0x3327e887,0xc74bfc93),LL(0xd2bab14c,0xe716458b),LL(0xe6b2e9c5,0x77d00d82),L_(0x161eafab)}},
   {{LL(0xe139a461,0x8fb0ef8c),LL(0x5b2aa48c,0x622967d6),LL(0xe9937cc2,0x00ef4c97),L_(0xb862b86d)}, {LL(0x42bfb0a9,0x97617d08),LL(0xfe0c2928,0x4f55cab8),LL(0xce44e424,0x625a38ef),L_(0x8594c3af)}},
   {{LL(0x239b2f0b,0x1b735668),LL(0xbbf9ef82,0x066337ae),LL(0x754e211c,0xf058688a),L_(0x95864d69)}, {LL(0x39a1596b,0x8beee136),LL(0x9150359b,0xd3b81681),LL(0x0ab949fd,0x3f4ac39d),L_(0xf6d5f8ee)}},
   {{LL(0xad05618b,0x4aaed3cb),LL(0xb594cda0,0x3d341ce5),LL(0x0216bf14,0x9e271d0d),L_(0x7efe2b03)}, {LL(0x68f95031,0x6a084fdc),LL(0x824376da,0xcda2825c),LL(0x6acd2b86,0x2deabee7),L_(0x55b4713a)}},
   {{LL(0x7d01995f,0x9b5408d1),LL(0x5ffa275a,0x5103e5b4),LL(0xfa6d3b64,0x3f3eff6d),L_(0x5bf0abda)}, {LL(0x13a776d1,0x97b27239),LL(0xcf4c8232,0x22a3f2f2),LL(0x170c8de4,0x047d7294),L_(0x43f7fc78)}},
   {{LL(0xab797a07,0x0b00d0bd),LL(0x4d42a1fe,0x92ead9f6),LL(0xc3907782,0xd65498ad),L_(0x716502c0)}, {LL(0x814b71a4,0xaa83d0a9),LL(0xd8e4c681,0xaf90bf17),LL(0xb2730e91,0x3b7c8d6f),L_(0xc3d966b3)}},
   {{LL(0x8da671b5,0xfee99849),LL(0x2e099be3,0xdbbe577f),LL(0x5f8e6df6,0x31915ae6),L_(0x2ba98b17)}, {LL(0x0d5b6ffc,0xe2f20ab0),LL(0x791aa104,0xdc69cc94),LL(0x2ae23401,0x5c5045ab),L_(0xd646c528)}},
   {{LL(0x93077179,0x50b8339a),LL(0x2874cd90,0x05d7c4a6),LL(0xcb1961a7,0x221ee1b5),L_(0x43f06891)}, {LL(0x0a41aae9,0xf9998038),LL(0xeac42395,0xd6ac7543),LL(0x042a0af5,0x36489fc2),L_(0x5fdad7bc)}},
   {{LL(0x3c56133d,0x9f01cd48),LL(0xc0ba445f,0x29263bac),LL(0x3e5d8713,0xc6107ee1),L_(0xecfad04a)}, {LL(0xb648509b,0x45f6365f),LL(0xeda195e8,0x4716fd8d),LL(0x2f2caf71,0x4fc9d505),L_(0x0c644ae1)}},
   {{LL(0x510aa881,0xbc63e799),LL(0x658bc0b0,0x4716a31d),LL(0x2713db77,0xabf827dd),L_(0x12824e7f)}, {LL(0x5c30bb60,0xb2616194),LL(0xfcdeef4c,0xf6fd28f3),LL(0x3fa68909,0xe9395e01),L_(0x42dfc220)}},
   {{LL(0x91f7da3d,0xe6bee7b8),LL(0xb1959a03,0x303e2e61),LL(0x9b779901,0xe7e57056),L_(0x1c2d553f)}, {LL(0x9abaa6a1,0x47995b48),LL(0xab4d653f,0x6d28c851),LL(0xf8e40f18,0xb70d2132),L_(0xbaae5b9b)}},
   {{LL(0xc8d7fa04,0x325e475b),LL(0x8ad5b01f,0x8d0c0363),LL(0x8fb55f0b,0xa691b273),L_(0xee832613)}, {LL(0x3a711dab,0xb146b4e3),LL(0x748d793c,0x81182539),LL(0x2911a867,0xdb940971),L_(0x24df2ed3)}},
   {{LL(0x0b071d1f,0x25961d64),LL(0x87754320,0xc47ba4de),LL(0x15d5a2ff,0xee280f81),L_(0xfc4877e3)}, {LL(0x1fbba89c,0xd08afabd),LL(0x8dd2825d,0xc881c4eb),LL(0xa36fe3f2,0x20840934),L_(0x398c3272)}},
   {{LL(0x13806339,0x51502992),LL(0x5f9bec8a,0x1bb87fb3),LL(0x878f8530,0xbcaa2cbf),L_(0xe99dba0e)}, {LL(0xde118aef,0x7a185dc3),LL(0x7f95d77f,0x2ea5905d),LL(0x3f419246,0x2b04f2d5),L_(0x7b04c078)}},
   {{LL(0x0fd96b35,0x597ba0a8),LL(0x0eb981ea,0xb6624fc6),LL(0xe6e26240,0xbfc4dac6),L_(0x35dd20af)}, {LL(0x61745187,0xe7b40563),LL(0x8d381cfb,0x554c1adc),LL(0x4013b9ec,0x354e81ee),L_(0x50a8d84c)}},
   {{LL(0x098f1196,0xccc6dc56),LL(0xe6735e77,0xe9fcb8bf),LL(0x629efebf,0x2ffffa9f),L_(0xc520680b)}, {LL(0x6d812701,0x732a53da),LL(0x4d7d1ede,0x248c465e),LL(0xa1f8c1a5,0x3dffe6e5),L_(0xc519ac0b)}},
   {{LL(0xcaf0a9d6,0xaef46030),LL(0xdd8a57c6,0x739000fd),LL(0xb0540672,0x8ef45968),L_(0x7dc6abd9)}, {LL(0x33da5d1c,0x5aafa6a3),LL(0x60865fae,0x6662d3dc),LL(0x9c28cea0,0xd3cb6b76),L_(0xbbb47176)}},
   {{LL(0x93070274,0xfe93910c),LL(0x86c3c000,0xd939b811),LL(0xe4bdec34,0xa7194bcc),L_(0x308782a6)}, {LL(0x640b22a9,0x710f1668),LL(0x1c395e5f,0xec57d6a5),LL(0x2f41d5d8,0xe7981956),L_(0x53f886b7)}},
   {{LL(0x67aad45f,0x6d08786e),LL(0x2cc15260,0x8bf6b065),LL(0x84169282,0xa53ab3f4),L_(0x49111025)}, {LL(0x714a4d69,0x8db73ccf),LL(0x49ee5fc0,0xaec354d3),LL(0xeb0a609a,0x26497f34),L_(0x752e2221)}},
   {{LL(0xbc1c0e24,0x23ccf8a7),LL(0xb6e96781,0x8d254256),LL(0x183ac489,0x907018f8),L_(0xf0ef92fc)}, {LL(0xc98e1d84,0xac3eb092),LL(0x63c56eb0,0x61b00609),LL(0xf8deac28,0x95f55a52),L_(0x5ae62b60)}},
   {{LL(0xa101bad5,0x8575f396),LL(0x643e6356,0x75613315),LL(0x3b84d8ca,0x87159fb6),L_(0xb1bb3787)}, {LL(0xeb901a9a,0x04257ae5),LL(0x76cc23a1,0xc722a719),LL(0xf70dd339,0xd9298acd),L_(0x91c26db4)}},
   {{LL(0xbcfc15f8,0xac24fa28),LL(0xa4b011f9,0x9cf8c10a),LL(0x77f2a9a2,0x897d9044),L_(0x0ed14d52)}, {LL(0xe210939f,0xa6397972),LL(0xe2308c70,0xc34aac84),LL(0x72030316,0x664a26ca),L_(0x291e9903)}},
   {{LL(0xd67a8d4c,0xd3c9cb6a),LL(0x38755541,0x115ea4ff),LL(0x4ad0effb,0x5c237f2b),L_(0x75646bb1)}, {LL(0x1c0c1001,0xa07ca1d3),LL(0x7f647322,0xc00e9dcd),LL(0xe24ec66d,0x819d18b8),L_(0x95c255a0)}},
   {{LL(0xd48e2f95,0x8209864f),LL(0xddb2d835,0x9d8a81eb),LL(0x41d055f3,0x92d42e0f),L_(0xe2f17df9)}, {LL(0xdaf29a59,0xc2c8d2a6),LL(0x061a667a,0xd108615b),LL(0xcc790c1c,0x0eb1df27),L_(0x6dc9536e)}},
   {{LL(0x5d7251ce,0xf7ca409f),LL(0x55a94793,0x8791f913),LL(0x6bbc768b,0x1ef210b3),L_(0x166d8931)}, {LL(0x24baa65a,0xa0bcf014),LL(0x621f63a2,0xa8abe2d3),LL(0xf790fc9d,0x60203dd4),L_(0x667b1fa7)}},
   {{LL(0x77f64ca2,0x2884e5da),LL(0xc59a883c,0x38a09cd0),LL(0x47f89d65,0x780aeefd),L_(0xd528d23d)}, {LL(0x9945faad,0x90523929),LL(0x6e9ee32f,0xa460502a),LL(0xb5fe2f01,0x1d21d327),L_(0x2883e925)}},
   {{LL(0xc3ff6cb2,0x8d7cbdb6),LL(0x32e5cd9d,0x2baad720),LL(0x2385860f,0x43f0f911),L_(0x4b6038f7)}, {LL(0xe096aed9,0x8ece6e04),LL(0x40da7078,0x964d3a26),LL(0xa20a6649,0x1a09dbaf),L_(0x709cfd74)}},
   {{LL(0x4a5af8b1,0xa2ce00d8),LL(0xa63a8b02,0x57b8f935),LL(0x35cb5ede,0x706b0821),L_(0x4173e4a2)}, {LL(0x74594a56,0xaae1c286),LL(0xaa0caca3,0xc1e054bb),LL(0x3c3b3e24,0x34698533),L_(0xe3b66255)}},
   {{LL(0xcb1b2f2a,0x4bf57c51),LL(0x164519e7,0x6b4ba459),LL(0xb55fe2b9,0xd37e29a7),L_(0x68e8549f)}, {LL(0x9b0eb441,0x9f4b4c99),LL(0x8f691f6b,0x59078e54),LL(0xa382964e,0x49524978),L_(0x7d3fe7f2)}},
   {{LL(0x3a14e17c,0xb90d4bd4),LL(0x087b646e,0xac93ca7a),LL(0x53865f43,0x2d659d4e),L_(0xb9db500d)}, {LL(0xdc91bb2e,0x17597d7a),LL(0x6ae5d43e,0xb1613ba6),LL(0x3a5e9fe1,0x4204e314),L_(0xd91a271a)}},
   {{LL(0xbf7ca431,0xcfce86aa),LL(0x16c2854c,0x51585545),LL(0x3999a9a1,0x5fd18a95),L_(0xa1e55106)}, {LL(0xd29f7bb5,0xd7192594),LL(0x7c17c721,0xb7a8c7e2),LL(0xf6856acd,0xdb82b841),L_(0x39017815)}},
   {{LL(0xe9631773,0xf155113d),LL(0xc2b2be30,0x09bb3adb),LL(0x1e571839,0xa98b5f59),L_(0x17a7266e)}, {LL(0xde672c40,0x02e9f20a),LL(0x9c53a705,0x68d00619),LL(0x476b9b10,0x144091ad),L_(0x98e56e88)}},
   {{LL(0x4746f916,0x02f6a9c6),LL(0x461d2e37,0x6a692ac8),LL(0x6fe1c2f9,0x8835c958),L_(0xaae29f67)}, {LL(0xaa57aa25,0x1573f8b0),LL(0x2aae0885,0x8359860b),LL(0xc9a1230d,0x930776e9),L_(0xd3f32fd7)}},
   {{LL(0xda3f170d,0xdbf89586),LL(0xfeac4726,0x6f39cb19),LL(0xe907285f,0x084c5773),L_(0xc72ade03)}, {LL(0x2f55c6f0,0x155c7e90),LL(0xe4fb90c2,0x50f9f0f0),LL(0x833af29a,0x87b3ac91),L_(0xeeb712d2)}},
   {{LL(0x1b2f2123,0x08fa5175),LL(0xad400dee,0xe9ac51f9),LL(0xdb50d10c,0x8bbf71ad),L_(0x1322cb4e)}, {LL(0x3efa91c3,0xb7ef36d2),LL(0xa5571f80,0x4c6187ca),LL(0x938aa540,0x6a106d1c),L_(0xfee7b99f)}},
   {{LL(0xc717d42e,0x0d738fbd),LL(0x9340ad37,0xf74cbf89),LL(0x923ef854,0x1a934a84),L_(0xdc9bcb14)}, {LL(0xf0ce91b3,0x3ed1d40f),LL(0x3e3d69c5,0x35833a46),LL(0x8cf302d1,0x60492caf),L_(0xee0ad716)}},
   {{LL(0xf9f98ca9,0x310d3206),LL(0xc40b6b35,0x67ddb804),LL(0xf6b370b9,0xcf2da667),L_(0x4a7e6061)}, {LL(0x57730157,0xd8d48240),LL(0x186ae81f,0x002b8bc3),LL(0x2133c198,0xa42e3f3d),L_(0xf8b9b6ca)}},
   {{LL(0x1ba04233,0x9ac56b2a),LL(0x55759656,0x927ea140),LL(0xd47c70b0,0x0f258220),L_(0x06bffc7e)}, {LL(0xe94dda94,0x453179d9),LL(0x92ffe53d,0x2b54517d),LL(0x2ac0af28,0xb924dbee),L_(0x7afd903c)}},
   {{LL(0x7c5bc163,0x7d2123b3),LL(0x61435520,0x1d58d680),LL(0x2b094145,0xeb4a95b5),L_(0x09b8c6a3)}, {LL(0x3c690f29,0x54e52454),LL(0x646b89b4,0x6bd00cf3),LL(0x2640a633,0x17b930ad),L_(0xa457233d)}},
   {{LL(0x05fb72f6,0xbfa90c2b),LL(0xfad26494,0x9f2c3cd4),LL(0xbfe60269,0xcd907456),L_(0xb3e48eca)}, {LL(0x6374f2bd,0xdf9ab844),LL(0x94fe7551,0x36c69779),LL(0x8ed01908,0x5a97c45a),L_(0x5dbf5641)}},
   {{LL(0xa14499cb,0x9dc2ad99),LL(0xe9019e0a,0x64554a27),LL(0x588acbee,0xfff03f99),L_(0xc48117a2)}, {LL(0x46bb4b9c,0x6e713362),LL(0x59d13072,0xfb7747ec),LL(0xc63c7628,0xefa02cb7),L_(0x39d7f934)}},
   {{LL(0x69d702d8,0x5f6d09ba),LL(0x2361ada1,0x920262ae),LL(0x5419445e,0x6f750612),L_(0xa30fa392)}, {LL(0xcf460268,0xd59c34bc),LL(0x32785073,0x76b69628),LL(0x72c99605,0xb431b539),L_(0xeec4de29)}},
   {{LL(0xc39fd75e,0x97507c26),LL(0x0c5debbb,0xd82ad3a6),LL(0x3c42310d,0xa01f4d9e),L_(0x5fdffd07)}, {LL(0xd1fedd01,0x5e2bab3e),LL(0x4ce43e20,0x8abe3a41),LL(0xe71cd8a2,0x0c4858ed),L_(0x8bd12b19)}},
   {{LL(0xbb2c8805,0xb6380f03),LL(0x59339488,0x7a463ca9),LL(0x920e2111,0x0fbdd870),L_(0xcf927c12)}, {LL(0x9fa1e546,0x4ea01a50),LL(0x20ee4123,0x38b747c2),LL(0xeeedf23b,0x65fa7940),L_(0x82b08346)}},
   {{LL(0x6a22e6e3,0x34651055),LL(0x1098cb76,0x269d82cd),LL(0xd6a624c2,0x5fd458b6),L_(0xb856ae59)}, {LL(0x6c36cae7,0xd687ba3d),LL(0x33a82abc,0x6963f047),LL(0xa949d735,0x1de89776),L_(0xc44fd2cc)}},
   {{LL(0x23ce7894,0xc54039f7),LL(0x1ad9c99d,0xaba8109a),LL(0xa0ae59e6,0x5269a308),L_(0x65e4eb8f)}, {LL(0x63914a92,0xe212c370),LL(0x2c3e1f0e,0xcaf748c2),LL(0x8869d4e8,0xc88e85fd),L_(0xbc2a9592)}},
   {{LL(0x0874ad96,0x0f4e6e42),LL(0x2bfeb37b,0x3e5a96cd),LL(0x6bb45b1b,0x53c918ad),L_(0xb8bd0546)}, {LL(0xfd07d3ba,0x0e33f7bc),LL(0x9e58beb5,0x58f58e4a),LL(0x26578e47,0x986f5206),L_(0x90135d8a)}},
},
/* digit=3 base_pwr=2^21 */
{
   {{LL(0x089bae53,0x4190a603),LL(0xec5c8b6b,0xa64a64f3),LL(0x8bca6d9c,0xba9eddb4),L_(0x3ebc0dcc)}, {LL(0x9555731f,0xa709d0a0),LL(0xc3d3be2f,0xa147a9eb),LL(0xdae0ded6,0xf474acd5),L_(0x965b8988)}},
   {{LL(0xa9158749,0xfb05e5ec),LL(0x7df5ec60,0x5b8ef888),LL(0x2d8170fd,0xd08f651c),L_(0xe08fb325)}, {LL(0x1ccffe04,0xa2b2a173),LL(0x20fa30ea,0xe5f33422),LL(0xd169c10d,0xc7e20c19),L_(0x366327ee)}},
   {{LL(0x5d85ed76,0xf62aa1bd),LL(0xe359773b,0x38737578),LL(0x2b782c08,0x31df9439),L_(0x80e166af)}, {LL(0x673b4b07,0xa7a7bd40),LL(0x50ca47c9,0x95ad7602),LL(0x8ed45ebe,0x4a7eebb4),L_(0x03e9fc9d)}},
   {{LL(0x33fde338,0xaba113fc),LL(0x0e6fb34d,0xc0022adf),LL(0xe2da2a97,0x491a39ca),L_(0x54e2ec96)}, {LL(0xb883d291,0x3d0946d9),LL(0x6f792b36,0xd8794d71),LL(0xd78ca8ec,0x13094248),L_(0x286898ab)}},
   {{LL(0x0c894527,0xb354ae4c),LL(0xf18c6f77,0xa39fcad9),LL(0xf685126a,0x7bd039dc),L_(0xfb3809c0)}, {LL(0xe8ec1ffb,0xc99f8944),LL(0x7c71a341,0x08ed706a),LL(0x21f4ed22,0x4c371e86),L_(0xa650cd97)}},
   {{LL(0x64586632,0x8e7cfcb0),LL(0xfcae4f5c,0xe83d4510),LL(0x62196ce2,0xc9a4fa0c),L_(0x83e1de7b)}, {LL(0x4871e08f,0x8f0bc09a),LL(0xfa78f5d2,0x1e32cbe6),LL(0xda32a5ee,0x23d217eb),L_(0xf8df814e)}},
   {{LL(0xefdc00b8,0x80a1f116),LL(0x670e56db,0x0b205db0),LL(0xd1c521ab,0xcaa4c48a),L_(0x41b37b7f)}, {LL(0xde1b3d20,0xf6421dce),LL(0xafa1a3e7,0x906ec304),LL(0x5c6bc442,0xbc2935cf),L_(0xbe169cc3)}},
   {{LL(0xa597c946,0xe5876159),LL(0x308bf446,0xcc637e99),LL(0x8047e56a,0xdb6a36f6),L_(0xe8293346)}, {LL(0x51f3fb7c,0x93a30fbd),LL(0xcd2f5b41,0x3e0a22c7),LL(0x0b7b25a0,0x3d9fe740),L_(0xaafd7816)}},
   {{LL(0xe59f76ea,0xc167cd8d),LL(0x0d3f30f7,0x07b77d41),LL(0x3b140b84,0x6554fb45),L_(0xc105ccc8)}, {LL(0x72d13268,0x69c72227),LL(0xf0450467,0x0ba87831),LL(0x88e4350b,0x1e0bbd29),L_(0x6f9f2dff)}},
   {{LL(0x8eb3482c,0x1e9d015e),LL(0xf4db6e3d,0x3a6f5318),LL(0x60df1945,0x91e766b6),L_(0x2499a2cf)}, {LL(0xe58a4308,0x0468acbe),LL(0x7a3c83da,0x9eeaeaca),LL(0xfeedfeca,0x95c8fd77),L_(0x9f35aedd)}},
   {{LL(0xdf4b1836,0x1d85a174),LL(0xefd882f6,0x2cdd973c),LL(0xfcbc0529,0xff17a943),L_(0x6d3cb4e6)}, {LL(0x7dbddba8,0x1d223311),LL(0x680691e3,0x6b1fb1d6),LL(0x2d7d7413,0x3d6056c3),L_(0xf379f1df)}},
   {{LL(0x75f0d3c3,0x5a2e3385),LL(0x66a60d5f,0xa99c6fca),LL(0x08231783,0x2621fe6f),L_(0x282e92dc)}, {LL(0x9beb8a2d,0xa53267dc),LL(0x48813229,0x2abc54c5),LL(0x18f63ff9,0x25a10917),L_(0x06fdd515)}},
   {{LL(0xd5250739,0x866c8a6c),LL(0xb47ff8e8,0xe99cef30),LL(0x360eff22,0x64925640),L_(0x392f59d5)}, {LL(0x58d96359,0x18c35e2f),LL(0xe8c753dd,0x5ea13624),LL(0xeef6fc01,0xdf425955),L_(0x3afc5278)}},
   {{LL(0x6599bea3,0xedc68cb7),LL(0x69d044d7,0x3950ba3f),LL(0xb3b58152,0x8eb64dda),L_(0x49c378c0)}, {LL(0xfcacec23,0xaf6b16d6),LL(0x47e931cc,0xbc7d2577),LL(0xcb7189bc,0xa9051236),L_(0x2aca5249)}},
   {{LL(0xb4223fb7,0xad339986),LL(0x9bcc65a5,0xe221979a),LL(0x278964b2,0x7810f1da),L_(0xf8f2be96)}, {LL(0x225ed2c6,0x7e983036),LL(0x59fa9f23,0xfcfa0c31),LL(0xbc86f69d,0x918a9c4a),L_(0xb726eddd)}},
   {{LL(0xcabaac34,0xac70d023),LL(0xe4e20984,0x7fc6e9c4),LL(0x0ffac935,0xcce8d103),L_(0x6efdaa99)}, {LL(0xd5d8263d,0xea3876d9),LL(0xa0b5e29a,0x1319056a),LL(0x65ae7e78,0x9fa645d3),L_(0x93cb4927)}},
   {{LL(0x09cc8e5f,0xac6cb27b),LL(0x414e945f,0xb503f62a),LL(0x73d9e98e,0x4809a03f),L_(0x2d14c85e)}, {LL(0xbf213bd6,0x0b05bc27),LL(0x74a3b39b,0x4314e710),LL(0x003b8462,0xed2336e3),L_(0xc6f085ad)}},
   {{LL(0x3c7d36a1,0xd9687e2e),LL(0xd82a965c,0x7a3f9fe9),LL(0x088401a6,0x068fa075),L_(0x07ba4f19)}, {LL(0x0f0c8a66,0xf91cec2f),LL(0xb6cda991,0xb14b14fb),LL(0xa7b1febe,0x50d069b1),L_(0x5d3d8cd6)}},
   {{LL(0x10626da8,0xf8949e74),LL(0x5c71b954,0xbe5cd298),LL(0x28006acc,0x6ba62a11),L_(0xe08cb4da)}, {LL(0xe8cb22bc,0xc4d932f1),LL(0x9059e525,0xb085b630),LL(0x72f4258d,0x4cc4c854),L_(0x0a971745)}},
   {{LL(0x1acb63c1,0xdaf48ca2),LL(0xa543ba94,0xe2648935),LL(0xd71707e5,0x809ef386),L_(0xe30d9ba3)}, {LL(0x12555c2b,0x69eeeae4),LL(0x6910f9ee,0xea668f6a),LL(0xe8c90ba6,0x1896385f),L_(0x12c1a610)}},
   {{LL(0x9f5a846d,0x2d268c8b),LL(0xd0a57e22,0xf1a092e2),LL(0x907afb7b,0x927d9d98),L_(0x4fb8839b)}, {LL(0x54540741,0x967900e2),LL(0xa9ff207a,0x628af581),LL(0xee9dbe19,0x1bad27e2),L_(0x37d4f91b)}},
   {{LL(0xd5bb2ea9,0x6069565b),LL(0x7e61e032,0x973dd1c6),LL(0x1034f977,0xa6921003),L_(0xf41e571f)}, {LL(0x58bdabb3,0xeaa8bfa5),LL(0x77a67171,0xe71c31a6),LL(0xf9f34af4,0xa78106eb),L_(0x7aed89ac)}},
   {{LL(0x9de5d621,0x8c1c931d),LL(0x2befbabd,0x5b08557f),LL(0x7e64aafe,0xb97c5d4d),L_(0xf99509a4)}, {LL(0x10d5a953,0x7372eddc),LL(0x47ce48d3,0x24c1d7d6),LL(0x28b7f369,0x1fd874f8),L_(0xfdab8a95)}},
   {{LL(0x886ba3df,0x7f6337ce),LL(0xe5943606,0x7b3e3023),LL(0x21b5b09f,0xa28353d5),L_(0x917b4519)}, {LL(0xd1005ca4,0xfd5ad004),LL(0x73dbf462,0xc7335825),LL(0x7f25c9d4,0x3b50b66b),L_(0x35b3bf78)}},
   {{LL(0x9b2e5ded,0x4b668d6b),LL(0xbe5e047a,0xd58c1f6d),LL(0x2bc0144b,0xee508f32),L_(0x9d2116d4)}, {LL(0x7efb1e4f,0x5d9a5637),LL(0x8ab6e9cc,0xcd52f6ad),LL(0x954172ff,0x7ca23907),L_(0x9fe1053c)}},
   {{LL(0x0b042690,0x711be7f8),LL(0x4c525b55,0x549592f5),LL(0xac72c62f,0x962efa73),L_(0x92c58056)}, {LL(0xe8cda600,0xa38c53ed),LL(0x917ebbe6,0xd1e2d2e6),LL(0x6ab0ae35,0x25682606),L_(0x0603c2dc)}},
   {{LL(0xd25fb822,0xb1a2e953),LL(0x22b420c7,0x030fabe0),LL(0xffe309b6,0xda6ba55a),L_(0xdce7859f)}, {LL(0x9c46c82d,0x96d10a75),LL(0x057f1caa,0xd24573e9),LL(0x72ae9f6f,0x8794d9eb),L_(0xab64c20f)}},
   {{LL(0xfc0d12fe,0xb289f4d7),LL(0xd9e2a94a,0x133fa143),LL(0x4c9fd385,0xecfdb1d1),L_(0x0675dcf1)}, {LL(0xa5804aec,0xaccd6ef1),LL(0xfad7706e,0x577c5c78),LL(0x0667c6b3,0x7a5f797a),L_(0x2207359c)}},
   {{LL(0x03a638e7,0xf3f519e2),LL(0x06628ac7,0x06647de5),LL(0xe38c99a3,0x86774ade),L_(0x8f775fea)}, {LL(0xe57995ae,0x1bf25ffc),LL(0xf9325842,0xcccd3a9f),LL(0x34f4502c,0xabcab086),L_(0x064e829c)}},
   {{LL(0x27f04a80,0x5fefab0b),LL(0x06295d26,0x2a4f8979),LL(0x234674d8,0x9c70b1f8),L_(0x01e05c18)}, {LL(0x49f25ab4,0x352749e2),LL(0xee248ea9,0x9909ef81),LL(0x3b3c0b4c,0x7556a699),L_(0x4df0080e)}},
   {{LL(0xeb394e8f,0xc8a345b1),LL(0xaf868ebc,0xbf72406a),LL(0x9a7fcfbb,0x959c94fa),L_(0xea271ece)}, {LL(0xb72d1e17,0x1ea91da8),LL(0x252b7410,0x004276b7),LL(0x560e8135,0x8ada4812),L_(0x4d0eaa23)}},
   {{LL(0x857c7d7b,0x11e78f96),LL(0x55ff3762,0xcd19ea8e),LL(0xa90b8b11,0xb557ab25),L_(0x98bfd175)}, {LL(0x436277e5,0x453f5ec0),LL(0xbd8cc123,0x74f19c3e),LL(0xe53f50d1,0xaf6d61de),L_(0xb809c27b)}},
   {{LL(0x70f6cada,0x22d1e307),LL(0xcafcf6b1,0xc39e2653),LL(0xb5d40ef0,0x602ae873),L_(0xb5bef707)}, {LL(0x37b8abfa,0x47aa28a0),LL(0xc7e7a44e,0xfe824073),LL(0x9f424f8c,0x39a88470),L_(0x08171c6a)}},
   {{LL(0x80811691,0xd43ff3e6),LL(0x5bfd96fc,0xb502656b),LL(0xa9d5f891,0xeb92efe6),L_(0x5ac4853d)}, {LL(0x1f5ef580,0x90443655),LL(0xe8ead180,0xb1c5c5f0),LL(0x84a289e5,0x98bbea4f),L_(0x8bd0878a)}},
   {{LL(0x7035018f,0xd233645a),LL(0x2cec3fe7,0x567fe1b2),LL(0xc6974f80,0xc7a73902),L_(0x0b481a08)}, {LL(0x9adde606,0x9eb32485),LL(0x9b2ec275,0x16392d5c),LL(0xdb3bea46,0x0653d374),L_(0xab7f0ce3)}},
   {{LL(0xe1cac3cf,0x89e007fd),LL(0xa77a41f1,0x3300e242),LL(0x6059351c,0xed2ec819),L_(0x02bfa9e8)}, {LL(0xba1e89c6,0xa9065e31),LL(0xd4e23efc,0x6a1da501),LL(0x9c16b8bc,0x465d2c9d),L_(0xa1b1cda1)}},
   {{LL(0x9ecf8570,0x54df839d),LL(0xb93666fd,0xb9514f4e),LL(0x1fed47ac,0x7e8268de),L_(0xbda2efca)}, {LL(0x5da57404,0x428af14c),LL(0x9ee2b07c,0x1471974b),LL(0xda44c29f,0xfffd22e4),L_(0x296f1761)}},
   {{LL(0x917860b8,0xb42b17e2),LL(0x6d4c5ddd,0xf2571ed2),LL(0xabe94b27,0x89be3310),L_(0xde98e510)}, {LL(0x346eec5a,0x9d4ca35e),LL(0x96df2661,0x11588b07),LL(0xb28f2b86,0x96395d9e),L_(0xc8d9fdcf)}},
   {{LL(0x031a12c5,0xdbc075c3),LL(0x54d26d6d,0x6ab8c6ef),LL(0x23c171f2,0xca47c96b),L_(0xecedbb06)}, {LL(0x8c7cf280,0x31539156),LL(0xe9bac065,0xb7662b30),LL(0x552f0920,0xef1cbf3a),L_(0x2cb42440)}},
   {{LL(0x9fba2bd9,0xed539df1),LL(0xd5c7355c,0x718a4f45),LL(0x5ced43ea,0x10d60144),L_(0x9618c03b)}, {LL(0x55e1556d,0x8b9bcb4a),LL(0xe3767fbb,0xf1a46d41),LL(0x2a5c6f77,0x8a009344),L_(0xa23cb6b8)}},
   {{LL(0x0db9a0c8,0xfd4a0f59),LL(0x7ea78a85,0x4d12f34e),LL(0x258d360e,0x951a2674),L_(0x7b3935b0)}, {LL(0x4a231357,0x75e447f0),LL(0x80a9bfbf,0xc988c231),LL(0x8cd933d2,0x1c914702),L_(0x8da2914d)}},
   {{LL(0xbe785bb8,0xbb8f1e8e),LL(0xbb61ba00,0xa9e04405),LL(0x018ee54b,0xf3ed084b),L_(0x9b33e2c5)}, {LL(0x796aad53,0x90c49201),LL(0xdac991d0,0x9174f367),LL(0x06286fbc,0xfa92d2f7),L_(0x8e51716d)}},
   {{LL(0xce4d1ca0,0xcd0c67a7),LL(0xe3fcc974,0x5197d346),LL(0x4936eae1,0x38aec941),L_(0x49b20db5)}, {LL(0xf9c12e70,0x8806a867),LL(0xfd73d3b4,0xfe592936),LL(0x8553e67e,0xbb731824),L_(0x5f399b3f)}},
   {{LL(0x7afb78c1,0x04dbb078),LL(0xd2ce86a5,0xbe71dc62),LL(0x636ee6c6,0x5c029f32),L_(0x90ef1fcb)}, {LL(0xa90f74fc,0x05c18efb),LL(0x812addf4,0x86cc7880),LL(0xdf2ff660,0x34c03bb8),L_(0xb1740945)}},
   {{LL(0x642146f4,0x18fd7ff7),LL(0x8a416ad5,0x8b9a8120),LL(0xb7d2ecb8,0x8226e0af),L_(0xbb5fb220)}, {LL(0x7d62e2a5,0xcaee42bb),LL(0xb76b0d06,0x9ae1502d),LL(0x31322587,0x678f208f),L_(0xa07b89ea)}},
   {{LL(0xdd6c08b2,0x5899ed84),LL(0x45083585,0x50d7aafb),LL(0x7217b173,0x04f9e7f1),L_(0x59198da1)}, {LL(0x5dd6befe,0x11d6cf0e),LL(0x908d5768,0xa9128791),LL(0x51f15b45,0x7dd32a0d),L_(0x58045077)}},
   {{LL(0xafc5b319,0x885c6636),LL(0xf7bef8ba,0xc50dd479),LL(0x478b8ac8,0x58aa9fce),L_(0x9d21686d)}, {LL(0xe233d315,0x541006e3),LL(0xdfc86dcf,0xdfefb8ef),LL(0xe98f7f2c,0x73e745a7),L_(0xfc620932)}},
   {{LL(0xbda630b0,0xd49816e2),LL(0x68c388d9,0xabd8c48b),LL(0x741de91e,0x3b976a04),L_(0xf24259fb)}, {LL(0x5a923f4f,0xcbbf2684),LL(0x56659522,0x8a51b8af),LL(0xea31954e,0x43c8b17e),L_(0x3d1de35f)}},
   {{LL(0x1067b268,0xddd685da),LL(0xa178f2e3,0xe840e9b4),LL(0x859daa46,0xc3965546),L_(0x65f74f36)}, {LL(0x3ebc87fc,0x31952a50),LL(0xd0b694c6,0x2ba1d2d4),LL(0x4f51b3a3,0x20b2748b),L_(0x54b61458)}},
   {{LL(0x450b7ea8,0x91be987e),LL(0x92390774,0x530b2e3f),LL(0x31cf3730,0xd8dc53f9),L_(0x490c450f)}, {LL(0xe4e70ece,0x604fd792),LL(0xb689243f,0xa20bf532),LL(0xb2d8f482,0xdc31ef3b),L_(0x506dd3c6)}},
   {{LL(0x89cb2010,0x1196c48c),LL(0x6755b53e,0x59cee16a),LL(0x6a0e74ef,0x2da63e64),L_(0xaa8ea0cd)}, {LL(0x661a0e7f,0xb013649e),LL(0x1ab0635d,0x56e8f0f1),LL(0x9858a79d,0x65938865),L_(0x3acb5b92)}},
   {{LL(0xf39d8f51,0x28b2de13),LL(0x6b120542,0xd10ea1ae),LL(0x5ea493a4,0xbe9b2f46),L_(0xcdd3cbfb)}, {LL(0x9af8b9c3,0x543265a5),LL(0xf3a605c4,0xe8ee990d),LL(0x92bcbe09,0x1d71ae8b),L_(0x4ad0176d)}},
   {{LL(0x77ad60e7,0x183493f0),LL(0x3e8ef423,0x4d9cc856),LL(0x065effba,0xa9ccf6fd),L_(0xb0500b2e)}, {LL(0x122bf6db,0xfb31da5d),LL(0xe16bd325,0x3de8eb0a),LL(0x2bb46b58,0xcdab125c),L_(0x8179b8bc)}},
   {{LL(0x72f249a0,0x54823e45),LL(0x263960f2,0xb1c1bcb0),LL(0x21ab8a5f,0x193b2d98),L_(0xb940533d)}, {LL(0xc80b7cd9,0x7586193c),LL(0x671090ab,0x40d0049f),LL(0x216acee5,0x6bb91540),L_(0xcc4fc3a7)}},
   {{LL(0xc26826e4,0x0e69184d),LL(0xe86f6c8e,0xfa8a33b4),LL(0xe88688e1,0x9beec95a),L_(0xd008257c)}, {LL(0xa638dd0b,0x35395ef4),LL(0xacddfb2b,0x005b4fcb),LL(0x086efe98,0x8a1a0ad3),L_(0xb4d51fd4)}},
   {{LL(0x9613f782,0xd8510368),LL(0xd5dedcbd,0x7ea65e3d),LL(0xab06f6a4,0xbd599498),L_(0xc5aa099e)}, {LL(0xf3eb18eb,0x32f40f7b),LL(0xad93fbe0,0xfde611a8),LL(0x40b951f1,0xf5751752),L_(0xfc37b6e1)}},
   {{LL(0xc4dba0cc,0xaae77bea),LL(0x58b91be1,0x7a9f5f3a),LL(0x6ad767a4,0x9b47cd71),L_(0x24d37cee)}, {LL(0x7268bfef,0x70ac8b25),LL(0x2da13b60,0x16423d93),LL(0xbadf18ad,0x62a037c8),L_(0x78224e07)}},
   {{LL(0x334baa48,0x6a2f2198),LL(0x1578e706,0x50c74263),LL(0xa273c223,0xe95f128a),L_(0x2b3feceb)}, {LL(0x0c8d88cc,0x4d1450aa),LL(0x247fea5e,0xb5c004a3),LL(0xcc56e421,0x32f2d692),L_(0x001d3250)}},
   {{LL(0x63905c2b,0x3d1a2b08),LL(0x1192eb20,0x7a96aa59),LL(0x194cd936,0xcbfbf813),L_(0xe3ba3ce4)}, {LL(0xdc1fe33e,0x8d6355cd),LL(0x36393a36,0x3c74701a),LL(0x6d3b3d2c,0x50d98fcf),L_(0x2da52d36)}},
   {{LL(0xa9690fb6,0xd68d0be2),LL(0x87600722,0x2fa3c8b8),LL(0x6a898b0f,0x64b1c25c),L_(0xf26df29e)}, {LL(0x18b74321,0x0562d2ac),LL(0x116fb9cd,0x1b89331d),LL(0xadee6e40,0x19e073f7),L_(0xdcd869ef)}},
   {{LL(0x39df6bf7,0xd9747842),LL(0x484e4a18,0x9ca1c1ce),LL(0x2181acfd,0x09d6f90f),L_(0xae25cfad)}, {LL(0xc2644ecc,0xb6aa912f),LL(0xeb2922c9,0x767976e7),LL(0xbe3728a6,0x3b8ce6be),L_(0x04eb6141)}},
   {{LL(0xd6e6d0b5,0x618ad07e),LL(0xc862e8d5,0x926d38e4),LL(0xca247f9d,0xaa268736),L_(0xbded92f5)}, {LL(0x1a6031fd,0x9481e659),LL(0x42b36d96,0xc8b3617c),LL(0xabfa2ad2,0xd3f46f02),L_(0x1cbbeccc)}},
   {{LL(0x2057edf3,0x2b94f0fe),LL(0x608a4b71,0x7ca57706),LL(0xdd35cef8,0x236745ba),L_(0xb4d5d169)}, {LL(0x99412994,0x03f0b33a),LL(0x826e0d5f,0x0c5801a2),LL(0x31eb9951,0xf2cc0540),L_(0xd954458a)}},
   {{LL(0xf649bf08,0x2c486c0b),LL(0xa5afc024,0x9417e436),LL(0xb063a531,0x38aa1ed3),L_(0x1558aff7)}, {LL(0x004fbb10,0x904e9ed7),LL(0x3494f0b6,0xd2555c07),LL(0x365e3d05,0x0f32df2c),L_(0x5c8f30d5)}},
},
/* digit=4 base_pwr=2^28 */
{
   {{LL(0x2f432139,0x6a7ebdd1),LL(0x676d690b,0x25d6c68a),LL(0xf169ac1a,0x70ef4a18),L_(0x3fe9e66e)}, {LL(0xa193f6f9,0x9b515115),LL(0x6b6b0581,0x4399b66b),LL(0xd0e88db2,0x52c4fcf1),L_(0x31c324fa)}},
   {{LL(0x62e391aa,0x403677bb),LL(0xe67af8a2,0xcd3fd0e0),LL(0x0b4be307,0xdcb51813),L_(0x88ce9bf6)}, {LL(0xe4877d66,0x0f788e98),LL(0x8400154f,0x1742afd0),LL(0x768edb4a,0x80428f26),L_(0x277da6c6)}},
   {{LL(0x0b15cc70,0x6933e246),LL(0x8f4bec34,0x8bc92958),LL(0x13b62772,0xc9121c92),L_(0x543662c4)}, {LL(0x4549aa77,0x1090a59c),LL(0xc62e7c6f,0x066a0956),LL(0xa38394a8,0x19c57ed5),L_(0x4823e8cf)}},
   {{LL(0x003e564b,0x61b100d2),LL(0x416f9398,0xaf3087cc),LL(0x8d0f5a48,0x11e5f34b),L_(0xe047da80)}, {LL(0xe946d928,0xc598885a),LL(0x455959c9,0x838b9ed4),LL(0xe69bdd02,0x6bdbe44c),L_(0xd3377883)}},
   {{LL(0xd1de4464,0xe93c7baa),LL(0x23daaa17,0xb54baf63),LL(0xcf91e74b,0x64e1431b),L_(0x1ae9e235)}, {LL(0x43fd6c3d,0x2c00590f),LL(0x0ba0b46e,0x98d151a0),LL(0x1ee4dc18,0xe61cc51e),L_(0x93805289)}},
   {{LL(0x15b970a3,0xa33f496c),LL(0xc65b315e,0x3f6d708b),LL(0x25723787,0x2ca2c551),L_(0xc7e943c6)}, {LL(0x9a10432d,0x1845a431),LL(0xb33bb1fe,0x81764017),LL(0xf5b5a1e6,0xbb41c15a),L_(0xf5bd0a9c)}},
   {{LL(0x042a23c5,0xe9864244),LL(0x73ab2381,0xbb7b2648),LL(0x0ed12b18,0xe3536d47),L_(0x0a040c86)}, {LL(0x9c5a68b2,0x550f5800),LL(0xa16b2d10,0xfa0aebe0),LL(0xd20e3020,0x40239ea3),L_(0x13feb317)}},
   {{LL(0x8e882fb8,0x12206859),LL(0xd53242c9,0x3e1a95f6),LL(0x1c2921f5,0x765eedc1),L_(0x84f582be)}, {LL(0x525bb499,0xf8de62be),LL(0xf9eee520,0xd0a27199),LL(0x209f3c9d,0x721633cd),L_(0xc2e9678c)}},
   {{LL(0xc8a35690,0x852acfe8),LL(0x92b626ca,0xdde1d1be),LL(0xf571ade2,0x918c76fa),L_(0x5b7490f5)}, {LL(0x38c8b0c0,0xdf1662bd),LL(0xbb1e6d8b,0x470868d3),LL(0xd821c745,0x9a06de8a),L_(0xcaec5c2c)}},
   {{LL(0x64f233d6,0xc1321d2a),LL(0x74538108,0x30546bc8),LL(0x2f1d65e6,0xc8d1c486),L_(0x8ecd7a1e)}, {LL(0xccea4b95,0x3466245a),LL(0x700ed4ab,0xf5d50910),LL(0xfa32badf,0xb5a66db1),L_(0xfc4883b2)}},
   {{LL(0xb4589512,0x962b479c),LL(0xc4a7efe4,0xb97069b4),LL(0xa0d20fc9,0xd0ef4040),L_(0x76eabbd4)}, {LL(0x38b07e76,0x17a75081),LL(0x4bc6d910,0x3d3dbafe),LL(0xff0f721e,0x36e896fb),L_(0x6146e5ef)}},
   {{LL(0xab1ff699,0x6fd3001f),LL(0xe508cb13,0x5b228934),LL(0x80a73390,0xcc02c93c),L_(0x6ad8073c)}, {LL(0x9444deec,0x5f92bcf6),LL(0x8847a6eb,0x06c7d42b),LL(0x758f4aa4,0x40b45a83),L_(0xdb4ada91)}},
   {{LL(0x0083de14,0xfff55c75),LL(0x661c826d,0x317fc21d),LL(0xb311e146,0x6bdaa80c),L_(0xcad158e9)}, {LL(0x69952fc9,0x4502f454),LL(0x239ecee1,0x74f37d8f),LL(0x2b344eb3,0x161e1cf8),L_(0x02ce3f80)}},
   {{LL(0x645e7684,0x0009b919),LL(0x53d14c44,0x8105efea),LL(0xcbb0101a,0xdf767e4c),L_(0xecc4eabb)}, {LL(0x0c2ac7fe,0x70cf6d6c),LL(0x0ede1159,0x17089527),LL(0x4162258f,0x5ed679cb),L_(0x5e706cdc)}},
   {{LL(0xcd1206e2,0x33b4ee64),LL(0x78fd4645,0xcab93e67),LL(0xb807ac0a,0x7e760bcf),L_(0x923934bf)}, {LL(0xfa2159f5,0xaf2ed832),LL(0x918e957c,0xe1d5b548),LL(0xdf411692,0xde9c3716),L_(0x2c76553a)}},
   {{LL(0x3d4c49ae,0xd3a72d08),LL(0xa5e4f840,0x03aca3cc),LL(0xd9bcbe83,0xf00706d6),L_(0x986b79c3)}, {LL(0x412a3304,0xefb01a68),LL(0xa0bb40d2,0x19e3729e),LL(0x2d182974,0x57b4bca1),L_(0x43c566de)}},
   {{LL(0xb3db024c,0x2f1c7c7b),LL(0xfb3c8bb1,0x86b88161),LL(0x1addae50,0x29206628),L_(0xa115d8d7)}, {LL(0xc7b9fa3c,0xcf3f272c),LL(0xf2a0db00,0xb0b04d71),LL(0x1a242dce,0x048ec06d),L_(0xbe72542f)}},
   {{LL(0xc6eb32f1,0x515f5aba),LL(0x995a0552,0x9e89cf10),LL(0x809459ef,0x2a52fccb),L_(0x4bf08352)}, {LL(0x31a7e6c7,0x94c73471),LL(0x507e309b,0xc1732a6e),LL(0x006ea760,0xd3410c08),L_(0xde332d38)}},
   {{LL(0xe9a863af,0x05608149),LL(0x971e9a21,0x80261bb1),LL(0xcf890a46,0x607c23ca),L_(0x8c2f3995)}, {LL(0xee178996,0xb8358125),LL(0xc384ccce,0x8630d610),LL(0x62e3cda6,0x637377fe),L_(0x36145889)}},
   {{LL(0xebe66e5d,0xa7ccaba3),LL(0x66ee0e61,0xe5d23ef3),LL(0x8b52d5b2,0x9b192439),L_(0xe2507e26)}, {LL(0xdebe6661,0xf0320f46),LL(0x164afe7b,0x06272a4d),LL(0xa9cd4c00,0x53be168f),L_(0xbd05717a)}},
   {{LL(0xb9a090f6,0x82377241),LL(0x84720302,0x8cab5c0e),LL(0x3e8eab14,0x8131dcb9),L_(0x8dffb402)}, {LL(0xe82db9aa,0xca87536f),LL(0x7476d50d,0xc99eb71b),LL(0x40036a93,0xad40e0b3),L_(0xae5c279a)}},
   {{LL(0x203e82ae,0xce0565d4),LL(0x4590aef4,0xf8303097),LL(0x813e461c,0x4d52d217),L_(0x64b75926)}, {LL(0x1774cb88,0x51483218),LL(0x4046569c,0x804c4220),LL(0xbd282e8b,0x394837d5),L_(0x5a6b2c1e)}},
   {{LL(0x7eb4bbcf,0x58408340),LL(0xa28c5b85,0x9cd1f588),LL(0xac9ddd98,0xdf706c18),L_(0x62a72bba)}, {LL(0x1ecd1293,0x7437fa2c),LL(0xafe0e361,0x5b13b49c),LL(0x29283b54,0x0c099515),L_(0x469c2258)}},
   {{LL(0x611af3b1,0x1869604e),LL(0x69cd9356,0xb7d10b48),LL(0x161651d5,0x8a8472c4),L_(0xa23f6947)}, {LL(0x8a84a570,0x9ca82d05),LL(0xdd9ffcc5,0xbdf0cb1e),LL(0x7c9cfaaf,0x8b227092),L_(0xc52e4c68)}},
   {{LL(0x344a8ce2,0x66b5d593),LL(0x661a0707,0x277d6af9),LL(0xf0ed8bfb,0x9f814a89),L_(0x68ff82da)}, {LL(0xff4e0f6a,0x54b6136e),LL(0x125a9d0a,0xe9e8996a),LL(0xdbfa6d51,0x0914dd93),L_(0xacae0da7)}},
   {{LL(0x326565a3,0x41cc5966),LL(0x8e32f402,0xa314cfc5),LL(0x99f4bcb7,0x0ba8496e),L_(0xcbcf93d4)}, {LL(0x58afb0bc,0xf2f01549),LL(0x22877383,0x775e4843),LL(0xf270905d,0xfeb2c8a5),L_(0x15d8ffe9)}},
   {{LL(0x111a2a90,0xc56f7437),LL(0x435cbc99,0x64f9c9bd),LL(0xedb87fdc,0xc1d19083),L_(0x702066e8)}, {LL(0xc8dbc37a,0x2c1c4233),LL(0x672ee8b0,0xc183ca5e),LL(0xfc342416,0xaa518aba),L_(0x34e2bf87)}},
   {{LL(0x13de8a41,0x3bd53f9f),LL(0x38707713,0x326f7696),LL(0x6251ddca,0x4c9a0c1a),L_(0xe7e323fd)}, {LL(0x6876f934,0xd33612f2),LL(0x93f34e7a,0x1b94f160),LL(0x0c42a981,0xdc0b91ab),L_(0x548c92c1)}},
   {{LL(0x4cbbb4e6,0x72ed1d97),LL(0x0de1c184,0x5a3987bf),LL(0x86904dc8,0x45379ece),L_(0xbe4c089a)}, {LL(0x87ac4c8e,0x6d5238ae),LL(0xa27b3f88,0xbf067204),LL(0x21e17b1e,0x122542d6),L_(0xd6e7acc3)}},
   {{LL(0x84631900,0x01b419d0),LL(0xdb03ac70,0x4d15d553),LL(0xc5304466,0x5f684940),L_(0x11655587)}, {LL(0x8f896988,0x196f70de),LL(0x771ca6aa,0x79277c70),LL(0x4a94022c,0xb69fe2d3),L_(0x076b1dc4)}},
   {{LL(0xb8ac3536,0xecd18f22),LL(0x225dd231,0x86439125),LL(0x23c21f5b,0x43b63acb),L_(0xf5b2b43c)}, {LL(0x9b9dfcde,0x7ec7a341),LL(0x362a2c90,0xd7db9d5b),LL(0x308c8fcb,0x608d86ad),L_(0x8858eb99)}},
   {{LL(0x89b3a627,0x82f0ce08),LL(0x8ffc3544,0x4c9d1c88),LL(0x1414581d,0xf2cd74da),L_(0x2ab74c29)}, {LL(0x47673821,0xd4c61a1b),LL(0x8b9a0584,0x4166899a),LL(0xd7114188,0x7b53c8e6),L_(0x4e91e1e4)}},
   {{LL(0x83a918f0,0xe472f5b7),LL(0x120a4cb4,0xa54211cb),LL(0xd29feaef,0xa568a4a5),L_(0x1f0a0815)}, {LL(0x80033652,0x52d9147c),LL(0x1dd52d78,0x73e66c07),LL(0x99e7538c,0x25ad4a06),L_(0x8a35cc7c)}},
   {{LL(0x45837327,0x3daaae35),LL(0x214d493c,0x0495de90),LL(0xc0f2cff5,0x5e7107a4),L_(0x46024f37)}, {LL(0x4ceabd92,0x0f39120b),LL(0xd7f5199d,0x26eb2e5e),LL(0x2dba9f4d,0x2755fd79),L_(0x54915591)}},
   {{LL(0xf3749f34,0xbd4c1fd7),LL(0xbe148855,0x0fc24339),LL(0x7f36ce43,0x41ceba92),L_(0x40c686c2)}, {LL(0xd073af02,0xa2355b2f),LL(0xeb937195,0xa16c5398),LL(0xa1f3ab05,0xf7d38672),L_(0x90828e20)}},
   {{LL(0x7c7ef0f9,0x37b7db08),LL(0x4de172b0,0x2ce3e250),LL(0x3f8b6d32,0x1253fef8),L_(0xc76ebff6)}, {LL(0x54208a07,0xc53e233e),LL(0x9a8ef0b3,0x6f68e154),LL(0x2d43fb48,0x9a94edf5),L_(0xc8891237)}},
   {{LL(0x85578ac6,0xf170445f),LL(0x115d7eb6,0xd46f93ef),LL(0x71666f11,0xdfdf4d27),L_(0xcd09d0bd)}, {LL(0x742f2abc,0x658ac044),LL(0xe75c6300,0x0f93b4da),LL(0xa0de2978,0x5ea75aec),L_(0xd9cc6143)}},
   {{LL(0x7e1b6b82,0x2a662d9d),LL(0xcd959485,0x38780971),LL(0x5ee58f00,0x69a1ad5b),L_(0x417599f7)}, {LL(0xfe8f4d61,0xc5110167),LL(0x471554ab,0x3201dfa1),LL(0x4e2c7d96,0xa6a4af88),L_(0xe49c13ae)}},
   {{LL(0x44dc3bbe,0x0a1a147d),LL(0x23d766f5,0x3c2cbc33),LL(0x5446ea58,0x737348f7),L_(0x49f6d997)}, {LL(0x2488e5f7,0x17464a0c),LL(0xf5ac023b,0x5e39103b),LL(0x1c89d9f7,0x5a4700e1),L_(0x918b1cd7)}},
   {{LL(0x1a52af4c,0xd9c4d8a5),LL(0xfb136a04,0x62eae24c),LL(0x627e23b3,0x2a546df5),L_(0x9518b383)}, {LL(0xc577abda,0xe74b275b),LL(0x9e8dd1b3,0xbe223e4a),LL(0xb28d50a9,0xc1bf5392),L_(0xf0c4dae3)}},
   {{LL(0x12450901,0x60cd114e),LL(0x19f30dd8,0xb57b5e92),LL(0x0a04952b,0xa5b0d428),L_(0x9fbf17a2)}, {LL(0x998f1456,0xdc88c74f),LL(0x2eaf445d,0x39f1ad33),LL(0x24d076c8,0xb7cb2b9f),L_(0x4da895c0)}},
   {{LL(0x2bde5f9c,0x296bb0ce),LL(0x1e460fe9,0x8b7ed12e),LL(0x0a2b9d60,0x779ddda6),L_(0x70ad87c8)}, {LL(0xda6b433e,0x1216c7ea),LL(0x015e5322,0xfbda2b1c),LL(0x08a2c2b4,0x5c4f2da2),L_(0xda0125d8)}},
   {{LL(0x7e27d6a4,0x5cbb2271),LL(0x0e27b02e,0xf69dabcb),LL(0x651b4dc0,0xf5b545b4),L_(0x89864ae4)}, {LL(0xe7bf1ebd,0xbfa85080),LL(0x0162c99f,0x358c27ef),LL(0xc148b34a,0x39fddb94),L_(0x7de290f5)}},
   {{LL(0xbe0c0f85,0x8d61f3ca),LL(0xe4f10dce,0x4cbdbb45),LL(0xd40a9135,0xdb4a6076),L_(0x869e961b)}, {LL(0x191ed5b2,0xc8ea7a1c),LL(0xaed2c272,0xe62f26a5),LL(0xdd267430,0x49228bdc),L_(0x196c9b69)}},
   {{LL(0xda4dc900,0x5f4b0d32),LL(0x357d26fc,0x47de3ca9),LL(0x8bc3e1fc,0x44a306a1),L_(0x6ebde62d)}, {LL(0xc4fcbc98,0x66c5b1ca),LL(0xc2a2d52f,0x6365559c),LL(0x67f40398,0x6a1916f6),L_(0x9eb2b762)}},
   {{LL(0xd3f693c6,0x97ae197e),LL(0xe2705942,0x4294f4ca),LL(0xb717c180,0xfe30cc2e),L_(0x7aa89b4d)}, {LL(0xb42643a1,0x3d1c1e8b),LL(0x1731a1a7,0x4e550c0f),LL(0x4cdde0b3,0x85e85391),L_(0xab7103b7)}},
   {{LL(0xe4b061b6,0x39f1ba7e),LL(0x21c91792,0xcfcb8f5f),LL(0xd1e29983,0x8102c46e),L_(0xaa0fed15)}, {LL(0xf3e438cb,0x1df20d6a),LL(0x659104e2,0x43818f1f),LL(0xd4f49239,0x3a84be64),L_(0xca582a23)}},
   {{LL(0x3118786e,0xce03f04a),LL(0x827606a4,0x107aaf6b),LL(0x10bcfdd2,0x277b88b9),L_(0xf2e4c5e4)}, {LL(0xb9b85218,0xb9213c6d),LL(0x161c54e8,0xebb4f0be),LL(0x2814c386,0x859e6045),L_(0xac244cf4)}},
   {{LL(0x7292e148,0xd6446675),LL(0xab795487,0x71b2851d),LL(0x5a242339,0x899b5dad),L_(0x8520bc7b)}, {LL(0x72df7577,0xa366ee1a),LL(0xc17b463a,0x30d222b1),LL(0xa0af7c17,0xddf53842),L_(0x85b5501f)}},
   {{LL(0x124bc359,0xc8d63689),LL(0xcd7b3fe1,0xbfb69ae2),LL(0x124ab94c,0x6971b236),L_(0x961e282f)}, {LL(0x7f403da9,0xb92ac7ad),LL(0xc76e5b2f,0x5d04c417),LL(0x8df11606,0x0055901f),L_(0x666181f1)}},
   {{LL(0xf49645d2,0x96fcbeb8),LL(0x745bf74b,0xe0b2a4ea),LL(0x24675bb4,0x531c5deb),L_(0xff3b4dca)}, {LL(0x560275f8,0xd74deba2),LL(0x97100731,0x02589d49),LL(0xa26f7f22,0xb1b02a6f),L_(0x396cb1a0)}},
   {{LL(0x7de707a4,0xc6165fe0),LL(0xc07709d3,0x6be7081c),LL(0xfdb883bd,0xb1a5a67b),L_(0xfd535ec8)}, {LL(0x25257079,0x8234b458),LL(0x45bb7b81,0x9c1b3f29),LL(0x1f4c2d8a,0xa1cee750),L_(0xcf55f516)}},
   {{LL(0x206c7d17,0x0e3c9acd),LL(0x75c44136,0xd3de8638),LL(0xbe92eabc,0xf4ca1355),L_(0x9e459c32)}, {LL(0xbf750404,0x4bc1af9e),LL(0xc06b8bfe,0x20fc4197),LL(0xe0b85f76,0x7d557c99),L_(0xff091685)}},
   {{LL(0xbe45dd17,0x90ad8901),LL(0x5513303a,0x62ba8a0a),LL(0x9b18a004,0x8286d6c6),L_(0x73c22592)}, {LL(0x126eaadf,0xf6402ecc),LL(0x2bbee8cd,0xa309b99e),LL(0xab06f9cc,0x42089652),L_(0xc4a5e5a0)}},
   {{LL(0xe1391274,0x6925cad9),LL(0x6d3b1102,0xe366ed4c),LL(0x2e19a20a,0x1593ba2f),L_(0x8c091708)}, {LL(0x9083afa5,0x711ad8d6),LL(0x1a65086c,0x5e8645a5),LL(0xc052325d,0x182117dd),L_(0x865d46f7)}},
   {{LL(0x6f70a66e,0xfb548313),LL(0x7a6b6eef,0xae61c7f4),LL(0xec447933,0x6a3d4ff3),L_(0x74866913)}, {LL(0x35042671,0x38316d02),LL(0xb9635b34,0x6a9889f8),LL(0x2db27177,0xac233369),L_(0x567be1c0)}},
   {{LL(0xbe9b9347,0xcf5edce0),LL(0xa18e3f0f,0xe93ce044),LL(0x05816a5b,0xa26ca8e1),L_(0x00da101c)}, {LL(0xacb5542b,0xe04bccb7),LL(0xa1f2efbe,0x470e98a4),LL(0x67b00a65,0xdd0dbd8b),L_(0xc0be2197)}},
   {{LL(0x788904b0,0xe5a39b63),LL(0x0a586a5d,0xc10656ab),LL(0xe22752b9,0xfc9ab49c),L_(0x72fc1c09)}, {LL(0x31249974,0x8b426974),LL(0xd51e5946,0x0c7e10a4),LL(0xa79d2b5d,0x8d8169eb),L_(0xe4ca8644)}},
   {{LL(0x93ac6205,0x923206a9),LL(0xd8894d89,0x311a6950),LL(0x992f9c30,0xd0679c8b),L_(0x0cde00fd)}, {LL(0x5d5f7311,0xf07baea8),LL(0x3f8e094d,0x00100a63),LL(0x3d3b3316,0xbfbe9e92),L_(0xd269b8cc)}},
   {{LL(0x3357ad87,0x88e7f567),LL(0x8dcb5c06,0x7dc56ca1),LL(0xc8f6e06c,0xa6e90095),L_(0x905b4a44)}, {LL(0x99de2654,0x63aa3a4e),LL(0x1e51e461,0x486476c9),LL(0x5fb4cf43,0x35d139e9),L_(0x8725096b)}},
   {{LL(0x016940d8,0x2005767b),LL(0x95ce3a0f,0x0dce7b68),LL(0xb593a6ed,0x2c5a09aa),L_(0x762f3669)}, {LL(0xe1f753d0,0x52242366),LL(0x88d38ecc,0x05e8fe24),LL(0x759292df,0x8843ff7b),L_(0x041ca14a)}},
   {{LL(0x8edf1adc,0x0062aa12),LL(0x4bff8a1d,0xb1aa7c7a),LL(0x28b33bc2,0x966aee37),L_(0xb4b3e7e7)}, {LL(0x2d9ceec2,0x7f31f05a),LL(0xc7be6aaf,0x10ae244f),LL(0xc2c97f7b,0x7401d050),L_(0x1475b928)}},
   {{LL(0xd008f9fc,0x3150b4aa),LL(0xd53ede1b,0xef6ad2c4),LL(0xe427c40d,0xf4e2d20a),L_(0xe5dd8b02)}, {LL(0xb370a814,0xac4f8ba7),LL(0xaaacf4ec,0x44dbb479),LL(0x8e6b2784,0xd9aeb4f2),L_(0x4019cd79)}},
   {{LL(0x004de7f7,0x680b826b),LL(0xc0a2ee02,0xc1ff25d4),LL(0xb25a4e2a,0x8b33b701),L_(0x3a2939d0)}, {LL(0x9621d94e,0xe6feb01b),LL(0xd1311810,0xca0e2b81),LL(0xa701c8f2,0xdbcecc81),L_(0xeee95871)}},
},
/* digit=5 base_pwr=2^35 */
{
   {{LL(0x22d3206b,0xb5c4c69b),LL(0xf507ee25,0x886af51e),LL(0x454a0f10,0xf21fe543),L_(0x29d7720a)}, {LL(0x22d2999a,0x7c33ec2a),LL(0xb4054a86,0xa4d8106e),LL(0xdb971595,0x8a74b5b0),L_(0x8850caf4)}},
   {{LL(0x3bc6ae19,0xa04cb662),LL(0xf30acc6a,0x4f889fa2),LL(0xd7560ec2,0xb8b6110c),L_(0x4f8fab99)}, {LL(0x6bc71596,0x4308e608),LL(0x2c9d6af5,0x10d12c43),LL(0xca6486c0,0x64c7b963),L_(0x28ca7953)}},
   {{LL(0xb662cfb6,0xbcc43ce9),LL(0x3055b2de,0x39c5bde1),LL(0x68461c60,0x4caf7c30),L_(0x84c1743c)}, {LL(0xab0e6d70,0xb5b3a77f),LL(0x0e121552,0xa5bc5e90),LL(0xc38f8d3d,0x6cee1666),L_(0xe3236cdc)}},
   {{LL(0x88d7b871,0x4220e847),LL(0xa785596f,0xc1fb7864),LL(0xfa4258dc,0xf4276ec4),L_(0xfc15e9f1)}, {LL(0x293a6fc6,0xa58a670d),LL(0x9382335f,0xd56aed8a),LL(0x3477647c,0x311f9766),L_(0x4f5dbddc)}},
   {{LL(0x6de91978,0x69933c7b),LL(0x21ea5af6,0xf13eda55),LL(0x4b132119,0x12e00aff),L_(0x0ff8497a)}, {LL(0x3253a2e3,0xb8a8554d),LL(0x7d054f07,0x9338ab00),LL(0xb896b1fc,0x91dbc7f3),L_(0x6470c920)}},
   {{LL(0x51929e02,0xc7b39de9),LL(0x93ff36ec,0x2b86f570),LL(0xa6f2dced,0x802aa4a0),L_(0xdbd9b943)}, {LL(0xa8bbc105,0x8c1a0edb),LL(0x7cecf63b,0x54aa43d1),LL(0x54af7ee1,0x78b1b385),L_(0x0498d501)}},
   {{LL(0xe2a3944b,0xf34b8646),LL(0xe5ba9123,0x61656a11),LL(0xc14ff9dd,0x42cbd03c),L_(0x2891aad9)}, {LL(0xd76dd4f7,0xf01f7308),LL(0x55808c68,0x793b012a),LL(0x2ceb5051,0x9d313349),L_(0x2f3b809e)}},
   {{LL(0x1cd0a45c,0x59881514),LL(0x919ddc75,0xa32929ee),LL(0xf87e0bab,0x3037e369),L_(0xa090d36b)}, {LL(0xb3841cb6,0x579ffa96),LL(0x38e224e4,0x904c8cca),LL(0x2cadda68,0x702d5e89),L_(0x12eff4f2)}},
   {{LL(0x92015c38,0x52239088),LL(0xfabf2649,0xd339e361),LL(0xe9f6b8df,0xe84977d6),L_(0x3f58519b)}, {LL(0xaa4bb20f,0x4357d0b8),LL(0x6c5351ad,0x8521ac57),LL(0x1253d822,0x9524c5cc),L_(0xfbce31d7)}},
   {{LL(0xc07ab316,0x2e93f070),LL(0xfcee759c,0x25c23713),LL(0x132e8c43,0x6a329ada),L_(0xec4a7a29)}, {LL(0xe8c7f287,0xd5e2b374),LL(0x36a40bf0,0xcf0c0149),LL(0x9da99c9f,0x25e16e85),L_(0x7730bda6)}},
   {{LL(0x4bd943c9,0x4e15a70d),LL(0x37a376cc,0x82e59c18),LL(0xad1329bc,0xe14b9f7a),L_(0xac8b6391)}, {LL(0x60924a1b,0xe412693b),LL(0xb1a885d1,0x20e7bad4),LL(0xdd7cdcc0,0x06667f09),L_(0xfb4d7bf2)}},
   {{LL(0xb6cb033c,0x350411eb),LL(0xc83be0ea,0x4b4e8d93),LL(0xc7612043,0xbb338804),L_(0xfb1e56ce)}, {LL(0xef5ca7d8,0x4b75557b),LL(0xfed92eee,0x5e0d7736),LL(0xc3259e9d,0x61eea795),L_(0xc28b498a)}},
   {{LL(0xbe56f422,0x7363a98e),LL(0xe21a8af4,0xb0ad92ac),LL(0x2ee0c526,0x081ad4e5),L_(0x50ce73f9)}, {LL(0xecb59a8c,0xfce29612),LL(0x52824a9e,0xdcb3b781),LL(0xb689cfc9,0xac6e83b5),L_(0x006e1f4f)}},
   {{LL(0x1f147d23,0xd0a972a8),LL(0xe471b9e4,0x346ed362),LL(0xcfd1906e,0xd7accdbc),L_(0x293a4390)}, {LL(0x79ac99cd,0x86a243d8),LL(0x06c24954,0xca928f18),LL(0x3453aba8,0x25495810),L_(0x36f891b2)}},
   {{LL(0xd91398ce,0xba90aff9),LL(0x4d72eeba,0x14e030a4),LL(0x81c31d9d,0xe70f3373),L_(0xa716b2aa)}, {LL(0x2213b99f,0x42ac4252),LL(0x6f657c45,0x415aca82),LL(0x7d074644,0x461026d7),L_(0x0dde683d)}},
   {{LL(0xa01e18e0,0x4b805781),LL(0x8dd02690,0x35c19d90),LL(0x1d04223e,0x83277841),L_(0x7ea96551)}, {LL(0xe06c5996,0x4c78c04d),LL(0x70dde026,0xc34eaf3c),LL(0xb58c0bf7,0xae645b00),L_(0x3afcf537)}},
   {{LL(0x76eb1d35,0x708a8083),LL(0x5f7d6f8d,0xc51a45c7),LL(0x141c5fc5,0x61a1cb60),L_(0x78ecbf19)}, {LL(0xba4b17cb,0x4cb03ebc),LL(0x9be1423d,0xf30899fe),LL(0xd63cb771,0x51acdaa0),L_(0xa32591dd)}},
   {{LL(0x88195009,0x8de09e86),LL(0x76e08928,0x70957a85),LL(0xa653c234,0x59256cc9),L_(0xd8d51507)}, {LL(0x5d7b9252,0x47309994),LL(0x0f159c21,0x3ff7ccc1),LL(0xfe9fe5bc,0x0339c473),L_(0x16784f00)}},
   {{LL(0x24c3bafd,0xf2b37280),LL(0x1cbd8aaf,0x2790844b),LL(0x8a600a73,0x2324a737),L_(0x644765ee)}, {LL(0x4ad1ba6a,0xb4dd3d30),LL(0xa23d1c2e,0x9d90ab4e),LL(0x953388e0,0x864110e9),L_(0xe9be065f)}},
   {{LL(0x9dfdba3f,0x9c32af38),LL(0x69ee0a06,0xf93a2f14),LL(0x1fd2a13b,0x38105c26),L_(0xfe12d778)}, {LL(0x3947424e,0xd4a50774),LL(0xd5b96217,0x8f7c85be),LL(0x0f0582fb,0x9f4dc144),L_(0x87629c9c)}},
   {{LL(0x6470ccfc,0x377dc9b6),LL(0x9e11a84c,0x09816331),LL(0x610e3f5c,0x9becf104),L_(0x87cc9a9b)}, {LL(0x62783e9f,0xbefa56b8),LL(0x89fa53e3,0x18a4bfcc),LL(0x0b3ed787,0xdb99856d),L_(0x68d6dbd1)}},
   {{LL(0x9620574e,0xdd0b49af),LL(0x7e561904,0x68703702),LL(0x3056fb70,0xab404bf8),L_(0x4911d8a5)}, {LL(0x0f24dcd1,0x91807631),LL(0x606c33cb,0x6847c0d5),LL(0x791f57f0,0x525adb32),L_(0x818f0a1f)}},
   {{LL(0xa84199ec,0x1094f386),LL(0xd1afe50b,0x558c85a0),LL(0xab294ef9,0x69998dcc),L_(0xe0d8fa53)}, {LL(0xa8d822a6,0x6ccabafd),LL(0x64a8c67c,0x4f2dce66),LL(0xd4a601a5,0x8082a5ca),L_(0x1c89a181)}},
   {{LL(0x49e538af,0x4855546c),LL(0x6d3c05ab,0x11f89f31),LL(0x44996dfb,0x9e6dd61e),L_(0xe5054752)}, {LL(0x053ae779,0x4c1c7978),LL(0xa7675178,0x49d3b92e),LL(0x72b800a4,0x307c1a34),L_(0xafc9ca2a)}},
   {{LL(0x73bd9113,0xf1254531),LL(0x1e52b09c,0xb3c304d4),LL(0xfc186009,0x579623ed),L_(0x0b5f7250)}, {LL(0x0a3ddce4,0xa4d79ef7),LL(0xff4c374a,0xd69618ef),LL(0x252cb608,0xf8e76342),L_(0xfcc58682)}},
   {{LL(0x1cbe9e0c,0x586d393a),LL(0x480cf461,0x2e2acea1),LL(0x13a215d4,0xe3cf0a14),L_(0x82460da2)}, {LL(0x38bbf760,0x188125b7),LL(0xcd5956cb,0xa80c3de5),LL(0x9e63be59,0xbfc652be),L_(0x275a3495)}},
   {{LL(0xccb821df,0x965c6e19),LL(0x0b8b60d7,0x03204648),LL(0xd6030341,0xc11f7050),L_(0x4e201a9c)}, {LL(0x5815f324,0xdd1a6a75),LL(0x73f70f06,0x7aaff126),LL(0xebc85955,0x11a05e07),L_(0x0cd2e37d)}},
   {{LL(0x34fb8b2f,0x7afa362d),LL(0xf7c5f96b,0x995871ed),LL(0xfa20c1e4,0x6be4e8e8),L_(0x02b63df7)}, {LL(0xf9949e2f,0x3453fb74),LL(0xf6fdf958,0x376ae7a8),LL(0x502b7475,0xadf32795),L_(0x20557243)}},
   {{LL(0xb1daffbe,0x446cdc61),LL(0x8eb329bf,0xefdaaa71),LL(0x369f125e,0x6c894dd3),L_(0x9db7bb62)}, {LL(0x4dd5f542,0xc4277490),LL(0x31b2dbea,0xfe4eccd1),LL(0xb650ae0a,0x20fa0957),L_(0x29da364e)}},
   {{LL(0x937b1868,0x3d0c5313),LL(0xaa1e00ca,0xb2ed18b4),LL(0xe23097a6,0x18e7fcd5),L_(0x790fdadc)}, {LL(0xf18eb299,0x00c47190),LL(0xf4068839,0x1797157e),LL(0x5b080743,0x257f4df1),L_(0xd4b97dd1)}},
   {{LL(0xcf771fe7,0x9e9f2679),LL(0x7fa3e72a,0x0a76d555),LL(0xec547443,0x303f3000),L_(0x74ba5f3a)}, {LL(0x56040059,0xe1305459),LL(0xf97b4e0b,0x58cce0ba),LL(0x7b3e3dc5,0xb3342459),L_(0x849b0cbe)}},
   {{LL(0xc043c4d3,0x156bffa3),LL(0xef4fdf72,0xc568dd1a),LL(0x0fb07e25,0x373456aa),L_(0x0c136024)}, {LL(0x9c743fd6,0x75063485),LL(0x09872cea,0x512889e4),LL(0x1f75eff7,0x2982d676),L_(0x46918abe)}},
   {{LL(0x19d6d0db,0x62d87801),LL(0x8ac63776,0xe6ac10ed),LL(0x3df57bb4,0xfb67a75d),L_(0x872c8f87)}, {LL(0xb22e582b,0x49548c67),LL(0xa4ab6141,0x6cb6b04c),LL(0x5ae2f1d3,0xccb21fe7),L_(0xcc4c5d51)}},
   {{LL(0x95de93f8,0x552b1d5a),LL(0xc91b6f0a,0xf6ce5a61),LL(0x26244e6c,0xebe1c6fb),L_(0xec4bbe5c)}, {LL(0x4fe79929,0x35cfef8c),LL(0x09fa8c0f,0xc7ce6957),LL(0x77fdabaa,0x4bbc3792),L_(0x94421437)}},
   {{LL(0xa29405cd,0x820d6808),LL(0x9b55ccda,0xf32ab874),LL(0x1046f37c,0xed3f6fb8),L_(0x07a02e6b)}, {LL(0x6ed01e06,0x6676830d),LL(0xf82f2f79,0x1cd99760),LL(0x35492433,0x79d53ab5),L_(0x16a996a8)}},
   {{LL(0x1f2f75b4,0xaa0f7a8f),LL(0xe4d29021,0x44a728bb),LL(0x9b86e6e0,0xdb26a514),L_(0xf567f4e0)}, {LL(0x47f066f6,0xc1decce5),LL(0x9de6495a,0xbb842ffb),LL(0xda23fbb5,0x6557417b),L_(0x7d23ff08)}},
   {{LL(0x5767ef80,0xc3c2aab8),LL(0x2e3254ff,0xb5203eb2),LL(0x48576b69,0x2e3a1aff),L_(0x4e27496a)}, {LL(0xd5e96792,0x497974dc),LL(0xb0b36f57,0x8169a5cf),LL(0x1d0467dd,0xbd12d521),L_(0xee061337)}},
   {{LL(0xa2a324b7,0x3a9a26ff),LL(0xfa637395,0x664652b4),LL(0xbc2048f0,0xf8ae9c4f),L_(0x7229d31e)}, {LL(0x499224d1,0xd86d5de2),LL(0xd7e8cbb1,0x33a4a4fb),LL(0xa98c305e,0x382a852f),L_(0x7668edae)}},
   {{LL(0xdf02b971,0x6d33747e),LL(0x6bae97b0,0x468b6590),LL(0xc0b8420d,0x0d1395e0),L_(0x5d523a37)}, {LL(0xb00603ce,0xf196c7bc),LL(0x940889f2,0x7b8dc7b3),LL(0xa914e0ea,0xce35713d),L_(0xcb59a392)}},
   {{LL(0x5a648ea8,0x984bd24c),LL(0x17b7b322,0x88be2da8),LL(0x33279a92,0xa03ade42),L_(0x3ac696ec)}, {LL(0xa75e5e17,0x36f23ece),LL(0x4faba2b9,0x7b477179),LL(0xf70fdee7,0x7bbc4522),L_(0xeb035a5f)}},
   {{LL(0x52c14e4f,0x5ddc6b6c),LL(0x150eef02,0xffc4fbef),LL(0xdd3f93bb,0x3696d783),L_(0x15012933)}, {LL(0xe0cad04f,0xff6e69b5),LL(0xc84755b9,0xd70bfdb4),LL(0x118ab913,0x2da1eb13),L_(0x99325702)}},
   {{LL(0x0bcc9e8a,0xdfd09cd8),LL(0xd7ddee82,0xf1564d08),LL(0x93460444,0x87b5bd4a),L_(0x0d5541f4)}, {LL(0x87dcbe2a,0x2e83bcba),LL(0xd3e283ed,0xd947e2cf),LL(0xe52b803b,0xb28eb570),L_(0x4a8c77d7)}},
   {{LL(0x7f6e4fa5,0x76362059),LL(0x18df4a69,0x41a44792),LL(0x70a8c001,0xf96be149),L_(0x69d0bc39)}, {LL(0xef115578,0x59f09aa4),LL(0x53e28bf7,0xc1354a01),LL(0x3ef52081,0x28ac324e),L_(0xa781c57c)}},
   {{LL(0xf28899e9,0xb145dd4d),LL(0xf8de49f2,0xc929d158),LL(0x7a359488,0xa5fe71f0),L_(0xea4e5a13)}, {LL(0x62263448,0x8252a9a9),LL(0x9870bd6d,0xa478d63b),LL(0xd7e75649,0x73608571),L_(0xb0499f6c)}},
   {{LL(0xf3fc02d9,0x63626f38),LL(0xee89a981,0xecb2dedb),LL(0x45b932da,0x25321bbf),L_(0xdd6b2382)}, {LL(0x73c34e68,0xbd649606),LL(0xdbca6a1d,0xf7cc4ce3),LL(0x1e699d4b,0x940460cb),L_(0x7938144e)}},
   {{LL(0x6625374c,0xa571f0bd),LL(0x50d17043,0x67316b9e),LL(0x7b24e59a,0x15047c8b),L_(0xe06a40d9)}, {LL(0x71fd751b,0x5bc5aeb6),LL(0xaeb386d4,0x77f99ebe),LL(0x40fabb40,0x62bdfdf5),L_(0x6aace159)}},
   {{LL(0x2c7b8857,0xde610cbe),LL(0xf6b35535,0xd1d9c553),LL(0xb7d2826b,0xb4edc29d),L_(0x03e1e5e0)}, {LL(0x05ce2cac,0x789b199e),LL(0x03331148,0x44bc5992),LL(0xa966c64a,0x0851cfcf),L_(0x1622ec1d)}},
   {{LL(0xb82bff1a,0x888b7c2b),LL(0x07cb3e37,0xade27f4e),LL(0xb017d5a9,0x7f7c933f),L_(0x42119d1d)}, {LL(0x2b1e659a,0x5ef83e45),LL(0x0109f6b8,0x91081a86),LL(0x6e139181,0x1b624dea),L_(0xa2bd127f)}},
   {{LL(0xa90158d0,0x54ae0fcd),LL(0x9a79307a,0x5d77cf42),LL(0x6d342e22,0xb177e174),L_(0x2e3ee119)}, {LL(0xf32ebada,0x5e84ad5e),LL(0x08e8f89c,0x1f21876e),LL(0x425e69c5,0x42b47533),L_(0x0b2e547d)}},
   {{LL(0x8c73ff85,0x041c2bca),LL(0xd43f468a,0xdff69743),LL(0x497dff29,0x58417877),L_(0x9551c8f0)}, {LL(0xea9a5efb,0x1e1b3a51),LL(0x7aa44dae,0x62164b97),LL(0x82e3854e,0xe5d6128d),L_(0x2f25adda)}},
   {{LL(0xce69fd62,0x92365b99),LL(0xdd4e8322,0x64c55482),LL(0xe083981e,0xbf7947a0),L_(0x5b678747)}, {LL(0xe12f61e8,0xc98f9109),LL(0x0ca71f21,0x7084eee8),LL(0xe31a901e,0x922f7c66),L_(0xeed6566d)}},
   {{LL(0x830c51a8,0x274ea5de),LL(0x52fe1ee1,0x6f8c93f7),LL(0x7dc25d86,0xd6285877),L_(0xda84cd8b)}, {LL(0x48aed2f0,0xbb649ac1),LL(0x1e3ba4b5,0x6976562f),LL(0x6f9c137d,0xbd251ef1),L_(0x5e5495d3)}},
   {{LL(0x239cae69,0x8392e4dc),LL(0xdc8ee05c,0x79a24fdc),LL(0x4eeeb0db,0x0d91ecaf),L_(0x31e89bf7)}, {LL(0x4b9c499d,0xeca9047c),LL(0xee9a09ba,0x2d00d0dc),LL(0x81d64f7f,0xea9e0b50),L_(0xc7b60917)}},
   {{LL(0xb40fe472,0xb48d4d3e),LL(0xb23798db,0x4985b85c),LL(0x4a87382d,0x829e3bb6),L_(0x492aac4a)}, {LL(0x59bdf861,0xf58a6463),LL(0x96935f47,0x28f41a65),LL(0x3396e951,0x6d21b8d5),L_(0xe244afe7)}},
   {{LL(0xe019d807,0x890b3389),LL(0x01b0423f,0xaf7bf53a),LL(0x6d8cdb00,0x68218ce9),L_(0x7809fc35)}, {LL(0xd3b4c83e,0x446c771a),LL(0xaae3f3c1,0x8e7a1eda),LL(0xfb834953,0x89128662),L_(0xff1c7590)}},
   {{LL(0xb5425304,0xe1d9d533),LL(0x52f4f30f,0xdcd37dd8),LL(0x3a3e06d4,0x6d0d3574),L_(0x9ec8c4a0)}, {LL(0x48b20906,0x06630324),LL(0x8d559ed9,0x334cec45),LL(0x295b5edf,0xd0a2d07d),L_(0x8d6e5cd0)}},
   {{LL(0x2b1a895f,0x42c70b2b),LL(0x4f4ae671,0x36254863),LL(0x536e7d65,0xfaa3003d),L_(0x94dc0a21)}, {LL(0xc4355888,0x1592950e),LL(0xb4369ba0,0x6d4496df),LL(0x5f932085,0x6309e764),L_(0xe5462315)}},
   {{LL(0xe10bf24d,0x08fa6e2e),LL(0xec60e2f4,0x4a8f6f4a),LL(0xa152beaf,0xcc5f8968),L_(0xbce9a53b)}, {LL(0xedca75db,0x076de5d8),LL(0x31df6025,0xfb2b47e7),LL(0xce591c62,0xead89b13),L_(0x52e8a741)}},
   {{LL(0x94fdce56,0x179fcba0),LL(0xc81fc0c7,0xa0dcd6da),LL(0x8eb90798,0xbdb2ded4),L_(0x820ef61d)}, {LL(0x9c335535,0x2b8a3cbb),LL(0xf522badb,0x7214d9a1),LL(0x93d7057a,0x0ff17306),L_(0x70098d33)}},
   {{LL(0x50cac57f,0x8e019361),LL(0x2f914dd7,0xa3854afa),LL(0x2acdf295,0x2848b0db),L_(0x0f138ea9)}, {LL(0x8c64c319,0x4ea1b039),LL(0x49ae634d,0x47440022),LL(0x113138a5,0x8ed0557e),L_(0x2b0adbad)}},
   {{LL(0x931ca67d,0xb33942ae),LL(0x1ecf958e,0xc163a1f5),LL(0xc66d2148,0x1f27ea10),L_(0x011365d7)}, {LL(0x267a659a,0xc4208dd5),LL(0xb8f4d08e,0x61b362a6),LL(0x3e553027,0x29499eb7),L_(0xea028ba1)}},
   {{LL(0xd2551fb7,0xa6f2df89),LL(0x5bf03737,0x40fa3c8b),LL(0x924c5819,0xce383c9e),L_(0x910aeacb)}, {LL(0x641dc85b,0xbfdc9b58),LL(0xe45c96a9,0xd63db8d5),LL(0x726cada6,0x8177cf88),L_(0xa738df5c)}},
   {{LL(0x0cdee643,0x21f30c42),LL(0x8cd0ef97,0xf0f5ccbd),LL(0xe6a63e2e,0xd4df1941),L_(0x57a01e9c)}, {LL(0x498e2890,0x9c025c36),LL(0xdc96eafb,0xc9df9cce),LL(0x95fe2aa2,0xae8f18c9),L_(0x239e5894)}},
   {{LL(0x52954a16,0x9c68e7cc),LL(0x22874bf5,0xac850b44),LL(0x64db3074,0x3b1f7071),L_(0xa14665fa)}, {LL(0x87ef6099,0x78fc6e68),LL(0x77eb0618,0x1edffe76),LL(0x6d531161,0xf5f0fda1),L_(0x429c094e)}},
},
/* digit=6 base_pwr=2^42 */
{
   {{LL(0x0a953500,0x570bd76b),LL(0x42617e4e,0x493624f9),LL(0xec1fce2d,0xbd807766),L_(0x5098d9d8)}, {LL(0xcbc6ada2,0x468a4f16),LL(0xada13650,0xb78fc3ac),LL(0xb5aa694d,0x1dce66de),L_(0x5318219a)}},
   {{LL(0xfedbc51c,0xd30a5800),LL(0x97115f9f,0x2745e2b8),LL(0xbb10a57f,0xeff5c174),L_(0xadf87884)}, {LL(0x589f5f94,0x162e3c5b),LL(0xa2bc843d,0x16aad5e5),LL(0x25478287,0xd6f94dfb),L_(0x31bb4258)}},
   {{LL(0xc180bac0,0x6c5c8427),LL(0x46351cc8,0xdf01fd21),LL(0xb964daab,0xdd1d4004),L_(0xc4d0e945)}, {LL(0x72fb966c,0xb82dabad),LL(0x7d7ac429,0x2268903b),LL(0xeed99c4c,0x7dafa986),L_(0xcbdad15b)}},
   {{LL(0x60c5d65f,0xe5e0aad8),LL(0xc7b54b13,0xc4b812b6),LL(0xe9d8af38,0x008e0bc3),L_(0x71a411c5)}, {LL(0x27cb0dc7,0xc482b232),LL(0x07289ac0,0x5042230e),LL(0x68d763c4,0xa128b566),L_(0xc589940b)}},
   {{LL(0x62562a0c,0x533b21ed),LL(0xb86a434f,0x398ba505),LL(0x5f1dadf5,0x5b2f618e),L_(0x35caca2b)}, {LL(0xcaf4b249,0x74cf7bbc),LL(0xb14dae65,0xd924780d),LL(0x27c04cc6,0xf5af0266),L_(0xc41f13b5)}},
   {{LL(0xfd136860,0x5727ef4b),LL(0x9fcc3538,0x2af297df),LL(0x957682ab,0xe33ed1c3),L_(0x9f83349f)}, {LL(0x64693518,0x5961423c),LL(0x5148178e,0x8650191e),LL(0x66e2c6e6,0x5986c53b),L_(0xd5c63ac2)}},
   {{LL(0x11b49c5a,0x9082e761),LL(0x5661867b,0x1815dbbc),LL(0x8ece3938,0x30442089),L_(0xeddde973)}, {LL(0xbf2eb7b4,0x3e2f678f),LL(0xd4f64e91,0x15fb42e7),LL(0x49fdfa46,0x0b104178),L_(0x78b6687a)}},
   {{LL(0x391a31fe,0x9f724691),LL(0x0ca4bf22,0x79581fa0),LL(0x730e0882,0x4bbc9bf5),L_(0x91d1e24a)}, {LL(0xd756f110,0x26de9371),LL(0xd72ae23c,0xf050810b),LL(0x6ca80e46,0x75b289b3),L_(0x777ca3a2)}},
   {{LL(0x8185a290,0x7d3fcac6),LL(0xebb7ace4,0x1f109f32),LL(0x48bb32eb,0x0be22833),L_(0x3f76d58e)}, {LL(0x1868a043,0x8e3df32b),LL(0x00f1bf36,0xd5e2b1fa),LL(0x8b204654,0xdaeaa58b),L_(0x3263322d)}},
   {{LL(0x742ad365,0x648b273d),LL(0x2e05e12c,0x9a066a23),LL(0x3a732a3d,0x691aae3b),L_(0x6cd91464)}, {LL(0x04fe3ff0,0x208c6c3b),LL(0xe4c05b8f,0x61fcbdb4),LL(0xd9e2218e,0x59cd9a06),L_(0x835930df)}},
   {{LL(0x33eb0d51,0xd984c49e),LL(0x888fc640,0xd7d25d0d),LL(0x9bb00fea,0x4da4ce0e),L_(0xaffa26c5)}, {LL(0x72396942,0x2307b843),LL(0xc6cd9202,0xc1d18e97),LL(0x3f8ed768,0x518df91e),L_(0xeadd2347)}},
   {{LL(0xc1d3ad41,0xaf9c00eb),LL(0xb90b0248,0x6b575b77),LL(0x45a5507d,0x2e1dac5f),L_(0x8db688a8)}, {LL(0xab53ef89,0xa655bbf0),LL(0x0fdb8f93,0xea6e6754),LL(0xfdb2b439,0x9751d174),L_(0xaef8bdee)}},
   {{LL(0x0aa602c4,0xb601c379),LL(0xdaadd78b,0x3a6533ad),LL(0xb65d4dfb,0xb3e4d63b),L_(0x653e4965)}, {LL(0xf35d8a67,0x8cbfa3ec),LL(0xbaaa17fd,0xd9cd6c4a),LL(0xe29a388e,0x50eade40),L_(0x2ec73453)}},
   {{LL(0xd1812921,0xbac1435a),LL(0xaa859487,0xf5d2dc01),LL(0xb5b2445b,0xe22a73a2),L_(0xaa1116ed)}, {LL(0x39d4f53c,0xf9d72a85),LL(0x616cb03b,0x27cfb1ab),LL(0xcdf54ee8,0x4cf9a426),L_(0xe0208859)}},
   {{LL(0x0749a748,0x5300262c),LL(0x7faf0467,0xbb0d7807),LL(0xff47f117,0x93b14cbc),L_(0x484a66c2)}, {LL(0x91118c52,0x654db6c9),LL(0x20220d13,0x6b9ef224),LL(0x1162cfdf,0x897e3ec2),L_(0xf29d6809)}},
   {{LL(0xf7481883,0x2c67eef0),LL(0x1a833a88,0x49af9414),LL(0x8aa0f2d7,0x5fc1c69f),L_(0x9f06aae9)}, {LL(0x2b6f561b,0x4ecfeaa5),LL(0x17fda35e,0xbf5973f7),LL(0x338ff3e0,0x0013029b),L_(0xe6215b79)}},
   {{LL(0x45cc9ed7,0x45530707),LL(0x3d6ab800,0xa1f8d140),LL(0x34dca331,0x5aa02638),L_(0x4dc086c8)}, {LL(0x5be308ce,0x6df633bb),LL(0x90bbb8d8,0xd7e74378),LL(0xc7c28c06,0x8dcb3bae),L_(0xa0c24314)}},
   {{LL(0x2b4b775f,0x7f227ccf),LL(0x8cd51cfe,0x8cb18663),LL(0x68ea58e3,0x9f7bcc04),L_(0x3003cc72)}, {LL(0xdb7fa1bc,0x5b982b6a),LL(0xd5ebd68b,0x64a1222b),LL(0x675b50df,0x9679336c),L_(0xa8fee552)}},
   {{LL(0x1db92746,0xe51e0a9e),LL(0x75eb0c92,0x584f0a75),LL(0x9af943f8,0xa54c748c),L_(0x28a759fb)}, {LL(0x59c61e20,0x1cea2c6d),LL(0xb0384781,0xd54f4a86),LL(0xf0f73ecd,0xb65d1ecc),L_(0xd26a77df)}},
   {{LL(0x6f8d4351,0xcb23a541),LL(0x51dd59db,0x3cb31aea),LL(0x24f458a5,0x7d0b3dda),L_(0x26b341e1)}, {LL(0xb5440aa5,0x80d2e582),LL(0x9e2fd334,0x1b5ad25e),LL(0x114215cd,0x794f6e44),L_(0xe3bb54b5)}},
   {{LL(0x119ef86c,0x01429cdc),LL(0x7fb2c053,0x994ee3e5),LL(0xee0b3bcd,0xcf236537),L_(0xca6a0781)}, {LL(0x9d76f90c,0x28ee066e),LL(0xe18e5f40,0x18225dd3),LL(0xb1e0ccfd,0x2a9ed4ef),L_(0xa109a3af)}},
   {{LL(0x9f94c8a8,0xad211acd),LL(0xe621f731,0xfa0ba148),LL(0x072030c8,0xa71de5a1),L_(0x106812c7)}, {LL(0x57d3fed5,0xeac9452f),LL(0x4ce2ef8a,0xff6172a2),LL(0xac163954,0x778bc2dd),L_(0x2e5cbdab)}},
   {{LL(0x6d06223c,0x0baf79a2),LL(0x9c81a4be,0xe2bee5ec),LL(0x9363638d,0x8a4d69e0),L_(0x687f0dee)}, {LL(0xdceee166,0xa6d3323c),LL(0x202ed4a9,0x354badbe),LL(0x1c483cb2,0x5bbe8c1a),L_(0x433b79ed)}},
   {{LL(0x2c4c3668,0x3828dbf7),LL(0xb8a0c6af,0x17810cb9),LL(0x9f12e96e,0xdd7d6cea),L_(0xe595dcab)}, {LL(0x5b28c732,0xad54b44b),LL(0xe8aa36bb,0xb904d92f),LL(0x11dcc1c4,0xcbe63415),L_(0xeff9d401)}},
   {{LL(0x29014f0c,0xfef9f39e),LL(0x1d099df3,0x4d9cac37),LL(0xbf9c0427,0xe387610f),L_(0x0cce0331)}, {LL(0x95f296eb,0x31cb09c4),LL(0xd62d1375,0x7a5fdf3c),LL(0x53290cd4,0xa90e25d2),L_(0xf1396054)}},
   {{LL(0x83b59271,0xf603ca7e),LL(0xe57e3a51,0xa2fe4449),LL(0x547b9c97,0x5234c386),L_(0x44edaf6d)}, {LL(0x6f332534,0xcea690d6),LL(0xf13946cc,0xdad5199c),LL(0x6ad6bb4e,0x9374003a),L_(0x36c2bacd)}},
   {{LL(0x9a4d019e,0x341c30ad),LL(0x7e1c2c9b,0x96ee26ce),LL(0x17e56dfd,0xb65db139),L_(0x9888703d)}, {LL(0x4d9f2cbf,0xe07d7042),LL(0x3e36f84d,0x2223785f),LL(0xb4b9c74d,0xbbd020f9),L_(0xed1bc384)}},
   {{LL(0x08bd06c5,0x4432f2cf),LL(0xa3a488e9,0xec0de874),LL(0x9efd00d7,0xd17c4829),L_(0xf5128936)}, {LL(0x83542187,0x35e93b48),LL(0xae41e05a,0x22977d11),LL(0xe030a465,0x41a1010a),L_(0x1db1bce3)}},
   {{LL(0xe96cbfbc,0x6fb1c1be),LL(0xfe5cfb57,0x46e0f43b),LL(0x401be370,0x29b356b7),L_(0xef6ca52e)}, {LL(0x1769847a,0x131c52a8),LL(0x3084c2e8,0x9a3bea64),LL(0x58a36c5b,0x8cfd7a22),L_(0xa076e835)}},
   {{LL(0xc50d8952,0x74382b8b),LL(0xecf54e6c,0x22c47953),LL(0x80a6564f,0x3378864a),L_(0xf29ac0df)}, {LL(0x87ce0bb5,0x9549b954),LL(0x71036c3c,0xfdefd453),LL(0xd3203c72,0x57b0b3ec),L_(0x7ee244a8)}},
   {{LL(0xc911b9a4,0x2a2ae110),LL(0xe00edfa7,0xc23eb58f),LL(0x6b68f39e,0x8f5215f7),L_(0x0b132723)}, {LL(0x750f7a6d,0xe7d50418),LL(0x85673786,0xd615c040),LL(0x231c0709,0x487f22bb),L_(0xf5e2a237)}},
   {{LL(0x2e68e2c2,0x95692ad2),LL(0x0b27bb26,0x72e6eaf5),LL(0x090be6f6,0x8b45b55f),L_(0xea84dd7c)}, {LL(0x8e110b28,0x928497fa),LL(0x971d5f28,0x100a822e),LL(0x6339a104,0x9ddbd5c4),L_(0x3fe08c7f)}},
   {{LL(0xdbfc1ac8,0x987de692),LL(0x1683a455,0xee7012c8),LL(0xb38e1de8,0xdd0a3684),L_(0xf01af5d9)}, {LL(0x1923ae0d,0x86ab28a0),LL(0xd4b30cec,0x3a16a7ed),LL(0x22c7c599,0x01cadc9e),L_(0xb5dd36ac)}},
   {{LL(0xa274d3dd,0x8baf2808),LL(0x60cc335f,0x7c5d7b61),LL(0xa633db28,0x732a899d),L_(0x7f9cd569)}, {LL(0xe146b7a9,0x04bce2d5),LL(0x1f59a0d7,0x8db165c3),LL(0x6d10e528,0xcbb166ea),L_(0x7fe10619)}},
   {{LL(0xa561b19d,0x04d81b3f),LL(0x88a7293b,0x9822961c),LL(0x1732c2f0,0x057b8676),L_(0x320d8861)}, {LL(0x8a935dc3,0x5bfbc590),LL(0x3e5d907e,0x1bdbcf0b),LL(0x2b35a3ff,0x1597d44b),L_(0x7605acfa)}},
   {{LL(0xe27abdf0,0x6112f38e),LL(0xf5ab47b4,0x12fd1d52),LL(0x5984ae07,0x4fccb7e8),L_(0x81972b3a)}, {LL(0x95532439,0x5d4f964b),LL(0x905ded37,0x60b2cf53),LL(0x6a31fa69,0xfb6ad50f),L_(0xd83dca88)}},
   {{LL(0x8f19b88b,0x1b0ac422),LL(0x711dde1a,0x9363a89c),LL(0xb5e309b2,0xac0aec46),L_(0x17ee236d)}, {LL(0x5719b962,0x600531c2),LL(0xce093e55,0xee69dc7c),LL(0xec04dd4e,0x59639b52),L_(0xb9e01d45)}},
   {{LL(0xca086bb7,0x1b94ace6),LL(0xb0fee2ec,0x3ed113f9),LL(0x5c0b8f2f,0xdfd59ee2),L_(0x6a154d90)}, {LL(0x462e146c,0x8fb75d59),LL(0xb692d5e7,0xc31c3f9e),LL(0x080f7773,0x1fbd50fb),L_(0xa4f8f9e4)}},
   {{LL(0x7b7f1009,0xf63f6da7),LL(0x2aa45a5e,0xf59e459b),LL(0x2d7d0b41,0x29c1bc69),L_(0x8aabebdc)}, {LL(0xaf1dea2d,0xb0063dbe),LL(0xc38e3c41,0x335900c9),LL(0x661ba0dd,0x99541913),L_(0xc19809da)}},
   {{LL(0x7540a87d,0xff252d35),LL(0xa1530a6c,0xaffcab72),LL(0xc8e9b570,0x9793a371),L_(0x3489107c)}, {LL(0xdcf94c42,0xf0b49c6c),LL(0x879022b6,0x35c46e5c),LL(0x069dd468,0x2c618b82),L_(0x64bde16d)}},
   {{LL(0x4335b182,0x2d08df94),LL(0xed7d7c9a,0x919ddb27),LL(0xe80a7f85,0x357baf28),L_(0xbb968f21)}, {LL(0xbcb38a46,0x8315da92),LL(0x96e91923,0x925476f6),LL(0x4e306972,0xc5ed2a12),L_(0x950bbe41)}},
   {{LL(0xcd3c90b3,0x39668cb5),LL(0x077167b6,0xaea10320),LL(0xfce255b9,0xb37bc6b8),L_(0xf3809a3f)}, {LL(0x860aeccb,0x7f61816d),LL(0x44d57109,0x80e57881),LL(0xb19cf5fd,0xdb202875),L_(0x42100d63)}},
   {{LL(0x5a155cef,0xaf246d29),LL(0x7f3965a6,0x329527e8),LL(0x8d8e3138,0x26bde524),L_(0x11b1a769)}, {LL(0x66b0792b,0xd87dfaed),LL(0x707808be,0x02a38ada),LL(0x04bb89d1,0xe52cbe87),L_(0x7ec2d88f)}},
   {{LL(0xded9227f,0x5423908e),LL(0x43d7285e,0xa6f72d9e),LL(0x49363917,0x6a49fc82),L_(0xf5d8d60b)}, {LL(0x067bfaaf,0xa0f34345),LL(0x665a2267,0x19294466),LL(0x1ac20d98,0xffa2e5fc),L_(0x4e6d33c1)}},
   {{LL(0x558d4d1f,0x9731dc1f),LL(0xf1f7a37e,0x6a85be9c),LL(0x69d32744,0xea51b61d),L_(0x05851633)}, {LL(0xa0b1e63d,0x04e8d8ba),LL(0x1c8bd021,0xa75506f1),LL(0xef76ca45,0xf099b725),L_(0xd230047c)}},
   {{LL(0xf47e0086,0x44613936),LL(0x62171491,0x0105d7a8),LL(0xa07a7b6e,0x9015ad40),L_(0x306c9555)}, {LL(0x166467b1,0x18925a9d),LL(0xaac18583,0x8a300595),LL(0x7d220577,0xbbd4708e),L_(0xf151aaec)}},
   {{LL(0x6807a39d,0x6e04c9ec),LL(0xb57b3b22,0xd4d3310c),LL(0xa3593866,0xa01b71df),L_(0xa63bab57)}, {LL(0x378ccde9,0x773f947b),LL(0xefc9bc76,0xe29288bb),LL(0xcb328f94,0xe12b780c),L_(0xb068b3ce)}},
   {{LL(0x3bf9b78d,0xc1916a89),LL(0x62555b0b,0xad0d163d),LL(0x00203bb0,0x82d7d36f),L_(0xd8764be5)}, {LL(0x91a430b0,0xf446925f),LL(0xa23e8076,0xedc1e316),LL(0xb903dea6,0x11610026),L_(0xd83fceb2)}},
   {{LL(0x8f55d6ab,0x2739ed0a),LL(0xcff10abf,0x3b4b462d),LL(0xe0b269ec,0x92761d90),L_(0xc1c0ca0e)}, {LL(0x2403dfd8,0xab5ce41d),LL(0x62aaaef4,0x9ebcf2d7),LL(0x5a3ff0da,0x3aa250ea),L_(0x8f10b784)}},
   {{LL(0x2fbc9b42,0xe5bbf62c),LL(0xa53f3f3f,0x2c19f9e2),LL(0x27dd1952,0xeada36e1),L_(0x40a88ecc)}, {LL(0x8eac382d,0xca0a31a3),LL(0xd318529b,0xcc8ff4c6),LL(0xefec04d1,0x49d3e4dd),L_(0x55c20eb0)}},
   {{LL(0x54347369,0xc5f830ce),LL(0xbf310209,0x66ba5a3d),LL(0x1372fe7d,0xab6a0dc0),L_(0xaf6004cc)}, {LL(0x65a2727f,0x8803af60),LL(0xa6088f51,0x91e29b12),LL(0x53f144ad,0xe77f4280),L_(0xd7d06ab8)}},
   {{LL(0x978295dd,0xc4969a0d),LL(0xdf19cc2f,0xabbb112c),LL(0x8bd7557f,0x554a3a06),L_(0x8fe8b8dc)}, {LL(0x516a1ab6,0xf261d772),LL(0x82e07bc5,0x4d16bca0),LL(0xd1d7da5e,0x45447ad3),L_(0x19044418)}},
   {{LL(0xe167f824,0x3956922e),LL(0xf66675c8,0xfcc5cc95),LL(0x20469977,0xf5f39465),L_(0x011863a2)}, {LL(0xaf5616e5,0x960a1daa),LL(0x66b9d7bb,0x2c7f8f70),LL(0x074a85f0,0xd9f9bf2e),L_(0x7dd3b230)}},
   {{LL(0x4e5dc71e,0xe5be9e2a),LL(0xb5995728,0xb7e24b2c),LL(0xa127aac9,0x23c8cef0),L_(0xf11df0db)}, {LL(0xe494dece,0xefcf6a97),LL(0x38cfdd79,0x936e7033),LL(0x899c799e,0x8c265685),L_(0x7cab4cd9)}},
   {{LL(0xdfd2bdce,0xe6160916),LL(0x480a36fd,0x01a7795c),LL(0x3f7c1b1d,0x91f84a8c),L_(0x85381912)}, {LL(0x14463148,0xb7408da8),LL(0xe263f58c,0xbd7b174a),LL(0x8ca85525,0x8c2f9aa1),L_(0x687fc42f)}},
   {{LL(0xa86b0be3,0xf5f78e41),LL(0x3455a3e9,0x135a0c71),LL(0x30c1c2c7,0xddf065a7),L_(0x3692f51b)}, {LL(0xebc66e40,0x0f5906c0),LL(0xbfc2af37,0x5229db63),LL(0x5fa05ed1,0xd9f88327),L_(0xfad22122)}},
   {{LL(0x6a2bd6b4,0xcbee3324),LL(0xe6fcf3f6,0xbfd3822c),LL(0xc3e92fcb,0x7c43be45),L_(0x6def0ea9)}, {LL(0xe2791c4e,0x72baebe0),LL(0xabf2c782,0xad049ecd),LL(0xff1c5bf6,0x21f7b8ae),L_(0xcada1649)}},
   {{LL(0xa204b629,0x2e618a14),LL(0xe44e287b,0x1ee56452),LL(0x9ec08fe0,0x3cbeeaa0),L_(0x41378416)}, {LL(0x65a854a6,0x3defdbe3),LL(0xdd8dc723,0xe7d6715a),LL(0x6d23302b,0xf1849878),L_(0x922ac03f)}},
   {{LL(0x826900c1,0xf4361242),LL(0x5150fa93,0xbc384de9),LL(0x1bf1d7e3,0x0106a9fc),L_(0x90827c46)}, {LL(0xb45c094f,0xd4951e89),LL(0x046acfdb,0xf1ea11f5),LL(0x38e792fd,0x2d8ca425),L_(0x07e228cc)}},
   {{LL(0x2b9cf3d8,0x889c6c44),LL(0x12d9fd78,0xc8a37703),LL(0xa6aed948,0x46f4af45),L_(0x4de1be3f)}, {LL(0x229dbecf,0x932b6857),LL(0x4b284bc1,0xa90d9c3e),LL(0xe27e82bc,0x9200852f),L_(0xe41775aa)}},
   {{LL(0x762bd870,0x50a8e4b7),LL(0x9d917e78,0xdf55385a),LL(0x4ac6dc2a,0xeb63b95b),L_(0x585b0c05)}, {LL(0xf896eb5a,0x876d7f21),LL(0xdd6e7325,0xbc4c0b28),LL(0x34a78682,0x55d49e0e),L_(0x7de26ccf)}},
   {{LL(0x0018e94a,0xf3430043),LL(0x09139313,0x6fccb693),LL(0x32e29483,0x9f0a6227),L_(0x1c584d56)}, {LL(0x4da9fbc2,0x11cca785),LL(0x53203116,0x349e04d6),LL(0xb1c8440b,0x0ec04b4a),L_(0xb4fd311a)}},
   {{LL(0x4e394c99,0x4a367cf0),LL(0x82ea7247,0x419778aa),LL(0x35a8f8e5,0xee0c8d61),L_(0xeb47eaf0)}, {LL(0xb8c3b039,0x7fc38516),LL(0xb4361b1e,0xedd99fe4),LL(0x24e2fb10,0x94b8860e),L_(0x1e4ac934)}},
   {{LL(0xb83e5822,0x9c7f7bc2),LL(0x85016181,0xec2c9c48),LL(0xae2e08c9,0x1d49bd53),L_(0xd51d0f76)}, {LL(0xe02702a5,0x574b1fa2),LL(0x275045e3,0x4b568161),LL(0xe63f9064,0xb270bab6),L_(0xcbacaf10)}},
},
/* digit=7 base_pwr=2^49 */
{
   {{LL(0x28c9f54b,0x55cc3b93),LL(0x81366649,0x9f91db20),LL(0x0bb5529e,0x66b83a45),L_(0x020a3d7b)}, {LL(0x84c6fb9a,0x6d9c718d),LL(0x4c39583c,0x6e5e8c08),LL(0x063ccb67,0x8cb7eb45),L_(0x01328b11)}},
   {{LL(0xc3d2cc22,0x7e06eec9),LL(0x2a100811,0x5a959809),LL(0xf4997f3a,0xf5c07857),L_(0xad98c158)}, {LL(0x5ae7c1cf,0xf408207d),LL(0xdb7d94f4,0x8cd5c2c0),LL(0x6379ef0a,0xf3bfa4da),L_(0xde775bfd)}},
   {{LL(0xd39752a4,0xd08dfe44),LL(0x9b19df92,0x208bc436),LL(0xfd14af8d,0xcc3f5ace),L_(0x9cb59da4)}, {LL(0x97f49ac2,0x880b584a),LL(0xe1fa2671,0x72d9f8ff),LL(0x76bf0449,0x0c870e3e),L_(0x6fb6d427)}},
   {{LL(0xbad01fec,0x70db0532),LL(0x12b3f8dd,0xf82bbcfa),LL(0x167ab18c,0x92abaaa2),L_(0x5e4d357c)}, {LL(0x4304ac9f,0x1fb69d3d),LL(0x7bd8f0e1,0x15e5d010),LL(0x47351320,0x84b31510),L_(0x6a03c6b3)}},
   {{LL(0x739b3599,0x5a2c8035),LL(0x91588e92,0xdac9ebee),LL(0x1b69af1a,0x451b95e7),L_(0x3517cee5)}, {LL(0xb64f8f05,0x1e09f842),LL(0x90930327,0x099f2550),LL(0xefa175e0,0x9e48451e),L_(0xedef520b)}},
   {{LL(0x43b4b068,0xa2492064),LL(0x1e8946dc,0xeddb27dd),LL(0x9916090e,0x82135311),L_(0x98fdccc1)}, {LL(0x4585ddd8,0x31f673e3),LL(0xd529f3ea,0xac4307ee),LL(0xb119ca51,0x428145ce),L_(0xcf53a57b)}},
   {{LL(0x3728e89e,0x38640e71),LL(0xa872dbd5,0x745bd8a2),LL(0x8f90052e,0xffeb6e87),L_(0x9a484b77)}, {LL(0xb85a7c0b,0xe045b6ce),LL(0xb3a39b3e,0xafd92044),LL(0xdbc53298,0x462a1b9b),L_(0x60a23eea)}},
   {{LL(0x050b6e45,0x8a1c84ea),LL(0xc38a15ff,0xc2315710),LL(0x5b502e99,0x67d1f3e4),L_(0xf85c7d6e)}, {LL(0xd3018572,0x7c34a744),LL(0x94062e0f,0x8b89ec47),LL(0x9579c983,0x35f7a266),L_(0x63ed902e)}},
   {{LL(0xbfd3d5c1,0x7e6607ad),LL(0x54fa21c1,0xbc8c84d5),LL(0x2c3baf13,0x177b275b),L_(0x39001a70)}, {LL(0x71f9447d,0xf4928a3c),LL(0x9356e91a,0xd23bd45a),LL(0x30b1d7ca,0x02772478),L_(0x8c5ecfb7)}},
   {{LL(0x5717dffc,0xcaa887a3),LL(0x4f7bef32,0xb1687f81),LL(0x5dec8cf9,0x8965403a),L_(0xcc362cae)}, {LL(0xb0dbc045,0x37f6ba20),LL(0x35e7cde5,0x6c266c6c),LL(0x0d54e5e6,0x02d7913b),L_(0xf4a0addb)}},
   {{LL(0xdde83971,0xcc348a37),LL(0x58983641,0x18fb2236),LL(0x237be719,0xd585a061),L_(0xdb2539f1)}, {LL(0x35a8a0dd,0x403565cc),LL(0x78351808,0x011a3765),LL(0x6202e956,0x7f54736b),L_(0x660d4fb3)}},
   {{LL(0x90a83185,0x6bc9ce97),LL(0xb22311a5,0xf3cc61c2),LL(0xe993a41b,0x85025abd),L_(0x432e7fa6)}, {LL(0x3a136918,0x42d19cdc),LL(0x18144346,0xe5b41dc3),LL(0x2bd73670,0xede484f4),L_(0xa93261b3)}},
   {{LL(0x62a3c5f3,0x38a7a728),LL(0x2660c15e,0x05a1c27b),LL(0xcbfe2670,0xe7715ea4),L_(0x803ef610)}, {LL(0xfff730b3,0x6f2eb06b),LL(0x3e73d7f3,0xafffff2e),LL(0x8fe4228f,0x2703c35b),L_(0x801d9246)}},
   {{LL(0xcf3f9f32,0xa7b3253d),LL(0xb347bed9,0xb88a3be1),LL(0x23367d55,0x50ebab37),L_(0xe6bc8d5f)}, {LL(0xb7d8a5f9,0x9df003d1),LL(0xd78ada2c,0x0bb08555),LL(0x4989b92e,0xf9b3e9f5),L_(0x2dbbe952)}},
   {{LL(0x1b0c7f5b,0xdc3c1c11),LL(0xfb949b08,0x8d1cf002),LL(0x010b8835,0x69483176),L_(0x73825d94)}, {LL(0x95d368f2,0x7caf4449),LL(0x3cd86c08,0xd50eaa5e),LL(0x8ba59225,0x6d5a74bd),L_(0x1e6e636f)}},
   {{LL(0xda9ad1f1,0x51eea6ca),LL(0x69de6020,0x3afc7be4),LL(0xf383a0c6,0xa7aeef7a),L_(0xec5ab9f7)}, {LL(0xe3d53f58,0xb56f14b0),LL(0x7b78f2d9,0xda9d3372),LL(0xb4e91769,0x0886a015),L_(0xb8c7fe1f)}},
   {{LL(0xbb4c1b2b,0x7b2d2c19),LL(0x3f05a7cf,0xc24e67f7),LL(0x8a978fa5,0x2541de9c),L_(0xb09f953f)}, {LL(0xbd943ed2,0xa6872621),LL(0x0275a86c,0x9f103457),LL(0xc3073baf,0xe68e138a),L_(0x70608786)}},
   {{LL(0x94c887ac,0x22def543),LL(0xc1d1a75a,0x70845389),LL(0x9d05dd03,0x13908388),L_(0xcac954e6)}, {LL(0xcee91dc6,0xfb37d06c),LL(0x42873dd7,0x0976cad9),LL(0x77686ddc,0xbd6e2bfe),L_(0x0a7aae21)}},
   {{LL(0xf93e73a3,0xd00f349f),LL(0x6d3f30d8,0xbdf0b725),LL(0x793a6f93,0x8e54b8c8),L_(0xa3ebe240)}, {LL(0xef42120a,0xd3635d7e),LL(0x7527fefe,0x954dc6a7),LL(0x7ecf5710,0xe5c193ac),L_(0xc5b359a4)}},
   {{LL(0xcee954fa,0x6a69e07e),LL(0x904b6a99,0x28557756),LL(0x74691c8a,0x377aafb2),L_(0xf8dfe49e)}, {LL(0xaf1c65b9,0xbc4bbf13),LL(0x32c37416,0xa07e7595),LL(0xa6bc7861,0xb481fbff),L_(0xab667972)}},
   {{LL(0x4ade6d39,0xd40e7121),LL(0x03318c55,0xe035cb6d),LL(0x5cc23063,0x7d21722c),L_(0x353dafc9)}, {LL(0x57f87e8e,0x47d32da3),LL(0x8e5c3c5f,0xdd769bba),LL(0x4c620a4a,0xc2e2dc49),L_(0x6ac8f16a)}},
   {{LL(0xe2c62094,0xe0aae2f2),LL(0x27e1fc19,0xe8068cdc),LL(0x9a5089ff,0xd066d3ec),L_(0x83ce796d)}, {LL(0x15285915,0x3d7bd85d),LL(0x2109b271,0xbfaee64f),LL(0x032bfaab,0x4c64918b),L_(0x28f7e9fb)}},
   {{LL(0xe9c1a122,0x81a3a910),LL(0x1ec15d92,0x1d6baa6b),LL(0x0de6b8aa,0x25d33d78),L_(0xdb94f1aa)}, {LL(0x84537b54,0xcda754fc),LL(0x0aa41bac,0x07667978),LL(0x26a33ede,0xaed5fcc3),L_(0x03b383eb)}},
   {{LL(0xc6a270c7,0x48a0860f),LL(0xe3c9d6e7,0xe8f8a0be),LL(0x1c9dda6d,0x37a0510c),L_(0x87290f47)}, {LL(0xb746897c,0xe0b9b992),LL(0x04968462,0x691f74eb),LL(0x79bcdef3,0xba7c23dd),L_(0x3c9dfbbc)}},
   {{LL(0xb9c1ad2a,0x4867362c),LL(0x9b7dcff4,0xea98891d),LL(0xf6be4fd8,0x9dd99069),L_(0xfc837cc5)}, {LL(0xc4b7695b,0xa104db38),LL(0xe73440a8,0xc1d73652),LL(0x557bc30e,0xca38a194),L_(0x0b9107de)}},
   {{LL(0x08821106,0xf9b3c662),LL(0x0572aa2e,0xa9b0c578),LL(0x6cf8f9ea,0xdc3bb3a4),L_(0x578fa229)}, {LL(0x935620c9,0x7b780978),LL(0xe2e55595,0xaa7bd737),LL(0x1ae5674a,0x7d70eb0e),L_(0x6287e785)}},
   {{LL(0x8161a173,0x60277149),LL(0x6418b451,0xe9fdf471),LL(0x008c73fe,0xc7a41b44),L_(0xb52d2690)}, {LL(0x6217c51d,0x07d5913e),LL(0x28af32f1,0x584cce8f),LL(0x70ace437,0x249d5a1f),L_(0xe2634904)}},
   {{LL(0x2aa8ec2f,0x91c3f75e),LL(0x7ac06d6f,0x3279d284),LL(0x8cbbcc96,0x6114dddf),L_(0xb3f952c3)}, {LL(0x18133afe,0x50a376bd),LL(0x95633a5e,0x281108f7),LL(0xf0277529,0x900af3d2),L_(0x2aac428e)}},
   {{LL(0x34f686e3,0xede089c0),LL(0xafe013ca,0x66ca4aa6),LL(0x24f9b482,0x7d9ab065),L_(0x7064130f)}, {LL(0x59e2f96d,0x88599e65),LL(0x1a8684d1,0x81cccdd4),LL(0x5b828088,0xa7784039),L_(0x274e2738)}},
   {{LL(0xa482a439,0x04a64674),LL(0xabf54d81,0x3a3f5066),LL(0x46144f06,0xc232e58f),L_(0x30cddd92)}, {LL(0x4aba099f,0xc2b48797),LL(0xfb8c420d,0x6eb5b117),LL(0xb4d97ee4,0x8d97d0d5),L_(0x8101b1e7)}},
   {{LL(0x3dd4c04f,0x98af21ef),LL(0xcd4340b9,0x57885c4e),LL(0xda56271d,0xcf1eb8ab),L_(0xcfc8c025)}, {LL(0x42508147,0x99747ee9),LL(0xdff889cf,0xbf581471),LL(0x16fb6025,0x22ae0ed8),L_(0x56d4b8ed)}},
   {{LL(0x3d08ef7e,0x0745f517),LL(0xc7781923,0x732c8246),LL(0x6cd5a87c,0xe1112158),L_(0x734d96e5)}, {LL(0x0ea91035,0xfbc17aeb),LL(0x7d6ae9b9,0x9fdb5814),LL(0x6b340fbc,0x7a81eba8),L_(0xa462be47)}},
   {{LL(0xd9df5a31,0xbad5359e),LL(0xcc17a45f,0x137fb217),LL(0x3962aaec,0x18daea3d),L_(0x24dea50b)}, {LL(0x585f316e,0x2db980d4),LL(0x94722851,0x890d4e8a),LL(0x17a80fda,0x0cd73f37),L_(0x0ef75926)}},
   {{LL(0x7ddc10cf,0x2994a0c5),LL(0x0d862f7d,0xe8d6d38f),LL(0xa326bcf2,0xb39a090d),L_(0xebe228c9)}, {LL(0x3fb2d1f5,0x4fe39e13),LL(0x292a329c,0xa1c6d77e),LL(0xa1f5d625,0x3f828ea8),L_(0x70ad621c)}},
   {{LL(0x997cd647,0xb7237a23),LL(0x94de2084,0xf796b3ba),LL(0x0c93caa4,0xd97e1daa),L_(0xb74fa915)}, {LL(0x4dfd621a,0x146706c7),LL(0x23c7b716,0xd56307f2),LL(0x54c6c8af,0xbe011ca0),L_(0x2ef6e4a4)}},
   {{LL(0xa3b6fdcc,0x33851601),LL(0xbee34e4c,0xd7da0ee1),LL(0x8e7c506d,0x2b92ce18),L_(0x62b378cc)}, {LL(0xc8c0d60c,0x95f16f73),LL(0x661c0891,0xdc466782),LL(0xf7df962d,0x6fdadac1),L_(0x8f3ef4f4)}},
   {{LL(0x81a3d47e,0x23206f02),LL(0x235bc318,0xf375259d),LL(0xec43017e,0xc5d4dab6),L_(0xf547238a)}, {LL(0x6f0f9561,0x25d0ebfb),LL(0x8357f1bb,0x8c478548),LL(0xe137e309,0x996a92a8),L_(0x629e8aa7)}},
   {{LL(0xa2b3a713,0x4a17680e),LL(0xa8c7b143,0xe242039e),LL(0xb84edf8c,0xf2acd3a2),L_(0x8b3a07a1)}, {LL(0x4d9e5baf,0x2f0d5861),LL(0xcd2f47fd,0xe42d316f),LL(0x0d4da406,0x0cd04a92),L_(0x8ec4fcad)}},
   {{LL(0x6dcda792,0x91a38da5),LL(0x76eb9c38,0xa4b8df93),LL(0x0a5b71ac,0x2d328a16),L_(0xf6122daa)}, {LL(0x9fdf7693,0xa1eceeb6),LL(0xe9555051,0x2371d345),LL(0x55020183,0xae8df945),L_(0xcee3a23b)}},
   {{LL(0x4fcaf8f6,0x88241ba4),LL(0x6839ac95,0x25f7328a),LL(0xb01a4c73,0x4e73d41b),L_(0xdbce70d9)}, {LL(0x470b6777,0x25457ffa),LL(0x4155bcf6,0x2545dcdb),LL(0x79b75849,0x3c25b38e),L_(0x8c33a19b)}},
   {{LL(0x5375d125,0xb2bc7aac),LL(0xc7b2eb56,0x434f55ea),LL(0x8112c880,0x15208222),L_(0x612138e3)}, {LL(0xd30babc8,0xa6392e60),LL(0x7fcbfb2b,0xc2c437cd),LL(0xb4541276,0x134be404),L_(0x3de33a7c)}},
   {{LL(0x2cb35099,0x27c80981),LL(0x6d995371,0x15e5f210),LL(0x1e16c2dd,0x0e59bd58),L_(0xfbad79fb)}, {LL(0x819dc1a9,0x481ca9ce),LL(0x6cb52c72,0x7ae12702),LL(0x7e9afc86,0xfda3af10),L_(0x41a661b7)}},
   {{LL(0xbab0c519,0xc421cd20),LL(0xb68b5cf7,0x1e9b3f9e),LL(0x107cec0d,0x81208f8b),L_(0x383162ed)}, {LL(0xf0c38656,0x2a8bfdfd),LL(0x0f977619,0xedf48fbf),LL(0x382decc1,0x8e5df4ce),L_(0xf0414c84)}},
   {{LL(0x3083dd2a,0x15f29276),LL(0x89f9b2a7,0x717244c4),LL(0x817b0a58,0x6ee121aa),L_(0xf56eeafc)}, {LL(0xcb4b4066,0x0bbfabbe),LL(0x52eb19cc,0x3641db33),LL(0xc2b09e5e,0xa82ae35b),L_(0xd9cf95b8)}},
   {{LL(0x8dada775,0x017073a4),LL(0x4f731dd0,0x7488377c),LL(0xd07b63f1,0x5e605b0d),L_(0x4769aa10)}, {LL(0xf9e533a6,0x26cfa7a4),LL(0xfac43672,0x077deba1),LL(0xbb074c5c,0xe02c4623),L_(0xb6536982)}},
   {{LL(0x711f0c09,0x0bd8823b),LL(0xca3a4054,0xe696bd3c),LL(0xbee559f1,0xac8c4662),L_(0x54b8d320)}, {LL(0xff20eb5d,0xfc9e136f),LL(0xafaff913,0x7fd1bf6f),LL(0x15a9349a,0x81208c2d),L_(0x0d06fd25)}},
   {{LL(0x5b98ceba,0xc4906f43),LL(0x3cb8d3d5,0x671d2405),LL(0xef621a84,0x42fee26b),L_(0xcb150abc)}, {LL(0xfa8ba5fe,0x8c081900),LL(0x2e00a64e,0x548ca8ab),LL(0x4b9fa7b4,0x7c4d0b48),L_(0x60c91243)}},
   {{LL(0x612a3e39,0x36d03f9a),LL(0x7049620b,0xf3a56627),LL(0x29c0b51d,0x50448eab),L_(0x3f0b2dd9)}, {LL(0x673df2ce,0x217c3b7c),LL(0x8eca2e20,0x17355feb),LL(0x20883efb,0x3afc2315),L_(0xc23211b5)}},
   {{LL(0x4fe29bd3,0x3c1d5c97),LL(0x00ef3bc9,0xb31d8854),LL(0xf1f069cc,0xc5bebf01),L_(0xf661ac1a)}, {LL(0x08af7536,0x54219e41),LL(0x25a7fa91,0x10eb2c79),LL(0xf217cbfb,0xf0d4c0bf),L_(0x4f2df7eb)}},
   {{LL(0x0daa5bcd,0x8f2773da),LL(0xb21c4cfa,0x88c3cc76),LL(0x52c04b62,0x590a0567),L_(0xf35f4a84)}, {LL(0xdf2a6679,0xcc2e1148),LL(0x04ae8b1c,0x340ba5a3),LL(0x92f66f03,0x4053731c),L_(0x20aba0c9)}},
   {{LL(0x55e48460,0x0a4bb9e8),LL(0x9ad08ec7,0x4d8d0e03),LL(0x54a4bb3f,0xc9bcb020),L_(0xf5564618)}, {LL(0x0bfda4fa,0x2f0cb8a9),LL(0x64275534,0xc49d2847),LL(0x33405050,0xd7e50869),L_(0xaf97cbcd)}},
   {{LL(0x6faccf10,0x96f1d3cc),LL(0x9f391b14,0x4db1c834),LL(0x5f6bed3c,0x008d20fd),L_(0xc79a2572)}, {LL(0xe2180f45,0x77d13f01),LL(0xb71f049c,0xf95c462e),LL(0xe088bd21,0xbd449f9b),L_(0x3a76432d)}},
   {{LL(0x0405bf97,0x12dbb0ce),LL(0x4f53adb9,0x2a497b10),LL(0x13ff9791,0x61674465),L_(0xa80fdde3)}, {LL(0xc0cc1d8e,0x50e102a8),LL(0x28391810,0x621706f7),LL(0xd119b153,0x47aca745),L_(0xeb9f0102)}},
   {{LL(0xc5273792,0x6dc8e415),LL(0xb01555de,0xe0184e8d),LL(0xc5669efc,0x14d3a409),L_(0x0c68a597)}, {LL(0x41d1a98a,0xed8dbd5f),LL(0xc5474f33,0x98419695),LL(0xd21a5750,0xf6b7aed6),L_(0x0d2d58af)}},
   {{LL(0x4cd02456,0xb63ca144),LL(0xa81eba20,0x84079574),LL(0xf6e2c559,0x4e439115),L_(0xe8f8d978)}, {LL(0x283b6289,0x8cf406a6),LL(0x23a5c7d1,0x4837878b),LL(0x4e16a72a,0x3db551d5),L_(0x88436ae6)}},
   {{LL(0x334d8d79,0x32172eb5),LL(0x7d243c47,0x1ed58d88),LL(0x4a4714a6,0xed3510a8),L_(0xfde7f4a6)}, {LL(0xb3d82e22,0x1ed13734),LL(0x4721f266,0xa661cacd),LL(0xf2889694,0x99b1258f),L_(0x45218687)}},
   {{LL(0xd82826c7,0xe91d500b),LL(0xcf933e7f,0xdfae0ba5),LL(0x2d1ad273,0x572c7767),L_(0x79dbaa6e)}, {LL(0x6b8ab0cc,0x17390031),LL(0x6c991f09,0x81e5fbb5),LL(0xbaab3192,0xfc86b907),L_(0x93c673b6)}},
   {{LL(0xbe8c8998,0x3543430b),LL(0xc2f96d5f,0x50196f53),LL(0xac730ca9,0x376185cf),L_(0xfeac300d)}, {LL(0xe73fa407,0x96df0a24),LL(0xf035d502,0xcdda229d),LL(0x9ca93571,0xd64df17a),L_(0x01970aaf)}},
   {{LL(0x7ca6efb8,0x471f2098),LL(0xa5fb90b3,0xd70e4a72),LL(0x2aa370a3,0x09b936b9),L_(0xcfb37658)}, {LL(0x3b357def,0xb2dfe05b),LL(0x1e8ce8f7,0x3594e46b),LL(0xcf8806b7,0xf062523d),L_(0xa1c74034)}},
   {{LL(0x03cfe393,0x02fd45e8),LL(0xf0d5ce43,0xf575564b),LL(0x86092a2d,0xc28ad111),L_(0x0990658e)}, {LL(0x70cbd16a,0x850acec1),LL(0x097c875b,0xc2704919),LL(0x79052ae9,0x651a1ec4),L_(0xdf5b00e0)}},
   {{LL(0xd20da620,0x7c4aacf3),LL(0x12306aef,0x51b1a962),LL(0x0369e1f2,0x86d193a6),L_(0x3bdb2181)}, {LL(0x93dcf8af,0x4d9c6a9a),LL(0x2e587a72,0xae32ac0a),LL(0xe30d76a5,0xaf97ed7f),L_(0x20fbaabf)}},
   {{LL(0x94bc93d0,0x044baaf3),LL(0x04047c5f,0xf8872ab5),LL(0xe4e3879a,0x436ef7e0),L_(0xed64bf04)}, {LL(0x11c91ee8,0xa9c7bdcc),LL(0xc1679bbd,0x006ae67f),LL(0x38ba10d2,0x83b598cb),L_(0xa99537cc)}},
   {{LL(0x01ef166f,0x18eb43c9),LL(0x5aa367ce,0x93100d60),LL(0x2dc9b947,0x066d6c04),L_(0x3a41602a)}, {LL(0xb9778f57,0xfcc4d377),LL(0x3dbf71fc,0x1d448cab),LL(0xc9ccbaae,0x0aacb390),L_(0x985954f5)}},
   {{LL(0xa4179427,0x1f6e5aab),LL(0x4d58f138,0x64717e7d),LL(0x72e75855,0x389e7a2b),L_(0x4b5aa1e6)}, {LL(0x12a92070,0x29d90bc2),LL(0xb35b2d2a,0x553f857a),LL(0xaca3b9cc,0x832c687a),L_(0xe2911a57)}},
},
/* digit=8 base_pwr=2^56 */
{
   {{LL(0xf505ff0a,0x56b1b68c),LL(0xa766a468,0x85630476),LL(0x7f0a8ccb,0x15b9ce0d),L_(0x8bfb375d)}, {LL(0x8741f066,0xe7944b6a),LL(0xe502fd6a,0x8026fb64),LL(0x079ac7ac,0x956dea42),L_(0x3fc3f258)}},
   {{LL(0xbd407e63,0x7157aab7),LL(0xde649372,0x62a53378),LL(0xec032ca1,0xdc9ac491),L_(0xaf00af85)}, {LL(0x5a1b0fb3,0x3d3a586c),LL(0xb1b4ed91,0x6cce27ca),LL(0x0da4ec47,0x2fb5dda7),L_(0xe69eb235)}},
   {{LL(0xa2c9ee34,0x4a718a90),LL(0x6813c47e,0xaf44fcce),LL(0xcc1aaacb,0xbae4540e),L_(0xfd6dde20)}, {LL(0xf4786198,0xa1399699),LL(0x47c1ef53,0xabb23a85),LL(0xff24f01d,0xa1779b32),L_(0xafe60f72)}},
   {{LL(0xa9ef8159,0x6e7fd2ab),LL(0x206a6f14,0x802458a0),LL(0x43a11c7f,0xf31bbf6c),L_(0xe2ff97fb)}, {LL(0x893d191b,0xb83101a9),LL(0x0b2c24c6,0x52ca6e06),LL(0xd00a24aa,0xc5ea4420),L_(0x80054822)}},
   {{LL(0x3173ee97,0x5d302b85),LL(0xa3ec0bc2,0x275a0fe0),LL(0x7469b929,0xde033220),L_(0x9823907b)}, {LL(0x06ebb14d,0x8d35908d),LL(0x668b8357,0x8a0b74c2),LL(0xcd96886f,0x388da84d),L_(0x8e759b7d)}},
   {{LL(0x82c45d8b,0xb0068631),LL(0xb87f256b,0x51283056),LL(0xb7562059,0x554d7fad),L_(0x68b8bed1)}, {LL(0xe9ccc163,0x5349ed31),LL(0xaa23f48d,0x0ef896e4),LL(0xac744603,0x125ee820),L_(0x150da1ca)}},
   {{LL(0x546f326b,0x16a4c78d),LL(0x26b49a2d,0x42a579ae),LL(0x69e5dc3a,0x50449596),L_(0x2af6b2c1)}, {LL(0xde7ccc7f,0x07439dc4),LL(0xf5736c67,0xacbc3d69),LL(0xeeb790dd,0x0cc523dd),L_(0xae2f8bbf)}},
   {{LL(0xf1c49667,0x1c12b2d5),LL(0x7628ecee,0x7bc41dbb),LL(0x0692cc8d,0x65398808),L_(0x5048f64f)}, {LL(0x2cde35bc,0xd880351d),LL(0xae75c4fc,0x131f80e2),LL(0x3e2ff89e,0x5f9ca860),L_(0x435a454d)}},
   {{LL(0xfc6b3603,0x1dbe11f0),LL(0x805cdd89,0x0ed82341),LL(0x0562fe66,0x5d6a45d5),L_(0xc9eb9f56)}, {LL(0x02dcc377,0x1fbcb272),LL(0x4b25d574,0x873a72d5),LL(0xb78a31c0,0x1466b27f),L_(0x468c47f5)}},
   {{LL(0xba96c374,0xb836e3e5),LL(0xf6743281,0x2c1ed972),LL(0x13f41d4f,0xd77d4340),L_(0x397a6846)}, {LL(0x748b4159,0xbd735496),LL(0x6560f3f3,0xeadd97bc),LL(0x61405ead,0x119c162c),L_(0x9870a7f1)}},
   {{LL(0x7c46ea86,0x4f81eb05),LL(0xdd4548bd,0xaad784b0),LL(0x49ea4452,0xff17a13f),L_(0x54ebcb9f)}, {LL(0x810464d0,0x3e7236a8),LL(0x4d44b2ce,0xfe62fc5f),LL(0xfeac4aaf,0x0d85cf3f),L_(0x7d7cb315)}},
   {{LL(0x4ea682b5,0xb0f78fa9),LL(0x82bbcdcf,0xa319bd52),LL(0x01559ec4,0x7cdb7afd),L_(0x86ff3957)}, {LL(0xc4e5e26b,0x6c320e71),LL(0x45b1afec,0x8d455eef),LL(0x9d12c298,0x6b93f2de),L_(0x3e21f849)}},
   {{LL(0xbb797884,0xc12b655d),LL(0x219c2865,0xa6ec705d),LL(0xc6daf8cf,0xf3ec2977),L_(0x10269fa7)}, {LL(0xf0951e7f,0x8de615f1),LL(0xb448dafb,0x0e454e35),LL(0xac03ce6a,0x0fd25b12),L_(0xb96afd44)}},
   {{LL(0xc1091d62,0x069f14f0),LL(0xc970df0c,0xa04696ef),LL(0x8cddee54,0x0475ddb9),L_(0xac228ae6)}, {LL(0xda3d6fc8,0xc73f5026),LL(0xbdeb5cb1,0x475ad247),LL(0x82098200,0x8694111c),L_(0xc11479ab)}},
   {{LL(0xf181f7b2,0x0c73fe98),LL(0x4aa502d3,0xe9de471e),LL(0x6c27909f,0x4f620050),L_(0xf76957c1)}, {LL(0xe0c4eaa1,0x1565b30a),LL(0xfba38431,0xb518ef0e),LL(0x65c7038d,0x82a6a124),L_(0xe7db47e4)}},
   {{LL(0x56f2fd55,0xfdab8dbd),LL(0x773fef23,0x198d2eb9),LL(0x011efbb9,0x8698e8b3),L_(0xa4f63ad2)}, {LL(0x8629eaf2,0x60003ff6),LL(0x5aebc019,0xd2455886),LL(0x00bcb9de,0x859f79cb),L_(0x6a9d21d2)}},
   {{LL(0x7a576508,0x8a59362e),LL(0x21480f43,0xbd8af994),LL(0x4a37a459,0xe55838d3),L_(0x36aab717)}, {LL(0xd4ef1a0b,0x1d33aa1b),LL(0x86dc1133,0xe457ffeb),LL(0x0a1e4a76,0xf9d8ea27),L_(0x7a9f7b61)}},
   {{LL(0xc6332530,0xf5ab77c1),LL(0xb7c034fa,0xcae4753a),LL(0x89204639,0xd0dd8159),L_(0x4ba725f9)}, {LL(0xae3159be,0xa510602d),LL(0xc72415fe,0xd6d54251),LL(0xa6bc7ccc,0x7d585ae6),L_(0xc90e9462)}},
   {{LL(0xb6723f1d,0xaed13815),LL(0x84f02e2a,0x6de52d8b),LL(0xffdbb5f9,0x3c9269f7),L_(0x5c61743e)}, {LL(0xb086030f,0xd5bc6378),LL(0xa9132224,0x607634ca),LL(0x12f65fd8,0x817445f8),L_(0xd0c982b8)}},
   {{LL(0xfb02a22b,0x40f7fe3d),LL(0xcb5018ef,0x34d86234),LL(0xf32b39b6,0x621a4bd4),L_(0xe7c33edc)}, {LL(0x974f5440,0x7972a383),LL(0x586d4bef,0x8b71269c),LL(0x81211aa8,0x8c46dfe1),L_(0x5b192417)}},
   {{LL(0xe3e15473,0x30ddafb0),LL(0xe56640a6,0x71464526),LL(0xe01ca53a,0x6b7735ff),L_(0xa4ba7379)}, {LL(0xfee843b7,0x184d5072),LL(0x1f2853f9,0x0b71f2dd),LL(0x0c65322e,0x77546dfc),L_(0x814529cf)}},
   {{LL(0xbb7e4da9,0xe44092c0),LL(0xdcf36025,0x6e08eec1),LL(0xa38b0382,0x2beaabb1),L_(0x57dab020)}, {LL(0x30b6f340,0x3fc25cd9),LL(0x16190d52,0x0d13ca42),LL(0x632514c8,0xc69b7cce),L_(0xcb82be1e)}},
   {{LL(0x3eb4e07f,0x79d31336),LL(0x78cc588c,0x986322bf),LL(0x5d3ca505,0x0951b466),L_(0x82b9278f)}, {LL(0x20636a0f,0x6c160f3d),LL(0x308d56c2,0x62ceeabd),LL(0x460ed898,0xcccc63ee),L_(0xd27b55ac)}},
   {{LL(0x81017a2d,0x2097cb5a),LL(0x6fb48059,0x423d9fb9),LL(0x8b4a82f5,0x3bf0f71b),L_(0xda304a5b)}, {LL(0x0620aba2,0x5d609c9d),LL(0xb32599e2,0x77106961),LL(0xea5b4567,0xc072d5ff),L_(0xf9f2dc57)}},
   {{LL(0xbe531971,0xdacba735),LL(0xbc32b9c9,0x4c64a7d8),LL(0x154849a8,0x29e4dd9b),L_(0x11dfc29e)}, {LL(0xbaa9dc51,0xc8616f73),LL(0x971ba9a3,0xf61450f4),LL(0x15776b0b,0x8325c930),L_(0x294a4ec5)}},
   {{LL(0x86967b01,0x16f8969b),LL(0x235d5c34,0x3233bbff),LL(0x57aa97db,0x5c79afae),L_(0x417467c4)}, {LL(0x2f6f6091,0xe7950968),LL(0x59ca92ce,0x9d943f83),LL(0xb1a7c34e,0x6203a921),L_(0x066a9647)}},
   {{LL(0x759f1a19,0xfb7e2683),LL(0x84d55477,0x96a642c6),LL(0xbdee722d,0x74bfcbf4),L_(0xf514f2bf)}, {LL(0x80d0ae79,0xba7058a3),LL(0x4713d1ae,0x6c24158b),LL(0xa6c17b5f,0x8941ddd8),L_(0x261d9ac7)}},
   {{LL(0x943bc0de,0xac5e5ce6),LL(0x76370390,0xd093dc4b),LL(0xfce7f694,0x7c34fbe8),L_(0xe3b42849)}, {LL(0x4b4bfe6f,0x8781925a),LL(0xe273ae40,0xfe87a94c),LL(0x777881ae,0x6fc64556),L_(0xb0f8e854)}},
   {{LL(0x498275e5,0x024a93d4),LL(0xa6b66d66,0x05a83965),LL(0x56c335af,0x0805e125),L_(0xe1e39e4e)}, {LL(0x059f9143,0x9bb438e6),LL(0x0e8dde7c,0x1ec8aae3),LL(0x06a38af8,0xa234eb6c),L_(0xd74870f8)}},
   {{LL(0x0883fffc,0x05b51c39),LL(0x62d8ac43,0x839148f1),LL(0x8eefcd87,0x3b151865),L_(0x3c047e9d)}, {LL(0x197a8a18,0x76ea2960),LL(0x82998a41,0x8a3902cd),LL(0x110b3b2f,0xbcfe60ba),L_(0x89b19d6d)}},
   {{LL(0xab6f3543,0x8765ec48),LL(0x36e47913,0x1b75763b),LL(0x767e806b,0x81bc9bd7),L_(0x38103684)}, {LL(0x0f165768,0xb8107b90),LL(0xbe9c24b5,0x54d331ef),LL(0x43623ea1,0xf06a675d),L_(0xcbeab5be)}},
   {{LL(0x8e21820b,0xd064269b),LL(0x1ccbf4e2,0x7020a8a8),LL(0xe21fbca7,0x5cee0a5c),L_(0x263913a5)}, {LL(0x8a6b9ebd,0xa3fc8bad),LL(0x047f1949,0x838a0019),LL(0xd1a5d116,0x35a0d475),L_(0x9e3419f5)}},
   {{LL(0x742ad243,0x36c9f808),LL(0x631145ae,0x2db0aada),LL(0x0c423277,0x4bf70210),L_(0x2f1584d5)}, {LL(0x6f50ff50,0xa19699d1),LL(0x8015f024,0xc8530fc5),LL(0xf9755c59,0x25538f41),L_(0x387ddb31)}},
   {{LL(0x1985efbe,0xe8509933),LL(0x9e321e61,0xb8ff1d78),LL(0x7dc68e42,0xf8206e77),L_(0x58e783b2)}, {LL(0xad8c32f2,0x906883cc),LL(0x6f41c769,0xa42d4644),LL(0xbb9527f5,0x38e2f27d),L_(0xcf17789e)}},
   {{LL(0xf5dd9f3b,0x1bf5a9f9),LL(0xae648e71,0xe8b2f305),LL(0x3f5f8275,0x93245241),L_(0x80482c3b)}, {LL(0xb0e6f457,0xef9f19aa),LL(0xaa9e665a,0x6dcb7ba1),LL(0xad522975,0xce379c80),L_(0xf8653e32)}},
   {{LL(0xeea9e807,0x19d1e570),LL(0x4023698a,0x5fb46b4e),LL(0xa19db2ab,0xd2800586),L_(0xaee5fc4c)}, {LL(0xe6f0c329,0x378c9ac9),LL(0xbcb82922,0x142d0bcf),LL(0xc3b8fa15,0x1b19035c),L_(0x1c3a4f74)}},
   {{LL(0x074cbefd,0xf201d616),LL(0x88de2216,0x703ef032),LL(0x409f13a6,0xfc1697ff),L_(0x1f4ee383)}, {LL(0x4f0ae8bc,0xc70f2c0a),LL(0x596702b2,0xf1e207d2),LL(0xaa0989d3,0xcc60a8c2),L_(0x5ba1fcbb)}},
   {{LL(0x4e7d0964,0x3f768dfd),LL(0xee929e0b,0x0dda310f),LL(0x171719ee,0x91e6ec5d),L_(0x48549719)}, {LL(0xb61e9fad,0xa3e2860b),LL(0xa30f11a1,0xa7e2c8e0),LL(0xf804fa42,0x58c435e7),L_(0xd81ff657)}},
   {{LL(0xd7d22038,0x3ee264ba),LL(0x6d74b27a,0xf3577276),LL(0x776b6b64,0xa97538c5),L_(0xdc494d53)}, {LL(0x89e972aa,0xe4acc7fd),LL(0xe5d1ac83,0xf7c63ea2),LL(0xf0a59879,0x9b5f787d),L_(0x0730aa6b)}},
   {{LL(0x6c13249b,0x5d202340),LL(0x15ca4cef,0xfbbc39ae),LL(0x03078647,0x7cbaad05),L_(0x7a152a3a)}, {LL(0xc0834b0b,0x5c759c2f),LL(0x1d48b5cf,0xc983dd84),LL(0x954731e0,0x67babf9e),L_(0xb890cb9b)}},
   {{LL(0xeefdbd63,0xfbda2538),LL(0xd314af3a,0x77f9df03),LL(0x333f8c94,0x07bcc285),L_(0xd37bbc81)}, {LL(0xadb373f3,0x7ba7309e),LL(0xe5686c59,0xb6c90f3a),LL(0x96218ea6,0xbfb2d48e),L_(0x7bc6fa2a)}},
   {{LL(0x5c8452f6,0x91273fc0),LL(0x9f330845,0x12ba358a),LL(0x8d3155c9,0x229515d9),L_(0x4e0d0091)}, {LL(0xaeed3a8c,0x3cca8a70),LL(0xc538ea5c,0x8528e080),LL(0x7f68ef37,0x412db800),L_(0xfc39477c)}},
   {{LL(0xc1f84a4a,0x202d77cf),LL(0x3f50f1ae,0x94eeda74),LL(0xc9d43dee,0x2163dc89),L_(0x82838e71)}, {LL(0xc916bedb,0x68b2f3b4),LL(0x028beb58,0x32389e93),LL(0x2f67136b,0x63c11941),L_(0xf0576d72)}},
   {{LL(0x5b467cc7,0x79a7908a),LL(0xfe08b743,0x237e35d8),LL(0xe4c9aa02,0x3121fc40),L_(0xe6d6660d)}, {LL(0x1c0060e4,0xcc2b942e),LL(0x54acd180,0xab155509),LL(0x4b3ac4d5,0xde75824c),L_(0x91ca23fa)}},
   {{LL(0x4201e6e0,0x46582482),LL(0xa927f3c9,0x78c500df),LL(0xe88a4310,0x1f445777),L_(0x3e33ca59)}, {LL(0x2a806dc7,0x71dd2626),LL(0x54a8126d,0xc14947d3),LL(0x9c986434,0xbec3ed13),L_(0x71db698f)}},
   {{LL(0x422ef39a,0x731eb54c),LL(0xa93b1752,0x61227804),LL(0x6f7bfa09,0x873d3a45),L_(0x4f66d06f)}, {LL(0x6fc87ed2,0xc97be3e6),LL(0xf6435b79,0x338ab476),LL(0x207b9b6d,0x1c6c74ae),L_(0xb64b6b70)}},
   {{LL(0xfd951bda,0x74ccc359),LL(0x4be5a10b,0x44c6bf22),LL(0xb790b940,0xc32a94b3),L_(0x098b38b5)}, {LL(0x57ac1bc0,0x4ef7c296),LL(0x18feb7b4,0x82d9c658),LL(0x6f3016a9,0xad72b80e),L_(0xbdf2db87)}},
   {{LL(0xcffa9aa8,0x0ca1adfd),LL(0xb13f2859,0xed4f750d),LL(0xd393e869,0x277e9cd1),L_(0x90b15b52)}, {LL(0x9386d23f,0xf705f504),LL(0x2c2d2fb1,0x98b6973f),LL(0xf49260e8,0xcb991398),L_(0xfed48a8d)}},
   {{LL(0xb2b27562,0x2687c047),LL(0xc3ca0fc2,0xe54d1276),LL(0xbb9d5b5b,0xe241d7c5),L_(0xef9d2be9)}, {LL(0xe9d4f361,0xfe8acc56),LL(0xc5011f32,0x5fc31f72),LL(0x8545eca9,0xfa8d83be),L_(0xc270dd4a)}},
   {{LL(0xe5151a0f,0xb3951e08),LL(0x78a942de,0xdd13a9d6),LL(0x57fae761,0xbc46ee6e),L_(0x83169c42)}, {LL(0x2d1c6395,0x340fc521),LL(0x669821ab,0x24a2322a),LL(0x09548072,0x92a505ce),L_(0x83ec59bb)}},
   {{LL(0x0a4dfff3,0x3ce8adae),LL(0x41ed0956,0x7f33cce3),LL(0xc0e55577,0xba8eb552),L_(0xa018b90e)}, {LL(0xa0f2bcda,0x67316cff),LL(0x5703fba5,0x58c433f9),LL(0x6078c1e4,0x99f221d3),L_(0x047eec80)}},
   {{LL(0x41f43d8c,0x81739c7f),LL(0x6f11b810,0xae918a0f),LL(0x1e314860,0x9ddc21df),L_(0x18e4d189)}, {LL(0x558db016,0xe78c5637),LL(0x439dc1dd,0x0de5812f),LL(0xa22acbf8,0x9f469794),L_(0x161460c0)}},
   {{LL(0xb19e5bf0,0xcd4a85b0),LL(0x7e2cff5d,0xfd828b89),LL(0x998dca20,0x3d5bd0f7),L_(0xbfc65978)}, {LL(0x176f7759,0x388987a0),LL(0x9b689b7c,0xed6c490d),LL(0x3ea487dd,0x5b3b94d0),L_(0xa77864e1)}},
   {{LL(0xdff87175,0x39413fad),LL(0x796a4cff,0xf20c6ce8),LL(0x978f94f4,0x2c42836e),L_(0xdabb2333)}, {LL(0x344fb7ef,0x683de92c),LL(0x5fe8df50,0xe072aff6),LL(0x705a1d2a,0xf68dcb2c),L_(0x16e49003)}},
   {{LL(0x04c5b257,0xc7d4f66d),LL(0x02ce592d,0x679405b4),LL(0x449fe24f,0x49708f6f),L_(0xe167be96)}, {LL(0x3f036191,0x7456adb8),LL(0xafdb07bc,0x63213856),LL(0xfc9ec5eb,0xb52ff28e),L_(0x55b069a6)}},
   {{LL(0x0dfb54c3,0x1b0cec1a),LL(0xf4484ca3,0xa32d497c),LL(0xde798874,0x05b5ac5f),L_(0x87ada89d)}, {LL(0x771e433f,0xbc022aca),LL(0x93b1313e,0x5af43a1b),LL(0xa4d9094a,0x71a6efc0),L_(0xd897f8e5)}},
   {{LL(0x8e365581,0x5f811bb2),LL(0x9a725a78,0x04edaa41),LL(0x1f6cfcf4,0xe1abeb72),L_(0x6d3872f8)}, {LL(0x6636d80e,0x90cec4c4),LL(0x400da758,0x751efdbb),LL(0x74d4d54d,0x6fed8555),L_(0x4bc5cda0)}},
   {{LL(0xed247186,0xb4a54351),LL(0xa9b678d4,0x349f3364),LL(0xf079b22c,0xcd89ee25),L_(0xec3738e6)}, {LL(0x9a75388b,0x1754341f),LL(0x3b3edf75,0xe93c6d27),LL(0x46732c66,0xcfead94a),L_(0xa689fe7e)}},
   {{LL(0x46839b8d,0x91cb6be2),LL(0xa3af0a2d,0x68643eeb),LL(0x37e8c56e,0x44cf93c9),L_(0xc55af358)}, {LL(0xb74e3e18,0x2941f1b6),LL(0x85525c49,0xe580c8b8),LL(0x7fe18358,0x32c35949),L_(0xbf4a513e)}},
   {{LL(0xf3da6204,0x8adf6261),LL(0x6619bed5,0xe7cd8eab),LL(0x5bb1a013,0xdac22d18),L_(0xe70d7542)}, {LL(0x64ef4466,0x0bfa91a1),LL(0x195a05ff,0xb9ce90fc),LL(0x52bb429e,0x34d0f00f),L_(0x904ed07d)}},
   {{LL(0x335940e2,0x4331ebb8),LL(0xf071319d,0x6f7c1728),LL(0x05e7c4f2,0x1da3b676),L_(0x7abf809c)}, {LL(0x1a1e3a9f,0x9290a717),LL(0xf68174d5,0xab7b1f25),LL(0xa93e4d8a,0x7f4cbc44),L_(0x93b784f5)}},
   {{LL(0x5640e88e,0x7b8373a5),LL(0x1bf6e31b,0xcdec2eb7),LL(0xb8888fa2,0x010ee8af),L_(0xa7cd3971)}, {LL(0x2d9ffc40,0x7e33265d),LL(0xc3866a61,0x26755d7a),LL(0x9e1bb3f0,0x9be25bb9),L_(0x1c158895)}},
   {{LL(0xcbffc265,0x187246e4),LL(0xcb1700db,0xcb116afc),LL(0x3aa73e7a,0x93e55e2b),L_(0xb2ba60e5)}, {LL(0xfa8aa297,0xbcfa0106),LL(0x253d2d5c,0x4919fb96),LL(0x9fa1e7c7,0x04ad0152),L_(0xd483d4ee)}},
   {{LL(0xce14b1f7,0xd95b8ba8),LL(0x1d6eb61c,0x20d43399),LL(0xc7ef0d06,0x47b3133d),L_(0xf255931e)}, {LL(0x5d5a0b15,0x60b98074),LL(0x2344f6b6,0x8b1f2bd2),LL(0x789f9baa,0x00302e85),L_(0xccdbabda)}},
},
/* digit=9 base_pwr=2^63 */
{
   {{LL(0xfb42e9d6,0x228b3a54),LL(0xd1e183b4,0x3efd699d),LL(0x4948ad88,0x18288fb9),L_(0xf45f24f0)}, {LL(0x23d7207d,0x37574acd),LL(0xe86d0db8,0x57c32be7),LL(0x1ccef3f9,0x451b09b7),L_(0xee7d7495)}},
   {{LL(0xa61d5325,0xd52a1c1e),LL(0xd08a8ae3,0xcb34540a),LL(0x750d587a,0x03e86cff),L_(0x27e73149)}, {LL(0x156ef6cf,0x527a9877),LL(0x96f4a96b,0x6ee444f1),LL(0xe4a1f249,0x038153e5),L_(0xd9998bb3)}},
   {{LL(0x2bb80640,0x281ef606),LL(0xee8331e0,0x73bbb0b4),LL(0x71264953,0xa04c6f5a),L_(0xecfe722b)}, {LL(0x85f7bcc2,0xbb4ca37a),LL(0x8ff53f4b,0x6e5931f6),LL(0x42ab81a0,0xb862be64),L_(0x8ba40994)}},
   {{LL(0xe1ee20f3,0x3c280032),LL(0xa9734b49,0xc6f2aff8),LL(0xb12f26d4,0x3686d2f0),L_(0xe0111b72)}, {LL(0x8bf62e5c,0x7a48daba),LL(0x069099d5,0x63c661a3),LL(0xaba55aed,0x6cdc05f3),L_(0x4ea7440f)}},
   {{LL(0x33a9ba23,0x4a15994f),LL(0xe99a66d4,0x022ffe96),LL(0x4d054e49,0x924ebb42),L_(0x155f0d8c)}, {LL(0x4d774618,0xe319a1e3),LL(0xe6d591f2,0x8f71c6dc),LL(0xe9b13d22,0x0834867a),L_(0x39cb42c0)}},
   {{LL(0x4edd01ad,0xb4a0a7e3),LL(0xd28d49c3,0xb472df44),LL(0xab2cbbd4,0x4cb47055),L_(0xcd15bd1f)}, {LL(0x31800301,0x16ef99e3),LL(0xb349d462,0x78a8538e),LL(0x746c4b41,0x4dce044a),L_(0x56cf52da)}},
   {{LL(0x7bf3d9bf,0xb1b9c2e9),LL(0x0196cebe,0xaaa33f41),LL(0xdb961af6,0x51023370),L_(0x68f286d7)}, {LL(0x8d6b25f5,0xad763cf9),LL(0xa07bb401,0xe4c3391f),LL(0xab25bfde,0xac71bbe8),L_(0x06a272b6)}},
   {{LL(0x33ad9d0f,0xd727b458),LL(0x6aab9dfc,0x340cf8e0),LL(0x695de110,0x078f906d),L_(0xa987a86e)}, {LL(0x65ed9ae8,0xd50493d4),LL(0x3eb5588c,0xb2673304),LL(0xce9abc84,0x101a021e),L_(0x2ec773ad)}},
   {{LL(0x492a4dda,0x7bdb0ab2),LL(0xb55a240a,0x08dd2523),LL(0x06998948,0xf475a7f2),L_(0x04f12f32)}, {LL(0xd2a94b0e,0x688143bd),LL(0x73021050,0x5bdf7bb4),LL(0x325ed38e,0x8114699d),L_(0x467cd567)}},
   {{LL(0x3e7baca8,0xcbfe120b),LL(0xe69958d1,0x56a88710),LL(0xa2e1385c,0x9586b08e),L_(0x8d8876e4)}, {LL(0xa05f5596,0xe749a357),LL(0x85028619,0x62d3a285),LL(0xf41d9b86,0x8430e884),L_(0xe0fe7436)}},
   {{LL(0x449eb800,0xe20a701b),LL(0xbc42935f,0x63dac155),LL(0xaf260958,0xb55c4fab),L_(0xb5f1b7b4)}, {LL(0xbaf2a69d,0x3e2c715a),LL(0x61856276,0x158e35b0),LL(0x447c4419,0xfaf57b99),L_(0xa5a859f6)}},
   {{LL(0xe7457d94,0xa50c7d9f),LL(0xaaf263e7,0x5bd93ac0),LL(0x27bdb99e,0x28bc6b58),L_(0xa291a83e)}, {LL(0xfed995c1,0xe155fe7a),LL(0xa92f7c72,0x441255d1),LL(0xfdebd3d9,0x54ea2480),L_(0x1d9b3247)}},
   {{LL(0x8cfadc9d,0x3bb14260),LL(0x40205f43,0xf6fea697),LL(0x1e5b99d9,0xf68ea4a3),L_(0x99449ff0)}, {LL(0xc42cccaf,0xb4680c53),LL(0x6506393b,0x1cded754),LL(0x9fa16281,0x355805ad),L_(0xa8869c1e)}},
   {{LL(0xa2b679a7,0x1fe90870),LL(0x60faf2a7,0x841d50a4),LL(0x74445704,0x97761552),L_(0xc6294a8b)}, {LL(0x88aef1a6,0x8306971e),LL(0x79313f41,0x8a08da74),LL(0x459c33b3,0x5ff5c231),L_(0xa47413df)}},
   {{LL(0x528acbac,0x8931a009),LL(0x1bb13907,0x64ec461c),LL(0x6e5d6bd9,0xf62b4e1f),L_(0x419ecc6e)}, {LL(0xb5b1b6f0,0x2e05fe26),LL(0xa9d7ebb7,0x5484c570),LL(0x46afc58b,0x2c8381f8),L_(0x243d0786)}},
   {{LL(0x67588c0e,0xa5a5886a),LL(0xd9832e54,0xf569db1f),LL(0xc764b06f,0x38ed1085),L_(0x7eb63b3e)}, {LL(0x11a2a8db,0x585bdb20),LL(0x8a62c89d,0x3d5436af),LL(0x5e301e9d,0x35b91587),L_(0x4e488867)}},
   {{LL(0x2e0bb083,0x9471249c),LL(0xcfed9152,0xd02dc9b6),LL(0xf83f6a42,0x859a1f08),L_(0x67c5e1b0)}, {LL(0xfa3c9fbb,0x40557a70),LL(0xa37c1374,0x180e4d9f),LL(0xecd05a92,0x7635cd74),L_(0xccd0d7f7)}},
   {{LL(0x0ad7acbd,0x02596d63),LL(0x81ed48b5,0xf3957977),LL(0x8ec54e25,0xfd23aeec),L_(0xa7513cf4)}, {LL(0xfa5cf86c,0x8f6097dd),LL(0x2a1c180b,0xea60f2b6),LL(0xacdc4130,0x253dbb93),L_(0x9daa0c3e)}},
   {{LL(0xece03d73,0x5f61e12d),LL(0x3336043a,0x1b5a6c4a),LL(0xa83c1fea,0x77fedd08),L_(0x64002d5d)}, {LL(0x6ed15df8,0x08aaa0a9),LL(0x210df7fe,0x6ccfd556),LL(0x6b43ad21,0x98ade169),L_(0x42a1a05e)}},
   {{LL(0x701d31e8,0x6e92d485),LL(0x8a6e2064,0x53a09da0),LL(0xc92637e7,0x4ee89ae2),L_(0xa8cbe2a0)}, {LL(0xc8b9f5ad,0x688111d2),LL(0x1fafd02e,0x75a9c058),LL(0x315d9dc5,0x0cd0654b),L_(0x02890b79)}},
   {{LL(0x24746253,0x131fab03),LL(0x1ac884b5,0xda0c0ec6),LL(0x8a1a27c7,0x66f76b97),L_(0x254a6db2)}, {LL(0x8c4da1c1,0x17450bac),LL(0xd2c4396b,0x988596ca),LL(0x2691986c,0xd9864a47),L_(0x31c0544b)}},
   {{LL(0x24fed0d9,0x2f0f74a0),LL(0xf3ddaad3,0x28c66344),LL(0xa0f85074,0x55736fde),L_(0x9f28ac0a)}, {LL(0x0c8f2cd4,0x5b08d2ff),LL(0xaaec443d,0x96848bd3),LL(0x9fd17e0f,0xb4213134),L_(0xf10e3114)}},
   {{LL(0x0077c0a8,0xd9301e91),LL(0xa6577039,0xdf36f0d6),LL(0x50e6f1b9,0xf5e86ab5),L_(0x58a7b434)}, {LL(0xe0f8d5dc,0x2f0371f4),LL(0x1d3d9546,0xeabede08),LL(0x39c8d544,0x12bf2df1),L_(0x57a123a0)}},
   {{LL(0x3878d553,0x817dc769),LL(0xfd0338c3,0x60a4d34c),LL(0xb99ba802,0x36e30856),L_(0x1dcffb49)}, {LL(0x8ba72940,0x665a28b4),LL(0xf1016ea1,0x9d6822ea),LL(0x86a30740,0xc3331f1b),L_(0x0205d1d8)}},
   {{LL(0x69c5e4bf,0x8628180f),LL(0xd7ae93c1,0x7ead547c),LL(0x227716ff,0x736914e5),L_(0x605a49ab)}, {LL(0xec82f273,0xa3fd7cc4),LL(0x99cf069a,0x60d1dc75),LL(0x93c53857,0xdc5b96cb),L_(0x080bcc0e)}},
   {{LL(0x5adf052d,0xf2335efb),LL(0x398d73b0,0x9cde465e),LL(0x00bb4ee5,0x212ffe0a),L_(0x7ab2ea4d)}, {LL(0x9a9ad333,0xe2f332a2),LL(0xeeeaddc8,0xbb4703c4),LL(0x2962d309,0x143e5fcc),L_(0x49553c63)}},
   {{LL(0x99cf8b3c,0xc941479d),LL(0x203c4316,0xe6b52643),LL(0x855dab4e,0x717ef700),L_(0x4ed7e83c)}, {LL(0xe00ddd9b,0x494ccfde),LL(0xe90530ff,0x94ed4ea6),LL(0x8222ad9a,0x4430a4d7),L_(0x252e55b2)}},
   {{LL(0x53d9c3c5,0x96506aa3),LL(0x3f484d6d,0x4d8a0741),LL(0x10e92da9,0xbd490fb2),L_(0xffd03214)}, {LL(0x5fa8cebc,0x95792257),LL(0x7dfe3775,0xe4e706c8),LL(0x9120d23f,0xe2412ec1),L_(0x1de70984)}},
   {{LL(0xa39f223e,0xcb479a89),LL(0xe7cc4f73,0x0ea9b422),LL(0x1382187c,0x9dc7e0af),L_(0x01c1bd80)}, {LL(0x530c072d,0x7d0256e7),LL(0xbd971088,0xbcc29991),LL(0x7ac3d22e,0x36e29a92),L_(0xecf0d361)}},
   {{LL(0x28113fdf,0xbb9273c9),LL(0x4c568852,0x6ab4dc1c),LL(0xf3f274f5,0xb258fcff),L_(0xdee7fe95)}, {LL(0xf7dd8fb6,0xea16df71),LL(0x3f7227a1,0xf774dfb9),LL(0x8865b774,0xc554b491),L_(0x10280ec0)}},
   {{LL(0x32e80670,0x0aaf5932),LL(0x600576a1,0x0829e9bc),LL(0x63228df7,0x7b64399d),L_(0xb12890cc)}, {LL(0xce0e14e3,0xf166eabe),LL(0xa3e9c521,0xa17a6f3d),LL(0x01f649df,0x33266d1a),L_(0xb179110c)}},
   {{LL(0x0d9efc66,0xaedb7e51),LL(0x2d780714,0x3d4cd6e3),LL(0x76c48d11,0x95533ca3),L_(0xef834a13)}, {LL(0xfcf0269c,0xbb0ea84c),LL(0x8bf5f008,0x2e995288),LL(0x0c5e907a,0x5462d73b),L_(0x1d6aecec)}},
   {{LL(0x80c6b892,0xb5bf76ef),LL(0x2e91623e,0xaf49e877),LL(0x1415663c,0x15fea421),L_(0x1f1c3707)}, {LL(0x71645fb0,0x03dd8cca),LL(0xb068031e,0x639980b2),LL(0xd50a50f4,0xbde1692d),L_(0xc8df6655)}},
   {{LL(0xb536b786,0xa49675be),LL(0x923698c8,0x410ff3c8),LL(0x9ea67a42,0x67f23bf8),L_(0x4444d11f)}, {LL(0xbb783ec9,0x05829e19),LL(0x18b3a870,0x79d71996),LL(0x8b8364e8,0x917428eb),L_(0xafd1b685)}},
   {{LL(0x391fe0d9,0x5cdc38f5),LL(0x8bdefd85,0x653fae2f),LL(0x60854dd0,0x5650f1c2),L_(0x3d41abd0)}, {LL(0xb709cc99,0x338d5d07),LL(0xd45495b6,0x840a9918),LL(0x10d2d56f,0x05c210e8),L_(0x2709cd44)}},
   {{LL(0xd04c44b0,0x3d513962),LL(0xb778f7da,0xedf97e8d),LL(0x237a1a75,0xb843c30f),L_(0xfe6c9001)}, {LL(0x52dcbf0f,0xb7f1d85e),LL(0xdfc37735,0x3f355190),LL(0x63ca258b,0xbcb01029),L_(0xaad826a1)}},
   {{LL(0x74f4c74e,0xf2d99edc),LL(0x3824eb60,0xcbcc9613),LL(0x96cabc5a,0xdb4188d8),L_(0xe5012748)}, {LL(0xbf865e18,0x100a7041),LL(0x73b6c9c4,0x922a5abc),LL(0x4b2e4ffc,0x3c179919),L_(0x35fd801d)}},
   {{LL(0x274742c8,0xfba7fe0c),LL(0x71487144,0x4d838b83),LL(0xa5684725,0xde29806b),L_(0xaf1f4b15)}, {LL(0x3c9d276d,0x0418df98),LL(0x3da078d2,0x970cee18),LL(0x9504ef04,0xd6fca127),L_(0x68282ff5)}},
   {{LL(0x2c2165ac,0xa4371233),LL(0xf24ea908,0x340a8ba9),LL(0xbf610642,0x751d8536),L_(0x169f256b)}, {LL(0xa6be26ab,0xa7afba2b),LL(0x66d0d844,0x20c6f4fd),LL(0x292f0814,0xb9773f57),L_(0xc95c32ed)}},
   {{LL(0x975db7a6,0x931f72a8),LL(0x2d2921a4,0x75be82f0),LL(0xaabec6ae,0x93be322a),L_(0xe0530ac5)}, {LL(0x4d6dd8a4,0x21ed4c4a),LL(0x3539fb21,0x702437c5),LL(0x54c5601d,0xff29a4db),L_(0x33f41e0a)}},
   {{LL(0x957fea7b,0x92370d49),LL(0x2b5f5d00,0x1feb014c),LL(0x9ce3670e,0xc246a235),L_(0xcfeee196)}, {LL(0x58ffdb8b,0xc688f198),LL(0xc992ad9e,0xee953054),LL(0x3828369e,0xc2172706),L_(0x9016c246)}},
   {{LL(0xf2b16a57,0xd8546f62),LL(0xf288a32a,0xf4996e54),LL(0xd866770d,0xa8e0a566),L_(0x41c5b1a5)}, {LL(0xa81aa974,0xc8eca336),LL(0x06f5713c,0x7317dc66),LL(0xaf5394d4,0x1c420a16),L_(0x05bdbb33)}},
   {{LL(0x3d683b90,0x8e6a5c3e),LL(0xf167321f,0xd4ab4b02),LL(0x4806da87,0x98fd2767),L_(0x23e3b0df)}, {LL(0xf9faf1d6,0xde00e6e4),LL(0xa65aaac7,0xb82c1e61),LL(0x603b9246,0x39526356),L_(0x6098239e)}},
   {{LL(0x1471049c,0xdf432ea3),LL(0x989bae16,0x08aae4b3),LL(0xcdff8338,0x1f3f8b31),L_(0xe12f868b)}, {LL(0xa96b85c5,0xc59a6e02),LL(0x3e2e2a6e,0x679432f4),LL(0x35623aa8,0x66bc9d0b),L_(0x419d9c04)}},
   {{LL(0xd12c1c29,0x3e403fad),LL(0x80a49b71,0xb3fa108c),LL(0xa16aaa21,0xc0552283),L_(0x941ac341)}, {LL(0x5f01091e,0x78d10e07),LL(0x9e2010ad,0x78aaa1d0),LL(0xc3d626dc,0xad4eebe0),L_(0x616d3812)}},
   {{LL(0x77d51621,0xa85af39f),LL(0xc44a4210,0x41d18ad2),LL(0xd700b9c9,0x00fd9c15),L_(0xaaeaf056)}, {LL(0xcb08ab64,0xcc55572b),LL(0x58e16fee,0xdedd8934),LL(0x224b2130,0x27e82ee6),L_(0xe838331f)}},
   {{LL(0xc76dfaad,0xf7314697),LL(0xc2d9578c,0x651c0514),LL(0x28f766e4,0xe5782e17),L_(0xdb1237a2)}, {LL(0x6bb64cbb,0xdcb19e7b),LL(0x382be491,0x9ac56c1c),LL(0xf76d352a,0x0749b544),L_(0x7968d6a8)}},
   {{LL(0x07203eb5,0xceb29d96),LL(0xe4a0f31c,0x3f66d60f),LL(0x62bd6086,0xc9e0da34),L_(0xd2a91a43)}, {LL(0xeb1e3a08,0xf153aa4a),LL(0x13715918,0x07234640),LL(0x5c701eb4,0x3cc2d39d),L_(0x21bc78c7)}},
   {{LL(0x481eb504,0xd3a03cad),LL(0x5f6a9d91,0x1d390d95),LL(0x3753e069,0x733ae857),L_(0x58532343)}, {LL(0x4de25ac2,0xc858b2c5),LL(0x423a2e5e,0xd0e29a40),LL(0x03c6d3e7,0xbd4e4f93),L_(0xb99af7a6)}},
   {{LL(0xf347240e,0x74665b47),LL(0x00f3a821,0xd4995fe7),LL(0xbb254497,0xae564546),L_(0x09235190)}, {LL(0x8aa0bb23,0x148c2b64),LL(0x44c9ade6,0xd4712255),LL(0x01120f40,0x2095b0d0),L_(0xc2cfde05)}},
   {{LL(0x7058189d,0x9096ca15),LL(0x0b95bea1,0xf2ebd4a0),LL(0x92a3cb12,0x403109c3),L_(0x67ede54a)}, {LL(0xc8d15887,0xd330e122),LL(0xe94c1522,0xbc6cec87),LL(0xcf570b75,0x8c60e503),L_(0x7b779c33)}},
   {{LL(0xc396d1d3,0xb7a09eba),LL(0xb7df5c36,0xa4ad2686),LL(0xda7c16c6,0xea76230d),L_(0xe12538a4)}, {LL(0x5c153671,0x35168d66),LL(0x22a3c5aa,0x7317fdf8),LL(0x90a1a3b8,0x488f516d),L_(0x4a17692d)}},
   {{LL(0x427a3fed,0x979163af),LL(0x5261367c,0x67750f39),LL(0x2b64c77f,0x121123d3),L_(0x57b5327b)}, {LL(0x1ae75801,0x60be5340),LL(0x4320f770,0x1c82e2fe),LL(0x80d5de38,0xbfa35c85),L_(0x804730a8)}},
   {{LL(0x7c65d352,0xfec3e94c),LL(0xe49a8416,0xe15daed0),LL(0xebea12bc,0xd561b8e1),L_(0x1b6939a7)}, {LL(0x80b4e8f4,0x689cf9a4),LL(0x0a52d61b,0xe93a4cce),LL(0x2707af62,0x00163171),L_(0x888d6c2f)}},
   {{LL(0xdaeed59d,0xcfab7c64),LL(0xd2cae7f9,0x5781a043),LL(0x10e60343,0x1e1d00e2),L_(0xb1b2d682)}, {LL(0x8a649d20,0xbc56c4a9),LL(0x87923cc3,0xa8b43dfb),LL(0x05fa3aa6,0x07ea55ed),L_(0x08a82010)}},
   {{LL(0x208d5b3c,0x5178b723),LL(0x10a520c7,0x2d1f55b2),LL(0xb1fffc0d,0xbbad0a50),L_(0xab6af890)}, {LL(0xce6aec24,0xf8407690),LL(0x2bcfca02,0x1c5424aa),LL(0x56944f46,0xaab35dc1),L_(0xa5c174f8)}},
   {{LL(0x957be638,0xd8227572),LL(0xfd6ac99b,0x7fcc54ba),LL(0x4081f3db,0x7e998f3d),L_(0x08e76dcc)}, {LL(0x3ffd1f3b,0x1c7b8f2b),LL(0x0551cbe8,0x76e4b694),LL(0x045398f7,0x93712a63),L_(0x31b716ac)}},
   {{LL(0x9ccd7f23,0xf2590188),LL(0x1074bc24,0x14031de7),LL(0x9d9df6f6,0x1a06a2f7),L_(0xf880ea5b)}, {LL(0x9cc577d5,0xe509ec7a),LL(0xc7021265,0x864b62e6),LL(0xe9189561,0x09271f5a),L_(0x910bdd23)}},
   {{LL(0x1f12f8d6,0x258fd822),LL(0x28518543,0xa7361b32),LL(0x338b1788,0xc4796256),L_(0x636fbcb3)}, {LL(0x4f5097e8,0x0aff7ca0),LL(0x9306ff92,0x3deb3281),LL(0x6a18552c,0x5ba5cce7),L_(0x563c7d80)}},
   {{LL(0xf184055c,0x0a2d6941),LL(0x682b34d2,0xba0b737d),LL(0x467f7fe4,0xa7e4e833),L_(0xfb7c4f69)}, {LL(0xceeed886,0xd5513495),LL(0xc4ac863c,0xcc4688b1),LL(0x74229c34,0xd5e5f34b),L_(0x729a99ef)}},
   {{LL(0x33e6967e,0xca92d3be),LL(0xacf9987e,0x27e6ecfd),LL(0x84c9d53f,0xa434a89c),L_(0x0ac3dbe1)}, {LL(0x3f16aded,0x05a4b104),LL(0xbbe7f4a2,0x9d1faddd),LL(0x2222c720,0x0091add4),L_(0x5e600780)}},
   {{LL(0x6e5923eb,0xe0a35b29),LL(0x5acac230,0x1ca3d3f2),LL(0x5089506f,0x57de9730),L_(0xf05a525b)}, {LL(0xc963a675,0x8c76bd08),LL(0x74d9f43b,0xdeecdd8c),LL(0xdca2c05b,0x4bcbac2f),L_(0xdc6633b9)}},
   {{LL(0x60601ff6,0x1bc30bde),LL(0xa7ddd558,0xcac4e4e0),LL(0xe21dd977,0x66baebc4),L_(0xb4d573cb)}, {LL(0x79ee6d3c,0x352d2cee),LL(0x3c8cf82d,0xe3db9857),LL(0x12474a96,0x8d29a0c1),L_(0x55049164)}},
   {{LL(0x3ff1fac3,0x8806d22d),LL(0x827b3141,0xe5b92d9d),LL(0xc0bcac45,0x45f1bee6),L_(0xe2c1046e)}, {LL(0x368d7809,0x0b973abe),LL(0xc60d4681,0x6f45c847),LL(0x05dd04f4,0x1ae3e5bf),L_(0x64bbe857)}},
},
/* digit=10 base_pwr=2^70 */
{
   {{LL(0xad2244ee,0xa40697e3),LL(0x6dcf122e,0x379dd18c),LL(0x179d681f,0x177fcb32),L_(0x2e1c6f2e)}, {LL(0xd9f2b14a,0xdfe5e464),LL(0x4c70a028,0xfcfc8017),LL(0x6183ea61,0x4d64db2b),L_(0xd053bce3)}},
   {{LL(0xa9c848c0,0xe6f3cecd),LL(0x3c37feb1,0x1953cde4),LL(0x857f0263,0xbbc0b11e),L_(0x091b8ddb)}, {LL(0x46acc3ea,0x8541cfa6),LL(0x772c3ab4,0x7c142776),LL(0x71e71be0,0xcd2d8bee),L_(0xd29d11b6)}},
   {{LL(0xc6b24ae6,0x4b9ffc6a),LL(0x8ee5427c,0x4a42092a),LL(0x997ab138,0xff6b61a7),L_(0x3b574c18)}, {LL(0xd6cb4977,0xbeade6c6),LL(0x315f47ce,0x2e2d5dc5),LL(0xc77bac27,0x39f2d0a5),L_(0x28b192db)}},
   {{LL(0xa902a170,0xdfa7909e),LL(0x374a8799,0x93925c6a),LL(0x121579dd,0xb4c9c6a6),L_(0x2b2a2fe1)}, {LL(0x421343f7,0x2dec18b9),LL(0x467e925a,0x4a7ce3d6),LL(0x51b39839,0x33d05082),L_(0x6dd45eec)}},
   {{LL(0xaf22f4e9,0xe4793da0),LL(0xce68ae65,0x6d70ff7a),LL(0xdf1eb924,0x67519764),L_(0xa6d12a17)}, {LL(0x9c9b86f4,0x1e13ad01),LL(0x43f0840b,0xebcc7b33),LL(0x648dece5,0x0b8a73e1),L_(0x9e1099c5)}},
   {{LL(0x3ee5fb2f,0x00e5b507),LL(0x16e7b26d,0x02eb10ab),LL(0x2fb5b17d,0xf5f94483),L_(0x964b96f7)}, {LL(0xb10c75ea,0xb577452b),LL(0x8083550f,0xcf241d80),LL(0x2f106bf6,0xe11783de),L_(0x0cfd54c0)}},
   {{LL(0x598729b4,0xd3bf6d02),LL(0x6afe9c51,0xfb3e35f6),LL(0xbb0f1cd8,0xb0816531),L_(0x27df4a0c)}, {LL(0x2f8214c2,0x72f5b39b),LL(0xc24bf200,0xdd44ad34),LL(0x39525315,0x2d46495a),L_(0x758c33c9)}},
   {{LL(0x6fa7c2c7,0x9c7bf299),LL(0x2cb77bd8,0x4a4734a4),LL(0xb7245ac9,0x0a33b879),L_(0xe50ef27d)}, {LL(0x3e4bd3ce,0xb91f6979),LL(0x1ca8036c,0xf8b8d90d),LL(0xbee50919,0xf053ecd4),L_(0xece7c952)}},
   {{LL(0xe88a9300,0x990b1bdd),LL(0x0420911c,0x516f881a),LL(0x4138b100,0xafe94d6f),L_(0x24ff9ef4)}, {LL(0x4c68e2d4,0x528a709b),LL(0xa50cb647,0x19ff7b20),LL(0xbb306765,0x907cbbc6),L_(0x74217b89)}},
   {{LL(0x0a371fe5,0x0bab31b6),LL(0xcccd2b33,0x69aaf848),LL(0x5f8e6b05,0xb7efd33b),L_(0x4cd501f3)}, {LL(0x117abd54,0x83d20ea2),LL(0xf475121c,0xa34d2ec5),LL(0x47427d11,0x6eb988c6),L_(0x7acc59a0)}},
   {{LL(0x96650b0d,0x591ee335),LL(0x16ee1f0b,0x26a19153),LL(0x27985f50,0x7a51d8bb),L_(0xfe74d9ab)}, {LL(0x276b185c,0xaa92cd9d),LL(0x9ed60020,0xb66c17eb),LL(0x2a0eac87,0x18315901),L_(0x8dbf60b9)}},
   {{LL(0xe73f9dd1,0xaaf7ef5f),LL(0x3d3b7078,0x315302f9),LL(0xf09fd9b4,0x00b8f278),L_(0xf7dd85ca)}, {LL(0x4140a60e,0x54e7df9d),LL(0xe305da30,0xaef051a2),LL(0x67dd41ab,0x88c0e226),L_(0xc437eba8)}},
   {{LL(0xaecd7f62,0x938be7f6),LL(0x7f0d8c8a,0xe465c032),LL(0x74f9489c,0xba77e9c0),L_(0x529b3458)}, {LL(0xbea77798,0xfea9f3ec),LL(0x93d7e4fe,0x6ec5677f),LL(0xce0cdb8b,0x1c3846b0),L_(0xe0a036d5)}},
   {{LL(0xa818d0c9,0x3b25f1dc),LL(0x640f0895,0xf998244a),LL(0x6a36628c,0x0a1631ee),L_(0x22c92416)}, {LL(0x79522c82,0xb5997002),LL(0xa3c8067e,0xbaf242be),LL(0xe28f2eda,0xca50379b),L_(0x0c2d44ef)}},
   {{LL(0x7e8a5165,0x1879da9a),LL(0xf1be18cc,0x4b28b798),LL(0x5708ec80,0x088da37b),L_(0xa743ca6b)}, {LL(0xa1588631,0xd86f2e95),LL(0x9bf8faf1,0x97dcfd6d),LL(0xae7389fc,0x784c1096),L_(0x1aa41e81)}},
   {{LL(0xf9a13e9f,0x2380678e),LL(0xe93e1c94,0x3f5ce533),LL(0x68003338,0xa4954b13),L_(0xfa7015a9)}, {LL(0x63905ece,0x3e892821),LL(0x19b8125b,0x4b7bb992),LL(0xd66a6ac0,0x8579e7ef),L_(0x7516c1ae)}},
   {{LL(0x21560c47,0x8051f521),LL(0xfd3b4b27,0xa4371e4f),LL(0x8947385a,0x8c0cdac3),L_(0xd7eaeef0)}, {LL(0xc46bf169,0x0639dd0c),LL(0x862785aa,0x9ec7e487),LL(0xd61bc685,0xd861b3a5),L_(0x83d3ee5e)}},
   {{LL(0x930059f4,0xb145ba07),LL(0x2aaf9072,0x30f8ae06),LL(0x2f3077a1,0xbc426169),L_(0xc99c4c21)}, {LL(0x6c57976c,0x6ee68683),LL(0x236b0866,0xb9cd4e70),LL(0xe6b22138,0xaa9f8c57),L_(0xc8cde662)}},
   {{LL(0xed492587,0xcedbdd1a),LL(0x51eec28a,0xe84b42a7),LL(0xd4deae6d,0x9e368887),L_(0xaded26ea)}, {LL(0x1bd18dec,0xf79fe229),LL(0xa610dbd9,0x0d650fde),LL(0xc8e53036,0x8d4db547),L_(0x6d7b9831)}},
   {{LL(0xb55dd50b,0x6d88e9da),LL(0xaf929b4d,0xbb1af641),LL(0x09f3ff8a,0x2c88cea6),L_(0x95aa1478)}, {LL(0xf6e2f2aa,0x4d819416),LL(0xe1f9b46c,0xce813081),LL(0x3fdecbde,0x368fe183),L_(0xce1f9cb4)}},
   {{LL(0x55d27a4e,0xb55224a5),LL(0xc92c61ce,0xbd06485e),LL(0xe42fc5df,0xbae2229f),L_(0x3afc274c)}, {LL(0xa273191e,0x85cd8bfc),LL(0x97118ed0,0x9c9c61a0),LL(0x5e361376,0xaaf47545),L_(0xcad4d556)}},
   {{LL(0xfb2444ed,0x18d8ff5c),LL(0x18b51ba2,0x3dc74bfd),LL(0xaf0eee18,0xfc1b2868),L_(0xbb73ea5a)}, {LL(0xd68ff57a,0x88d94a2f),LL(0x730a7041,0x162bcf15),LL(0xd183ab03,0x9867c6b2),L_(0xf198711f)}},
   {{LL(0x817437d0,0xf225ca3a),LL(0xebadfce9,0x2afa4968),LL(0x7b328600,0x09069b77),L_(0x1091afcf)}, {LL(0x82fb3189,0xdcc4bc5d),LL(0xf1844abf,0x49ace070),LL(0xf655f51c,0x7e393926),L_(0xa022dcd5)}},
   {{LL(0x13b3c6d8,0x32d143d6),LL(0x07002476,0x37eefa5a),LL(0x028e4eac,0xf7ed9d04),L_(0x65e3cb55)}, {LL(0x22c86f9d,0x1cab2511),LL(0xdf940bc6,0x50c1e3f3),LL(0x734bf045,0x2643f52d),L_(0x11593f75)}},
   {{LL(0xeaec70bc,0x1759457b),LL(0xcee75815,0x1ef41bc8),LL(0xbfd1d973,0x5f64050d),L_(0x6b3087eb)}, {LL(0xda22e615,0xc045345f),LL(0x776dfb51,0x3f9aff61),LL(0x07f4f3a2,0xb17003a3),L_(0xb6a6cf4d)}},
   {{LL(0x9276438a,0x53f04642),LL(0x8525ef52,0x1397f8d6),LL(0x9a0888e4,0xde6f955e),L_(0xda028b89)}, {LL(0x4292050e,0x2711d946),LL(0xd109e67e,0x70541c9f),LL(0x19b5f6f0,0x75754b8c),L_(0x85c8d6d5)}},
   {{LL(0x3073bee0,0x0a0de3cd),LL(0x725322a6,0x3ecd54ab),LL(0xabf6f3c4,0xeb4d3502),L_(0x6dfedcca)}, {LL(0x12c56dbf,0x08adcae7),LL(0x33d4bd69,0x0b5e164a),LL(0xcfe1b693,0x659e68b8),L_(0x82c86924)}},
   {{LL(0xc347067e,0xd351d4d7),LL(0x141df0b6,0x5fe38868),LL(0xaaee8341,0x6a97a114),L_(0xaafe4f6c)}, {LL(0xc0912e86,0x3ce36b09),LL(0x4106201e,0xd6174b63),LL(0x40106385,0xfda3c463),L_(0x9e471096)}},
   {{LL(0x21aa62bd,0x15f6014c),LL(0xb5217d60,0xf4c26c02),LL(0x421c3d60,0xcc23ecec),L_(0xc0f18f53)}, {LL(0x08c04fe6,0x1ae450ba),LL(0xf66ae880,0x7520566d),LL(0x2e31cf5b,0x1b3a8046),L_(0x916c70ac)}},
   {{LL(0x47d17a16,0xd287d1ca),LL(0x3e47dc5d,0x291a00ca),LL(0x42cffa9b,0xc6ddb9b7),L_(0xb751c3c5)}, {LL(0x6b576fb2,0x7f3bfdc3),LL(0x8676c057,0x05c5c887),LL(0x2f6a12cc,0x4e43cafd),L_(0x273385a8)}},
   {{LL(0xae17a9c0,0x1d472cf8),LL(0xb53ac163,0x1569367f),LL(0xf1c57e24,0x5f7dfde3),L_(0x53f9522b)}, {LL(0x45eaeed1,0xf3d29713),LL(0x3c47cbaf,0x7a261ef7),LL(0x561e0755,0xbe952c4a),L_(0x6bcd178c)}},
   {{LL(0x63cc0fbd,0x24b41db6),LL(0x36a421ac,0x34bd0f29),LL(0xdc178877,0x619cb9a4),L_(0xa266089e)}, {LL(0x7b84d003,0x9fd8767d),LL(0xba9a6d1f,0xa5381512),LL(0x8cbdf583,0x46c198f4),L_(0x23f49a00)}},
   {{LL(0x65bc3562,0x281cc4e9),LL(0xc3d2804e,0x7562f859),LL(0x22a16679,0x638a7949),L_(0x7ec2aa7b)}, {LL(0xe6c7330d,0x187243a5),LL(0xb99ca6d8,0xee353b2a),LL(0x2d4040d0,0xbd604fb9),L_(0x8cefbced)}},
   {{LL(0x1fc30151,0xfbd9e1c4),LL(0xd10937a9,0x0f7ffd5e),LL(0x2fa67094,0xe7b103db),L_(0xb57b1840)}, {LL(0x39530ea9,0x1368261d),LL(0x719e1d06,0xc1336cad),LL(0x192c11f5,0xa6cd197f),L_(0x9b41db84)}},
   {{LL(0x37505def,0xd68a5a49),LL(0xddd06586,0xb0511d2a),LL(0x52664a68,0x2621ebc3),L_(0x6f5736f5)}, {LL(0x2fcaf173,0xd267f258),LL(0xbacb7a85,0xe5e678b7),LL(0x5c0b0c67,0xaa7c18e3),L_(0x166ad375)}},
   {{LL(0x669558fe,0x9a21bdfe),LL(0x8411fa8b,0x06c5b604),LL(0xde70b24d,0xf5bd9bc8),L_(0xd7443ed1)}, {LL(0x0a439533,0x2b84584e),LL(0x3ea3f8f2,0x2cf8fe06),LL(0x099fde13,0xbc0565bd),L_(0x8fd32c81)}},
   {{LL(0xc291b778,0x7e78ce16),LL(0x12d214f6,0x6d2b8e47),LL(0x5f204c64,0xaf966aac),L_(0x62d03da4)}, {LL(0xe9be215a,0x0fc086e5),LL(0x14494699,0x77fd96e7),LL(0xcc71f940,0xec60a687),L_(0x65805046)}},
   {{LL(0x4b6c94b4,0x1ee6996b),LL(0x8eb81a1a,0x5a56077f),LL(0x547ea701,0x71308d2f),L_(0x9126c06e)}, {LL(0x9fdcaccb,0x5fecdc40),LL(0x49cacc43,0xd04d1212),LL(0x89bc7a7f,0x6331b109),L_(0x93db1b8f)}},
   {{LL(0x2b6470fb,0x18292505),LL(0x93a8f9cc,0xe0d6f3f0),LL(0x8d658c23,0x33eef875),L_(0xc22c9665)}, {LL(0xb09b3b96,0x2081f85d),LL(0xbea17435,0x11959ada),LL(0x6f44daf5,0x5ff2900d),L_(0xdea368bd)}},
   {{LL(0x224521dd,0xfcb125d5),LL(0x097e443e,0x7d3c0cbb),LL(0x96ff5378,0x0cce254b),L_(0x562b2166)}, {LL(0xc45e09f3,0x73e36236),LL(0x992060e2,0xf844b9b5),LL(0xa968d30e,0x139e42f9),L_(0xe5b89488)}},
   {{LL(0xb57b80af,0x3e86ab5b),LL(0xb177e3f9,0x8e7a956e),LL(0x95050abc,0x580bb577),L_(0x75270515)}, {LL(0xa96f2da7,0x0ce33d22),LL(0xcab8647e,0xf5e4958e),LL(0x139f525e,0x9b1f28f4),L_(0xa2e38250)}},
   {{LL(0xf097a8be,0x193d7b50),LL(0xbe57f373,0x326744fe),LL(0x192cee98,0x66699aba),L_(0x69e0895a)}, {LL(0x51e1d17f,0xe787c995),LL(0xb3f66818,0x7acab4f1),LL(0x5ee5c7cd,0xe1645254),L_(0xb7d10719)}},
   {{LL(0x29d8986c,0x17022e01),LL(0x2f237075,0x0ccc4d78),LL(0x01453fbb,0xba3615f3),L_(0xa703f343)}, {LL(0xf538646c,0xb5aab8cf),LL(0x30b0a7e8,0x4a42c722),LL(0x33588983,0x8a8547cc),L_(0x64a4028c)}},
   {{LL(0x82ecbbdc,0xbbdec907),LL(0x6c0cb953,0x1807ae9c),LL(0xf8e40104,0xca5b89e3),L_(0x5e56c8f6)}, {LL(0x41f32d88,0x040b1ce4),LL(0x76bd7772,0xbd9bda0d),LL(0x396f023e,0x6bb455f6),L_(0x4243e710)}},
   {{LL(0x82b34b90,0xff00cfce),LL(0x7a3a34e7,0xfea1370a),LL(0x59745afe,0x13dec9bb),L_(0x00460ef4)}, {LL(0xc0ccb5ce,0x664b5767),LL(0x467ad23b,0xe91ea77c),LL(0xd6ac872c,0xc30f32ed),L_(0x7527a687)}},
   {{LL(0x4a73f787,0xf2877171),LL(0xa577b4e7,0x05b283db),LL(0x7ac6ab0b,0xbbb22aab),L_(0xce8c4374)}, {LL(0x2cf4237f,0x5d2444bb),LL(0x1362bb86,0x354270cf),LL(0xa58498c9,0x03a3218e),L_(0x4585f77c)}},
   {{LL(0x17d3d4da,0xe1c0eb99),LL(0xf69271e6,0xde536a3e),LL(0x239476a2,0x3701952a),L_(0xb94d2063)}, {LL(0x25bb5fc2,0x39ed615a),LL(0x44f390c3,0x6a9dd585),LL(0xa4418ea6,0x1ce9df6c),L_(0x76cd5c55)}},
   {{LL(0xa10e49d6,0xf5dcc900),LL(0x62fd7874,0x2e74ef98),LL(0x74f61b6c,0x9d3a5873),L_(0x10ca6b61)}, {LL(0x470595b7,0x848244dc),LL(0x81922f08,0xf7db6c87),LL(0xbc236621,0xb2a50ed6),L_(0x0cbc779a)}},
   {{LL(0x274cea83,0x4914c419),LL(0xb8a19540,0x1bc1a1ba),LL(0x957adbf1,0xd2aac8dc),L_(0x64d6ca82)}, {LL(0x96ac9d7f,0xdb015eca),LL(0x42372196,0x03b15d50),LL(0x05451b09,0xe16c7207),L_(0x463bc66d)}},
   {{LL(0x01c4dfe9,0xfc7e965f),LL(0xb8358066,0x8fc594e9),LL(0xa1915c8c,0x8d26603b),L_(0xfdddd6dc)}, {LL(0xc4b1e185,0x2166a1d2),LL(0x79912871,0xad0b86ad),LL(0xe96aa265,0x98a560dc),L_(0x43b8858b)}},
   {{LL(0xa1e7a822,0x2d3f3785),LL(0x95813382,0xcc85095a),LL(0x89e2322f,0x3568d961),L_(0xfbe11c03)}, {LL(0xe15474b7,0xb0011b61),LL(0xb6d17af8,0x94e2b66a),LL(0xfa35ee62,0x88f30207),L_(0x496e0cfb)}},
   {{LL(0x00a8605a,0x233a4ea2),LL(0xed225296,0xfb49bd9b),LL(0x18a0b19f,0x49f70f9b),L_(0xfd4a3974)}, {LL(0x3d19e5f7,0x6935d1e0),LL(0x4ca49b02,0x167ee2db),LL(0x465d1799,0x83dab70c),L_(0x3925587b)}},
   {{LL(0x06b4ef1a,0xadc74707),LL(0x97b00e28,0x7b776a0d),LL(0x4ea78215,0x29de362e),L_(0xe0075db8)}, {LL(0xd1a8437d,0xcd9ea162),LL(0x4071fbdd,0xd6f65966),LL(0x5861958d,0xb0f23643),L_(0xfb6d160c)}},
   {{LL(0xe1dbc405,0x1cf01490),LL(0xfd9abd17,0x781604fe),LL(0xfdabdb45,0xfe106ce3),L_(0x28aec19b)}, {LL(0x0765d40f,0x5dea7b0e),LL(0x9dd7eace,0x05e88870),LL(0x7aa326d2,0x58144fba),L_(0x0c0b55eb)}},
   {{LL(0x4b6bbeb3,0xf492b1fe),LL(0x80211303,0x00d1a0a3),LL(0xa0a42f81,0x812c4f34),L_(0x0513dab5)}, {LL(0x701129ed,0x2b196b3d),LL(0x8292825d,0x14497eea),LL(0xfccdef2f,0x8309d41f),L_(0xa6c098eb)}},
   {{LL(0x4e01759a,0xe5e506e5),LL(0x34984e59,0x5bdb0568),LL(0xc78ed701,0x023cf8a4),L_(0x8818f370)}, {LL(0xadc05868,0xedb22c00),LL(0x61bbfcfe,0x1eb0f72a),LL(0x4b1c28bf,0x87de5c20),L_(0x13f66db8)}},
   {{LL(0x2c5c67ae,0x311b814c),LL(0x18478ee8,0x98f391ee),LL(0x5fc08d55,0x3d6c85b2),L_(0x2a4234ef)}, {LL(0xc63b3540,0xef0ad114),LL(0xfa0e9b48,0x19926812),LL(0xded54ff1,0x8d2d8068),L_(0xe6033e94)}},
   {{LL(0xfcdb3c46,0x567ac24a),LL(0x31e079f2,0x5c7d3b6c),LL(0x876a06e1,0xd77d6297),L_(0xabca81b2)}, {LL(0xcd9c326f,0x27382ebd),LL(0xbd7f7ff0,0x8c9f6089),LL(0xb11d9de0,0x31e36d28),L_(0xba48fe41)}},
   {{LL(0x8c0ffdf0,0x845700c1),LL(0x659750c3,0x9a4535d8),LL(0xbfa2263b,0x46779254),L_(0x426c9332)}, {LL(0xea64e4f2,0xc0627926),LL(0x919159e1,0xd72486ed),LL(0xb8633c7d,0x62049111),L_(0x57a02e38)}},
   {{LL(0x6d5a56c1,0xc0ef9c32),LL(0x87339367,0x1398d5b9),LL(0xd690f5fe,0x4cb8372b),L_(0x74004a4c)}, {LL(0xa8b7608c,0x62fc7f7e),LL(0xc75eca95,0x7fcc6560),LL(0x87991514,0x1a3939cc),L_(0x9eeab34e)}},
   {{LL(0xee335372,0x6dda426f),LL(0xe95ee7c3,0x7a84b92e),LL(0x5637d166,0x0cc0434e),L_(0x108326a5)}, {LL(0x4c69a1c3,0x66908f8e),LL(0x768a4dc5,0x93e7c968),LL(0x24086621,0x0390f0a6),L_(0x34cd95fd)}},
   {{LL(0xffd9518c,0xa579c74a),LL(0xf4d1cd8b,0x245e5d80),LL(0xa682fc6c,0xcf5b3a95),L_(0x13389ce6)}, {LL(0x8944bc8c,0xc24a47b4),LL(0xf37dbb35,0x67296189),LL(0x9d47aaf7,0x196773ed),L_(0xab42affb)}},
   {{LL(0x6c1fae2e,0x78457340),LL(0x0583dc84,0xea2de616),LL(0x6a0c10e4,0xbf46a7bb),L_(0xe6adebac)}, {LL(0xb7cad754,0x3e924b67),LL(0xbf5e7329,0x93303eef),LL(0x9cb33af1,0x5460942b),L_(0x1005e77d)}},
   {{LL(0x5dd3bfdc,0xa399de31),LL(0x867305ce,0xbc0f8f54),LL(0x2200477c,0xd31a5a81),L_(0x9ac2ed5d)}, {LL(0x59ceee51,0xa5a011eb),LL(0x9cb94041,0x53fce407),LL(0xf7a7e8fe,0x26131e4c),L_(0xbcae3bd4)}},
},
/* digit=11 base_pwr=2^77 */
{
   {{LL(0xe3b818f5,0xe00bddec),LL(0x44e43c61,0x4681ea2b),LL(0xc584d310,0xb6fe7f05),L_(0x951b39e7)}, {LL(0x81eb4c72,0x1eae5fc5),LL(0x518f5441,0xdae90140),LL(0xc505aeda,0x216fb5f2),L_(0xbb781e43)}},
   {{LL(0x3cf8ec7d,0xf9a62c2a),LL(0xe0d93869,0x23fabd24),LL(0x6d3f0de8,0xb3106bbb),L_(0xb0f77641)}, {LL(0x6c7c7f7d,0x9b298ebb),LL(0x184aed09,0x9f30709a),LL(0x334394d4,0x79c1d22f),L_(0xd95473d2)}},
   {{LL(0xae76cd6a,0xcb7cb589),LL(0xf3fc070e,0x3ecf8591),LL(0x60dba7d6,0xe19d60f2),L_(0xe420c869)}, {LL(0x59b7cb97,0x097e4daa),LL(0x707e030c,0x7f52c8de),LL(0x0dcda4fd,0x9803c9fe),L_(0x2bbc44ce)}},
   {{LL(0x2ff400c0,0xfabcb249),LL(0x10355326,0xacc122ab),LL(0x91ed19cf,0x4f316ca1),L_(0xd20efc4e)}, {LL(0x940a18e1,0x1a2e490c),LL(0x2b83ac01,0x029cb6e5),LL(0x0fd81ad0,0x8da694f8),L_(0xfcad677f)}},
   {{LL(0xb1b32829,0x248ec9b3),LL(0x11ef88d2,0x35de7b93),LL(0xda678713,0x370ca0ea),L_(0x7a293625)}, {LL(0x1a056e86,0xddfa0a4d),LL(0x1f856294,0xfe05d52e),LL(0x6bd2e637,0xf728ccb4),L_(0x91baf444)}},
   {{LL(0x74014196,0x9e4adc25),LL(0x8981411a,0xf964b6c4),LL(0x7eb4dd2d,0x58e82e86),L_(0x97c312ce)}, {LL(0x153d2149,0xd7b84e08),LL(0xc95c8ff2,0x4e547787),LL(0xd7ac3655,0x7546fc40),L_(0x504a9f59)}},
   {{LL(0x3f1e5f11,0x9edb0425),LL(0x476b8958,0x63c34818),LL(0xf14565b6,0x4bab1b96),L_(0x3d3ca262)}, {LL(0x18ef180c,0xfd4e9a47),LL(0x0043c8ff,0xa1174dbb),LL(0xafaf8911,0x0b1e0a8d),L_(0xcbeb2d8e)}},
   {{LL(0x486cbfa7,0xebbefd50),LL(0x44fc634e,0xe20aab71),LL(0x9ddeace8,0x08a9107e),L_(0x909eff4c)}, {LL(0x746a522f,0xf816fa9e),LL(0xefd70514,0x864324da),LL(0x598db9df,0x0094b64c),L_(0xc6634c12)}},
   {{LL(0xe65d8204,0xd6fb5d4b),LL(0xadb1a9c2,0xec38f10c),LL(0xc8ec0f39,0x1908657d),L_(0xa25710b3)}, {LL(0x87e1d90e,0x311947c3),LL(0xe187cb1c,0x04d9654b),LL(0x1bae3f87,0xafb6e7b5),L_(0x44ea2977)}},
   {{LL(0xe4a3b821,0xee18a382),LL(0x60b120f0,0x4002bc41),LL(0x91636563,0x29fd66f4),L_(0x6538f012)}, {LL(0x300af347,0xda52b57a),LL(0x9a0bd35b,0xa08ae1b7),LL(0xcee45c95,0xafb1fcae),L_(0x042232d0)}},
   {{LL(0xe06efe03,0x54397157),LL(0xd06ed8f6,0xf4ce3dfb),LL(0x67644c3e,0xaa8e6af8),L_(0xa26425e0)}, {LL(0xbbd40e3b,0x6fb8009d),LL(0xdfa5a71f,0xbf5753ee),LL(0xc4fb8d84,0x413c2883),L_(0xa71e26d8)}},
   {{LL(0xe358be61,0xae3de433),LL(0x45610301,0x9f32cd87),LL(0x4ec1365e,0xb95b637d),L_(0x41dbce62)}, {LL(0x0840486b,0x9637a66e),LL(0xc6d80952,0x522810a8),LL(0x928915a0,0x761cc88a),L_(0x1da0d501)}},
   {{LL(0x208620d1,0x37a59562),LL(0xb4ca54d1,0xc246b32e),LL(0x5128864c,0x8aad87e4),L_(0x559dacdc)}, {LL(0x9d93608a,0x89718ac7),LL(0xea353321,0x48d40286),LL(0x2012406b,0xc599f213),L_(0x1eb2fe94)}},
   {{LL(0x6807d450,0xba62fad2),LL(0xa7fededf,0x07ea62b0),LL(0xd58195f9,0xd3d6b5bf),L_(0x1ad1505f)}, {LL(0x87583b1c,0x6f93c77a),LL(0xaf96dc4e,0xa959c022),LL(0x20624dad,0x4fa52512),L_(0x3481aabe)}},
   {{LL(0x9310eef9,0xba1c1390),LL(0x9fb8e966,0xcce0400b),LL(0xb9c0dea0,0xd59d1993),L_(0xe8a99c53)}, {LL(0x243bcc60,0xb44f7477),LL(0x97335f28,0x1bd266f4),LL(0x6f550c82,0x3749bb24),L_(0x67501a21)}},
   {{LL(0x35f741c9,0x2ddbdad6),LL(0x5e95c8ee,0xbeac98ef),LL(0x74fa6bf8,0xf68aac5d),L_(0xe404df62)}, {LL(0x13d824a6,0xf3252763),LL(0x934f4963,0x4c5b99db),LL(0x65ea023c,0x39f3ad81),L_(0xf2ab45c6)}},
   {{LL(0x513ad1f3,0x6c9e60dd),LL(0xd82ebdb2,0xeb595e39),LL(0xf4278e99,0xf76b6d5d),L_(0x177bfa74)}, {LL(0xda5b699a,0x0698c8f7),LL(0x8af11d92,0x022b9e4d),LL(0xdcfeb87d,0xdf966b87),L_(0x790b8b3b)}},
   {{LL(0xe6576cdc,0x0d632c2d),LL(0x82b42706,0xe4079724),LL(0xeb7e23be,0x67d13334),L_(0x6bef8f5e)}, {LL(0xab6f7a84,0x5b9d7907),LL(0xb943b3e3,0x3b2e4de9),LL(0xe1142403,0xe61e180e),L_(0x3186f5ab)}},
   {{LL(0x54c38bc7,0xee73f3c8),LL(0x1c021526,0xcdd71fb7),LL(0xd5f0330f,0xbcd8ae19),L_(0x34f15f5c)}, {LL(0x3f1e795b,0x0f2d33f8),LL(0xf2452d93,0xe4d47a55),LL(0x0c4b0f70,0x322c677d),L_(0x311431b5)}},
   {{LL(0xe32f7152,0xd222f30a),LL(0x32c8e92e,0xd5265ce0),LL(0xf9051d2c,0x34079f7e),L_(0xca51b73e)}, {LL(0x752d0df7,0xe4529fd5),LL(0x3ee05e30,0xd39fd02a),LL(0x762211ba,0x957b54a8),L_(0x7bd842d1)}},
   {{LL(0xca5c5b9b,0x8578a05a),LL(0xf931c6ff,0x5854ceb6),LL(0x9c1a24ae,0xc7ad186b),L_(0x3aae7e80)}, {LL(0x405ad033,0xcd94ffe8),LL(0x02d60546,0xc24532db),LL(0x6d8dfc51,0xf70532cd),L_(0xa7523214)}},
   {{LL(0xf6e5bcb6,0x3898ac49),LL(0x7ef15aa2,0x0dc0b7cb),LL(0xc92d3e61,0x6f146470),L_(0x20a65e93)}, {LL(0xa99d1e34,0x40c2ff2e),LL(0x5f5f4b1f,0x8bce3569),LL(0xef24d37f,0xf5eb8c60),L_(0xd0f3625a)}},
   {{LL(0x770ff147,0x04af9657),LL(0x0ddb7f8a,0x5d6a9432),LL(0xf5679a0e,0xfa164f3a),L_(0x75f9e25f)}, {LL(0xb341ef73,0xd2b96f53),LL(0xc75b8bfb,0x111da06d),LL(0x141dae1e,0xf86c989c),L_(0x1a015ae9)}},
   {{LL(0xd89c49d6,0x7440165b),LL(0x252d3538,0xbc50c381),LL(0xc7935e6e,0xfc485299),L_(0xafc2b1d4)}, {LL(0x7b5b7847,0x8b5606a8),LL(0x8714fa4e,0xc8ca9e3a),LL(0xd015855f,0x03679b60),L_(0x14d9de29)}},
   {{LL(0x5d13fc8a,0x7fd6372a),LL(0x1fc76dfc,0xd3c20f3f),LL(0x835ac98b,0xec54c6b2),L_(0x98cbc171)}, {LL(0xe5e2a3e4,0x7ab5c98a),LL(0x577a8e9f,0x6c990ebd),LL(0x84028fa5,0xed0d2362),L_(0xb28e8d67)}},
   {{LL(0x277610eb,0xd8c85ac2),LL(0x8e20b175,0xc11f2394),LL(0xbc82df9f,0x763f7da0),L_(0xc32cc81d)}, {LL(0x85f8381c,0xb02c89cd),LL(0xd305dbab,0x2a33f683),LL(0x7f0a99f0,0xd40823f0),L_(0xf958bb39)}},
   {{LL(0x08d09d3c,0xee84ea33),LL(0xc5f043e5,0xe195e3e9),LL(0x13f7ae30,0x18438628),L_(0xb6ffaed5)}, {LL(0x4012f22a,0x9cb2856d),LL(0x9a5c926d,0x6dd2b787),LL(0xad05632b,0x4d11a1b7),L_(0xf0f2c69d)}},
   {{LL(0x8e9a6388,0x60b10c54),LL(0x79943f17,0x0474b959),LL(0x15056515,0x70dc2681),L_(0xc9a17b25)}, {LL(0xb6a005e4,0x4d9c29fe),LL(0x5978aada,0x9f751987),LL(0x6e84fa9d,0xb01cfa96),L_(0xd9553cbf)}},
   {{LL(0xe72b7959,0x14645ddf),LL(0x50569b84,0x6a35f872),LL(0x76814e1a,0x6931ce3c),L_(0xd1694dea)}, {LL(0x9877446f,0x8fb4ec92),LL(0xcbd84ece,0xd0d64612),LL(0x70c2d8a8,0x6f1a63f5),L_(0x2ca687fd)}},
   {{LL(0x73d59d85,0x66719258),LL(0x78b1b1f1,0x3528fe8e),LL(0x372aba7b,0xf72aecd5),L_(0x049424c7)}, {LL(0x0a8225e9,0x1299651f),LL(0x7a3d9b21,0xcdfa1798),LL(0xeea3b43d,0x6296bed4),L_(0x965623a1)}},
   {{LL(0xe79e4875,0x19faf24e),LL(0xee2a75e2,0xfce9de0f),LL(0x2608d8d7,0x5dbd9aa4),L_(0x8dd2d8bb)}, {LL(0xa1d5ac1f,0x96024e09),LL(0x59608369,0x88c11a15),LL(0x3109b985,0x4c72f34b),L_(0x073affe7)}},
   {{LL(0xa7d5999d,0x7163417c),LL(0xe75a8d8d,0x58690b9f),LL(0x34a71e84,0xf46ea0c1),L_(0xa472172f)}, {LL(0xdb5e6f96,0x5941f31e),LL(0x45ef8849,0xff03761f),LL(0x813628f8,0xff6e6be2),L_(0x2f6a6923)}},
   {{LL(0x37f746c4,0xf2dab7c0),LL(0xee4fac0c,0x94b2339d),LL(0x442b5427,0xd1572489),L_(0x1b63f7f9)}, {LL(0x77b93d36,0x86beafa0),LL(0x931a009d,0x89006e01),LL(0xc54e5dcf,0xce454abb),L_(0xe3d31b67)}},
   {{LL(0x5dbfaab4,0x2bfb9672),LL(0x00a46605,0x0f7940ce),LL(0xd0e900f4,0xac391030),L_(0x56b396f4)}, {LL(0xd65db7d6,0x111e6518),LL(0x0e375257,0x574c6063),LL(0x0569eb0d,0x6386031f),L_(0xa43939dc)}},
   {{LL(0xe79fa54d,0x790064ff),LL(0x19bae3a4,0x040699eb),LL(0x278b676c,0x2bfc1210),L_(0x5fd221ef)}, {LL(0xc41696b3,0xaddaabbc),LL(0x3e58e744,0x4ef37496),LL(0x86211a5e,0x27a20439),L_(0x06b96609)}},
   {{LL(0xa9a97859,0x31e7bef8),LL(0x6b521dbf,0x39d7de6d),LL(0xbb74a0f3,0x41f4bd94),L_(0x79699c57)}, {LL(0xfd6b25ba,0x6af2b090),LL(0xb9a903de,0x17aa75d4),LL(0x250d78d4,0x565e3c29),L_(0x674edb33)}},
   {{LL(0x580583bb,0xfa502a79),LL(0x4f9ab886,0x9fc3a870),LL(0xb5e6f604,0xcd18776f),L_(0xf25cc6e4)}, {LL(0xea80a2a0,0x9822aa1a),LL(0x2c1f0788,0x3a8b80d3),LL(0xb4be8a38,0x671f0157),L_(0x5b11123b)}},
   {{LL(0x9b6b4d48,0x54914216),LL(0x1256ef4e,0xb5f866f9),LL(0x075b53ab,0x5ecb1889),L_(0x56e20726)}, {LL(0x69455333,0x09f26254),LL(0xd571b91b,0x2b2fad5a),LL(0x328d62f6,0xde6fb9c6),L_(0xe065c610)}},
   {{LL(0xfd0dd09b,0x45144de7),LL(0xdc428059,0x74b10792),LL(0x9aff2cca,0x9af32379),L_(0x7429380e)}, {LL(0x49635e31,0xe3565603),LL(0xdea72af4,0xe2386988),LL(0x2bf6a43f,0x5a2add5c),L_(0x6b6ae6f7)}},
   {{LL(0x13448c75,0x9ab52f76),LL(0x061d7ef8,0x4392015e),LL(0x010840be,0xae0d756d),L_(0x97508e7b)}, {LL(0xd15d5084,0xadc78971),LL(0x4d5b80a4,0x7c713bfc),LL(0xf9aaef5c,0x256809bb),L_(0xa1ec2e50)}},
   {{LL(0xfbb42651,0x74a6dd49),LL(0x0393f719,0x977434ad),LL(0x4e57160b,0xac7e00ee),L_(0xe124fa0e)}, {LL(0x7a6f9567,0x2389bd87),LL(0x80b4129f,0xfb1db8c0),LL(0x0e8aaba3,0x1051debe),L_(0x4a190f97)}},
   {{LL(0x477f6b19,0x2156959b),LL(0x37e2ab37,0x910a6757),LL(0xfe9b7158,0x540652e0),L_(0xd3215a27)}, {LL(0x7976f02e,0xc0c597ef),LL(0xceb6dd56,0xe44ded06),LL(0x09436cb3,0x07e11a08),L_(0xc752116d)}},
   {{LL(0xf6b2fb7d,0xe154e3b5),LL(0x3dd689de,0xc3b95a13),LL(0xecec9673,0xf4ebc67c),L_(0xdf67ffb0)}, {LL(0x01c1c880,0x33ce85c6),LL(0x58ffaf1b,0x033751df),LL(0x741f05bf,0x2535c38a),L_(0xffe3294f)}},
   {{LL(0x308eb02c,0x4c921d98),LL(0x1c787299,0x5bd1a752),LL(0xbf10f0e6,0x4f5e9e39),L_(0xdd4daa07)}, {LL(0x18aefaaa,0x92fcbe07),LL(0xf29c8618,0x200b4b85),LL(0x84305705,0x8d7c9a43),L_(0x19694aa8)}},
   {{LL(0x3a51e7ce,0x680c4324),LL(0xb1ac0982,0x0b339790),LL(0xa4032aca,0xdf576261),L_(0x92007d10)}, {LL(0x50fc383f,0x755cfb3b),LL(0xb9ea853a,0x702d89aa),LL(0x28725f39,0xa32de41e),L_(0x8310cc56)}},
   {{LL(0xe3293ae8,0x4319f044),LL(0x6a31f14c,0xd8080331),LL(0xf0a68d54,0xa79121b3),L_(0xf8ad5c75)}, {LL(0x3c85f6ea,0x1aca6f0d),LL(0xd728ef75,0xb8a8d44f),LL(0x66ed37bf,0x244f6df5),L_(0x5253f731)}},
   {{LL(0x40e52539,0x7e5a8bd2),LL(0x9d03a5af,0x2a775548),LL(0xa4eb71ab,0xa305f15d),L_(0x1dc7fb76)}, {LL(0x68a7c637,0x20487a25),LL(0x6deaadda,0x8c18ae20),LL(0xe0f4999d,0x9bb84678),L_(0xc451bb12)}},
   {{LL(0x27b6061c,0xc977cb30),LL(0xd39b851a,0x430b3684),LL(0xcd656ca9,0xa19898ab),L_(0xafd81332)}, {LL(0xfa31c77d,0x2bb5e6b0),LL(0xc3157a16,0x60d12981),LL(0x8acd8d9d,0x69cbb158),L_(0x6d76f690)}},
   {{LL(0xe9d8d3f2,0xc8ff4785),LL(0x2bca8c99,0x2e87c4ea),LL(0xa2caaf78,0xf6be10a7),L_(0x2100372c)}, {LL(0x8cdcc69a,0x761e39c6),LL(0xdb53800d,0xd8745be4),LL(0xc599f6e5,0xc995d474),L_(0xcb2cc9b4)}},
   {{LL(0x35b78d83,0x39687ef6),LL(0xe8d0ac03,0x91a4e746),LL(0x921b3cbc,0xc64c35ba),L_(0xa500e776)}, {LL(0x4f3feb91,0xe79c45af),LL(0x60ad3863,0xdbd9485d),LL(0x0ce33aaf,0xaee36834),L_(0x4ce8962e)}},
   {{LL(0x8023ad1d,0xb38c8506),LL(0xc359f6e1,0x0673f87d),LL(0x31837b1d,0x98af1b3a),L_(0xf716fe6e)}, {LL(0x3024d432,0x3830cd5b),LL(0x52930fa5,0x60d8b1ad),LL(0x0ad62261,0x097dfa97),L_(0xebeb447d)}},
   {{LL(0xf447ab0f,0xe409d4ec),LL(0xe80a1b86,0x58d6899f),LL(0x1aaed1d3,0x9eac7e33),L_(0x80bb831f)}, {LL(0x2a9866ba,0x986f7e0b),LL(0x005a4dc6,0x43a73638),LL(0x52827ff0,0x293c91f5),L_(0xcdd9e93e)}},
   {{LL(0x691e28ae,0x2c6558ef),LL(0xd0ac15f2,0xe56ad3e9),LL(0xa247b120,0x573e9688),L_(0xd0627f7e)}, {LL(0x375dee4b,0x22887f9e),LL(0xd31e0ba7,0x7d20b431),LL(0x1eea7f55,0x4423561d),L_(0xf9581051)}},
   {{LL(0xd4243ea6,0x42a54363),LL(0xeb2affe7,0x619c06c6),LL(0x972ce84c,0xaeca0047),L_(0x4d9efd23)}, {LL(0x4e0c1d41,0x173fea08),LL(0x1d890ba5,0x4bd9d779),LL(0xfea0ad92,0x5f0b0f42),L_(0x5619484b)}},
   {{LL(0xe4e12bd4,0x6cdbcee2),LL(0xc11ff4f3,0xfe8d06f2),LL(0xa260adc3,0x5bb4d1a2),L_(0x33e0dbb6)}, {LL(0x27018d5f,0xcaa2ba14),LL(0x415b8922,0x74b9b719),LL(0xb1fbde1d,0xe7f63f33),L_(0xb014bfff)}},
   {{LL(0xeb046732,0xab510c97),LL(0xee7b323c,0x9bf44d3f),LL(0x37854732,0x23d6cc1a),L_(0xdbbcc348)}, {LL(0xf773172b,0x64d5f135),LL(0x6fb100cc,0xb0c07c01),LL(0xb27f28f5,0x980b1419),L_(0xca8834c2)}},
   {{LL(0x01f6b795,0x1b1c75ca),LL(0x8c28e4a6,0x6f2aae0d),LL(0x2cbdb244,0x1341e16e),L_(0x5a726ab8)}, {LL(0xd482dc14,0x32264865),LL(0x4f4a819e,0x1e1e75b0),LL(0xba655f22,0x37367c7d),L_(0xd157593e)}},
   {{LL(0xaa3ab9fd,0x46d94f54),LL(0xc9397bab,0x3628a867),LL(0x65e56a09,0xf946f4e0),L_(0xa259e975)}, {LL(0x372139d2,0xb5060783),LL(0xccc37d6a,0xeef39a01),LL(0xf709343a,0xfb056c2e),L_(0x89705d36)}},
   {{LL(0xb9d0b061,0xeb841e0e),LL(0x7663ad0e,0x0dc12b25),LL(0xb1420c31,0x5792fb1a),L_(0xaebf5ced)}, {LL(0xfb7eb601,0xeaecfdd5),LL(0xd5142858,0xc9d7159e),LL(0xe56ab53a,0xdbaff47d),L_(0xb56ba899)}},
   {{LL(0x64ac917b,0xd69321cf),LL(0xa06234ff,0xe8fd532c),LL(0x98b8cc13,0x4be3eee6),L_(0xadbfb76c)}, {LL(0xb07602f3,0x57f0cdf3),LL(0x01e45447,0xa4676560),LL(0xb307918b,0xa0a5d2a4),L_(0xf7e16814)}},
   {{LL(0x87a7be06,0x6c92c96b),LL(0x865c8f6c,0x666baece),LL(0xf96358e6,0xc0b20165),L_(0x10dbd508)}, {LL(0xc18e3e53,0x990a46d4),LL(0x1d1a7d70,0x0002a616),LL(0x6c2d572e,0xa87b8f93),L_(0x03a7ba98)}},
   {{LL(0x8d933af2,0x6d9f3e8c),LL(0x2221d899,0x3a4a28c9),LL(0x7c844ad4,0x142035bd),L_(0x11c04771)}, {LL(0x2de90904,0xf02e1ad1),LL(0x78ab7d85,0x9793442b),LL(0x301c5cee,0x1feac0e2),L_(0x95568955)}},
   {{LL(0x48a5ccac,0xc5b0ee88),LL(0xf045ebf3,0x1ecaeea9),LL(0xc1ea77c4,0x4e10f859),L_(0xa44993af)}, {LL(0xf6bfe532,0x38d373de),LL(0x5c460d85,0x9d4beaec),LL(0x2bdad16c,0x264105cb),L_(0xb8fc3b3e)}},
   {{LL(0xd715e784,0x2edb771c),LL(0xd4ae16cd,0x5451781f),LL(0xc8055a0a,0xdee3f3e2),L_(0xc08d1ca2)}, {LL(0x3a1c4bfb,0x4252a1b8),LL(0x6d687314,0xbca7011a),LL(0xba935b81,0x3b383652),L_(0x52c1b85d)}},
},
/* digit=12 base_pwr=2^84 */
{
   {{LL(0x793c7e62,0xb92b8aef),LL(0x6354d2c9,0xd358ac7c),LL(0x5a4d9e35,0x0381968a),L_(0x52d9f13a)}, {LL(0x36ed6955,0xc63b9fe6),LL(0x3743142f,0x726db145),LL(0x6e186e53,0xaea91efc),L_(0x3de25aa0)}},
   {{LL(0x4c1ff250,0xf8921757),LL(0xced4216d,0xb6db16c1),LL(0x28cf0cfe,0xd524c877),L_(0x03bdd0b5)}, {LL(0xdfd0f98a,0xd91831dd),LL(0xd131c59b,0x980f4d7d),LL(0x47533185,0xf492c767),L_(0x63c30af1)}},
   {{LL(0x0cc8e4dd,0x79a5ddcb),LL(0x495cdcc7,0x26e26606),LL(0x7f199f53,0x9d2ff41e),L_(0xbe613bf2)}, {LL(0x36bd8348,0x967f0753),LL(0x76c36728,0xc4ce6462),LL(0x02af3c45,0xee5a166b),L_(0x0bda3025)}},
   {{LL(0xff47da80,0x63718529),LL(0x473e976c,0xa699d496),LL(0x88aeb436,0x368ce839),L_(0x94aabe19)}, {LL(0x0bf4b797,0xb19a3ec6),LL(0x7d50b7ff,0xcd07cc98),LL(0x81c2bf46,0x8c4eb78c),L_(0xb0a7cd1f)}},
   {{LL(0x2ddf8350,0x56f0849c),LL(0x5739e59e,0x4be4bf01),LL(0x1fc6493a,0x8b91a3a3),L_(0xd401b24b)}, {LL(0xe89ec9f3,0x9decd9c5),LL(0x17f178e9,0xe63f9369),LL(0xfa904473,0x7a09efc0),L_(0xca5c00a9)}},
   {{LL(0xe3f640d4,0x6aca6f60),LL(0x4bda0113,0x574789e1),LL(0x12ccd894,0xb4db3fdf),L_(0xb39be04f)}, {LL(0xf1e4603b,0xb4705565),LL(0xccbe5154,0xef713fb6),LL(0xd84fec44,0x11964906),L_(0x9ceede3f)}},
   {{LL(0x0244c62d,0x13111bdb),LL(0x175d3f77,0x14dda4d9),LL(0xe0971342,0x21244b52),L_(0xd6fa5642)}, {LL(0x2ee8560d,0x68da34f1),LL(0xb9933942,0x387164b3),LL(0xd7d7f791,0xee292b1d),L_(0x4382d48d)}},
   {{LL(0x2a7dfaac,0xe877a208),LL(0xaf9b8432,0x77afbba8),LL(0x1257224a,0xbb09b923),L_(0x1501cefc)}, {LL(0x38985dbe,0xe061421f),LL(0xaa863b6a,0x00e2d3d6),LL(0x2733782a,0x1a9bf0b7),L_(0x6912b5aa)}},
   {{LL(0x70c5c0c6,0x38d01224),LL(0xa0259e70,0x84a934e4),LL(0x8e85d7ef,0xfd49df96),L_(0x6bb4a230)}, {LL(0xa15b7d76,0x58b99e52),LL(0x3c2be4ba,0x2dc32c4c),LL(0xe49fcb40,0x4937a76d),L_(0xdc9b73dc)}},
   {{LL(0x5ce8f858,0xa516c9ed),LL(0x41610138,0xd3f7221c),LL(0x71d5b83a,0x8795351f),L_(0x9a45e682)}, {LL(0xef8c4cff,0x9edacc8d),LL(0x17bd9c0b,0x9166ced1),LL(0x871cc69c,0x57013f03),L_(0xb6b7b6e2)}},
   {{LL(0xf3f01792,0x4855e4b1),LL(0x124c67e8,0x7556de49),LL(0xd4d7f50b,0x494ae118),L_(0xf16b9834)}, {LL(0xab9ad587,0x3a6a8d30),LL(0x18d69af9,0xa950fac0),LL(0x7f8ed091,0x9daffa88),L_(0xba6e7c59)}},
   {{LL(0x8f831a7c,0x011788f5),LL(0x9c64aa0c,0xf480aaea),LL(0xf896522c,0xcc8f1ea7),L_(0x772627b7)}, {LL(0xe1227c86,0x72bb812a),LL(0x57f8a759,0x6e202e06),LL(0xc7e7a397,0x4c06faeb),L_(0xeea7a14b)}},
   {{LL(0x46853977,0x0c456e84),LL(0xbacf3d29,0x94a82d15),LL(0x861ef909,0x17a88c53),L_(0x25d01a4d)}, {LL(0xaa44d1ac,0x0b6e87fe),LL(0x066301cc,0xce900509),LL(0x51763eac,0xe4c44c3a),L_(0x243f5364)}},
   {{LL(0xf02ccf3e,0x69fe2b3e),LL(0x623f7c06,0xa6634ac0),LL(0xa81b9914,0xba6dd8e5),L_(0xd33612e7)}, {LL(0xcf35703c,0xc948497c),LL(0x7184b686,0x3725232b),LL(0xd5e74279,0xfa105e4b),L_(0x4bfe5bea)}},
   {{LL(0xbbed4591,0x03d46274),LL(0xfcb464d3,0xc3c74b90),LL(0xdcb4ccb3,0xf6934477),L_(0xbdf563a1)}, {LL(0xdae24867,0x7c07acc7),LL(0x4d4331eb,0x05fd6424),LL(0x768b2706,0x022d151f),L_(0xf65a4027)}},
   {{LL(0xfff51bbb,0x5cbe2653),LL(0x32707706,0xbec1f2c5),LL(0x9739588e,0x1d1ff202),L_(0xe51c696a)}, {LL(0xfa60c3f4,0xdd740417),LL(0x25fcc7e6,0x6a0e351e),LL(0xac2dfbc7,0x0d72b69e),L_(0x1ff6f6ad)}},
   {{LL(0x3eaa7b81,0x3ade75e7),LL(0x3ac64888,0x4036d025),LL(0xd957b110,0x7be01ff5),L_(0x33f835d6)}, {LL(0xc6af558a,0x45081b72),LL(0x0c08887d,0x137ffdce),LL(0x4fc01e35,0x84d416f7),L_(0x15716e5d)}},
   {{LL(0xb47ca6c4,0xc4cafbc2),LL(0xcd118e18,0xac25eb7f),LL(0x78d7034b,0x4d198a7c),L_(0x9ba63f25)}, {LL(0xdb8a8725,0x81c83f82),LL(0x9eb5e40b,0xdd6d729d),LL(0x3b1941e5,0x27b87b90),L_(0xb7229f73)}},
   {{LL(0x492e405b,0x59afdea3),LL(0xf6e5451f,0x74cd3ea2),LL(0x3c7977fd,0x8262d80b),L_(0x0ff79532)}, {LL(0xd5eea209,0x85d46815),LL(0x41a59faa,0xe5277e25),LL(0x815bfdff,0xf25c4d84),L_(0x26a648ca)}},
   {{LL(0x1ce66149,0xc47be93e),LL(0x8807158f,0x150ece78),LL(0x71da23a8,0xf790aca9),L_(0x26957d5b)}, {LL(0xeaf83d16,0x571af637),LL(0x3a3ff68a,0xbbf0d967),LL(0xb02eef0e,0x78555759),L_(0x1609b893)}},
   {{LL(0x38ec85bb,0xa99f1706),LL(0x7a4aae59,0x12f3c66b),LL(0xc84493c2,0xf1b134ce),L_(0x8f2fb8ff)}, {LL(0xbfc251dd,0x8f6f7a7f),LL(0x73cd2591,0x7b6bc242),LL(0xf68459a2,0xf4b9ddab),L_(0x1d8a7278)}},
   {{LL(0x995f94c3,0xa1fd53d1),LL(0x3ad117de,0xae71c050),LL(0x75e31939,0x58a5dfbd),L_(0x3251c208)}, {LL(0xda1a36d0,0x8436c37c),LL(0x65ab378a,0x80258188),LL(0x3a685733,0xe6172e78),L_(0xfa11b001)}},
   {{LL(0x54abde7c,0x18e18996),LL(0x48c2dc2c,0x3b62d97b),LL(0x814d21da,0x198afe84),L_(0xf76ecf91)}, {LL(0x0ca9c69a,0xb918572e),LL(0xf53020dd,0x249d565a),LL(0x365cb3e0,0xa6a9ebf4),L_(0xf8ad1ad6)}},
   {{LL(0xb5e96d0f,0xf1859230),LL(0xcb1be872,0x64eca0e1),LL(0xfaa10d6a,0x11273c1d),L_(0x240336fa)}, {LL(0xe56d43c3,0x6bfe9508),LL(0x88653660,0x12a7c212),LL(0x7b0d301c,0xd427be59),L_(0xbd85a9ef)}},
   {{LL(0xbc70a5f7,0x72600cd7),LL(0xe9cae1c4,0xee97ffd5),LL(0x3d1457c8,0x96865fe9),L_(0x9a1bc1e7)}, {LL(0x1518b86c,0x7727a4f5),LL(0x09a2a9db,0x1ed0a203),LL(0x028d0a99,0x9656e0c0),L_(0x7f3b01db)}},
   {{LL(0xcb2d1081,0xb1f41608),LL(0x060c3752,0xf5804976),LL(0x6efb7250,0x0bd8bd31),L_(0x79016736)}, {LL(0xc3ebc844,0x8739f62c),LL(0xf14247f9,0xb2896f39),LL(0x29b78fc9,0x5b2e893a),L_(0x5f2b1e7f)}},
   {{LL(0x058470b5,0x380ddadd),LL(0x014c7c14,0xddf90df8),LL(0xc2d7837b,0x78897ca3),L_(0x5567dd34)}, {LL(0xe3308b08,0x00642d55),LL(0xf735f752,0x9ca8b38a),LL(0x8afd8b87,0x6eb02b4e),L_(0x120f5319)}},
   {{LL(0x2f0ecf3f,0x8fc31698),LL(0x41d511e3,0x8873639c),LL(0xbdc4fb65,0x9841bbd5),L_(0x207bfaea)}, {LL(0x8e82da75,0x38fcb8d9),LL(0x6ba5dcee,0x05619b1f),LL(0x21f2d336,0xf78fd1c0),L_(0xc6f634fd)}},
   {{LL(0x8258ddc6,0x09a676ca),LL(0x414f2daa,0x23041ee7),LL(0x4cbc9d7b,0x3b973302),L_(0xb409e95d)}, {LL(0x7027e9ce,0xd16a5ca8),LL(0xf92a6554,0x473c41d1),LL(0x153e5025,0xa08470dd),L_(0xae4dfb2b)}},
   {{LL(0xb57dcbad,0x5b3c44c3),LL(0x45031642,0x1130bf58),LL(0x335c400b,0xd4b606ca),L_(0x6b34a475)}, {LL(0xbd2244f8,0xf3ab94fe),LL(0x746295e2,0x68723d0a),LL(0x4b9b612a,0x9d957566),L_(0x510e67f8)}},
   {{LL(0x9a9a5ab2,0xdc56f264),LL(0x9f7dfe87,0x8565f7b7),LL(0xe3386448,0x104ea5b5),L_(0xac4c4feb)}, {LL(0x5964e7d7,0xb2f8650c),LL(0x186e3514,0xf781aadb),LL(0x58c2701b,0x81927ca6),L_(0x81220e7e)}},
   {{LL(0x6cffb688,0x727a95e9),LL(0xfb0e9702,0x83b28617),LL(0xcca5f38c,0xdb71dd75),L_(0xad31c5dd)}, {LL(0xf33897ae,0x3ff02651),LL(0x683b0e57,0x380268df),LL(0xc6e32a5b,0x4ab25a51),L_(0x283f183b)}},
   {{LL(0x48906979,0x08ea3767),LL(0xf7f76eae,0x744d1934),LL(0x222dd487,0x5b8927a0),L_(0xdc4a955f)}, {LL(0xf36b1f09,0x605b6b77),LL(0xc80579f6,0xc4a4588d),LL(0x4cf36f45,0x94402fad),L_(0x0a226060)}},
   {{LL(0xe0e79649,0x77db1b5d),LL(0x4fe36153,0xd0b5aff6),LL(0xb32ea8ef,0xa09373a7),L_(0xd4f68719)}, {LL(0x116381ec,0xd8828433),LL(0x8c0026e9,0x7e2c0fab),LL(0x16af4c4a,0x8f7a154b),L_(0xd4672394)}},
   {{LL(0x32df0ada,0xe432397e),LL(0xd0196172,0xd07b25cb),LL(0x114f01ee,0x12143827),L_(0x2f06e8aa)}, {LL(0x9a3c110f,0x00bcfcbc),LL(0x76ee873b,0xb9f30c05),LL(0x8ff02da4,0x786da906),L_(0x7e7e6c65)}},
   {{LL(0x0cb411eb,0xc8866c5b),LL(0x0d7b5118,0x5022b82e),LL(0xef5be83e,0xb37c41ce),L_(0x4b76c64d)}, {LL(0x17ee0e9a,0x3d28da11),LL(0x8ef7cb6d,0xdd638761),LL(0xa3c8a2d5,0x4369465e),L_(0xc8c85ab8)}},
   {{LL(0x5cda4d54,0x08818978),LL(0x17f1f556,0x62350317),LL(0x251cc26a,0x686c57b8),L_(0x865fb805)}, {LL(0xe9b31ebe,0x19c7f60e),LL(0x23566eda,0x56fbedbe),LL(0x47f95f9f,0x867b619f),L_(0x1958dbaa)}},
   {{LL(0x6427adad,0xfc7b0692),LL(0x44b01395,0x899725d1),LL(0x0d676d87,0x075607b2),L_(0xe2ae6225)}, {LL(0xf9c3cc31,0x56d68f4a),LL(0x7e636075,0xb883986d),LL(0x1ffba2be,0x5c1ff02f),L_(0xc237448d)}},
   {{LL(0x06c3092f,0xc3ea5575),LL(0x52c5f87e,0x2926437b),LL(0xf135c025,0x593535ee),L_(0xe0965b4b)}, {LL(0x33672517,0x54fedc17),LL(0x28d4b11a,0x39103346),LL(0xe5297d9a,0xb0d741c2),L_(0x39caed90)}},
   {{LL(0xf87698f2,0x340fdc0a),LL(0x1b8201fc,0x4ba2f34c),LL(0x248c333c,0x5f236490),L_(0x12ee4946)}, {LL(0x52913c22,0xcbc6b04f),LL(0xee23385a,0xe9ac2163),LL(0x630f7175,0x0000016a),L_(0x3f6e71ce)}},
   {{LL(0x953882bf,0xc6c3de30),LL(0xf99b211a,0x34fd7b9c),LL(0x549ffc42,0x5ffb8aa4),L_(0xbc4e1e52)}, {LL(0x8cc9069f,0xe552f859),LL(0xf54af34b,0xe153c954),LL(0xd61618d2,0x5bb20166),L_(0x5776aea6)}},
   {{LL(0x546f1f93,0x535ea11d),LL(0x565a5471,0x3aceac15),LL(0x4200ddef,0x8ee2aacd),L_(0x2493fe70)}, {LL(0xd0228894,0x048f6036),LL(0xc2e41220,0x37388c77),LL(0x3193a2ea,0x3c56b2e4),L_(0xc4609ea4)}},
   {{LL(0xcae3c391,0x4276f085),LL(0x57d5dc17,0x5fbdd403),LL(0xa8a6408a,0xd70d74f7),L_(0x98acac12)}, {LL(0x75cbb8e7,0x93cd39a9),LL(0xb4c10685,0xe460014e),LL(0x56a7276e,0x2685e5f9),L_(0x3eeed659)}},
   {{LL(0x4c4c8866,0x2d701e0d),LL(0x22e61da3,0x73a590ba),LL(0x45111480,0x403fd833),L_(0x6080d968)}, {LL(0x41cddacd,0x2957083a),LL(0x38eb9ccb,0xed0801ce),LL(0x1be2ace4,0x726301f8),L_(0x002d8909)}},
   {{LL(0xdf6a5e6d,0xd1f92359),LL(0xf591d929,0x221a78d5),LL(0xdc207a6c,0x5e691f61),L_(0x20b8f8e8)}, {LL(0xa52ee137,0xaeb5ac09),LL(0x7eacabb9,0x95e85345),LL(0x0985df07,0x04b8e782),L_(0xf8047be0)}},
   {{LL(0x2128e372,0x84928309),LL(0x4f347f1e,0x62f5aa58),LL(0x7d0fc140,0xf365c6c6),L_(0xa0e662a2)}, {LL(0x7f46460f,0x8d246f44),LL(0x9dcc1860,0x05e6f03d),LL(0x5bfa3ba3,0xd9a27ded),L_(0xb281e327)}},
   {{LL(0x5bf0784f,0x6bab642b),LL(0xbff128cd,0x0d3bef83),LL(0x268f3894,0xfc73c613),L_(0xcd24728c)}, {LL(0xdefd474c,0xd40e3e59),LL(0xf065f5d7,0xb7f90c59),LL(0x15d0cf49,0x4ad712bb),L_(0x70e00818)}},
   {{LL(0xd2fe399c,0xf1c993ca),LL(0xa2f0722e,0x4e4c53ef),LL(0xe7c142ac,0x8af34ce1),L_(0xce47aba3)}, {LL(0x7f0fff87,0xb6c1f1fe),LL(0x49dac9d9,0x0f23c16d),LL(0x1baa4d4d,0xdbfb3b98),L_(0xd927e1a8)}},
   {{LL(0xe7a09b61,0xde4413aa),LL(0xad27b0b5,0xbaf8e054),LL(0x0077242e,0x5d3ac419),L_(0xe50d3185)}, {LL(0xb407f82a,0x6b5c4ee9),LL(0x4a8b1fad,0x6d04cfb4),LL(0x30ac4e43,0xe0253dce),L_(0xf07461c4)}},
   {{LL(0x5a25f143,0x01947898),LL(0x818e99b8,0x35e2c880),LL(0x05ed8f7c,0x09f78f1a),L_(0x2fb78a6e)}, {LL(0xc9d82b2c,0x4c2f0ccd),LL(0xfdd48d7b,0xaf9e328c),LL(0xe380dbfc,0x9600539c),L_(0xbb3a3bc6)}},
   {{LL(0x54969bd7,0xea2bbe07),LL(0x394029b6,0x262c8bfa),LL(0xc8ab0425,0x90339a91),L_(0xda368bdd)}, {LL(0xd540c3e3,0x0fe39b20),LL(0xc20f1366,0x5dbdfc74),LL(0x8075ba9a,0xdc668935),L_(0xa20bdd31)}},
   {{LL(0x457e42b6,0x513bd423),LL(0x02a9ea32,0x37ec475b),LL(0xe477baa7,0x52b0bc16),L_(0x4583bc9a)}, {LL(0xcce869a0,0x287b6484),LL(0xf11b84b0,0xe1aa5cec),LL(0x7925b118,0x4ab367b2),L_(0x8ff3e481)}},
   {{LL(0xe318b3c1,0xb6f137ff),LL(0x0ffc4673,0x73596e44),LL(0xf9b0dfda,0x2bd4103b),L_(0x56acfc3b)}, {LL(0xd8a47cb2,0xa87de433),LL(0x046c39cd,0x69413a97),LL(0xdebd1720,0x11fa933d),L_(0x3f6c6fec)}},
   {{LL(0x47ed0eca,0xae4601fc),LL(0x06034734,0x110171a4),LL(0x104d85dd,0xca3c77a5),L_(0x8bca0d2e)}, {LL(0xaee2efa8,0xc78c356e),LL(0x70ae3402,0xcedf0097),LL(0x322e72ad,0x07c1fe3f),L_(0xd0253c44)}},
   {{LL(0x26e6bed6,0xc2f400a6),LL(0x2291a21d,0x7258c335),LL(0x287eac13,0x1680907d),L_(0x03a00f86)}, {LL(0x4aa4df86,0xbadddbc2),LL(0xc5e15b28,0xc0fcaacd),LL(0xf30664e1,0xb1606671),L_(0xc87b6603)}},
   {{LL(0x81f0973b,0x1873a8e4),LL(0x5f55c7e2,0xecb9e193),LL(0x33f083bf,0x95c9fed8),L_(0xe416fc1c)}, {LL(0x13f90aa7,0x02778f81),LL(0x07eef5cc,0xc73195fa),LL(0xa94149e0,0xc8c44cfd),L_(0x2d270d3f)}},
   {{LL(0x1a3813b8,0x7ca9f957),LL(0xdb514ec8,0xb674cd7d),LL(0x9718dd0a,0xe2bf5a99),L_(0x325fe3bd)}, {LL(0x25692bfa,0x4f0b0620),LL(0x78990d74,0x5191688d),LL(0x5e70b57e,0xdfbda68c),L_(0x6e9d11e2)}},
   {{LL(0x134142aa,0xd61361ba),LL(0xf97cf589,0xb62b767a),LL(0x7b21af77,0xf64ebc1a),L_(0x03a72d66)}, {LL(0x6883434c,0x4b021c90),LL(0x39df1cca,0x2fe04352),LL(0xb84c6c06,0x9b6a3082),L_(0x62692bf5)}},
   {{LL(0x33f0c495,0x94bf9bf4),LL(0x79c2f57b,0x85cdb0a8),LL(0xe0d00e0a,0x49f82bc9),L_(0xb9c9a088)}, {LL(0xce773e24,0x1f589bd0),LL(0x11cd1677,0x1892643b),LL(0x8f952d00,0xb5eec978),L_(0x016b826e)}},
   {{LL(0x87c07fe6,0x8cd03f0a),LL(0xf4d9e042,0xa3443289),LL(0xb07e4630,0x361c9252),L_(0x5bbca5ce)}, {LL(0xb600853c,0xe687c31f),LL(0x7bd9a485,0x90550b79),LL(0xe200e18a,0x0e254cdc),L_(0x50dce2e6)}},
   {{LL(0x3c92fc46,0x4b5e8398),LL(0x284e641b,0xf7347214),LL(0x0556fa64,0xa75c4fae),L_(0x4cc4b972)}, {LL(0xaebb3e9c,0x90fa7977),LL(0x6a40b04c,0xa133ea2d),LL(0x97401c28,0x613bfe61),L_(0x443845a0)}},
   {{LL(0xe2f9a715,0x532b2d67),LL(0x482b7779,0x065c4d2c),LL(0x123aa16e,0x6886e7e9),L_(0x9147c317)}, {LL(0x40c6dfd4,0x1530f2ae),LL(0xaed04ac0,0xd86577c2),LL(0xa4111d16,0xd86157ff),L_(0x7406594c)}},
   {{LL(0x23598a12,0xafae6c6e),LL(0xed33d4b3,0xa2858c95),LL(0x008ec0c9,0xfe186e4e),L_(0x7e4ed374)}, {LL(0x9cb08278,0x2534f6e7),LL(0x798a9b30,0xe5f0b67d),LL(0xfcf997cc,0xb58b2b67),L_(0x2f620d55)}},
   {{LL(0x3d465f68,0xe8013df7),LL(0x5dcfcfb0,0x3e4bfa75),LL(0x7a765b73,0xe0fbd7bb),L_(0x4e7374be)}, {LL(0x186b6aed,0xfe9b1e77),LL(0x375f54f1,0xf7011562),LL(0x55dc463e,0x202fbb6d),L_(0x8d697af5)}},
},
/* digit=13 base_pwr=2^91 */
{
   {{LL(0xe165ca1e,0xbcdbe444),LL(0xcb4a833d,0x76643977),LL(0xb07c1b51,0xef7d725e),L_(0xdc3b82a8)}, {LL(0x21f29146,0xaa5a7ac8),LL(0x01e0c54e,0xa51481d9),LL(0x2b2d5197,0xd220aca6),L_(0x736d91c9)}},
   {{LL(0x87d54b6b,0x74d6d0ba),LL(0x9af89e67,0x723937b3),LL(0xada42e0e,0xda49ba83),L_(0x83b9c4ec)}, {LL(0xa3055000,0x12dfdcad),LL(0x7e0a2cb4,0xa98d5243),LL(0xf7e3a9a3,0x6ed0d332),L_(0x1f8c692d)}},
   {{LL(0x6e661a1d,0x634cef8a),LL(0xff5ab01d,0x35d65531),LL(0x6fe3ecd9,0xc1015db1),L_(0xfcc7ca4c)}, {LL(0x769b7e20,0x795d4926),LL(0xed944d52,0x6db9a053),LL(0xf5f4129a,0x8f02bb95),L_(0x4007343b)}},
   {{LL(0x75cd1b48,0xab70364f),LL(0x6b243c97,0xf0408ddf),LL(0x8b9a0873,0x0ef0ffd5),L_(0xb6e8c899)}, {LL(0xbe3a87dc,0x2e1c11d8),LL(0x154379bf,0x6ed01611),LL(0x4382cb75,0xc876e0d3),L_(0xc42ec7d1)}},
   {{LL(0x0e4e37cf,0x1ce64f93),LL(0xa2eb2c00,0x1061f84c),LL(0xe1a22446,0x450ed02e),L_(0x4205e69b)}, {LL(0x644aa51b,0xc76dda32),LL(0x62db01d8,0xb244a09f),LL(0x633bdf3e,0x95c86470),L_(0xf9a6b309)}},
   {{LL(0xab41ec83,0x71aba88e),LL(0x590711c2,0x3366c35b),LL(0x0b6ab428,0x87bfe939),L_(0x1d7e1090)}, {LL(0x1d44d3e7,0x7ebcdd74),LL(0x2a937fd4,0x42a83b75),LL(0x3da3197e,0xc4155a57),L_(0xf1cea938)}},
   {{LL(0xb74ca945,0x6a4807e3),LL(0xc0c52eb7,0x474fa11d),LL(0x12ba26fc,0xda7a7914),L_(0x001d4756)}, {LL(0x34ddd360,0xce833467),LL(0x3290dbb2,0x0e517e33),LL(0x2b0a342a,0x579c91eb),L_(0xc27c2657)}},
   {{LL(0xb129c61a,0x6983b44b),LL(0xf27d3e69,0x463c956f),LL(0xe4a043b6,0x5a97c010),L_(0xc8f8a3c7)}, {LL(0x6cfa78d3,0xb9a3137c),LL(0x5655ed33,0xfc8d858e),LL(0x0b6dca4a,0xc1a3c9bb),L_(0x630b3026)}},
   {{LL(0xe0a6194c,0x082a09e7),LL(0xffe89fc6,0x8699cb3c),LL(0x9f4f85f6,0x0b13732a),L_(0xba6e1981)}, {LL(0xa2917bd0,0x26eb2b17),LL(0xa4afe844,0x0b4bc106),LL(0x91a5d7b8,0xeeac1c21),L_(0xe71f5fe0)}},
   {{LL(0x2f9e0c0c,0x3f0bf1e3),LL(0x031e908e,0xbd33fa00),LL(0xbc16afc6,0x6e2d4301),L_(0x90a23e58)}, {LL(0x99df250f,0x2549fd20),LL(0x52f4ef15,0xa724a018),LL(0x5b4c8dd8,0x1a5fbbc3),L_(0x7457c84e)}},
   {{LL(0xbdbe5cc7,0xa3742f59),LL(0xe0a87786,0x6287ad4f),LL(0xe7aa9e66,0xba62daa3),L_(0x65327959)}, {LL(0xa06c023b,0x7da31b20),LL(0x7fbf3e8d,0x94bea960),LL(0xe1a7865a,0x877c379e),L_(0xea576b89)}},
   {{LL(0xca000d60,0xaa616a39),LL(0x662a3f35,0xcacdc37a),LL(0xdd7d18e3,0xee30f511),L_(0xf25f9c46)}, {LL(0x7bee7ee6,0x3e51e652),LL(0x81e804d2,0xdd692c2b),LL(0xb0aaaab1,0xa52da544),L_(0x89ee1376)}},
   {{LL(0x3e7d3b9f,0x97063082),LL(0xf2d42cdd,0x846e4346),LL(0x9cb9423c,0x6ed009de),L_(0x746461d4)}, {LL(0xd30da2a3,0x6b0d52f7),LL(0xe78fe609,0x0b06f17a),LL(0x6887a9cd,0x8730c3cd),L_(0x0f586a94)}},
   {{LL(0x91370dc2,0xf21a3a39),LL(0x38aa8cf8,0x4835ded5),LL(0x3dd4e3e5,0xf1f81f18),L_(0xee849585)}, {LL(0x8aa2dfcc,0xb30df028),LL(0x1e4514ba,0x24f52ca4),LL(0x757e6772,0x3b2305c3),L_(0xa3525eaa)}},
   {{LL(0x0e45f1c6,0xb94b1aa1),LL(0xebe6c1cc,0xa3e2b74f),LL(0x1c4d6e37,0xfa33270b),L_(0x5d3856a1)}, {LL(0x62ec6306,0x77b4e047),LL(0x63f53943,0x36b59c35),LL(0x7466b59e,0xaea29806),L_(0xb1050d57)}},
   {{LL(0x4fc503ac,0x0fd330ac),LL(0x0f394d0b,0xd80c9e62),LL(0xe4d013eb,0x99b62540),L_(0xaa1c17a9)}, {LL(0x0a2e0bc8,0x8f6f2742),LL(0x7366a618,0x3a715ba2),LL(0x88e9ae3e,0x8818aa14),L_(0xd2fe924a)}},
   {{LL(0x1cf5fda8,0xfd68c56e),LL(0xd02cd00e,0x77ce9ac0),LL(0x5f260a21,0x9c693311),L_(0x7c1f3266)}, {LL(0x878fca42,0x5b2551ab),LL(0x9f545dde,0x365af76c),LL(0x19f87afe,0x019bf350),L_(0x58397798)}},
   {{LL(0x2ced9c6c,0x46ea826a),LL(0xc4da4263,0xe4112274),LL(0x43744264,0xbfc3581d),L_(0xcc933457)}, {LL(0xb80bff4c,0x9cbecb36),LL(0xe0071ca1,0xb9509e61),LL(0xcf0cfc02,0x67a2c9e1),L_(0x2f9bd24a)}},
   {{LL(0x48a08314,0xa57e540e),LL(0xd472cc15,0xd2110f43),LL(0x0b60e719,0xcd92d76f),L_(0xca1c9c30)}, {LL(0x44377160,0x8103a772),LL(0xa305744b,0xf08411da),LL(0x54b38b68,0xd6e1cdb8),L_(0xb252c82f)}},
   {{LL(0xf1e98090,0x935833d6),LL(0xd0e4995f,0x530418e5),LL(0x87f60eef,0x933733ca),L_(0x379092ba)}, {LL(0x423ea862,0xd4945af5),LL(0x60eeff1a,0x165baec1),LL(0x1d6d1df0,0x4bee290d),L_(0xe87f42f5)}},
   {{LL(0xcf66861f,0xb222f293),LL(0x080ba4e8,0xb45f15c2),LL(0x8ed3f3c1,0xfdee70ba),L_(0x58ce1c82)}, {LL(0x46cc54a7,0x8490a252),LL(0x6ed1e744,0x7d319886),LL(0x0602a509,0x585e149e),L_(0x7fa41773)}},
   {{LL(0xcd741439,0xf0a3e291),LL(0x74dc9c31,0x41b8528c),LL(0xc78baacd,0x01ba1bb0),L_(0xe5b73558)}, {LL(0x1ce4898f,0x9cf50575),LL(0x5421bd61,0x5a0b632c),LL(0x241cb4c7,0xc421cfb2),L_(0x9ba297a8)}},
   {{LL(0x9bac4e41,0xf0b66ab7),LL(0xd87b31c9,0x9f0067bf),LL(0xcf835fb0,0x2f9c67c3),L_(0xaee5e6cf)}, {LL(0x228157ee,0xd04298dd),LL(0x0411e445,0xeac052a3),LL(0xbec90b82,0xe6adb2bb),L_(0xb30b9c0e)}},
   {{LL(0xdcd6bf3b,0xda605905),LL(0x5f01a30b,0x0b1bb0c1),LL(0x7c053c41,0x40f7d79d),L_(0xc39feb03)}, {LL(0x98a4669a,0x3d73c359),LL(0xe472e81d,0x4a94f93e),LL(0xdea2bf34,0xed66fd6f),L_(0xddc57518)}},
   {{LL(0x904c89ad,0x7095dc91),LL(0x07f66d27,0x2423fe30),LL(0x9c4ec652,0xfd209ae0),L_(0x35803afd)}, {LL(0x94522cf2,0x51ed5eae),LL(0xc9356091,0xb3740f2a),LL(0x19e52bee,0x0676dac6),L_(0xd87e6c27)}},
   {{LL(0x4ef0a025,0x83877c70),LL(0x39e61abc,0x09e709b4),LL(0xa88e05f1,0x7885ad28),L_(0x53bfba53)}, {LL(0x9b73b3c4,0x8e433b9e),LL(0xd6a78477,0x7394d932),LL(0xcefdc6a1,0x3ce9d748),L_(0x27d9ed71)}},
   {{LL(0xda4320d3,0xe57aeaa8),LL(0x7dbbbc39,0x06fa3110),LL(0x5c942e4e,0x7d7aed19),L_(0xe660d0ba)}, {LL(0xadbc0281,0xff40dd89),LL(0xcdf314b5,0xb1ee8372),LL(0xa625d183,0x9d4344c2),L_(0x54f2035c)}},
   {{LL(0x73c8d3ee,0xd120f88d),LL(0xe3275570,0x060ff4c7),LL(0xe344b66d,0x8d5e8c7f),L_(0x1cdb23ae)}, {LL(0x23a7070c,0xce526de2),LL(0xe64e6355,0x4b694305),LL(0x5edf30dc,0x7e00559a),L_(0x3501ee2b)}},
   {{LL(0x93c5d80a,0xa6698b0c),LL(0x91c3e912,0xd21b90b5),LL(0xbb24391d,0xe1e797f6),L_(0x0526a13b)}, {LL(0xe2efd168,0x41e4a8d3),LL(0x63a82271,0x92a0c47e),LL(0xf6fd3e09,0xd1ade148),L_(0x85041faa)}},
   {{LL(0xd58b2165,0x7675921a),LL(0xb30c5e67,0xa292beda),LL(0x9a004497,0xb5aac131),L_(0xcaf13420)}, {LL(0xa8c50521,0xced86c69),LL(0xc60fb50a,0x9774900b),LL(0x8cd4dd3d,0xdec98657),L_(0x5092d4ea)}},
   {{LL(0x829ecf62,0x9fb10f97),LL(0x67b904af,0xef94ed3f),LL(0xeebc687c,0xdae52182),L_(0x13aadd13)}, {LL(0x964738db,0x59a01f8a),LL(0xc7db8080,0x608e9fae),LL(0x5433ce08,0x8b26a71a),L_(0x0c3a71bb)}},
   {{LL(0x09326b40,0xd29e534e),LL(0x215aa2f4,0x727f40f5),LL(0x5f00b26c,0x3c54a3cb),L_(0x71ec2585)}, {LL(0x14b424a2,0x4c553b92),LL(0xf86c23e2,0x06c97184),LL(0xdd6cfc3e,0x0fb40ab0),L_(0x4647a750)}},
   {{LL(0xb818ca3b,0xa51a2b16),LL(0x1bbce0ea,0xb716ceb6),LL(0x7e78ccf9,0xc6736591),L_(0xbaa1d2e3)}, {LL(0xe68eaa6c,0x6b8866a9),LL(0xc5b9e6fb,0xa4607970),LL(0xc39a4975,0x712336e6),L_(0x86d2dbfa)}},
   {{LL(0x37f2fc9e,0x106db861),LL(0xc495534c,0x4b840f41),LL(0x75c25359,0x05a076ec),L_(0xf2a1fa9b)}, {LL(0x22526757,0xbab703f5),LL(0x2b054aeb,0x8e748f87),LL(0x9e9ed0df,0x42646b39),L_(0xd95a6334)}},
   {{LL(0x22dfe0f7,0xbcf088fd),LL(0x373df018,0x58f99bf3),LL(0x2646d8ce,0xa20f1b7c),L_(0x3bc4db1d)}, {LL(0xc7206a4f,0xf3f061d0),LL(0x7fc330d4,0x97fe5b1c),LL(0xd64fa0dc,0x93be4afe),L_(0x239a905a)}},
   {{LL(0x465eb830,0x9491ca54),LL(0x3a2368e4,0x096991e5),LL(0x840339bc,0x60229ac3),L_(0x0f74bf45)}, {LL(0x167045a8,0x6bb23f63),LL(0x6df7548f,0x7c1e871a),LL(0xbd4ae1af,0xa5c7651d),L_(0x85fa24af)}},
   {{LL(0xc10316ff,0xd9ea606e),LL(0x72e722f7,0x1d7722d9),LL(0x02dae11f,0x6f289df9),L_(0x4f148db1)}, {LL(0xb66ca6f2,0xe8df6026),LL(0x0dc6bfc0,0x6f25ab93),LL(0xc2c848fa,0x726e5bf2),L_(0x0bc8372a)}},
   {{LL(0x29431ba8,0xe513cc60),LL(0xb368f5d3,0xf213ba74),LL(0x2acdbaae,0x077919ee),L_(0x69129848)}, {LL(0x3dbdea30,0x5c03af05),LL(0x6892cac4,0x81b53f34),LL(0x13156876,0x86df2c46),L_(0x15d63bed)}},
   {{LL(0x20319c6f,0x85ef7365),LL(0x7ea45182,0x2f38b49b),LL(0x7d4f7ca6,0x2e79b2d0),L_(0x5befddd6)}, {LL(0xcfcb5a29,0xeb6e8177),LL(0xd7c05293,0xeb5b17b7),LL(0xfebc1ad3,0xc2ef9919),L_(0x5ef848b7)}},
   {{LL(0x628b133f,0xec0fd6a2),LL(0x4834ba2e,0x71d618c4),LL(0x70062560,0xee7cbb5c),L_(0xa240c28a)}, {LL(0x0c5f4e11,0xe8bfcc1d),LL(0x36f2363a,0xdaf65ead),LL(0x2c7fdead,0xee2e792a),L_(0xfe7f2c9f)}},
   {{LL(0xb9045bc3,0x21f1b865),LL(0x7a207076,0xf7c65eaf),LL(0xa289580c,0x26c6a4fb),L_(0xac8c38be)}, {LL(0x2c1dd729,0x603a204c),LL(0xc96f5b0f,0x7ec2e50b),LL(0x07974ac0,0xdd93d467),L_(0xd384c831)}},
   {{LL(0x27da0d82,0x130cf277),LL(0xb2d57bad,0x7cd69a13),LL(0x2e92cb98,0x0b8251c6),L_(0x04325b28)}, {LL(0x35f746a8,0x630e8e9d),LL(0xcda5985f,0xe5c307ba),LL(0xf6775b8c,0x9cc53c03),L_(0x466530c7)}},
   {{LL(0x660bcc85,0x9b3d46a0),LL(0x6911dd6e,0xfec92d25),LL(0xf5c72cad,0xaf7d9d93),L_(0x53a3ea3c)}, {LL(0x244c7ae6,0xd879f3e1),LL(0xeaba3d43,0x57858904),LL(0x4c921338,0x3b59edc0),L_(0x7ee8ea79)}},
   {{LL(0x0d019290,0x64c557a0),LL(0x70b7979e,0xa66689a7),LL(0x482d3b9a,0x08f6017c),L_(0xecf693f0)}, {LL(0x90c3c46a,0xe8816977),LL(0x7c8670c4,0xf94ce04b),LL(0x05b8eed1,0xa8713371),L_(0x0ebddf9f)}},
   {{LL(0xe8f0dfc1,0x50dc1442),LL(0x884f22ec,0x104f20a7),LL(0x5f885c0b,0x4da67f62),L_(0x6b4ecc34)}, {LL(0x8460d391,0x0181b70c),LL(0x3f227a67,0xffedd7e8),LL(0xafb114cc,0x5b40835b),L_(0x2f5bb436)}},
   {{LL(0x2da90ee3,0xdaec76bb),LL(0x6deef6a5,0x05e4b337),LL(0xe70560f3,0x8e70fab8),L_(0xdf0c450d)}, {LL(0x76bc05d5,0x916202a7),LL(0x05568dde,0x84cf17af),LL(0x89f0919f,0xb5670da6),L_(0x97387cce)}},
   {{LL(0x54cdc952,0xc269a349),LL(0xae4d60ad,0xa15fdc88),LL(0x27a5a550,0x560e5ad7),L_(0xfd2940f8)}, {LL(0x23e1393c,0x86446f69),LL(0x2f2ee978,0x918a434e),LL(0xa481c71a,0xd1f9e2af),L_(0x99fe0c80)}},
   {{LL(0xecc46dd9,0x4c354b31),LL(0x9d2731ca,0x5917267d),LL(0x9583d77a,0x549fab4d),L_(0x20743202)}, {LL(0xfa4579de,0xc9afa8d5),LL(0x6ee70bf6,0x679e9b14),LL(0x71181b55,0x34dfc171),L_(0x186cb701)}},
   {{LL(0x92faa6da,0x784904dc),LL(0xef91aeb1,0x99e8f15a),LL(0x466ca745,0x9dba0e94),L_(0x3efbbed0)}, {LL(0x9c629c4a,0x818f1160),LL(0x238404b3,0x1649074d),LL(0x7f378102,0x9026e5ef),L_(0xd9ddeb02)}},
   {{LL(0xcb042e26,0x50f516e2),LL(0x359a1ffe,0xf69c04f0),LL(0xea7c7fb8,0x2be043ec),L_(0xe26d37f4)}, {LL(0x4f292184,0xd8d23c81),LL(0x97f967f3,0xa3d950fe),LL(0xf97e68cf,0x0cf39dbb),L_(0x0f3321ca)}},
   {{LL(0xf1e4bd62,0xd200dfdf),LL(0x890af600,0x9e7aabf2),LL(0x1e4962b1,0xd80a6211),L_(0x78562fa6)}, {LL(0x2ca32240,0xdb5a70a7),LL(0x68abbf8d,0xb27e8e16),LL(0xbffea5c8,0x99114570),L_(0x6e5308c1)}},
   {{LL(0x4fe00738,0xdfd77bdc),LL(0x16448af4,0x0bebe0b7),LL(0xcb49f26a,0x9f4c424c),L_(0x7c95aa03)}, {LL(0x84b25349,0xd5948dce),LL(0x93acdc43,0xa92b381d),LL(0x7efb93e4,0xa8e311a0),L_(0xe08e348f)}},
   {{LL(0x8f67c168,0x96c4f394),LL(0xfa10af74,0x457899a3),LL(0x489d01bc,0x1a4d0c16),L_(0x5457e489)}, {LL(0x5c85d68a,0x1331b4d9),LL(0x84eb66f4,0x97024d48),LL(0x8054cf7a,0xb191cf89),L_(0x3e7eb0cb)}},
   {{LL(0xd307d077,0x8d0cc3ce),LL(0xa6a534ca,0xa8bd8bc1),LL(0xa917d88e,0xc01d22dc),L_(0xba878e3c)}, {LL(0x6dc0cccd,0x927b4947),LL(0xfb3f69e6,0x8cef47b8),LL(0x44185c64,0x586c00a9),L_(0x151ed75b)}},
   {{LL(0x2be07ecc,0x72a46aa9),LL(0x519ea031,0x5b289abe),LL(0x75f6f741,0x1a09f429),L_(0x0df7057c)}, {LL(0xe83d8445,0x0511f341),LL(0xb8c291d5,0xcd82da51),LL(0xbd814e25,0xa548461b),L_(0xdfcbbc5c)}},
   {{LL(0x481967d8,0xf029f707),LL(0x3ec594c8,0x9b2f8b93),LL(0xaacdcc45,0x76cebcbf),L_(0x6e7c6e83)}, {LL(0xaec54ae3,0xa2a8a4fe),LL(0xb47c1c29,0xea7ebbf6),LL(0xebdc1ae0,0xed49a01f),L_(0x63ac12b4)}},
   {{LL(0x532eddda,0xaa4b3b46),LL(0x7e957ed9,0x54c46d48),LL(0xe3393b10,0xde08b775),L_(0x8522afee)}, {LL(0x04a30d0e,0x5f73d8d1),LL(0x39d26476,0xa070b5a5),LL(0x7e58e0f3,0x213036c0),L_(0xa0e99b0f)}},
   {{LL(0x180c99f6,0x8a626b91),LL(0xbe1888a3,0xa37b0a55),LL(0xc627fd2e,0xd55c88b6),L_(0x75c0ce87)}, {LL(0x143ea75e,0x96806d01),LL(0xdce56555,0xb31db013),LL(0xa0f7c5ea,0x286d7145),L_(0xd7ff4a5f)}},
   {{LL(0x7eb22eef,0x9f5793c8),LL(0xf0bdad25,0x3795ea86),LL(0xde77b0de,0x308c67ea),L_(0x506de99f)}, {LL(0xa761521a,0x9d70efe9),LL(0xf8b6908d,0xbeee111a),LL(0xa22f9208,0xabb48b09),L_(0x305250b9)}},
   {{LL(0x54347d1c,0x04d51ba3),LL(0x69790978,0x0cf10a39),LL(0x55c72c45,0x340e8617),L_(0xcd9b6116)}, {LL(0x897f38e5,0x80b0004a),LL(0x8b257151,0xa9ff195d),LL(0xae7d889b,0xd95f9a8b),L_(0xf7035183)}},
   {{LL(0xf9445254,0x96f17da2),LL(0x21e71bcf,0x87497b69),LL(0x1dde1b45,0x3f515a1d),L_(0x1f620fc3)}, {LL(0xbb05c2ca,0xd2651945),LL(0xa7f48789,0x5426bd20),LL(0xa9e9b7f9,0x048d20b9),L_(0x2efd58e7)}},
   {{LL(0xfddc7154,0x095a8f25),LL(0x6fa2edc8,0x675fd35e),LL(0xad7dc081,0xf2b8fbbb),L_(0x88a7b80f)}, {LL(0x6cabea84,0x2ca3a2a5),LL(0x00c09332,0x1047dcd4),LL(0x0d577a61,0x54d150a9),L_(0x1837cf45)}},
   {{LL(0xe243bea6,0xec967631),LL(0x281b5c05,0x7c40da6a),LL(0x580362ba,0x832277a8),L_(0x702cc617)}, {LL(0x1ca409f6,0x7335bdc8),LL(0x26119f95,0x735f11ed),LL(0xa27ab674,0xc7055ffb),L_(0x652580b5)}},
   {{LL(0x3dba43f3,0x8c6adeda),LL(0xebd0f498,0xa42d63e8),LL(0x993107d0,0xb75b4edf),L_(0x0232f8ad)}, {LL(0x720ca302,0xa2592945),LL(0x48d03da4,0x473bf917),LL(0xda871c01,0x8ff4aaa7),L_(0x746e818b)}},
},
/* digit=14 base_pwr=2^98 */
{
   {{LL(0x3e28ba8b,0x23800fab),LL(0x8b12655a,0x99a4d041),LL(0x66a4179d,0x33023ec9),L_(0x1205108d)}, {LL(0xdbb13a66,0xa87c96ac),LL(0x402e3b9a,0x98998891),LL(0xd3dbe7de,0x6b327511),L_(0x5723ac5b)}},
   {{LL(0xa70cb5da,0x7ab235bd),LL(0xbede2315,0xac5f3af5),LL(0x5f9a6aaf,0xe6d56464),L_(0x357d2d21)}, {LL(0x3156d6fe,0x95a60b40),LL(0x49f4f0e2,0x9e63e29f),LL(0xdca3ea62,0x4c739342),L_(0xa16d4457)}},
   {{LL(0xe1948d0e,0xdcc5b89a),LL(0x463327ef,0xd5a2f46b),LL(0xadb9328a,0xbfce0e4d),L_(0x0d62b545)}, {LL(0x1dd89a4b,0x349d4bd5),LL(0xa76133fc,0xc19125f9),LL(0xb4284724,0xf5a817aa),L_(0xf7ca3909)}},
   {{LL(0x7fc121e6,0x0655039e),LL(0xbc44db37,0x42a0f87f),LL(0x3d4e983e,0x67d89bbe),L_(0x85435ace)}, {LL(0x18346e4d,0xbe29cd2e),LL(0xed39f9e8,0xe8b5a373),LL(0x55cac6ca,0x3d4406d2),L_(0xde2aeda4)}},
   {{LL(0x6b7d56c0,0x8fabc04a),LL(0xc39c7ee8,0xbb51ed42),LL(0xc619fe58,0x0816fbf5),L_(0xf268e46c)}, {LL(0xe0f981bd,0x3b1bba1f),LL(0xce226865,0x27136de9),LL(0x4f9ab1d9,0xfcbc42d6),L_(0x13efb949)}},
   {{LL(0x4f54831f,0xf8119c34),LL(0x75d60569,0x24503e1c),LL(0xb3a82995,0x063e48b4),L_(0xc7bc1aec)}, {LL(0xf21ff6a7,0xd626619e),LL(0xa7f182b8,0x5a64395d),LL(0x533f72eb,0x1d2aa02e),L_(0x69953a2d)}},
   {{LL(0x64b57fc3,0x91516c9b),LL(0x0913a322,0xb1936549),LL(0xa0961a2f,0xa7d10cb1),L_(0x06e4869d)}, {LL(0x8f465d18,0xdc8793e6),LL(0x9f17e3a2,0x9a3dd9de),LL(0x06fc6243,0xe925133a),L_(0xb7622bbe)}},
   {{LL(0x784462ed,0x84e7489a),LL(0xc981ba62,0xe5d5e019),LL(0x792db847,0xa77e3ed1),L_(0xf4b6792a)}, {LL(0x36f5a622,0xed208d22),LL(0xfc7dc4a1,0x953f894f),LL(0x87c2b024,0x095b0ea7),L_(0x88016bc4)}},
   {{LL(0x486f7168,0x59119396),LL(0x4b8755e3,0xb37c8ef0),LL(0x19aa2514,0x554f2e58),L_(0x1d544a58)}, {LL(0xaa9858fe,0x1d4dba6e),LL(0xcc8e5a35,0xa400007e),LL(0x4036ed62,0xf70ca123),L_(0x0fd793e9)}},
   {{LL(0x292ba598,0x269da084),LL(0x2503b0ce,0xdd28f9d0),LL(0xb3025472,0xeb061c62),L_(0xfe9ae248)}, {LL(0x79d8d16b,0xd2906b24),LL(0xeaa5d6f9,0xdee30126),LL(0xaf62d50d,0xc2b140a8),L_(0x4dbbcce0)}},
   {{LL(0x57f41736,0xa7cf5774),LL(0x3293ff9c,0x238c8acf),LL(0x1f10127e,0x739c1785),L_(0xc01f9711)}, {LL(0x761037ad,0x68df6a03),LL(0x50fcfcf4,0xc797349c),LL(0x340f3712,0x53cced84),L_(0xba988480)}},
   {{LL(0xc57ef65f,0xed47d3b7),LL(0x6051e486,0x623f7228),LL(0x2e29a4c3,0xfb40723c),L_(0x77353f89)}, {LL(0xda6cb60f,0xf4eea8fd),LL(0x672e091a,0xd086a7ab),LL(0xef34d20e,0x96e8e707),L_(0x2e5db0f7)}},
   {{LL(0xd47d7453,0x089e80e8),LL(0x259bf71c,0x99cb2528),LL(0xcd52c81c,0x9b67f891),L_(0x106aa44c)}, {LL(0xb3fd57c9,0x84cca31f),LL(0x492159ac,0xd208e111),LL(0x99fc08c2,0x9a577c2f),L_(0xde194e1f)}},
   {{LL(0x3ffcd5a2,0x1a869e94),LL(0xac2dd0d1,0x0101716e),LL(0x0d98495e,0x926fca45),L_(0x230d5720)}, {LL(0xd7a376f8,0xda97aa6f),LL(0xe0f36b43,0x043a0c8c),LL(0xbdee74a9,0x9f4f1e97),L_(0x616d2e77)}},
   {{LL(0xc462e14f,0x4ce190f6),LL(0xe408ab06,0x8983aaf7),LL(0xaec26bc8,0xea864657),L_(0xa9c6ad27)}, {LL(0xe823a8e1,0xab862943),LL(0x327c78b5,0x42d64507),LL(0x751ec6f1,0x18bed81e),L_(0x1c240eff)}},
   {{LL(0xc5905237,0x291c3272),LL(0x766b4a0b,0x7507a05b),LL(0x6e56a096,0xafb97a2d),L_(0xe3cbbbe3)}, {LL(0x44beffab,0x2369de34),LL(0x4c78c1e4,0x287f5c70),LL(0x698e8f90,0x14e1a4eb),L_(0xa87a74e8)}},
   {{LL(0x4ca2c62e,0x0fb3705f),LL(0xcbf75eae,0xe59c7ef1),LL(0x58dee01f,0x1a149a9e),L_(0x67ff1a09)}, {LL(0xa2bee6a8,0xf63b9864),LL(0x7e111af7,0x602716b7),LL(0xf6f23814,0xc6967a32),L_(0x682916a3)}},
   {{LL(0x94c36bb8,0x53aa5941),LL(0x7dc26ca6,0x57977663),LL(0x3a7f3859,0x4c6d1720),L_(0x87d2b98e)}, {LL(0xce189ddd,0x379e4e5c),LL(0xc6bec0ca,0xafc9ee7e),LL(0x69db9b01,0x4e3aaa80),L_(0x4f1a3e1f)}},
   {{LL(0x9f2327ea,0x948cf978),LL(0xe1a7d9ed,0x932e601a),LL(0xabd060c2,0xe2c8026f),L_(0xd02845ba)}, {LL(0x4568e0ed,0xd2fcd20c),LL(0x9fc8ca8b,0x224394d0),LL(0x0b403598,0xfa897231),L_(0xd2ef4cab)}},
   {{LL(0x61ffdada,0xc9608223),LL(0xd3a9e728,0xbfde5d50),LL(0x9fcf28b0,0x23db313b),L_(0xe226f1f0)}, {LL(0xc362ae30,0xdd26c697),LL(0xa3a7479f,0xbe0de20e),LL(0x4871cbd5,0xc0ad648f),L_(0x30414c9b)}},
   {{LL(0x93a58cef,0xe22f8df4),LL(0xdef38d08,0x40b0ba7f),LL(0xfacc23b1,0x7b3f5e97),L_(0x00fb9a2e)}, {LL(0x926c141d,0xa6cbdc36),LL(0x87cb811c,0xefb43047),LL(0x6214d9f5,0x06806fd9),L_(0x5ba8f23b)}},
   {{LL(0xc5b2ff25,0x86e33c2c),LL(0x23423298,0xde9bc0ea),LL(0xf6e30434,0xc242970e),L_(0xbf53cf6b)}, {LL(0x3df6ea77,0x84f61ee9),LL(0x8140c42b,0x1d03d30e),LL(0xe917c40a,0x70d1d6e0),L_(0x95de99fd)}},
   {{LL(0xca391889,0x396774da),LL(0xc249bd59,0xc8c93775),LL(0x24f81111,0xfe365e8c),L_(0xdcd6e625)}, {LL(0x680035a8,0xe01d308c),LL(0x65e9e9bf,0xe2741d32),LL(0x5cab99f6,0x441a1645),L_(0xf759c45d)}},
   {{LL(0x858ad4cf,0x1d7326e1),LL(0x52fad307,0x7ee589d6),LL(0x3aa389bc,0x9cfd5023),L_(0x3feb82b3)}, {LL(0x03b8a60e,0x4a98e744),LL(0x22a198b9,0x2c871f83),LL(0x5fd22f5d,0x95dcd425),L_(0xe40ca457)}},
   {{LL(0xb6c77ce3,0x0baec1d0),LL(0x899726d2,0xe604c5b2),LL(0x9503a4a3,0x8faff5a9),L_(0x735bcc96)}, {LL(0x6c5c0a3c,0x294b5dc8),LL(0x27659e3e,0x63327cba),LL(0xb11539c5,0xf82de3fd),L_(0xdafe9d6c)}},
   {{LL(0x2b031402,0xb712f885),LL(0x4ba7d548,0xc6d1f271),LL(0x627d0816,0xebc722ce),L_(0x4e21b679)}, {LL(0x2f1104b0,0x3d488791),LL(0x654d615e,0x7a45f251),LL(0x38e30e5a,0xbe0c2b4a),L_(0xd07392c8)}},
   {{LL(0xc3944e46,0x91582e10),LL(0x245a5f24,0xbf72cd72),LL(0x1d74fe95,0x32fc2d8c),L_(0x8d68efa6)}, {LL(0xc63d3e55,0xb73ed81a),LL(0xd7c363e3,0x81d7988d),LL(0x4b18aee3,0x8563cd35),L_(0xf296ee0a)}},
   {{LL(0xde9cd058,0x6115b5de),LL(0x12e07056,0x494a27b8),LL(0xab33f925,0xdadb7f44),L_(0x9611fc0f)}, {LL(0x7c525248,0x135db673),LL(0x0014d0b2,0x13d5d03c),LL(0x34df44be,0x498b5aab),L_(0x3b9bf0a3)}},
   {{LL(0xb6d60031,0x8b47b859),LL(0x7583945e,0xd5d1a023),LL(0xe1eb9f0d,0xdc6bf5ac),L_(0x7f50540d)}, {LL(0xd48c1aad,0x7dcd0fa2),LL(0xd6e884e8,0x7ff3b8ae),LL(0x0fda3c41,0x42d34849),L_(0x156b5074)}},
   {{LL(0xf41aefe8,0x9db1951d),LL(0x6f1aaef3,0xc78e5a0a),LL(0xe07d36c2,0xaf67c45f),L_(0x598114fb)}, {LL(0x491a5763,0x1883388f),LL(0xb079f72e,0xfa868602),LL(0xee4b11df,0x20bd4ca5),L_(0x9dd3598b)}},
   {{LL(0x4fe58726,0x2f515e06),LL(0x0a130c64,0x03928fde),LL(0x21d929ee,0xed92766f),L_(0x2370b592)}, {LL(0x26410775,0x461ee104),LL(0x11fd327c,0xbf2200cc),LL(0xaac6ed8b,0xaad362f8),L_(0xe2c21338)}},
   {{LL(0xa0369424,0x7702c304),LL(0x90cc96b4,0x88180dc5),LL(0xec9108c9,0xea844162),L_(0x79d4d2e8)}, {LL(0xf7c7120d,0x9b79933f),LL(0x9b1c7dfd,0x73191637),LL(0xf8a798d5,0x4a0a1e22),L_(0x7439503e)}},
   {{LL(0xa7716be7,0x25112356),LL(0x41a91a95,0xf0234300),LL(0x12be9cf2,0xa09c00f4),L_(0x87b213f6)}, {LL(0xb0626c0f,0x3cc23f7c),LL(0xa44cb6f8,0xee84c654),LL(0xda068194,0x999bf774),L_(0xe19968dc)}},
   {{LL(0x20c86fd9,0xa3bbb5a6),LL(0xe07280d1,0xdced85e1),LL(0xf0df9658,0xbd50dd87),L_(0x4715fc8b)}, {LL(0x90915daf,0xf3d02094),LL(0x64a96f25,0x5a8039b5),LL(0x661822c3,0x8b298e54),L_(0xd8ec764d)}},
   {{LL(0x4146852b,0xcef9767a),LL(0x3dfab91a,0x3b280141),LL(0x85522368,0xdbf3bb79),L_(0x79b36fcb)}, {LL(0x730e7127,0x05fa18c1),LL(0xdbafd1f2,0xf1172af0),LL(0xdd549dff,0x83c7657f),L_(0xd82dc78f)}},
   {{LL(0x35ef27c5,0xad652989),LL(0x58b32f27,0xeb421196),LL(0x976af3d3,0x4b505f2e),L_(0x64c1cfc3)}, {LL(0xf3dae0f8,0x59913cfd),LL(0xbab37657,0x2039f0fc),LL(0xf066519c,0x8887e569),L_(0x3d975cff)}},
   {{LL(0xb1eb152a,0x425ddccb),LL(0xd5368e78,0xe11b71d6),LL(0x12b23c3e,0x33076aa1),L_(0x4e626b1c)}, {LL(0x4b18cb13,0x23f1e78c),LL(0xe86079cd,0x875b887a),LL(0x908738ce,0x244ce9b9),L_(0xe51d741d)}},
   {{LL(0xd6758071,0xb2d63b63),LL(0xf0e2cdd8,0x30007109),LL(0x90f8143d,0x75829dd6),L_(0x2d23a43f)}, {LL(0x5e5327c1,0x47ca8667),LL(0xa526104e,0xf04921aa),LL(0x85467645,0xb0f0e275),L_(0xafc61eeb)}},
   {{LL(0x87b4b9ac,0x7ee79f80),LL(0xf367484b,0x27ef21cc),LL(0xbfc8c258,0x7fe068cc),L_(0x7c8262c2)}, {LL(0x572f615c,0xc96069f0),LL(0xd5c6e6e1,0x5f2c8d88),LL(0xd8f70ecc,0x24073300),L_(0xc55f13ed)}},
   {{LL(0x33b47b28,0x82d22a51),LL(0xe38ffddd,0x480198ae),LL(0x03df60cb,0xa964fa9d),L_(0x9bfb3890)}, {LL(0xee5cec96,0x96707a63),LL(0xf9f9338a,0x22ae2467),LL(0x5d9e9ac0,0x0acd29e1),L_(0x854609dd)}},
   {{LL(0xb29eb007,0x84bcad3a),LL(0x9a527f75,0x761f8288),LL(0xc16f294f,0xdaf38a99),L_(0xfdebcc4a)}, {LL(0x9c537eba,0x8a1d04d0),LL(0x760244a9,0x03627bce),LL(0x97af96d2,0x79ecc48b),L_(0xebf508a7)}},
   {{LL(0xa6400e31,0xa9338a43),LL(0x4ccbe898,0x4312a928),LL(0xf79ce659,0xa7f6528b),L_(0x0efdd8f6)}, {LL(0x0c3ef14c,0x7002f54a),LL(0x859eefc9,0xacd6c70b),LL(0x3146204e,0x25b282fd),L_(0x9e570ac1)}},
   {{LL(0x7cfc8794,0x50227a05),LL(0x96cc2182,0x5a9a9895),LL(0xed491384,0xa5e9dc33),L_(0xef480409)}, {LL(0xf767a743,0x48fc5412),LL(0x7b84b8b7,0xb18a17fb),LL(0x22c04cfa,0xa2e99567),L_(0x5b8dc942)}},
   {{LL(0x45e7789f,0xb80a1876),LL(0x7a0670f5,0xaff77eda),LL(0x99c0dcff,0xceba003e),L_(0x737b1478)}, {LL(0x169db41d,0x88ff60f7),LL(0xa8a91790,0xe2c4f8ba),LL(0xd0c75bfd,0x8ebfd0f8),L_(0xaf779b34)}},
   {{LL(0x0401de06,0x62fec3e1),LL(0xd969780a,0x99a90daf),LL(0xafc171d9,0x647f2bf8),L_(0x16ea68f8)}, {LL(0x41dbf4ac,0x83f12d02),LL(0xb63e2588,0xa64d6924),LL(0xb69cbad9,0x08b7948f),L_(0x1a740ee4)}},
   {{LL(0x4eff7cff,0xd3ad093d),LL(0x97774b8e,0x520858a7),LL(0x1438ff5b,0x90acd42f),L_(0x0c55b2c6)}, {LL(0xdb613141,0xf3524dc0),LL(0x1f19f151,0x4d56c7a8),LL(0xc5b4f62c,0xe1899b7c),L_(0x7240d638)}},
   {{LL(0xeb1a717d,0xf9ad69aa),LL(0x9a96c604,0x32f476a8),LL(0xd851c4f0,0x816d9822),L_(0xf895d68e)}, {LL(0x6f708c56,0x914c2f2f),LL(0xdd941975,0x64e1cdce),LL(0x1765af52,0x9bdc107c),L_(0x770ebc8f)}},
   {{LL(0xc745e953,0xe5bd052c),LL(0x70e13a7c,0x1d26e2b6),LL(0x0c4379c6,0x0121f8bc),L_(0x71124e71)}, {LL(0x87d6954b,0xe43a3a54),LL(0xd9134113,0x9bfa2ae6),LL(0xbbbe7daf,0x3a1c2b24),L_(0x3950f72a)}},
   {{LL(0x6a18f59a,0xa71550b0),LL(0xdab1491f,0x73428fb9),LL(0x12caa6e9,0xb08da5e0),L_(0xf505a6eb)}, {LL(0x3f48afb0,0x79842176),LL(0xcf510150,0x2962dfb0),LL(0xafdce054,0x87a8f6f2),L_(0x8158b7f4)}},
   {{LL(0x438e4a3a,0xe2eb279d),LL(0x6af97a49,0x42b5d181),LL(0xd6c5159e,0xcf78aa46),L_(0x572c05de)}, {LL(0x50428178,0x0bf9ef59),LL(0x5b7148bf,0x479ccacb),LL(0xbe4b9632,0x101cdc13),L_(0x7ed1a6b8)}},
   {{LL(0xf0214364,0xb580b66f),LL(0xc6feaf5c,0x4e810e3e),LL(0xa5cb0934,0x02cfec16),L_(0xc2e35b6b)}, {LL(0xddb45407,0x6e09365d),LL(0x5b4d0976,0x76d217cd),LL(0x405714ef,0x892fc650),L_(0x75be4796)}},
   {{LL(0xf30d07fb,0xe024a3b5),LL(0xffdc8787,0xe4416883),LL(0xbb8a1baa,0x16ab4631),L_(0xf8688223)}, {LL(0x2f0f8322,0x09c1692b),LL(0x0dd7a6f9,0x13a2837f),LL(0x3a94d17d,0x784e210d),L_(0x58b82c4e)}},
   {{LL(0x75a9d2ec,0x4df5c032),LL(0xa0b09f2e,0x9989a169),LL(0x8ad59d7f,0x92ef323e),L_(0x9dfd994c)}, {LL(0x641354fe,0x9c294c51),LL(0x0784cc2a,0xa2a1bccd),LL(0xa7c397c2,0x9d3b5c1e),L_(0x5217298a)}},
   {{LL(0x1d43acad,0x5fffdecb),LL(0x15425166,0xd4c107c2),LL(0x3ad2c54b,0x29b56266),L_(0xcbfffc28)}, {LL(0x2c3f6f4b,0xb4051ff3),LL(0xe925b548,0x65af8f85),LL(0xd94421d3,0x61d3c552),L_(0x0c2f7520)}},
   {{LL(0x6d77a3b7,0x70ed4768),LL(0xfa81c450,0x9fe1ff87),LL(0xbb762aa7,0x72d6d774),L_(0x7b08eae2)}, {LL(0x7dbd50f7,0x1432d1e1),LL(0xc28ea0a8,0xd0acc833),LL(0x57bf54b2,0x163d3f1c),L_(0x996fa17f)}},
   {{LL(0x4583d724,0xb5fbaafe),LL(0x53d923d7,0x051dd3b4),LL(0x8aeef402,0xc59594fb),L_(0xd7267ef5)}, {LL(0x5e8a7497,0xede0bede),LL(0xb019e637,0x6171e5a5),LL(0x1be98aa9,0xb69b47a8),L_(0xd848486b)}},
   {{LL(0xddef3d83,0x95f000be),LL(0xc417bef7,0x615e1192),LL(0xef4c07bd,0x74d082e1),L_(0xc8cd53f2)}, {LL(0xc2fe846c,0xfb026020),LL(0xaab1ecd2,0x31978a04),LL(0x47b08bd0,0xf6ef41f3),L_(0x386f4e41)}},
   {{LL(0x0974e562,0x806b3be2),LL(0xd292e87d,0xb4c666b3),LL(0xb31604a0,0x08cba3d2),L_(0x5a2daba1)}, {LL(0xd6e9ea3f,0xff881f9f),LL(0x4da3f147,0xbaac2563),LL(0x8b3ea4f6,0x7f14ef98),L_(0x931e6a5d)}},
   {{LL(0x09ed1548,0xe8b9b58f),LL(0x3a4e460d,0xbdf4bff4),LL(0x410540d4,0xba715bf7),L_(0x9faf6e57)}, {LL(0x58d1763d,0x13d8563b),LL(0x83a48da3,0x3effa46a),LL(0xbc420fce,0x54fecfb6),L_(0x8ea8b02c)}},
   {{LL(0xcc8fc2ff,0x9f084f50),LL(0x15421912,0x4d3ec704),LL(0x69e1810f,0xa93a4b35),L_(0x66d7f71e)}, {LL(0x75523ca2,0x9b5d9b14),LL(0x85c36f3f,0x8615c97c),LL(0xf837f0c4,0x1acc3982),L_(0x4526c2f7)}},
   {{LL(0x61301784,0x91bdbe3d),LL(0x5c462b80,0xe5304d0f),LL(0x16f4a53a,0x2452c556),L_(0xf2ad5a49)}, {LL(0x92965e0e,0x961c76d6),LL(0x92031bde,0x3e75de1d),LL(0x43fc1b46,0xba2806c7),L_(0x7ea59306)}},
   {{LL(0xe2e84c48,0xaf7fedc1),LL(0x44f370ac,0x7473cfd6),LL(0x7fa03a08,0x366491a9),L_(0x65caef0d)}, {LL(0x8b501288,0x3a469595),LL(0x6dc5f00f,0x67981e43),LL(0x87cfba83,0xebb7f488),L_(0x1fc8150f)}},
   {{LL(0x0adc88a9,0x70a3e6c5),LL(0xf79b97ae,0xc2b70fcd),LL(0x0c92525b,0xfd612292),L_(0x1f877d7a)}, {LL(0x2017f6eb,0x2b249ae0),LL(0x78a6422a,0x083dff80),LL(0x0659a5e4,0x12934c87),L_(0x7a1fca11)}},
   {{LL(0x2ef56139,0xe4d85de6),LL(0x0b0acedc,0xf2c33f4b),LL(0x6091bc19,0x04526cb6),L_(0xac1ce3ea)}, {LL(0x24bfc4bb,0xb54024a6),LL(0x57196c24,0x909d86a2),LL(0x42e04322,0x1b1d5d2a),L_(0xbb31992a)}},
},
/* digit=15 base_pwr=2^105 */
{
   {{LL(0xb476bc0f,0x5f4861c3),LL(0x9a1b156b,0xed89263c),LL(0x45b930e9,0xbeb5a955),L_(0x29358e8a)}, {LL(0xaad0495f,0x7923968b),LL(0x127f5983,0x3abc6adf),LL(0xf5aef19c,0x22713358),L_(0x0329be36)}},
   {{LL(0x97c8de6b,0x842632ce),LL(0xd5e766ad,0xfcb4c194),LL(0xbab1e908,0xdb59f894),L_(0xa103e5b9)}, {LL(0xe0644e8b,0xc4c67972),LL(0x2cd3e628,0x340af5c5),LL(0xe2690b88,0x787c5ec2),L_(0x153e0f49)}},
   {{LL(0x7e2abe1e,0x01d9f5c6),LL(0x9acab18d,0x4187566a),LL(0x96fbb4bc,0x43543f3e),L_(0x19e99acc)}, {LL(0x911c12bc,0x4d4f6942),LL(0x2373babd,0x05a419a8),LL(0x0c662933,0xdd524336),L_(0x60570f05)}},
   {{LL(0x5607a4f3,0x98a32e87),LL(0x9cbcb156,0x841d648d),LL(0x00a72b88,0x85f705a8),L_(0xb60a7751)}, {LL(0x30e3dc7e,0x3ef7d9b6),LL(0xc5078f17,0xeec26796),LL(0x20110dec,0xb521d60e),L_(0x324aa16f)}},
   {{LL(0x547a71b7,0x7053cd14),LL(0xc75d38b5,0x3ecd5cad),LL(0xff26fb53,0xfa6fa1ea),L_(0x89904b74)}, {LL(0xba1aaedf,0xb9800605),LL(0x0d1c07cb,0x517f26e6),LL(0x7605e424,0xe982130b),L_(0xc4ed9c66)}},
   {{LL(0x06117460,0xb3f704f4),LL(0x2c1c043d,0xed0726e9),LL(0x3e68dbd8,0xbe354e8d),L_(0xa6382386)}, {LL(0xa4136478,0x26378a40),LL(0x04646666,0xcc8a8909),LL(0x88fbb7ce,0xc603785d),L_(0xb1f815a2)}},
   {{LL(0xee89450b,0x43fe75c8),LL(0x9a29229c,0x16a051c0),LL(0x9b1a9170,0x7d4b9fc3),L_(0x133dd4ca)}, {LL(0x738df325,0x53ac8646),LL(0x6dcfad8d,0x5c600cb6),LL(0xe4115708,0x6ff11203),L_(0xbc9689c3)}},
   {{LL(0xaab0cefc,0xd81789dc),LL(0x7b457d74,0x67ca91b0),LL(0x5a87275f,0x096f997d),L_(0x5d4f53a6)}, {LL(0x3d1c72aa,0xc86a176a),LL(0x0c5247ba,0xbb03ebf7),LL(0xb389c0f5,0x325d67b1),L_(0xdc511ddb)}},
   {{LL(0xe0e68838,0xa0dc02ac),LL(0xca8b2e56,0xcb1f32e9),LL(0x59515fbd,0x152ad207),L_(0xa0ec8c40)}, {LL(0xaf08ac54,0x990ce0e8),LL(0x094829b8,0x62062c81),LL(0xdb93ce78,0x53af9d31),L_(0x980f663f)}},
   {{LL(0xa663c4b2,0x0afcea40),LL(0x1410da31,0xeb72dea5),LL(0xc4dd3dce,0xefc18ea4),L_(0xc844e9d6)}, {LL(0x827776e0,0xb2b1ab80),LL(0x89238185,0xeef3941c),LL(0xc007c089,0xcc238164),L_(0x10df0d3c)}},
   {{LL(0xccb1a15d,0xd9103e73),LL(0xec1702f8,0x34d31501),LL(0x67e64440,0x4b15a941),L_(0xaf4f5d2e)}, {LL(0x8b86c479,0x82fa4170),LL(0xbc874d39,0xb17316c7),LL(0x63f968d3,0x7b71afa1),L_(0xc6c567c6)}},
   {{LL(0x8941fa38,0x1b353d4b),LL(0xfbfeb69e,0x76d964d5),LL(0xaecee209,0xdaae33fb),L_(0xc8d8f5e9)}, {LL(0x47c07e3b,0x4e52dc91),LL(0x87741a88,0x92359ae5),LL(0xa5009004,0x72c650f3),L_(0xc505e3b9)}},
   {{LL(0x660afa5f,0x2c204e5a),LL(0x9f6af3aa,0x70aa9a5d),LL(0x0423f2f9,0xa67cf879),L_(0xe29129c8)}, {LL(0x58a3703e,0x47ebe4a0),LL(0xc220fd1a,0x8f060a95),LL(0xa34e4a66,0x1fd2a0dc),L_(0x7c1fb322)}},
   {{LL(0x1bfb2217,0xcad10b11),LL(0xca3e4b04,0xc1ab5423),LL(0x00ce7df7,0x9c84d269),L_(0xa1f31fb1)}, {LL(0x8a721b19,0x45e72360),LL(0xf80ab752,0xa59a87d1),LL(0x38935f38,0x36966819),L_(0xebd1b2a6)}},
   {{LL(0x657a1f10,0x28dbeb1d),LL(0x4f41e2ad,0xafbe6ad3),LL(0x1c0782c2,0x60b0cb1e),L_(0xb609d7c2)}, {LL(0xae37eb9c,0x403c7b8e),LL(0x3d308948,0x374fa1b2),LL(0xd8503866,0x0326fd01),L_(0xd472fecf)}},
   {{LL(0xa5b224c1,0x98673100),LL(0x35c900a8,0x7e857bdd),LL(0xdb5df675,0x63415d28),L_(0x6b676a73)}, {LL(0x04cd81e5,0x3e21afd0),LL(0xa33929eb,0x2e610def),LL(0x717e43a4,0x6bf414b5),L_(0x9c7a3c47)}},
   {{LL(0xd475f817,0xf9ea4384),LL(0x6b738d34,0x671b40ad),LL(0xac317163,0xd197efee),L_(0x2438948a)}, {LL(0x66a989c1,0xebcc3f73),LL(0x1fc5f207,0x7913fae7),LL(0x9fdbdbb6,0xddc39008),L_(0x663a5b4e)}},
   {{LL(0xe781f5a2,0xa670dec4),LL(0xaeb71ef5,0xb1196ff6),LL(0xa4973f38,0xf981cb1a),L_(0xe1b496e0)}, {LL(0xcfa69883,0x8185f4e4),LL(0xdcd9014d,0xfea1489d),LL(0x01c7c910,0x33bfde50),L_(0x0a1d6d42)}},
   {{LL(0x0b6d7bc6,0x0c74186c),LL(0xf93c6e65,0x21c134ed),LL(0x5c4b9d40,0x9b7f55b4),L_(0x62194b5f)}, {LL(0xf0967474,0x07af0f2c),LL(0xead94ac6,0x43b53483),LL(0xfc351f8f,0x6a02d6dc),L_(0x96c34c95)}},
   {{LL(0x8a77d8e2,0x06eb2a9e),LL(0xb29f094a,0x5a50f6db),LL(0x8cc86e58,0x40844e37),L_(0x2f8aef05)}, {LL(0x3ed6ee27,0x864a532f),LL(0x8ade6f97,0xbafa3f05),LL(0x78f8e230,0xadc126ed),L_(0x8c70fd00)}},
   {{LL(0x61f69a9b,0xecdb5c0e),LL(0x3c35a3f6,0xac81a7f8),LL(0xe0f4235e,0x49ad3737),L_(0x162c0cf0)}, {LL(0x9bbaae92,0x6d262e32),LL(0x3611d970,0x92ea2e05),LL(0x56f9b02c,0x569e559b),L_(0xd700ade3)}},
   {{LL(0x1248f104,0xab2f2393),LL(0xf45846e5,0x2e1305dc),LL(0xaa46d2c4,0x1222d842),L_(0xf5f64176)}, {LL(0xfa36ca68,0x1b067c95),LL(0xb8990097,0xbdbafd4f),LL(0xd2f67212,0xe2b50ce3),L_(0x7ac5ce8c)}},
   {{LL(0x2f610f1e,0x754843cd),LL(0x77e241fc,0xeb4f1114),LL(0xa0d9cbeb,0x55cb5b7b),L_(0x98136c68)}, {LL(0x4f9041bc,0xef262223),LL(0x523e661c,0x38002545),LL(0x30661570,0x3e7c7958),L_(0x5ea1ccfb)}},
   {{LL(0x43726846,0x736e6684),LL(0xa762a548,0x4ea61054),LL(0x4ea95a8f,0xa90f1dd7),L_(0xf1a7ecac)}, {LL(0x052ff525,0xdb175bab),LL(0x176f58b3,0x113349c6),LL(0x66d2f2dd,0xf30e4d83),L_(0x047e3e70)}},
   {{LL(0x3053884b,0x2523bfbe),LL(0xdb1ab97e,0x32d988c1),LL(0xa7f43905,0xed30067f),L_(0xe0063138)}, {LL(0x315aac55,0x0fe3afe0),LL(0xef7cd5f9,0x25c62d9c),LL(0xbceebc63,0x7475ff1c),L_(0x37aef5f2)}},
   {{LL(0xd9f33d01,0x21d9dd88),LL(0xfe303f84,0x0e4f3aea),LL(0x403a10c7,0xdc250e4e),L_(0xcf99ada4)}, {LL(0x20afdb06,0x16293016),LL(0xdf77e00f,0x21ef71a7),LL(0x017aea99,0xb51701db),L_(0xb9a51f04)}},
   {{LL(0xb223220b,0xaefa0a3e),LL(0x3c6160d1,0xd7af87c7),LL(0xb6871dd0,0xcb336211),L_(0x30015933)}, {LL(0x605c6481,0x87d9707a),LL(0x18fd4882,0x367d0209),LL(0xf138a307,0xeeb3123d),L_(0xce3f684e)}},
   {{LL(0x1ad5ce5f,0xdb674dfd),LL(0xd843e892,0xc8958516),LL(0xedfd1c19,0xc7988b87),L_(0x3bdd2895)}, {LL(0x9b80b679,0x09bd2af2),LL(0x294ff4e2,0x3150d204),LL(0x2ad4535f,0x2fc2c26d),L_(0x1ebd13d4)}},
   {{LL(0x0284d812,0x0bdfba7b),LL(0x94e42bd5,0xabd7b483),LL(0x512c1be7,0x10be5ddc),L_(0x6bc6d702)}, {LL(0x0bf271cb,0xd21e8820),LL(0x636561c6,0x0b1793f9),LL(0xe4ef4a35,0xda161673),L_(0x3d69a37a)}},
   {{LL(0xf8513215,0xc206f7b0),LL(0xa0f11bca,0x9944badb),LL(0x23eeb3f7,0xe6050ed7),L_(0x98889901)}, {LL(0x2ce1ebe3,0xc76ba3ec),LL(0x4edc3ee6,0x3e534941),LL(0xaab7f4a9,0xc6e7d7b5),L_(0xf6442856)}},
   {{LL(0x6d4746dd,0x5b3638e6),LL(0xd78b290a,0x6b3e0d8f),LL(0x77aef8ca,0xfac422d4),L_(0x0feb906d)}, {LL(0x3bc30209,0xdc0b9001),LL(0x331327a2,0xe68c779f),LL(0x5d146cb4,0x6fb5da21),L_(0x3f887bb5)}},
   {{LL(0x13fd6d90,0xb80d181c),LL(0x2be6d108,0x288db1c9),LL(0xb342ddca,0x7004d85b),L_(0xbe7db7ec)}, {LL(0x0ba4c981,0xc1d565da),LL(0xb82063a3,0xaf5c7b86),LL(0xaa90bf00,0xbf4ed46e),L_(0x3a15f177)}},
   {{LL(0x7f0b547c,0x49eb74f8),LL(0x146e10ea,0x95205278),LL(0x1256c788,0x346ea27d),L_(0x3f761b6a)}, {LL(0x6a1a4872,0x4ec42ca7),LL(0x150f9515,0xec53ab18),LL(0xda268f4d,0xe31f64a6),L_(0xe4706b89)}},
   {{LL(0x331bfc4e,0x0b4a9c48),LL(0xc8ae2f2d,0xa61d3700),LL(0x58cc7b40,0x3e9cb1a4),L_(0x31facd26)}, {LL(0x10a69f57,0x3b34d496),LL(0xd018498f,0xb4f2bc1e),LL(0xcbad3ed9,0xa8867a65),L_(0x0fb7952a)}},
   {{LL(0x060d829d,0xe832b583),LL(0xa4bc1e98,0x376e4266),LL(0x4ca3a485,0x2da2b54a),L_(0xbbcd02db)}, {LL(0xce23e38a,0xd59bf323),LL(0x02c44a56,0x71d50f01),LL(0x366f36f1,0x0dbb411d),L_(0xf30d32a2)}},
   {{LL(0x000bf15a,0x52b2e706),LL(0x19599554,0x83c05edb),LL(0xe133dbc2,0x659be3d5),L_(0x90ee1c4f)}, {LL(0xe38dcd45,0x70cd081f),LL(0xba72fb3b,0xc7e5be8d),LL(0xff5688ad,0x3ae40925),L_(0xb751810a)}},
   {{LL(0x7b159833,0x071a076f),LL(0x01bb4b01,0x67a90721),LL(0xd05cc7e4,0xc3b59e15),L_(0x14e1dc86)}, {LL(0xbb61bd7a,0x1ee66adf),LL(0x743d3ae3,0xdf5f0047),LL(0x75487d37,0xb272a24e),L_(0x18ad2d25)}},
   {{LL(0xecfd1c78,0x69935fe6),LL(0xefec1497,0x0149938a),LL(0xb3e61f40,0x1edeaf36),L_(0x2b4abccf)}, {LL(0xe6be1d5b,0x6554f5f2),LL(0x0fe91c0b,0x443c9e6c),LL(0x613c238b,0xf8ba68b0),L_(0x0b0ae516)}},
   {{LL(0xed42491a,0xbf0bc611),LL(0x9c608ede,0x7040894e),LL(0xc0f244de,0x0b0057eb),L_(0xd601464b)}, {LL(0xf956aa49,0x825a361b),LL(0x4ff78a4b,0xb7556c66),LL(0x6aed48bf,0x1ae6fb3a),L_(0xba8a9318)}},
   {{LL(0x87f39f99,0x09e379b4),LL(0x9e14e886,0x3fce44cd),LL(0x2fb4d3e3,0x75711ae3),L_(0x036fce90)}, {LL(0x85e9ac2b,0x2542f70e),LL(0xa4a903d2,0x8303b7ee),LL(0x95852fe8,0x637b247b),L_(0x012141cc)}},
   {{LL(0x2165af3e,0x9af74288),LL(0x41ab3d86,0x8945cab0),LL(0x29c26ab0,0xcccef77c),L_(0xea5535b0)}, {LL(0x8648e931,0xc11f99b1),LL(0xb7f340c5,0x7dcfe320),LL(0xfc229594,0x8f5c0876),L_(0xf20b5874)}},
   {{LL(0x48b7cd0a,0xba865a55),LL(0x6fae5df7,0xf611d6ed),LL(0x2fe39b65,0x8329316a),L_(0xc07b24cb)}, {LL(0xdf303162,0xd09a26e9),LL(0x1af7d473,0xb6a75d27),LL(0x375af56e,0x62993ad0),L_(0x957236cb)}},
   {{LL(0xd5155d78,0x3e2c48a3),LL(0x427f5e06,0x0f15720c),LL(0x25837ce2,0xcae65a88),L_(0x966e8b1e)}, {LL(0x78a5dd4d,0x9e77f90a),LL(0x306bd895,0xd3918029),LL(0xd4ed09b8,0xa2585522),L_(0x47a8a9ab)}},
   {{LL(0xd18d6ff6,0x1e6ed174),LL(0x8c957fe6,0xfb94845d),LL(0x54818bf4,0x2495a8a9),L_(0xdd977b92)}, {LL(0x5993b230,0x05a5967d),LL(0x2647b9d2,0x3902c0f8),LL(0x53df8e0c,0x496697af),L_(0x5247a639)}},
   {{LL(0x7b7c8590,0x8424e43f),LL(0x682c252e,0xf452918a),LL(0x0bb6bcff,0x17581f41),L_(0x040d50d1)}, {LL(0x91b07886,0xd070dc3a),LL(0x686cbe64,0x9e8c81ee),LL(0x3485fd98,0xdfd534bf),L_(0xf0aec283)}},
   {{LL(0xc3e77fdc,0x4458831d),LL(0xbfa63a77,0xdcd45b1b),LL(0xe6ad6640,0xeffc865e),L_(0xa2922966)}, {LL(0xf55dbb61,0xae34f5fe),LL(0xec39a211,0xcad5ffd6),LL(0x5d5de65c,0x1ffcbe06),L_(0xae549602)}},
   {{LL(0xc0237c46,0xf7d9eb1a),LL(0x008bf1c7,0x578b9779),LL(0xdce3e9ef,0x7341adb2),L_(0xe6aba225)}, {LL(0xbfdcbb1f,0x52473c28),LL(0x58df1458,0xcea6114e),LL(0xcbca23e4,0x64a68f69),L_(0xdc93a2bf)}},
   {{LL(0x2ad64bbf,0x12f4285d),LL(0x289cd434,0xe31b4490),LL(0x2614d391,0x5132c5c4),L_(0xd24f8d84)}, {LL(0x0a9c7374,0xb496fb6f),LL(0x945d4d6b,0xaa619fb0),LL(0x64d58ea0,0x226ba20e),L_(0x78fadda9)}},
   {{LL(0x328976b5,0x15c734ef),LL(0x71f5ab8b,0x49d5ba93),LL(0xd644b74d,0x8a5a08f7),L_(0x5b9eaf3f)}, {LL(0xcd33e0a1,0x40783f59),LL(0xfc35396c,0x91276e7f),LL(0xdf4e44eb,0x698ace20),L_(0x85de5335)}},
   {{LL(0xbe2724e6,0x4ca6d828),LL(0xc01805fa,0xab7f4b0a),LL(0x9b659d62,0xb0bfec7f),L_(0x21ba53df)}, {LL(0xcb83946a,0x3ca3ffdb),LL(0xaaee16f0,0xe089ddef),LL(0x2829c022,0x8266c12f),L_(0x570a8b63)}},
   {{LL(0x7f62cdfb,0x27c7435e),LL(0xc2862c3c,0x56a78e59),LL(0x577136cc,0xeb454a5e),L_(0x41bf3bc1)}, {LL(0x76737cb9,0xd5ff2a5c),LL(0xce2b3472,0xa9a9af2c),LL(0xabfed878,0xdae3ae7c),L_(0x076254af)}},
   {{LL(0x8125e260,0xfbee92e0),LL(0x288eeccf,0xeb8bf90b),LL(0x9cfb5c57,0x059c298f),L_(0x53598af7)}, {LL(0xc5e99b56,0x888186ae),LL(0x3084705f,0x696f873d),LL(0x86f1d164,0xc59e084e),L_(0xa44fd074)}},
   {{LL(0xeb77883e,0xd2343b34),LL(0xa40a6bf8,0xc2267f63),LL(0x798ceee2,0x27b865e6),L_(0xe4570eb1)}, {LL(0x60596c23,0x8f87d684),LL(0xf0ce5967,0x06984779),LL(0x66b1f5c5,0xcbcc053f),L_(0x2f8869b5)}},
   {{LL(0x7620dc06,0x0d7b69de),LL(0x17f0c4dc,0x9d39bfa6),LL(0xa20e8614,0xabbfb492),L_(0xc7dc6a2c)}, {LL(0xe61a635a,0x2759d405),LL(0xcc56f093,0x966b6ab3),LL(0x049b5aa5,0x4e1f566b),L_(0xbec87c76)}},
   {{LL(0x57acbf9f,0x40e840f5),LL(0x1f1c6ca8,0x0ffd84c8),LL(0x2d21e27b,0x9ca6123e),L_(0x1d5e2c46)}, {LL(0xda38f612,0x5531352e),LL(0x009d49a6,0x3be34fe2),LL(0x43dffb38,0xd020075d),L_(0x2f8a3d38)}},
   {{LL(0x89bc38a6,0x5ded0d34),LL(0xd70c960d,0x82054d24),LL(0x08b78436,0xbcbebb9e),L_(0x662ab949)}, {LL(0xea6945f0,0x1d7a378c),LL(0xcd6abb3b,0x2b017a9f),LL(0x991364b4,0x4dbff4b2),L_(0x6e088465)}},
   {{LL(0xca476359,0xf7fff479),LL(0x8b812163,0xe17baaee),LL(0xad939c3e,0xd700d28b),L_(0xe8837ac3)}, {LL(0xbb3a964c,0x26712ef2),LL(0xd89edbf6,0xe0390b08),LL(0xeaa73bdc,0x536ef396),L_(0x19770554)}},
   {{LL(0xb20e2aec,0x7f3d231f),LL(0xe904178b,0xa6924ffd),LL(0x7ff297bf,0xad6fdd69),L_(0x6deef5b5)}, {LL(0x6c6f7f42,0xda650bfd),LL(0xc9fb4677,0x702cfe74),LL(0xd99a935f,0x93d1a532),L_(0xbfdd1320)}},
   {{LL(0x49e4811b,0x0e07db7d),LL(0xfed77923,0x0dbb53c0),LL(0x0f05670c,0x112a7ade),L_(0x6e79da66)}, {LL(0x3436a7c3,0x201e8aa2),LL(0x567d0dd8,0xd3bae1e2),LL(0x9a0b34dd,0x6a730912),L_(0x5c656713)}},
   {{LL(0x8ee826b6,0x9522c2ac),LL(0x1cf99a2e,0x6bbbf4b2),LL(0xbfb9ae13,0x1fe851c4),L_(0x91215bb4)}, {LL(0x8e9f0b90,0xbf89747c),LL(0x6286cdba,0x0e20c83e),LL(0x792aeb4f,0x25b00fa8),L_(0x61f48d5b)}},
   {{LL(0x5762ff25,0xa1ce4a2b),LL(0xbe45da68,0xe3b7ba58),LL(0x63ee425c,0x98b17978),L_(0xd26e1784)}, {LL(0x8832d22b,0x5fcc8cf0),LL(0xecb80c24,0x54bb743c),LL(0x3e21c95d,0x1df7f32b),L_(0xb9cc6c4a)}},
   {{LL(0xc522b190,0x61766f4c),LL(0xa5778295,0x2fc3322e),LL(0x34307914,0x57f4defb),L_(0x71c105c8)}, {LL(0xe02ecefc,0x2096768e),LL(0xce51bf21,0xad887721),LL(0x3117821e,0xf475048e),L_(0x2b073db2)}},
   {{LL(0xa7c80b61,0x7067894f),LL(0x4426eaf6,0x99c9ba1c),LL(0xd1c53be2,0x68f854e3),L_(0xce0157db)}, {LL(0x5b23fe91,0x5112a49d),LL(0xa0471f83,0x97b550cf),LL(0x87e0a2d1,0x9f64998d),L_(0x344b223b)}},
   {{LL(0xb77380be,0x3ad984a7),LL(0x3bd4bc94,0x216f1748),LL(0xc17ca930,0x4aee7a72),L_(0xfcd30768)}, {LL(0x02746210,0xa1f41f76),LL(0x6fdb5587,0x670f4013),LL(0x3c2763bb,0x0145b604),L_(0x6e18b2a2)}},
},
/* digit=16 base_pwr=2^112 */
{
   {{LL(0xca8e1890,0xa72d343b),LL(0x006aff79,0x07464235),LL(0x1ec739c9,0x136d5ba5),L_(0xce9ce64b)}, {LL(0x721071fb,0xb11ea220),LL(0x7bfe943d,0xfb7789dc),LL(0x57dad905,0xac86bd3f),L_(0x5eeba8a6)}},
   {{LL(0xac243d83,0xb0b49013),LL(0x31f50c04,0x6bbaa790),LL(0xd8fab554,0x1d543cec),L_(0x2fb0a285)}, {LL(0x15602d83,0x3256cf69),LL(0xf3bd35dd,0x4a5d9d40),LL(0x8a264f24,0x7dff3a00),L_(0x24830923)}},
   {{LL(0x4400cf66,0x3267fa61),LL(0x7dfb1e52,0xfe343741),LL(0x34e86055,0x552bcdda),L_(0x131f5b56)}, {LL(0x6065029d,0x40256e8c),LL(0x61212776,0x6fac4167),LL(0x3807bd7c,0xbba74ace),L_(0xa2ce6ab3)}},
   {{LL(0x954e93a7,0xfd51954f),LL(0xd0706483,0xbe73a498),LL(0x89215ea5,0x45e8a87b),L_(0xf3cf4080)}, {LL(0xcdccb303,0x85c8552d),LL(0x09212e02,0xdb8ab951),LL(0x10c2badf,0x19c8db39),L_(0x7b08661c)}},
   {{LL(0xdafc0f6f,0x9e15fa46),LL(0x148c6f0e,0x89a68d97),LL(0xdb00031e,0x4cc0f2bb),L_(0xc3df9d3a)}, {LL(0x340e9460,0x1ff47bb7),LL(0x620b581f,0x7ef8908d),LL(0x57588030,0x4277ac9d),L_(0xfc122be2)}},
   {{LL(0x5631310d,0x1336d2ba),LL(0xcd5bed69,0x039396d8),LL(0xedd85963,0xac1f77dc),L_(0xea62aa43)}, {LL(0x40cf73fd,0xf970f801),LL(0x25fc53f3,0x7927be46),LL(0xf9d1654e,0x9f9753cc),L_(0xb61c3720)}},
   {{LL(0x1f400c41,0x96dca6f7),LL(0x6461f973,0x29585547),LL(0x1f57a9e0,0xb90ca0df),L_(0xb010f7d3)}, {LL(0xd1cf0605,0x21188094),LL(0xf21e0d4d,0xcfd8c7ab),LL(0x5f3468a5,0xde0d0f24),L_(0x4e99f334)}},
   {{LL(0x0e472341,0x70d9012f),LL(0x7c2bf741,0x876213ff),LL(0x47c7e115,0xb309c07a),L_(0x977513df)}, {LL(0x7682fd32,0xbe1ac3e8),LL(0xb7e947d5,0x1c6b708a),LL(0x09c3181e,0x41e0c220),L_(0x8b26dfc1)}},
   {{LL(0x4c02bbfb,0xfa332320),LL(0xbb0f1fc0,0x1dd657b9),LL(0x68034aed,0x784ebda5),L_(0x58fb05ab)}, {LL(0x0677d37f,0x32cca1fc),LL(0x83e2269c,0x2c9407c6),LL(0xea0312b9,0x33fa100f),L_(0x1761c87e)}},
   {{LL(0x40a8cd2a,0xf4f00cb5),LL(0x58c8e52b,0x05e2b3e0),LL(0x99489e99,0x5593b9ee),L_(0x7924f02b)}, {LL(0x464ae656,0x0334e2ed),LL(0x2a2bf206,0x55ec8d2c),LL(0x41ac277d,0xa7b08209),L_(0x0c5c176d)}},
   {{LL(0x3ac595c7,0x1aabcb3f),LL(0x1450434d,0x93706f44),LL(0xcf82a49f,0x0e6c721c),L_(0xec192907)}, {LL(0x92676891,0xfecc2ffd),LL(0x451e72f3,0xc8f49751),LL(0x81bf062a,0xe8852ddc),L_(0xf84db443)}},
   {{LL(0xb1dfeb18,0x32a2e886),LL(0x8b6ac261,0x074b9898),LL(0x3c40d7a6,0x1ea4ca4c),L_(0xf7653f1d)}, {LL(0x6940eab6,0x2b2347ff),LL(0x56338adb,0xca0ffd59),LL(0x314b0e58,0x8e442705),L_(0x79527139)}},
   {{LL(0xbe4dbd9f,0xd63c1700),LL(0x1101a669,0x9b252cda),LL(0xd06390a9,0x501cc189),L_(0x0fb428cd)}, {LL(0x8dcdeb0d,0x0a3108aa),LL(0x47aa9e0e,0x71a44ce5),LL(0x9c9d8b17,0xe86804cc),L_(0x61ad5482)}},
   {{LL(0xfe987606,0x4432954f),LL(0xebe782ea,0x4d668855),LL(0x11e7dc71,0xf9379d6a),L_(0xfb5f9f2d)}, {LL(0xebb7e729,0x98c8071e),LL(0x3aaef819,0x23ed5017),LL(0x3d2ba5a3,0x0cd97039),L_(0x2ceebe22)}},
   {{LL(0xf7bd7afe,0x6646c443),LL(0xc3bd60cf,0x242cedce),LL(0x7c4048fa,0x5e74dbe8),L_(0xd29b1f65)}, {LL(0xc5c513c7,0x91eb1130),LL(0xbb7b08f7,0x9edbdadd),LL(0x3c5f214f,0x6e37d0fb),L_(0x85ea7863)}},
   {{LL(0x46abff2c,0x87af66ad),LL(0x26b2f70c,0x6dfaf224),LL(0x5494f6a8,0xf84408cd),L_(0xa24b6575)}, {LL(0xb3e6fa13,0xfb2baa35),LL(0x7fdee2e4,0x48b82a51),LL(0x82709ae1,0xd231d3ab),L_(0x8897d9e8)}},
   {{LL(0x9b54a957,0x43e3a795),LL(0xc93592ff,0x7df94193),LL(0x9aebe35d,0x3c64092f),L_(0x1d7432bf)}, {LL(0xd212a26f,0x608467be),LL(0x61dd14ba,0x92d589f9),LL(0x1c6431e0,0xb4aaeed1),L_(0x53076cd4)}},
   {{LL(0x20a03c9c,0x064ac09e),LL(0x3e9b39b0,0xe7dae697),LL(0xd8bc5839,0xc348b809),L_(0x8ab0974b)}, {LL(0xcf8f0968,0xf886797d),LL(0x9cb572eb,0xeaa08b56),LL(0x9a9ce85b,0xa4a7eaf7),L_(0x0e1605e0)}},
   {{LL(0x7cf927b4,0xbe868d7f),LL(0xeca36ddb,0xd5b365e4),LL(0x86bfcb64,0x9b8656b8),L_(0x2ab84426)}, {LL(0x82271896,0x01fe86cb),LL(0x15f75f0e,0x672910dd),LL(0x4c2d70f5,0xf239468a),L_(0x365ce9d3)}},
   {{LL(0xfbc74fe2,0xd342696f),LL(0xe0e930b3,0x1de7ca12),LL(0xcb67c28f,0x163565e8),L_(0x312a9e2c)}, {LL(0x197ec1e3,0x54d76d68),LL(0x3b508871,0x11c4ebab),LL(0x71ee55a2,0x4dfa9969),L_(0x0a025c8a)}},
   {{LL(0xe7055396,0xc327e039),LL(0xf0b0b5d0,0x5f514a6a),LL(0xb462d687,0xa70b2c73),L_(0x842f549f)}, {LL(0x161da539,0x864e3870),LL(0x0b583d9c,0xde80546e),LL(0x1693a8f3,0x5135df56),L_(0xadecd5ac)}},
   {{LL(0xc575df9d,0xe8f3d99a),LL(0x04c439d1,0xbba7ca9c),LL(0x79057450,0xed15dbf2),L_(0x557c06cf)}, {LL(0x7e76a8b5,0x09b41a1e),LL(0x0bdc4dcf,0xf356b834),LL(0xf085eab4,0xdab0c931),L_(0xb48d6df1)}},
   {{LL(0x2605d170,0x4a27c44b),LL(0x2e80ab05,0x5425f3dc),LL(0xdd9e1376,0x242b815a),L_(0xa910e496)}, {LL(0x8b98999a,0x3f111c1c),LL(0x0c5343f8,0x29db22da),LL(0x10be67ce,0xf1c6ef62),L_(0x4ae541a2)}},
   {{LL(0xbec6e451,0xc5669a27),LL(0xff0e3e5f,0xabcdf52c),LL(0xe911fae7,0x85cd9be2),L_(0x4150b30c)}, {LL(0x9e505adc,0xab441ae6),LL(0x92888eaf,0x3ad89345),LL(0x0dd62a7e,0x1b7b37eb),L_(0xe9e642a8)}},
   {{LL(0xc78817e4,0x15d73eae),LL(0x70f4b3ee,0x4965b1f3),LL(0x70ff29ef,0xe8787785),L_(0xe5dce106)}, {LL(0x6012d2ef,0x36b7ab1a),LL(0x21a140c0,0x943b6ec7),LL(0xffda816f,0x9bd299e0),L_(0x5968ca97)}},
   {{LL(0x6cb8a41d,0xb0d9d8cc),LL(0x6b2cc953,0xa58bcb0c),LL(0x932fd885,0x8fc62139),L_(0xb782380e)}, {LL(0x5794203a,0x83692f14),LL(0xb2525747,0x8e1b841e),LL(0x2213b2d6,0xc34bb49f),L_(0x0270ba84)}},
   {{LL(0x4fa6af14,0x3b925a7e),LL(0xe649f779,0x23113463),LL(0xa3785af0,0x20ce9dd7),L_(0xdc5dbf9f)}, {LL(0x3c0ce37f,0xa5d677f4),LL(0xcdc8d1f8,0x793a5749),LL(0xd8d3e9cd,0xb698eb06),L_(0x5cbd0981)}},
   {{LL(0xfe510f94,0x1e752015),LL(0xf090a1eb,0x65fa8e80),LL(0xfc5f6857,0x111a1741),L_(0xbaf89d9f)}, {LL(0x4cc335c5,0x1b1a1910),LL(0xbfa9551b,0x24829f96),LL(0x24a70dba,0xc5ef4f06),L_(0x17130b26)}},
   {{LL(0x1febf14a,0xfcb5b042),LL(0x911935f4,0x2a9fcf44),LL(0x53ed6fc0,0x90634826),L_(0x4421e383)}, {LL(0x98983bff,0xc4143e7a),LL(0x12ccf145,0xfc4163c9),LL(0x5bf8b6c9,0x4f765081),L_(0x73883a6d)}},
   {{LL(0xef044be0,0x997c821a),LL(0xe367cf8a,0x3560d6df),LL(0x3d0455b7,0xe8c7bcd6),L_(0x92643ab4)}, {LL(0x26239fb1,0x547dbeb1),LL(0xb5294491,0x2538d97d),LL(0x07450e45,0x472a47f1),L_(0x5e3d3784)}},
   {{LL(0x23d0e3a3,0x6372566a),LL(0x4d62de9c,0x9858907c),LL(0x7a323978,0x78458eba),L_(0x4b2a04ff)}, {LL(0xae2094af,0x6819a84c),LL(0x0eccbf7e,0xe0c337be),LL(0xb93eec58,0x2a02a086),L_(0x03b76658)}},
   {{LL(0xbfc94c0f,0x6aa0f607),LL(0x0cbee27c,0x4123775a),LL(0xae1d1601,0xbbef8dbb),L_(0xe7149786)}, {LL(0xf56932af,0xbe686d03),LL(0x91ce3c48,0x3e33d4ba),LL(0x3d6535b6,0xa72c8f85),L_(0x2c9fb75e)}},
   {{LL(0x4a47050e,0x39dd7c74),LL(0xeebc8c18,0x2f4a6be4),LL(0xa2f0fb3a,0xba0f68a4),L_(0x1c5cad2c)}, {LL(0x85d6ef4d,0x3f59abfa),LL(0x3a302769,0x47cf5f7e),LL(0xed44c056,0xcd7fd1b9),L_(0x2e974b80)}},
   {{LL(0x691eccfe,0xc8f85c87),LL(0xf503e456,0x34a4ce86),LL(0xb53e74a3,0x8eb76194),L_(0xc4c55423)}, {LL(0x22da99fd,0x153fefba),LL(0xdbbf4e12,0x5438e1b0),LL(0x80b3f694,0x65187422),L_(0x9f8c212c)}},
   {{LL(0xab8f55ff,0x0fcf5a38),LL(0x6da81ce6,0x0b057c12),LL(0xdff4777f,0x4a9fa32b),L_(0x602640c6)}, {LL(0x2624de9e,0xa3b241c6),LL(0x2a545f4f,0x0dd2feaa),LL(0x4cf94266,0x8a894ff1),L_(0xb5260296)}},
   {{LL(0x2194d8b6,0x7320211c),LL(0x391c4f62,0xd2de2551),LL(0x6d039386,0xbaa68de6),L_(0xc3a5b977)}, {LL(0xf3b1162b,0xbca0bb5c),LL(0x3643a502,0xd1a3ad93),LL(0xe1f5b66b,0xa430cadc),L_(0xb32e7ba4)}},
   {{LL(0x4c8d2ee8,0xd36efcf0),LL(0xb6b2c682,0xc22137cd),LL(0x1024c516,0x8688f303),L_(0x99e728ff)}, {LL(0x3fb84765,0x777d2fde),LL(0xd972734f,0x9a00c6af),LL(0x25251454,0xf34097fd),L_(0x424404cf)}},
   {{LL(0xdfdcead3,0x3079239d),LL(0xb5ff46f3,0x80c77858),LL(0x36a3735f,0xc9986723),L_(0xd572091f)}, {LL(0x6645ed98,0x0dabffd5),LL(0x0da45042,0x4b0c6093),LL(0xf6ffaf50,0xbffef0a3),L_(0xbc330b9d)}},
   {{LL(0x7d149925,0x3b961737),LL(0x8d236f27,0xc18d27e0),LL(0xa5f014ae,0xee0ae6b2),L_(0xe50d6b9c)}, {LL(0xc1b92639,0x2b5f008e),LL(0x36e3aea7,0x8996a3a6),LL(0x2f8e4e4e,0x5ccc02df),L_(0x9f683f7c)}},
   {{LL(0x3a45a099,0x7caa8790),LL(0x6c2cdda1,0xf82f4e65),LL(0xa9cdf2c5,0xb634f00e),L_(0xd5eec010)}, {LL(0x4012fecf,0xa6a18c31),LL(0xde736bf0,0x153b6c27),LL(0xc1059186,0xb7a29e2b),L_(0x9b23923d)}},
   {{LL(0xc90ac0c7,0x7000c5a2),LL(0xb80400e7,0x2f6fc4cf),LL(0x147ccc10,0x9e3b070e),L_(0xfef094f1)}, {LL(0x0ab36cd7,0xefcc38e0),LL(0xf5b77a62,0xa70d4a46),LL(0x2c523e42,0x02efb3f7),L_(0x210639da)}},
   {{LL(0xda4106f5,0x330449c9),LL(0x7b06f635,0x30d85516),LL(0x9efabcec,0xe879f10c),L_(0x3bd206ba)}, {LL(0x75c081fb,0x0312bdf3),LL(0x88e64b25,0xb372be76),LL(0x3a8f90eb,0x3e832f84),L_(0x14f38ac0)}},
   {{LL(0xc7331aa0,0x4bafc9dc),LL(0xb6461cb6,0xae9ec59c),LL(0x266ba391,0xf40aa6ef),L_(0x4715355a)}, {LL(0x33ae64ed,0x1adab377),LL(0x716f224c,0x04c2b6cd),LL(0x32b71c3a,0x7e89026d),L_(0xccbd7f4e)}},
   {{LL(0x095925d5,0x7de43130),LL(0x0bbfe1df,0xa8605297),LL(0x10701822,0x0bc2e01c),L_(0xf68b86eb)}, {LL(0x816f32a8,0xad3f4af4),LL(0x67e192cc,0xe9dc6c49),LL(0xd5fef609,0x52b649b2),L_(0xf428af30)}},
   {{LL(0x9f45ede9,0xe78176a5),LL(0x0b536242,0x0ba30aad),LL(0xdbdd8b20,0x503b274a),L_(0x51151bfa)}, {LL(0x4d7ac01b,0x393d4fa5),LL(0xdc4e6926,0x8e9d7c61),LL(0xe68387c4,0xaaf0a35e),L_(0x06168aa4)}},
   {{LL(0x35b79ca0,0x2143ebc0),LL(0x4be98451,0x85664008),LL(0x1016322a,0x5db7c140),L_(0x8229e729)}, {LL(0x4e10aefd,0xc19bdf57),LL(0xc8df52e4,0x07fc2589),LL(0x89b605d1,0x2a200cdd),L_(0x0c94090c)}},
   {{LL(0x02211b9e,0x4588f75f),LL(0x0c217b21,0x90c9218a),LL(0x90f7c009,0xef37649d),L_(0xddfaf799)}, {LL(0x35885f85,0x3d0294b0),LL(0x510fffdd,0xe646d7e9),LL(0x4a21ba9d,0x6c9d2cfa),L_(0xa7ada78e)}},
   {{LL(0xcf1bb365,0x1db377d9),LL(0x63e01d3e,0x8b74ddb8),LL(0x8c9ad4b9,0x6bb31b8b),L_(0x2b26fec2)}, {LL(0x858449ed,0xfeaf9d41),LL(0xe2652b65,0x700223fb),LL(0x03b48402,0xee2505a9),L_(0x2438e370)}},
   {{LL(0x9e8ea415,0x900098e5),LL(0xd8a6e8e7,0xe478cc9a),LL(0xd8452a93,0x2e2cd25b),L_(0x569bbc5f)}, {LL(0x5d22bbf4,0x80e8caa9),LL(0x3f76b251,0x10dc6eaf),LL(0xeba89ffb,0x18acb601),L_(0xf763dfc0)}},
   {{LL(0xfed0a316,0x24370f95),LL(0xf5f4a36c,0xc06ba373),LL(0x527f5322,0x43417948),L_(0xe0b12bf9)}, {LL(0xa7bf2afb,0x6aa04127),LL(0xb3183162,0x87eaa68e),LL(0x1070f9c1,0xe002ccee),L_(0x5b0c2b4a)}},
   {{LL(0xc7a653c7,0xfca649b9),LL(0xb223ef3b,0xf66cb526),LL(0x4aa0b0e6,0x10592f0a),L_(0x3d9b366f)}, {LL(0x61473660,0xab61a041),LL(0xaa721fec,0x1b173f43),LL(0x0a96a721,0x06be6fd2),L_(0xb3b3cccc)}},
   {{LL(0x87a37531,0xfda2de36),LL(0xd8576cc3,0x94a1bbc9),LL(0x22b5be7f,0xaad70136),L_(0xc40c9235)}, {LL(0xfb07dd6f,0x9edbb744),LL(0x501faa81,0xccf5dbd3),LL(0x60c831a6,0x6f69ca6e),L_(0x813a83b7)}},
   {{LL(0xa96bdf09,0xdbe3e245),LL(0x4f871623,0xacfa6c70),LL(0xd2c645ee,0x18c48541),L_(0xc037f50e)}, {LL(0xcb36c731,0x08342ef7),LL(0x68cc0cb0,0x4e89abf4),LL(0x0c4fa826,0x186c66ac),L_(0x8f28f4b4)}},
   {{LL(0x2d46a77a,0x837493ce),LL(0xf41079e1,0x15f6dc1a),LL(0x1a861704,0x0e538567),L_(0x03e474ae)}, {LL(0xa4564b51,0xa188c7ea),LL(0x76610ad6,0x3c36716c),LL(0x706eb273,0x8025f126),L_(0xb5435d24)}},
   {{LL(0x28638fa1,0xc4faeda2),LL(0xd9ddd6dd,0xcb83b696),LL(0xe2be7ac0,0x4eea6f06),L_(0x2e6abaae)}, {LL(0xaad7f53d,0x99f8dc45),LL(0xc8502a73,0x8477e1b0),LL(0xa0aa0199,0xcbc5ed2a),L_(0x38689f49)}},
   {{LL(0xd7981750,0xfbd4773e),LL(0xb3548b14,0x01d933bf),LL(0x18cea1f5,0xb3c65f79),L_(0x3bc435fc)}, {LL(0x3414a439,0x0706d138),LL(0xca5b9f90,0xb2166606),LL(0x14b87e77,0x864343f5),L_(0x8a82e1fd)}},
   {{LL(0xa2f61a19,0x18cc1c82),LL(0x0026972b,0x0e88aaf9),LL(0x1da022c6,0x2265f4c6),L_(0xb898d6b7)}, {LL(0xc0e9ecae,0xa53eabdc),LL(0xef23c0bd,0xd1e853a6),LL(0x98e844f5,0x05f9b046),L_(0x7d2e87fd)}},
   {{LL(0xf9a4fe94,0xe7c2033d),LL(0x1c9b4b3f,0x3037306e),LL(0xf61d0ee2,0xdb5834bf),L_(0x28b5943b)}, {LL(0x4c22b3ea,0x5de6ece6),LL(0xf70c350f,0xa4e1deaa),LL(0x1c3bfb3c,0xd4c8be79),L_(0xe75564bc)}},
   {{LL(0x16c64cef,0xab2cc7d7),LL(0xa4b41a9f,0xa5534f63),LL(0x912b7248,0xe6362cfb),L_(0x6a214d2e)}, {LL(0x9c7005e2,0xc25479e5),LL(0xf1652743,0x6a7f2578),LL(0x3bcb10e6,0x9be1e8e0),L_(0xbede8ea6)}},
   {{LL(0x459cf3fc,0x31630308),LL(0x706d16a7,0xf60bf227),LL(0x9f2f55c9,0xed0db8c6),L_(0xaf0d2787)}, {LL(0x6dd2c868,0xb9e7f043),LL(0xce8001ee,0xa2ee914c),LL(0xb8247bef,0x658e5458),L_(0x78f60572)}},
   {{LL(0x6bc3d716,0x5c6843cd),LL(0xacaada1c,0x1bd0e72e),LL(0xcabaa3a2,0x36370b66),L_(0x507a17a3)}, {LL(0xebae3c4d,0x481b026d),LL(0x48821bb8,0x4cd76456),LL(0x5ed8e5b9,0x78f46113),L_(0x6817009d)}},
   {{LL(0x029887e4,0x7560c538),LL(0x75119a00,0x92989cd8),LL(0xe00d03bb,0xef71ab39),L_(0xb863df63)}, {LL(0x5f6dd06c,0xfedea07c),LL(0xddc4f0d5,0xd3b8a96e),LL(0xc43e8f08,0x36d08634),L_(0x9efb1f02)}},
   {{LL(0x3215c001,0xba05b2e5),LL(0x949d9990,0xa3503f0e),LL(0x76e4e629,0x6c281fa8),L_(0xdccad172)}, {LL(0xbcf0bc10,0x6b13eba6),LL(0x70f4d75c,0x1b941283),LL(0xd34756fd,0x70986d34),L_(0x8f71b0bd)}},
   {{LL(0x2576d875,0xe10a7afc),LL(0x0a9b3e69,0x411dab1f),LL(0xf0719c41,0x046cc610),L_(0x9d1611b2)}, {LL(0x1ba334cc,0x6b7a1ff0),LL(0x24d9cfba,0x47e8a2e5),LL(0x5bd30854,0xa740e80a),L_(0x1159e67f)}},
},
/* digit=17 base_pwr=2^119 */
{
   {{LL(0x7c2ca549,0x5898fc38),LL(0xa3566efe,0xa757b52e),LL(0x24bb02bf,0x91ab4316),L_(0x00e46ee0)}, {LL(0x9ec42857,0x8c02881c),LL(0x1d20ad25,0xb9a28540),LL(0xb01ce79f,0x40d6ae86),L_(0x40b2c06e)}},
   {{LL(0x4fd7f402,0x259779c5),LL(0xafc1d993,0x3aa4fd10),LL(0x86dcdc83,0x77170d32),L_(0x8441f4f3)}, {LL(0x9892b126,0x46144a81),LL(0x58df739b,0xc6549963),LL(0x344ddb0c,0x01761502),L_(0xd9bcbeb6)}},
   {{LL(0xaf8e61b1,0x66de5a2d),LL(0xbc92b6dd,0xcb8a0bac),LL(0x1127f258,0x3d18ed03),L_(0x3982ee92)}, {LL(0x8ca9b244,0x0deb6e5b),LL(0xcf198f32,0x7cefc1f3),LL(0xdbaf1a5d,0x373c1801),L_(0xf524b8bb)}},
   {{LL(0x746fcfe2,0x1bb01e23),LL(0xdc5e2b63,0x38352329),LL(0x5df11c5b,0x68e597b5),L_(0x44ec319c)}, {LL(0x84b36d32,0x0e27f5b0),LL(0xe5706d53,0x06c3753d),LL(0xe67b3281,0xa7111b5d),L_(0x8fa5ee02)}},
   {{LL(0x5141455b,0xe429fedc),LL(0x0035c84e,0x353b3d90),LL(0x2ea6f135,0xa3e74746),L_(0xb9e352c0)}, {LL(0xa8f82237,0x220bb555),LL(0xb02520ea,0xecc581df),LL(0xe4b94f8e,0x16dbb87f),L_(0xfbff49d6)}},
   {{LL(0xc2eb5b9f,0xda91cf48),LL(0xdb33cda7,0x821b171b),LL(0x006b3e24,0x489cbfa3),L_(0x4461fe49)}, {LL(0xc4e7389d,0x4d93f4e1),LL(0xfe8e5310,0xf28f63e5),LL(0xf3d0d29d,0x94deaf7c),L_(0x3d33e2ad)}},
   {{LL(0x1e948480,0x453d6838),LL(0x8f1b71e2,0xd842ed96),LL(0x92302500,0xf76bc681),L_(0x17e6ad54)}, {LL(0x5d04302d,0x66dcecf9),LL(0xe97810bb,0x37c278e5),LL(0x13de9f06,0x30c9e7f2),L_(0xf16b622f)}},
   {{LL(0xe415829a,0x436d6b94),LL(0x794d49e6,0x541c7bda),LL(0x702ede17,0x5afaacb1),L_(0x3b5ca68c)}, {LL(0xc9496ea8,0x9ca7b73f),LL(0x50a1e66e,0xc5ce627a),LL(0x762e4f6f,0xc90185b1),L_(0x6705418b)}},
   {{LL(0x52188496,0x52070a5e),LL(0x9065afef,0x510e288b),LL(0xe6df0c7e,0x441709b3),L_(0xa1396423)}, {LL(0xa8018a37,0x66868b26),LL(0x14f84f84,0x28ce78c1),LL(0x1fa90793,0x19c083f9),L_(0xfd8d88ee)}},
   {{LL(0x2458c92b,0x5330a979),LL(0x77ea9ee9,0xdba786fa),LL(0xce00962f,0xae5f476a),L_(0x05915824)}, {LL(0x1219b9cd,0x8cdf0300),LL(0x660a19e4,0xdc707120),LL(0xc70e0b11,0xfdd0563d),L_(0x41d445a6)}},
   {{LL(0x6102cfd1,0x53d54a0a),LL(0xa5729cf8,0x4fc590fa),LL(0xea907455,0xea0c5a91),L_(0x59a871ce)}, {LL(0xb4020096,0xe025ffb0),LL(0x7e7fb9d6,0x140025ac),LL(0xb7954f0e,0x46ca7ee6),L_(0x010dbf77)}},
   {{LL(0x6a28fa6b,0x2d7e46eb),LL(0x034b69ed,0x1ae23996),LL(0x51bc6b67,0x3c25755a),L_(0xe9bf6717)}, {LL(0x579d75fc,0x06b03b1f),LL(0xfc3e6ce3,0xe0b2026f),LL(0x723756d9,0x58938357),L_(0xb4731ae9)}},
   {{LL(0x5cbab1e2,0x42ddfb70),LL(0x33bbd9fd,0x6d346512),LL(0xeb7aa22f,0x725fdf2d),L_(0x88eb52e0)}, {LL(0x48e63519,0xf1fc169f),LL(0x297a7972,0x91aa8dc7),LL(0xb92c9671,0x9c2b4849),L_(0xa557aadd)}},
   {{LL(0x4fe2181c,0x58182a19),LL(0x27b6bd7a,0x0c4b58a3),LL(0x1c72e55a,0xd4f2f098),L_(0x6b53e542)}, {LL(0x0bdee531,0xb5e11022),LL(0xa8f7b191,0xa8bb8315),LL(0xe62460d4,0xbdf84150),L_(0x74ea071f)}},
   {{LL(0x1ca83a09,0xe03c9ae4),LL(0x54fef132,0x7e2ef688),LL(0xa900e0bd,0x1a602274),L_(0x3698f6bb)}, {LL(0xc6654794,0x57e365a9),LL(0xbb500025,0x2b39b18b),LL(0xf16e71f6,0xa0dfe4bb),L_(0x9d6899d1)}},
   {{LL(0x1d2f288c,0x8e577078),LL(0x8e833532,0x8ebb1ef6),LL(0xfb9fbcd0,0xf5ead3c9),L_(0xe31f0c20)}, {LL(0xf16fa294,0xc462ce8b),LL(0x6466c045,0xc1068645),LL(0x85e7447b,0x3e1abc03),L_(0xbba798a0)}},
   {{LL(0x41e55979,0x122d19ea),LL(0x158f4312,0xfd5b3495),LL(0x297c5747,0xa7b750fd),L_(0x735aa3f7)}, {LL(0xd2156123,0x312d6128),LL(0x31e8bfa3,0x4456ccf3),LL(0xcc9cdadc,0x5094cff5),L_(0x9ea9beca)}},
   {{LL(0x1ea96169,0x616ee979),LL(0x78257a1a,0x60b3f06d),LL(0xec98f98d,0x609d2d2d),L_(0xd1b3e514)}, {LL(0x1100f9ce,0x2e59e232),LL(0x8087f780,0x34e1a9a4),LL(0x0b40b853,0x95c2865d),L_(0x8fa97627)}},
   {{LL(0x297f6423,0x1eac58fa),LL(0x7eb748d9,0x6fa36840),LL(0x0d0b673e,0x955ae146),L_(0x72be4e8e)}, {LL(0xb20412e2,0x22b08d06),LL(0xa4616bf8,0x57a2dcba),LL(0xc55b68a6,0x42138fbd),L_(0x8fa529ef)}},
   {{LL(0xdb26dfa2,0xf1766b07),LL(0xdfd72d39,0x050e85e5),LL(0xad6b741f,0x59466485),L_(0x5fceaf74)}, {LL(0xbc9edaeb,0x0c6b2254),LL(0x453c9eae,0xfca1477c),LL(0x9c00d050,0xa3817166),L_(0xa87cf553)}},
   {{LL(0xdbbc12a2,0xad146ff5),LL(0x39990110,0x33dcec4a),LL(0x3d48a808,0xfe9fea01),L_(0x8c4aac12)}, {LL(0x2cefb34a,0x0ad5c79b),LL(0x0a9a20f7,0xaca81076),LL(0x942b38be,0x9475c474),L_(0x218031b5)}},
   {{LL(0xc173f9cf,0x4c0dcf94),LL(0xcc4c491e,0x61d12afb),LL(0xfa790f55,0x12a6c289),L_(0x37abf7be)}, {LL(0xd04d2ae3,0xcb586fd2),LL(0x192166df,0x3c04c5b4),LL(0x7ee1f233,0xbba9c768),L_(0xb69b46cc)}},
   {{LL(0x7c1e6077,0x1443ddcc),LL(0x6345a210,0x70d81228),LL(0x5f035127,0xc8273e38),L_(0xdafd6230)}, {LL(0xcf227ac1,0x4e794379),LL(0xb3e2d215,0x916e6825),LL(0x138aca38,0x4afb3539),L_(0x4ce64d55)}},
   {{LL(0x463d181b,0x6b581764),LL(0xc440084f,0xa9bac44f),LL(0x088767cd,0xf0d004bb),L_(0x94e1ba30)}, {LL(0x55abae1e,0xa1a18412),LL(0xa732c539,0x6d71f955),LL(0x896710ea,0x3608f75d),L_(0xa067bb07)}},
   {{LL(0xbc080af4,0x9891098f),LL(0x65c3a1bc,0x00befda2),LL(0x3321e866,0xb04ac43d),L_(0x438e5b9d)}, {LL(0x5fd8c4d4,0xe260d024),LL(0xc4dcba5f,0x19886f71),LL(0x5e5f5f4c,0x37b1a0c5),L_(0xd3c102f6)}},
   {{LL(0x58ee2dea,0x4e2d8d4c),LL(0xe6546299,0x8042631e),LL(0x262d0cb2,0x2df96d37),L_(0x349292cc)}, {LL(0x04a3c0d5,0x2fddedd3),LL(0x452a0a52,0xd5418050),LL(0x2293a069,0xc6053861),L_(0xfe3040eb)}},
   {{LL(0x3a50a7c9,0x02a166d5),LL(0x861af22b,0x8a435126),LL(0x449c202c,0xb053bb62),L_(0xbb664aff)}, {LL(0x9083b8c2,0xd6f78a70),LL(0x14a57ed2,0x8e5943f4),LL(0x6c5eedb1,0x591d0e00),L_(0x0f079d9e)}},
   {{LL(0x2036fb32,0x4ee2bac6),LL(0xc62978b5,0xce8c9892),LL(0x0ee684b9,0x56d30723),L_(0xb8993d5e)}, {LL(0x2fee5c92,0xfc172755),LL(0xc8622a7c,0x43c48f91),LL(0x4279a0bd,0x78cfee0e),L_(0x3826553c)}},
   {{LL(0x4315c335,0x9060272c),LL(0xe29ecb65,0x5d0bbf12),LL(0xdd976fea,0x53d3e99f),L_(0x278ebed1)}, {LL(0x7fc69ec0,0x7a33fe7e),LL(0x13d33afb,0x9b8a7301),LL(0x19d16dca,0xe09a47ad),L_(0x6e0ed440)}},
   {{LL(0xd0327c5a,0xb55799e4),LL(0xf145e8f9,0x7408bf29),LL(0x97e6d3d7,0x4580043d),L_(0x5f599a51)}, {LL(0x517f2eda,0x38414d94),LL(0xe09d35b1,0x37f611a1),LL(0x31551461,0xce073290),L_(0x1044bab9)}},
   {{LL(0x24eb973b,0x045e586d),LL(0x4d8f2d61,0xa3318071),LL(0x1d2b5b32,0xb68b7f81),L_(0x10bccde6)}, {LL(0xcd3e0e0e,0x0578898c),LL(0x1e460079,0x927cc37b),LL(0x31bad31c,0x3d22477a),L_(0x8c371b62)}},
   {{LL(0xa374b8d9,0xbc19767b),LL(0x3ed7db29,0xa97161c1),LL(0xcc843c60,0x45c30a38),L_(0x508e3e86)}, {LL(0x433b6dbe,0x1457fcb6),LL(0xd8c78825,0xed4ed464),LL(0xc3997703,0x2b88e6d2),L_(0xa1f2140c)}},
   {{LL(0x953f63bf,0x2a452961),LL(0x62444dca,0x6a62e296),LL(0x2d1e976e,0x1d95f51c),L_(0x2bbeb1e0)}, {LL(0x62f53f4b,0x052daa65),LL(0x827170a2,0x5cb51b06),LL(0xc319b1bb,0x3a8ec55d),L_(0xda0c72a5)}},
   {{LL(0x71e73523,0x898419da),LL(0x35d9c257,0x49a774fb),LL(0x55758231,0x4e49ddae),L_(0xd5c3703e)}, {LL(0x4d4edb63,0x28d3f23e),LL(0xa38dc679,0x41807b47),LL(0x749ea0b8,0x15724faa),L_(0x68bb7f4b)}},
   {{LL(0x5af98b88,0x0f227def),LL(0x24af0afb,0x27396d04),LL(0x1cd97cef,0xa2ad5165),L_(0x678f117f)}, {LL(0xce22032e,0x03432db3),LL(0x5cb93ad2,0x92eb23b1),LL(0xcc96e20a,0x47d3d48c),L_(0x5f1e84c2)}},
   {{LL(0x163136e0,0xc4293aee),LL(0x7f0db6e5,0xca0a189e),LL(0x9069c8c2,0xa088318f),L_(0xf4210d55)}, {LL(0xb0d80206,0x13bd2b27),LL(0x34f3150f,0xcd0efd69),LL(0xe152eb4e,0x0fcf1527),L_(0x32e6885b)}},
   {{LL(0xcb923f09,0xed8906b9),LL(0x7eba0ca2,0x95cab4ce),LL(0xc91dbf27,0xcf44aa02),L_(0xf0948c8c)}, {LL(0x1a9b31dd,0xbe7b7487),LL(0x88539394,0xc84b72c8),LL(0x938fd6cd,0x15b31f6a),L_(0x1a1c6b6b)}},
   {{LL(0xce67988c,0xc2869ed0),LL(0x61ab3c9f,0x882c2379),LL(0x3bb8c353,0xb69781cd),L_(0x95d9d719)}, {LL(0xedcef064,0x3de229ea),LL(0xc3895533,0x84ff6e10),LL(0xb6955022,0x5ac0d29e),L_(0x6e2ba9f7)}},
   {{LL(0x3cc87419,0xfe2a24c3),LL(0xe3406a2f,0x3d474db8),LL(0x67beb236,0xf6867bb0),L_(0x5b2cd123)}, {LL(0x74e31d6b,0x2ff473bf),LL(0xed6d88b5,0xeadd2ad4),LL(0x046a1f51,0x9b3025d4),L_(0xf4a61d8b)}},
   {{LL(0x40eaccf0,0x99e19903),LL(0x7e9f5e57,0x1b67fa53),LL(0x00b84a60,0xbabc0faa),L_(0x6de1d67d)}, {LL(0x06fe7a95,0x04041366),LL(0x06672f11,0x5f522ae3),LL(0x0dd51cb4,0xd4012aa6),L_(0xd5ce357e)}},
   {{LL(0xfbc38722,0x538e45da),LL(0x5f595162,0xf57a7cab),LL(0xe917805d,0x56a8bf52),L_(0xf639993d)}, {LL(0x31ba0865,0xe876b4e9),LL(0x3eaf3c58,0xb08987af),LL(0x0ef4538a,0xbd9e742b),L_(0x95404cfd)}},
   {{LL(0x9eb63b57,0x1453ad21),LL(0xf6e400ed,0x627008e7),LL(0xa4ababfc,0xd78f1c8d),L_(0x4ebc5852)}, {LL(0xcc21b23c,0x4e1ef090),LL(0x7415270e,0xb8b3701a),LL(0x921cd6ee,0x472b1f2f),L_(0x765126f0)}},
   {{LL(0x59e9fe85,0xade7ead3),LL(0x1b413797,0xb8da2677),LL(0xeabbca1c,0xdb705e2f),L_(0xa1e12595)}, {LL(0x65e95e7d,0xd1990099),LL(0x79fa6f2e,0x2b7f7bbd),LL(0x4d4d9e0a,0xbde588b7),L_(0xc0ac97e6)}},
   {{LL(0x20ff9acf,0x23cf39f7),LL(0xca0fac28,0x1f72fbb4),LL(0x4b179b83,0xb04c01ae),L_(0xfaca0fdf)}, {LL(0x7173749c,0x7b3cc769),LL(0x3d2bf7fd,0x0cbf34b6),LL(0xf3f2fea4,0x6369c1d9),L_(0x866e7f8e)}},
   {{LL(0xb873e85f,0x6eb23ad8),LL(0x8b9475ca,0x2855072f),LL(0x22d9fd01,0x4e9a3adc),L_(0x813270cc)}, {LL(0xfbf27bf5,0x292b8bdc),LL(0x3db7f857,0xe9f00375),LL(0xf6ba5b62,0x1b8a570a),L_(0xe20966e7)}},
   {{LL(0x05a5a71b,0x19b8e7ac),LL(0xe13277a1,0x9a4abf1f),LL(0x6d26fe41,0x8f8b7d68),L_(0x0ce42976)}, {LL(0xaa0c781d,0x48e32e14),LL(0x0ee98d37,0x13c9b527),LL(0x3bb6c2c9,0x4db74aeb),L_(0x208cd0d9)}},
   {{LL(0xc2685698,0x47ef78f2),LL(0x6c1b0959,0x9bb6bd13),LL(0x2ccd97f7,0xa303c278),L_(0xf9abf5b1)}, {LL(0xafdb9d40,0x0276fd8a),LL(0x2be1d396,0xdb26cd04),LL(0x726beb4a,0xe09517f7),L_(0x4e0b499e)}},
   {{LL(0x6f715206,0x5979197a),LL(0x82ffa270,0xb44c8267),LL(0x52cb678a,0x260e7e25),L_(0xc541f700)}, {LL(0x18439449,0x33983071),LL(0xb1d2f59c,0x19d945b4),LL(0x11eef46b,0xf2b04fe3),L_(0x9fba737f)}},
   {{LL(0x214431d6,0xbcadb425),LL(0x4d1d9a21,0x941b766d),LL(0xe2a1f314,0x2170c6a6),L_(0xc15b9466)}, {LL(0x6c1b9a4b,0xd605b2bc),LL(0x1578a3c5,0x65c9904b),LL(0x96b6befc,0xbe0b58ef),L_(0xc4f7e33c)}},
   {{LL(0x5a93994e,0x3fcf43e1),LL(0xb19e11a6,0xf5f793fc),LL(0x5ad2ed77,0x6ff2a142),L_(0xe587c79f)}, {LL(0x23c53b77,0xb2ee2d92),LL(0xa6fa95d1,0x023b2c75),LL(0xac5ec115,0x67b40c94),L_(0x41c55249)}},
   {{LL(0x05d09dcb,0x6b551c3c),LL(0xc6c2dc20,0x2e81c56f),LL(0x1b5233cb,0x0e6faafe),L_(0x4565862d)}, {LL(0x43368b7b,0x08edb0b1),LL(0xb2d31417,0x29c77be9),LL(0xa0fa599c,0x77052491),L_(0xb2ac3ec1)}},
   {{LL(0x3a2c0bb0,0x0302d5f3),LL(0x04cf0b83,0xfda301ea),LL(0xbd701c39,0xfb565431),L_(0x5eee993f)}, {LL(0x2230bcaa,0xb5c3681a),LL(0x5d285d50,0xa9dd4df3),LL(0x8618542b,0x8c1d7c52),L_(0xd80996cc)}},
   {{LL(0x85ebe7cd,0xf0fe462f),LL(0x12c3c8ab,0x752eb1b0),LL(0xb6349271,0x2f148e5a),L_(0xc0513e4e)}, {LL(0xdaa709f7,0xd043e9e0),LL(0x340f7e04,0x7842c3a1),LL(0xb4acedf4,0x5d6d0c14),L_(0xfa56de6f)}},
   {{LL(0x807fa877,0x435b6be9),LL(0xdb7a9959,0x65c2c690),LL(0x24fae090,0x3cd0982b),L_(0xcc274d62)}, {LL(0x721b08d7,0x02f3dea6),LL(0x2a9bf937,0x052c0001),LL(0xadae4c04,0x61de7f6d),L_(0xba3c48a3)}},
   {{LL(0xdfe50f8c,0x2faaf032),LL(0x7e8bb31a,0x51dbadf4),LL(0xd937aa83,0x072c01c9),L_(0xbfe57ab1)}, {LL(0x5d66c435,0x87c5d208),LL(0x4ce95747,0xfb211069),LL(0x7ee66e9b,0xf01cc54c),L_(0x48478c99)}},
   {{LL(0xb35f128c,0x957c7ee6),LL(0x50dc83b5,0xf963430c),LL(0xf0f63a80,0xb4578760),L_(0x842ee70e)}, {LL(0x57d57cbd,0xc891093b),LL(0x24bc2867,0xa548c5ff),LL(0x0f85fa2f,0x47bb6a4a),L_(0x18cd7a2e)}},
   {{LL(0x813aaaa5,0x40773919),LL(0x798cdd8c,0xb6d9db4e),LL(0x68858372,0xd2c5ede9),L_(0xb240c94c)}, {LL(0x79a51f60,0x62fa25b7),LL(0x89e94f52,0xac054f6f),LL(0x0332347b,0xf207ad5e),L_(0x9e15c0d4)}},
   {{LL(0x19041ef2,0x6d748e07),LL(0x298a62fb,0x3b9bdf40),LL(0x8f7fb37f,0x35f944a9),L_(0x92926ab1)}, {LL(0x51a4355a,0xdc488cfe),LL(0x3f355ad1,0x814db3fc),LL(0x84b0d885,0x29d8cb57),L_(0x98d39682)}},
   {{LL(0xfe4b50e2,0x07c8c799),LL(0x4957a527,0x3418ab11),LL(0xec94fd37,0x601424db),L_(0xf147c20f)}, {LL(0x420c1861,0x1adc80ee),LL(0x7d9440bd,0x09ee96a9),LL(0xee7355ab,0xd1adde68),L_(0xba4d5d2f)}},
   {{LL(0x14379251,0x4dc18b12),LL(0xebadf0ef,0xfee19a43),LL(0xbc8d4cc9,0x8da37771),L_(0xd555d58a)}, {LL(0xd71b4107,0x42fdc281),LL(0x2e05719e,0xac68f304),LL(0xdc3ed2bf,0x4c1b070d),L_(0x03356492)}},
   {{LL(0x0c94988c,0x8da99bff),LL(0xa1e580f6,0xe79a8b42),LL(0xcf6d6c6e,0xfdabb4f2),L_(0x80f7da41)}, {LL(0x1a4121f1,0x6a7dfe84),LL(0x36467882,0x6b06eea9),LL(0x30e07e6a,0x943abc5d),L_(0x8ae444d2)}},
   {{LL(0x2f87568d,0x2441fb99),LL(0x0599179e,0xe8a9e90a),LL(0x54669582,0x47767b19),L_(0x6fa66e89)}, {LL(0x19290f64,0x4596bd1b),LL(0x93f0e068,0x9dab54b3),LL(0xe4e3af20,0xff7a84e2),L_(0x951ce962)}},
   {{LL(0x0b49df59,0x466ca295),LL(0x16afa866,0xc6b06127),LL(0x1f38aad5,0xf4e1677b),L_(0x672e7252)}, {LL(0x29732848,0x3a4e5076),LL(0xfcdff811,0xcc6ceb5b),LL(0xfe6adcac,0x8a4243ce),L_(0xb04666fc)}},
   {{LL(0xe74fffdd,0xc1e32f38),LL(0xd93030cb,0xf9561e97),LL(0x15f3f603,0x39bbca36),L_(0x90c8c914)}, {LL(0x3734c1a1,0xa53fc681),LL(0xb0c4475f,0x158ec0bb),LL(0xed39d189,0xa57cf95c),L_(0xd562a81e)}},
},
/* digit=18 base_pwr=2^126 */
{
   {{LL(0x5cfdbf76,0x02c6b679),LL(0xace24453,0x8195dc3a),LL(0xa58b929b,0xbe2f8e64),L_(0x9a3a7efb)}, {LL(0x9f7af479,0x7fa4b042),LL(0x8ad04303,0xb747bbcf),LL(0x426eb7ce,0xa67998c5),L_(0xdb040e40)}},
   {{LL(0x7d551504,0x3bfd76b2),LL(0x0361658a,0xa3a2cf8c),LL(0xa3b9f1bd,0x66db4cfa),L_(0x4799c86b)}, {LL(0x27afe24e,0x01c04fa0),LL(0x0689618b,0xaa0383bc),LL(0x5b052124,0x524802bb),L_(0x25748110)}},
   {{LL(0xfd9c3522,0xb542168a),LL(0xf1dd3dc0,0x435f3e05),LL(0x3556cca8,0xf3b21a79),L_(0x0d1838e7)}, {LL(0x0a1d87d1,0xbbfa360f),LL(0x43bd0f7a,0xdcacc277),LL(0xfdb15d31,0xa293f5a9),L_(0xf9285692)}},
   {{LL(0x960b1fe7,0x7da35372),LL(0xefe3b9e7,0xb15275ca),LL(0x08b0fe30,0x9c0da2e3),L_(0xcef9bd5d)}, {LL(0xfc9156e7,0x087c3eea),LL(0x4386ea9f,0x13e94876),LL(0x5543dce0,0x057ad0af),L_(0xc5162840)}},
   {{LL(0xf9c98a4a,0xfeb313d1),LL(0x398fba1b,0x0e02712b),LL(0x52f53c92,0x98909181),L_(0xfa320278)}, {LL(0xebd66101,0x906d44a0),LL(0x427688fa,0xc8c39420),LL(0x1c7282db,0x3901fb30),L_(0xaea2389a)}},
   {{LL(0x6775078d,0x1424671a),LL(0xd8f3ae0f,0x4ebcdd7f),LL(0x80279f72,0xd5a77238),L_(0x602ce4df)}, {LL(0xc083918a,0x12e9402d),LL(0xdd095d1b,0x02858419),LL(0x6cb15ad2,0x0927adfc),L_(0xa38c1285)}},
   {{LL(0xfc6fbaf6,0xec392752),LL(0x1ec20f01,0xeed1ebd2),LL(0x5cac318e,0x47520f8c),L_(0x407d9032)}, {LL(0x8ec0f4fd,0xe2f3dac1),LL(0x43dd79ec,0x35a74b40),LL(0x9236a81d,0x6bccf29e),L_(0x00ac0a42)}},
   {{LL(0xc4755991,0x163c66e0),LL(0x849061a0,0xdd950bf5),LL(0x70208a9e,0x7bf95257),L_(0x07a10de2)}, {LL(0xb0813c25,0x45c2891a),LL(0x15468f60,0x450f087f),LL(0x73d5ebe9,0x387e234b),L_(0xa32ac9f8)}},
   {{LL(0xe8ff8e91,0x962ab070),LL(0x570fcd62,0xd3ce94dc),LL(0xb1757c3e,0x1ad55d1b),L_(0x7fa592e8)}, {LL(0xa8ad3f09,0x76fcda7d),LL(0x73d3b392,0x348ab2da),LL(0xd6cf8e99,0xca16fca1),L_(0xac456eec)}},
   {{LL(0xc7486909,0x5f929900),LL(0x240b92b4,0x27ccfc23),LL(0x0f7d51a1,0x632731fb),L_(0xf12e9902)}, {LL(0x735b1778,0x47671b3c),LL(0x985803eb,0xec4d7c8a),LL(0x63e1c219,0x608cf2d5),L_(0x109df03c)}},
   {{LL(0x09438c99,0xb25c71f5),LL(0x0829cedc,0xbe60921f),LL(0x483d485d,0x15305f6f),L_(0x60591120)}, {LL(0xba32c7ab,0xbcdea716),LL(0xbeeee884,0x2b65df3c),LL(0x88e6ef43,0xc3d80182),L_(0x4b2dd1d3)}},
   {{LL(0x1bdc3884,0x482c3679),LL(0x6b421466,0x6f172c00),LL(0xa58b570f,0x2587649f),L_(0x3e4ee4bd)}, {LL(0xef99f5b4,0xf5a1388a),LL(0x98494de5,0x5a8f54da),LL(0x8f25a37c,0x50e9cc72),L_(0x1ce8e0f0)}},
   {{LL(0x72cca43e,0x119587d3),LL(0x8e668562,0xa3b06524),LL(0x0554888c,0xd291f0fd),L_(0xae58bc85)}, {LL(0xd1139d9b,0x48a506c1),LL(0x034f448b,0x04ec28f0),LL(0x94dd32bc,0x2bd2abc9),L_(0xb9e588a6)}},
   {{LL(0x4988c3b7,0xe1e13901),LL(0x24b16627,0x4231b66d),LL(0x4f52ca62,0x749a749b),L_(0x8bb11e14)}, {LL(0x03150154,0x3dfafe07),LL(0x71bdac04,0x73419c6b),LL(0x40bd5232,0x1587ab8c),L_(0x7d2d4a5e)}},
   {{LL(0xf231ad9a,0xe7ffdae6),LL(0x0f11fde4,0xa13a1e41),LL(0xc78059f3,0xf4713c18),L_(0x2d2f0ff7)}, {LL(0xfde9d338,0x5fe86c6a),LL(0xa381beae,0x9d423777),LL(0x734b9092,0x77f56796),L_(0x82131b3f)}},
   {{LL(0x3a2c7153,0x612650a1),LL(0xc29b399c,0x72e7ea2b),LL(0xd673cbc5,0x60425f21),L_(0xece9a8ea)}, {LL(0xfd08bc21,0x394b09dd),LL(0x1b69fef5,0x58e594d5),LL(0x141fc2ff,0xeab0baec),L_(0x85999248)}},
   {{LL(0x5c8c18cf,0xf4b4d1ab),LL(0xe1f44cb9,0xdca5073e),LL(0xd86fbb1f,0x32861f51),L_(0x242811d3)}, {LL(0x038cf05a,0x20b39e70),LL(0xc64c1332,0xd6198042),LL(0xeb68ae09,0x93c66ba1),L_(0x7aee0eb8)}},
   {{LL(0xc4c03565,0xf47a18f7),LL(0x992398ea,0x808ebaad),LL(0x7414fdc9,0x325c8a8e),L_(0x42a7e664)}, {LL(0x0900ad27,0x9f053caf),LL(0xc8869d15,0x2b52f07a),LL(0x1878089f,0x11f347a4),L_(0x19ee0c05)}},
   {{LL(0xf8345a24,0x3e51ac6c),LL(0xb254a9d3,0x57f4ec7a),LL(0xe2b48c1d,0x916d1214),L_(0x3732c0ab)}, {LL(0x5045ad37,0x0cb5ec90),LL(0x010093ba,0x1eea394a),LL(0x4a4cfff6,0x4aaf2a44),L_(0x7feda10c)}},
   {{LL(0x370622e3,0x58625a1a),LL(0xea895fc8,0xe2d5d70a),LL(0xcac0e4d3,0x53bc48b9),L_(0x1d43df08)}, {LL(0x681301c8,0xee75a210),LL(0xe1a8bb95,0x097c56b0),LL(0x7e7b6ce5,0x4a384a6d),L_(0x26d0db83)}},
   {{LL(0x3ebba7eb,0x2bb95cad),LL(0x9e42faba,0x5398a161),LL(0x0f33c612,0xf67c4703),L_(0x6ef93df3)}, {LL(0xbb0fa5e1,0x8ad685b2),LL(0x8655685c,0xb1c226c7),LL(0xdd2eb2bd,0x5e194627),L_(0x523e0fd5)}},
   {{LL(0x3d42f37f,0x301fcb42),LL(0x80fb78c3,0x647b20c7),LL(0xfaa1d374,0x3d0a850d),L_(0x051bb6db)}, {LL(0xbe9d92c9,0xc7fc5cad),LL(0x38f0078b,0xae641b58),LL(0x91a544af,0x0c9540c5),L_(0x6cae5443)}},
   {{LL(0x33b2702e,0x93ebe56e),LL(0x71946cb7,0xff835a5d),LL(0xc82366c4,0xbfe6dc77),L_(0x27a97fdc)}, {LL(0x2bb71ad5,0x80d9e527),LL(0x16d42ed2,0x21ce1d14),LL(0x18abf494,0x0398f8d3),L_(0xa190c5d3)}},
   {{LL(0x87b49d63,0x495ac575),LL(0x6bf10350,0xb50a8115),LL(0x2674f39c,0x90ac41f9),L_(0xaedc843c)}, {LL(0x20e79ef7,0x60ca4b9c),LL(0x2b7ba708,0x67e85483),LL(0xc070ae10,0x2e4b0764),L_(0x0c3aa1f7)}},
   {{LL(0x9b6c96e8,0x555cdffb),LL(0xd2310fa6,0x1116970b),LL(0x42f99112,0x9d84f404),L_(0x8d7bdc74)}, {LL(0x452face1,0x43e3cff2),LL(0x1c149edf,0xf602e240),LL(0x442dde2f,0x1288dcc9),L_(0xb3089de8)}},
   {{LL(0x1ad5fcdc,0x3fe61e5f),LL(0x20fa93a3,0x2e793d8a),LL(0xb50f92b7,0xbfce0907),L_(0x632ab7a5)}, {LL(0x8a0e5e0a,0xc89531e2),LL(0x6f57401d,0x4059a329),LL(0xf7bd6eb0,0x5942987b),L_(0xceffd87f)}},
   {{LL(0xacfc439f,0x74161b3d),LL(0x647fc617,0x3b17df59),LL(0xac7f6cc8,0xc485b2c8),L_(0x0414a8d3)}, {LL(0x69c118ce,0xd7de72d0),LL(0x1d6064d4,0x1eff6d80),LL(0x9fd6a735,0x8ee9cb9d),L_(0xf104d7a9)}},
   {{LL(0x252fb3dd,0x776f7b1a),LL(0x2b058af6,0xb6f63fd9),LL(0x0d06abe2,0xb0461b36),L_(0x90f6c358)}, {LL(0x5c9d58a0,0x9a72e438),LL(0x24e0b184,0xce596569),LL(0x6facf219,0x5f6753e5),L_(0x5f23a218)}},
   {{LL(0x87cb85e1,0x4fd2cc25),LL(0xadb7e3c5,0x53c5cae5),LL(0x4cc67b4e,0x4c80d589),L_(0x6a578011)}, {LL(0x4f08551a,0xc04163a3),LL(0x41978a17,0x52d7161e),LL(0x4701d626,0x7acb9d4e),L_(0x7040eb22)}},
   {{LL(0x25c4f3fa,0xb117faf3),LL(0x734cd26d,0xb2bfac77),LL(0x071d299e,0x1ad0de2a),L_(0x826417df)}, {LL(0x1049bb39,0x487c15ef),LL(0x5432a8df,0x74da9484),LL(0xa5c29870,0xe2f76737),L_(0x89d1f699)}},
   {{LL(0xe58a9a06,0xe366cc79),LL(0xcd96f1e7,0x497f4836),LL(0x19aa24f5,0xbe89a16c),L_(0xea6d5b99)}, {LL(0x4105a3ce,0x1afa0bd4),LL(0x8da4ca09,0x4aa54b7f),LL(0xb278c556,0x07697581),L_(0xea0b2435)}},
   {{LL(0x69260167,0x233ddda4),LL(0xbce451f2,0x055176b1),LL(0xe21ca9dd,0x33fc9573),L_(0x608477ee)}, {LL(0xb1111c08,0x37f4a8b6),LL(0x2f253ba7,0x60c86d48),LL(0xec1f4a44,0x3ae7c4bb),L_(0x1c649260)}},
   {{LL(0x418877ba,0xf186ffa7),LL(0xca88dd8a,0xe435d41d),LL(0x2e8c5c07,0xd65638d7),L_(0x8d28689a)}, {LL(0x50763ccc,0xa2ef4b71),LL(0xfa5e89e0,0xbab1be9a),LL(0xea4b66ad,0x2a17ea7b),L_(0x5744fef5)}},
   {{LL(0x511ae7ab,0x1c88c915),LL(0xd377e6af,0x55991086),LL(0xbd38c04f,0x43c62134),L_(0x7e11eacb)}, {LL(0x715ff436,0x3aa19a12),LL(0xe65b9643,0xc019b147),LL(0x0069d240,0x7d7a8265),L_(0xc6fc4f7d)}},
   {{LL(0x00db9dd8,0xb94a17c8),LL(0x95308d83,0xb2b6edac),LL(0x88fbad84,0xd14a8ca8),L_(0xcec36252)}, {LL(0x7824a283,0xf71bccba),LL(0xd90035ec,0xd172d468),LL(0x55a73d3d,0x07f0cdbc),L_(0x6c21797b)}},
   {{LL(0x3b33cd4c,0x5162797e),LL(0xa87e9702,0x6fc60b78),LL(0x2ab93ff3,0xe2a31c19),L_(0x4288abf4)}, {LL(0x297e2ccf,0x80be51bb),LL(0x388680ed,0x058de4bc),LL(0x8dedbe92,0x7da1a829),L_(0xbffcb08c)}},
   {{LL(0x7008c033,0xac1e2c00),LL(0x1d6fb3ed,0xd70668f0),LL(0x9f1545b9,0xfb03795c),L_(0xbfecfd6e)}, {LL(0x83ab63e2,0xac0e74d0),LL(0xcbb46331,0x481489c9),LL(0x01849565,0x79c317d8),L_(0x9186b0db)}},
   {{LL(0x1b42ef85,0xe3b9e5c1),LL(0xa44cb510,0x4bcaad24),LL(0xc72bcdea,0xd47f22e3),L_(0x23ac79ad)}, {LL(0xffacee46,0x1062a849),LL(0xd5193e66,0xfb9adffd),LL(0xbb25e4cf,0x600c0353),L_(0x5ab3c0ce)}},
   {{LL(0xd8f9674f,0x1f36e1c6),LL(0x2a414d7c,0xc2f2d11a),LL(0x23202a91,0xbdd27859),L_(0xd77095b4)}, {LL(0xe447fde6,0xe5a99ae1),LL(0xd0fffc9e,0x5835b80b),LL(0x490dac48,0xacfdbe9c),L_(0x9079d090)}},
   {{LL(0x65c70a3a,0x550c3e82),LL(0x448361bb,0x40b9aa3e),LL(0xfeab6ae4,0x41969a52),L_(0x74461138)}, {LL(0x37a53426,0x47381f18),LL(0x2be030b8,0xd6e199e3),LL(0xb091abda,0x4da4cd53),L_(0x7df8f1e3)}},
   {{LL(0xb6274829,0x9fea92c3),LL(0xab2fcd01,0xcf88fa14),LL(0xd804cba2,0xed8709e8),L_(0x19141717)}, {LL(0x820090bf,0x88fb9ad9),LL(0xe105506b,0x370d09cc),LL(0x8cb2c13e,0x4e6bda23),L_(0xcaba970f)}},
   {{LL(0x5ad6bf8f,0x9fcd4871),LL(0x160191eb,0xa47d5c67),LL(0x826242d1,0x99634c20),L_(0xdfe7289d)}, {LL(0xbd0af1f7,0x4e07eba8),LL(0xc0910d55,0x18abee83),LL(0x7e042c87,0x6d7f5229),L_(0xd74d6e82)}},
   {{LL(0x4747a07d,0x6efbc0e2),LL(0xcc3c4e8e,0x9088fde2),LL(0xf21ea1eb,0x05c108d8),L_(0x37aae682)}, {LL(0x0d08ac9d,0xf2c37fd2),LL(0x9a51bef8,0x4d3bd6c1),LL(0x5ac834a7,0x2a412b12),L_(0xac985dae)}},
   {{LL(0xf5833d2b,0x39bd0099),LL(0xb0ef3648,0x886d33b3),LL(0xc1936e74,0x97226670),L_(0x3eb64bfc)}, {LL(0x408e784b,0xfa065545),LL(0x6cb29cd8,0x96246832),LL(0x010fbc36,0x6b24b9f4),L_(0x72e08b5c)}},
   {{LL(0xaa5c541e,0xe0daf350),LL(0x2fb0ee94,0x86c5cee7),LL(0xac57f680,0xb753971e),L_(0x6b0ccf83)}, {LL(0xbcb76004,0xc3b1be67),LL(0x48b857e0,0x3f637d0f),LL(0x81e2b0c6,0x595ebecb),L_(0x7151293e)}},
   {{LL(0x9a2c561d,0xb788bb43),LL(0x65e9c6ef,0x6cf04373),LL(0x10062256,0x891a7719),L_(0xe6f377fd)}, {LL(0x9d5d462c,0x7355204a),LL(0xe23556ab,0x08f2e4fc),LL(0x188872ce,0x84eba15d),L_(0xea03bca4)}},
   {{LL(0x37bcd5fb,0x159bb9de),LL(0x08807416,0x412b6267),LL(0x496eba6c,0x5b5b2f5c),L_(0xff4f34bf)}, {LL(0xd57ab42b,0x376ec0fc),LL(0xe48904ea,0x62f02fea),LL(0x96bf9fdc,0x9b04b325),L_(0x003a468c)}},
   {{LL(0x85de6fe5,0x3d1be5ab),LL(0x62509881,0xf11ef926),LL(0x65d5f607,0x244caa05),L_(0xc38d7540)}, {LL(0xc1a149a8,0x0b239ba6),LL(0x19b8e994,0x53224bb3),LL(0x1f25f647,0xd2cd0a34),L_(0x7f481820)}},
   {{LL(0x303ac802,0xdceb71b5),LL(0xf395c0f2,0x9bd5860b),LL(0x6febd95b,0x01d8efcf),L_(0x0b358ed0)}, {LL(0xc95f4540,0xdadfd46c),LL(0xdc5d961e,0xc1f026dd),LL(0x221b24a2,0x6d21a125),L_(0xc0f16b10)}},
   {{LL(0x03e43472,0x3ece8318),LL(0x54252e96,0x430e9b3f),LL(0x97edd631,0x020e1e99),L_(0x90685aae)}, {LL(0xad7d0431,0xb7771434),LL(0x6df64310,0x9647f95c),LL(0xa647e2b0,0xd6ddd8f3),L_(0x52ff5927)}},
   {{LL(0xe48eb4a9,0x3fd7f726),LL(0xbb02013a,0xcd6dd255),LL(0x6e17def8,0x5ef8f2c6),L_(0x1b970bb2)}, {LL(0x867ed64e,0x1236fa4f),LL(0xf7a7e274,0x459207e6),LL(0xffa4c978,0x5a8179b1),L_(0x99219388)}},
   {{LL(0xcb87940a,0x4b407e59),LL(0x844408fa,0xa05e3d79),LL(0x742669b3,0x0751f86d),L_(0xb00b365a)}, {LL(0x7860c6e5,0x3a8360fb),LL(0x4147cefc,0x10209a6b),LL(0x0393026e,0x7a7f6231),L_(0x2e39fa34)}},
   {{LL(0xe7f6f33d,0x149f6c77),LL(0xbbcd3afa,0x9a9bdb39),LL(0x742b5b3a,0x899f0118),L_(0xb8dc2766)}, {LL(0x573badd0,0x14c00e4c),LL(0x6924c12d,0x190dfe39),LL(0x4e4111e9,0xdf09100c),L_(0x2987ca68)}},
   {{LL(0xe5fa2136,0x65fad088),LL(0x01cff054,0x9c1c2f99),LL(0x3002769e,0xdb9e3912),L_(0xe9ad46db)}, {LL(0xa7493cf2,0x6e70449f),LL(0x34ab7ab5,0x1ed50609),LL(0xfda4dfd0,0x91982d5a),L_(0x89e4ea03)}},
   {{LL(0xb358f84b,0x132822e9),LL(0x1ad76044,0x9d9ae4a1),LL(0x3e8a1d95,0xb981b100),L_(0x37f04df1)}, {LL(0x364692e2,0x0b17bd09),LL(0x0b34cd00,0x5d63a307),LL(0x447278ad,0x07bde57a),L_(0x6e16122f)}},
   {{LL(0xa6d64780,0x3221832c),LL(0x8fc6ce6c,0x3dbd0cb9),LL(0x6582853a,0x9cdf65d9),L_(0x708e9258)}, {LL(0xf6bfe6cf,0x273611ec),LL(0xef43a507,0xa78d98b3),LL(0x86406043,0x88f84624),L_(0x4396769b)}},
   {{LL(0xbca28b2c,0x25903255),LL(0x6b8a2c7e,0x255e1a29),LL(0x567e51fc,0x06f1e0ff),L_(0xdbcad515)}, {LL(0xf590a256,0xe7fa9d15),LL(0x627c717b,0xe236aabd),LL(0x3efff1d7,0xa592bf53),L_(0xe58ce7d7)}},
   {{LL(0x47f49527,0x6bc49db6),LL(0x2993ad98,0xc9fd48de),LL(0xcd8ed14b,0x104437d1),L_(0x419b6d76)}, {LL(0x80c19d84,0x5ceae531),LL(0x2692fc9f,0x7b8ba2ce),LL(0x96208c12,0x5f17f19b),L_(0x34a83cc6)}},
   {{LL(0x35c24843,0x3d238cb1),LL(0x6bb879b5,0x818fcd3b),LL(0xeaa5d8cb,0x137d82f5),L_(0x769e2a68)}, {LL(0x1bcbebc0,0x0ddf279d),LL(0x2ae26372,0xda0b62e6),LL(0x79e90c0c,0x27e773c7),L_(0x3451edcf)}},
   {{LL(0x8b2767f2,0x65ddd79a),LL(0x34988e35,0x9ab07761),LL(0xd93aac10,0x814a0e96),L_(0x76ec0b96)}, {LL(0x802069a0,0xe6639b9a),LL(0x9143cd9d,0x23a6da3d),LL(0xfb9b096c,0x77b915d6),L_(0x4353464a)}},
   {{LL(0xc93ba38f,0x103b0148),LL(0x52d3720b,0x9d45cf17),LL(0x3c3a2d72,0x0ddae18b),L_(0x3543af35)}, {LL(0x10ee7a77,0x90846157),LL(0x51cb6d6c,0xa8824c71),LL(0x5541df28,0x9f0df2e6),L_(0xa96cf0e6)}},
   {{LL(0xdb058979,0x51a147b3),LL(0xbd4c751a,0xba41fdd4),LL(0x5b132575,0x81839452),L_(0x9b0ddfdb)}, {LL(0x785de02d,0x2e14f24c),LL(0x54dad63a,0x052a799c),LL(0xd7c22290,0xdcbca7aa),L_(0x77c13acc)}},
   {{LL(0xe8859b95,0x4c326ed2),LL(0xe3bb354a,0x57333802),LL(0xa148004d,0xb7008ce3),L_(0x85367d7f)}, {LL(0x2d4febd4,0xe73d1a3f),LL(0x93d871ad,0x1f4f245d),LL(0x7768c969,0xd9c87876),L_(0xb38a617a)}},
   {{LL(0x085dbc17,0xaf8f400d),LL(0x2a94dc1a,0x93399c4a),LL(0xf8d2e936,0x370e4c06),L_(0xca35bd61)}, {LL(0x0fb73083,0x2c1682bc),LL(0x00b6e1a8,0x4742b84d),LL(0x6d92e1f4,0x9bb1736d),L_(0x3e9b6293)}},
},
/* digit=19 base_pwr=2^133 */
{
   {{LL(0x2a85b5cc,0x6c35afbc),LL(0xf6f30e96,0xb02b80b7),LL(0x669a8252,0x4dd8bbbe),L_(0xb9b1b311)}, {LL(0x28535ef6,0xbbcbf46b),LL(0xe4d59c99,0x20d09a27),LL(0x63242255,0x877172dc),L_(0x853f9c6a)}},
   {{LL(0x432984a1,0xd5de71e9),LL(0xe7f42edd,0xb421581f),LL(0x0b9c53bf,0xc8c4ee60),L_(0x82b1e37e)}, {LL(0x51492ea8,0x4a10fc29),LL(0x7502a8c1,0x69016e9c),LL(0x2dc30077,0xcd3e0527),L_(0x2e0b7f76)}},
   {{LL(0x161f4880,0xc21b4aab),LL(0x95f715ce,0x44db3fe0),LL(0xff0073cd,0xdb4528e5),L_(0xdbc69fc1)}, {LL(0x5161785c,0x8bfc4101),LL(0xe2c44c67,0x22e31d16),LL(0xa7624b31,0x1b45072c),L_(0xebf2704a)}},
   {{LL(0xfd1ff163,0x540d3ef9),LL(0xe3c8a7e4,0xd48da416),LL(0x39e285d6,0xccac1ee5),L_(0x8d098605)}, {LL(0x09862475,0xaae2485c),LL(0x424cb101,0x4df1fb56),LL(0xd81ebf03,0xe045acd6),L_(0xec4b922d)}},
   {{LL(0x3fbc727e,0xe3ace3b3),LL(0xfb5f0701,0x39644794),LL(0x1534cde5,0x25904a79),L_(0xf9421b0d)}, {LL(0xd3275fa8,0x0528f7ca),LL(0x8a9c39d8,0xdea09c8b),LL(0xdd383932,0x458a7c5f),L_(0xbaf06aaf)}},
   {{LL(0x1adf0fac,0x193b780b),LL(0x5abd6eef,0x5042f2da),LL(0xd76b78fd,0x06442ef6),L_(0x0cdf9e58)}, {LL(0xfef055d2,0x8a5c2657),LL(0xfae0a442,0x0bf16e33),LL(0x6c2b778b,0xb5adbc98),L_(0xf374d05a)}},
   {{LL(0x846ef971,0x0c6f4ae2),LL(0xb657d5f6,0xd2466841),LL(0xabf1920f,0x70a6775e),L_(0x15909e3c)}, {LL(0xcede3a4d,0xe0c73f43),LL(0x563843f3,0x15282f23),LL(0xe15912d0,0x2e80308d),L_(0xd16c8835)}},
   {{LL(0x767329dd,0xb00d2849),LL(0x96ae8e24,0x10d39711),LL(0x118cc563,0x4ed443bf),L_(0x9157acc3)}, {LL(0x755122ae,0x14b54c12),LL(0x07d6ccc1,0x4fd87692),LL(0x05ec4685,0xf50f2de3),L_(0x1a923061)}},
   {{LL(0x3480f7d1,0xc692cc02),LL(0xbe07dcf7,0x71999891),LL(0x5f04bf18,0x0d99300c),L_(0xa062deba)}, {LL(0xb9eb55fc,0xa3451ea5),LL(0x02666d4c,0x1acb5351),LL(0x2fc90780,0x564f33fe),L_(0xa2bd47b9)}},
   {{LL(0xb70680a0,0x6da6ee27),LL(0xdab5f17d,0x76f48c1f),LL(0x36f25faf,0x16a32d1e),L_(0xbe5fdc6d)}, {LL(0xcd1b3276,0xd0d79fc8),LL(0x0ee979a6,0x2e5c89f7),LL(0xc670e327,0x6435ef83),L_(0x55425ae2)}},
   {{LL(0x8e0dd4c4,0xcf395e16),LL(0xd8954092,0x86079f76),LL(0x516938e6,0xc57f3db0),L_(0xe85d4e50)}, {LL(0x4eabd92c,0x611ea78f),LL(0xfafbe19a,0x8d26fbc1),LL(0xe5ff8f07,0xe41d9e6e),L_(0x9e9c0da2)}},
   {{LL(0xcfcf1446,0x86e3cdb5),LL(0x8cb41178,0x3fbc43f0),LL(0x5615e2b5,0x74d58d7a),L_(0x4e6b82ce)}, {LL(0x0c36a5a8,0xd4c6f917),LL(0x374bbfd9,0xc3b3ae3b),LL(0x15aadf46,0x363a6cec),L_(0x30dd6649)}},
   {{LL(0xe325634b,0xe43c3ade),LL(0x120bb9ae,0xb587b6a1),LL(0x73ef55fc,0xbe8a8508),L_(0xbb18ab2b)}, {LL(0xc135e4d5,0x84facee1),LL(0xb277aa2c,0x67fba354),LL(0x742a25ca,0x1cc06cae),L_(0xd0d24d20)}},
   {{LL(0x3aa2917b,0x0714ac71),LL(0x2ffe0519,0x178aa56a),LL(0x843d58c9,0x7eddcb6d),L_(0x1e60abe4)}, {LL(0x4b221bdc,0xb10fd184),LL(0xccf9c609,0x113f37e1),LL(0xa0d4b627,0xe398ed34),L_(0x7b6aa811)}},
   {{LL(0x95472b76,0xa960c0c2),LL(0x735c5d62,0xf7232900),LL(0x0f8b2a35,0x54dfa089),L_(0x0b186f7a)}, {LL(0x7962dd9a,0x3e7eb8a1),LL(0xc1614c55,0x0188ff65),LL(0x8c7e1e1d,0x390e7c1c),L_(0xf9b0b2f9)}},
   {{LL(0x66e2227a,0x11277d69),LL(0xd9cc122b,0x5181f585),LL(0x3f2ab21d,0x2cbd87e4),L_(0x494cfb1e)}, {LL(0xd57e150f,0xae7bbb8f),LL(0x9977c3ac,0xee685251),LL(0x397ea438,0xb224446c),L_(0x7ed9dec7)}},
   {{LL(0x3c14cf6c,0xc59accd1),LL(0xb5432f0f,0xb6446b96),LL(0x8b88a937,0xc5589b93),L_(0x02d9f9ac)}, {LL(0x007ea9ac,0x8302d1bb),LL(0x43908c95,0x087096dc),LL(0xb05f1092,0x83349c39),L_(0x5061c674)}},
   {{LL(0xf6734878,0xf0350dc8),LL(0x9371eb01,0x4edf29b1),LL(0xce86a9ba,0x47861a0c),L_(0x9aad7247)}, {LL(0x16443994,0xa559d5e8),LL(0x979093ec,0xa58da55e),LL(0x0fe17efc,0xb512322d),L_(0xae7b8c8e)}},
   {{LL(0xf66cc404,0x3ad82960),LL(0x9f1ec261,0x07a3082c),LL(0x77a36e96,0x3914c9d2),L_(0x9b1a4daf)}, {LL(0x238844ea,0xec701058),LL(0xcfa77625,0xc70140df),LL(0xe60b3f2a,0x0c5b4abf),L_(0x138b4b1c)}},
   {{LL(0xa5d35437,0x174442e5),LL(0x1802aa28,0x2aa23227),LL(0x26f71e89,0x4a4c5681),L_(0x7ae33021)}, {LL(0x7106d5d6,0x5d0cad43),LL(0x2893d185,0xe47a47cc),LL(0x7e9947d1,0xc9f7bf57),L_(0xf2c79dea)}},
   {{LL(0x607d7e89,0x3a838f3a),LL(0xfbe1be02,0xf73959fe),LL(0x229d491d,0x207a3044),L_(0x9f62e957)}, {LL(0xd08dcf90,0x2e1d3c53),LL(0x86b7e5aa,0x229a31a5),LL(0x3fbda02f,0x7c96f0d5),L_(0xbda44f9b)}},
   {{LL(0x6e006c73,0xf1aeae69),LL(0xae77f574,0x08648f34),LL(0x0423c397,0xf1f4d153),L_(0x47451433)}, {LL(0xcd40d0b0,0x3a807b62),LL(0xe8e961f8,0xa745762a),LL(0xd89c3d2a,0x467e86a7),L_(0x6ac80359)}},
   {{LL(0xb9798433,0xb36ecce6),LL(0x3b7324a4,0x5be87e7d),LL(0xbb9babf8,0x98c1904a),L_(0xf1e380ce)}, {LL(0x03c9dc36,0x01b08224),LL(0x2eeb6052,0x1ae12594),LL(0xd0d4fef9,0x19322045),L_(0x256a539e)}},
   {{LL(0xdcff0296,0xc94659b0),LL(0x2f0568f5,0x39aab164),LL(0xed8a79a8,0x50d8c266),L_(0x757c7d99)}, {LL(0x4445c1ec,0x66c509e4),LL(0xb792533a,0x70d28954),LL(0x9e0c5aad,0x34caeea9),L_(0xe5b193ca)}},
   {{LL(0x8188c9ad,0x195602fb),LL(0x701a9997,0x76ac1458),LL(0x3992cbca,0xe1e8b8e7),L_(0x8ec536e5)}, {LL(0x4d168e61,0x29c79b39),LL(0x29bff6f9,0xca81be8e),LL(0x596ff95f,0x0a0ae347),L_(0x661b4a6c)}},
   {{LL(0xd0970633,0x8e767a03),LL(0x473ec4b4,0x7b96613b),LL(0xeb4d7d16,0x98a8ebb4),L_(0xc4e28600)}, {LL(0x1487660b,0x6beb5ab7),LL(0x1ef6cbd4,0x71463757),LL(0x83dde9df,0x1762ff65),L_(0x9ddef882)}},
   {{LL(0x0d56a9f5,0xe6f97e52),LL(0x2cfac1c4,0xe924c291),LL(0x9480052a,0x61546993),L_(0xcee96715)}, {LL(0x4783b246,0xdc612b8a),LL(0x01036872,0x40ce9da3),LL(0x9e6d56f5,0x0543bf02),L_(0x40b8eb7d)}},
   {{LL(0xaa29fa33,0xbb67b973),LL(0xd7ceddb4,0x733e1da7),LL(0x7bd3859f,0xe95a7ea2),L_(0xc85424d7)}, {LL(0xc42d076f,0x5d21362a),LL(0x2591a7e7,0x8674ee1d),LL(0x52817220,0xc9d8453d),L_(0xa286acd4)}},
   {{LL(0x7fece4b3,0xaf830115),LL(0x6e587151,0x0a71eb17),LL(0x3d8da1f0,0x7afbb075),L_(0xf8e3bab4)}, {LL(0xea1fecf8,0x3ab8344c),LL(0x6bb840b0,0xb4392fd5),LL(0xae7bd8be,0x73177038),L_(0xed8b2ca2)}},
   {{LL(0x50dfe6f9,0xadd91143),LL(0x680d82d3,0x88a2c506),LL(0x88f503de,0x37393204),L_(0x746b676f)}, {LL(0xc6591d03,0xaf979503),LL(0x5a9be716,0x79a8b931),LL(0x0e682183,0x609ab748),L_(0xfe8299c8)}},
   {{LL(0xc3535c6b,0x193fd320),LL(0x8682c4c0,0xff4f04b8),LL(0xd566872d,0x1407f3c3),L_(0x16439d5d)}, {LL(0xac2e95ba,0xd6726481),LL(0xb7bc2fb3,0xb0b0ea32),LL(0xb4c2b53d,0xfeaaece6),L_(0x4349edc0)}},
   {{LL(0x649e4b51,0xae9bd277),LL(0x6c678048,0x4268b40c),LL(0x52da3782,0xae4a71b1),L_(0x1e1cd89e)}, {LL(0x4f079bbd,0x2f691fc6),LL(0xf69884c3,0xfe5d33d3),LL(0x19020a41,0xdda1c483),L_(0x77d02be4)}},
   {{LL(0x7ea53289,0x8529164c),LL(0x5d80aafd,0xd09dfaea),LL(0x1e830c7c,0xa9f0f09e),L_(0xfc3140e2)}, {LL(0x5135dc82,0xdc2ab04a),LL(0x59befe02,0xa0b35f50),LL(0xa44595d6,0xc1e15529),L_(0xf19abe7c)}},
   {{LL(0x3a5d5f42,0xa12bcced),LL(0xb9b8ec3a,0xb87d7dc6),LL(0xad1e4dcf,0xb63e0921),L_(0xfcf65288)}, {LL(0x0c0e6332,0x863fe95a),LL(0x5cd2ee3e,0x42a677d4),LL(0x60da7b43,0xb8a5bd09),L_(0xc426f7df)}},
   {{LL(0x51ea4dd8,0x6e5213a1),LL(0xfd69d7c4,0xdf5f1e1a),LL(0xd30cf23b,0x37d44e1e),L_(0x7097147f)}, {LL(0x321e5347,0x12d98f14),LL(0x32dad5dc,0x76d7e7b8),LL(0xdfe1a803,0xc8d86798),L_(0xeccd5411)}},
   {{LL(0xa7aae910,0x74006fc6),LL(0xd7d803d8,0xa34c1849),LL(0xec84dc01,0xae309a70),L_(0xa34e0e32)}, {LL(0x42af799e,0x235ed631),LL(0xcfee558c,0x8802c5f3),LL(0xcd97224f,0x1640e7f8),L_(0x92280213)}},
   {{LL(0x535c56a8,0xc7ccf419),LL(0xd687e53b,0x31ef82e6),LL(0x9a24176e,0x01470242),L_(0x01aae44b)}, {LL(0xf12919c6,0x9fcc8bac),LL(0xc772707a,0x2bbb2bc0),LL(0xeeb32998,0x1fd556e7),L_(0x6b1da8b4)}},
   {{LL(0x10943f72,0x3fe0c227),LL(0x929fe46b,0xf1163a98),LL(0x55829b86,0xb0593728),L_(0x46968b3a)}, {LL(0x98063840,0x861c28fa),LL(0x49238b1c,0x238ae42f),LL(0xe2c2f08f,0xc3669826),L_(0xd635b05a)}},
   {{LL(0x0b716b89,0x1cbd7071),LL(0xac172491,0x9bedfd64),LL(0x70337bd6,0x5806f542),L_(0xba62de08)}, {LL(0xff8ac4cd,0xc789f899),LL(0x778e6f6e,0xcabb9b96),LL(0x7cb61f89,0x9e11ca0e),L_(0x54802afa)}},
   {{LL(0xb4f60ac0,0xa293c1cf),LL(0x98d81eae,0x3ce25589),LL(0x4a05fd65,0xf7776c88),L_(0x31edd830)}, {LL(0x7102bdbe,0x6b003741),LL(0x530442c5,0x619c469b),LL(0x2717506f,0x993316f5),L_(0x977de534)}},
   {{LL(0x731f0430,0x1f4b647b),LL(0x63a5f62f,0xa8649b7d),LL(0x184bd1bf,0x9eb8bb02),L_(0xd878238c)}, {LL(0x2268c556,0xee715bb4),LL(0x0c09795f,0x5ba6433a),LL(0x4a0ef111,0x5e80a1d2),L_(0xa6ed70be)}},
   {{LL(0x1a0ec85a,0x02e27463),LL(0x66b5fa13,0x7351bd1f),LL(0xf1afa0ff,0x262f06e4),L_(0x55328bec)}, {LL(0x17a8c5cc,0x359db888),LL(0x56654a88,0xbbf24e90),LL(0x31866e60,0xe08705c4),L_(0x615a5b05)}},
   {{LL(0x2741b093,0xd9bae789),LL(0xa18e5f26,0x959b66a0),LL(0xfb66461f,0x52d89638),L_(0xf79de1ee)}, {LL(0x3b41960d,0x63f080c4),LL(0xf0c1209c,0x1356ad3c),LL(0x8927edab,0x1cca87fa),L_(0x0d7779de)}},
   {{LL(0xed47b255,0x4ffe196c),LL(0x591b255f,0x0bda5bcf),LL(0xf529da71,0xa919472f),L_(0xc172114d)}, {LL(0x4edb3f70,0x727df75d),LL(0x0b0bba01,0x7a3418a3),LL(0x18ae3fad,0x7de531b5),L_(0x60f6010b)}},
   {{LL(0xeec6a320,0xc8ab7f39),LL(0x4d9c100a,0x42812137),LL(0xb3ed07de,0x2ce9d1a7),L_(0xf4f877cc)}, {LL(0xbf31e6a0,0xe5d2ab30),LL(0x7ebabf38,0xc0c95215),LL(0xce3cfa6e,0x85ffa6d1),L_(0xfa3306d0)}},
   {{LL(0xfe179a15,0x014e69fa),LL(0xe40ec6e7,0x949a6258),LL(0x8f04f43e,0xf3db5670),L_(0xb9883ee7)}, {LL(0xf31a374c,0xeab3fc5d),LL(0xedf8720a,0x70831223),LL(0xf1d8e66c,0x32cbb568),L_(0xc1af6691)}},
   {{LL(0x955bb333,0xb78b642a),LL(0x65a42ec3,0x18885922),LL(0xdde23959,0x216642c3),L_(0xcdc7c4de)}, {LL(0x4a6ce38f,0x82ee3788),LL(0xec771036,0xd5430b69),LL(0xefdcbefc,0xe82b1033),L_(0x5b9fd0f1)}},
   {{LL(0x856e9dad,0xd7c8b168),LL(0x94661f99,0x034f792d),LL(0x968a0514,0x5e1f223f),L_(0xf2936613)}, {LL(0x9468ee68,0xf109d747),LL(0x8c1958b8,0x35d5c367),LL(0xd1baaa07,0x58b0e5bd),L_(0x04265560)}},
   {{LL(0x03eb3ddb,0x3c1c78f2),LL(0x73aa994d,0x77b06810),LL(0x9d13c879,0x5ffbf962),L_(0x329365bf)}, {LL(0xeb311f82,0x151d1b5b),LL(0x9b162681,0xa0010f38),LL(0x67d4adbd,0x2d7119cb),L_(0x112111da)}},
   {{LL(0x3a3f8979,0x8de63a52),LL(0x0bc054e9,0xcecf35b9),LL(0xac3474f8,0xd39c2e22),L_(0x8a24d244)}, {LL(0xe1c3169e,0x72de5749),LL(0xf7cd28ca,0x1f3311ce),LL(0xaba378ca,0xacf3d959),L_(0xc4c178a4)}},
   {{LL(0x41028a7a,0x4c196a71),LL(0xeddb3096,0x07052fa2),LL(0x78d05008,0x4a5cea44),L_(0x4e82cf55)}, {LL(0xe4836382,0xbc157855),LL(0x42628bb0,0x6906bb4c),LL(0x3bec9a03,0x414298ee),L_(0x71c76fb4)}},
   {{LL(0x2dc6d80f,0x451c89a8),LL(0x0eb5ca81,0xb205946f),LL(0x85231812,0xdd102f79),L_(0xcfe70175)}, {LL(0x79cbba78,0x74f5620f),LL(0xa68f72ba,0x925ad282),LL(0x107305c7,0xa0b90dac),L_(0x71b34217)}},
   {{LL(0xd11e6b55,0xdf55e61e),LL(0x74013fe7,0x39b4240c),LL(0x384b1c58,0x117f4d5d),L_(0xb417f0cc)}, {LL(0xdee67f19,0xa5e101d1),LL(0x6a59354e,0x057e8fde),LL(0xdd613a29,0x3d053f50),L_(0x49698601)}},
   {{LL(0x0e1e0dac,0xbc28da28),LL(0x350510df,0x6476441e),LL(0xa5492aa3,0x488b3ebb),L_(0x7335580b)}, {LL(0xe4198903,0x150fa7db),LL(0x5080e530,0x1e0113ef),LL(0x41fe0adc,0x151c555b),L_(0xefba1701)}},
   {{LL(0x4e602721,0xda6ab8ed),LL(0x01c286ca,0x82be4f7e),LL(0x8f870fa0,0x67983e93),L_(0x011517ba)}, {LL(0xd1bb4bf4,0x2ffdf949),LL(0xb8ff2b68,0x4a318fa3),LL(0x57a7f08b,0x2df51505),L_(0x7f4a552d)}},
   {{LL(0xc07ee494,0x8a38d900),LL(0x15f78485,0xd40347a2),LL(0x40936fbc,0xdc403083),L_(0x26d9bf80)}, {LL(0x87df50e9,0x6682494a),LL(0x658eecc3,0x88a94156),LL(0xe2c9cc47,0xf4a7dc3b),L_(0x6742c074)}},
   {{LL(0x58b3b3f6,0xaf6f6f61),LL(0x1be73cf3,0xb37f3459),LL(0x74ffb428,0xb29ede9a),L_(0x92e686e4)}, {LL(0xa4431924,0xdfa5e88f),LL(0xe1001a22,0x01fe6a9c),LL(0xf029cdcc,0x55c88ddd),L_(0xd03747ad)}},
   {{LL(0xc359e1a8,0x69bc7a5b),LL(0x0a7f469b,0x79ced652),LL(0x8cb61ff4,0x860090cf),L_(0x25e6acf5)}, {LL(0x5ae7246e,0x7c9a5e58),LL(0xd04ea633,0x62e6dab4),LL(0x3dccdf35,0x679b9a9c),L_(0xe0d5601a)}},
   {{LL(0x9f6c163f,0x75393ffa),LL(0x503ff7fc,0x8a5ba74c),LL(0xad84a980,0x6a8d3ebd),L_(0xb711116c)}, {LL(0xdb3d29d6,0x508cd2da),LL(0x298e39d1,0xadc29bed),LL(0xd182ffce,0xf35bcd9e),L_(0x56f60d5a)}},
   {{LL(0x18ec5621,0xbb37476c),LL(0x126c6b62,0x00881f6d),LL(0x12e961ae,0xc23fa1b7),L_(0x620a85d2)}, {LL(0x116a94f1,0x18d50f3c),LL(0x90b6bf4d,0xc1b565ee),LL(0xcce6cb44,0xb9db57be),L_(0x43232f2b)}},
   {{LL(0x7e516745,0x33f18556),LL(0x34242913,0x303ca557),LL(0xecfa2caa,0xe5298223),L_(0x54c9b89f)}, {LL(0x5a38eea9,0x2bb85e07),LL(0x3640c336,0xbe45ca34),LL(0x23f72dcc,0xe33bfd7f),L_(0x1138d09f)}},
   {{LL(0x2532b3de,0x9635cec4),LL(0x99decb38,0xecdd16cc),LL(0xb6e94853,0xc90493fd),L_(0xe03d588e)}, {LL(0x7c31a9ad,0xbbcd5eac),LL(0x1a5efdc4,0x24031f80),LL(0xbd3e5f35,0xc0bbee30),L_(0x91354a2e)}},
   {{LL(0x46fe272f,0xe4fe7dd5),LL(0x40ccc046,0xcb3af2d4),LL(0xef734198,0xe37fbe0f),L_(0xd7d730ce)}, {LL(0xc1b5100f,0xe81b74b4),LL(0xccbbf03c,0xfbbba07c),LL(0xaf1eca00,0xb202f7d1),L_(0x6c3a5371)}},
   {{LL(0xb759588a,0x3c293b4f),LL(0x50ebe82c,0xccc60f4b),LL(0xe5c199da,0x9c0164e0),L_(0x09ae68d1)}, {LL(0xdbb33342,0xe57a6b34),LL(0x764aa682,0x71131364),LL(0xd2b124d2,0xc37e19f6),L_(0x0684de69)}},
},
/* digit=20 base_pwr=2^140 */
{
   {{LL(0x2e5b54d1,0x3a2fc0ae),LL(0x3a0ae219,0x810b8800),LL(0x71d39325,0xeb53d991),L_(0x1b364d9f)}, {LL(0xc225e922,0xe634fa4b),LL(0xfa979c3b,0x48c1302d),LL(0x0315e213,0x7a46c430),L_(0x7d3dfc00)}},
   {{LL(0xf75af9f3,0x69c6dab4),LL(0xcad74055,0xbd0a143f),LL(0x5f980b30,0xf457bc61),L_(0x08ab6cd4)}, {LL(0xa472e295,0x5a3b5bb9),LL(0x4a0eec60,0x1e647946),LL(0xa4e2c617,0xeb13a610),L_(0xea8ba726)}},
   {{LL(0xb1f39e26,0x8208d95b),LL(0x1d126957,0xd33801fe),LL(0x38314570,0x2b3db8bc),L_(0x68f43b86)}, {LL(0x06523bb4,0x2fa1baa7),LL(0xb3650b8b,0x7b7200e3),LL(0x222d46b5,0x5ab609c3),L_(0xe25b7f78)}},
   {{LL(0xf09c3b4a,0x2601c0c2),LL(0x10c7e593,0xb71c88f8),LL(0xd433020b,0x163a4b4f),L_(0x6ea15982)}, {LL(0xd7a199df,0x559abf4b),LL(0x168cc6db,0x8c58a3fd),LL(0x5e92d42d,0xcad8eab0),L_(0x0f718cd3)}},
   {{LL(0x5222ebab,0x1eccb2dd),LL(0x45eee5ae,0x6cca2e8f),LL(0xf0f806c3,0x93a2e216),L_(0x0b896b9f)}, {LL(0x2753730f,0x21ac6aa6),LL(0xe860a374,0x93ff82bd),LL(0xa6d03d5f,0xc336e305),L_(0x5a3ad08c)}},
   {{LL(0x05094bb6,0x7e65ba97),LL(0x927ff44f,0x69a96369),LL(0xcfc2ebd4,0x8e5841d8),L_(0xcd919ed2)}, {LL(0x7d67ba45,0xd72479fa),LL(0xf9079703,0xb458951d),LL(0x11b2590d,0x12b5f89b),L_(0x9e78739c)}},
   {{LL(0xb68ddf1d,0x64971999),LL(0xcc5dc1ba,0x0f78a155),LL(0xc7f1ee3f,0xbbfeebe2),L_(0xa475b47a)}, {LL(0xae6b1a67,0xd00f3342),LL(0xaa2739d6,0xcbc8eae8),LL(0x6bcba57f,0x08421638),L_(0xf50c5c89)}},
   {{LL(0x21fce16e,0x56ea720e),LL(0xc14eb574,0xb07c02a7),LL(0xe5e93b80,0x1a141ce9),L_(0xf206da59)}, {LL(0x400443af,0xff6ca106),LL(0x85cacb30,0xe7e89a41),LL(0xe54c35f6,0x73777848),L_(0xeecb8a06)}},
   {{LL(0x27098970,0x27c35c5a),LL(0x88bff7c0,0xd3495797),LL(0x2aa31cbd,0x84083f2a),L_(0xee8007b7)}, {LL(0xd736e6e1,0x7c2ce5db),LL(0x46d1faf9,0x7eccf318),LL(0x0d96af5f,0x7894ccc6),L_(0x143733e6)}},
   {{LL(0x38e5c20b,0xf8d1a779),LL(0x74b4b8c0,0x1407c96c),LL(0xaa226358,0x0f677dc4),L_(0xbce88332)}, {LL(0xae06373b,0xa1eff8f8),LL(0x6f4d4505,0x46e8175d),LL(0x37fa6b50,0x12414006),L_(0x078af83c)}},
   {{LL(0xaa9911e2,0x6038786a),LL(0x043fa7bc,0x9ec4cf47),LL(0x39594053,0xeeffb86e),L_(0xf30130d5)}, {LL(0x93573962,0x8e9bddbb),LL(0x6b822690,0xa5656b7e),LL(0xfd18acdc,0xf63c9ead),L_(0xa29c50ef)}},
   {{LL(0x7a68f812,0x875129d8),LL(0x23706eff,0x43107ab7),LL(0x343d01b4,0x0f0278ce),L_(0xd1d2ff3f)}, {LL(0x947dc2e4,0x62e80eed),LL(0x76f993b1,0xb237e733),LL(0xd8eb588f,0xc6cb2916),L_(0x3e93bfcf)}},
   {{LL(0x2323cfa1,0x40dc414f),LL(0x02fc59aa,0x81bf9064),LL(0xf4b7afef,0x2f863790),L_(0xdf1eb2ce)}, {LL(0x2da3a955,0xc2be20ad),LL(0x05119c9a,0x826e16e4),LL(0xa27da69d,0x59ba9dd2),L_(0x50be3370)}},
   {{LL(0x702f4a30,0xe5b817ac),LL(0x62b0b4a9,0x732be8dd),LL(0x0b744a18,0x49eb11c0),L_(0xdb8066e2)}, {LL(0x61315a81,0x60bc3f3b),LL(0x9db8094e,0xa033694a),LL(0x1a336970,0xe5728d57),L_(0x98764bb8)}},
   {{LL(0xa31f05a5,0x471336b0),LL(0xc97e0c25,0xe1b3c5c4),LL(0x40016340,0xd00385b4),L_(0x8681b939)}, {LL(0xf3dbd1c9,0xfb0504e6),LL(0x190db5e8,0xf84aaa64),LL(0x73d7879b,0x5e7d8612),L_(0x8f037a48)}},
   {{LL(0x4bb632f0,0xb920e089),LL(0x81a11e46,0x5fd26f9a),LL(0xad7ce91c,0xd873ff38),L_(0x463f5eef)}, {LL(0xe973b348,0xa997a44a),LL(0x67e4b48b,0x7c4b4d96),LL(0x76b8b69f,0xc007a35a),L_(0x0ebb0cc7)}},
   {{LL(0x066bf5fc,0xc6331e3d),LL(0xd49fdc60,0x01bb84a6),LL(0x3f1c848c,0xae53c6d2),L_(0xa68e50ed)}, {LL(0x64a19a00,0xa6271228),LL(0x1c7298ae,0x6345be32),LL(0x34a3a582,0x6666c91c),L_(0x1bb3ce87)}},
   {{LL(0xe3fc9fee,0xabcce7ab),LL(0xfa1beb7d,0xdf35c272),LL(0x9cab965d,0x7c86ce27),L_(0x145e311e)}, {LL(0xadb37a6d,0x3f4563ca),LL(0xbd40ee7b,0xc5e44069),LL(0x077a6fa1,0x6ab23a8b),L_(0xdc9e2258)}},
   {{LL(0xe14e323f,0x27ccdc44),LL(0x8d1e93bc,0x9fd47cad),LL(0xac370974,0xf5fc7d6a),L_(0xbb95cf97)}, {LL(0x3c654b26,0x65e7b71e),LL(0x0d3d8936,0x69b3aaf9),LL(0x6cfc89cd,0xcfd74f9d),L_(0x46c8cccc)}},
   {{LL(0xe59aef1b,0x82c35cac),LL(0x4b6e4cd2,0xd343c61a),LL(0x5bf959ed,0x77b74152),L_(0x14153467)}, {LL(0xabc9828c,0xc645fe14),LL(0x44ed7180,0x251c7bd5),LL(0xb00a2c9f,0x94c68e2e),L_(0xcd40eadd)}},
   {{LL(0x90c9c5ec,0xdb397bc5),LL(0x0bb14c2d,0x0029fc46),LL(0x2727c32b,0xbdde6547),L_(0xf5120538)}, {LL(0x8748603f,0xca65a20c),LL(0xb71dac7a,0x79d195d6),LL(0x7b4ac903,0xf1e5f556),L_(0x83bcccf1)}},
   {{LL(0xd8d5a0e2,0x35177eb8),LL(0xb2a6346d,0xe17edde4),LL(0xcdb1c427,0xc5c607ff),L_(0xf6acdcca)}, {LL(0x7045642b,0xf198276e),LL(0x5ddfa215,0xa8387d90),LL(0xc1a3402a,0xc070af80),L_(0x0cf324a8)}},
   {{LL(0x89428955,0x4be4c6c0),LL(0xddf4319c,0x076b38c2),LL(0xcc96208b,0xd1d87cd0),L_(0xc3e2cfee)}, {LL(0xb63b328c,0x1fa3dfb1),LL(0x89c310af,0xbcb33f40),LL(0xab955eb2,0x1e3849a6),L_(0x4e9da2a4)}},
   {{LL(0x572060d6,0x4a409b86),LL(0x4a32be16,0x01fee4f3),LL(0x5593d29d,0x1f66b255),L_(0xa18a71de)}, {LL(0xdc92993c,0x871121b8),LL(0xd5a52a49,0x2885efde),LL(0xdb77ca39,0x35178007),L_(0xe87f7b56)}},
   {{LL(0xb0b54185,0x7b11bf25),LL(0x78cced92,0x5f580048),LL(0x568cf974,0x1aebc2f4),L_(0x90437303)}, {LL(0x380e1852,0xe1d4cf60),LL(0x81708f04,0x335221a8),LL(0xfcd5e396,0xe8e310f1),L_(0x67164e9b)}},
   {{LL(0x5cc646e4,0x4435eb3f),LL(0x309bf5d1,0xc4516a98),LL(0xfa094c38,0x0f0f519b),L_(0xa7712b6a)}, {LL(0x063b90c9,0x13b443e8),LL(0x3fc80f73,0x588eabab),LL(0xc2640659,0x6f1b036b),L_(0xcc979824)}},
   {{LL(0x15f68a6c,0x2c10fe82),LL(0xfd4280b4,0x4be94dbf),LL(0x0f01bc51,0x8ebbdd0a),L_(0xe8a94761)}, {LL(0xf5c8cf6e,0x3331db35),LL(0xe9e32c12,0xee34fb22),LL(0xdfdd390f,0x7d9acfa0),L_(0xf46a833d)}},
   {{LL(0x4db03d63,0xb1f7d640),LL(0x50d5c66d,0x77b9e67a),LL(0xd311bbb8,0x2df06964),L_(0x4d4065dd)}, {LL(0xb1ba8be7,0xc5f2e7e7),LL(0xa84be5a0,0x3c7c5264),LL(0x704fb715,0x72ab97c4),L_(0xe211a72c)}},
   {{LL(0xde6eb928,0x22e459d0),LL(0x262ef282,0x78e351ef),LL(0xcdae125f,0x400cd2e5),L_(0xc83f210b)}, {LL(0x1982c6e5,0xb3e5fe6c),LL(0x87c8591a,0x13591995),LL(0x83fd93d2,0x072fa5b0),L_(0x7d701f58)}},
   {{LL(0xae57420d,0x2dda42fb),LL(0xa6490705,0x0c8f6d52),LL(0x5e3ec836,0x974661b3),L_(0x77bc552c)}, {LL(0xca185c95,0xf43d8647),LL(0x351407f7,0x3094919d),LL(0x93522ec1,0x2effc179),L_(0x49a92d02)}},
   {{LL(0xad6aa522,0x1178f9fd),LL(0x88ddd25b,0x7c35a7b4),LL(0x4985102d,0x0d3de452),L_(0xf0277afe)}, {LL(0x9952ce22,0xf3571744),LL(0x53a80f8c,0x33bbd8c5),LL(0xa11d84cd,0x8cf3c9a4),L_(0x7920692b)}},
   {{LL(0x19d5d3aa,0x3259f507),LL(0x8dd2cea1,0xfaa534d0),LL(0xb1de59c3,0x964e43aa),L_(0xd9cd158a)}, {LL(0xb8fe9ccf,0xf7812a29),LL(0x483a8ca3,0x5a40b462),LL(0xf93b681f,0xeafa2982),L_(0x22d0852e)}},
   {{LL(0xccafe782,0x4e3898e7),LL(0x178b364e,0x122c0954),LL(0x1e46cbf3,0x07005a6a),L_(0x017cc249)}, {LL(0x1f834707,0xe3d63870),LL(0x9df8269b,0x84c46f73),LL(0xf32797a5,0x1360de40),L_(0x70a39a32)}},
   {{LL(0xe6611eb8,0x58005efe),LL(0xefaea42e,0x9d406314),LL(0xcdf53ee0,0x8d9f77b8),L_(0x2be3de21)}, {LL(0x3152090c,0x219fc4d6),LL(0xdfbb4054,0x2e867eb0),LL(0x370b0058,0x0df48ecb),L_(0x67c50061)}},
   {{LL(0xd35d5628,0x7d760756),LL(0x3cb96c10,0xaff6e332),LL(0x87378266,0xc87eb4e6),L_(0xbb1b1686)}, {LL(0x719b7dba,0xf625cc5d),LL(0x7f6f6bfb,0xc5c21343),LL(0xd1702bfc,0xd7459a05),L_(0xb5757892)}},
   {{LL(0x07b102d7,0x2285b7c5),LL(0x77777ad0,0x90a0453e),LL(0x2a9f4627,0xa5cb03f7),L_(0xbdaaa412)}, {LL(0xd6bb4325,0x6fb99eeb),LL(0x8aa2c4b5,0xc61bae52),LL(0xec729246,0x9b9f8d87),L_(0x995887b7)}},
   {{LL(0x714a54b4,0x786ccb7c),LL(0xae2ffd66,0x1df96294),LL(0xb5661480,0x0b294654),L_(0x42cc5e69)}, {LL(0xc6f02a04,0x1634f1da),LL(0x21f0caa7,0x75d2244a),LL(0x80410ce5,0x6597b349),L_(0xb9e50c3a)}},
   {{LL(0xd19e2b9f,0xbefea27e),LL(0xca541338,0x53b70277),LL(0x0bc9bd26,0xd300e1fc),L_(0xed2c5b84)}, {LL(0x935f9273,0x731c6b5d),LL(0x10fed3e2,0xac105b56),LL(0x6d0357ab,0x30c7c4b6),L_(0xfdf04b77)}},
   {{LL(0x32c68e8f,0x6198d0bf),LL(0x49f0ac87,0x02928249),LL(0x5908bfa6,0x2b72f231),L_(0x1e0626b8)}, {LL(0x87d56448,0xb70c5fdd),LL(0xaffd0d74,0xc8176ae0),LL(0x30464156,0xbe7c0da7),L_(0xf7d2ffb9)}},
   {{LL(0xffa548f9,0xd71029f6),LL(0x2ac8267b,0x5922838c),LL(0x4f8afc3b,0x2d787218),L_(0x10f9a786)}, {LL(0xb358d2a8,0xf19c8b54),LL(0x5209c3c7,0xf590721f),LL(0x0212109d,0x4e16a382),L_(0x3cd77b54)}},
   {{LL(0xef4b9410,0x4e89b491),LL(0x77501651,0x417ff9fd),LL(0xdbb5243c,0xd24a6908),L_(0xc09c5df1)}, {LL(0x2c39f4c4,0xeb16d02d),LL(0xb28822ac,0x4cd4a553),LL(0xbf46b5bb,0xde8a18e1),L_(0x47566077)}},
   {{LL(0x0937292b,0xa4a4f179),LL(0xdff0cf1f,0x01b86895),LL(0x34f0c71b,0x3fe5a3b7),L_(0xae1fff30)}, {LL(0xe080e0da,0x95b01982),LL(0x5317a578,0x3064e918),LL(0x8872e118,0x154f511a),L_(0x28ad7f46)}},
   {{LL(0x9dbd0087,0x3c4162c3),LL(0x88636424,0x58631293),LL(0x69f32e25,0xf3a9fd92),L_(0x049b5c0f)}, {LL(0x8d93c391,0xd8613d4d),LL(0xcf94b297,0x74bd10f5),LL(0xf99f0a15,0xf6d69ab3),L_(0x8d722881)}},
   {{LL(0x67ba729a,0x2829f720),LL(0xc6f3cc19,0x88fffa70),LL(0xf0c5f76b,0x0d44b294),L_(0xda7df6fd)}, {LL(0x033be0dd,0x35516898),LL(0x4c9c487f,0x50dfac1a),LL(0x312edd81,0x14a64ecf),L_(0x2db649b2)}},
   {{LL(0x4933ac70,0x4e65d480),LL(0x17bb4dfc,0x8f9141c2),LL(0x7b8827b2,0x6ffab99a),L_(0x788afe5d)}, {LL(0x15192ba1,0x6ed82d80),LL(0x75adf2d6,0x7be9df26),LL(0x99ce22b1,0x957976c9),L_(0xf8f37372)}},
   {{LL(0x2204ff70,0x3ad3db41),LL(0xa4617ca1,0x7ca2697e),LL(0x8f228ced,0xc881f9da),L_(0x86ddd01a)}, {LL(0x6e35e7e8,0xd1467d74),LL(0x41eb80aa,0xa4eadfce),LL(0x875ce5dd,0xaf17e700),L_(0x398d4c2f)}},
   {{LL(0x3e59ab08,0x165575cd),LL(0x7c2f636f,0x55e6cf6d),LL(0x89f1290e,0xeeb00b1e),L_(0x2cccc148)}, {LL(0x264e5c6c,0x7200e28c),LL(0x5409cce2,0xaf2b4dc4),LL(0xf8e920e1,0x3586fdc4),L_(0x11976312)}},
   {{LL(0xab519f43,0x45934ad0),LL(0xa0a5bb1b,0x0eda4d94),LL(0xbcff29b2,0x9f749e30),L_(0x2f73a7d6)}, {LL(0x2a833e74,0x9ed5bbc8),LL(0x111b6548,0x83945e37),LL(0x7b77f3f9,0x9b568171),L_(0xf1563350)}},
   {{LL(0x999312c1,0xbde3fb88),LL(0xaa153efc,0x4ead50eb),LL(0xf24dc769,0x91718221),L_(0x3207691f)}, {LL(0x1445f929,0xa962b883),LL(0x8a1416f3,0x6db04bb2),LL(0x2992be20,0x388328fb),L_(0xa7c4a359)}},
   {{LL(0x92efc55b,0x709d7f62),LL(0xf678058c,0x08e33ff6),LL(0x9db19687,0x8d882bae),L_(0x81afb521)}, {LL(0x462b58d3,0xce4ef2a4),LL(0xc6ecf132,0x21f09167),LL(0xab9bbd79,0x3831d678),L_(0x4c11a697)}},
   {{LL(0xda7c3371,0x8a72049c),LL(0xc3e5138e,0xc7decc6d),LL(0x9235657e,0xf870194a),L_(0xeddbefb7)}, {LL(0x306c9631,0x85d843d8),LL(0x912ed2a5,0x27180693),LL(0xdb45568f,0x453cbf16),L_(0x410259cc)}},
   {{LL(0x3f11cacd,0x10263ac7),LL(0x69fa2805,0xb832a378),LL(0x6120289b,0x86d5285d),L_(0xece730a0)}, {LL(0xabd38f00,0xbf1197bb),LL(0x286f4724,0xdbbb1a26),LL(0x0b0ef425,0xf397181d),L_(0x24bb4f4f)}},
   {{LL(0xe24a9a47,0x162331a8),LL(0x536a5ea6,0x2bfbcebe),LL(0x88c29411,0x3da5a14f),L_(0x9cb76097)}, {LL(0x5145ce76,0xef6b8093),LL(0x59e86f3c,0xe0a56e13),LL(0x817764eb,0x58eed1b6),L_(0x8010fc3d)}},
   {{LL(0x0b1262b6,0x9a5b60b7),LL(0x108b3068,0x14ec42e4),LL(0x1312c3e2,0x8dd4b81a),L_(0x0253e9dc)}, {LL(0x253ff39b,0x5eadccd1),LL(0xbd75013a,0x87febbae),LL(0xc60daeca,0x20b2a500),L_(0xbea0d3fb)}},
   {{LL(0x20482528,0xfb3fb2a4),LL(0x72db2c7a,0x3620dd94),LL(0x864ce0ff,0xe0956799),L_(0x2d7ec9e2)}, {LL(0x86bb3cca,0xdfc7056e),LL(0x77bfd4dc,0xf75d4a37),LL(0x0f42ffde,0x110fb370),L_(0xf2040e52)}},
   {{LL(0x38389a23,0x2100e0ba),LL(0x6fd68890,0xbf38ce28),LL(0x67978ca8,0xb3a94049),L_(0x279171ad)}, {LL(0x4b5e0e0d,0x0230458f),LL(0x41165c1f,0xb73bda2e),LL(0x8eadb8c9,0xe9eaef42),L_(0x2345e24e)}},
   {{LL(0x9fe296bd,0x41f15f29),LL(0xaa889c09,0x8ccda12e),LL(0x6adf563e,0x559edd7a),L_(0xdd414cb0)}, {LL(0x84d15464,0xad8084b6),LL(0xbef296d5,0x3238cf0a),LL(0x0ee342aa,0x074068ff),L_(0xc9f3a4e6)}},
   {{LL(0x374364db,0x4db1a4b3),LL(0x983e29af,0x190ecab8),LL(0xc26d69a7,0x8e174c76),L_(0x2739f19a)}, {LL(0xea361aa4,0x4c902613),LL(0xcd082b89,0xbbc1e2c1),LL(0xa0661a39,0xd0352acc),L_(0x6d5bd738)}},
   {{LL(0x6b165ee3,0x0f46deb2),LL(0xe05a9d3e,0xbaae3236),LL(0x68d899f5,0x2fcbd485),L_(0xd426fbf8)}, {LL(0xfce9e8db,0x70b0b8e7),LL(0xbb075381,0x24c285b0),LL(0xeb323221,0x933c7339),L_(0x90f21e83)}},
   {{LL(0xa437ec78,0x65c3ea6f),LL(0x26f5292f,0x16eebdc4),LL(0x2f10ff0d,0xd0e0dfce),L_(0xe59106aa)}, {LL(0x47d3151e,0x334ef2fb),LL(0xe1d13288,0xa14975cc),LL(0xb2dd6c80,0xe941b704),L_(0x428af6d0)}},
   {{LL(0xac3814f9,0xf4357fa6),LL(0xe9634ed9,0x3e28518f),LL(0x92f270b2,0xb8fe4e04),L_(0x4ce90044)}, {LL(0xc454c0af,0x9137e5a6),LL(0xb5cf536c,0xff154124),LL(0xaa1f4551,0x0024f90b),L_(0x981784b9)}},
   {{LL(0x9e485500,0xb04608de),LL(0x5269aa1a,0x6eb17af7),LL(0x8ce1ebee,0x3d279465),L_(0xc66381e8)}, {LL(0x0b5ed745,0xc4fa7ed6),LL(0xdeb29303,0x601d19f0),LL(0x510a3c8d,0x2ce166ff),L_(0xaa227120)}},
   {{LL(0x1c8b8392,0x992398c6),LL(0xa3c32177,0x98eae201),LL(0xcd19850d,0x70b1a326),L_(0x9505296f)}, {LL(0x958225ae,0x41bcad26),LL(0x0d11515a,0x938c115e),LL(0x04883b49,0x5a5bf602),L_(0x72d1e180)}},
   {{LL(0x597adc60,0x5898ed4e),LL(0x63896a2f,0xa9c6e64f),LL(0x82a0ed92,0x71349776),L_(0x71b9b0ee)}, {LL(0x216fc62a,0xd360630c),LL(0xbd440771,0xa2285bdf),LL(0xa49907da,0x28e79328),L_(0xad498ec7)}},
},
/* digit=21 base_pwr=2^147 */
{
   {{LL(0x9dfbcf8e,0x5e338a94),LL(0x0324aba6,0x9f1b5252),LL(0x970769c2,0x32ea5aeb),L_(0x82c3b64b)}, {LL(0xfeabe87b,0x80346bd1),LL(0x98111ddd,0x63f11cc5),LL(0x2ef9e31e,0x75819a03),L_(0x6e0e0fd4)}},
   {{LL(0xead2c5e0,0xd481d37a),LL(0x676ef7cb,0x83660242),LL(0x6ddd542a,0x4bcdf56d),L_(0x0807b6ee)}, {LL(0x88b2540c,0x99cfbaa1),LL(0x440a4586,0x92cdeb4c),LL(0xaab76f82,0x62bde53d),L_(0x31ca293a)}},
   {{LL(0x88edf264,0x279a665d),LL(0xc73f3e56,0x4837e985),LL(0x51cce02b,0x925d55fe),L_(0xba48033b)}, {LL(0xfbc2fa3e,0x3c3ccfff),LL(0x09219d1a,0xa32d9d41),LL(0x7bba7cbf,0xf5c59984),L_(0x1f9a5c08)}},
   {{LL(0x1175b9ba,0x4079f189),LL(0x150e6d32,0xe9e1ab30),LL(0x8b6cc92b,0x2d425400),L_(0x274ec0c2)}, {LL(0x8f0527d9,0x6d261b01),LL(0x4ef086f3,0x63cc5685),LL(0x29dec1b9,0x7f963f71),L_(0xea97e94e)}},
   {{LL(0xd50b2092,0x9e517139),LL(0xf9867b99,0x32ba3680),LL(0xbd760d63,0xafdf07e9),L_(0xa9b6e9ec)}, {LL(0x8b644aef,0x6d61dc07),LL(0x93df1b2e,0xe8d9afc6),LL(0x3387d03a,0x32d7f165),L_(0x759fb34d)}},
   {{LL(0x41d99c15,0x08345264),LL(0x34f3b1e1,0x17c124bd),LL(0x20468107,0xc8425e31),L_(0xa035ed92)}, {LL(0x19019adc,0x15129a66),LL(0xd78546b5,0x4c880bae),LL(0x2a03f320,0x04faf0ec),L_(0xf88d60ed)}},
   {{LL(0xcbda1d0c,0x99a20d9a),LL(0x31f03c7f,0xc7f9c96f),LL(0xbdab0f5d,0xcfc1004b),L_(0xc665b5f0)}, {LL(0x24cf476d,0x57bee516),LL(0xbaab1072,0xcf987028),LL(0x64de4da0,0xa4efa899),L_(0xb3c284d3)}},
   {{LL(0xec9ac930,0xb6ef6e4f),LL(0x7bd6b2eb,0xf97982f6),LL(0xb87b9f50,0x3506344d),L_(0xe38cc59c)}, {LL(0x0df2358f,0x98614bb4),LL(0x4daab44e,0xa73a33ba),LL(0x3957de53,0xc680567a),L_(0xcfa42f96)}},
   {{LL(0x6e9aa82b,0xad09cde6),LL(0x89749594,0x88dae77c),LL(0x686f22a8,0xe7736848),L_(0xd741b21a)}, {LL(0x9e20d489,0x08ab4cce),LL(0xb87c7bef,0xcc175c96),LL(0x982331fd,0x177b6473),L_(0x63140d6e)}},
   {{LL(0xb3fe0b33,0x061731af),LL(0x785b32f1,0xdee055b2),LL(0xc296cafa,0x1de52e72),L_(0xb10ce663)}, {LL(0xc5a7f1be,0xc6a06418),LL(0x50016567,0x1713586c),LL(0x4502900b,0x2c001a59),L_(0x450b28d5)}},
   {{LL(0xcab8d38d,0xc5913490),LL(0x9e1e0ce7,0x343c325b),LL(0xc5f9091b,0xe7590273),L_(0x44739896)}, {LL(0xee5b12cc,0x449da236),LL(0x3913a8d2,0xd0e8b775),LL(0xd9db43d7,0x0204d37f),L_(0xbcc28709)}},
   {{LL(0xfda66aba,0x0ba00094),LL(0x00c25d60,0x034594b0),LL(0xb0ccb9b4,0xaecf0812),L_(0xebecd41b)}, {LL(0x95fb80f1,0xe2cb6f9c),LL(0xe9f56be8,0x33f77310),LL(0xab12b176,0x77e2c3e0),L_(0xa7a303fc)}},
   {{LL(0xaa4bbfa4,0x0ca6bd41),LL(0x829f1bab,0x9f37381d),LL(0xf8aa85f0,0x63a37ba6),L_(0xe64c6ed3)}, {LL(0xb1db7703,0x9ba22c56),LL(0x14c74bf2,0x26369a5a),LL(0x57cdabdc,0xa2f36ff4),L_(0x69603458)}},
   {{LL(0xd1e5c2d6,0x049ce7f9),LL(0xdda045cf,0x6a8702d5),LL(0xae642723,0x444a4ca8),L_(0xc9e8efd0)}, {LL(0xbc74c0b8,0xee8ff912),LL(0xa5f0a182,0x7cacfd02),LL(0xf34c0a0e,0x0bdcdecc),L_(0xf174835c)}},
   {{LL(0xad6aaeda,0x262941fe),LL(0x50dd89e9,0x052230cb),LL(0x52bfaad6,0x3d3651a5),L_(0x42da9f6d)}, {LL(0x1803593d,0x72f24771),LL(0xfb5c67bc,0xf8d5c69c),LL(0x6f3ebdef,0xa8074e96),L_(0x5d95664e)}},
   {{LL(0x3cfb3680,0xfbd1a191),LL(0x14bfa4e1,0xd4f7de76),LL(0xf38ce345,0xfcdb2abd),L_(0x45ebd21e)}, {LL(0x1d84093e,0xb59f480a),LL(0x3957e934,0xcc218197),LL(0x7c1b8e5b,0xe4604f76),L_(0xc58f2c48)}},
   {{LL(0xd2d407e1,0xc8a88d72),LL(0x21606f53,0x9385e572),LL(0xfba9c9f7,0xfb7d83f0),L_(0xf57dce48)}, {LL(0x37eda284,0xfb80fd4f),LL(0xc8023a07,0xf3549107),LL(0x71a25cd3,0x7a9a0b44),L_(0xb3c48734)}},
   {{LL(0x885801a9,0xf3798b62),LL(0x9856b7c0,0x08f77ad9),LL(0x04ab2722,0x471ef449),L_(0xe5f3587d)}, {LL(0xc09f1c74,0x21b96d5c),LL(0xb0c73d23,0x5e2a228b),LL(0xa3366107,0x69323a1b),L_(0x759bf017)}},
   {{LL(0x11492138,0x80978740),LL(0x4e3559fa,0x456876b0),LL(0xe7559081,0x56ad6e70),L_(0xdd9852be)}, {LL(0x8ce4faa0,0x0e7594de),LL(0xe4523a8a,0x3999a1a1),LL(0x4a40d4e5,0x8ca5687b),L_(0xfaf902fd)}},
   {{LL(0x854da328,0xd2ade9c1),LL(0x683ea003,0xb2369f45),LL(0x6bbf8ead,0x26a2a12e),L_(0x65fa7d53)}, {LL(0x44971ebd,0x6154c3a8),LL(0xd7f809d8,0x9e3b11ad),LL(0x20ed9167,0x24d19af8),L_(0xa692a83a)}},
   {{LL(0x26ad917a,0xd8e0ff42),LL(0x90e27172,0xd10a1313),LL(0xa90bf44d,0x8cb2abd0),L_(0x130d21df)}, {LL(0xea4790cd,0xa3cb4c6a),LL(0x65186f8e,0xfd47c901),LL(0x0c9c373c,0x3514478f),L_(0x5307b692)}},
   {{LL(0x80001060,0xd8568305),LL(0x34f0a25d,0x6a578b8b),LL(0x039f50cc,0x0eadf2ee),L_(0x6a3cf739)}, {LL(0x8ca51f3b,0x6a9a2a61),LL(0x5c707fb6,0x92f53d4f),LL(0x22360abf,0x756df232),L_(0xa7a12676)}},
   {{LL(0xa54ee092,0x232b0eae),LL(0xfac26fa0,0x2030a397),LL(0x91c0d8b3,0xc85b9885),L_(0xcd36972d)}, {LL(0xf99f3cb8,0x2b28cccb),LL(0xa2197bb4,0xd5dc11dd),LL(0xb00034b6,0x22200d9d),L_(0x398b8d08)}},
   {{LL(0x48067657,0x3e7e898a),LL(0x14230734,0x429f6872),LL(0x7bef4c47,0xe3a1abc8),L_(0xb3564180)}, {LL(0xb41998c2,0xeed81406),LL(0xa7ce6e62,0xe40d75be),LL(0xf02dd5b1,0x9d22967d),L_(0xa3ff259d)}},
   {{LL(0x669ca4f5,0x25d2230b),LL(0x2d777318,0x7178f19a),LL(0x80147bac,0x6f460f52),L_(0x22279805)}, {LL(0x7e64abfe,0x20550e61),LL(0x7880c924,0x79d1e82f),LL(0x6428d5fc,0x9a19cfda),L_(0xd1beb76b)}},
   {{LL(0xba10e2eb,0xe36b37d3),LL(0x5d87cbda,0x2be422df),LL(0x404781e8,0xc8f5e0fe),L_(0xa1bc9eda)}, {LL(0xd68a56ff,0x23299058),LL(0x970d11fc,0x7d7df62c),LL(0x785e8445,0x8ed8b412),L_(0xbad1345d)}},
   {{LL(0xea68889d,0xce7aa168),LL(0xd74f6437,0x77a09f9a),LL(0x1f1dc772,0xf497718e),L_(0x36827efd)}, {LL(0x53f8a4a5,0x978b617c),LL(0x4cb29feb,0x11c9e4d8),LL(0x6a64b734,0xe04077a2),L_(0x86b8cf3a)}},
   {{LL(0x81d83b84,0xed710d97),LL(0x3a39d170,0x9d798cd7),LL(0xef236d5b,0x0683a7e0),L_(0xb9141322)}, {LL(0x295bb2f8,0x05109f73),LL(0xdade7c00,0xb994c8eb),LL(0xf251df45,0xc11452de),L_(0xb48e9a05)}},
   {{LL(0xab19c0be,0x4aaf45f4),LL(0x17039bf0,0x12b12bdf),LL(0xb97b72ee,0x30ce8053),L_(0xa7a81755)}, {LL(0xa5e18f4a,0xc81db67a),LL(0x80753cae,0x1c4a9982),LL(0xe3019a4d,0xd1c81400),L_(0x80f8c3da)}},
   {{LL(0xeb75d498,0x165046c8),LL(0x030a06a1,0x232d37a1),LL(0x09c6053c,0x3932bda3),L_(0xcb39f998)}, {LL(0x95de7723,0x87fe7976),LL(0x0499f81d,0x216feab0),LL(0x7ae298d0,0x2959c264),L_(0x1edb6b93)}},
   {{LL(0x916476f9,0x6ab52744),LL(0xdc732139,0x45a91c78),LL(0xbd7da6c3,0x5ac1ef95),L_(0xe2e0217e)}, {LL(0x874209cf,0x63f9997a),LL(0x5cecf1c0,0x9d1320f0),LL(0x5082d35e,0x7b11e2cc),L_(0x04cc9aa3)}},
   {{LL(0x481e157c,0xfecff2a6),LL(0x4f8b44cc,0xa01ff932),LL(0xd783714c,0x9c0232a4),L_(0x90d1c36e)}, {LL(0xad070ffb,0xbd5e004d),LL(0xf22c1c14,0xec3d0dd1),LL(0x77b24ec6,0xc4fac858),L_(0xf15be1db)}},
   {{LL(0xa4425706,0xfa634b73),LL(0xc11ab7e2,0xe2196b67),LL(0x39a0b0f1,0xa62656eb),L_(0xdf5e8e89)}, {LL(0x10e754f8,0x35af495a),LL(0x059e7fc2,0x8885a6c4),LL(0x5721bb41,0xfcc7a8b4),L_(0x18530aec)}},
   {{LL(0xce8caf63,0x1efc0864),LL(0x7e7fb20d,0x45a5d6d7),LL(0x319cbee0,0x32603543),L_(0x216780a7)}, {LL(0x728ec24d,0x771c5008),LL(0x332b0e8d,0x88a3ec11),LL(0x27842539,0xfd61d211),L_(0x176fee74)}},
   {{LL(0x636c68f5,0x6d5e6c03),LL(0x8f5f36df,0x99e754bc),LL(0xff864830,0x04fc8d4b),L_(0x35ae923b)}, {LL(0x8c2df569,0xd03159ad),LL(0x6c4574ce,0x802cc48f),LL(0x95e44ad6,0xecc940e1),L_(0xacd13e60)}},
   {{LL(0xaeaeda46,0x24bc05c7),LL(0x55d8d9f7,0x6d33fac1),LL(0xba1700c6,0xb1dbe56e),L_(0x12671c92)}, {LL(0xa3906992,0x7c873618),LL(0x28f4b581,0x5d43aa50),LL(0xc27395ee,0x31a82a43),L_(0x84bd2bdf)}},
   {{LL(0x85fa613e,0x19e04eb7),LL(0xd6d7b2f3,0xacf5f895),LL(0x7d71e73c,0x5dd55589),L_(0x6b4ddaaf)}, {LL(0x5cbbc28d,0xb3874051),LL(0xb395499e,0xf0404258),LL(0x56ce4cbb,0x5961b289),L_(0x2960f1ca)}},
   {{LL(0x08af573f,0x739615c5),LL(0x72405988,0xdcdfa5f7),LL(0x03300371,0xe254b688),L_(0x6195855d)}, {LL(0xe4e87561,0xd298a295),LL(0xf9205a13,0x156d1dbc),LL(0x11ffb781,0xe00ca07c),L_(0xd638a7b0)}},
   {{LL(0x9e23a234,0x970fe955),LL(0x62d115d2,0x4374bbb8),LL(0x4db47695,0xb3652081),L_(0x6565367b)}, {LL(0x49a6f033,0x53a17dff),LL(0xbb00a05a,0xef6118bc),LL(0xc7f0a623,0x2971e39b),L_(0x091d591c)}},
   {{LL(0xde40fbc4,0x3c51221c),LL(0xd273fed4,0xda08dbf2),LL(0x7eaf4c1f,0xc3cadac4),L_(0x155b21cd)}, {LL(0x8e6c2580,0x824bf439),LL(0x6d0eb93f,0x16f47b06),LL(0x2b5af50d,0xe79201e3),L_(0x50f56685)}},
   {{LL(0x2cbaab9e,0xaf65e93d),LL(0x64cb8361,0xdcc125a4),LL(0x2b08724e,0x5d31d941),L_(0x128434b7)}, {LL(0x11800db2,0xd87b7933),LL(0x6349bb61,0x41b8d379),LL(0x8eabd8ce,0x46f9abc2),L_(0xf5ec4590)}},
   {{LL(0x226b273d,0xde41b0d7),LL(0xfcdd55ff,0x16980cb0),LL(0x5f8a8d74,0xad3ba05b),L_(0x5c84894a)}, {LL(0x8dc5f338,0x1041454e),LL(0x7425541f,0xc1a5e43c),LL(0xf6c08550,0xbe1bce56),L_(0x48f611af)}},
   {{LL(0xf3ceb53f,0x31e207aa),LL(0x505aaa25,0x7b5d7c2b),LL(0x9df3ed0d,0x6ff3aa7e),L_(0xecdfa689)}, {LL(0xe6296cd5,0x4660e301),LL(0xaac1b188,0x8e86f81b),LL(0xe2709e66,0x1690d7e4),L_(0x9ad43bd5)}},
   {{LL(0x31a8090d,0x9e4966d9),LL(0xe9803ed7,0x2923afcc),LL(0x94a51c0a,0xc6ffb2f1),L_(0xac8d73e5)}, {LL(0x7e9362e3,0x87ee51c6),LL(0x0f2f29a5,0x9df72731),LL(0xcf07fe4d,0xf7993328),L_(0x11a086fc)}},
   {{LL(0x366308d2,0x482c8576),LL(0x7008cc6c,0x04061b0e),LL(0x4ea34321,0xea139dd1),L_(0x0db793be)}, {LL(0xd163362c,0x2f88c485),LL(0x46daeb88,0x20d2b379),LL(0xa9529777,0x8fc9a2fe),L_(0x685903b2)}},
   {{LL(0x4733e75d,0x160fbbaf),LL(0x85dc20a6,0x8dfd433f),LL(0x40ce263c,0x33ee3772),L_(0xf18c955f)}, {LL(0xbc745c54,0x7c56f68e),LL(0x629b2ed0,0x7ba316a5),LL(0xd3751f88,0x9be35b3d),L_(0x87658375)}},
   {{LL(0x65718fca,0x4c71f19d),LL(0x2e6a930b,0x58ad9bf9),LL(0x6ce370be,0x9c7c9090),L_(0x9fe1a225)}, {LL(0xd3e1755a,0xdfc19acc),LL(0xbc4095c2,0xeb61289d),LL(0x102343b7,0xa07aed0a),L_(0x75593f8f)}},
   {{LL(0x892da157,0x6e87ec4f),LL(0xe6abe1ed,0xe3973fff),LL(0x32e1d48d,0x9c1aa3b0),L_(0xbc16fee9)}, {LL(0x64341200,0x1a098c5d),LL(0xa549cfc8,0xabb4ca0a),LL(0xdccd7aa4,0xe2b2179c),L_(0xf0addee3)}},
   {{LL(0xa357e9f5,0xec32e6a1),LL(0x4e4d505b,0x0d55532a),LL(0xd584c5e2,0x708100e5),L_(0xf05981fa)}, {LL(0x9188d927,0xff71c065),LL(0x30b8c50e,0x099da401),LL(0xa5cf657a,0x00705bdb),L_(0x003593a1)}},
   {{LL(0x634a3d0c,0x1e13fadd),LL(0xbfa0d04d,0xbc86e569),LL(0xcc81bd28,0x39bdb34f),L_(0x7ba2330c)}, {LL(0xa09b0aa7,0xb5b45e68),LL(0xf331bfd7,0xf478e7fb),LL(0xdd8ec1b5,0xb3f734e7),L_(0x9064cd97)}},
   {{LL(0x156f54eb,0x6a4afc5f),LL(0x4194a70e,0x4e2a403a),LL(0x523cff28,0x03d08e2c),L_(0xa27e7edc)}, {LL(0x994e73a6,0xfc7c7e86),LL(0x38873c54,0xa0320256),LL(0x805abf3c,0x414e3976),L_(0x93991ee1)}},
   {{LL(0xcf5e586e,0xa5d47503),LL(0xb3fe93fc,0xcd1bfca9),LL(0xb7d168b6,0x4a438e8c),L_(0x581ebd7a)}, {LL(0xbbc7084b,0x1e7c3d14),LL(0x7749a42d,0x66b2f40c),LL(0x86e109ed,0x853c9b29),L_(0x069c5f0b)}},
   {{LL(0x51236167,0x2fb43299),LL(0x2b3c8790,0x48986b12),LL(0xc7e0dd08,0x9d9741bf),L_(0x51f341ad)}, {LL(0x9e792a48,0x6eb8a76d),LL(0x1cd0ce38,0x20f5141d),LL(0x2698a77b,0x3099a97a),L_(0x6b8968f5)}},
   {{LL(0xdcec6ecf,0x5f3a4ac2),LL(0xf86e098a,0x8c8b8e0a),LL(0x3194f379,0xc98e48aa),L_(0x886d05b8)}, {LL(0xe568b10f,0xbd9be136),LL(0x488df6bf,0xa20174d8),LL(0x9833b00c,0xfb9f6810),L_(0x5fb8ce68)}},
   {{LL(0xb887fe68,0x0996c579),LL(0xe74e34d6,0xdd48ea52),LL(0xbc8f1cc6,0x49ddbd76),L_(0x3e421dda)}, {LL(0xd57b1ea4,0x2fcb399b),LL(0xf599d67b,0xde52325a),LL(0x1de7c3a8,0x60ffdfcb),L_(0x41018e1d)}},
   {{LL(0xda8c07fc,0x5803c900),LL(0x1834f6f5,0xab321f3b),LL(0x0269aa98,0x51b8942e),L_(0xbd9cdd7b)}, {LL(0xad131077,0xf11dba7e),LL(0x62d1e404,0xf313e57d),LL(0xa355486e,0x3e7a1a96),L_(0xa6d151c2)}},
   {{LL(0x7855f86c,0x5bfcbb65),LL(0xd3cf6d12,0x06e90560),LL(0x509c593c,0x36beb1c6),L_(0x50bb9be7)}, {LL(0x89a08a0b,0x461e9080),LL(0xf6ebcef8,0x278adf68),LL(0x53fb4ce1,0x1d3a4bb3),L_(0x84afbe92)}},
   {{LL(0xda5ddb0e,0x36690883),LL(0x66809ebd,0xa2cf5571),LL(0x6d7b75a0,0xb7561328),L_(0x7a0b342b)}, {LL(0xdbdf8401,0xc6b75bd0),LL(0x8b464a87,0xbcc0ebce),LL(0x145091bc,0x33cace99),L_(0x723fe6fa)}},
   {{LL(0xcfc2a3f9,0xfa386bf6),LL(0x5ea0874e,0x7749b51c),LL(0xa8387298,0xbc4e6a04),L_(0x02c73e59)}, {LL(0x42c5707c,0xa4b7c2f1),LL(0x2fdbbf21,0xea7c0fdd),LL(0xb933bbc7,0x874e47bf),L_(0x4c88b532)}},
   {{LL(0xc90a29b6,0xb79dad0a),LL(0x94b27e28,0x833287b1),LL(0x47dac83a,0xb3a5c35a),L_(0xace3bd6b)}, {LL(0x318b5daa,0x0d91b2e7),LL(0x00f91b6c,0x2490c7d0),LL(0xd6d7f96d,0x390f1d25),L_(0x92ea6b8b)}},
   {{LL(0x4f12bc87,0xa7f8fb95),LL(0xf0e6068b,0x6fd2151b),LL(0xed56b92b,0x64cd2ac4),L_(0xd3d78730)}, {LL(0x6abfc96c,0xf0d95041),LL(0x8db9e812,0xa3638d2c),LL(0x6447eba8,0xd45b6fa7),L_(0x3b769af7)}},
   {{LL(0x54f9cda5,0x26f167e9),LL(0x8aa2617b,0x869821a5),LL(0x52ead3d1,0x41988291),L_(0x5273a693)}, {LL(0x32358d32,0x18ca4d80),LL(0xb1bbd253,0xb8619557),LL(0x58878e33,0x64d8aee8),L_(0xe8908e43)}},
   {{LL(0x64ed9a67,0x5a517622),LL(0x6d83d019,0x742a30e3),LL(0x969711c0,0xa7fc9902),L_(0x225d81d7)}, {LL(0x8b9e5575,0x25981f3f),LL(0xe02b87e7,0x78b544b4),LL(0xaec410d7,0x35630262),L_(0xb1115eaa)}},
   {{LL(0xb95b1070,0x83f05d8d),LL(0x4dc83cf2,0x225fcf2b),LL(0x523db1cd,0x09d299d6),L_(0xe968d6f7)}, {LL(0xb132ecd5,0xa5406081),LL(0xc545ab6a,0x645c2ab4),LL(0xcf6e254a,0x592bcabb),L_(0x599530e8)}},
},
/* digit=22 base_pwr=2^154 */
{
   {{LL(0xda598881,0xa46f08d8),LL(0xb9fe1aa5,0x4cf987a8),LL(0x32c26e96,0x1b37edc0),L_(0xcc6e653e)}, {LL(0x8ea05891,0x37d78fa4),LL(0x44dcd6fa,0xb6453df3),LL(0x637ed305,0x72834a3e),L_(0x6cab1cc9)}},
   {{LL(0x1ff1de02,0x2efe30a1),LL(0xd9e804cc,0xfc2c0f38),LL(0xb9c440fa,0xb6ba8131),L_(0xd2f7d77a)}, {LL(0x60e1b4f3,0x88842b69),LL(0x96f7f20a,0x008db8f2),LL(0x9a618fff,0xf13ceac5),L_(0x6577f60c)}},
   {{LL(0xd0b89306,0xbd482556),LL(0x1653fafd,0xc3361427),LL(0x39c09ddd,0x10b62eeb),L_(0x79f6d014)}, {LL(0xd744c03c,0x78d81b00),LL(0xe8963c96,0xec51067e),LL(0x8c8f6a4c,0x63b70385),L_(0x551fa28d)}},
   {{LL(0xf611c717,0x433a33e9),LL(0x2417a7ce,0x5f8001f9),LL(0x1d9155b1,0x7a6ca298),L_(0xd78e16eb)}, {LL(0xc11dfcf9,0x3ab289c1),LL(0xee48adb0,0x14996507),LL(0xc998c5c9,0x8dd6fbf2),L_(0x959f66af)}},
   {{LL(0x82c3a705,0x23a0b239),LL(0x63c09a4f,0xa1f25d1e),LL(0x021b685f,0x7c69331c),L_(0xb1b0820e)}, {LL(0xf86bcb46,0x81992f9e),LL(0x370cbdbc,0x0d158241),LL(0xeecaf1b3,0xaeba071a),L_(0x5ebd44d9)}},
   {{LL(0x17d040bb,0x193df858),LL(0x7a008e70,0x858d93f7),LL(0x8d7ec281,0x81377001),L_(0x3062b2db)}, {LL(0xffd70fd5,0x11e91ce3),LL(0xdb62847d,0xf3cecc63),LL(0x9a492d59,0x42b1e033),L_(0x3c5e448f)}},
   {{LL(0x9609b50b,0x48abe62e),LL(0xf00a790d,0x5b827b1e),LL(0xe9c56ab8,0xcfc6643d),L_(0x7a101d9e)}, {LL(0x63e008bb,0xe901d5c6),LL(0xc8b5cd29,0x26d03ac2),LL(0x5a6c0d3c,0xf3eacfba),L_(0x437c7e62)}},
   {{LL(0x42de8327,0x7ef25e15),LL(0xc3c0b7b4,0xf84d2f14),LL(0x2d411980,0x9d372287),L_(0xb86b3e9a)}, {LL(0x7f225094,0x6682e29b),LL(0xc7ddb5da,0x7638e99e),LL(0x80ad8746,0xc5272ca9),L_(0x67e42112)}},
   {{LL(0xb19ef327,0x2db21213),LL(0x26d65fa2,0xa797a69c),LL(0x27bc92ce,0x780c1507),L_(0x62bfbfb4)}, {LL(0x02fe91fa,0x90f53f41),LL(0x306b2065,0xd9ad25ed),LL(0x70add874,0x97f0fe05),L_(0xef3f82de)}},
   {{LL(0xf3beb3c0,0xdb734eed),LL(0x49ac57da,0x4035d15e),LL(0x0575235f,0x12e6b4bb),L_(0x84ba90a7)}, {LL(0x2a740735,0x2e72cdc8),LL(0xde178a6e,0x32f40243),LL(0x2c8f9b53,0x04316bb2),L_(0x4d9021c4)}},
   {{LL(0xbcf71c73,0x4a88c71f),LL(0x7c1da7fb,0x646fc8a7),LL(0x7be262ad,0x26f4415e),L_(0x8d16df29)}, {LL(0xbe32e30e,0x6e047967),LL(0xf7050746,0x935f128c),LL(0x40048ee7,0x671c053f),L_(0x7e714450)}},
   {{LL(0xa89d04f2,0x126208fb),LL(0xc3f913fb,0xcea88f22),LL(0x81ebbd24,0xfbf9d09c),L_(0x60be0397)}, {LL(0x554a7a99,0xab78592d),LL(0xee7a8b54,0xbca7876a),LL(0x58636e68,0x5cb0cea7),L_(0x56d9eb18)}},
   {{LL(0xf22ebb87,0xe2adb3e2),LL(0xe08bf721,0x9944dfb5),LL(0xcf7b1348,0x88809e01),L_(0x9dd0883a)}, {LL(0x57335ddf,0xfb2d7aec),LL(0x9e8ac3f6,0x91eff023),LL(0x3aa9be67,0x87a97238),L_(0x56210cb7)}},
   {{LL(0x88f902c7,0x515e6712),LL(0x989ff215,0x47c3afa5),LL(0x8011a541,0x16b0e517),L_(0xc8b38bfd)}, {LL(0x4bc31c65,0x23c98753),LL(0xc13358b8,0x3b82c767),LL(0x5edc147b,0xae944131),L_(0xbf36526b)}},
   {{LL(0x8561f47c,0xbb74f4a2),LL(0xf57da9fb,0x4788db1a),LL(0xe6c34f9a,0xae99e920),L_(0x72b76a37)}, {LL(0xbdc2a795,0xd46a4b66),LL(0x377b9f4d,0x1b102f7a),LL(0x2dbc27cc,0x1c0d51f0),L_(0x63d0ccbb)}},
   {{LL(0xf20ce9cb,0x76b14e73),LL(0x5173ac48,0xe87b13ab),LL(0xac29f6c0,0xde2af466),L_(0xbf4405a4)}, {LL(0x5cd87093,0xe0a96171),LL(0xc72bdfba,0x6f175bde),LL(0x14f78dd7,0x2c991165),L_(0xf55ecef7)}},
   {{LL(0x94e2d286,0x6d393d5e),LL(0x05e626ee,0x250d4b04),LL(0x225ed4d7,0x53d41ea6),L_(0x5d5b37e5)}, {LL(0x6770d691,0x70c4e492),LL(0xf4527dfb,0xdeeb2466),LL(0x1522deea,0x333509e1),L_(0xad711cce)}},
   {{LL(0x5bd9660b,0x072fecfa),LL(0xba0c11d9,0xeac96039),LL(0xa6fa784d,0xa105395f),L_(0x3a914aba)}, {LL(0x7ecb3cf0,0x5b5ec37c),LL(0x6c8c0e8b,0x8123cd86),LL(0x79696f79,0x69daa4dd),L_(0xbf1748ba)}},
   {{LL(0xa78decff,0x16919c0f),LL(0x43014c2d,0xfbc474bc),LL(0xd14b6a12,0x6d705b75),L_(0x395d7526)}, {LL(0xa7378944,0x4c3aec7f),LL(0x3f4fd283,0x8737fb98),LL(0x4114b4f9,0x6a63c71b),L_(0xcf962aae)}},
   {{LL(0x99ce3cf2,0x085cede3),LL(0x39a2603a,0x87ec3837),LL(0xb203a75d,0xbdc1e809),L_(0xf3640f89)}, {LL(0xd93d4aae,0x8d9d5264),LL(0xd480817b,0xd455e58c),LL(0x8492b58b,0xe3368ba5),L_(0x24b8bfe6)}},
   {{LL(0x5ec2512d,0x64ef9737),LL(0x6907b7bc,0x78d38667),LL(0xf6e9c509,0x33b42737),L_(0x55ab9706)}, {LL(0xfaa748d1,0xf586dde6),LL(0x76b6d961,0x154becfd),LL(0xad1c2111,0x13031322),L_(0x1f36dd8c)}},
   {{LL(0xa1af32ba,0xe83a229c),LL(0xe6ae70d0,0x4ec7d7fe),LL(0x5b9ca0d3,0x06479e73),L_(0x17cdf808)}, {LL(0xad7c2138,0xd020f0ab),LL(0xca397995,0x85bbbbc2),LL(0x70369103,0x3d37e46a),L_(0x67d7bcef)}},
   {{LL(0xb1610c52,0x920986d3),LL(0x9e2b8580,0xbe1fdaa6),LL(0x1a1086a7,0xf038b9fd),L_(0x76630d46)}, {LL(0x62b5dc1c,0xf2780d63),LL(0x479ba46d,0x97c26f70),LL(0x41a1c690,0x953b6640),L_(0xeee5927b)}},
   {{LL(0x50cfff54,0x7a6dcf7f),LL(0x2905feea,0xd9932f1f),LL(0x5bba1fd8,0xd274195e),L_(0x596a73e6)}, {LL(0xd4297f7a,0xfa2e2812),LL(0x83723184,0xc5b16141),LL(0x21f09629,0x7ce090e3),L_(0xb6dc2849)}},
   {{LL(0x7d63db18,0xb254b1d4),LL(0xfee577f0,0xaf830093),LL(0x6accfc0b,0x775f693f),L_(0x483bf6a8)}, {LL(0x0e851bfe,0x74582d9f),LL(0x6731bfc7,0x7ffa6bf2),LL(0xf7d39569,0xa67faa6c),L_(0xc9418804)}},
   {{LL(0xf9316a6d,0xb5eb4f28),LL(0x7a12fe1e,0x8cfe6abf),LL(0x4a7fc70e,0xd820b6bc),L_(0x319719c3)}, {LL(0x54fb2143,0xbd26f428),LL(0x66648099,0x64ab8443),LL(0xb662d063,0x7fbe15d7),L_(0xf16d93b4)}},
   {{LL(0x0457afa4,0x951453e4),LL(0x8f99c3ae,0xfd74995c),LL(0x422aab2c,0xec8547fb),L_(0x73be771b)}, {LL(0xe916d5c5,0x49e82e71),LL(0x7b15c573,0x3a725205),LL(0xaa7e7ca5,0xfd867aa0),L_(0x58e19238)}},
   {{LL(0x31434cbf,0x31e49c5c),LL(0x2d0ee2e6,0xaf660f6c),LL(0x3aaf91fd,0x271cbc87),L_(0xcb833e79)}, {LL(0x520b6d31,0x36952e46),LL(0xf7a6bc06,0x6695996b),LL(0xda163497,0x1f6ccb97),L_(0xb5f271bf)}},
   {{LL(0xa5259a0d,0x606962f8),LL(0x82d18205,0xf314ac21),LL(0x4a97cf98,0xde06a403),L_(0x23969744)}, {LL(0x43ff18b9,0xba5e083d),LL(0x22229508,0x3789db89),LL(0x376f004b,0x1c53a79e),L_(0x8ecb5e4a)}},
   {{LL(0x3757ad7d,0xfde8f5af),LL(0xdaab7a45,0x313f26c0),LL(0x6cba2e40,0x0b7de7e1),L_(0x31db2861)}, {LL(0x90144bb6,0xf98e6b34),LL(0xa8d3e811,0x0ec080af),LL(0xc4961d3b,0x4bad9114),L_(0x10d64cdd)}},
   {{LL(0x1ff6dfd6,0xadd9fbd0),LL(0xca3624bb,0x4c069f73),LL(0x73a43aed,0x91301b03),L_(0xc382f632)}, {LL(0x9d2aab1b,0x83cfddff),LL(0xf9e8d9a9,0x66a4ad61),LL(0xb9fbd92f,0x25a8b014),L_(0x760229d4)}},
   {{LL(0xc29bf199,0x95ce63b8),LL(0x225d1531,0x1e469d80),LL(0xf1862d72,0x143d1930),L_(0x89faba0c)}, {LL(0xac073163,0x36978268),LL(0x2f3bfacc,0x077ad847),LL(0x8ecd6948,0x817ba1fc),L_(0xad14a0f1)}},
   {{LL(0x4b5edd16,0x9cf6404b),LL(0x0fcc7993,0xf9fdd94b),LL(0x94227ce7,0x36c49b8b),L_(0x1add1a97)}, {LL(0x06eb3245,0xddf10442),LL(0x478ddbe9,0xd0e75910),LL(0x598d307b,0xa7d77b58),L_(0x7c2d4285)}},
   {{LL(0x640b82b6,0x6ae40efb),LL(0x7296f54f,0xc265b407),LL(0xbb8c6473,0xa3248f4d),L_(0x096d9531)}, {LL(0xb52d1b77,0x72de2ce6),LL(0x71d14af5,0x2df2b5bc),LL(0x8bb3132f,0xd5a9918e),L_(0x1ee08068)}},
   {{LL(0x7efa31af,0x3a3e84c4),LL(0xdb4c764e,0x41d38660),LL(0x06967d91,0x9a2fe8e7),L_(0x3d0cd72c)}, {LL(0x81428c4a,0x663d93ab),LL(0xd9a81336,0x463057c2),LL(0xe092d510,0x7d2f5c90),L_(0x45eb443a)}},
   {{LL(0x482cdab4,0x29b44f02),LL(0x09859196,0x88f3dd7a),LL(0x274e7582,0x8a9337d3),L_(0xe9cba9ed)}, {LL(0x30787ea1,0xab1d6ae1),LL(0x1aa44fa6,0x72445ee4),LL(0x8201c08b,0x2f9c1703),L_(0xd128d342)}},
   {{LL(0xdf8f55c0,0x04dbb488),LL(0x4b130afb,0xa603695f),LL(0x010d7b25,0xf211bcec),L_(0xb7f03740)}, {LL(0xf354eee1,0x24a2aec6),LL(0xf116e286,0x473d0ffe),LL(0x4b6af9fa,0x974e5b38),L_(0x19cbf327)}},
   {{LL(0x3d701f1c,0x8816e065),LL(0x52f1347b,0xd04d2137),LL(0x733b5b5e,0xe33961b0),L_(0x8db340e4)}, {LL(0x02cb30d8,0x8fdad9c6),LL(0xb3cc8f09,0x63fea3f5),LL(0x36371a82,0x36432409),L_(0x9dd5d017)}},
   {{LL(0x3ac09ce0,0xc45e9f48),LL(0xce25491e,0x0b828bf6),LL(0xf8ea793c,0x27eb6c22),L_(0x6371b206)}, {LL(0xb1d7932a,0x80e32e78),LL(0x61eaf6df,0x8cf6f356),LL(0xdc4f44ca,0x387b7804),L_(0x4b85bf6e)}},
   {{LL(0x1b431c86,0x2c7dc474),LL(0x18a6a123,0x6c31f68d),LL(0x53665bca,0xaa43dbfb),L_(0x8d40fe26)}, {LL(0x7d646aab,0x6723ea8e),LL(0xbade3e08,0xcd9dea27),LL(0xc7adbb73,0x2f3a4dc9),L_(0xa74a2565)}},
   {{LL(0x7fa9020f,0x9ab672ef),LL(0x9cc7bfa2,0xa0040dfb),LL(0x2ac7eb07,0x368be66e),L_(0x5665ccf4)}, {LL(0xba527a40,0xb250a7ab),LL(0x465d7f98,0xa9c591cc),LL(0x8d2b97aa,0x7b979097),L_(0xa50d476d)}},
   {{LL(0x4b3aef47,0x8b68fb1a),LL(0x542fb7f1,0xadc18f71),LL(0xd96b7a34,0xb020cf53),L_(0xdb72531d)}, {LL(0xdf800c86,0xc73ecdcb),LL(0x1c4cd34a,0x47adcb2b),LL(0x039494e8,0xd23f79c7),L_(0xa21edcfd)}},
   {{LL(0x849075c6,0x17c4dc71),LL(0x5ce9dcb0,0xe5cdb7bf),LL(0x25936d77,0xebe08b04),L_(0x70fc7789)}, {LL(0x25213d26,0x7f3546f4),LL(0x7563a201,0xc56c68df),LL(0x10bfdb20,0xdb570af0),L_(0xeb5d3434)}},
   {{LL(0x106086c7,0xef345020),LL(0xd0e87d76,0x18e51b40),LL(0x222e0b10,0xe86e0a88),L_(0x6fa3b5c9)}, {LL(0x410e6c90,0x9da8796a),LL(0xabdf6b2a,0x4fe0bb39),LL(0x7ead1796,0xbbc76905),L_(0x8c71560c)}},
   {{LL(0x7da50c4d,0x890d8ce1),LL(0xb5cc05b2,0x2a181e5a),LL(0x7de1e29b,0x866cd575),L_(0xfc749a26)}, {LL(0x41ec00b6,0x7eaa0bb5),LL(0x886271b4,0x2328a91e),LL(0xed069b46,0x65bd26ca),L_(0x2f59240d)}},
   {{LL(0x4df1cae4,0xd954621f),LL(0xcbae178c,0xdf03015e),LL(0xe71c4857,0x70950b7d),L_(0xc8f7ab52)}, {LL(0xc0de4c81,0x52a1a22d),LL(0x24d86b6f,0x23da02f5),LL(0x70b94a09,0xd450bfb1),L_(0x0d983cdd)}},
   {{LL(0x3e6f9967,0x5619c34a),LL(0x77d8a74f,0x47b1c13c),LL(0x71f4dc89,0xfa0612da),L_(0xb127ccb2)}, {LL(0xcca647d0,0x879fb15f),LL(0x205b4589,0x5e51a2db),LL(0xe8b6c8f4,0xaad791a8),L_(0x1490d5ca)}},
   {{LL(0x1b020bba,0x32c5bb8f),LL(0x28810885,0xc2383a3f),LL(0x2ceffdff,0x14a9a0ad),L_(0x7970ea64)}, {LL(0xc1169c12,0x8708e80a),LL(0xbbf83ccb,0x5a067425),LL(0x5cc8baac,0x800c8eba),L_(0x98c8b359)}},
   {{LL(0x8f103cfd,0x22c84664),LL(0xdbd93971,0x993bf6c5),LL(0x697a6de7,0x6f50eaa6),L_(0x87cb9042)}, {LL(0x9314a935,0x5f70b39d),LL(0x2af4f24a,0xc319862f),LL(0xd8a16a0e,0x8f4571e4),L_(0xa7b9ed81)}},
   {{LL(0xf4ec13c6,0x6e1de980),LL(0x725de358,0x0fffcef7),LL(0x85652ca3,0x50b9f863),L_(0xd5f0033d)}, {LL(0x312ea9c7,0x19c015bb),LL(0x948391b7,0x92417df8),LL(0xaf6e5ed6,0x33c66538),L_(0x96bfa703)}},
   {{LL(0xab942a96,0x069d03c0),LL(0x47f60293,0xc848a1bd),LL(0xc32780f6,0x3bfeba8e),L_(0xc2e0eb4e)}, {LL(0x4521b895,0x29bd4112),LL(0xc233f03a,0xe259118f),LL(0x574da3b6,0xf4f6a7ac),L_(0xae0e0260)}},
   {{LL(0x16cf432a,0xbb8f2924),LL(0x5a087018,0x455c67ee),LL(0x8c5485dd,0x351ad0f1),L_(0xf7d7679e)}, {LL(0x318e4ede,0x2bdc4896),LL(0x277bcf81,0x8b26b0fb),LL(0xf147c5b4,0x87185cc7),L_(0xdf48c9e6)}},
   {{LL(0x196afebb,0xdb75cc0f),LL(0x17222dc7,0x225c2aa1),LL(0x3dd256a1,0x3a012a3b),L_(0x7ec5696a)}, {LL(0x25bf2c5f,0x9381bed5),LL(0x4f4cdfd3,0x0f7e0324),LL(0x7b5b13a4,0xbf5f2e84),L_(0x2639cce3)}},
   {{LL(0xf2e480ae,0xdb334a58),LL(0x6da64943,0xe10cd107),LL(0xfce96a35,0x6ad5dfad),L_(0xa599a89e)}, {LL(0x26a35a0e,0xeddc6625),LL(0x2ef746d1,0x5f6fe79f),LL(0xa912c3f0,0xd6859ab6),L_(0x1ac3aba5)}},
   {{LL(0x705906c9,0x0e5c7c8c),LL(0xb050ceeb,0x5f2aa907),LL(0x53e4bd2f,0x710688b3),L_(0x19a215e7)}, {LL(0xb2485b2a,0x9a042c99),LL(0xd22cc140,0xf04c2421),LL(0xb5f7e2eb,0x7f1f59f8),L_(0x154453dd)}},
   {{LL(0xc0354be0,0x9473341b),LL(0x9f33393e,0xb5b4c257),LL(0xb3dc51a3,0x09fb2116),L_(0x0ee5b84f)}, {LL(0x3e9814a0,0xf91ac675),LL(0x816d6ca6,0x506f79fd),LL(0xc0c5edef,0x0cfcd5ed),L_(0xb71c05e5)}},
   {{LL(0x400799f5,0x17c9bd2a),LL(0x4c0897b1,0xd4538ae6),LL(0x7a16a169,0x0f41e0dd),L_(0xa88a35f1)}, {LL(0x4037d80f,0x5e5fc718),LL(0x85b8baf5,0xee15dcea),LL(0xd33b70cd,0x5dd4f3b8),L_(0x5ddf49c1)}},
   {{LL(0x0095bcdb,0x795a3892),LL(0xedb3e7f2,0x6c1761f4),LL(0x33c80575,0x26818153),L_(0xf7414144)}, {LL(0x943d7d31,0x6f311ca0),LL(0x24d6e452,0xe928eca1),LL(0xe7193165,0x66d1a963),L_(0x89fa62a1)}},
   {{LL(0x8faef020,0x375aa47c),LL(0x70c638b9,0xc033edb6),LL(0xf167acbd,0xddf9d01a),L_(0x364bf41a)}, {LL(0xf8bb63ab,0x7d82f66c),LL(0x8862080b,0xedb30cf7),LL(0x0448ac9c,0x4dbdd638),L_(0x971eaebd)}},
   {{LL(0xad55ab2a,0x30f6e053),LL(0x4b4739f2,0xa878cbee),LL(0xd62b9aef,0x3bbafe26),L_(0xb84b80ac)}, {LL(0xc8ef692b,0xd03fc3fd),LL(0xb6803cd9,0xe63bac37),LL(0x81afcd95,0x205a14b1),L_(0x7a0ade0e)}},
   {{LL(0xff50050c,0x2d735063),LL(0x4b1c54dd,0xab849fa5),LL(0xda4629d6,0xe95c14ae),L_(0x6f216326)}, {LL(0x74ba5b90,0x2fae20c6),LL(0x3112101e,0x5e3d1209),LL(0xbce52748,0xfcd465ea),L_(0x53fd0b9b)}},
   {{LL(0x2d491553,0x068c0d92),LL(0x1f9856a8,0x37cee3f9),LL(0x3408da24,0xc9b9b412),L_(0xc63b4cb7)}, {LL(0x74b30760,0xb7836b39),LL(0x200fd925,0x339dec72),LL(0x55a6ddf0,0x2b7f407e),L_(0x0fb2a9d1)}},
   {{LL(0xa824468a,0xe36962f5),LL(0xfae22eee,0x0a42cdab),LL(0xfe866c3e,0x8acadf81),L_(0xa3f599f7)}, {LL(0xcdd076d7,0xe8b6b61e),LL(0xb4b13551,0x46d40c51),LL(0xcdd8f22e,0xb9099888),L_(0xe1b8ed2e)}},
   {{LL(0x60051d31,0x7e8f146e),LL(0x3818b39f,0x4cd2aa83),LL(0x16886905,0xf677bde6),L_(0x022ae138)}, {LL(0x98b45d19,0x5746a458),LL(0xacf62d52,0x70f9fca7),LL(0x63279bc6,0x4ee46705),L_(0xfd285b9c)}},
},
/* digit=23 base_pwr=2^161 */
{
   {{LL(0xf0d700ae,0xbad78996),LL(0xc931bee3,0x1b3ef234),LL(0xe3c1a2d8,0x7cd507de),L_(0x58fc32e1)}, {LL(0x83e229c6,0x01135f5f),LL(0xfb8b7b35,0x8d5e43a4),LL(0x413750b4,0x2dffb3fe),L_(0x6a265953)}},
   {{LL(0x8c78030e,0x796bb2af),LL(0x56c1932c,0x7d56b42a),LL(0xf443c29c,0x6ff56bc3),L_(0xbdd22bb5)}, {LL(0x95857542,0x21178547),LL(0x22575d25,0xbdc732f6),LL(0x7ffb29d7,0x4ffc71d3),L_(0xd6ef14c3)}},
   {{LL(0x3a92228c,0xbc9613ab),LL(0x5920119c,0x304f1f53),LL(0xc3d2b2e2,0x9633ba13),L_(0xe7d31665)}, {LL(0xc95ebff4,0xe622933c),LL(0xec5df7bb,0x9abfde78),LL(0x50f7f156,0x7feb42b1),L_(0xeb17ef8e)}},
   {{LL(0x25b9b992,0x1a08721f),LL(0xeeddf997,0xbf152cb2),LL(0xb6ad560a,0x3c761d64),L_(0x52dd9f57)}, {LL(0xc0d79f78,0x39b5966e),LL(0xca15f167,0x562740a0),LL(0xea78b847,0xd9642afb),L_(0x37eea476)}},
   {{LL(0xd57560f1,0xda4b72e5),LL(0xa3072640,0x68c344ac),LL(0x36a057e0,0x4754e3c5),L_(0x4faa9820)}, {LL(0xe786962f,0xf20160bd),LL(0xa80ee38d,0x77ea80d1),LL(0xc8e173dc,0x665dbf2e),L_(0x2aba1be0)}},
   {{LL(0xb6e8e52b,0x228ed7a9),LL(0x2d501abf,0x5d4bca44),LL(0xe4f5d2f6,0x3dad807b),L_(0x0b4aa28f)}, {LL(0x006c4c0f,0xe672b21f),LL(0x0888b653,0x614c5f64),LL(0xa3e477ae,0x226f83cd),L_(0x0afdcedc)}},
   {{LL(0xd5a1f5d3,0xe7476654),LL(0x562eb536,0x7a0ad378),LL(0x7f5bedc0,0x2887f756),L_(0x8581b7e9)}, {LL(0x9986fd24,0x160c3504),LL(0xdde60b62,0x06b5ca84),LL(0x8a8ca29b,0xb94770da),L_(0x3ecfeca1)}},
   {{LL(0x28c355ca,0xb615d5a7),LL(0x8ea9ecfb,0x2c8d2d4f),LL(0x6be624fe,0x7f81ee5d),L_(0x45035a93)}, {LL(0x97611fa5,0xc3d23100),LL(0xe9c562fc,0x3ccce59a),LL(0xf451c22a,0x6ee17a32),L_(0x932ddcfa)}},
   {{LL(0xfa127ca0,0x6337e908),LL(0xe970ce2d,0x839c1246),LL(0x58d5ae88,0x253afb1a),L_(0x88db9fe4)}, {LL(0x4550894d,0x3d74de43),LL(0xa90b033e,0x46e1ab37),LL(0x132dd86c,0xed163aa4),L_(0xc88fa0d9)}},
   {{LL(0xf244b8ec,0xa0e34d4b),LL(0xa5b69894,0x9ac53a57),LL(0x495e2cc1,0x987653cd),L_(0x5e895b41)}, {LL(0xf2fd1824,0xf289cc20),LL(0xc41882ef,0x6ce96956),LL(0x3329b3be,0x4acf73b8),L_(0x9efd2db0)}},
   {{LL(0x3bb5fbc8,0xdd8e91ba),LL(0xf8f6f870,0x8acac0fe),LL(0x728bf4b5,0x93d4067d),L_(0xb9c4891b)}, {LL(0xaef29279,0xf8bf9b95),LL(0x1e5e8506,0xdf786531),LL(0x11821ebf,0x99083ad2),L_(0x7e9d0c84)}},
   {{LL(0x83dcfec2,0x85547390),LL(0x48ab6926,0xc7ed1aa0),LL(0x1badb306,0xdd5f7613),L_(0x4641e82a)}, {LL(0xf53ff561,0x3d43a296),LL(0x58c32dfb,0x7b2e7561),LL(0x3ffcd8f7,0xc250c17a),L_(0xfacc3394)}},
   {{LL(0xda8507d4,0x99456e76),LL(0xe9a40562,0x9ba62d11),LL(0xa165351f,0xd0dd1798),L_(0xbabb6293)}, {LL(0x853dcb7c,0xb83a079a),LL(0xa5cd33ea,0x26c9aa1e),LL(0xf120f051,0x6109e625),L_(0x774354d8)}},
   {{LL(0x14c27c7a,0x8d82b0bc),LL(0xbafdf31f,0xe88bea46),LL(0x7639e998,0xd2a8308f),L_(0x3be638b1)}, {LL(0xc5e7e09d,0x54bc114e),LL(0xeb650088,0xd1baff80),LL(0xebdef489,0x902bb33d),L_(0xc3e46171)}},
   {{LL(0x45911106,0xc86362a5),LL(0x56cc1bef,0xce560fcf),LL(0x3b4b4fc8,0xf9436f6d),L_(0x370a01fd)}, {LL(0xa8fad83b,0x632c516d),LL(0x916aa8c7,0xae217497),LL(0xe3ad18e9,0xe18b971a),L_(0x72c7e5a4)}},
   {{LL(0x9b05d089,0xb483969f),LL(0xf8cf3d39,0xcfd11869),LL(0x1d89563e,0xa602a006),L_(0x91cd1395)}, {LL(0x71d49ace,0xdbdaa432),LL(0x0966d0a9,0x1c3623ed),LL(0x166bebe1,0xf1b4da0f),L_(0x433f9113)}},
   {{LL(0x86d4180d,0x6fb1d5d9),LL(0xd89da757,0x031c30db),LL(0x2939bdad,0xa42134e8),L_(0x6b13fefc)}, {LL(0x1baf8cf7,0xb37f662e),LL(0x413f92df,0xed2b1683),LL(0x985c742a,0x1de586e7),L_(0x5f4904de)}},
   {{LL(0x01f2ea9b,0x03906d91),LL(0xb13c4090,0x9d1c566b),LL(0xf3affee6,0x6830a295),L_(0xb7fc1017)}, {LL(0xa0a0056c,0x45d24eb6),LL(0xcb2d95c2,0x622219f3),LL(0x8c20bcd3,0x3379e3da),L_(0x6dc4634a)}},
   {{LL(0x3e8ae0a0,0xaefdff77),LL(0x97e82574,0xe4ee96dc),LL(0x112d1838,0x20061ca4),L_(0x0060cd67)}, {LL(0x20825ad2,0x11499a1e),LL(0x8b6913cb,0x1c239b7a),LL(0x67ff5e87,0x1680aa3a),L_(0xe823c3ec)}},
   {{LL(0xd6ad0217,0xd3ac8a55),LL(0xd2b4232a,0xf6ddec57),LL(0x1b8abf52,0x2fcb86b4),L_(0x19bc7365)}, {LL(0x1f2c9fd1,0xa051bf99),LL(0x1aec4d94,0x5a88054d),LL(0xfbc4f987,0xa954ceb9),L_(0x05b104cc)}},
   {{LL(0x4ccd6345,0x40690896),LL(0x4799ff54,0x3b0d618b),LL(0xbc95c20a,0xc0b5b3de),L_(0x290686c2)}, {LL(0x488b692d,0xca443d1d),LL(0x6b65747b,0x823770ad),LL(0xf2987b72,0xf039026d),L_(0xb9b37c0a)}},
   {{LL(0x8dd8e6e6,0xd360d772),LL(0xd81505e8,0x994209e4),LL(0x40816745,0xc9836e31),L_(0x9dca53d3)}, {LL(0x07ea2add,0xd820c03f),LL(0xb45d3f5d,0x33d13943),LL(0xedf3a3c9,0xe0ce5e44),L_(0x459a2236)}},
   {{LL(0x3f40e1dd,0x35ffc3c1),LL(0x273c8e42,0x9e607b7b),LL(0xfecec30d,0xda6a8389),L_(0x8e8dc1c3)}, {LL(0xe7f0596c,0x6fe132fd),LL(0xc3565543,0x8889d9c6),LL(0x951334fa,0x770ed88f),L_(0x4d1d476c)}},
   {{LL(0x542a815c,0x20e67927),LL(0x488e81fb,0x7f7fe269),LL(0xd968afb4,0xb0057d32),L_(0xb8d04c1b)}, {LL(0x85a86d34,0xee01ab82),LL(0xef38d717,0x5e5709d8),LL(0x830deaea,0xfba87128),L_(0xbf20a3cb)}},
   {{LL(0x9efe5459,0x15750d83),LL(0x8d8facef,0x4d132c09),LL(0xef0f3af1,0xd1a4206b),L_(0x64ebf883)}, {LL(0x5e56d99d,0x5b6d2c3b),LL(0x3c48ef20,0xce5ccc62),LL(0x153580c5,0x534ba912),L_(0xa02f889c)}},
   {{LL(0x349d9804,0x1f2a638d),LL(0xcb233fe2,0x9d44fa20),LL(0x792d0e00,0x62d03b08),L_(0xd6646af8)}, {LL(0x180ef2f5,0x4a8172cc),LL(0x82199f97,0x401272d5),LL(0xf879213c,0x8048de29),L_(0x51bdf94f)}},
   {{LL(0x9c9a3b84,0x75f12d75),LL(0x743c75d7,0xaadd691d),LL(0xc13528c2,0x0f8f0fc7),L_(0x969b010e)}, {LL(0x9b8f4442,0x2acbaab0),LL(0xdfc6ee9c,0x4678a76c),LL(0x23ac3530,0xc00ef611),L_(0xdfc77ba6)}},
   {{LL(0x3fe1d500,0xd561d679),LL(0xac256d68,0x7e3b2fc9),LL(0xb7f44d84,0x1d8cc6f2),L_(0xddd4acc6)}, {LL(0xccec3e4c,0x83d18ef0),LL(0x6b5fbe51,0xf9877554),LL(0xe616da91,0x04e7aef1),L_(0x7079d8b4)}},
   {{LL(0xacf8b8b3,0xc0f7d69f),LL(0x2242759d,0x0a034401),LL(0x9949d06b,0xde898701),L_(0x8b24618f)}, {LL(0xa9a593b3,0x3976254d),LL(0xb0bd22ab,0x776af23a),LL(0x2125ed2b,0xa5f28d83),L_(0xa1c543ea)}},
   {{LL(0x4bd74e55,0xe7ee1b28),LL(0xf2a06317,0xec3c82fa),LL(0x784961c1,0xf759d96d),L_(0x5670b8e7)}, {LL(0x2db86f19,0xafedb099),LL(0x8fbfe519,0x19d6aa87),LL(0x7246a16a,0x7503e23c),L_(0x23ac9af3)}},
   {{LL(0x952bc75c,0x87836e61),LL(0xc31169c7,0x278dc1c5),LL(0x111050a4,0xa7810165),L_(0x446bda32)}, {LL(0x490d4091,0xa876bcf8),LL(0x32692fd4,0x7e43410e),LL(0xd11e3c6f,0xe2aece33),L_(0xd325fefc)}},
   {{LL(0x209ded15,0x8c31bbf9),LL(0x0e3623bf,0x818a6e0b),LL(0xbb5c266d,0xc5f4aaef),L_(0xa7179b54)}, {LL(0x15383b16,0xed087186),LL(0x375cb291,0xbbe070e0),LL(0x047af636,0x73198d27),L_(0x8f1c0b06)}},
   {{LL(0x4cc95bc4,0x77e38347),LL(0xc22c3ac3,0x83e4ab76),LL(0xea1c777a,0xab8de890),L_(0x8b9d1132)}, {LL(0x959ae60a,0x2a2fde96),LL(0x78f20c5e,0x6184d8f4),LL(0xa11f0041,0x302a6212),L_(0xba7689dc)}},
   {{LL(0x376e162d,0x8c2d9e1f),LL(0x50cda9df,0x7ba9efe3),LL(0xa6158df9,0xadbbeaa9),L_(0x4778c93d)}, {LL(0x4165d3a8,0xd1a49e09),LL(0x6eb6664c,0x276faed0),LL(0xb72ac28c,0xb8fb2671),L_(0xfc493918)}},
   {{LL(0x3f2f65a3,0xe846a0c3),LL(0xc3ffb873,0xb4100b80),LL(0xd98b14c7,0xcd3fcccf),L_(0xec8c8522)}, {LL(0xde06d3b7,0x27093099),LL(0xc435552e,0x5b30c12f),LL(0x731b5ca4,0x6d3545bb),L_(0xcf95c612)}},
   {{LL(0x06a11b1f,0x67e83944),LL(0x77e47a85,0x840987d9),LL(0x41b4a73c,0xf14ab27f),L_(0x45673153)}, {LL(0x5e20b8e1,0x29737735),LL(0x3a31ee99,0xdcbc9d02),LL(0x89cb9a71,0x3c610ccb),L_(0x92d6ab9e)}},
   {{LL(0x7dc21a89,0x24dd0312),LL(0xfe3af3cf,0xbd1479bd),LL(0x1fa695ef,0x49e9da9d),L_(0x35d8485e)}, {LL(0x4caa0416,0x1be78d4d),LL(0x9a8e5ede,0x06a82fad),LL(0xdb4e20a3,0xb9902b0e),L_(0x762073cc)}},
   {{LL(0x26109f70,0xba8c73ab),LL(0xfd983b8b,0xe8f78dd1),LL(0x3921c9f7,0x9c6ee5d8),L_(0x7f5a5177)}, {LL(0x14bbc8be,0xab53e7b2),LL(0x7db07019,0xfff92c17),LL(0x01f9a8fb,0x2588492a),L_(0x9ea1e98d)}},
   {{LL(0x924eb396,0x31997788),LL(0xa9a0407f,0xbc33538a),LL(0x26fdd725,0xa3068d9c),L_(0x2aba0c14)}, {LL(0xfb38ce06,0x6d07282f),LL(0x2d5ed5c3,0x0f9417d3),LL(0x52b3241d,0xdbd63903),L_(0x11f4b8bc)}},
   {{LL(0x88a000ac,0x35994c37),LL(0xccc2b918,0x337027af),LL(0xf27570cb,0x870c688b),L_(0x8c91a8af)}, {LL(0x458bc212,0x5459f2f5),LL(0x9157cd7c,0x7d5dcbea),LL(0x021d9845,0xdbbbf2c4),L_(0x3319b3fc)}},
   {{LL(0x4a370724,0xaeba325e),LL(0xaecdc264,0xaa40e67e),LL(0x9b24587f,0x859c1b6a),L_(0xf7648c83)}, {LL(0x9da1346e,0x7005a3f3),LL(0xdf465967,0x7311149a),LL(0x2411ca0d,0xd689abfe),L_(0x1452c503)}},
   {{LL(0x2a06a445,0xcc290bf3),LL(0xf34968d7,0x65fd04ce),LL(0x7228d7c1,0x48e02e21),L_(0x17e25b2b)}, {LL(0x20f72ec8,0xa53a154b),LL(0x1035d2b6,0x8b5f77f7),LL(0xe8ab36c6,0xbcf0eb3e),L_(0x63fa883e)}},
   {{LL(0xa46b410c,0x0b53b1cd),LL(0x2088982f,0x44825888),LL(0x21364c98,0x45bf87a7),L_(0xeeb978c5)}, {LL(0x77561ed1,0x4e53377b),LL(0x093e9c15,0xe15699a0),LL(0xd8249f4f,0x02b46436),L_(0x64f236ee)}},
   {{LL(0x5bf80f33,0x3940ee66),LL(0x6164d05e,0x98afb7d2),LL(0xb0056da3,0x872642d0),L_(0xa7bd3d66)}, {LL(0xc227063a,0xce37f406),LL(0xd8371363,0x97040937),LL(0x90362154,0x009ca301),L_(0xe59d0b2d)}},
   {{LL(0x812ca945,0xcf247bdf),LL(0x73b26b51,0xb0c27a65),LL(0xdf33ad4d,0x1f6d55bd),L_(0x4ed64955)}, {LL(0xc4a11d3d,0xa18aee15),LL(0x8ca23082,0xbd802f68),LL(0x20da76f8,0x2453f147),L_(0x4606958c)}},
   {{LL(0xfa30299c,0x99950738),LL(0x06b63065,0xbe157ae3),LL(0x9853453b,0x88e04e19),L_(0x456e72cc)}, {LL(0xfcb616c9,0xb168037b),LL(0xd3c22f8d,0xce3f0f0a),LL(0xd989f0f6,0x9699fc87),L_(0x273ac5b1)}},
   {{LL(0x85664bdc,0x273a933d),LL(0x24f52dc3,0xfcf79a03),LL(0x36bc3c34,0x3e91b0cc),L_(0x61de4790)}, {LL(0x58e3b120,0x8f4b6789),LL(0x44bdcadc,0xb12252e0),LL(0x830bedee,0xbb9ac7e8),L_(0xbda5fd11)}},
   {{LL(0x44861e3d,0x37a97134),LL(0x079c1b7a,0x6798891d),LL(0xeaf11509,0x3012e200),L_(0x61b63246)}, {LL(0xa87c42aa,0xc0246bd8),LL(0xb92f89e2,0xed286525),LL(0x3fff4a96,0x44cf1045),L_(0x6e70ae3f)}},
   {{LL(0x5068198a,0x6615d342),LL(0xb3a83dd9,0x0e23ae21),LL(0x0c4c00f7,0x6cbcc29c),L_(0x84d9fe95)}, {LL(0x5e865fd0,0x30e1885b),LL(0x82a4da98,0x5fcfdb20),LL(0xbcc23ab7,0x259db6ce),L_(0x306cf56e)}},
   {{LL(0x3622f7f7,0x08624c7b),LL(0x32c9c5fc,0x01b193ac),LL(0x38295bfb,0xfb49c594),L_(0xdcfe3634)}, {LL(0x103965ff,0x832aec92),LL(0xe5cda111,0xdb928d8e),LL(0x98fc1728,0xac496bac),L_(0x3f057842)}},
   {{LL(0x69b7107b,0x7bdd13e7),LL(0x189dee30,0x5aa65303),LL(0x01cf6016,0x02a07254),L_(0x7f6053e7)}, {LL(0x409cab89,0xc9d79b48),LL(0xd467929d,0xe17deac4),LL(0x2a4257b0,0x3ed68c3c),L_(0x0209a951)}},
   {{LL(0xde7ecacc,0xb50fa3a8),LL(0x568377b7,0x7926440e),LL(0x992fa9d6,0xf3995e07),L_(0xf09351e3)}, {LL(0xaf199c40,0xde00386c),LL(0x7c21c039,0xacbcdf7b),LL(0x983e3f5f,0x970f1581),L_(0xfde7bb8d)}},
   {{LL(0x31d0cb64,0xa5991d35),LL(0x727713ca,0x20cf37ff),LL(0x81781a16,0x946557d9),L_(0x8b2de3fc)}, {LL(0xd1639576,0xb61c0935),LL(0xefe6ec67,0xcaf4adb1),LL(0x0483381f,0xbfd4710c),L_(0x3cb8617a)}},
   {{LL(0x19bcb25a,0xfb2c55b6),LL(0x83398c47,0xe6a45901),LL(0xfc55bdd9,0xd6aad463),L_(0x733b8806)}, {LL(0x4da45d48,0x600c9640),LL(0x72a69c3e,0x7bb5666e),LL(0x25c8da01,0xa863f55c),L_(0x6b73f445)}},
   {{LL(0x8ff6b908,0xb5811bfb),LL(0xdea6b1b8,0x3864ff6e),LL(0x6f29d6ab,0x244c832f),L_(0xb885707b)}, {LL(0x6b307d64,0x19847d9d),LL(0xbb7684f6,0x85f7c9ef),LL(0xc0f2d53c,0x20ef5736),L_(0x9e06880d)}},
   {{LL(0x8c5dd4e2,0x9a3cdc81),LL(0x6fd2d1fe,0x6a09971a),LL(0xfe48b29f,0xa14e785b),L_(0x7afeb3e3)}, {LL(0xb175e3fc,0x15c1eebf),LL(0xdc5c271b,0xc21625b9),LL(0xd51bbdf3,0x4095982d),L_(0xea6a656f)}},
   {{LL(0x3418b0dc,0x8df3827e),LL(0xad96082c,0xae05d810),LL(0x0c7308d4,0x747332eb),L_(0x3e790568)}, {LL(0xf43bfba5,0xb410e2a5),LL(0xa45bcc07,0x6ea69371),LL(0x34e11697,0x68990fab),L_(0xd0d95f71)}},
   {{LL(0x46088da2,0x2b70f4c2),LL(0x95663747,0x2b81bbe7),LL(0x18e76686,0x0510d4ae),L_(0xa806fc64)}, {LL(0x718b9269,0x74a39c45),LL(0x11681c31,0x39edf452),LL(0x7f1ee73e,0xd70c5b25),L_(0x145296c5)}},
   {{LL(0x38adcbf7,0xb90e977c),LL(0xa73e17c7,0x589db798),LL(0x3b0459aa,0xd910c80d),L_(0x4a996387)}, {LL(0xa2014673,0x09cc1462),LL(0x4ff007ad,0x9deb8885),LL(0x6f3bf601,0x9137e94d),L_(0x3d0461a8)}},
   {{LL(0x49eeb574,0xdf5e90f6),LL(0x03f3fdc3,0xf38e094e),LL(0x2f05f4ba,0xefe1e30f),L_(0x5dab20a6)}, {LL(0x25f380be,0xfbe83a6a),LL(0x815eac18,0x96d53b04),LL(0x2c89a65a,0x13f75b6a),L_(0x326a87f4)}},
   {{LL(0xb57a062a,0x99fa6314),LL(0x29c19e9b,0x29ce4a74),LL(0x5d5a6ffb,0x84d62c60),L_(0x4e8640e3)}, {LL(0x63890d47,0x01830a3e),LL(0xa50855ff,0x86e7548d),LL(0xb9b3fb5b,0xbf0727b4),L_(0xa1cac932)}},
   {{LL(0x93078c89,0x08fe3e54),LL(0x7f8ba9f8,0xa70bd6a5),LL(0xfff3b260,0xa621223a),L_(0xda5b8cc0)}, {LL(0x5c0ab9e1,0xb352caaf),LL(0x52e875c5,0x57c08401),LL(0x888687cb,0xd159fefe),L_(0x788f0705)}},
   {{LL(0x50a69ca4,0xe7ee73d3),LL(0x71f9e278,0x50682750),LL(0x0511bea3,0x147b2ef8),L_(0x908389b2)}, {LL(0x0d09e4fb,0xb9505161),LL(0x7ab5b1bb,0xf8cfd83f),LL(0xa16a5843,0xeb15fb80),L_(0x8840233d)}},
   {{LL(0xedb21e1a,0xfbf89123),LL(0xb350235f,0xba1f0507),LL(0x601ae7a3,0x487a5b44),L_(0x168cfefc)}, {LL(0x1632dc61,0x27f10999),LL(0x3f7f4917,0x224754d4),LL(0x576cbe66,0x93695572),L_(0x1697b707)}},
},
/* digit=24 base_pwr=2^168 */
{
   {{LL(0x897a6b26,0x1aa93494),LL(0x07822bd8,0x0eecf33d),LL(0x690a0450,0x411c39e9),L_(0xddfd5a23)}, {LL(0x752f4021,0x5d8da0af),LL(0x74dcec38,0xc6e5a1f0),LL(0xa51c46ee,0x6b77f2b0),L_(0x1367898a)}},
   {{LL(0x10f0de19,0xd8b1efba),LL(0x5d75e9ca,0x831a06cb),LL(0xdea9f60f,0x7af3d6fd),L_(0x95065ae4)}, {LL(0x379b23db,0xe0c7e800),LL(0x22ced186,0x5d673e52),LL(0xc6698d7f,0x95367fa5),L_(0x9310a67e)}},
   {{LL(0x288d3aae,0x7322a3b2),LL(0x1b5b3f21,0xbe251fd5),LL(0x1fe6e87b,0xcad2ee21),L_(0xb5fd4e5b)}, {LL(0x91f82182,0xe5ae89d4),LL(0xc2853eeb,0x3262d99e),LL(0x9036d2d0,0xcb159584),L_(0x45627827)}},
   {{LL(0xf3073506,0xf3411909),LL(0x2470f5ee,0x94dcf7c1),LL(0xf0be84fe,0xade34d2c),L_(0x90636c20)}, {LL(0xb2827737,0x180fa04c),LL(0x2db09b0f,0x26143ea2),LL(0x0a0909aa,0x83bcc559),L_(0x281cf9aa)}},
   {{LL(0xb3d17628,0x38d6fee0),LL(0x8f859667,0x5ac1dd54),LL(0x86559609,0xca6e59bf),L_(0x6854b4e8)}, {LL(0x61f3a762,0x23a24b87),LL(0x5840ce8a,0x0173425e),LL(0x946b6bbd,0x00fd3d85),L_(0x8a095b45)}},
   {{LL(0x156ebcbc,0x74d544f1),LL(0x4c355669,0x3127ea7e),LL(0x1ea60e82,0x099e116c),L_(0xd960a70d)}, {LL(0xd32fd59f,0xd60a4a59),LL(0x305faf07,0x3cf7063d),LL(0x61b80d24,0xe6b8e5c2),L_(0xd428e0cc)}},
   {{LL(0x22dd5fee,0x26d21295),LL(0x5d2ed352,0xd4c31ad3),LL(0x3daf91ee,0x44afa3c4),L_(0xd529e7b2)}, {LL(0x17333fb8,0xd7ed0b47),LL(0xcc9faccf,0xe9201771),LL(0xc98aa7de,0xc9757998),L_(0xce2c4e14)}},
   {{LL(0x1ae8d0a4,0x098fc503),LL(0x3d3f8532,0x41d10295),LL(0x50253fa1,0x23d266c8),L_(0xb38286d6)}, {LL(0xfb63eee1,0x99903092),LL(0x48a64b98,0x9100a99c),LL(0xbc43643a,0xfe159bd8),L_(0x2f08eae4)}},
   {{LL(0xb10e23d2,0x5b31f9b6),LL(0xc558d554,0x5c151d19),LL(0xe3c46403,0x19a6f592),L_(0x2211131c)}, {LL(0xbb6a51b6,0x4d8f19e7),LL(0x8c4d4f34,0x551259ee),LL(0x347e8cae,0xad36aa89),L_(0x5a959e3c)}},
   {{LL(0x9f2c968b,0x178db2dd),LL(0x8674b5e1,0xc9ba55bc),LL(0xce21dcf1,0xd3884f45),L_(0x7664513a)}, {LL(0xbb703b97,0x7a1b3e56),LL(0xc9e299e2,0xc4a588c4),LL(0xe96de0f0,0x8d3f0ffc),L_(0x5aba434e)}},
   {{LL(0x8685946b,0x3206b59f),LL(0xbf15b53c,0x7e914149),LL(0x2fff52c6,0xfec55940),L_(0xc3791f65)}, {LL(0x78dad0a9,0x394de0c1),LL(0x5c5fed8c,0x72c6919f),LL(0x674b4d59,0x0e57743b),L_(0xd7315975)}},
   {{LL(0x3dad8066,0x7fa4f80a),LL(0xc574aafe,0x60f2414d),LL(0x25994c45,0x15293405),L_(0xa00f9231)}, {LL(0x0f201926,0xf98edea5),LL(0xd0bcf974,0xaf5b2ad4),LL(0xa3c74ad6,0x448873ac),L_(0x8ff67b2d)}},
   {{LL(0x75e7728e,0x510cb9a4),LL(0xbaf15c58,0x54d76825),LL(0x234a8fab,0x0023d4d4),L_(0x9d6e89b9)}, {LL(0x2cce0b43,0x374850ff),LL(0x6bf4faef,0x32a29d1b),LL(0xdd164002,0x799cb3ee),L_(0xfc7797a7)}},
   {{LL(0x26f100f7,0x34261e54),LL(0xcf17aa4d,0x589b344b),LL(0x801ccc35,0x0c7a04fa),L_(0x762b3408)}, {LL(0x9578af64,0x002ca131),LL(0x0870f41d,0xe0fba28a),LL(0x70686201,0x276eaa26),L_(0xf592842e)}},
   {{LL(0xcf7ff5db,0xb78b71bc),LL(0xa79ca9e4,0x625166ca),LL(0xe7aab4c3,0x095aa863),L_(0x980df4d9)}, {LL(0xfc336519,0x75ac5ed8),LL(0xcc69d2f9,0xf211966f),LL(0x429f98f1,0x10ce7510),L_(0x5736d4ee)}},
   {{LL(0xf3b848f9,0x8da51d84),LL(0xb2abe419,0xc9d1a3db),LL(0xefa624dc,0xc275c72e),L_(0x8b684df9)}, {LL(0xc682bec0,0x75f1bf6e),LL(0xfc8f823d,0x3f9f9e0a),LL(0xa497166b,0xb3363382),L_(0x3241d9b6)}},
   {{LL(0xe748b559,0x9a65f283),LL(0x47d0d598,0xeb923237),LL(0x600f8097,0xdfc5d530),L_(0x06c1d340)}, {LL(0x8c26801d,0x409b596b),LL(0xe50cc47b,0x2d0a1f78),LL(0x71018c42,0x4f9b2683),L_(0x2de345b4)}},
   {{LL(0x2273bfc3,0xd352b3d9),LL(0xfb7e2c37,0xb02762b6),LL(0xfb5035dc,0xc27af22c),L_(0xe5cf5a24)}, {LL(0x304d3903,0xf2e49e03),LL(0x31c42940,0x8e2941f7),LL(0xb3e476b2,0x3130439d),L_(0x729a7e6b)}},
   {{LL(0x2fb4a6c0,0x554fbba4),LL(0x1228460b,0xcbf47250),LL(0xda7db077,0xf5e0be2b),L_(0x8a989e94)}, {LL(0xbed3a180,0xe15aded1),LL(0x56931b09,0x2a994a1d),LL(0x7fb6175b,0x36407d3f),L_(0x785ad12c)}},
   {{LL(0x7466b3e3,0x296e5cc4),LL(0x6178d140,0x29e191e4),LL(0x9088d05b,0x1f9dc468),L_(0x631396d3)}, {LL(0x2665b7af,0x86ffc6cd),LL(0x04f52e9a,0x099491df),LL(0x4b21e0a9,0x27375e3f),L_(0x242c7c60)}},
   {{LL(0x974f4c4b,0x064ca079),LL(0xa02c0b2b,0xba3c8a64),LL(0xa1558503,0x13a69d2a),L_(0x3d67919d)}, {LL(0x148d5568,0x2bfc83bf),LL(0x3a653e93,0x1054b748),LL(0x69ae9c83,0x0a4cc47c),L_(0xf624d321)}},
   {{LL(0xacbde616,0x9e3e45cf),LL(0xd947c8c7,0x6374e6b2),LL(0xa439634d,0x5b100526),L_(0x7b5067f8)}, {LL(0xb742e1b9,0xfc26652d),LL(0x701d8d64,0x6746e242),LL(0x277501f0,0x262c9dad),L_(0xd3dd957f)}},
   {{LL(0x09a69072,0x9f4be9d7),LL(0xcf4ac2d6,0x20d76239),LL(0xe04b6795,0xf3257cbf),L_(0xec24a41c)}, {LL(0x48529c37,0xb3eb3d13),LL(0xcc1689ae,0x35003a26),LL(0x8712e8c7,0xafd7b5e3),L_(0x868af920)}},
   {{LL(0x620ac627,0xd27017eb),LL(0x75a3408c,0x28e8bc69),LL(0x7de280af,0xfab61ca0),L_(0xb76f92d0)}, {LL(0x3e502b01,0x528c5328),LL(0xffede226,0x05139646),LL(0x9bf76d0b,0x4d2256e2),L_(0xdd96e876)}},
   {{LL(0x4eb92472,0xe00a0552),LL(0x896685cb,0xfa213a96),LL(0xf0d4a520,0xe8f5fdb6),L_(0x0f380c61)}, {LL(0x5288ae5e,0x4d63b6c6),LL(0xca87cdad,0x7d544fbe),LL(0x69639e2d,0x941d9935),L_(0xed61f0e4)}},
   {{LL(0x31d063fa,0x72ebb946),LL(0x43d0a9df,0xa9c082ad),LL(0x53424334,0x10a77134),L_(0xdf849ce1)}, {LL(0x5a1815d9,0xdf56c5f3),LL(0x58feabaa,0x6bbd3105),LL(0x787ed1e2,0xa802a275),L_(0xa542b49b)}},
   {{LL(0x70042e82,0x0646662b),LL(0x33793ddc,0x6dfd5259),LL(0x873a716b,0x3ae6c62d),L_(0xd34ef5f0)}, {LL(0xb3489aa8,0x9974e0c1),LL(0x5b201375,0xae617d06),LL(0x25cf8d2d,0xfac50e53),L_(0xf5028682)}},
   {{LL(0x82900f26,0x20822d19),LL(0xdb5b7f7b,0xc050b43f),LL(0x610bf8a8,0xe4e5ddab),L_(0x59f6f46f)}, {LL(0x0bd3d213,0xd3a9f7a0),LL(0x29ffdb56,0x8a060b79),LL(0x1aed0fa9,0x7cd570f8),L_(0xa1cdd0bb)}},
   {{LL(0x0f6923b1,0xb9d66a95),LL(0xf9463b5d,0x6b4aafc6),LL(0x9dcb5191,0x7e8de98d),L_(0x3a174474)}, {LL(0xafeb6788,0xd8a6df5b),LL(0xb00ae812,0x1127ce66),LL(0x595dbbd9,0xab858cbb),L_(0x20456b6d)}},
   {{LL(0xf945d541,0x6e092bb4),LL(0x4b66cf94,0x27b14454),LL(0x7eca44eb,0x6daca5c9),L_(0x44fbdf91)}, {LL(0xfc99b4f1,0xcb39349a),LL(0x9f292892,0x611f2265),LL(0x7d2e1348,0xb91c9d21),L_(0x2511f796)}},
   {{LL(0x8d960ed0,0xc742fe68),LL(0xaeca436c,0x7a253140),LL(0xa736c99b,0x7053b3ba),L_(0x4df3a08c)}, {LL(0x1c4bf5fd,0x35ff874b),LL(0xa50437d0,0x85d7a155),LL(0xbb3ee79c,0x28f141d8),L_(0xa309d53a)}},
   {{LL(0x18a6fdae,0x052107c1),LL(0x4c1843ed,0x7a4941a3),LL(0x5b395717,0x10eeec3a),L_(0x41bed6c7)}, {LL(0xc77d493c,0xe3bf062d),LL(0xe2632a75,0x99abf215),LL(0xccb37627,0xf5482813),L_(0x65bddca7)}},
   {{LL(0x86262170,0x97f943dd),LL(0xe4f707eb,0x28ad9142),LL(0x1fca8434,0xe200b711),L_(0x5979d82e)}, {LL(0x7eeaf938,0xa0864853),LL(0x073f049c,0xdcbbbd26),LL(0xefca8545,0x5e115ac7),L_(0x81b25231)}},
   {{LL(0xfa392d8c,0x0b83214e),LL(0x6e208498,0x5535cb0e),LL(0xd7f17a61,0x9d606b9f),L_(0xf9ff6be9)}, {LL(0x532d4992,0x78a1ebcc),LL(0x8412a690,0xc80751aa),LL(0x3fc39e84,0xa4039f18),L_(0x16cc85fa)}},
   {{LL(0xb6b3e3bf,0xb29dd0f5),LL(0x3f372e07,0xfc4475a6),LL(0x1010f76a,0x499ca40b),L_(0xc9d3a123)}, {LL(0x505bb647,0x1a039620),LL(0x898b067c,0x71c1fd31),LL(0x8fba57f2,0x920244b9),L_(0x2f20dfbe)}},
   {{LL(0x09e67c8d,0xadba6513),LL(0xf1db6fa1,0x004a0b2d),LL(0x780eff8e,0x6530b59a),L_(0xdcf21d60)}, {LL(0x8ecdce5b,0x767463fd),LL(0x65b13c4b,0x70cc1a2b),LL(0xaf0f593a,0xb10b6a9a),L_(0x05c89578)}},
   {{LL(0x040645b4,0xae7f7819),LL(0x678205ed,0x421d2a1d),LL(0xcc32c0b0,0x2492206a),L_(0x43d4ee55)}, {LL(0xb779e43f,0x2296a21f),LL(0xd485f134,0x5fce8d33),LL(0x7dae00fa,0xd7fd8f23),L_(0x9c357857)}},
   {{LL(0xc2d0a0f7,0x683b4c52),LL(0xd1014606,0x457af899),LL(0x2de4b78e,0xed10d363),L_(0xe5547d1b)}, {LL(0x84fc2002,0x40cf6189),LL(0x1fa00e4b,0x6b948df1),LL(0x2643b489,0xa90cf1b7),L_(0x862ff071)}},
   {{LL(0xf9cd4743,0x2c263fe3),LL(0x3df3e881,0x7af59647),LL(0x4a452825,0xf8b4e248),L_(0xff9860e0)}, {LL(0xe9e940c2,0xfaa7bd3a),LL(0x4dcae86a,0x829a4b08),LL(0x0392152c,0x8e97fa6c),L_(0xfc62b042)}},
   {{LL(0xf8c2cf32,0x38ca48d4),LL(0xc8c6f843,0xb6e872e4),LL(0x1a5701ad,0x8137a142),L_(0x3332d11c)}, {LL(0xd8632d31,0xf65f8d02),LL(0xc57e0eee,0xed896e7d),LL(0x84c4a8f0,0x474d64c3),L_(0x9a310ea4)}},
   {{LL(0x1facebce,0x90943f9c),LL(0xabcd06f1,0xc0153740),LL(0x4bf7868b,0x2460277b),L_(0xf15a1c15)}, {LL(0xe70f731b,0xbbc3520f),LL(0xa0084581,0x1881feb6),LL(0x8c12c484,0x4461e279),L_(0x2cb0b794)}},
   {{LL(0x420f669d,0xc6234887),LL(0x1336960e,0x961fd208),LL(0x6690aec7,0x8dcc7988),L_(0xea8d5acf)}, {LL(0x6a7ab2c0,0xad19e666),LL(0x221a923e,0xf44304de),LL(0x3d295ba2,0x840fc746),L_(0xa849c9c6)}},
   {{LL(0xffb23eb2,0x796ddb67),LL(0x7f2ece47,0xb74ebddd),LL(0xfa40fdb7,0x6e9b86f6),L_(0x3753a599)}, {LL(0x517edc52,0xb0d32903),LL(0x9ce52e22,0x4e38e3ff),LL(0x609b3583,0xbd4bbc16),L_(0x470387a5)}},
   {{LL(0xc2767e71,0xb5992b12),LL(0x15e316ec,0x722e7585),LL(0x331c37c2,0x4f66d67c),L_(0x5a547f8e)}, {LL(0xad349dda,0x2343f37d),LL(0xc7ecc98e,0x991a0d20),LL(0x2f36fbc0,0x32fd452b),L_(0xbcd300ae)}},
   {{LL(0xe9e7130e,0x3965b7db),LL(0xfbb383d5,0xa434d230),LL(0xd5ddeb7e,0x5de1379b),L_(0x89236512)}, {LL(0x26844221,0x71474c32),LL(0x359ac63d,0x86ab382d),LL(0x6a2a41ee,0xb41d45a2),L_(0x4543bf02)}},
   {{LL(0xc03175e0,0xdd45b154),LL(0x51a93461,0xc32e33b5),LL(0x8e87a379,0x6030bb55),L_(0xe0f74319)}, {LL(0x1ee1fb1c,0x180c566b),LL(0xf64d3bd0,0xd5501439),LL(0x117f7e05,0x9fe6d9b3),L_(0x276cdeae)}},
   {{LL(0x79701f00,0x90d2c4af),LL(0x8ee8e755,0x19fa045c),LL(0x34e9da6a,0xea446558),L_(0x56f8560b)}, {LL(0x22273b04,0xf104982d),LL(0x7a1b70ed,0x0b6be603),LL(0x5df6c184,0xb3a79a2f),L_(0x3e690209)}},
   {{LL(0xb6a58003,0x506cdad2),LL(0x1cd413bf,0xcc9e3e86),LL(0x5af9735a,0x4669ee32),L_(0xf133df4e)}, {LL(0xa97f8799,0x343b9133),LL(0xd2df1205,0x5ed1097e),LL(0xc0f593eb,0xe1f7e724),L_(0x03f70d2b)}},
   {{LL(0xfbde70cb,0xe1aefce0),LL(0x28e77727,0x953078f3),LL(0xea43f71f,0xb1afcd33),L_(0xe72f4294)}, {LL(0x42df1489,0x5c609190),LL(0xd85d4064,0x9d39f16a),LL(0xd4734dcb,0xc6a7ed93),L_(0x73444dba)}},
   {{LL(0xc80deb54,0x786041c5),LL(0x2e784b12,0x73806b24),LL(0xa5394485,0xd5ed489c),L_(0x7c1354f6)}, {LL(0xc98caa8f,0x6f5aed20),LL(0x04a2cef9,0x343f789d),LL(0x6a61903c,0x0943ad59),L_(0x74525b1c)}},
   {{LL(0xdd468a2b,0x41295f9e),LL(0x79058179,0x767bb930),LL(0x8356b028,0x8630275b),L_(0x7d99704d)}, {LL(0x9677c30c,0xfe64fec1),LL(0xb6ad9348,0x7bfdd07e),LL(0x323c5204,0xff571db9),L_(0xabd2794c)}},
   {{LL(0xc6bef1c2,0x39e5ddd7),LL(0xa00ac9eb,0x4166efe7),LL(0x08022d52,0xec00b4ca),L_(0x89c305e2)}, {LL(0xdff9eb38,0x2f6d8e32),LL(0x1c9dee31,0x38a38882),LL(0x063cbbc8,0xd453f2df),L_(0x500b8e1b)}},
   {{LL(0x2a4b58b5,0x0d2d67f5),LL(0x704fba9c,0xb9671248),LL(0x3b601783,0xc86eeaf3),L_(0x3a076280)}, {LL(0xc6604758,0x057735ba),LL(0x3c7fe621,0xa0f157e0),LL(0xba90bf98,0x7ea681b4),L_(0xc697f17d)}},
   {{LL(0xf242b313,0x2fb2d012),LL(0x8b404954,0x301b1ef8),LL(0x3864cc2c,0xe8bf6c74),L_(0x27928a4a)}, {LL(0x37e42bc1,0x129459b3),LL(0xb079e9c8,0x95c688c9),LL(0xa8c506fd,0x2be35d42),L_(0xabd7b8a2)}},
   {{LL(0x8d7f03a4,0x0c337ebf),LL(0x9f5cd253,0x2b5789a5),LL(0x3bc3df04,0x808378fd),L_(0x57ce1b57)}, {LL(0x8394999d,0x9173aca8),LL(0x29069aea,0x68a97a68),LL(0x172c5e76,0xd7f9c6b8),L_(0x2bb82de1)}},
   {{LL(0x26324b43,0x069b62db),LL(0x3a04b8c8,0x2d6c75c6),LL(0xbb1fb7ef,0x8e5dc130),L_(0xdf0fbfa5)}, {LL(0x427bae31,0x8d6bb598),LL(0xae962471,0x4a1d29d8),LL(0x6a0b5064,0x1f5aad58),L_(0xcb90fbe1)}},
   {{LL(0xf5f95663,0x99c6500a),LL(0x182ee0a7,0xfc57412c),LL(0x9acab63a,0xd7a3f799),L_(0x579dffc7)}, {LL(0xb294554e,0x2e7244f9),LL(0x7a563978,0xc577a189),LL(0xd9421c12,0x4a33607f),L_(0xca35ca5e)}},
   {{LL(0xcf4cc636,0x476c2c16),LL(0x84dd3019,0x40d43930),LL(0x9f6a9cbb,0x8de41867),L_(0xa0782cd2)}, {LL(0x8e9807cc,0x335870d6),LL(0x873bb057,0x237820cd),LL(0xd2763265,0x817bb568),L_(0xac61e5ba)}},
   {{LL(0x277845cc,0xb4216df3),LL(0x28a00ba3,0x4ae7f387),LL(0x8d18d55a,0x83dc382b),L_(0x9929ba4b)}, {LL(0xace5a497,0xc4acbdd9),LL(0x1a149f46,0xfee31905),LL(0x03677afd,0x7f382dd5),L_(0x038b4fd6)}},
   {{LL(0x39c19776,0x7b1469c5),LL(0x770906b4,0xefd3e00c),LL(0x39b76a30,0xb757175c),L_(0x77b00db2)}, {LL(0xea73fb31,0x00e5eb12),LL(0x564b2b5a,0x5dec4d76),LL(0x16670a8b,0x6cedfc0a),L_(0xfc26f8d3)}},
   {{LL(0x0f967635,0x5f72f035),LL(0x492f15fb,0x9fbbd203),LL(0x8a57f34e,0xca278cc2),L_(0x136d3def)}, {LL(0x3233be88,0x56bc1e3d),LL(0xb7d079d2,0x167694bb),LL(0xb849beca,0xc12eafab),L_(0xf42c9064)}},
   {{LL(0xc22b8b50,0x35110b41),LL(0x3593434b,0xc03f7534),LL(0x2a685ee5,0x38e7d616),L_(0xd80d02e7)}, {LL(0x681060ac,0x47c0e163),LL(0xa5dbbbef,0xaf6a4e94),LL(0x58c7a38b,0x00b0f102),L_(0xbe4a965e)}},
   {{LL(0xa0bfdebf,0x00ea08c9),LL(0x0a49eb36,0x19299c59),LL(0x4a5b799c,0x1389e249),L_(0x6bb607bd)}, {LL(0x003f62bb,0x21dd6f70),LL(0x648ac9f7,0x0d28002f),LL(0x46ddb53c,0x123f4a38),L_(0x0b454608)}},
   {{LL(0x6cd705df,0xdc0b4f05),LL(0xa26650ba,0x584543cc),LL(0x9623d15d,0x52a5b5a4),L_(0x8530847b)}, {LL(0x95176358,0x2b9634a4),LL(0x88564796,0xb76d8613),LL(0x8cd268e3,0xdd09c6f7),L_(0xfbbf3c6c)}},
},
/* digit=25 base_pwr=2^175 */
{
   {{LL(0x40600691,0x076ba1e2),LL(0xaf9141d1,0x3cbbece3),LL(0x90d62387,0xeae39108),L_(0x39ae931a)}, {LL(0xef1b81b3,0x7c192920),LL(0x51f12b8a,0x07e50a2b),LL(0xf1cfbc63,0xaa15f8c6),L_(0x26ff0bea)}},
   {{LL(0x9f2ac6a5,0x5ac7dc31),LL(0x5a5fa2f2,0xd3536baf),LL(0x918f6160,0x19e26cfd),L_(0x3d611b68)}, {LL(0x58b92257,0xfbe0d264),LL(0xa4b3a16e,0x6f69ee33),LL(0xab7e3e38,0xfb87b2d4),L_(0xe4cf7a53)}},
   {{LL(0x4428fe8d,0x28c21968),LL(0xe6afbadf,0x82133ac1),LL(0x08d0d89a,0x79f622a4),L_(0xfc61f01d)}, {LL(0x0f6397fa,0x9afa4014),LL(0x1819a677,0x8b6b825f),LL(0x9e89a4ea,0xf6189381),L_(0x4c306973)}},
   {{LL(0x16b5009f,0xd0fd8e2d),LL(0x85f596e9,0x75cd4af4),LL(0xc650d10a,0xc0ff364a),L_(0xaa432f6b)}, {LL(0xb9e1fd35,0x60cd9ff6),LL(0xa81387b4,0xf5ffdf83),LL(0x1fb57eae,0x0f74dad8),L_(0x77ebce28)}},
   {{LL(0xcd3d74f0,0xe7801dbf),LL(0xc705f5f3,0x4cae30c4),LL(0x892e14fd,0xee1d51c0),L_(0x235f35c6)}, {LL(0xcb840ec4,0xc690d3bf),LL(0x496db46c,0x6a393aed),LL(0xf95db976,0xa0c8c115),L_(0x3a2ff859)}},
   {{LL(0xad9729df,0x21208e24),LL(0x50a51554,0x8a669134),LL(0x76da2c79,0x739f33c4),L_(0x47f7655f)}, {LL(0x3faf5d73,0x6df3da71),LL(0xda653a05,0x23d729bd),LL(0xe0dbba03,0x7efefa7f),L_(0x8546d123)}},
   {{LL(0xa29be43b,0x63a12c13),LL(0x81bccb17,0x0f3de027),LL(0xb8ad1f49,0x2ee0f6a0),L_(0xeca64ff4)}, {LL(0x2a0247bd,0x6c14c6a4),LL(0x27d0478a,0x15b008e7),LL(0x9dc8b9de,0xe6cf3269),L_(0xde4a727b)}},
   {{LL(0x8a41f62f,0xe9139920),LL(0xafcf2338,0xe343f86a),LL(0x8626eb92,0x330ff008),L_(0x62bc5b8f)}, {LL(0x770a0f51,0x9ed3ab3d),LL(0xacdfce8c,0x87533c3f),LL(0x198a1e9a,0x36ceed7d),L_(0x366edc66)}},
   {{LL(0xad4f0edd,0x9eb6ef96),LL(0xa0790c0e,0x8f2b7875),LL(0x8212d5cc,0xa20f0405),L_(0x2e6749a5)}, {LL(0x43c2e598,0x0b5bf3aa),LL(0xf0598778,0x695b110e),LL(0x6b2a153e,0xacd67e10),L_(0x9a4287cf)}},
   {{LL(0xc9194ba8,0x1df015df),LL(0x5545ae11,0xe5e57b96),LL(0xf86d852d,0x34894ebb),L_(0x61b42915)}, {LL(0xaab92979,0xb25bf6b0),LL(0x2f858332,0xf88d9382),LL(0x082b80c9,0xa64fc13d),L_(0xdb34263a)}},
   {{LL(0x4fc8d228,0x6c1819aa),LL(0x46f4e2d8,0x790019d3),LL(0x3e6d5835,0x31b5f8c4),L_(0x0471eabf)}, {LL(0xd70615ba,0x7377bdb1),LL(0x4f0b793d,0x3d8b3ee2),LL(0x1e0b124d,0x8be72522),L_(0xcb228953)}},
   {{LL(0xd05e1a54,0x3d345d79),LL(0x4e4eb11a,0x9b0c74e2),LL(0x4f7cb301,0xd3f60831),L_(0x9b37cac3)}, {LL(0xdd153410,0xd42ac14a),LL(0x4b82b6e1,0x67daba89),LL(0x062085e8,0x293aa2ea),L_(0xdafbaa02)}},
   {{LL(0x57e823a2,0x66dc4352),LL(0xeee6c065,0x9d1334ad),LL(0xe974bd6b,0x51129e57),L_(0xc6e911b2)}, {LL(0x4a591897,0x143fbbc0),LL(0xe567aa72,0xfcf32ecc),LL(0xd98a2bac,0x985f3567),L_(0x4940de3e)}},
   {{LL(0x810e3b3c,0xabba8dd8),LL(0xbc9b4d54,0xd024fdfa),LL(0x09ce7445,0x56df5156),L_(0x869985ec)}, {LL(0xa279f60b,0xd64936f1),LL(0xf9f4dd90,0x147e3ae6),LL(0xb0502e71,0x9b4a0a61),L_(0x2b7784e2)}},
   {{LL(0x10ea913e,0x228b551a),LL(0x132cf6ee,0xd39b6d23),LL(0xb41622ce,0x1166f594),L_(0x4f82b870)}, {LL(0x771a8cbc,0xc4a0434c),LL(0x1f3f48d6,0x4e8415e6),LL(0x185bfa20,0x0549b016),L_(0xab597880)}},
   {{LL(0x8f78c75c,0x7783a236),LL(0xcacbfea9,0x185d7d80),LL(0x37384da3,0xd767d6c6),L_(0x1b40ecf0)}, {LL(0xb58df6c4,0xcb35e135),LL(0x1bdb10dc,0xdede089c),LL(0x51f82512,0x032e4873),L_(0x86fff3dd)}},
   {{LL(0xae08ea09,0x993762bf),LL(0x3c5e8711,0xe9e1ac75),LL(0xc3fb58ec,0x5942a654),L_(0xdc39e05c)}, {LL(0x6f00a879,0x4c7c1a57),LL(0x76f183b7,0xb5b4e070),LL(0x02f3f5a3,0xc7635031),L_(0x95a5b1bf)}},
   {{LL(0x1973898f,0x44aeebfb),LL(0x6fdec0d8,0x22963505),LL(0x4dccb72d,0xc61c3dc6),L_(0x29d98bbe)}, {LL(0xc79d4f82,0x0b4d124c),LL(0x60e1bdf3,0xab61b974),LL(0x60907706,0x502b0c26),L_(0x52fb8b7c)}},
   {{LL(0x9cc7f5da,0xa0c27cd2),LL(0xa30b225d,0x39c837e6),LL(0x14e5e017,0x90beb6db),L_(0xca71c799)}, {LL(0x3452a918,0x6406e7a2),LL(0x8b449af9,0x44086ede),LL(0xd95a3674,0xe83116d2),L_(0xc945cb30)}},
   {{LL(0x757467e5,0x195a49f1),LL(0x19bb9a9f,0xbd375548),LL(0xdd63e900,0x32a666f4),L_(0xbb5bae80)}, {LL(0xb06e326c,0x71975390),LL(0x7200d75c,0xa8071dcb),LL(0x9bb8f1f0,0x34371f37),L_(0xd14c5988)}},
   {{LL(0x9fc79376,0x330b4d2c),LL(0x4766d5c5,0x89487193),LL(0x96d443b1,0xb683d913),L_(0x55cd4038)}, {LL(0x25e49098,0x0b19d841),LL(0x4de7c5cc,0xc211c53d),LL(0xb34627f6,0x63d637ee),L_(0x05d48e84)}},
   {{LL(0x2fb2a30d,0xa9ab961d),LL(0x587b4227,0x7f2e1082),LL(0x9bacdb2f,0xf2fed11b),L_(0x4327e6c8)}, {LL(0xf5f7ca94,0x59f2340a),LL(0xdae690d8,0xa241b47f),LL(0x20f9a82b,0x04266881),L_(0xb60ba27e)}},
   {{LL(0xfc8a6083,0x4b7ea198),LL(0xf7bd165d,0x0439da93),LL(0xaff22d2a,0x6e2c180a),L_(0x5fa28be1)}, {LL(0xb5422e8a,0x8d8de819),LL(0x8623fa12,0x1c7aa689),LL(0xf1b013b4,0xecdeea2c),L_(0x1412065d)}},
   {{LL(0x1b8b14e9,0x949b80c2),LL(0x9e45f562,0x7634d5e7),LL(0xeacf4d90,0x89e47023),L_(0xed86f825)}, {LL(0x2a28ee33,0xa46549d3),LL(0xcc9d20c3,0x91467c95),LL(0xb5aaafc8,0xaad820ff),L_(0xfa9b0b7b)}},
   {{LL(0x3a989cf8,0x56a19c64),LL(0x694699c4,0x6a9264c2),LL(0x2172bedd,0xfcd8d182),L_(0x71a622e6)}, {LL(0xabaee9e8,0x096356a3),LL(0x5df0e982,0xb3c8273f),LL(0xbff93f13,0x15499fe5),L_(0x035d8bec)}},
   {{LL(0xe7f02675,0x8665d519),LL(0x0b392d7e,0x3114f640),LL(0xda3e7314,0x27724fe5),L_(0xb0000497)}, {LL(0x9626e05c,0x92676514),LL(0x86de4867,0x70fdd068),LL(0xbfb7e396,0xc9127734),L_(0xec308376)}},
   {{LL(0x889048c2,0x13351e00),LL(0x9ee3edde,0x8a42c8c0),LL(0xe9b0df44,0x039a9289),L_(0x3ba48603)}, {LL(0x9ba675fa,0x1bc5a4ed),LL(0x28dbe0e2,0xc1c91604),LL(0x33d13a46,0x683a466a),L_(0x6f0e40d3)}},
   {{LL(0x6f169f5d,0x49b1e542),LL(0x5467b60c,0x6be4fa18),LL(0x63a775bc,0xf10e3ad8),L_(0x555978a5)}, {LL(0x2b932386,0xe2647c33),LL(0x2d98173b,0xc10e0d58),LL(0xa46de613,0xd4d665e8),L_(0x8c2f942e)}},
   {{LL(0x76f88409,0x83f96957),LL(0x11159549,0xb83760f5),LL(0xa26fb69f,0xada04407),L_(0xe1ffa9b1)}, {LL(0x7985249d,0xf63dc16c),LL(0xb8212592,0xa9f05a9e),LL(0x8f2b706f,0xa84d6d60),L_(0x0952d255)}},
   {{LL(0x9ef2ca53,0xc1ad35cf),LL(0x8063b121,0xf54f3d1d),LL(0x2c69c142,0x2eddbe16),L_(0xe20cbf04)}, {LL(0x58b9d3e5,0xc6a6486c),LL(0x5f1b4348,0x30a9fb2a),LL(0xaa7af663,0xbd211312),L_(0xf09747fa)}},
   {{LL(0x6aeadb7e,0xb3384570),LL(0xad4b300b,0x80077723),LL(0x7e0408aa,0x92c8f393),L_(0xcdb75950)}, {LL(0x91a66b10,0x01c71c42),LL(0x98806233,0xd3e2ebf9),LL(0x4ccfab0f,0x2c91c51d),L_(0x0daddeb0)}},
   {{LL(0xa5136305,0x1656bed1),LL(0xf9475d4e,0xd5bd811f),LL(0x29f67636,0x71b2886d),L_(0xce5ffc47)}, {LL(0xf8ec4c6c,0xed19aefd),LL(0x00fbe22e,0x3a69ea47),LL(0x1a593c52,0xed7635e4),L_(0xfea5dbc5)}},
   {{LL(0xb6a32091,0x0a800d89),LL(0x3e952cbe,0xe00a730d),LL(0xb849574d,0x95e45fa4),L_(0xd0b6f06f)}, {LL(0xc0b38793,0xca47bacc),LL(0x3aea6517,0xa1c9b9db),LL(0x98d62545,0xf79f468d),L_(0xe75cf2cd)}},
   {{LL(0x6d96b85c,0x92cfaed8),LL(0x8aa22d94,0xe9d09bd1),LL(0x951cb3e9,0xcdf3e590),L_(0xc6443f1d)}, {LL(0x5ee4705a,0xa83030bd),LL(0x4ab4bb2e,0xbbdfbd6c),LL(0x126bf8d3,0xab5c4613),L_(0x78abc625)}},
   {{LL(0x1d099c24,0xb14045e9),LL(0x75f0192a,0xc8658ab0),LL(0x66a1d69f,0x488c7367),L_(0x27671771)}, {LL(0x10f50e10,0x41dc68c8),LL(0xdbf2b527,0x56c509b4),LL(0xaa6e15c2,0xc052c755),L_(0x6264eb14)}},
   {{LL(0xf102c97a,0x361df88d),LL(0x8133758d,0xfcb2818c),LL(0x89d87e0a,0x8313147e),L_(0x156564f5)}, {LL(0x92996691,0x0305e3bb),LL(0xb530ebe1,0x3ea90757),LL(0x75ed509b,0xe3da205e),L_(0x8810aa6d)}},
   {{LL(0x0936433b,0x66bf33ad),LL(0x5cdc1d84,0x20489613),LL(0x40fff511,0x02569cc0),L_(0xa46f7fa8)}, {LL(0x6bea5f0b,0x03ae7465),LL(0x6ca8cba2,0xa43eaabe),LL(0x9d34e4a3,0x338f15f5),L_(0xf175cde9)}},
   {{LL(0x37a46a76,0x43fb29da),LL(0x0f113fed,0x1e4e18be),LL(0xc9bc3624,0x0891f557),L_(0xf2de428e)}, {LL(0x6cb133b6,0x72c78284),LL(0x0bca6469,0x920b91e7),LL(0x4ad07f92,0x9aca547e),L_(0x254d2e0c)}},
   {{LL(0x48b9afe1,0xbedf71c7),LL(0xfdf235a0,0x7b745717),LL(0x707d37d5,0x519516e4),L_(0xb70d0d1d)}, {LL(0xf179e489,0x026e2e98),LL(0x7b9cd555,0xa64e85e9),LL(0x05a8ce5e,0x490bbe2c),L_(0xfd0ac7d5)}},
   {{LL(0x5ef0e444,0x4518decb),LL(0x5704b61f,0x84b0fcd7),LL(0x05e3a6d4,0x761243d0),L_(0x09fdcc95)}, {LL(0xf157aba1,0x52cc970f),LL(0x2f40eec4,0x545ea4c0),LL(0x3cfbc5b3,0x82e0b382),L_(0xd24582c3)}},
   {{LL(0x3def68b9,0x91b73501),LL(0x63a13aee,0xf4c2f983),LL(0x64a186c4,0xd30674ab),L_(0xae849f85)}, {LL(0xd639edb0,0xbd3a7bd8),LL(0x4e22b52c,0xcf732f2a),LL(0xdee6130a,0xa16934a2),L_(0xa208c2f8)}},
   {{LL(0xee275071,0xacddc898),LL(0xdcfd6055,0xc137eacf),LL(0x4226123c,0x22c7c6d7),L_(0x18caa3b3)}, {LL(0x575ea43d,0x27a1cc63),LL(0xcfd9c842,0x1d810b5b),LL(0x435cc5b0,0x98cdd25d),L_(0x03338f7f)}},
   {{LL(0x1bca7595,0x39e40f6e),LL(0xa3f28110,0xf4b9f93a),LL(0xbb9b4a37,0x6d2f9be1),L_(0xc7ef39dd)}, {LL(0x288d64fa,0xb0848ac0),LL(0x9e6a4583,0xe3934b4d),LL(0x65ecc897,0x7961633c),L_(0x12939ce7)}},
   {{LL(0x69332018,0xdf1e1997),LL(0x0eb1a850,0x55fdefde),LL(0xcfe64a91,0x6d23295e),L_(0xf848ec96)}, {LL(0x6ccd6db0,0x8a058163),LL(0x4a1159c7,0x8e9a68bc),LL(0x145557f9,0x44e87592),L_(0x744a6684)}},
   {{LL(0x570af433,0xd2524bb7),LL(0x71ee45c1,0xe5696936),LL(0xae2e62f4,0x90655f39),L_(0x4aa6dd85)}, {LL(0xce93159e,0x170e2563),LL(0xcd77d544,0xb60f5503),LL(0x1637c5b9,0xbbddebd8),L_(0x116cc13c)}},
   {{LL(0xa79f3d54,0xda76037a),LL(0xe567b31a,0x1b943930),LL(0x81454a32,0x8ce526e9),L_(0xa68887d1)}, {LL(0x7c692a2a,0x5ccae9e7),LL(0x2aa4281c,0xbad7967a),LL(0x27a06af9,0x67a8cecd),L_(0xb0666bd6)}},
   {{LL(0x18d87b3d,0x3bf215b9),LL(0xf3b762b4,0x13a2d20a),LL(0xc7ded91e,0xf2dddefc),L_(0x491f86d1)}, {LL(0x9175dde7,0x1a33ec16),LL(0xbfa502ae,0xde7bc4a6),LL(0x8b90c1c4,0x011d4fdb),L_(0xae49e966)}},
   {{LL(0x339d5841,0x30030ab7),LL(0x9b2323cb,0x93a59a0c),LL(0x3dfca103,0x6d677f76),L_(0x615cef3a)}, {LL(0x5b19bbca,0xd8927084),LL(0x143864e2,0x0d3dc8c8),LL(0xe2a53ef5,0x7bfce607),L_(0x98d95aec)}},
   {{LL(0xbfe42fd9,0xe4480a82),LL(0x674699df,0xa3a3e505),LL(0xf6bd5725,0xa3bae3b5),L_(0x93074bef)}, {LL(0x2f753cf5,0xc278e6d8),LL(0x9075dc94,0x042fbabf),LL(0x66f70e54,0x2912d169),L_(0x77ae21bf)}},
   {{LL(0x8e1def19,0xba0002d6),LL(0x9d21da23,0xa31c1302),LL(0xc9cb3757,0xc0b1cdab),L_(0x478c82a0)}, {LL(0x8ebd0b98,0x708a627c),LL(0x95df97f0,0x00716397),LL(0x036873bf,0x9b0c552c),L_(0x31bdcc00)}},
   {{LL(0xa7f4f676,0xdb1b2796),LL(0x16f6892d,0x287bb1a9),LL(0x513e742d,0x83e7ad99),L_(0x5ba54190)}, {LL(0xd83035ed,0x3ab45168),LL(0xd55025e1,0xc5e72511),LL(0x57264580,0xa5b527ed),L_(0x246fe503)}},
   {{LL(0x2956f5b6,0x9456aa98),LL(0x72287d14,0x2d9d56ca),LL(0xbd98da03,0x783e5a79),L_(0x1c038840)}, {LL(0x278213f9,0x44dd9214),LL(0x8abbd0fb,0x1ae00f0f),LL(0x546492b8,0x9365b6f9),L_(0x638f9886)}},
   {{LL(0x2772a4d8,0xe568c53d),LL(0x56dd3ec8,0xc5798254),LL(0xad2418b2,0x40cc9856),L_(0x5fe0c00d)}, {LL(0xcc8ceaa1,0x1cdd7255),LL(0x17304349,0xd65e8630),LL(0xe17ec8e0,0xb4dc2a26),L_(0x62eb4c57)}},
   {{LL(0xb31137c7,0xdb4ee5e0),LL(0xa85ff057,0x79aa7312),LL(0x0fb19074,0xf1a5850d),L_(0xb1c8c20f)}, {LL(0xa65022af,0x407175d5),LL(0xc25f5ad9,0x1debba7b),LL(0x0ed43d07,0xa2739236),L_(0x55db40b6)}},
   {{LL(0xeb7ff1d6,0xbb91225a),LL(0xb6ec7895,0x7915895d),LL(0xec277689,0x53e7c8ad),L_(0x503d01fd)}, {LL(0x427cc57e,0xff30aa1b),LL(0x789118fb,0x28a3c0f3),LL(0x45dce70e,0x7758b97b),L_(0x2ad05594)}},
   {{LL(0x79df6beb,0x64d25f38),LL(0x7c1271cb,0x1d5f4772),LL(0xafba6d13,0x24a4fbdb),L_(0x68588b5a)}, {LL(0x9bd0327d,0x4408d2f3),LL(0xb9962d65,0x0c0efa60),LL(0x1adff959,0x3e504930),L_(0x1a7248ef)}},
   {{LL(0xc91f5609,0x50371a5d),LL(0x10a09de0,0x8f3d868e),LL(0x544a1f87,0x05603976),L_(0x9bb49da7)}, {LL(0x4f023f6b,0x2686c470),LL(0x22de25c7,0xad3c86da),LL(0x7b318aae,0x0e37c5eb),L_(0x6e0af008)}},
   {{LL(0x9cc2caa4,0xc4103bd9),LL(0x28c77b28,0x2b106e89),LL(0xc44c9d88,0x02942cad),L_(0x045b3511)}, {LL(0xae7c29af,0x985fde90),LL(0x3e691e0d,0x5ea3d69a),LL(0x66200341,0xa7fa897a),L_(0xc1b7c5c2)}},
   {{LL(0xf2dba47e,0x858b20b7),LL(0x94f1ad67,0x80c7e010),LL(0xcb454e18,0xe75e697e),L_(0xea8630aa)}, {LL(0xa6af9eed,0x7a69400e),LL(0x7d8e1f8e,0x73d9d6c2),LL(0x87354e0a,0xd949265a),L_(0xddd78248)}},
   {{LL(0xd65411f3,0xbb95794c),LL(0x56456f06,0xdb7b7288),LL(0xb5869a96,0x7de9cd89),L_(0x4dd26048)}, {LL(0x235bdd25,0x6b7c47bf),LL(0x3a2f738b,0xb8e2adff),LL(0x43a158b6,0x6452e6ec),L_(0xb59a08ed)}},
   {{LL(0x3b9a55fb,0xfd294202),LL(0xd446ba91,0x77e86f2f),LL(0x0bfb0e2c,0xb347e482),L_(0xda5432f0)}, {LL(0x811a63d8,0xda7568f2),LL(0xfd584f1c,0x32ddb6b6),LL(0x3e5b388e,0xfbef98fa),L_(0xa2090daa)}},
   {{LL(0x29e9bf1c,0xebc6de84),LL(0x8601a460,0x77ae022e),LL(0x0fe7b2f5,0x51718d1f),L_(0x768e05f3)}, {LL(0x8f61fd67,0x107bee81),LL(0xdafff971,0x45e70a15),LL(0x775aa267,0xf5af9168),L_(0xa74ac1aa)}},
   {{LL(0xcc4ccaf5,0x7770d24c),LL(0x6c27e8ad,0x931b3ac4),LL(0x0c21256b,0x5eb3480c),L_(0x24df253e)}, {LL(0xd60ece5b,0x32c4c950),LL(0x304257df,0xfa6d0590),LL(0x3821aab8,0xcfd48043),L_(0xe9ed7945)}},
   {{LL(0xa563da21,0xf927be40),LL(0xa625fc39,0x65697957),LL(0xc708ccff,0xc7c05c00),L_(0xe3551df0)}, {LL(0x1db01e2b,0x4a824ae6),LL(0x00aff255,0x21fd2abf),LL(0xdc1865ba,0x489f78db),L_(0xc47678ec)}},
},
/* digit=26 base_pwr=2^182 */
{
   {{LL(0xa7314858,0x4d52ad98),LL(0x44075a5f,0xa2ad3d54),LL(0x5f264197,0xa5da6ace),L_(0x1bd94b24)}, {LL(0x1046d21c,0xaddbfb84),LL(0x9deccfa5,0xdb3e6a82),LL(0x38b0cd4d,0x0ae5bd28),L_(0xcdefc743)}},
   {{LL(0x11028ec7,0x9f68d2fc),LL(0xdb80636b,0xda67d396),LL(0x399cbe22,0x058ae3aa),L_(0x9a20e964)}, {LL(0xa4afe4c5,0x0655d446),LL(0x82be9278,0xd1f696f1),LL(0x02a07951,0x78c7207b),L_(0x5473af3e)}},
   {{LL(0xa430f529,0x48532007),LL(0xe1daa6d9,0xf9c913ed),LL(0x8ffcdbb0,0xbd0a98e5),L_(0x9b2bdde3)}, {LL(0x3eec4099,0x58bd4134),LL(0x4dcdfb2b,0x524e44ad),LL(0x29153443,0x7457124c),L_(0x9190c343)}},
   {{LL(0x0c51f106,0xe34c46c7),LL(0x1a8f4562,0xa808bcef),LL(0x6cddda1e,0xb1bb8fc9),L_(0xa502ea7e)}, {LL(0x9e63e9ac,0x9421a0eb),LL(0x124b21a9,0x09abe1ac),LL(0x634ea315,0xfc9b2b69),L_(0x4d2d4d06)}},
   {{LL(0xd47adb81,0xa4e345b6),LL(0x370847bf,0xc8a3f346),LL(0x242713b8,0x10782ff2),L_(0xbc286c44)}, {LL(0x0501d40b,0xbe6fa011),LL(0x72197eb9,0xe784765d),LL(0x23d8a846,0x9c366368),L_(0xe57509e0)}},
   {{LL(0xb79e2ec1,0x9344ebd4),LL(0x41626ac2,0xee0983f2),LL(0xb61e001b,0x1c404975),L_(0x05deb3c3)}, {LL(0xafd8bc28,0x10e30e79),LL(0x4f5429b7,0x90c587f7),LL(0x0cc2b3bc,0xe64edacb),L_(0x2a32f215)}},
   {{LL(0x683db291,0xa18ec5e3),LL(0x597408da,0x0cdbd7b0),LL(0xb4dc90d1,0x9e4003b1),L_(0xf0377012)}, {LL(0x99f01d42,0x2f737bf9),LL(0x138b7393,0x2c744a14),LL(0xbf91bb7d,0x4390dd24),L_(0xd3eb9506)}},
   {{LL(0x7da97135,0x1f5f9c08),LL(0xcbb88522,0x60a61939),LL(0x590e948d,0xd63bd64a),L_(0x45a94350)}, {LL(0xc3f04dfe,0x8b9c0e7f),LL(0xc46a6c09,0x24ed2a64),LL(0xe5cb409f,0x6aa2f787),L_(0x90fd8928)}},
   {{LL(0xa6f20c60,0xc519de26),LL(0xc3ac3715,0xb2ee7698),LL(0x915329a3,0x0ca44fa0),L_(0xe61346f3)}, {LL(0x22cc3439,0xe4d76c97),LL(0x37f7afee,0x360a76e2),LL(0x6bdfe1ff,0x536926c2),L_(0xc048c2a4)}},
   {{LL(0x33feb155,0x57103466),LL(0x9e779a72,0xf0ac4da3),LL(0xf0cf02a3,0xc4a65bf2),L_(0xc464a2f5)}, {LL(0x9f49b51f,0x771064c3),LL(0x545f80d2,0xf119551c),LL(0x142107aa,0xc2378a52),L_(0x233fc912)}},
   {{LL(0xd5e26db1,0x2c57b6eb),LL(0xe31036a2,0x2260e245),LL(0x18945c38,0xa0a4a4f0),L_(0xa70971ce)}, {LL(0x51a8f045,0x78d2269e),LL(0x3f54f031,0x08084351),LL(0xd1a3f072,0xcf63943b),L_(0x1bbad8a4)}},
   {{LL(0x450a90c1,0x7535f2ce),LL(0x2202c2ff,0x29cb37b3),LL(0xad07269a,0xc576098a),L_(0x233792b7)}, {LL(0x368c1c44,0x5864342f),LL(0x02886c17,0x2dca7f73),LL(0xb1a39467,0x74bab9ee),L_(0xe841def5)}},
   {{LL(0x8df7b590,0x7f584614),LL(0x0ec350c2,0x3c0acdb3),LL(0x2da65636,0xb160863f),L_(0xe9a43081)}, {LL(0xb0ebfa84,0xce3663d0),LL(0xafbf1395,0x69ff1f32),LL(0xdfa06d4c,0x9aa85d56),L_(0x4f103987)}},
   {{LL(0xc94ce26f,0x7df657b9),LL(0xf5ac31c8,0x0b939fc2),LL(0x1a02ac9f,0xd28fa99b),L_(0x6407f45c)}, {LL(0x9ca2e994,0x06494081),LL(0xe60e26a5,0x6719d6cd),LL(0x44cfce93,0x4db05d61),L_(0xf189ccff)}},
   {{LL(0x665df7e1,0x16a82c07),LL(0xd179a921,0x85dffbcc),LL(0x51227dd4,0x1d15b312),L_(0x129de8ff)}, {LL(0x28e8d109,0xe117dd1d),LL(0x3f1a519b,0x3db3c554),LL(0xd473a86f,0xe470bd93),L_(0x48ca17dc)}},
   {{LL(0x2cd73074,0xce23cb8f),LL(0x025cce52,0x27d52714),LL(0x7128e28f,0x60a7af16),L_(0xee37af3a)}, {LL(0x291c3d92,0x228c1193),LL(0x8bcaaced,0x559ca84b),LL(0x60835724,0x5f7c29e2),L_(0x42db013e)}},
   {{LL(0x80630e79,0xd7266cd9),LL(0x5ce7356c,0xb190aae6),LL(0xc0bdd70b,0x086e72f7),L_(0x3319c072)}, {LL(0xef5d0e48,0x4ae3ba69),LL(0x086343f9,0x95fb3fdf),LL(0x80bc2aba,0xbf3ecc51),L_(0x37cadfe3)}},
   {{LL(0xaab46c1f,0x66dd80d9),LL(0x32cd82a6,0xaf92ed97),LL(0x11e8daf7,0x6268a0b4),L_(0xbd297ce7)}, {LL(0x1771ff6b,0x971bdef8),LL(0x8c4eabfb,0x446b9cea),LL(0xcbfa339f,0x83d9f14e),L_(0xbd36bd3d)}},
   {{LL(0x38641225,0xe3a86074),LL(0xf0d2a9f0,0x46459726),LL(0x68a87b42,0x27fad851),L_(0x465da389)}, {LL(0x6ec9b313,0xa438dfac),LL(0x1024a804,0x2f7651af),LL(0x3e3b2c83,0x9e904c3e),L_(0xb0cbbf5a)}},
   {{LL(0x3daddf8b,0x4f3cafc7),LL(0x4571821c,0xc7f93897),LL(0xb59cdc87,0x564a1eff),L_(0xd3fc8e11)}, {LL(0x32b8d1f1,0x210954cc),LL(0x485b3e16,0x0b89424a),LL(0xa39a8650,0xc71b961e),L_(0x50587567)}},
   {{LL(0x938ffcc2,0x3f39a45a),LL(0x39ff9b1a,0x22ec814a),LL(0xe9320257,0x293c62a7),L_(0x30d60ad1)}, {LL(0xfe028f50,0x51699000),LL(0xecc36d79,0x687388c0),LL(0x782811c0,0xb9b6100c),L_(0xd4aec2a8)}},
   {{LL(0x4fa8529f,0xbf6700b9),LL(0xdb67bc3c,0xb75d6fc1),LL(0x81a05a44,0x2a63deb4),L_(0x2c00d649)}, {LL(0x4f57cb6a,0xc429bae2),LL(0x962a325d,0x45995d71),LL(0xf818f011,0x07c3cde9),L_(0x8b0fa963)}},
   {{LL(0x1767b13e,0xdcf6f7fc),LL(0xd9cd3792,0x4c91cbcf),LL(0xa1b347a0,0x9daa99a0),L_(0x6b0d6566)}, {LL(0xe85fb43d,0x448838bb),LL(0xe794fda6,0x9a62c0b7),LL(0xb6b1fe7b,0xc0cb0a63),L_(0xd09a8df9)}},
   {{LL(0x0510ae51,0x4c395b4e),LL(0x8e70bc90,0x49d7c304),LL(0x04b3e853,0x9d40e63f),L_(0x4a2dc6ce)}, {LL(0x6ec0b89c,0x36339e46),LL(0x385a4594,0xb1a1676b),LL(0x455a09ad,0x9304dd88),L_(0xc6ff5bf9)}},
   {{LL(0x7502d288,0xe085cdf7),LL(0x187b4002,0xa35b991a),LL(0x4c4e720e,0xab690dee),L_(0x86287cf7)}, {LL(0x05d1c250,0xe2c63b4e),LL(0x9adbcac5,0x6b60d74e),LL(0xddd28bc7,0x4248e502),L_(0xef074338)}},
   {{LL(0xc29777cd,0x896670f3),LL(0x08020d96,0x3a144e73),LL(0xe06c4b5f,0x9786e6bf),L_(0x3da82523)}, {LL(0x4a099ef4,0x4fe3491e),LL(0x99758e3b,0x14bb92ff),LL(0x0c3b82b6,0xbab9fec3),L_(0x45c4baf6)}},
   {{LL(0xc471a0d6,0xb5a9a1df),LL(0x4e7e7ab2,0x56e0f739),LL(0xce6cd002,0x7c86bb7d),L_(0xcadc14a3)}, {LL(0xf7d0edab,0xbef7e6fa),LL(0xbff62993,0xd5068119),LL(0x0a2850cc,0xc5820cdd),L_(0x92bdfe5c)}},
   {{LL(0x8c84d27f,0x484daaee),LL(0xd799a2f4,0x612b63b1),LL(0x550daad1,0xef0ac41a),L_(0x8fbf8f4e)}, {LL(0x377207e9,0x84f7e136),LL(0x7bdb6e8f,0xfd5152ed),LL(0xd89df99b,0x395821cc),L_(0x20d76719)}},
   {{LL(0x3fd703c0,0xae387d38),LL(0x760e59a1,0x28b52395),LL(0x97e8f74f,0xbc044138),L_(0xa21350e4)}, {LL(0x86d227b5,0x02533549),LL(0xe3e6583e,0x2b3bb618),LL(0xc654b122,0xa829cd8c),L_(0x39c79315)}},
   {{LL(0xe57a78bc,0x040ffaef),LL(0xb7bea5b5,0x0bea9607),LL(0xdb5241af,0x372d1796),L_(0x940a8a94)}, {LL(0x4374d692,0xf1499afd),LL(0xcabac4a3,0x4a7889cd),LL(0xb861df89,0x0cb6a234),L_(0xa96fe9bb)}},
   {{LL(0x7682c515,0x5eedd184),LL(0x4434f8b6,0x5b6b1177),LL(0x2a8d7505,0x86f300e0),L_(0x0e35d724)}, {LL(0x5d047332,0xfaf29ff0),LL(0x4c814382,0xea27ade1),LL(0x141d9196,0xd1bc5f13),L_(0xade4deb9)}},
   {{LL(0x5fdcb5d6,0x406ace5b),LL(0x1ea243b0,0x088fb509),LL(0xd6070207,0x22934b39),L_(0xd9cd49e0)}, {LL(0x0d4662d6,0x4d0f3174),LL(0xc755b33c,0x4eef4ac5),LL(0xf8e1a986,0xc50467c8),L_(0x18a3ef6a)}},
   {{LL(0x747a7c15,0x4607dcec),LL(0x614bcea6,0x5f5ba78f),LL(0x68ae9194,0xf3906a4b),L_(0x42d006dc)}, {LL(0xde1842dc,0x4cb64b0d),LL(0x9dbec629,0xba893ced),LL(0x0bf6a2f2,0x057751b4),L_(0xa9fae60b)}},
   {{LL(0x6b278395,0x3c12fc74),LL(0x017d14d0,0xc71b3f43),LL(0xc6c97422,0xa51b04ab),L_(0x626c81c6)}, {LL(0x3b8a9b8d,0x238bcf03),LL(0x5e266a2a,0x6ca507ab),LL(0x23764003,0x51803104),L_(0x5f7162bf)}},
   {{LL(0xfae6aeea,0xa5dd759e),LL(0x522db1a2,0x5e4a05cb),LL(0x09f2baed,0xf85a6ac1),L_(0x323304d4)}, {LL(0x5e98632e,0x1360791b),LL(0x1db93b31,0xb1ea2fc8),LL(0x1b768230,0xabf3fed8),L_(0x87d85b61)}},
   {{LL(0x79a68d97,0x761f1b7c),LL(0x0b49a02f,0x3ccdde40),LL(0xbb854598,0x77637e3e),L_(0x0022290f)}, {LL(0xdfc1715a,0xf0700420),LL(0xe0410aa3,0x4a5a58b2),LL(0x6d34be85,0xf6bc18c0),L_(0x46cdba65)}},
   {{LL(0xe6b65986,0xb9773b59),LL(0xc5228020,0xd35b5bb3),LL(0x453c7ac2,0xf11276b9),L_(0x7df2cd47)}, {LL(0xcbee2352,0xf5a8c36a),LL(0x179c274d,0xba21cd5c),LL(0xa5c5bb12,0x11df901f),L_(0xe17d00c3)}},
   {{LL(0xb8d6f9d0,0x3ed4c79a),LL(0x8883a1db,0x60bbaf97),LL(0xfdd5776e,0x2f54ff27),L_(0xcb776738)}, {LL(0x589db69e,0x9a98b106),LL(0x79c8cc7c,0x9590d453),LL(0xdc2c596a,0xc7ee141c),L_(0x6b47161b)}},
   {{LL(0xb554abc5,0x543685f9),LL(0xc2395393,0x329ac98a),LL(0x9a0b36b0,0xaae1a4bd),L_(0xd5d4e38e)}, {LL(0x0637ec30,0xc2e17c96),LL(0xd7e5c5cf,0x54bb5d81),LL(0x2bef0600,0x9ed78a64),L_(0x561fb810)}},
   {{LL(0x4b821219,0x2fe8fb71),LL(0x1a1e9b3b,0x4feedce1),LL(0x3925b0ea,0xbfb2449b),L_(0xb054dfbe)}, {LL(0x29864158,0x28647223),LL(0xa5055940,0xe2ec01b2),LL(0x31e1b9a5,0x23b5a4c4),L_(0x195035c8)}},
   {{LL(0xbdf1c841,0xcc50c48c),LL(0x209f3af8,0x1489a642),LL(0x181662d4,0x4b465365),L_(0x91296676)}, {LL(0x9e845d49,0x138dff73),LL(0xed5f5189,0xe47473ab),LL(0xc0ded518,0x40df6431),L_(0x723fb4ad)}},
   {{LL(0x38970226,0x112db661),LL(0x3569136f,0x7a1d7c16),LL(0xb8e53775,0xdd45fe78),L_(0xfaa04e8d)}, {LL(0xb657c7ea,0xf6371e53),LL(0x86806849,0xf29213b2),LL(0xf4779601,0x4c858ecd),L_(0xa4992880)}},
   {{LL(0x77e1f87e,0x89673149),LL(0x156e1198,0x03cb4796),LL(0x39eb8237,0xecde7783),L_(0x6d9c66e5)}, {LL(0xe3363ded,0xd2f38b47),LL(0x3fba7225,0x0a4d15cb),LL(0x088944d2,0x4f8035c2),L_(0x33f1f404)}},
   {{LL(0xa1983b54,0x51cb66d5),LL(0x58d0680a,0xaf231cb5),LL(0xcb3e58aa,0x1d2325d3),L_(0x13427358)}, {LL(0x6d4b8ce2,0x7dbb0bf1),LL(0x85f983bc,0x3bec3e74),LL(0x0f88839b,0x8a991a31),L_(0x6b3985b4)}},
   {{LL(0xf774901f,0xdf68b707),LL(0xde5b911a,0x0c634df4),LL(0xeed92206,0xfc74f9a9),L_(0xbcecf38d)}, {LL(0x3a74e355,0x96e739bf),LL(0x0a6c0e27,0x9a315c68),LL(0x82400ede,0xbba8f32c),L_(0x27d5bb2b)}},
   {{LL(0xa27fb54f,0x3131df10),LL(0x042f1a03,0xac9267a7),LL(0xe003b634,0xa22e7895),L_(0x6dd1e8b0)}, {LL(0x4591bb21,0x90518b2b),LL(0x5efe4007,0x9c565a5c),LL(0x88b87e5d,0x63c085ee),L_(0xf6155e0f)}},
   {{LL(0x9a0ef584,0x841e4add),LL(0x1f02ab50,0x7b5e812b),LL(0x67a50be5,0xe6f206a7),L_(0xf1e42056)}, {LL(0x737ad369,0x2febb4c6),LL(0x657a0dc7,0x93b35fb8),LL(0xbbf3484b,0xea4f285b),L_(0x020b154e)}},
   {{LL(0xa95bc0c4,0xdbbe1817),LL(0x91e4e9f4,0x8583001a),LL(0xbf43b7fd,0xe4667773),L_(0x52865635)}, {LL(0xec4299f7,0x5962ba86),LL(0x28fe8ab8,0x3849c7d1),LL(0xd449338b,0xd636805c),L_(0xa1efd1c9)}},
   {{LL(0xa552a401,0xdfd448aa),LL(0xb92e258d,0x3f8fa817),LL(0x35b6d45c,0xa42ce187),L_(0x0d195771)}, {LL(0x0ca7ae66,0x672eaebb),LL(0x60feb26e,0x34d002fc),LL(0x94989870,0xb3b0ede4),L_(0xc41d2caa)}},
   {{LL(0x75191aa5,0x5261f5a6),LL(0x330e8a48,0xce0728c4),LL(0xf61c72d3,0x33ca48fd),L_(0xffeebdb0)}, {LL(0x7973a221,0x7ae9478e),LL(0x73383a11,0x2d6cf157),LL(0xafb5d296,0x689cfdb3),L_(0x32706d23)}},
   {{LL(0x5b0f9106,0x0bd73e3b),LL(0x62c3a48b,0x2c31c735),LL(0x0c0a3d1e,0xacf48d35),L_(0x48c97582)}, {LL(0x9aa2cb41,0x01b5bb28),LL(0xaf780b08,0xd14e0836),LL(0x2a0a2d75,0xabcaf322),L_(0x13f7f83f)}},
   {{LL(0xf123c708,0xb992a79f),LL(0x19a0cf96,0xada9dff4),LL(0x98adc9dd,0x6a4d3e6c),L_(0x40f8faf2)}, {LL(0x676dfe03,0xef011d37),LL(0x7e67e105,0xb029aa6b),LL(0x87eb4d53,0x45add6b3),L_(0xce77f195)}},
   {{LL(0xfdc3ee1e,0x3d9993bb),LL(0x8adaa277,0x9c19b8ff),LL(0x1fca3093,0x7433657e),L_(0x8caf0d86)}, {LL(0xcbbc468c,0x85112380),LL(0x3dbd4315,0xa4c53f5c),LL(0x9d410bfb,0x374f5e1e),L_(0xf2fcce49)}},
   {{LL(0x058ab78a,0x532b7587),LL(0xd3243b22,0xf94c79ad),LL(0x11c1365e,0xb045d4a7),L_(0xb70776bc)}, {LL(0x35abd993,0xadb798da),LL(0xe29a9f85,0x449472ae),LL(0x04e83977,0x4ef414c2),L_(0xd94bbca4)}},
   {{LL(0xe9445da7,0x69a5c563),LL(0x189faa10,0x013c6f09),LL(0x11ff4092,0x270bddfa),L_(0x13f400ef)}, {LL(0xab739108,0x54c1f415),LL(0x5467f97f,0xb88a9ecf),LL(0x8a19f9f7,0xd713b7c7),L_(0xbecb24c4)}},
   {{LL(0xa9451b5f,0xd9e41b7c),LL(0x154bb407,0xb69c99b7),LL(0x9e3d7f55,0x0a7b54c7),L_(0x135359b7)}, {LL(0xd2d3aed3,0x84d9fef7),LL(0x5d18cdb1,0x61c13355),LL(0x142d7261,0x1cba3904),L_(0xac50bd6c)}},
   {{LL(0xb8b1ee7a,0x709ebb7d),LL(0xec49c916,0xd7ba31a4),LL(0x16f1cdf7,0x9b4ca4e1),L_(0xd65f4a10)}, {LL(0xad080359,0xf98d2d14),LL(0x03b182b1,0xe7c36b4e),LL(0x060b7206,0xb11e15e8),L_(0x6d96c56b)}},
   {{LL(0xf8d054a3,0x8f77e5e4),LL(0x72cd83b6,0x93dad056),LL(0x25a5b5bf,0x25a2e699),L_(0x6fdd7673)}, {LL(0x0063d830,0x90a6d4fe),LL(0xd576b9f6,0x3c44f207),LL(0xbcb25092,0xad2b4418),L_(0xefb79bc4)}},
   {{LL(0x01e94264,0xfba463c6),LL(0x67359a81,0xfc726fd8),LL(0x2c69efd6,0x280e7df6),L_(0x74569519)}, {LL(0xa6c619bb,0xe2c76093),LL(0xb18f6667,0x8f9466f6),LL(0xbd326eec,0x88831553),L_(0x87869f49)}},
   {{LL(0x88c46f2f,0xbaa57c7c),LL(0x3b207add,0x7299422c),LL(0x4ed0514a,0xb373586a),L_(0x309bffcc)}, {LL(0xc5186b1b,0x461b005b),LL(0xc0f53c53,0xbbee060f),LL(0xc558624a,0xcda6e3b1),L_(0x6faa11b6)}},
   {{LL(0x1f345075,0xf0680ab2),LL(0xcbfa17ef,0xe657cfab),LL(0xcddb5b82,0x41c78a0d),L_(0x20234c33)}, {LL(0x65c9a0da,0xc2fe9028),LL(0xf4f1f1bc,0x14464393),LL(0xfc36a9bc,0x75517f43),L_(0x7adc3f7c)}},
   {{LL(0xc9f99571,0xba51387a),LL(0x49c36869,0xfba899b3),LL(0x03e373c6,0xb8b4767a),L_(0xb07a3f0c)}, {LL(0x7fc1b90f,0xe85c5039),LL(0x808f702e,0x94d61e9d),LL(0xdddd86d9,0x7011d76f),L_(0xfae1c7e7)}},
   {{LL(0xf98efe50,0x80fa818b),LL(0x31e4ef2b,0xf1aadc09),LL(0x68d478dc,0xb3f48a44),L_(0x7630fe61)}, {LL(0xaa830a4a,0xf7c94ece),LL(0x9a533238,0x04218b55),LL(0x5e7ce696,0x3b49d1b1),L_(0x52a95933)}},
   {{LL(0xc81b286e,0x0d20898e),LL(0x9d1c985c,0x3f1398f8),LL(0xc62ed966,0xce3b7676),L_(0xeb820161)}, {LL(0x4dd1834e,0x60f14618),LL(0xe6d270e0,0x16b9b346),LL(0x80e6594b,0x9e645987),L_(0x879e807c)}},
},
/* digit=27 base_pwr=2^189 */
{
   {{LL(0xe215ad69,0x0703230b),LL(0x9c23cc7f,0xcb4845fc),LL(0xa6d601b3,0x3f2e4ff0),L_(0x8b1b5936)}, {LL(0xd006b554,0xe0882f46),LL(0x402ec9ab,0xd39b85a4),LL(0x47de2c7e,0x555b4d4e),L_(0xeaa5bf4d)}},
   {{LL(0xc42ab049,0xf78b555a),LL(0x971ade84,0x9092aa3c),LL(0x4e8a2ef9,0x5ad4ea73),L_(0x0425d602)}, {LL(0x2b7e58ba,0x791d9122),LL(0x227d0459,0xf3147395),LL(0xc9d765a1,0xe4f1346c),L_(0x28b65397)}},
   {{LL(0xceec40d6,0xe7d16970),LL(0x871eef82,0x361a2299),LL(0xb5fdd432,0xb27a33c5),L_(0x1d066c2a)}, {LL(0x1d4d340e,0xe97230fd),LL(0x862b60db,0x5852f3b4),LL(0x593d4118,0x539f6530),L_(0x8436da66)}},
   {{LL(0xe5f6d86b,0x2425af9e),LL(0xead58c17,0xe4dac347),LL(0xb5cbb50e,0xccdd8dd9),L_(0x86b3650e)}, {LL(0x95382a07,0xdc320634),LL(0xa3e665cd,0xe9848376),LL(0x1e2d319e,0x33fae415),L_(0xeca579c1)}},
   {{LL(0x372307a3,0x6a9d1fc4),LL(0x89b65031,0x50fbf109),LL(0x23a987e1,0xfde88e7a),L_(0x2dd37a50)}, {LL(0x2947bdb3,0xcebc549b),LL(0x820f3f12,0xab66e245),LL(0xb7bcaa87,0x694bc71e),L_(0x8e6e09c1)}},
   {{LL(0xc6572555,0xfe8a98ef),LL(0x63ff767a,0x71371393),LL(0xc5f7f4d1,0x0e5335e1),L_(0x7cfba5a2)}, {LL(0x96ccd504,0x77da1976),LL(0x67e59eaa,0x16544c87),LL(0x504acfd5,0x96be881c),L_(0xbeef988a)}},
   {{LL(0x3faff788,0x82325976),LL(0x575e852e,0x05c5a091),LL(0x2c72e2f2,0xe0cc66d6),L_(0x1927e387)}, {LL(0xb872a6ad,0x4d0b0121),LL(0x7da1cf59,0x5bcfc963),LL(0x26204672,0xdb25f5a7),L_(0x0a8b1258)}},
   {{LL(0xe8b35d2e,0x7d69f703),LL(0x20d3988b,0xc271bab1),LL(0x6c24b0c1,0xaab259f9),L_(0x5e65b843)}, {LL(0x51fcf150,0x75e5c2e1),LL(0xea29f462,0x8993613b),LL(0x1834a191,0x8c9eae94),L_(0xf967c0ea)}},
   {{LL(0x41292928,0x7e9bdb88),LL(0x00adf08d,0xc2ca609c),LL(0xa0138acf,0x8dfb7875),L_(0xd93b37ee)}, {LL(0x3666deb2,0x749de945),LL(0xbfd574b2,0xe95a61ae),LL(0xe75abd5a,0x67c65b64),L_(0x13366a27)}},
   {{LL(0x21088150,0xb08be400),LL(0xe06fcef2,0x86d73e23),LL(0xe0ba359f,0x607c14ca),L_(0x382ff71f)}, {LL(0xe5594024,0x6404f03c),LL(0x3f8a49f7,0x148ec4e9),LL(0x7853702e,0x4887d191),L_(0x04d3f5e6)}},
   {{LL(0xd2a149ac,0x67857251),LL(0xdb805369,0x8e8ea485),LL(0xd63c66e4,0xa73ff057),L_(0x587ee86f)}, {LL(0xfb2b3fde,0xa10a6759),LL(0xbd927cb1,0xfc748cd7),LL(0xc799e97c,0x6f65ee19),L_(0x4703e5a8)}},
   {{LL(0xa25a7805,0xaa2f24f7),LL(0xba32a728,0x7fcf37bf),LL(0xb7ad366d,0xa0ac6a50),L_(0xb220a09e)}, {LL(0xcb955e95,0xd5db1087),LL(0x44a3150c,0x65f007ad),LL(0x09784fd1,0xf6f05498),L_(0xc1e6c0b0)}},
   {{LL(0xc19f511e,0x06b7160e),LL(0xbb692e48,0xc84c111e),LL(0x15498ecf,0xb77b124d),L_(0xbb092f5d)}, {LL(0xc61f57ba,0x74e3ca97),LL(0x7b0969cb,0xd5839144),LL(0xf86a2cf8,0x6d59ed66),L_(0xdeabc883)}},
   {{LL(0x9f404ee0,0x17174a2e),LL(0x852eae73,0x1bb80f22),LL(0x1e348f1a,0x0374bff3),L_(0xeacc5568)}, {LL(0x8d863212,0x5585a76c),LL(0xa921615e,0x66b5bc71),LL(0x1b74c1fd,0x26867d50),L_(0x52fb37fb)}},
   {{LL(0xd2f49002,0xb96496b7),LL(0xe996dbee,0xa5746d1f),LL(0x39670ef3,0x8094e471),L_(0x9c159e05)}, {LL(0x286d79a8,0x4226465a),LL(0xb7fd505e,0xbf1aca6c),LL(0xf2355f67,0xe46e77c4),L_(0xe0fbfc8b)}},
   {{LL(0xea2be4c9,0xc386d032),LL(0x16b11ec1,0x828f0d4f),LL(0x75da11d8,0x94c7a19f),L_(0xfda19041)}, {LL(0xd7f40cb4,0xcd2bfd30),LL(0x0e5ba2ee,0xb240b8db),LL(0x2339d87a,0x375d7766),L_(0x24af1974)}},
   {{LL(0xd5cb473f,0x0c7c5c0e),LL(0x06c1fc0c,0xb684879d),LL(0x1d94e717,0xb65c6a50),L_(0x0b30821a)}, {LL(0xd7821eb5,0xefa6a39d),LL(0x6459b552,0x727bdd40),LL(0xb7d7e4b6,0x18598afa),L_(0x00b319ed)}},
   {{LL(0x703608d7,0xf884841c),LL(0x8dad7679,0xe15515c9),LL(0xb306db5d,0xb36c0754),L_(0x5229df9a)}, {LL(0x979aad60,0xaadea0d7),LL(0x970bb22c,0x6c1b7c90),LL(0x298689dc,0xf3b57c01),L_(0x89fe17ad)}},
   {{LL(0x428e4fe4,0x76218fd4),LL(0xceff7d9d,0x975a4605),LL(0x63a383cf,0x238b4e91),L_(0x99f29b16)}, {LL(0xccf0edbc,0xe11ea485),LL(0x7ae47927,0x93601402),LL(0x7cd7a8d0,0x0d9f157a),L_(0xb3fa2b87)}},
   {{LL(0x9e4d4311,0xff049f88),LL(0x85fe8244,0xcf8e1ceb),LL(0x5d2ff231,0xef0f5b1a),L_(0x63ccdc00)}, {LL(0x959a2ad3,0x8ba95ec0),LL(0xa8a44260,0xbd876759),LL(0x6ed75cfa,0x3e43af21),L_(0x8464e573)}},
   {{LL(0x444c26f1,0x65d29b4c),LL(0x1583a353,0xee5f7609),LL(0x43193114,0x92998e59),L_(0x704abb0e)}, {LL(0x99be8a63,0x2a7e16f6),LL(0x4c98adbf,0xe1fc70da),LL(0xa5a25fbe,0x44dc73bb),L_(0x8fc488f7)}},
   {{LL(0xa3bfed42,0x784f9377),LL(0xe7bcd034,0x07fc5bf3),LL(0x89f2f24f,0xbec66d27),L_(0x4917e013)}, {LL(0x3a3c1ba3,0xf0cde4d7),LL(0x0961875a,0x69bd4eb3),LL(0xa260b4df,0x59faeb8c),L_(0xbeeb1631)}},
   {{LL(0x21f06827,0x280f3962),LL(0xa088a1e1,0xcf16263e),LL(0x1dbee473,0x1692a131),L_(0x05a158ea)}, {LL(0x283c279b,0x0a11b121),LL(0x0e13e263,0x8ed3889a),LL(0xdee544e9,0x5dcaae12),L_(0x8f1fbb34)}},
   {{LL(0x87dc2c5f,0x5d2e90ad),LL(0xb9ad99ac,0xed6943f8),LL(0x0c329529,0xaa459865),L_(0x96ffc387)}, {LL(0x34567b61,0x13516c2e),LL(0xa0aa883b,0x39122bf4),LL(0xbfa72fcc,0xa825f1f8),L_(0x97da099b)}},
   {{LL(0x67b5c14e,0x14bced82),LL(0xf4fdf535,0xff66cf51),LL(0x689f6f0d,0x71814baa),L_(0x5b8ff385)}, {LL(0x247c8262,0x9c1ba7ce),LL(0x907e4e69,0x4b578253),LL(0xff0ce18c,0x2458cb25),L_(0xef5cd0aa)}},
   {{LL(0xb909db13,0xfee8b8ef),LL(0xa4710d21,0x1107af8a),LL(0x54895b98,0xb541a6b0),L_(0x526b0ca4)}, {LL(0xa7097ce7,0x16947322),LL(0x31b294e5,0x29af3ae4),LL(0x1b4cb2b5,0x98687802),L_(0xfbcc3509)}},
   {{LL(0x1586105f,0xb150cf07),LL(0xe16b93aa,0xa4bbc36a),LL(0x253eb0dd,0x984df68e),L_(0x87118c04)}, {LL(0xcb079f80,0x480076ec),LL(0x0dcde0c7,0x6b489d1e),LL(0x18bdffd8,0x16aa0174),L_(0x1f265d34)}},
   {{LL(0x59614a61,0xee0e75bd),LL(0x5617d582,0x52348cb9),LL(0x3e671da9,0x0190d33a),L_(0x3bee1eff)}, {LL(0xaa01d03e,0xe617dbcb),LL(0x09cde5ef,0x84c57057),LL(0x888eade7,0xfc85f3d6),L_(0x359daa39)}},
   {{LL(0xd48bf3db,0xb342c647),LL(0x4a00b224,0x40b2d73f),LL(0xb894175c,0x58a94e88),L_(0x70cf2206)}, {LL(0xb4cd9a9f,0xf20b5560),LL(0xdca48b9e,0x1a01b8da),LL(0x7707c57e,0xa0a1a133),L_(0xc33e6b03)}},
   {{LL(0x727e08d2,0x9c8970a1),LL(0x4c16f18a,0xf1b1124a),LL(0x57b552ca,0xda495e0b),L_(0x1a9ee7da)}, {LL(0xe079a731,0x1ad4dac4),LL(0x06f7634a,0xf219d184),LL(0x1b77c61d,0x110655eb),L_(0xd5bf028c)}},
   {{LL(0xf20ff378,0x21c59244),LL(0x51b05ff6,0x7d017e8f),LL(0xaf13b9eb,0x2b5c0017),L_(0x4aa9bdff)}, {LL(0x2371404f,0x703d8718),LL(0xaf5e4830,0x1cdec0ba),LL(0x52cc936a,0xf59d1925),L_(0x013abb12)}},
   {{LL(0x0e22d6ca,0x1082f990),LL(0x2c9f0b9e,0xf4898bd8),LL(0x666e06e5,0x703ab683),L_(0x6f1b6308)}, {LL(0x2a6a3005,0xc8b722db),LL(0xd103845d,0x3ef671a0),LL(0x2459c28d,0xd0ce016b),L_(0x60884f40)}},
   {{LL(0x59320b15,0x7f7ba17f),LL(0xe194bda2,0x52a3af1a),LL(0xbb90a873,0x2977890c),L_(0x3dc3fa11)}, {LL(0x13cb3938,0x1ec1818b),LL(0x2f976a14,0x813e2d47),LL(0x6ef670b1,0x547bbc14),L_(0x7a6392b3)}},
   {{LL(0xbee8917d,0xfa44c4a2),LL(0xd1652cf8,0x11ddbdcd),LL(0x7b98c98d,0xeeaab800),L_(0xb60ccb53)}, {LL(0x699cc945,0x20bc5779),LL(0x3d3d111a,0x5d171d19),LL(0xf1c69a57,0x4586a0f5),L_(0x047748d6)}},
   {{LL(0xcb2dd3a2,0xbfb2046d),LL(0x1ba40992,0x4ab9b7d4),LL(0x492e1976,0x8fa3d075),L_(0xb62a0315)}, {LL(0x9faa1011,0x08bcfaa8),LL(0x55ee388d,0xa8201ba0),LL(0xb50f4e76,0x48d16444),L_(0x947b5775)}},
   {{LL(0x7bb21f8c,0x27f91139),LL(0x9933e071,0x7328102f),LL(0x16712a5a,0xc739c006),L_(0x75d0d886)}, {LL(0x0d97d602,0x95eed6e2),LL(0x5fd2d047,0x4657933f),LL(0xfa45dd80,0x9d54f1fb),L_(0x741b92d0)}},
   {{LL(0x4b499dd3,0xe5114e38),LL(0xc2275c0e,0xc9aa88b0),LL(0xb4d5582b,0x3e016601),L_(0x25741fdf)}, {LL(0xe08edbac,0x4514d2eb),LL(0xc4ccb66a,0x16e96390),LL(0xb57016cb,0x59b84d70),L_(0xf83fe7e5)}},
   {{LL(0x697291df,0x7a6974e3),LL(0xcecae61e,0x747d2ddd),LL(0xa93265cf,0x1f1fb9f1),L_(0xf3c9e95d)}, {LL(0x05b8ad90,0x76333190),LL(0x5ef96caf,0x2e7e8b35),LL(0x26204dea,0x66ec9969),L_(0x8ee499c3)}},
   {{LL(0x7d3e7f82,0x1abe8df3),LL(0x0e4c49fd,0xf5621bfb),LL(0x5430fd38,0x5d968105),L_(0xb2efc301)}, {LL(0x7fa73503,0xde3d9216),LL(0x191e5b5e,0x358512db),LL(0x32722378,0xa0892090),L_(0xd92f1352)}},
   {{LL(0xfec9743d,0x757c94ef),LL(0x9649e576,0xf290f932),LL(0x79587916,0x8be7ad8d),L_(0xadd01304)}, {LL(0x61f6de8f,0x29445ca0),LL(0x285b7eef,0x1426c1d9),LL(0xae55ea09,0x962f7357),L_(0x983ad1c5)}},
   {{LL(0xdfa17534,0x40d7b215),LL(0x3897f22b,0x4a689975),LL(0x0a2968a8,0xae9ce773),L_(0x5eeabf53)}, {LL(0xb2a472ce,0xd7707d10),LL(0x3f8692b7,0x11cc24bf),LL(0x416e3f6c,0x0e77bce9),L_(0xf53b376e)}},
   {{LL(0xe52b9bd7,0x211044a2),LL(0xb9dbcba3,0x32202391),LL(0x7e52000c,0x2961ea4a),L_(0x12e1a413)}, {LL(0xae4f775c,0xc87793ae),LL(0x36916c16,0xadc59df4),LL(0x93f3db96,0xf4f9dbfb),L_(0x2bfda0b4)}},
   {{LL(0x04c1afdd,0xbbbbeb9c),LL(0xcc1030ad,0x0de0dfaa),LL(0x49bedb1d,0x1869975f),L_(0x6a004fbc)}, {LL(0x0b0e188c,0x51c16437),LL(0x71e53119,0x3e18f624),LL(0x5e139c48,0x60f866a3),L_(0x0784d6e0)}},
   {{LL(0xc173ecc1,0x97f340e7),LL(0xa61d043f,0x444c1c32),LL(0x4c35a67a,0x8581b05b),L_(0x44182222)}, {LL(0x731fc41e,0x45bca0e4),LL(0x8bf96ead,0x853c2a7b),LL(0x7c86a091,0x2ad3d962),L_(0x7ea985d4)}},
   {{LL(0x5d1d3e02,0x1ef28f3e),LL(0xcbf512f8,0x72330e86),LL(0xdc8ed43f,0x516f1183),L_(0xec32dc2b)}, {LL(0xd7b96122,0xc45e5ada),LL(0x505156da,0xdf67ff42),LL(0x983acb8c,0x30958d17),L_(0xe6ec6def)}},
   {{LL(0x569438de,0x1072cfdf),LL(0xaaf2a64f,0xda01f192),LL(0x33454609,0x8b880e7d),L_(0xa7a46a05)}, {LL(0x35cc6b77,0x7354fe07),LL(0xd5821805,0xa74b9400),LL(0x0fc1cbb1,0x0e58985d),L_(0x9f7a9dad)}},
   {{LL(0x84ac106e,0x9dc30814),LL(0xe1ebef86,0xcd23c165),LL(0x852da87b,0xed5cdd01),L_(0x21bc828d)}, {LL(0xce6842d7,0x5aadc859),LL(0xe522baaa,0xe89ad998),LL(0x608b4b16,0xee466b6e),L_(0x0c33a24a)}},
   {{LL(0x8243d368,0x4e1fae12),LL(0xf397b191,0xe6b7d57c),LL(0x2940f1c8,0x47dd818f),L_(0x68c77c89)}, {LL(0xe08f2ebf,0xba40969e),LL(0x8ffaa799,0x102a6172),LL(0x712df945,0x997feef1),L_(0x4da4261d)}},
   {{LL(0x455b0c0c,0x1b0afa51),LL(0x69fea568,0xf8ba1286),LL(0xdfc334a6,0xa2587e32),L_(0xf61b674f)}, {LL(0x060208cb,0x62eb8d23),LL(0xdd643088,0xa1a3f3bd),LL(0x5cde0036,0x53b8bdc0),L_(0x1c5ce488)}},
   {{LL(0xb0c1bb5a,0xa64a20a5),LL(0xd22b7f88,0xf22948de),LL(0x0628283c,0x75129c40),L_(0x6efeb06e)}, {LL(0x4000d1fd,0x76f8c148),LL(0x181e8b58,0x7b823883),LL(0xc758648b,0x3d1fbafb),L_(0x069f31f1)}},
   {{LL(0x2b15bfdf,0x78f3872c),LL(0xbba94fb7,0x49778942),LL(0x852ef997,0x48ddef24),L_(0x066ad60c)}, {LL(0x38c99352,0xc1ba735f),LL(0x790d395b,0x30382b1c),LL(0x7616ea81,0x61b408b8),L_(0x9278e0eb)}},
   {{LL(0xc0f1bdbb,0x15c9a84f),LL(0xae384504,0x1419e604),LL(0xf4814e2b,0x1b8b0e26),L_(0x2b97a1b4)}, {LL(0xf8656f5b,0x260ef3d0),LL(0x5db22d7e,0xc76c3691),LL(0x2e7c6bf8,0x7c1471d1),L_(0x7bdec3b1)}},
   {{LL(0xa27bd94f,0xb71fd22e),LL(0x7f17f289,0xbbacb216),LL(0xd4564a47,0x34b284d4),L_(0xb75a3120)}, {LL(0x09622583,0x986c8a28),LL(0x197cc0c1,0x208425a0),LL(0xcc3ae593,0x7da4599c),L_(0x54ebd08e)}},
   {{LL(0x2d21c888,0x9addab68),LL(0xbd0c6b5c,0xcad27cf0),LL(0x7ec99646,0xaa918115),L_(0x8ee2e6eb)}, {LL(0xea74e73d,0xd9db5cd3),LL(0x2d90dfec,0x719d6c8b),LL(0xf48ec5bb,0x7732d381),L_(0x2daa3783)}},
   {{LL(0xfe67cafa,0x4087fb31),LL(0x64d12677,0xa7b372c6),LL(0xd6d5d34f,0x5aa57c66),L_(0xeebb82c5)}, {LL(0x03f99f8c,0x0b3a848f),LL(0x4bcc334e,0x3076a517),LL(0xafc73b46,0xbdc15e76),L_(0xba3353c1)}},
   {{LL(0x666e86e2,0x3f49b089),LL(0xe4f392d0,0xb96b8ccf),LL(0xc0bd57d7,0x4a84d25e),L_(0x8b9c3a47)}, {LL(0xdd447511,0x88262b51),LL(0x5d6233aa,0xf3ee05e8),LL(0x1da165fe,0xba39a0b7),L_(0x56569118)}},
   {{LL(0x4b2743e8,0x55aff6c3),LL(0xffb26779,0x0356100f),LL(0x581f3f38,0xc259ed17),L_(0x0527ce42)}, {LL(0x9f2d81ee,0xeb967185),LL(0x592962f6,0x58f98402),LL(0xbc866b82,0x9a5ee9a2),L_(0xb8fcf82b)}},
   {{LL(0x50d6a742,0x7b7d67a1),LL(0x6bed8eca,0x0cc91aac),LL(0x74e18696,0x3ac5ac85),L_(0x4777aaae)}, {LL(0xfcd5973d,0xaf83cba8),LL(0x114e541f,0x44df12cc),LL(0x49c90e0c,0x6ffe97ac),L_(0xd22caea2)}},
   {{LL(0xf6ec95c0,0xad0a7c5c),LL(0xcf68be2d,0xef894370),LL(0x46042df0,0x44cc34c3),L_(0x83baaad0)}, {LL(0xda2df354,0xe5be5f7b),LL(0x232306d2,0x09c99b41),LL(0x30e3711b,0x7b102ef8),L_(0x90ca6b87)}},
   {{LL(0x9041d06a,0x24f85c28),LL(0x1c82027d,0xc05d59fb),LL(0xc82c4968,0xf294dde1),L_(0xa0dabc8e)}, {LL(0xc334c99a,0xeadad609),LL(0xa2d74908,0x47f2b704),LL(0xea64f015,0x3b0536e3),L_(0x948d4413)}},
   {{LL(0x90fcfab4,0x4cd8e163),LL(0x7e58fb10,0x9bc6ae53),LL(0x3a2788f9,0xc860f6b9),L_(0x673a28d4)}, {LL(0x95e581f4,0xc5e46853),LL(0x62c5a4e9,0x6d07718f),LL(0x059005a0,0x5a617a3d),L_(0x06267ff2)}},
   {{LL(0xe3e68d85,0x2af744e2),LL(0xae8162d5,0x16487042),LL(0xbe28dd7a,0x950dd089),L_(0xcd76243e)}, {LL(0x6a083312,0x887a306e),LL(0xc5cf5f6e,0x963509bb),LL(0x6076457e,0xd58cb75e),L_(0xa5a9063c)}},
   {{LL(0xe722f017,0x6f63cd75),LL(0xa7571852,0x216d76b1),LL(0x2ab669d0,0xd42dd087),L_(0xee6f2f6d)}, {LL(0x097743cb,0x0a7d1e20),LL(0xfa5d28f5,0x21dad6f5),LL(0x9441fb90,0x8466e082),L_(0x4a673507)}},
   {{LL(0x55c0ab69,0x980ae610),LL(0xc6222c7e,0x5c17d9ff),LL(0x3030f359,0x9b8f42e0),L_(0xbbf50262)}, {LL(0xd0aba7f0,0x469ee64e),LL(0x3ef5e0ea,0x45ce7ede),LL(0x0e1c1519,0xb57bce91),L_(0x806bb655)}},
},
/* digit=28 base_pwr=2^196 */
{
   {{LL(0x18e92c63,0x4760716e),LL(0x789ae13c,0x03cdfbf4),LL(0xf578d1e6,0xc6470ce9),L_(0xd42e2969)}, {LL(0x4db766a1,0x5f1cf230),LL(0x733456c6,0x1237a68d),LL(0x592bb19a,0x91e21ef9),L_(0x8814ad36)}},
   {{LL(0x91a052d2,0xbefb4b7b),LL(0x6eb484b6,0xf5d7b19f),LL(0xa01ef7f5,0xdc02e33e),L_(0xf8ac22f1)}, {LL(0x1ee85810,0x0cbad014),LL(0xc2d60cca,0x0ecada19),LL(0x78a4abe0,0x375f545c),L_(0xbb2518b8)}},
   {{LL(0xe924bce1,0xe94cf6cf),LL(0x721159cc,0x3c03f7b7),LL(0xd001dacc,0xba328884),L_(0x6ca68824)}, {LL(0x99214785,0x857c60d0),LL(0x3e1c1a16,0x6d9d01de),LL(0x4e3a8a4a,0xe5100cc6),L_(0x2fadd498)}},
   {{LL(0x21699c26,0x96e332c1),LL(0xba35b5c5,0x3b2527e2),LL(0x32a3227e,0x3d572604),L_(0x3287caa4)}, {LL(0xfa83e20a,0xd3d8aa48),LL(0xd8f05722,0x42872cdc),LL(0x211c6e5e,0x789832b8),L_(0xd7095c24)}},
   {{LL(0x76f2859a,0x8e5e0a08),LL(0xa554b8a7,0x49d7fe17),LL(0x3cfefa32,0xde9b7d08),L_(0xb712685d)}, {LL(0x3f072319,0xd942a6e3),LL(0xc242dae3,0x105e78a0),LL(0x5c2986c1,0xca0a3ad3),L_(0x58984f5a)}},
   {{LL(0x90f303c9,0xbb8f3d66),LL(0x75e4f894,0x9f67adfd),LL(0x597d61ec,0xc5900752),L_(0xbb05ff05)}, {LL(0x608daf57,0x78265562),LL(0x8bb8c320,0x31f9a44f),LL(0x2afed365,0x9604812d),L_(0x2ecd08d1)}},
   {{LL(0x254ed32d,0x4f8511c4),LL(0xf62b6ad5,0x437b9a4f),LL(0xe0ab6bd6,0x9849b3b9),L_(0x437c1eb4)}, {LL(0x99f67517,0x60c66156),LL(0xe2da9ad4,0xe7f23f9d),LL(0xc72b155a,0xe933cc08),L_(0xad7576ea)}},
   {{LL(0x83df61c0,0x019aa18e),LL(0x06661e71,0x159ba381),LL(0x2a662ce2,0xc9069497),L_(0x5ff8cdb7)}, {LL(0x90a3120f,0x22ef5986),LL(0x62a400b7,0x4c18595b),LL(0xb2915c27,0x1fe38da6),L_(0xffba34b0)}},
   {{LL(0xeb9b1c9e,0x6335b063),LL(0x6133838a,0xfc5ad1b9),LL(0x76c52676,0xce767a1c),L_(0x5e465185)}, {LL(0x3794090c,0x93f4d92f),LL(0xce0d54ed,0x25dd3904),LL(0x742e50dc,0xabc94edc),L_(0x78bba80f)}},
   {{LL(0x887eb7e0,0x4b83b6cd),LL(0x81b49586,0x1b46a818),LL(0x1fa505f1,0xae4acd7d),L_(0x832817ef)}, {LL(0xc574766b,0xd6ef910f),LL(0xd59182a2,0x2419996a),LL(0x764ce5bd,0x323e203f),L_(0x63236d14)}},
   {{LL(0xf567e8d7,0x118cb519),LL(0x8ea6b928,0x1c12c8c9),LL(0x56dd68b6,0x395935d8),L_(0xbd116bcd)}, {LL(0xfc1ac8c8,0x5ced4a6b),LL(0xbb4ec3a0,0x5a7b4c09),LL(0x63f4b4af,0x1f24cb4f),L_(0xf9b9d0b1)}},
   {{LL(0x596243e7,0x18a50232),LL(0x25f531db,0xaa27b297),LL(0x8b3f0f16,0xa0701098),L_(0x6a6aef50)}, {LL(0x60fe4fb4,0xfb3a537d),LL(0x8c64961c,0xd4b3fc26),LL(0x83f4d9b6,0x440a2f19),L_(0x7696fcba)}},
   {{LL(0xf34d3bad,0xc0f2402b),LL(0x5360ffc2,0x2a9eb601),LL(0x8d33aed5,0x7da3a904),L_(0x4dde5b44)}, {LL(0x3a35e974,0x6c75adbb),LL(0x0c9938a7,0xd7a77183),LL(0xf4ef9f5d,0x30114b89),L_(0xbb08ab61)}},
   {{LL(0x80255f87,0xdf042ef7),LL(0x481cae35,0x2a918359),LL(0x6fed5580,0x3de7d914),L_(0xa4e1bcfb)}, {LL(0xcd8bcdde,0x85cd2cff),LL(0x4370738c,0x03a1d543),LL(0xf95a95e6,0x9814d14c),L_(0xe0fd8dcb)}},
   {{LL(0xf9193246,0xc27ed927),LL(0x655d6ffb,0xe4ef91cd),LL(0x43a4fc18,0xde7270a7),L_(0xcba9f29e)}, {LL(0xa246b9ef,0x2e139df8),LL(0x5cc92d84,0xc362ff6d),LL(0xc5a596a0,0x6a5bdae7),L_(0xd5e4697c)}},
   {{LL(0xdd86fb5f,0xa1aad671),LL(0xf4cdb11a,0x2ccfca9e),LL(0x35eb1979,0x06b67d2c),L_(0xd3a50661)}, {LL(0xcf9afa3f,0x47a43129),LL(0x2c80dfe0,0x0f28c072),LL(0x1e137541,0x2528a436),L_(0x4a30bc3a)}},
   {{LL(0xb7a122bc,0xa27429d1),LL(0xac029aa3,0x8704523f),LL(0xe555d79c,0xc26117ae),L_(0xeee94c98)}, {LL(0xd57ea228,0xb5866e80),LL(0x3b83d3dc,0x8fda3fab),LL(0x497236f4,0xa896a908),L_(0xc1eba30c)}},
   {{LL(0xa3fc2c9c,0x0b1d2a88),LL(0x6a45073a,0x8af7a77a),LL(0x72b5edce,0xc7a3b1fe),L_(0xac5268b4)}, {LL(0x018d5428,0x8589b539),LL(0x25c0f37d,0x22f0783b),LL(0x8cf1d803,0x9d3fe1d2),L_(0x15093e7c)}},
   {{LL(0x8d1c5128,0xac83f07c),LL(0x6eba86a7,0x4ddb4944),LL(0x7769b4f0,0xce08325a),L_(0x306da9b2)}, {LL(0xe9629e93,0x9fd70869),LL(0xbe70ef68,0xa27b01af),LL(0x9649ac38,0x59634c20),L_(0x260c709f)}},
   {{LL(0x32f4b675,0xe36b9e35),LL(0x56c9f551,0x59ec0560),LL(0x9a19b345,0x7b1c335b),L_(0xb90484f8)}, {LL(0x2e951d15,0x0b5ef9ec),LL(0x8c6e220c,0xac2613e4),LL(0x4ddb2066,0xe5c29249),L_(0x2b898fcc)}},
   {{LL(0x90d06a78,0xb7d4dedc),LL(0x3cc11efc,0xf6a2c79e),LL(0xc5d55aae,0xebad4304),L_(0x139832e2)}, {LL(0x73f6f29a,0x0089eaff),LL(0xb6ab35e5,0xa043d1f1),LL(0x35af18a2,0x363f80f9),L_(0x8c8c5655)}},
   {{LL(0x800e76ab,0x3cad20e2),LL(0xd54cc179,0xd6ee4412),LL(0xc7630400,0x8ece816d),L_(0x4347fa49)}, {LL(0x46a3c678,0xe22f3ed6),LL(0x83d72b28,0x5e7ffb13),LL(0xa89efb1f,0xc07fe6b1),L_(0xf679cfd6)}},
   {{LL(0x1e9baa6c,0xdd4159b6),LL(0x90409ff6,0xc1a8d493),LL(0x0509d403,0x0d946df5),L_(0x46153d5b)}, {LL(0x2b496957,0x1229cd0e),LL(0x25841f04,0xac5550dc),LL(0x163487ec,0x1f5ac4bb),L_(0x7a894b16)}},
   {{LL(0xf830d51c,0xd0fd0fde),LL(0x95e0fb87,0x393b8885),LL(0x12d4a0db,0xbac4ce3b),L_(0x4aecb41e)}, {LL(0x949595c3,0xf6b866b2),LL(0xc85cee8f,0xadac3140),LL(0x929c773e,0x22cf7705),L_(0xf90855df)}},
   {{LL(0x05ee1c57,0xb938eeef),LL(0x054fb615,0x87e5ccbf),LL(0x7d1ce423,0x624c46af),L_(0xc8279b91)}, {LL(0x70e96354,0x6fc664fb),LL(0x957fa9ea,0x0ec18ae0),LL(0x86d71f58,0x8216ab5f),L_(0x4fdcf503)}},
   {{LL(0x6a1895bd,0x99acbf5f),LL(0xea1c792e,0x2f4551f2),LL(0x3d4924d2,0x09c0ca0b),L_(0xb94190da)}, {LL(0xa95f5a84,0xaccafdf8),LL(0x6c1aa007,0x7635c97f),LL(0xe4eccac7,0x07821f33),L_(0x3125c30a)}},
   {{LL(0x9eee4fb3,0x76375e85),LL(0xa2da6002,0x6d9d3924),LL(0x851663d2,0x903300d5),L_(0xe4304412)}, {LL(0x16885ea2,0xf13f93d6),LL(0xac6cc7e1,0x765c7a8f),LL(0xc233fc52,0xfa884feb),L_(0x889b7699)}},
   {{LL(0xea0d9656,0x649b4877),LL(0xb98eb821,0x0e89d083),LL(0xda85057d,0xe8c391ca),L_(0xc53a37ec)}, {LL(0x5ab13638,0x3c241d52),LL(0x1c63bf16,0xe5732bef),LL(0xe6f1e93f,0x882db63a),L_(0xacf7e8f4)}},
   {{LL(0xb0ae8194,0xd18b015c),LL(0xf734d021,0xe38e3807),LL(0x62b04fde,0xea71f882),L_(0xe9f4d402)}, {LL(0xb7452054,0x3d8f7744),LL(0xad9abc71,0x68a3e02d),LL(0x088ad020,0x25dfde15),L_(0x250f4123)}},
   {{LL(0x5076af69,0xe0e96eac),LL(0x46b7097a,0xd8c13c78),LL(0x19ddbba5,0x3ecd0bd0),L_(0x1b6e3a84)}, {LL(0x74eefc3e,0xfc22adbe),LL(0xbae23185,0xad1b79ab),LL(0x09ed4ed6,0x062eb7e7),L_(0xe1ce81e7)}},
   {{LL(0x06e9bfca,0x4649146e),LL(0x91702d2f,0xb30f9778),LL(0xbfdaf3cb,0xcf8f23f1),L_(0xd403a4bb)}, {LL(0x13bbb0cb,0x33bbb504),LL(0x5d48ee37,0x17c9202c),LL(0x79ec8717,0x5a9ca7a8),L_(0x76092de3)}},
   {{LL(0xfe5a3569,0x1f9e6c97),LL(0x17ed7898,0x2727d871),LL(0x668c28ad,0x745aa2ed),L_(0xcf9ce234)}, {LL(0x83f03768,0x4a783728),LL(0xbf5955af,0xadfd1b46),LL(0x92891a4a,0xea43a792),L_(0xc67457a9)}},
   {{LL(0x44cf762b,0xb8dd47fc),LL(0xff42200e,0xe06ffe0d),LL(0xa70a5da7,0x6cbd4092),L_(0x6d058c66)}, {LL(0x0f9eb44c,0x110c3a99),LL(0x67048ffd,0x990449de),LL(0x82fc6b0f,0x30242c32),L_(0xa769318b)}},
   {{LL(0x2521a3f3,0xbb708ced),LL(0x7620c902,0xe32cd4d5),LL(0x75832ff6,0x3a89190f),L_(0x4632df7f)}, {LL(0xc48dd2c2,0x48e4bd13),LL(0x62dc813c,0x4888006f),LL(0x9c55f000,0x36a4457f),L_(0xa330078b)}},
   {{LL(0xb664f023,0xc1f3d1a9),LL(0xe37d1ff2,0xa5e8a776),LL(0xed06b0a5,0xb1d3c33d),L_(0x939c8f4c)}, {LL(0x836de412,0xf53390d9),LL(0x9e6ebae1,0x35e4cf7b),LL(0x28276f39,0xd69e1327),L_(0x19f5bac2)}},
   {{LL(0x7885d34d,0xafd90d23),LL(0x0e84e363,0xbf4cd59c),LL(0xabb80eda,0x07e40274),L_(0x95656580)}, {LL(0x0b980b41,0xa8a767b0),LL(0xc90af857,0x560c2286),LL(0x465bb86e,0xbf658fdd),L_(0x7eb455a7)}},
   {{LL(0x49a8618f,0xda192000),LL(0xad70ec43,0xd0fca141),LL(0xcb55de32,0x7d66c4af),L_(0xa952a38d)}, {LL(0x06ab4e56,0xdceb099e),LL(0xed3e0675,0xa87bfc4d),LL(0x635a862a,0xb9f4c7da),L_(0x8d6f1c2a)}},
   {{LL(0xfd911863,0x6f166fdc),LL(0x0c46e6e4,0x31dda475),LL(0x0813e798,0x9066cff4),L_(0x2f759557)}, {LL(0x65e21d85,0xbd44613d),LL(0x06317041,0x5b7e81b6),LL(0x6879039b,0x6853d2d8),L_(0x80566bad)}},
   {{LL(0xe802ae6d,0xd83655d3),LL(0x7c1a0bb8,0x58e0b41c),LL(0xa5228881,0x00c5e925),L_(0xe9c51c0b)}, {LL(0x0ab1b93f,0x0234b858),LL(0x9a00e742,0xe3d995c9),LL(0xf774358e,0x0eb69043),L_(0xd46eb256)}},
   {{LL(0xe178f2a2,0xe386daab),LL(0x9a305482,0xe1fa3e14),LL(0x7957d9d7,0xf3f52055),L_(0xd1a9ce7b)}, {LL(0xd284eae6,0x303b5ae8),LL(0xac2ad67a,0x7abad7f1),LL(0xb7b58e09,0x7dc5f5f1),L_(0x6c99d250)}},
   {{LL(0xc25aa814,0x29507cf4),LL(0x37c15d8e,0xa6beff47),LL(0x738d9ba6,0xeb76b42c),L_(0x0b91ab81)}, {LL(0x0d166346,0xac1dd9b7),LL(0xb974e300,0x5e6f1484),LL(0xd55b5832,0x9748b154),L_(0xbfcdc886)}},
   {{LL(0x767c2496,0xaa791b18),LL(0xb611f73f,0x6dd24074),LL(0x6fd43d11,0xde586616),L_(0xff2a1da0)}, {LL(0xd73b7439,0x7da57054),LL(0x5051af3a,0x9ca8008f),LL(0xa7d98205,0x80c04759),L_(0xd4db7bec)}},
   {{LL(0xaef4ac4b,0x9051c152),LL(0xd36418fc,0x3e90a332),LL(0x2adcc4ec,0xa2d25b82),L_(0xf9315c6a)}, {LL(0x062c149b,0x6fec476a),LL(0xbdd2e105,0xb614d883),LL(0xdc2bc2a6,0xd473b242),L_(0x9d87dbe8)}},
   {{LL(0xadf54761,0x4031a20d),LL(0x5001f11c,0x908ea1cd),LL(0x86e3fa8d,0xa7c80719),L_(0x36408819)}, {LL(0x4573ac01,0xaf0e8125),LL(0x430d59f8,0xcf05666d),LL(0x5b426c42,0x1c62cb71),L_(0x1a2c3f95)}},
   {{LL(0x840aabc5,0x42998c5b),LL(0x9b1ab1ad,0x41b44d81),LL(0xe7ad335c,0xd2f18816),L_(0x9b21d6a9)}, {LL(0x31ed5509,0x09dc1280),LL(0xc3771787,0x5bacd41e),LL(0x69b13c81,0x5419b128),L_(0x472d8eda)}},
   {{LL(0x47816cca,0xc26aa120),LL(0xb9665455,0xd20f8647),LL(0x3bac4cd2,0x4b96b490),L_(0x15632e30)}, {LL(0x82eca1ec,0x03c2ca5f),LL(0xeac8678b,0x8d2f8677),LL(0x5772568c,0xbef62c1e),L_(0x3b53ce81)}},
   {{LL(0xee48fbfb,0xbe66a12a),LL(0x753cb0aa,0xe362e842),LL(0x38781cdd,0xcf1c41a4),L_(0xab22df0a)}, {LL(0xb208a69c,0x7e4c97e4),LL(0x3869cfcb,0x72ee6ffc),LL(0xb43a4e85,0x6ab97c59),L_(0x35c0f24a)}},
   {{LL(0x98013b12,0x292f0d68),LL(0x3e2b7aaa,0x42f99c9b),LL(0xe2ba5f7b,0x77ff29a4),L_(0xe21d90f9)}, {LL(0xd9c0cb58,0xbc0b01cf),LL(0xe3677335,0x4d244a26),LL(0x8fee4b54,0x2b1a301a),L_(0x5d1511ed)}},
   {{LL(0xa5f95387,0xe5e95789),LL(0xd73285ea,0x392f5392),LL(0x2e506fa2,0xecb1a91e),L_(0x79d54bca)}, {LL(0xbdc25307,0x7bf48edd),LL(0x9f75d7b8,0xe8a12662),LL(0x08014deb,0xaf188a86),L_(0xbfada230)}},
   {{LL(0xd011c31b,0xa63e4dbc),LL(0xe55bd873,0xd7f84005),LL(0xbe01f07e,0x1b09710f),L_(0x6e6cb66a)}, {LL(0x5fc5038d,0x955bdbc5),LL(0xcecf1493,0xca6b6d16),LL(0x861a0878,0x084d6464),L_(0x0d0e35ed)}},
   {{LL(0x5abc70f8,0xcdca5ed4),LL(0xdb335ec9,0x1e0fc344),LL(0xa8df4bea,0xcaea8450),L_(0xf564ec58)}, {LL(0x361561bd,0xffeefed9),LL(0x193c8457,0xd89a00be),LL(0xddabba77,0xc99c9580),L_(0xa8ec469f)}},
   {{LL(0x0423395a,0xf3c9cc68),LL(0xa888f556,0x852e0732),LL(0xaaa99a77,0x79ee145a),L_(0x4760cde2)}, {LL(0x7dbf680b,0xd543c51b),LL(0x4d61119d,0xef9d767f),LL(0xb57c70b1,0x6d605846),L_(0x8392010f)}},
   {{LL(0x4f6c403d,0x79dcc934),LL(0xd988c0b3,0x48395681),LL(0x66a38840,0x4cdffd50),L_(0x026d6f1b)}, {LL(0x0fd100a8,0xd01d8faf),LL(0xc2150428,0x37a79216),LL(0x453f0c92,0xff7a816c),L_(0x84b14e80)}},
   {{LL(0x8f9b1871,0xa7823318),LL(0xab4b5d45,0xe70810de),LL(0x1d9dae80,0xce1d1836),L_(0x1f8f6e55)}, {LL(0x3fe286a2,0x8d6bd5d6),LL(0xd272957e,0xe02405c8),LL(0xc59f4815,0x15f1375a),L_(0x1dcaf5b0)}},
   {{LL(0xd83c608e,0x6c316cb3),LL(0x5f19c736,0x28464849),LL(0x9bb6aded,0xadea31b9),L_(0x1090e65a)}, {LL(0x3ad3fb92,0xfec08eda),LL(0xad397892,0xf01270f2),LL(0xb2c262c0,0xe12a0551),L_(0xcd11a65d)}},
   {{LL(0xd00f765f,0x5db707b5),LL(0xf10d38e7,0x7a2ef5be),LL(0x5bbaa9f8,0x1692548b),L_(0xa34228d1)}, {LL(0x2a6743e2,0x654e5d41),LL(0xf2342e4d,0xbe7af05e),LL(0x53e19ef6,0x254b33f9),L_(0x2609459a)}},
   {{LL(0x0a41c216,0x7007f545),LL(0xefba2361,0xd7626b6f),LL(0x72c3ffc3,0x600f4b0f),L_(0x68bfc655)}, {LL(0x0f831a13,0x13f14da7),LL(0x6efd514e,0x9b2809ea),LL(0xcc761584,0x949da3c4),L_(0xd3524443)}},
   {{LL(0xab16d7fb,0xeca99655),LL(0x98853678,0xf946fea2),LL(0x959586ec,0x2b4d5496),L_(0x882a160e)}, {LL(0x7fe0f8f3,0xdf2a1b88),LL(0x52e13aac,0x0f3ca0d4),LL(0x4a5dde81,0xc432a9d8),L_(0xcafd8d25)}},
   {{LL(0xae4273ff,0x39455fc9),LL(0x0c20053d,0x2cb371b8),LL(0x04a4c142,0x14b7b1d1),L_(0x38e2582a)}, {LL(0xbc61e230,0x20e68183),LL(0xd15dab61,0x5cc3f0a5),LL(0xba541287,0x62766fdb),L_(0x6619e0a4)}},
   {{LL(0xf1589655,0x3bc818e2),LL(0x746af8f1,0x7a905de6),LL(0xbbe6fd69,0x53589554),L_(0x446a9ab9)}, {LL(0x42c6c834,0xc30fa665),LL(0xf4d47c43,0x8776e4e7),LL(0x431ea6ae,0x9dafb227),L_(0x4ff395ed)}},
   {{LL(0xe3b10ef9,0x2c57d4e0),LL(0xa4cceb3e,0xf2b56f06),LL(0xff6cbcb5,0x8918c037),L_(0x268bc00c)}, {LL(0x782ed32e,0xa92f2fb0),LL(0x76415814,0x5aed9645),LL(0x6859753a,0x0a6cac9a),L_(0x399773b8)}},
   {{LL(0x115dc4ae,0x7652379c),LL(0xc59e872e,0xfceb6ec3),LL(0x1fa3657f,0xa77b28fb),L_(0xa7619402)}, {LL(0x26333b61,0xa85f48be),LL(0xe180ae18,0x671848d6),LL(0x68f2c963,0x3145c6a2),L_(0xe4d30712)}},
   {{LL(0x71516dad,0xb3ff7310),LL(0x00082b09,0xb2aa0684),LL(0x0de3deec,0x33ecf9d4),L_(0xab6cc532)}, {LL(0xeb77b453,0xd8a0ae1b),LL(0x2a618d55,0xab8ca810),LL(0x3993ca70,0x0f2549ab),L_(0xcc02ada7)}},
   {{LL(0xe8cde28b,0xb5a7fc92),LL(0xec8a7e38,0x130ca680),LL(0x786e2a86,0x882cb9d7),L_(0x103bbacc)}, {LL(0xc662b5a7,0x7395bf15),LL(0x4be2ec80,0x5a3d7841),LL(0xb287766f,0x4eb9dbc0),L_(0x54c7f36b)}},
},
/* digit=29 base_pwr=2^203 */
{
   {{LL(0x11dfaf3f,0x663a8bf5),LL(0xf1bd6f1b,0x5a8b6d6f),LL(0x2de2cf36,0x4f49967b),L_(0x49fde755)}, {LL(0x1ea11bab,0x9956f1ec),LL(0x69f14061,0xf4e97d30),LL(0xc5c29863,0x516db87b),L_(0xd4456e50)}},
   {{LL(0x6d6d1495,0x78b8a6b8),LL(0x5019a365,0x9a15b171),LL(0x025726c3,0xd611e283),L_(0x68f7b67f)}, {LL(0x9367aca7,0xb4e7c5cf),LL(0x5ed6c698,0xfb9fa295),LL(0x6ea9c71d,0x2c0dd03d),L_(0x6ad6ab61)}},
   {{LL(0x74645aca,0xc1c1b8cc),LL(0xa1ea038f,0x6422554f),LL(0x9d8f1f3b,0x5e44aefb),L_(0xbc0d302a)}, {LL(0xd0763fa0,0x47ec400c),LL(0x9049574e,0x3240a34f),LL(0x9778ebf4,0xe0adc015),L_(0xe0348d22)}},
   {{LL(0x42b0ea67,0x2b432e4e),LL(0x97be15f2,0x51920440),LL(0xfb123348,0x752805f0),L_(0x095f8861)}, {LL(0x428d08f5,0xe4fd179c),LL(0xbfe10a89,0x4c2603c8),LL(0xc1a95174,0xb5e94f0f),L_(0x38bffbb7)}},
   {{LL(0xfa8bbea8,0x31d5bf10),LL(0x3b628b52,0x747269c9),LL(0x9f017556,0x50b6b23f),L_(0x2051232c)}, {LL(0xf36708ff,0x18b1fb8b),LL(0x58660126,0x731cf4a4),LL(0x85343658,0xcdefb5cb),L_(0xee7c02af)}},
   {{LL(0x517736b8,0x842b9439),LL(0x79b770df,0x017ab239),LL(0x2ca8d23d,0x29cf116b),L_(0xd89fd3b6)}, {LL(0xa037ebda,0x20362fb9),LL(0x17dff9f9,0xd010fe68),LL(0xb8350a9e,0xcbd44b83),L_(0x3adbe1f1)}},
   {{LL(0x3d4cc86b,0x171138f4),LL(0xd002fd82,0x7c670c94),LL(0xed715f0d,0x14d543b9),L_(0x770fe818)}, {LL(0x684456e3,0x2c123603),LL(0x3f8f258d,0xba0fe630),LL(0xa3564d33,0x9d216804),L_(0x435f77e5)}},
   {{LL(0xa6e5dc69,0xabe6aeac),LL(0x5e84d762,0xde0a4f86),LL(0x538499e7,0x1d74cb44),L_(0xbfb3e36a)}, {LL(0xbbcfa69a,0x198dddb7),LL(0xfcbe1984,0x873a7f1f),LL(0xb7b8bbcc,0x80cf7321),L_(0x00ac08de)}},
   {{LL(0x78482416,0xe874537f),LL(0xbbb4692e,0x221e1e86),LL(0x02f17d20,0xe7057c36),L_(0xec132fea)}, {LL(0x45b4351e,0x89b168f7),LL(0x7f9c9480,0x9d1de6f5),LL(0xf9e94168,0xf34e2409),L_(0x13e022d0)}},
   {{LL(0xc854b8a6,0x243d2e96),LL(0x358d9af4,0x8938cb33),LL(0xb47467ae,0x81a7eea1),L_(0x470b2dab)}, {LL(0x7063770a,0x3d8103c4),LL(0x9a2755c4,0xafa79d2a),LL(0x9228b07c,0x5b3e4741),L_(0xbc9b37bd)}},
   {{LL(0xd005caa3,0x8739f51c),LL(0x3e9880c9,0x48357287),LL(0xe30595ad,0x9d3ff1f0),L_(0x4b286b55)}, {LL(0xcc729699,0x89be0141),LL(0xe2576d49,0x27c29e5a),LL(0x00a64de8,0x1e63717f),L_(0x0c898bea)}},
   {{LL(0x020255bf,0xd89a7c51),LL(0x489bd20b,0x1abc0a1a),LL(0xe7ac0a69,0xf9bbb011),L_(0xbe7ead5c)}, {LL(0xfd43757d,0x5bf6715d),LL(0x2573c5cb,0x74bb4d9c),LL(0x30483451,0x7ccb3fa6),L_(0x0071ca44)}},
   {{LL(0x4c06b933,0x83211263),LL(0xd01481a8,0x868c1469),LL(0x48c43d0d,0xd5277f78),L_(0x32c5572b)}, {LL(0xd4224814,0xb34b30d1),LL(0x780e2016,0x750a0da3),LL(0x43e3806b,0xd4ad90ce),L_(0x97399684)}},
   {{LL(0x31b3565e,0xe410e718),LL(0xb76d2e77,0x78c21e10),LL(0x0df8f583,0x6f7bfcdc),L_(0x210d7777)}, {LL(0x740c2b91,0x04ef15f1),LL(0x87836a34,0x03a4d422),LL(0x44c267d1,0x2fbcd218),L_(0xbb549183)}},
   {{LL(0xe76d9d5d,0xa541189c),LL(0x3029c3b6,0xd7480312),LL(0x277c1d19,0x761b9d9e),L_(0xd09fd429)}, {LL(0x0b301fd2,0x8fe5cbdf),LL(0xf3cbac6c,0x42373c0c),LL(0xcbbe3e5d,0x4dd89b19),L_(0x75e821e8)}},
   {{LL(0x9cbc0a32,0xf92173f5),LL(0x3b6ebbae,0xca2337d8),LL(0x44ad6eda,0x49918d50),L_(0x71e12775)}, {LL(0xfc7a6248,0x7c6c9ea3),LL(0xa76fc6d0,0xe1e9ac5f),LL(0x95bb0f9e,0x0e5cdfb0),L_(0x3505e20d)}},
   {{LL(0x1671e3f6,0xc0a26f33),LL(0xf36a466b,0x6a156434),LL(0x769bd94c,0x96b0b77f),L_(0xbaa9cde2)}, {LL(0xec341956,0x961d12a2),LL(0xe383e51b,0xb1257ac5),LL(0xb680bdbd,0xfbf75271),L_(0x1f5805cf)}},
   {{LL(0xa993566d,0x205ad8c6),LL(0x84ebf013,0x1874691d),LL(0xf24e1951,0x749aace4),L_(0xaaf03f06)}, {LL(0x49610345,0x33c5f5b4),LL(0x378cf813,0x9fbc337a),LL(0x8ac53e4c,0x9b1441f3),L_(0x0113566b)}},
   {{LL(0x2f5bdd05,0x8145a740),LL(0xd10f5321,0x525fb54c),LL(0xe73eb2a8,0x4da9ccbf),L_(0xf35011e6)}, {LL(0xe92c65ec,0x375da80a),LL(0x3b493784,0xc35b0096),LL(0xe99fa1c1,0xbfcdd019),L_(0xfcd0fffb)}},
   {{LL(0x3783ad2b,0x44724690),LL(0x6169a512,0xbbc2508f),LL(0xb54ae1fa,0x97426a68),L_(0x84a843c6)}, {LL(0x1439b1e1,0x57ee4649),LL(0x9fa3b852,0x224d6e57),LL(0xcf4188dd,0xb9fbee93),L_(0xc6f126b9)}},
   {{LL(0xa302b5dc,0xa34571c9),LL(0x48662048,0xb1b8b7f2),LL(0xfa26591d,0xf6f0b8d2),L_(0xc7f51cd9)}, {LL(0x58d23928,0x5bc438ce),LL(0x66200c18,0x69768b41),LL(0xe6f1c95d,0x6844e3a4),L_(0x6e17cd42)}},
   {{LL(0xcf358165,0xddd1ff7d),LL(0xf244b120,0x73e9c17f),LL(0xe6046902,0x40c26687),L_(0x1691fe67)}, {LL(0x933ab02a,0x6e105d9c),LL(0x9b50eded,0x02d570b4),LL(0x00b6a88f,0xe9b08b55),L_(0x4b82631a)}},
   {{LL(0xa09f3f50,0x04530c1e),LL(0xf4435606,0x5195314d),LL(0x9b79c17b,0x9f045501),L_(0x839380ae)}, {LL(0x1cbc90d3,0x2d608e31),LL(0x9b83bfd2,0x913acbf4),LL(0x10db4136,0x22cc1949),L_(0x3a2e7962)}},
   {{LL(0xd3e26092,0xec19a27b),LL(0xe5ad9b86,0xddac453d),LL(0xc7d0db50,0xb0a1794d),L_(0x12340a6f)}, {LL(0x20e8dc2d,0x8bcb0707),LL(0x54438da3,0xe23756c5),LL(0x33597066,0x68e9b910),L_(0x192ee4ac)}},
   {{LL(0x64c640f8,0x6b3b3e11),LL(0x6076299f,0xe52af985),LL(0x210c46f7,0x11bcefd8),L_(0x4c36d81d)}, {LL(0x5bed674c,0xd49ab218),LL(0xb72e4c99,0x32664eea),LL(0x0e9348eb,0xa88c581f),L_(0xd67c4ee9)}},
   {{LL(0x2d66a175,0xe899e96c),LL(0xb64bf9de,0x5fb2ed8e),LL(0x43b5e3af,0xa7cc5c88),L_(0x656b15b3)}, {LL(0xdd7873f4,0xc5c4e24a),LL(0x112512c6,0x6324347c),LL(0x26ff024e,0xccd67ae6),L_(0x4f8c1dc8)}},
   {{LL(0xe0a8b3e4,0x95b13b3d),LL(0x9e553300,0x85e3e1dc),LL(0x7fafd447,0xf8451257),L_(0x6a113d4a)}, {LL(0x65687ab1,0xedde7f06),LL(0xaf37550f,0x901e09b8),LL(0x0b1d7c55,0x69e02d8f),L_(0xcfd6985a)}},
   {{LL(0xe1246244,0xecd3e1aa),LL(0xbe482681,0xfb799ecf),LL(0x47798831,0x9fac48a4),L_(0xafbbf20e)}, {LL(0x3d17bb47,0xd6f1c5bc),LL(0xfe2c6138,0x1b8d361a),LL(0xd3d7cf0a,0x0fdeccc7),L_(0xa05a2e1d)}},
   {{LL(0x3a510fa4,0x6b3cd8ac),LL(0xa16dcddd,0xaee76b4c),LL(0x0fdf0fc4,0x3cb7db3e),L_(0x5144175e)}, {LL(0x2669bf74,0xdaa284b0),LL(0xb41e11e0,0xa0a2a2a4),LL(0x334a53c5,0xb6f86d54),L_(0xea0526d6)}},
   {{LL(0x142fbd3e,0xef18bf5c),LL(0x268ce667,0xc11a8ae3),LL(0x70f67534,0xb9cffb18),L_(0xc3038e8d)}, {LL(0x941d9303,0x29b2c2c3),LL(0x4d1a3a20,0x4205ec1c),LL(0xe2c18665,0x5ab3157b),L_(0x7a60e7b3)}},
   {{LL(0xf39b6495,0x1c5c3f1d),LL(0x28e72014,0x609a5ec7),LL(0x32068427,0x47c37b5e),L_(0x94c1b6e7)}, {LL(0x39767baa,0x5f5b4e71),LL(0x77567891,0x05496c1a),LL(0x55a4f7f2,0xc6f93ff6),L_(0x8f654b8b)}},
   {{LL(0xdb846749,0x8fa02752),LL(0xfcbae4b8,0xc21478c8),LL(0xcff78948,0x22632d48),L_(0x70d86a48)}, {LL(0xa8b86e1b,0x00132e4f),LL(0x3721a7ea,0x0457eac2),LL(0xdcc0060a,0x0c2cbce2),L_(0x38fb42c3)}},
   {{LL(0x802a7a9c,0x5ea4abc7),LL(0x5f3751e3,0x2aa4c525),LL(0xac364b4b,0x495dd3d3),L_(0xa6c1d838)}, {LL(0x89d692f9,0xba68755b),LL(0x465b3970,0x72a043d5),LL(0xfe4e336e,0xfc16f7fe),L_(0x2c759722)}},
   {{LL(0x3d83071b,0xdd3a0176),LL(0xcb5a7236,0x61809f55),LL(0xa2bbecb2,0x1f553745),L_(0x46d5deeb)}, {LL(0x2602bcbb,0x4c542035),LL(0xa92075fc,0xc39e8592),LL(0xa99e29ec,0x31176d99),L_(0x7ca41c07)}},
   {{LL(0xb54001ac,0x39ce1948),LL(0xbff084c9,0xe3b4d5e5),LL(0x43984570,0x05ef0ee6),L_(0x0a861937)}, {LL(0xa8b0d940,0x1288a0c3),LL(0x52997092,0x3fa9ab6b),LL(0x24210cbf,0xec238c31),L_(0x9e2872c4)}},
   {{LL(0x30423d94,0x40fd474a),LL(0x5b8538a3,0xec2ead24),LL(0x0d208047,0x1cb60ff4),L_(0x761298e8)}, {LL(0x213bda9b,0x1b6d44ff),LL(0xbcf74593,0x5ddf4ac8),LL(0x746b2f76,0x679d4dee),L_(0x32233e28)}},
   {{LL(0x1482adcb,0x0c718f19),LL(0x66cd9b3c,0x03bfd19e),LL(0x3b3a6f85,0xa9ef5cce),L_(0xca539a7c)}, {LL(0xd847f363,0xed44fad7),LL(0x4db83c78,0xe9c8d39e),LL(0x5a9002ed,0x57ef09d7),L_(0x47b485ec)}},
   {{LL(0x7d5b0115,0xe4537555),LL(0x4cc9c9c5,0x6d1d53a0),LL(0xec59f90b,0x55f05058),L_(0xaa97605c)}, {LL(0xc10edab1,0x66f84f5d),LL(0x8c26f4dc,0x9097a136),LL(0xff1f32e7,0xda9cb1ce),L_(0x92d5f2c4)}},
   {{LL(0x918a45ed,0x5ec4e734),LL(0xffb048ed,0x861a69b7),LL(0xb9ca6eb1,0x4f8b9e40),L_(0x089b9558)}, {LL(0xd5b022cc,0x43e91e88),LL(0xcca4bf58,0xe5fa6721),LL(0x21a8592f,0x2180b560),L_(0xa9e4e373)}},
   {{LL(0x73dd816a,0x95e834c2),LL(0x85c32c46,0x52b23dc2),LL(0xad18d469,0xd0c03244),L_(0xf46c7fff)}, {LL(0xdbde0560,0xde2297b5),LL(0x8c32b174,0x8f0937ea),LL(0x28635d97,0x8b431269),L_(0x47d44429)}},
   {{LL(0xc8d22f39,0x28c78e40),LL(0x93181781,0x61fdf274),LL(0xcb623c4d,0x98d40a79),L_(0xdda4b06c)}, {LL(0xd808e325,0x3b688d0d),LL(0xfe3fd141,0xd51bdb7c),LL(0xedd15e12,0x3a965d08),L_(0xf3097f75)}},
   {{LL(0xde0a2f39,0xbda9cc54),LL(0xa6bfe2fa,0xa29b11da),LL(0xf02a2f58,0x1b6a851d),L_(0xcb819b77)}, {LL(0x8a57fd87,0x30780f8e),LL(0x9f0d56af,0x2b7ce89d),LL(0xf8a06fed,0x27c807cb),L_(0xb99605d9)}},
   {{LL(0x46b40b76,0x6786d96f),LL(0x72255ae1,0x80ccf20e),LL(0xc29277a1,0xb9deb14f),L_(0xb01102d2)}, {LL(0xaeb5c04b,0x1c8e7e81),LL(0xaf4a6e1a,0xaec9e1df),LL(0x06c802c5,0x4637b70a),L_(0x21f2e3c1)}},
   {{LL(0x9ffc5f28,0xb4a7c608),LL(0x3a6619c7,0x4be0db0a),LL(0x5deb2377,0x24dd90ec),L_(0x7e94da01)}, {LL(0x31617c3e,0x3788bd2b),LL(0x80b9433f,0x0c1368f4),LL(0xd49fdafb,0x47c8e8f1),L_(0x2be1c836)}},
   {{LL(0x5ef0901d,0x7f036146),LL(0xf5f17b3f,0xdca93bb1),LL(0x45a7571b,0x4e68fb20),L_(0x68857bb8)}, {LL(0x62732201,0xa24d2c8c),LL(0x2d13dee5,0xed2b0948),LL(0x88581796,0x118e784d),L_(0xf0493f53)}},
   {{LL(0xa0e66eb2,0x58d860e3),LL(0x8971f724,0x9083bee2),LL(0xb4f43e7e,0xb5941f8a),L_(0xb537d4a0)}, {LL(0xb975e31d,0xa327b762),LL(0x0a0d438e,0x9be00cfd),LL(0x23118e99,0x2345f880),L_(0xe40f8c30)}},
   {{LL(0xe9728c3b,0xcbd96a38),LL(0x1197dd6b,0xe0af89d0),LL(0xb47615a0,0x381bf7e1),L_(0x251a8f8f)}, {LL(0x008018e9,0xad578807),LL(0x34717cff,0x2b57e965),LL(0x5fbc83f2,0xa4f9d4fb),L_(0x64b6d093)}},
   {{LL(0x69093077,0xf183d4de),LL(0xf6acf5ed,0xbaeb9a58),LL(0x0cc30970,0x181d2aa6),L_(0x53b37c4b)}, {LL(0xce44dcb5,0x2ce916ba),LL(0x38654a56,0x59bee1db),LL(0xaba6d625,0x09ca53b2),L_(0xcdfe1968)}},
   {{LL(0x91d466d2,0x9e6894a8),LL(0xf7c98ba1,0x7219e547),LL(0x8e28cba4,0xe28c0250),L_(0xdc432b47)}, {LL(0x43c25678,0x52a05293),LL(0xff3120b7,0x789cbfc0),LL(0xd4520462,0x3107ac59),L_(0x92fc422a)}},
   {{LL(0x41f9db9c,0x378db33c),LL(0x1b7e14c2,0x6dc570a1),LL(0x2e5d4aaf,0x536bff4a),L_(0x885f68ab)}, {LL(0xf657a8f3,0xeaff53c3),LL(0x8ea881df,0xe2da9fbd),LL(0xce5d6df3,0x037450b2),L_(0x6899f186)}},
   {{LL(0xa41f3714,0xe8227713),LL(0xb158373d,0xc20b72bd),LL(0x73e5ad11,0x0d1473e2),L_(0x963a9a91)}, {LL(0xc3c89f91,0x6796b002),LL(0x32ad74a8,0x978ceeca),LL(0xcad42d90,0xa157eba3),L_(0x64fb791b)}},
   {{LL(0x6d7b048c,0x91b7250b),LL(0xb6fc8bc6,0x1b7c37d7),LL(0x207504d2,0xa04f258e),L_(0x35b71223)}, {LL(0x69296811,0x000189fb),LL(0x8cf221df,0xf5202462),LL(0xe9692259,0xc7e62afa),L_(0x2482c08d)}},
   {{LL(0xb34b2712,0xa6fa3a44),LL(0xe579dd13,0x77c307c4),LL(0x8643df69,0x57d9dfb8),L_(0x9f01f782)}, {LL(0x5573d8e5,0xb5c2c766),LL(0xbb840380,0xe0195f6a),LL(0x99e8ca50,0x6c698864),L_(0x31b4df34)}},
   {{LL(0xc836d70e,0xfa87a978),LL(0x6e27995c,0x2d67fe21),LL(0xc806a202,0xc065a04b),L_(0x93cff90b)}, {LL(0x6731446f,0xbb867036),LL(0x9cdfc841,0xe8066219),LL(0x0a935f1c,0xa9d96eb8),L_(0xc15bd4d8)}},
   {{LL(0x60b97b73,0xd003f54a),LL(0xa4a49d7a,0x63dd3e3b),LL(0xa79d7693,0xa15f7dc6),L_(0xbd4d523d)}, {LL(0x1e02bb50,0x6dbfe7b7),LL(0xcc88e7c3,0xb0a1bb28),LL(0x8264318f,0x5e5fd5b2),L_(0x5f2fa1e2)}},
   {{LL(0xe76ef3c7,0xec0ebc09),LL(0x1936faf8,0xd297aff2),LL(0xc8c40d08,0x01dd06de),L_(0x54d85f2a)}, {LL(0x2bc0d559,0x8e9d926c),LL(0xcbcea653,0x99783016),LL(0x37d07340,0x0beb45e4),L_(0x3de7fc81)}},
   {{LL(0x564ba5a3,0x32058809),LL(0x9850a91e,0x3af0e1db),LL(0x5464c544,0x1452f874),L_(0xb9637ddd)}, {LL(0xeb527a99,0x7298be18),LL(0xdf5911d2,0x16aa8506),LL(0xf6dfd8a8,0x8bb252ea),L_(0x839f8d56)}},
   {{LL(0x89f63fb2,0xc7fe9490),LL(0xa2fc5db9,0x99c60e8f),LL(0x47b2b00a,0x142e5001),L_(0x557178b5)}, {LL(0x081ac8ff,0x7281e45c),LL(0x80053d16,0x92d81832),LL(0x335bfb8b,0x3285f848),L_(0xe9e17269)}},
   {{LL(0xdbe51600,0xd38c0683),LL(0xd98fa793,0x990e1e00),LL(0x14117a61,0x0d3b9c08),L_(0x3e3d6538)}, {LL(0x7a4ddb24,0x374f82aa),LL(0x538b9bf1,0x0c5bcaf1),LL(0x1cc16fb8,0xe0c1bced),L_(0xc0de70fa)}},
   {{LL(0xd35b0a61,0x96448e89),LL(0x98c48df5,0x8846f415),LL(0x50f58327,0x4770c98e),L_(0x197ced21)}, {LL(0x5ff2d0d7,0x1af1733c),LL(0x2bca7ffe,0xa04c1cce),LL(0xeac8a367,0x13278f06),L_(0x5dd0431b)}},
   {{LL(0xd19368fc,0xa83c536e),LL(0x02379421,0x118f2771),LL(0x218d25f6,0x90caeeec),L_(0xf254a0e4)}, {LL(0xc0dc699c,0x44d824bf),LL(0x0bbdc5a2,0x5903a15f),LL(0x21933c71,0xec2c7cff),L_(0x0a756aae)}},
   {{LL(0x97dfc53d,0xd8438b57),LL(0x1d02bb13,0x42584478),LL(0x25ef520b,0x3dd8667d),L_(0xe3e22c9e)}, {LL(0xc5d6139c,0x0d51408e),LL(0x0227f2bc,0xd3e4dd71),LL(0x090e4a02,0x2fc23bc1),L_(0xf9171e03)}},
   {{LL(0x11deaeb1,0x8d85409b),LL(0xd1184135,0x617f0460),LL(0xf357dae1,0x7b364c45),L_(0xb9d690d1)}, {LL(0x4d915f34,0x0c5f601c),LL(0xcb872f3a,0xe0fe9ec0),LL(0xbccb4f0f,0xf7bfadef),L_(0x9f017f8b)}},
   {{LL(0x83cff561,0x15442e1c),LL(0x62cd98e6,0xab89d9c9),LL(0x58dcba94,0x7426c5f8),L_(0x80d0a8cf)}, {LL(0x449af58b,0x52648e23),LL(0x3a628b1d,0xdb61eac8),LL(0xf0844217,0x9c140c5f),L_(0xb3cc7bfa)}},
},
/* digit=30 base_pwr=2^210 */
{
   {{LL(0xd6e670a1,0xf0426be1),LL(0xb4f44241,0x14838573),LL(0xefc4394a,0x80845fb2),L_(0x90205f0f)}, {LL(0x99c56ec6,0x8a6cba70),LL(0x4345224a,0x6149abf7),LL(0xfc80d60c,0x9c7703f2),L_(0x3d5a7643)}},
   {{LL(0x730ebcfd,0xe4b299d2),LL(0x79702991,0x6cdbe6f3),LL(0xad58884b,0xff023a31),L_(0x18b5c6ce)}, {LL(0x51fd8a3b,0x433c522f),LL(0x3e4907b4,0xc1579271),LL(0x3c1c0282,0xe4454779),L_(0xb4df9ee5)}},
   {{LL(0x48953bf7,0x658675d7),LL(0xdf2f4ac2,0x8faf514a),LL(0xdde660e1,0x50213318),L_(0x64826f9f)}, {LL(0x8433d7fa,0xadcc3e45),LL(0xbc26833e,0xcc5ad72a),LL(0xde547aae,0x619251e2),L_(0xf96f73d1)}},
   {{LL(0x548b77dd,0xba3e8f2e),LL(0x728cadb5,0x21cbbb1a),LL(0x0feebd0a,0xdb740393),L_(0xdcffa1d6)}, {LL(0xb6848db5,0xa742cd6f),LL(0x549a4fca,0x39f63442),LL(0x3d24aeeb,0x426c42e9),L_(0xb67d7f28)}},
   {{LL(0x408e3341,0x96cc626e),LL(0xd2b3305b,0x6a9bb18a),LL(0x355d47d4,0x33f53993),L_(0x960f4207)}, {LL(0x16218c03,0x329378d3),LL(0x1004c020,0x70de3eb8),LL(0xa9117cf1,0x0e5dbd85),L_(0x73f55c21)}},
   {{LL(0xb9a63bdd,0x456357c9),LL(0x1b3ded00,0x8e077f94),LL(0x0f0f2d4f,0x51648f45),L_(0xab4e5da6)}, {LL(0x3963e9df,0xe96f0202),LL(0xcb55e2cd,0x3b87f53b),LL(0x7b3cc4d5,0x0090498b),L_(0xdfb4de74)}},
   {{LL(0x98927a32,0xba64358f),LL(0x757e6fbb,0xccb5e233),LL(0x95336360,0x50d4d37c),L_(0xac96f5a7)}, {LL(0x93c38d36,0x53008bb8),LL(0x33869046,0x5e587a5b),LL(0xfb1262f2,0x1e3e57ec),L_(0x407af3f6)}},
   {{LL(0xc458ba03,0x9d6ad457),LL(0x78e956e9,0x82640533),LL(0xb94af545,0x32977879),L_(0xd20f35a3)}, {LL(0x6224b353,0x9c65ff21),LL(0x5c8f99f9,0x4a46fdf7),LL(0x4ce9a00e,0x307b3d8d),L_(0x798d2cf9)}},
   {{LL(0x77b2fb3b,0x3b8e6ef5),LL(0xb6ca3273,0xe32b3b97),LL(0xe260b8d0,0x9c66c2a4),L_(0x05eb84cf)}, {LL(0xaf957b78,0xbe1f3ecf),LL(0x1e35adca,0x9d69d314),LL(0xa175edc2,0x4681a0ae),L_(0xc6715ff6)}},
   {{LL(0xee864b12,0xe9c88636),LL(0x32143c93,0xf0c1e0f3),LL(0x434818e2,0x25a929c9),L_(0x4c806ec9)}, {LL(0x20527edc,0xfe3e80ac),LL(0xa0372526,0xd5fc27b6),LL(0x5279eb8a,0xee97419a),L_(0x98dbe31d)}},
   {{LL(0xcbc47797,0xe8bb79d3),LL(0xa6d3938a,0xf225d84a),LL(0x0b2f88a1,0xf279df50),L_(0x59a54b2e)}, {LL(0x8cd34576,0x1390a7ac),LL(0xc9c37e25,0x1130b1eb),LL(0x0777586f,0x3aa2516a),L_(0x42cece95)}},
   {{LL(0x3964d2a4,0xa47e21c1),LL(0x5f9875c7,0xf918e47e),LL(0x2ecbc86c,0x4ac56066),L_(0x5ea26fc2)}, {LL(0xa5a826e0,0x31e429d6),LL(0x944c3736,0xd7082b19),LL(0xcf2fc959,0x02c3927e),L_(0x3bb33cf9)}},
   {{LL(0x27da5082,0xcf56d753),LL(0x64f98e4d,0xd40b530a),LL(0x1c7766cc,0xb20cfe07),L_(0xe351283a)}, {LL(0x18e30244,0xdc9dc465),LL(0x3bee62a5,0xcccf7fb0),LL(0xb2e1f645,0xfe54699d),L_(0x18ee6ea1)}},
   {{LL(0xdf71a0f1,0xd2fe136d),LL(0xe5f658a8,0x5a88e4dd),LL(0xd83b5473,0x94a1a0d2),L_(0x586d7529)}, {LL(0x1d163bf2,0x0748fda6),LL(0xe98d9b7d,0x69880bc0),LL(0xf0def9e5,0xc4e12cd5),L_(0xf40402da)}},
   {{LL(0x585fff04,0x2956c762),LL(0x257681d9,0x3ba8ab0b),LL(0x696e0262,0xfd84ba5b),L_(0x0675debf)}, {LL(0xa783c064,0xef9b82a5),LL(0xe38868a1,0xdaa2e717),LL(0xcdcdf922,0x292fb9c9),L_(0x269d71d2)}},
   {{LL(0x477cabed,0xb5fae58e),LL(0x55622b3a,0x6ba8e28a),LL(0xff23a08d,0x56c2b54e),L_(0x866ba05b)}, {LL(0x037720d9,0xa737dae5),LL(0xdc5131fd,0xb93a1d0b),LL(0xbb1176e6,0x860b7b0d),L_(0xf51c76d2)}},
   {{LL(0x5845cd10,0x6bd647f2),LL(0xe811642e,0x9a4283ed),LL(0x0776f557,0x2c9fb90d),L_(0xb8cebf02)}, {LL(0xb1612f57,0x643fb865),LL(0xc4902b6c,0x88ea78f7),LL(0x74aac273,0xfd2878e7),L_(0x1a76b1b7)}},
   {{LL(0xb6983e76,0xf609c0af),LL(0x0cfea42c,0x123027ca),LL(0x7a72c3c7,0xb9872dc7),L_(0x1ae5bcf6)}, {LL(0x57a545e2,0x9fe44848),LL(0x2a90f4d7,0x24f2f0f2),LL(0xdcdd6bf2,0x32eb5ac2),L_(0x939219ad)}},
   {{LL(0x2540ecf0,0x267059b6),LL(0x2669c8c4,0x6d717377),LL(0xb6ba656d,0xc80a4182),L_(0xa67b63ae)}, {LL(0xac5cdbce,0x7acd503b),LL(0x98520b74,0xf8884f71),LL(0x0688ae94,0x47fc66da),L_(0xacf9786e)}},
   {{LL(0x9b68b84a,0xde210bb3),LL(0x74430a09,0x98b4770a),LL(0xf0e810c7,0x85cefd4f),L_(0xba1cac0a)}, {LL(0xe9e74a58,0x35194ff1),LL(0x6452e158,0x520f3a2b),LL(0x8ed85e25,0xd8535bc9),L_(0x3bc32918)}},
   {{LL(0xd77f4a13,0xd3467c72),LL(0x3046a69b,0x3a350288),LL(0x996236b5,0xf02f1b90),L_(0xdbce524f)}, {LL(0x3a7711d1,0xdcc369a6),LL(0xa2eda832,0x81b1b5d5),LL(0xcb12490a,0x33444c56),L_(0xb4685d00)}},
   {{LL(0x69d66472,0x6fe5a181),LL(0xa316d37d,0x11b6d85b),LL(0x75ec9a2b,0x8b98fad0),L_(0x0c5e4465)}, {LL(0xde810eaf,0xbd855a8d),LL(0x7661287b,0x436e58d0),LL(0x9b044bfa,0x0f23ddbb),L_(0x7abd77aa)}},
   {{LL(0xc57c3228,0xe7d89f58),LL(0x490f55aa,0x2e88de2d),LL(0x3aa72314,0xdea3a145),L_(0x159c120c)}, {LL(0x5f68084a,0x840bb946),LL(0x1516983e,0xdcbbb7b4),LL(0x527c0f73,0x0d4d39e1),L_(0xe3846304)}},
   {{LL(0x1fed9837,0x56c53d5f),LL(0x8c3176ff,0x752789ed),LL(0x26489626,0x94c341e1),L_(0xa5b14d8d)}, {LL(0x3cb481d4,0xb4f1f612),LL(0x1a2f793c,0x94eb82ca),LL(0x600ebda4,0x6ba99e2b),L_(0xd1571ee0)}},
   {{LL(0x960bacd7,0x122b6639),LL(0x41a70d6d,0xbaf725d5),LL(0x50c3e831,0x37bdeb4c),L_(0x948ab46f)}, {LL(0x7cc81d1e,0x1a55a55a),LL(0x3b20c1e6,0xa6276eba),LL(0x81e5763b,0x4ddc462f),L_(0xffa876e2)}},
   {{LL(0x8d1558c7,0x62e4c32c),LL(0x7f945839,0xfc3257d8),LL(0xa1e798f7,0xfe755aec),L_(0x5524084e)}, {LL(0xcebab559,0x02c2a9e2),LL(0xdc627b0e,0xf8a1716c),LL(0x7c827c47,0xbcf0b634),L_(0x677ad978)}},
   {{LL(0xe3c947ec,0x74af562c),LL(0x8713ce0f,0x1493b8fd),LL(0x6951ea3a,0x23d2ab0f),L_(0x761f3a78)}, {LL(0xb7530283,0x098fb6af),LL(0xeb48e791,0xccff2003),LL(0x160c9294,0xf1992cc6),L_(0x2ed7be4f)}},
   {{LL(0xff03f6d9,0x02ad20b1),LL(0x30165084,0x17295d20),LL(0xb8bee43f,0x67757e6c),L_(0x0eedd6d6)}, {LL(0x61ba2edd,0x5acf9123),LL(0x8dee8798,0x4442dc9a),LL(0xa4c9925a,0x9970bc21),L_(0x54e339cf)}},
   {{LL(0x250b4244,0x7f335cdf),LL(0x5019bc77,0xa892b382),LL(0x15ce5f7d,0x4f31c43e),L_(0xfa69d995)}, {LL(0xae609392,0x1d57d52c),LL(0x20dc7c66,0x6e6a7489),LL(0x01a82796,0xa8fa2376),L_(0x861ac30c)}},
   {{LL(0xadcafee5,0x56acbf11),LL(0x7b20a0c7,0xa63a1d19),LL(0xbec3d69c,0x5e324e3c),L_(0xd3d97b66)}, {LL(0x64b971b3,0xc585e140),LL(0x2b8e18cf,0xd08772ba),LL(0xe0b2513f,0x6e7d6c03),L_(0x7e1a6116)}},
   {{LL(0x9bc4e6f4,0x23bf2ce8),LL(0x82122cee,0x0b5f01ee),LL(0x318ba204,0xa93e44d4),L_(0xd2180b18)}, {LL(0xd6ef00ac,0xd9d85de8),LL(0xd7d06345,0x3062eb10),LL(0x5bea35e9,0x952d933c),L_(0x72bd41e8)}},
   {{LL(0x72791646,0x60e9cb5e),LL(0x107232fb,0x644a5b6e),LL(0x004f8684,0xa23e4299),L_(0xefa3c56b)}, {LL(0x964aa750,0xad436903),LL(0xbb576a94,0x041c2a91),LL(0xbd787460,0x3513e3fe),L_(0x4066ae12)}},
   {{LL(0x09a3d049,0x627d1807),LL(0x7365dc29,0xe00d7aec),LL(0x0dec445a,0x96d5022f),L_(0xf4343e09)}, {LL(0x515a9b8f,0x00d16dd6),LL(0xeeaa6e35,0x97242fab),LL(0x220f953e,0x91ecc397),L_(0xaf714cd0)}},
   {{LL(0x79198e5a,0x6dd11272),LL(0xce83daf3,0x9169c8b8),LL(0x6ce5dec2,0xce89d045),L_(0xbafd94a5)}, {LL(0x1563c0ce,0x577ca0e0),LL(0x1e36fd91,0x5a836e51),LL(0x060d3970,0x729ba81d),L_(0x8b7e4fbb)}},
   {{LL(0x16781a85,0xdbaa195f),LL(0xb43de333,0xec92b939),LL(0x2ccf4a4e,0x036ff9cf),L_(0xafc236c4)}, {LL(0x6221e156,0xd7468cbb),LL(0x9a645200,0xffcbfaac),LL(0x5b2deae6,0xb553572e),L_(0xd4efa381)}},
   {{LL(0xd257958d,0xc9536b4a),LL(0x10222aaa,0x9ab6a135),LL(0x84934c38,0xdf7ca8e3),L_(0x5e5dc148)}, {LL(0x93660c96,0x4d98778c),LL(0xb7c8140c,0xd69648d2),LL(0x69c09498,0xb0ea1d36),L_(0x7e0cd235)}},
   {{LL(0x62e6becb,0xe9443a72),LL(0x94747e78,0xd494860b),LL(0x2415da80,0xf50cdd9d),L_(0x65d2fa97)}, {LL(0x48dbba8d,0x3ca3104f),LL(0xaf39be74,0xbd695019),LL(0x0441383f,0x393e3175),L_(0x4e5bd1d4)}},
   {{LL(0xf6d61041,0x70145555),LL(0xc463ec32,0x8fa5018e),LL(0x1e5a98cb,0x59b3adf7),L_(0xeb7c80fc)}, {LL(0x529f5fa8,0x5234550b),LL(0x4eb610a4,0x2ac1098d),LL(0xdc8db9bb,0x52106057),L_(0xbc3a065e)}},
   {{LL(0x62bb8d08,0x6e360f55),LL(0xee7f7e1c,0xe795bf91),LL(0x7411bd70,0x627a5d54),L_(0xb191903e)}, {LL(0x323d2db5,0x6bb808ed),LL(0x484c498f,0xded0b935),LL(0x3cef8b7c,0xcc379eda),L_(0xe1f0ee29)}},
   {{LL(0x106cde5b,0x21a0558f),LL(0x2ea40699,0x84c5579e),LL(0x057a8487,0x5834fcc4),L_(0xb35b630b)}, {LL(0x6779af9f,0x2d1d0e72),LL(0x6560d0d6,0xd90a0248),LL(0xeb026a8b,0x4d572129),L_(0x720991c7)}},
   {{LL(0x22ee217f,0x657c8db7),LL(0x5e4f7ccd,0x12f06bfb),LL(0x8351fe6c,0xa43443b2),L_(0x48c1ec10)}, {LL(0x01489ccd,0xb376d6e2),LL(0xc77d6dfd,0x727638f2),LL(0x0f953513,0xeae9d0a1),L_(0x91a7a744)}},
   {{LL(0xb85754e4,0x59454f27),LL(0xd2c2004d,0xe70796e3),LL(0xacc344aa,0x775a61d7),L_(0x1c186106)}, {LL(0xa1656bbb,0xdcf00b60),LL(0x4e3f081c,0x1af1503d),LL(0xcaa9cad3,0x1e4e523b),L_(0x816a09ca)}},
   {{LL(0x0b303a9c,0xd468c096),LL(0x86aa5ecf,0x1910f732),LL(0x26d08884,0xe43dffa8),L_(0x6501f291)}, {LL(0xa597567a,0x88248cd9),LL(0x4711c711,0x5263d7c9),LL(0xa4245344,0xcfde7c18),L_(0x4111cd00)}},
   {{LL(0x7d78f32c,0x0df3b4b0),LL(0x9cf9b276,0xf4191861),LL(0x9dd92d6a,0x6444967a),L_(0x5f3bbd8f)}, {LL(0xa9ba118e,0x1a4b4820),LL(0x2d798278,0x2aa6dcc4),LL(0xe9681e55,0x0ce3d37c),L_(0x739d5ad5)}},
   {{LL(0xdd02bc4d,0xcf20f024),LL(0xf4697953,0x90e93ff7),LL(0x789e6af7,0x1050c0c2),L_(0x383c22fa)}, {LL(0x60720c41,0x5cc2adcc),LL(0x2ace7063,0xf57806de),LL(0xf6f6aaf0,0x31c60f20),L_(0x7b350e28)}},
   {{LL(0xcc10306b,0xf49e5f61),LL(0x1a92eddc,0x58cb04a9),LL(0x365a5041,0xc89ba67e),L_(0xa32fc59f)}, {LL(0x65f980db,0xdb9db26b),LL(0x09322a72,0x93261bc9),LL(0x3fda4217,0xc6c63b34),L_(0xddeb9986)}},
   {{LL(0x148cbb74,0x77b280f1),LL(0xc18c7395,0x7ee57ca3),LL(0x4cd0382a,0x37436f93),L_(0x66528f8e)}, {LL(0x6e4b7e57,0x2dd6704a),LL(0x83b283eb,0x7f7d547a),LL(0x54972ce8,0xbca75a4c),L_(0xa9f10506)}},
   {{LL(0x7605a526,0xee53ff2e),LL(0x5105c866,0x09187c91),LL(0x02b0042e,0x1f9de3bb),L_(0x4f9500fd)}, {LL(0x880380ec,0x8cb6dd5c),LL(0xaa04e960,0x2a74a12c),LL(0xa0112a70,0xec0e66ff),L_(0xec305363)}},
   {{LL(0x4b9381fc,0xa014524b),LL(0x400fd142,0xcb74eb3a),LL(0xf02c64c2,0xd9684416),L_(0x8da612c7)}, {LL(0x06a80727,0x4e54713b),LL(0x7cde46af,0xfe9ec578),LL(0x8380cc14,0x5013084d),L_(0x80963cc8)}},
   {{LL(0x349f5079,0x6dcbbe48),LL(0x6b6125b6,0xceefada1),LL(0x67b840bc,0x16c79367),L_(0x570c02ad)}, {LL(0x361a38dd,0x8552cefd),LL(0xa8167ca5,0x286e6e20),LL(0x3636e865,0x9c4305d9),L_(0x43d5af7e)}},
   {{LL(0x7f9a80cd,0x413cd659),LL(0x9c95b00e,0x310a2a72),LL(0x64a28802,0x6ef01bfc),L_(0x858e4b79)}, {LL(0xb1565b88,0x113a4fdb),LL(0xd390a92a,0xc04a736a),LL(0x774c1209,0x63557ff4),L_(0x54e47eaa)}},
   {{LL(0x940daa93,0x27760c7d),LL(0x9e3147b9,0xd15d7a6a),LL(0xb9bb9711,0x89363a50),L_(0xe852edfe)}, {LL(0x94740e26,0x43ee10a5),LL(0x2d6acef6,0xaa297031),LL(0x03fc12e9,0x20890385),L_(0x2d45580b)}},
   {{LL(0x87453ec7,0x4101d16f),LL(0xbebb2e76,0xf89a5338),LL(0x533638ee,0x05bdd7f6),L_(0x92a0bd39)}, {LL(0x56483d09,0x4c760096),LL(0x81c6243b,0x37294333),LL(0x5119eebb,0x049c6a06),L_(0xc410f246)}},
   {{LL(0xd5e79013,0x51167f9a),LL(0x12be7d2f,0x90e63900),LL(0xa41ce391,0x0572301a),L_(0x0a761ede)}, {LL(0x8bb599f2,0xe5dca064),LL(0x3dad4d6c,0x88bd9756),LL(0x82d548d7,0x79a33fd1),L_(0x76d2d77b)}},
   {{LL(0x801eaf93,0x77f53014),LL(0xa4c2daab,0xd48faf97),LL(0x40a7d787,0x604bbbf3),L_(0xcc41bd91)}, {LL(0x9de9fab8,0x4c6757d4),LL(0xe55da08f,0x9a37321e),LL(0xf50459ca,0xf8d3a03d),L_(0x8252e5cb)}},
   {{LL(0xc0ad0a5d,0xaaca1be1),LL(0xb4f7cd9f,0x6f65ac78),LL(0xfae3835d,0xf685dd47),L_(0x6c5b897b)}, {LL(0x8e631946,0x03e5ac2e),LL(0x71afc215,0x4dafffd8),LL(0xce6c64d0,0xa1bdc366),L_(0xe59eacca)}},
   {{LL(0xcde53a1b,0x28e18006),LL(0x428bd7b1,0xec3e3023),LL(0x3bf0eb6c,0x69ce21e0),L_(0xfbe09c69)}, {LL(0xc6ab638b,0xf7c3e582),LL(0x3283bc61,0xe88ef77b),LL(0x20ead3f8,0x1f935f24),L_(0x3b040b57)}},
   {{LL(0x20008407,0xa23c1b84),LL(0x41a0017c,0xa6773559),LL(0xc09598e5,0xffd68469),L_(0x1da39b13)}, {LL(0x221c586e,0xb19fcd20),LL(0xe71a0fa0,0xa76b6f70),LL(0x83e812db,0xb605b69d),L_(0x0efdfc84)}},
   {{LL(0x28c66881,0x656d6dd0),LL(0x58a3f7d2,0x7540ad09),LL(0x1396fe6c,0x52fddb17),L_(0x39edad21)}, {LL(0xa2cc1fe5,0xfb57f569),LL(0x1072b4e0,0xddfbbf25),LL(0x94372dad,0x0a26aff2),L_(0x88f38fd7)}},
   {{LL(0x8fdfcf33,0x6d4895a3),LL(0xff2438e1,0x19da7037),LL(0x67f22bef,0x3cfaa48e),L_(0x6bf9fc2b)}, {LL(0x0a03dd57,0x7b4df30e),LL(0x956e9ec2,0xd632b168),LL(0xe6cb0b1c,0xfc98eaec),L_(0x7dbf18a3)}},
   {{LL(0x8b21bd41,0xf20afcc8),LL(0x9b4028ed,0x9fb97c45),LL(0x4e476a55,0x6bf05b58),L_(0x2b0c488d)}, {LL(0xec47e2f5,0x01d3ca74),LL(0xc0d6bd28,0x7a3a23e0),LL(0x95f353c8,0x010298ee),L_(0x034e53ca)}},
   {{LL(0x6fa3f957,0x80ff52bb),LL(0x50412d11,0xf58be861),LL(0xbe2f9f7b,0xe84a58ad),L_(0x83284261)}, {LL(0xa5a7f060,0xc605b12b),LL(0x5da82b44,0x86fa207e),LL(0x837f9c22,0x10d656cc),L_(0x454d1413)}},
   {{LL(0xb270f308,0xde3d1dc4),LL(0x6619a39c,0xf9fa04cf),LL(0xaaa9723b,0x3910ae67),L_(0x86fdb715)}, {LL(0x1046b659,0x3782c29a),LL(0x5c35eb01,0xd10183f6),LL(0x76b4be96,0xdd570ea6),L_(0xf56b5810)}},
   {{LL(0x237ac7ff,0x9ec12515),LL(0x3f080f23,0xed07ceb7),LL(0xc3509a49,0x08bba00b),L_(0x07998613)}, {LL(0xd7bef81a,0x5c36ecb2),LL(0x75c6840a,0x9bd57323),LL(0x3e8f5265,0x2c678e3c),L_(0xc7f1cc73)}},
},
/* digit=31 base_pwr=2^217 */
{
   {{LL(0x0993fc0d,0x2409728c),LL(0x2d5fd352,0x7af57f46),LL(0x28f3cb14,0xee72bd56),L_(0x21c2f494)}, {LL(0xf22e2fef,0xac73118d),LL(0x283e1b56,0x490bb371),LL(0xf00c3802,0xaa6c7b0a),L_(0xee304f23)}},
   {{LL(0x2619d224,0x5b89b52b),LL(0x433d8d17,0x1df61e0b),LL(0xec27bd08,0x49353c25),L_(0xf18b23f5)}, {LL(0xc525714f,0x2c798eec),LL(0x91011438,0x6bd787c1),LL(0xfaa77bac,0x6e2db36a),L_(0x2a3b730b)}},
   {{LL(0xf5bd48f0,0xea09f154),LL(0x18c36c89,0x45e66f63),LL(0xff1cf2a0,0x1d069cd7),L_(0x672fde0f)}, {LL(0x07bd9cc6,0xf1f4e5b9),LL(0x86238146,0xa40cdc9d),LL(0xcdc7fd6b,0x00bcd404),L_(0x51a105eb)}},
   {{LL(0xd8d6b178,0x348659fa),LL(0x0afc95de,0x724963c6),LL(0x478c3432,0x1f2f8d2f),L_(0xaa0e8781)}, {LL(0x49e60cff,0xbf3811fe),LL(0x526d1fb9,0x50f264cd),LL(0xf802b4ae,0xf613e178),L_(0xd3df21bc)}},
   {{LL(0x45bfb8c0,0x40af4179),LL(0x3b133f03,0x6e8997cf),LL(0x9b5dda1d,0xc2f372b3),L_(0x3948b7d7)}, {LL(0x55f88793,0x99028593),LL(0xbcf7e27f,0x405ba3c5),LL(0x1de8aa5f,0xb2fc7cd3),L_(0xaadbe95a)}},
   {{LL(0xd00dadb4,0xff2c8be9),LL(0x26721264,0x8d199a29),LL(0xbd6c65e6,0x7f2093a5),L_(0x21e5d488)}, {LL(0x03e74b8c,0xa000a3ec),LL(0x68530945,0xeb8e9efa),LL(0xd1f6fc3c,0x3f15aab2),L_(0x9b5d8cbb)}},
   {{LL(0xb9fa3849,0x45f74205),LL(0x97a4da7b,0x121a4ca8),LL(0x6d246547,0x819c4b2d),L_(0x8255bc16)}, {LL(0x156c1f2a,0xf9d396e1),LL(0x5502b719,0xe8758b6d),LL(0x8b9b6ca8,0x94965c57),L_(0x3c7a5ffc)}},
   {{LL(0x75175beb,0xc11d6984),LL(0xbad3d4ba,0x3c966d66),LL(0x67c5bea4,0xb82cec9b),L_(0x93b6322c)}, {LL(0x1f09f72e,0x9661c046),LL(0x769c204a,0x12a03826),LL(0x35880fdc,0xc215f2cd),L_(0x4f5bf77f)}},
   {{LL(0x4fdd677c,0x8bce7ecf),LL(0x847b9ee6,0x62abc7d2),LL(0x8cb1f02b,0xfffe0edc),L_(0x62c22364)}, {LL(0x48727496,0xdcd7d51d),LL(0xa9cd4c50,0xfe1661b1),LL(0x30edb5dc,0x82ea29e2),L_(0xcc4403d4)}},
   {{LL(0xc5db9960,0xed02ad3a),LL(0x66e7f904,0x194f3505),LL(0x2877668b,0xe7bcfa22),L_(0x873070fa)}, {LL(0xf3636bf8,0xf013615b),LL(0x2fe9dfb1,0xeec498a2),LL(0x51ff974b,0x56961e83),L_(0xcd87825f)}},
   {{LL(0xb4807fe6,0x5b989bd9),LL(0x2f915297,0x161be465),LL(0xcf0387e7,0x80e7dd9f),L_(0x2e7c6dd7)}, {LL(0xb221d5c4,0x861c2ffd),LL(0x1df12d2f,0x36db9960),LL(0x048668bc,0x9b862a1d),L_(0x2b64be2a)}},
   {{LL(0x02a0d3cc,0x644173f7),LL(0xc3611806,0x2103668f),LL(0x78a6d7ae,0xe22b26dd),L_(0x1dd89345)}, {LL(0x39193a47,0xd1587982),LL(0x7692c1c8,0x9106104e),LL(0xecf6b1c4,0x0a975fb0),L_(0x0d6babd5)}},
   {{LL(0xf550f1a0,0x4222b0ec),LL(0xb5c91a5b,0x78338ceb),LL(0x0ec3e064,0x9d0d2af2),L_(0x660e553e)}, {LL(0x1d9a4d16,0xa632918f),LL(0xff1f387c,0x7bd6a5fc),LL(0xd795c509,0x67e40e1b),L_(0xdada8e3d)}},
   {{LL(0x7f1d2d28,0x6eb26316),LL(0x4e1593ef,0x93bd25dd),LL(0x8698f986,0x2d0a715c),L_(0x8e8b3499)}, {LL(0x41550af3,0x0b43672a),LL(0x132c1c9a,0xad35a8ca),LL(0xfd973661,0xea97e0b3),L_(0x3b811a56)}},
   {{LL(0x87e35efe,0x3edb12df),LL(0xf66872e6,0x31c81b67),LL(0xc41cad87,0x0cfa43fd),L_(0x43f4aa49)}, {LL(0x209b0e7b,0xcfa7ab7a),LL(0xafd31516,0xe582ff27),LL(0xc78d7ba1,0xea786f69),L_(0x2090d8fa)}},
   {{LL(0x073b3ba4,0x7ee5008f),LL(0x28c86852,0xf81d75be),LL(0xd457685a,0xc2987502),L_(0xe596765e)}, {LL(0xd0f9d913,0xfb631b84),LL(0x42aaadd8,0xe5cce6ba),LL(0xf7d72923,0x2257f5b6),L_(0x260ae885)}},
   {{LL(0x6ca309b3,0xdb84c9b9),LL(0xdbb13e44,0x0f9da41e),LL(0xa7294694,0xc1443c26),L_(0xe02af6c2)}, {LL(0xfe44c5f2,0x293f667a),LL(0x52bbc8cd,0x5f04fc4a),LL(0x7e1de72e,0x6907cdc1),L_(0xf8a9aeab)}},
   {{LL(0xa3dd55e7,0x3fade182),LL(0xd7401dde,0x03aa0312),LL(0xc85449b7,0xc47549c9),L_(0x6ec51217)}, {LL(0x55e72225,0xae146938),LL(0x4bbe9886,0x7c810544),LL(0xee349410,0x9f320ada),L_(0x4fdf9e41)}},
   {{LL(0x2f1465cb,0xf506f849),LL(0xf58881e9,0xb38fe4e3),LL(0xd6feb622,0xedfd3c65),L_(0xb163b10b)}, {LL(0x1345454b,0xb8f8a7bb),LL(0xb10b02be,0x11df874d),LL(0x1e5816ca,0x0ad84066),L_(0x9cd4eb66)}},
   {{LL(0x4f8d4868,0x413431bb),LL(0x9ea4aaec,0xd1ee0d6b),LL(0x135f5f2b,0x9e4519f3),L_(0xd3e6e8f7)}, {LL(0x7803a8e3,0xfa8179f8),LL(0x318fb330,0x53800116),LL(0xe0cb106b,0x95e1ca87),L_(0xfb4ce48c)}},
   {{LL(0xec15776f,0xf3d74e54),LL(0xb7fb663d,0x4fd65071),LL(0x6706a48f,0x7f8a7cfb),L_(0x52d7353d)}, {LL(0x811679af,0xc42a9ef3),LL(0x5bd192eb,0x5ed520f8),LL(0x2b1ff6c1,0xaa53702a),L_(0x5388fa6f)}},
   {{LL(0x2813a7fe,0x423e53a3),LL(0x0baae890,0x3d22e91f),LL(0x08234191,0x9bfec6fd),L_(0xc8f4d373)}, {LL(0xfbdbc5c1,0x119e3476),LL(0x19587b51,0xfeed51b0),LL(0x1c276e15,0x951dd485),L_(0x98048932)}},
   {{LL(0x21e06022,0xbe2d77e9),LL(0x67d742de,0x0c1663f5),LL(0x7983fd13,0x7115e508),L_(0xb9406bb8)}, {LL(0x4eba1a04,0xe45eb297),LL(0xdf3a7991,0x5693a9fc),LL(0x03ba9fe5,0x25d8f862),L_(0x6ea3053f)}},
   {{LL(0x93e92c0c,0xefbe9105),LL(0x266e3900,0x1cee1814),LL(0xe042e34f,0xbec6b919),L_(0x5cb15a6f)}, {LL(0xd41a287a,0x7568673f),LL(0xfe7b7595,0x4deaa29d),LL(0x3e67572e,0x2d437dfd),L_(0x03299218)}},
   {{LL(0xfea564c7,0xc2d6c313),LL(0x3d460901,0x3e809da5),LL(0x591fe51a,0xeb85aa17),L_(0x44d65538)}, {LL(0x45dff18c,0xea596f90),LL(0x48a2f0df,0x4708877e),LL(0xc2cabfd0,0x17f7dae8),L_(0x31a2b7ff)}},
   {{LL(0x21d1b602,0xb389e984),LL(0xb19669c8,0x806cbce5),LL(0x729a4109,0x53062f9c),L_(0x779cbefe)}, {LL(0x3bd3b5fd,0xa29b29ca),LL(0x6119b4c5,0xd386838e),LL(0xeb10107e,0x91c7a000),L_(0xd43d23aa)}},
   {{LL(0xa2b87da8,0xb2465acd),LL(0x954d5e2a,0x935ce923),LL(0xcc9f7e66,0xa5432c5d),L_(0xd4ffbeb9)}, {LL(0x02a952ef,0x686dc22d),LL(0xb3aa6ec4,0x272f791c),LL(0xf9a59b07,0xd8d7140f),L_(0x94e1344f)}},
   {{LL(0x63bd0826,0x7dd1f80d),LL(0x241f38ad,0x61111bf9),LL(0x72ffb980,0xff97f61b),L_(0xa3d97bc2)}, {LL(0x29de7bfe,0xd4e6e73a),LL(0xb4b19841,0xf25ef407),LL(0x5fd346b8,0xa14d6ec3),L_(0x366b72b2)}},
   {{LL(0x700b11e0,0xb6fe5b75),LL(0xe3a6e8d3,0xf936e40c),LL(0x4ed1cc1b,0x9f69a8a4),L_(0xe6f9dbe8)}, {LL(0xf722e6ee,0xa4d858ef),LL(0x00a65edd,0x39a6e0aa),LL(0x9733fc57,0xd4ab5744),L_(0x38f910bb)}},
   {{LL(0xf3acc856,0x338c1ad4),LL(0x4a664d78,0x9d2c02fd),LL(0xd6ec4c42,0xaee2829b),L_(0xededd826)}, {LL(0xb9a4651d,0x417e0805),LL(0x2c83b6ff,0x1f17faef),LL(0x65e8562a,0x54ffde80),L_(0x8972a5fc)}},
   {{LL(0x639320ae,0x49c2561a),LL(0xd2b48b48,0x05a4bc45),LL(0x38241d91,0x7dc5afe2),L_(0xc399df03)}, {LL(0x01fe9bae,0xc98d8a83),LL(0x666e5386,0xb711940d),LL(0x33895285,0xb43f7afa),L_(0xb804c149)}},
   {{LL(0x4d95806d,0xd0681518),LL(0x93423c26,0x537054cb),LL(0xc2eabb37,0xea12b99c),L_(0x9297a177)}, {LL(0x791e1723,0xddfee3a2),LL(0x4de65e15,0x21639c41),LL(0x7f3bad81,0xe43ae1fc),L_(0x7af9ba54)}},
   {{LL(0x1981ac7c,0x13762e73),LL(0x5afe2e21,0x76679941),LL(0xa21b9a1b,0xd03deaed),L_(0xd498e9f2)}, {LL(0xc88d2fa4,0x6ae69ee8),LL(0x749eed42,0x14c5a38c),LL(0x5f4c63df,0x95aa2f5a),L_(0x890e5e3e)}},
   {{LL(0xda52bcab,0xdea7e049),LL(0xbf3bb703,0x6a4a37e1),LL(0xc64a209c,0xddb56124),L_(0x6858318b)}, {LL(0xb063f261,0x9e3a0203),LL(0xf0dbd02b,0xf2de2a02),LL(0xa216442b,0x5ebba836),L_(0xc0cbf7a0)}},
   {{LL(0xc6f7be46,0x9bdf3b81),LL(0xe5bfa2bd,0x471a44c8),LL(0xef3da843,0x7b3913de),L_(0x107bba5b)}, {LL(0xd5502ac9,0xda5362cd),LL(0x81eecff8,0x51e4974d),LL(0x151060a4,0xd9ea79af),L_(0xe89b5db6)}},
   {{LL(0x176cf8de,0x310d44fc),LL(0x4509feca,0x8d199332),LL(0x25aa7254,0xd40c4268),L_(0xa958ef77)}, {LL(0x4c15f926,0x5dae221b),LL(0x71fc93e9,0xb47feada),LL(0xc3e3cb95,0x0f45640e),L_(0x6b7cb33b)}},
   {{LL(0x2033fbaf,0xc74fe8dc),LL(0x8c9ef406,0x8a528a60),LL(0xb6fe758c,0x04aa9cd6),L_(0x4f8b3877)}, {LL(0xbbebde8e,0x9f568191),LL(0x65aed657,0x88ffeed7),LL(0x85d0ebdf,0x7968220e),L_(0x04b5f3d4)}},
   {{LL(0x2004c53a,0x275a3075),LL(0x3a2a339d,0x686f881e),LL(0x8a7f68a5,0x5e03cf06),L_(0xd663fea8)}, {LL(0x5c2f3d32,0x83dd1968),LL(0x6151157a,0xbccda799),LL(0x620979ba,0x6d82e1aa),L_(0x3ac652d1)}},
   {{LL(0x4bd23c39,0x9f8e323d),LL(0xed2fa4ec,0xaa59f591),LL(0x554cf5e9,0x14820bd1),L_(0xc72e2e81)}, {LL(0x9ce69d25,0xe15fa1bd),LL(0x5f0c938d,0x6c3a0be6),LL(0xdf2d997a,0x33617c0a),L_(0x3087e812)}},
   {{LL(0x5df1d007,0x7f81aff5),LL(0xd6ea063d,0xbe630797),LL(0x631cf97e,0x46b66ca3),L_(0x3bfc14a4)}, {LL(0xd751c657,0x3a379a18),LL(0xaa5dfd05,0x87fb3d3b),LL(0xf3af9e8f,0x95b8ff29),L_(0x928c3c53)}},
   {{LL(0xcfeb637a,0xe65a2e0c),LL(0xa11c784f,0xe52dcc1f),LL(0xd33e59b1,0x2f4c50c4),L_(0xe15fe3e7)}, {LL(0x8e0494e2,0xe61d0a02),LL(0xdff24b95,0xa0c563a3),LL(0xd0e84cbb,0xc9c516d5),L_(0xbc6f5dd3)}},
   {{LL(0x6f4d5a61,0x828d8489),LL(0x55c5a959,0xe0f2a11a),LL(0xed8f7fcb,0x2be30973),L_(0x0b24dc92)}, {LL(0x835ad623,0xc98877f5),LL(0xd599613c,0x62b30316),LL(0x43a0bc0b,0x34b1b90a),L_(0xa8edd1ca)}},
   {{LL(0x25bb0089,0x126c6d31),LL(0xb92a7bd7,0xb4a63c35),LL(0xd9952b28,0xb5f3accf),L_(0xd21fa7b1)}, {LL(0x470e5d4b,0x6ac1deda),LL(0x7d8ef38b,0x7ddc00b8),LL(0x26d207ac,0xc01bf44b),L_(0x58c8cee8)}},
   {{LL(0x43e06d65,0x5d2a899b),LL(0x94d31b36,0x787531d0),LL(0x3050256d,0xd6d74149),L_(0xaa4efe0c)}, {LL(0xa5994d61,0x9b7a733e),LL(0x491dc4d2,0xab4d4ef7),LL(0x7540af9a,0x358ed8d5),L_(0x17dc58f7)}},
   {{LL(0x2cf09935,0xb13a4f1f),LL(0x4d20428f,0xaa77cf17),LL(0xf17077d7,0x27509ac6),L_(0x58852489)}, {LL(0xdeaa5893,0x64ad83cf),LL(0xc965f9f3,0xeb7c2c1c),LL(0x9a3903b0,0x2677b45a),L_(0x45839bea)}},
   {{LL(0x1b4a3efd,0x6c1eb10c),LL(0x24b30b08,0x30469bc0),LL(0x44f85d04,0x688bbe3b),L_(0xc74ccccb)}, {LL(0x40ec199e,0x94d1ce02),LL(0x14d9a43c,0x1e8a57b2),LL(0xc940698d,0xf3a40155),L_(0xefe2f127)}},
   {{LL(0x7ced76b7,0x9378bdc8),LL(0xfbd86f1e,0xd3de6697),LL(0x38c7a9cc,0x965226a4),L_(0x434dbb0c)}, {LL(0x1d9de00e,0xb9259728),LL(0xc36bf7c6,0xe5e2b0b4),LL(0x0a12f12b,0xb21834ca),L_(0x0134eeee)}},
   {{LL(0x2cbf4902,0xad55e742),LL(0xa30d570d,0xceb87a93),LL(0xdaaa178c,0x525f0059),L_(0xf38af4e6)}, {LL(0xf041ead7,0xa001bb9f),LL(0xa05f60e2,0x951d1144),LL(0x701418ea,0xfd38f820),L_(0x27e433c5)}},
   {{LL(0x4f2b74e1,0xcf1f6417),LL(0x3570870f,0x2d368dff),LL(0x85875e05,0x96e8ab61),L_(0xefc6f006)}, {LL(0x139ced83,0x0206e9aa),LL(0x6f6540da,0x1f1ac51c),LL(0x2a1b80e0,0x9caa872e),L_(0x895193a8)}},
   {{LL(0xbe7a3924,0x40d4de6c),LL(0x288f99a9,0xa9bbbd0c),LL(0x74a2cf89,0xad2e6432),L_(0x52ddf73d)}, {LL(0x50719c7b,0xef978612),LL(0xbb41a387,0x44b2c373),LL(0x319f4bf7,0x40e92907),L_(0x9884b224)}},
   {{LL(0xcb2e3230,0xe2fde7bc),LL(0x282179db,0xb286e49e),LL(0x7897f73f,0x14a26c02),L_(0x07ab717c)}, {LL(0x0974c087,0xb3e012dd),LL(0xa65b35a3,0xb211a759),LL(0xb888ddca,0xc0879433),L_(0x8819e222)}},
   {{LL(0xa98ed0e0,0x32acb172),LL(0x9f902cc1,0x30fb222b),LL(0x2ea4f419,0xec378b71),L_(0xd036aa23)}, {LL(0xfc7c459e,0x0f8fe474),LL(0x3e16c057,0xea879818),LL(0x134bd529,0xee28b685),L_(0x867d1b08)}},
   {{LL(0x7f52225a,0x49cdbae9),LL(0x5c1f02d0,0x22253ab0),LL(0xd83d7d9e,0x1b0a0f31),L_(0x2c287cd4)}, {LL(0xa7c8e376,0xf27c6df7),LL(0x74534873,0xb5e78384),LL(0x5a696a14,0x8118c158),L_(0xfc06292e)}},
   {{LL(0xa94b2f8a,0x5e909156),LL(0xbd1a69f7,0x0e89f809),LL(0x703cb9be,0x5a474912),L_(0xcedd645a)}, {LL(0xfc513c30,0xdeeac8ab),LL(0x2ed53772,0x8d081de8),LL(0x5e6b86ab,0xc7b731bc),L_(0x5ce1f8a0)}},
   {{LL(0xa68e3ae6,0xcc023bb5),LL(0xdf3b4492,0xd2e02e8e),LL(0x87e45b7d,0x0ae46dae),L_(0xe1f7eac7)}, {LL(0xe31f07c6,0xb2f3733a),LL(0x72718cd2,0x58504950),LL(0x85e9b624,0x626b0871),L_(0xc20aa08f)}},
   {{LL(0x3383e911,0x44129030),LL(0x8b7e591b,0x1fa18b39),LL(0xc7753dad,0xf476b49f),L_(0x2fc4d6c3)}, {LL(0x43355b4f,0xe26efb18),LL(0x4dd384fc,0xb345e6e0),LL(0x478d3761,0xc574295c),L_(0x421851ef)}},
   {{LL(0xc576e693,0x92253e01),LL(0x5a61d65d,0xa9b92c8f),LL(0xd4924563,0x1e712886),L_(0x2dd1532f)}, {LL(0x502854fc,0xa3f1ded9),LL(0x39cdfbfc,0xb3758c76),LL(0xb32cde92,0x57960c59),L_(0x40ad0b76)}},
   {{LL(0x7c7eb74a,0x33392cdf),LL(0x52977c85,0x3035bc66),LL(0x147af937,0x6dd913b5),L_(0x4ec4b1c5)}, {LL(0x3cd27826,0x699ab1e8),LL(0x438d69df,0x674459d8),LL(0xdf94b6a0,0xbc60bb02),L_(0x60dae203)}},
   {{LL(0x98237ec1,0x01cf54f3),LL(0xc3cf28be,0xc86cd9e7),LL(0xfa448ad2,0xf8c1cf70),L_(0x59baadb3)}, {LL(0x6e0c6ab4,0xb0a595fa),LL(0x60b75e6c,0xaef06103),LL(0xb92101cb,0x99833d0b),L_(0x4be8d444)}},
   {{LL(0x76a063e0,0xedd8ac9f),LL(0xcb571bbf,0xfbb746f6),LL(0x5a13acc6,0x68203714),L_(0x79914deb)}, {LL(0x698725d3,0x5c3bbaeb),LL(0x211e58cd,0x8019d5ed),LL(0xdef6ac5e,0x776cb363),L_(0x93cb4804)}},
   {{LL(0x2c9e2d1e,0x7d49d5cb),LL(0x55e82125,0xb44a7110),LL(0xa13cee20,0x7e46156a),L_(0x7d7e5719)}, {LL(0x293b532f,0x629c9b2f),LL(0x5fd28ac1,0x4766f7bb),LL(0xdc393f35,0x8ea6f6e3),L_(0x289c68b1)}},
   {{LL(0xf29e017b,0x63596f57),LL(0x31f49f27,0xb9a200a0),LL(0x7c86680a,0x0b1f95fd),L_(0x02fc0e10)}, {LL(0x61a4ad54,0x8f2843cd),LL(0xca4cab38,0x5392a44e),LL(0xace069cf,0xf56c80ff),L_(0x270d9677)}},
   {{LL(0x8e969ba3,0x6352c6f5),LL(0x67e09224,0xd3d5c18a),LL(0x5be06f9a,0x14efc35d),L_(0x38c6a0a2)}, {LL(0x0263394a,0xbe2d9aa3),LL(0x6d0b11a9,0x72815d01),LL(0x6c28404f,0x5538c4c6),L_(0x0af532d8)}},
   {{LL(0x27afc27a,0x588f5f00),LL(0xb4277028,0x7fd91a63),LL(0x2477f29e,0x5fe5d31d),L_(0x68c2db2a)}, {LL(0x3c6bab6a,0x2a5ebb99),LL(0x7b9c3f49,0x8847424d),LL(0xbf734579,0xbf266e3b),L_(0xfe65a7a9)}},
},
/* digit=32 base_pwr=2^224 */
{
   {{LL(0x390f347f,0xb7478766),LL(0xeaa013ed,0x69774be3),LL(0x1cf3e562,0xe8c99b44),L_(0x142c4251)}, {LL(0x6addc55a,0x5ac149fb),LL(0x9c824754,0x251be20c),LL(0x6b1233b2,0xb9c09eeb),L_(0xc5afba96)}},
   {{LL(0x772d2e43,0xb7c40ccd),LL(0xbf4d6687,0x4a5ac532),LL(0x88025f52,0x80c953d4),L_(0xe3dc6d83)}, {LL(0xa265b97a,0xb7668ae2),LL(0x50faab59,0xa19f0872),LL(0x235fc5b5,0xac299c46),L_(0x1378e170)}},
   {{LL(0xe16661d5,0xb43ccb67),LL(0x47792f74,0x13fe8d1f),LL(0x3dbeda1c,0xb688499b),L_(0x5b9b5ea1)}, {LL(0x6ec4def3,0x6f26a9f6),LL(0xc4515542,0x45056f3c),LL(0x8e4ebca1,0x1f557c40),L_(0xd6441259)}},
   {{LL(0xa24e4556,0xea6ea3c9),LL(0xc6c1108f,0xb4010098),LL(0xcc83bf8e,0x2c30408f),L_(0x7aa17b0c)}, {LL(0x89219d24,0x06aed6af),LL(0xc003650d,0xbe94120f),LL(0x72d853fe,0x66238f2e),L_(0xf46a15a7)}},
   {{LL(0xf0756f21,0xf16a11ac),LL(0xccc9dc6a,0xc74dff4d),LL(0x71ab7d63,0x54c72750),L_(0x5a001c63)}, {LL(0xcf7f1e0d,0x226cb517),LL(0xb6f26d84,0x09440d66),LL(0x3fc36947,0x9f9ee30a),L_(0xa1549b9d)}},
   {{LL(0x9d51f064,0x4a2b2246),LL(0xa65e1e3b,0xfddd99d1),LL(0x671cf2f3,0x6b9d08f0),L_(0x162efd24)}, {LL(0x4fa36b3d,0x3316fc3e),LL(0x84ae1987,0xd5adf51b),LL(0x7d0dc886,0xdbe094f5),L_(0xc7b0730d)}},
   {{LL(0xb0e743f2,0x1bb62b05),LL(0xe8e8411e,0x6fd286a0),LL(0xd192f8c2,0xe45573f0),L_(0xc1f3a5f2)}, {LL(0x96b5af59,0x26419174),LL(0x6c1325f6,0x50d9352d),LL(0x2f81e001,0xa91d228e),L_(0x0ff7a9b2)}},
   {{LL(0x1a7e3bbc,0x88e39d85),LL(0x42262241,0xc7cb4750),LL(0x3ad07ade,0x918fdfa4),L_(0xcb771a46)}, {LL(0xd39e0dbc,0x18ba36b7),LL(0xeb29d84d,0x3f5d4732),LL(0x1383d57f,0xf57833f6),L_(0xfda5e179)}},
   {{LL(0x024fa042,0x47af4689),LL(0x6f69c44b,0xe8d6a5c3),LL(0x5e4d4fa0,0xd6821d55),L_(0x46eb8384)}, {LL(0xc59ebc7b,0xcdeb889d),LL(0x871d5809,0xeaf69758),LL(0x1029a1da,0x5cf9245d),L_(0x1bcc2546)}},
   {{LL(0x491af658,0xfd8a4d1e),LL(0xdb1de26b,0x677407f1),LL(0x8aa6deb0,0x12fee6da),L_(0x1e010832)}, {LL(0xeb1d6fd4,0xd9702487),LL(0x9e928c81,0x0d5e0508),LL(0x79014b3a,0x844adfba),L_(0xa7623853)}},
   {{LL(0x6333a497,0x2e0124e6),LL(0x2b6dbd37,0xacc3158a),LL(0xcb4fc42b,0xd7cdda0b),L_(0xd5d8f543)}, {LL(0x1a1d4227,0x485584ae),LL(0xdd6eb7f3,0xa25706fd),LL(0x2e41e3f8,0x65859cc9),L_(0xeca30c9e)}},
   {{LL(0xef4c8e5d,0xd5f8805d),LL(0x66979e6c,0xc2437937),LL(0x1b2ea5ec,0xf8ebb830),L_(0x416a9eb2)}, {LL(0x87c226a3,0x6e8ad20c),LL(0x15ff6e9d,0x784bfc8d),LL(0x0865593f,0xee384b9b),L_(0xe0eeda2c)}},
   {{LL(0x148e4a75,0x642f55af),LL(0xbfeb8487,0xc162ad9e),LL(0xfa3ad249,0x7145f39b),L_(0xd0f1abf1)}, {LL(0x4053c61b,0x4defbbd5),LL(0x893a0505,0x1244854d),LL(0xf3e2f26d,0xf54c27a7),L_(0x6f12b38f)}},
   {{LL(0xa1ac9684,0x5d26514a),LL(0x7cad1aa3,0x3b5e65fa),LL(0x74faf86d,0xfeab4314),L_(0x645cca78)}, {LL(0x3c8ed8d4,0x43f00055),LL(0xb72096cb,0x8df847ba),LL(0x7a1950ed,0x071d954a),L_(0x5337fbf2)}},
   {{LL(0x33140bfb,0xb32000f6),LL(0x5f6dea95,0x27eab135),LL(0xf4446577,0x19091ccf),L_(0x74118cb1)}, {LL(0x765e46b5,0x9210256d),LL(0x9aacef2c,0x3ce74727),LL(0x1ca9b6d8,0xb1ab02c7),L_(0xa870996c)}},
   {{LL(0x36f1ae02,0x308cbf7f),LL(0x552396fc,0x748ef11b),LL(0x7b2fda3a,0x4146f800),L_(0xaebf97a3)}, {LL(0xb4291275,0x61fb1316),LL(0xf1c9012c,0x894b0da7),LL(0x0142bd07,0x9b191445),L_(0xbab9da1b)}},
   {{LL(0xe1f1f426,0x5e508e4a),LL(0xfa8ca29d,0x4b94de05),LL(0x27a0f674,0x4cfdd6de),L_(0xee0206bb)}, {LL(0x4b46ce88,0x978ed51e),LL(0xa6bbc41f,0x6bee5364),LL(0xe52c7867,0x006c7072),L_(0x32478f51)}},
   {{LL(0x5382793a,0x217f1202),LL(0x04851e7d,0x3c689851),LL(0xd7767c09,0x75b3d632),L_(0x4c08238d)}, {LL(0xab3ed287,0x3858c44e),LL(0xcd97e8ec,0x0c103d5b),LL(0x62318c47,0x93355b9b),L_(0xe2405794)}},
   {{LL(0xe36e5cd7,0xde835c5d),LL(0x92a463c3,0x1eba7f4e),LL(0x3b10b96a,0x4b4acd48),L_(0x5f5bd93b)}, {LL(0xd4e11813,0xf537b955),LL(0xedc9c8f4,0xf31bab45),LL(0xf2362c8d,0xe58a0000),L_(0x970cf697)}},
   {{LL(0xc005e339,0x8e7b1729),LL(0x882e0b3e,0xf8b77b70),LL(0xbf2d6013,0x490fbccd),L_(0xc0848551)}, {LL(0x5c4916a2,0x73f7a7d8),LL(0x17b850e8,0xde0ad5e0),LL(0xfcc68ded,0x9cbe988b),L_(0x8d09bdd2)}},
   {{LL(0xf00570c3,0x6266d7a9),LL(0x7f0f6292,0x4291a62a),LL(0x47c40355,0x1cfc61e3),L_(0x7a59e7d3)}, {LL(0x4ce6c101,0xbddfbabb),LL(0x84b7a613,0xe92b59c0),LL(0x89396fd0,0x0f5147ce),L_(0x2b7f355a)}},
   {{LL(0x9d82dd91,0x8af9951f),LL(0x4d2ceb91,0x78598c17),LL(0xcc451e87,0x6d59d3b5),L_(0xc116fdef)}, {LL(0xd20d8318,0x79ead74d),LL(0x813a5852,0x8ba20d33),LL(0xc12ffcbc,0xcd67731a),L_(0x3fdf17e1)}},
   {{LL(0x5e3aff2a,0x764582dd),LL(0xefaccc05,0x05c889a6),LL(0x63018ab6,0x6f606211),L_(0x6b706f5e)}, {LL(0xac9cf98f,0x2663583f),LL(0x2a01814c,0xde1ab3e0),LL(0xe29eede6,0x756a8f01),L_(0x10f5f095)}},
   {{LL(0x363dd819,0xa38d34f9),LL(0x591f8fe0,0xc1ebeead),LL(0xc31b8775,0xae2b4a7d),L_(0x55de9380)}, {LL(0xd500cd22,0x63877737),LL(0x566cf0c3,0x04743650),LL(0x1ad0462c,0x5c1a2a83),L_(0xcfd11aad)}},
   {{LL(0x4c679d95,0xbc66bc8e),LL(0x9f39feb3,0xd84b8c56),LL(0x21f0dec8,0xd2ff18e6),L_(0x99e3e6c4)}, {LL(0x08b2eebe,0xcce55b7e),LL(0xf3fabc3c,0xb0ed283f),LL(0x5c869424,0x4f7a9f81),L_(0xc660df04)}},
   {{LL(0xd868008a,0x302b25d6),LL(0xfbdef76c,0xd9a10be6),LL(0x97f6ff80,0xcc0decf0),L_(0x26e1f975)}, {LL(0xbb6408c0,0x4f37e254),LL(0x61eee678,0x9682814e),LL(0x6a0d8fc5,0xd84014df),L_(0xb7e3a2d4)}},
   {{LL(0xaa81602b,0xde3d73e7),LL(0x3178216e,0xe66f2866),LL(0x2f89efcd,0xdcbe0246),L_(0xb937b3c2)}, {LL(0x72d505b5,0xacaa30cf),LL(0x7f06975e,0x66c190f4),LL(0x714b6102,0xd23f9579),L_(0xc1321142)}},
   {{LL(0x3c07cee1,0x8271dfd0),LL(0x6e2e57bd,0x85e2b2d2),LL(0xec736ab9,0xbcfb4116),L_(0x1cc014d1)}, {LL(0x57b0096e,0x02178015),LL(0xbcc0c7bd,0xda722aae),LL(0xdf2643e7,0xedc952bb),L_(0x36ab26fe)}},
   {{LL(0x76041355,0xc562a324),LL(0xccd2af21,0xa955496c),LL(0x8cbd4ea2,0x9f800082),L_(0x7bbad5e0)}, {LL(0x53fd9c07,0xbfd65f54),LL(0xfa87a290,0xd0f0709a),LL(0xd25d631b,0xe063926d),L_(0x0f3d4539)}},
   {{LL(0xd48fa29e,0x6bf5b024),LL(0x1a63c568,0xea32fa48),LL(0x7bad1c42,0x62df4304),L_(0x38b54b11)}, {LL(0x97d1cfa8,0x94995152),LL(0x7f7b64f5,0x9e00e462),LL(0xd25d1128,0xbdbbd673),L_(0x4703bb51)}},
   {{LL(0x304a08ec,0x2509ad1a),LL(0x9885b7f4,0x9877088f),LL(0xf70447a7,0x9bfc053a),L_(0x1b77852c)}, {LL(0xaf70aa1f,0x7dc9ba14),LL(0xa43807b7,0xff3589c4),LL(0x02c669a5,0x220e1871),L_(0x991d8093)}},
   {{LL(0x86209a7e,0xc434b863),LL(0x6c74aee9,0x10b6d0ca),LL(0x072efedb,0x19982c91),L_(0xac91d19f)}, {LL(0xbfb13837,0xbc64a3c6),LL(0xcea49a1b,0xab8a0475),LL(0xe25c1282,0x37f9ab3c),L_(0x57709b27)}},
   {{LL(0x422a938d,0x63a0bab7),LL(0xbdede240,0x2dffb3e9),LL(0x8ec0a031,0xbbb0cddc),L_(0xf6b96225)}, {LL(0xfa46994b,0x3dff08ca),LL(0xf46b51e6,0xb21694f6),LL(0xfac7173a,0x3f2c5530),L_(0xe5090eda)}},
   {{LL(0x5e18c4f3,0x585b98ed),LL(0x4cb0c03a,0x2784af9f),LL(0xb6dc9a2d,0x126ee071),L_(0x3331172f)}, {LL(0x56db8dc6,0xba3aa739),LL(0x71ac411d,0x16a260a5),LL(0x816a4f1b,0xa88c4198),L_(0xe409dbda)}},
   {{LL(0x5fd428ad,0xccb6c49d),LL(0xa20ba7c3,0xe7066136),LL(0x6782fea2,0x83b7b5ce),L_(0xf28c836e)}, {LL(0x4fe78dbc,0x64f82fd1),LL(0x523afba8,0xe21bc6e2),LL(0x024ddda9,0x65d29391),L_(0x4cec16d6)}},
   {{LL(0xfd04b964,0xaff84dd9),LL(0x1594f62b,0x4eb6067b),LL(0xa8de453c,0xeb65c1b7),L_(0x30a103ff)}, {LL(0x6903c7e6,0xf76afad7),LL(0xa91cd374,0x2f9fd25b),LL(0xa33359fe,0x0b99c0c0),L_(0x9015867f)}},
   {{LL(0x8a65619a,0xb366fcc7),LL(0xdde98289,0xec03d9a8),LL(0xe45737f9,0x30c2ee70),L_(0xa6bb364a)}, {LL(0x67ffeeef,0xe00c866a),LL(0x8177a004,0xac4b7104),LL(0x49205953,0x0106e5ac),L_(0x5d4c1327)}},
   {{LL(0x2e836cb7,0xdebcecc7),LL(0xfdc46d5f,0x9b0df65d),LL(0xa125951c,0x5633f2c4),L_(0x04c2afc1)}, {LL(0x89476731,0xacd14df8),LL(0xb5f970f1,0x9f48abbc),LL(0x0973ccd0,0x236b3580),L_(0x02b2c84a)}},
   {{LL(0x88394318,0xcb13811c),LL(0x0d2dd5a3,0x9bbda12c),LL(0xd8b5d0f6,0x4f399d9a),L_(0xa46887b9)}, {LL(0x2ae73578,0x10165160),LL(0xddf54326,0xc04d1a88),LL(0xf080be9c,0xeee34275),L_(0x047237fe)}},
   {{LL(0x7aab908a,0x11fceb11),LL(0xfaf4fbdf,0x96bea707),LL(0x66bfd60d,0x442c2a46),L_(0x28cf0358)}, {LL(0x6d51778b,0x4cebfe24),LL(0x6c01d714,0xaf283f0e),LL(0x11d66f5a,0x8cc29c26),L_(0xcf7771ef)}},
   {{LL(0x76ad7d8f,0xf90d16be),LL(0xb89933d4,0x2a97f299),LL(0xc8f0695f,0x5a6af165),L_(0x82274dbf)}, {LL(0x1826ea5a,0x225b8bb2),LL(0x5e138d61,0x16c62f5f),LL(0x3e2273d1,0x6b42f9f3),L_(0xaa4b331c)}},
   {{LL(0x5659f40e,0xa9d80f3d),LL(0x6773ac5a,0x6ed38547),LL(0x970b6067,0x78700052),L_(0x2f129047)}, {LL(0x4d7032cd,0x0393eefa),LL(0x04368569,0x98e16738),LL(0x5180fe6f,0x3014d2a0),L_(0x6bd99185)}},
   {{LL(0x1988a375,0x299011dd),LL(0xe9270a23,0x40dded09),LL(0x54a9528b,0x7af57c36),L_(0x8e613981)}, {LL(0xb388faf2,0xff5fc081),LL(0xd7dd7f17,0x2ccd8f6a),LL(0x421cb6c5,0x590b13ce),L_(0x8f9e6852)}},
   {{LL(0xe7870572,0x6513f9c8),LL(0xd5bf1a5d,0x2f88c206),LL(0x1942fc9b,0x6ec5f13f),L_(0x70273957)}, {LL(0xf1998de6,0x5bb80bbb),LL(0xcededd82,0x923e0aaa),LL(0xa7c7ee32,0xfa06b05d),L_(0x86a8080f)}},
   {{LL(0xe016efc2,0x9510adb0),LL(0x8a998afb,0x9a9168be),LL(0xfaf28d30,0x0863e4f3),L_(0xdcf64ed3)}, {LL(0xa0beb82c,0x1ea72a0f),LL(0x78d5ce7b,0x932f17af),LL(0xafe82b0c,0xde25a969),L_(0x1f0a50fe)}},
   {{LL(0x496b6eb7,0x3d5e21a1),LL(0xddb5610d,0x36b95de4),LL(0x435cd100,0xcc34b698),L_(0xf7218db6)}, {LL(0xf643607b,0xc23a0cb7),LL(0x3b787bdf,0x2299d421),LL(0x71fcc010,0xb0e2a897),L_(0xd6c69c33)}},
   {{LL(0x0241244a,0x1cd5d5da),LL(0xe94a402b,0xf0ca4eca),LL(0x26944f06,0x737d6c5b),L_(0xd0af2018)}, {LL(0xc1e2ad0c,0x1b523220),LL(0xa8a984b3,0xc8ebc391),LL(0x39663eef,0xbf4b3e0e),L_(0x0297dfd1)}},
   {{LL(0x0ac5f597,0xc8d20d13),LL(0xf915a4c4,0x00b7e16a),LL(0x2921fb10,0x6b467b73),L_(0x226421aa)}, {LL(0xf2ede5a3,0xf4bdad8c),LL(0xb5512d8c,0xce15cba2),LL(0x2df92000,0x4979cf5a),L_(0xd5159856)}},
   {{LL(0x7ac29a8f,0xffb25428),LL(0xad4ff433,0x6903617b),LL(0x22c48ae6,0xdcdd2832),L_(0xee468cb2)}, {LL(0x6769e2fd,0xad56cbbc),LL(0x1ac2be8d,0xc8f4bea8),LL(0x11bdfbf2,0x4c080ea0),L_(0xd326fcc1)}},
   {{LL(0x264b7510,0x3c2a21b2),LL(0x4c900a2e,0x10bf375c),LL(0x540b6e6d,0x40c9b00d),L_(0xf0db08ca)}, {LL(0x1f0e8e4f,0x75ea8b18),LL(0x23e0167d,0x4ae7f6f7),LL(0x3ce76be8,0xc2578bd9),L_(0x274ef866)}},
   {{LL(0x2d2787bd,0x06e96b07),LL(0xa0cc2074,0x3859e8e4),LL(0x2181d378,0xbc50ac87),L_(0x46b62277)}, {LL(0xc39d3dd1,0x66a073ee),LL(0xbc771f86,0x2bf462de),LL(0x47275780,0xa7216ad9),L_(0xd3c1c667)}},
   {{LL(0x359b7048,0x37e5b8f4),LL(0xfea02084,0x16126c46),LL(0xe9a5f0b8,0xbf587847),L_(0x53933d2b)}, {LL(0x1c7944f6,0x492b64e3),LL(0x8257b251,0xf2983977),LL(0x8d9ec843,0xfe9686a0),L_(0xeeea3972)}},
   {{LL(0xac24ef0b,0xb93a8f55),LL(0xf5712604,0xfa720e89),LL(0x6981acc8,0xf73a981c),L_(0x86cc6e25)}, {LL(0xd19071bc,0xa6c75878),LL(0x18cf2a94,0x15519e91),LL(0x5737dd50,0x4c1f9f0e),L_(0x3114afeb)}},
   {{LL(0xec690133,0xbbc2c60e),LL(0x19e92b39,0x1a4e7efe),LL(0xb3de20d1,0x43d40348),L_(0x693597af)}, {LL(0x9b86b502,0xa7c941f1),LL(0x4568fdc8,0x38d7607d),LL(0x92d2afed,0x50493b74),L_(0xa510adc1)}},
   {{LL(0x29a4b55c,0x60867e2e),LL(0xab4d38b7,0xc6a2be65),LL(0x0b8a018f,0x2c774449),L_(0xc11883d8)}, {LL(0x3703e4c4,0xa50bce89),LL(0x9bcfd81b,0xc601a985),LL(0xf5811bc5,0x80f1e9c2),L_(0xe9b876e5)}},
   {{LL(0xb21301b8,0x368a0741),LL(0x7cf8b51b,0x57d524cc),LL(0x85f0eb37,0xb5926249),L_(0x08e70a34)}, {LL(0xdd6d17da,0xe9a1940d),LL(0x4d61fee9,0x905e7d89),LL(0x24af91e7,0xa85a5970),L_(0x41f4907d)}},
   {{LL(0xa38b5cfe,0x33182545),LL(0x7b2ef570,0x10b61fe2),LL(0x2a780101,0x8d1180df),L_(0xab954cb9)}, {LL(0x8b6a7232,0x88d81ace),LL(0x0d62a965,0xa35ed73b),LL(0xaa16f924,0x2bc7cca4),L_(0x2d99dcc8)}},
   {{LL(0x3c5881cd,0xb1395709),LL(0x21c56351,0xe2ce53b6),LL(0x41b20f4e,0x067af3fb),L_(0xbd9ab15f)}, {LL(0x2f06e7bc,0x1523f3df),LL(0x9de5e7e4,0xb26c33fb),LL(0xef0cf37f,0x6c9e48d4),L_(0xc9fc2851)}},
   {{LL(0xcb75ecba,0xb81124f0),LL(0xea0ed825,0xad146b33),LL(0xf4f08526,0x8e19f5c1),L_(0x820a4f79)}, {LL(0xa1d54353,0x6eaedecb),LL(0xefe607bc,0xe5ecca5d),LL(0x75968ada,0x21596aa9),L_(0xcdf9ca14)}},
   {{LL(0x4cdb6b68,0xdaef28c2),LL(0x2be4550b,0xdd0b6f86),LL(0xee673998,0x8162c783),L_(0x4e1fb36c)}, {LL(0x58a6ed93,0xab7e3385),LL(0xb682150c,0x3081637e),LL(0x02a0814f,0x9c692620),L_(0x3bb22aea)}},
   {{LL(0x6bc24027,0xa03f91a5),LL(0xd356a2bd,0x0f664f1a),LL(0x04625b5a,0xf350f847),L_(0x2eacfb01)}, {LL(0x4e04629e,0x63af93f4),LL(0xba005c26,0x8ec3c009),LL(0x7964a6f9,0xc56edb4d),L_(0x14b146d7)}},
   {{LL(0x64910001,0xcae5ffdd),LL(0x51fbb8f7,0x657b3f67),LL(0xa961e968,0xbe43f447),L_(0x2a774b0b)}, {LL(0xbfbe3d28,0x4e174bef),LL(0xb12fde0a,0x11d92d90),LL(0x85b59e4f,0x421eac26),L_(0x4c480a51)}},
   {{LL(0x83f1230c,0xc3a1a862),LL(0xfe8ceef6,0xbef83f8d),LL(0x0469540a,0xd231a2a4),L_(0x0828f4c0)}, {LL(0xc896c8cb,0x2c034efa),LL(0x1744d906,0xc9a26995),LL(0x2a85a8d6,0x28f5775a),L_(0x58717292)}},
   {{LL(0x12ab27ce,0xf12dc791),LL(0xbcc5e464,0xec36c787),LL(0xb881c819,0x2bfca9a4),L_(0x7e60531f)}, {LL(0xd8af4418,0x28ea4eef),LL(0x9ef6ada6,0xe0a7044d),LL(0xf56fbcf0,0x534e2057),L_(0x9857c41f)}},
}
};
#else
const __ALIGN64 P224_POINT_AFFINE ec_p224r1_precomputed[33][64] = {
/* digit=0 base_pwr=2^0 */
{
   {{LL(0xbc905227,0x6018bfaa),LL(0xf22fe220,0xf96bec04),LL(0x6dd3af9b,0xa21b5e60),L_(0x92f5b516)}, {LL(0x2edca1e6,0x05335a6b),LL(0xe8c15513,0x03dfe878),LL(0xaea9c5ae,0x614786f1),L_(0x100c1218)}},
   {{LL(0x4ca9a1ed,0x5650df9b),LL(0xbe27f7a3,0x2a0f1689),LL(0x10c911f7,0xcafb50f5),L_(0x18dd00ac)}, {LL(0x02aac79c,0xc90ae186),LL(0xe72e600f,0x76cc1019),LL(0x0c84ced0,0x5cabb880),L_(0x6a5db1a2)}},
   {{LL(0xe37710d8,0xe0b02223),LL(0xab22ca96,0xc3cf1c5a),LL(0x00505bb6,0xaf8b6496),L_(0x799eb566)}, {LL(0x9a73b174,0x04144e62),LL(0xe434f88f,0x357940ab),LL(0xafe10c35,0x6dfa492c),L_(0x8c1b1da3)}},
   {{LL(0x0a72faf7,0x0860856c),LL(0xc03cd01a,0xcb021705),LL(0x4256e88a,0x89f0c6a0),L_(0x7bf0ef17)}, {LL(0x6743e274,0x98933269),LL(0x64b26855,0x2058f0ba),LL(0x7ee104d3,0x8c87db62),L_(0x73e0d4c7)}},
   {{LL(0x0b10cc51,0xf8dfc283),LL(0xac3c8962,0x975a451e),LL(0x39211b77,0xf7f4fe95),L_(0x2bd053a8)}, {LL(0x2b552e74,0x61a9e695),LL(0xc350ef91,0x11bcf037),LL(0x42b54f79,0xc858da82),L_(0x9b052162)}},
   {{LL(0xde577c0e,0x1a8aec73),LL(0xe82df964,0xe0cd01d3),LL(0x1addb3a1,0xf25fe17c),L_(0x21b6fe25)}, {LL(0x5fae6c33,0x0f5c7709),LL(0x327f8848,0xf2d2c41c),LL(0x9b6731a6,0x162f9e1c),L_(0x2383dee4)}},
   {{LL(0x90462821,0x68646e35),LL(0x5048b145,0x1e2526ba),LL(0xc81df812,0x7ed50814),L_(0x7665453e)}, {LL(0x610f2186,0x1fb32889),LL(0x9d0eba26,0xed8ceeed),LL(0xce5eb25e,0x0e5952e3),L_(0xa229bf84)}},
   {{LL(0xb4c9d866,0x7bd696a2),LL(0x1de5d041,0xa3bebfe9),LL(0x8e4a5712,0x1b59038c),L_(0x21e8f814)}, {LL(0x1bc1ff37,0x7558734a),LL(0x38a20402,0x0976258f),LL(0xc4e19066,0x2199c364),L_(0xd65a0dfb)}},
   {{LL(0x1c1ecc78,0x743eda4a),LL(0x9f0b4f26,0x2f993a4f),LL(0x94f5e9ba,0x79d4095a),L_(0x846b2c39)}, {LL(0xe07e55b7,0x909bf3f1),LL(0xbacf5e6a,0x77e1db6c),LL(0xe733d627,0x5114ac9d),L_(0x70889bf7)}},
   {{LL(0xb808145d,0xb4f2672c),LL(0x328e0ac0,0x3e8abf6a),LL(0xc1f86e2c,0x9d0cdd53),L_(0x5ac34abe)}, {LL(0xccddcf00,0x02883b6f),LL(0x225faa54,0x873d7e4c),LL(0xdbc0f8ee,0x2a687506),L_(0x722981c9)}},
   {{LL(0xac358ca9,0xe1bdd5ca),LL(0xde5d2c4d,0xd0dee305),LL(0x0f060758,0xd6d89093),L_(0x9397da27)}, {LL(0x649ea143,0x08181c2c),LL(0x79888920,0x6c63713a),LL(0xc1182bf0,0x45e48d79),L_(0x0e48dd82)}},
   {{LL(0xf6495da2,0xa37b7441),LL(0xed4d0ea3,0x2b0b8e19),LL(0x571d38d9,0x517af941),L_(0x70792626)}, {LL(0x1305b613,0x8cf21835),LL(0x208ffd66,0x5f59deb1),LL(0xbc243af0,0xa71404ae),L_(0x6d228d4b)}},
   {{LL(0x489c8db3,0xffbe563f),LL(0x83630fe6,0xbc99fb0a),LL(0x68ade63c,0x398eacf0),L_(0xc5e095c8)}, {LL(0x83fdacb6,0x800df824),LL(0xa3ffe74f,0x442ed616),LL(0xb4dc6bf8,0x3f8060b4),L_(0x12ab633c)}},
   {{LL(0xf4109222,0xcdbd6eec),LL(0x0e9441ba,0x1ef814eb),LL(0xf4550d2e,0xb3a9b5e4),L_(0x47c1182a)}, {LL(0x1544e90f,0x3f51f9f9),LL(0x9ed249e3,0x825b5794),LL(0x6516bbc3,0x3d1419d2),L_(0x92de7285)}},
   {{LL(0x55bfd51a,0xb16ed1af),LL(0x464afb0a,0xd91345bb),LL(0xc5e26f62,0x64a35d22),L_(0x16880cc7)}, {LL(0x05015843,0x70203aa0),LL(0x11e4ffa1,0x05a83a63),LL(0xdaf3972b,0x9422f10e),L_(0x5dbc0e12)}},
   {{LL(0x6a4bd1fa,0xb3f66de6),LL(0x2ec3ac7c,0xfdf9a0e9),LL(0x471a2a7f,0xb9c51b5f),L_(0x8c4bc81e)}, {LL(0xfa29681b,0x9c409bd0),LL(0xd979b98e,0x169405c2),LL(0x48be80cb,0xf22d3fc0),L_(0xbba8bdbd)}},
   {{LL(0x86ade48c,0x79cec3fb),LL(0x3f83b939,0xb9622253),LL(0x5523244b,0x538d543e),L_(0x07ba7ce6)}, {LL(0x12bb38c0,0xb4e20b8c),LL(0x70a52728,0xe6873448),LL(0x2a2e7758,0x2954321a),L_(0x07a8f3fd)}},
   {{LL(0x3591891f,0x27a72a86),LL(0x845865a8,0x482460f5),LL(0x70114363,0x3b204e52),L_(0xb84ab43b)}, {LL(0xd5f98054,0xc4a1a96b),LL(0xa4e237e3,0x0c0e97bb),LL(0x6bc4b6e9,0xfb212c8c),L_(0x335f11e1)}},
   {{LL(0x716b6ade,0xea007122),LL(0x2a638e5a,0x08a75055),LL(0x4e35e1f6,0xe28f8fa3),L_(0xe40528c7)}, {LL(0x7393ad17,0x4e27959e),LL(0xd8f5fd90,0x433c708c),LL(0xe5b33144,0x4c003b64),L_(0x6c3deb98)}},
   {{LL(0xc57bd47b,0x238addb6),LL(0x8d685d17,0x703c82d4),LL(0xa0691b17,0x14a5d58e),L_(0xcd7fb5fc)}, {LL(0xf9cefcb6,0xfd4764c3),LL(0xbb38397b,0x7b23c9fd),LL(0x23cf8d21,0x1d7b0c07),L_(0x7dafc83c)}},
   {{LL(0x869c61a2,0x8d8b998b),LL(0x6f781395,0x5cc8116d),LL(0x1f271989,0xa8992ade),L_(0x78c70a2c)}, {LL(0xb2334f8f,0x1fadf19a),LL(0x28c19018,0x71d5904e),LL(0x939bd9ec,0x534ec549),L_(0x77091fc9)}},
   {{LL(0x22be0a45,0xb12549d7),LL(0xca28a29e,0xb64feadc),LL(0xfa13b698,0xd61f5f42),L_(0x5d34326e)}, {LL(0x7ea2feda,0xd797283d),LL(0xc9ceaa43,0x5e1f2529),LL(0x113c4da8,0x4d22be7a),L_(0x12036cda)}},
   {{LL(0x746daf60,0x210782c4),LL(0x64b5fd56,0x358ace9d),LL(0x1008fd0f,0xaf231c63),L_(0x285dcb4b)}, {LL(0x17203da1,0x3534d554),LL(0xf8542368,0x30962d47),LL(0xf7552ee7,0xa1ab0080),L_(0x11208856)}},
   {{LL(0x7708ee69,0x981a94f1),LL(0xd9d79e3e,0x50822535),LL(0x2632b017,0x21031b41),L_(0x8876f1a5)}, {LL(0xa43cc165,0xfb24577f),LL(0xcd6f28d7,0x67a5fdbc),LL(0x583ccbb4,0xdf198e9a),L_(0xf49fccdb)}},
   {{LL(0xec729f5c,0x347bddd0),LL(0x77f89e52,0x4576657a),LL(0x79a42c59,0x3a4d5c62),L_(0xdc6f4cf3)}, {LL(0x72672c8e,0x34edf6de),LL(0xa9cc925d,0xc615c175),LL(0x264b7c52,0x6b6c91f7),L_(0x78e30c64)}},
   {{LL(0x3c93deb5,0x454fe5e1),LL(0xc107ca82,0xdaedb454),LL(0x895933d6,0xc3d92afc),L_(0xc23f9ad4)}, {LL(0x62afc583,0xb1ad4196),LL(0x9336c30d,0x122626f8),LL(0x56df6aa3,0x88c5db33),L_(0x454135c4)}},
   {{LL(0xb5862dd9,0xfb7f7ec7),LL(0xfd622a86,0xf9ca08fc),LL(0xf3fb356b,0xa3379ec2),L_(0x3726b3ee)}, {LL(0x5680ab95,0x1f4b780d),LL(0x72773b06,0x181ad0ef),LL(0x9c77acf2,0xc5ed6c19),L_(0xd82c650d)}},
   {{LL(0x57cdf957,0x2e235ce3),LL(0x919ef962,0x3b66736f),LL(0x02924c24,0x9611c7f6),L_(0x5a87ed36)}, {LL(0x2cd39c41,0x0ca01257),LL(0x8cec5f84,0x36cea89b),LL(0x576c9d6d,0xc0386b9b),L_(0xdf235412)}},
   {{LL(0x3ff1d621,0xb26ad028),LL(0x0117a093,0x3395a449),LL(0xbb9b7453,0xac8575b5),L_(0x9638e810)}, {LL(0x19b52295,0x61b7e04c),LL(0x917e2f50,0xea595a39),LL(0xb7e04864,0xf641867a),L_(0xf34ee0e8)}},
   {{LL(0xf9c67813,0x0d5cd555),LL(0xcd9d1329,0x15609313),LL(0x071a1814,0xcec729a1),L_(0x973d765a)}, {LL(0xd82b0a64,0xe9fd6113),LL(0xe16f07d8,0xf4a523e7),LL(0x095fdd3b,0x601388e8),L_(0xd8da4ec4)}},
   {{LL(0x5d150589,0x6e0c12f8),LL(0xe3db4074,0xcc4b0246),LL(0xe25d613a,0x98b62cd3),L_(0xc7d0dccf)}, {LL(0xca8e4caa,0x1f7fb23f),LL(0x6eff4dea,0x41da9f53),LL(0x44527d49,0x9e1f47c6),L_(0x3f6e7e3d)}},
   {{LL(0xb17f74ed,0xe460275b),LL(0x3f409a21,0x56dc0e69),LL(0xaef13524,0x055bebff),L_(0x40a3d6e6)}, {LL(0x5a0a6116,0x52aae002),LL(0x5d1a1a61,0x65c91bf7),LL(0x55229ca9,0x837c7497),L_(0x7a1145b5)}},
   {{LL(0x724287d7,0x86eb5493),LL(0x2b764d91,0x7078641f),LL(0x72b46e9e,0x0bd8afe8),L_(0xfaf8c9a7)}, {LL(0xeea543b7,0x7400cd52),LL(0x409c38a5,0x0a6c2e1b),LL(0x9604a54b,0xe2c2c60a),L_(0xd18594c9)}},
   {{LL(0x771d897f,0x80cd8b3c),LL(0x4efc7c63,0x3b961927),LL(0x56c6e625,0x8d0c1d5a),L_(0x5533d0e3)}, {LL(0x4b4c8803,0xee49b988),LL(0xc5802272,0x617aabd8),LL(0xb63d84cc,0xb309e31a),L_(0x0fa8942f)}},
   {{LL(0x6ef3ab06,0xfbca1b97),LL(0xa18bd2bb,0xd8911f8d),LL(0x639e011c,0xb731d607),L_(0xde5b279e)}, {LL(0x0706cd09,0x8e5b6703),LL(0xc24e209d,0x4d443d16),LL(0xdc2ea468,0xccc5dbca),L_(0x9aba875f)}},
   {{LL(0xf86e4ee7,0xb38b3cb9),LL(0xdc2bb49b,0xd001bff5),LL(0xad920b0c,0x5f833048),L_(0xd67959b4)}, {LL(0x3c687113,0x9cac7077),LL(0x84cba202,0x2bf74e82),LL(0x92b4f4b1,0xfcfe6f73),L_(0x3154d972)}},
   {{LL(0x9afdccc2,0xf2645ccf),LL(0xe60a2a1d,0x170bc3dd),LL(0xc06a289b,0x8b3174cc),L_(0xb5e5b73e)}, {LL(0xc289c4c6,0xfcbc66cf),LL(0x3f41c58d,0xea32e7fb),LL(0xfdc7bcec,0x2d89d32d),L_(0x34aa50e4)}},
   {{LL(0x31801b59,0xcf102dd0),LL(0xf19df129,0x28f82869),LL(0x1600629e,0x6d3edc02),L_(0x3c98d6b7)}, {LL(0xf21cabe7,0x31851ac8),LL(0x09f782ac,0x037f5693),LL(0x3f33781d,0xb0a0ce6c),L_(0xe96df7ee)}},
   {{LL(0x4821f126,0x43b3629d),LL(0x502c3b9c,0x3319053f),LL(0xa3b32151,0x4038845d),L_(0xea3b4ccc)}, {LL(0xedfaa06d,0x00044014),LL(0xd835277d,0xd5eb20d5),LL(0x1a8a108b,0x38cb0a6b),L_(0x30a8f927)}},
   {{LL(0xb6e363d5,0xa45c3df2),LL(0x96c532cd,0x00c26ab0),LL(0x85098966,0x6d03e325),L_(0xea0d51b3)}, {LL(0x8eec84d2,0x9134e03a),LL(0x04327df4,0xbbc8d92d),LL(0x4ffdb217,0x6e97a4bb),L_(0xc9fa9177)}},
   {{LL(0x231d20c6,0xa36c8848),LL(0x3fcb294a,0xcb7e8b54),LL(0xc4b2c9f8,0x2f8b6de0),L_(0x49f1993b)}, {LL(0xb8860526,0xfdcf530d),LL(0xb895a0fe,0x49da0f6a),LL(0xefc627b6,0x4e9fb807),L_(0xdbc0d2c1)}},
   {{LL(0x64eae114,0x23074bd5),LL(0x48d5696a,0xb246633b),LL(0xefdb4759,0xc6af484b),L_(0xde4380e0)}, {LL(0xe5997f29,0x8979b2ec),LL(0x507b7708,0x530f4963),LL(0x55ee3298,0xccf127a1),L_(0x32413bed)}},
   {{LL(0x687018fd,0x7c723f5d),LL(0x86c9f952,0x2221f721),LL(0x93828714,0x13d720d3),L_(0x45d38213)}, {LL(0xe61c9b06,0xc198db55),LL(0x598140b2,0xbe38af8f),LL(0x621f284e,0x18f09bf6),L_(0x192fd52c)}},
   {{LL(0x65cd0db7,0xbb33fff7),LL(0xf9ca36b0,0x44ac0a55),LL(0x7a9b66be,0xc9308cae),L_(0x837d663e)}, {LL(0x1b4048cc,0x5b7ecc1b),LL(0x4ee4991e,0x7fa30aad),LL(0x7e09ad23,0xff1f76e3),L_(0xbc48338a)}},
   {{LL(0xa178a33b,0xdd724842),LL(0x914d8342,0xa5d7a87b),LL(0x30befc70,0x69874db8),L_(0xec19410e)}, {LL(0xcfcdbfcf,0xe3d6346a),LL(0xde9edbbd,0x84c65475),LL(0xde39cef5,0xe4efb3c1),L_(0xc10a727b)}},
   {{LL(0x493ae0bb,0xbc282cc0),LL(0xe7ebcd87,0xa024a0e9),LL(0xfa3a15cd,0x240e6a89),L_(0xa4a7db8d)}, {LL(0xbba5d02e,0x95b6f1e0),LL(0x86491854,0x15b8172e),LL(0x158b8956,0x5eb7385c),L_(0xa8d22179)}},
   {{LL(0x92f507ce,0x9137f80a),LL(0xf48fb253,0xcb924115),LL(0x2be56c53,0x0703cdda),L_(0xb4cf511f)}, {LL(0x0b8faa80,0xb73eae3f),LL(0x58a8b927,0xf227ad25),LL(0xdbca2fc4,0x06231927),L_(0x409b12c6)}},
   {{LL(0x5983f1a8,0xee7ac222),LL(0xf6df6ac6,0x48c69cbf),LL(0xa7dc8e21,0xbd701252),L_(0x6d2da921)}, {LL(0x483083a2,0xc0d30797),LL(0x003aeb7c,0xd4ec2e63),LL(0xbda6fdc0,0x15074c14),L_(0x55d107b3)}},
   {{LL(0x29e4a336,0x600e1a17),LL(0xd60d3b73,0xa542f637),LL(0x556905e7,0xc7209080),L_(0x3f5144ad)}, {LL(0x0b55b0fe,0xa6cf1f95),LL(0xb8f822b9,0x9d44e115),LL(0x1acfe880,0xe820af09),L_(0x0a84bb85)}},
   {{LL(0xc0bba383,0x02857752),LL(0x497eb0a9,0x8b30f555),LL(0xf120846d,0xac6401e1),L_(0xf72d6212)}, {LL(0xcb20c291,0x84fbf362),LL(0xa175ab61,0xecddc9da),LL(0xe1839421,0x63952a13),L_(0xbe8f97bb)}},
   {{LL(0x24f4b136,0xf0524e07),LL(0x8fd9fda0,0x138ee6ed),LL(0x1770de3d,0xedc6985e),L_(0x6c2f8154)}, {LL(0x997ae150,0xb5a50098),LL(0xd64a082c,0x05ed2b30),LL(0xe84702e1,0x4701e013),L_(0xcb1ea5c4)}},
   {{LL(0xa2d01aa4,0x5c0291c5),LL(0x7cbe7024,0x87f294c6),LL(0xdc575065,0xf7c66733),L_(0x66e96e2a)}, {LL(0x236da943,0xb2bb31c0),LL(0xef556ba5,0xefda656e),LL(0x0d4ba22a,0xd90a6c4a),L_(0xe51714f5)}},
   {{LL(0xd7b84537,0x6ae52eb4),LL(0x5303af34,0x4989ea91),LL(0x58155fb2,0x51865588),L_(0xff60915f)}, {LL(0xb2f24a3f,0x4a65a557),LL(0x4126452d,0xc5c558ab),LL(0x39d391fe,0x71637238),L_(0x2d627920)}},
   {{LL(0xc6f1655c,0xda9557a2),LL(0x9f1d5a1c,0xad9724f4),LL(0x02bd8dec,0x286ae065),L_(0xaae19e79)}, {LL(0x66f88629,0x409bbff9),LL(0x02b7c84e,0xa035e3ce),LL(0xfa9ce7d1,0xb11a05b6),L_(0xae613f5a)}},
   {{LL(0x7d3144d8,0xeeb7e587),LL(0x4daf5c9e,0x63105e87),LL(0x24b31279,0x358ad95d),L_(0x7ca7cd50)}, {LL(0x8b96f130,0xa1f8699a),LL(0x885e7bf5,0x36720d2e),LL(0x19da3cf7,0x0c3e259d),L_(0xbb7ab1a9)}},
   {{LL(0x3ebef322,0x18a10476),LL(0xd9293d5e,0x93fc7b17),LL(0x2dd5dfa6,0x453f9b19),L_(0xb064e79e)}, {LL(0x2e375ef2,0xde52e14c),LL(0x6557a54b,0xb86d2dac),LL(0xcdf1057b,0x8978f37a),L_(0x60169743)}},
   {{LL(0x01ae0873,0xf209f351),LL(0xc477ed75,0x1c3d4f63),LL(0xfd2dee37,0x23082359),L_(0x5b85b81c)}, {LL(0x226bc4e5,0xc1dc2c14),LL(0xfac25e9a,0x4a88888c),LL(0xd11ebfc0,0x5bb60dfa),L_(0x6deca182)}},
   {{LL(0x6d7442e7,0x1a37fe3d),LL(0xceca55fd,0x64b6847e),LL(0x54c71408,0x9321abab),L_(0x42538c89)}, {LL(0x8f6064c2,0x0fe08518),LL(0x0baba2f8,0x48846076),LL(0x18d88b42,0x5027824f),L_(0x9fa6cb27)}},
   {{LL(0x2fefa18b,0x0a5f6d1a),LL(0x81d4f0b6,0xaeecb1e3),LL(0xe2cc44ed,0x1a3c357a),L_(0xa72825d3)}, {LL(0xe129986d,0x78fb4e9b),LL(0x872596fe,0x6b166a06),LL(0x1f456692,0x55649b1b),L_(0xff6c52c4)}},
   {{LL(0xbb9f7823,0x0c7e73f7),LL(0x066a275b,0x4c6140bb),LL(0x1d71b6bb,0xc0d4cee9),L_(0x6b9a5b6c)}, {LL(0x8f080a17,0xf403aa8c),LL(0x2c5d59b4,0xb53cebd4),LL(0x45561f5f,0x176891fe),L_(0x7b917d84)}},
   {{LL(0xffa27853,0x4d6d2499),LL(0xc40d40a1,0xb5248259),LL(0x5a3e6570,0xb7125d0c),L_(0x27a24b64)}, {LL(0x37979bfd,0x785afdfc),LL(0x695b2c23,0x1a4c7732),LL(0xe1b6b02f,0xa84e1866),L_(0x09ceeb7b)}},
   {{LL(0x11dbcdce,0x4243fd62),LL(0x68c988bf,0x9892ddcf),LL(0xdc0c4d01,0x5638c3c1),L_(0x8f1f8fc8)}, {LL(0xa42313cf,0xfcf531d5),LL(0x5446a19a,0x1417aa25),LL(0x7e0b8323,0x41b2315e),L_(0x8e4521f0)}},
   {{LL(0xd6c61ac2,0x9d056208),LL(0x63bfbe05,0xa70e6aa2),LL(0x566c60a7,0x483b996a),L_(0x53b69d35)}, {LL(0xa38d32bd,0xc38fefeb),LL(0x27b8ad3b,0x57f15c47),LL(0x689ae7bd,0x3e0d8010),L_(0xfa0d354d)}},
   {{LL(0x2587df15,0x904f01e9),LL(0x815dc8cc,0xbce7f04f),LL(0x88aa4d08,0xd7f1a354),L_(0x4478e55e)}, {LL(0x5bfae678,0x299925d3),LL(0x661104da,0x6a708cd7),LL(0x74d5a757,0x97968c25),L_(0x8a264b4a)}},
},
/* digit=1 base_pwr=2^7 */
{
   {{LL(0xd1a6df99,0x084a7e0f),LL(0xce960ef7,0x0b68659c),LL(0xdf5b6590,0xe1cee8dc),L_(0xbb593215)}, {LL(0xf5210072,0xccbda01d),LL(0xb592eea4,0x4c7a3391),LL(0x01bb7a1c,0xa7484df7),L_(0x4b70313b)}},
   {{LL(0xcb74232c,0xb62e25de),LL(0xb8cba58b,0xc5c8bed0),LL(0xfb8ee6c4,0x4617192d),L_(0x967548ce)}, {LL(0xeaf8b0a4,0xdc2453a5),LL(0x71f81200,0x4d7f2e27),LL(0xf43e697c,0x2e81523e),L_(0x9d94ae94)}},
   {{LL(0x37eb6d85,0x29543b70),LL(0xc7648cad,0xe67d4661),LL(0xbf221e31,0xde581875),L_(0xded3fb1b)}, {LL(0xc3db2400,0x82bba001),LL(0x656ecd6b,0x60a68af1),LL(0x5e5ccc27,0x619f6149),L_(0xa5bb89fa)}},
   {{LL(0xe80619b6,0x25f0799b),LL(0xd58585cc,0xeb6dfe0c),LL(0x9bf5f7d0,0x63a366a0),L_(0xff7302b6)}, {LL(0x927212bf,0x02c160de),LL(0xf4586c45,0x09ea0af5),LL(0xfe05e98d,0xbae9ad71),L_(0xc8fc9d86)}},
   {{LL(0x65b1d33e,0x7f5ed157),LL(0x36faea3e,0x1ed1728d),LL(0x39babd58,0x0137cf58),L_(0x4d9cad56)}, {LL(0x2546cd9c,0xd0dfb090),LL(0x4316891a,0xb59a51ab),LL(0x6476def4,0xc5dce06b),L_(0x7c93323f)}},
   {{LL(0x2313823a,0x80c0cfc1),LL(0x753f5cf1,0x27734f88),LL(0x587454f5,0xb527a9af),L_(0x315583c4)}, {LL(0x049823f3,0x25a1f7dc),LL(0xd3e3e75f,0xb3ea257d),LL(0x1d11fd76,0xafd0ae51),L_(0x1190d505)}},
   {{LL(0xc1bc485d,0x407824fd),LL(0x59b8241e,0xa0e87d81),LL(0x71256aba,0x76217f94),L_(0x64a95b3e)}, {LL(0x8b3e508c,0xe501c9b4),LL(0xfbc90f77,0x08a19d4d),LL(0xfb512fa2,0x59ab4cb8),L_(0xc1d78806)}},
   {{LL(0xe4756697,0xaf2eb7f6),LL(0x05273793,0x5c0e9df7),LL(0xc52db668,0x2b7064bd),L_(0x8eabb5ee)}, {LL(0x43184721,0xd4b9fc59),LL(0x8bf316a9,0x1343040b),LL(0x88ef261a,0xb9864942),L_(0x212e47d7)}},
   {{LL(0x0f601217,0xc3ce93c3),LL(0x1d875805,0x40d758a5),LL(0xc0aa59c7,0x54ae6327),L_(0x7111e036)}, {LL(0x111b9e3d,0x09cd3a96),LL(0xcc42b514,0x6bcd40ee),LL(0xb04993c8,0xbcda6d48),L_(0x620f7a62)}},
   {{LL(0xbabee792,0x030c3495),LL(0x74030e51,0xd426f7b4),LL(0x457bca6d,0x81319526),L_(0xf21490eb)}, {LL(0x2498c159,0x0b2bd04a),LL(0x8221d32e,0x12985a92),LL(0xcff74849,0x837d8e0e),L_(0x2ca32bf2)}},
   {{LL(0x139de2dc,0xf41adf6c),LL(0xef7b6eba,0xb7dc7dc5),LL(0xede2ea87,0x6e2033f0),L_(0x0647be07)}, {LL(0xcd213865,0x287f4ece),LL(0x4d3993a9,0x50f400e9),LL(0x0fc6614f,0x89db6d9e),L_(0x8f95a105)}},
   {{LL(0xc42a6b84,0x9e9a2ddd),LL(0xeeea986a,0xf3f5ab2d),LL(0xabdd54a4,0xc69f9a88),L_(0xb87d1977)}, {LL(0x8affd663,0xb4602516),LL(0xa2399ce4,0xc75f7a3e),LL(0xfbd56bb7,0xe4aa58aa),L_(0xa1149cb4)}},
   {{LL(0xbb083ae4,0x96aff3db),LL(0x896df53d,0x2d9b4c87),LL(0x77208838,0x2b1d390f),L_(0x02221513)}, {LL(0x408ee7f5,0xc9a98c1e),LL(0x3460329f,0xb6789b9e),LL(0xf0a9cf2d,0xcc0aecb1),L_(0x14bf88b6)}},
   {{LL(0xf99e1bc2,0xf9d97ac1),LL(0x32112cbf,0x31242e73),LL(0xe520dfcf,0xba519709),L_(0xb278a649)}, {LL(0xaada88db,0xdf453d98),LL(0x7de50c69,0x48d83f42),LL(0xa35584c3,0x7efe383e),L_(0xafd2f996)}},
   {{LL(0x18351406,0x987fc6a6),LL(0x33c7e3ed,0xf2b213b9),LL(0x17bc0ecf,0xd5c3d1eb),L_(0xe316f3eb)}, {LL(0xc603801d,0x037728cd),LL(0x0890b2c8,0xc5cf397d),LL(0xde6243a1,0xaf004bfd),L_(0xb5bd010e)}},
   {{LL(0x81c21907,0x140e4fb4),LL(0x4a85940e,0xb07d29b4),LL(0x65c368fc,0x6829786b),L_(0x11ef4637)}, {LL(0x7c01e8b8,0x5c2ad4ba),LL(0xd4385c66,0x8b692602),LL(0xca528c39,0x09175739),L_(0x8545624f)}},
   {{LL(0x3e778b36,0x332aa21b),LL(0x69ac9b72,0x7761d7c9),LL(0xf6af5223,0xded7e546),L_(0x2e42f629)}, {LL(0xde909623,0x254a8b2a),LL(0xd20517cf,0x6fa060cc),LL(0x7cccaea0,0x723e3222),L_(0x17b8fcf9)}},
   {{LL(0x00e54d7f,0x80274626),LL(0xc377f0eb,0xf63104d2),LL(0x448cb4f0,0x86f7822c),L_(0x905f1a28)}, {LL(0x635d1d1a,0x74c9d852),LL(0x03d82ded,0x2f27a18d),LL(0xa7638f56,0x7df0f814),L_(0xa98013e6)}},
   {{LL(0xe3919fa2,0x11b8e6be),LL(0x71e986fa,0x2038bc38),LL(0x1a7c24cb,0x547f1d30),L_(0xd082b3c8)}, {LL(0x4937319c,0x14498e9f),LL(0x96e06bdf,0xd9e8ae36),LL(0x71231dd7,0x099cc29c),L_(0x52cc6fea)}},
   {{LL(0xbfa21d6a,0x9934b257),LL(0x1c360412,0x2552b461),LL(0xfad28b1a,0x87244f2d),L_(0xf0f3af7a)}, {LL(0xe091c63c,0x224579d6),LL(0x8b1b9bd5,0x7927774a),LL(0xa4177fae,0x39e5970f),L_(0xfbd55d05)}},
   {{LL(0xf8a953dc,0x00e1559f),LL(0x1dc11998,0xa70e63b9),LL(0xf8851e82,0x48191bdc),L_(0x7132a05f)}, {LL(0x6d398f42,0xc96682f5),LL(0x3029087f,0xc2927516),LL(0xd7d600fe,0x6435359a),L_(0x103c49c9)}},
   {{LL(0x43c60bfa,0xbe98cc32),LL(0xdf3dae98,0xe899fd7e),LL(0x6f6a67d4,0x4805ca93),L_(0xb569900e)}, {LL(0x532ddf8e,0xb5f6725b),LL(0x912c5876,0x9818fcef),LL(0xa6151612,0xa9419c51),L_(0xb659688d)}},
   {{LL(0x544bdd85,0xf568cb9c),LL(0x486d2f59,0xe070e7d2),LL(0xa9680c77,0xf00849de),L_(0xcaecbf55)}, {LL(0xc159fff5,0x9f8b9614),LL(0xff12efb3,0x0503644e),LL(0xf325f0bc,0x58ef4f26),L_(0xc0fb619b)}},
   {{LL(0xbb5a090c,0x2a905098),LL(0x34e237a4,0xeb89d751),LL(0xb46f0674,0xc23b6e08),L_(0xf504a980)}, {LL(0x84d934a6,0x7446341f),LL(0xfe8d0cb7,0x2d200177),LL(0x428fe6ef,0x663531ec),L_(0x82b91738)}},
   {{LL(0xf0622227,0xe7d3d99c),LL(0xea4400e4,0x87323665),LL(0x0421f1d4,0xb81113de),L_(0x3c97ee34)}, {LL(0x121ae441,0x14b9bfe5),LL(0xae37e0d9,0xab8ce456),LL(0xfb0b5381,0xeebf8b75),L_(0xd0e53ea8)}},
   {{LL(0x4ab996fa,0x3bca93de),LL(0x12681ec0,0x9c224825),LL(0x51f0a250,0x4390fa94),L_(0xe12f0e92)}, {LL(0xa17f9bd0,0x87ca94cd),LL(0x707670f5,0x8eb876e3),LL(0x8c10d532,0x6ab8e456),L_(0xad2be3fc)}},
   {{LL(0x00191032,0xbe58c480),LL(0x20928c0f,0xf0df6b45),LL(0x51d6cd1a,0x3a2b0d2e),L_(0xdae2e721)}, {LL(0x6e0dcc33,0x17ad59a0),LL(0x4848071c,0x24ff8cf7),LL(0xb1342fd7,0x514eda6b),L_(0x8d26ef04)}},
   {{LL(0xaede35a9,0x43fadb36),LL(0x00e95336,0xda078971),LL(0x89ff847e,0x757ad40d),L_(0xc2d2fb8d)}, {LL(0x063bed05,0xfbec9b9f),LL(0xc9f9426c,0xa2938400),LL(0xed5c65dd,0xd3c116f5),L_(0xcc982965)}},
   {{LL(0xbf6d76ac,0x4a94b590),LL(0xdd6503fb,0x43e3ebef),LL(0x6d7848f4,0xe0fe0161),L_(0xb547e19d)}, {LL(0x503ad71c,0x14c43727),LL(0xeef3ccce,0xafbcb090),LL(0xbb5f8c82,0x3bf8d893),L_(0x2cff74c5)}},
   {{LL(0xa2d0d278,0xd8be2204),LL(0x0ccdf937,0xc9fa15ec),LL(0xac17e25a,0x91b10a2f),L_(0x61c0b94e)}, {LL(0x12c8ef80,0x087fa01d),LL(0x1b1a2421,0xb6f728bd),LL(0xd198666a,0xcef80aee),L_(0x76851ee8)}},
   {{LL(0x6112903d,0x89add916),LL(0xd329af40,0x1c13960f),LL(0xf15aa481,0x48b52034),L_(0x0726b2a7)}, {LL(0x2d1bf701,0xb5d87e2f),LL(0xb0c178da,0xc7e5ba8c),LL(0x6e4a90fb,0xfd04fbda),L_(0xe99dc375)}},
   {{LL(0xce2f6f2f,0xae50cdc8),LL(0x37383201,0x5c73b50e),LL(0x59a0da00,0x6b833239),L_(0xe9260200)}, {LL(0x54c575fb,0xfdd6dfac),LL(0xea53c24e,0x761a3a3d),LL(0xb3095407,0xca8c1a6e),L_(0x4e4fe43a)}},
   {{LL(0x56e12b9f,0xa3fc4119),LL(0xfb6450ca,0xf5bdb563),LL(0xe34b58ec,0x178622a7),L_(0xd55ba9dd)}, {LL(0x9e9ffc06,0xfbbaa19e),LL(0x1f5955f9,0x42f09c2b),LL(0xb2d6157b,0x0c83351e),L_(0x325e63cd)}},
   {{LL(0x623598c2,0xae4a61b7),LL(0xc98d16f5,0x47014b1b),LL(0xddd6a618,0xc1476c47),L_(0x7f025b55)}, {LL(0x7eb38a44,0x31a26c96),LL(0x374db7ac,0xc5ea2b2a),LL(0xa5a136e8,0x13b6043f),L_(0x43d33d2b)}},
   {{LL(0x07cb2294,0x210180c0),LL(0x173b6efc,0x782987ea),LL(0xab40f28e,0x4a4f4f9e),L_(0xce66440a)}, {LL(0x591b5ef9,0xf6832896),LL(0x49522fda,0x38f56fa2),LL(0x5abb2a56,0x5a65a98b),L_(0x6a059efd)}},
   {{LL(0xca04bf7c,0x022f4557),LL(0x7a97541a,0x657a19e3),LL(0xc772f98c,0xa19ab123),L_(0x0e388f1a)}, {LL(0x82db0ead,0xd6b0ec00),LL(0x312059b5,0xdeadd9cc),LL(0x9dab24ce,0x1c2e2051),L_(0xc5487b00)}},
   {{LL(0x7647407b,0x489801b8),LL(0x05173f8e,0x56034023),LL(0x26d5313c,0x29b7a999),L_(0xf53aa77c)}, {LL(0x6f0fc77d,0xfdc7c2f1),LL(0xe7c7d3e1,0x006982c8),LL(0xcb5af1f9,0x9323d960),L_(0x7cb0661a)}},
   {{LL(0x76954498,0xa4de751e),LL(0xcad7d975,0x6dc87029),LL(0x805eb5f7,0x1cfb5ce3),L_(0x6e5f7742)}, {LL(0x94bbb1b3,0x43b64965),LL(0x3e36f135,0xaf8d3158),LL(0xc4fd0925,0xeb438d66),L_(0x5568e790)}},
   {{LL(0xc7a170c2,0xb2973b56),LL(0x51fcb9a5,0x9e4dbac9),LL(0x2cb87666,0xcf3bacaf),L_(0xf1a40af8)}, {LL(0xa2effdcf,0x20a60ee8),LL(0x47e2556f,0x60509402),LL(0x676121bd,0x745c0ff2),L_(0x1082eb4f)}},
   {{LL(0x3407acc6,0xb6d85b13),LL(0xb4a1c32a,0x7e9049e0),LL(0x9fc877ce,0x64c1005f),L_(0x928ec4e2)}, {LL(0xb606ab7c,0xe272e2ce),LL(0xacbb37ae,0x97ac2d38),LL(0x954f3f1f,0xa0e40e6d),L_(0x1630287d)}},
   {{LL(0xdf2474e8,0x1f05663e),LL(0x7d0f40a6,0x234012a8),LL(0x89989069,0xc2481dbc),L_(0xb865be0e)}, {LL(0xf495b126,0x452948bf),LL(0x27e891c7,0xa43ea9bb),LL(0x280bca9e,0x8a34a29a),L_(0xdb3ca4d6)}},
   {{LL(0xae198472,0x848b5387),LL(0xa735717b,0xcce778e0),LL(0x4288a820,0x34cf4dad),L_(0xdcd93303)}, {LL(0x9e011fee,0x30c9f97b),LL(0x7df2569f,0xcba3ae4d),LL(0xdb99670d,0x1db11c4c),L_(0xac0ab6d0)}},
   {{LL(0x9d825f39,0x2980d39c),LL(0x1edad965,0x5a3ddebc),LL(0x9c96ad62,0x7f86e997),L_(0xa6296b18)}, {LL(0xac938f79,0x7a80b2ac),LL(0x8d1c685b,0xe8b1101a),LL(0x055eb004,0x6c8daec0),L_(0xae9ffa1b)}},
   {{LL(0xcaf9f5b9,0x88ad0652),LL(0x977ba6f8,0x35f6f9f7),LL(0xbced09e3,0x51bdbe8d),L_(0xaf059927)}, {LL(0xd40e66f8,0xa071947a),LL(0xde4b10ab,0x2ee8b51f),LL(0x9757d231,0x4f80de84),L_(0x84da5f77)}},
   {{LL(0xdf7efaf1,0x3121c652),LL(0xbb4de8a3,0xd7749653),LL(0xb988c422,0x493a2d8b),L_(0x7dd3a0cf)}, {LL(0x98e08451,0x2760de2f),LL(0x05ba148d,0xcf07c50e),LL(0xe5f81246,0x1a00e240),L_(0x89c9fdf9)}},
   {{LL(0x9d26d17d,0x89b4fc25),LL(0x35eb5721,0xc765d35e),LL(0x45f96752,0xbfefae3b),L_(0x65d4fcd7)}, {LL(0xf9a4089c,0x22ecc154),LL(0x3b66fffb,0x5053964b),LL(0xdcd4205f,0x1e216001),L_(0x9258ce3b)}},
   {{LL(0x1db56ac0,0x7ae4875c),LL(0xdd9688b4,0x4a976987),LL(0x4fa88f83,0xd67fc048),L_(0xcf37f9f8)}, {LL(0x49b6eedc,0xedcaacaf),LL(0x38ff0886,0xd26b609b),LL(0x29f6f1cc,0xa9ceab78),L_(0x2a5e8359)}},
   {{LL(0xf6f8a0a2,0x95c6ea4c),LL(0x0080fd45,0x61916ad2),LL(0x8497eb60,0xb0ce6cae),L_(0x75ea62e8)}, {LL(0x50c80949,0xa31d0581),LL(0x9b2a0c1a,0x4cec302b),LL(0x0bfcd85b,0x36484967),L_(0xe5f11201)}},
   {{LL(0x6e5e0c26,0x6df2fa39),LL(0x59949304,0x248a89e4),LL(0xdcc52026,0x67b816f2),L_(0xc824943c)}, {LL(0xee7ad3ec,0x5bb4e55c),LL(0xfa312fb7,0xdaa70d9e),LL(0xacc63cd9,0x9155d520),L_(0x42aa4030)}},
   {{LL(0x6abedf03,0x22a83e65),LL(0x2814f9c9,0xb90ac85d),LL(0x088ae4b8,0x65e3e716),L_(0x7975b338)}, {LL(0xf19b0bfc,0xfbebb9d9),LL(0xfd3c7946,0x7355b4aa),LL(0xef331b93,0xfcb3b647),L_(0x9f037b66)}},
   {{LL(0x4e49fdb8,0x76e1b0d3),LL(0xa7cfa8e6,0x94293ef4),LL(0x588501b4,0xd8359bfc),L_(0x50d236b6)}, {LL(0x3cd45f3b,0xee63deea),LL(0x49004988,0x98af1b2e),LL(0xd1f2806f,0x12a331f8),L_(0xe108a9ab)}},
   {{LL(0x3577e68c,0x7e2770f6),LL(0x08275599,0xe8913f4b),LL(0x77ead765,0xe027fdcb),L_(0x2d381d9d)}, {LL(0x9a6f6903,0xf459aa5e),LL(0x38d25f4d,0x24bf5d49),LL(0x130316d1,0xd808c825),L_(0x53bfc24b)}},
   {{LL(0xe048c7d9,0x33437c46),LL(0xce7c4312,0x2ce661a7),LL(0xe77ba9cf,0x46b5165f),L_(0x98d33543)}, {LL(0xaa89fa67,0xd3057aed),LL(0x81116061,0x142332ba),LL(0xb60e8644,0x3eb16842),L_(0x632f4f19)}},
   {{LL(0x35fc5d9c,0x9f0177bd),LL(0x63fb6870,0x63dcb419),LL(0x4fc638bc,0x0d4a9edd),L_(0x5f4b4b0c)}, {LL(0xd3a1ffcb,0x2be34520),LL(0x4dfd2636,0x6ee7420c),LL(0x449c7ffa,0xca54fb79),L_(0x1723c63b)}},
   {{LL(0x447c4951,0x07a8fa48),LL(0x5a4bf1f5,0xfe67f509),LL(0x0d70f3ac,0xdbc48919),L_(0xc6e3be9d)}, {LL(0xc5e2455e,0xc9b41ae6),LL(0x1cb83c2d,0x1f329372),LL(0xaab1f6eb,0xa76cbb84),L_(0x13b46361)}},
   {{LL(0x7f11366d,0xd38d4421),LL(0xd7176b60,0x9815fc31),LL(0x1e6e5fa0,0x42bf5236),L_(0xaf44dc92)}, {LL(0x15c5a009,0x03ac6a2a),LL(0xabc5024d,0x8f3eaace),LL(0x9eb61a43,0x1a7ff3ad),L_(0x67fd7c39)}},
   {{LL(0xf6ee6118,0x4446a48b),LL(0xd5ff7dbe,0x08776dba),LL(0x62dd1551,0x856288fd),L_(0xf3f982ab)}, {LL(0xb1d08623,0x59e29beb),LL(0x6ab6d270,0xca32b4fc),LL(0x1744a1d3,0xad3745b8),L_(0xd398bc25)}},
   {{LL(0x3831190b,0xaaeece81),LL(0x93c12ffb,0xd4655489),LL(0x6dc6a194,0xdbb627e1),L_(0xb1087770)}, {LL(0x91571bcc,0xf5fb9008),LL(0x4ad3caf9,0xf421b41e),LL(0xeaac6d91,0x01604108),L_(0xe70c380f)}},
   {{LL(0xa2f0a872,0x276d7a20),LL(0xdacb5ad6,0xa823fb57),LL(0x507ee8a9,0xe6fef46d),L_(0x8d424c56)}, {LL(0xa385d034,0x2785c811),LL(0x7ba7caa5,0xbcf3686a),LL(0x8ac8344a,0x3bf9ea41),L_(0x63abeaac)}},
   {{LL(0x8075a441,0xfea1ac4e),LL(0x2be4fe2a,0x0fedea77),LL(0xb3c70edf,0x2398f701),L_(0xbafd9f1f)}, {LL(0xbbbf63ac,0x9c1eb75a),LL(0x06bc9e31,0x9c3a2d57),LL(0xee68eb75,0x4d73d0da),L_(0x5c2caf9e)}},
   {{LL(0xe663f1d8,0x31f69d33),LL(0x2870e42c,0x46cf68c3),LL(0xc7fa33cc,0xb4e3665a),L_(0x4310b502)}, {LL(0x48f183b7,0x154b0c79),LL(0x225295c9,0x32607f60),LL(0x1e044a43,0xc1030470),L_(0xf5ab45de)}},
   {{LL(0x25aa9de1,0x88786097),LL(0x8cfe807b,0x045b9699),LL(0x2e64a0a5,0x921c1980),L_(0xaa9a7dc4)}, {LL(0x8b058dee,0xbe46c84c),LL(0x47ac068a,0x5b271860),LL(0xfe16e851,0x48f0acc8),L_(0x76ccb233)}},
   {{LL(0x31bafd7f,0x7c0c6e0d),LL(0xdade195f,0xe2d4fff8),LL(0x2b1da468,0x95ee5ad6),L_(0xee119e9e)}, {LL(0xfebeb585,0x307f35dd),LL(0xbf55c718,0x406d6f39),LL(0xe2245982,0xd6be6d55),L_(0x7d132e1a)}},
   {{LL(0x002c4e28,0x4f02beba),LL(0x8d9e9ed0,0xc26d73f0),LL(0x21de614f,0x38b705d3),L_(0x68620b5e)}, {LL(0x23429db7,0xf4967923),LL(0xb79a106a,0x198f9eec),LL(0x9eb6af93,0x6bb259cc),L_(0xec4f9488)}},
},
/* digit=2 base_pwr=2^14 */
{
   {{LL(0xdcabdf3b,0xd382ee31),LL(0x8d88ff9c,0xeeccf67f),LL(0xe97abee6,0xc6941824),L_(0x90a4b5a6)}, {LL(0x47a31930,0x7f45fc06),LL(0x876a0ffe,0x7b1af995),LL(0x4d9341e8,0x248e916d),L_(0x398b68bc)}},
   {{LL(0x082378c3,0x4bf3bcae),LL(0x8cfd71b3,0x82c65899),LL(0xb15f6e52,0x5919e8e7),L_(0x2217b0ec)}, {LL(0x6c251213,0xce97fdbe),LL(0xd1051883,0xf8d74874),LL(0x1738524d,0x44e69e4f),L_(0x98671ed4)}},
   {{LL(0x54e9eb42,0x1bc6447d),LL(0xcb1bcc20,0x65482174),LL(0x63694975,0x72869bff),L_(0x63c463b1)}, {LL(0xf6aea0ba,0xa70d16cd),LL(0x12e9b82e,0xa212780d),LL(0x14730725,0xdc25b909),L_(0x48e6eff8)}},
   {{LL(0x8d673443,0x01c6af2b),LL(0xce9a229f,0x24e6fbce),LL(0xc6a67d3a,0xbbe5e1ed),L_(0xf65af52d)}, {LL(0x964e75d2,0xb97b0d10),LL(0x8e72295a,0x09007530),LL(0x67a932fe,0xc85f611d),L_(0x7b6da974)}},
   {{LL(0xfaef942d,0xfc2eb46d),LL(0xc735b7a5,0x51096872),LL(0x75df38bf,0x75ab062b),L_(0x82f7953f)}, {LL(0x1f5151d3,0x587ef818),LL(0x723a33d5,0xf3f93876),LL(0x0eae8822,0x5b5cb708),L_(0x97da4376)}},
   {{LL(0x36982471,0x165fe970),LL(0x5a5ca19b,0x2ff27508),LL(0x89cd99ad,0x98c66330),L_(0xd0ea6970)}, {LL(0x0397edc8,0xd2a0cfc6),LL(0x49c7caa4,0x7e36813d),LL(0x722feffe,0x829c6b89),L_(0x3cc14e57)}},
   {{LL(0xe76a6991,0x394ea602),LL(0xb318aa05,0x1348f9f3),LL(0x34e40071,0x3b2e0279),L_(0xa37218ff)}, {LL(0x5da0385d,0x0b77484a),LL(0x487e3215,0xbbb8ff2e),LL(0x7e2ac222,0xf54f007f),L_(0x916f6355)}},
   {{LL(0x306867f1,0x271f814c),LL(0x45d99845,0x455790c2),LL(0x038f4fe6,0x58008770),L_(0x1a0c7df5)}, {LL(0xa3ffa256,0x2ae5134e),LL(0x791baa3a,0xfa061cdf),LL(0x2bb3a978,0xa4fe1b53),L_(0xd5d1144c)}},
   {{LL(0x1840dd40,0x24b14e0e),LL(0x8cc62de0,0xc72a3b9b),LL(0x5ca5c5d6,0x8625f308),L_(0x30593cfa)}, {LL(0x6afba385,0x5e8c195d),LL(0xb4f4df35,0x6bdd2d33),LL(0x0975350a,0xeff53d99),L_(0x20a0786d)}},
   {{LL(0x3fb95973,0xc84c9bda),LL(0xe3fb8d0f,0x172afe39),LL(0xd069e505,0x2d143bfb),L_(0xcaad811f)}, {LL(0xaf5715f5,0x9c2db5f2),LL(0xf0b61247,0x6e7b80b3),LL(0x71ec23d9,0x4f3f9687),L_(0x8dce56a9)}},
   {{LL(0xf5d591b8,0x0c9c04f2),LL(0x4e3ba616,0x6d31163c),LL(0x530f24db,0x66c42b68),L_(0xefcb7af5)}, {LL(0x1378c67f,0x777097e8),LL(0xf5bcc425,0x3f1062b8),LL(0x9ee5175c,0xb09a9492),L_(0x8ba812ad)}},
   {{LL(0x12b71604,0x0ee3063d),LL(0xf5bf2d6e,0x22611499),LL(0x853772f7,0x8f349671),L_(0xdf452ffe)}, {LL(0xdd3b39e2,0x3acb7c05),LL(0xa199b0f1,0x34c3e766),LL(0x9381a0b0,0x74081fec),L_(0x613e10dc)}},
   {{LL(0x49667bc1,0xd57108d0),LL(0x93327b6d,0x6834cfcd),LL(0xf7fb9009,0x9b0baa9c),L_(0x032637ce)}, {LL(0xd797b7f9,0x13c1353f),LL(0x66821e80,0x6f581f7f),LL(0x48a60c42,0x7f9e960b),L_(0xcad0afca)}},
   {{LL(0x474d3704,0xcfd42398),LL(0xd4011a09,0x1c1a54ac),LL(0x3c4fd36b,0xd3eac930),L_(0xffda655a)}, {LL(0x7c4dc3a4,0xdce7096a),LL(0x75a38366,0x732dfde0),LL(0xa1787513,0xe804b476),L_(0x86101a86)}},
   {{LL(0x926b6b5d,0xe5f713d5),LL(0x9c17bdda,0x3d019ddd),LL(0xd52c15ec,0x3e7f7e62),L_(0xfaca5ec2)}, {LL(0x6609394f,0x694f2696),LL(0x283219a8,0x69aa76bd),LL(0x07cee438,0x234e3949),L_(0x77c5f3f2)}},
   {{LL(0x90a49b53,0xf501aeb8),LL(0x8debc123,0xc50fe003),LL(0x8b89e79b,0xa04a6f7a),L_(0x51dcf521)}, {LL(0xff3922de,0xc4ed1911),LL(0xcefa7526,0x0ce76928),LL(0x8af23a3e,0x918974b5),L_(0x3dda3d79)}},
   {{LL(0xdfd72a80,0xdea21ead),LL(0xbaf90df8,0x8e3f7ba6),LL(0x1f678c3b,0x6df4e74e),L_(0xc2ad132e)}, {LL(0xc5a010e7,0xd7eea56a),LL(0x77bf90b7,0x389d8452),LL(0xc6ed7b9d,0x2e489cfc),L_(0xdd237e98)}},
   {{LL(0x914f8bfe,0xbcee9591),LL(0x21b5804c,0x7fab5561),LL(0xec5af129,0xd4d265a9),L_(0x7f0919d3)}, {LL(0xc74bfc94,0xd2bab14c),LL(0x1a3e2e12,0xe6b2e9c5),LL(0x77d00d82,0x161eafab),L_(0xccd81779)}},
   {{LL(0x8fb0ef8d,0x5b2aa48c),LL(0x43630c37,0xe9937cc2),LL(0x00ef4c97,0xb862b86d),L_(0x1ec65b9f)}, {LL(0x97617d09,0xfe0c2928),LL(0x92157b61,0xce44e423),LL(0x625a38ef,0x8594c3af),L_(0xbd404f57)}},
   {{LL(0x1b735669,0xbbf9ef82),LL(0x29fe66b9,0x754e211b),LL(0xf058688a,0x95864d69),L_(0xdc64d0f5)}, {LL(0x8beee137,0x9150359b),LL(0x0d596fec,0x0ab949fd),LL(0x3f4ac39d,0xf6d5f8ee),L_(0xc65ea695)}},
   {{LL(0x4aaed3cc,0xb594cda0),LL(0xea397e70,0x0216bf13),LL(0x9e271d0d,0x7efe2b03),L_(0x52fa9e75)}, {LL(0x6a084fdd,0x824376da),LL(0x369bd28d,0x6acd2b86),LL(0x2deabee7,0x55b4713a),L_(0x9706afcf)}},
   {{LL(0x9b5408d2,0x5ffa275a),LL(0xce057f13,0xfa6d3b63),LL(0x3f3eff6d,0x5bf0abda),L_(0x82fe66a1)}, {LL(0x97b2723a,0xcf4c8232),LL(0x364b69c3,0x170c8de3),LL(0x047d7294,0x43f7fc78),L_(0xec58892f)}},
   {{LL(0x0b00d0be,0x4d42a1fe),LL(0x3e6453fd,0xc3907782),LL(0xd65498ad,0x716502c0),L_(0x548685f9)}, {LL(0xaa83d0aa,0xd8e4c681),LL(0x30dc30bb,0xb2730e91),LL(0x3b7c8d6f,0xc3d966b3),L_(0x7eb48e5c)}},
   {{LL(0xfee9984a,0x2e099be3),LL(0x6964c934,0x5f8e6df6),LL(0x31915ae6,0x2ba98b17),L_(0x72598e4b)}, {LL(0xe2f20ab1,0x791aa104),LL(0xe9c53c90,0x2ae23400),LL(0x5c5045ab,0xd646c528),L_(0xf2a49004)}},
   {{LL(0x50b8339b,0x2874cd90),LL(0x98df361f,0xcb1961a6),LL(0x221ee1b5,0x43f06891),L_(0x6cf88e87)}, {LL(0xf9998039,0xeac42395),LL(0xe0ee202c,0x042a0af4),LL(0x36489fc2,0x5fdad7bc),L_(0xf5be5517)}},
   {{LL(0x9f01cd49,0xc0ba445f),LL(0x657c4ee9,0x3e5d8712),LL(0xc6107ee1,0xecfad04a),L_(0xc3a9ecc3)}, {LL(0x45f63660,0xeda195e8),LL(0xfd5f4e28,0x2f2caf70),LL(0x4fc9d505,0x0c644ae1),L_(0x49b7af65)}},
   {{LL(0xbc63e79a,0x658bc0b0),LL(0x98214b9e,0x2713db76),LL(0xabf827dd,0x12824e7f),L_(0xaef5577f)}, {LL(0xb2616195,0xfcdeef4c),LL(0x532de453,0x3fa68909),LL(0xe9395e01,0x42dfc220),L_(0xa3cf44a0)}},
   {{LL(0xe6bee7b9,0xb1959a03),LL(0xc236089e,0x9b779900),LL(0xe7e57056,0x1c2d553f),L_(0x6e0825c3)}, {LL(0x47995b49,0xab4d653f),LL(0x07e36ef2,0xf8e40f18),LL(0xb70d2132,0xbaae5b9b),L_(0x6545595f)}},
   {{LL(0x325e475c,0x8ad5b01f),LL(0x55e3fd67,0x8fb55f0b),LL(0xa691b273,0xee832613),L_(0x372805fc)}, {LL(0xb146b4e4,0x748d793c),LL(0xbb8942e4,0x2911a866),LL(0xdb940971,0x24df2ed3),L_(0xc58ee255)}},
   {{LL(0x25961d65,0x87754320),LL(0xcf82c1fd,0x15d5a2fe),LL(0xee280f81,0xfc4877e3),L_(0xf4f8e2e1)}, {LL(0xd08afabe,0x8dd2825d),LL(0xe83d6d87,0xa36fe3f1),LL(0x20840934,0x398c3272),L_(0xe0445764)}},
   {{LL(0x51502993,0x5f9bec8a),LL(0x2f38e2ec,0x878f852f),LL(0xbcaa2cbf,0xe99dba0e),L_(0xec7f9cc7)}, {LL(0x7a185dc4,0x7f95d77f),LL(0x0cb71b4c,0x3f419246),LL(0x2b04f2d5,0x7b04c078),L_(0x21ee7511)}},
   {{LL(0x597ba0a9,0x0eb981ea),LL(0xc63bbafb,0xe6e2623f),LL(0xbfc4dac6,0x35dd20af),L_(0xf02694cb)}, {LL(0xe7b40564,0x8d381cfb),LL(0xb6c06c63,0x4013b9eb),LL(0x354e81ee,0x50a8d84c),L_(0x9e8bae79)}},
   {{LL(0xccc6dc57,0xe6735e77),LL(0xf38bca55,0x629efebe),LL(0x2ffffa9f,0xc520680b),L_(0xf670ee6a)}, {LL(0x732a53db,0x4d7d1ede),LL(0x920d6d5f,0xa1f8c1a4),LL(0x3dffe6e5,0xc519ac0b),L_(0x927ed8ff)}},
   {{LL(0xaef46031,0xdd8a57c6),LL(0x3e80aad3,0xb0540672),LL(0x8ef45968,0x7dc6abd9),L_(0x350f562a)}, {LL(0x5aafa6a4,0x60865fae),LL(0x9a3d30f8,0x9c28ce9f),LL(0xd3cb6b76,0xbbb47176),L_(0xcc25a2e4)}},
   {{LL(0xfe93910d,0x86c3c000),LL(0x6c40ba85,0xe4bdec34),LL(0xa7194bcc,0x308782a6),L_(0x6cf8fd8c)}, {LL(0x710f1669,0x1c395e5f),LL(0x5062f94e,0x2f41d5d8),LL(0xe7981956,0x53f886b7),L_(0x9bf4dd57)}},
   {{LL(0x6d08786f,0x2cc15260),LL(0xf3a184c4,0x84169281),LL(0xa53ab3f4,0x49111025),L_(0x98552ba1)}, {LL(0x8db73cd0,0x49ee5fc0),LL(0x200da23c,0xeb0a609a),LL(0x26497f34,0x752e2221),L_(0x8eb5b297)}},
   {{LL(0x23ccf8a8,0xb6e96781),LL(0x4941507a,0x183ac489),LL(0x907018f8,0xf0ef92fc),L_(0x43e3f1dc)}, {LL(0xac3eb093,0x63c56eb0),LL(0x2b3e238d,0xf8deac28),LL(0x95f55a52,0x5ae62b60),L_(0x3671e27c)}},
   {{LL(0x8575f397,0x643e6356),LL(0x1662edea,0x3b84d8ca),LL(0x87159fb6,0xb1bb3787),L_(0x5efe452b)}, {LL(0x04257ae6,0x76cc23a1),LL(0xb2b2c1b3,0xf70dd339),LL(0xd9298acd,0x91c26db4),L_(0x146fe566)}},
   {{LL(0xac24fa29,0xa4b011f9),LL(0x59f4d702,0x77f2a9a2),LL(0x897d9044,0x0ed14d52),L_(0x4303ea08)}, {LL(0xa6397973,0xe2308c70),LL(0xa55b4023,0x72030316),LL(0x664a26ca,0x291e9903),L_(0x1def6c61)}},
   {{LL(0xd3c9cb6b,0x38755541),LL(0xe7d9324b,0x4ad0effa),LL(0x5c237f2b,0x75646bb1),L_(0x298572b4)}, {LL(0xa07ca1d4,0x7f647322),LL(0xdc1aadce,0xe24ec66c),LL(0x819d18b8,0x95c255a0),L_(0xe3f3efff)}},
   {{LL(0x82098650,0xddb2d835),LL(0x7218b180,0x41d055f3),LL(0x92d42e0f,0xe2f17df9),L_(0x2b71d06b)}, {LL(0xc2c8d2a7,0x061a667a),LL(0xabfafbb4,0xcc790c1c),LL(0x0eb1df27,0x6dc9536e),L_(0x250d65a7)}},
   {{LL(0xf7ca40a0,0x55a94793),LL(0xe5044ae1,0x6bbc768a),LL(0x1ef210b3,0x166d8931),L_(0xa28dae32)}, {LL(0xa0bcf015,0x621f63a2),LL(0xcd66892d,0xf790fc9c),LL(0x60203dd4,0x667b1fa7),L_(0xdb4559a6)}},
   {{LL(0x2884e5db,0xc59a883c),LL(0xb096e972,0x47f89d64),LL(0x780aeefd,0xd528d23d),L_(0x8809b35e)}, {LL(0x9052392a,0x6e9ee32f),LL(0x3da64ad7,0xb5fe2f01),LL(0x1d21d327,0x2883e925),L_(0x66ba0553)}},
   {{LL(0x8d7cbdb7,0x32e5cd9d),LL(0xefaa43d2,0x2385860e),LL(0x43f0f911,0x4b6038f7),L_(0x3c00934e)}, {LL(0x8ece6e05,0x40da7078),LL(0x76e3e8ff,0xa20a6649),LL(0x1a09dbaf,0x709cfd74),L_(0x1f695127)}},
   {{LL(0xa2ce00d9,0xa63a8b02),LL(0xa213f1e6,0x35cb5edd),LL(0x706b0821,0x4173e4a2),L_(0xb5a5074f)}, {LL(0xaae1c287,0xaa0caca3),LL(0x36399f11,0x3c3b3e24),LL(0x34698533,0xe3b66255),L_(0x8ba6b5aa)}},
   {{LL(0x4bf57c52,0x164519e7),LL(0x3666d383,0xb55fe2b9),LL(0xd37e29a7,0x68e8549f),L_(0x34e4d0d6)}, {LL(0x9f4b4c9a,0x8f691f6b),LL(0xf4164295,0xa382964d),LL(0x49524978,0x7d3fe7f2),L_(0x64f14bbf)}},
   {{LL(0xb90d4bd5,0x087b646e),LL(0xe6a8abf6,0x53865f42),LL(0x2d659d4e,0xb9db500d),L_(0xc5eb1e84)}, {LL(0x17597d7b,0x6ae5d43e),LL(0x8df2f6d4,0x3a5e9fe1),LL(0x4204e314,0xd91a271a),L_(0x236e44d2)}},
   {{LL(0xcfce86ab,0x16c2854c),LL(0x10d4f976,0x3999a9a1),LL(0x5fd18a95,0xa1e55106),L_(0x40835bcf)}, {LL(0xd7192595,0x7c17c721),LL(0x8a484397,0xf6856acd),LL(0xdb82b841,0x39017815),L_(0x2d60844b)}},
   {{LL(0xf155113e,0xc2b2be30),LL(0xf31e524e,0x1e571838),LL(0xa98b5f59,0x17a7266e),L_(0x169ce88d)}, {LL(0x02e9f20b,0x9c53a705),LL(0x47373259,0x476b9b10),LL(0x144091ad,0x98e56e88),L_(0x2198d3c0)}},
   {{LL(0x02f6a9c7,0x461d2e37),LL(0xb1b023de,0x6fe1c2f8),LL(0x8835c958,0xaae29f67),L_(0xb8b906ea)}, {LL(0x1573f8b1,0x2aae0885),LL(0x2db13030,0xc9a1230d),LL(0x930776e9,0xd3f32fd7),L_(0x55a855db)}},
   {{LL(0xdbf89587,0xfeac4726),LL(0x4978e226,0xe907285f),LL(0x084c5773,0xc72ade03),L_(0x25c0e8f3)}, {LL(0x155c7e91,0xe4fb90c2),LL(0x804fb7e0,0x833af299),LL(0x87b3ac91,0xeeb712d2),L_(0xd0aa3910)}},
   {{LL(0x08fa5176,0xad400dee),LL(0x04db731c,0xdb50d10c),LL(0x8bbf71ad,0x1322cb4e),L_(0xe4d0dedd)}, {LL(0xb7ef36d3,0xa5571f80),LL(0x8b5c198d,0x938aa53f),LL(0x6a106d1c,0xfee7b99f),L_(0xc1056e3d)}},
   {{LL(0x0d738fbe,0x9340ad37),LL(0xbe6493b7,0x923ef854),LL(0x1a934a84,0xdc9bcb14),L_(0x38e82bd2)}, {LL(0x3ed1d410,0x3e3d69c5),LL(0x2651cbf9,0x8cf302d1),LL(0x60492caf,0xee0ad716),L_(0x0f316e4d)}},
   {{LL(0x310d3207,0xc40b6b35),LL(0x61d744ad,0xf6b370b9),LL(0xcf2da667,0x4a7e6061),L_(0x06067357)}, {LL(0xd8d48241,0x186ae81f),LL(0x579e8d1a,0x2133c197),LL(0xa42e3f3d,0xf8b9b6ca),L_(0xa88cfea9)}},
   {{LL(0x9ac56b2b,0x55759656),LL(0xae1ee373,0xd47c70af),LL(0x0f258220,0x06bffc7e),L_(0xe45fbdcd)}, {LL(0x453179da,0x92ffe53d),LL(0x14a22c11,0x2ac0af28),LL(0xb924dbee,0x7afd903c),L_(0x16b2256c)}},
   {{LL(0x7d2123b4,0x61435520),LL(0x99b497e3,0x2b094144),LL(0xeb4a95b5,0x09b8c6a3),L_(0x83a43e9d)}, {LL(0x54e52455,0x646b89b4),LL(0xa8391c1c,0x2640a632),LL(0x17b930ad,0xa457233d),L_(0xc396f0d7)}},
   {{LL(0xbfa90c2c,0xfad26494),LL(0xa527afca,0xbfe60268),LL(0xcd907456,0xb3e48eca),L_(0xfa048d0a)}, {LL(0xdf9ab845,0x94fe7551),LL(0x9a3b8a36,0x8ed01907),LL(0x5a97c45a,0x5dbf5641),L_(0x9c8b0d43)}},
   {{LL(0x9dc2ad9a,0xe9019e0a),LL(0x0599e3f2,0x588acbee),LL(0xfff03f99,0xc48117a2),L_(0x5ebb6635)}, {LL(0x6e713363,0x59d13072),LL(0x42329388,0xc63c7628),LL(0xefa02cb7,0x39d7f934),L_(0xb944b464)}},
   {{LL(0x5f6d09bb,0x2361ada1),LL(0xfbd96586,0x5419445d),LL(0x6f750612,0xa30fa392),L_(0x9628fd28)}, {LL(0xd59c34bd,0x32785073),LL(0x45fc9890,0x72c99605),LL(0xb431b539,0xeec4de29),L_(0x30b9fd98)}},
   {{LL(0x97507c27,0x0c5debbb),LL(0x9bcaab04,0x3c42310d),LL(0xa01f4d9e,0x5fdffd07),L_(0x3c6028a2)}, {LL(0x5e2bab3f,0x4ce43e20),LL(0x5cbd1742,0xe71cd8a2),LL(0x0c4858ed,0x8bd12b19),L_(0x2e0122ff)}},
   {{LL(0xb6380f04,0x59339488),LL(0x3572c4ae,0x920e2111),LL(0x0fbdd870,0xcf927c12),L_(0x44d377fb)}, {LL(0x4ea01a51,0x20ee4123),LL(0xd8592d08,0xeeedf23a),LL(0x65fa7940,0x82b08346),L_(0x605e1aba)}},
   {{LL(0x34651056,0x1098cb76),LL(0x90c069b0,0xd6a624c1),LL(0x5fd458b6,0xb856ae59),L_(0x95dd191d)}, {LL(0xd687ba3e,0x33a82abc),LL(0xd59abb2e,0xa949d734),LL(0x1de89776,0xc44fd2cc),L_(0x93c93519)}},
   {{LL(0xc54039f8,0x1ad9c99d),LL(0xcf76892e,0xa0ae59e5),LL(0x5269a308,0x65e4eb8f),L_(0xdc31876c)}, {LL(0xe212c371,0x2c3e1f0e),LL(0x2e889354,0x8869d4e8),LL(0xc88e85fd,0xbc2a9592),L_(0x9c6eb56e)}},
   {{LL(0x0f4e6e43,0x2bfeb37b),LL(0x46cf4463,0x6bb45b1a),LL(0x53c918ad,0xb8bd0546),L_(0xf78b526a)}, {LL(0x0e33f7bd,0x9e58beb5),LL(0x55fd6204,0x26578e47),LL(0x986f5206,0x90135d8a),L_(0x02f82c46)}},
},
/* digit=3 base_pwr=2^21 */
{
   {{LL(0x4190a604,0xec5c8b6b),LL(0xaee61346,0x8bca6d9b),LL(0xba9eddb4,0x3ebc0dcc),L_(0xf76451ad)}, {LL(0xa709d0a1,0xc3d3be2f),LL(0x369d1d0a,0xdae0ded6),LL(0xf474acd5,0x965b8988),L_(0x6aaa8ce1)}},
   {{LL(0xfb05e5ed,0x7df5ec60),LL(0x04a47fd1,0x2d8170fd),LL(0xd08f651c,0xe08fb325),L_(0x56ea78b7)}, {LL(0xa2b2a174,0x20fa30ea),LL(0x02c33226,0xd169c10d),LL(0xc7e20c19,0x366327ee),L_(0xe33001fc)}},
   {{LL(0xf62aa1be,0xe359773b),LL(0x95f962ee,0x2b782c07),LL(0x31df9439,0x80e166af),L_(0xa27a128a)}, {LL(0xa7a7bd41,0x50ca47c9),LL(0xfce8c109,0x8ed45ebd),LL(0x4a7eebb4,0x03e9fc9d),L_(0x98c4b4f9)}},
   {{LL(0xaba113fd,0x0e6fb34d),LL(0xf4000e17,0xe2da2a96),LL(0x491a39ca,0x54e2ec96),L_(0xcc021cc8)}, {LL(0x3d0946da,0x6f792b36),LL(0x90fd2002,0xd78ca8ec),LL(0x13094248,0x286898ab),L_(0x477c2d6f)}},
   {{LL(0xb354ae4d,0xf18c6f77),LL(0xb0291000,0xf6851269),LL(0x7bd039dc,0xfb3809c0),L_(0xf376bad9)}, {LL(0xc99f8945,0x7c71a341),LL(0xf1d99065,0x21f4ed21),LL(0x4c371e86,0xa650cd97),L_(0x1713e005)}},
   {{LL(0x8e7cfcb1,0xfcae4f5c),LL(0x4c95ab42,0x62196ce2),LL(0xc9a4fa0c,0x83e1de7b),L_(0x9ba799ce)}, {LL(0x8f0bc09b,0xfa78f5d2),LL(0x66a4ac75,0xda32a5ed),LL(0x23d217eb,0xf8df814e),L_(0xb78e1f71)}},
   {{LL(0x80a1f117,0x670e56db),LL(0xfafc5e68,0xd1c521aa),LL(0xcaa4c48a,0x41b37b7f),L_(0x1023ff48)}, {LL(0xf6421dcf,0xafa1a3e7),LL(0x6e8a0024,0x5c6bc442),LL(0xbc2935cf,0xbe169cc3),L_(0x21e4c2e0)}},
   {{LL(0xe587615a,0x308bf446),LL(0x71fb47df,0x8047e56a),LL(0xdb6a36f6,0xe8293346),L_(0x5a6836ba)}, {LL(0x93a30fbe,0xcd2f5b41),LL(0x8ffe1e43,0x0b7b259f),LL(0x3d9fe740,0xaafd7816),L_(0xae0c0484)}},
   {{LL(0xc167cd8e,0x0d3f30f7),LL(0xed56f42b,0x3b140b83),LL(0x6554fb45,0xc105ccc8),L_(0x1a608916)}, {LL(0x69c72228,0xf0450467),LL(0x7e79aa99,0x88e4350a),LL(0x1e0bbd29,0x6f9f2dff),L_(0x8d2ecd98)}},
   {{LL(0x1e9d015f,0xf4db6e3d),LL(0xc9229b44,0x60df1944),LL(0x91e766b6,0x2499a2cf),L_(0x714cb7d4)}, {LL(0x0468acbf,0x7a3c83da),LL(0x84752dd2,0xfeedfeca),LL(0x95c8fd77,0x9f35aedd),L_(0x1a75bcf8)}},
   {{LL(0x1d85a175,0xefd882f6),LL(0x0c28af72,0xfcbc0529),LL(0xff17a943,0x6d3cb4e6),L_(0x20b4e7ca)}, {LL(0x1d223312,0x680691e3),LL(0xe8dd8d7e,0x2d7d7412),LL(0x3d6056c3,0xf379f1df),L_(0x82422458)}},
   {{LL(0x5a2e3386,0x66a60d5f),LL(0x1f8d438d,0x08231783),LL(0x2621fe6f,0x282e92dc),L_(0x8a0f2c3d)}, {LL(0xa53267dd,0x48813229),LL(0xc6a7def2,0x18f63ff8),LL(0x25a10917,0x06fdd515),L_(0x641475d3)}},
   {{LL(0x866c8a6d,0xb47ff8e8),LL(0xbec1f669,0x360eff22),LL(0x64925640,0x392f59d5),L_(0x2adaf8c7)}, {LL(0x18c35e30,0xe8c753dd),LL(0xb77a997d,0xeef6fc00),LL(0xdf425955,0x3afc5278),L_(0xa7269ca7)}},
   {{LL(0xedc68cb8,0x69d044d7),LL(0x9eea78e2,0xb3b58151),LL(0x8eb64dda,0x49c378c0),L_(0x9a66415d)}, {LL(0xaf6b16d7,0x47e931cc),LL(0xb92a119a,0xcb7189bc),LL(0xa9051236,0x2aca5249),L_(0x035313dd)}},
   {{LL(0xad339987,0x9bcc65a5),LL(0x9643d751,0x278964b2),LL(0x7810f1da,0xf8f2be96),L_(0x4bddc049)}, {LL(0x7e983037,0x59fa9f23),LL(0x1f58def7,0xbc86f69d),LL(0x918a9c4a,0xb726eddd),L_(0xdda12d3a)}},
   {{LL(0xac70d024,0xe4e20984),LL(0x4a8195f8,0x0ffac935),LL(0xcce8d103,0x6efdaa99),L_(0x354553cc)}, {LL(0xea3876da,0xa0b5e29a),LL(0xe8f12ba7,0x65ae7e77),LL(0x9fa645d3,0x93cb4927),L_(0x2a27d9c3)}},
   {{LL(0xac6cb27c,0x414e945f),LL(0xbed08489,0x73d9e98d),LL(0x4809a03f,0x2d14c85e),L_(0xf63371a1)}, {LL(0x0b05bc28,0x74a3b39b),LL(0x023622e6,0x003b8462),LL(0xed2336e3,0xc6f085ad),L_(0x40dec42a)}},
   {{LL(0xd9687e2f,0xd82a965c),LL(0xb6bcd68a,0x088401a5),LL(0x068fa075,0x07ba4f19),L_(0xc382c95f)}, {LL(0xf91cec30,0xb6cda991),LL(0xc0579f61,0xa7b1febd),LL(0x50d069b1,0x5d3d8cd6),L_(0xf0f3759a)}},
   {{LL(0xf8949e75,0x5c71b954),LL(0xcebf4040,0x28006acb),LL(0x6ba62a11,0xe08cb4da),L_(0xef9d9258)}, {LL(0xc4d932f2,0x9059e525),LL(0x9950d8ec,0x72f4258d),LL(0x4cc4c854,0x0a971745),L_(0x1734dd44)}},
   {{LL(0xdaf48ca3,0xa543ba94),LL(0xfd2fecf6,0xd71707e4),LL(0x809ef386,0xe30d9ba3),L_(0xe5349c3f)}, {LL(0x69eeeae5,0x6910f9ee),LL(0xfcbbeb95,0xe8c90ba5),LL(0x1896385f,0x12c1a610),L_(0xedaaa3d5)}},
   {{LL(0x2d268c8c,0xd0a57e22),LL(0x90fb174f,0x907afb7b),LL(0x927d9d98,0x4fb8839b),L_(0x60a57b93)}, {LL(0x967900e3,0xa9ff207a),LL(0xb6defcc2,0xee9dbe18),LL(0x1bad27e2,0x37d4f91b),L_(0xababf8bf)}},
   {{LL(0x6069565c,0x7e61e032),LL(0x6cf9006f,0x1034f977),LL(0xa6921003,0xf41e571f),L_(0x2a44d157)}, {LL(0xeaa8bfa6,0x77a67171),LL(0x3fd9dd59,0xf9f34af4),LL(0xa78106eb,0x7aed89ac),L_(0xa742544d)}},
   {{LL(0x8c1c931e,0x2befbabd),LL(0xf8ee2ba0,0x7e64aafd),LL(0xb97c5d4d,0xf99509a4),L_(0x621a29df)}, {LL(0x7372eddd,0x47ce48d3),LL(0x35978129,0x28b7f368),LL(0x1fd874f8,0xfdab8a95),L_(0xef2a56ad)}},
   {{LL(0x7f6337cf,0xe5943606),LL(0x03a9d402,0x21b5b09f),LL(0xa28353d5,0x917b4519),L_(0x77945c21)}, {LL(0xfd5ad005,0x73dbf462),LL(0x9833b4c9,0x7f25c9d4),LL(0x3b50b66b,0x35b3bf78),L_(0x2effa35c)}},
   {{LL(0x4b668d6c,0xbe5e047a),LL(0x70ba7d5a,0x2bc0144b),LL(0xee508f32,0x9d2116d4),L_(0x64d1a213)}, {LL(0x5d9a5638,0x8ab6e9cc),LL(0x4c4e14fc,0x954172ff),LL(0x7ca23907,0x9fe1053c),L_(0x8104e1b1)}},
   {{LL(0x711be7f9,0x4c525b55),LL(0x5f99b985,0xac72c62e),LL(0x962efa73,0x92c58056),L_(0xf4fbd970)}, {LL(0xa38c53ee,0x917ebbe6),LL(0xbab078e6,0x6ab0ae35),LL(0x25682606,0x0603c2dc),L_(0x17325a00)}},
   {{LL(0xb1a2e954,0x22b420c7),LL(0xd56f6402,0xffe309b5),LL(0xda6ba55a,0xdce7859f),L_(0x2da047de)}, {LL(0x96d10a76,0x057f1caa),LL(0x6e8c3c16,0x72ae9f6f),LL(0x8794d9eb,0xab64c20f),L_(0x63b937d3)}},
   {{LL(0xb289f4d8,0xd9e2a94a),LL(0x0f4cb441,0x4c9fd385),LL(0xecfdb1d1,0x0675dcf1),L_(0x03f2ed02)}, {LL(0xaccd6ef2,0xfad7706e),LL(0xfcfca764,0x0667c6b2),LL(0x7a5f797a,0x2207359c),L_(0x5a7fb514)}},
   {{LL(0xf3f519e3,0x06628ac7),LL(0x0a0ab6cc,0xe38c99a2),LL(0x86774ade,0x8f775fea),L_(0xfc59c719)}, {LL(0x1bf25ffd,0xf9325842),LL(0xb246d04d,0x34f4502c),LL(0xabcab086,0x064e829c),L_(0x1a866a52)}},
   {{LL(0x5fefab0c,0x06295d26),LL(0x523fd3f9,0x234674d7),LL(0x9c70b1f8,0x01e05c18),L_(0xd80fb580)}, {LL(0x352749e3,0xee248ea9),LL(0xe2fc4a35,0x3b3c0b4b),LL(0x7556a699,0x4df0080e),L_(0xb60da54c)}},
   {{LL(0xc8a345b2,0xaf868ebc),LL(0xaaab8ef9,0x9a7fcfbb),LL(0x959c94fa,0xea271ece),L_(0x14c6b171)}, {LL(0x1ea91da9,0x252b7410),LL(0xb76f94ce,0x560e8134),LL(0x8ada4812,0x4d0eaa23),L_(0x48d2e1e9)}},
   {{LL(0x11e78f97,0x55ff3762),LL(0x52966809,0xa90b8b11),LL(0xb557ab25,0x98bfd175),L_(0x7a838285)}, {LL(0x453f5ec1,0xbd8cc123),LL(0xb8541423,0xe53f50d0),LL(0xaf6d61de,0xb809c27b),L_(0xbc9d881b)}},
   {{LL(0x22d1e308,0xcafcf6b1),LL(0x3494f12d,0xb5d40ef0),LL(0x602ae873,0xb5bef707),L_(0x8f093526)}, {LL(0x47aa28a1,0xc7e7a44e),LL(0x363aec6d,0x9f424f8c),LL(0x39a88470,0x08171c6a),L_(0xc8475406)}},
   {{LL(0xd43ff3e7,0x5bfd96fc),LL(0x35837bfc,0xa9d5f891),LL(0xeb92efe6,0x5ac4853d),L_(0x7f7ee96f)}, {LL(0x90443656,0xe8ead180),LL(0xd124bb70,0x84a289e4),LL(0x98bbea4f,0x8bd0878a),L_(0xe0a10a80)}},
   {{LL(0xd233645b,0x2cec3fe7),LL(0xc6b4e341,0xc6974f7f),LL(0xc7a73902,0x0b481a08),L_(0x8fcafe71)}, {LL(0x9eb32486,0x9b2ec275),LL(0xb1171362,0xdb3bea45),LL(0x0653d374,0xab7f0ce3),L_(0x652219fa)}},
   {{LL(0x89e007fe,0xa77a41f1),LL(0x14cba611,0x6059351c),LL(0xed2ec819,0x02bfa9e8),L_(0x1e353c31)}, {LL(0xa9065e32,0xd4e23efc),LL(0x243c2ec7,0x9c16b8bc),LL(0x465d2c9d,0xa1b1cda1),L_(0x45e1763a)}},
   {{LL(0x54df839e,0xb93666fd),LL(0x5820d4be,0x1fed47ac),LL(0x7e8268de,0xbda2efca),L_(0x61307a90)}, {LL(0x428af14d,0x9ee2b07c),LL(0x72170b4f,0xda44c29e),LL(0xfffd22e4,0x296f1761),L_(0xa25a8bfc)}},
   {{LL(0xb42b17e3,0x6d4c5ddd),LL(0x83cf7f8a,0xabe94b27),LL(0x89be3310,0xde98e510),L_(0x6e879f48)}, {LL(0x9d4ca35f,0x96df2661),LL(0x45c77761,0xb28f2b85),LL(0x96395d9e,0xc8d9fdcf),L_(0xcb9113a6)}},
   {{LL(0xdbc075c4,0x54d26d6d),LL(0x6dd2d9b4,0x23c171f1),LL(0xca47c96b,0xecedbb06),L_(0xfce5ed3b)}, {LL(0x31539157,0xe9bac065),LL(0x43e31db0,0x552f0920),LL(0xef1cbf3a,0x2cb42440),L_(0x73830d80)}},
   {{LL(0xed539df2,0xd5c7355c),LL(0x11447b1e,0x5ced43ea),LL(0x10d60144,0x9618c03b),L_(0x6045d427)}, {LL(0x8b9bcb4b,0xe3767fbb),LL(0x4785c2ae,0x2a5c6f77),LL(0x8a009344,0xa23cb6b8),L_(0xaa1eaa93)}},
   {{LL(0xfd4a0f5a,0x7ea78a85),LL(0x5acc9416,0x258d360d),LL(0x951a2674,0x7b3935b0),L_(0xf2465f38)}, {LL(0x75e447f1,0x80a9bfbf),LL(0x13abd588,0x8cd933d2),LL(0x1c914702,0x8da2914d),L_(0xb5dceca9)}},
   {{LL(0xbb8f1e8f,0xbb61ba00),LL(0x68589fbd,0x018ee54b),LL(0xf3ed084b,0x9b33e2c5),L_(0x4187a448)}, {LL(0x90c49202,0xdac991d0),LL(0x0adfa0ba,0x06286fbc),LL(0xfa92d2f7,0x8e51716d),L_(0x869552ad)}},
   {{LL(0xcd0c67a8,0xe3fcc974),LL(0x1fe4efe6,0x4936eae1),LL(0x38aec941,0x49b20db5),L_(0x31b2e360)}, {LL(0x8806a868,0xfd73d3b4),LL(0xf81a57a6,0x8553e67e),LL(0xbb731824,0x5f399b3f),L_(0x063ed190)}},
   {{LL(0x04dbb079,0xd2ce86a5),LL(0x396d5523,0x636ee6c6),LL(0x5c029f32,0x90ef1fcb),L_(0x8504873f)}, {LL(0x05c18efc,0x812addf4),LL(0x2fdbed7c,0xdf2ff660),LL(0x34c03bb8,0xb1740945),L_(0x56f08b04)}},
   {{LL(0x18fd7ff8,0x8a416ad5),LL(0xefbbc814,0xb7d2ecb7),LL(0x8226e0af,0xbb5fb220),L_(0x9bdeb90c)}, {LL(0xcaee42bc,0xb76b0d06),LL(0x184432d2,0x31322587),LL(0x678f208f,0xa07b89ea),L_(0x829d1d5b)}},
   {{LL(0x5899ed85,0x45083585),LL(0x2e43b3ad,0x7217b173),LL(0x04f9e7f1,0x59198da1),L_(0x2293f74e)}, {LL(0x11d6cf0f,0x908d5768),LL(0x06e9468f,0x51f15b45),LL(0x7dd32a0d,0x58045077),L_(0xa2294102)}},
   {{LL(0x885c6637,0xf7bef8ba),LL(0x74d38792,0x478b8ac8),LL(0x58aa9fce,0x9d21686d),L_(0x503a4ce7)}, {LL(0x541006e4,0xdfc86dcf),LL(0xc2238c04,0xe98f7f2c),LL(0x73e745a7,0xfc620932),L_(0x1dcc2ceb)}},
   {{LL(0xd49816e3,0x68c388d9),LL(0x697ef53b,0x741de91e),LL(0x3b976a04,0xf24259fb),L_(0x4259cf50)}, {LL(0xcbbf2685,0x56659522),LL(0xe4e3f7fe,0xea31954d),LL(0x43c8b17e,0x3d1de35f),L_(0xa56dc0b1)}},
   {{LL(0xddd685db,0xa178f2e3),LL(0xf8a89c1c,0x859daa45),LL(0xc3965546,0x65f74f36),L_(0xef984d98)}, {LL(0x31952a51,0xd0b694c6),LL(0x6a5e5ad0,0x4f51b3a2),LL(0x20b2748b,0x54b61458),L_(0xc1437804)}},
   {{LL(0x91be987f,0x92390774),LL(0x9816ace7,0x31cf372f),LL(0xd8dc53f9,0x490c450f),L_(0xbaf48158)}, {LL(0x604fd793,0xb689243f),LL(0x86f30400,0xb2d8f482),LL(0xdc31ef3b,0x506dd3c6),L_(0x1b18f132)}},
   {{LL(0x1196c48d,0x6755b53e),LL(0xe39a017a,0x6a0e74ee),LL(0x2da63e64,0xaa8ea0cd),L_(0x7634dff0)}, {LL(0xb013649f,0x1ab0635d),LL(0xbd02ff70,0x9858a79c),LL(0x65938865,0x3acb5b92),L_(0x99e5f181)}},
   {{LL(0x28b2de14,0x6b120542),LL(0xc4ac30ff,0x5ea493a4),LL(0xbe9b2f46,0xcdd3cbfb),L_(0x0c6270af)}, {LL(0x543265a6,0xf3a605c4),LL(0x83e752d0,0x92bcbe09),LL(0x1d71ae8b,0x4ad0176d),L_(0x6507463d)}},
   {{LL(0x183493f1,0x3e8ef423),LL(0xc54a293d,0x065effb9),LL(0xa9ccf6fd,0xb0500b2e),L_(0x88529f19)}, {LL(0xfb31da5e,0xe16bd325),LL(0x5014e1e5,0x2bb46b57),LL(0xcdab125c,0x8179b8bc),L_(0xedd40925)}},
   {{LL(0x54823e46,0x263960f2),LL(0x24b40650,0x21ab8a5f),LL(0x193b2d98,0xb940533d),L_(0x8d0db660)}, {LL(0x7586193d,0x671090ab),LL(0x08db8178,0x216acee5),LL(0x6bb91540,0xcc4fc3a7),L_(0x37f48327)}},
   {{LL(0x0e69184e,0xe86f6c8e),LL(0xbcf25a98,0xe88688e1),LL(0x9beec95a,0xd008257c),L_(0x3d97d91c)}, {LL(0x35395ef5,0xacddfb2b),LL(0xa6942cd6,0x086efe97),LL(0x8a1a0ad3,0xb4d51fd4),L_(0x59c722f5)}},
   {{LL(0xd8510369,0xd5dedcbd),LL(0x14ba55bf,0xab06f6a4),LL(0xbd599498,0xc5aa099e),L_(0x69ec087e)}, {LL(0x32f40f7c,0xad93fbe0),LL(0xf1d12a93,0x40b951f1),LL(0xf5751752,0xfc37b6e1),L_(0x0c14e715)}},
   {{LL(0xaae77beb,0x58b91be1),LL(0x3f7b0006,0x6ad767a4),LL(0x9b47cd71,0x24d37cee),L_(0x3b245f34)}, {LL(0x70ac8b26,0x2da13b60),LL(0x88aafd82,0xbadf18ac),LL(0x62a037c8,0x78224e07),L_(0x8d974011)}},
   {{LL(0x6a2f2199,0x1578e706),LL(0x8412ecab,0xa273c222),LL(0xe95f128a,0x2b3feceb),L_(0xccb455b8)}, {LL(0x4d1450ab,0x247fea5e),LL(0xc24d8d6f,0xcc56e420),LL(0x32f2d692,0x001d3250),L_(0xf3727734)}},
   {{LL(0x3d1a2b09,0x1192eb20),LL(0xde270684,0x194cd935),LL(0xcbfbf813,0xe3ba3ce4),L_(0x9c6fa3d5)}, {LL(0x8d6355ce,0x36393a36),LL(0x18945358,0x6d3b3d2c),LL(0x50d98fcf,0x2da52d36),L_(0x23e01cc2)}},
   {{LL(0xd68d0be3,0x87600722),LL(0xd90cd86e,0x6a898b0e),LL(0x64b1c25c,0xf26df29e),L_(0x5696f04a)}, {LL(0x0562d2ad,0x116fb9cd),LL(0x3440763e,0xadee6e3f),LL(0x19e073f7,0xdcd869ef),L_(0xe748bcdf)}},
   {{LL(0xd9747843,0x484e4a18),LL(0xd6812dc5,0x2181acfc),LL(0x09d6f90f,0xae25cfad),L_(0xc6209409)}, {LL(0xb6aa9130,0xeb2922c9),LL(0x38ddc5b3,0xbe3728a6),LL(0x3b8ce6be,0x04eb6141),L_(0x3d9bb134)}},
   {{LL(0x618ad07f,0xc862e8d5),LL(0x69540999,0xca247f9d),LL(0xaa268736,0xbded92f5),L_(0x29192f4b)}, {LL(0x9481e65a,0x42b36d96),LL(0xe3139379,0xabfa2ad1),LL(0xd3f46f02,0x1cbbeccc),L_(0xe59fce03)}},
   {{LL(0x2b94f0ff,0x608a4b71),LL(0x9cfd64f9,0xdd35cef7),LL(0x236745ba,0xb4d5d169),L_(0xdfa8120d)}, {LL(0x03f0b33b,0x826e0d5f),LL(0xa5992b36,0x31eb9950),LL(0xf2cc0540,0xd954458a),L_(0x66bed66c)}},
   {{LL(0x2c486c0c,0xa5afc024),LL(0x8a61a33e,0xb063a531),LL(0x38aa1ed3,0x1558aff7),L_(0x09b640f8)}, {LL(0x904e9ed8,0x3494f0b6),LL(0xd2a51717,0x365e3d04),LL(0x0f32df2c,0x5c8f30d5),L_(0xffb044f0)}},
},
/* digit=4 base_pwr=2^28 */
{
   {{LL(0x6a7ebdd2,0x676d690b),LL(0x5519e7c3,0xf169ac19),LL(0x70ef4a18,0x3fe9e66e),L_(0xd0bcdec7)}, {LL(0x9b515116,0x6b6b0581),LL(0xe52dad64,0xd0e88db1),LL(0x52c4fcf1,0x31c324fa),L_(0x5e6c0907)}},
   {{LL(0x403677bc,0xe67af8a2),LL(0x3023628a,0x0b4be307),LL(0xdcb51813,0x88ce9bf6),L_(0x9d1c6e56)}, {LL(0x0f788e99,0x8400154f),LL(0xfbca2d36,0x768edb49),LL(0x80428f26,0x277da6c6),L_(0x1b78829a)}},
   {{LL(0x6933e247,0x8f4bec34),LL(0x96def5c8,0x13b62771),LL(0xc9121c92,0x543662c4),L_(0xf4ea3390)}, {LL(0x1090a59d,0xc62e7c6f),LL(0x4bb3b3cd,0xa38394a7),LL(0x19c57ed5,0x4823e8cf),L_(0xbab65589)}},
   {{LL(0x61b100d3,0x416f9398),LL(0xaf6ede17,0x8d0f5a47),LL(0x11e5f34b,0xe047da80),L_(0xffc1a9b5)}, {LL(0xc598885b,0x455959c9),LL(0x6cd277fc,0xe69bdd02),LL(0x6bdbe44c,0xd3377883),L_(0x16b926d8)}},
   {{LL(0xe93c7bab,0x23daaa17),LL(0x8729f3c7,0xcf91e74b),LL(0x64e1431b,0x1ae9e235),L_(0x2e21bb9c)}, {LL(0x2c005910,0x0ba0b46e),LL(0xdccebddd,0x1ee4dc17),LL(0xe61cc51e,0x93805289),L_(0xbc0293c3)}},
   {{LL(0xa33f496d,0xc65b315e),LL(0x5526e12e,0x25723786),LL(0x2ca2c551,0xc7e943c6),L_(0xea468f5d)}, {LL(0x1845a432,0xb33bb1fe),LL(0x1b868344,0xf5b5a1e6),LL(0xbb41c15a,0xf5bd0a9c),L_(0x65efbcd3)}},
   {{LL(0xe9864245,0x73ab2381),LL(0xbfa54a0d,0x0ed12b17),LL(0xe3536d47,0x0a040c86),L_(0xfbd5dc3b)}, {LL(0x550f5801,0xa16b2d10),LL(0x96655492,0xd20e3020),LL(0x40239ea3,0x13feb317),L_(0x63a5974e)}},
   {{LL(0x1220685a,0xd53242c9),LL(0xcca2c5ae,0x1c2921f4),LL(0x765eedc1,0x84f582be),L_(0x7177d048)}, {LL(0xf8de62bf,0xf9eee520),LL(0x22fe2632,0x209f3c9d),LL(0x721633cd,0xc2e9678c),L_(0xada44b67)}},
   {{LL(0x852acfe9,0x92b626ca),LL(0xa685284e,0xf571ade2),LL(0x918c76fa,0x5b7490f5),L_(0x375ca970)}, {LL(0xdf1662be,0xbb1e6d8b),LL(0x7fd11993,0xd821c744),LL(0x9a06de8a,0xcaec5c2c),L_(0xc7374f40)}},
   {{LL(0xc1321d2b,0x74538108),LL(0x95469f9e,0x2f1d65e5),LL(0xc8d1c486,0x8ecd7a1e),L_(0x9b0dcc2a)}, {LL(0x3466245b,0x700ed4ab),LL(0xc2bf54a5,0xfa32badf),LL(0xb5a66db1,0xfc4883b2),L_(0x3315b46b)}},
   {{LL(0x962b479d,0xc4a7efe4),LL(0x6dc8fec6,0xa0d20fc9),LL(0xd0ef4040,0x76eabbd4),L_(0x4ba76aee)}, {LL(0x17a75082,0x4bc6d910),LL(0x75ee3974,0xff0f721d),LL(0x36e896fb,0x6146e5ef),L_(0xc74f818a)}},
   {{LL(0x6fd30020,0xe508cb13),LL(0x06427fcd,0x80a73390),LL(0xcc02c93c,0x6ad8073c),L_(0x54e00967)}, {LL(0x5f92bcf7,0x8847a6eb),LL(0x9b0cb317,0x758f4aa3),LL(0x40b45a83,0xdb4ada91),L_(0x6bbb2114)}},
   {{LL(0xfff55c76,0x661c826d),LL(0x3203a031,0xb311e145),LL(0x6bdaa80c,0xcad158e9),L_(0xff7c21ec)}, {LL(0x4502f455,0x239ecee1),LL(0xde88ad58,0x2b344eb2),LL(0x161e1cf8,0x02ce3f80),L_(0x966ad037)}},
   {{LL(0x0009b91a,0x53d14c44),LL(0xe564666e,0xcbb01019),LL(0xdf767e4c,0xecc4eabb),L_(0x9ba1897c)}, {LL(0x70cf6d6d,0x0ede1159),LL(0x23335d25,0x4162258e),LL(0x5ed679cb,0x5e706cdc),L_(0xf3d53802)}},
   {{LL(0x33b4ee65,0x78fd4645),LL(0x97cb4549,0xb807ac0a),LL(0x7e760bcf,0x923934bf),L_(0x32edf91e)}, {LL(0xaf2ed833,0x918e957c),LL(0xdbf70f3d,0xdf411692),LL(0xde9c3716,0x2c76553a),L_(0x05dea60b)}},
   {{LL(0xd3a72d09,0xa5e4f840),LL(0x40f8ed7a,0xd9bcbe82),LL(0xf00706d6,0x986b79c3),L_(0xc2b3b652)}, {LL(0xefb01a69,0xa0bb40d2),LL(0x5b0da5a2,0x2d182973),LL(0x57b4bca1,0x43c566de),L_(0xbed5ccfc)}},
   {{LL(0x2f1c7c7c,0xfb3c8bb1),LL(0x3a9383ad,0x1addae50),LL(0x29206628,0xa115d8d7),L_(0x4c24fdb4)}, {LL(0xcf3f272d,0xf2a0db00),LL(0x786a47ad,0x1a242dce),LL(0x048ec06d,0xbe72542f),L_(0x384605c4)}},
   {{LL(0x515f5abb,0x995a0552),LL(0x65750201,0x809459ef),LL(0x2a52fccb,0x4bf08352),L_(0x3914cd0f)}, {LL(0x94c73472,0x507e309b),LL(0xf31b1135,0x006ea75f),LL(0xd3410c08,0xde332d38),L_(0xce581939)}},
   {{LL(0x0560814a,0x971e9a21),LL(0x69ce7f60,0xcf890a46),LL(0x607c23ca,0x8c2f3995),L_(0x16579c51)}, {LL(0xb8358126,0xc384ccce),LL(0x74485fa6,0x62e3cda6),LL(0x637377fe,0x36145889),L_(0x11e8766a)}},
   {{LL(0xa7ccaba4,0x66ee0e61),LL(0xd1b8ad50,0x8b52d5b2),LL(0x9b192439,0xe2507e26),L_(0x141991a3)}, {LL(0xf0320f47,0x164afe7b),LL(0xe4e590ae,0xa9cd4bff),LL(0x53be168f,0xbd05717a),L_(0x2141999f)}},
   {{LL(0x82377242,0x84720302),LL(0x464bed04,0x3e8eab14),LL(0x8131dcb9,0x8dffb402),L_(0x465f6f0a)}, {LL(0xca875370,0x7476d50d),LL(0xb1cc70c5,0x40036a93),LL(0xad40e0b3,0xae5c279a),L_(0x17d24656)}},
   {{LL(0xce0565d5,0x4590aef4),LL(0x186eb345,0x813e461c),LL(0x4d52d217,0x64b75926),L_(0xdfc17d52)}, {LL(0x51483219,0x4046569c),LL(0x97c10da8,0xbd282e8a),LL(0x394837d5,0x5a6b2c1e),L_(0xe88b3478)}},
   {{LL(0x58408341,0xa28c5b85),LL(0x1b86b157,0xac9ddd98),LL(0xdf706c18,0x62a72bba),L_(0x814b4431)}, {LL(0x7437fa2d,0xafe0e361),LL(0x79e0c72f,0x29283b53),LL(0x0c099515,0x469c2258),L_(0xe132ed6d)}},
   {{LL(0x1869604f,0x69cd9356),LL(0x18ebfef9,0x161651d5),LL(0x8a8472c4,0xa23f6947),L_(0x9ee50c4f)}, {LL(0x9ca82d06,0xdd9ffcc5),LL(0x4875708e,0x7c9cfaaf),LL(0x8b227092,0xc52e4c68),L_(0x757b5a90)}},
   {{LL(0x66b5d594,0x661a0707),LL(0x5bc7f7db,0xf0ed8bfa),LL(0x9f814a89,0x68ff82da),L_(0xcbb5731e)}, {LL(0x54b6136f,0x125a9d0a),LL(0xe936a8d4,0xdbfa6d51),LL(0x0914dd93,0xacae0da7),L_(0x00b1f096)}},
   {{LL(0x41cc5967,0x8e32f402),LL(0xd57a3568,0x99f4bcb6),LL(0x0ba8496e,0xcbcf93d4),L_(0xcd9a9a5d)}, {LL(0xf2f0154a,0x22877383),LL(0xd00df8ff,0xf270905c),LL(0xfeb2c8a5,0x15d8ffe9),L_(0xa7504f44)}},
   {{LL(0xc56f7438,0x435cbc99),LL(0x7613f44d,0xedb87fdb),LL(0xc1d19083,0x702066e8),L_(0xeee5d570)}, {LL(0x2c1c4234,0x672ee8b0),LL(0x8a5f8dd8,0xfc342416),LL(0xaa518aba,0x34e2bf87),L_(0x37243c86)}},
   {{LL(0x3bd53fa0,0x38707713),LL(0x464e00d7,0x6251ddc9),LL(0x4c9a0c1a,0xe7e323fd),L_(0xec2175bf)}, {LL(0xd33612f3,0x93f34e7a),LL(0x840bea94,0x0c42a980),LL(0xdc0b91ab,0x548c92c1),L_(0x978906cc)}},
   {{LL(0x72ed1d98,0x0de1c184),LL(0xa6f53ca5,0x86904dc7),LL(0x45379ece,0xbe4c089a),L_(0xb3444b1a)}, {LL(0x6d5238af,0xa27b3f88),LL(0x46b2be92,0x21e17b1e),LL(0x122542d6,0xd6e7acc3),L_(0x7853b372)}},
   {{LL(0x01b419d1,0xdb03ac70),LL(0xd178ee53,0xc5304465),LL(0x5f684940,0x11655587),L_(0x7b9ce700)}, {LL(0x196f70df,0x771ca6aa),LL(0x08b0e5f8,0x4a94022c),LL(0xb69fe2d3,0x076b1dc4),L_(0x70769678)}},
   {{LL(0xecd18f23,0x225dd231),LL(0x3eefc65b,0x23c21f5b),LL(0x43b63acb,0xf5b2b43c),L_(0x4753caca)}, {LL(0x7ec7a342,0x362a2c90),LL(0x73799a39,0x308c8fcb),LL(0x608d86ad,0x8858eb99),L_(0x64620322)}},
   {{LL(0x82f0ce09,0x8ffc3544),LL(0xd650c2af,0x1414581c),LL(0xf2cd74da,0x2ab74c29),L_(0x764c59d9)}, {LL(0xd4c61a1c,0x8b9a0584),LL(0x88cdc1bb,0xd7114187),LL(0x7b53c8e6,0x4e91e1e4),L_(0xb898c7df)}},
   {{LL(0xe472f5b8,0x120a4cb4),LL(0x28eb2abb,0xd29feaef),LL(0xa568a4a5,0x1f0a0815),L_(0x7c56e710)}, {LL(0x52d9147d,0x1dd52d78),LL(0xf3e9a259,0x99e7538b),LL(0x25ad4a06,0x8a35cc7c),L_(0x7ffcc9ae)}},
   {{LL(0x3daaae36,0x214d493c),LL(0x4a1951b7,0xc0f2cff4),LL(0x5e7107a4,0x46024f37),L_(0xba7c8cd9)}, {LL(0x0f39120c,0xd7f5199d),LL(0x73d5ebf0,0x2dba9f4c),LL(0x2755fd79,0x54915591),L_(0xb315426e)}},
   {{LL(0xbd4c1fd8,0xbe148855),LL(0x0336e26d,0x7f36ce43),LL(0x41ceba92,0x40c686c2),L_(0x0c8b60cc)}, {LL(0xa2355b30,0xeb937195),LL(0x71e0029a,0xa1f3ab05),LL(0xf7d38672,0x90828e20),L_(0x2f8c50fe)}},
   {{LL(0x37b7db09,0x4de172b0),LL(0xa962d349,0x3f8b6d31),LL(0x1253fef8,0xc76ebff6),L_(0x83810f07)}, {LL(0xc53e233f,0x9a8ef0b3),LL(0xc3896b5b,0x2d43fb47),LL(0x9a94edf5,0xc8891237),L_(0xabdf75f9)}},
   {{LL(0xf1704460,0x115d7eb6),LL(0x59c71eb5,0x71666f11),LL(0xdfdf4d27,0xcd09d0bd),L_(0x7aa8753a)}, {LL(0x658ac045,0xe75c6300),LL(0x83c2df96,0xa0de2977),LL(0x5ea75aec,0xd9cc6143),L_(0x8bd0d544)}},
   {{LL(0x2a662d9e,0xcd959485),LL(0xb69374f3,0x5ee58eff),LL(0x69a1ad5b,0x417599f7),L_(0x81e4947e)}, {LL(0xc5110168,0x471554ab),LL(0x30912d02,0x4e2c7d96),LL(0xa6a4af88,0xe49c13ae),L_(0x0170b29f)}},
   {{LL(0x0a1a147e,0x23d766f5),LL(0x8108f7f1,0x5446ea57),LL(0x737348f7,0x49f6d997),L_(0xbb23c442)}, {LL(0x17464a0d,0xf5ac023b),LL(0x82c1f632,0x1c89d9f6),LL(0x5a4700e1,0x918b1cd7),L_(0xdb771a09)}},
   {{LL(0xd9c4d8a6,0xfb136a04),LL(0x7d3d9198,0x627e23b2),LL(0x2a546df5,0x9518b383),L_(0xe5ad50b4)}, {LL(0xe74b275c,0x9e8dd1b3),LL(0x8399ea24,0xb28d50a9),LL(0xc1bf5392,0xf0c4dae3),L_(0x3a885426)}},
   {{LL(0x60cd114f,0x19f30dd8),LL(0xc7c06793,0x0a04952a),LL(0xa5b0d428,0x9fbf17a2),L_(0xedbaf6ff)}, {LL(0xdc88c750,0x2eaf445d),LL(0xd380c189,0x24d076c7),LL(0xb7cb2b9f,0x4da895c0),L_(0x6670ebaa)}},
   {{LL(0x296bb0cf,0x1e460fe9),LL(0xb75d30ca,0x0a2b9d5f),LL(0x779ddda6,0x70ad87c8),L_(0xd421a064)}, {LL(0x1216c7eb,0x015e5322),LL(0xd6456e5a,0x08a2c2b4),LL(0x5c4f2da2,0xda0125d8),L_(0x2594bcc2)}},
   {{LL(0x5cbb2272,0x0e27b02e),LL(0x74c5826f,0x651b4dc0),LL(0xf5b545b4,0x89864ae4),L_(0x81d8295c)}, {LL(0xbfa85081,0x0162c99f),LL(0x1d4b46ac,0xc148b34a),LL(0x39fddb94,0x7de290f5),L_(0x1840e143)}},
   {{LL(0x8d61f3cb,0xe4f10dce),LL(0x0ac9caca,0xd40a9135),LL(0xdb4a6076,0x869e961b),L_(0x41f3f07b)}, {LL(0xc8ea7a1d,0xaed2c272),LL(0xff4dfc57,0xdd26742f),LL(0x49228bdc,0x196c9b69),L_(0xe6e12a4e)}},
   {{LL(0x5f4b0d33,0x357d26fc),LL(0x222c05a9,0x8bc3e1fc),LL(0x44a306a1,0x6ebde62d),L_(0x25b23700)}, {LL(0x66c5b1cb,0xc2a2d52f),LL(0x28621234,0x67f40398),LL(0x6a1916f6,0x9eb2b762),L_(0x3b034368)}},
   {{LL(0x97ae197f,0xe2705942),LL(0x168b8890,0xb717c180),LL(0xfe30cc2e,0x7aa89b4d),L_(0x2c096c3a)}, {LL(0x3d1c1e8c,0x1731a1a7),LL(0x027b4fb0,0x4cdde0b3),LL(0x85e85391,0xab7103b7),L_(0x4bd9bc5f)}},
   {{LL(0x39f1ba7f,0x21c91792),LL(0xb47bf115,0xd1e29983),LL(0x8102c46e,0xaa0fed15),L_(0x1b4f9e4a)}, {LL(0x1df20d6b,0x659104e2),LL(0x3765c7ea,0xd4f49239),LL(0x3a84be64,0xca582a23),L_(0x0c1bc735)}},
   {{LL(0xce03f04b,0x827606a4),LL(0x419327d9,0x10bcfdd1),LL(0x277b88b9,0xf2e4c5e4),L_(0xcee78792)}, {LL(0xb9213c6e,0x161c54e8),LL(0xa56d42d6,0x2814c386),LL(0x859e6045,0xac244cf4),L_(0x4647ade8)}},
   {{LL(0xd6446676,0xab795487),LL(0xe4456665,0x5a242338),LL(0x899b5dad,0x8520bc7b),L_(0x8d6d1eb8)}, {LL(0xa366ee1b,0xc17b463a),LL(0xa3b19828,0xa0af7c16),LL(0xddf53842,0x85b5501f),L_(0x8d208a89)}},
   {{LL(0xc8d6368a,0xcd7b3fe1),LL(0xd2025e3b,0x124ab94b),LL(0x6971b236,0x961e282f),L_(0xedb43ca7)}, {LL(0xb92ac7ae,0xc76e5b2f),LL(0xdc4501c0,0x8df11605),LL(0x0055901f,0x666181f1),L_(0x80bfc257)}},
   {{LL(0x96fcbeb9,0x745bf74b),LL(0xd548eabc,0x24675bb4),LL(0x531c5deb,0xff3b4dca),L_(0x0b69ba2e)}, {LL(0xd74deba3,0x97100731),LL(0x585b1341,0xa26f7f21),LL(0xb1b02a6f,0x396cb1a0),L_(0xa9fd8a08)}},
   {{LL(0xc6165fe1,0xc07709d3),LL(0xe9ce0fc0,0xfdb883bc),LL(0xb1a5a67b,0xfd535ec8),L_(0x8218f85c)}, {LL(0x8234b459,0x45bb7b81),LL(0xc140afa2,0x1f4c2d89),LL(0xa1cee750,0xcf55f516),L_(0xdada8f87)}},
   {{LL(0x0e3c9ace,0x75c44136),LL(0xf44b034f,0xbe92eabb),LL(0xf4ca1355,0x9e459c32),L_(0xdf9382e9)}, {LL(0x4bc1af9f,0xc06b8bfe),LL(0xe071459b,0xe0b85f75),LL(0x7d557c99,0xff091685),L_(0x408afbfc)}},
   {{LL(0x90ad8902,0x5513303a),LL(0x21006721,0x9b18a004),LL(0x8286d6c6,0x73c22592),L_(0x41ba22e9)}, {LL(0xf6402ecd,0x2bbee8cd),LL(0xb578647d,0xab06f9cb),LL(0x42089652,0xc4a5e5a0),L_(0xed915521)}},
   {{LL(0x6925cada,0x6d3b1102),LL(0xc49fffc0,0x2e19a20a),LL(0x1593ba2f,0x8c091708),L_(0x1ec6ed8c)}, {LL(0x711ad8d7,0x1a65086c),LL(0xef09f54a,0xc052325c),LL(0x182117dd,0x865d46f7),L_(0x6f7c505b)}},
   {{LL(0xfb548314,0x7a6b6eef),LL(0x1dd26e62,0xec447933),LL(0x6a3d4ff3,0x74866913),L_(0x908f5992)}, {LL(0x38316d03,0xb9635b34),LL(0x9f9cb069,0x2db27176),LL(0xac233369,0x567be1c0),L_(0xcafbd98f)}},
   {{LL(0xcf5edce1,0xa18e3f0f),LL(0xa7d8738b,0x05816a5b),LL(0xa26ca8e1,0x00da101c),L_(0x41646cb9)}, {LL(0xe04bccb8,0xa1f2efbe),LL(0xf3c3eccf,0x67b00a64),LL(0xdd0dbd8b,0xc0be2197),L_(0x534aabd5)}},
   {{LL(0xe5a39b64,0x0a586a5d),LL(0x398f5b5b,0xe22752b9),LL(0xfc9ab49c,0x72fc1c09),L_(0x8776fb50)}, {LL(0x8b426975,0xd51e5946),LL(0x3da2aa18,0xa79d2b5c),LL(0x8d8169eb,0xe4ca8644),L_(0xcedb668c)}},
   {{LL(0x923206aa,0xd8894d89),LL(0xc4c6cb55,0x992f9c2f),LL(0xd0679c8b,0x0cde00fd),L_(0x6c539dfb)}, {LL(0xf07baea9,0x3f8e094d),LL(0x5d6f7d74,0x3d3b3315),LL(0xbfbe9e92,0xd269b8cc),L_(0xa2a08cef)}},
   {{LL(0x88e7f568,0x8dcb5c06),LL(0xb11d1a28,0xc8f6e06b),LL(0xa6e90095,0x905b4a44),L_(0xcca85279)}, {LL(0x63aa3a4f,0x1e51e461),LL(0xe2429d1d,0x5fb4cf42),LL(0x35d139e9,0x8725096b),L_(0x6621d9ac)}},
   {{LL(0x2005767c,0x95ce3a0f),LL(0x0f37bc40,0xb593a6ec),LL(0x2c5a09aa,0x762f3669),L_(0xfe96bf28)}, {LL(0x52242367,0x88d38ecc),LL(0xe7e051f4,0x759292de),LL(0x8843ff7b,0x041ca14a),L_(0x1e08ac30)}},
   {{LL(0x0062aa13,0x4bff8a1d),LL(0x40899756,0x28b33bc2),LL(0x966aee37,0xb4b3e7e7),L_(0x7120e524)}, {LL(0x7f31f05b,0xc7be6aaf),LL(0x3e4b1311,0xc2c97f7a),LL(0x7401d050,0x1475b928),L_(0xd263113e)}},
   {{LL(0x3150b4ab,0xd53ede1b),LL(0xbf73ccc0,0xe427c40d),LL(0xf4e2d20a,0xe5dd8b02),L_(0x2ff70604)}, {LL(0xac4f8ba8,0xaaacf4ec),LL(0xf84c5c8d,0x8e6b2783),LL(0xd9aeb4f2,0x4019cd79),L_(0x4c8f57ec)}},
   {{LL(0x680b826c,0xc0a2ee02),LL(0xc24d0dcb,0xb25a4e29),LL(0x8b33b701,0x3a2939d0),L_(0xffb21809)}, {LL(0xe6feb01c,0xd1311810),LL(0x603004cf,0xa701c8f2),LL(0xdbcecc81,0xeee95871),L_(0x69de26b2)}},
},
/* digit=5 base_pwr=2^35 */
{
   {{LL(0xb5c4c69c,0xf507ee25),LL(0xab3e1589,0x454a0f0f),LL(0xf21fe543,0x29d7720a),L_(0xdd2cdf95)}, {LL(0x7c33ec2b,0xb4054a86),LL(0xc7aaaa08,0xdb971594),LL(0x8a74b5b0,0x8850caf4),L_(0xdd2d6666)}},
   {{LL(0xa04cb663,0xf30acc6a),LL(0x8b4f4dbb,0xd7560ec1),LL(0xb8b6110c,0x4f8fab99),L_(0xc43951e7)}, {LL(0x4308e609,0x2c9d6af5),LL(0x7c9841d9,0xca6486bf),LL(0x64c7b963,0x28ca7953),L_(0x9438ea6a)}},
   {{LL(0xbcc43cea,0x3055b2de),LL(0xf0288d97,0x68461c5f),LL(0x4caf7c30,0x84c1743c),L_(0x499d304a)}, {LL(0xb5b3a780,0x0e121552),LL(0x50cacc00,0xc38f8d3d),LL(0x6cee1666,0xe3236cdc),L_(0x54f19290)}},
   {{LL(0x4220e848,0xa785596f),LL(0x4ad330d5,0xfa4258dc),LL(0xf4276ec4,0xfc15e9f1),L_(0x7728478f)}, {LL(0xa58a670e,0x9382335f),LL(0xfea55d50,0x3477647b),LL(0x311f9766,0x4f5dbddc),L_(0xd6c5903a)}},
   {{LL(0x69933c7c,0x21ea5af6),LL(0x5f27f3cd,0x4b132119),LL(0x12e00aff,0x0ff8497a),L_(0x9216e688)}, {LL(0xb8a8554e,0x7d054f07),LL(0xc58c4de3,0xb896b1fb),LL(0x91dbc7f3,0x6470c920),L_(0xcdac5d1d)}},
   {{LL(0xc7b39dea,0x93ff36ec),LL(0x7d199372,0xa6f2dcec),LL(0x802aa4a0,0xdbd9b943),L_(0xae6d61fe)}, {LL(0x8c1a0edc,0x7cecf63b),LL(0xfd6604d6,0x54af7ee0),LL(0x78b1b385,0x0498d501),L_(0x57443efb)}},
   {{LL(0xf34b8647,0xe5ba9123),LL(0x4408fe5c,0xc14ff9dd),LL(0x42cbd03c,0x2891aad9),L_(0x1d5c6bb5)}, {LL(0xf01f7309,0x55808c68),LL(0x50a8d621,0x2ceb5051),LL(0x9d313349,0x2f3b809e),L_(0x28922b09)}},
   {{LL(0x59881515,0x919ddc75),LL(0xbff9ce4a,0xf87e0baa),LL(0x3037e369,0xa090d36b),L_(0xe32f5ba4)}, {LL(0x579ffa97,0x38e224e4),LL(0x43d0a980,0x2cadda68),LL(0x702d5e89,0x12eff4f2),L_(0x4c7be34a)}},
   {{LL(0x52239089,0xfabf2649),LL(0x653b3f99,0xe9f6b8df),LL(0xe84977d6,0x3f58519b),L_(0x6dfea3c8)}, {LL(0x4357d0b9,0x6c5351ad),LL(0x2f6d5e66,0x1253d822),LL(0x9524c5cc,0xfbce31d7),L_(0x55b44df1)}},
   {{LL(0x2e93f071,0xfcee759c),LL(0xe63cea29,0x132e8c42),LL(0x6a329ada,0xec4a7a29),L_(0x3f854cea)}, {LL(0xd5e2b375,0x36a40bf0),LL(0xb7d3f3d0,0x9da99c9f),LL(0x25e16e85,0x7730bda6),L_(0x17380d79)}},
   {{LL(0x4e15a70e,0x37a376cc),LL(0xcebedfe1,0xad1329bb),LL(0xe14b9f7a,0xac8b6391),L_(0xb426bc37)}, {LL(0xe412693c,0xb1a885d1),LL(0x817a04ef,0xdd7cdcbf),LL(0x06667f09,0xfb4d7bf2),L_(0x9f6db5e5)}},
   {{LL(0x350411ec,0xc83be0ea),LL(0x021990cf,0xc7612043),LL(0xbb338804,0xfb1e56ce),L_(0x4934fcc4)}, {LL(0x4b75557c,0xfed92eee),LL(0x4d6a1f0e,0xc3259e9d),LL(0x61eea795,0xc28b498a),L_(0x10a35828)}},
   {{LL(0x7363a98f,0xe21a8af4),LL(0x6f0486ce,0x2ee0c526),LL(0x081ad4e5,0x50ce73f9),L_(0x41a90bde)}, {LL(0xfce29613,0x52824a9e),LL(0xc969520d,0xb689cfc9),LL(0xac6e83b5,0x006e1f4f),L_(0x134a6574)}},
   {{LL(0xd0a972a9,0xe471b9e4),LL(0x53835085,0xcfd1906d),LL(0xd7accdbc,0x293a4390),L_(0xe0eb82dd)}, {LL(0x86a243d9,0x06c24954),LL(0x443f28e5,0x3453aba8),LL(0x25495810,0x36f891b2),L_(0x86536633)}},
   {{LL(0xba90affa,0x4d72eeba),LL(0xedf3c972,0x81c31d9c),LL(0xe70f3373,0xa716b2aa),L_(0x26ec6732)}, {LL(0x42ac4253,0x6f657c45),LL(0x636e8421,0x7d074643),LL(0x461026d7,0x0dde683d),L_(0xddec4661)}},
   {{LL(0x4b805782,0x8dd02690),LL(0xd5dfb670,0x1d04223d),LL(0x83277841,0x7ea96551),L_(0x5fe1e720)}, {LL(0x4c78c04e,0x70dde026),LL(0xa3bb08d2,0xb58c0bf7),LL(0xae645b00,0x3afcf537),L_(0x1f93a66a)}},
   {{LL(0x708a8084,0x5f7d6f8d),LL(0x3c0562fc,0x141c5fc5),LL(0x61a1cb60,0x78ecbf19),L_(0x8914e2cb)}, {LL(0x4cb03ebd,0x9be1423d),LL(0xad53b1c9,0xd63cb771),LL(0x51acdaa0,0xa32591dd),L_(0x45b4e835)}},
   {{LL(0x8de09e87,0x76e08928),LL(0xf8aeca8e,0xa653c233),LL(0x59256cc9,0xd8d51507),L_(0x77e6aff7)}, {LL(0x47309995,0x0f159c21),LL(0x9d735f13,0xfe9fe5bb),LL(0x0339c473,0x16784f00),L_(0xa2846dae)}},
   {{LL(0xf2b37281,0x1cbd8aaf),LL(0x4c543f48,0x8a600a72),LL(0x2324a737,0x644765ee),L_(0xdb3c4503)}, {LL(0xb4dd3d31,0xa23d1c2e),LL(0xe86265b8,0x953388df),LL(0x864110e9,0xe9be065f),L_(0xb52e4596)}},
   {{LL(0x9c32af39,0x69ee0a06),LL(0x9737e953,0x1fd2a13b),LL(0x38105c26,0xfe12d778),L_(0x620245c1)}, {LL(0xd4a50775,0xd5b96217),LL(0xc8c3c80c,0x0f0582fa),LL(0x9f4dc144,0x87629c9c),L_(0xc6b8bdb2)}},
   {{LL(0x377dc9b7,0x9e11a84c),LL(0x6df2302d,0x610e3f5b),LL(0x9becf104,0x87cc9a9b),L_(0x9b8f3304)}, {LL(0xbefa56b9,0x89fa53e3),LL(0x7b1cfe6b,0x0b3ed786),LL(0xdb99856d,0x68d6dbd1),L_(0x9d87c161)}},
   {{LL(0xdd0b49b0,0x7e561904),LL(0xfe908e50,0x3056fb6f),LL(0xab404bf8,0x4911d8a5),L_(0x69dfa8b2)}, {LL(0x91807632,0x606c33cb),LL(0x776c9da6,0x791f57ef),LL(0x525adb32,0x818f0a1f),L_(0xf0db232f)}},
   {{LL(0x1094f387,0xd1afe50b),LL(0xfdce1f8c,0xab294ef8),LL(0x69998dcc,0xe0d8fa53),L_(0x57be6614)}, {LL(0x6ccabafe,0x64a8c67c),LL(0xf805f10c,0xd4a601a4),LL(0x8082a5ca,0x1c89a181),L_(0x5727dd5a)}},
   {{LL(0x4855546d,0x6d3c05ab),LL(0x5bddd7e0,0x44996dfa),LL(0x9e6dd61e,0xe5054752),L_(0xb61ac751)}, {LL(0x4c1c7979,0xa7675178),LL(0x4f0ea0a7,0x72b800a3),LL(0x307c1a34,0xafc9ca2a),L_(0xfac51887)}},
   {{LL(0xf1254532,0x1e52b09c),LL(0x278095e7,0xfc186009),LL(0x579623ed,0x0b5f7250),L_(0x8c426eed)}, {LL(0xa4d79ef8,0xff4c374a),LL(0xe0d3f5d3,0x252cb607),LL(0xf8e76342,0xfcc58682),L_(0xf5c2231c)}},
   {{LL(0x586d393b,0x480cf461),LL(0x4ae96cad,0x13a215d3),LL(0xe3cf0a14,0x82460da2),L_(0xe34161f4)}, {LL(0x188125b8,0xcd5956cb),LL(0xe0c83545,0x9e63be58),LL(0xbfc652be,0x275a3495),L_(0xc74408a0)}},
   {{LL(0x965c6e1a,0x0b8b60d7),LL(0xcfd86827,0xd6030340),LL(0xc11f7050,0x4e201a9c),L_(0x3347de21)}, {LL(0xdd1a6a76,0x73f70f06),LL(0xd2c5e44a,0xebc85954),LL(0x11a05e07,0x0cd2e37d),L_(0xa7ea0cdc)}},
   {{LL(0x7afa362e,0xf7c5f96b),LL(0xce53fd1c,0xfa20c1e3),LL(0x6be4e8e8,0x02b63df7),L_(0xcb0474d1)}, {LL(0x3453fb75,0xf6fdf958),LL(0x30ff85d7,0x502b7475),LL(0xadf32795,0x20557243),L_(0x066b61d1)}},
   {{LL(0x446cdc62,0x8eb329bf),LL(0xa1b5aa2f,0x369f125e),LL(0x6c894dd3,0x9db7bb62),L_(0x4e250042)}, {LL(0xc4277491,0x31b2dbea),LL(0x4c24c213,0xb650ae0a),LL(0x20fa0957,0x29da364e),L_(0xb22a0abe)}},
   {{LL(0x3d0c5314,0xaa1e00ca),LL(0x4668311c,0xe23097a6),LL(0x18e7fcd5,0x790fdadc),L_(0x6c84e798)}, {LL(0x00c47191,0xf4068839),LL(0x0925c817,0x5b080743),LL(0x257f4df1,0xd4b97dd1),L_(0x0e714d67)}},
   {{LL(0x9e9f267a,0x7fa3e72a),LL(0xd9edf53c,0xec547442),LL(0x303f3000,0x74ba5f3a),L_(0x3088e019)}, {LL(0xe130545a,0xf97b4e0b),LL(0xaed0e113,0x7b3e3dc4),LL(0xb3342459,0x849b0cbe),L_(0xa9fbffa7)}},
   {{LL(0x156bffa4,0xef4fdf72),LL(0x85aca1ed,0x0fb07e25),LL(0x373456aa,0x0c136024),L_(0x3fbc3b2d)}, {LL(0x75063486,0x09872cea),LL(0xed9cc9ba,0x1f75eff6),LL(0x2982d676,0x46918abe),L_(0x638bc02a)}},
   {{LL(0x62d87802,0x8ac63776),LL(0x0082e1c8,0x3df57bb4),LL(0xfb67a75d,0x872c8f87),L_(0xe6292f25)}, {LL(0x49548c68,0xa4ab6141),LL(0x1ee50877,0x5ae2f1d3),LL(0xccb21fe7,0xcc4c5d51),L_(0x4dd1a7d5)}},
   {{LL(0x552b1d5b,0xc91b6f0a),LL(0x8cacee59,0x26244e6c),LL(0xebe1c6fb,0xec4bbe5c),L_(0x6a216c08)}, {LL(0x35cfef8d,0x09fa8c0f),LL(0x17b60280,0x77fdabaa),LL(0x4bbc3792,0x94421437),L_(0xb01866d7)}},
   {{LL(0x820d6809,0x9b55ccda),LL(0x95bebe41,0x1046f37c),LL(0xed3f6fb8,0x07a02e6b),L_(0x5d6bfa33)}, {LL(0x6676830e,0xf82f2f79),LL(0x8ba9b566,0x35492432),LL(0x79d53ab5,0x16a996a8),L_(0x912fe1fa)}},
   {{LL(0xaa0f7a90,0xe4d29021),LL(0x63d69e6f,0x9b86e6df),LL(0xdb26a514,0xf567f4e0),L_(0xe0d08a4c)}, {LL(0xc1decce6,0x9de6495a),LL(0x037496f1,0xda23fbb5),LL(0x6557417b,0x7d23ff08),L_(0xb80f990a)}},
   {{LL(0xc3c2aab9,0x2e3254ff),LL(0x0c882e32,0x48576b69),LL(0x2e3a1aff,0x4e27496a),L_(0xa8981080)}, {LL(0x497974dd,0xb0b36f57),LL(0x57530d61,0x1d0467dd),LL(0xbd12d521,0xee061337),L_(0x2a16986e)}},
   {{LL(0x3a9a2700,0xfa637395),LL(0x08e9776b,0xbc2048f0),LL(0xf8ae9c4f,0x7229d31e),L_(0x5d5cdb49)}, {LL(0xd86d5de3,0xd7e8cbb1),LL(0x7d36c9cc,0xa98c305d),LL(0x382a852f,0x7668edae),L_(0xb66ddb2f)}},
   {{LL(0x6d33747f,0x6bae97b0),LL(0x258e1f01,0xc0b8420d),LL(0x0d1395e0,0x5d523a37),L_(0x20fd468f)}, {LL(0xf196c7bd,0x940889f2),LL(0x2b93cb81,0xa914e0ea),LL(0xce35713d,0xcb59a392),L_(0x4ff9fc32)}},
   {{LL(0x984bd24d,0x17b7b322),LL(0xe322bc50,0x33279a91),LL(0xa03ade42,0x3ac696ec),L_(0xa59b7158)}, {LL(0x36f23ecf,0x4faba2b9),LL(0x22a5cf90,0xf70fdee7),LL(0x7bbc4522,0xeb035a5f),L_(0x58a1a1e9)}},
   {{LL(0x5ddc6b6d,0x150eef02),LL(0x52864a3e,0xdd3f93bb),LL(0x3696d783,0x15012933),L_(0xad3eb1b1)}, {LL(0xff6e69b6,0xc84755b9),LL(0xb7d6ce03,0x118ab913),LL(0x2da1eb13,0x99325702),L_(0x1f352fb1)}},
   {{LL(0xdfd09cd9,0xd7ddee82),LL(0xfd22eb92,0x93460443),LL(0x87b5bd4a,0x0d5541f4),L_(0xf4336176)}, {LL(0x2e83bcbb,0xd3e283ed),LL(0x6124a0f9,0xe52b803b),LL(0xb28eb570,0x4a8c77d7),L_(0x782341d6)}},
   {{LL(0x7636205a,0x18df4a69),LL(0xc1129737,0x70a8c000),LL(0xf96be149,0x69d0bc39),L_(0x8091b05b)}, {LL(0x59f09aa5,0x53e28bf7),LL(0xb0469f79,0x3ef52081),LL(0x28ac324e,0xa781c57c),L_(0x10eeaa88)}},
   {{LL(0xb145dd4e,0xf8de49f2),LL(0xbbb26b41,0x7a359488),LL(0xa5fe71f0,0xea4e5a13),L_(0x0d776617)}, {LL(0x8252a9aa,0x9870bd6d),LL(0x069f0a83,0xd7e75649),LL(0x73608571,0xb0499f6c),L_(0x9dd9cbb8)}},
   {{LL(0x63626f39,0xee89a981),LL(0xe0aee1b4,0x45b932da),LL(0x25321bbf,0xdd6b2382),L_(0x0c03fd27)}, {LL(0xbd649607,0xdbca6a1d),LL(0x6b8f9b4b,0x1e699d4b),LL(0x940460cb,0x7938144e),L_(0x8c3cb198)}},
   {{LL(0xa571f0be,0x50d17043),LL(0xcd56a2ea,0x7b24e599),LL(0x15047c8b,0xe06a40d9),L_(0x99dac8b4)}, {LL(0x5bc5aeb7,0xaeb386d4),LL(0xe9f713d9,0x40fabb3f),LL(0x62bdfdf5,0x6aace159),L_(0x8e028ae5)}},
   {{LL(0xde610cbf,0xf6b35535),LL(0xfe554daa,0xb7d2826a),LL(0xb4edc29d,0x03e1e5e0),L_(0xd38477a9)}, {LL(0x789b199f,0x03331148),LL(0x4a8a863e,0xa966c649),LL(0x0851cfcf,0x1622ec1d),L_(0xfa31d354)}},
   {{LL(0x888b7c2c,0x07cb3e37),LL(0x660e7e68,0xb017d5a9),LL(0x7f7c933f,0x42119d1d),L_(0x47d400e6)}, {LL(0x5ef83e46,0x0109f6b8),LL(0xbc268020,0x6e139180),LL(0x1b624dea,0xa2bd127f),L_(0xd4e19a66)}},
   {{LL(0x54ae0fce,0x9a79307a),LL(0x06792812,0x6d342e22),LL(0xb177e174,0x2e3ee119),L_(0x56fea730)}, {LL(0x5e84ad5f,0x08e8f89c),LL(0x12504248,0x425e69c5),LL(0x42b47533,0x0b2e547d),L_(0x0cd14526)}},
   {{LL(0x041c2bcb,0xd43f468a),LL(0x6c6a96c8,0x497dff29),LL(0x58417877,0x9551c8f0),L_(0x738c007b)}, {LL(0x1e1b3a52,0x7aa44dae),LL(0x4cb0aa92,0x82e3854e),LL(0xe5d6128d,0x2f25adda),L_(0x1565a105)}},
   {{LL(0x92365b9a,0xdd4e8322),LL(0x332f51e4,0xe083981e),LL(0xbf7947a0,0x5b678747),L_(0x3196029e)}, {LL(0xc98f910a,0x0ca71f21),LL(0x51b450d0,0xe31a901e),LL(0x922f7c66,0xeed6566d),L_(0x1ed09e18)}},
   {{LL(0x274ea5df,0x52fe1ee1),LL(0xf298e59f,0x7dc25d85),LL(0xd6285877,0xda84cd8b),L_(0x7cf3ae58)}, {LL(0xbb649ac2,0x1e3ba4b5),LL(0xb225291f,0x6f9c137c),LL(0xbd251ef1,0x5e5495d3),L_(0xb7512d10)}},
   {{LL(0x8392e4dd,0xdc8ee05c),LL(0x9d3efe45,0x4eeeb0da),LL(0x0d91ecaf,0x31e89bf7),L_(0xdc635197)}, {LL(0xeca9047d,0xee9a09ba),LL(0x789d1a79,0x81d64f7e),LL(0xea9e0b50,0xc7b60917),L_(0xb463b663)}},
   {{LL(0xb48d4d3f,0xb23798db),LL(0xfd959cce,0x4a87382c),LL(0x829e3bb6,0x492aac4a),L_(0x4bf01b8e)}, {LL(0xf58a6464,0x96935f47),LL(0x82b212c6,0x3396e950),LL(0x6d21b8d5,0xe244afe7),L_(0xa642079f)}},
   {{LL(0x890b338a,0x01b0423f),LL(0x8f95cd41,0x6d8cdb00),LL(0x68218ce9,0x7809fc35),L_(0x1fe627f9)}, {LL(0x446c771b,0xaae3f3c1),LL(0x622ee718,0xfb834953),LL(0x89128662,0xff1c7590),L_(0x2c4b37c2)}},
   {{LL(0xe1d9d534,0x52f4f30f),LL(0x9215d0dc,0x3a3e06d4),LL(0x6d0d3574,0x9ec8c4a0),L_(0x4abdacfc)}, {LL(0x06630325,0x8d559ed9),LL(0x7bfef54b,0x295b5ede),LL(0xd0a2d07d,0x8d6e5cd0),L_(0xb74df6fa)}},
   {{LL(0x42c70b2c,0x4f4ae671),LL(0x613fd1c2,0x536e7d64),LL(0xfaa3003d,0x94dc0a21),L_(0xd4e576a1)}, {LL(0x1592950f,0xb4369ba0),LL(0x3179ef67,0x5f932085),LL(0x6309e764,0xe5462315),L_(0x3bcaa778)}},
   {{LL(0x08fa6e2f,0xec60e2f4),LL(0x2b9b6197,0xa152beaf),LL(0xcc5f8968,0xbce9a53b),L_(0x1ef40db3)}, {LL(0x076de5d9,0x31df6025),LL(0xe8f5bdc2,0xce591c62),LL(0xead89b13,0x52e8a741),L_(0x12358a25)}},
   {{LL(0x179fcba1,0xc81fc0c7),LL(0x35daa530,0x8eb90798),LL(0xbdb2ded4,0x820ef61d),L_(0x6b0231aa)}, {LL(0x2b8a3cbc,0xf522badb),LL(0x0e482ed6,0x93d7057a),LL(0x0ff17306,0x70098d33),L_(0x63ccaacb)}},
   {{LL(0x8e019362,0x2f914dd7),LL(0xf4501079,0x2acdf294),LL(0x2848b0db,0x0f138ea9),L_(0xaf353a81)}, {LL(0x4ea1b03a,0x49ae634d),LL(0xd3a8c33b,0x113138a4),LL(0x8ed0557e,0x2b0adbad),L_(0x739b3ce7)}},
   {{LL(0xb33942af,0x1ecf958e),LL(0x54804872,0xc66d2148),LL(0x1f27ea10,0x011365d7),L_(0x6ce35983)}, {LL(0xc4208dd6,0xb8f4d08e),LL(0x882dc840,0x3e553026),LL(0x29499eb7,0xea028ba1),L_(0xd9859a66)}},
   {{LL(0xa6f2df8a,0x5bf03737),LL(0x134f5c42,0x924c5819),LL(0xce383c9e,0x910aeacb),L_(0x2daae049)}, {LL(0xbfdc9b59,0xe45c96a9),LL(0x3a5b8130,0x726cada6),LL(0x8177cf88,0xa738df5c),L_(0x9be237a5)}},
   {{LL(0x21f30c43,0x8cd0ef97),LL(0xfdd4b300,0xe6a63e2d),LL(0xd4df1941,0x57a01e9c),L_(0xf32119bd)}, {LL(0x9c025c37,0xdc96eafb),LL(0x136dc55e,0x95fe2aa2),LL(0xae8f18c9,0x239e5894),L_(0xb671d770)}},
   {{LL(0x9c68e7cd,0x22874bf5),LL(0xff1a555a,0x64db3073),LL(0x3b1f7071,0xa14665fa),L_(0xad6ab5ea)}, {LL(0x78fc6e69,0x77eb0618),LL(0xa6cf5f0f,0x6d531160),LL(0xf5f0fda1,0x429c094e),L_(0x78109f67)}},
},
/* digit=6 base_pwr=2^42 */
{
   {{LL(0x570bd76c,0x42617e4e),LL(0x53cb59f9,0xec1fce2c),LL(0xbd807766,0x5098d9d8),L_(0xf56acb00)}, {LL(0x468a4f17,0xada13650),LL(0x8356714e,0xb5aa694d),LL(0x1dce66de,0x5318219a),L_(0x3439525e)}},
   {{LL(0xd30a5801,0x97115f9f),LL(0x2621a7d4,0xbb10a57f),LL(0xeff5c174,0xadf87884),L_(0x01243ae4)}, {LL(0x162e3c5c,0xa2bc843d),LL(0x6f4a3579,0x25478286),LL(0xd6f94dfb,0x31bb4258),L_(0xa760a06c)}},
   {{LL(0x6c5c8428,0x46351cc8),LL(0xa082b7e1,0xb964daab),LL(0xdd1d4004,0xc4d0e945),L_(0x3e7f4540)}, {LL(0xb82dabae,0x7d7ac429),LL(0x956426a7,0xeed99c4b),LL(0x7dafa986,0xcbdad15b),L_(0x8d046994)}},
   {{LL(0xe5e0aad9,0xc7b54b13),LL(0x257de915,0xe9d8af38),LL(0x008e0bc3,0x71a411c5),L_(0x9f3a29a1)}, {LL(0xc482b233,0x07289ac0),LL(0x780d30d5,0x68d763c3),LL(0xa128b566,0xc589940b),L_(0xd834f239)}},
   {{LL(0x533b21ee,0xb86a434f),LL(0x9be1cf11,0x5f1dadf4),LL(0x5b2f618e,0x35caca2b),L_(0x9da9d5f4)}, {LL(0x74cf7bbd,0xb14dae65),LL(0xa4192a56,0x27c04cc6),LL(0xf5af0266,0xc41f13b5),L_(0x350b4db7)}},
   {{LL(0x5727ef4c,0x9fcc3538),LL(0x2806003f,0x957682ab),LL(0xe33ed1c3,0x9f83349f),L_(0x02ec97a0)}, {LL(0x5961423d,0x5148178e),LL(0xeab94e36,0x66e2c6e5),LL(0x5986c53b,0xd5c63ac2),L_(0x9b96cae8)}},
   {{LL(0x9082e762,0x5661867b),LL(0x29ca7816,0x8ece3937),LL(0x30442089,0xeddde973),L_(0xee4b63a6)}, {LL(0x3e2f6790,0xd4f64e91),LL(0xd529fa9b,0x49fdfa45),LL(0x0b104178,0x78b6687a),L_(0x40d1484c)}},
   {{LL(0x9f724692,0x0ca4bf22),LL(0xb272519e,0x730e0881),LL(0x4bbc9bf5,0x91d1e24a),L_(0xc6e5ce02)}, {LL(0x26de9372,0xd72ae23c),LL(0xc7a7721b,0x6ca80e46),LL(0x75b289b3,0x777ca3a2),L_(0x28a90ef0)}},
   {{LL(0x7d3fcac7,0xebb7ace4),LL(0xa09641c2,0x48bb32ea),LL(0x0be22833,0x3f76d58e),L_(0x7e7a5d70)}, {LL(0x8e3df32c,0x00f1bf36),LL(0xee4b523d,0x8b204653),LL(0xdaeaa58b,0x3263322d),L_(0xe7975fbd)}},
   {{LL(0x648b273e,0x2e05e12c),LL(0x0e313d88,0x3a732a3d),LL(0x691aae3b,0x6cd91464),L_(0x8bd52c9b)}, {LL(0x208c6c3c,0xe4c05b8f),LL(0x66fafda4,0xd9e2218d),LL(0x59cd9a06,0x835930df),L_(0xfb01c010)}},
   {{LL(0xd984c49f,0x888fc640),LL(0x0bbd6a5e,0x9bb00fea),LL(0x4da4ce0e,0xaffa26c5),L_(0xcc14f2af)}, {LL(0x2307b844,0xc6cd9202),LL(0x340af7d9,0x3f8ed768),LL(0x518df91e,0xeadd2347),L_(0x8dc696be)}},
   {{LL(0xaf9c00ec,0xb90b0248),LL(0x2d2b08b8,0x45a5507d),LL(0x2e1dac5f,0x8db688a8),L_(0x3e2c52bf)}, {LL(0xa655bbf1,0x0fdb8f93),LL(0x95c256dd,0xfdb2b439),LL(0x9751d174,0xaef8bdee),L_(0x54ac1077)}},
   {{LL(0xb601c37a,0xdaadd78b),LL(0x450b3671,0xb65d4dfa),LL(0xb3e4d63b,0x653e4965),L_(0xf559fd3c)}, {LL(0x8cbfa3ed,0xbaaa17fd),LL(0xcd2af6b1,0xe29a388e),LL(0x50eade40,0x2ec73453),L_(0x0ca27599)}},
   {{LL(0xbac1435b,0xaa859487),LL(0xc7540522,0xb5b2445b),LL(0xe22a73a2,0xaa1116ed),L_(0x2e7ed6df)}, {LL(0xf9d72a86,0x616cb03b),LL(0x61a4a6e7,0xcdf54ee7),LL(0x4cf9a426,0xe0208859),L_(0xc62b0ac4)}},
   {{LL(0x5300262d,0x7faf0467),LL(0xc2571f4f,0xff47f116),LL(0x93b14cbc,0x484a66c2),L_(0xf8b658b8)}, {LL(0x654db6ca,0x20220d13),LL(0xfcb07e76,0x1162cfde),LL(0x897e3ec2,0xf29d6809),L_(0x6eee73ae)}},
   {{LL(0x2c67eef1,0x1a833a88),LL(0x40f7ac97,0x8aa0f2d7),LL(0x5fc1c69f,0x9f06aae9),L_(0x08b7e77d)}, {LL(0x4ecfeaa6,0x17fda35e),LL(0xeac8ca12,0x338ff3df),LL(0x0013029b,0xe6215b79),L_(0xd490a9e5)}},
   {{LL(0x45530708,0x3d6ab800),LL(0xe7c57017,0x34dca330),LL(0x5aa02638,0x4dc086c8),L_(0xba336129)}, {LL(0x6df633bc,0x90bbb8d8),LL(0x33ca4c46,0xc7c28c06),LL(0x8dcb3bae,0xa0c24314),L_(0xa41cf732)}},
   {{LL(0x7f227cd0,0x8cd51cfe),LL(0xb7fcfdc2,0x68ea58e2),LL(0x9f7bcc04,0x3003cc72),L_(0xd4b488a1)}, {LL(0x5b982b6b,0xd5ebd68b),LL(0x4020c3e7,0x675b50df),LL(0x9679336c,0xa8fee552),L_(0x24805e44)}},
   {{LL(0xe51e0a9f,0x75eb0c92),LL(0x760831bb,0x9af943f7),LL(0xa54c748c,0x28a759fb),L_(0xe246d8ba)}, {LL(0x1cea2c6e,0xb0384781),LL(0x2f1568a6,0xf0f73ecd),LL(0xb65d1ecc,0xd26a77df),L_(0xa639e1e0)}},
   {{LL(0xcb23a542,0x51dd59db),LL(0xac405e3b,0x24f458a4),LL(0x7d0b3dda,0x26b341e1),L_(0x9072bcaf)}, {LL(0x80d2e583,0x9e2fd334),LL(0xd09edd03,0x114215cc),LL(0x794f6e44,0xe3bb54b5),L_(0x4abbf55b)}},
   {{LL(0x01429cdd,0x7fb2c053),LL(0xaaeddc51,0xee0b3bcc),LL(0xcf236537,0xca6a0781),L_(0xee610794)}, {LL(0x28ee066f,0xe18e5f40),LL(0xb59956df,0xb1e0ccfc),LL(0x2a9ed4ef,0xa109a3af),L_(0x628906f4)}},
   {{LL(0xad211ace,0xe621f731),LL(0x99a069f0,0x072030c8),LL(0xa71de5a1,0x106812c7),L_(0x606b3758)}, {LL(0xeac94530,0x4ce2ef8a),LL(0x57357177,0xac163954),LL(0x778bc2dd,0x2e5cbdab),L_(0xa82c012b)}},
   {{LL(0x0baf79a3,0x9c81a4be),LL(0x4fc50828,0x9363638d),LL(0x8a4d69e0,0x687f0dee),L_(0x92f9ddc4)}, {LL(0xa6d3323d,0x202ed4a9),LL(0x123a8f24,0x1c483cb2),LL(0x5bbe8c1a,0x433b79ed),L_(0x23111e9a)}},
   {{LL(0x3828dbf8,0xb8a0c6af),LL(0x43cd4321,0x9f12e96d),LL(0xdd7d6cea,0xe595dcab),L_(0xd3b3c998)}, {LL(0xad54b44c,0xe8aa36bb),LL(0x142da061,0x11dcc1c4),LL(0xcbe63415,0xeff9d401),L_(0xa4d738ce)}},
   {{LL(0xfef9f39f,0x1d099df3),LL(0x769dfb43,0xbf9c0426),LL(0xe387610f,0x0cce0331),L_(0xd6feb0f4)}, {LL(0x31cb09c5,0xd62d1375),LL(0x10527627,0x53290cd4),LL(0xa90e25d2,0xf1396054),L_(0x6a0d6915)}},
   {{LL(0xf603ca7f,0xe57e3a51),LL(0x26b3d6ba,0x547b9c97),LL(0x5234c386,0x44edaf6d),L_(0x7c4a6d8f)}, {LL(0xcea690d7,0xf13946cc),LL(0x4a083ed0,0x6ad6bb4e),LL(0x9374003a,0x36c2bacd),L_(0x90ccdacc)}},
   {{LL(0x341c30ae,0x7e1c2c9b),LL(0x313b286c,0x17e56dfd),LL(0xb65db139,0x9888703d),L_(0x65b2fe62)}, {LL(0xe07d7043,0x3e36f84d),LL(0x6fc2a51e,0xb4b9c74c),LL(0xbbd020f9,0xed1bc384),L_(0xb260d341)}},
   {{LL(0x4432f2d0,0xa3a488e9),LL(0xf4caef39,0x9efd00d6),LL(0xd17c4829,0xf5128936),L_(0xf742f93b)}, {LL(0x35e93b49,0xae41e05a),LL(0xa5eb9e98,0xe030a464),LL(0x41a1010a,0x1db1bce3),L_(0x7cabde79)}},
   {{LL(0x6fb1c1bf,0xfe5cfb57),LL(0x304db3f7,0x401be370),LL(0x29b356b7,0xef6ca52e),L_(0x16934044)}, {LL(0x131c52a9,0x3084c2e8),LL(0xb1a56ede,0x58a36c5a),LL(0x8cfd7a22,0xa076e835),L_(0xe8967b86)}},
   {{LL(0x74382b8c,0xecf54e6c),LL(0xe7d202a5,0x80a6564e),LL(0x3378864a,0xf29ac0df),L_(0x3af276ae)}, {LL(0x9549b955,0x71036c3c),LL(0x85bde008,0xd3203c72),LL(0x57b0b3ec,0x7ee244a8),L_(0x7831f44b)}},
   {{LL(0x2a2ae111,0xe00edfa7),LL(0x8b506f33,0x6b68f39e),LL(0x8f5215f7,0x0b132723),L_(0x36ee465c)}, {LL(0xe7d50419,0x85673786),LL(0x4b253aad,0x231c0709),LL(0x487f22bb,0xf5e2a237),L_(0x8af08593)}},
   {{LL(0x95692ad3,0x0b27bb26),LL(0xa14fcdb7,0x090be6f5),LL(0x8b45b55f,0xea84dd7c),L_(0xd1971d3e)}, {LL(0x928497fb,0x971d5f28),LL(0x9e1b8d56,0x6339a103),LL(0x9ddbd5c4,0x3fe08c7f),L_(0x71eef4d8)}},
   {{LL(0x987de693,0x1683a455),LL(0xca6c2d90,0xb38e1de8),LL(0xdd0a3684,0xf01af5d9),L_(0x2403e538)}, {LL(0x86ab28a1,0xd4b30cec),LL(0x533a55fa,0x22c7c598),LL(0x01cadc9e,0xb5dd36ac),L_(0xe6dc51f3)}},
   {{LL(0x8baf2809,0x60cc335f),LL(0x1ed24f3e,0xa633db28),LL(0x732a899d,0x7f9cd569),L_(0x5d8b2c23)}, {LL(0x04bce2d6,0x1f59a0d7),LL(0x6ef81d6c,0x6d10e528),LL(0xcbb166ea,0x7fe10619),L_(0x1eb94857)}},
   {{LL(0x04d81b40,0x88a7293b),LL(0x3d8447b9,0x1732c2f0),LL(0x057b8676,0x320d8861),L_(0x5a9e4e63)}, {LL(0x5bfbc591,0x3e5d907e),LL(0xa66f2cce,0x2b35a3fe),LL(0x1597d44b,0x7605acfa),L_(0x756ca23d)}},
   {{LL(0x6112f38f,0xf5ab47b4),LL(0xf577db42,0x5984ae06),LL(0x4fccb7e8,0x81972b3a),L_(0x1d854210)}, {LL(0x5d4f964c,0x905ded37),LL(0xf605f38c,0x6a31fa68),LL(0xfb6ad50f,0xd83dca88),L_(0x6aacdbc7)}},
   {{LL(0x1b0ac423,0x711dde1a),LL(0x227d6127,0xb5e309b2),LL(0xac0aec46,0x17ee236d),L_(0x70e64775)}, {LL(0x600531c3,0xce093e55),LL(0x458395de,0xec04dd4e),LL(0x59639b52,0xb9e01d45),L_(0xa8e6469e)}},
   {{LL(0x1b94ace7,0xb0fee2ec),LL(0x08d97fb0,0x5c0b8f2f),LL(0xdfd59ee2,0x6a154d90),L_(0x35f79449)}, {LL(0x8fb75d5a,0xb692d5e7),LL(0x094a540a,0x080f7773),LL(0x1fbd50fb,0xa4f8f9e4),L_(0xb9d1eb94)}},
   {{LL(0xf63f6da8,0x2aa45a5e),LL(0x711d55a4,0x2d7d0b41),LL(0x29c1bc69,0x8aabebdc),L_(0x8480eff7)}, {LL(0xb0063dbf,0xc38e3c41),LL(0xe276eaf6,0x661ba0dc),LL(0x99541913,0xc19809da),L_(0x50e215d3)}},
   {{LL(0xff252d36,0xa1530a6c),LL(0x253d53ef,0xc8e9b570),LL(0x9793a371,0x3489107c),L_(0x8abf5783)}, {LL(0xf0b49c6d,0x879022b6),LL(0x12bdba9e,0x069dd468),LL(0x2c618b82,0x64bde16d),L_(0x2306b3be)}},
   {{LL(0x2d08df95,0xed7d7c9a),LL(0xd4d38ca9,0xe80a7f84),LL(0x357baf28,0xbb968f21),L_(0xbcca4e7e)}, {LL(0x8315da93,0x96e91923),LL(0x4f08013c,0x4e306972),LL(0xc5ed2a12,0x950bbe41),L_(0x434c75ba)}},
   {{LL(0x39668cb6,0x077167b6),LL(0x7bdd93d3,0xfce255b9),LL(0xb37bc6b8,0xf3809a3f),L_(0x32c36f4d)}, {LL(0x7f61816e,0x44d57109),LL(0x06f0654c,0xb19cf5fd),LL(0xdb202875,0x42100d63),L_(0x79f51335)}},
   {{LL(0xaf246d2a,0x7f3965a6),LL(0x8caa84d7,0x8d8e3137),LL(0x26bde524,0x11b1a769),L_(0xa5eaa311)}, {LL(0xd87dfaee,0x707808be),LL(0x69540405,0x04bb89d0),LL(0xe52cbe87,0x7ec2d88f),L_(0x994f86d5)}},
   {{LL(0x5423908f,0x43d7285e),LL(0x85d0501d,0x49363917),LL(0x6a49fc82,0xf5d8d60b),L_(0x2126dd81)}, {LL(0xa0f34346,0x665a2267),LL(0x1fa53f15,0x1ac20d97),LL(0xffa2e5fc,0x4e6d33c1),L_(0xf9840551)}},
   {{LL(0x9731dc20,0xf1f7a37e),LL(0xc0130bbb,0x69d32743),LL(0xea51b61d,0x05851633),L_(0xaa72b2e1)}, {LL(0x04e8d8bb,0x1c8bd021),LL(0x4806ed2e,0xef76ca45),LL(0xf099b725,0xd230047c),L_(0x5f4e19c3)}},
   {{LL(0x44613937,0x62171491),LL(0xf583d82e,0xa07a7b6d),LL(0x9015ad40,0x306c9555),L_(0x0b81ff7a)}, {LL(0x18925a9e,0xaac18583),LL(0xa0946d46,0x7d220576),LL(0xbbd4708e,0xf151aaec),L_(0xe99b984f)}},
   {{LL(0x6e04c9ed,0xb57b3b22),LL(0x3cdad4a9,0xa3593866),LL(0xa01b71df,0xa63bab57),L_(0x97f85c63)}, {LL(0x773f947c,0xefc9bc76),LL(0x1a1f56a4,0xcb328f94),LL(0xe12b780c,0xb068b3ce),L_(0xc8733217)}},
   {{LL(0xc1916a8a,0x62555b0b),LL(0xe906cdca,0x00203baf),LL(0x82d7d36f,0xd8764be5),L_(0xc4064873)}, {LL(0xf4469260,0xa23e8076),LL(0x7f6613c6,0xb903dea6),LL(0x11610026,0xd83fceb2),L_(0x6e5bcf50)}},
   {{LL(0x2739ed0b,0xcff10abf),LL(0xcaa11cd8,0xe0b269eb),LL(0x92761d90,0xc1c0ca0e),L_(0x70aa2955)}, {LL(0xab5ce41e,0x62aaaef4),LL(0xc2c0d2af,0x5a3ff0d9),LL(0x3aa250ea,0x8f10b784),L_(0xdbfc2028)}},
   {{LL(0xe5bbf62d,0xa53f3f3f),LL(0x5bd69524,0x27dd1951),LL(0xeada36e1,0x40a88ecc),L_(0xd04364be)}, {LL(0xca0a31a4,0xd318529b),LL(0x5b3c2cf3,0xefec04d1),LL(0x49d3e4dd,0x55c20eb0),L_(0x7153c7d3)}},
   {{LL(0xc5f830cf,0xbf310209),LL(0xbaeecda6,0x1372fe7c),LL(0xab6a0dc0,0xaf6004cc),L_(0xabcb8c97)}, {LL(0x8803af61,0xa6088f51),LL(0xf7850d91,0x53f144ac),LL(0xe77f4280,0xd7d06ab8),L_(0x9a5d8d81)}},
   {{LL(0xc4969a0e,0xdf19cc2f),LL(0x433da709,0x8bd7557f),LL(0x554a3a06,0x8fe8b8dc),L_(0x687d6a23)}, {LL(0xf261d773,0x82e07bc5),LL(0x9e80d756,0xd1d7da5d),LL(0x45447ad3,0x19044418),L_(0xae95e54a)}},
   {{LL(0x3956922f,0xf66675c8),LL(0xde2dc4b9,0x20469977),LL(0xf5f39465,0x011863a2),L_(0x1e9807dc)}, {LL(0x960a1dab,0x66b9d7bb),LL(0xdbd5a655,0x074a85ef),LL(0xd9f9bf2e,0x7dd3b230),L_(0x50a9e91b)}},
   {{LL(0xe5be9e2b,0xb5995728),LL(0x0640124a,0xa127aac9),LL(0x23c8cef0,0xf11df0db),L_(0xb1a238e2)}, {LL(0xefcf6a98,0x38cfdd79),LL(0x78034f01,0x899c799e),LL(0x8c265685,0x7cab4cd9),L_(0x1b6b2132)}},
   {{LL(0xe6160917,0x480a36fd),LL(0xe17a372a,0x3f7c1b1c),LL(0x91f84a8c,0x85381912),L_(0x202d4232)}, {LL(0xb7408da9,0xe263f58c),LL(0xd1c14892,0x8ca85524),LL(0x8c2f9aa1,0x687fc42f),L_(0xebb9ceb8)}},
   {{LL(0xf5f78e42,0x3455a3e9),LL(0xbbc51854,0x30c1c2c6),LL(0xddf065a7,0x3692f51b),L_(0x5794f41d)}, {LL(0x0f5906c1,0xbfc2af37),LL(0x3df049a3,0x5fa05ed1),LL(0xd9f88327,0xfad22122),L_(0x143991c0)}},
   {{LL(0xcbee3325,0xe6fcf3f6),LL(0x29ff58e0,0xc3e92fcb),LL(0x7c43be45,0x6def0ea9),L_(0x95d4294c)}, {LL(0x72baebe1,0xabf2c782),LL(0x8f7dbb1b,0xff1c5bf6),LL(0x21f7b8ae,0xcada1649),L_(0x1d86e3b2)}},
   {{LL(0x2e618a15,0xe44e287b),LL(0xc0ea1a7b,0x9ec08fdf),LL(0x3cbeeaa0,0x41378416),L_(0x5dfb49d7)}, {LL(0x3defdbe4,0xdd8dc723),LL(0x4d7ec600,0x6d23302b),LL(0xf1849878,0x922ac03f),L_(0x9a57ab5a)}},
   {{LL(0xf4361243,0x5150fa93),LL(0x3ea14eaa,0x1bf1d7e3),LL(0x0106a9fc,0x90827c46),L_(0x7d96ff3f)}, {LL(0xd4951e8a,0x046acfdb),LL(0xa6461b44,0x38e792fd),LL(0x2d8ca425,0x07e228cc),L_(0x4ba3f6b1)}},
   {{LL(0x889c6c45,0x12d9fd78),LL(0xf4406adb,0xa6aed947),LL(0x46f4af45,0x4de1be3f),L_(0xd4630c28)}, {LL(0x932b6858,0x4b284bc1),LL(0xcbab5b0d,0xe27e82bb),LL(0x9200852f,0xe41775aa),L_(0xdd624131)}},
   {{LL(0x50a8e4b8,0x9d917e78),LL(0x558110ca,0x4ac6dc2a),LL(0xeb63b95b,0x585b0c05),L_(0x89d42790)}, {LL(0x876d7f22,0xdd6e7325),LL(0xb4e2f682,0x34a78682),LL(0x55d49e0e,0x7de26ccf),L_(0x076914a6)}},
   {{LL(0xf3430044,0x09139313),LL(0x6fe59fdd,0x32e29482),LL(0x9f0a6227,0x1c584d56),L_(0xffe716b6)}, {LL(0x11cca786,0x53203116),LL(0x82480098,0xb1c8440a),LL(0x0ec04b4a,0xb4fd311a),L_(0xb256043e)}},
   {{LL(0x4a367cf1,0x82ea7247),LL(0x8fd0c543,0x35a8f8e4),LL(0xee0c8d61,0xeb47eaf0),L_(0xb1c6b367)}, {LL(0x7fc38517,0xb4361b1e),LL(0xa69d501d,0x24e2fb10),LL(0x94b8860e,0x1e4ac934),L_(0x473c4fc7)}},
   {{LL(0x9c7f7bc3,0x85016181),LL(0xa46af46a,0xae2e08c9),LL(0x1d49bd53,0xd51d0f76),L_(0x47c1a7de)}, {LL(0x574b1fa3,0x275045e3),LL(0x2b7d8406,0xe63f9064),LL(0xb270bab6,0xcbacaf10),L_(0x1fd8fd5b)}},
},
/* digit=7 base_pwr=2^49 */
{
   {{LL(0x55cc3b94,0x81366649),LL(0xc85bd06b,0x0bb5529d),LL(0x66b83a45,0x020a3d7b),L_(0xd7360ab5)}, {LL(0x6d9c718e,0x4c39583c),LL(0xf32587a2,0x063ccb66),LL(0x8cb7eb45,0x01328b11),L_(0x7b390466)}},
   {{LL(0x7e06eeca,0x2a100811),LL(0x1e68642b,0xf4997f3a),LL(0xf5c07857,0xad98c158),L_(0x3c2d33de)}, {LL(0xf408207e,0xdb7d94f4),LL(0xe7bd848f,0x6379ef09),LL(0xf3bfa4da,0xde775bfd),L_(0xa5183e31)}},
   {{LL(0xd08dfe45,0x9b19df92),LL(0xf42316da,0xfd14af8c),LL(0xcc3f5ace,0x9cb59da4),L_(0x2c68ad5c)}, {LL(0x880b584b,0xe1fa2671),LL(0x0ace93c1,0x76bf0449),LL(0x0c870e3e,0x6fb6d427),L_(0x680b653e)}},
   {{LL(0x70db0533,0x12b3f8dd),LL(0xb2fbdce6,0x167ab18c),LL(0x92abaaa2,0x5e4d357c),L_(0x452fe014)}, {LL(0x1fb69d3e,0x7bd8f0e1),LL(0x58ea7caf,0x4735131f),LL(0x84b31510,0x6a03c6b3),L_(0xbcfb5361)}},
   {{LL(0x5a2c8036,0x91588e92),LL(0x4e652187,0x1b69af1a),LL(0x451b95e7,0x3517cee5),L_(0x8c64ca67)}, {LL(0x1e09f843,0x90930327),LL(0xbfeeb455,0xefa175df),LL(0x9e48451e,0xedef520b),L_(0x49b070fb)}},
   {{LL(0xa2492065,0x1e8946dc),LL(0x318fd845,0x9916090e),LL(0x82135311,0x98fdccc1),L_(0xbc4b4f98)}, {LL(0x31f673e4,0xd529f3ea),LL(0xf1c8e5c6,0xb119ca50),LL(0x428145ce,0xcf53a57b),L_(0xba7a2228)}},
   {{LL(0x38640e72,0xa872dbd5),LL(0xab84c140,0x8f90052d),LL(0xffeb6e87,0x9a484b77),L_(0xc8d71762)}, {LL(0xe045b6cf,0xb3a39b3e),LL(0x68339c4f,0xdbc53298),LL(0x462a1b9b,0x60a23eea),L_(0x47a583f5)}},
   {{LL(0x8a1c84eb,0xc38a15ff),LL(0xc73cc555,0x5b502e98),LL(0x67d1f3e4,0xf85c7d6e),L_(0xfaf491bb)}, {LL(0x7c34a745,0x94062e0f),LL(0x5e8b71b9,0x9579c983),LL(0x35f7a266,0x63ed902e),L_(0x2cfe7a8e)}},
   {{LL(0x7e6607ae,0x54fa21c1),LL(0x7c605a96,0x2c3baf13),LL(0x177b275b,0x39001a70),L_(0x402c2a3f)}, {LL(0xf4928a3d,0x9356e91a),LL(0x443518d7,0x30b1d7ca),LL(0x02772478,0x8c5ecfb7),L_(0x8e06bb83)}},
   {{LL(0xcaa887a4,0x4f7bef32),LL(0x08805f7d,0x5dec8cf9),LL(0x8965403a,0xcc362cae),L_(0xa8e82004)}, {LL(0x37f6ba21,0x35e7cde5),LL(0x1d022cb1,0x0d54e5e6),LL(0x02d7913b,0xf4a0addb),L_(0x4f243fbb)}},
   {{LL(0xcc348a38,0x58983641),LL(0xf6e35ba7,0x237be718),LL(0xd585a061,0xdb2539f1),L_(0x2217c68f)}, {LL(0x403565cd,0x78351808),LL(0x36c2d842,0x6202e955),LL(0x7f54736b,0x660d4fb3),L_(0xca575f23)}},
   {{LL(0x6bc9ce98,0xb22311a5),LL(0x84749347,0xe993a41b),LL(0x85025abd,0x432e7fa6),L_(0x6f57ce7b)}, {LL(0x42d19cdd,0x18144346),LL(0x1fc786db,0x2bd73670),LL(0xede484f4,0xa93261b3),L_(0xc5ec96e8)}},
   {{LL(0x38a7a729,0x2660c15e),LL(0x6845886e,0xcbfe266f),LL(0xe7715ea4,0x803ef610),L_(0x9d5c3a0d)}, {LL(0x6f2eb06c,0x3e73d7f3),LL(0xaff72fe1,0x8fe4228f),LL(0x2703c35b,0x801d9246),L_(0x0008cf4d)}},
   {{LL(0xa7b3253e,0xb347bed9),LL(0x87c9db13,0x23367d55),LL(0x50ebab37,0xe6bc8d5f),L_(0x30c060ce)}, {LL(0x9df003d2,0xd78ada2c),LL(0xc3892b4e,0x4989b92d),LL(0xf9b3e9f5,0x2dbbe952),L_(0x48275a07)}},
   {{LL(0xdc3c1c12,0xfb949b08),LL(0xa8296f5d,0x010b8834),LL(0x69483176,0x73825d94),L_(0xe4f380a5)}, {LL(0x7caf444a,0x3cd86c08),LL(0x6ae21350,0x8ba59225),LL(0x6d5a74bd,0x1e6e636f),L_(0x6a2c970e)}},
   {{LL(0x51eea6cb,0x69de6020),LL(0x15974dd5,0xf383a0c6),LL(0xa7aeef7a,0xec5ab9f7),L_(0x25652e0f)}, {LL(0xb56f14b1,0x7b78f2d9),LL(0xbe7272ca,0xb4e91769),LL(0x0886a015,0xb8c7fe1f),L_(0x1c2ac0a8)}},
   {{LL(0x7b2d2c1a,0x3f05a7cf),LL(0x7d9a8322,0x8a978fa5),LL(0x2541de9c,0xb09f953f),L_(0x44b3e4d5)}, {LL(0xa6872622,0x0275a86c),LL(0x5ca47329,0xc3073baf),LL(0xe68e138a,0x70608786),L_(0x426bc12e)}},
   {{LL(0x22def544,0xc1d1a75a),LL(0x054cdb35,0x9d05dd03),LL(0x13908388,0xcac954e6),L_(0x6b377854)}, {LL(0xfb37d06d,0x42873dd7),LL(0xd85fe89f,0x77686ddb),LL(0xbd6e2bfe,0x0a7aae21),L_(0x3116e23a)}},
   {{LL(0xd00f34a0,0x6d3f30d8),LL(0xb72f2ac8,0x793a6f93),LL(0x8e54b8c8,0xa3ebe240),L_(0x06c18c5d)}, {LL(0xd3635d7f,0x7527fefe),LL(0x848fd8b1,0x7ecf5710),LL(0xe5c193ac,0xc5b359a4),L_(0x10bdedf6)}},
   {{LL(0x6a69e07f,0x904b6a99),LL(0xf73ecc50,0x74691c89),LL(0x377aafb2,0xf8dfe49e),L_(0x3116ab06)}, {LL(0xbc4bbf14,0x32c37416),LL(0x4f9adb4e,0xa6bc7861),LL(0xb481fbff,0xab667972),L_(0x50e39a47)}},
   {{LL(0xd40e7122,0x03318c55),LL(0x2b1438a6,0x5cc23063),LL(0x7d21722c,0x353dafc9),L_(0xb52192c7)}, {LL(0x47d32da4,0x8e5c3c5f),LL(0x356f1a48,0x4c620a4a),LL(0xc2e2dc49,0x6ac8f16a),L_(0xa8078172)}},
   {{LL(0xe0aae2f3,0x27e1fc19),LL(0xcaccad70,0x9a5089ff),LL(0xd066d3ec,0x83ce796d),L_(0x1d39df6c)}, {LL(0x3d7bd85e,0x2109b271),LL(0xd4d73f64,0x032bfaaa),LL(0x4c64918b,0x28f7e9fb),L_(0xead7a6eb)}},
   {{LL(0x81a3a911,0x1ec15d92),LL(0x072d4b8d,0x0de6b8aa),LL(0x25d33d78,0xdb94f1aa),L_(0x163e5ede)}, {LL(0xcda754fd,0x0aa41bac),LL(0x8bb9f4cc,0x26a33edd),LL(0xaed5fcc3,0x03b383eb),L_(0x7bac84ac)}},
   {{LL(0x48a08610,0xe3c9d6e7),LL(0xaf9b1185,0x1c9dda6d),LL(0x37a0510c,0x87290f47),L_(0x395d8f39)}, {LL(0xe0b9b993,0x04968462),LL(0x2065fe67,0x79bcdef3),LL(0xba7c23dd,0x3c9dfbbc),L_(0x48b97684)}},
   {{LL(0x4867362d,0x9b7dcff4),LL(0xa45a3647,0xf6be4fd8),LL(0x9dd99069,0xfc837cc5),L_(0x463e52d6)}, {LL(0xa104db39,0xe73440a8),LL(0x868e9fad,0x557bc30e),LL(0xca38a194,0x0b9107de),L_(0x3b4896a5)}},
   {{LL(0xf9b3c663,0x0572aa2e),LL(0xb232d67e,0x6cf8f9e9),LL(0xdc3bb3a4,0x578fa229),L_(0xf77deefa)}, {LL(0x7b780979,0xe2e55595),LL(0x3dd1f800,0x1ae5674a),LL(0x7d70eb0e,0x6287e785),L_(0x6ca9df37)}},
   {{LL(0x6027714a,0x6418b451),LL(0x6b5f95e4,0x008c73fe),LL(0xc7a41b44,0xb52d2690),L_(0x7e9e5e8d)}, {LL(0x07d5913f,0x28af32f1),LL(0xba6493ac,0x70ace436),LL(0x249d5a1f,0xe2634904),L_(0x9de83ae3)}},
   {{LL(0x91c3f75f,0x7ac06d6f),LL(0x5d22beb3,0x8cbbcc95),LL(0x6114dddf,0xb3f952c3),L_(0xd55713d1)}, {LL(0x50a376be,0x95633a5e),LL(0x402443f5,0xf0277528),LL(0x900af3d2,0x2aac428e),L_(0xe7ecc502)}},
   {{LL(0xede089c1,0xafe013ca),LL(0x9bc0d189,0x24f9b481),LL(0x7d9ab065,0x7064130f),L_(0xcb09791d)}, {LL(0x88599e66,0x1a8684d1),LL(0xdbafc741,0x5b828087),LL(0xa7784039,0x274e2738),L_(0xa61d0693)}},
   {{LL(0x04a64675,0xabf54d81),LL(0xdec1f49f,0x46144f05),LL(0xc232e58f,0x30cddd92),L_(0x5b7d5bc7)}, {LL(0xc2b48798,0xfb8c420d),LL(0xb96fbab6,0xb4d97ee3),LL(0x8d97d0d5,0x8101b1e7),L_(0xb545f661)}},
   {{LL(0x98af21f0,0xcd4340b9),LL(0x955d1c9d,0xda56271c),LL(0xcf1eb8ab,0xcfc8c025),L_(0xc22b3fb1)}, {LL(0x99747eea,0xdff889cf),LL(0x01a895b8,0x16fb6025),LL(0x22ae0ed8,0x56d4b8ed),L_(0xbdaf7eb9)}},
   {{LL(0x0745f518,0xc7781923),LL(0xb03571c4,0x6cd5a87b),LL(0xe1112158,0x734d96e5),L_(0xc2f71082)}, {LL(0xfbc17aec,0x7d6ae9b9),LL(0xae846849,0x6b340fbb),LL(0x7a81eba8,0xa462be47),L_(0xf156efcb)}},
   {{LL(0xbad5359f,0xcc17a45f),LL(0xed5f0c48,0x3962aaeb),LL(0x18daea3d,0x24dea50b),L_(0x2620a5cf)}, {LL(0x2db980d5,0x94722851),LL(0xe16c7ff8,0x17a80fd9),LL(0x0cd73f37,0x0ef75926),L_(0xa7a0ce92)}},
   {{LL(0x2994a0c6,0x0d862f7d),LL(0x66b2e45e,0xa326bcf2),LL(0xb39a090d,0xebe228c9),L_(0x8223ef31)}, {LL(0x4fe39e14,0x292a329c),LL(0xe179a973,0xa1f5d624),LL(0x3f828ea8,0x70ad621c),L_(0xc04d2e0b)}},
   {{LL(0xb7237a24,0x94de2084),LL(0x91138a01,0x0c93caa4),LL(0xd97e1daa,0xb74fa915),L_(0x668329b9)}, {LL(0x146706c8,0x23c7b716),LL(0x23606a0c,0x54c6c8af),LL(0xbe011ca0,0x2ef6e4a4),L_(0xb2029de6)}},
   {{LL(0x33851602,0xbee34e4c),LL(0x7b910cad,0x8e7c506d),LL(0x2b92ce18,0x62b378cc),L_(0x5c490234)}, {LL(0x95f16f74,0x661c0891),LL(0xa5073d8e,0xf7df962d),LL(0x6fdadac1,0x8f3ef4f4),L_(0x373f29f4)}},
   {{LL(0x23206f03,0x235bc318),LL(0x7518fa1b,0xec43017e),LL(0xc5d4dab6,0xf547238a),L_(0x7e5c2b82)}, {LL(0x25d0ebfc,0x8357f1bb),LL(0xfb571aa9,0xe137e308),LL(0x996a92a8,0x629e8aa7),L_(0x90f06a9f)}},
   {{LL(0x4a17680f,0xa8c7b143),LL(0x84f5aab1,0xb84edf8c),LL(0xf2acd3a2,0x8b3a07a1),L_(0x5d4c58ed)}, {LL(0x2f0d5862,0xcd2f47fd),LL(0x31cb8d1e,0x0d4da406),LL(0x0cd04a92,0x8ec4fcad),L_(0xb261a451)}},
   {{LL(0x91a38da6,0x76eb9c38),LL(0x12868725,0x0a5b71ac),LL(0x2d328a16,0xf6122daa),L_(0x9232586e)}, {LL(0xa1eceeb7,0xe9555051),LL(0xc35149d8,0x55020182),LL(0xae8df945,0xcee3a23b),L_(0x6020896d)}},
   {{LL(0x88241ba5,0x6839ac95),LL(0x75c22b80,0xb01a4c72),LL(0x4e73d41b,0xdbce70d9),L_(0xb035070a)}, {LL(0x25457ffb,0x4155bcf6),LL(0x6c514452,0x79b75848),LL(0x3c25b38e,0x8c33a19b),L_(0xb8f49889)}},
   {{LL(0xb2bc7aad,0xc7b2eb56),LL(0x96c5270f,0x8112c87f),LL(0x15208222,0x612138e3),L_(0xac8a2edb)}, {LL(0xa6392e61,0x7fcbfb2b),LL(0x95cfe395,0xb4541276),LL(0x134be404,0x3de33a7c),L_(0x2cf45438)}},
   {{LL(0x27c80982,0x6d995371),LL(0x429942a9,0x1e16c2dc),LL(0x0e59bd58,0xfbad79fb),L_(0xd34caf67)}, {LL(0x481ca9cf,0x6cb52c72),LL(0xfc7ee8ab,0x7e9afc85),LL(0xfda3af10,0x41a661b7),L_(0x7e623e57)}},
   {{LL(0xc421cd21,0xb68b5cf7),LL(0xd94c04b7,0x107cec0c),LL(0x81208f8b,0x383162ed),L_(0x454f3ae7)}, {LL(0x2a8bfdfe,0x0f977619),LL(0xdeb81615,0x382decc1),LL(0x8e5df4ce,0xf0414c84),L_(0x0f3c79aa)}},
   {{LL(0x15f29277,0x89f9b2a7),LL(0xa1f621ee,0x817b0a57),LL(0x6ee121aa,0xf56eeafc),L_(0xcf7c22d6)}, {LL(0x0bbfabbf,0x52eb19cc),LL(0x018d1b99,0xc2b09e5e),LL(0xa82ae35b,0xd9cf95b8),L_(0x34b4bf9a)}},
   {{LL(0x017073a5,0x4f731dd0),LL(0x0235def1,0xd07b63f1),LL(0x5e605b0d,0x4769aa10),L_(0x7252588b)}, {LL(0x26cfa7a5,0xfac43672),LL(0x01631f47,0xbb074c5c),LL(0xe02c4623,0xb6536982),L_(0x061acc5a)}},
   {{LL(0x0bd8823c,0xca3a4054),LL(0x57b5c945,0xbee559f1),LL(0xac8c4662,0x54b8d320),L_(0x8ee0f3f7)}, {LL(0xfc9e1370,0xafaff913),LL(0x7ef2aacc,0x15a9349a),LL(0x81208c2d,0x0d06fd25),L_(0x00df14a3)}},
   {{LL(0xc4906f44,0x3cb8d3d5),LL(0xc2b5f2bf,0xef621a83),LL(0x42fee26b,0xcb150abc),L_(0xa4673146)}, {LL(0x8c081901,0x2e00a64e),LL(0x4f184ea9,0x4b9fa7b4),LL(0x7c4d0b48,0x60c91243),L_(0x05745a02)}},
   {{LL(0x36d03f9b,0x7049620b),LL(0x54cfa460,0x29c0b51d),LL(0x50448eab,0x3f0b2dd9),L_(0x9ed5c1c7)}, {LL(0x217c3b7d,0x8eca2e20),LL(0x7e7352b9,0x20883efa),LL(0x3afc2315,0xc23211b5),L_(0x98c20d32)}},
   {{LL(0x3c1d5c98,0x00ef3bc9),LL(0x03002427,0xf1f069cc),LL(0xc5bebf01,0xf661ac1a),L_(0xb01d642d)}, {LL(0x54219e42,0x25a7fa91),LL(0x199aa1af,0xf217cbfa),LL(0xf0d4c0bf,0x4f2df7eb),L_(0xf7508aca)}},
   {{LL(0x8f2773db,0xb21c4cfa),LL(0x966e2843,0x52c04b61),LL(0x590a0567,0xf35f4a84),L_(0xf255a433)}, {LL(0xcc2e1149,0x04ae8b1c),LL(0x13360c1c,0x92f66f03),LL(0x4053731c,0x20aba0c9),L_(0x20d59987)}},
   {{LL(0x0a4bb9e9,0x9ad08ec7),LL(0xa3719263,0x54a4bb3e),LL(0xc9bcb020,0xf5564618),L_(0xaa1b7ba0)}, {LL(0x2f0cb8aa,0x64275534),LL(0xd09acd41,0x3340504f),LL(0xd7e50869,0xaf97cbcd),L_(0xf4025b06)}},
   {{LL(0x96f1d3cd,0x9f391b14),LL(0xbd5e9744,0x5f6bed3b),LL(0x008d20fd,0xc79a2572),L_(0x905330f0)}, {LL(0x77d13f02,0xb71f049c),LL(0xdb745573,0xe088bd21),LL(0xbd449f9b,0x3a76432d),L_(0x1de7f0bb)}},
   {{LL(0x12dbb0cf,0x4f53adb9),LL(0x2e4f3aa7,0x13ff9790),LL(0x61674465,0xa80fdde3),L_(0xfbfa4069)}, {LL(0x50e102a9,0x28391810),LL(0x22e32485,0xd119b153),LL(0x47aca745,0xeb9f0102),L_(0x3f33e272)}},
   {{LL(0x6dc8e416,0xb01555de),LL(0xa53f861f,0xc5669efc),LL(0x14d3a409,0x0c68a597),L_(0x3ad8c86e)}, {LL(0xed8dbd60,0xc5474f33),LL(0xda13401f,0xd21a574f),LL(0xf6b7aed6,0x0d2d58af),L_(0xbe2e5676)}},
   {{LL(0xb63ca145,0xa81eba20),LL(0xd0d7b9ca,0xf6e2c558),LL(0x4e439115,0xe8f8d978),L_(0xb32fdbaa)}, {LL(0x8cf406a7,0x23a5c7d1),LL(0x7072ea14,0x4e16a729),LL(0x3db551d5,0x88436ae6),L_(0xd7c49d77)}},
   {{LL(0x32172eb6,0x7d243c47),LL(0x52231b01,0x4a4714a5),LL(0xed3510a8,0xfde7f4a6),L_(0xccb27287)}, {LL(0x1ed13735,0x4721f266),LL(0x5a39f8ef,0xf2889694),LL(0x99b1258f,0x45218687),L_(0x4c27d1de)}},
   {{LL(0xe91d500c,0xcf933e7f),LL(0xb7d6326c,0x2d1ad273),LL(0x572c7767,0x79dbaa6e),L_(0x27d7d939)}, {LL(0x17390032,0x6c991f09),LL(0xed70ac81,0xbaab3191),LL(0xfc86b907,0x93c673b6),L_(0x94754f34)}},
   {{LL(0x3543430c,0xc2f96d5f),LL(0x0ea5f8eb,0xac730ca9),LL(0x376185cf,0xfeac300d),L_(0x41737668)}, {LL(0x96df0a25,0xf035d502),LL(0xb519c6a4,0x9ca93571),LL(0xd64df17a,0x01970aaf),L_(0x18c05bf9)}},
   {{LL(0x471f2099,0xa5fb90b3),LL(0x53b53a2a,0x2aa370a3),LL(0x09b936b9,0xcfb37658),L_(0x83591048)}, {LL(0xb2dfe05c,0x1e8ce8f7),LL(0x70ca625a,0xcf8806b6),LL(0xf062523d,0xa1c74034),L_(0xc4ca8211)}},
   {{LL(0x02fd45e9,0xf0d5ce43),LL(0xf94539de,0x86092a2c),LL(0xc28ad111,0x0990658e),L_(0xfc301c6d)}, {LL(0x850acec2,0x097c875b),LL(0x333c1a83,0x79052ae9),LL(0x651a1ec4,0xdf5b00e0),L_(0x8f342e96)}},
   {{LL(0x7c4aacf4,0x12306aef),LL(0x23bf4f82,0x0369e1f2),LL(0x86d193a6,0x3bdb2181),L_(0x2df259e0)}, {LL(0x4d9c6a9b,0x2e587a72),LL(0x420fa4b9,0xe30d76a5),LL(0xaf97ed7f,0x20fbaabf),L_(0x6c230751)}},
   {{LL(0x044baaf4,0x04047c5f),LL(0x8d43be85,0xe4e3879a),LL(0x436ef7e0,0xed64bf04),L_(0x6b436c30)}, {LL(0xa9c7bdcd,0xc1679bbd),LL(0x12340567,0x38ba10d1),LL(0x83b598cb,0xa99537cc),L_(0xee36e118)}},
   {{LL(0x18eb43ca,0x5aa367ce),LL(0x94ff23cf,0x2dc9b946),LL(0x066d6c04,0x3a41602a),L_(0xfe10e991)}, {LL(0xfcc4d378,0x3dbf71fc),LL(0xd6bc1c02,0xc9ccbaad),LL(0x0aacb390,0x985954f5),L_(0x468870a9)}},
   {{LL(0x1f6e5aac,0x4d58f138),LL(0x088912a4,0x72e75855),LL(0x389e7a2b,0x4b5aa1e6),L_(0x5be86bd9)}, {LL(0x29d90bc3,0xb35b2d2a),LL(0x67e8a5ea,0xaca3b9cb),LL(0x832c687a,0xe2911a57),L_(0xed56df90)}},
},
/* digit=8 base_pwr=2^56 */
{
   {{LL(0x56b1b68d,0xa766a468),LL(0x7a690380,0x7f0a8ccb),LL(0x15b9ce0d,0x8bfb375d),L_(0x0afa00f6)}, {LL(0xe7944b6b,0xe502fd6a),LL(0x0768ebca,0x079ac7ac),LL(0x956dea42,0x3fc3f258),L_(0x78be0f9a)}},
   {{LL(0x7157aab8,0xde649372),LL(0x1fe5b1db,0xec032ca1),LL(0xdc9ac491,0xaf00af85),L_(0x42bf819d)}, {LL(0x3d3a586d,0xb1b4ed91),LL(0xc6e9377d,0x0da4ec46),LL(0x2fb5dda7,0xe69eb235),L_(0xa5e4f04d)}},
   {{LL(0x4a718a91,0x6813c47e),LL(0x520eeb02,0xcc1aaacb),LL(0xbae4540e,0xfd6dde20),L_(0x5d3611cc)}, {LL(0xa139969a,0x47c1ef53),LL(0xa02a9c1d,0xff24f01d),LL(0xa1779b32,0xafe60f72),L_(0x0b879e68)}},
   {{LL(0x6e7fd2ac,0x206a6f14),LL(0x2a13d9f9,0x43a11c7f),LL(0xf31bbf6c,0xe2ff97fb),L_(0x56107ea7)}, {LL(0xb83101aa,0x0b2c24c6),LL(0xdc078721,0xd00a24a9),LL(0xc5ea4420,0x80054822),L_(0x76c2e6e5)}},
   {{LL(0x5d302b86,0xa3ec0bc2),LL(0x58cdfe77,0x7469b928),LL(0xde033220,0x9823907b),L_(0xce8c1169)}, {LL(0x8d35908e,0x668b8357),LL(0x90f7260f,0xcd96886e),LL(0x388da84d,0x8e759b7d),L_(0xf9144eb3)}},
   {{LL(0xb0068632,0xb87f256b),LL(0xd3ec8de1,0xb7562058),LL(0x554d7fad,0x68b8bed1),L_(0x7d3ba275)}, {LL(0x5349ed32,0xaa23f48d),LL(0xf8c55847,0xac744602),LL(0x125ee820,0x150da1ca),L_(0x16333e9d)}},
   {{LL(0x16a4c78e,0x26b49a2d),LL(0x9714ac19,0x69e5dc39),LL(0x50449596,0x2af6b2c1),L_(0xab90cd95)}, {LL(0x07439dc5,0xf5736c67),LL(0x8b3909e8,0xeeb790dd),LL(0x0cc523dd,0xae2f8bbf),L_(0x21833381)}},
   {{LL(0x1c12b2d6,0x7628ecee),LL(0x6d88b422,0x0692cc8d),LL(0x65398808,0x5048f64f),L_(0x0e3b6999)}, {LL(0xd880351e,0xae75c4fc),LL(0x3ffdb69e,0x3e2ff89d),LL(0x5f9ca860,0x435a454d),L_(0xd321ca44)}},
   {{LL(0x1dbe11f1,0x805cdd89),LL(0x0b435944,0x0562fe66),LL(0x5d6a45d5,0xc9eb9f56),L_(0x0394c9fd)}, {LL(0x1fbcb273,0x4b25d574),LL(0x8a17364c,0xb78a31bf),LL(0x1466b27f,0x468c47f5),L_(0xfd233c89)}},
   {{LL(0xb836e3e6,0xf6743281),LL(0xe6b59ce6,0x13f41d4e),LL(0xd77d4340,0x397a6846),L_(0x45693c8c)}, {LL(0xbd735497,0x6560f3f3),LL(0x5f68d915,0x61405ead),LL(0x119c162c,0x9870a7f1),L_(0x8b74bea7)}},
   {{LL(0x4f81eb06,0xdd4548bd),LL(0x271e6f36,0x49ea4452),LL(0xff17a13f,0x54ebcb9f),L_(0x83b9157a)}, {LL(0x3e7236a9,0x4d44b2ce),LL(0x7f67612f,0xfeac4aaf),LL(0x0d85cf3f,0x7d7cb315),L_(0x7efb9b30)}},
   {{LL(0xb0f78faa,0x82bbcdcf),LL(0xf1c04007,0x01559ec3),LL(0x7cdb7afd,0x86ff3957),L_(0xb1597d4b)}, {LL(0x6c320e72,0x45b1afec),LL(0x522b415a,0x9d12c298),LL(0x6b93f2de,0x3e21f849),L_(0x3b1a1d95)}},
   {{LL(0xc12b655e,0x219c2865),LL(0x6265e8e1,0xc6daf8cf),LL(0xf3ec2977,0x10269fa7),L_(0x4486877c)}, {LL(0x8de615f2,0xb448dafb),LL(0xfeda6cb4,0xac03ce69),LL(0x0fd25b12,0xb96afd44),L_(0x0f6ae181)}},
   {{LL(0x069f14f1,0xc970df0c),LL(0x614fb451,0x8cddee54),LL(0x0475ddb9,0xac228ae6),L_(0x3ef6e29e)}, {LL(0xc73f5027,0xbdeb5cb1),LL(0x2198420f,0x82098200),LL(0x8694111c,0xc11479ab),L_(0x25c29038)}},
   {{LL(0x0c73fe99,0x4aa502d3),LL(0xdb603ed0,0x6c27909f),LL(0x4f620050,0xf76957c1),L_(0x0e7e084e)}, {LL(0x1565b30b,0xfba38431),LL(0x95ddd9af,0x65c7038d),LL(0x82a6a124,0xe7db47e4),L_(0x1f3b155f)}},
   {{LL(0xfdab8dbe,0x773fef23),LL(0x70802c0e,0x011efbb8),LL(0x8698e8b3,0xa4f63ad2),L_(0xa90d02ab)}, {LL(0x60003ff7,0x5aebc019),LL(0x586f4378,0x00bcb9de),LL(0x859f79cb,0x6a9d21d2),L_(0x79d6150e)}},
   {{LL(0x8a59362f,0x21480f43),LL(0x37e25e9c,0x4a37a459),LL(0xe55838d3,0x36aab717),L_(0x85a89af8)}, {LL(0x1d33aa1c,0x86dc1133),LL(0xb94719f6,0x0a1e4a76),LL(0xf9d8ea27,0x7a9f7b61),L_(0x2b10e5f5)}},
   {{LL(0xf5ab77c2,0xb7c034fa),LL(0x91179a6a,0x89204639),LL(0xd0dd8159,0x4ba725f9),L_(0x39ccdad0)}, {LL(0xa510602e,0xc72415fe),LL(0x85069c0f,0xa6bc7ccc),LL(0x7d585ae6,0xc90e9462),L_(0x51cea642)}},
   {{LL(0xaed13816,0x84f02e2a),LL(0x24576ca8,0xffdbb5f9),LL(0x3c9269f7,0x5c61743e),L_(0x498dc0e3)}, {LL(0xd5bc6379,0xa9132224),LL(0x10fc37d9,0x12f65fd8),LL(0x817445f8,0xd0c982b8),L_(0x4f79fcf1)}},
   {{LL(0x40f7fe3e,0xcb5018ef),LL(0x2fdb045f,0xf32b39b6),LL(0x621a4bd4,0xe7c33edc),L_(0x04fd5dd5)}, {LL(0x7972a384,0x586d4bef),LL(0x22c07adc,0x81211aa8),LL(0x8c46dfe1,0x5b192417),L_(0x68b0abc0)}},
   {{LL(0x30ddafb1,0xe56640a6),LL(0x55279999,0xe01ca53a),LL(0x6b7735ff,0xa4ba7379),L_(0x1c1eab8d)}, {LL(0x184d5073,0x1f2853f9),LL(0x0a5a3694,0x0c65322e),LL(0x77546dfc,0x814529cf),L_(0x0117bc49)}},
   {{LL(0xe44092c1,0xdcf36025),LL(0x29873c6a,0xa38b0382),LL(0x2beaabb1,0x57dab020),L_(0x4481b257)}, {LL(0x3fc25cda,0x16190d52),LL(0x3dcabd82,0x632514c7),LL(0xc69b7cce,0xcb82be1e),L_(0xcf490cc0)}},
   {{LL(0x79d31337,0x78cc588c),LL(0xd718033e,0x5d3ca504),LL(0x0951b466,0x82b9278f),L_(0xc14b1f81)}, {LL(0x6c160f3e,0x308d56c2),LL(0x833254cc,0x460ed897),LL(0xcccc63ee,0xd27b55ac),L_(0xdf9c95f1)}},
   {{LL(0x2097cb5b,0x6fb48059),LL(0xc33f19e6,0x8b4a82f4),LL(0x3bf0f71b,0xda304a5b),L_(0x7efe85d3)}, {LL(0x5d609c9e,0xb32599e2),LL(0x7d311503,0xea5b4566),LL(0xc072d5ff,0xf9f2dc57),L_(0xf9df545e)}},
   {{LL(0xdacba736,0xbc32b9c9),LL(0x0ab7c149,0x154849a8),LL(0x29e4dd9b,0x11dfc29e),L_(0x41ace68f)}, {LL(0xc8616f74,0x971ba9a3),LL(0xb0be2d45,0x15776b0b),LL(0x8325c930,0x294a4ec5),L_(0x455623af)}},
   {{LL(0x16f8969c,0x235d5c34),LL(0xb8ca3700,0x57aa97da),LL(0x5c79afae,0x417467c4),L_(0x796984ff)}, {LL(0xe7950969,0x59ca92ce),LL(0xcd03a014,0xb1a7c34d),LL(0x6203a921,0x066a9647),L_(0xd0909f6f)}},
   {{LL(0xfb7e2684,0x84d55477),LL(0x0c455cdf,0xbdee722d),LL(0x74bfcbf4,0xf514f2bf),L_(0x8a60e5e7)}, {LL(0xba7058a4,0x4713d1ae),LL(0xecf4c404,0xa6c17b5e),LL(0x8941ddd8,0x261d9ac7),L_(0x7f2f5187)}},
   {{LL(0xac5e5ce7,0x76370390),LL(0x64cf9d29,0xfce7f694),LL(0x7c34fbe8,0xe3b42849),L_(0x6bc43f22)}, {LL(0x8781925b,0xe273ae40),LL(0x49d3a7bb,0x777881ae),LL(0x6fc64556,0xb0f8e854),L_(0xb4b40191)}},
   {{LL(0x024a93d5,0xa6b66d66),LL(0x4f2aaf4a,0x56c335ae),LL(0x0805e125,0xe1e39e4e),L_(0xb67d8a1b)}, {LL(0x9bb438e7,0x0e8dde7c),LL(0x24683c26,0x06a38af7),LL(0xa234eb6c,0xd74870f8),L_(0xfa606ebd)}},
   {{LL(0x05b51c3a,0x62d8ac43),LL(0x8c1548ed,0x8eefcd86),LL(0x3b151865,0x3c047e9d),L_(0xf77c0004)}, {LL(0x76ea2961,0x82998a41),LL(0xa3b38ce5,0x110b3b2e),LL(0xbcfe60ba,0x89b19d6d),L_(0xe68575e8)}},
   {{LL(0x8765ec49,0x36e47913),LL(0xc6e4ab7e,0x767e806a),LL(0x81bc9bd7,0x38103684),L_(0x5490cabd)}, {LL(0xb8107b91,0xbe9c24b5),LL(0x63e98957,0x43623ea0),LL(0xf06a675d,0xcbeab5be),L_(0xf0e9a898)}},
   {{LL(0xd064269c,0x1ccbf4e2),LL(0xfe422ab3,0xe21fbca6),LL(0x5cee0a5c,0x263913a5),L_(0x71de7df5)}, {LL(0xa3fc8bae,0x047f1949),LL(0x0df59ed6,0xd1a5d116),LL(0x35a0d475,0x9e3419f5),L_(0x75946143)}},
   {{LL(0x36c9f809,0x631145ae),LL(0xa1db7d1d,0x0c423276),LL(0x4bf70210,0x2f1584d5),L_(0x8bd52dbd)}, {LL(0xa19699d2,0x8015f024),LL(0x37a40f15,0xf9755c59),LL(0x25538f41,0x387ddb31),L_(0x90af00b0)}},
   {{LL(0xe8509934,0x9e321e61),LL(0xd2850d36,0x7dc68e41),LL(0xf8206e77,0x58e783b2),L_(0xe67a1042)}, {LL(0x906883cd,0x6f41c769),LL(0x51b97936,0xbb9527f5),LL(0x38e2f27d,0xcf17789e),L_(0x5273cd0e)}},
   {{LL(0x1bf5a9fa,0xae648e71),LL(0xde909240,0x3f5f8275),LL(0x93245241,0x80482c3b),L_(0x0a2260c5)}, {LL(0xef9f19ab,0xaa9e665a),LL(0x1eb26ff8,0xad522975),LL(0xce379c80,0xf8653e32),L_(0x4f190ba9)}},
   {{LL(0x19d1e571,0x4023698a),LL(0x4e5e5355,0xa19db2ab),LL(0xd2800586,0xaee5fc4c),L_(0x115617f9)}, {LL(0x378c9aca,0xbcb82922),LL(0xfb1dcef8,0xc3b8fa14),LL(0x1b19035c,0x1c3a4f74),L_(0x190f3cd7)}},
   {{LL(0xf201d617,0x88de2216),LL(0x778baf2f,0x409f13a5),LL(0xfc1697ff,0x1f4ee383),L_(0xf8b34103)}, {LL(0xc70f2c0b,0x596702b2),LL(0x40ecf08e,0xaa0989d3),LL(0xcc60a8c2,0x5ba1fcbb),L_(0xb0f51744)}},
   {{LL(0x3f768dfe,0xee929e0b),LL(0x5c573a73,0x171719ed),LL(0x91e6ec5d,0x48549719),L_(0xb182f69c)}, {LL(0xa3e2860c,0xa30f11a1),LL(0x5e01688d,0xf804fa42),LL(0x58c435e7,0xd81ff657),L_(0x49e16053)}},
   {{LL(0x3ee264bb,0x6d74b27a),LL(0xcb2992ae,0x776b6b64),LL(0xa97538c5,0xdc494d53),L_(0x282ddfc8)}, {LL(0xe4acc7fe,0xe5d1ac83),LL(0x81afb14c,0xf0a59879),LL(0x9b5f787d,0x0730aa6b),L_(0x76168d56)}},
   {{LL(0x5d202341,0x15ca4cef),LL(0x67cf5e49,0x03078647),LL(0x7cbaad05,0x7a152a3a),L_(0x93ecdb65)}, {LL(0x5c759c30,0x1d48b5cf),LL(0x8a07288f,0x954731e0),LL(0x67babf9e,0xb890cb9b),L_(0x3f7cb4f5)}},
   {{LL(0xfbda2539,0xd314af3a),LL(0x66f79c66,0x333f8c94),LL(0x07bcc285,0xd37bbc81),L_(0x1102429d)}, {LL(0x7ba7309f,0xe5686c59),LL(0x647c832d,0x96218ea6),LL(0xbfb2d48e,0x7bc6fa2a),L_(0x524c8c0d)}},
   {{LL(0x91273fc1,0x9f330845),LL(0x6f3e8880,0x8d3155c8),LL(0x229515d9,0x4e0d0091),L_(0xa37bad0a)}, {LL(0x3cca8a71,0xc538ea5c),LL(0x34161b0c,0x7f68ef37),LL(0x412db800,0xfc39477c),L_(0x5112c574)}},
   {{LL(0x202d77d0,0x3f50f1ae),LL(0x56e724be,0xc9d43dee),LL(0x2163dc89,0x82838e71),L_(0x3e07b5b6)}, {LL(0x68b2f3b5,0x028beb58),LL(0xfb4f5d6e,0x2f67136a),LL(0x63c11941,0xf0576d72),L_(0x36e94125)}},
   {{LL(0x79a7908b,0xfe08b743),LL(0x7ec4b29f,0xe4c9aa01),LL(0x3121fc40,0xe6d6660d),L_(0xa4b98339)}, {LL(0xcc2b942f,0x54acd180),LL(0xc715b5ed,0x4b3ac4d4),LL(0xde75824c,0x91ca23fa),L_(0xe3ff9f1c)}},
   {{LL(0x46582483,0xa927f3c9),LL(0xbac6e7bf,0xe88a430f),LL(0x1f445777,0x3e33ca59),L_(0xbdfe1920)}, {LL(0x71dd2627,0x54a8126d),LL(0xebc9b59a,0x9c986433),LL(0xbec3ed13,0x71db698f),L_(0xd57f9239)}},
   {{LL(0x731eb54d,0xa93b1752),LL(0xa3516b9e,0x6f7bfa08),LL(0x873d3a45,0x4f66d06f),L_(0xbdd10c66)}, {LL(0xc97be3e7,0xf6435b79),LL(0xa3533348,0x207b9b6c),LL(0x1c6c74ae,0xb64b6b70),L_(0x9037812e)}},
   {{LL(0x74ccc35a,0x4be5a10b),LL(0x425bdafc,0xb790b940),LL(0xc32a94b3,0x098b38b5),L_(0x026ae426)}, {LL(0x4ef7c297,0x18feb7b4),LL(0xda85e218,0x6f3016a8),LL(0xad72b80e,0xbdf2db87),L_(0xa853e440)}},
   {{LL(0x0ca1adfe,0xb13f2859),LL(0xbd4a0fb5,0xd393e869),LL(0x277e9cd1,0x90b15b52),L_(0x30056558)}, {LL(0xf705f505,0x2c2d2fb1),LL(0x2c3d697e,0xf49260e8),LL(0xcb991398,0xfed48a8d),L_(0x6c792dc1)}},
   {{LL(0x2687c048,0xc3ca0fc2),LL(0x97ff87d8,0xbb9d5b5b),LL(0xe241d7c5,0xef9d2be9),L_(0x4d4d8a9e)}, {LL(0xfe8acc57,0xc5011f32),LL(0x499812d3,0x8545eca9),LL(0xfa8d83be,0xc270dd4a),L_(0x162b0c9f)}},
   {{LL(0xb3951e09,0x78a942de),LL(0xc228c3e5,0x57fae761),LL(0xbc46ee6e,0x83169c42),L_(0x1aeae5f1)}, {LL(0x340fc522,0x669821ab),LL(0x51be95bf,0x09548071),LL(0x92a505ce,0x83ec59bb),L_(0xd2e39c6b)}},
   {{LL(0x3ce8adaf,0x41ed0956),LL(0x8981ccd6,0xc0e55576),LL(0xba8eb552,0xa018b90e),L_(0xf5b2000d)}, {LL(0x67316d00,0x5703fba5),LL(0xf9b6f0d3,0x6078c1e3),LL(0x99f221d3,0x047eec80),L_(0x5f0d4326)}},
   {{LL(0x81739c80,0x6f11b810),LL(0xf085c79b,0x1e31485f),LL(0x9ddc21df,0x18e4d189),L_(0xbe0bc274)}, {LL(0xe78c5638,0x439dc1dd),LL(0x63733145,0xa22acbf7),LL(0x9f469794,0x161460c0),L_(0xaa724fea)}},
   {{LL(0xcd4a85b1,0x7e2cff5d),LL(0xaf20e779,0x998dca20),LL(0x3d5bd0f7,0xbfc65978),L_(0x4e61a410)}, {LL(0x388987a1,0x9b689b7c),LL(0x04dbc066,0x3ea487dd),LL(0x5b3b94d0,0xa77864e1),L_(0xe89088a7)}},
   {{LL(0x39413fae,0x796a4cff),LL(0xd204de5d,0x978f94f4),LL(0x2c42836e,0xdabb2333),L_(0x20078e8b)}, {LL(0x683de92d,0x5fe8df50),LL(0x14c267e5,0x705a1d2a),LL(0xf68dcb2c,0x16e49003),L_(0xcbb04811)}},
   {{LL(0xc7d4f66e,0x02ce592d),LL(0x6c59b80b,0x449fe24e),LL(0x49708f6f,0xe167be96),L_(0xfb3a4da9)}, {LL(0x7456adb9,0xafdb07bc),LL(0xa22499e7,0xfc9ec5ea),LL(0xb52ff28e,0x55b069a6),L_(0xc0fc9e6f)}},
   {{LL(0x1b0cec1b,0xf4484ca3),LL(0xb1289e3f,0xde798873),LL(0x05b5ac5f,0x87ada89d),L_(0xf204ab3d)}, {LL(0xbc022acb,0x93b1313e),LL(0xd2127d5a,0xa4d90949),LL(0x71a6efc0,0xd897f8e5),L_(0x88e1bcc1)}},
   {{LL(0x5f811bb3,0x9a725a78),LL(0x9323ffc2,0x1f6cfcf3),LL(0xe1abeb72,0x6d3872f8),L_(0x71c9aa7f)}, {LL(0x90cec4c5,0x400da758),LL(0xdb55d5c9,0x74d4d54c),LL(0x6fed8555,0x4bc5cda0),L_(0x99c927f2)}},
   {{LL(0xb4a54352,0xa9b678d4),LL(0x21c3a4ea,0xf079b22c),LL(0xcd89ee25,0xec3738e6),L_(0x12db8e7a)}, {LL(0x17543420,0x3b3edf75),LL(0x83b1a5b2,0x46732c66),LL(0xcfead94a,0xa689fe7e),L_(0x658ac775)}},
   {{LL(0x91cb6be3,0xa3af0a2d),LL(0xaee7da78,0x37e8c56d),LL(0x44cf93c9,0xc55af358),L_(0xb97c6473)}, {LL(0x2941f1b7,0x85525c49),LL(0x9ccf06d0,0x7fe18358),LL(0x32c35949,0xbf4a513e),L_(0x48b1c1e8)}},
   {{LL(0x8adf6262,0x6619bed5),LL(0xdba7f0af,0x5bb1a013),LL(0xdac22d18,0xe70d7542),L_(0x0c259dfc)}, {LL(0x0bfa91a2,0x195a05ff),LL(0x1ebdd562,0x52bb429e),LL(0x34d0f00f,0x904ed07d),L_(0x9b10bb9a)}},
   {{LL(0x4331ebb9,0xf071319d),LL(0xa2d5580a,0x05e7c4f1),LL(0x1da3b676,0x7abf809c),L_(0xcca6bf1e)}, {LL(0x9290a718,0xf68174d5),LL(0xc59959c4,0xa93e4d89),LL(0x7f4cbc44,0x93b784f5),L_(0xe5e1c561)}},
   {{LL(0x7b8373a6,0x1bf6e31b),LL(0x242d1745,0xb8888fa2),LL(0x010ee8af,0xa7cd3971),L_(0xa9bf1772)}, {LL(0x7e33265e,0xc3866a61),LL(0x541559ba,0x9e1bb3ef),LL(0x9be25bb9,0x1c158895),L_(0xd26003c0)}},
   {{LL(0x187246e5,0xcb1700db),LL(0x97112d61,0x3aa73e7a),LL(0x93e55e2b,0xb2ba60e5),L_(0x34003d9b)}, {LL(0xbcfa0107,0x253d2d5c),LL(0x43a49e2d,0x9fa1e7c7),LL(0x04ad0152,0xd483d4ee),L_(0x05755d69)}},
   {{LL(0xd95b8ba9,0x1d6eb61c),LL(0xeee8e590,0xc7ef0d05),LL(0x47b3133d,0xf255931e),L_(0x31eb4e09)}, {LL(0x60b98075,0x2344f6b6),LL(0xe87936e7,0x789f9ba9),LL(0x00302e85,0xccdbabda),L_(0xa2a5f4eb)}},
},
/* digit=9 base_pwr=2^63 */
{
   {{LL(0x228b3a55,0xd1e183b4),LL(0x3a405373,0x4948ad88),LL(0x18288fb9,0xf45f24f0),L_(0x04bd162a)}, {LL(0x37574ace,0xe86d0db8),LL(0x7b9a4c64,0x1ccef3f8),LL(0x451b09b7,0xee7d7495),L_(0xdc28df83)}},
   {{LL(0xd52a1c1f,0xd08a8ae3),LL(0x7151a72f,0x750d587a),LL(0x03e86cff,0x27e73149),L_(0x59e2acdb)}, {LL(0x527a9878,0x96f4a96b),LL(0x84533bc0,0xe4a1f248),LL(0x038153e5,0xd9998bb3),L_(0xea910931)}},
   {{LL(0x281ef607,0xee8331e0),LL(0x9f73b6f4,0x71264952),LL(0xa04c6f5a,0xecfe722b),L_(0xd447f9c0)}, {LL(0xbb4ca37b,0x8ff53f4b),LL(0xf450eeb8,0x42ab819f),LL(0xb862be64,0x8ba40994),L_(0x7a08433e)}},
   {{LL(0x3c280033,0xa9734b49),LL(0xa8e0d0eb,0xb12f26d4),LL(0x3686d2f0,0xe0111b72),L_(0x1e11df0d)}, {LL(0x7a48dabb,0x069099d5),LL(0xefbc8fff,0xaba55aec),LL(0x6cdc05f3,0x4ea7440f),L_(0x7409d1a4)}},
   {{LL(0x4a159950,0xe99a66d4),LL(0x35d9b8b9,0x4d054e48),LL(0x924ebb42,0x155f0d8c),L_(0xcc5645dd)}, {LL(0xe319a1e4,0xe6d591f2),LL(0xdce90cf4,0xe9b13d21),LL(0x0834867a,0x39cb42c0),L_(0xb288b9e8)}},
   {{LL(0xb4a0a7e4,0xd28d49c3),LL(0x034fe0f1,0xab2cbbd4),LL(0x4cb47055,0xcd15bd1f),L_(0xb122fe53)}, {LL(0x16ef99e4,0xb349d462),LL(0xaa28568f,0x746c4b40),LL(0x4dce044a,0x56cf52da),L_(0xce7ffcff)}},
   {{LL(0xb1b9c2ea,0x0196cebe),LL(0x26971900,0xdb961af6),LL(0x51023370,0x68f286d7),L_(0x840c2641)}, {LL(0xad763cfa,0xa07bb401),LL(0x722e5f14,0xab25bfde),LL(0xac71bbe8,0x06a272b6),L_(0x7294da0b)}},
   {{LL(0xd727b459,0x6aab9dfc),LL(0x67ba95ef,0x695de10f),LL(0x078f906d,0xa987a86e),L_(0xcc5262f1)}, {LL(0xd50493d5,0x3eb5588c),LL(0x1854cdec,0xce9abc84),LL(0x101a021e,0x2ec773ad),L_(0x9a126518)}},
   {{LL(0x7bdb0ab3,0xb55a240a),LL(0x520772fd,0x06998947),LL(0xf475a7f2,0x04f12f32),L_(0xb6d5b226)}, {LL(0x688143be,0x73021050),LL(0x2e88c6c2,0x325ed38e),LL(0x8114699d,0x467cd567),L_(0x2d56b4f2)}},
   {{LL(0xcbfe120c,0xe69958d1),LL(0x952433b8,0xa2e1385b),LL(0x9586b08e,0x8d8876e4),L_(0xc1845358)}, {LL(0xe749a358,0x85028619),LL(0x0332f81b,0xf41d9b86),LL(0x8430e884,0xe0fe7436),L_(0x5fa0aa6a)}},
   {{LL(0xe20a701c,0xbc42935f),LL(0xa8797955,0xaf260957),LL(0xb55c4fab,0xb5f1b7b4),L_(0xbb614800)}, {LL(0x3e2c715b,0x61856276),LL(0xd080dc4d,0x447c4418),LL(0xfaf57b99,0xa5a859f6),L_(0x450d5963)}},
   {{LL(0xa50c7da0,0xaaf263e7),LL(0x431eb854,0x27bdb99e),LL(0x28bc6b58,0xa291a83e),L_(0x18ba826c)}, {LL(0xe155fe7b,0xa92f7c72),LL(0x42ebeb92,0xfdebd3d9),LL(0x54ea2480,0x1d9b3247),L_(0x01266a3f)}},
   {{LL(0x3bb14261,0x40205f43),LL(0x83f98334,0x1e5b99d9),LL(0xf68ea4a3,0x99449ff0),L_(0x73052363)}, {LL(0xb4680c54,0x6506393b),LL(0xe10ba403,0x9fa16280),LL(0x355805ad,0xa8869c1e),L_(0x3bd33351)}},
   {{LL(0x1fe90871,0x60faf2a7),LL(0x26d3ca4b,0x74445704),LL(0x97761552,0xc6294a8b),L_(0x5d498659)}, {LL(0x8306971f,0x79313f41),LL(0x12b7cc1a,0x459c33b3),LL(0x5ff5c231,0xa47413df),L_(0x77510e5a)}},
   {{LL(0x8931a00a,0x1bb13907),LL(0xb77711c8,0x6e5d6bd8),LL(0xf62b4e1f,0x419ecc6e),L_(0xad753454)}, {LL(0x2e05fe27,0xa9d7ebb7),LL(0x0a367c60,0x46afc58b),LL(0x2c8381f8,0x243d0786),L_(0x4a4e4910)}},
   {{LL(0xa5a5886b,0xd9832e54),LL(0x5cc2672d,0xc764b06f),LL(0x38ed1085,0x7eb63b3e),L_(0x98a773f2)}, {LL(0x585bdb21,0x8a62c89d),LL(0x4ef6df8a,0x5e301e9c),LL(0x35b91587,0x4e488867),L_(0xee5d5725)}},
   {{LL(0x9471249d,0xcfed9152),LL(0xfe397a39,0xf83f6a41),LL(0x859a1f08,0x67c5e1b0),L_(0xd1f44f7d)}, {LL(0x40557a71,0xa37c1374),LL(0x124aed5a,0xecd05a92),LL(0x7635cd74,0xccd0d7f7),L_(0x05c36045)}},
   {{LL(0x02596d64,0x81ed48b5),LL(0xfe6d2634,0x8ec54e24),LL(0xfd23aeec,0xa7513cf4),L_(0xf5285343)}, {LL(0x8f6097de,0x2a1c180b),LL(0xe4bdeb22,0xacdc4130),LL(0x253dbb93,0x9daa0c3e),L_(0x05a30794)}},
   {{LL(0x5f61e12e,0x3336043a),LL(0x083aa9bd,0xa83c1fea),LL(0x77fedd08,0x64002d5d),L_(0x131fc28d)}, {LL(0x08aaa0aa,0x210df7fe),LL(0xdba1334e,0x6b43ad20),LL(0x98ade169,0x42a1a05e),L_(0x912ea208)}},
   {{LL(0x6e92d486,0x8a6e2064),LL(0xc3bdcf88,0xc92637e6),LL(0x4ee89ae2,0xa8cbe2a0),L_(0x8fe2ce18)}, {LL(0x688111d3,0x1fafd02e),LL(0x3e63b605,0x315d9dc5),LL(0x0cd0654b,0x02890b79),L_(0x37460a53)}},
   {{LL(0x131fab04,0x1ac884b5),LL(0xfe807119,0x8a1a27c6),LL(0x66f76b97,0x254a6db2),L_(0xdb8b9dad)}, {LL(0x17450bad,0xd2c4396b),LL(0x24d3388b,0x2691986c),LL(0xd9864a47,0x31c0544b),L_(0x73b25e3f)}},
   {{LL(0x2f0f74a1,0xf3ddaad3),LL(0x4dc5341d,0xa0f85073),LL(0x55736fde,0x9f28ac0a),L_(0xdb012f27)}, {LL(0x5b08d300,0xaaec443d),LL(0xa313b8a7,0x9fd17e0e),LL(0xb4213134,0xf10e3114),L_(0xf370d32c)}},
   {{LL(0xd9301e92,0xa6577039),LL(0xdfaeb17e,0x50e6f1b8),LL(0xf5e86ab5,0x58a7b434),L_(0xff883f58)}, {LL(0x2f0371f5,0x1d3d9546),LL(0xcbb7b3e4,0x39c8d544),LL(0x12bf2df1,0x57a123a0),L_(0x1f072a24)}},
   {{LL(0x817dc76a,0xfd0338c3),LL(0x991da89f,0xb99ba801),LL(0x36e30856,0x1dcffb49),L_(0xc7872aad)}, {LL(0x665a28b5,0xf1016ea1),LL(0x290f4c2a,0x86a30740),LL(0xc3331f1b,0x0205d1d8),L_(0x7458d6c0)}},
   {{LL(0x86281810,0xd7ae93c1),LL(0xe873393b,0x227716fe),LL(0x736914e5,0x605a49ab),L_(0x963a1b41)}, {LL(0xa3fd7cc5,0x99cf069a),LL(0x4d54cee8,0x93c53857),LL(0xdc5b96cb,0x080bcc0e),L_(0x137d0d8d)}},
   {{LL(0xf2335efc,0x398d73b0),LL(0xf7bd4b8b,0x00bb4ee4),LL(0x212ffe0a,0x7ab2ea4d),L_(0xa520fad3)}, {LL(0xe2f332a3,0xeeeaddc8),LL(0x55e1d6f7,0x2962d309),LL(0x143e5fcc,0x49553c63),L_(0x65652ccd)}},
   {{LL(0xc941479e,0x203c4316),LL(0x8084b17f,0x855dab4e),LL(0x717ef700,0x4ed7e83c),L_(0x663074c4)}, {LL(0x494ccfdf,0xe90530ff),LL(0x74fb2c41,0x8222ad9a),LL(0x4430a4d7,0x252e55b2),L_(0x1ff22265)}},
   {{LL(0x96506aa4,0x3f484d6d),LL(0xa163cb06,0x10e92da8),LL(0xbd490fb2,0xffd03214),L_(0xac263c3b)}, {LL(0x95792258,0x7dfe3775),LL(0x448fd584,0x9120d23f),LL(0xe2412ec1,0x1de70984),L_(0xa0573144)}},
   {{LL(0xcb479a8a,0xe7cc4f73),LL(0xb248d660,0x1382187b),LL(0x9dc7e0af,0x01c1bd80),L_(0x5c60ddc2)}, {LL(0x7d0256e8,0xbd971088),LL(0x0fcea0be,0x7ac3d22e),LL(0x36e29a92,0xecf0d361),L_(0xacf3f8d3)}},
   {{LL(0xbb9273ca,0x4c568852),LL(0x92c61bfb,0xf3f274f4),LL(0xb258fcff,0xdee7fe95),L_(0xd7eec021)}, {LL(0xea16df72,0x3f7227a1),LL(0xef526f6f,0x8865b774),LL(0xc554b491,0x10280ec0),L_(0x0822704a)}},
   {{LL(0x0aaf5933,0x600576a1),LL(0x3b11f02c,0x63228df6),LL(0x7b64399d,0xb12890cc),L_(0xcd17f990)}, {LL(0xf166eabf,0xa3e9c521),LL(0x6f888420,0x01f649df),LL(0x33266d1a,0xb179110c),L_(0x31f1eb1d)}},
   {{LL(0xaedb7e52,0x2d780714),LL(0x4aebd349,0x76c48d10),LL(0x95533ca3,0xef834a13),L_(0xf261039a)}, {LL(0xbb0ea84d,0x8bf5f008),LL(0x2b897924,0x0c5e907a),LL(0x5462d73b,0x1d6aecec),L_(0x030fd964)}},
   {{LL(0xb5bf76f0,0x2e91623e),LL(0x3010a109,0x1415663c),LL(0x15fea421,0x1f1c3707),L_(0x7f39476e)}, {LL(0x03dd8ccb,0xb068031e),LL(0xd4fde062,0xd50a50f3),LL(0xbde1692d,0xc8df6655),L_(0x8e9ba050)}},
   {{LL(0xa49675bf,0x923698c8),LL(0xf646ab4e,0x9ea67a41),LL(0x67f23bf8,0x4444d11f),L_(0x4ac9487a)}, {LL(0x05829e1a,0x18b3a870),LL(0x354f585f,0x8b8364e8),LL(0x917428eb,0xafd1b685),L_(0x4487c137)}},
   {{LL(0x5cdc38f6,0x8bdefd85),LL(0x9e5f8f08,0x60854dcf),LL(0x5650f1c2,0x3d41abd0),L_(0xc6e01f27)}, {LL(0x338d5d08,0xd45495b6),LL(0x3b1465b1,0x10d2d56f),LL(0x05c210e8,0x2709cd44),L_(0x48f63367)}},
   {{LL(0x3d513963,0xb778f7da),LL(0xbe45c33d,0x237a1a75),LL(0xb843c30f,0xfe6c9001),L_(0x2fb3bb50)}, {LL(0xb7f1d85f,0xdfc37735),LL(0x9212109f,0x63ca258a),LL(0xbcb01029,0xaad826a1),L_(0xad2340f1)}},
   {{LL(0xf2d99edd,0x3824eb60),LL(0x40c15d61,0x96cabc5a),LL(0xdb4188d8,0xe5012748),L_(0x8b0b38b2)}, {LL(0x100a7042,0x73b6c9c4),LL(0x51b0b8d4,0x4b2e4ffc),LL(0x3c179919,0x35fd801d),L_(0x4079a1e8)}},
   {{LL(0xfba7fe0d,0x71487144),LL(0x74cace4b,0xa5684724),LL(0xde29806b,0xaf1f4b15),L_(0xd8b8bd38)}, {LL(0x0418df99,0x3da078d2),LL(0xd3aa1585,0x9504ef03),LL(0xd6fca127,0x68282ff5),L_(0xc362d893)}},
   {{LL(0xa4371234,0xf24ea908),LL(0x602bf155,0xbf610641),LL(0x751d8536,0x169f256b),L_(0xd3de9a54)}, {LL(0xa7afba2c,0x66d0d844),LL(0xc7851ba8,0x292f0813),LL(0xb9773f57,0xc95c32ed),L_(0x5941d955)}},
   {{LL(0x931f72a9,0x2d2921a4),LL(0x0d1c3a96,0xaabec6ae),LL(0x93be322a,0xe0530ac5),L_(0x68a2485a)}, {LL(0x21ed4c4b,0x3539fb21),LL(0xbd921069,0x54c5601c),LL(0xff29a4db,0x33f41e0a),L_(0xb292275c)}},
   {{LL(0x92370d4a,0x2b5f5d00),LL(0xb56aebc7,0x9ce3670d),LL(0xc246a235,0xcfeee196),L_(0x6a801585)}, {LL(0xc688f199,0xc992ad9e),LL(0x47950bdf,0x3828369e),LL(0xc2172706,0x9016c246),L_(0xa7002475)}},
   {{LL(0xd8546f63,0xf288a32a),LL(0xe74ad8ab,0xd866770d),LL(0xa8e0a566,0x41c5b1a5),L_(0x0d4e95a9)}, {LL(0xc8eca337,0x06f5713c),LL(0x1b3285da,0xaf5394d4),LL(0x1c420a16,0x05bdbb33),L_(0x57e5568c)}},
   {{LL(0x8e6a5c3f,0xf167321f),LL(0x12138692,0x4806da87),LL(0x98fd2767,0x23e3b0df),L_(0xc297c470)}, {LL(0xde00e6e5,0xa65aaac7),LL(0xb2271037,0x603b9246),LL(0x39526356,0x6098239e),L_(0x06050e2a)}},
   {{LL(0xdf432ea4,0x989bae16),LL(0x1d1be94f,0xcdff8337),LL(0x1f3f8b31,0xe12f868b),L_(0xeb8efb64)}, {LL(0xc59a6e03,0x3e2e2a6e),LL(0x10ffb8b9,0x35623aa8),LL(0x66bc9d0b,0x419d9c04),L_(0x56947a3b)}},
   {{LL(0x3e403fae,0x80a49b71),LL(0x85262cb5,0xa16aaa21),LL(0xc0552283,0x941ac341),L_(0x2ed3e3d7)}, {LL(0x78d10e08,0x9e2010ad),LL(0xd7abaaee,0xc3d626db),LL(0xad4eebe0,0x616d3812),L_(0xa0fef6e2)}},
   {{LL(0xa85af3a0,0xc44a4210),LL(0xb9a6a0f3,0xd700b9c8),LL(0x00fd9c15,0xaaeaf056),L_(0x882ae9df)}, {LL(0xcc55572c,0x58e16fee),LL(0xa9e63498,0x224b2130),LL(0x27e82ee6,0xe838331f),L_(0x34f7549c)}},
   {{LL(0xf7314698,0xc2d9578c),LL(0x2c89ffc1,0x28f766e4),LL(0xe5782e17,0xdb1237a2),L_(0x38920553)}, {LL(0xdcb19e7c,0x382be491),LL(0x067bb8d7,0xf76d352a),LL(0x0749b544,0x7968d6a8),L_(0x9449b345)}},
   {{LL(0xceb29d97,0xe4a0f31c),LL(0x468714c4,0x62bd6085),LL(0xc9e0da34,0xd2a91a43),L_(0xf8dfc14b)}, {LL(0xf153aa4b,0x13715918),LL(0xf2418048,0x5c701eb3),LL(0x3cc2d39d,0x21bc78c7),L_(0x14e1c5f8)}},
   {{LL(0xd3a03cae,0x5f6a9d91),LL(0x6557c299,0x3753e068),LL(0x733ae857,0x58532343),L_(0xb7e14afc)}, {LL(0xc858b2c6,0x423a2e5e),LL(0x1ec4f502,0x03c6d3e7),LL(0xbd4e4f93,0xb99af7a6),L_(0xb21da53e)}},
   {{LL(0x74665b48,0x00f3a821),LL(0xc7e083f5,0xbb254497),LL(0xae564546,0x09235190),L_(0x0cb8dbf2)}, {LL(0x148c2b65,0x44c9ade6),LL(0x5f11dd78,0x01120f40),LL(0x2095b0d0,0xc2cfde05),L_(0x755f44dd)}},
   {{LL(0x9096ca16,0x0b95bea1),LL(0x6343ed3d,0x92a3cb12),LL(0x403109c3,0x67ede54a),L_(0x8fa7e763)}, {LL(0xd330e123,0xe94c1522),LL(0x853e450e,0xcf570b75),LL(0x8c60e503,0x7b779c33),L_(0x372ea779)}},
   {{LL(0xb7a09ebb,0xb7df5c36),LL(0x6843f859,0xda7c16c6),LL(0xea76230d,0xe12538a4),L_(0x3c692e2d)}, {LL(0x35168d67,0x22a3c5aa),LL(0xcf2d3469,0x90a1a3b7),LL(0x488f516d,0x4a17692d),L_(0xa3eac98f)}},
   {{LL(0x979163b0,0x5261367c),LL(0xa9ef4f26,0x2b64c77e),LL(0x121123d3,0x57b5327b),L_(0xbd85c013)}, {LL(0x60be5341,0x4320f770),LL(0x376a3aff,0x80d5de37),LL(0xbfa35c85,0x804730a8),L_(0xe518a7ff)}},
   {{LL(0xfec3e94d,0xe49a8416),LL(0x5dc38222,0xebea12bc),LL(0xd561b8e1,0x1b6939a7),L_(0x839a2cae)}, {LL(0x689cf9a5,0x0a52d61b),LL(0x69ef35c2,0x2707af62),LL(0x00163171,0x888d6c2f),L_(0x7f4b170c)}},
   {{LL(0xcfab7c65,0xd2cae7f9),LL(0x327075e0,0x10e60343),LL(0x1e1d00e2,0xb1b2d682),L_(0x25112a63)}, {LL(0xbc56c4aa,0x87923cc3),LL(0x3318db1b,0x05fa3aa6),LL(0x07ea55ed,0x08a82010),L_(0x759b62e0)}},
   {{LL(0x5178b724,0x10a520c7),LL(0x4dacb0ee,0xb1fffc0c),LL(0xbbad0a50,0xab6af890),L_(0xdf72a4c4)}, {LL(0xf8407691,0x2bcfca02),LL(0xeabf10ce,0x56944f45),LL(0xaab35dc1,0xa5c174f8),L_(0x319513dc)}},
   {{LL(0xd8227573,0xfd6ac99b),LL(0x15483af2,0x4081f3db),LL(0x7e998f3d,0x08e76dcc),L_(0x6a8419c8)}, {LL(0x1c7b8f2c,0x0551cbe8),LL(0xb6e1d5cf,0x045398f6),LL(0x93712a63,0x31b716ac),L_(0xc002e0c5)}},
   {{LL(0xf2590189,0x1074bc24),LL(0xb0d09d0a,0x9d9df6f5),LL(0x1a06a2f7,0xf880ea5b),L_(0x633280dd)}, {LL(0xe509ec7b,0xc7021265),LL(0x2310dabb,0xe9189561),LL(0x09271f5a,0x910bdd23),L_(0x633a882b)}},
   {{LL(0x258fd823,0x28518543),LL(0xc6491408,0x338b1787),LL(0xc4796256,0x636fbcb3),L_(0xe0ed072a)}, {LL(0x0aff7ca1,0x9306ff92),LL(0x8d3bca69,0x6a18552b),LL(0x5ba5cce7,0x563c7d80),L_(0xb0af6818)}},
   {{LL(0x0a2d6942,0x682b34d2),LL(0xab8f78d9,0x467f7fe4),LL(0xa7e4e833,0xfb7c4f69),L_(0x0e7bfaa4)}, {LL(0xd5513496,0xc4ac863c),LL(0x9b356137,0x74229c34),LL(0xd5e5f34b,0x729a99ef),L_(0x3111277a)}},
   {{LL(0xca92d3bf,0xacf9987e),LL(0x5bcd837b,0x84c9d53e),LL(0xa434a89c,0x0ac3dbe1),L_(0xcc196982)}, {LL(0x05a4b105,0xbbe7f4a2),LL(0xdc365bca,0x2222c71f),LL(0x0091add4,0x5e600780),L_(0xc0e95213)}},
   {{LL(0xe0a35b2a,0x5acac230),LL(0x8afcf7dd,0x5089506e),LL(0x57de9730,0xf05a525b),L_(0x91a6dc15)}, {LL(0x8c76bd09,0x74d9f43b),LL(0xa8508401,0xdca2c05b),LL(0x4bcbac2f,0xdc6633b9),L_(0x369c598b)}},
   {{LL(0x1bc30bdf,0xa7ddd558),LL(0x2b2504d6,0xe21dd977),LL(0x66baebc4,0xb4d573cb),L_(0x9f9fe00a)}, {LL(0x352d2cef,0x3c8cf82d),LL(0x5dca0593,0x12474a96),LL(0x8d29a0c1,0x55049164),L_(0x861192c4)}},
   {{LL(0x8806d22e,0x827b3141),LL(0x25ab2860,0xc0bcac45),LL(0x45f1bee6,0xe2c1046e),L_(0xc00e053d)}, {LL(0x0b973abf,0xc60d4681),LL(0xa5d34050,0x05dd04f3),LL(0x1ae3e5bf,0x64bbe857),L_(0xc97287f7)}},
},
/* digit=10 base_pwr=2^70 */
{
   {{LL(0xa40697e4,0x6dcf122e),LL(0xe4c0167a,0x179d681e),LL(0x177fcb32,0x2e1c6f2e),L_(0x52ddbb12)}, {LL(0xdfe5e465,0x4c70a028),LL(0xd6ef3161,0x6183ea61),LL(0x4d64db2b,0xd053bce3),L_(0x260d4eb6)}},
   {{LL(0xe6f3cece,0x3c37feb1),LL(0xc31c16a4,0x857f0262),LL(0xbbc0b11e,0x091b8ddb),L_(0x5637b740)}, {LL(0x8541cfa7,0x772c3ab4),LL(0xc2c0eb60,0x71e71bdf),LL(0xcd2d8bee,0xd29d11b6),L_(0xb9533c16)}},
   {{LL(0x4b9ffc6b,0x8ee5427c),LL(0x10f45410,0x997ab138),LL(0xff6b61a7,0x3b574c18),L_(0x394db51a)}, {LL(0xbeade6c7,0x315f47ce),LL(0x04f8a73c,0xc77bac27),LL(0x39f2d0a5,0x28b192db),L_(0x2934b689)}},
   {{LL(0xdfa7909f,0x374a8799),LL(0x3c94fdda,0x121579dd),LL(0xb4c9c6a6,0x2b2a2fe1),L_(0x56fd5e90)}, {LL(0x2dec18ba,0x467e925a),LL(0x8c9027cd,0x51b39838),LL(0x33d05082,0x6dd45eec),L_(0xbdecbc09)}},
   {{LL(0xe4793da1,0xce68ae65),LL(0x1c93f463,0xdf1eb924),LL(0x67519764,0xa6d12a17),L_(0x50dd0b17)}, {LL(0x1e13ad02,0x43f0840b),LL(0x88680227,0x648dece5),LL(0x0b8a73e1,0x9e1099c5),L_(0x6364790c)}},
   {{LL(0x00e5b508,0x16e7b26d),LL(0x41d10bda,0x2fb5b17c),LL(0xf5f94483,0x964b96f7),L_(0xc11a04d1)}, {LL(0xb577452c,0x8083550f),LL(0x8030936a,0x2f106bf6),LL(0xe11783de,0x0cfd54c0),L_(0x4ef38a16)}},
   {{LL(0xd3bf6d03,0x6afe9c51),LL(0x54c55faa,0xbb0f1cd8),LL(0xb0816531,0x27df4a0c),L_(0xa678d64c)}, {LL(0x72f5b39c,0xc24bf200),LL(0x0cc6c1f6,0x39525315),LL(0x2d46495a,0x758c33c9),L_(0xd07deb3e)}},
   {{LL(0x9c7bf29a,0x2cb77bd8),LL(0xb9eef76b,0xb7245ac8),LL(0x0a33b879,0xe50ef27d),L_(0x90583d39)}, {LL(0xb91f697a,0x1ca8036c),LL(0x3704acdb,0xbee50919),LL(0xf053ecd4,0xece7c952),L_(0xc1b42c32)}},
   {{LL(0x990b1bde,0x0420911c),LL(0x39fa1b1a,0x4138b100),LL(0xafe94d6f,0x24ff9ef4),L_(0x17756d00)}, {LL(0x528a709c,0xa50cb647),LL(0x66685df4,0xbb306764),LL(0x907cbbc6,0x74217b89),L_(0xb3971d2c)}},
   {{LL(0x0bab31b7,0xcccd2b33),LL(0x73e2182d,0x5f8e6b04),LL(0xb7efd33b,0x4cd501f3),L_(0xf5c8e01b)}, {LL(0x83d20ea3,0xf475121c),LL(0xb4c7ec19,0x47427d10),LL(0x6eb988c6,0x7acc59a0),L_(0xee8542ac)}},
   {{LL(0x591ee336,0x16ee1f0b),LL(0xbd069c60,0x27985f4f),LL(0x7a51d8bb,0xfe74d9ab),L_(0x699af4f3)}, {LL(0xaa92cd9e,0x9ed60020),LL(0xddd73047,0x2a0eac86),LL(0x18315901,0x8dbf60b9),L_(0xd894e7a4)}},
   {{LL(0xaaf7ef60,0x3d3b7078),LL(0x1892a0ca,0xf09fd9b4),LL(0x00b8f278,0xf7dd85ca),L_(0x18c0622f)}, {LL(0x54e7df9e,0xe305da30),LL(0xf030f7b0,0x67dd41aa),LL(0x88c0e226,0xc437eba8),L_(0xbebf59f2)}},
   {{LL(0x938be7f7,0x7f0d8c8a),LL(0x93333f94,0x74f9489c),LL(0xba77e9c0,0x529b3458),L_(0x5132809e)}, {LL(0xfea9f3ed,0x93d7e4fe),LL(0x2d6cdf17,0xce0cdb8b),LL(0x1c3846b0,0xe0a036d5),L_(0x41588868)}},
   {{LL(0x3b25f1dd,0x640f0895),LL(0xa1b0f513,0x6a36628c),LL(0x0a1631ee,0x22c92416),L_(0x57e72f37)}, {LL(0xb5997003,0xa3c8067e),LL(0x34446f40,0xe28f2eda),LL(0xca50379b,0x0c2d44ef),L_(0x86add37e)}},
   {{LL(0x1879da9b,0xf1be18cc),LL(0xc9b308fd,0x5708ec7f),LL(0x088da37b,0xa743ca6b),L_(0x8175ae9b)}, {LL(0xd86f2e96,0x9bf8faf1),LL(0x3935839e,0xae7389fc),LL(0x784c1096,0x1aa41e81),L_(0x5ea779cf)}},
   {{LL(0x2380678f,0xe93e1c94),LL(0x38fe23d2,0x68003338),LL(0xa4954b13,0xfa7015a9),L_(0x065ec161)}, {LL(0x3e892822,0x19b8125b),LL(0xaf0c1860,0xd66a6abf),LL(0x8579e7ef,0x7516c1ae),L_(0x9c6fa132)}},
   {{LL(0x8051f522,0xfd3b4b27),LL(0xc58d2a96,0x89473859),LL(0x8c0cdac3,0xd7eaeef0),L_(0xdea9f3b9)}, {LL(0x0639dd0d,0x862785aa),LL(0x6333d5f0,0xd61bc685),LL(0xd861b3a5,0x83d3ee5e),L_(0x3b940e97)}},
   {{LL(0xb145ba08,0x2aaf9072),LL(0xc3f907fa,0x2f3077a0),LL(0xbc426169,0xc99c4c21),L_(0x6cffa60c)}, {LL(0x6ee68684,0x236b0866),LL(0x2624e5dc,0xe6b22138),LL(0xaa9f8c57,0xc8cde662),L_(0x93a86894)}},
   {{LL(0xcedbdd1b,0x51eec28a),LL(0xd594682e,0xd4deae6d),LL(0x9e368887,0xaded26ea),L_(0x12b6da79)}, {LL(0xf79fe22a,0xa610dbd9),LL(0x29369dca,0xc8e53035),LL(0x8d4db547,0x6d7b9831),L_(0xe42e7214)}},
   {{LL(0x6d88e9db,0xaf929b4d),LL(0x7078cb4c,0x09f3ff8a),LL(0x2c88cea6,0x95aa1478),L_(0x4aa22af5)}, {LL(0x4d819417,0xe1f9b46c),LL(0xc564232b,0x3fdecbde),LL(0x368fe183,0xce1f9cb4),L_(0x091d0d56)}},
   {{LL(0xb55224a6,0xc92c61ce),LL(0x12d8c2ac,0xe42fc5df),LL(0xbae2229f,0x3afc274c),L_(0xaa2d85b2)}, {LL(0x85cd8bfd,0x97118ed0),LL(0x3f0f7abe,0x5e361376),LL(0xaaf47545,0xcad4d556),L_(0x5d8ce6e2)}},
   {{LL(0x18d8ff5d,0x18b51ba2),LL(0x38eb90ea,0xaf0eee18),LL(0xfc1b2868,0xbb73ea5a),L_(0x04dbbb13)}, {LL(0x88d94a30,0x730a7041),LL(0xecbbc48f,0xd183ab02),LL(0x9867c6b2,0xf198711f),L_(0x29700a86)}},
   {{LL(0xf225ca3b,0xebadfce9),LL(0xac6e8138,0x7b3285ff),LL(0x09069b77,0x1091afcf),L_(0x7e8bc830)}, {LL(0xdcc4bc5e,0xf1844abf),LL(0xcca811f9,0xf655f51b),LL(0x7e393926,0xa022dcd5),L_(0x7d04ce77)}},
   {{LL(0x32d143d7,0x07002476),LL(0x4ba2c132,0x028e4eab),LL(0xf7ed9d04,0x65e3cb55),L_(0xec4c3928)}, {LL(0x1cab2512,0xdf940bc6),LL(0x738a5390,0x734bf044),LL(0x2643f52d,0x11593f75),L_(0xdd379063)}},
   {{LL(0x1759457c,0xcee75815),LL(0x09e08c84,0xbfd1d973),LL(0x5f64050d,0x6b3087eb),L_(0x15138f44)}, {LL(0xc0453460,0x776dfb51),LL(0x19bde576,0x07f4f3a2),LL(0xb17003a3,0xb6a6cf4d),L_(0x25dd19eb)}},
   {{LL(0x53f04643,0x8525ef52),LL(0xa60e3c60,0x9a0888e3),LL(0xde6f955e,0xda028b89),L_(0x6d89bc76)}, {LL(0x2711d947,0xd109e67e),LL(0xb2e621ad,0x19b5f6ef),LL(0x75754b8c,0x85c8d6d5),L_(0xbd6dfaf2)}},
   {{LL(0x0a0de3ce,0x725322a6),LL(0x6f41138b,0xabf6f3c3),LL(0xeb4d3502,0x6dfedcca),L_(0xcf8c4120)}, {LL(0x08adcae8,0x33d4bd69),LL(0x1e238409,0xcfe1b692),LL(0x659e68b8,0x82c86924),L_(0xed3a9241)}},
   {{LL(0xd351d4d8,0x141df0b6),LL(0x232a8ee6,0xaaee8341),LL(0x6a97a114,0xaafe4f6c),L_(0x3cb8f982)}, {LL(0x3ce36b0a,0x4106201e),LL(0x96a879e9,0x40106385),LL(0xfda3c463,0x9e471096),L_(0x3f6ed17a)}},
   {{LL(0x15f6014d,0xb5217d60),LL(0x166ccebf,0x421c3d60),LL(0xcc23ecec,0xc0f18f53),L_(0xde559d43)}, {LL(0x1ae450bb,0xf66ae880),LL(0x7de0a653,0x2e31cf5a),LL(0x1b3a8046,0x916c70ac),L_(0xf73fb01a)}},
   {{LL(0xd287d1cb,0x3e47dc5d),LL(0x70eb7ae0,0x42cffa9a),LL(0xc6ddb9b7,0xb751c3c5),L_(0xb82e85ea)}, {LL(0x7f3bfdc4,0x8676c057),LL(0x711d3839,0x2f6a12cb),LL(0x4e43cafd,0x273385a8),L_(0x94a8904e)}},
   {{LL(0x1d472cf9,0xb53ac163),LL(0xc380e03f,0xf1c57e23),LL(0x5f7dfde3,0x53f9522b),L_(0x51e85640)}, {LL(0xf3d29714,0x3c47cbaf),LL(0xc0110dc8,0x561e0754),LL(0xbe952c4a,0x6bcd178c),L_(0xba15112f)}},
   {{LL(0x24b41db7,0x36a421ac),LL(0x98891ee6,0xdc178876),LL(0x619cb9a4,0xa266089e),L_(0x9c33f043)}, {LL(0x9fd8767e,0xba9a6d1f),LL(0x20bce515,0x8cbdf583),LL(0x46c198f4,0x23f49a00),L_(0x847b2ffd)}},
   {{LL(0x281cc4ea,0xc3d2804e),LL(0xdb1f2dbb,0x22a16678),LL(0x638a7949,0x7ec2aa7b),L_(0x9a43ca9e)}, {LL(0x187243a6,0xb99ca6d8),LL(0xd4fc6e37,0x2d4040d0),LL(0xbd604fb9,0x8cefbced),L_(0x1938ccf3)}},
   {{LL(0xfbd9e1c5,0xd10937a9),LL(0x2f42feaf,0x2fa67093),LL(0xe7b103db,0xb57b1840),L_(0xe03cfeaf)}, {LL(0x1368261e,0x719e1d06),LL(0xfa867b56,0x192c11f4),LL(0xa6cd197f,0x9b41db84),L_(0xc6acf157)}},
   {{LL(0xd68a5a4a,0xddd06586),LL(0xe7a17b19,0x52664a67),LL(0x2621ebc3,0x6f5736f5),L_(0xc8afa211)}, {LL(0xd267f259,0xbacb7a85),LL(0x15b16a2a,0x5c0b0c67),LL(0xaa7c18e3,0x166ad375),L_(0xd0350e8d)}},
   {{LL(0x9a21bdff,0x8411fa8b),LL(0x6d5b0f02,0xde70b24c),LL(0xf5bd9bc8,0xd7443ed1),L_(0x996aa702)}, {LL(0x2b84584f,0x3ea3f8f2),LL(0x373c9339,0x099fde12),LL(0xbc0565bd,0x8fd32c81),L_(0xf5bc6acd)}},
   {{LL(0x7e78ce17,0x12d214f6),LL(0x2fbd45bf,0x5f204c64),LL(0xaf966aac,0x62d03da4),L_(0x3d6e4888)}, {LL(0x0fc086e6,0x14494699),LL(0x61bbb841,0xcc71f940),LL(0xec60a687,0x65805046),L_(0x1641dea6)}},
   {{LL(0x1ee6996c,0x8eb81a1a),LL(0xa5c29c33,0x547ea700),LL(0x71308d2f,0x9126c06e),L_(0xb4936b4c)}, {LL(0x5fecdc41,0x49cacc43),LL(0x7029bedd,0x89bc7a7f),LL(0x6331b109,0x93db1b8f),L_(0x60235335)}},
   {{LL(0x18292506,0x93a8f9cc),LL(0x0c3b64eb,0x8d658c23),LL(0x33eef875,0xc22c9665),L_(0xd49b8f05)}, {LL(0x2081f85e,0xbea17435),LL(0xc230d670,0x6f44daf4),LL(0x5ff2900d,0xdea368bd),L_(0x4f64c46a)}},
   {{LL(0xfcb125d6,0x097e443e),LL(0x9f812e98,0x96ff5377),LL(0x0cce254b,0x562b2166),L_(0xddbade23)}, {LL(0x73e36237,0x992060e2),LL(0xbca2c3a8,0xa968d30e),LL(0x139e42f9,0xe5b89488),L_(0x3ba1f60d)}},
   {{LL(0x3e86ab5c,0xb177e3f9),LL(0x43f6161d,0x95050abc),LL(0x580bb577,0x75270515),L_(0x4a847f51)}, {LL(0x0ce33d23,0xcab8647e),LL(0x9f53c335,0x139f525e),LL(0x9b1f28f4,0xa2e38250),L_(0x5690d259)}},
   {{LL(0x193d7b51,0xbe57f373),LL(0x22feedbc,0x192cee98),LL(0x66699aba,0x69e0895a),L_(0x0f685742)}, {LL(0xe787c996,0xb3f66818),LL(0xccac8670,0x5ee5c7cc),LL(0xe1645254,0xb7d10719),L_(0xae1e2e81)}},
   {{LL(0x17022e02,0x2f237075),LL(0x36a4e5e4,0x01453fba),LL(0xba3615f3,0xa703f343),L_(0xd6276794)}, {LL(0xb5aab8d0,0x30b0a7e8),LL(0x3f7b2b8e,0x33588983),LL(0x8a8547cc,0x64a4028c),L_(0x0ac79b94)}},
   {{LL(0xbbdec908,0x6c0cb953),LL(0x9af46a78,0xf8e40103),LL(0xca5b89e3,0x5e56c8f6),L_(0x7d134424)}, {LL(0x040b1ce5,0x76bd7772),LL(0xff8f0795,0x396f023d),LL(0x6bb455f6,0x4243e710),L_(0xbe0cd278)}},
   {{LL(0xff00cfcf,0x7a3a34e7),LL(0x8154829a,0x59745afe),LL(0x13dec9bb,0x00460ef4),L_(0x7d4cb470)}, {LL(0x664b5768,0x467ad23b),LL(0xa9eb5d4a,0xd6ac872c),LL(0xc30f32ed,0x7527a687),L_(0x3f334a32)}},
   {{LL(0xf2877172,0xa577b4e7),LL(0x50267b62,0x7ac6ab0a),LL(0xbbb22aab,0xce8c4374),L_(0xb58c0879)}, {LL(0x5d2444bc,0x1362bb86),LL(0x6236944e,0xa58498c8),LL(0x03a3218e,0x4585f77c),L_(0xd30bdc81)}},
   {{LL(0xe1c0eb9a,0xf69271e6),LL(0xf6273f18,0x239476a1),LL(0x3701952a,0xb94d2063),L_(0xe82c2b26)}, {LL(0x39ed615b,0x44f390c3),LL(0x90593547,0xa4418ea5),LL(0x1ce9df6c,0x76cd5c55),L_(0xda44a03e)}},
   {{LL(0xf5dcc901,0x62fd7874),LL(0xcf83396e,0x74f61b6b),LL(0x9d3a5873,0x10ca6b61),L_(0x5ef1b62a)}, {LL(0x848244dd,0x81922f08),LL(0x3ee1023e,0xbc236621),LL(0xb2a50ed6,0x0cbc779a),L_(0xb8fa6a49)}},
   {{LL(0x4914c41a,0xb8a19540),LL(0x430e8c3d,0x957adbf0),LL(0xd2aac8dc,0x64d6ca82),L_(0xd8b3157d)}, {LL(0xdb015ecb,0x42372196),LL(0x9a5dfacf,0x05451b08),LL(0xe16c7207,0x463bc66d),L_(0x69536281)}},
   {{LL(0xfc7e9660,0xb8358066),LL(0x918a74d2,0xa1915c8b),LL(0x8d26603b,0xfdddd6dc),L_(0xfe3b2017)}, {LL(0x2166a1d3,0x79912871),LL(0x71bd6832,0xe96aa265),LL(0x98a560dc,0x43b8858b),L_(0x3b4e1e7b)}},
   {{LL(0x2d3f3786,0x95813382),LL(0x6e6cb17c,0x89e2322f),LL(0x3568d961,0xfbe11c03),L_(0x5e1857de)}, {LL(0xb0011b62,0xb6d17af8),LL(0x76372b21,0xfa35ee62),LL(0x88f30207,0x496e0cfb),L_(0x1eab8b49)}},
   {{LL(0x233a4ea3,0xed225296),LL(0xfbf21df5,0x18a0b19e),LL(0x49f70f9b,0xfd4a3974),L_(0xff579fa6)}, {LL(0x6935d1e1,0x4ca49b02),LL(0x5398c8d2,0x465d1798),LL(0x83dab70c,0x3925587b),L_(0xc2e61a09)}},
   {{LL(0xadc74708,0x97b00e28),LL(0x822c5927,0x4ea78214),LL(0x29de362e,0xe0075db8),L_(0xf94b10e6)}, {LL(0xcd9ea163,0x4071fbdd),LL(0xa89e9ce3,0x5861958d),LL(0xb0f23643,0xfb6d160c),L_(0x2e57bc83)}},
   {{LL(0x1cf01491,0xfd9abd17),LL(0x59f1c903,0xfdabdb45),LL(0xfe106ce3,0x28aec19b),L_(0x1e243bfb)}, {LL(0x5dea7b0f,0x9dd7eace),LL(0x0d4e5c7f,0x7aa326d1),LL(0x58144fba,0x0c0b55eb),L_(0xf89a2bf1)}},
   {{LL(0xf492b1ff,0x80211303),LL(0x4c3d5f56,0xa0a42f80),LL(0x812c4f34,0x0513dab5),L_(0xb494414d)}, {LL(0x2b196b3e,0x8292825d),LL(0x845aa8d7,0xfccdef2e),LL(0x8309d41f,0xa6c098eb),L_(0x8feed613)}},
   {{LL(0xe5e506e6,0x34984e59),LL(0xa9dc7b02,0xc78ed700),LL(0x023cf8a4,0x8818f370),L_(0xb1fe8a66)}, {LL(0xedb22c01,0x61bbfcfe),LL(0xcc714f92,0x4b1c28be),LL(0x87de5c20,0x13f66db8),L_(0x523fa798)}},
   {{LL(0x311b814d,0x18478ee8),LL(0xc54ff99c,0x5fc08d54),LL(0x3d6c85b2,0x2a4234ef),L_(0xd3a39852)}, {LL(0xef0ad115,0xfa0e9b48),LL(0xdfcd9d52,0xded54ff0),LL(0x8d2d8068,0xe6033e94),L_(0x39c4cac0)}},
   {{LL(0x567ac24b,0x31e079f2),LL(0x595877b2,0x876a06e1),LL(0xd77d6297,0xabca81b2),L_(0x0324c3ba)}, {LL(0x27382ebe,0xbd7f7ff0),LL(0x5a3b92f8,0xb11d9de0),LL(0x31e36d28,0xba48fe41),L_(0x3263cd91)}},
   {{LL(0x845700c2,0x659750c3),LL(0x265533c8,0xbfa2263b),LL(0x46779254,0x426c9332),L_(0x73f00210)}, {LL(0xc0627927,0x919159e1),LL(0xc1896bdf,0xb8633c7d),LL(0x62049111,0x57a02e38),L_(0x159b1b0e)}},
   {{LL(0xc0ef9c33,0x87339367),LL(0x80f32c7a,0xd690f5fd),LL(0x4cb8372b,0x74004a4c),L_(0x92a5a93f)}, {LL(0x62fc7f7f,0xc75eca95),LL(0x2883c5ec,0x87991514),LL(0x1a3939cc,0x9eeab34e),L_(0x57489f74)}},
   {{LL(0x6dda4270,0xe95ee7c3),LL(0x68b80ca0,0x5637d166),LL(0x0cc0434e,0x108326a5),L_(0x11ccac8e)}, {LL(0x66908f8f,0x768a4dc5),LL(0xe0516b2b,0x24086620),LL(0x0390f0a6,0x34cd95fd),L_(0xb3965e3d)}},
   {{LL(0xa579c74b,0xf4d1cd8b),LL(0x2437af0c,0xa682fc6c),LL(0xcf5b3a95,0x13389ce6),L_(0x0026ae74)}, {LL(0xc24a47b5,0xf37dbb35),LL(0xf06e1e15,0x9d47aaf6),LL(0x196773ed,0xab42affb),L_(0x76bb4374)}},
   {{LL(0x78457341,0x0583dc84),LL(0x564d9444,0x6a0c10e4),LL(0xbf46a7bb,0xe6adebac),L_(0x93e051d2)}, {LL(0x3e924b68,0xbf5e7329),LL(0x4afb1643,0x9cb33af1),LL(0x5460942b,0x1005e77d),L_(0x483528ac)}},
   {{LL(0xa399de32,0x867305ce),LL(0x19e34f30,0x2200477c),LL(0xd31a5a81,0x9ac2ed5d),L_(0xa22c4024)}, {LL(0xa5a011ec,0x9cb94041),LL(0xadcbd258,0xf7a7e8fd),LL(0x26131e4c,0xbcae3bd4),L_(0xa63111af)}},
},
/* digit=11 base_pwr=2^77 */
{
   {{LL(0xe00bdded,0x44e43c61),LL(0x2a3a0320,0xc584d310),LL(0xb6fe7f05,0x951b39e7),L_(0x1c47e70b)}, {LL(0x1eae5fc6,0x518f5441),LL(0x5cd44db2,0xc505aeda),LL(0x216fb5f2,0xbb781e43),L_(0x7e14b38e)}},
   {{LL(0xf9a62c2b,0xe0d93869),LL(0x60f3a9a1,0x6d3f0de7),LL(0xb3106bbb,0xb0f77641),L_(0xc3071383)}, {LL(0x9b298ebc,0x184aed09),LL(0x0bacf017,0x334394d4),LL(0x79c1d22f,0xd95473d2),L_(0x93838083)}},
   {{LL(0xcb7cb58a,0xf3fc070e),LL(0xed4652fb,0x60dba7d5),LL(0xe19d60f2,0xe420c869),L_(0x51893296)}, {LL(0x097e4dab,0x707e030c),LL(0xd90a9475,0x0dcda4fc),LL(0x9803c9fe,0x2bbc44ce),L_(0xa6483469)}},
   {{LL(0xfabcb24a,0x10355326),LL(0xdcb5236b,0x91ed19ce),LL(0x4f316ca1,0xd20efc4e),L_(0xd00bff40)}, {LL(0x1a2e490d,0x2b83ac01),LL(0x96a6cfc6,0x0fd81acf),LL(0x8da694f8,0xfcad677f),L_(0x6bf5e71f)}},
   {{LL(0x248ec9b4,0x11ef88d2),LL(0xe791a3bc,0xda678712),LL(0x370ca0ea,0x7a293625),L_(0x4e4cd7d7)}, {LL(0xddfa0a4e,0x1f856294),LL(0x180b43b4,0x6bd2e637),LL(0xf728ccb4,0x91baf444),L_(0xe5fa917a)}},
   {{LL(0x9e4adc26,0x8981411a),LL(0x6d65f85a,0x7eb4dd2d),LL(0x58e82e86,0x97c312ce),L_(0x8bfebe6a)}, {LL(0xd7b84e09,0xc95c8ff2),LL(0x639198d0,0xd7ac3654),LL(0x7546fc40,0x504a9f59),L_(0xeac2deb7)}},
   {{LL(0x9edb0426,0x476b8958),LL(0xa2e1a729,0xf14565b5),LL(0x4bab1b96,0x3d3ca262),L_(0xc0e1a0ef)}, {LL(0xfd4e9a48,0x0043c8ff),LL(0xba0665c7,0xafaf8910),LL(0x0b1e0a8d,0xcbeb2d8e),L_(0xe710e7f4)}},
   {{LL(0xebbefd51,0x44fc634e),LL(0x2a776b18,0x9ddeace8),LL(0x08a9107e,0x909eff4c),L_(0xb7934059)}, {LL(0xf816fa9f,0xefd70514),LL(0xfaad7709,0x598db9de),LL(0x0094b64c,0xc6634c12),L_(0x8b95add1)}},
   {{LL(0xd6fb5d4c,0xadb1a9c2),LL(0xd2967310,0xc8ec0f39),LL(0x1908657d,0xa25710b3),L_(0x19a27dfc)}, {LL(0x311947c4,0xe187cb1c),LL(0x8cbb3e59,0x1bae3f86),LL(0xafb6e7b5,0x44ea2977),L_(0x781e26f2)}},
   {{LL(0xee18a383,0x60b120f0),LL(0x24a67462,0x91636563),LL(0x29fd66f4,0x6538f012),L_(0x1b5c47df)}, {LL(0xda52b57b,0x9a0bd35b),LL(0xd095d4fe,0xcee45c94),LL(0xafb1fcae,0x042232d0),L_(0xcff50cb9)}},
   {{LL(0x54397158,0xd06ed8f6),LL(0xd53d3bfe,0x67644c3e),LL(0xaa8e6af8,0xa26425e0),L_(0x1f9101fd)}, {LL(0x6fb8009e,0xdfa5a71f),LL(0x7b2b6229,0xc4fb8d84),LL(0x413c2883,0xa71e26d8),L_(0x442bf1c5)}},
   {{LL(0xae3de434,0x45610301),LL(0x828b8be8,0x4ec1365e),LL(0xb95b637d,0x41dbce62),L_(0x1ca7419f)}, {LL(0x9637a66f,0xc6d80952),LL(0x5a685913,0x9289159f),LL(0x761cc88a,0x1da0d501),L_(0xf7bfb795)}},
   {{LL(0x37a59563,0xb4ca54d1),LL(0xe2ccd3ff,0x5128864b),LL(0x8aad87e4,0x559dacdc),L_(0xdf79df2f)}, {LL(0x89718ac8,0xea353321),LL(0xe6676310,0x2012406a),LL(0xc599f213,0x1eb2fe94),L_(0x626c9f76)}},
   {{LL(0xba62fad3,0xa7fededf),LL(0x6ff23700,0xd58195f8),LL(0xd3d6b5bf,0x1ad1505f),L_(0x97f82bb0)}, {LL(0x6f93c77b,0xaf96dc4e),LL(0x30b1fb3e,0x20624dad),LL(0x4fa52512,0x3481aabe),L_(0x78a7c4e4)}},
   {{LL(0xba1c1391,0x9fb8e966),LL(0x5ff12f04,0xb9c0dea0),LL(0xd59d1993,0xe8a99c53),L_(0x6cef1107)}, {LL(0xb44f7478,0x97335f28),LL(0x400e3354,0x6f550c81),LL(0x3749bb24,0x67501a21),L_(0xdbc433a0)}},
   {{LL(0x2ddbdad7,0x5e95c8ee),LL(0xf4a3dab8,0x74fa6bf7),LL(0xf68aac5d,0xe404df62),L_(0xca08be37)}, {LL(0xf3252764,0x934f4963),LL(0x6033be81,0x65ea023b),LL(0x39f3ad81,0xf2ab45c6),L_(0xec27db5a)}},
   {{LL(0x6c9e60de,0xd82ebdb2),LL(0x3c94302c,0xf4278e99),LL(0xf76b6d5d,0x177bfa74),L_(0xaec52e0d)}, {LL(0x0698c8f8,0x8af11d92),LL(0xdc8707e7,0xdcfeb87c),LL(0xdf966b87,0x790b8b3b),L_(0x25a49666)}},
   {{LL(0x0d632c2e,0x82b42706),LL(0xca5f0400,0xeb7e23be),LL(0x67d13334,0x6bef8f5e),L_(0x19a89324)}, {LL(0x5b9d7908,0xb943b3e3),LL(0xe69dc86d,0xe1142402),LL(0xe61e180e,0x3186f5ab),L_(0x5490857c)}},
   {{LL(0xee73f3c9,0x1c021526),LL(0x229aab7e,0xd5f0330f),LL(0xbcd8ae19,0x34f15f5c),L_(0xab3c7439)}, {LL(0x0f2d33f9,0xf2452d93),LL(0x23f2f3b0,0x0c4b0f70),LL(0x322c677d,0x311431b5),L_(0xc0e186a5)}},
   {{LL(0xd222f30b,0x32c8e92e),LL(0xb855ce32,0xf9051d2c),LL(0x34079f7e,0xca51b73e),L_(0x1cd08eae)}, {LL(0xe4529fd6,0x3ee05e30),LL(0x48ccde21,0x762211ba),LL(0x957b54a8,0x7bd842d1),L_(0x8ad2f209)}},
   {{LL(0x8578a05b,0xf931c6ff),LL(0x22b12a51,0x9c1a24ae),LL(0xc7ad186b,0x3aae7e80),L_(0x35a3a465)}, {LL(0xcd94ffe9,0x02d60546),LL(0x02a0030e,0x6d8dfc51),LL(0xf70532cd,0xa7523214),L_(0xbfa52fcd)}},
   {{LL(0x3898ac4a,0x7ef15aa2),LL(0x04a67481,0xc92d3e61),LL(0x6f146470,0x20a65e93),L_(0x091a434a)}, {LL(0x40c2ff2f,0x5f5f4b1f),LL(0x356b539d,0xef24d37f),LL(0xf5eb8c60,0xd0f3625a),L_(0x5662e1cc)}},
   {{LL(0x04af9658,0x0ddb7f8a),LL(0xd47a8579,0xf5679a0d),LL(0xfa164f3a,0x75f9e25f),L_(0x88f00eb9)}, {LL(0xd2b96f54,0xc75b8bfb),LL(0xc45f8fe0,0x141dae1d),LL(0xf86c989c,0x1a015ae9),L_(0x4cbe108d)}},
   {{LL(0x7440165c,0x252d3538),LL(0x94ed0d57,0xc7935e6e),LL(0xfc485299,0xafc2b1d4),L_(0x2763b62a)}, {LL(0x8b5606a9,0x8714fa4e),LL(0x44261681,0xd015855f),LL(0x03679b60,0x14d9de29),L_(0x84a487b9)}},
   {{LL(0x7fd6372b,0x1fc76dfc),LL(0x30d60bc9,0x835ac98b),LL(0xec54c6b2,0x98cbc171),L_(0xa2ec0376)}, {LL(0x7ab5c98b,0x577a8e9f),LL(0x527bb2a1,0x84028fa5),LL(0xed0d2362,0xb28e8d67),L_(0x1a1d5c1c)}},
   {{LL(0xd8c85ac3,0x8e20b175),LL(0xe895347f,0xbc82df9e),LL(0x763f7da0,0xc32cc81d),L_(0xd889ef15)}, {LL(0xb02c89ce,0xd305dbab),LL(0xb02c2e9f,0x7f0a99ef),LL(0xd40823f0,0xf958bb39),L_(0x7a07c7e4)}},
   {{LL(0xee84ea34,0xc5f043e5),LL(0xea668125,0x13f7ae2f),LL(0x18438628,0xb6ffaed5),L_(0xf72f62c4)}, {LL(0x9cb2856e,0x9a5c926d),LL(0xade5a9b1,0xad05632a),LL(0x4d11a1b7,0xf0f2c69d),L_(0xbfed0dd6)}},
   {{LL(0x60b10c55,0x79943f17),LL(0x930f1ce1,0x15056514),LL(0x70dc2681,0xc9a17b25),L_(0x71659c78)}, {LL(0x4d9c29ff,0x5978aada),LL(0x56151f6b,0x6e84fa9d),LL(0xb01cfa96,0xd9553cbf),L_(0x495ffa1c)}},
   {{LL(0x14645de0,0x50569b84),LL(0x516171cb,0x76814e1a),LL(0x6931ce3c,0xd1694dea),L_(0x18d486a7)}, {LL(0x8fb4ec93,0xcbd84ece),LL(0x694d8a81,0x70c2d8a8),LL(0x6f1a63f5,0x2ca687fd),L_(0x6788bb91)}},
   {{LL(0x66719259,0x78b1b1f1),LL(0xa8fe9c13,0x372aba7a),LL(0xf72aecd5,0x049424c7),L_(0x8c2a627b)}, {LL(0x12996520,0x7a3d9b21),LL(0xd87c3d81,0xeea3b43c),LL(0x6296bed4,0x965623a1),L_(0xf57dda17)}},
   {{LL(0x19faf24f,0xee2a75e2),LL(0xe4882684,0x2608d8d7),LL(0x5dbd9aa4,0x8dd2d8bb),L_(0x1861b78b)}, {LL(0x96024e0a,0x59608369),LL(0x2a96c634,0x3109b985),LL(0x4c72f34b,0x073affe7),L_(0x5e2a53e1)}},
   {{LL(0x7163417d,0xe75a8d8d),LL(0x003ea53c,0x34a71e84),LL(0xf46ea0c1,0xa472172f),L_(0x582a6663)}, {LL(0x5941f31f,0x45ef8849),LL(0xda61e5b5,0x813628f8),LL(0xff6e6be2,0x2f6a6923),L_(0x24a1906a)}},
   {{LL(0xf2dab7c1,0xee4fac0c),LL(0xcca97a61,0x442b5426),LL(0xd1572489,0x1b63f7f9),L_(0xc808b93c)}, {LL(0x86beafa1,0x931a009d),LL(0x00b9ab37,0xc54e5dcf),LL(0xce454abb,0xe3d31b67),L_(0x8846c2ca)}},
   {{LL(0x2bfb9673,0x00a46605),LL(0x6d38eb82,0xd0e900f3),LL(0xac391030,0x56b396f4),L_(0xa240554c)}, {LL(0x111e6519,0x0e375257),LL(0x2daa1839,0x0569eb0d),LL(0x6386031f,0xa43939dc),L_(0x29a2482a)}},
   {{LL(0x79006500,0x19bae3a4),LL(0xeba63f38,0x278b676b),LL(0x2bfc1210,0x5fd221ef),L_(0x18605ab3)}, {LL(0xaddaabbd,0x3e58e744),LL(0x130a0b49,0x86211a5e),LL(0x27a20439,0x06b96609),L_(0x3be9694d)}},
   {{LL(0x31e7bef9,0x6b521dbf),LL(0xe38156c6,0xbb74a0f2),LL(0x41f4bd94,0x79699c57),L_(0x565687a7)}, {LL(0x6af2b091,0xb9a903de),LL(0x15159b8e,0x250d78d4),LL(0x565e3c29,0x674edb33),L_(0x0294da46)}},
   {{LL(0xfa502a7a,0x4f9ab886),LL(0xf7c92c2b,0xb5e6f603),LL(0xcd18776f,0xf25cc6e4),L_(0xa7fa7c45)}, {LL(0x9822aa1b,0x2c1f0788),LL(0x250c2373,0xb4be8a38),LL(0x671f0157,0x5b11123b),L_(0x157f5d60)}},
   {{LL(0x54914217,0x1256ef4e),LL(0x5163b441,0x075b53ab),LL(0x5ecb1889,0x56e20726),L_(0x6494b2b8)}, {LL(0x09f26255,0xd571b91b),LL(0x9475008d,0x328d62f5),LL(0xde6fb9c6,0xe065c610),L_(0x96baaccd)}},
   {{LL(0x45144de8,0xdc428059),LL(0x71bed82d,0x9aff2cca),LL(0x9af32379,0x7429380e),L_(0x02f22f65)}, {LL(0xe3565604,0xdea72af4),LL(0x2b9bc7b9,0x2bf6a43f),LL(0x5a2add5c,0x6b6ae6f7),L_(0xb69ca1cf)}},
   {{LL(0x9ab52f77,0x061d7ef8),LL(0x56d68dd3,0x010840bd),LL(0xae0d756d,0x97508e7b),L_(0xecbb738b)}, {LL(0xadc78972,0x4d5b80a4),LL(0x4dce8c80,0xf9aaef5c),LL(0x256809bb,0xa1ec2e50),L_(0x2ea2af7c)}},
   {{LL(0x74a6dd4a,0x0393f719),LL(0x93285afe,0x4e57160b),LL(0xac7e00ee,0xe124fa0e),L_(0x044bd9af)}, {LL(0x2389bd88,0x80b4129f),LL(0x758d4e27,0x0e8aaba3),LL(0x1051debe,0x4a190f97),L_(0x85906a99)}},
   {{LL(0x2156959c,0x37e2ab37),LL(0xd889d270,0xfe9b7157),LL(0x540652e0,0xd3215a27),L_(0xb88094e7)}, {LL(0xc0c597f0,0xceb6dd56),LL(0x5dc4dd34,0x09436cb3),LL(0x07e11a08,0xc752116d),L_(0x86890fd2)}},
   {{LL(0xe154e3b6,0x3dd689de),LL(0xba6c5590,0xecec9673),LL(0xf4ebc67c,0xdf67ffb0),L_(0x094d0483)}, {LL(0x33ce85c7,0x58ffaf1b),LL(0x04f91a5f,0x741f05be),LL(0x2535c38a,0xffe3294f),L_(0xfe3e3780)}},
   {{LL(0x4c921d99,0x1c787299),LL(0x8c60577e,0xbf10f0e5),LL(0x4f5e9e39,0xdd4daa07),L_(0xcf714fd4)}, {LL(0x92fcbe08,0xf29c8618),LL(0x38ba462f,0x84305704),LL(0x8d7c9a43,0x19694aa8),L_(0xe7510556)}},
   {{LL(0x680c4325,0xb1ac0982),LL(0x45857f5e,0xa4032ac9),LL(0xdf576261,0x92007d10),L_(0xc5ae1832)}, {LL(0x755cfb3c,0xb9ea853a),LL(0xc129c1e9,0x28725f38),LL(0xa32de41e,0x8310cc56),L_(0xaf03c7c1)}},
   {{LL(0x4319f045,0x6a31f14c),LL(0xbb313e19,0xf0a68d54),LL(0xa79121b3,0xf8ad5c75),L_(0x1cd6c518)}, {LL(0x1aca6f0e,0xd728ef75),LL(0xf52ecb39,0x66ed37be),LL(0x244f6df5,0x5253f731),L_(0xc37a0916)}},
   {{LL(0x7e5a8bd3,0x9d03a5af),LL(0x6b5c7a81,0xa4eb71aa),LL(0xa305f15d,0x1dc7fb76),L_(0xbf1adac7)}, {LL(0x20487a26,0x6deaadda),LL(0xf4c07457,0xe0f4999c),LL(0x9bb84678,0xc451bb12),L_(0x975839c9)}},
   {{LL(0xc977cb31,0xd39b851a),LL(0x6ac13ca0,0xcd656ca8),LL(0xa19898ab,0xafd81332),L_(0xd849f9e4)}, {LL(0x2bb5e6b1,0xc3157a16),LL(0x5b02f0fe,0x8acd8d9d),LL(0x69cbb158,0x6d76f690),L_(0x05ce3883)}},
   {{LL(0xc8ff4786,0x2bca8c99),LL(0x186098dc,0xa2caaf78),LL(0xf6be10a7,0x2100372c),L_(0x16272c0e)}, {LL(0x761e39c7,0xdb53800d),LL(0x6551227e,0xc599f6e5),LL(0xc995d474,0xcb2cc9b4),L_(0x73233966)}},
   {{LL(0x39687ef7,0xe8d0ac03),LL(0xc75c74c9,0x921b3cbb),LL(0xc64c35ba,0xa500e776),L_(0xca48727d)}, {LL(0xe79c45b0,0x60ad3863),LL(0x2b1933ee,0x0ce33aaf),LL(0xaee36834,0x4ce8962e),L_(0xb0c0146f)}},
   {{LL(0xb38c8507,0xc359f6e1),LL(0x8697a59a,0x31837b1c),LL(0x98af1b3a,0xf716fe6e),L_(0x7fdc52e3)}, {LL(0x3830cd5c,0x52930fa5),LL(0x90fd85df,0x0ad62260),LL(0x097dfa97,0xebeb447d),L_(0xcfdb2bce)}},
   {{LL(0xe409d4ed,0xe80a1b86),LL(0x4d1e34ae,0x1aaed1d3),LL(0x9eac7e33,0x80bb831f),L_(0x0bb854f1)}, {LL(0x986f7e0c,0x005a4dc6),LL(0x6e3f9cf2,0x52827fef),LL(0x293c91f5,0xcdd9e93e),L_(0xd5679946)}},
   {{LL(0x2c6558f0,0xd0ac15f2),LL(0x4e88fc97,0xa247b120),LL(0x573e9688,0xd0627f7e),L_(0x96e1d752)}, {LL(0x22887f9f,0xd31e0ba7),LL(0xb47ea27c,0x1eea7f54),LL(0x4423561d,0xf9581051),L_(0xc8a211b5)}},
   {{LL(0x42a54364,0xeb2affe7),LL(0x35c0456c,0x972ce84c),LL(0xaeca0047,0x4d9efd23),L_(0x2bdbc15a)}, {LL(0x173fea09,0x1d890ba5),LL(0x99e5f4ba,0xfea0ad91),LL(0x5f0b0f42,0x5619484b),L_(0xb1f3e2bf)}},
   {{LL(0x6cdbcee3,0xc11ff4f3),LL(0xe36e32c6,0xa260adc3),LL(0x5bb4d1a2,0x33e0dbb6),L_(0x1b1ed42c)}, {LL(0xcaa2ba15,0x415b8922),LL(0x9bbb4478,0xb1fbde1c),LL(0xe7f63f33,0xb014bfff),L_(0xd8fe72a1)}},
   {{LL(0xab510c98,0xee7b323c),LL(0x86f8b471,0x37854732),LL(0x23d6cc1a,0xdbbcc348),L_(0x14fb98ce)}, {LL(0x64d5f136,0x6fb100cc),LL(0xa833932c,0xb27f28f5),LL(0x980b1419,0xca8834c2),L_(0x088ce8d5)}},
   {{LL(0x1b1c75cb,0x8c28e4a6),LL(0x712165a2,0x2cbdb243),LL(0x1341e16e,0x5a726ab8),L_(0xfe09486b)}, {LL(0x32264866,0x4f4a819e),LL(0xf2a151c4,0xba655f21),LL(0x37367c7d,0xd157593e),L_(0x2b7d23ec)}},
   {{LL(0x46d94f55,0xc9397bab),LL(0xe0636264,0x65e56a08),LL(0xf946f4e0,0xa259e975),L_(0x55c54603)}, {LL(0xb5060784,0xccc37d6a),LL(0x2614d3d3,0xf709343a),LL(0xfb056c2e,0x89705d36),L_(0xc8dec62e)}},
   {{LL(0xeb841e0f,0x7663ad0e),LL(0xc791db86,0xb1420c30),LL(0x5792fb1a,0xaebf5ced),L_(0x462f4f9f)}, {LL(0xeaecfdd6,0xd5142858),LL(0xc555cb9f,0xe56ab53a),LL(0xdbaff47d,0xb56ba899),L_(0x048149ff)}},
   {{LL(0xd69321d0,0xa06234ff),LL(0x4da9e4a7,0x98b8cc13),LL(0x4be3eee6,0xadbfb76c),L_(0x9b536e85)}, {LL(0x57f0cdf4,0x01e45447),LL(0x54dd6853,0xb307918b),LL(0xa0a5d2a4,0xf7e16814),L_(0x4f89fd0d)}},
   {{LL(0x6c92c96c,0x865c8f6c),LL(0xee136cd4,0xf96358e5),LL(0xc0b20165,0x10dbd508),L_(0x785841fa)}, {LL(0x990a46d5,0x1d1a7d70),LL(0xc190e469,0x6c2d572d),LL(0xa87b8f93,0x03a7ba98),L_(0x3e71c1ad)}},
   {{LL(0x6d9f3e8d,0x2221d899),LL(0xc7dd63bb,0x7c844ad3),LL(0x142035bd,0x11c04771),L_(0x726cc50e)}, {LL(0xf02e1ad2,0x78ab7d85),LL(0xc57c4d2f,0x301c5ced),LL(0x1feac0e2,0x95568955),L_(0xd216f6fc)}},
   {{LL(0xc5b0ee89,0xf045ebf3),LL(0x6770bb55,0xc1ea77c3),LL(0x4e10f859,0xa44993af),L_(0xb75a3354)}, {LL(0x38d373df,0x5c460d85),LL(0x940bd01e,0x2bdad16c),LL(0x264105cb,0xb8fc3b3e),L_(0x09401ace)}},
   {{LL(0x2edb771d,0xd4ae16cd),LL(0x2b675fa3,0xc8055a0a),LL(0xdee3f3e2,0xc08d1ca2),L_(0x28ea187c)}, {LL(0x4252a1b9,0x6d687314),LL(0xf6c34d15,0xba935b80),LL(0x3b383652,0x52c1b85d),L_(0xc5e3b405)}},
},
/* digit=12 base_pwr=2^84 */
{
   {{LL(0xb92b8af0,0x6354d2c9),LL(0x4c952ade,0x5a4d9e35),LL(0x0381968a,0x52d9f13a),L_(0x86c3819e)}, {LL(0xc63b9fe7,0x3743142f),LL(0xa95b1a9a,0x6e186e52),LL(0xaea91efc,0x3de25aa0),L_(0xc91296ab)}},
   {{LL(0xf8921758,0xced4216d),LL(0x02fb0911,0x28cf0cfe),LL(0xd524c877,0x03bdd0b5),L_(0xb3e00db0)}, {LL(0xd91831de,0xd131c59b),LL(0x77e04707,0x47533185),LL(0xf492c767,0x63c30af1),L_(0x202f0676)}},
   {{LL(0x79a5ddcc,0x495cdcc7),LL(0x33ab4ae3,0x7f199f52),LL(0x9d2ff41e,0xbe613bf2),L_(0xf3371b23)}, {LL(0x967f0754,0x76c36728),LL(0xfb8be7aa,0x02af3c44),LL(0xee5a166b,0x0bda3025),L_(0xc9427cb8)}},
   {{LL(0x6371852a,0x473e976c),LL(0xa5e1af16,0x88aeb436),LL(0x368ce839,0x94aabe19),L_(0x00b82580)}, {LL(0xb19a3ec7,0x7d50b7ff),LL(0xd8fc842f,0x81c2bf45),LL(0x8c4eb78c,0xb0a7cd1f),L_(0xf40b4869)}},
   {{LL(0x56f0849d,0x5739e59e),LL(0x79c44251,0x1fc64939),LL(0x8b91a3a3,0xd401b24b),L_(0xd2207cb0)}, {LL(0x9decd9c6,0x17f178e9),LL(0xcede5d5c,0xfa904473),LL(0x7a09efc0,0xca5c00a9),L_(0x1761360d)}},
   {{LL(0x6aca6f61,0x4bda0113),LL(0x3b3dcab5,0x12ccd894),LL(0xb4db3fdf,0xb39be04f),L_(0x1c09bf2c)}, {LL(0xb4705566,0xccbe5154),LL(0xe1559ff1,0xd84fec44),LL(0x11964906,0x9ceede3f),L_(0x0e1b9fc5)}},
   {{LL(0x13111bdc,0x175d3f77),LL(0x17226b06,0xe0971341),LL(0x21244b52,0xd6fa5642),L_(0xfdbb39d3)}, {LL(0x68da34f2,0xb9933942),LL(0x6759bac0,0xd7d7f790),LL(0xee292b1d,0x4382d48d),L_(0xd117a9f3)}},
   {{LL(0xe877a209,0xaf9b8432),LL(0xa22db654,0x12572249),LL(0xbb09b923,0x1501cefc),L_(0xd5820554)}, {LL(0xe0614220,0xaa863b6a),LL(0x397b3194,0x27337829),LL(0x1a9bf0b7,0x6912b5aa),L_(0xc767a242)}},
   {{LL(0x38d01225,0xa0259e70),LL(0xf56ef5aa,0x8e85d7ee),LL(0xfd49df96,0x6bb4a230),L_(0x8f3a3f3a)}, {LL(0x58b99e53,0x3c2be4ba),LL(0xcf1ea9c2,0xe49fcb3f),LL(0x4937a76d,0xdc9b73dc),L_(0x5ea4828a)}},
   {{LL(0xa516c9ee,0x41610138),LL(0x30e01a74,0x71d5b83a),LL(0x8795351f,0x9a45e682),L_(0xa31707a8)}, {LL(0x9edacc8e,0x17bd9c0b),LL(0x80f31bd0,0x871cc69c),LL(0x57013f03,0xb6b7b6e2),L_(0x1073b301)}},
   {{LL(0x4855e4b2,0x124c67e8),LL(0x6946f5db,0xd4d7f50b),LL(0x494ae118,0xf16b9834),L_(0x0c0fe86e)}, {LL(0x3a6a8d31,0x18d69af9),LL(0x54ebd047,0x7f8ed091),LL(0x9daffa88,0xba6e7c59),L_(0x54652a79)}},
   {{LL(0x011788f6,0x9c64aa0c),LL(0x8403c566,0xf896522c),LL(0xcc8f1ea7,0x772627b7),L_(0x707ce584)}, {LL(0x72bb812b,0x57f8a759),LL(0x4f42aa8c,0xc7e7a397),LL(0x4c06faeb,0xeea7a14b),L_(0x1edd837a)}},
   {{LL(0x0c456e85,0xbacf3d29),LL(0xdb2d668c,0x861ef908),LL(0x17a88c53,0x25d01a4d),L_(0xb97ac689)}, {LL(0x0b6e87ff,0x066301cc),LL(0x78d4d6b5,0x51763eac),LL(0xe4c44c3a,0x243f5364),L_(0x55bb2e54)}},
   {{LL(0x69fe2b3f,0x623f7c06),LL(0x969019fe,0xa81b9914),LL(0xba6dd8e5,0xd33612e7),L_(0x0fd330c2)}, {LL(0xc948497d,0x7184b686),LL(0x065a9367,0xd5e74279),LL(0xfa105e4b,0x4bfe5bea),L_(0x30ca8fc4)}},
   {{LL(0x03d46275,0xfcb464d3),LL(0x7fb49121,0xdcb4ccb3),LL(0xf6934477,0xbdf563a1),L_(0x4412ba6f)}, {LL(0x7c07acc8,0x4d4331eb),LL(0xe0dfac8b,0x768b2705),LL(0x022d151f,0xf65a4027),L_(0x251db799)}},
   {{LL(0x5cbe2654,0x32707706),LL(0xbeb70e80,0x9739588e),LL(0x1d1ff202,0xe51c696a),L_(0x000ae445)}, {LL(0xdd740418,0x25fcc7e6),LL(0x646ef912,0xac2dfbc7),LL(0x0d72b69e,0x1ff6f6ad),L_(0x059f3c0c)}},
   {{LL(0x3ade75e8,0x3ac64888),LL(0x7ee14ba6,0xd957b10f),LL(0x7be01ff5,0x33f835d6),L_(0xc155847f)}, {LL(0x45081b73,0x0c08887d),LL(0xda2f5358,0x4fc01e34),LL(0x84d416f7,0x15716e5d),L_(0x3950aa76)}},
   {{LL(0xc4cafbc3,0xcd118e18),LL(0x60a29243,0x78d7034b),LL(0x4d198a7c,0x9ba63f25),L_(0x4b83593c)}, {LL(0x81c83f83,0x9eb5e40b),LL(0xb8f7f9c2,0x3b1941e5),LL(0x27b87b90,0xb7229f73),L_(0x247578db)}},
   {{LL(0x59afdea4,0xf6e5451f),LL(0xbdfb7efd,0x3c7977fc),LL(0x8262d80b,0x0ff79532),L_(0xb6d1bfa5)}, {LL(0x85d46816,0x41a59faa),LL(0xbb16202e,0x815bfdff),LL(0xf25c4d84,0x26a648ca),L_(0x2a115df7)}},
   {{LL(0xc47be93f,0x8807158f),LL(0x31f52fc1,0x71da23a7),LL(0xf790aca9,0x26957d5b),L_(0xe3199eb7)}, {LL(0x571af638,0x3a3ff68a),LL(0xa6e9167d,0xb02eef0e),LL(0x78555759,0x1609b893),L_(0x1507c2ea)}},
   {{LL(0xa99f1707,0x7a4aae59),LL(0x4be04c26,0xc84493c1),LL(0xf1b134ce,0x8f2fb8ff),L_(0xc7137a45)}, {LL(0x8f6f7a80,0x73cd2591),LL(0x3b2e141f,0xf68459a2),LL(0xf4b9ddab,0x1d8a7278),L_(0x403dae23)}},
   {{LL(0xa1fd53d2,0x3ad117de),LL(0x47d15513,0x75e31939),LL(0x58a5dfbd,0x3251c208),L_(0x66a06b3d)}, {LL(0x8436c37d,0x65ab378a),LL(0x5a3fb858,0x3a685733),LL(0xe6172e78,0xfa11b001),L_(0x25e5c930)}},
   {{LL(0x18e18997,0x48c2dc2c),LL(0x900eb7f7,0x814d21d9),LL(0x198afe84,0xf76ecf91),L_(0xab542184)}, {LL(0xb918572f,0xf53020dd),LL(0x31471cf4,0x365cb3df),LL(0xa6a9ebf4,0xf8ad1ad6),L_(0xf3563966)}},
   {{LL(0xf1859231,0xcb1be872),LL(0x1ad60df0,0xfaa10d6a),LL(0x11273c1d,0x240336fa),L_(0x4a1692f1)}, {LL(0x6bfe9509,0x88653660),LL(0xf81505d5,0x7b0d301b),LL(0xd427be59,0xbd85a9ef),L_(0x1a92bc3d)}},
   {{LL(0x72600cd8,0xe9cae1c4),LL(0xab08a5cc,0x3d1457c8),LL(0x96865fe9,0x9a1bc1e7),L_(0x438f5a09)}, {LL(0x7727a4f6,0x09a2a9db),LL(0x33e95a6f,0x028d0a98),LL(0x9656e0c0,0x7f3b01db),L_(0xeae74794)}},
   {{LL(0xb1f41609,0x060c3752),LL(0xc0ad59f7,0x6efb7250),LL(0x0bd8bd31,0x79016736),L_(0x34d2ef7f)}, {LL(0x8739f62d,0xf14247f9),LL(0x7675377d,0x29b78fc9),LL(0x5b2e893a,0x5f2b1e7f),L_(0x3c1437bc)}},
   {{LL(0x380ddade,0x014c7c14),LL(0xe37d7ead,0xc2d7837a),LL(0x78897ca3,0x5567dd34),L_(0xfa7b8f4b)}, {LL(0x00642d56,0xf735f752),LL(0x7fd93e92,0x8afd8b87),LL(0x6eb02b4e,0x120f5319),L_(0x1ccf74f8)}},
   {{LL(0x8fc31699,0x41d511e3),LL(0xb78232db,0xbdc4fb64),LL(0x9841bbd5,0x207bfaea),L_(0xd0f130c1)}, {LL(0x38fcb8da,0x6ba5dcee),LL(0x93e47594,0x21f2d335),LL(0xf78fd1c0,0xc6f634fd),L_(0x717d258b)}},
   {{LL(0x09a676cb,0x414f2daa),LL(0xa55cfcad,0x4cbc9d7a),LL(0x3b973302,0xb409e95d),L_(0x7da7223a)}, {LL(0xd16a5ca9,0xf92a6554),LL(0xb7642b9f,0x153e5024),LL(0xa08470dd,0xae4dfb2b),L_(0x8fd81632)}},
   {{LL(0x5b3c44c4,0x45031642),LL(0xc6ae8b05,0x335c400a),LL(0xd4b606ca,0x6b34a475),L_(0x4a823453)}, {LL(0xf3ab94ff,0x746295e2),LL(0x25948202,0x4b9b612a),LL(0x9d957566,0x510e67f8),L_(0x42ddbb08)}},
   {{LL(0xdc56f265,0x9f7dfe87),LL(0x20005269,0xe3386448),LL(0x104ea5b5,0xac4c4feb),L_(0x6565a54e)}, {LL(0xb2f8650d,0x186e3514),LL(0x50e692b2,0x58c2701b),LL(0x81927ca6,0x81220e7e),L_(0xa69b1829)}},
   {{LL(0x727a95ea,0xfb0e9702),LL(0xf0b23c9f,0xcca5f38b),LL(0xdb71dd75,0xad31c5dd),L_(0x93004978)}, {LL(0x3ff02652,0x683b0e57),LL(0x2b3b008d,0xc6e32a5b),LL(0x4ab25a51,0x283f183b),L_(0x0cc76852)}},
   {{LL(0x08ea3768,0xf7f76eae),LL(0xbcdd82ad,0x222dd486),LL(0x5b8927a0,0xdc4a955f),L_(0xb76f9687)}, {LL(0x605b6b78,0xc80579f6),LL(0xb80f7796,0x4cf36f45),LL(0x94402fad,0x0a226060),L_(0x0c94e0f7)}},
   {{LL(0x77db1b5e,0x4fe36153),LL(0xb19d463f,0xb32ea8ef),LL(0xa09373a7,0xd4f68719),L_(0x1f1869b7)}, {LL(0xd8828434,0x8c0026e9),LL(0x8f8f9197,0x16af4c49),LL(0x8f7a154b,0xd4672394),L_(0xee9c7e14)}},
   {{LL(0xe432397f,0xd0196172),LL(0x035a30a5,0x114f01ee),LL(0x12143827,0x2f06e8aa),L_(0xcd20f526)}, {LL(0x00bcfcbd,0x76ee873b),LL(0x542f1d14,0x8ff02da4),LL(0x786da906,0x7e7e6c65),L_(0x65c3eef1)}},
   {{LL(0xc8866c5c,0x0d7b5118),LL(0x5cd6ca19,0xef5be83d),LL(0xb37c41ce,0x4b76c64d),L_(0xf34bee15)}, {LL(0x3d28da12,0x8ef7cb6d),LL(0xf55195fb,0xa3c8a2d4),LL(0x4369465e,0xc8c85ab8),L_(0xe811f166)}},
   {{LL(0x08818979,0x17f1f556),LL(0xbf0f506b,0x251cc269),LL(0x686c57b8,0x865fb805),L_(0xa325b2ac)}, {LL(0x19c7f60f,0x23566eda),LL(0x40af0c7c,0x47f95f9f),LL(0x867b619f,0x1958dbaa),L_(0x164ce142)}},
   {{LL(0xfc7b0693,0x44b01395),LL(0xedbed37e,0x0d676d86),LL(0x075607b2,0xe2ae6225),L_(0x9bd85253)}, {LL(0x56d68f4b,0x7e636075),LL(0xb247649e,0x1ffba2be),LL(0x5c1ff02f,0xc237448d),L_(0x063c33cf)}},
   {{LL(0xc3ea5576,0x52c5f87e),LL(0x2fe94caa,0xf135c024),LL(0x593535ee,0xe0965b4b),L_(0xf93cf6d1)}, {LL(0x54fedc18,0x28d4b11a),LL(0x6c77585d,0xe5297d99),LL(0xb0d741c2,0x39caed90),L_(0xcc98dae9)}},
   {{LL(0x340fdc0b,0x1b8201fc),LL(0x44198c3e,0x248c333c),LL(0x5f236490,0x12ee4946),L_(0x0789670e)}, {LL(0xcbc6b050,0xee23385a),LL(0x3c3d5d85,0x630f7175),LL(0x0000016a,0x3f6e71ce),L_(0xad6ec3de)}},
   {{LL(0xc6c3de31,0xf99b211a),LL(0xca35fe5b,0x549ffc41),LL(0x5ffb8aa4,0xbc4e1e52),L_(0x6ac77d41)}, {LL(0xe552f85a,0xf54af34b),LL(0x6e1ccff3,0xd61618d2),LL(0x5bb20166,0x5776aea6),L_(0x7336f961)}},
   {{LL(0x535ea11e,0x565a5471),LL(0x8f3dcba8,0x4200ddee),LL(0x8ee2aacd,0x2493fe70),L_(0xab90e06d)}, {LL(0x048f6037,0xc2e41220),LL(0x075b150b,0x3193a2ea),LL(0x3c56b2e4,0xc4609ea4),L_(0x2fdd776c)}},
   {{LL(0x4276f086,0x57d5dc17),LL(0x2aa19794,0xa8a6408a),LL(0xd70d74f7,0x98acac12),L_(0x351c3c6f)}, {LL(0x93cd39aa,0xb4c10685),LL(0x5a2bba35,0x56a7276e),LL(0x2685e5f9,0x3eeed659),L_(0x8a344719)}},
   {{LL(0x2d701e0e,0x22e61da3),LL(0xbff21920,0x4511147f),LL(0x403fd833,0x6080d968),L_(0xb3b3779a)}, {LL(0x2957083b,0x38eb9ccb),LL(0x2ed5dc9b,0x1be2ace4),LL(0x726301f8,0x002d8909),L_(0xbe322533)}},
   {{LL(0xd1f9235a,0xf591d929),LL(0x0184d742,0xdc207a6c),LL(0x5e691f61,0x20b8f8e8),L_(0x2095a193)}, {LL(0xaeb5ac0a,0x7eacabb9),LL(0x3b17347c,0x0985df07),LL(0x04b8e782,0xf8047be0),L_(0x5ad11ec9)}},
   {{LL(0x8492830a,0x4f347f1e),LL(0x841e8dca,0x7d0fc13f),LL(0xf365c6c6,0xa0e662a2),L_(0xded71c8e)}, {LL(0x8d246f45,0x9dcc1860),LL(0x852d364c,0x5bfa3ba2),LL(0xd9a27ded,0xb281e327),L_(0x80b9b9f1)}},
   {{LL(0x6bab642c,0xbff128cd),LL(0x692c67d2,0x268f3893),LL(0xfc73c613,0xcd24728c),L_(0xa40f87b1)}, {LL(0xd40e3e5a,0xf065f5d7),LL(0x96f653a5,0x15d0cf49),LL(0x4ad712bb,0x70e00818),L_(0x2102b8b4)}},
   {{LL(0xf1c993cb,0xa2f0722e),LL(0x214a8d8b,0xe7c142ac),LL(0x8af34ce1,0xce47aba3),L_(0x2d01c664)}, {LL(0xb6c1f1ff,0x49dac9d9),LL(0x8e33c0f4,0x1baa4d4c),LL(0xdbfb3b98,0xd927e1a8),L_(0x80f00079)}},
   {{LL(0xde4413ab,0xad27b0b5),LL(0xa2997bb5,0x0077242e),LL(0x5d3ac419,0xe50d3185),L_(0x185f649f)}, {LL(0x6b5c4eea,0x4a8b1fad),LL(0x210cc7de,0x30ac4e43),LL(0xe0253dce,0xf07461c4),L_(0x4bf807d6)}},
   {{LL(0x01947899,0x818e99b8),LL(0x9008b9c3,0x05ed8f7b),LL(0x09f78f1a,0x2fb78a6e),L_(0xa5da0ebd)}, {LL(0x4c2f0cce,0xfdd48d7b),LL(0x79765db8,0xe380dbfc),LL(0x9600539c,0xbb3a3bc6),L_(0x3627d4d4)}},
   {{LL(0xea2bbe08,0x394029b6),LL(0x7ac327d1,0xc8ab0424),LL(0x90339a91,0xda368bdd),L_(0xab696429)}, {LL(0x0fe39b21,0xc20f1366),LL(0x32fec057,0x8075ba9a),LL(0xdc668935,0xa20bdd31),L_(0x2abf3c1d)}},
   {{LL(0x513bd424,0x02a9ea32),LL(0x7d6a8a11,0xe477baa6),LL(0x52b0bc16,0x4583bc9a),L_(0xba81bd4a)}, {LL(0x287b6485,0xf11b84b0),LL(0xae92c68c,0x7925b118),LL(0x4ab367b2,0x8ff3e481),L_(0x33179660)}},
   {{LL(0xb6f13800,0x0ffc4673),LL(0x56722205,0xf9b0dfda),LL(0x2bd4103b,0x56acfc3b),L_(0x1ce74c3f)}, {LL(0xa87de434,0x046c39cd),LL(0x41e5b749,0xdebd1720),LL(0x11fa933d,0x3f6c6fec),L_(0x275b834e)}},
   {{LL(0xae4601fd,0x06034734),LL(0x58ee806e,0x104d85dc),LL(0xca3c77a5,0x8bca0d2e),L_(0xb812f136)}, {LL(0xc78c356f,0x70ae3402),LL(0x7dc1f03f,0x322e72ad),LL(0x07c1fe3f,0xd0253c44),L_(0x511d1058)}},
   {{LL(0xc2f400a7,0x2291a21d),LL(0x993f820b,0x287eac12),LL(0x1680907d,0x03a00f86),L_(0xd919412a)}, {LL(0xbadddbc3,0xc5e15b28),LL(0x0ba18a53,0xf30664e1),LL(0xb1606671,0xc87b6603),L_(0xb55b207a)}},
   {{LL(0x1873a8e5,0x5f55c7e2),LL(0x6eaa78ce,0x33f083bf),LL(0x95c9fed8,0xe416fc1c),L_(0x7e0f68c5)}, {LL(0x02778f82,0x07eef5cc),LL(0xdb2aa0a1,0xa94149df),LL(0xc8c44cfd,0x2d270d3f),L_(0xec06f559)}},
   {{LL(0x7ca9f958,0xdb514ec8),LL(0xd0ace135,0x9718dd09),LL(0xe2bf5a99,0x325fe3bd),L_(0xe5c7ec48)}, {LL(0x4f0b0621,0x78990d74),LL(0x76fa9487,0x5e70b57d),LL(0xdfbda68c,0x6e9d11e2),L_(0xda96d406)}},
   {{LL(0xd61361bb,0xf97cf589),LL(0xc96cb924,0x7b21af76),LL(0xf64ebc1a,0x03a72d66),L_(0xecbebd56)}, {LL(0x4b021c91,0x39df1cca),LL(0x9863869e,0xb84c6c05),LL(0x9b6a3082,0x62692bf5),L_(0x977cbcb4)}},
   {{LL(0x94bf9bf5,0x79c2f57b),LL(0xb9be753d,0xe0d00e09),LL(0x49f82bc9,0xb9c9a088),L_(0xcc0f3b6b)}, {LL(0x1f589bd1,0x11cd1677),LL(0xe709a25f,0x8f952cff),LL(0xb5eec978,0x016b826e),L_(0x3188c1dc)}},
   {{LL(0x8cd03f0b,0xf4d9e042),LL(0x2b04b26f,0xb07e4630),LL(0x361c9252,0x5bbca5ce),L_(0x783f801a)}, {LL(0xe687c320,0x7bd9a485),LL(0x465590b5,0xe200e18a),LL(0x0e254cdc,0x50dce2e6),L_(0x49ff7ac4)}},
   {{LL(0x4b5e8399,0x284e641b),LL(0x33c76e5a,0x0556fa64),LL(0xa75c4fae,0x4cc4b972),L_(0xc36d03ba)}, {LL(0x90fa7978,0x6a40b04c),LL(0x4fef28c9,0x97401c28),LL(0x613bfe61,0x443845a0),L_(0x5144c164)}},
   {{LL(0x532b2d68,0x482b7779),LL(0xe955f441,0x123aa16d),LL(0x6886e7e9,0x9147c317),L_(0x1d0658eb)}, {LL(0x1530f2af,0xaed04ac0),LL(0x192c5796,0xa4111d16),LL(0xd86157ff,0x7406594c),L_(0xbf39202c)}},
   {{LL(0xafae6c6f,0xed33d4b3),LL(0xc5df16a7,0x008ec0c8),LL(0xfe186e4e,0x7e4ed374),L_(0xdca675ee)}, {LL(0x2534f6e8,0x798a9b30),LL(0x82a138f5,0xfcf997cc),LL(0xb58b2b67,0x2f620d55),L_(0x634f7d88)}},
   {{LL(0xe8013df8,0x5dcfcfb0),LL(0x7b9259dd,0x7a765b72),LL(0xe0fbd7bb,0x4e7374be),L_(0xc2b9a098)}, {LL(0xfe9b1e78,0x375f54f1),LL(0x0f6c804f,0x55dc463e),LL(0x202fbb6d,0x8d697af5),L_(0xe7949513)}},
},
/* digit=13 base_pwr=2^91 */
{
   {{LL(0xbcdbe445,0xcb4a833d),LL(0x57ca0395,0xb07c1b51),LL(0xef7d725e,0xdc3b82a8),L_(0x1e9a35e2)}, {LL(0xaa5a7ac9,0x01e0c54e),LL(0xc707131f,0x2b2d5196),LL(0xd220aca6,0x736d91c9),L_(0xde0d6eba)}},
   {{LL(0x74d6d0bb,0x9af89e67),LL(0xfa0e831e,0xada42e0d),LL(0xda49ba83,0x83b9c4ec),L_(0x782ab495)}, {LL(0x12dfdcae,0x7e0a2cb4),LL(0x4c92a243,0xf7e3a9a3),LL(0x6ed0d332,0x1f8c692d),L_(0x5cfab000)}},
   {{LL(0x634cef8b,0xff5ab01d),LL(0xa43c6f4e,0x6fe3ecd8),LL(0xc1015db1,0xfcc7ca4c),L_(0x9199e5e3)}, {LL(0x795d4927,0xed944d52),LL(0xe4551e73,0xf5f41299),LL(0x8f02bb95,0x4007343b),L_(0x896481e0)}},
   {{LL(0xab703650,0x6b243c97),LL(0x660da927,0x8b9a0873),LL(0x0ef0ffd5,0xb6e8c899),L_(0x8a32e4b8)}, {LL(0x2e1c11d9,0x154379bf),LL(0x2d0a9ded,0x4382cb75),LL(0xc876e0d3,0xc42ec7d1),L_(0x41c57824)}},
   {{LL(0x1ce64f94,0xa2eb2c00),LL(0x1eb0301b,0xe1a22445),LL(0x450ed02e,0x4205e69b),L_(0xf1b1c831)}, {LL(0xc76dda33,0x62db01d8),LL(0x168f45ba,0x633bdf3e),LL(0x95c86470,0xf9a6b309),L_(0x9bb55ae5)}},
   {{LL(0x71aba88f,0x590711c2),LL(0xdea8afde,0x0b6ab427),LL(0x87bfe939,0x1d7e1090),L_(0x54be137d)}, {LL(0x7ebcdd75,0x2a937fd4),LL(0x5fed0f5c,0x3da3197d),LL(0xc4155a57,0xf1cea938),L_(0xe2bb2c19)}},
   {{LL(0x6a4807e4,0xc0c52eb7),LL(0xfe9c4a62,0x12ba26fb),LL(0xda7a7914,0x001d4756),L_(0x48b356bb)}, {LL(0xce833468,0x3290dbb2),LL(0x432f5193,0x2b0a3429),LL(0x579c91eb,0xc27c2657),L_(0xcb222ca0)}},
   {{LL(0x6983b44c,0xf27d3e69),LL(0xf7665b89,0xe4a043b5),LL(0x5a97c010,0xc8f8a3c7),L_(0x4ed639e6)}, {LL(0xb9a3137d,0x5655ed33),LL(0x6987fe61,0x0b6dca4a),LL(0xc1a3c9bb,0x630b3026),L_(0x9305872d)}},
   {{LL(0x082a09e8,0xffe89fc6),LL(0x673fe488,0x9f4f85f6),LL(0x0b13732a,0xba6e1981),L_(0x1f59e6b4)}, {LL(0x26eb2b18,0xa4afe844),LL(0xaddd3cd6,0x91a5d7b7),LL(0xeeac1c21,0xe71f5fe0),L_(0x5d6e8430)}},
   {{LL(0x3f0bf1e4,0x031e908e),LL(0xecd2060c,0xbc16afc5),LL(0x6e2d4301,0x90a23e58),L_(0xd061f3f4)}, {LL(0x2549fd21,0x52f4ef15),LL(0x4103c527,0x5b4c8dd8),LL(0x1a5fbbc3,0x7457c84e),L_(0x6620daf1)}},
   {{LL(0xa3742f5a,0xe0a87786),LL(0x20460a16,0xe7aa9e66),LL(0xba62daa3,0x65327959),L_(0x4241a339)}, {LL(0x7da31b21,0x7fbf3e8d),LL(0x352aab9b,0xe1a7865a),LL(0x877c379e,0xea576b89),L_(0x5f93fdc5)}},
   {{LL(0xaa616a3a,0x662a3f35),LL(0x94cdd0da,0xdd7d18e3),LL(0xee30f511,0xf25f9c46),L_(0x35fff2a0)}, {LL(0x3e51e653,0x81e804d2),LL(0x5957ab11,0xb0aaaab1),LL(0xa52da544,0x89ee1376),L_(0x8411811a)}},
   {{LL(0x97063083,0xf2d42cdd),LL(0xc2eb7ee5,0x9cb9423b),LL(0x6ed009de,0x746461d4),L_(0xc182c461)}, {LL(0x6b0d52f8,0xe78fe609),LL(0xde14941d,0x6887a9cc),LL(0x8730c3cd,0x0f586a94),L_(0x2cf25d5d)}},
   {{LL(0xf21a3a3a,0x38aa8cf8),LL(0xd96cec97,0x3dd4e3e4),LL(0xf1f81f18,0xee849585),L_(0x6ec8f23e)}, {LL(0xb30df029,0x1e4514ba),LL(0xaf980c70,0x757e6771),LL(0x3b2305c3,0xa3525eaa),L_(0x755d2034)}},
   {{LL(0xb94b1aa2,0xebe6c1cc),LL(0xb228a915,0x1c4d6e36),LL(0xfa33270b,0x5d3856a1),L_(0xf1ba0e3a)}, {LL(0x77b4e048,0x63f53943),LL(0x99a1ff3b,0x7466b59d),LL(0xaea29806,0xb1050d57),L_(0x9d139cfa)}},
   {{LL(0x0fd330ad,0x0f394d0b),LL(0x27d1a20e,0xe4d013eb),LL(0x99b62540,0xaa1c17a9),L_(0xb03afc54)}, {LL(0x8f6f2743,0x7366a618),LL(0x449f676a,0x88e9ae3d),LL(0x8818aa14,0xd2fe924a),L_(0xf5d1f438)}},
   {{LL(0xfd68c56f,0xd02cd00e),LL(0x94c49868,0x5f260a20),LL(0x9c693311,0x7c1f3266),L_(0xe30a0258)}, {LL(0x5b2551ac,0x9f545dde),LL(0xbdeac1ae,0x19f87afd),LL(0x019bf350,0x58397798),L_(0x787035be)}},
   {{LL(0x46ea826b,0xc4da4263),LL(0x10febee0,0x43744264),LL(0xbfc3581d,0xcc933457),L_(0xd3126394)}, {LL(0x9cbecb37,0xe0071ca1),LL(0x715c9dad,0xcf0cfc02),LL(0x67a2c9e1,0x2f9bd24a),L_(0x47f400b4)}},
   {{LL(0xa57e540f,0xd472cc15),LL(0x1ab19257,0x0b60e719),LL(0xcd92d76f,0xca1c9c30),L_(0xb75f7cec)}, {LL(0x8103a773,0xa305744b),LL(0x34bb833a,0x54b38b68),LL(0xd6e1cdb8,0xb252c82f),L_(0xbbc88ea0)}},
   {{LL(0x935833d7,0xd0e4995f),LL(0x44ed9975,0x87f60eef),LL(0x933733ca,0x379092ba),L_(0x0e167f70)}, {LL(0xd4945af6,0x60eeff1a),LL(0x589a5723,0x1d6d1def),LL(0x4bee290d,0xe87f42f5),L_(0xbdc1579e)}},
   {{LL(0xb222f294,0x080ba4e8),LL(0x83c59be1,0x8ed3f3c1),LL(0xfdee70ba,0x58ce1c82),L_(0x309979e1)}, {LL(0x8490a253,0x6ed1e744),LL(0xc3fded2d,0x0602a508),LL(0x585e149e,0x7fa41773),L_(0xb933ab59)}},
   {{LL(0xf0a3e292,0x74dc9c31),LL(0x0f2c66c5,0xc78baacd),LL(0x01ba1bb0,0xe5b73558),L_(0x328bebc7)}, {LL(0x9cf50576,0x5421bd61),LL(0x76efecbb,0x241cb4c6),LL(0xc421cfb2,0x9ba297a8),L_(0xe31b7671)}},
   {{LL(0xf0b66ab8,0xd87b31c9),LL(0x3aacb600,0xcf835fb0),LL(0x2f9c67c3,0xaee5e6cf),L_(0x6453b1bf)}, {LL(0xd04298de,0x0411e445),LL(0x0d41aa91,0xbec90b82),LL(0xe6adb2bb,0xb30b9c0e),L_(0xdd7ea812)}},
   {{LL(0xda605906,0x5f01a30b),LL(0xe7f26ffc,0x7c053c40),LL(0x40f7d79d,0xc39feb03),L_(0x232940c5)}, {LL(0x3d73c35a,0xe472e81d),LL(0xe3395fd8,0xdea2bf33),LL(0xed66fd6f,0xddc57518),L_(0x675b9966)}},
   {{LL(0x7095dc92,0x07f66d27),LL(0xb47087dd,0x9c4ec651),LL(0xfd209ae0,0x35803afd),L_(0x6fb37653)}, {LL(0x51ed5eaf,0xc9356091),LL(0x47c63c1c,0x19e52bee),LL(0x0676dac6,0xd87e6c27),L_(0x6badd30e)}},
   {{LL(0x83877c71,0x39e61abc),LL(0x58d7a9d9,0xa88e05f0),LL(0x7885ad28,0x53bfba53),L_(0xb10f5fdb)}, {LL(0x8e433b9f,0xd6a78477),LL(0x0f088cf6,0xcefdc6a1),LL(0x3ce9d748,0x27d9ed71),L_(0x648c4c3c)}},
   {{LL(0xe57aeaa9,0x7dbbbc39),LL(0xe13d51e3,0x5c942e4d),LL(0x7d7aed19,0xe660d0ba),L_(0x25bcdf2d)}, {LL(0xff40dd8a,0xcdf314b5),LL(0x5faa85f3,0xa625d183),LL(0x9d4344c2,0x54f2035c),L_(0x5243fd7f)}},
   {{LL(0xd120f88e,0xe3275570),LL(0x79d8c8b5,0xe344b66c),LL(0x8d5e8c7f,0x1cdb23ae),L_(0x8c372c12)}, {LL(0xce526de3,0xe64e6355),LL(0x6f104a11,0x5edf30db),LL(0x7e00559a,0x3501ee2b),L_(0xdc58f8f4)}},
   {{LL(0xa6698b0d,0x91c3e912),LL(0x65e168bf,0xbb24391d),LL(0xe1e797f6,0x0526a13b),L_(0x6c3a27f6)}, {LL(0x41e4a8d4,0x63a82271),LL(0x759095e6,0xf6fd3e09),LL(0xd1ade148,0x85041faa),L_(0x1d102e98)}},
   {{LL(0x7675921b,0xb30c5e67),LL(0x781de03f,0x9a004497),LL(0xb5aac131,0xcaf13420),L_(0x2a74de9b)}, {LL(0xced86c6a,0xc60fb50a),LL(0x4039952c,0x8cd4dd3d),LL(0xdec98657,0x5092d4ea),L_(0x573afadf)}},
   {{LL(0x9fb10f98,0x67b904af),LL(0x7233bca1,0xeebc687c),LL(0xdae52182,0x13aadd13),L_(0x7d61309e)}, {LL(0x59a01f8b,0xc7db8080),LL(0xf6d5d889,0x5433ce07),LL(0x8b26a71a,0x0c3a71bb),L_(0x69b8c725)}},
   {{LL(0xd29e534f,0x215aa2f4),LL(0x7bb1ac35,0x5f00b26b),LL(0x3c54a3cb,0x71ec2585),L_(0xf6cd94c0)}, {LL(0x4c553b93,0xf86c23e2),LL(0x1b7d9626,0xdd6cfc3d),LL(0x0fb40ab0,0x4647a750),L_(0xeb4bdb5e)}},
   {{LL(0xa51a2b17,0x1bbce0ea),LL(0x6f2f98f1,0x7e78ccf9),LL(0xc6736591,0xbaa1d2e3),L_(0x47e735c5)}, {LL(0x6b8866aa,0xc5b9e6fb),LL(0x8aef23dc,0xc39a4975),LL(0x712336e6,0x86d2dbfa),L_(0x19715594)}},
   {{LL(0x106db862,0xc495534c),LL(0x83770bdf,0x75c25358),LL(0x05a076ec,0xf2a1fa9b),L_(0xc80d0362)}, {LL(0xbab703f6,0x2b054aeb),LL(0xb0c6f6de,0x9e9ed0de),LL(0x42646b39,0xd95a6334),L_(0xddad98a9)}},
   {{LL(0xbcf088fe,0x373df018),LL(0x7bd97cea,0x2646d8cd),LL(0xa20f1b7c,0x3bc4db1d),L_(0xdd201f09)}, {LL(0xf3f061d1,0x7fc330d4),LL(0x5f1ec56b,0xd64fa0dc),LL(0x93be4afe,0x239a905a),L_(0x38df95b1)}},
   {{LL(0x9491ca55,0x3a2368e4),LL(0x4fc84a15,0x840339bb),LL(0x60229ac3,0x0f74bf45),L_(0xb9a147d0)}, {LL(0x6bb23f64,0x6df7548f),LL(0x928eccc2,0xbd4ae1ae),LL(0xa5c7651d,0x85fa24af),L_(0xe98fba58)}},
   {{LL(0xd9ea606f,0x72e722f7),LL(0xde7a39d8,0x02dae11e),LL(0x6f289df9,0x4f148db1),L_(0x3efce901)}, {LL(0xe8df6027,0x0dc6bfc0),LL(0x25925285,0xc2c848fa),LL(0x726e5bf2,0x0bc8372a),L_(0x4993590e)}},
   {{LL(0xe513cc61,0xb368f5d3),LL(0x1b56d61c,0x2acdbaae),LL(0x077919ee,0x69129848),L_(0xd6bce458)}, {LL(0x5c03af06,0x6892cac4),LL(0xbf732964,0x13156875),LL(0x86df2c46,0x15d63bed),L_(0xc24215d0)}},
   {{LL(0x85ef7366,0x7ea45182),LL(0x4f6a510a,0x7d4f7ca5),LL(0x2e79b2d0,0x5befddd6),L_(0xdfce6391)}, {LL(0xeb6e8178,0xd7c05293),LL(0xbb2671e0,0xfebc1ad3),LL(0xc2ef9919,0x5ef848b7),L_(0x3034a5d7)}},
   {{LL(0xec0fd6a3,0x4834ba2e),LL(0xd4612c03,0x7006255f),LL(0xee7cbb5c,0xa240c28a),L_(0x9d74ecc1)}, {LL(0xe8bfcc1e,0x36f2363a),LL(0xe755acbe,0x2c7fdeac),LL(0xee2e792a,0xfe7f2c9f),L_(0xf3a0b1ef)}},
   {{LL(0x21f1b866,0x7a207076),LL(0xb0caba72,0xa289580c),LL(0x26c6a4fb,0xac8c38be),L_(0x46fba43d)}, {LL(0x603a204d,0xc96f5b0f),LL(0xaae0bc34,0x07974abf),LL(0xdd93d467,0xd384c831),L_(0xd3e228d7)}},
   {{LL(0x130cf278,0xb2d57bad),LL(0xa4b0a795,0x2e92cb97),LL(0x0b8251c6,0x04325b28),L_(0xd825f27e)}, {LL(0x630e8e9e,0xcda5985f),LL(0x1bba4e62,0xf6775b8c),LL(0x9cc53c03,0x466530c7),L_(0xca08b958)}},
   {{LL(0x9b3d46a1,0x6911dd6e),LL(0x64d4f9aa,0xf5c72cad),LL(0xaf7d9d93,0x53a3ea3c),L_(0x99f4337b)}, {LL(0xd879f3e2,0xeaba3d43),LL(0x7bd203ea,0x4c921337),LL(0x3b59edc0,0x7ee8ea79),L_(0xdbb3851a)}},
   {{LL(0x64c557a1,0x70b7979e),LL(0xb3681c37,0x482d3b99),LL(0x08f6017c,0xecf693f0),L_(0xf2fe6d70)}, {LL(0xe8816978,0x7c8670c4),LL(0x8a10a4b5,0x05b8eed1),LL(0xa8713371,0x0ebddf9f),L_(0x6f3c3b96)}},
   {{LL(0x50dc1443,0x884f22ec),LL(0xf9400068,0x5f885c0a),LL(0x4da67f62,0x6b4ecc34),L_(0x170f203f)}, {LL(0x0181b70d,0x3f227a67),LL(0x844eab79,0xafb114cc),LL(0x5b40835b,0x2f5bb436),L_(0x7b9f2c6f)}},
   {{LL(0xdaec76bc,0x6deef6a5),LL(0x338dc21a,0xe70560f2),LL(0x8e70fab8,0xdf0c450d),L_(0xd256f11d)}, {LL(0x916202a8,0x05568dde),LL(0xfb8b1d84,0x89f0919e),LL(0xb5670da6,0x97387cce),L_(0x8943fa2b)}},
   {{LL(0xc269a34a,0xae4d60ad),LL(0xf62da5da,0x27a5a54f),LL(0x560e5ad7,0xfd2940f8),L_(0xab3236ae)}, {LL(0x86446f6a,0x2f2ee978),LL(0xb56b7c8a,0xa481c719),LL(0xd1f9e2af,0x99fe0c80),L_(0xdc1ec6c4)}},
   {{LL(0x4c354b32,0x9d2731ca),LL(0x45db9456,0x9583d77a),LL(0x549fab4d,0x20743202),L_(0x133b9227)}, {LL(0xc9afa8d6,0x6ee70bf6),LL(0x61e414f2,0x71181b55),LL(0x34dfc171,0x186cb701),L_(0x05ba8622)}},
   {{LL(0x784904dd,0xef91aeb1),LL(0x2ce39834,0x466ca745),LL(0x9dba0e94,0x3efbbed0),L_(0x6d055926)}, {LL(0x818f1161,0x238404b3),LL(0xb2aba397,0x7f378101),LL(0x9026e5ef,0xd9ddeb02),L_(0x639d63b6)}},
   {{LL(0x50f516e3,0x359a1ffe),LL(0xc1a03316,0xea7c7fb8),LL(0x2be043ec,0xe26d37f4),L_(0x34fbd1da)}, {LL(0xd8d23c82,0x97f967f3),LL(0xf3027282,0xf97e68ce),LL(0x0cf39dbb,0x0f3321ca),L_(0xb0d6de7c)}},
   {{LL(0xd200dfe0,0x890af600),LL(0x905f6954,0x1e4962b1),LL(0xd80a6211,0x78562fa6),L_(0x0e1b429e)}, {LL(0xdb5a70a8,0x68abbf8d),LL(0xdf21b056,0xbffea5c7),LL(0x99114570,0x6e5308c1),L_(0xd35cddc0)}},
   {{LL(0xdfd77bdd,0x16448af4),LL(0x5bcbe7ef,0xcb49f269),LL(0x9f4c424c,0x7c95aa03),L_(0xb01ff8c8)}, {LL(0xd5948dcf,0x93acdc43),LL(0x2ddd8b66,0x7efb93e4),LL(0xa8e311a0,0xe08e348f),L_(0x7b4dacb7)}},
   {{LL(0x96c4f395,0xfa10af74),LL(0xd4e05b0b,0x489d01bb),LL(0x1a4d0c16,0x5457e489),L_(0x70983e98)}, {LL(0x1331b4da,0x84eb66f4),LL(0xf38823d2,0x8054cf79),LL(0xb191cf89,0x3e7eb0cb),L_(0xa37a2976)}},
   {{LL(0x8d0cc3cf,0xa6a534ca),LL(0x7bc55c38,0xa917d88e),LL(0xc01d22dc,0xba878e3c),L_(0x2cf82f89)}, {LL(0x927b4948,0xfb3f69e6),LL(0xfab01485,0x44185c63),LL(0x586c00a9,0x151ed75b),L_(0x923f3333)}},
   {{LL(0x72a46aaa,0x519ea031),LL(0x8709198a,0x75f6f740),LL(0x1a09f429,0x0df7057c),L_(0xd41f8134)}, {LL(0x0511f342,0xb8c291d5),LL(0xb5c05e96,0xbd814e25),LL(0xa548461b,0xdfcbbc5c),L_(0x17c27bbb)}},
   {{LL(0xf029f708,0x3ec594c8),LL(0xe348f36b,0xaacdcc44),LL(0x76cebcbf,0x6e7c6e83),L_(0xb7e69828)}, {LL(0xa2a8a4ff,0xb47c1c29),LL(0x994406d9,0xebdc1ae0),LL(0xed49a01f,0x63ac12b4),L_(0x513ab51d)}},
   {{LL(0xaa4b3b47,0x7e957ed9),LL(0xa7f34b22,0xe3393b0f),LL(0xde08b775,0x8522afee),L_(0xacd12226)}, {LL(0x5f73d8d2,0x39d26476),LL(0xa513c2b3,0x7e58e0f2),LL(0x213036c0,0xa0e99b0f),L_(0xfb5cf2f2)}},
   {{LL(0x8a626b92,0xbe1888a3),LL(0xbb87a44b,0xc627fd2d),LL(0xd55c88b6,0x75c0ce87),L_(0xe7f3660a)}, {LL(0x96806d02,0xdce56555),LL(0xc75c5771,0xa0f7c5e9),LL(0x286d7145,0xd7ff4a5f),L_(0xebc158a2)}},
   {{LL(0x9f5793c9,0xf0bdad25),LL(0xb6481975,0xde77b0dd),LL(0x308c67ea,0x506de99f),L_(0x814dd111)}, {LL(0x9d70efea,0xf8b6908d),LL(0x664f6334,0xa22f9208),LL(0xabb48b09,0x305250b9),L_(0x589eade6)}},
   {{LL(0x04d51ba4,0x69790978),LL(0x61258755,0x55c72c44),LL(0x340e8617,0xcd9b6116),L_(0xabcb82e4)}, {LL(0x80b0004b,0x8b257151),LL(0x337e5242,0xae7d889b),LL(0xd95f9a8b,0xf7035183),L_(0x7680c71b)}},
   {{LL(0x96f17da3,0x21e71bcf),LL(0x808dcdbd,0x1dde1b45),LL(0x3f515a1d,0x1f620fc3),L_(0x06bbadac)}, {LL(0xd2651946,0xa7f48789),LL(0x0f2c7fea,0xa9e9b7f9),LL(0x048d20b9,0x2efd58e7),L_(0x44fa3d36)}},
   {{LL(0x095a8f26,0x6fa2edc8),LL(0x653c44b2,0xad7dc081),LL(0xf2b8fbbb,0x88a7b80f),L_(0x02238eac)}, {LL(0x2ca3a2a6,0x00c09332),LL(0x7cf3c758,0x0d577a60),LL(0x54d150a9,0x1837cf45),L_(0x9354157c)}},
   {{LL(0xec967632,0x281b5c05),LL(0x5e849910,0x580362ba),LL(0x832277a8,0x702cc617),L_(0x1dbc415a)}, {LL(0x7335bdc9,0x26119f95),LL(0x90031be3,0xa27ab673),LL(0xc7055ffb,0x652580b5),L_(0xe35bf60a)}},
   {{LL(0x8c6adedb,0xebd0f498),LL(0xe1e7a7db,0x993107cf),LL(0xb75b4edf,0x0232f8ad),L_(0xc245bc0d)}, {LL(0xa2592946,0x48d03da4),LL(0xb9489c19,0xda871c00),LL(0x8ff4aaa7,0x746e818b),L_(0x8df35cfe)}},
},
/* digit=14 base_pwr=2^98 */
{
   {{LL(0x23800fac,0x8b12655a),LL(0xd7cd8acc,0x66a4179c),LL(0x33023ec9,0x1205108d),L_(0xc1d74575)}, {LL(0xa87c96ad,0x402e3b9a),LL(0x744ac2f7,0xd3dbe7de),LL(0x6b327511,0x5723ac5b),L_(0x244ec59a)}},
   {{LL(0x7ab235be,0xbede2315),LL(0x536bf0cf,0x5f9a6aaf),LL(0xe6d56464,0x357d2d21),L_(0x58f34a26)}, {LL(0x95a60b41,0x49f4f0e2),LL(0xcfbab99d,0xdca3ea61),LL(0x4c739342,0xa16d4457),L_(0xcea92902)}},
   {{LL(0xdcc5b89b,0x463327ef),LL(0xb7378179,0xadb9328a),LL(0xbfce0e4d,0x0d62b545),L_(0x1e6b72f2)}, {LL(0x349d4bd6,0xa76133fc),LL(0xdf69c044,0xb4284723),LL(0xf5a817aa,0xf7ca3909),L_(0xe22765b5)}},
   {{LL(0x0655039f,0xbc44db37),LL(0xc2621a65,0x3d4e983d),LL(0x67d89bbe,0x85435ace),L_(0x803ede1a)}, {LL(0xbe29cd2f,0xed39f9e8),LL(0x00ea11c0,0x55cac6ca),LL(0x3d4406d2,0xde2aeda4),L_(0xe7cb91b3)}},
   {{LL(0x8fabc04b,0xc39c7ee8),LL(0x26cf4402,0xc619fe58),LL(0x0816fbf5,0xf268e46c),L_(0x9482a940)}, {LL(0x3b1bba20,0xce226865),LL(0x080cefa6,0x4f9ab1d9),LL(0xfcbc42d6,0x13efb949),L_(0x1f067e43)}},
   {{LL(0xf8119c35,0x75d60569),LL(0x73a4c13b,0xb3a82994),LL(0x063e48b4,0xc7bc1aec),L_(0xb0ab7ce1)}, {LL(0xd626619f,0xa7f182b8),LL(0x4c843004,0x533f72eb),LL(0x1d2aa02e,0x69953a2d),L_(0x0de00959)}},
   {{LL(0x91516c9c,0x0913a322),LL(0x1648e50c,0xa0961a2f),LL(0xa7d10cb1,0x06e4869d),L_(0x9b4a803d)}, {LL(0xdc8793e7,0x9f17e3a2),LL(0x298436f6,0x06fc6243),LL(0xe925133a,0xb7622bbe),L_(0x70b9a2e8)}},
   {{LL(0x84e7489b,0xc981ba62),LL(0x5e1a4306,0x792db847),LL(0xa77e3ed1,0xf4b6792a),L_(0x87bb9d13)}, {LL(0xed208d23,0xfc7dc4a1),LL(0xcc352f71,0x87c2b023),LL(0x095b0ea7,0x88016bc4),L_(0xc90a59de)}},
   {{LL(0x59119397,0x4b8755e3),LL(0xfbec0058,0x19aa2513),LL(0x554f2e58,0x1d544a58),L_(0xb7908e98)}, {LL(0x1d4dba6f,0xcc8e5a35),LL(0x4e98597c,0x4036ed62),LL(0xf70ca123,0x0fd793e9),L_(0x5567a702)}},
   {{LL(0x269da085,0x2503b0ce),LL(0x06549f68,0xb3025472),LL(0xeb061c62,0xfe9ae248),L_(0xd6d45a68)}, {LL(0xd2906b25,0xeaa5d6f9),LL(0x58bbd291,0xaf62d50d),LL(0xc2b140a8,0x4dbbcce0),L_(0x86272e95)}},
   {{LL(0xa7cf5775,0x3293ff9c),LL(0x7b80a205,0x1f10127d),LL(0x739c1785,0xc01f9711),L_(0xa80be8ca)}, {LL(0x68df6a04,0x50fcfcf4),LL(0x3da76c49,0x340f3712),LL(0x53cced84,0xba988480),L_(0x89efc853)}},
   {{LL(0xed47d3b8,0x6051e486),LL(0x27be6887,0x2e29a4c3),LL(0xfb40723c,0x77353f89),L_(0x3a8109a1)}, {LL(0xf4eea8fe,0x672e091a),LL(0xaaf35dba,0xef34d20e),LL(0x96e8e707,0x2e5db0f7),L_(0x259349f1)}},
   {{LL(0x089e80e9,0x259bf71c),LL(0x6e48997b,0xcd52c81c),LL(0x9b67f891,0x106aa44c),L_(0x2b828bad)}, {LL(0x84cca320,0x492159ac),LL(0x860638da,0x99fc08c2),LL(0x9a577c2f,0xde194e1f),L_(0x4c02a837)}},
   {{LL(0x1a869e95,0xac2dd0d1),LL(0x40fe4710,0x0d98495d),LL(0x926fca45,0x230d5720),L_(0xc0032a5e)}, {LL(0xda97aa70,0xe0f36b43),LL(0xdbdd8384,0xbdee74a8),LL(0x9f4f1e97,0x616d2e77),L_(0x285c8908)}},
   {{LL(0x4ce190f7,0xe408ab06),LL(0x4de68c46,0xaec26bc8),LL(0xea864657,0xa9c6ad27),L_(0x3b9d1eb1)}, {LL(0xab862944,0x327c78b5),LL(0x2af9ede8,0x751ec6f1),LL(0x18bed81e,0x1c240eff),L_(0x17dc571f)}},
   {{LL(0x291c3273,0x766b4a0b),LL(0x3a97f292,0x6e56a096),LL(0xafb97a2d,0xe3cbbbe3),L_(0x3a6fadc9)}, {LL(0x2369de35,0x4c78c1e4),LL(0x6d3e5c1b,0x698e8f8f),LL(0x14e1a4eb,0xa87a74e8),L_(0xbb410055)}},
   {{LL(0x0fb37060,0xcbf75eae),LL(0x323f451f,0x58dee01f),LL(0x1a149a9e,0x67ff1a09),L_(0xb35d39d2)}, {LL(0xf63b9865,0x7e111af7),LL(0x02e5fd5f,0xf6f23814),LL(0xc6967a32,0x682916a3),L_(0x5d411958)}},
   {{LL(0x53aa5942,0x7dc26ca6),LL(0xec5ae21b,0x3a7f3858),LL(0x4c6d1720,0x87d2b98e),L_(0x6b3c9448)}, {LL(0x379e4e5d,0xc6bec0ca),LL(0x7de28c5b,0x69db9b01),LL(0x4e3aaa80,0x4f1a3e1f),L_(0x31e76223)}},
   {{LL(0x948cf979,0xe1a7d9ed),LL(0x32518804,0xabd060c2),LL(0xe2c8026f,0xd02845ba),L_(0x60dcd816)}, {LL(0xd2fcd20d,0x9fc8ca8b),LL(0x67ac75bd,0x0b403597),LL(0xfa897231,0xd2ef4cab),L_(0xba971f13)}},
   {{LL(0xc9608224,0xd3a9e728),LL(0x21de382a,0x9fcf28b0),LL(0x23db313b,0xe226f1f0),L_(0x9e002526)}, {LL(0xdd26c698,0xa3a7479f),LL(0x8170903e,0x4871cbd5),LL(0xc0ad648f,0x30414c9b),L_(0x3c9d51d0)}},
   {{LL(0xe22f8df5,0xdef38d08),LL(0xd456476e,0xfacc23b0),LL(0x7b3f5e97,0x00fb9a2e),L_(0x6c5a7311)}, {LL(0xa6cbdc37,0x87cb811c),LL(0x82204464,0x6214d9f5),LL(0x06806fd9,0x5ba8f23b),L_(0x6d93ebe3)}},
   {{LL(0x86e33c2d,0x23423298),LL(0xa44ec00f,0xf6e30434),LL(0xc242970e,0xbf53cf6b),L_(0x3a4d00db)}, {LL(0x84f61eea,0x8140c42b),LL(0x5afabd85,0xe917c409),LL(0x70d1d6e0,0x95de99fd),L_(0xc2091589)}},
   {{LL(0x396774db,0xc249bd59),LL(0x93024ffe,0x24f81111),LL(0xfe365e8c,0xdcd6e625),L_(0x35c6e777)}, {LL(0xe01d308d,0x65e9e9bf),LL(0x4a7452da,0x5cab99f6),LL(0x441a1645,0xf759c45d),L_(0x97ffca58)}},
   {{LL(0x1d7326e2,0x52fad307),LL(0x04705ea5,0x3aa389bc),LL(0x9cfd5023,0x3feb82b3),L_(0x7a752b31)}, {LL(0x4a98e745,0x22a198b9),LL(0x303fc591,0x5fd22f5c),LL(0x95dcd425,0xe40ca457),L_(0xfc4759f2)}},
   {{LL(0x0baec1d1,0x899726d2),LL(0x9ccc4295,0x9503a4a3),LL(0x8faff5a9,0x735bcc96),L_(0x4938831d)}, {LL(0x294b5dc9,0x27659e3e),LL(0xcf8e86f6,0xb11539c4),LL(0xf82de3fd,0xdafe9d6c),L_(0x93a3f5c4)}},
   {{LL(0xb712f886,0x4ba7d548),LL(0xf1d50673,0x627d0815),LL(0xebc722ce,0x4e21b679),L_(0xd4fcebfe)}, {LL(0x3d488792,0x654d615e),LL(0xa956f701,0x38e30e59),LL(0xbe0c2b4a,0xd07392c8),L_(0xd0eefb50)}},
   {{LL(0x91582e11,0x245a5f24),LL(0x83071bb8,0x1d74fe95),LL(0x32fc2d8c,0x8d68efa6),L_(0x3c6bb1ba)}, {LL(0xb73ed81b,0xd7c363e3),LL(0x4814d6e2,0x4b18aee3),LL(0x8563cd35,0xf296ee0a),L_(0x39c2c1ab)}},
   {{LL(0x6115b5df,0x12e07056),LL(0x27e6f810,0xab33f925),LL(0xdadb7f44,0x9611fc0f),L_(0x21632fa8)}, {LL(0x135db674,0x0014d0b2),LL(0x90282284,0x34df44bd),LL(0x498b5aab,0x3b9bf0a3),L_(0x83adadb8)}},
   {{LL(0x8b47b85a,0x7583945e),LL(0x8ca7a054,0xe1eb9f0d),LL(0xdc6bf5ac,0x7f50540d),L_(0x4929ffcf)}, {LL(0x7dcd0fa3,0xd6e884e8),LL(0x547fd35b,0x0fda3c41),LL(0x42d34849,0x156b5074),L_(0x2b73e553)}},
   {{LL(0x9db1951e,0x6f1aaef3),LL(0xbba949f2,0xe07d36c2),LL(0xaf67c45f,0x598114fb),L_(0x0be51018)}, {LL(0x18833890,0xb079f72e),LL(0x43a0dd65,0xee4b11df),LL(0x20bd4ca5,0x9dd3598b),L_(0xb6e5a89d)}},
   {{LL(0x2f515e07,0x0a130c64),LL(0x53781704,0x21d929ed),LL(0xed92766f,0x2370b592),L_(0xb01a78da)}, {LL(0x461ee105,0x11fd327c),LL(0xe5630841,0xaac6ed8a),LL(0xaad362f8,0xe2c21338),L_(0xd9bef88b)}},
   {{LL(0x7702c305,0x90cc96b4),LL(0x284ea1e9,0xec9108c9),LL(0xea844162,0x79d4d2e8),L_(0x5fc96bdc)}, {LL(0x9b799340,0x9b1c7dfd),LL(0x6ae02844,0xf8a798d5),LL(0x4a0a1e22,0x7439503e),L_(0x0838edf3)}},
   {{LL(0x25112357,0x41a91a95),LL(0x9794aee7,0x12be9cf2),LL(0xa09c00f4,0x87b213f6),L_(0x588e9419)}, {LL(0x3cc23f7d,0xa44cb6f8),LL(0x9ee73263,0xda068194),LL(0x999bf774,0xe19968dc),L_(0x4f9d93f1)}},
   {{LL(0xa3bbb5a7,0xe07280d1),LL(0xfdb5f5ba,0xf0df9657),LL(0xbd50dd87,0x4715fc8b),L_(0xdf379027)}, {LL(0xf3d02095,0x64a96f25),LL(0xeb119764,0x661822c2),LL(0x8b298e54,0xd8ec764d),L_(0x6f6ea251)}},
   {{LL(0xcef9767b,0x3dfab91a),LL(0x7c6e866c,0x85522367),LL(0xdbf3bb79,0x79b36fcb),L_(0xbeb97ad5)}, {LL(0x05fa18c2,0xdbafd1f2),LL(0x64259c17,0xdd549dff),LL(0x83c7657f,0xd82dc78f),L_(0x8cf18ed9)}},
   {{LL(0xad65298a,0x58b32f27),LL(0x2131395b,0x976af3d3),LL(0x4b505f2e,0x64c1cfc3),L_(0xca10d83b)}, {LL(0x59913cfe,0xbab37657),LL(0x1414d1f4,0xf066519c),LL(0x8887e569,0x3d975cff),L_(0x0c251f08)}},
   {{LL(0x425ddccc,0xd5368e78),LL(0x93068700,0x12b23c3e),LL(0x33076aa1,0x4e626b1c),L_(0x4e14ead6)}, {LL(0x23f1e78d,0xe86079cd),LL(0xd274538d,0x908738cd),LL(0x244ce9b9,0xe51d741d),L_(0xb4e734ed)}},
   {{LL(0xb2d63b64,0xf0e2cdd8),LL(0x0675f17a,0x90f8143d),LL(0x75829dd6,0x2d23a43f),L_(0x298a7f8f)}, {LL(0x47ca8668,0xa526104e),LL(0x4e9c496b,0x85467645),LL(0xb0f0e275,0xafc61eeb),L_(0xa1acd83f)}},
   {{LL(0x7ee79f81,0xf367484b),LL(0xafa3db78,0xbfc8c257),LL(0x7fe068cc,0x7c8262c2),L_(0x784b4654)}, {LL(0xc96069f1,0xd5c6e6e1),LL(0xb65beee4,0xd8f70ecb),LL(0x24073300,0xc55f13ed),L_(0xa8d09ea4)}},
   {{LL(0x82d22a52,0xe38ffddd),LL(0x7bb613d6,0x03df60ca),LL(0xa964fa9d,0x9bfb3890),L_(0xcc4b84d8)}, {LL(0x96707a64,0xf9f9338a),LL(0x110b10fd,0x5d9e9ac0),LL(0x0acd29e1,0x854609dd),L_(0x11a3136a)}},
   {{LL(0x84bcad3b,0x9a527f75),LL(0x28be328f,0xc16f294f),LL(0xdaf38a99,0xfdebcc4a),L_(0x4d614ff9)}, {LL(0x8a1d04d1,0x760244a9),LL(0x9fb5fa88,0x97af96d1),LL(0x79ecc48b,0xebf508a7),L_(0x63ac8146)}},
   {{LL(0xa9338a44,0x4ccbe898),LL(0xe952b759,0xf79ce658),LL(0xa7f6528b,0x0efdd8f6),L_(0x59bff1cf)}, {LL(0x7002f54b,0x859eefc9),LL(0xb915b857,0x3146204d),LL(0x25b282fd,0x9e570ac1),L_(0xf3c10eb4)}},
   {{LL(0x50227a06,0x96cc2182),LL(0xd7972029,0xed491383),LL(0xa5e9dc33,0xef480409),L_(0x8303786c)}, {LL(0x48fc5413,0x7b84b8b7),LL(0xa8f1bf3e,0x22c04cfa),LL(0xa2e99567,0x5b8dc942),L_(0x089858bd)}},
   {{LL(0xb80a1877,0x7a0670f5),LL(0xf5def779,0x99c0dcfe),LL(0xceba003e,0x737b1478),L_(0xba188761)}, {LL(0x88ff60f8,0xa8a91790),LL(0xf962acd7,0xd0c75bfc),LL(0x8ebfd0f8,0xaf779b34),L_(0xe9624be3)}},
   {{LL(0x62fec3e2,0xd969780a),LL(0x9daaebb5,0xafc171d8),LL(0x647f2bf8,0x16ea68f8),L_(0xfbfe21fa)}, {LL(0x83f12d03,0xb63e2588),LL(0xe8295dd0,0xb69cbad8),LL(0x08b7948f,0x1a740ee4),L_(0xbe240b54)}},
   {{LL(0xd3ad093e,0x97774b8e),LL(0xa107d5a6,0x1438ff5a),LL(0x90acd42f,0x0c55b2c6),L_(0xb1008301)}, {LL(0xf3524dc1,0x1f19f151),LL(0x28b7f8e9,0xc5b4f62c),LL(0xe1899b7c,0x7240d638),L_(0x249ecebf)}},
   {{LL(0xf9ad69ab,0x9a96c604),LL(0x1e0ee825,0xd851c4f0),LL(0x816d9822,0xf895d68e),L_(0x14e58e83)}, {LL(0x914c2f30,0xdd941975),LL(0xd4525a24,0x1765af51),LL(0x9bdc107c,0x770ebc8f),L_(0x908f73aa)}},
   {{LL(0xe5bd052d,0x70e13a7c),LL(0xe46ccc09,0x0c4379c5),LL(0x0121f8bc,0x71124e71),L_(0x38ba16ad)}, {LL(0xe43a3a55,0xd9134113),LL(0x23d0c031,0xbbbe7daf),LL(0x3a1c2b24,0x3950f72a),L_(0x78296ab5)}},
   {{LL(0xa71550b1,0xdab1491f),LL(0xdd5b8553,0x12caa6e8),LL(0xb08da5e0,0xf505a6eb),L_(0x95e70a66)}, {LL(0x79842177,0xcf510150),LL(0x68ab8f60,0xafdce053),LL(0x87a8f6f2,0x8158b7f4),L_(0xc0b75050)}},
   {{LL(0xe2eb279e,0x6af97a49),LL(0x86441bbb,0xd6c5159d),LL(0xcf78aa46,0x572c05de),L_(0xbc71b5c6)}, {LL(0x0bf9ef5a,0x5b7148bf),LL(0x97df4c43,0xbe4b9631),LL(0x101cdc13,0x7ed1a6b8),L_(0xafbd7e88)}},
   {{LL(0xb580b670,0xc6feaf5c),LL(0x3ea251a2,0xa5cb0934),LL(0x02cfec16,0xc2e35b6b),L_(0x0fdebc9c)}, {LL(0x6e09365e,0x5b4d0976),LL(0x54866bd4,0x405714ef),LL(0x892fc650,0x75be4796),L_(0x224babf9)}},
   {{LL(0xe024a3b6,0xffdc8787),LL(0xd74e707e,0xbb8a1baa),LL(0x16ab4631,0xf8688223),L_(0x0cf2f805)}, {LL(0x09c1692c,0x0dd7a6f9),LL(0x42b206a1,0x3a94d17c),LL(0x784e210d,0x58b82c4e),L_(0xd0f07cde)}},
   {{LL(0x4df5c033,0xa0b09f2e),LL(0x0f337455,0x8ad59d7f),LL(0x92ef323e,0x9dfd994c),L_(0x8a562d14)}, {LL(0x9c294c52,0x0784cc2a),LL(0x06b511cb,0xa7c397c2),LL(0x9d3b5c1e,0x5217298a),L_(0x9becab02)}},
   {{LL(0x5fffdecc,0x15425166),LL(0xf204b46f,0x3ad2c54a),LL(0x29b56266,0xcbfffc28),L_(0xe2bc5353)}, {LL(0xb4051ff4,0xe925b548),LL(0x91eefed0,0xd94421d2),LL(0x61d3c552,0x0c2f7520),L_(0xd3c090b5)}},
   {{LL(0x70ed4769,0xfa81c450),LL(0x0d59a33e,0xbb762aa7),LL(0x72d6d774,0x7b08eae2),L_(0x92885c49)}, {LL(0x1432d1e2,0xc28ea0a8),LL(0x4e6a192a,0x57bf54b2),LL(0x163d3f1c,0x996fa17f),L_(0x8242af09)}},
   {{LL(0xb5fbaaff,0x53d923d7),LL(0x4aa1aad8,0x8aeef401),LL(0xc59594fb,0xd7267ef5),L_(0xba7c28dc)}, {LL(0xede0bedf,0xb019e637),LL(0xbffc5a3c,0x1be98aa8),LL(0xb69b47a8,0xd848486b),L_(0xa1758b69)}},
   {{LL(0x95f000bf,0xc417bef7),LL(0x3f4d4f15,0xef4c07bd),LL(0x74d082e1,0xc8cd53f2),L_(0x2210c27d)}, {LL(0xfb026021,0xaab1ecd2),LL(0xf4960e70,0x47b08bcf),LL(0xf6ef41f3,0x386f4e41),L_(0x3d017b94)}},
   {{LL(0x806b3be3,0xd292e87d),LL(0xbe3b4c15,0xb316049f),LL(0x08cba3d2,0x5a2daba1),L_(0xf68b1a9e)}, {LL(0xff881fa0,0x4da3f147),LL(0x91960fa2,0x8b3ea4f6),LL(0x7f14ef98,0x931e6a5d),L_(0x291615c1)}},
   {{LL(0xe8b9b590,0x3a4e460d),LL(0xc7e1d53c,0x410540d3),LL(0xba715bf7,0x9faf6e57),L_(0xf612eab8)}, {LL(0x13d8563c,0x83a48da3),LL(0x97d11aa7,0xbc420fcd),LL(0x54fecfb6,0x8ea8b02c),L_(0xa72e89c3)}},
   {{LL(0x9f084f51,0x15421912),LL(0x19ce8a03,0x69e1810f),LL(0xa93a4b35,0x66d7f71e),L_(0x33703d01)}, {LL(0x9b5d9b15,0x85c36f3f),LL(0xfb68061e,0xf837f0c3),LL(0x1acc3982,0x4526c2f7),L_(0x8aadc35e)}},
   {{LL(0x91bdbe3e,0x5c462b80),LL(0x46606493,0x16f4a53a),LL(0x2452c556,0xf2ad5a49),L_(0x9ecfe87c)}, {LL(0x961c76d7,0x92031bde),LL(0xd10c3c2b,0x43fc1b45),LL(0xba2806c7,0x7ea59306),L_(0x6d69a1f2)}},
   {{LL(0xaf7fedc2,0x44f370ac),LL(0x575c1c1e,0x7fa03a08),LL(0x366491a9,0x65caef0d),L_(0x1d17b3b8)}, {LL(0x3a469596,0x6dc5f00f),LL(0xf2e830cb,0x87cfba82),LL(0xebb7f488,0x1fc8150f),L_(0x74afed78)}},
   {{LL(0x70a3e6c6,0xf79b97ae),LL(0xcd939876,0x0c92525a),LL(0xfd612292,0x1f877d7a),L_(0xf5237757)}, {LL(0x2b249ae1,0x78a6422a),LL(0x2855f66b,0x0659a5e3),LL(0x12934c87,0x7a1fca11),L_(0xdfe80915)}},
   {{LL(0xe4d85de7,0x0b0acedc),LL(0x21b8a084,0x6091bc19),LL(0x04526cb6,0xac1ce3ea),L_(0xd10a9ec7)}, {LL(0xb54024a7,0x57196c24),LL(0xb55d4b5d,0x42e04321),LL(0x1b1d5d2a,0xbb31992a),L_(0xdb403b45)}},
},
/* digit=15 base_pwr=2^105 */
{
   {{LL(0x5f4861c4,0x9a1b156b),LL(0xa1ffe24b,0x45b930e9),LL(0xbeb5a955,0x29358e8a),L_(0x4b8943f1)}, {LL(0x7923968c,0x127f5983),LL(0xe58cb43e,0xf5aef19b),LL(0x22713358,0x0329be36),L_(0x552fb6a1)}},
   {{LL(0x842632cf,0xd5e766ad),LL(0x947d9fff,0xbab1e908),LL(0xdb59f894,0xa103e5b9),L_(0x68372195)}, {LL(0xc4c67973,0x2cd3e628),LL(0x146f4450,0xe2690b88),LL(0x787c5ec2,0x153e0f49),L_(0x1f9bb175)}},
   {{LL(0x01d9f5c7,0x9acab18d),LL(0xbfb21488,0x96fbb4bb),LL(0x43543f3e,0x19e99acc),L_(0x81d541e2)}, {LL(0x4d4f6943,0x2373babd),LL(0x96c02c64,0x0c662932),LL(0xdd524336,0x60570f05),L_(0x6ee3ed44)}},
   {{LL(0x98a32e88,0x9cbcb156),LL(0xda250980,0x00a72b87),LL(0x85f705a8,0xb60a7751),L_(0xa9f85b0d)}, {LL(0x3ef7d9b7,0xc5078f17),LL(0x1fa64414,0x20110dec),LL(0xb521d60e,0x324aa16f),L_(0xcf1c2382)}},
   {{LL(0x7053cd15,0xc75d38b5),LL(0x9347ce64,0xff26fb52),LL(0xfa6fa1ea,0x89904b74),L_(0xab858e49)}, {LL(0xb9800606,0x0d1c07cb),LL(0x0b99d5c5,0x7605e424),LL(0xe982130b,0xc4ed9c66),L_(0x45e55121)}},
   {{LL(0xb3f704f5,0x2c1c043d),LL(0xf3189b49,0x3e68dbd7),LL(0xbe354e8d,0xa6382386),L_(0xf9ee8ba0)}, {LL(0x26378a41,0x04646666),LL(0x709ded81,0x88fbb7ce),LL(0xc603785d,0xb1f815a2),L_(0x5bec9b88)}},
   {{LL(0x43fe75c9,0x9a29229c),LL(0x052996cb,0x9b1a9170),LL(0x7d4b9fc3,0x133dd4ca),L_(0x1176baf5)}, {LL(0x53ac8647,0x6dcfad8d),LL(0xcfedffdb,0xe4115707),LL(0x6ff11203,0xbc9689c3),L_(0x8c720cdb)}},
   {{LL(0xd81789dd,0x7b457d74),LL(0x127b60ac,0x5a87275f),LL(0x096f997d,0x5d4f53a6),L_(0x554f3104)}, {LL(0xc86a176b,0x0c5247ba),LL(0xf8205ea1,0xb389c0f4),LL(0x325d67b1,0xdc511ddb),L_(0xc2e38d56)}},
   {{LL(0xa0dc02ad,0xca8b2e56),LL(0xac05bb21,0x59515fbd),LL(0x152ad207,0xa0ec8c40),L_(0x1f1977c8)}, {LL(0x990ce0e9,0x094829b8),LL(0x110ed8d5,0xdb93ce78),LL(0x53af9d31,0x980f663f),L_(0x50f753ac)}},
   {{LL(0x0afcea41,0x1410da31),LL(0x91d6a357,0xc4dd3dce),LL(0xefc18ea4,0xc844e9d6),L_(0x599c3b4e)}, {LL(0xb2b1ab81,0x89238185),LL(0x716b0afc,0xc007c089),LL(0xcc238164,0x10df0d3c),L_(0x7d888920)}},
   {{LL(0xd9103e74,0xec1702f8),LL(0x0184b65e,0x67e64440),LL(0x4b15a941,0xaf4f5d2e),L_(0x334e5ea3)}, {LL(0x82fa4171,0xbc874d39),LL(0x3cf9db40,0x63f968d3),LL(0x7b71afa1,0xc6c567c6),L_(0x74793b87)}},
   {{LL(0x1b353d4c,0xfbfeb69e),LL(0x001b5f0d,0xaecee209),LL(0xdaae33fb,0xc8d8f5e9),L_(0x76be05c8)}, {LL(0x4e52dc92,0x87741a88),LL(0xd9f61920,0xa5009003),LL(0x72c650f3,0xc505e3b9),L_(0xb83f81c5)}},
   {{LL(0x2c204e5b,0x9f6af3aa),LL(0xd6b594bc,0x0423f2f8),LL(0xa67cf879,0xe29129c8),L_(0x99f505a1)}, {LL(0x47ebe4a1,0xc220fd1a),LL(0xe7a97ad3,0xa34e4a65),LL(0x1fd2a0dc,0x7c1fb322),L_(0xa75c8fc2)}},
   {{LL(0xcad10b12,0xca3e4b04),LL(0xdda6763a,0x00ce7df6),LL(0x9c84d269,0xa1f31fb1),L_(0xe404dde9)}, {LL(0x45e72361,0xf80ab752),LL(0x300ca2ea,0x38935f38),LL(0x36966819,0xebd1b2a6),L_(0x758de4e7)}},
   {{LL(0x28dbeb1e,0x4f41e2ad),LL(0x153889e3,0x1c0782c2),LL(0x60b0cb1e,0xb609d7c2),L_(0x9a85e0f0)}, {LL(0x403c7b8f,0x3d308948),LL(0xe5878d4e,0xd8503865),LL(0x0326fd01,0xd472fecf),L_(0x51c81464)}},
   {{LL(0x98673101,0x35c900a8),LL(0x2437a09e,0xdb5df675),LL(0x63415d28,0x6b676a73),L_(0x5a4ddb3f)}, {LL(0x3e21afd1,0xa33929eb),LL(0x332e8fd4,0x717e43a3),LL(0x6bf414b5,0x9c7a3c47),L_(0xfb327e1b)}},
   {{LL(0xf9ea4385,0x6b738d34),LL(0x3b9138c4,0xac317163),LL(0xd197efee,0x2438948a),L_(0x2b8a07e9)}, {LL(0xebcc3f74,0x1fc5f207),LL(0xdfbd84a8,0x9fdbdbb5),LL(0xddc39008,0x663a5b4e),L_(0x9956763f)}},
   {{LL(0xa670dec5,0xaeb71ef5),LL(0x989b6598,0xa4973f38),LL(0xf981cb1a,0xe1b496e0),L_(0x187e0a5e)}, {LL(0x8185f4e5,0xdcd9014d),LL(0xce47e120,0x01c7c910),LL(0x33bfde50,0x0a1d6d42),L_(0x3059677d)}},
   {{LL(0x0c74186d,0xf93c6e65),LL(0x2d2eb0b3,0x5c4b9d3f),LL(0x9b7f55b4,0x62194b5f),L_(0xf492843a)}, {LL(0x07af0f2d,0xead94ac6),LL(0x344ba8f7,0xfc351f8f),LL(0x6a02d6dc,0x96c34c95),L_(0x0f698b8c)}},
   {{LL(0x06eb2a9f,0xb29f094a),LL(0xe4c8cfbd,0x8cc86e57),LL(0x40844e37,0x2f8aef05),L_(0x7588271e)}, {LL(0x864a5330,0x8ade6f97),LL(0xf9d12d2c,0x78f8e22f),LL(0xadc126ed,0x8c70fd00),L_(0xc12911d9)}},
   {{LL(0xecdb5c0f,0x3c35a3f6),LL(0x0e784293,0xe0f4235e),LL(0x49ad3737,0x162c0cf0),L_(0x9e096565)}, {LL(0x6d262e33,0x3611d970),LL(0x2ea4dc97,0x56f9b02c),LL(0x569e559b,0xd700ade3),L_(0x6445516e)}},
   {{LL(0xab2f2394,0xf45846e5),LL(0x405bf6e0,0xaa46d2c3),LL(0x1222d842,0xf5f64176),L_(0xedb70efc)}, {LL(0x1b067c96,0xb8990097),LL(0xb7f1c7b7,0xd2f67212),LL(0xe2b50ce3,0x7ac5ce8c),L_(0x05c93598)}},
   {{LL(0x754843ce,0x77e241fc),LL(0x1ab02032,0xa0d9cbeb),LL(0x55cb5b7b,0x98136c68),L_(0xd09ef0e2)}, {LL(0xef262224,0x523e661c),LL(0x87906701,0x3066156f),LL(0x3e7c7958,0x5ea1ccfb),L_(0xb06fbe44)}},
   {{LL(0x736e6685,0xa762a548),LL(0x9218789a,0x4ea95a8e),LL(0xa90f1dd7,0xf1a7ecac),L_(0xbc8d97ba)}, {LL(0xdb175bac,0x176f58b3),LL(0x16633eeb,0x66d2f2dc),LL(0xf30e4d83,0x047e3e70),L_(0xfad00adb)}},
   {{LL(0x2523bfbf,0xdb1ab97e),LL(0x632d110c,0xa7f43904),LL(0xed30067f,0xe0063138),L_(0xcfac77b5)}, {LL(0x0fe3afe1,0xef7cd5f9),LL(0x5720d9f1,0xbceebc62),LL(0x7475ff1c,0x37aef5f2),L_(0xcea553ab)}},
   {{LL(0x21d9dd89,0xfe303f84),LL(0xe84277eb,0x403a10c6),LL(0xdc250e4e,0xcf99ada4),L_(0x260cc2ff)}, {LL(0x16293017,0xdf77e00f),LL(0x429f4cad,0x017aea98),LL(0xb51701db,0xb9a51f04),L_(0xdf5024fa)}},
   {{LL(0xaefa0a3f,0x3c6160d1),LL(0x89d2a9d2,0xb6871dd0),LL(0xcb336211,0x30015933),L_(0x4ddcddf5)}, {LL(0x87d9707b,0x18fd4882),LL(0x96d9668a,0xf138a306),LL(0xeeb3123d,0xce3f684e),L_(0x9fa39b7f)}},
   {{LL(0xdb674dfe,0xd843e892),LL(0xe36b5375,0xedfd1c18),LL(0xc7988b87,0x3bdd2895),L_(0xe52a31a1)}, {LL(0x09bd2af3,0x294ff4e2),LL(0xccd1887d,0x2ad4535e),LL(0x2fc2c26d,0x1ebd13d4),L_(0x647f4987)}},
   {{LL(0x0bdfba7c,0x94e42bd5),LL(0xae5c8c95,0x512c1be6),LL(0x10be5ddc,0x6bc6d702),L_(0xfd7b27ee)}, {LL(0xd21e8821,0x636561c6),LL(0x170a05c4,0xe4ef4a34),LL(0xda161673,0x3d69a37a),L_(0xf40d8e35)}},
   {{LL(0xc206f7b1,0xa0f11bca),LL(0x9195ecf0,0x23eeb3f7),LL(0xe6050ed7,0x98889901),L_(0x07aecdeb)}, {LL(0xc76ba3ed,0x4edc3ee6),LL(0x6b353524,0xaab7f4a8),LL(0xc6e7d7b5,0xf6442856),L_(0xd31e141d)}},
   {{LL(0x5b3638e7,0xd78b290a),LL(0xd885546c,0x77aef8c9),LL(0xfac422d4,0x0feb906d),L_(0x92b8b923)}, {LL(0xdc0b9002,0x331327a2),LL(0x224f79a8,0x5d146cb4),LL(0x6fb5da21,0x3f887bb5),L_(0xc43cfdf7)}},
   {{LL(0xb80d181d,0x2be6d108),LL(0x3c8b1f59,0xb342ddc9),LL(0x7004d85b,0xbe7db7ec),L_(0xec029270)}, {LL(0xc1d565db,0xb82063a3),LL(0xbb014507,0xaa90beff),LL(0xbf4ed46e,0x3a15f177),L_(0xf45b367f)}},
   {{LL(0x49eb74f9,0x146e10ea),LL(0x142ba6f4,0x1256c788),LL(0x346ea27d,0x3f761b6a),L_(0x80f4ab84)}, {LL(0x4ec42ca8,0x150f9515),LL(0x566df38a,0xda268f4d),LL(0xe31f64a6,0xe4706b89),L_(0x95e5b78e)}},
   {{LL(0x0b4a9c49,0xc8ae2f2d),LL(0xd939334e,0x58cc7b3f),LL(0x3e9cb1a4,0x31facd26),L_(0xcce403b2)}, {LL(0x3b34d497,0xd018498f),LL(0xc5995b75,0xcbad3ed8),LL(0xa8867a65,0x0fb7952a),L_(0xef5960a9)}},
   {{LL(0xe832b584,0xa4bc1e98),LL(0x3d7bc503,0x4ca3a484),LL(0x2da2b54a,0xbbcd02db),L_(0xf9f27d63)}, {LL(0xd59bf324,0x02c44a56),LL(0x3ff8f28b,0x366f36f1),LL(0x0dbb411d,0xf30d32a2),L_(0x31dc1c76)}},
   {{LL(0x52b2e707,0x19599554),LL(0x83cc5035,0xe133dbc1),LL(0x659be3d5,0x90ee1c4f),L_(0xfff40ea6)}, {LL(0x70cd0820,0xba72fb3b),LL(0xab738bd2,0xff5688ad),LL(0x3ae40925,0xb751810a),L_(0x1c7232bb)}},
   {{LL(0x071a0770,0x01bb4b01),LL(0xe2be9f54,0xd05cc7e3),LL(0xc3b59e15,0x14e1dc86),L_(0x84ea67cd)}, {LL(0x1ee66ae0,0x743d3ae3),LL(0x9ac0bdc1,0x75487d37),LL(0xb272a24e,0x18ad2d25),L_(0x449e4286)}},
   {{LL(0x69935fe7,0xefec1497),LL(0xee46b002,0xb3e61f3f),LL(0x1edeaf36,0x2b4abccf),L_(0x1302e388)}, {LL(0x6554f5f3,0x0fe91c0b),LL(0x2afabbc7,0x613c238b),LL(0xf8ba68b0,0x0b0ae516),L_(0x1941e2a5)}},
   {{LL(0xbf0bc612,0x9c608ede),LL(0x5d82d268,0xc0f244de),LL(0x0b0057eb,0xd601464b),L_(0x12bdb6e6)}, {LL(0x825a361c,0x4ff78a4b),LL(0xb0ac16af,0x6aed48bf),LL(0x1ae6fb3a,0xba8a9318),L_(0x06a955b7)}},
   {{LL(0x09e379b5,0x9e14e886),LL(0xc7c1e466,0x2fb4d3e2),LL(0x75711ae3,0x036fce90),L_(0x780c6067)}, {LL(0x2542f70f,0xa4a903d2),LL(0x08ed6419,0x95852fe8),LL(0x637b247b,0x012141cc),L_(0x7a1653d5)}},
   {{LL(0x9af74289,0x41ab3d86),LL(0xaaab79ee,0x29c26aaf),LL(0xcccef77c,0xea5535b0),L_(0xde9a50c2)}, {LL(0xc11f99b2,0xb7f340c5),LL(0x0418cc51,0xfc229594),LL(0x8f5c0876,0xf20b5874),L_(0x79b716cf)}},
   {{LL(0xba865a56,0x6fae5df7),LL(0x3ec9a3f7,0x2fe39b65),LL(0x8329316a,0xc07b24cb),L_(0xb74832f6)}, {LL(0xd09a26ea,0x1af7d473),LL(0x95d78e89,0x375af56e),LL(0x62993ad0,0x957236cb),L_(0x20cfce9e)}},
   {{LL(0x3e2c48a4,0x427f5e06),LL(0xe42acf84,0x25837ce1),LL(0xcae65a88,0x966e8b1e),L_(0x2aeaa288)}, {LL(0x9e77f90b,0x306bd895),LL(0x4c375d76,0xd4ed09b8),LL(0xa2585522,0x47a8a9ab),L_(0x875a22b3)}},
   {{LL(0x1e6ed175,0x8c957fe6),LL(0xcd21f453,0x54818bf4),LL(0x2495a8a9,0xdd977b92),L_(0x2e72900a)}, {LL(0x05a5967e,0x2647b9d2),LL(0x92967328,0x53df8e0b),LL(0x496697af,0x5247a639),L_(0xa66c4dd0)}},
   {{LL(0x8424e440,0x682c252e),LL(0x6fcf171a,0x0bb6bcff),LL(0x17581f41,0x040d50d1),L_(0x84837a70)}, {LL(0xd070dc3b,0x686cbe64),LL(0x303cfa74,0x3485fd98),LL(0xdfd534bf,0xf0aec283),L_(0x6e4f877a)}},
   {{LL(0x4458831e,0xbfa63a77),LL(0xa0bbdaf7,0xe6ad6640),LL(0xeffc865e,0xa2922966),L_(0x3c188024)}, {LL(0xae34f5ff,0xec39a211),LL(0xc033bb37,0x5d5de65c),LL(0x1ffcbe06,0xae549602),L_(0x0aa2449f)}},
   {{LL(0xf7d9eb1b,0x008bf1c7),LL(0x17af13bf,0xdce3e9ef),LL(0x7341adb2,0xe6aba225),L_(0x3fdc83ba)}, {LL(0x52473c29,0x58df1458),LL(0x8e82cc6d,0xcbca23e4),LL(0x64a68f69,0xdc93a2bf),L_(0x402344e1)}},
   {{LL(0x12f4285e,0x289cd434),LL(0x0df1904f,0x2614d391),LL(0x5132c5c4,0xd24f8d84),L_(0xd529b441)}, {LL(0xb496fb70,0x945d4d6b),LL(0xb4fe1324,0x64d58e9f),LL(0x226ba20e,0x78fadda9),L_(0xf5638c8c)}},
   {{LL(0x15c734f0,0x71f5ab8b),LL(0x7c5f3148,0xd644b74c),LL(0x8a5a08f7,0x5b9eaf3f),L_(0xcd76894b)}, {LL(0x40783f5a,0xfc35396c),LL(0x5e5b4f20,0xdf4e44eb),LL(0x698ace20,0x85de5335),L_(0x32cc1f5f)}},
   {{LL(0x4ca6d829,0xc01805fa),LL(0x69a66ff0,0x9b659d62),LL(0xb0bfec7f,0x21ba53df),L_(0x41d8db1a)}, {LL(0x3ca3ffdc,0xaaee16f0),LL(0xac0d7259,0x2829c022),LL(0x8266c12f,0x570a8b63),L_(0x347c6b96)}},
   {{LL(0x27c7435f,0xc2862c3c),LL(0xd60a5c54,0x577136cb),LL(0xeb454a5e,0x41bf3bc1),L_(0x809d3205)}, {LL(0xd5ff2a5d,0xce2b3472),LL(0x201d2be5,0xabfed878),LL(0xdae3ae7c,0x076254af),L_(0x898c8347)}},
   {{LL(0xfbee92e1,0x288eeccf),LL(0x6cb1db6b,0x9cfb5c57),LL(0x059c298f,0x53598af7),L_(0x7eda1da0)}, {LL(0x888186af,0x3084705f),LL(0x2f592293,0x86f1d164),LL(0xc59e084e,0xa44fd074),L_(0x3a1664aa)}},
   {{LL(0xd2343b35,0xa40a6bf8),LL(0xad9e07a1,0x798ceee2),LL(0x27b865e6,0xe4570eb1),L_(0x148877c2)}, {LL(0x8f87d685,0xf0ce5967),LL(0x66f1b39c,0x66b1f5c4),LL(0xcbcc053f,0x2f8869b5),L_(0x9fa693dd)}},
   {{LL(0x0d7b69df,0x17f0c4dc),LL(0x135a9bac,0xa20e8614),LL(0xabbfb492,0xc7dc6a2c),L_(0x89df23fa)}, {LL(0x2759d406,0xcc56f093),LL(0x7c85ce0d,0x049b5aa5),LL(0x4e1f566b,0xbec87c76),L_(0x19e59ca6)}},
   {{LL(0x40e840f6,0x1f1c6ca8),LL(0x67aa4467,0x2d21e27a),LL(0x9ca6123e,0x1d5e2c46),L_(0xa8534061)}, {LL(0x5531352f,0x009d49a6),LL(0x161c45f4,0x43dffb38),LL(0xd020075d,0x2f8a3d38),L_(0x25c709ee)}},
   {{LL(0x5ded0d35,0xd70c960d),LL(0x0bc185ca,0x08b78436),LL(0xbcbebb9e,0x662ab949),L_(0x7643c75a)}, {LL(0x1d7a378d,0xcd6abb3b),LL(0x156ac08f,0x991364b4),LL(0x4dbff4b2,0x6e088465),L_(0x1596ba10)}},
   {{LL(0xf7fff47a,0x8b812163),LL(0xabc30e47,0xad939c3e),LL(0xd700d28b,0xe8837ac3),L_(0x35b89ca7)}, {LL(0x26712ef3,0xd89edbf6),LL(0x9b73a154,0xeaa73bdc),LL(0x536ef396,0x19770554),L_(0x44c569b4)}},
   {{LL(0x7f3d2320,0xe904178b),LL(0x58a07ae9,0x7ff297bf),LL(0xad6fdd69,0x6deef5b5),L_(0x4df1d514)}, {LL(0xda650bfe,0xc9fb4677),LL(0xdc9c7db6,0xd99a935e),LL(0x93d1a532,0xbfdd1320),L_(0x939080be)}},
   {{LL(0x0e07db7e,0xfed77923),LL(0x579fd4db,0x0f05670b),LL(0x112a7ade,0x6e79da66),L_(0xb61b7ee5)}, {LL(0x201e8aa3,0x567d0dd8),LL(0x07f189a5,0x9a0b34dd),LL(0x6a730912,0x5c656713),L_(0xcbc9583d)}},
   {{LL(0x9522c2ad,0x1cf99a2e),LL(0xfaa41b68,0xbfb9ae12),LL(0x1fe851c4,0x91215bb4),L_(0x7117d94a)}, {LL(0xbf89747d,0x6286cdba),LL(0x9cbfd3ce,0x792aeb4e),LL(0x25b00fa8,0x61f48d5b),L_(0x7160f470)}},
   {{LL(0xa1ce4a2c,0xbe45da68),LL(0x3b1ab97d,0x63ee425c),LL(0x98b17978,0xd26e1784),L_(0xa89d00db)}, {LL(0x5fcc8cf1,0xecb80c24),LL(0xdcee4667,0x3e21c95c),LL(0x1df7f32b,0xb9cc6c4a),L_(0x77cd2dd5)}},
   {{LL(0x61766f4d,0xa5778295),LL(0xf4e5e3be,0x34307913),LL(0x57f4defb,0x71c105c8),L_(0x3add4e70)}, {LL(0x2096768f,0xce51bf21),LL(0x8db7461d,0x3117821e),LL(0xf475048e,0x2b073db2),L_(0x1fd13104)}},
   {{LL(0x70678950,0x4426eaf6),LL(0x4191c57d,0xd1c53be2),LL(0x68f854e3,0xce0157db),L_(0x5837f49f)}, {LL(0x5112a49e,0xa0471f83),LL(0xf2d94f60,0x87e0a2d0),LL(0x9f64998d,0x344b223b),L_(0xa4dc016f)}},
   {{LL(0x3ad984a8,0x3bd4bc94),LL(0xd8e29806,0xc17ca92f),LL(0x4aee7a72,0xfcd30768),L_(0x488c7f42)}, {LL(0xa1f41f77,0x6fdb5587),LL(0x6983a223,0x3c2763ba),LL(0x0145b604,0x6e18b2a2),L_(0xfd8b9df0)}},
},
/* digit=16 base_pwr=2^112 */
{
   {{LL(0xa72d343c,0x006aff79),LL(0xd1d45ac5,0x1ec739c8),LL(0x136d5ba5,0xce9ce64b),L_(0x3571e770)}, {LL(0xb11ea221,0x7bfe943d),LL(0x6d87fbd7,0x57dad905),LL(0xac86bd3f,0x5eeba8a6),L_(0x8def8e05)}},
   {{LL(0xb0b49014,0x31f50c04),LL(0x17dee513,0xd8fab554),LL(0x1d543cec,0x2fb0a285),L_(0x53dbc27d)}, {LL(0x3256cf6a,0xf3bd35dd),LL(0x5fbdcac3,0x8a264f23),LL(0x7dff3a00,0x24830923),L_(0xea9fd27d)}},
   {{LL(0x3267fa62,0x7dfb1e52),LL(0x423506a7,0x34e86055),LL(0x552bcdda,0x131f5b56),L_(0xbbff309a)}, {LL(0x40256e8d,0x61212776),LL(0xd0114404,0x3807bd7b),LL(0xbba74ace,0xa2ce6ab3),L_(0x9f9afd63)}},
   {{LL(0xfd519550,0xd0706483),LL(0x53c2383f,0x89215ea5),LL(0x45e8a87b,0xf3cf4080),L_(0x6ab16c59)}, {LL(0x85c8552e,0x09212e02),LL(0xa9576c54,0x10c2badf),LL(0x19c8db39,0x7b08661c),L_(0x32334cfd)}},
   {{LL(0x9e15fa47,0x148c6f0e),LL(0x64a29d06,0xdb00031e),LL(0x4cc0f2bb,0xc3df9d3a),L_(0x2503f091)}, {LL(0x1ff47bb8,0x620b581f),LL(0xb30724ed,0x5758802f),LL(0x4277ac9d,0xfc122be2),L_(0xcbf16ba0)}},
   {{LL(0x1336d2bb,0xcd5bed69),LL(0x59c4c7e5,0xedd85962),LL(0xac1f77dc,0xea62aa43),L_(0xa9cecef3)}, {LL(0xf970f802,0x25fc53f3),LL(0xb9f73243,0xf9d1654d),LL(0x9f9753cc,0xb61c3720),L_(0xbf308c03)}},
   {{LL(0x96dca6f8,0x6461f973),LL(0x48986188,0x1f57a9df),LL(0xb90ca0df,0xb010f7d3),L_(0xe0bff3bf)}, {LL(0x21188095,0xf21e0d4d),LL(0xa1a7cdb0,0x5f3468a5),LL(0xde0d0f24,0x4e99f334),L_(0x2e30f9fb)}},
   {{LL(0x70d90130,0x7c2bf741),LL(0x95a93740,0x47c7e114),LL(0xb309c07a,0x977513df),L_(0xf1b8dcbf)}, {LL(0xbe1ac3e9,0xb7e947d5),LL(0x92ee6dbc,0x09c3181d),LL(0x41e0c220,0x8b26dfc1),L_(0x897d02ce)}},
   {{LL(0xfa332321,0xbb0f1fc0),LL(0x69d913b4,0x68034aec),LL(0x784ebda5,0x58fb05ab),L_(0xb3fd4405)}, {LL(0x32cca1fd,0x83e2269c),LL(0x330bdb45,0xea0312b8),LL(0x33fa100f,0x1761c87e),L_(0xf9882c81)}},
   {{LL(0xf4f00cb6,0x58c8e52b),LL(0x468b810a,0x99489e98),LL(0x5593b9ee,0x7924f02b),L_(0xbf5732d6)}, {LL(0x0334e2ee,0x2a2bf206),LL(0x9c377382,0x41ac277c),LL(0xa7b08209,0x0c5c176d),L_(0xb9b519aa)}},
   {{LL(0x1aabcb40,0x1450434d),LL(0xce36050b,0xcf82a49e),LL(0x0e6c721c,0xec192907),L_(0xc53a6a39)}, {LL(0xfecc2ffe,0x451e72f3),LL(0x5b5bffe2,0x81bf062a),LL(0xe8852ddc,0xf84db443),L_(0x6d98976f)}},
   {{LL(0x32a2e887,0x8b6ac261),LL(0xb92b83b0,0x3c40d7a5),LL(0x1ea4ca4c,0xf7653f1d),L_(0x4e2014e8)}, {LL(0x2b234800,0x56338adb),LL(0x3350e80f,0x314b0e58),LL(0x8e442705,0x79527139),L_(0x96bf154a)}},
   {{LL(0xd63c1701,0x1101a669),LL(0x5972ea79,0xd06390a9),LL(0x501cc189,0x0fb428cd),L_(0x41b24261)}, {LL(0x0a3108ab,0x47aa9e0e),LL(0xff7237f2,0x9c9d8b16),LL(0xe86804cc,0x61ad5482),L_(0x723214f3)}},
   {{LL(0x44329550,0xebe782ea),LL(0x4bfefe5b,0x11e7dc71),LL(0xf9379d6a,0xfb5f9f2d),L_(0x016789fa)}, {LL(0x98c8071f,0x3aaef819),LL(0x0fa53740,0x3d2ba5a3),LL(0x0cd97039,0x2ceebe22),L_(0x144818d7)}},
   {{LL(0x6646c444,0xc3bd60cf),LL(0x1bea68cc,0x7c4048fa),LL(0x5e74dbe8,0xd29b1f65),L_(0x08428502)}, {LL(0x91eb1131,0xbb7b08f7),LL(0x64a0eea4,0x3c5f214f),LL(0x6e37d0fb,0x85ea7863),L_(0x3a3aec39)}},
   {{LL(0x87af66ae,0x26b2f70c),LL(0xb4a6f150,0x5494f6a7),LL(0xf84408cd,0xa24b6575),L_(0xb95400d4)}, {LL(0xfb2baa36,0x7fdee2e4),LL(0xfc9f2464,0x82709ae0),LL(0xd231d3ab,0x8897d9e8),L_(0x4c1905ed)}},
   {{LL(0x43e3a796,0xc93592ff),LL(0x194deaea,0x9aebe35d),LL(0x3c64092f,0x1d7432bf),L_(0x64ab56a9)}, {LL(0x608467bf,0x61dd14ba),LL(0x64e82c68,0x1c6431e0),LL(0xb4aaeed1,0x53076cd4),L_(0x2ded5d91)}},
   {{LL(0x064ac09f,0x3e9b39b0),LL(0x087b2333,0xd8bc5839),LL(0xc348b809,0x8ab0974b),L_(0xdf5fc364)}, {LL(0xf886797e,0x9cb572eb),LL(0xba2f94be,0x9a9ce85b),LL(0xa4a7eaf7,0x0e1605e0),L_(0x3070f698)}},
   {{LL(0xbe868d80,0xeca36ddb),LL(0x52ac8d98,0x86bfcb64),LL(0x9b8656b8,0x2ab84426),L_(0x8306d84c)}, {LL(0x01fe86cc,0x15f75f0e),LL(0xe9502973,0x4c2d70f4),LL(0xf239468a,0x365ce9d3),L_(0x7dd8e76a)}},
   {{LL(0xd3426970,0xe0e930b3),LL(0x19af19f4,0xcb67c28f),LL(0x163565e8,0x312a9e2c),L_(0x0438b01e)}, {LL(0x54d76d69,0x3b508871),LL(0x2b43ad8e,0x71ee55a1),LL(0x4dfa9969,0x0a025c8a),L_(0xe6813e1d)}},
   {{LL(0xc327e03a,0xf0b0b5d0),LL(0x46569e00,0xb462d687),LL(0xa70b2c73,0x842f549f),L_(0x18faac6a)}, {LL(0x864e3871,0x0b583d9c),LL(0xf49df9a7,0x1693a8f2),LL(0x5135df56,0xadecd5ac),L_(0xe9e25ac7)}},
   {{LL(0xe8f3d99b,0x04c439d1),LL(0x811daa39,0x79057450),LL(0xed15dbf2,0x557c06cf),L_(0x3a8a2063)}, {LL(0x09b41a1f,0x0bdc4dcf),LL(0x71cd60e9,0xf085eab4),LL(0xdab0c931,0xb48d6df1),L_(0x8189574b)}},
   {{LL(0x4a27c44c,0x2e80ab05),LL(0x7a2bc54c,0xdd9e1375),LL(0x242b815a,0xa910e496),L_(0xd9fa2e90)}, {LL(0x3f111c1d,0x0c5343f8),LL(0xb573bc74,0x10be67cd),LL(0xf1c6ef62,0x4ae541a2),L_(0x74676666)}},
   {{LL(0xc5669a28,0xff0e3e5f),LL(0x6a94d97d,0xe911fae7),LL(0x85cd9be2,0x4150b30c),L_(0x41391baf)}, {LL(0xab441ae7,0x92888eaf),LL(0xd928ee21,0x0dd62a7d),LL(0x1b7b37eb,0xe9e642a8),L_(0x61afa524)}},
   {{LL(0x15d73eaf,0x70f4b3ee),LL(0x10edc9d7,0x70ff29ef),LL(0xe8787785,0xe5dce106),L_(0x3877e81c)}, {LL(0x36b7ab1b,0x21a140c0),LL(0xf44e41b6,0xffda816e),LL(0x9bd299e0,0x5968ca97),L_(0x9fed2d11)}},
   {{LL(0xb0d9d8cd,0x6b2cc953),LL(0x12446f29,0x932fd885),LL(0x8fc62139,0xb782380e),L_(0x93475be3)}, {LL(0x83692f15,0xb2525747),LL(0xe5afa458,0x2213b2d5),LL(0xc34bb49f,0x0270ba84),L_(0xa86bdfc6)}},
   {{LL(0x3b925a7f,0xe649f779),LL(0x72b7e377,0xa3785aef),LL(0x20ce9dd7,0xdc5dbf9f),L_(0xb05950ec)}, {LL(0xa5d677f5,0xcdc8d1f8),LL(0xb5473ac8,0xd8d3e9cc),LL(0xb698eb06,0x5cbd0981),L_(0xc3f31c81)}},
   {{LL(0x1e752016,0xf090a1eb),LL(0x644b9e14,0xfc5f6857),LL(0x111a1741,0xbaf89d9f),L_(0x01aef06c)}, {LL(0x1b1a1911,0xbfa9551b),LL(0x7145d55b,0x24a70db9),LL(0xc5ef4f06,0x17130b26),L_(0xb33cca3b)}},
   {{LL(0xfcb5b043,0x911935f4),LL(0x4a8bc08e,0x53ed6fbf),LL(0x90634826,0x4421e383),L_(0xe0140eb6)}, {LL(0xc4143e7b,0x12ccf145),LL(0x94d99fc8,0x5bf8b6c9),LL(0x4f765081,0x73883a6d),L_(0x6767c401)}},
   {{LL(0x997c821b,0xe367cf8a),LL(0x246522bf,0x3d0455b7),LL(0xe8c7bcd6,0x92643ab4),L_(0x10fbb420)}, {LL(0x547dbeb2,0xb5294491),LL(0x4b5c792e,0x07450e44),LL(0x472a47f1,0x5e3d3784),L_(0xd9dc604f)}},
   {{LL(0x6372566b,0x4d62de9c),LL(0xbc29741f,0x7a323977),LL(0x78458eba,0x4b2a04ff),L_(0xdc2f1c5d)}, {LL(0x6819a84d,0x0eccbf7e),LL(0x8ee3cc6d,0xb93eec58),LL(0x2a02a086,0x03b76658),L_(0x51df6b51)}},
   {{LL(0x6aa0f608,0x0cbee27c),LL(0x00ecc369,0xae1d1601),LL(0xbbef8dbb,0xe7149786),L_(0x4036b3f1)}, {LL(0xbe686d04,0x91ce3c48),LL(0x339d0769,0x3d6535b6),LL(0xa72c8f85,0x2c9fb75e),L_(0x0a96cd51)}},
   {{LL(0x39dd7c75,0xeebc8c18),LL(0x799170f2,0xa2f0fb39),LL(0xba0f68a4,0x1c5cad2c),L_(0xb5b8faf2)}, {LL(0x3f59abfb,0x3a302769),LL(0xcda64ecb,0xed44c055),LL(0xcd7fd1b9,0x2e974b80),L_(0x7a2910b3)}},
   {{LL(0xc8f85c88,0xf503e456),LL(0x9dc39b84,0xb53e74a2),LL(0x8eb76194,0xc4c55423),L_(0x96e13302)}, {LL(0x153fefbb,0xdbbf4e12),LL(0x77137bad,0x80b3f693),LL(0x65187422,0x9f8c212c),L_(0xdd256603)}},
   {{LL(0x0fcf5a39,0x6da81ce6),LL(0xb694d211,0xdff4777e),LL(0x4a9fa32b,0x602640c6),L_(0x5470aa01)}, {LL(0xa3b241c7,0x2a545f4f),LL(0x33f7dd48,0x4cf94265),LL(0x8a894ff1,0xb5260296),L_(0xd9db2162)}},
   {{LL(0x7320211d,0x391c4f62),LL(0xf472fe07,0x6d039385),LL(0xbaa68de6,0xc3a5b977),L_(0xde6b274a)}, {LL(0xbca0bb5d,0x3643a502),LL(0xc554c3be,0xe1f5b66b),LL(0xa430cadc,0xb32e7ba4),L_(0x0c4ee9d5)}},
   {{LL(0xd36efcf1,0xb6b2c682),LL(0x0eae66b5,0x1024c516),LL(0x8688f303,0x99e728ff),L_(0xb372d118)}, {LL(0x777d2fdf,0xd972734f),LL(0xd9b90e14,0x25251453),LL(0xf34097fd,0x424404cf),L_(0xc047b89b)}},
   {{LL(0x3079239e,0xb5ff46f3),LL(0x60a4632b,0x36a3735f),LL(0xc9986723,0xd572091f),L_(0x2023152d)}, {LL(0x0dabffd6,0x0da45042),LL(0xb1524e2b,0xf6ffaf4f),LL(0xbffef0a3,0xbc330b9d),L_(0x99ba1268)}},
   {{LL(0x3b961738,0x8d236f27),LL(0x3ea1c105,0xa5f014ae),LL(0xee0ae6b2,0xe50d6b9c),L_(0x82eb66db)}, {LL(0x2b5f008f,0x36e3aea7),LL(0x4b4fc9df,0x2f8e4e4e),LL(0x5ccc02df,0x9f683f7c),L_(0x3e46d9c7)}},
   {{LL(0x7caa8791,0x6c2cdda1),LL(0x3274eefe,0xa9cdf2c5),LL(0xb634f00e,0xd5eec010),L_(0xc5ba5f67)}, {LL(0xa6a18c32,0xde736bf0),LL(0x554e6af6,0xc1059185),LL(0xb7a29e2b,0x9b23923d),L_(0xbfed0131)}},
   {{LL(0x7000c5a3,0xb80400e7),LL(0xf87a8596,0x147ccc0f),LL(0x9e3b070e,0xfef094f1),L_(0x36f53f39)}, {LL(0xefcc38e1,0xf5b77a62),LL(0xb1c0b71d,0x2c523e41),LL(0x02efb3f7,0x210639da),L_(0xf54c9329)}},
   {{LL(0x330449ca,0x7b06f635),LL(0x0b195c0b,0x9efabcec),LL(0xe879f10c,0x3bd206ba),L_(0x25bef90b)}, {LL(0x0312bdf4,0x88e64b25),LL(0x29334071,0x3a8f90eb),LL(0x3e832f84,0x14f38ac0),L_(0x8a3f7e05)}},
   {{LL(0x4bafc9dd,0xb6461cb6),LL(0x75d1e03c,0x266ba391),LL(0xf40aa6ef,0x4715355a),L_(0x38cce560)}, {LL(0x1adab378,0x716f224c),LL(0x38711bba,0x32b71c39),LL(0x7e89026d,0xccbd7f4e),L_(0xcc519b13)}},
   {{LL(0x7de43131,0x0bbfe1df),LL(0xb1b9786c,0x10701821),LL(0x0bc2e01c,0xf68b86eb),L_(0xf6a6da2b)}, {LL(0xad3f4af5,0x67e192cc),LL(0x6b4b9ef1,0xd5fef609),LL(0x52b649b2,0xf428af30),L_(0x7e90cd58)}},
   {{LL(0xe78176a6,0x0b536242),LL(0xaae8f896,0xdbdd8b1f),LL(0x503b274a,0x51151bfa),L_(0x60ba1217)}, {LL(0x393d4fa6,0xdc4e6926),LL(0xdc183c7c,0xe68387c3),LL(0xaaf0a35e,0x06168aa4),L_(0xb2853fe5)}},
   {{LL(0x2143ebc1,0x4be98451),LL(0xbb1ddca8,0x10163229),LL(0x5db7c140,0x8229e729),L_(0xca486360)}, {LL(0xc19bdf58,0xc8df52e4),LL(0x560cd486,0x89b605d0),LL(0x2a200cdd,0x0c94090c),L_(0xb1ef5103)}},
   {{LL(0x4588f760,0x0c217b21),LL(0x92ea3d28,0x90f7c008),LL(0xef37649d,0xddfaf799),L_(0xfddee462)}, {LL(0x3d0294b1,0x510fffdd),LL(0x1bcf376e,0x4a21ba9d),LL(0x6c9d2cfa,0xa7ada78e),L_(0xca77a07b)}},
   {{LL(0x1db377da,0x63e01d3e),LL(0x5a90911d,0x8c9ad4b9),LL(0x6bb31b8b,0x2b26fec2),L_(0x30e44c9b)}, {LL(0xfeaf9d42,0xe2652b65),LL(0xf5866de8,0x03b48401),LL(0xee2505a9,0x2438e370),L_(0x7a7bb613)}},
   {{LL(0x900098e6,0xd8a6e8e7),LL(0x830770af,0xd8452a93),LL(0x2e2cd25b,0x569bbc5f),L_(0x61715beb)}, {LL(0x80e8caaa,0x3f76b251),LL(0x6dff2aa3,0xeba89ffa),LL(0x18acb601,0xf763dfc0),L_(0xa2dd440c)}},
   {{LL(0x24370f96,0xf5f4a36c),LL(0xbf3c4689,0x527f5322),LL(0x43417948,0xe0b12bf9),L_(0x012f5cea)}, {LL(0x6aa04128,0xb3183162),LL(0x2fa9d189,0x1070f9c1),LL(0xe002ccee,0x5b0c2b4a),L_(0x5840d505)}},
   {{LL(0xfca649ba,0xb223ef3b),LL(0xbe1308ed,0x4aa0b0e6),LL(0x10592f0a,0x3d9b366f),L_(0x3859ac39)}, {LL(0xab61a042,0xaa721fec),LL(0x7c5e75a3,0x0a96a720),LL(0x06be6fd2,0xb3b3cccc),L_(0x9eb8c9a0)}},
   {{LL(0xfda2de37,0xd8576cc3),LL(0x1c4530fa,0x22b5be7f),LL(0xaad70136,0xc40c9235),L_(0x785c8acf)}, {LL(0x9edbb745,0x501faa81),LL(0xc7fdb942,0x60c831a6),LL(0x6f69ca6e,0x813a83b7),L_(0x04f82291)}},
   {{LL(0xdbe3e246,0x4f871623),LL(0x56664b79,0xd2c645ee),LL(0x18c48541,0xc037f50e),L_(0x569420f7)}, {LL(0x08342ef8,0x68cc0cb0),LL(0x19c07325,0x0c4fa826),LL(0x186c66ac,0x8f28f4b4),L_(0x34c938cf)}},
   {{LL(0x837493cf,0xf41079e1),LL(0x433d8394,0x1a861703),LL(0x0e538567,0x03e474ae),L_(0xd2b95886)}, {LL(0xa188c7eb,0x76610ad6),LL(0xe08cbcbd,0x706eb272),LL(0x8025f126,0xb5435d24),L_(0x5ba9b4af)}},
   {{LL(0xc4faeda3,0xd9ddd6dd),LL(0xf3e74637,0xe2be7abf),LL(0x4eea6f06,0x2e6abaae),L_(0xd79c705f)}, {LL(0x99f8dc46,0xc8502a73),LL(0x2f4fd6ed,0xa0aa0199),LL(0xcbc5ed2a,0x38689f49),L_(0x55280ac3)}},
   {{LL(0xfbd4773f,0xb3548b14),LL(0xd9714b0f,0x18cea1f4),LL(0xb3c65f79,0x3bc435fc),L_(0x2867e8b0)}, {LL(0x0706d139,0xca5b9f90),LL(0xe62b0a3f,0x14b87e76),LL(0x864343f5,0x8a82e1fd),L_(0xcbeb5bc7)}},
   {{LL(0x18cc1c83,0x0026972b),LL(0xb17ec512,0x1da022c5),LL(0x2265f4c6,0xb898d6b7),L_(0x5d09e5e7)}, {LL(0xa53eabdd,0xef23c0bd),LL(0x92d24054,0x98e844f5),LL(0x05f9b046,0x7d2e87fd),L_(0x3f161352)}},
   {{LL(0xe7c2033e,0x1c9b4b3f),LL(0x29dc2f02,0xf61d0ee2),LL(0xdb5834bf,0x28b5943b),L_(0x065b016c)}, {LL(0x5de6ece7,0xf70c350f),LL(0xf1049294,0x1c3bfb3b),LL(0xd4c8be79,0xe75564bc),L_(0xb3dd4c16)}},
   {{LL(0xab2cc7d8,0xa4b41a9f),LL(0xbc199c52,0x912b7247),LL(0xe6362cfb,0x6a214d2e),L_(0xe939b311)}, {LL(0xc25479e6,0xf1652743),LL(0x06ef2b5a,0x3bcb10e6),LL(0x9be1e8e0,0xbede8ea6),L_(0x638ffa1e)}},
   {{LL(0x31630309,0x706d16a7),LL(0x3ba8e623,0x9f2f55c9),LL(0xed0db8c6,0xaf0d2787),L_(0xba630c04)}, {LL(0xb9e7f044,0xce8001ee),LL(0x10c159b4,0xb8247bef),LL(0x658e5458,0x78f60572),L_(0x922d3798)}},
   {{LL(0x5c6843ce,0xacaada1c),LL(0x8794be44,0xcabaa3a1),LL(0x36370b66,0x507a17a3),L_(0x943c28ea)}, {LL(0x481b026e,0x48821bb8),LL(0x3885a0a3,0x5ed8e5b9),LL(0x78f46113,0x6817009d),L_(0x1451c3b3)}},
   {{LL(0x7560c539,0x75119a00),LL(0x953124bc,0xe00d03ba),LL(0xef71ab39,0xb863df63),L_(0xfd67781c)}, {LL(0xfedea07d,0xddc4f0d5),LL(0x332679da,0xc43e8f08),LL(0x36d08634,0x9efb1f02),L_(0xa0922f94)}},
   {{LL(0xba05b2e6,0x949d9990),LL(0xd565ff0f,0x76e4e628),LL(0x6c281fa8,0xdccad172),L_(0xcdea3fff)}, {LL(0x6b13eba7,0x70f4d75c),LL(0xd884ce93,0xd34756fc),LL(0x70986d34,0x8f71b0bd),L_(0x430f43f0)}},
   {{LL(0xe10a7afd,0x0a9b3e69),LL(0x66948394,0xf0719c40),LL(0x046cc610,0x9d1611b2),L_(0xda89278b)}, {LL(0x6b7a1ff1,0x24d9cfba),LL(0x638bd7b1,0x5bd30853),LL(0xa740e80a,0x1159e67f),L_(0xe45ccb34)}},
},
/* digit=17 base_pwr=2^119 */
{
   {{LL(0x5898fc39,0xa3566efe),LL(0x23845a77,0x24bb02bf),LL(0x91ab4316,0x00e46ee0),L_(0x83d35ab7)}, {LL(0x8c02881d,0x1d20ad25),LL(0x5866ad97,0xb01ce79f),LL(0x40d6ae86,0x40b2c06e),L_(0x613bd7a9)}},
   {{LL(0x259779c6,0xafc1d993),LL(0x8a7cf112,0x86dcdc82),LL(0x77170d32,0x8441f4f3),L_(0xb0280bfe)}, {LL(0x46144a82,0x58df739b),LL(0x5ee74a89,0x344ddb0c),LL(0x01761502,0xd9bcbeb6),L_(0x676d4eda)}},
   {{LL(0x66de5a2e,0xbc92b6dd),LL(0x7b186d5d,0x1127f258),LL(0x3d18ed03,0x3982ee92),L_(0x50719e4f)}, {LL(0x0deb6e5c,0xcf198f32),LL(0x09997437,0xdbaf1a5d),LL(0x373c1801,0xf524b8bb),L_(0x73564dbc)}},
   {{LL(0x1bb01e24,0xdc5e2b63),LL(0xaca4f30b,0x5df11c5a),LL(0x68e597b5,0x44ec319c),L_(0x8b90301e)}, {LL(0x0e27f5b1,0xe5706d53),LL(0x8b76e26f,0xe67b3280),LL(0xa7111b5d,0x8fa5ee02),L_(0x7b4c92ce)}},
   {{LL(0xe429fedd,0x0035c84e),LL(0x867c82eb,0x2ea6f134),LL(0xa3e74746,0xb9e352c0),L_(0xaebebaa5)}, {LL(0x220bb556,0xb02520ea),LL(0x95bda416,0xe4b94f8e),LL(0x16dbb87f,0xfbff49d6),L_(0x5707ddc9)}},
   {{LL(0xda91cf49,0xdb33cda7),LL(0x450672ba,0x006b3e24),LL(0x489cbfa3,0x4461fe49),L_(0x3d14a461)}, {LL(0x4d93f4e2,0xfe8e5310),LL(0xb7769c82,0xf3d0d29d),LL(0x94deaf7c,0x3d33e2ad),L_(0x3b18c763)}},
   {{LL(0x453d6839,0x8f1b71e2),LL(0xf6d77216,0x923024ff),LL(0xf76bc681,0x17e6ad54),L_(0xe16b7b80)}, {LL(0x66dcecfa,0xe97810bb),LL(0x94c6a912,0x13de9f05),LL(0x30c9e7f2,0xf16b622f),L_(0xa2fbcfd3)}},
   {{LL(0x436d6b95,0x794d49e6),LL(0x3831fe74,0x702ede17),LL(0x5afaacb1,0x3b5ca68c),L_(0x1bea7d66)}, {LL(0x9ca7b740,0x50a1e66e),LL(0x8f17d122,0x762e4f6f),LL(0xc90185b1,0x6705418b),L_(0x36b69158)}},
   {{LL(0x52070a5f,0x9065afef),LL(0xa326ad21,0xe6df0c7d),LL(0x441709b3,0xa1396423),L_(0xade77b6a)}, {LL(0x66868b27,0x14f84f84),LL(0xd0d002f8,0x1fa90792),LL(0x19c083f9,0xfd8d88ee),L_(0x57fe75c9)}},
   {{LL(0x5330a97a,0x77ea9ee9),LL(0x00005025,0xce00962f),LL(0xae5f476a,0x05915824),L_(0xdba736d5)}, {LL(0x8cdf0301,0x660a19e4),LL(0xee8a2aed,0xc70e0b10),LL(0xfdd0563d,0x41d445a6),L_(0xede64633)}},
   {{LL(0x53d54a0b,0xa5729cf8),LL(0xb0c860cb,0xea907454),LL(0xea0c5a91,0x59a871ce),L_(0x9efd302f)}, {LL(0xe025ffb1,0x7e7fb9d6),LL(0xc8022642,0xb7954f0d),LL(0x46ca7ee6,0x010dbf77),L_(0x4bfdff6a)}},
   {{LL(0x2d7e46ec,0x034b69ed),LL(0x850b3401,0x51bc6b66),LL(0x3c25755a,0xe9bf6717),L_(0x95d70595)}, {LL(0x06b03b20,0xfc3e6ce3),LL(0x384f786b,0x723756d9),LL(0x58938357,0xb4731ae9),L_(0xa8628a04)}},
   {{LL(0x42ddfb71,0x33bbd9fd),LL(0xc9ef16f4,0xeb7aa22e),LL(0x725fdf2d,0x88eb52e0),L_(0xa3454e1e)}, {LL(0xf1fc16a0,0x297a7972),LL(0xda90c2e0,0xb92c9670),LL(0x9c2b4849,0xa557aadd),L_(0xb719cae7)}},
   {{LL(0x58182a1a,0x27b6bd7a),LL(0x5c2d70bf,0x1c72e559),LL(0xd4f2f098,0x6b53e542),L_(0xb01de7e4)}, {LL(0xb5e11023,0xa8f7b191),LL(0xb49a6846,0xe62460d3),LL(0xbdf84150,0x74ea071f),L_(0xf4211acf)}},
   {{LL(0xe03c9ae5,0x54fef132),LL(0x9ad73091,0xa900e0bc),LL(0x1a602274,0x3698f6bb),L_(0xe357c5f7)}, {LL(0x57e365aa,0xbb500025),LL(0xf19ef91f,0xf16e71f5),LL(0xa0dfe4bb,0x9d6899d1),L_(0x399ab86c)}},
   {{LL(0x8e577079,0x8e833532),LL(0xabea4782,0xfb9fbccf),LL(0xf5ead3c9,0xe31f0c20),L_(0xe2d0d774)}, {LL(0xc462ce8c,0x6466c045),LL(0xb27628d9,0x85e7447b),LL(0x3e1abc03,0xbba798a0),L_(0x0e905d6c)}},
   {{LL(0x122d19eb,0x158f4312),LL(0x3f408e0e,0x297c5747),LL(0xa7b750fd,0x735aa3f7),L_(0xbe1aa687)}, {LL(0x312d6129,0x31e8bfa3),LL(0x166c2e16,0xcc9cdadc),LL(0x5094cff5,0x9ea9beca),L_(0x2dea9edd)}},
   {{LL(0x616ee97a,0x78257a1a),LL(0x7f5d51d6,0xec98f98c),LL(0x609d2d2d,0xd1b3e514),L_(0xe1569e97)}, {LL(0x2e59e233,0x8087f780),LL(0x45e2a372,0x0b40b852),LL(0x95c2865d,0x8fa97627),L_(0xeeff0632)}},
   {{LL(0x1eac58fb,0x7eb748d9),LL(0x9922cc63,0x0d0b673d),LL(0x955ae146,0x72be4e8e),L_(0xd6809bdd)}, {LL(0x22b08d07,0xa4616bf8),LL(0x09a6ef9c,0xc55b68a6),LL(0x42138fbd,0x8fa529ef),L_(0x4dfbed1e)}},
   {{LL(0xf1766b08,0xdfd72d39),LL(0xe0356587,0xad6b741e),LL(0x59466485,0x5fceaf74),L_(0x24d9205e)}, {LL(0x0c6b2255,0x453c9eae),LL(0xb9402267,0x9c00d050),LL(0xa3817166,0xa87cf553),L_(0x43612515)}},
   {{LL(0xad146ff6,0x39990110),LL(0x0f98feec,0x3d48a808),LL(0xfe9fea01,0x8c4aac12),L_(0x2443ed5e)}, {LL(0x0ad5c79c,0x0a9a20f7),LL(0xd997c3c0,0x942b38bd),LL(0x9475c474,0x218031b5),L_(0xd3104cb6)}},
   {{LL(0x4c0dcf95,0xcc4c491e),LL(0x234524ca,0xfa790f55),LL(0x12a6c289,0x37abf7be),L_(0x3e8c0631)}, {LL(0xcb586fd3,0x192166df),LL(0x0c51f097,0x7ee1f233),LL(0xbba9c768,0xb69b46cc),L_(0x2fb2d51d)}},
   {{LL(0x1443ddcd,0x6345a210),LL(0xecf6729f,0x5f035126),LL(0xc8273e38,0xdafd6230),L_(0x83e19f89)}, {LL(0x4e79437a,0xb3e2d215),LL(0x6090e2e6,0x138aca38),LL(0x4afb3539,0x4ce64d55),L_(0x30dd853f)}},
   {{LL(0x6b581765,0xc440084f),LL(0xeff7dc6a,0x088767cc),LL(0xf0d004bb,0x94e1ba30),L_(0xb9c2e7e5)}, {LL(0xa1a18413,0xa732c539),LL(0xc31da773,0x896710e9),LL(0x3608f75d,0xa067bb07),L_(0xaa5451e2)}},
   {{LL(0x98910990,0x65c3a1bc),LL(0xbcc70896,0x3321e865),LL(0xb04ac43d,0x438e5b9d),L_(0x43f7f50c)}, {LL(0xe260d025,0xc4dcba5f),LL(0x79613445,0x5e5f5f4b),LL(0x37b1a0c5,0xd3c102f6),L_(0xa0273b2c)}},
   {{LL(0x4e2d8d4d,0xe6546299),LL(0xd9309108,0x262d0cb1),LL(0x2df96d37,0x349292cc),L_(0xa711d216)}, {LL(0x2fddedd4,0x452a0a52),LL(0xd9e54125,0x2293a068),LL(0xc6053861,0xfe3040eb),L_(0xfb5c3f2b)}},
   {{LL(0x02a166d6,0x861af22b),LL(0xc493f8ef,0x449c202b),LL(0xb053bb62,0xbb664aff),L_(0xc5af5837)}, {LL(0xd6f78a71,0x14a57ed2),LL(0x1edcfcb6,0x6c5eedb1),LL(0x591d0e00,0x0f079d9e),L_(0x6f7c473e)}},
   {{LL(0x4ee2bac7,0xc62978b5),LL(0xeec393c4,0x0ee684b8),LL(0x56d30723,0xb8993d5e),L_(0xdfc904ce)}, {LL(0xfc172756,0xc8622a7c),LL(0x73b2ec23,0x4279a0bc),LL(0x78cfee0e,0x3826553c),L_(0xd011a36e)}},
   {{LL(0x9060272d,0xe29ecb65),LL(0xa0218247,0xdd976fe9),LL(0x53d3e99f,0x278ebed1),L_(0xbcea3ccb)}, {LL(0x7a33fe7f,0x13d33afb),LL(0x1b5111c1,0x19d16dca),LL(0xe09a47ad,0x6e0ed440),L_(0x80396140)}},
   {{LL(0xb55799e5,0xf145e8f9),LL(0x443b3b83,0x97e6d3d7),LL(0x4580043d,0x5f599a51),L_(0x2fcd83a6)}, {LL(0x38414d95,0xe09d35b1),LL(0x8975407b,0x31551460),LL(0xce073290,0x1044bab9),L_(0xae80d126)}},
   {{LL(0x045e586e,0x4d8f2d61),LL(0xc81d17ac,0x1d2b5b31),LL(0xb68b7f81,0x10bccde6),L_(0xdb1468c5)}, {LL(0x0578898d,0x1e460079),LL(0x5fbad189,0x31bad31c),LL(0x3d22477a,0x8c371b62),L_(0x32c1f1f2)}},
   {{LL(0xbc19767c,0x3ed7db29),LL(0x4ce61a9a,0xcc843c60),LL(0x45c30a38,0x508e3e86),L_(0x5c8b4727)}, {LL(0x1457fcb7,0xd8c78825),LL(0x308a4222,0xc3997703),LL(0x2b88e6d2,0xa1f2140c),L_(0xbcc49242)}},
   {{LL(0x2a452962,0x62444dca),LL(0xffa24655,0x2d1e976d),LL(0x1d95f51c,0x2bbeb1e0),L_(0x6ac09c41)}, {LL(0x052daa66,0x827170a2),LL(0xbfaa5a51,0xc319b1ba),LL(0x3a8ec55d,0xda0c72a5),L_(0x9d0ac0b5)}},
   {{LL(0x898419db,0x35d9c257),LL(0xbb8eaa1e,0x55758230),LL(0x4e49ddae,0xd5c3703e),L_(0x8e18cadd)}, {LL(0x28d3f23f,0xa38dc679),LL(0x8ecf56aa,0x749ea0b7),LL(0x15724faa,0x68bb7f4b),L_(0xb2b1249d)}},
   {{LL(0x0f227df0,0x24af0afb),LL(0x8232f88c,0x1cd97cee),LL(0xa2ad5165,0x678f117f),L_(0xa5067478)}, {LL(0x03432db4,0x5cb93ad2),LL(0x610d26df,0xcc96e20a),LL(0x47d3d48c,0x5f1e84c2),L_(0x31ddfcd2)}},
   {{LL(0xc4293aef,0x7f0db6e5),LL(0xe03b4f7e,0x9069c8c1),LL(0xa088318f,0xf4210d55),L_(0xe9cec920)}, {LL(0x13bd2b28,0x34f3150f),LL(0x7de6ff6f,0xe152eb4e),LL(0x0fcf1527,0x32e6885b),L_(0x4f27fdfa)}},
   {{LL(0xed8906ba,0x7eba0ca2),LL(0x615cf3d7,0xc91dbf27),LL(0xcf44aa02,0xf0948c8c),L_(0x346dc0f7)}, {LL(0xbe7b7488,0x88539394),LL(0xe2e6a4a5,0x938fd6cc),LL(0x15b31f6a,0x1a1c6b6b),L_(0xe564ce23)}},
   {{LL(0xc2869ed1,0x61ab3c9f),LL(0x5693bc05,0x3bb8c353),LL(0xb69781cd,0x95d9d719),L_(0x31986774)}, {LL(0x3de229eb,0xc3895533),LL(0x72ce5e74,0xb6955022),LL(0x5ac0d29e,0x6e2ba9f7),L_(0x12310f9c)}},
   {{LL(0xfe2a24c4,0xe3406a2f),LL(0x7a0fc1d1,0x67beb235),LL(0xf6867bb0,0x5b2cd123),L_(0xc3378be7)}, {LL(0x2ff473c0,0xed6d88b5),LL(0x5fc0483f,0x046a1f51),LL(0x9b3025d4,0xf4a61d8b),L_(0x8b1ce295)}},
   {{LL(0x99e19904,0x7e9f5e57),LL(0x5c52c743,0x00b84a5f),LL(0xbabc0faa,0x6de1d67d),L_(0xbf153310)}, {LL(0x04041367,0x06672f11),LL(0x6650a578,0x0dd51cb3),LL(0xd4012aa6,0xd5ce357e),L_(0xf901856b)}},
   {{LL(0x538e45db,0x5f595162),LL(0xf13e03cd,0xe917805d),LL(0x56a8bf52,0xf639993d),L_(0x043c78de)}, {LL(0xe876b4ea,0x3eaf3c58),LL(0xe2439014,0x0ef45389),LL(0xbd9e742b,0x95404cfd),L_(0xce45f79b)}},
   {{LL(0x1453ad22,0xf6e400ed),LL(0x0126443e,0xa4ababfc),LL(0xd78f1c8d,0x4ebc5852),L_(0x6149c4a9)}, {LL(0x4e1ef091,0x7415270e),LL(0x84d52256,0x921cd6ee),LL(0x472b1f2f,0x765126f0),L_(0x33de4dc4)}},
   {{LL(0xade7ead4,0x1b413797),LL(0x12c424fc,0xeabbca1c),LL(0xdb705e2f,0xa1e12595),L_(0xa616017b)}, {LL(0xd199009a,0x79fa6f2e),LL(0x9168da3a,0x4d4d9e09),LL(0xbde588b7,0xc0ac97e6),L_(0x9a16a183)}},
   {{LL(0x23cf39f8,0xca0fac28),LL(0x40729683,0x4b179b82),LL(0xb04c01ae,0xfaca0fdf),L_(0xdf006531)}, {LL(0x7b3cc76a,0x3d2bf7fd),LL(0x7e32a952,0xf3f2fea3),LL(0x6369c1d9,0x866e7f8e),L_(0x8e8c8b64)}},
   {{LL(0x6eb23ad9,0x8b9475ca),LL(0xe0c8ef8e,0x22d9fd00),LL(0x4e9a3adc,0x813270cc),L_(0x478c17a1)}, {LL(0x292b8bdd,0x3db7f857),LL(0xe5e27f6a,0xf6ba5b62),LL(0x1b8a570a,0xe20966e7),L_(0x040d840b)}},
   {{LL(0x19b8e7ad,0xe13277a1),LL(0x9ff0663a,0x6d26fe40),LL(0x8f8b7d68,0x0ce42976),L_(0xfa5a58e5)}, {LL(0x48e32e15,0x0ee98d37),LL(0xbdd62d44,0x3bb6c2c8),LL(0x4db74aeb,0x208cd0d9),L_(0x55f387e3)}},
   {{LL(0x47ef78f3,0x6c1b0959),LL(0x5e1f13ab,0x2ccd97f7),LL(0xa303c278,0xf9abf5b1),L_(0x3d97a968)}, {LL(0x0276fd8b,0x2be1d396),LL(0x8b026a44,0x726beb4a),LL(0xe09517f7,0x4e0b499e),L_(0x502462c0)}},
   {{LL(0x5979197b,0x82ffa270),LL(0x23bdd46d,0x52cb678a),LL(0x260e7e25,0xc541f700),L_(0x908eadfa)}, {LL(0x33983072,0xb1d2f59c),LL(0x321cd9fd,0x11eef46a),LL(0xf2b04fe3,0x9fba737f),L_(0xe7bc6bb7)}},
   {{LL(0xbcadb426,0x4d1d9a21),LL(0xb55fa843,0xe2a1f313),LL(0x2170c6a6,0xc15b9466),L_(0xdebbce2a)}, {LL(0xd605b2bd,0x1578a3c5),LL(0xd1e52a96,0x96b6befb),LL(0xbe0b58ef,0xc4f7e33c),L_(0x93e465b5)}},
   {{LL(0x3fcf43e2,0xb19e11a6),LL(0x508b2d4a,0x5ad2ed77),LL(0x6ff2a142,0xe587c79f),L_(0xa56c66b2)}, {LL(0xb2ee2d93,0xa6fa95d1),LL(0x260067ec,0xac5ec114),LL(0x67b40c94,0x41c55249),L_(0xdc3ac489)}},
   {{LL(0x6b551c3d,0xc6c2dc20),LL(0x3452633a,0x1b5233ca),LL(0x0e6faafe,0x4565862d),L_(0xfa2f6235)}, {LL(0x08edb0b2,0xb2d31417),LL(0x6cfe0764,0xa0fa599b),LL(0x77052491,0xb2ac3ec1),L_(0xbcc97485)}},
   {{LL(0x0302d5f4,0x04cf0b83),LL(0x37cf0d9a,0xbd701c39),LL(0xfb565431,0x5eee993f),L_(0xc5d3f450)}, {LL(0xb5c3681b,0x5d285d50),LL(0xcc0e0a9d,0x8618542a),LL(0x8c1d7c52,0xd80996cc),L_(0xddcf4356)}},
   {{LL(0xf0fe4630,0x12c3c8ab),LL(0xfb1a997d,0xb6349270),LL(0x2f148e5a,0xc0513e4e),L_(0x7a141833)}, {LL(0xd043e9e1,0x340f7e04),LL(0x52e9cd98,0xb4acedf4),LL(0x5d6d0c14,0xfa56de6f),L_(0x2558f609)}},
   {{LL(0x435b6bea,0xdb7a9959),LL(0xe6426f07,0x24fae08f),LL(0x3cd0982b,0xcc274d62),L_(0x7f805789)}, {LL(0x02f3dea7,0x2a9bf937),LL(0x774708d8,0xadae4c03),LL(0x61de7f6d,0xba3c48a3),L_(0x8de4f729)}},
   {{LL(0x2faaf033,0x7e8bb31a),LL(0x31c0bd80,0xd937aa83),LL(0x072c01c9,0xbfe57ab1),L_(0x201af074)}, {LL(0x87c5d209,0x4ce95747),LL(0x5887d49e,0x7ee66e9b),LL(0xf01cc54c,0x48478c99),L_(0xa2993bcb)}},
   {{LL(0x957c7ee7,0x50dc83b5),LL(0xacc25598,0xf0f63a80),LL(0xb4578760,0x842ee70e),L_(0x4ca0ed74)}, {LL(0xc891093c,0x24bc2867),LL(0xfd1e42bc,0x0f85fa2e),LL(0x47bb6a4a,0x18cd7a2e),L_(0xa82a8343)}},
   {{LL(0x4077391a,0x798cdd8c),LL(0x381485f3,0x68858372),LL(0xd2c5ede9,0xb240c94c),L_(0x7ec5555b)}, {LL(0x62fa25b8,0x89e94f52),LL(0x25aa6ecf,0x0332347b),LL(0xf207ad5e,0x9e15c0d4),L_(0x865ae0a0)}},
   {{LL(0x6d748e08,0x298a62fb),LL(0x549ffe32,0x8f7fb37e),LL(0x35f944a9,0x92926ab1),L_(0xe6fbe10e)}, {LL(0xdc488cff,0x3f355ad1),LL(0xd2f1e956,0x84b0d884),LL(0x29d8cb57,0x98d39682),L_(0xae5bcaa6)}},
   {{LL(0x07c8c79a,0x4957a527),LL(0x3263fbf3,0xec94fd37),LL(0x601424db,0xf147c20f),L_(0x01b4af1e)}, {LL(0x1adc80ef,0x7d9440bd),LL(0x4bfaaf0a,0xee7355aa),LL(0xd1adde68,0xba4d5d2f),L_(0xbdf3e79f)}},
   {{LL(0x4dc18b13,0xebadf0ef),LL(0x13192c94,0xbc8d4cc9),LL(0x8da37771,0xd555d58a),L_(0xebc86daf)}, {LL(0x42fdc282,0x2e05719e),LL(0x8384340b,0xdc3ed2bf),LL(0x4c1b070d,0x03356492),L_(0x28e4bef9)}},
   {{LL(0x8da99c00,0xa1e580f6),LL(0xf42f23ce,0xcf6d6c6d),LL(0xfdabb4f2,0x80f7da41),L_(0xf36b6774)}, {LL(0x6a7dfe85,0x36467882),LL(0x8548109a,0x30e07e69),LL(0x943abc5d,0x8ae444d2),L_(0xe5bede0f)}},
   {{LL(0x2441fb9a,0x0599179e),LL(0x18313f97,0x54669582),LL(0x47767b19,0x6fa66e89),L_(0xd078a973)}, {LL(0x4596bd1c,0x93f0e068),LL(0xb6d46417,0xe4e3af1f),LL(0xff7a84e2,0x951ce962),L_(0xe6d6f09c)}},
   {{LL(0x466ca296,0x16afa866),LL(0xd1fa4080,0x1f38aad4),LL(0xf4e1677b,0x672e7252),L_(0xf4b620a7)}, {LL(0x3a4e5077,0xfcdff811),LL(0xf5e013a3,0xfe6adcab),LL(0x8a4243ce,0xb04666fc),L_(0xd68cd7b8)}},
   {{LL(0xc1e32f39,0xd93030cb),LL(0xe0a61e74,0x15f3f603),LL(0x39bbca36,0x90c8c914),L_(0x18b00023)}, {LL(0xa53fc682,0xb0c4475f),LL(0x4cc3825c,0xed39d188),LL(0xa57cf95c,0xd562a81e),L_(0xc8cb3e5f)}},
},
/* digit=18 base_pwr=2^126 */
{
   {{LL(0x02c6b67a,0xace24453),LL(0xde939bb0,0xa58b929a),LL(0xbe2f8e64,0x9a3a7efb),L_(0xa302408a)}, {LL(0x7fa4b043,0x8ad04303),LL(0x56c2b048,0x426eb7ce),LL(0xa67998c5,0xdb040e40),L_(0x60850b87)}},
   {{LL(0x3bfd76b3,0x0361658a),LL(0x20f7e490,0xa3b9f1bd),LL(0x66db4cfa,0x4799c86b),L_(0x82aaeafc)}, {LL(0x01c04fa1,0x0689618b),LL(0xd1b3660a,0x5b052123),LL(0x524802bb,0x25748110),L_(0xd8501db2)}},
   {{LL(0xb542168b,0xf1dd3dc0),LL(0x40fb7327,0x3556cca8),LL(0xf3b21a79,0x0d1838e7),L_(0x0263cade)}, {LL(0xbbfa3610,0x43bd0f7a),LL(0xe6ca4a48,0xfdb15d30),LL(0xa293f5a9,0xf9285692),L_(0xf5e2782f)}},
   {{LL(0x7da35373,0xefe3b9e7),LL(0x475d95b1,0x08b0fe30),LL(0x9c0da2e3,0xcef9bd5d),L_(0x69f4e019)}, {LL(0x087c3eeb,0x4386ea9f),LL(0x107a9f5d,0x5543dce0),LL(0x057ad0af,0xc5162840),L_(0x036ea919)}},
   {{LL(0xfeb313d2,0x398fba1b),LL(0x07cbfb75,0x52f53c92),LL(0x98909181,0xfa320278),L_(0x063675b6)}, {LL(0x906d44a1,0x427688fa),LL(0xb499f521,0x1c7282db),LL(0x3901fb30,0xaea2389a),L_(0x14299eff)}},
   {{LL(0x1424671b,0xd8f3ae0f),LL(0xb631e50c,0x80279f71),LL(0xd5a77238,0x602ce4df),L_(0x988af873)}, {LL(0x12e9402e,0xdd095d1b),LL(0xc30915a3,0x6cb15ad1),LL(0x0927adfc,0xa38c1285),L_(0x3f7c6e76)}},
   {{LL(0xec392753,0x1ec20f01),LL(0xeb41a6c8,0x5cac318e),LL(0x47520f8c,0x407d9032),L_(0x0390450a)}, {LL(0xe2f3dac2,0x43dd79ec),LL(0xc468403d,0x9236a81c),LL(0x6bccf29e,0x00ac0a42),L_(0x713f0b03)}},
   {{LL(0x163c66e1,0x849061a0),LL(0xa20a6586,0x70208a9e),LL(0x7bf95257,0x07a10de2),L_(0x3b8aa66f)}, {LL(0x45c2891b,0x15468f60),LL(0xf59044a4,0x73d5ebe8),LL(0x387e234b,0xa32ac9f8),L_(0x4f7ec3db)}},
   {{LL(0x962ab071,0x570fcd62),LL(0xbcce236d,0xb1757c3e),LL(0x1ad55d1b,0x7fa592e8),L_(0x1700716f)}, {LL(0x76fcda7e,0x73d3b392),LL(0xdd37f1e3,0xd6cf8e98),LL(0xca16fca1,0xac456eec),L_(0x5752c0f7)}},
   {{LL(0x5f929901,0x240b92b4),LL(0xef15652c,0x0f7d51a0),LL(0x632731fb,0xf12e9902),L_(0x38b796f7)}, {LL(0x47671b3d,0x985803eb),LL(0x5fa89402,0x63e1c219),LL(0x608cf2d5,0x109df03c),L_(0x8ca4e888)}},
   {{LL(0xb25c71f6,0x0829cedc),LL(0xc7a41eb8,0x483d485c),LL(0x15305f6f,0x60591120),L_(0xf6bc7367)}, {LL(0xbcdea717,0xbeeee884),LL(0xe598a6e7,0x88e6ef42),LL(0xc3d80182,0x4b2dd1d3),L_(0x45cd3855)}},
   {{LL(0x482c367a,0x6b421466),LL(0x8af36484,0xa58b570e),LL(0x2587649f,0x3e4ee4bd),L_(0xe423c77c)}, {LL(0xf5a1388b,0x98494de5),LL(0x4a294a8e,0x8f25a37c),LL(0x50e9cc72,0x1ce8e0f0),L_(0x10660a4c)}},
   {{LL(0x119587d4,0x8e668562),LL(0x167d0962,0x0554888c),LL(0xd291f0fd,0xae58bc85),L_(0x8d335bc2)}, {LL(0x48a506c2,0x034f448b),LL(0xd5ffc68b,0x94dd32bb),LL(0x2bd2abc9,0xb9e588a6),L_(0x2eec6265)}},
   {{LL(0xe1e13902,0x24b16627),LL(0x8bba7a24,0x4f52ca61),LL(0x749a749b,0x8bb11e14),L_(0xb6773c49)}, {LL(0x3dfafe08,0x71bdac04),LL(0x76569dbf,0x40bd5231),LL(0x1587ab8c,0x7d2d4a5e),L_(0xfceafeac)}},
   {{LL(0xe7ffdae7,0x0f11fde4),LL(0x936bcbdb,0xc78059f3),LL(0xf4713c18,0x2d2f0ff7),L_(0x0dce5266)}, {LL(0x5fe86c6b,0xa381beae),LL(0x9b2c0aaf,0x734b9092),LL(0x77f56796,0x82131b3f),L_(0x02162cc8)}},
   {{LL(0x612650a2,0xc29b399c),LL(0xad145b7e,0xd673cbc4),LL(0x60425f21,0xece9a8ea),L_(0xc5d38ead)}, {LL(0x394b09de,0x1b69fef5),LL(0x55ee50f6,0x141fc2ff),LL(0xeab0baec,0x85999248),L_(0x02f743df)}},
   {{LL(0xf4b4d1ac,0xe1f44cb9),LL(0x3931200d,0xd86fbb1f),LL(0x32861f51,0x242811d3),L_(0xa373e731)}, {LL(0x20b39e71,0xc64c1332),LL(0xd9a6709c,0xeb68ae08),LL(0x93c66ba1,0x7aee0eb8),L_(0xfc730fa6)}},
   {{LL(0xf47a18f8,0x992398ea),LL(0x454ef012,0x7414fdc9),LL(0x325c8a8e,0x42a7e664),L_(0x3b3fca9b)}, {LL(0x9f053cb0,0xc8869d15),LL(0x34539da1,0x1878089e),LL(0x11f347a4,0x19ee0c05),L_(0xf6ff52d9)}},
   {{LL(0x3e51ac6d,0xb254a9d3),LL(0x5029469e,0xe2b48c1d),LL(0x916d1214,0x3732c0ab),L_(0x07cba5dc)}, {LL(0x0cb5ec91,0x010093ba),LL(0x6f2fe681,0x4a4cfff5),LL(0x4aaf2a44,0x7feda10c),L_(0xafba52c9)}},
   {{LL(0x58625a1b,0xea895fc8),LL(0x19dbf9ed,0xcac0e4d3),LL(0x53bc48b9,0x1d43df08),L_(0xc8f9dd1d)}, {LL(0xee75a211,0xe1a8bb95),LL(0x718f5878,0x7e7b6ce4),LL(0x4a384a6d,0x26d0db83),L_(0x97ecfe38)}},
   {{LL(0x2bb95cae,0x9e42faba),LL(0x9254494c,0x0f33c611),LL(0xf67c4703,0x6ef93df3),L_(0xc1445815)}, {LL(0x8ad685b3,0x8655685c),LL(0x6cd1cca8,0xdd2eb2bd),LL(0x5e194627,0x523e0fd5),L_(0x44f05a1f)}},
   {{LL(0x301fcb43,0x80fb78c3),LL(0xa1be1446,0xfaa1d373),LL(0x3d0a850d,0x051bb6db),L_(0xc2bd0c81)}, {LL(0xc7fc5cae,0x38f0078b),LL(0x6d01ae21,0x91a544af),LL(0x0c9540c5,0x6cae5443),L_(0x41626d37)}},
   {{LL(0x93ebe56f,0x71946cb7),LL(0x3335ca8b,0xc82366c4),LL(0xbfe6dc77,0x27a97fdc),L_(0xcc4d8fd2)}, {LL(0x80d9e528,0x16d42ed2),LL(0x4d8537e9,0x18abf493),LL(0x0398f8d3,0xa190c5d3),L_(0xd448e52b)}},
   {{LL(0x495ac576,0x6bf10350),LL(0x3cbf1e78,0x2674f39c),LL(0x90ac41f9,0xaedc843c),L_(0x784b629d)}, {LL(0x60ca4b9d,0x2b7ba708),LL(0x88cff37a,0xc070ae0f),LL(0x2e4b0764,0x0c3aa1f7),L_(0xdf186109)}},
   {{LL(0x555cdffc,0xd2310fa6),LL(0xac832df3,0x42f99111),LL(0x9d84f404,0x8d7bdc74),L_(0x64936918)}, {LL(0x43e3cff3,0x1c149edf),LL(0x3b328f21,0x442dde2f),LL(0x1288dcc9,0xb3089de8),L_(0xbad0531f)}},
   {{LL(0x3fe61e60,0x20fa93a3),LL(0x494f3a66,0xb50f92b6),LL(0xbfce0907,0x632ab7a5),L_(0xe52a0324)}, {LL(0xc89531e3,0x6f57401d),LL(0xca680133,0xf7bd6eaf),LL(0x5942987b,0xceffd87f),L_(0x75f1a1f6)}},
   {{LL(0x74161b3e,0x647fc617),LL(0xe81422f8,0xac7f6cc7),LL(0xc485b2c8,0x0414a8d3),L_(0x5303bc61)}, {LL(0xd7de72d1,0x1d6064d4),LL(0x88c0864e,0x9fd6a734),LL(0x8ee9cb9d,0xf104d7a9),L_(0x963ee732)}},
   {{LL(0x776f7b1b,0x2b058af6),LL(0xdc25f3b6,0x0d06abe1),LL(0xb0461b36,0x90f6c358),L_(0xdad04c23)}, {LL(0x9a72e439,0x24e0b184),LL(0x2af6be09,0x6facf219),LL(0x5f6753e5,0x5f23a218),L_(0xa362a760)}},
   {{LL(0x4fd2cc26,0xadb7e3c5),LL(0xdb9150c6,0x4cc67b4d),LL(0x4c80d589,0x6a578011),L_(0x78347a1f)}, {LL(0xc04163a4,0x41978a17),LL(0xa1df6b38,0x4701d625),LL(0x7acb9d4e,0x7040eb22),L_(0xb0f7aae6)}},
   {{LL(0xb117faf4,0x734cd26d),LL(0xd884a071,0x071d299d),LL(0x1ad0de2a,0x826417df),L_(0xda3b0c06)}, {LL(0x487c15f0,0x5432a8df),LL(0x85244fbd,0xa5c2986f),LL(0xe2f76737,0x89d1f699),L_(0xefb644c7)}},
   {{LL(0xe366cc7a,0xcd96f1e7),LL(0x2f09e23c,0x19aa24f5),LL(0xbe89a16c,0xea6d5b99),L_(0x1a7565fa)}, {LL(0x1afa0bd5,0x8da4ca09),LL(0x8baaef4d,0xb278c555),LL(0x07697581,0xea0b2435),L_(0xbefa5c32)}},
   {{LL(0x233ddda5,0xbce451f2),LL(0x6e777818,0xe21ca9dc),LL(0x33fc9573,0x608477ee),L_(0x96d9fe99)}, {LL(0x37f4a8b7,0x2f253ba7),LL(0x11d98950,0xec1f4a44),LL(0x3ae7c4bb,0x1c649260),L_(0x4eeee3f8)}},
   {{LL(0xf186ffa8,0xca88dd8a),LL(0x25be4bd7,0x2e8c5c07),LL(0xd65638d7,0x8d28689a),L_(0xbe778846)}, {LL(0xa2ef4b72,0xfa5e89e0),LL(0x0b27fb66,0xea4b66ad),LL(0x2a17ea7b,0x5744fef5),L_(0xaf89c334)}},
   {{LL(0x1c88c916,0xd377e6af),LL(0xa6b3f831,0xbd38c04e),LL(0x43c62134,0x7e11eacb),L_(0xaee51855)}, {LL(0x3aa19a13,0xe65b9643),LL(0x3179a57d,0x0069d240),LL(0x7d7a8265,0xc6fc4f7d),L_(0x8ea00bca)}},
   {{LL(0xb94a17c9,0x95308d83),LL(0xb3928b84,0x88fbad83),LL(0xd14a8ca8,0xcec36252),L_(0xff246228)}, {LL(0xf71bccbb,0xd90035ec),LL(0x499776eb,0x55a73d3d),LL(0x07f0cdbc,0x6c21797b),L_(0x87db5d7d)}},
   {{LL(0x5162797f,0xa87e9702),LL(0xaaf9d8c4,0x2ab93ff2),LL(0xe2a31c19,0x4288abf4),L_(0xc4cc32b4)}, {LL(0x80be51bc,0x388680ed),LL(0x2f0c118b,0x8dedbe91),LL(0x7da1a829,0xbffcb08c),L_(0xd681d331)}},
   {{LL(0xac1e2c01,0x1d6fb3ed),LL(0x470f2923,0x9f1545b9),LL(0xfb03795c,0xbfecfd6e),L_(0x8ff73fcd)}, {LL(0xac0e74d1,0xcbb46331),LL(0xcbbfedab,0x01849564),LL(0x79c317d8,0x9186b0db),L_(0x7c549c1e)}},
   {{LL(0xe3b9e5c2,0xa44cb510),LL(0x670d9ca9,0xc72bcde9),LL(0xd47f22e3,0x23ac79ad),L_(0xe4bd107b)}, {LL(0x1062a84a,0xd5193e66),LL(0xfb47ce43,0xbb25e4cf),LL(0x600c0353,0x5ab3c0ce),L_(0x005311ba)}},
   {{LL(0x1f36e1c7,0x2a414d7c),LL(0x9bec3869,0x23202a91),LL(0xbdd27859,0xd77095b4),L_(0x270698b1)}, {LL(0xe5a99ae2,0xd0fffc9e),LL(0x3c7db5f1,0x490dac48),LL(0xacfdbe9c,0x9079d090),L_(0x1bb8021a)}},
   {{LL(0x550c3e83,0x448361bb),LL(0xa680b478,0xfeab6ae3),LL(0x41969a52,0x74461138),L_(0x9a38f5c6)}, {LL(0x47381f19,0x2be030b8),LL(0x0e86ce09,0xb091abda),LL(0x4da4cd53,0x7df8f1e3),L_(0xc85acbda)}},
   {{LL(0x9fea92c4,0xab2fcd01),LL(0x85b0423d,0xd804cba2),LL(0xed8709e8,0x19141717),L_(0x49d8b7d7)}, {LL(0x88fb9ada,0xe105506b),LL(0xb90d9a8b,0x8cb2c13d),LL(0x4e6bda23,0xcaba970f),L_(0x7dff6f41)}},
   {{LL(0x9fcd4872,0x160191eb),LL(0xff541bf6,0x826242d0),LL(0x99634c20,0xdfe7289d),L_(0xa5294071)}, {LL(0x4e07eba9,0xc0910d55),LL(0xd5b6e07a,0x7e042c86),LL(0x6d7f5229,0xd74d6e82),L_(0x42f50e09)}},
   {{LL(0x6efbc0e3,0xcc3c4e8e),LL(0xd7d09e5f,0xf21ea1ea),LL(0x05c108d8,0x37aae682),L_(0xb8b85f83)}, {LL(0xf2c37fd3,0x9a51bef8),LL(0x5a44835e,0x5ac834a6),LL(0x2a412b12,0xac985dae),L_(0xf2f75363)}},
   {{LL(0x39bd009a,0xb0ef3648),LL(0x7df070de,0xc1936e74),LL(0x97226670,0x3eb64bfc),L_(0x0a7cc2d5)}, {LL(0xfa065546,0x6cb29cd8),LL(0xd6b2e07d,0x010fbc35),LL(0x6b24b9f4,0x72e08b5c),L_(0xbf7187b5)}},
   {{LL(0xe0daf351,0x2fb0ee94),LL(0x31222305,0xac57f680),LL(0xb753971e,0x6b0ccf83),L_(0x55a3abe2)}, {LL(0xc3b1be68,0x48b857e0),LL(0xfc1add13,0x81e2b0c5),LL(0x595ebecb,0x7151293e),L_(0x43489ffc)}},
   {{LL(0xb788bb44,0x65e9c6ef),LL(0x071c9990,0x10062256),LL(0x891a7719,0xe6f377fd),L_(0x65d3a9e3)}, {LL(0x7355204b,0xe23556ab),LL(0xa6502b28,0x188872cd),LL(0x84eba15d,0xea03bca4),L_(0x62a2b9d4)}},
   {{LL(0x159bb9df,0x08807416),LL(0x78e83862,0x496eba6b),LL(0x5b5b2f5c,0xff4f34bf),L_(0xc8432a05)}, {LL(0x376ec0fd,0xe48904ea),LL(0x386ae415,0x96bf9fdc),LL(0x9b04b325,0x003a468c),L_(0x2a854bd5)}},
   {{LL(0x3d1be5ac,0x62509881),LL(0x76fd690b,0x65d5f607),LL(0x244caa05,0xc38d7540),L_(0x7a21901b)}, {LL(0x0b239ba7,0x19b8e994),LL(0x14c3955b,0x1f25f647),LL(0xd2cd0a34,0x7f481820),L_(0x3e5eb658)}},
   {{LL(0xdceb71b6,0xf395c0f2),LL(0xcc104e0d,0x6febd95a),LL(0x01d8efcf,0x0b358ed0),L_(0xcfc537fe)}, {LL(0xdadfd46d,0xdc5d961e),LL(0x8b4f6c1d,0x221b24a2),LL(0x6d21a125,0xc0f16b10),L_(0x36a0bac0)}},
   {{LL(0x3ece8319,0x54252e96),LL(0x46f2cfb1,0x97edd630),LL(0x020e1e99,0x90685aae),L_(0xfc1bcb8e)}, {LL(0xb7771435,0x6df64310),LL(0x43c4fd8d,0xa647e2b0),LL(0xd6ddd8f3,0x52ff5927),L_(0x5282fbcf)}},
   {{LL(0x3fd7f727,0xbb02013a),LL(0xb1fc86fe,0x6e17def8),LL(0x5ef8f2c6,0x1b970bb2),L_(0x1b714b57)}, {LL(0x1236fa50,0xf7a7e274),LL(0xcc10de34,0xffa4c977),LL(0x5a8179b1,0x99219388),L_(0x798129b2)}},
   {{LL(0x4b407e5a,0x844408fa),LL(0x6be5d183,0x742669b3),LL(0x0751f86d,0xb00b365a),L_(0x34786bf6)}, {LL(0x3a8360fc,0x4147cefc),LL(0x88816150,0x0393026d),LL(0x7a7f6231,0x2e39fa34),L_(0x879f391b)}},
   {{LL(0x149f6c78,0xbbcd3afa),LL(0x8292ce76,0x742b5b3a),LL(0x899f0118,0xb8dc2766),L_(0x18090cc3)}, {LL(0x14c00e4d,0x6924c12d),LL(0x7049ac09,0x4e4111e8),LL(0xdf09100c,0x2987ca68),L_(0xa8c45230)}},
   {{LL(0x65fad089,0x01cff054),LL(0x821650cf,0x3002769e),LL(0xdb9e3912,0xe9ad46db),L_(0x1a05deca)}, {LL(0x6e7044a0,0x34ab7ab5),LL(0xc61e42fb,0xfda4dfcf),LL(0x91982d5a,0x89e4ea03),L_(0x58b6c30e)}},
   {{LL(0x132822ea,0x1ad76044),LL(0x50f3dcec,0x3e8a1d95),LL(0xb981b100,0x37f04df1),L_(0x4ca707b5)}, {LL(0x0b17bd0a,0x0b34cd00),LL(0x93aa35e9,0x447278ac),LL(0x07bde57a,0x6e16122f),L_(0xc9b96d1e)}},
   {{LL(0x3221832d,0x8fc6ce6c),LL(0xe4935439,0x65828539),LL(0x9cdf65d9,0x708e9258),L_(0x5929b880)}, {LL(0x273611ed,0xef43a507),LL(0x9e4d7f82,0x86406043),LL(0x88f84624,0x4396769b),L_(0x09401931)}},
   {{LL(0x25903256,0x6b8a2c7e),LL(0xe200a555,0x567e51fb),LL(0x06f1e0ff,0xdbcad515),L_(0x435d74d4)}, {LL(0xe7fa9d16,0x627c717b),LL(0xd7c74d13,0x3efff1d7),LL(0xa592bf53,0xe58ce7d7),L_(0x0a6f5daa)}},
   {{LL(0x6bc49db7,0x2993ad98),LL(0x11f1de05,0xcd8ed14b),LL(0x104437d1,0x419b6d76),L_(0xb80b6ad9)}, {LL(0x5ceae532,0x2692fc9f),LL(0xfc4d4052,0x96208c11),LL(0x5f17f19b,0x34a83cc6),L_(0x7f3e627c)}},
   {{LL(0x3d238cb2,0x6bb879b5),LL(0xb752157e,0xeaa5d8ca),LL(0x137d82f5,0x769e2a68),L_(0xca3db7bd)}, {LL(0x0ddf279e,0x2ae26372),LL(0xf5d74ea6,0x79e90c0b),LL(0x27e773c7,0x3451edcf),L_(0xe4341440)}},
   {{LL(0x65ddd79b,0x34988e35),LL(0x25d7df53,0xd93aac10),LL(0x814a0e96,0x76ec0b96),L_(0x74d8980e)}, {LL(0xe6639b9b,0x9143cd9d),LL(0xa3c743dd,0xfb9b096b),LL(0x77b915d6,0x4353464a),L_(0x7fdf9660)}},
   {{LL(0x103b0149,0x52d3720b),LL(0x668172a6,0x3c3a2d72),LL(0x0ddae18b,0x3543af35),L_(0x36c45c71)}, {LL(0x90846158,0x51cb6d6c),LL(0xb970c6e8,0x5541df27),LL(0x9f0df2e6,0xa96cf0e6),L_(0xef118589)}},
   {{LL(0x51a147b4,0xbd4c751a),LL(0x9547874d,0x5b132575),LL(0x81839452,0x9b0ddfdb),L_(0x24fa7687)}, {LL(0x2e14f24d,0x54dad63a),LL(0x7d8859c9,0xd7c2228f),LL(0xdcbca7aa,0x77c13acc),L_(0x87a21fd3)}},
   {{LL(0x4c326ed3,0xe3bb354a),LL(0x3fb8d397,0xa148004d),LL(0xb7008ce3,0x85367d7f),L_(0x177a646b)}, {LL(0xe73d1a40,0x93d871ad),LL(0x4c9f1031,0x7768c968),LL(0xd9c87876,0xb38a617a),L_(0xd2b0142c)}},
   {{LL(0xaf8f400e,0x2a94dc1a),LL(0x9b975861,0xf8d2e935),LL(0x370e4c06,0xca35bd61),L_(0xf7a243e9)}, {LL(0x2c1682bd,0x00b6e1a8),LL(0x56f9e8d0,0x6d92e1f3),LL(0x9bb1736d,0x3e9b6293),L_(0xf048cf7d)}},
},
/* digit=19 base_pwr=2^133 */
{
   {{LL(0x6c35afbd,0xf6f30e96),LL(0xdab13683,0x669a8251),LL(0x4dd8bbbe,0xb9b1b311),L_(0xd57a4a34)}, {LL(0xbbcbf46c,0xe4d59c99),LL(0x4923f91d,0x63242254),LL(0x877172dc,0x853f9c6a),L_(0xd7aca10a)}},
   {{LL(0xd5de71ea,0xe7f42edd),LL(0xf74adcc0,0x0b9c53be),LL(0xc8c4ee60,0x82b1e37e),L_(0xbcd67b5f)}, {LL(0x4a10fc2a,0x7502a8c1),LL(0xba4a9d44,0x2dc30076),LL(0xcd3e0527,0x2e0b7f76),L_(0xaeb6d158)}},
   {{LL(0xc21b4aac,0x95f715ce),LL(0x5afa8860,0xff0073cc),LL(0xdb4528e5,0xdbc69fc1),L_(0xe9e0b780)}, {LL(0x8bfc4102,0xe2c44c67),LL(0x74449572,0xa7624b30),LL(0x1b45072c,0xebf2704a),L_(0xae9e87a4)}},
   {{LL(0x540d3efa,0xe3c8a7e4),LL(0xd1ad9579,0x39e285d6),LL(0xccac1ee5,0x8d098605),L_(0x02e00e9d)}, {LL(0xaae2485d,0x424cb101),LL(0x57781fcb,0xd81ebf02),LL(0xe045acd6,0xec4b922d),L_(0xf679db8b)}},
   {{LL(0xe3ace3b4,0xfb5f0701),LL(0x7920ba12,0x1534cde4),LL(0x25904a79,0xf9421b0d),L_(0xc0438d82)}, {LL(0x0528f7cb,0x8a9c39d8),LL(0xb1c7fc33,0xdd383932),LL(0x458a7c5f,0xbaf06aaf),L_(0x2cd8a058)}},
   {{LL(0x193b780c,0x5abd6eef),LL(0x6b220286,0xd76b78fc),LL(0x06442ef6,0x0cdf9e58),L_(0xe520f054)}, {LL(0x8a5c2658,0xfae0a442),LL(0x0ae1c405,0x6c2b778b),LL(0xb5adbc98,0xf374d05a),L_(0x010faa2e)}},
   {{LL(0x0c6f4ae3,0xb657d5f6),LL(0x56b561b2,0xabf1920f),LL(0x70a6775e,0x15909e3c),L_(0x7b91068f)}, {LL(0xe0c73f44,0x563843f3),LL(0xe4066970,0xe15912cf),LL(0x2e80308d,0xd16c8835),L_(0x3121c5b3)}},
   {{LL(0xb00d284a,0x96ae8e24),LL(0x8746c0ee,0x118cc562),LL(0x4ed443bf,0x9157acc3),L_(0x898cd623)}, {LL(0x14b54c13,0x07d6ccc1),LL(0xc5299940,0x05ec4684),LL(0xf50f2de3,0x1a923061),L_(0x8aaedd52)}},
   {{LL(0xc692cc03,0xbe07dcf7),LL(0xa61a9062,0x5f04bf17),LL(0x0d99300c,0xa062deba),L_(0xcb7f082f)}, {LL(0xa3451ea6,0x02666d4c),LL(0xd4b6a94d,0x2fc9077f),LL(0x564f33fe,0xa2bd47b9),L_(0x4614aa04)}},
   {{LL(0x6da6ee28,0xdab5f17d),LL(0x2dfb0cbf,0x36f25faf),LL(0x16a32d1e,0xbe5fdc6d),L_(0x48f97f60)}, {LL(0xd0d79fc9,0x0ee979a6),LL(0xfb77bc6d,0xc670e326),LL(0x6435ef83,0x55425ae2),L_(0x32e4cd8a)}},
   {{LL(0xcf395e17,0xd8954092),LL(0x1415743a,0x516938e6),LL(0xc57f3db0,0xe85d4e50),L_(0x71f22b3c)}, {LL(0x611ea790,0xfafbe19a),LL(0xdbd2d4ed,0xe5ff8f06),LL(0xe41d9e6e,0x9e9c0da2),L_(0xb15426d4)}},
   {{LL(0x86e3cdb6,0x8cb41178),LL(0x0f8b5836,0x5615e2b5),LL(0x74d58d7a,0x4e6b82ce),L_(0x3030ebba)}, {LL(0xd4c6f918,0x374bbfd9),LL(0xcfea53e3,0x15aadf45),LL(0x363a6cec,0x30dd6649),L_(0xf3c95a58)}},
   {{LL(0xe43c3adf,0x120bb9ae),LL(0x98ad19ec,0x73ef55fc),LL(0xbe8a8508,0xbb18ab2b),L_(0x1cda9cb5)}, {LL(0x84facee2,0xb277aa2c),LL(0x29318829,0x742a25ca),LL(0x1cc06cae,0xd0d24d20),L_(0x3eca1b2b)}},
   {{LL(0x0714ac72,0x2ffe0519),LL(0x522d36e5,0x843d58c8),LL(0x7eddcb6d,0x1e60abe4),L_(0xc55d6e85)}, {LL(0xb10fd185,0xccf9c609),LL(0x5c6153bd,0xa0d4b626),LL(0xe398ed34,0x7b6aa811),L_(0xb4dde424)}},
   {{LL(0xa960c0c3,0x735c5d62),LL(0x8c6a5476,0x0f8b2a35),LL(0x54dfa089,0x0b186f7a),L_(0x6ab8d48a)}, {LL(0x3e7eb8a2,0xc1614c55),LL(0x7aebdcff,0x8c7e1e1c),LL(0x390e7c1c,0xf9b0b2f9),L_(0x869d2266)}},
   {{LL(0x11277d6a,0xd9cc122b),LL(0xb86417ff,0x3f2ab21c),LL(0x2cbd87e4,0x494cfb1e),L_(0x991ddd86)}, {LL(0xae7bbb90,0x9977c3ac),LL(0xc3e66760,0x397ea438),LL(0xb224446c,0x7ed9dec7),L_(0x2a81eaf1)}},
   {{LL(0xc59accd2,0xb5432f0f),LL(0xf2593b02,0x8b88a936),LL(0xc5589b93,0x02d9f9ac),L_(0xc3eb3094)}, {LL(0x8302d1bc,0x43908c95),LL(0x08ef4088,0xb05f1091),LL(0x83349c39,0x5061c674),L_(0xff815654)}},
   {{LL(0xf0350dc9,0x9371eb01),LL(0x45527229,0xce86a9ba),LL(0x47861a0c,0x9aad7247),L_(0x098cb788)}, {LL(0xa559d5e9,0x979093ec),LL(0xbbd1def2,0x0fe17efb),LL(0xb512322d,0xae7b8c8e),L_(0xe9bbc66c)}},
   {{LL(0x3ad82961,0x9f1ec261),LL(0xfe0fcc30,0x77a36e95),LL(0x3914c9d2,0x9b1a4daf),L_(0x09933bfc)}, {LL(0xec701059,0xcfa77625),LL(0xea8985c9,0xe60b3f29),LL(0x0c5b4abf,0x138b4b1c),L_(0xdc77bb16)}},
   {{LL(0x174442e6,0x1802aa28),LL(0xd075865e,0x26f71e88),LL(0x4a4c5681,0x7ae33021),L_(0x5a2cabc9)}, {LL(0x5d0cad44,0x2893d185),LL(0x55811da2,0x7e9947d1),LL(0xc9f7bf57,0xf2c79dea),L_(0x8ef92a2a)}},
   {{LL(0x3a838f3b,0xfbe1be02),LL(0x57b6d887,0x229d491d),LL(0x207a3044,0x9f62e957),L_(0x9f828177)}, {LL(0x2e1d3c54,0x86b7e5aa),LL(0xf3280135,0x3fbda02e),LL(0x7c96f0d5,0xbda44f9b),L_(0x2f723070)}},
   {{LL(0xf1aeae6a,0xae77f574),LL(0x7664fba7,0x0423c396),LL(0xf1f4d153,0x47451433),L_(0x91ff938d)}, {LL(0x3a807b63,0xe8e961f8),LL(0x748646da,0xd89c3d2a),LL(0x467e86a7,0x6ac80359),L_(0x32bf2f50)}},
   {{LL(0xb36ecce7,0x3b7324a4),LL(0x156202b0,0xbb9babf8),LL(0x98c1904a,0xf1e380ce),L_(0x46867bcd)}, {LL(0x01b08225,0x2eeb6052),LL(0x1eab01ca,0xd0d4fef8),LL(0x19322045,0x256a539e),L_(0xfc3623ca)}},
   {{LL(0xc94659b1,0x2f0568f5),LL(0x16a9b3fa,0xed8a79a8),LL(0x50d8c266,0x757c7d99),L_(0x2300fd6a)}, {LL(0x66c509e5,0xb792533a),LL(0xb5184b40,0x9e0c5aac),LL(0x34caeea9,0xe5b193ca),L_(0xbbba3e14)}},
   {{LL(0x195602fc,0x701a9997),LL(0xf834de05,0x3992cbc9),LL(0xe1e8b8e7,0x8ec536e5),L_(0x7e773653)}, {LL(0x29c79b3a,0x29bff6f9),LL(0x17984cef,0x596ff95f),LL(0x0a0ae347,0x661b4a6c),L_(0xb2e9719f)}},
   {{LL(0x8e767a04,0x473ec4b4),LL(0x4c2d676e,0xeb4d7d16),LL(0x98a8ebb4,0xc4e28600),L_(0x2f68f9cd)}, {LL(0x6beb5ab8,0x1ef6cbd4),LL(0x85cd9d62,0x83dde9de),LL(0x1762ff65,0x9ddef882),L_(0xeb7899f5)}},
   {{LL(0xe6f97e53,0x2cfac1c4),LL(0xf67b6c86,0x94800529),LL(0x61546993,0xcee96715),L_(0xf2a9560b)}, {LL(0xdc612b8b,0x01036872),LL(0x88524fe9,0x9e6d56f4),LL(0x0543bf02,0x40b8eb7d),L_(0xb87c4dba)}},
   {{LL(0xbb67b974,0xd7ceddb4),LL(0x1d6817da,0x7bd3859f),LL(0xe95a7ea2,0xc85424d7),L_(0x55d605cd)}, {LL(0x5d21362b,0x2591a7e7),LL(0x4aa1f58c,0x52817220),LL(0xc9d8453d,0xa286acd4),L_(0x3bd2f891)}},
   {{LL(0xaf830116,0x6e587151),LL(0x8a5ecfca,0x3d8da1ef),LL(0x7afbb075,0xf8e3bab4),L_(0x80131b4d)}, {LL(0x3ab8344d,0x6bb840b0),LL(0x9e591ccd,0xae7bd8be),LL(0x73177038,0xed8b2ca2),L_(0x15e01308)}},
   {{LL(0xadd91144,0x680d82d3),LL(0xd982abff,0x88f503dd),LL(0x37393204,0x746b676f),L_(0xaf201907)}, {LL(0xaf979504,0x5a9be716),LL(0x4001d634,0x0e682183),LL(0x609ab748,0xfe8299c8),L_(0x39a6e2fd)}},
   {{LL(0x193fd321,0x8682c4c0),LL(0xc2a26123,0xd566872d),LL(0x1407f3c3,0x16439d5d),L_(0x3caca395)}, {LL(0xd6726482,0xb7bc2fb3),LL(0x5cdf7fec,0xb4c2b53d),LL(0xfeaaece6,0x4349edc0),L_(0x53d16a46)}},
   {{LL(0xae9bd278,0x6c678048),LL(0xa706ff5d,0x52da3781),LL(0xae4a71b1,0x1e1cd89e),L_(0x9b61b4af)}, {LL(0x2f691fc7,0xf69884c3),LL(0x4d64cf90,0x19020a41),LL(0xdda1c483,0x77d02be4),L_(0xb0f86443)}},
   {{LL(0x8529164d,0x5d80aafd),LL(0x4f432d73,0x1e830c7c),LL(0xa9f0f09e,0xfc3140e2),L_(0x815acd77)}, {LL(0xdc2ab04b,0x59befe02),LL(0xf1e93bd2,0xa44595d5),LL(0xc1e15529,0xf19abe7c),L_(0xaeca237e)}},
   {{LL(0xa12bccee,0xb9b8ec3a),LL(0xf2dadd08,0xad1e4dce),LL(0xb63e0921,0xfcf65288),L_(0xc5a2a0be)}, {LL(0x863fe95b,0x5cd2ee3e),LL(0x4eb4db06,0x60da7b42),LL(0xb8a5bd09,0xc426f7df),L_(0xf3f19cce)}},
   {{LL(0x6e5213a2,0xfd69d7c4),LL(0x31496bf2,0xd30cf23b),LL(0x37d44e1e,0x7097147f),L_(0xae15b228)}, {LL(0x12d98f15,0x32dad5dc),LL(0xa8f63aff,0xdfe1a802),LL(0xc8d86798,0xeccd5411),L_(0xcde1acb9)}},
   {{LL(0x74006fc7,0xd7d803d8),LL(0x4af70159,0xec84dc01),LL(0xae309a70,0xa34e0e32),L_(0x585516f0)}, {LL(0x235ed632,0xcfee558c),LL(0xcab23f91,0xcd97224e),LL(0x1640e7f8,0x92280213),L_(0xbd508662)}},
   {{LL(0xc7ccf41a,0xd687e53b),LL(0x854bd98e,0x9a24176d),LL(0x01470242,0x01aae44b),L_(0xaca3a958)}, {LL(0x9fcc8bad,0xc772707a),LL(0x1ce44586,0xeeb32998),LL(0x1fd556e7,0x6b1da8b4),L_(0x0ed6e63a)}},
   {{LL(0x3fe0c228,0x929fe46b),LL(0x01aa7a0a,0x55829b86),LL(0xb0593728,0x46968b3a),L_(0xef6bc08e)}, {LL(0x861c28fb,0x49238b1c),LL(0xbb911c6f,0xe2c2f08e),LL(0xc3669826,0xd635b05a),L_(0x67f9c7c0)}},
   {{LL(0x1cbd7072,0xac172491),LL(0xa75f68ed,0x70337bd5),LL(0x5806f542,0xba62de08),L_(0xf48e9477)}, {LL(0xc789f89a,0x778e6f6e),LL(0xca466063,0x7cb61f89),LL(0x9e11ca0e,0x54802afa),L_(0x00753b33)}},
   {{LL(0xa293c1d0,0x98d81eae),LL(0xf1d86049,0x4a05fd64),LL(0xf7776c88,0x31edd830),L_(0x4b09f540)}, {LL(0x6b003742,0x530442c5),LL(0xd29f0459,0x2717506e),LL(0x993316f5,0x977de534),L_(0x8efd4242)}},
   {{LL(0x1f4b647c,0x63a5f62f),LL(0x1b839fad,0x184bd1bf),LL(0x9eb8bb02,0xd878238c),L_(0x8ce0fbd0)}, {LL(0xee715bb5,0x0c09795f),LL(0x7e0f0890,0x4a0ef110),LL(0x5e80a1d2,0xa6ed70be),L_(0xdd973aaa)}},
   {{LL(0x02e27464,0x66b5fa13),LL(0x8d608579,0xf1afa0fe),LL(0x262f06e4,0x55328bec),L_(0xe5f137a6)}, {LL(0x359db889,0x56654a88),LL(0xd39b145c,0x31866e5f),LL(0xe08705c4,0x615a5b05),L_(0xe8573a34)}},
   {{LL(0xd9bae78a,0xa18e5f26),LL(0xbcdd1733,0xfb66461e),LL(0x52d89638,0xf79de1ee),L_(0xd8be4f6d)}, {LL(0x63f080c5,0xf0c1209c),LL(0x4e984349,0x8927edaa),LL(0x1cca87fa,0x0d7779de),L_(0xc4be69f3)}},
   {{LL(0x4ffe196d,0x591b255f),LL(0xf9220e24,0xf529da70),LL(0xa919472f,0xc172114d),L_(0x12b84dab)}, {LL(0x727df75e,0x0b0bba01),LL(0xc90f5813,0x18ae3fac),LL(0x7de531b5,0x60f6010b),L_(0xb124c090)}},
   {{LL(0xc8ab7f3a,0x4d9c100a),LL(0x3147c457,0xb3ed07de),LL(0x2ce9d1a7,0xf4f877cc),L_(0x11395ce0)}, {LL(0xe5d2ab31,0x7ebabf38),LL(0x7ffb38b5,0xce3cfa6e),LL(0x85ffa6d1,0xfa3306d0),L_(0x40ce1960)}},
   {{LL(0x014e69fb,0xe40ec6e7),LL(0x92b1fc6d,0x8f04f43e),LL(0xf3db5670,0xb9883ee7),L_(0x01e865eb)}, {LL(0xeab3fc5e,0xedf8720a),LL(0x639d496f,0xf1d8e66c),LL(0x32cbb568,0xc1af6691),L_(0x0ce5c8b4)}},
   {{LL(0xb78b642b,0x65a42ec3),LL(0xade40c55,0xdde23958),LL(0x216642c3,0xcdc7c4de),L_(0x6aa44ccd)}, {LL(0x82ee3789,0xec771036),LL(0x1fafeef8,0xefdcbefc),LL(0xe82b1033,0x5b9fd0f1),L_(0xb5931c71)}},
   {{LL(0xd7c8b169,0x94661f99),LL(0x88be16da,0x968a0513),LL(0x5e1f223f,0xf2936613),L_(0x7a916253)}, {LL(0xf109d748,0x8c1958b8),LL(0xca3eb1cf,0xd1baaa06),LL(0x58b0e5bd,0x04265560),L_(0x6b971198)}},
   {{LL(0x3c1c78f3,0x73aa994d),LL(0x7b9ba5eb,0x9d13c878),LL(0x5ffbf962,0x329365bf),L_(0xfc14c225)}, {LL(0x151d1b5c,0x9b162681),LL(0x8b322eba,0x67d4adbd),LL(0x2d7119cb,0x112111da),L_(0x14cee07e)}},
   {{LL(0x8de63a53,0x0bc054e9),LL(0x090ebf32,0xac3474f8),LL(0xd39c2e22,0x8a24d244),L_(0xc5c07687)}, {LL(0x72de574a,0xf7cd28ca),LL(0x00f6286c,0xaba378ca),LL(0xacf3d959,0xc4c178a4),L_(0x1e3ce962)}},
   {{LL(0x4c196a72,0xeddb3096),LL(0x4807ba1c,0x78d05007),LL(0x4a5cea44,0x4e82cf55),L_(0xbefd7586)}, {LL(0xbc157856,0x42628bb0),LL(0x4d8a1ece,0x3bec9a03),LL(0x414298ee,0x71c76fb4),L_(0x1b7c9c7e)}},
   {{LL(0x451c89a9,0x0eb5ca81),LL(0xdfcc6c7e,0x85231811),LL(0xdd102f79,0xcfe70175),L_(0xd23927f1)}, {LL(0x74f56210,0xa68f72ba),LL(0x0c268cfa,0x107305c7),LL(0xa0b90dac,0x71b34217),L_(0x86344588)}},
   {{LL(0xdf55e61f,0x74013fe7),LL(0x0ad28f61,0x384b1c58),LL(0x117f4d5d,0xb417f0cc),L_(0x2ee194ab)}, {LL(0xa5e101d2,0x6a59354e),LL(0xe4650ef7,0xdd613a28),LL(0x3d053f50,0x49698601),L_(0x211980e7)}},
   {{LL(0xbc28da29,0x350510df),LL(0x729451ca,0xa5492aa2),LL(0x488b3ebb,0x7335580b),L_(0xf1e1f254)}, {LL(0x150fa7dc,0x5080e530),LL(0x021a9cf2,0x41fe0adc),LL(0x151c555b,0xefba1701),L_(0x1be676fd)}},
   {{LL(0xda6ab8ee,0x01c286ca),LL(0xd11e769f,0x8f870f9f),LL(0x67983e93,0x011517ba),L_(0xb19fd8df)}, {LL(0x2ffdf94a,0xb8ff2b68),LL(0x1becdb97,0x57a7f08b),LL(0x2df51505,0x7f4a552d),L_(0x2e44b40c)}},
   {{LL(0x8a38d901,0x15f78485),LL(0x94822c36,0x40936fbc),LL(0xdc403083,0x26d9bf80),L_(0x3f811b6c)}, {LL(0x6682494b,0x658eecc3),LL(0x1088923f,0xe2c9cc47),LL(0xf4a7dc3b,0x6742c074),L_(0x7820af17)}},
   {{LL(0xaf6f6f62,0x1be73cf3),LL(0x0c32e84f,0x74ffb428),LL(0xb29ede9a,0x92e686e4),L_(0xa74c4c0a)}, {LL(0xdfa5e890,0xe1001a22),LL(0xa64183c0,0xf029cdcb),LL(0x55c88ddd,0xd03747ad),L_(0x5bbce6dc)}},
   {{LL(0x69bc7a5c,0x0a7f469b),LL(0x3d28b7fa,0x8cb61ff4),LL(0x860090cf,0x25e6acf5),L_(0x3ca61e58)}, {LL(0x7c9a5e59,0xd04ea633),LL(0xbdcdff22,0x3dccdf34),LL(0x679b9a9c,0xe0d5601a),L_(0xa518db92)}},
   {{LL(0x75393ffb,0x503ff7fc),LL(0x29c7bd8b,0xad84a980),LL(0x6a8d3ebd,0xb711116c),L_(0x6093e9c1)}, {LL(0x508cd2db,0x298e39d1),LL(0x88ffc5c3,0xd182ffce),LL(0xf35bcd9e,0x56f60d5a),L_(0x24c2d62a)}},
   {{LL(0xbb37476d,0x126c6b62),LL(0x1974758e,0x12e961ad),LL(0xc23fa1b7,0x620a85d2),L_(0xe713a9df)}, {LL(0x18d50f3d,0x90b6bf4d),LL(0xd31ffadf,0xcce6cb43),LL(0xb9db57be,0x43232f2b),L_(0xee956b0f)}},
   {{LL(0x33f18557,0x34242913),LL(0xae8e0c9c,0xecfa2ca9),LL(0xe5298223,0x54c9b89f),L_(0x81ae98bb)}, {LL(0x2bb85e08,0x3640c336),LL(0x187eb8dd,0x23f72dcc),LL(0xe33bfd7f,0x1138d09f),L_(0xa5c71157)}},
   {{LL(0x9635cec5,0x99decb38),LL(0x120fcaaa,0xb6e94853),LL(0xc90493fd,0xe03d588e),L_(0xdacd4c22)}, {LL(0xbbcd5ead,0x1a5efdc4),LL(0xa034c92d,0xbd3e5f34),LL(0xc0bbee30,0x91354a2e),L_(0x83ce5653)}},
   {{LL(0xe4fe7dd6,0x40ccc046),LL(0x12391a03,0xef734198),LL(0xe37fbe0f,0xd7d730ce),L_(0xb901d8d1)}, {LL(0xe81b74b5,0xccbbf03c),LL(0xbd70b08b,0xaf1eca00),LL(0xb202f7d1,0x6c3a5371),L_(0x3e4aeff1)}},
   {{LL(0x3c293b50,0x50ebe82c),LL(0x841f67d5,0xe5c199da),LL(0x9c0164e0,0x09ae68d1),L_(0x48a6a776)}, {LL(0xe57a6b35,0x764aa682),LL(0x4cc646a6,0xd2b124d2),LL(0xc37e19f6,0x0684de69),L_(0x244cccbe)}},
},
/* digit=20 base_pwr=2^140 */
{
   {{LL(0x3a2fc0af,0x3a0ae219),LL(0xaf66dcd1,0x71d39324),LL(0xeb53d991,0x1b364d9f),L_(0xd1a4ab2f)}, {LL(0xe634fa4c,0xfa979c3b),LL(0x0ae7194f,0x0315e213),LL(0x7a46c430,0x7d3dfc00),L_(0x3dda16de)}},
   {{LL(0x69c6dab5,0xcad74055),LL(0xb4650e32,0x5f980b30),LL(0xf457bc61,0x08ab6cd4),L_(0x08a5060d)}, {LL(0x5a3b5bba,0x4a0eec60),LL(0xc2d75bdb,0xa4e2c616),LL(0xeb13a610,0xea8ba726),L_(0x5b8d1d6b)}},
   {{LL(0x8208d95c,0x1d126957),LL(0x852ba024,0x38314570),LL(0x2b3db8bc,0x68f43b86),L_(0x4e0c61da)}, {LL(0x2fa1baa8,0xb3650b8b),LL(0x81c43c97,0x222d46b4),LL(0x5ab609c3,0xe25b7f78),L_(0xf9adc44c)}},
   {{LL(0x2601c0c3,0x10c7e593),LL(0xa7b8c442,0xd433020b),LL(0x163a4b4f,0x6ea15982),L_(0x0f63c4b6)}, {LL(0x559abf4c,0x168cc6db),LL(0x63fa3ddc,0x5e92d42d),LL(0xcad8eab0,0x0f718cd3),L_(0x285e6621)}},
   {{LL(0x1eccb2de,0x45eee5ae),LL(0xbeed1a3a,0xf0f806c2),LL(0x93a2e216,0x0b896b9f),L_(0xaddd1455)}, {LL(0x21ac6aa7,0xe860a374),LL(0xbb52f5cc,0xa6d03d5e),LL(0xc336e305,0x5a3ad08c),L_(0xd8ac8cf1)}},
   {{LL(0x7e65ba98,0x927ff44f),LL(0x6eb2af1f,0xcfc2ebd3),LL(0x8e5841d8,0xcd919ed2),L_(0xfaf6b44a)}, {LL(0xd72479fb,0xf9079703),LL(0x31c04f62,0x11b2590d),LL(0x12b5f89b,0x9e78739c),L_(0x829845bb)}},
   {{LL(0x6497199a,0xcc5dc1ba),LL(0xc6068072,0xc7f1ee3e),LL(0xbbfeebe2,0xa475b47a),L_(0x497220e3)}, {LL(0xd00f3343,0xaa2739d6),LL(0x7a34054f,0x6bcba57f),LL(0x08421638,0xf50c5c89),L_(0x5194e599)}},
   {{LL(0x56ea720f,0xc14eb574),LL(0xd278e415,0xe5e93b7f),LL(0x1a141ce9,0xf206da59),L_(0xde031e92)}, {LL(0xff6ca107,0x85cacb30),LL(0x27ecddf0,0xe54c35f6),LL(0x73777848,0xeecb8a06),L_(0xbffbbc51)}},
   {{LL(0x27c35c5b,0x88bff7c0),LL(0xfa52e107,0x2aa31cbc),LL(0x84083f2a,0xee8007b7),L_(0xd8f67690)}, {LL(0x7c2ce5dc,0x46d1faf9),LL(0x5603d9f9,0x0d96af5f),LL(0x7894ccc6,0x143733e6),L_(0x28c9191f)}},
   {{LL(0xf8d1a77a,0x74b4b8c0),LL(0x4ced8b77,0xaa226357),LL(0x0f677dc4,0xbce88332),L_(0xc71a3df5)}, {LL(0xa1eff8f9,0x6f4d4505),LL(0xf4ee4e98,0x37fa6b4f),LL(0x12414006,0x078af83c),L_(0x51f9c8c5)}},
   {{LL(0x6038786b,0x043fa7bc),LL(0x495de129,0x39594053),LL(0xeeffb86e,0xf30130d5),L_(0x5566ee1e)}, {LL(0x8e9bddbc,0x6b822690),LL(0x38bca4e0,0xfd18acdc),LL(0xf63c9ead,0xa29c50ef),L_(0x6ca8c69e)}},
   {{LL(0x875129d9,0x23706eff),LL(0xbd7972c9,0x343d01b3),LL(0x0f0278ce,0xd1d2ff3f),L_(0x859707ee)}, {LL(0x62e80eee,0x76f993b1),LL(0x46b5aa17,0xd8eb588f),LL(0xc6cb2916,0x3e93bfcf),L_(0x6b823d1c)}},
   {{LL(0x40dc4150,0x02fc59aa),LL(0xa4e36005,0xf4b7afee),LL(0x2f863790,0xdf1eb2ce),L_(0xdcdc305f)}, {LL(0xc2be20ae,0x05119c9a),LL(0xb011c039,0xa27da69c),LL(0x59ba9dd2,0x50be3370),L_(0xd25c56ab)}},
   {{LL(0xe5b817ad,0x62b0b4a9),LL(0xe35b330d,0x0b744a17),LL(0x49eb11c0,0xdb8066e2),L_(0x8fd0b5d0)}, {LL(0x60bc3f3c,0x9db8094e),LL(0x0164c3cb,0x1a336970),LL(0xe5728d57,0x98764bb8),L_(0x9ecea57f)}},
   {{LL(0x471336b1,0xc97e0c25),LL(0x84d2cb69,0x40016340),LL(0xd00385b4,0x8681b939),L_(0x5ce0fa5b)}, {LL(0xfb0504e7,0x190db5e8),LL(0xec267c2d,0x73d7879b),LL(0x5e7d8612,0x8f037a48),L_(0x0c242e37)}},
   {{LL(0xb920e08a,0x81a11e46),LL(0xab88a28a,0xad7ce91b),LL(0xd873ff38,0x463f5eef),L_(0xb449cd10)}, {LL(0xa997a44b,0x67e4b48b),LL(0x65bf00de,0x76b8b69f),LL(0xc007a35a,0x0ebb0cc7),L_(0x168c4cb8)}},
   {{LL(0xc6331e3e,0xd49fdc60),LL(0x08277aa2,0x3f1c848b),LL(0xae53c6d2,0xa68e50ed),L_(0xf9940a04)}, {LL(0xa6271229,0x1c7298ae),LL(0xc7e75832,0x34a3a581),LL(0x6666c91c,0x1bb3ce87),L_(0x9b5e6600)}},
   {{LL(0xabcce7ac,0xfa1beb7d),LL(0xc3326260,0x9cab965d),LL(0x7c86ce27,0x145e311e),L_(0x1c036012)}, {LL(0x3f4563cb,0xbd40ee7b),LL(0x7397bad6,0x077a6fa1),LL(0x6ab23a8b,0xdc9e2258),L_(0x524c8593)}},
   {{LL(0x27ccdc45,0x8d1e93bc),LL(0x8122aeec,0xac370974),LL(0xf5fc7d6a,0xbb95cf97),L_(0x1eb1cdc1)}, {LL(0x65e7b71f,0x0d3d8936),LL(0xa618f61f,0x6cfc89cc),LL(0xcfd74f9d,0x46c8cccc),L_(0xc39ab4da)}},
   {{LL(0x82c35cad,0x4b6e4cd2),LL(0xb8deb535,0x5bf959ed),LL(0x77b74152,0x14153467),L_(0x1a6510e5)}, {LL(0xc645fe15,0x44ed7180),LL(0xd0e5fe61,0xb00a2c9e),LL(0x94c68e2e,0xcd40eadd),L_(0x54367d74)}},
   {{LL(0xdb397bc6,0x0bb14c2d),LL(0x90f3c232,0x2727c32a),LL(0xbdde6547,0xf5120538),L_(0x6f363a14)}, {LL(0xca65a20d,0xb71dac7a),LL(0x0119f615,0x7b4ac903),LL(0xf1e5f556,0x83bcccf1),L_(0x78b79fc1)}},
   {{LL(0x35177eb9,0xb2a6346d),LL(0xba547ec6,0xcdb1c427),LL(0xc5c607ff,0xf6acdcca),L_(0x272a5f1e)}, {LL(0xf198276f,0x5ddfa215),LL(0x187de1bb,0xc1a3402a),LL(0xc070af80,0x0cf324a8),L_(0x8fba9bd5)}},
   {{LL(0x4be4c6c1,0xddf4319c),LL(0x90adc217,0xcc96208a),LL(0xd1d87cd0,0xc3e2cfee),L_(0x76bd76ab)}, {LL(0x1fa3dfb2,0x89c310af),LL(0x72ee71cc,0xab955eb2),LL(0x1e3849a6,0x4e9da2a4),L_(0x49c4cd74)}},
   {{LL(0x4a409b87,0x4a32be16),LL(0x591f45c9,0x5593d29c),LL(0x1f66b255,0xa18a71de),L_(0xa8df9f2a)}, {LL(0x871121b9,0xd5a52a49),LL(0x0518891a,0xdb77ca39),LL(0x35178007,0xe87f7b56),L_(0x236d66c4)}},
   {{LL(0x7b11bf26,0x78cced92),LL(0x100d41cd,0x568cf974),LL(0x1aebc2f4,0x90437303),L_(0x4f4abe7b)}, {LL(0xe1d4cf61,0x81708f04),LL(0x6b6039fa,0xfcd5e395),LL(0xe8e310f1,0x67164e9b),L_(0xc7f1e7ae)}},
   {{LL(0x4435eb40,0x309bf5d1),LL(0x2117b17c,0xfa094c38),LL(0x0f0f519b,0xa7712b6a),L_(0xa339b91c)}, {LL(0x13b443e9,0x3fc80f73),LL(0x5eca3c74,0xc2640658),LL(0x6f1b036b,0xcc979824),L_(0xf9c46f37)}},
   {{LL(0x2c10fe83,0xfd4280b4),LL(0x61dfd82b,0x0f01bc50),LL(0x8ebbdd0a,0xe8a94761),L_(0xea097594)}, {LL(0x3331db36,0xe9e32c12),LL(0xe3fdca90,0xdfdd390f),LL(0x7d9acfa0,0xf46a833d),L_(0x0a373092)}},
   {{LL(0xb1f7d641,0x50d5c66d),LL(0xc56a23dd,0xd311bbb7),LL(0x2df06964,0x4d4065dd),L_(0xb24fc29d)}, {LL(0xc5f2e7e8,0xa84be5a0),LL(0xee36de4b,0x704fb714),LL(0x72ab97c4,0xe211a72c),L_(0x4e457419)}},
   {{LL(0x22e459d1,0x262ef282),LL(0x57520b17,0xcdae125f),LL(0x400cd2e5,0xc83f210b),L_(0x219146d8)}, {LL(0xb3e5fe6d,0x87c8591a),LL(0x2cdbe07a,0x83fd93d1),LL(0x072fa5b0,0x7d701f58),L_(0xe67d391b)}},
   {{LL(0x2dda42fc,0xa6490705),LL(0xbae6af5f,0x5e3ec835),LL(0x974661b3,0x77bc552c),L_(0x51a8bdf3)}, {LL(0xf43d8648,0x351407f7),LL(0xfaacee32,0x93522ec0),LL(0x2effc179,0x49a92d02),L_(0x35e7a36b)}},
   {{LL(0x1178f9fe,0x88ddd25b),LL(0x29a04cd6,0x4985102d),LL(0x0d3de452,0xf0277afe),L_(0x52955ade)}, {LL(0xf3571745,0x53a80f8c),LL(0xcd0ea6e7,0xa11d84cc),LL(0x8cf3c9a4,0x7920692b),L_(0x66ad31de)}},
   {{LL(0x3259f508,0x8dd2cea1),LL(0x147b087a,0xb1de59c3),LL(0x964e43aa,0xd9cd158a),L_(0xe62a2c56)}, {LL(0xf7812a2a,0x483a8ca3),LL(0x133f5131,0xf93b681f),LL(0xeafa2982,0x22d0852e),L_(0x47016331)}},
   {{LL(0x4e3898e8,0x178b364e),LL(0xdedbf0d6,0x1e46cbf2),LL(0x07005a6a,0x017cc249),L_(0x3350187e)}, {LL(0xe3d63871,0x9df8269b),LL(0xa447b67a,0xf32797a4),LL(0x1360de40,0x70a39a32),L_(0xe07cb8f9)}},
   {{LL(0x58005eff,0xefaea42e),LL(0x83a181cc,0xcdf53ee0),LL(0x8d9f77b8,0x2be3de21),L_(0x199ee148)}, {LL(0x219fc4d7,0xdfbb4054),LL(0x5fd887bc,0x370b0057),LL(0x0df48ecb,0x67c50061),L_(0xceadf6f4)}},
   {{LL(0x7d760757,0x3cb96c10),LL(0x8354395a,0x87378266),LL(0xc87eb4e6,0xbb1b1686),L_(0x2ca2a9d8)}, {LL(0xf625cc5e,0x7f6f6bfb),LL(0x375d90fd,0xd1702bfc),LL(0xd7459a05,0xb5757892),L_(0x8e648246)}},
   {{LL(0x2285b7c6,0x77777ad0),LL(0x98514815,0x2a9f4626),LL(0xa5cb03f7,0xbdaaa412),L_(0xf84efd29)}, {LL(0x6fb99eec,0x8aa2c4b5),LL(0x9cd6f177,0xec729246),LL(0x9b9f8d87,0x995887b7),L_(0x2944bcdb)}},
   {{LL(0x786ccb7d,0xae2ffd66),LL(0x8f43b748,0xb566147f),LL(0x0b294654,0x42cc5e69),L_(0x8eb5ab4c)}, {LL(0x1634f1db,0x21f0caa7),LL(0x3cc24e4e,0x80410ce5),LL(0x6597b349,0xb9e50c3a),L_(0x390fd5fc)}},
   {{LL(0xbefea27f,0xca541338),LL(0x25552e16,0x0bc9bd26),LL(0xd300e1fc,0xed2c5b84),L_(0x2e61d461)}, {LL(0x731c6b5e,0x10fed3e2),LL(0x3f6fedc9,0x6d0357ab),LL(0x30c7c4b6,0xfdf04b77),L_(0x6ca06d8d)}},
   {{LL(0x6198d0c0,0x49f0ac87),LL(0x355910d8,0x5908bfa5),LL(0x2b72f231,0x1e0626b8),L_(0xcd397171)}, {LL(0xb70c5fde,0xaffd0d74),LL(0x4feccf28,0x30464156),LL(0xbe7c0da7,0xf7d2ffb9),L_(0x782a9bb8)}},
   {{LL(0xd71029f7,0x2ac8267b),LL(0x58c7cc85,0x4f8afc3b),LL(0x2d787218,0x10f9a786),L_(0x005ab707)}, {LL(0xf19c8b55,0x5209c3c7),LL(0xa8e944c7,0x0212109d),LL(0x4e16a382,0x3cd77b54),L_(0x4ca72d58)}},
   {{LL(0x4e89b492,0x77501651),LL(0x30cb8e0d,0xdbb5243c),LL(0xd24a6908,0xc09c5df1),L_(0x10b46bf0)}, {LL(0xeb16d02e,0xb28822ac),LL(0x790e9a17,0xbf46b5ba),LL(0xde8a18e1,0x47566077),L_(0xd3c60b3c)}},
   {{LL(0xa4a4f17a,0xdff0cf1f),LL(0x0aef91c0,0x34f0c71a),LL(0x3fe5a3b7,0xae1fff30),L_(0xf6c8d6d5)}, {LL(0x95b01983,0x5317a578),LL(0x10e5c9f2,0x8872e118),LL(0x154f511a,0x28ad7f46),L_(0x1f7f1f26)}},
   {{LL(0x3c4162c4,0x88636424),LL(0xf620131a,0x69f32e24),LL(0xf3a9fd92,0x049b5c0f),L_(0x6242ff79)}, {LL(0xd8613d4e,0xcf94b297),LL(0x0250d486,0xf99f0a15),LL(0xf6d69ab3,0x8d722881),L_(0x726c3c6f)}},
   {{LL(0x2829f721,0xc6f3cc19),LL(0xf0ba6d0a,0xf0c5f76a),LL(0x0d44b294,0xda7df6fd),L_(0x98458d66)}, {LL(0x35516899,0x4c9c487f),LL(0x541b8cf7,0x312edd80),LL(0x14a64ecf,0x2db649b2),L_(0xfcc41f23)}},
   {{LL(0x4e65d481,0x17bb4dfc),LL(0xd8c4ee32,0x7b8827b1),LL(0x6ffab99a,0x788afe5d),L_(0xb6cc5390)}, {LL(0x6ed82d81,0x75adf2d6),LL(0x91030ac7,0x99ce22b0),LL(0x957976c9,0xf8f37372),L_(0xeae6d45f)}},
   {{LL(0x3ad3db42,0xa4617ca1),LL(0x9ea768ee,0x8f228cec),LL(0xc881f9da,0x86ddd01a),L_(0xddfb0090)}, {LL(0xd1467d75,0x41eb80aa),LL(0x1320c7b6,0x875ce5dd),LL(0xaf17e700,0x398d4c2f),L_(0x91ca1818)}},
   {{LL(0x165575ce,0x7c2f636f),LL(0x94407a75,0x89f1290d),LL(0xeeb00b1e,0x2cccc148),L_(0xc1a654f8)}, {LL(0x7200e28d,0x5409cce2),LL(0xd579aa30,0xf8e920e0),LL(0x3586fdc4,0x11976312),L_(0xd9b1a394)}},
   {{LL(0x45934ad1,0xa0a5bb1b),LL(0xba2becd7,0xbcff29b1),LL(0x9f749e30,0x2f73a7d6),L_(0x54ae60bd)}, {LL(0x9ed5bbc9,0x111b6548),LL(0xae179cab,0x7b77f3f8),LL(0x9b568171,0xf1563350),L_(0xd57cc18c)}},
   {{LL(0xbde3fb89,0xaa153efc),LL(0xe84063ac,0xf24dc768),LL(0x91718221,0x3207691f),L_(0x666ced3f)}, {LL(0xa962b884,0x8a1416f3),LL(0x81f644db,0x2992be1f),LL(0x388328fb,0xa7c4a359),L_(0xebba06d7)}},
   {{LL(0x709d7f63,0xf678058c),LL(0x9bd30551,0x9db19686),LL(0x8d882bae,0x81afb521),L_(0x6d103aa5)}, {LL(0xce4ef2a5,0xc6ecf132),LL(0x681bea3a,0xab9bbd78),LL(0x3831d678,0x4c11a697),L_(0xb9d4a72d)}},
   {{LL(0x8a72049d,0xc3e5138e),LL(0xa25affde,0x9235657e),LL(0xf870194a,0xeddbefb7),L_(0x2583cc8f)}, {LL(0x85d843d9,0x912ed2a5),LL(0x57849cc4,0xdb45568e),LL(0x453cbf16,0x410259cc),L_(0xcf9369cf)}},
   {{LL(0x10263ac8,0x69fa2805),LL(0xf7446e45,0x6120289a),LL(0x86d5285d,0xece730a0),L_(0xc0ee3533)}, {LL(0xbf1197bc,0x286f4724),LL(0x878ea926,0x0b0ef425),LL(0xf397181d,0x24bb4f4f),L_(0x542c7100)}},
   {{LL(0x162331a9,0x536a5ea6),LL(0x0e466905,0x88c29411),LL(0x3da5a14f,0x9cb76097),L_(0x1db565b9)}, {LL(0xef6b8094,0x59e86f3c),LL(0x31eb3c89,0x817764eb),LL(0x58eed1b6,0x8010fc3d),L_(0xaeba318a)}},
   {{LL(0x9a5b60b8,0x108b3068),LL(0x1ffea59a,0x1312c3e1),LL(0x8dd4b81a,0x0253e9dc),L_(0xf4ed9d4a)}, {LL(0x5eadccd2,0xbd75013a),LL(0xad3eaf49,0xc60daec9),LL(0x20b2a500,0xbea0d3fb),L_(0xdac00c65)}},
   {{LL(0xfb3fb2a5,0x72db2c7a),LL(0x566902bc,0x864ce0fe),LL(0xe0956799,0x2d7ec9e2),L_(0xdfb7dad8)}, {LL(0xdfc7056f,0x77bfd4dc),LL(0x7e188701,0x0f42ffde),LL(0x110fb370,0xf2040e52),L_(0x7944c336)}},
   {{LL(0x2100e0bb,0x6fd68890),LL(0xf771684b,0x67978ca7),LL(0xb3a94049,0x279171ad),L_(0xc7c765dd)}, {LL(0x02304590,0x41165c1f),LL(0x0299e83b,0x8eadb8c9),LL(0xe9eaef42,0x2345e24e),L_(0xb4a1f1f3)}},
   {{LL(0x41f15f2a,0xaa889c09),LL(0x2cb037eb,0x6adf563e),LL(0x559edd7a,0xdd414cb0),L_(0x601d6943)}, {LL(0xad8084b7,0xbef296d5),LL(0xb70a236e,0x0ee342a9),LL(0x074068ff,0xc9f3a4e6),L_(0x7b2eab9c)}},
   {{LL(0x4db1a4b4,0x983e29af),LL(0x50522f93,0xc26d69a6),LL(0x8e174c76,0x2739f19a),L_(0xc8bc9b25)}, {LL(0x4c902614,0xcd082b89),LL(0xa5f7fd65,0xa0661a39),LL(0xd0352acc,0x6d5bd738),L_(0x15c9e55c)}},
   {{LL(0x0f46deb3,0xe05a9d3e),LL(0x25c49119,0x68d899f5),LL(0x2fcbd485,0xd426fbf8),L_(0x94e9a11d)}, {LL(0x70b0b8e8,0xbb075381),LL(0x21ac6e8b,0xeb323221),LL(0x933c7339,0x90f21e83),L_(0x03161725)}},
   {{LL(0x65c3ea70,0x26f5292f),LL(0xbb26aa3c,0x2f10ff0c),LL(0xd0e0dfce,0xe59106aa),L_(0x5bc81388)}, {LL(0x334ef2fc,0xe1d13288),LL(0xe91c8aea,0xb2dd6c7f),LL(0xe941b704,0x428af6d0),L_(0xb82ceae2)}},
   {{LL(0xf4357fa7,0xe9634ed9),LL(0xea606688,0x92f270b1),LL(0xb8fe4e04,0x4ce90044),L_(0x53c7eb07)}, {LL(0x9137e5a7,0xb5cf536c),LL(0xc36a01d3,0xaa1f4551),LL(0x0024f90b,0x981784b9),L_(0x3bab3f51)}},
   {{LL(0xb04608df,0x5269aa1a),LL(0x0cf9cff7,0x8ce1ebee),LL(0x3d279465,0xc66381e8),L_(0x61b7ab00)}, {LL(0xc4fa7ed7,0xdeb29303),LL(0x6b7bf135,0x510a3c8c),LL(0x2ce166ff,0xaa227120),L_(0xf4a128bb)}},
   {{LL(0x992398c7,0xa3c32177),LL(0xb5766593,0xcd19850c),LL(0x70b1a326,0x9505296f),L_(0xe3747c6e)}, {LL(0x41bcad27,0x0d11515a),LL(0x290e370c,0x04883b49),LL(0x5a5bf602,0x72d1e180),L_(0x6a7dda52)}},
   {{LL(0x5898ed4f,0x63896a2f),LL(0x0341c2af,0x82a0ed92),LL(0x71349776,0x71b9b0ee),L_(0xa68523a0)}, {LL(0xd360630d,0xbd440771),LL(0xc3982209,0xa49907d9),LL(0x28e79328,0xad498ec7),L_(0xde9039d6)}},
},
/* digit=21 base_pwr=2^147 */
{
   {{LL(0x5e338a95,0x0324aba6),LL(0x3d1721e0,0x970769c2),LL(0x32ea5aeb,0x82c3b64b),L_(0x62043072)}, {LL(0x80346bd2,0x98111ddd),LL(0x629d0540,0x2ef9e31e),LL(0x75819a03,0x6e0e0fd4),L_(0x01541785)}},
   {{LL(0xd481d37b,0x676ef7cb),LL(0x6e38c822,0x6ddd542a),LL(0x4bcdf56d,0x0807b6ee),L_(0x152d3a20)}, {LL(0x99cfbaa2,0x440a4586),LL(0x1b803f58,0xaab76f82),LL(0x62bde53d,0x31ca293a),L_(0x774dabf4)}},
   {{LL(0x279a665e,0xc73f3e56),LL(0xd125dbe9,0x51cce02a),LL(0x925d55fe,0xba48033b),L_(0x77120d9c)}, {LL(0x3c3cd000,0x09219d1a),LL(0x9ef0977f,0x7bba7cbf),LL(0xf5c59984,0x1f9a5c08),L_(0x043d05c2)}},
   {{LL(0x4079f18a,0x150e6d32),LL(0xfb5764ea,0x8b6cc92a),LL(0x2d425400,0x274ec0c2),L_(0xee8a4646)}, {LL(0x6d261b02,0x4ef086f3),LL(0xf2d17e5e,0x29dec1b8),LL(0x7f963f71,0xea97e94e),L_(0x70fad827)}},
   {{LL(0x9e51713a,0xf9867b99),LL(0x07c55712,0xbd760d63),LL(0xafdf07e9,0xa9b6e9ec),L_(0x2af4df6e)}, {LL(0x6d61dc08,0x93df1b2e),LL(0x743dfab5,0x3387d03a),LL(0x32d7f165,0x759fb34d),L_(0x749bb511)}},
   {{LL(0x08345265,0x34f3b1e1),LL(0x599ac0d2,0x20468106),LL(0xc8425e31,0xa035ed92),L_(0xbe2663eb)}, {LL(0x15129a67,0xd78546b5),LL(0x6589a68a,0x2a03f31f),LL(0x04faf0ec,0xf88d60ed),L_(0xe6fe6524)}},
   {{LL(0x99a20d9b,0x31f03c7f),LL(0x93d3e67b,0xbdab0f5d),LL(0xcfc1004b,0xc665b5f0),L_(0x3425e2f4)}, {LL(0x57bee517,0xbaab1072),LL(0xf467b795,0x64de4d9f),LL(0xa4efa899,0xb3c284d3),L_(0xdb30b893)}},
   {{LL(0xb6ef6e50,0x7bd6b2eb),LL(0xe6144c26,0xb87b9f50),LL(0x3506344d,0xe38cc59c),L_(0x136536d0)}, {LL(0x98614bb5,0x4daab44e),LL(0xb52c6949,0x3957de52),LL(0xc680567a,0xcfa42f96),L_(0xf20dca71)}},
   {{LL(0xad09cde7,0x89749594),LL(0xf7758fa7,0x686f22a7),LL(0xe7736848,0xd741b21a),L_(0x916557d5)}, {LL(0x08ab4ccf,0xb87c7bef),LL(0x6a38311f,0x982331fd),LL(0x177b6473,0x63140d6e),L_(0x61df2b77)}},
   {{LL(0x061731b0,0x785b32f1),LL(0x92de60e5,0xc296cafa),LL(0x1de52e72,0xb10ce663),L_(0x4c01f4cd)}, {LL(0xc6a06419,0x50016567),LL(0xdcbb4a2a,0x4502900a),LL(0x2c001a59,0x450b28d5),L_(0x3a580e42)}},
   {{LL(0xc5913491,0x9e1e0ce7),LL(0xfef505e8,0xc5f9091a),LL(0xe7590273,0x44739896),L_(0x35472c73)}, {LL(0x449da237,0x3913a8d2),LL(0xbf43ca41,0xd9db43d7),LL(0x0204d37f,0xbcc28709),L_(0x11a4ed34)}},
   {{LL(0x0ba00095,0x00c25d60),LL(0x00ebff6a,0xb0ccb9b4),LL(0xaecf0812,0xebecd41b),L_(0x02599546)}, {LL(0xe2cb6f9d,0xe9f56be8),LL(0xc9f2f401,0xab12b175),LL(0x77e2c3e0,0xa7a303fc),L_(0x6a047f0f)}},
   {{LL(0x0ca6bd42,0x829f1bab),LL(0x4982f7c1,0xf8aa85f0),LL(0x63a37ba6,0xe64c6ed3),L_(0x55b4405c)}, {LL(0x9ba22c57,0x14c74bf2),LL(0xd812115d,0x57cdabdb),LL(0xa2f36ff4,0x69603458),L_(0x4e2488fd)}},
   {{LL(0x049ce7fa,0xdda045cf),LL(0x3c6cc5ab,0xae642723),LL(0x444a4ca8,0xc9e8efd0),L_(0x2e1a3d2a)}, {LL(0xee8ff913,0xa5f0a182),LL(0x3921bdba,0xf34c0a0e),LL(0x0bdcdecc,0xf174835c),L_(0x438b3f48)}},
   {{LL(0x262941ff,0x50dd89e9),LL(0xb28cdfa5,0x52bfaad5),LL(0x3d3651a5,0x42da9f6d),L_(0x52955126)}, {LL(0x72f24772,0xfb5c67bc),LL(0x10d91fd9,0x6f3ebdef),LL(0xa8074e96,0x5d95664e),L_(0xe7fca6c3)}},
   {{LL(0xfbd1a192,0x14bfa4e1),LL(0x11f314f6,0xf38ce345),LL(0xfcdb2abd,0x45ebd21e),L_(0xc304c980)}, {LL(0xb59f480b,0x3957e934),LL(0xe9a58ad5,0x7c1b8e5a),LL(0xe4604f76,0xc58f2c48),L_(0xe27bf6c2)}},
   {{LL(0xc8a88d73,0x21606f53),LL(0x6659ed53,0xfba9c9f7),LL(0xfb7d83f0,0xf57dce48),L_(0x2d2bf81f)}, {LL(0xfb80fd50,0xc8023a07),LL(0x2b42338b,0x71a25cd3),LL(0x7a9a0b44,0xb3c48734),L_(0xc8125d7c)}},
   {{LL(0xf3798b63,0x9856b7c0),LL(0x914f7c82,0x04ab2721),LL(0x471ef449,0xe5f3587d),L_(0x77a7fe57)}, {LL(0x21b96d5d,0xb0c73d23),LL(0x1ec93eff,0xa3366107),LL(0x69323a1b,0x759bf017),L_(0x3f60e38c)}},
   {{LL(0x80978741,0x4e3559fa),LL(0x56b197e8,0xe7559080),LL(0x56ad6e70,0xdd9852be),L_(0xeeb6dec8)}, {LL(0x0e7594df,0xe4523a8a),LL(0xc67e9c41,0x4a40d4e4),LL(0x8ca5687b,0xfaf902fd),L_(0x731b0560)}},
   {{LL(0xd2ade9c2,0x683ea003),LL(0x3784426d,0x6bbf8ead),LL(0x26a2a12e,0x65fa7d53),L_(0x7ab25cd8)}, {LL(0x6154c3a9,0xd7f809d8),LL(0xe2d2306a,0x20ed9166),LL(0x24d19af8,0xa692a83a),L_(0xbb68e143)}},
   {{LL(0xd8e0ff43,0x90e27172),LL(0xf7b7a48d,0xa90bf44c),LL(0x8cb2abd0,0x130d21df),L_(0xd9526e86)}, {LL(0xa3cb4c6b,0x65186f8e),LL(0xe78f59ce,0x0c9c373c),LL(0x3514478f,0x5307b692),L_(0x15b86f33)}},
   {{LL(0xd8568306,0x34f0a25d),LL(0xea579beb,0x039f50cb),LL(0x0eadf2ee,0x6a3cf739),L_(0x7fffefa0)}, {LL(0x6a9a2a62,0x5c707fb6),LL(0x1f9a5c8a,0x22360abf),LL(0x756df232,0xa7a12676),L_(0x735ae0c5)}},
   {{LL(0x232b0eaf,0xfac26fa0),LL(0xc57f8429,0x91c0d8b2),LL(0xc85b9885,0xcd36972d),L_(0x5ab11f6e)}, {LL(0x2b28cccc,0xa2197bb4),LL(0xcf7b4e95,0xb00034b6),LL(0x22200d9d,0x398b8d08),L_(0x0660c348)}},
   {{LL(0x3e7e898b,0x14230734),LL(0x8aa5dec9,0x7bef4c46),LL(0xe3a1abc8,0xb3564180),L_(0xb7f989a9)}, {LL(0xeed81407,0xa7ce6e62),LL(0x98270e80,0xf02dd5b1),LL(0x9d22967d,0xa3ff259d),L_(0x4be6673e)}},
   {{LL(0x25d2230c,0x2d777318),LL(0xd815968f,0x80147bab),LL(0x6f460f52,0x22279805),L_(0x99635b0b)}, {LL(0x20550e62,0x7880c924),LL(0xf836942d,0x6428d5fb),LL(0x9a19cfda,0xd1beb76b),L_(0x819b5402)}},
   {{LL(0xe36b37d4,0x5d87cbda),LL(0xe5f505ca,0x404781e7),LL(0xc8f5e0fe,0xa1bc9eda),L_(0x45ef1d15)}, {LL(0x23299059,0x970d11fc),LL(0x54084d2b,0x785e8445),LL(0x8ed8b412,0xbad1345d),L_(0x2975a901)}},
   {{LL(0xce7aa169,0xd74f6437),LL(0x62092837,0x1f1dc772),LL(0xf497718e,0x36827efd),L_(0x15977763)}, {LL(0x978b617d,0x4cb29feb),LL(0x65c2897d,0x6a64b733),LL(0xe04077a2,0x86b8cf3a),L_(0xac075b5b)}},
   {{LL(0xed710d98,0x3a39d170),LL(0x1f51c85b,0xef236d5b),LL(0x0683a7e0,0xb9141322),L_(0x7e27c47c)}, {LL(0x05109f74,0xdade7c00),LL(0xe2f07be3,0xf251df44),LL(0xc11452de,0xb48e9a05),L_(0xd6a44d08)}},
   {{LL(0x4aaf45f5,0x17039bf0),LL(0xbdcaec9d,0xb97b72ed),LL(0x30ce8053,0xa7a81755),L_(0x54e63f42)}, {LL(0xc81db67b,0x80753cae),LL(0xc22c28cc,0xe3019a4c),LL(0xd1c81400,0x80f8c3da),L_(0x5a1e70b6)}},
   {{LL(0x165046c9,0x030a06a1),LL(0x0ea30c39,0x09c6053c),LL(0x3932bda3,0xcb39f998),L_(0x148a2b68)}, {LL(0x87fe7977,0x0499f81d),LL(0xb74e61d3,0x7ae298cf),LL(0x2959c264,0x1edb6b93),L_(0x6a2188dd)}},
   {{LL(0x6ab52745,0xdc732139),LL(0xd70d9371,0xbd7da6c2),LL(0x5ac1ef95,0xe2e0217e),L_(0x6e9b8907)}, {LL(0x63f9997b,0x5cecf1c0),LL(0x24552abf,0x5082d35e),LL(0x7b11e2cc,0x04cc9aa3),L_(0x78bdf631)}},
   {{LL(0xfecff2a7,0x4f8b44cc),LL(0xe83e0eae,0xd783714b),LL(0x9c0232a4,0x90d1c36e),L_(0xb7e1ea84)}, {LL(0xbd5e004e,0xf22c1c14),LL(0x99441dcc,0x77b24ec6),LL(0xc4fac858,0xf15be1db),L_(0x52f8f005)}},
   {{LL(0xfa634b74,0xc11ab7e2),LL(0x865bc26d,0x39a0b0f1),LL(0xa62656eb,0xdf5e8e89),L_(0x5bbda8fa)}, {LL(0x35af495b,0x059e7fc2),LL(0x996cfbbc,0x5721bb40),LL(0xfcc7a8b4,0x18530aec),L_(0xef18ab08)}},
   {{LL(0x1efc0865,0x7e7fb20d),LL(0x1432863a,0x319cbee0),LL(0x32603543,0x216780a7),L_(0x3173509d)}, {LL(0x771c5009,0x332b0e8d),LL(0xfb32ae5e,0x27842538),LL(0xfd61d211,0x176fee74),L_(0x8d713db3)}},
   {{LL(0x6d5e6c04,0x8f5f36df),LL(0xfd53bdb1,0xff86482f),LL(0x04fc8d4b,0x35ae923b),L_(0x9c93970b)}, {LL(0xd03159ae,0x6c4574ce),LL(0x0c5ab9f8,0x95e44ad6),LL(0xecc940e1,0xacd13e60),L_(0x73d20a97)}},
   {{LL(0x24bc05c8,0x55d8d9f7),LL(0x1be2d507,0xba1700c6),LL(0xb1dbe56e,0x12671c92),L_(0x515125ba)}, {LL(0x7c873619,0x28f4b581),LL(0x00d413e2,0xc27395ee),LL(0x31a82a43,0x84bd2bdf),L_(0x5c6f966e)}},
   {{LL(0x19e04eb8,0xd6d7b2f3),LL(0x32f059d3,0x7d71e73c),LL(0x5dd55589,0x6b4ddaaf),L_(0x7a059ec2)}, {LL(0xb3874052,0xb395499e),LL(0x4cfc04e5,0x56ce4cbb),LL(0x5961b289,0x2960f1ca),L_(0xa3443d73)}},
   {{LL(0x739615c6,0x72405988),LL(0xe58efd36,0x03300370),LL(0xe254b688,0x6195855d),L_(0xf750a8c1)}, {LL(0xd298a296,0xf9205a13),LL(0xfa55931d,0x11ffb780),LL(0xe00ca07c,0xd638a7b0),L_(0x1b178a9f)}},
   {{LL(0x970fe956,0x62d115d2),LL(0xe1985dec,0x4db47694),LL(0xb3652081,0x6565367b),L_(0x61dc5dcc)}, {LL(0x53a17e00,0xbb00a05a),LL(0x390808ef,0xc7f0a623),LL(0x2971e39b,0x091d591c),L_(0xb6590fcd)}},
   {{LL(0x3c51221d,0xd273fed4),LL(0xb849d7b6,0x7eaf4c1f),LL(0xc3cadac4,0x155b21cd),L_(0x21bf043c)}, {LL(0x824bf43a,0x6d0eb93f),LL(0xa560a086,0x2b5af50c),LL(0xe79201e3,0x50f56685),L_(0x7193da80)}},
   {{LL(0xaf65e93e,0x64cb8361),LL(0x097bd142,0x2b08724e),LL(0x5d31d941,0x128434b7),L_(0xd3455462)}, {LL(0xd87b7934,0x6349bb61),LL(0x5338e12b,0x8eabd8cd),LL(0x46f9abc2,0xf5ec4590),L_(0xee7ff24e)}},
   {{LL(0xde41b0d8,0xfcdd55ff),LL(0x390333ed,0x5f8a8d73),LL(0xad3ba05b,0x5c84894a),L_(0xdd94d8c3)}, {LL(0x1041454f,0x7425541f),LL(0x4f6bd774,0xf6c08550),LL(0xbe1bce56,0x48f611af),L_(0x723a0cc8)}},
   {{LL(0x31e207ab,0x505aaa25),LL(0x6f2c316a,0x9df3ed0d),LL(0x6ff3aa7e,0xecdfa689),L_(0x0c314ac1)}, {LL(0x4660e302,0xaac1b188),LL(0x74b064f0,0xe2709e66),LL(0x1690d7e4,0x9ad43bd5),L_(0x19d6932b)}},
   {{LL(0x9e4966da,0xe9803ed7),LL(0x5acbb8d9,0x94a51c09),LL(0xc6ffb2f1,0xac8d73e5),L_(0xce57f6f3)}, {LL(0x87ee51c7,0x0f2f29a5),LL(0x1c8a8a14,0xcf07fe4d),LL(0xf7993328,0x11a086fc),L_(0x816c9d1d)}},
   {{LL(0x482c8577,0x7008cc6c),LL(0x3a6923e0,0x4ea34320),LL(0xea139dd1,0x0db793be),L_(0xc99cf72e)}, {LL(0x2f88c486,0x46daeb88),LL(0xf235e9a5,0xa9529776),LL(0x8fc9a2fe,0x685903b2),L_(0x2e9cc9d4)}},
   {{LL(0x160fbbb0,0x85dc20a6),LL(0xd5312a9c,0x40ce263b),LL(0x33ee3772,0xf18c955f),L_(0xb8cc18a3)}, {LL(0x7c56f68f,0x629b2ed0),LL(0x381772f9,0xd3751f88),LL(0x9be35b3d,0x87658375),L_(0x438ba3ac)}},
   {{LL(0x4c71f19e,0x2e6a930b),LL(0xbe1f2bc3,0x6ce370bd),LL(0x9c7c9090,0x9fe1a225),L_(0x9a8e7036)}, {LL(0xdfc19acd,0xbc4095c2),LL(0xbf429df7,0x102343b7),LL(0xa07aed0a,0x75593f8f),L_(0x2c1e8aa6)}},
   {{LL(0x6e87ec50,0xe6abe1ed),LL(0x6cc4e156,0x32e1d48d),LL(0x9c1aa3b0,0xbc16fee9),L_(0x76d25ea9)}, {LL(0x1a098c5e,0xa549cfc8),LL(0x0fe8dc0a,0xdccd7aa4),LL(0xe2b2179c,0xf0addee3),L_(0x9bcbee00)}},
   {{LL(0xec32e6a2,0x4e4d505b),LL(0xb0ad3d1f,0xd584c5e1),LL(0x708100e5,0xf05981fa),L_(0x5ca8160b)}, {LL(0xff71c066,0x30b8c50e),LL(0x9b267d28,0xa5cf6579),LL(0x00705bdb,0x003593a1),L_(0x6e7726d9)}},
   {{LL(0x1e13fade,0xbfa0d04d),LL(0x1fd12275,0xcc81bd28),LL(0x39bdb34f,0x7ba2330c),L_(0x9cb5c2f4)}, {LL(0xb5b45e69,0xf331bfd7),LL(0x9513f2a2,0xdd8ec1b5),LL(0xb3f734e7,0x9064cd97),L_(0x5f64f559)}},
   {{LL(0x6a4afc60,0x4194a70e),LL(0x63999525,0x523cff27),LL(0x03d08e2c,0xa27e7edc),L_(0xea90ab15)}, {LL(0xfc7c7e87,0x38873c54),LL(0x398075fc,0x805abf3c),LL(0x414e3976,0x93991ee1),L_(0x66b18c5a)}},
   {{LL(0xa5d47504,0xb3fe93fc),LL(0x9c7a5517,0xb7d168b6),LL(0x4a438e8c,0x581ebd7a),L_(0x30a1a792)}, {LL(0x1e7c3d15,0x7749a42d),LL(0x2279fc57,0x86e109ed),LL(0x853c9b29,0x069c5f0b),L_(0x4438f7b5)}},
   {{LL(0x2fb4329a,0x2b3c8790),LL(0x99bbcc79,0xc7e0dd07),LL(0x9d9741bf,0x51f341ad),L_(0xaedc9e99)}, {LL(0x6eb8a76e,0x1cd0ce38),LL(0xbf6e3e65,0x2698a77a),LL(0x3099a97a,0x6b8968f5),L_(0x6186d5b8)}},
   {{LL(0x5f3a4ac3,0xf86e098a),LL(0x6977fcd9,0x3194f379),LL(0xc98e48aa,0x886d05b8),L_(0x23139131)}, {LL(0xbd9be137,0x488df6bf),LL(0x876a25e7,0x9833b00c),LL(0xfb9f6810,0x5fb8ce68),L_(0x1a974ef1)}},
   {{LL(0x0996c57a,0xe74e34d6),LL(0x95d0e8ba,0xbc8f1cc6),LL(0x49ddbd76,0x3e421dda),L_(0x47780198)}, {LL(0x2fcb399c,0xf599d67b),LL(0xb3cd50fe,0x1de7c3a8),LL(0x60ffdfcb,0x41018e1d),L_(0x2a84e15c)}},
   {{LL(0x5803c901,0x1834f6f5),LL(0x85be2737,0x0269aa98),LL(0x51b8942e,0xbd9cdd7b),L_(0x2573f804)}, {LL(0xf11dba7f,0x62d1e404),LL(0xa026f5f4,0xa355486e),LL(0x3e7a1a96,0xa6d151c2),L_(0x52ecef89)}},
   {{LL(0x5bfcbb66,0xd3cf6d12),LL(0x7f3efdcc,0x509c593b),LL(0x36beb1c6,0x50bb9be7),L_(0x87aa0794)}, {LL(0x461e9081,0xf6ebcef8),LL(0xb12b6973,0x53fb4ce0),LL(0x1d3a4bb3,0x84afbe92),L_(0x765f75f5)}},
   {{LL(0x36690884,0x66809ebd),LL(0x7d2d307f,0x6d7b75a0),LL(0xb7561328,0x7a0b342b),L_(0x25a224f2)}, {LL(0xc6b75bd1,0x8b464a87),LL(0x98a06fcf,0x145091bc),LL(0x33cace99,0x723fe6fa),L_(0x24207bff)}},
   {{LL(0xfa386bf7,0x5ea0874e),LL(0x470c5915,0xa8387298),LL(0xbc4e6a04,0x02c73e59),L_(0x303d5c07)}, {LL(0xa4b7c2f2,0x2fdbbf21),LL(0x2d418059,0xb933bbc7),LL(0x874e47bf,0x4c88b532),L_(0xbd3a8f84)}},
   {{LL(0xb79dad0b,0x94b27e28),LL(0x4c3cb167,0x47dac83a),LL(0xb3a5c35a,0xace3bd6b),L_(0x36f5d64a)}, {LL(0x0d91b2e8,0x00f91b6c),LL(0x561c257a,0xd6d7f96c),LL(0x390f1d25,0x92ea6b8b),L_(0xce74a256)}},
   {{LL(0xa7f8fb96,0xf0e6068b),LL(0xbee4d1a2,0xed56b92a),LL(0x64cd2ac4,0xd3d78730),L_(0xb0ed4379)}, {LL(0xf0d95042,0x8db9e812),LL(0x0e235698,0x6447eba8),LL(0xd45b6fa7,0x3b769af7),L_(0x95403694)}},
   {{LL(0x26f167ea,0x8aa2617b),LL(0xdb91ef4a,0x52ead3d0),LL(0x41988291,0x5273a693),L_(0xab06325b)}, {LL(0x18ca4d81,0xb1bbd253),LL(0xea972289,0x58878e32),LL(0x64d8aee8,0xe8908e43),L_(0xcdca72ce)}},
   {{LL(0x5a517623,0x6d83d019),LL(0xd917cb4a,0x969711bf),LL(0xa7fc9902,0x225d81d7),L_(0x9b126599)}, {LL(0x25981f40,0xe02b87e7),LL(0x04539a29,0xaec410d7),LL(0x35630262,0xb1115eaa),L_(0x7461aa8b)}},
   {{LL(0x83f05d8e,0x4dc83cf2),LL(0xdbbadf9b,0x523db1cc),LL(0x09d299d6,0xe968d6f7),L_(0x46a4ef90)}, {LL(0xa5406082,0xc545ab6a),LL(0x158f1789,0xcf6e254a),LL(0x592bcabb,0x599530e8),L_(0x4ecd132b)}},
},
/* digit=22 base_pwr=2^154 */
{
   {{LL(0xa46f08d9,0xb9fe1aa5),LL(0x27531029,0x32c26e96),LL(0x1b37edc0,0xcc6e653e),L_(0x25a6777f)}, {LL(0x37d78fa5,0x44dcd6fa),LL(0x44e59684,0x637ed305),LL(0x72834a3e,0x6cab1cc9),L_(0x715fa76f)}},
   {{LL(0x2efe30a2,0xd9e804cc),LL(0x1c1ded3a,0xb9c440fa),LL(0xb6ba8131,0xd2f7d77a),L_(0xe00e21fe)}, {LL(0x88842b6a,0x96f7f20a),LL(0x616f6de5,0x9a618ffe),LL(0xf13ceac5,0x6577f60c),L_(0x9f1e4b0d)}},
   {{LL(0xbd482557,0x1653fafd),LL(0x93eea72d,0x39c09ddd),LL(0x10b62eeb,0x79f6d014),L_(0x2f476cfa)}, {LL(0x78d81b01,0xe8963c96),LL(0xc395c6ba,0x8c8f6a4c),LL(0x63b70385,0x551fa28d),L_(0x28bb3fc4)}},
   {{LL(0x433a33ea,0x2417a7ce),LL(0x5591c910,0x1d9155b1),LL(0x7a6ca298,0xd78e16eb),L_(0x09ee38e9)}, {LL(0x3ab289c2,0xee48adb0),LL(0xd5b76200,0xc998c5c8),LL(0x8dd6fbf2,0x959f66af),L_(0x3ee20307)}},
   {{LL(0x23a0b23a,0x63c09a4f),LL(0x24b60423,0x021b685f),LL(0x7c69331c,0xb1b0820e),L_(0x7d3c58fb)}, {LL(0x81992f9f,0x370cbdbc),LL(0x05814d87,0xeecaf1b3),LL(0xaeba071a,0x5ebd44d9),L_(0x079434ba)}},
   {{LL(0x193df859,0x7a008e70),LL(0x9d5dd4b2,0x8d7ec280),LL(0x81377001,0x3062b2db),L_(0xe82fbf45)}, {LL(0x11e91ce4,0xdb62847d),LL(0xf3a5dc38,0x9a492d59),LL(0x42b1e033,0x3c5e448f),L_(0x0028f02b)}},
   {{LL(0x48abe62f,0xf00a790d),LL(0xf18c3029,0xe9c56ab7),LL(0xcfc6643d,0x7a101d9e),L_(0x69f64af5)}, {LL(0xe901d5c7,0xc8b5cd29),LL(0x8ab0437d,0x5a6c0d3b),LL(0xf3eacfba,0x437c7e62),L_(0x9c1ff745)}},
   {{LL(0x7ef25e16,0xc3c0b7b4),LL(0x3b2bb23b,0x2d411980),LL(0x9d372287,0xb86b3e9a),L_(0xbd217cd9)}, {LL(0x6682e29c,0xc7ddb5da),LL(0xf55b3a32,0x80ad8745),LL(0xc5272ca9,0x67e42112),L_(0x80ddaf6c)}},
   {{LL(0x2db21214,0x26d65fa2),LL(0x593699c3,0x27bc92ce),LL(0x780c1507,0x62bfbfb4),L_(0x4e610cd9)}, {LL(0x90f53f42,0x306b2065),LL(0xdcabb7e7,0x70add873),LL(0x97f0fe05,0xef3f82de),L_(0xfd016e06)}},
   {{LL(0xdb734eee,0x49ac57da),LL(0x33f4851e,0x0575235f),LL(0x12e6b4bb,0x84ba90a7),L_(0x0c414c40)}, {LL(0x2e72cdc9,0xde178a6e),LL(0x5d680978,0x2c8f9b52),LL(0x04316bb2,0x4d9021c4),L_(0xd58bf8cb)}},
   {{LL(0x4a88c720,0x7c1da7fb),LL(0x2166e51a,0x7be262ad),LL(0x26f4415e,0x8d16df29),L_(0x4308e38d)}, {LL(0x6e047968,0xf7050746),LL(0x5191f59a,0x40048ee7),LL(0x671c053f,0x7e714450),L_(0x41cd1cf2)}},
   {{LL(0x126208fc,0xc3f913fb),LL(0x77459414,0x81ebbd24),LL(0xfbf9d09c,0x60be0397),L_(0x5762fb0e)}, {LL(0xab78592e,0xee7a8b54),LL(0x11f20203,0x58636e68),LL(0x5cb0cea7,0x56d9eb18),L_(0xaab58567)}},
   {{LL(0xe2adb3e3,0xe08bf721),LL(0x8b739b3c,0xcf7b1348),LL(0x88809e01,0x9dd0883a),L_(0x0dd14479)}, {LL(0xfb2d7aed,0x9e8ac3f6),LL(0xe9234e02,0x3aa9be66),LL(0x87a97238,0x56210cb7),L_(0xa8cca221)}},
   {{LL(0x515e6713,0x989ff215),LL(0xd0bcb26c,0x8011a540),LL(0x16b0e517,0xc8b38bfd),L_(0x7706fd39)}, {LL(0x23c98754,0xc13358b8),LL(0x8745e3cc,0x5edc147a),LL(0xae944131,0xbf36526b),L_(0xb43ce39b)}},
   {{LL(0xbb74f4a3,0xf57da9fb),LL(0xcceacf96,0xe6c34f99),LL(0xae99e920,0x72b76a37),L_(0x7a9e0b84)}, {LL(0xd46a4b67,0x377b9f4d),LL(0xd8d2d70f,0x2dbc27cb),LL(0x1c0d51f0,0x63d0ccbb),L_(0x423d586b)}},
   {{LL(0x76b14e74,0x5173ac48),LL(0xda87fd76,0xac29f6c0),LL(0xde2af466,0xbf4405a4),L_(0x0df31635)}, {LL(0xe0a96172,0xc72bdfba),LL(0xcbefcc71,0x14f78dd6),LL(0x2c991165,0xf55ecef7),L_(0xa3278f6d)}},
   {{LL(0x6d393d5f,0x05e626ee),LL(0xb9f01d8a,0x225ed4d6),LL(0x53d41ea6,0x5d5b37e5),L_(0x6b1d2d7a)}, {LL(0x70c4e493,0xf4527dfb),LL(0x465bfaf7,0x1522deea),LL(0x333509e1,0xad711cce),L_(0x988f296f)}},
   {{LL(0x072fecfb,0xba0c11d9),LL(0x46a2c644,0xa6fa784d),LL(0xa105395f,0x3a914aba),L_(0xa42699f5)}, {LL(0x5b5ec37d,0x6c8c0e8b),LL(0xffef0a76,0x79696f78),LL(0x69daa4dd,0xbf1748ba),L_(0x8134c310)}},
   {{LL(0x16919c10,0x43014c2d),LL(0xa35261bb,0xd14b6a12),LL(0x6d705b75,0x395d7526),L_(0x58721301)}, {LL(0x4c3aec80,0x3f4fd283),LL(0x2e6f84dc,0x4114b4f9),LL(0x6a63c71b,0xcf962aae),L_(0x58c876bc)}},
   {{LL(0x085cede4,0x39a2603a),LL(0x21ba7529,0xb203a75d),LL(0xbdc1e809,0xf3640f89),L_(0x6631c30e)}, {LL(0x8d9d5265,0xd480817b),LL(0xad93303a,0x8492b58b),LL(0xe3368ba5,0x24b8bfe6),L_(0x26c2b552)}},
   {{LL(0x64ef9738,0x6907b7bc),LL(0xd795d794,0xf6e9c508),LL(0x33b42737,0x55ab9706),L_(0xa13daed3)}, {LL(0xf586dde7,0x76b6d961),LL(0x0ff335ce,0xad1c2111),LL(0x13031322,0x1f36dd8c),L_(0x0558b72f)}},
   {{LL(0xe83a229d,0xe6ae70d0),LL(0xf0770ab8,0x5b9ca0d2),LL(0x06479e73,0x17cdf808),L_(0x5e50cd46)}, {LL(0xd020f0ac,0xca397995),LL(0x3337dcfa,0x70369103),LL(0x3d37e46a,0x67d7bcef),L_(0x5283dec8)}},
   {{LL(0x920986d4,0x9e2b8580),LL(0x6f80e6f8,0x1a1086a7),LL(0xf038b9fd,0x76630d46),L_(0x4e9ef3ae)}, {LL(0xf2780d64,0x479ba46d),LL(0xfa784b8c,0x41a1c68f),LL(0x953b6640,0xeee5927b),L_(0x9d4a23e4)}},
   {{LL(0x7a6dcf80,0x2905feea),LL(0x2a632e73,0x5bba1fd8),LL(0xd274195e,0x596a73e6),L_(0xaf3000ac)}, {LL(0xfa2e2813,0x83723184),LL(0x99dae0bb,0x21f09629),LL(0x7ce090e3,0xb6dc2849),L_(0x2bd68086)}},
   {{LL(0xb254b1d5,0xfee577f0),LL(0x2ce6dbab,0x6accfc0b),LL(0x775f693f,0x483bf6a8),L_(0x829c24e8)}, {LL(0x74582da0,0x6731bfc7),LL(0x8e7f87f0,0xf7d39568),LL(0xa67faa6c,0xc9418804),L_(0xf17ae402)}},
   {{LL(0xb5eb4f29,0x7a12fe1e),LL(0x862fd52c,0x4a7fc70e),LL(0xd820b6bc,0x319719c3),L_(0x06ce9593)}, {LL(0xbd26f429,0x66648099),LL(0xb9a6a586,0xb662d062),LL(0x7fbe15d7,0xf16d93b4),L_(0xab04debd)}},
   {{LL(0x951453e5,0x8f99c3ae),LL(0x01cc4900,0x422aab2c),LL(0xec8547fb,0x73be771b),L_(0xfba8505c)}, {LL(0x49e82e72,0x7b15c573),LL(0x238927ca,0xaa7e7ca5),LL(0xfd867aa0,0x58e19238),L_(0x16e92a3b)}},
   {{LL(0x31e49c5d,0x2d0ee2e6),LL(0xe0a95c2b,0x3aaf91fc),LL(0x271cbc87,0xcb833e79),L_(0xcebcb341)}, {LL(0x36952e47,0xf7a6bc06),LL(0xb8a1069c,0xda163496),LL(0x1f6ccb97,0xb5f271bf),L_(0xadf492cf)}},
   {{LL(0x606962f9,0x82d18205),LL(0x983a462e,0x4a97cf98),LL(0xde06a403,0x23969744),L_(0x5ada65f3)}, {LL(0xba5e083e,0x22229508),LL(0x7b88f442,0x376f004a),LL(0x1c53a79e,0x8ecb5e4a),L_(0xbc00e747)}},
   {{LL(0xfde8f5b0,0xdaab7a45),LL(0x6896d43d,0x6cba2e3f),LL(0x0b7de7e1,0x31db2861),L_(0xc8a85283)}, {LL(0xf98e6b35,0xa8d3e811),LL(0x9ed4cc65,0xc4961d3a),LL(0x4bad9114,0x10d64cdd),L_(0x6febb44a)}},
   {{LL(0xadd9fbd1,0xca3624bb),LL(0x6bfd7f49,0x73a43aec),LL(0x91301b03,0xc382f632),L_(0xe009202a)}, {LL(0x83cfde00,0xf9e8d9a9),LL(0x03cf587c,0xb9fbd92f),LL(0x25a8b014,0x760229d4),L_(0x62d554e5)}},
   {{LL(0x95ce63b9,0x225d1531),LL(0xe0e28f19,0xf1862d71),LL(0x143d1930,0x89faba0c),L_(0x3d640e67)}, {LL(0x36978269,0x2f3bfacc),LL(0xb38209aa,0x8ecd6947),LL(0x817ba1fc,0xad14a0f1),L_(0x53f8ce9d)}},
   {{LL(0x9cf6404c,0x0fcc7993),LL(0x455cb661,0x94227ce7),LL(0x36c49b8b,0x1add1a97),L_(0xb4a122ea)}, {LL(0xddf10443,0x478ddbe9),LL(0xd7d28b55,0x598d307a),LL(0xa7d77b58,0x7c2d4285),L_(0xf914cdbb)}},
   {{LL(0x6ae40efc,0x7296f54f),LL(0x267136bd,0xbb8c6473),LL(0xa3248f4d,0x096d9531),L_(0x9bf47d4a)}, {LL(0x72de2ce7,0x71d14af5),LL(0xe31fd133,0x8bb3132e),LL(0xd5a9918e,0x1ee08068),L_(0x4ad2e489)}},
   {{LL(0x3a3e84c5,0xdb4c764e),LL(0xc0cdb80f,0x06967d90),LL(0x9a2fe8e7,0x3d0cd72c),L_(0x8105ce51)}, {LL(0x663d93ac,0xd9a81336),LL(0xc772e40c,0xe092d50f),LL(0x7d2f5c90,0x45eb443a),L_(0x7ebd73b6)}},
   {{LL(0x29b44f03,0x09859196),LL(0xd120b82e,0x274e7581),LL(0x8a9337d3,0xe9cba9ed),L_(0xb7d3254c)}, {LL(0xab1d6ae2,0x1aa44fa6),LL(0xa2bcdd85,0x8201c08a),LL(0x2f9c1703,0xd128d342),L_(0xcf87815f)}},
   {{LL(0x04dbb489,0x4b130afb),LL(0x8592bf1f,0x010d7b25),LL(0xf211bcec,0xb7f03740),L_(0x2070aa40)}, {LL(0x24a2aec7,0xf116e286),LL(0x3a91fedf,0x4b6af9fa),LL(0x974e5b38,0x19cbf327),L_(0x0cab111f)}},
   {{LL(0x8816e066,0x52f1347b),LL(0x0dbd4053,0x733b5b5e),LL(0xe33961b0,0x8db340e4),L_(0xc28fe0e4)}, {LL(0x8fdad9c7,0xb3cc8f09),LL(0x66c9d4cd,0x36371a81),LL(0x36432409,0x9dd5d017),L_(0xfd34cf28)}},
   {{LL(0xc45e9f49,0xce25491e),LL(0x464328d6,0xf8ea793b),LL(0x27eb6c22,0x6371b206),L_(0xc53f6320)}, {LL(0x80e32e79,0x61eaf6df),LL(0x3ece8680,0xdc4f44ca),LL(0x387b7804,0x4b85bf6e),L_(0x4e286cd6)}},
   {{LL(0x2c7dc475,0x18a6a123),LL(0x87751313,0x53665bc9),LL(0xaa43dbfb,0x8d40fe26),L_(0xe4bce37a)}, {LL(0x6723ea8f,0xbade3e08),LL(0x4b0254d2,0xc7adbb73),LL(0x2f3a4dc9,0xa74a2565),L_(0x829b9555)}},
   {{LL(0x9ab672f0,0x9cc7bfa2),LL(0x1fad100a,0x2ac7eb07),LL(0x368be66e,0x5665ccf4),L_(0x8056fdf1)}, {LL(0xb250a7ac,0x465d7f98),LL(0x64180c0c,0x8d2b97aa),LL(0x7b979097,0xa50d476d),L_(0x45ad85c0)}},
   {{LL(0x8b68fb1b,0x542fb7f1),LL(0xf8fc7eb8,0xd96b7a33),LL(0xb020cf53,0xdb72531d),L_(0xb4c510b9)}, {LL(0xc73ecdcc,0x1c4cd34a),LL(0x272dd7b1,0x039494e8),LL(0xd23f79c7,0xa21edcfd),L_(0x207ff37a)}},
   {{LL(0x17c4dc72,0x5ce9dcb0),LL(0x6a5e2d85,0x25936d77),LL(0xebe08b04,0x70fc7789),L_(0x7b6f8a3a)}, {LL(0x7f3546f5,0x7563a201),LL(0xea8da605,0x10bfdb1f),LL(0xdb570af0,0xeb5d3434),L_(0xdadec2da)}},
   {{LL(0xef345021,0xd0e87d76),LL(0x2945a207,0x222e0b0f),LL(0xe86e0a88,0x6fa3b5c9),L_(0xef9f7939)}, {LL(0x9da8796b,0xabdf6b2a),LL(0x90ef27c9,0x7ead1795),LL(0xbbc76905,0x8c71560c),L_(0xbef19370)}},
   {{LL(0x890d8ce2,0xb5cc05b2),LL(0xa7bd2aa7,0x7de1e29a),LL(0x866cd575,0xfc749a26),L_(0x825af3b3)}, {LL(0x7eaa0bb6,0x886271b4),LL(0x6514a9d4,0xed069b45),LL(0x65bd26ca,0x2f59240d),L_(0xbe13ff4a)}},
   {{LL(0xd9546220,0xcbae178c),LL(0x2cf4cc42,0xe71c4857),LL(0x70950b7d,0xc8f7ab52),L_(0xb20e351c)}, {LL(0x52a1a22e,0x24d86b6f),LL(0xe4b84f76,0x70b94a08),LL(0xd450bfb1,0x0d983cdd),L_(0x3f21b37f)}},
   {{LL(0x5619c34b,0x77d8a74f),LL(0x86215aa3,0x71f4dc88),LL(0xfa0612da,0xb127ccb2),L_(0xc1906699)}, {LL(0x879fb160,0x205b4589),LL(0x2af7eaab,0xe8b6c8f4),LL(0xaad791a8,0x1490d5ca),L_(0x3359b830)}},
   {{LL(0x32c5bb90,0x28810885),LL(0xdd3a45f9,0x2ceffdfe),LL(0x14a9a0ad,0x7970ea64),L_(0xe4fdf446)}, {LL(0x8708e80b,0xbbf83ccb),LL(0x1b1d1037,0x5cc8baac),LL(0x800c8eba,0x98c8b359),L_(0x3ee963ee)}},
   {{LL(0x22c84665,0xdbd93971),LL(0x284c33c2,0x697a6de7),LL(0x6f50eaa6,0x87cb9042),L_(0x70efc303)}, {LL(0x5f70b39e,0x2af4f24a),LL(0x562e2f64,0xd8a16a0e),LL(0x8f4571e4,0xa7b9ed81),L_(0x6ceb56cb)}},
   {{LL(0x6e1de981,0x725de358),LL(0x04ebe2bd,0x85652ca3),LL(0x50b9f863,0xd5f0033d),L_(0x0b13ec3a)}, {LL(0x19c015bc,0x948391b7),LL(0xc37027bf,0xaf6e5ed5),LL(0x33c66538,0x96bfa703),L_(0xced15639)}},
   {{LL(0x069d03c1,0x47f60293),LL(0x73dccc53,0xc32780f6),LL(0x3bfeba8e,0xc2e0eb4e),L_(0x546bd56a)}, {LL(0x29bd4113,0xc233f03a),LL(0x277aca24,0x574da3b6),LL(0xf4f6a7ac,0xae0e0260),L_(0xbade476b)}},
   {{LL(0xbb8f2925,0x5a087018),LL(0x5c2bab18,0x8c5485dc),LL(0x351ad0f1,0xf7d7679e),L_(0xe930bcd6)}, {LL(0x2bdc4897,0x277bcf81),LL(0xbcb4ffd9,0xf147c5b3),LL(0x87185cc7,0xdf48c9e6),L_(0xce71b122)}},
   {{LL(0xdb75cc10,0x17222dc7),LL(0x3bc7295c,0x3dd256a0),LL(0x3a012a3b,0x7ec5696a),L_(0xe6950145)}, {LL(0x9381bed6,0x4f4cdfd3),LL(0x353d2f83,0x7b5b13a3),LL(0xbf5f2e84,0x2639cce3),L_(0xda40d3a1)}},
   {{LL(0xdb334a59,0x6da64943),LL(0xd3f151b5,0xfce96a35),LL(0x6ad5dfad,0xa599a89e),L_(0x0d1b7f52)}, {LL(0xeddc6626,0x2ef746d1),LL(0x861341ad,0xa912c3ef),LL(0xd6859ab6,0x1ac3aba5),L_(0xd95ca5f2)}},
   {{LL(0x0e5c7c8d,0xb050ceeb),LL(0xcf83afd0,0x53e4bd2e),LL(0x710688b3,0x19a215e7),L_(0x8fa6f937)}, {LL(0x9a042c9a,0xd22cc140),LL(0xa2947f4b,0xb5f7e2eb),LL(0x7f1f59f8,0x154453dd),L_(0x4db7a4d6)}},
   {{LL(0x9473341c,0x9f33393e),LL(0x75ea0e37,0xb3dc51a3),LL(0x09fb2116,0x0ee5b84f),L_(0x3fcab420)}, {LL(0xf91ac676,0x816d6ca6),LL(0x8f078e9d,0xc0c5edee),LL(0x0cfcd5ed,0xb71c05e5),L_(0xc167eb60)}},
   {{LL(0x17c9bd2b,0x4c0897b1),LL(0x145b24db,0x7a16a169),LL(0x0f41e0dd,0xa88a35f1),L_(0xbff8660b)}, {LL(0x5e5fc719,0x85b8baf5),LL(0x2e4db4f9,0xd33b70cd),LL(0x5dd4f3b8,0x5ddf49c1),L_(0xbfc827f1)}},
   {{LL(0x795a3893,0xedb3e7f2),LL(0x6cad1ecf,0x33c80574),LL(0x26818153,0xf7414144),L_(0xff6a4325)}, {LL(0x6f311ca1,0x24d6e452),LL(0x7d6669d2,0xe7193165),LL(0x66d1a963,0x89fa62a1),L_(0x6bc282cf)}},
   {{LL(0x375aa47d,0x70c638b9),LL(0x4fe2ddd6,0xf167acbd),LL(0xddf9d01a,0x364bf41a),L_(0x70510fe0)}, {LL(0x7d82f66d,0x8862080b),LL(0xe66e70a2,0x0448ac9c),LL(0x4dbdd638,0x971eaebd),L_(0x07449c55)}},
   {{LL(0x30f6e054,0x4b4739f2),LL(0x55ce7718,0xd62b9aef),LL(0x3bbafe26,0xb84b80ac),L_(0x52aa54d6)}, {LL(0xd03fc3fe,0xb6803cd9),LL(0xaf2b1562,0x81afcd95),LL(0x205a14b1,0x7a0ade0e),L_(0x371096d5)}},
   {{LL(0x2d735064,0x4b1c54dd),LL(0xaad4a4b1,0xda4629d6),LL(0xe95c14ae,0x6f216326),L_(0x00affaf4)}, {LL(0x2fae20c7,0x3112101e),LL(0xd2f76d99,0xbce52747),LL(0xfcd465ea,0x53fd0b9b),L_(0x8b45a470)}},
   {{LL(0x068c0d93,0x1f9856a8),LL(0x6517f94c,0x3408da23),LL(0xc9b9b412,0xc63b4cb7),L_(0xd2b6eaad)}, {LL(0xb7836b3a,0x200fd925),LL(0xa850f3d2,0x55a6ddef),LL(0x2b7f407e,0x0fb2a9d1),L_(0x8b4cf8a0)}},
   {{LL(0xe36962f6,0xfae22eee),LL(0xb2671435,0xfe866c3d),LL(0x8acadf81,0xa3f599f7),L_(0x57dbb976)}, {LL(0xe8b6b61f,0xb4b13551),LL(0x14a48328,0xcdd8f22e),LL(0xb9099888,0xe1b8ed2e),L_(0x322f8929)}},
   {{LL(0x7e8f146f,0x3818b39f),LL(0xacd7c7b4,0x16886904),LL(0xf677bde6,0x022ae138),L_(0x9ffae2cf)}, {LL(0x5746a459,0xacf62d52),LL(0x09ae59c0,0x63279bc6),LL(0x4ee46705,0xfd285b9c),L_(0x674ba2e7)}},
},
/* digit=23 base_pwr=2^161 */
{
   {{LL(0xbad78997,0xc931bee3),LL(0x0c15f2e2,0xe3c1a2d8),LL(0x7cd507de,0x58fc32e1),L_(0x0f28ff52)}, {LL(0x01135f60,0xfb8b7b35),LL(0x11406d6a,0x413750b4),LL(0x2dffb3fe,0x6a265953),L_(0x7c1dd63a)}},
   {{LL(0x796bb2b0,0x56c1932c),LL(0x09ceb738,0xf443c29c),LL(0x6ff56bc3,0xbdd22bb5),L_(0x7387fcf2)}, {LL(0x21178548,0x22575d25),LL(0x534ca838,0x7ffb29d7),LL(0x4ffc71d3,0xd6ef14c3),L_(0x6a7a8abe)}},
   {{LL(0xbc9613ac,0x5920119c),LL(0x6ae141df,0xc3d2b2e1),LL(0x9633ba13,0xe7d31665),L_(0xc56ddd74)}, {LL(0xe622933d,0xec5df7bb),LL(0x641e9e6c,0x50f7f156),LL(0x7feb42b1,0xeb17ef8e),L_(0x36a1400c)}},
   {{LL(0x1a087220,0xeeddf997),LL(0xe4cee644,0xb6ad5609),LL(0x3c761d64,0x52dd9f57),L_(0xda46466e)}, {LL(0x39b5966f,0xca15f167),LL(0x16fee018,0xea78b847),LL(0xd9642afb,0x37eea476),L_(0x3f286088)}},
   {{LL(0xda4b72e6,0xa3072640),LL(0x3e38a59d,0x36a057e0),LL(0x4754e3c5,0x4faa9820),L_(0x2a8a9f0f)}, {LL(0xf20160be,0xa80ee38d),LL(0x5f711700,0xc8e173dc),LL(0x665dbf2e,0x2aba1be0),L_(0x187969d1)}},
   {{LL(0x228ed7aa,0x2d501abf),LL(0x1434af6f,0xe4f5d2f6),LL(0x3dad807b,0x0b4aa28f),L_(0x49171ad5)}, {LL(0xe672b220,0x0888b653),LL(0x61b8ab73,0xa3e477ad),LL(0x226f83cd,0x0afdcedc),L_(0xff93b3f1)}},
   {{LL(0xe7476655,0x562eb536),LL(0x4facc94b,0x7f5bedc0),LL(0x2887f756,0x8581b7e9),L_(0x2a5e0a2d)}, {LL(0x160c3505,0xdde60b62),LL(0xa03cc7a8,0x8a8ca29a),LL(0xb94770da,0x3ecfeca1),L_(0x667902dc)}},
   {{LL(0xb615d5a8,0x8ea9ecfb),LL(0x55508319,0x6be624fd),LL(0x7f81ee5d,0x45035a93),L_(0xd73caa36)}, {LL(0xc3d23101,0xe9c562fc),LL(0xd42e053f,0xf451c229),LL(0x6ee17a32,0x932ddcfa),L_(0x689ee05b)}},
   {{LL(0x6337e909,0xe970ce2d),LL(0x7dae8ee6,0x58d5ae88),LL(0x253afb1a,0x88db9fe4),L_(0x05ed8360)}, {LL(0x3d74de44,0xa90b033e),LL(0x8c323484,0x132dd86b),LL(0xed163aa4,0xc88fa0d9),L_(0xbaaf76b3)}},
   {{LL(0xa0e34d4c,0xa5b69894),LL(0x8d09f343,0x495e2cc1),LL(0x987653cd,0x5e895b41),L_(0x0dbb4714)}, {LL(0xf289cc21,0xc41882ef),LL(0x5fe6817a,0x3329b3be),LL(0x4acf73b8,0x9efd2db0),L_(0x0d02e7dc)}},
   {{LL(0xdd8e91bb,0xf8f6f870),LL(0xc680bcc6,0x728bf4b4),LL(0x93d4067d,0xb9c4891b),L_(0xc44a0438)}, {LL(0xf8bf9b96,0x1e5e8506),LL(0x8e6af7aa,0x11821ebf),LL(0x99083ad2,0x7e9d0c84),L_(0x510d6d87)}},
   {{LL(0x85547391,0x48ab6926),LL(0x4bca1962,0x1badb306),LL(0xdd5f7613,0x4641e82a),L_(0x7c23013e)}, {LL(0x3d43a297,0x58c32dfb),LL(0x706e6ac2,0x3ffcd8f7),LL(0xc250c17a,0xfacc3394),L_(0x0ac00a9f)}},
   {{LL(0x99456e77,0xe9a40562),LL(0x762b34e5,0xa165351f),LL(0xd0dd1798,0xbabb6293),L_(0x257af82c)}, {LL(0xb83a079b,0xa5cd33ea),LL(0xac07759a,0xf120f050),LL(0x6109e625,0x774354d8),L_(0x7ac23484)}},
   {{LL(0x8d82b0bd,0xbafdf31f),LL(0xfd4e66c0,0x7639e997),LL(0xd2a8308f,0x3be638b1),L_(0xeb3d8386)}, {LL(0x54bc114f,0xeb650088),LL(0x97a2e01d,0xebdef489),LL(0x902bb33d,0xc3e46171),L_(0x3a181f63)}},
   {{LL(0xc86362a6,0x56cc1bef),LL(0x13e720d5,0x3b4b4fc8),LL(0xf9436f6d,0x370a01fd),L_(0xba6eeefa)}, {LL(0x632c516e,0x916aa8c7),LL(0x571c4cd2,0xe3ad18e9),LL(0xe18b971a,0x72c7e5a4),L_(0x570527c5)}},
   {{LL(0xb48396a0,0xf8cf3d39),LL(0x6ad6e8f2,0x1d89563e),LL(0xa602a006,0x91cd1395),L_(0x64fa2f77)}, {LL(0xdbdaa433,0x0966d0a9),LL(0x8e0abebb,0x166bebe0),LL(0xf1b4da0f,0x433f9113),L_(0x8e2b6532)}},
   {{LL(0x6fb1d5da,0xd89da757),LL(0x89f048e8,0x2939bdac),LL(0xa42134e8,0x6b13fefc),L_(0x792be7f3)}, {LL(0xb37f662f,0x413f92df),LL(0x08daa37a,0x985c742a),LL(0x1de586e7,0x5f4904de),L_(0xe4507309)}},
   {{LL(0x03906d92,0xb13c4090),LL(0x9f0f4106,0xf3affee5),LL(0x6830a295,0xb7fc1017),L_(0xfe0d1565)}, {LL(0x45d24eb7,0xcb2d95c2),LL(0x02c21f5f,0x8c20bcd3),LL(0x3379e3da,0x6dc4634a),L_(0x5f5ffa94)}},
   {{LL(0xaefdff78,0x97e82574),LL(0x2379777c,0x112d1838),LL(0x20061ca4,0x0060cd67),L_(0xc1751f60)}, {LL(0x11499a1f,0x8b6913cb),LL(0x3ca5f64c,0x67ff5e86),LL(0x1680aa3a,0xe823c3ec),L_(0xdf7da52e)}},
   {{LL(0xd3ac8a56,0xd2b4232a),LL(0xcd8aee6e,0x1b8abf52),LL(0x2fcb86b4,0x19bc7365),L_(0x2952fde9)}, {LL(0xa051bf9a,0x1aec4d94),LL(0x79b4a51e,0xfbc4f986),LL(0xa954ceb9,0x05b104cc),L_(0xe0d3602f)}},
   {{LL(0x40690897,0x4799ff54),LL(0x87dac4d0,0xbc95c209),LL(0xc0b5b3de,0x290686c2),L_(0xb3329cbb)}, {LL(0xca443d1e,0x6b65747b),LL(0xcac2d9da,0xf2987b71),LL(0xf039026d,0xb9b37c0a),L_(0xb77496d3)}},
   {{LL(0xd360d773,0xd81505e8),LL(0x271af0ca,0x40816745),LL(0xc9836e31,0x9dca53d3),L_(0x7227191a)}, {LL(0xd820c040,0xb45d3f5d),LL(0x3bbb6420,0xedf3a3c8),LL(0xe0ce5e44,0x459a2236),L_(0xf815d523)}},
   {{LL(0x35ffc3c2,0x273c8e42),LL(0xdda15d58,0xfecec30c),LL(0xda6a8389,0x8e8dc1c3),L_(0xc0bf1e23)}, {LL(0x6fe132fe,0xc3565543),LL(0x707a3332,0x951334fa),LL(0x770ed88f,0x4d1d476c),L_(0x180fa694)}},
   {{LL(0x20e67928,0x488e81fb),LL(0xd3aa63c5,0xd968afb3),LL(0xb0057d32,0xb8d04c1b),L_(0xabd57ea4)}, {LL(0xee01ab83,0xef38d717),LL(0xe3ff770c,0x830deae9),LL(0xfba87128,0xbf20a3cb),L_(0x7a5792cc)}},
   {{LL(0x15750d84,0x8d8facef),LL(0xec118062,0xef0f3af0),LL(0xd1a4206b,0x64ebf883),L_(0x6101aba7)}, {LL(0x5b6d2c3c,0x3c48ef20),LL(0x2cb3a5ff,0x153580c5),LL(0x534ba912,0xa02f889c),L_(0xa1a92663)}},
   {{LL(0x1f2a638e,0xcb233fe2),LL(0xd1e29224,0x792d0dff),LL(0x62d03b08,0xd6646af8),L_(0xcb6267fc)}, {LL(0x4a8172cd,0x82199f97),LL(0x582165ca,0xf879213b),LL(0x8048de29,0x51bdf94f),L_(0xe7f10d0b)}},
   {{LL(0x75f12d76,0x743c75d7),LL(0x4777a4a1,0xc13528c2),LL(0x0f8f0fc7,0x969b010e),L_(0x6365c47c)}, {LL(0x2acbaab1,0xdfc6ee9c),LL(0xe207ebae,0x23ac352f),LL(0xc00ef611,0xdfc77ba6),L_(0x6470bbbe)}},
   {{LL(0xd561d67a,0xac256d68),LL(0xbe1d04c9,0xb7f44d83),LL(0x1d8cc6f2,0xddd4acc6),L_(0xc01e2b00)}, {LL(0x83d18ef1,0x6b5fbe51),LL(0xc673b3a0,0xe616da91),LL(0x04e7aef1,0x7079d8b4),L_(0x3313c1b4)}},
   {{LL(0xc0f7d6a0,0x2242759d),LL(0xb6fbfcb4,0x9949d06a),LL(0xde898701,0x8b24618f),L_(0x5307474d)}, {LL(0x3976254e,0xb0bd22ab),LL(0x211085ed,0x2125ed2b),LL(0xa5f28d83,0xa1c543ea),L_(0x565a6c4d)}},
   {{LL(0xe7ee1b29,0xf2a06317),LL(0x3813d14f,0x784961c1),LL(0xf759d96d,0x5670b8e7),L_(0xb428b1ab)}, {LL(0xafedb09a,0x8fbfe519),LL(0x478f19a0,0x7246a169),LL(0x7503e23c,0x23ac9af3),L_(0xd24790e7)}},
   {{LL(0x87836e62,0xc31169c7),LL(0xbcb98921,0x111050a3),LL(0xa7810165,0x446bda32),L_(0x6ad438a4)}, {LL(0xa876bcf9,0x32692fd4),LL(0xc750819f,0xd11e3c6e),LL(0xe2aece33,0xd325fefc),L_(0xb6f2bf6f)}},
   {{LL(0x8c31bbfa,0x0e3623bf),LL(0xa2285b20,0xbb5c266c),LL(0xc5f4aaef,0xa7179b54),L_(0xdf6212eb)}, {LL(0xed087187,0x375cb291),LL(0xd118abf6,0x047af635),LL(0x73198d27,0x8f1c0b06),L_(0xeac7c4ea)}},
   {{LL(0x77e38348,0xc22c3ac3),LL(0xd0ae073a,0xea1c7779),LL(0xab8de890,0x8b9d1132),L_(0xb336a43c)}, {LL(0x2a2fde97,0x78f20c5e),LL(0xf71fbefe,0xa11f0040),LL(0x302a6212,0xba7689dc),L_(0x6a6519f6)}},
   {{LL(0x8c2d9e20,0x50cda9df),LL(0xb3180610,0xa6158df8),LL(0xadbbeaa9,0x4778c93d),L_(0xc891e9d3)}, {LL(0xd1a49e0a,0x6eb6664c),LL(0x68d58278,0xb72ac28b),LL(0xb8fb2671,0xfc493918),L_(0xbe9a2c58)}},
   {{LL(0xe846a0c4,0xc3ffb873),LL(0xf33f7123,0xd98b14c6),LL(0xcd3fcccf,0xec8c8522),L_(0xc0d09a5d)}, {LL(0x2709309a,0xc435552e),LL(0x393794e6,0x731b5ca4),LL(0x6d3545bb,0xcf95c612),L_(0x21f92c49)}},
   {{LL(0x67e83945,0x77e47a85),LL(0x8aaaa2f8,0x41b4a73b),LL(0xf14ab27f,0x45673153),L_(0xf95ee4e1)}, {LL(0x29737736,0x3a31ee99),LL(0x3add55e3,0x89cb9a71),LL(0x3c610ccb,0x92d6ab9e),L_(0xa1df471f)}},
   {{LL(0x24dd0313,0xfe3af3cf),LL(0x3ad69446,0x1fa695ef),LL(0x49e9da9d,0x35d8485e),L_(0x823de577)}, {LL(0x1be78d4e,0x9a8e5ede),LL(0x535233c3,0xdb4e20a2),LL(0xb9902b0e,0x762073cc),L_(0xb355fbea)}},
   {{LL(0xba8c73ac,0xfd983b8b),LL(0x0f082d41,0x3921c9f7),LL(0x9c6ee5d8,0x7f5a5177),L_(0xd9ef6090)}, {LL(0xab53e7b3,0x7db07019),LL(0x14b4f4d5,0x01f9a8fb),LL(0x2588492a,0x9ea1e98d),L_(0xeb443742)}},
   {{LL(0x31997789,0xa9a0407f),LL(0x4e820720,0x26fdd725),LL(0xa3068d9c,0x2aba0c14),L_(0x6db14c6a)}, {LL(0x6d072830,0x2d5ed5c3),LL(0x0acce5d9,0x52b3241d),LL(0xdbd63903,0x11f4b8bc),L_(0x04c731fa)}},
   {{LL(0x35994c38,0xccc2b918),LL(0xbc10285b,0xf27570ca),LL(0x870c688b,0x8c91a8af),L_(0x775fff54)}, {LL(0x5459f2f6,0x9157cd7c),LL(0xc2e98dfc,0x021d9844),LL(0xdbbbf2c4,0x3319b3fc),L_(0xba743dee)}},
   {{LL(0xaeba325f,0xaecdc264),LL(0xf477eda2,0x9b24587e),LL(0x859c1b6a,0xf7648c83),L_(0xb5c8f8dc)}, {LL(0x7005a3f4,0xdf465967),LL(0x10b24908,0x2411ca0d),LL(0xd689abfe,0x1452c503),L_(0x625ecb92)}},
   {{LL(0xcc290bf4,0xf34968d7),LL(0x9003a913,0x7228d7c0),LL(0x48e02e21,0x17e25b2b),L_(0xd5f95bbb)}, {LL(0xa53a154c,0x1035d2b6),LL(0xac56a6bf,0xe8ab36c5),LL(0xbcf0eb3e,0x63fa883e),L_(0xdf08d138)}},
   {{LL(0x0b53b1ce,0x2088982f),LL(0xe8ed9994,0x21364c97),LL(0x45bf87a7,0xeeb978c5),L_(0x5b94bef4)}, {LL(0x4e53377c,0x093e9c15),LL(0x58acb871,0xd8249f4f),LL(0x02b46436,0x64f236ee),L_(0x88a9e12f)}},
   {{LL(0x3940ee67,0x6164d05e),LL(0xf4a7c705,0xb0056da2),LL(0x872642d0,0xa7bd3d66),L_(0xa407f0cd)}, {LL(0xce37f407,0xd8371363),LL(0x592b0f71,0x90362154),LL(0x009ca301,0xe59d0b2d),L_(0x3dd8f9c6)}},
   {{LL(0xcf247be0,0x73b26b51),LL(0x31ef23aa,0xdf33ad4d),LL(0x1f6d55bd,0x4ed64955),L_(0x7ed356bb)}, {LL(0xa18aee16,0x8ca23082),LL(0x82214ca5,0x20da76f8),LL(0x2453f147,0x4606958c),L_(0x3b5ee2c3)}},
   {{LL(0x99950739,0x06b63065),LL(0xb845a47f,0x9853453b),LL(0x88e04e19,0x456e72cc),L_(0x05cfd664)}, {LL(0xb168037c,0xd3c22f8d),LL(0xcaf525d3,0xd989f0f6),LL(0x9699fc87,0x273ac5b1),L_(0x0349e937)}},
   {{LL(0x273a933e,0x24f52dc3),LL(0x825de5df,0x36bc3c34),LL(0x3e91b0cc,0x61de4790),L_(0x7a99b424)}, {LL(0x8f4b678a,0x44bdcadc),LL(0x0a060400,0x830bedee),LL(0xbb9ac7e8,0xbda5fd11),L_(0xa71c4ee0)}},
   {{LL(0x37a97135,0x079c1b7a),LL(0xac1ea75a,0xeaf11508),LL(0x3012e200,0x61b63246),L_(0xbb79e1c3)}, {LL(0xc0246bd9,0xb92f89e2),LL(0x95a4a7cf,0x3fff4a96),LL(0x44cf1045,0x6e70ae3f),L_(0x5783bd56)}},
   {{LL(0x6615d343,0xb3a83dd9),LL(0x5e8bc7ab,0x0c4c00f6),LL(0x6cbcc29c,0x84d9fe95),L_(0xaf97e676)}, {LL(0x30e1885c,0x82a4da98),LL(0xbe563af0,0xbcc23ab6),LL(0x259db6ce,0x306cf56e),L_(0xa179a030)}},
   {{LL(0x08624c7c,0x32c9c5fc),LL(0x37d48ba3,0x38295bfa),LL(0xfb49c594,0xdcfe3634),L_(0xc9dd0809)}, {LL(0x832aec93,0xe5cda111),LL(0xebcbf38d,0x98fc1727),LL(0xac496bac,0x3f057842),L_(0xefc69a01)}},
   {{LL(0x7bdd13e8,0x189dee30),LL(0xc45d637e,0x01cf6015),LL(0x02a07254,0x7f6053e7),L_(0x9648ef85)}, {LL(0xc9d79b49,0xd467929d),LL(0x221a964d,0x2a4257b0),LL(0x3ed68c3c,0x0209a951),L_(0xbf635477)}},
   {{LL(0xb50fa3a9,0x568377b7),LL(0x57a50eda,0x992fa9d6),LL(0xf3995e07,0xf09351e3),L_(0x21813534)}, {LL(0xde00386d,0x7c21c039),LL(0x5bd67bbb,0x983e3f5f),LL(0x970f1581,0xfde7bb8d),L_(0x50e663c0)}},
   {{LL(0xa5991d36,0x727713ca),LL(0x52a00363,0x81781a15),LL(0x946557d9,0x8b2de3fc),L_(0xce2f349c)}, {LL(0xb61c0936,0xefe6ec67),LL(0x9c584327,0x0483381f),LL(0xbfd4710c,0x3cb8617a),L_(0x2e9c6a8a)}},
   {{LL(0xfb2c55b7,0x83398c47),LL(0x00610b5b,0xfc55bdd9),LL(0xd6aad463,0x733b8806),L_(0xe6434da6)}, {LL(0x600c9641,0x72a69c3e),LL(0xc959c3b6,0x25c8da00),LL(0xa863f55c,0x6b73f445),L_(0xb25ba2b8)}},
   {{LL(0xb5811bfc,0xdea6b1b8),LL(0xc85bb876,0x6f29d6aa),LL(0x244c832f,0xb885707b),L_(0x700946f8)}, {LL(0x19847d9e,0xbb7684f6),LL(0xf1284753,0xc0f2d53b),LL(0x20ef5736,0x9e06880d),L_(0x94cf829c)}},
   {{LL(0x9a3cdc82,0x6fd2d1fe),LL(0xf6676bfc,0xfe48b29e),LL(0xa14e785b,0x7afeb3e3),L_(0x73a22b1e)}, {LL(0x15c1eec0,0xdc5c271b),LL(0x738c09b5,0xd51bbdf3),LL(0x4095982d,0xea6a656f),L_(0x4e8a1c04)}},
   {{LL(0x8df3827f,0xad96082c),LL(0xe21e88ec,0x0c7308d3),LL(0x747332eb,0x3e790568),L_(0xcbe74f24)}, {LL(0xb410e2a6,0xa45bcc07),LL(0x62e28f16,0x34e11697),LL(0x68990fab,0xd0d95f71),L_(0x0bc4045b)}},
   {{LL(0x2b70f4c3,0x95663747),LL(0x718a4989,0x18e76685),LL(0x0510d4ae,0xa806fc64),L_(0xb9f7725e)}, {LL(0x74a39c46,0x11681c31),LL(0xab7986bb,0x7f1ee73d),LL(0xd70c5b25,0x145296c5),L_(0x8e746d97)}},
   {{LL(0xb90e977d,0xa73e17c7),LL(0x914b838f,0x3b0459a9),LL(0xd910c80d,0x4a996387),L_(0xc7523409)}, {LL(0x09cc1463,0x4ff007ad),LL(0x3feccef8,0x6f3bf601),LL(0x9137e94d,0x3d0461a8),L_(0x5dfeb98d)}},
   {{LL(0xdf5e90f7,0x03f3fdc3),LL(0x3d7cbec2,0x2f05f4ba),LL(0xefe1e30f,0x5dab20a6),L_(0xb6114a8c)}, {LL(0xfbe83a6b,0x815eac18),LL(0xbcc8bbc2,0x2c89a659),LL(0x13f75b6a,0x326a87f4),L_(0xda0c7f42)}},
   {{LL(0x99fa6315,0x29c19e9b),LL(0xdf48509e,0x5d5a6ffa),LL(0x84d62c60,0x4e8640e3),L_(0x4a85f9d6)}, {LL(0x01830a3f,0xa50855ff),LL(0xea7061d4,0xb9b3fb5a),LL(0xbf0727b4,0xa1cac932),L_(0x9c76f2b9)}},
   {{LL(0x08fe3e55,0x7f8ba9f8),LL(0x3a13632e,0xfff3b260),LL(0xa621223a,0xda5b8cc0),L_(0x6cf87377)}, {LL(0xb352cab0,0x52e875c5),LL(0xb3cb3de2,0x888687ca),LL(0xd159fefe,0x788f0705),L_(0xa3f5461f)}},
   {{LL(0xe7ee73d4,0x71f9e278),LL(0xa10ec3f4,0x0511bea2),LL(0x147b2ef8,0x908389b2),L_(0xaf59635c)}, {LL(0xb9505162,0x7ab5b1bb),LL(0x05d9bd3a,0xa16a5843),LL(0xeb15fb80,0x8840233d),L_(0xf2f61b05)}},
   {{LL(0xfbf89124,0xb350235f),LL(0xa7d12321,0x601ae7a3),LL(0x487a5b44,0x168cfefc),L_(0x124de1e6)}, {LL(0x27f1099a,0x3f7f4917),LL(0x387a3135,0x576cbe65),LL(0x93695572,0x1697b707),L_(0xe9cd239f)}},
},
/* digit=24 base_pwr=2^168 */
{
   {{LL(0x1aa93495,0x07822bd8),LL(0x98675e63,0x690a044f),LL(0x411c39e9,0xddfd5a23),L_(0x768594da)}, {LL(0x5d8da0b0,0x74dcec38),LL(0x3c14e211,0xa51c46ee),LL(0x6b77f2b0,0x1367898a),L_(0x8ad0bfdf)}},
   {{LL(0xd8b1efbb,0x5d75e9ca),LL(0x940ae4e4,0xdea9f60e),LL(0x7af3d6fd,0x95065ae4),L_(0xef0f21e7)}, {LL(0xe0c7e801,0x22ced186),LL(0x9502622d,0xc6698d7e),LL(0x95367fa5,0x9310a67e),L_(0xc864dc25)}},
   {{LL(0x7322a3b3,0x1b5b3f21),LL(0xe6b25a83,0x1fe6e87a),LL(0xcad2ee21,0xb5fd4e5b),L_(0xd772c552)}, {LL(0xe5ae89d5,0xc2853eeb),LL(0xc45afb20,0x9036d2cf),LL(0xcb159584,0x45627827),L_(0x6e07de7e)}},
   {{LL(0xf341190a,0x2470f5ee),LL(0x87e42cc7,0xf0be84fe),LL(0xade34d2c,0x90636c20),L_(0x0cf8cafa)}, {LL(0x180fa04d,0x2db09b0f),LL(0xd896b5d9,0x0a0909a9),LL(0x83bcc559,0x281cf9aa),L_(0x4d7d88c9)}},
   {{LL(0x38d6fee1,0x8f859667),LL(0x0e93537c,0x86559609),LL(0xca6e59bf,0x6854b4e8),L_(0x4c2e89d8)}, {LL(0x23a24b88,0x5840ce8a),LL(0x6366e9c0,0x946b6bbc),LL(0x00fd3d85,0x8a095b45),L_(0x9e0c589e)}},
   {{LL(0x74d544f2,0x4c355669),LL(0x4696a73a,0x1ea60e81),LL(0x099e116c,0xd960a70d),L_(0xea914344)}, {LL(0xd60a4a5a,0x305faf07),LL(0x1026dbdc,0x61b80d24),LL(0xe6b8e5c2,0xd428e0cc),L_(0x2cd02a61)}},
   {{LL(0x26d21296,0x5d2ed352),LL(0xf7a07ac1,0x3daf91ed),LL(0x44afa3c4,0xd529e7b2),L_(0xdd22a012)}, {LL(0xd7ed0b48,0xcc9faccf),LL(0x00535729,0xc98aa7de),LL(0xc9757998,0xce2c4e14),L_(0xe8ccc048)}},
   {{LL(0x098fc504,0x3d3f8532),LL(0x5cb9d339,0x50253fa0),LL(0x23d266c8,0xb38286d6),L_(0xe5172f5c)}, {LL(0x99903093,0x48a64b98),LL(0x8c64987d,0xbc43643a),LL(0xfe159bd8,0x2f08eae4),L_(0x049c111f)}},
   {{LL(0x5b31f9b7,0xc558d554),LL(0x0d2340eb,0xe3c46403),LL(0x19a6f592,0x2211131c),L_(0x4ef1dc2e)}, {LL(0x4d8f19e8,0x8c4d4f34),LL(0x107caba4,0x347e8cae),LL(0xad36aa89,0x5a959e3c),L_(0x4495ae4a)}},
   {{LL(0x178db2de,0x8674b5e1),LL(0x68e6ec47,0xce21dcf1),LL(0xd3884f45,0x7664513a),L_(0x60d36975)}, {LL(0x7a1b3e57,0xc9e299e2),LL(0x8015c45b,0xe96de0f0),LL(0x8d3f0ffc,0x5aba434e),L_(0x448fc469)}},
   {{LL(0x3206b5a0,0xbf15b53c),LL(0x0516d5b4,0x2fff52c6),LL(0xfec55940,0xc3791f65),L_(0x797a6b95)}, {LL(0x394de0c2,0x5c5fed8c),LL(0xeba16248,0x674b4d58),LL(0x0e57743b,0xd7315975),L_(0x87252f57)}},
   {{LL(0x7fa4f80b,0xc574aafe),LL(0x9e9fc1b3,0x25994c44),LL(0x15293405,0xa00f9231),L_(0xc2527f9a)}, {LL(0xf98edea6,0xd0bcf974),LL(0xbe7b43fa,0xa3c74ad5),LL(0x448873ac,0x8ff67b2d),L_(0xf0dfe6da)}},
   {{LL(0x510cb9a5,0xbaf15c58),LL(0xcabedab3,0x234a8faa),LL(0x0023d4d4,0x9d6e89b9),L_(0x8a188d72)}, {LL(0x37485100,0x6bf4faef),LL(0x5f70a85e,0xdd164001),LL(0x799cb3ee,0xfc7797a7),L_(0xd331f4bd)}},
   {{LL(0x34261e55,0xcf17aa4d),LL(0x7f8c3542,0x801ccc34),LL(0x0c7a04fa,0x762b3408),L_(0xd90eff09)}, {LL(0x002ca132,0x0870f41d),LL(0x767451ee,0x70686201),LL(0x276eaa26,0xf592842e),L_(0x6a87509c)}},
   {{LL(0xb78b71bd,0xa79ca9e4),LL(0x31d15ca5,0xe7aab4c3),LL(0x095aa863,0x980df4d9),L_(0x30800a25)}, {LL(0x75ac5ed9,0xcc69d2f9),LL(0xee44fb88,0x429f98f1),LL(0x10ce7510,0x5736d4ee),L_(0x03cc9ae7)}},
   {{LL(0x8da51d85,0xb2abe419),LL(0xbd89ecd4,0xefa624dc),LL(0xc275c72e,0x8b684df9),L_(0x0c47b707)}, {LL(0x75f1bf6f,0xfc8f823d),LL(0x06225cca,0xa497166b),LL(0xb3363382,0x3241d9b6),L_(0x397d4140)}},
   {{LL(0x9a65f284,0x47d0d598),LL(0xd2dae790,0x600f8097),LL(0xdfc5d530,0x06c1d340),L_(0x18b74aa7)}, {LL(0x409b596c,0xe50cc47b),LL(0xb9309f95,0x71018c41),LL(0x4f9b2683,0x2de345b4),L_(0x73d97fe3)}},
   {{LL(0xd352b3da,0xfb7e2c37),LL(0xd29b2279,0xfb5035db),LL(0xc27af22c,0xe5cf5a24),L_(0xdd8c403d)}, {LL(0xf2e49e04,0x31c42940),LL(0xbe767afa,0xb3e476b1),LL(0x3130439d,0x729a7e6b),L_(0xcfb2c6fd)}},
   {{LL(0x554fbba5,0x1228460b),LL(0xfba91910,0xda7db076),LL(0xf5e0be2b,0x8a989e94),L_(0xd04b5940)}, {LL(0xe15aded2,0x56931b09),LL(0xe96ceb9d,0x7fb6175a),LL(0x36407d3f,0x785ad12c),L_(0x412c5e80)}},
   {{LL(0x296e5cc5,0x6178d140),LL(0x9e4845c7,0x9088d05a),LL(0x1f9dc468,0x631396d3),L_(0x8b994c1d)}, {LL(0x86ffc6ce,0x04f52e9a),LL(0x2ffa498e,0x4b21e0a8),LL(0x27375e3f,0x242c7c60),L_(0xd99a4851)}},
   {{LL(0x064ca07a,0xa02c0b2b),LL(0x518bd6af,0xa1558503),LL(0x13a69d2a,0x3d67919d),L_(0x68b0b3b5)}, {LL(0x2bfc83c0,0x3a653e93),LL(0x24e20cb0,0x69ae9c82),LL(0x0a4cc47c,0xf624d321),L_(0xeb72aa98)}},
   {{LL(0x9e3e45d0,0xd947c8c7),LL(0x1032ccc8,0xa439634d),LL(0x5b100526,0x7b5067f8),L_(0x534219ea)}, {LL(0xfc26652e,0x701d8d64),LL(0x1e89c3fb,0x277501f0),LL(0x262c9dad,0xd3dd957f),L_(0x48bd1e47)}},
   {{LL(0x9f4be9d8,0xcf4ac2d6),LL(0x2a7df2ab,0xe04b6794),LL(0xf3257cbf,0xec24a41c),L_(0xf6596f8e)}, {LL(0xb3eb3d14,0xcc1689ae),LL(0x7d52d65d,0x8712e8c6),LL(0xafd7b5e3,0x868af920),L_(0xb7ad63c9)}},
   {{LL(0xd27017ec,0x75a3408c),LL(0x8af38290,0x7de280ae),LL(0xfab61ca0,0xb76f92d0),L_(0x9df539d9)}, {LL(0x528c5329,0xffede226),LL(0x4363c147,0x9bf76d0a),LL(0x4d2256e2,0xdd96e876),L_(0xc1afd4ff)}},
   {{LL(0xe00a0553,0x896685cb),LL(0x48da5f08,0xf0d4a520),LL(0xe8f5fdb6,0x0f380c61),L_(0xb146db8e)}, {LL(0x4d63b6c7,0xca87cdad),LL(0xcfdcfe1c,0x69639e2c),LL(0x941d9935,0xed61f0e4),L_(0xad7751a2)}},
   {{LL(0x72ebb947,0x43d0a9df),LL(0xdb90e6a7,0x53424333),LL(0x10a77134,0xdf849ce1),L_(0xce2f9c06)}, {LL(0xdf56c5f4,0x58feabaa),LL(0xc5d546de,0x787ed1e1),LL(0xa802a275,0xa542b49b),L_(0xa5e7ea27)}},
   {{LL(0x0646662c,0x33793ddc),LL(0xde0180db,0x873a716a),LL(0x3ae6c62d,0xd34ef5f0),L_(0x8ffbd17e)}, {LL(0x9974e0c2,0x5b201375),LL(0x61aa17ae,0x25cf8d2d),LL(0xfac50e53,0xf5028682),L_(0x4cb76558)}},
   {{LL(0x20822d1a,0xdb5b7f7b),LL(0x42e0c365,0x610bf8a8),LL(0xe4e5ddab,0x59f6f46f),L_(0x7d6ff0da)}, {LL(0xd3a9f7a1,0x29ffdb56),LL(0x95d9dd8c,0x1aed0fa8),LL(0x7cd570f8,0xa1cdd0bb),L_(0xf42c2ded)}},
   {{LL(0xb9d66a96,0xf9463b5d),LL(0x7ab3d377,0x9dcb5190),LL(0x7e8de98d,0x3a174474),L_(0xf096dc4f)}, {LL(0xd8a6df5c,0xb00ae812),LL(0xc11335ee,0x595dbbd8),LL(0xab858cbb,0x20456b6d),L_(0x50149878)}},
   {{LL(0x6e092bb5,0x4b66cf94),LL(0x20f71995,0x7eca44eb),LL(0x6daca5c9,0x44fbdf91),L_(0x06ba2abf)}, {LL(0xcb39349b,0x9f292892),LL(0x5db8d756,0x7d2e1348),LL(0xb91c9d21,0x2511f796),L_(0x03664b0f)}},
   {{LL(0xc742fe69,0xaeca436c),LL(0x07bb4010,0xa736c99b),LL(0x7053b3ba,0x4df3a08c),L_(0x7269f130)}, {LL(0x35ff874c,0xa50437d0),LL(0xa2239752,0xbb3ee79b),LL(0x28f141d8,0xa309d53a),L_(0xe3b40a03)}},
   {{LL(0x052107c2,0x4c1843ed),LL(0x92f03f51,0x5b395716),LL(0x10eeec3a,0x41bed6c7),L_(0xe7590252)}, {LL(0xe3bf062e,0xe2632a75),LL(0x61293b51,0xccb37627),LL(0xf5482813,0x65bddca7),L_(0x3882b6c4)}},
   {{LL(0x97f943de,0xe4f707eb),LL(0xaed3b2b2,0x1fca8433),LL(0xe200b711,0x5979d82e),L_(0x79d9de90)}, {LL(0xa0864854,0x073f049c),LL(0x5ba6b65e,0xefca8545),LL(0x5e115ac7,0x81b25231),L_(0x811506c8)}},
   {{LL(0x0b83214f,0x6e208498),LL(0x4f6ef89a,0xd7f17a61),LL(0x9d606b9f,0xf9ff6be9),L_(0x05c6d274)}, {LL(0x78a1ebcd,0x8412a690),LL(0x1b349b3c,0x3fc39e84),LL(0xa4039f18,0x16cc85fa),L_(0xacd2b66e)}},
   {{LL(0xb29dd0f6,0x3f372e07),LL(0xb2f85965,0x1010f76a),LL(0x499ca40b,0xc9d3a123),L_(0x494c1c41)}, {LL(0x1a039621,0x898b067c),LL(0xc21db378,0x8fba57f1),LL(0x920244b9,0x2f20dfbe),L_(0xafa449b9)}},
   {{LL(0xadba6514,0xf1db6fa1),LL(0x0a3087ba,0x780eff8d),LL(0x6530b59a,0xdcf21d60),L_(0xf6198373)}, {LL(0x767463fe,0x65b13c4b),LL(0xff99e886,0xaf0f5939),LL(0xb10b6a9a,0x05c89578),L_(0x713231a5)}},
   {{LL(0xae7f781a,0x678205ed),LL(0x46236fd1,0xcc32c0af),LL(0x2492206a,0x43d4ee55),L_(0xfbf9ba4c)}, {LL(0x2296a220,0xd485f134),LL(0x17487172,0x7dae00fa),LL(0xd7fd8f23,0x9c357857),L_(0x48861bc1)}},
   {{LL(0x683b4c53,0xd1014606),LL(0x084b9990,0x2de4b78e),LL(0xed10d363,0xe5547d1b),L_(0x3d2f5f09)}, {LL(0x40cf618a,0x1fa00e4b),LL(0xf090adf3,0x2643b488),LL(0xa90cf1b7,0x862ff071),L_(0x7b03dffe)}},
   {{LL(0x2c263fe4,0x3df3e881),LL(0x74c2dd8a,0x4a452825),LL(0xf8b4e248,0xff9860e0),L_(0x0632b8bd)}, {LL(0xfaa7bd3b,0x4dcae86a),LL(0x6c838bca,0x0392152c),LL(0x8e97fa6c,0xfc62b042),L_(0x1616bf3e)}},
   {{LL(0x38ca48d5,0xc8c6f843),LL(0xafab4216,0x1a5701ad),LL(0x8137a142,0x3332d11c),L_(0x073d30ce)}, {LL(0xf65f8d03,0xc57e0eee),LL(0xc5ec9bae,0x84c4a8f0),LL(0x474d64c3,0x9a310ea4),L_(0x279cd2cf)}},
   {{LL(0x90943f9d,0xabcd06f1),LL(0xdfc2230e,0x4bf7868a),LL(0x2460277b,0xf15a1c15),L_(0xe0531432)}, {LL(0xbbc35210,0xa0084581),LL(0xff9171d1,0x8c12c483),LL(0x4461e279,0x2cb0b794),L_(0x18f08ce5)}},
   {{LL(0xc6234888,0x1336960e),LL(0xd82f38a5,0x6690aec6),LL(0x8dcc7988,0xea8d5acf),L_(0xbdf09963)}, {LL(0xad19e667,0x221a923e),LL(0x5ebdb79e,0x3d295ba2),LL(0x840fc746,0xa849c9c6),L_(0x95854d40)}},
   {{LL(0x796ddb68,0x7f2ece47),LL(0xb700fc8f,0xfa40fdb7),LL(0x6e9b86f6,0x3753a599),L_(0x004dc14e)}, {LL(0xb0d32904,0x9ce52e22),LL(0x9fb7c051,0x609b3582),LL(0xbd4bbc16,0x470387a5),L_(0xae8123ae)}},
   {{LL(0xb5992b13,0x15e316ec),LL(0x34a4f3f6,0x331c37c2),LL(0x4f66d67c,0x5a547f8e),L_(0x3d89818f)}, {LL(0x2343f37e,0xc7ecc98e),LL(0x464eaafa,0x2f36fbc0),LL(0x32fd452b,0xbcd300ae),L_(0x52cb6226)}},
   {{LL(0x3965b7dc,0xfbb383d5),LL(0x8e1be53e,0xd5ddeb7e),LL(0x5de1379b,0x89236512),L_(0x1618ecf2)}, {LL(0x71474c33,0x359ac63d),LL(0xad2f7a4e,0x6a2a41ed),LL(0xb41d45a2,0x4543bf02),L_(0xd97bbddf)}},
   {{LL(0xdd45b155,0x51a93461),LL(0x835fa995,0x8e87a379),LL(0x6030bb55,0xe0f74319),L_(0x3fce8a20)}, {LL(0x180c566c,0xf64d3bd0),LL(0xf4320f55,0x117f7e04),LL(0x9fe6d9b3,0x276cdeae),L_(0xe11e04e4)}},
   {{LL(0x90d2c4b0,0x8ee8e755),LL(0x936a235c,0x34e9da69),LL(0xea446558,0x56f8560b),L_(0x868fe100)}, {LL(0xf104982e,0x7a1b70ed),LL(0x2d932107,0x5df6c183),LL(0xb3a79a2f,0x3e690209),L_(0xddd8c4fc)}},
   {{LL(0x506cdad3,0x1cd413bf),LL(0x8343be89,0x5af9735a),LL(0x4669ee32,0xf133df4e),L_(0x495a7ffd)}, {LL(0x343b9134,0xd2df1205),LL(0x08509117,0xc0f593eb),LL(0xe1f7e724,0x03f70d2b),L_(0x56807867)}},
   {{LL(0xe1aefce1,0x28e77727),LL(0x910ee9be,0xea43f71f),LL(0xb1afcd33,0xe72f4294),L_(0x04218f35)}, {LL(0x5c609191,0xd85d4064),LL(0xe01905f3,0xd4734dca),LL(0xc6a7ed93,0x73444dba),L_(0xbd20eb77)}},
   {{LL(0x786041c6,0x2e784b12),LL(0x3b8e5678,0xa5394485),LL(0xd5ed489c,0x7c1354f6),L_(0x37f214ac)}, {LL(0x6f5aed21,0x04a2cef9),LL(0xfdcc232c,0x6a61903b),LL(0x0943ad59,0x74525b1c),L_(0x36735571)}},
   {{LL(0x41295f9f,0x79058179),LL(0x53c2435b,0x8356b028),LL(0x8630275b,0x7d99704d),L_(0x22b975d5)}, {LL(0xfe64fec2,0xb6ad9348),LL(0x1275938a,0x323c5204),LL(0xff571db9,0xabd2794c),L_(0x69883cf4)}},
   {{LL(0x39e5ddd8,0xa00ac9eb),LL(0x0825e1a9,0x08022d52),LL(0xec00b4ca,0x89c305e2),L_(0x39410e3e)}, {LL(0x2f6d8e33,0x1c9dee31),LL(0x189d73ba,0x063cbbc8),LL(0xd453f2df,0x500b8e1b),L_(0x200614c8)}},
   {{LL(0x0d2d67f6,0x704fba9c),LL(0xe3b26afd,0x3b601782),LL(0xc86eeaf3,0x3a076280),L_(0xd5b4a74b)}, {LL(0x057735bb,0x3c7fe621),LL(0x67519f38,0xba90bf98),LL(0x7ea681b4,0xc697f17d),L_(0x399fb8a8)}},
   {{LL(0x2fb2d013,0x8b404954),LL(0x225dd20b,0x3864cc2c),LL(0xe8bf6c74,0x27928a4a),L_(0x0dbd4ced)}, {LL(0x129459b4,0xb079e9c8),LL(0xcdaab48a,0xa8c506fc),LL(0x2be35d42,0xabd7b8a2),L_(0xc81bd43f)}},
   {{LL(0x0c337ec0,0x9f5cd253),LL(0xb8d68d49,0x3bc3df03),LL(0x808378fd,0x57ce1b57),L_(0x7280fc5c)}, {LL(0x9173aca9,0x29069aea),LL(0xec3e1405,0x172c5e75),LL(0xd7f9c6b8,0x2bb82de1),L_(0x7c6b6663)}},
   {{LL(0x069b62dc,0x3a04b8c8),LL(0x539ec109,0xbb1fb7ee),LL(0x8e5dc130,0xdf0fbfa5),L_(0xd9cdb4bd)}, {LL(0x8d6bb599,0xae962471),LL(0x8c98d809,0x6a0b5063),LL(0x1f5aad58,0xcb90fbe1),L_(0xbd8451cf)}},
   {{LL(0x99c6500b,0x182ee0a7),LL(0xf250978f,0x9acab63a),LL(0xd7a3f799,0x579dffc7),L_(0x0a06a99d)}, {LL(0x2e7244fa,0x7a563978),LL(0x780bf6d7,0xd9421c12),LL(0x4a33607f,0xca35ca5e),L_(0x4d6baab2)}},
   {{LL(0x476c2c17,0x84dd3019),LL(0x1020ff66,0x9f6a9cbb),LL(0x8de41867,0xa0782cd2),L_(0x30b339ca)}, {LL(0x335870d7,0x873bb057),LL(0xb2102899,0xd2763264),LL(0x817bb568,0xac61e5ba),L_(0x7167f834)}},
   {{LL(0xb4216df4,0x28a00ba3),LL(0x72603953,0x8d18d559),LL(0x83dc382b,0x9929ba4b),L_(0xd887ba34)}, {LL(0xc4acbdda,0x1a149f46),LL(0xabc8bd9c,0x03677afd),LL(0x7f382dd5,0x038b4fd6),L_(0x531a5b69)}},
   {{LL(0x7b1469c6,0x770906b4),LL(0x29957782,0x39b76a30),LL(0xb757175c,0x77b00db2),L_(0xc63e688a)}, {LL(0x00e5eb13,0x564b2b5a),LL(0x486048a7,0x16670a8b),LL(0x6cedfc0a,0xfc26f8d3),L_(0x158c04cf)}},
   {{LL(0x5f72f036,0x492f15fb),LL(0xaf524838,0x8a57f34d),LL(0xca278cc2,0x136d3def),L_(0xf06989cb)}, {LL(0x56bc1e3e,0xb7d079d2),LL(0x48aa5343,0xb849bec9),LL(0xc12eafab,0xf42c9064),L_(0xcdcc4178)}},
   {{LL(0x35110b42,0x3593434b),LL(0x826b0084,0x2a685ee5),LL(0x38e7d616,0xd80d02e7),L_(0x3dd474b0)}, {LL(0x47c0e164,0xa5dbbbef),LL(0x177aaf40,0x58c7a38b),LL(0x00b0f102,0xbe4a965e),L_(0x97ef9f54)}},
   {{LL(0x00ea08ca,0x0a49eb36),LL(0xb9e97b18,0x4a5b799b),LL(0x1389e249,0x6bb607bd),L_(0x5f402141)}, {LL(0x21dd6f71,0x648ac9f7),LL(0x0d6762ea,0x46ddb53b),LL(0x123f4a38,0x0b454608),L_(0xffc09d45)}},
   {{LL(0xdc0b4f06,0xa26650ba),LL(0xc51c49ab,0x9623d15c),LL(0x52a5b5a4,0x8530847b),L_(0x9328fa21)}, {LL(0x2b9634a5,0x88564796),LL(0x4c84e96b,0x8cd268e3),LL(0xdd09c6f7,0xfbbf3c6c),L_(0x6ae89ca8)}},
},
/* digit=25 base_pwr=2^175 */
{
   {{LL(0x076ba1e3,0xaf9141d1),LL(0x7d1bf374,0x90d62386),LL(0xeae39108,0x39ae931a),L_(0xbf9ff96f)}, {LL(0x7c192921,0x51f12b8a),LL(0xf7008bde,0xf1cfbc62),LL(0xaa15f8c6,0x26ff0bea),L_(0x10e47e4d)}},
   {{LL(0x5ac7dc32,0x5a5fa2f2),LL(0x727e3254,0x918f6160),LL(0x19e26cfd,0x3d611b68),L_(0x60d5395b)}, {LL(0xfbe0d265,0xa4b3a16e),LL(0xc823108a,0xab7e3e37),LL(0xfb87b2d4,0xe4cf7a53),L_(0xa746dda9)}},
   {{LL(0x28c21969,0xe6afbadf),LL(0xc63c394e,0x08d0d899),LL(0x79f622a4,0xfc61f01d),L_(0xbbd70173)}, {LL(0x9afa4015,0x1819a677),LL(0x9acf1a59,0x9e89a4e9),LL(0xf6189381,0x4c306973),L_(0xf09c6806)}},
   {{LL(0xd0fd8e2e,0x85f596e9),LL(0x8c824b93,0xc650d109),LL(0xc0ff364a,0xaa432f6b),L_(0xe94aff61)}, {LL(0x60cd9ff7,0xa81387b4),LL(0xafe1dcb8,0x1fb57eae),LL(0x0f74dad8,0x77ebce28),L_(0x461e02cb)}},
   {{LL(0xe7801dc0,0xc705f5f3),LL(0x19eba5b4,0x892e14fd),LL(0xee1d51c0,0x235f35c6),L_(0x32c28b10)}, {LL(0xc690d3c0,0x496db46c),LL(0x35bd49b1,0xf95db976),LL(0xa0c8c115,0x3a2ff859),L_(0x347bf13c)}},
   {{LL(0x21208e25,0x50a51554),LL(0x37fdbb13,0x76da2c79),LL(0x739f33c4,0x47f7655f),L_(0x5268d621)}, {LL(0x6df3da72,0xda653a05),LL(0x63868730,0xe0dbba02),LL(0x7efefa7f,0x8546d123),L_(0xc050a28d)}},
   {{LL(0x63a12c14,0x81bccb17),LL(0xb1d9c462,0xb8ad1f48),LL(0x2ee0f6a0,0xeca64ff4),L_(0x5d641bc5)}, {LL(0x6c14c6a5,0x27d0478a),LL(0x3fb250a4,0x9dc8b9dd),LL(0xe6cf3269,0xde4a727b),L_(0xd5fdb843)}},
   {{LL(0xe9139921,0xafcf2338),LL(0x6d85ee99,0x8626eb92),LL(0x330ff008,0x62bc5b8f),L_(0x75be09d1)}, {LL(0x9ed3ab3e,0xacdfce8c),LL(0xfe5d4b90,0x198a1e99),LL(0x36ceed7d,0x366edc66),L_(0x88f5f0af)}},
   {{LL(0x9eb6ef97,0xa0790c0e),LL(0x3c7a8752,0x8212d5cc),LL(0xa20f0405,0x2e6749a5),L_(0x52b0f123)}, {LL(0x0b5bf3ab,0xf0598778),LL(0xad1df6a6,0x6b2a153d),LL(0xacd67e10,0x9a4287cf),L_(0xbc3d1a68)}},
   {{LL(0x1df015e0,0x5545ae11),LL(0xaefec73e,0xf86d852d),LL(0x34894ebb,0x61b42915),L_(0x36e6b458)}, {LL(0xb25bf6b1,0x2f858332),LL(0xa346bcfb,0x082b80c9),LL(0xa64fc13d,0xdb34263a),L_(0x5546d687)}},
   {{LL(0x6c1819ab,0x46f4e2d8),LL(0xc8c8ebfb,0x3e6d5834),LL(0x31b5f8c4,0x0471eabf),L_(0xb0372dd8)}, {LL(0x7377bdb2,0x4f0b793d),LL(0x1491549c,0x1e0b124d),LL(0x8be72522,0xcb228953),L_(0x28f9ea46)}},
   {{LL(0x3d345d7a,0x4e4eb11a),LL(0x6b6a8f36,0x4f7cb301),LL(0xd3f60831,0x9b37cac3),L_(0x2fa1e5ac)}, {LL(0xd42ac14b,0x4b82b6e1),LL(0x44efee99,0x062085e8),LL(0x293aa2ea,0xdafbaa02),L_(0x22eacbf0)}},
   {{LL(0x66dc4353,0xeee6c065),LL(0xf4fb584f,0xe974bd6a),LL(0x51129e57,0xc6e911b2),L_(0xa817dc5e)}, {LL(0x143fbbc1,0xe567aa72),LL(0x474c4763,0xd98a2bac),LL(0x985f3567,0x4940de3e),L_(0xb5a6e769)}},
   {{LL(0xabba8dd9,0xbc9b4d54),LL(0x51333936,0x09ce7445),LL(0x56df5156,0x869985ec),L_(0x7ef1c4c4)}, {LL(0xd64936f2,0xf9f4dd90),LL(0xb6f830f1,0xb0502e70),LL(0x9b4a0a61,0x2b7784e2),L_(0x5d8609f5)}},
   {{LL(0x228b551b,0x132cf6ee),LL(0xe485fe61,0xb41622cd),LL(0x1166f594,0x4f82b870),L_(0xef156ec2)}, {LL(0xc4a0434d,0x1f3f48d6),LL(0xc59ea2a2,0x185bfa1f),LL(0x0549b016,0xab597880),L_(0x88e57344)}},
   {{LL(0x7783a237,0xcacbfea9),LL(0xa7d644dc,0x37384da2),LL(0xd767d6c6,0x1b40ecf0),L_(0x708738a4)}, {LL(0xcb35e136,0x1bdb10dc),LL(0x946bff60,0x51f82512),LL(0x032e4873,0x86fff3dd),L_(0x4a72093c)}},
   {{LL(0x993762c0,0x3c5e8711),LL(0x97ea967e,0xc3fb58ec),LL(0x5942a654,0xdc39e05c),L_(0x51f715f7)}, {LL(0x4c7c1a58,0x76f183b7),LL(0x24b588e9,0x02f3f5a3),LL(0xc7635031,0x95a5b1bf),L_(0x90ff5787)}},
   {{LL(0x44aeebfc,0x6fdec0d8),LL(0x3c09be94,0x4dccb72c),LL(0xc61c3dc6,0x29d98bbe),L_(0xe68c7671)}, {LL(0x0b4d124d,0x60e1bdf3),LL(0x72ff08f6,0x60907706),LL(0x502b0c26,0x52fb8b7c),L_(0x3862b07e)}},
   {{LL(0xa0c27cd3,0xa30b225d),LL(0xd6902dc0,0x14e5e016),LL(0x90beb6db,0xca71c799),L_(0x63380a26)}, {LL(0x6406e7a3,0x8b449af9),LL(0x785b17f6,0xd95a3673),LL(0xe83116d2,0xc945cb30),L_(0xcbad56e8)}},
   {{LL(0x195a49f2,0x19bb9a9f),LL(0x32abbd2d,0xdd63e900),LL(0x32a666f4,0xbb5bae80),L_(0x8a8b981b)}, {LL(0x71975391,0x7200d75c),LL(0x58755037,0x9bb8f1f0),LL(0x34371f37,0xd14c5988),L_(0x4f91cd94)}},
   {{LL(0x330b4d2d,0x4766d5c5),LL(0x29100509,0x96d443b1),LL(0xb683d913,0x55cd4038),L_(0x60386c8a)}, {LL(0x0b19d842,0x4de7c5cc),LL(0xe7f655d5,0xb34627f5),LL(0x63d637ee,0x05d48e84),L_(0xda1b6f68)}},
   {{LL(0xa9ab961e,0x587b4227),LL(0xaee0b38f,0x9bacdb2e),LL(0xf2fed11b,0x4327e6c8),L_(0xd04d5cf3)}, {LL(0x59f2340b,0xdae690d8),LL(0x98397f13,0x20f9a82b),LL(0x04266881,0xb60ba27e),L_(0x0a08356c)}},
   {{LL(0x4b7ea199,0xf7bd165d),LL(0x00c43b16,0xaff22d2a),LL(0x6e2c180a,0x5fa28be1),L_(0x03759f7d)}, {LL(0x8d8de81a,0x8623fa12),LL(0xd1bcd513,0xf1b013b3),LL(0xecdeea2c,0x1412065d),L_(0x4abdd176)}},
   {{LL(0x949b80c3,0x9e45f562),LL(0x91bfead0,0xeacf4d8f),LL(0x89e47023,0xed86f825),L_(0xe474eb17)}, {LL(0xa46549d4,0xcc9d20c3),LL(0xbb6f6ac8,0xb5aaafc7),LL(0xaad820ff,0xfa9b0b7b),L_(0xd5d711cd)}},
   {{LL(0x56a19c65,0x694699c4),LL(0xa52b01ba,0x2172bedc),LL(0xfcd8d182,0x71a622e6),L_(0xc5676308)}, {LL(0x096356a4,0x5df0e982),LL(0x5f771127,0xbff93f13),LL(0x15499fe5,0x035d8bec),L_(0x54511618)}},
   {{LL(0x8665d51a,0x0b392d7e),LL(0x19051cb5,0xda3e7314),LL(0x27724fe5,0xb0000497),L_(0x180fd98b)}, {LL(0x92676515,0x86de4867),LL(0x0724b0c4,0xbfb7e396),LL(0xc9127734,0xec308376),L_(0x69d91fa4)}},
   {{LL(0x13351e01,0x9ee3edde),LL(0x12d31182,0xe9b0df44),LL(0x039a9289,0x3ba48603),L_(0x776fb73e)}, {LL(0x1bc5a4ee,0x28dbe0e2),LL(0x5d6f8bfe,0x33d13a46),LL(0x683a466a,0x6f0e40d3),L_(0x64598a06)}},
   {{LL(0x49b1e543,0x5467b60c),LL(0xdafb9975,0x63a775bb),LL(0xf10e3ad8,0x555978a5),L_(0x90e960a3)}, {LL(0xe2647c34,0x2d98173b),LL(0xeca130de,0xa46de612),LL(0xd4d665e8,0x8c2f942e),L_(0xd46cdc7a)}},
   {{LL(0x83f96958,0x11159549),LL(0x2f2fe4fe,0xa26fb69f),LL(0xada04407,0xe1ffa9b1),L_(0x89077bf7)}, {LL(0xf63dc16d,0xb8212592),LL(0x23757f3b,0x8f2b706f),LL(0xa84d6d60,0x0952d255),L_(0x867adb63)}},
   {{LL(0xc1ad35d0,0x8063b121),LL(0x94420770,0x2c69c142),LL(0x2eddbe16,0xe20cbf04),L_(0x610d35ad)}, {LL(0xc6a6486d,0x5f1b4348),LL(0x8963cf0f,0xaa7af662),LL(0xbd211312,0xf09747fa),L_(0xa7462c1b)}},
   {{LL(0xb3384571,0xad4b300b),LL(0xeaf252a1,0x7e0408a9),LL(0x92c8f393,0xcdb75950),L_(0x95152482)}, {LL(0x01c71c43,0x98806233),LL(0x65895709,0x4ccfab0f),LL(0x2c91c51d,0x0daddeb0),L_(0x6e5994f0)}},
   {{LL(0x1656bed2,0xf9475d4e),LL(0x7ad0e424,0x29f67636),LL(0x71b2886d,0xce5ffc47),L_(0x5aec9cfb)}, {LL(0xed19aefe,0x00fbe22e),LL(0x335636b3,0x1a593c52),LL(0xed7635e4,0xfea5dbc5),L_(0x0713b394)}},
   {{LL(0x0a800d8a,0x3e952cbe),LL(0x96ad939e,0xb849574d),LL(0x95e45fa4,0xd0b6f06f),L_(0x495cdf6f)}, {LL(0xca47bacd,0x3aea6517),LL(0x627d416e,0x98d62545),LL(0xf79f468d,0xe75cf2cd),L_(0x3f4c786d)}},
   {{LL(0x92cfaed9,0x8aa22d94),LL(0x5767542d,0x951cb3e9),LL(0xcdf3e590,0xc6443f1d),L_(0x926947a4)}, {LL(0xa83030be,0x4ab4bb2e),LL(0x1ac42dc6,0x126bf8d3),LL(0xab5c4613,0x78abc625),L_(0xa11b8fa6)}},
   {{LL(0xb14045ea,0x75f0192a),LL(0xe56f26d4,0x66a1d69e),LL(0x488c7367,0x27671771),L_(0xe2f663dc)}, {LL(0x41dc68c9,0xdbf2b527),LL(0x67ba17c4,0xaa6e15c1),LL(0xc052c755,0x6264eb14),L_(0xef0af1f0)}},
   {{LL(0x361df88e,0x8133758d),LL(0xedb54b06,0x89d87e0a),LL(0x8313147e,0x156564f5),L_(0x0efd3686)}, {LL(0x0305e3bc,0xb530ebe1),LL(0xd1426de8,0x75ed509a),LL(0xe3da205e,0x8810aa6d),L_(0x6d66996f)}},
   {{LL(0x66bf33ae,0x5cdc1d84),LL(0x297ed94e,0x40fff510),LL(0x02569cc0,0xa46f7fa8),L_(0xf6c9bcc5)}, {LL(0x03ae7466,0x6ca8cba2),LL(0x102909c9,0x9d34e4a3),LL(0x338f15f5,0xf175cde9),L_(0x9415a0f5)}},
   {{LL(0x43fb29db,0x0f113fed),LL(0x55f28334,0xc9bc3623),LL(0x0891f557,0xf2de428e),L_(0xc85b958a)}, {LL(0x72c78285,0x0bca6469),LL(0xfebcc59d,0x4ad07f91),LL(0x9aca547e,0x254d2e0c),L_(0x934ecc4a)}},
   {{LL(0xbedf71c8,0xfdf235a0),LL(0xc42e06f8,0x707d37d4),LL(0x519516e4,0xb70d0d1d),L_(0xb746501f)}, {LL(0x026e2e99,0x7b9cd555),LL(0x97c86a72,0x05a8ce5e),LL(0x490bbe2c,0xfd0ac7d5),L_(0x0e861b77)}},
   {{LL(0x4518decc,0x5704b61f),LL(0xe3a1e11b,0x05e3a6d3),LL(0x761243d0,0x09fdcc95),L_(0xa10f1bbc)}, {LL(0x52cc9710,0x2f40eec4),LL(0x45b65061,0x3cfbc5b3),LL(0x82e0b382,0xd24582c3),L_(0x0ea8545f)}},
   {{LL(0x91b73502,0x63a13aee),LL(0x32b2623c,0x64a186c4),LL(0xd30674ab,0xae849f85),L_(0xc2109747)}, {LL(0xbd3a7bd9,0x4e22b52c),LL(0xa5ad1cda,0xdee6130a),LL(0xa16934a2,0xa208c2f8),L_(0x29c61250)}},
   {{LL(0xacddc899,0xdcfd6055),LL(0xaf5f3b40,0x4226123c),LL(0x22c7c6d7,0x18caa3b3),L_(0x11d8af8f)}, {LL(0x27a1cc64,0xcfd9c842),LL(0x74dfaf98,0x435cc5af),LL(0x98cdd25d,0x03338f7f),L_(0xa8a15bc3)}},
   {{LL(0x39e40f6f,0xa3f28110),LL(0x10846ecf,0xbb9b4a37),LL(0x6d2f9be1,0xc7ef39dd),L_(0xe4358a6b)}, {LL(0xb0848ac1,0x9e6a4583),LL(0x0c20b047,0x65ecc897),LL(0x7961633c,0x12939ce7),L_(0xd7729b06)}},
   {{LL(0xdf1e1998,0x0eb1a850),LL(0xbf310ff6,0xcfe64a90),LL(0x6d23295e,0xf848ec96),L_(0x96ccdfe8)}, {LL(0x8a058164,0x4a1159c7),LL(0xfb67d66c,0x145557f8),LL(0x44e87592,0x744a6684),L_(0x93329250)}},
   {{LL(0xd2524bb8,0x71ee45c1),LL(0x3c745d69,0xae2e62f4),LL(0x90655f39,0x4aa6dd85),L_(0xa8f50bcd)}, {LL(0x170e2564,0xcd77d544),LL(0x84a26aa1,0x1637c5b9),LL(0xbbddebd8,0x116cc13c),L_(0x316cea62)}},
   {{LL(0xda76037b,0xe567b31a),LL(0xc3337684,0x81454a31),LL(0x8ce526e9,0xa68887d1),L_(0x5860c2ac)}, {LL(0x5ccae9e8,0x2aa4281c),LL(0x3740c0a4,0x27a06af9),LL(0x67a8cecd,0xb0666bd6),L_(0x8396d5d6)}},
   {{LL(0x3bf215ba,0xf3b762b4),LL(0x2c7b4d47,0xc7ded91d),LL(0xf2dddefc,0x491f86d1),L_(0xe72784c3)}, {LL(0x1a33ec17,0xbfa502ae),LL(0x6ff1a28d,0x8b90c1c4),LL(0x011d4fdb,0xae49e966),L_(0x6e8a2219)}},
   {{LL(0x30030ab8,0x9b2323cb),LL(0xc742f24d,0x3dfca102),LL(0x6d677f76,0x615cef3a),L_(0xcc62a7bf)}, {LL(0xd8927085,0x143864e2),LL(0x68578492,0xe2a53ef4),LL(0x7bfce607,0x98d95aec),L_(0xa4e64436)}},
   {{LL(0xe4480a83,0x674699df),LL(0x638814de,0xf6bd5725),LL(0xa3bae3b5,0x93074bef),L_(0x401bd027)}, {LL(0xc278e6d9,0x9075dc94),LL(0x33a4f7b4,0x66f70e53),LL(0x2912d169,0x77ae21bf),L_(0xd08ac30b)}},
   {{LL(0xba0002d7,0x9d21da23),LL(0x313a021b,0xc9cb3757),LL(0xc0b1cdab,0x478c82a0),L_(0x71e210e7)}, {LL(0x708a627d,0x95df97f0),LL(0x8f2e6f2f,0x036873be),LL(0x9b0c552c,0x31bdcc00),L_(0x7142f468)}},
   {{LL(0xdb1b2797,0x16f6892d),LL(0xd070a81f,0x513e742c),LL(0x83e7ad99,0x5ba54190),L_(0x580b098a)}, {LL(0x3ab45169,0xd55025e1),LL(0x9e175afe,0x57264580),LL(0xa5b527ed,0x246fe503),L_(0x27cfca13)}},
   {{LL(0x9456aa99,0x72287d14),LL(0x56f44c80,0xbd98da02),LL(0x783e5a79,0x1c038840),L_(0xd6a90a4a)}, {LL(0x44dd9215,0x8abbd0fb),LL(0x42622308,0x546492b7),LL(0x9365b6f9,0x638f9886),L_(0xd87dec07)}},
   {{LL(0xe568c53e,0x56dd3ec8),LL(0xecec272c,0xad2418b1),LL(0x40cc9856,0x5fe0c00d),L_(0xd88d5b28)}, {LL(0x1cdd7256,0x17304349),LL(0xa2eb70d1,0xe17ec8e0),LL(0xb4dc2a26,0x62eb4c57),L_(0x3373155f)}},
   {{LL(0xdb4ee5e1,0xa85ff057),LL(0x2cbbaad9,0x0fb19074),LL(0xf1a5850d,0xb1c8c20f),L_(0x4ceec839)}, {LL(0x407175d6,0xc25f5ad9),LL(0xc43bdd2a,0x0ed43d06),LL(0xa2739236,0x55db40b6),L_(0x59afdd51)}},
   {{LL(0xbb91225b,0xb6ec7895),LL(0x64957b33,0xec277689),LL(0x53e7c8ad,0x503d01fd),L_(0x14800e2a)}, {LL(0xff30aa1c,0x789118fb),LL(0x6b208671,0x45dce70d),LL(0x7758b97b,0x2ad05594),L_(0xbd833a82)}},
   {{LL(0x64d25f39,0x7c1271cb),LL(0x973eb35d,0xafba6d12),LL(0x24a4fbdb,0x68588b5a),L_(0x86209415)}, {LL(0x4408d2f4,0xb9962d65),LL(0xa7df2cdd,0x1adff958),LL(0x3e504930,0x1a7248ef),L_(0x642fcd83)}},
   {{LL(0x50371a5e,0x10a09de0),LL(0x585cdc97,0x544a1f87),LL(0x05603976,0x9bb49da7),L_(0x36e0a9f7)}, {LL(0x2686c471,0x22de25c7),LL(0xfc3ec645,0x7b318aad),LL(0x0e37c5eb,0x6e0af008),L_(0xb0fdc095)}},
   {{LL(0xc4103bda,0x28c77b28),LL(0xc7d3392d,0xc44c9d87),LL(0x02942cad,0x045b3511),L_(0x633d355c)}, {LL(0x985fde91,0x3e691e0d),LL(0x0d200049,0x66200341),LL(0xa7fa897a,0xc1b7c5c2),L_(0x5183d651)}},
   {{LL(0x858b20b8,0x94f1ad67),LL(0x73a3848e,0xcb454e18),LL(0xe75e697e,0xea8630aa),L_(0x0d245b82)}, {LL(0x7a69400f,0x7d8e1f8e),LL(0x1a8975af,0x87354e0a),LL(0xd949265a,0xddd78248),L_(0x59506113)}},
   {{LL(0xbb95794d,0x56456f06),LL(0xb1cf847b,0xb5869a96),LL(0x7de9cd89,0x4dd26048),L_(0x29abee0d)}, {LL(0x6b7c47c0,0x3a2f738b),LL(0xdc3e8b24,0x43a158b5),LL(0x6452e6ec,0xb59a08ed),L_(0xdca422db)}},
   {{LL(0xfd294203,0xd446ba91),LL(0xb382c52a,0x0bfb0e2b),LL(0xb347e482,0xda5432f0),L_(0xc465aa05)}, {LL(0xda7568f3,0xfd584f1c),LL(0xb3f81a8e,0x3e5b388d),LL(0xfbef98fa,0xa2090daa),L_(0x7ee59c28)}},
   {{LL(0xebc6de85,0x8601a460),LL(0xa197c14a,0x0fe7b2f4),LL(0x51718d1f,0x768e05f3),L_(0xd61640e4)}, {LL(0x107bee82,0xdafff971),LL(0xd549077c,0x775aa266),LL(0xf5af9168,0xa74ac1aa),L_(0x709e0299)}},
   {{LL(0x7770d24d,0x6c27e8ad),LL(0x5f6805b9,0x0c21256b),LL(0x5eb3480c,0x24df253e),L_(0x33b3350b)}, {LL(0x32c4c951,0x304257df),LL(0xd07bd3eb,0x3821aab8),LL(0xcfd48043,0xe9ed7945),L_(0x29f131a5)}},
   {{LL(0xf927be41,0xa625fc39),LL(0x0acd5378,0xc708ccff),LL(0xc7c05c00,0xe3551df0),L_(0x5a9c25df)}, {LL(0x4a824ae7,0x00aff255),LL(0x3fad48ea,0xdc1865b9),LL(0x489f78db,0xc47678ec),L_(0xe24fe1d5)}},
},
/* digit=26 base_pwr=2^182 */
{
   {{LL(0x4d52ad99,0x44075a5f),LL(0x49de85ac,0x5f264197),LL(0xa5da6ace,0x1bd94b24),L_(0x58ceb7a8)}, {LL(0xaddbfb85,0x9deccfa5),LL(0xeb853c9e,0x38b0cd4c),LL(0x0ae5bd28,0xcdefc743),L_(0xefb92de4)}},
   {{LL(0x9f68d2fd,0xdb80636b),LL(0xeb6a625d,0x399cbe21),LL(0x058ae3aa,0x9a20e964),L_(0xeefd7139)}, {LL(0x0655d447,0x82be9278),LL(0x76a67bb6,0x02a07951),LL(0x78c7207b,0x5473af3e),L_(0x5b501b3b)}},
   {{LL(0x48532008,0xe1daa6d9),LL(0x9dfa0916,0x8ffcdbb0),LL(0xbd0a98e5,0x9b2bdde3),L_(0x5bcf0ad7)}, {LL(0x58bd4135,0x4dcdfb2b),LL(0x913a8546,0x29153442),LL(0x7457124c,0x9190c343),L_(0xc113bf67)}},
   {{LL(0xe34c46c8,0x1a8f4562),LL(0xb45aadf5,0x6cddda1d),LL(0xb1bb8fc9,0xa502ea7e),L_(0xf3ae0efa)}, {LL(0x9421a0ec,0x124b21a9),LL(0xa80fcb58,0x634ea314),LL(0xfc9b2b69,0x4d2d4d06),L_(0x619c1654)}},
   {{LL(0xa4e345b7,0x370847bf),LL(0x9d1ecec7,0x242713b8),LL(0x10782ff2,0xbc286c44),L_(0x2b85247f)}, {LL(0xbe6fa012,0x72197eb9),LL(0xec864a68,0x23d8a845),LL(0x9c366368,0xe57509e0),L_(0xfafe2bf5)}},
   {{LL(0x9344ebd5,0x41626ac2),LL(0xa5a7b2b3,0xb61e001b),LL(0x1c404975,0x05deb3c3),L_(0x4861d13f)}, {LL(0x10e30e7a,0x4f5429b7),LL(0x409e441f,0x0cc2b3bc),LL(0xe64edacb,0x2a32f215),L_(0x502743d8)}},
   {{LL(0xa18ec5e4,0x597408da),LL(0x75198a41,0xb4dc90d0),LL(0x9e4003b1,0xf0377012),L_(0x97c24d6f)}, {LL(0x2f737bfa,0x138b7393),LL(0xc6646756,0xbf91bb7c),LL(0x4390dd24,0xd3eb9506),L_(0x660fe2be)}},
   {{LL(0x1f5f9c09,0xcbb88522),LL(0xde4f8a6e,0x590e948c),LL(0xd63bd64a,0x45a94350),L_(0x82568ecb)}, {LL(0x8b9c0e80,0xc46a6c09),LL(0xe8dd7862,0xe5cb409e),LL(0x6aa2f787,0x90fd8928),L_(0x3c0fb202)}},
   {{LL(0xc519de27,0xc3ac3715),LL(0x59e082f8,0x915329a3),LL(0x0ca44fa0,0xe61346f3),L_(0x590df3a0)}, {LL(0xe4d76c98,0x37f7afee),LL(0x58d6ab1b,0x6bdfe1fe),LL(0x536926c2,0xc048c2a4),L_(0xdd33cbc7)}},
   {{LL(0x57103467,0x9e779a72),LL(0x24aafef8,0xf0cf02a3),LL(0xc4a65bf2,0xc464a2f5),L_(0xcc014eab)}, {LL(0x771064c4,0x545f80d2),LL(0x90630a3b,0x142107aa),LL(0xc2378a52,0x233fc912),L_(0x60b64ae1)}},
   {{LL(0x2c57b6ec,0xe31036a2),LL(0xf8434ff6,0x18945c37),LL(0xa0a4a4f0,0xa70971ce),L_(0x2a1d924f)}, {LL(0x78d2269f,0x3f54f031),LL(0x59b13396,0xd1a3f071),LL(0xcf63943b,0x1bbad8a4),L_(0xae570fbb)}},
   {{LL(0x7535f2cf,0x2202c2ff),LL(0x6ed5c874,0xad072699),LL(0xc576098a,0x233792b7),L_(0xbaf56f3f)}, {LL(0x58643430,0x02886c17),LL(0x64569bb7,0xb1a39466),LL(0x74bab9ee,0xe841def5),L_(0xc973e3bc)}},
   {{LL(0x7f584615,0x0ec350c2),LL(0xca028343,0x2da65635),LL(0xb160863f,0xe9a43081),L_(0x72084a70)}, {LL(0xce3663d1,0xafbf1395),LL(0x1aeb19b6,0xdfa06d4c),LL(0x9aa85d56,0x4f103987),L_(0x4f14057c)}},
   {{LL(0x7df657ba,0xf5ac31c8),LL(0xd4e08231,0x1a02ac9e),LL(0xd28fa99b,0x6407f45c),L_(0x36b31d91)}, {LL(0x06494082,0xe60e26a5),LL(0x03bcc061,0x44cfce93),LL(0x4db05d61,0xf189ccff),L_(0x635d166c)}},
   {{LL(0x16a82c08,0xd179a921),LL(0xec3df3ad,0x51227dd3),LL(0x1d15b312,0x129de8ff),L_(0x99a2081f)}, {LL(0xe117dd1e,0x3f1a519b),LL(0x669c965d,0xd473a86e),LL(0xe470bd93,0x48ca17dc),L_(0xd7172ef7)}},
   {{LL(0xce23cb90,0x025cce52),LL(0x54ac5788,0x7128e28e),LL(0x60a7af16,0xee37af3a),L_(0xd328cf8c)}, {LL(0x228c1194,0x8bcaaced),LL(0x7eb8e5dd,0x60835723),LL(0x5f7c29e2,0x42db013e),L_(0xd6e3c26e)}},
   {{LL(0xd7266cda,0x5ce7356c),LL(0x31f3b95f,0xc0bdd70b),LL(0x086e72f7,0x3319c072),L_(0x7f9cf187)}, {LL(0x4ae3ba6a,0x086343f9),LL(0x85584e27,0x80bc2aba),LL(0xbf3ecc51,0x37cadfe3),L_(0x10a2f1b8)}},
   {{LL(0x66dd80da,0x32cd82a6),LL(0x5a4759b6,0x11e8daf7),LL(0x6268a0b4,0xbd297ce7),L_(0x554b93e1)}, {LL(0x971bdef9,0x8c4eabfb),LL(0x5bdd9c55,0xcbfa339e),LL(0x83d9f14e,0xbd36bd3d),L_(0xe88e0095)}},
   {{LL(0xe3a86075,0xf0d2a9f0),LL(0x7ea9a94b,0x68a87b41),LL(0x27fad851,0x465da389),L_(0xc79beddb)}, {LL(0xa438dfad,0x1024a804),LL(0x9e4004c2,0x3e3b2c82),LL(0x9e904c3e,0xb0cbbf5a),L_(0x91364ced)}},
   {{LL(0x4f3cafc8,0x4571821c),LL(0x05a71822,0xb59cdc87),LL(0x564a1eff,0xd3fc8e11),L_(0xc2522075)}, {LL(0x210954cd,0x485b3e16),LL(0x3e42143b,0xa39a864f),LL(0xc71b961e,0x50587567),L_(0xcd472e0f)}},
   {{LL(0x3f39a45b,0x39ff9b1a),LL(0xb67c7e0c,0xe9320256),LL(0x293c62a7,0x30d60ad1),L_(0x6c70033e)}, {LL(0x51699001,0xecc36d79),LL(0x66761810,0x782811c0),LL(0xb9b6100c,0xd4aec2a8),L_(0x01fd70b0)}},
   {{LL(0xbf6700ba,0xdb67bc3c),LL(0x0705c260,0x81a05a44),LL(0x2a63deb4,0x2c00d649),L_(0xb057ad61)}, {LL(0xc429bae3,0x962a325d),LL(0x94f128db,0xf818f010),LL(0x07c3cde9,0x8b0fa963),L_(0xb0a83496)}},
   {{LL(0xdcf6f7fd,0xd9cd3792),LL(0x63f97d0d,0xa1b3479f),LL(0x9daa99a0,0x6b0d6566),L_(0xe8984ec2)}, {LL(0x448838bc,0xe794fda6),LL(0x82c274f4,0xb6b1fe7b),LL(0xc0cb0a63,0xd09a8df9),L_(0x17a04bc3)}},
   {{LL(0x4c395b4f,0x8e70bc90),LL(0x4ee87155,0x04b3e852),LL(0x9d40e63f,0x4a2dc6ce),L_(0xfaef51af)}, {LL(0x36339e47,0x385a4594),LL(0x20622007,0x455a09ad),LL(0x9304dd88,0xc6ff5bf9),L_(0x913f4764)}},
   {{LL(0xe085cdf8,0x187b4002),LL(0x185e6ba2,0x4c4e720e),LL(0xab690dee,0x86287cf7),L_(0x8afd2d78)}, {LL(0xe2c63b4f,0x9adbcac5),LL(0x7132999e,0xddd28bc6),LL(0x4248e502,0xef074338),L_(0xfa2e3db0)}},
   {{LL(0x896670f4,0x08020d96),LL(0xfcabc640,0xe06c4b5e),LL(0x9786e6bf,0x3da82523),L_(0x3d688833)}, {LL(0x4fe3491f,0x99758e3b),LL(0x5ec531f3,0x0c3b82b5),LL(0xbab9fec3,0x45c4baf6),L_(0xb5f6610c)}},
   {{LL(0xb5a9a1e0,0x4e7e7ab2),LL(0x1b52980f,0xce6cd002),LL(0x7c86bb7d,0xcadc14a3),L_(0x3b8e5f2a)}, {LL(0xbef7e6fb,0xbff62993),LL(0xccd76ec4,0x0a2850cc),LL(0xc5820cdd,0x92bdfe5c),L_(0x082f1255)}},
   {{LL(0x484daaef,0xd799a2f4),LL(0xedb03630,0x550daad0),LL(0xef0ac41a,0x8fbf8f4e),L_(0x737b2d81)}, {LL(0x84f7e137,0x7bdb6e8f),LL(0x34c35ad6,0xd89df99b),LL(0x395821cc,0x20d76719),L_(0xc88df817)}},
   {{LL(0xae387d39,0x760e59a1),LL(0x688c2755,0x97e8f74e),LL(0xbc044138,0xa21350e4),L_(0xc028fc40)}, {LL(0x0253354a,0xe3e6583e),LL(0xb20dddcd,0xc654b121),LL(0xa829cd8c,0x39c79315),L_(0x792dd84b)}},
   {{LL(0x040ffaf0,0xb7bea5b5),LL(0xf1650ec3,0xdb5241ae),LL(0x372d1796,0x940a8a94),L_(0x1a858744)}, {LL(0xf1499afe,0xcabac4a3),LL(0x8ded605f,0xb861df88),LL(0x0cb6a234,0xa96fe9bb),L_(0xbc8b296e)}},
   {{LL(0x5eedd185,0x4434f8b6),LL(0xd1edd68c,0x2a8d7504),LL(0x86f300e0,0x0e35d724),L_(0x897d3aeb)}, {LL(0xfaf29ff1,0x4c814382),LL(0x472c2113,0x141d9196),LL(0xd1bc5f13,0xade4deb9),L_(0xa2fb8cce)}},
   {{LL(0x406ace5c,0x1ea243b0),LL(0x686c6adf,0xd6070206),LL(0x22934b39,0xd9cd49e0),L_(0xa0234a2a)}, {LL(0x4d0f3175,0xc755b33c),LL(0x5c35ad9b,0xf8e1a985),LL(0xc50467c8,0x18a3ef6a),L_(0xf2b99d2a)}},
   {{LL(0x4607dced,0x614bcea6),LL(0xd3d623a4,0x68ae9193),LL(0xf3906a4b,0x42d006dc),L_(0x8b8583eb)}, {LL(0x4cb64b0e,0x9dbec629),LL(0x98a17fc9,0x0bf6a2f2),LL(0x057751b4,0xa9fae60b),L_(0x21e7bd24)}},
   {{LL(0x3c12fc75,0x017d14d0),LL(0x3242c2d8,0xc6c97422),LL(0xa51b04ab,0x626c81c6),L_(0x94d87c6b)}, {LL(0x238bcf04,0x5e266a2a),LL(0xa82fa338,0x23764002),LL(0x51803104,0x5f7162bf),L_(0xc4756473)}},
   {{LL(0xa5dd759f,0x522db1a2),LL(0x5930b4b5,0x09f2baed),LL(0xf85a6ac1,0x323304d4),L_(0x05195116)}, {LL(0x1360791c,0x1db93b31),LL(0x108292f6,0x1b768230),LL(0xabf3fed8,0x87d85b61),L_(0xa1679cd2)}},
   {{LL(0x761f1b7d,0x0b49a02f),LL(0xb6746bd7,0xbb854597),LL(0x77637e3e,0x0022290f),L_(0x86597269)}, {LL(0xf0700421,0xe0410aa3),LL(0x2a1bca0c,0x6d34be85),LL(0xf6bc18c0,0x46cdba65),L_(0x203e8ea6)}},
   {{LL(0xb9773b5a,0xc5228020),LL(0xba11b539,0x453c7ac2),LL(0xf11276b9,0x7df2cd47),L_(0x1949a67a)}, {LL(0xf5a8c36b,0x179c274d),LL(0x860ff0ae,0xa5c5bb12),LL(0x11df901f,0xe17d00c3),L_(0x3411dcae)}},
   {{LL(0x3ed4c79b,0x8883a1db),LL(0x1992a967,0xfdd5776e),LL(0x2f54ff27,0xcb776738),L_(0x47290630)}, {LL(0x9a98b107,0x79c8cc7c),LL(0xee2e8af1,0xdc2c5969),LL(0xc7ee141c,0x6b47161b),L_(0xa7624962)}},
   {{LL(0x543685fa,0xc2395393),LL(0xe7ef754f,0x9a0b36af),LL(0xaae1a4bd,0xd5d4e38e),L_(0x4aab543b)}, {LL(0xc2e17c97,0xd7e5c5cf),LL(0x5af349b1,0x2bef05ff),LL(0x9ed78a64,0x561fb810),L_(0xf9c813d0)}},
   {{LL(0x2fe8fb72,0x1a1e9b3b),LL(0x9b70eefa,0x3925b0e9),LL(0xbfb2449b,0xb054dfbe),L_(0xb47dede7)}, {LL(0x28647224,0xa5055940),LL(0x0c72430a,0x31e1b9a5),LL(0x23b5a4c4,0x195035c8),L_(0xd679bea8)}},
   {{LL(0xcc50c48d,0x209f3af8),LL(0xd27b6e83,0x181662d3),LL(0x4b465365,0x91296676),L_(0x420e37bf)}, {LL(0x138dff74,0xed5f5189),LL(0x82f8d0f4,0xc0ded518),LL(0x40df6431,0x723fb4ad),L_(0x617ba2b7)}},
   {{LL(0x112db662,0x3569136f),LL(0xb2b47e3c,0xb8e53774),LL(0xdd45fe78,0xfaa04e8d),L_(0xc768fdda)}, {LL(0xf6371e54,0x86806849),LL(0xa8e9db9c,0xf4779601),LL(0x4c858ecd,0xa4992880),L_(0x49a83816)}},
   {{LL(0x8967314a,0x156e1198),LL(0x7bad4014,0x39eb8236),LL(0xecde7783,0x6d9c66e5),L_(0x881e0782)}, {LL(0xd2f38b48,0x3fba7225),LL(0xed8353b8,0x088944d1),LL(0x4f8035c2,0x33f1f404),L_(0x1cc9c213)}},
   {{LL(0x51cb66d6,0x58d0680a),LL(0x50bb5809,0xcb3e58aa),LL(0x1d2325d3,0x13427358),L_(0x5e67c4ac)}, {LL(0x7dbb0bf2,0x85f983bc),LL(0xa937cb56,0x0f88839a),LL(0x8a991a31,0x6b3985b4),L_(0x92b4731e)}},
   {{LL(0xdf68b708,0xde5b911a),LL(0x03d7de13,0xeed92206),LL(0xfc74f9a9,0xbcecf38d),L_(0x088b6fe1)}, {LL(0x96e739c0,0x0a6c0e27),LL(0xd4a63fbd,0x82400edd),LL(0xbba8f32c,0x27d5bb2b),L_(0xc58b1cab)}},
   {{LL(0x3131df11,0x042f1a03),LL(0x4f121cf6,0xe003b634),LL(0xa22e7895,0x6dd1e8b0),L_(0x5d804ab1)}, {LL(0x90518b2c,0x5efe4007),LL(0xe1e8157d,0x88b87e5c),LL(0x63c085ee,0xf6155e0f),L_(0xba6e44df)}},
   {{LL(0x841e4ade,0x1f02ab50),LL(0x156d76af,0x67a50be5),LL(0xe6f206a7,0xf1e42056),L_(0x65f10a7c)}, {LL(0x2febb4c7,0x657a0dc7),LL(0x072e3321,0xbbf3484b),LL(0xea4f285b,0x020b154e),L_(0x8c852c97)}},
   {{LL(0xdbbe1818,0x91e4e9f4),LL(0x2edec0de,0xbf43b7fd),LL(0xe4667773,0x52865635),L_(0x56a43f3c)}, {LL(0x5962ba87,0x28fe8ab8),LL(0x248c61c8,0xd449338b),LL(0xd636805c,0xa1efd1c9),L_(0x13bd6609)}},
   {{LL(0xdfd448ab,0xb92e258d),LL(0xe4e24c18,0x35b6d45b),LL(0xa42ce187,0x0d195771),L_(0x5aad5bff)}, {LL(0x672eaebc,0x60feb26e),LL(0x4177b162,0x9498986f),LL(0xb3b0ede4,0xc41d2caa),L_(0xf358519a)}},
   {{LL(0x5261f5a7,0x330e8a48),LL(0x43204369,0xf61c72d3),LL(0x33ca48fd,0xffeebdb0),L_(0x8ae6e55b)}, {LL(0x7ae9478f,0x73383a11),LL(0xa6e09378,0xafb5d295),LL(0x689cfdb3,0x32706d23),L_(0x868c5ddf)}},
   {{LL(0x0bd73e3c,0x62c3a48b),LL(0x8741583b,0x0c0a3d1d),LL(0xacf48d35,0x48c97582),L_(0xa4f06efa)}, {LL(0x01b5bb29,0xaf780b08),LL(0x6bf0d377,0x2a0a2d75),LL(0xabcaf322,0x13f7f83f),L_(0x655d34bf)}},
   {{LL(0xb992a7a0,0x19a0cf96),LL(0x9ecda6fc,0x98adc9dd),LL(0x6a4d3e6c,0x40f8faf2),L_(0x0edc38f8)}, {LL(0xef011d38,0x7e67e105),LL(0x1797a86e,0x87eb4d53),LL(0x45add6b3,0xce77f195),L_(0x989201fd)}},
   {{LL(0x3d9993bc,0x8adaa277),LL(0x99dda71d,0x1fca3093),LL(0x7433657e,0x8caf0d86),L_(0x023c11e2)}, {LL(0x85112381,0x3dbd4315),LL(0x708185e8,0x9d410bfb),LL(0x374f5e1e,0xf2fcce49),L_(0x3443b974)}},
   {{LL(0x532b7588,0xd3243b22),LL(0xfed73137,0x11c1365d),LL(0xb045d4a7,0xb70776bc),L_(0xfa754876)}, {LL(0xadb798db,0xe29a9f85),LL(0x7a404c41,0x04e83976),LL(0x4ef414c2,0xd94bbca4),L_(0xca54266d)}},
   {{LL(0x69a5c564,0x189faa10),LL(0xea80ccb0,0x11ff4091),LL(0x270bddfa,0x13f400ef),L_(0x16bba259)}, {LL(0x54c1f416,0x5467f97f),LL(0x63fe2fd7,0x8a19f9f7),LL(0xd713b7c7,0xbecb24c4),L_(0x548c6ef8)}},
   {{LL(0xd9e41b7d,0x154bb407),LL(0x5fe1b516,0x9e3d7f55),LL(0x0a7b54c7,0x135359b7),L_(0x56bae4a1)}, {LL(0x84d9fef8,0x5d18cdb1),LL(0x3494e228,0x142d7261),LL(0x1cba3904,0xac50bd6c),L_(0x2d2c512d)}},
   {{LL(0x709ebb7e,0xec49c916),LL(0x906c201e,0x16f1cdf7),LL(0x9b4ca4e1,0xd65f4a10),L_(0x474e1186)}, {LL(0xf98d2d15,0x03b182b1),LL(0x94cb6ea7,0x060b7206),LL(0xb11e15e8,0x6d96c56b),L_(0x52f7fca7)}},
   {{LL(0x8f77e5e5,0x72cd83b6),LL(0x8cab24f9,0x25a5b5bf),LL(0x25a2e699,0x6fdd7673),L_(0x072fab5d)}, {LL(0x90a6d4ff,0xd576b9f6),LL(0x3ca8ca37,0xbcb25091),LL(0xad2b4418,0xefb79bc4),L_(0xff9c27d0)}},
   {{LL(0xfba463c7,0x67359a81),LL(0xfe5bb23c,0x2c69efd5),LL(0x280e7df6,0x74569519),L_(0xfe16bd9c)}, {LL(0xe2c76094,0xb18f6667),LL(0x365a80b1,0xbd326eec),LL(0x88831553,0x87869f49),L_(0x5939e645)}},
   {{LL(0xbaa57c7d,0x3b207add),LL(0xfb5db15b,0x4ed05149),LL(0xb373586a,0x309bffcc),L_(0x773b90d1)}, {LL(0x461b005c,0xc0f53c53),LL(0x8106712a,0xc558624a),LL(0xcda6e3b1,0x6faa11b6),L_(0x3ae794e5)}},
   {{LL(0xf0680ab3,0xcbfa17ef),LL(0x058c2020,0xcddb5b82),LL(0x41c78a0d,0x20234c33),L_(0xe0cbaf8b)}, {LL(0xc2fe9029,0xf4f1f1bc),LL(0x7a0fe46d,0xfc36a9bb),LL(0x75517f43,0x7adc3f7c),L_(0x9a365f26)}},
   {{LL(0xba51387b,0x49c36869),LL(0xc5a22f24,0x03e373c6),LL(0xb8b4767a,0xb07a3f0c),L_(0x36066a8f)}, {LL(0xe85c503a,0x808f702e),LL(0x1497d7ac,0xdddd86d9),LL(0x7011d76f,0xfae1c7e7),L_(0x803e46f1)}},
   {{LL(0x80fa818c,0x31e4ef2b),LL(0xeb39da59,0x68d478dc),LL(0xb3f48a44,0x7630fe61),L_(0x067101b0)}, {LL(0xf7c94ecf,0x9a533238),LL(0xaea4959f,0x5e7ce695),LL(0x3b49d1b1,0x52a95933),L_(0x557cf5b6)}},
   {{LL(0x0d20898f,0x9d1c985c),LL(0x072ec166,0xc62ed966),LL(0xce3b7676,0xeb820161),L_(0x37e4d792)}, {LL(0x60f14619,0xe6d270e0),LL(0x648b3694,0x80e6594a),LL(0x9e645987,0x879e807c),L_(0xb22e7cb2)}},
},
/* digit=27 base_pwr=2^189 */
{
   {{LL(0x0703230c,0x9c23cc7f),LL(0xad5df365,0xa6d601b3),LL(0x3f2e4ff0,0x8b1b5936),L_(0x1dea5297)}, {LL(0xe0882f47,0x402ec9ab),LL(0xa3a23af8,0x47de2c7e),LL(0x555b4d4e,0xeaa5bf4d),L_(0x2ff94aac)}},
   {{LL(0xf78b555b,0x971ade84),LL(0x54bd5a85,0x4e8a2ef9),LL(0x5ad4ea73,0x0425d602),L_(0x3bd54fb7)}, {LL(0x791d9123,0x227d0459),LL(0x1e92cc4f,0xc9d765a1),LL(0xe4f1346c,0x28b65397),L_(0xd481a746)}},
   {{LL(0xe7d16971,0x871eef82),LL(0x0506636f,0xb5fdd432),LL(0xb27a33c5,0x1d066c2a),L_(0x3113bf2a)}, {LL(0xe97230fe,0x862b60db),LL(0x75a027c2,0x593d4117),LL(0x539f6530,0x8436da66),L_(0xe2b2cbf2)}},
   {{LL(0x2425af9f,0xead58c17),LL(0xcad19bb2,0xb5cbb50e),LL(0xccdd8dd9,0x86b3650e),L_(0x1a092795)}, {LL(0xdc320635,0xa3e665cd),LL(0x7ebcad7d,0x1e2d319e),LL(0x33fae415,0xeca579c1),L_(0x6ac7d5f9)}},
   {{LL(0x6a9d1fc5,0x89b65031),LL(0x881ef8ac,0x23a987e0),LL(0xfde88e7a,0x2dd37a50),L_(0xc8dcf85d)}, {LL(0xcebc549c,0x820f3f12),LL(0xd4ae9ff8,0xb7bcaa86),LL(0x694bc71e,0x8e6e09c1),L_(0xd6b8424d)}},
   {{LL(0xfe8a98f0,0x63ff767a),LL(0x378e38e8,0xc5f7f4d1),LL(0x0e5335e1,0x7cfba5a2),L_(0x39a8daab)}, {LL(0x77da1977,0x67e59eaa),LL(0xad21218b,0x504acfd4),LL(0x96be881c,0xbeef988a),L_(0x69332afc)}},
   {{LL(0x82325977,0x575e852e),LL(0x45759819,0x2c72e2f1),LL(0xe0cc66d6,0x1927e387),L_(0xc0500878)}, {LL(0x4d0b0122,0x7da1cf59),LL(0x14427010,0x26204672),LL(0xdb25f5a7,0x0a8b1258),L_(0x478d5953)}},
   {{LL(0x7d69f704,0x20d3988b),LL(0xab2517df,0x6c24b0c1),LL(0xaab259f9,0x5e65b843),L_(0x174ca2d2)}, {LL(0x75e5c2e2,0xea29f462),LL(0xdb90528b,0x1834a190),LL(0x8c9eae94,0xf967c0ea),L_(0xae030eb0)}},
   {{LL(0x7e9bdb89,0x00adf08d),LL(0x03f389c4,0xa0138acf),LL(0x8dfb7875,0xd93b37ee),L_(0xbed6d6d8)}, {LL(0x749de946,0xbfd574b2),LL(0x1fc14060,0xe75abd5a),LL(0x67c65b64,0x13366a27),L_(0xc999214e)}},
   {{LL(0xb08be401,0xe06fcef2),LL(0xa7dfbf73,0xe0ba359e),LL(0x607c14ca,0x382ff71f),L_(0xdef77eb0)}, {LL(0x6404f03d,0x3f8a49f7),LL(0xf9e8050d,0x7853702d),LL(0x4887d191,0x04d3f5e6),L_(0x1aa6bfdc)}},
   {{LL(0x67857252,0xdb805369),LL(0x612fee31,0xd63c66e4),LL(0xa73ff057,0x587ee86f),L_(0x2d5eb654)}, {LL(0xa10a675a,0xbd927cb1),LL(0xf79fccb5,0xc799e97c),LL(0x6f65ee19,0x4703e5a8),L_(0x04d4c022)}},
   {{LL(0xaa2f24f8,0xba32a728),LL(0x2229afc4,0xb7ad366d),LL(0xa0ac6a50,0xb220a09e),L_(0x5da587fb)}, {LL(0xd5db1088,0x44a3150c),LL(0x31856642,0x09784fd1),LL(0xf6f05498,0xc1e6c0b0),L_(0x346aa16b)}},
   {{LL(0x06b7160f,0xbb692e48),LL(0x89eb623c,0x15498ecf),LL(0xb77b124d,0xbb092f5d),L_(0x3e60aee2)}, {LL(0x74e3ca98,0x7b0969cb),LL(0x9ba2e8fe,0xf86a2cf8),LL(0x6d59ed66,0xdeabc883),L_(0x39e0a846)}},
   {{LL(0x17174a2f,0x852eae73),LL(0xbaf85e02,0x1e348f19),LL(0x0374bff3,0xeacc5568),L_(0x60bfb120)}, {LL(0x5585a76d,0xa921615e),LL(0xf43bee83,0x1b74c1fc),LL(0x26867d50,0x52fb37fb),L_(0x7279cdee)}},
   {{LL(0xb96496b8,0xe996dbee),LL(0x7868fd21,0x39670ef3),LL(0x8094e471,0x9c159e05),L_(0x2d0b6ffe)}, {LL(0x4226465b,0xb7fd505e),LL(0xe7884414,0xf2355f66),LL(0xe46e77c4,0xe0fbfc8b),L_(0xd7928658)}},
   {{LL(0xc386d033,0x16b11ec1),LL(0x6cbaf218,0x75da11d8),LL(0x94c7a19f,0xfda19041),L_(0x15d41b37)}, {LL(0xcd2bfd31,0x0e5ba2ee),LL(0x8a34c58f,0x2339d87a),LL(0x375d7766,0x24af1974),L_(0x280bf34c)}},
   {{LL(0x0c7c5c0f,0x06c1fc0c),LL(0x8c4fcedc,0x1d94e717),LL(0xb65c6a50,0x0b30821a),L_(0x2a34b8c1)}, {LL(0xefa6a39e,0x6459b552),LL(0x49fdfbf5,0xb7d7e4b6),LL(0x18598afa,0x00b319ed),L_(0x287de14b)}},
   {{LL(0xf884841d,0x8dad7679),LL(0x518b1ea0,0xb306db5d),LL(0xb36c0754,0x5229df9a),L_(0x8fc9f729)}, {LL(0xaadea0d8,0x970bb22c),LL(0x03b629f0,0x298689dc),LL(0xf3b57c01,0x89fe17ad),L_(0x686552a0)}},
   {{LL(0x76218fd5,0xceff7d9d),LL(0xd9e895e9,0x63a383ce),LL(0x238b4e91,0x99f29b16),L_(0xbd71b01c)}, {LL(0xe11ea486,0x7ae47927),LL(0x605101be,0x7cd7a8d0),LL(0x0d9f157a,0xb3fa2b87),L_(0x330f1244)}},
   {{LL(0xff049f89,0x85fe8244),LL(0x6ddb5ffc,0x5d2ff231),LL(0xef0f5b1a,0x63ccdc00),L_(0x61b2bcef)}, {LL(0x8ba95ec1,0xa8a44260),LL(0x5321922c,0x6ed75cfa),LL(0x3e43af21,0x8464e573),L_(0x6a65d52d)}},
   {{LL(0x65d29b4d,0x1583a353),LL(0x32ab9cfa,0x43193114),LL(0x92998e59,0x704abb0e),L_(0xbbb3d90f)}, {LL(0x2a7e16f7,0x4c98adbf),LL(0x7bbafb3d,0xa5a25fbe),LL(0x44dc73bb,0x8fc488f7),L_(0x6641759d)}},
   {{LL(0x784f9378,0xe7bcd034),LL(0xabbc4935,0x89f2f24e),LL(0xbec66d27,0x4917e013),L_(0x5c4012be)}, {LL(0xf0cde4d8,0x0961875a),LL(0xa3f96a56,0xa260b4de),LL(0x59faeb8c,0xbeeb1631),L_(0xc5c3e45d)}},
   {{LL(0x280f3963,0xa088a1e1),LL(0xf1068e65,0x1dbee472),LL(0x1692a131,0x05a158ea),L_(0xde0f97d9)}, {LL(0x0a11b122,0x0e13e263),LL(0xb70fb035,0xdee544e8),LL(0x5dcaae12,0x8f1fbb34),L_(0xd7c3d865)}},
   {{LL(0x5d2e90ae,0xb9ad99ac),LL(0x75457057,0x0c329529),LL(0xaa459865,0x96ffc387),L_(0x7823d3a1)}, {LL(0x13516c2f,0xa0aa883b),LL(0x6d68a755,0xbfa72fcb),LL(0xa825f1f8,0x97da099b),L_(0xcba9849f)}},
   {{LL(0x14bced83,0xf4fdf535),LL(0x671c909f,0x689f6f0d),LL(0x71814baa,0x5b8ff385),L_(0x984a3eb2)}, {LL(0x9c1ba7cf,0x907e4e69),LL(0x6fd404b5,0xff0ce18b),LL(0x2458cb25,0xef5cd0aa),L_(0xdb837d9e)}},
   {{LL(0xfee8b8f0,0xa4710d21),LL(0xca118a9d,0x54895b97),LL(0xb541a6b0,0x526b0ca4),L_(0x46f624ed)}, {LL(0x16947323,0x31b294e5),LL(0xd0b8b7cb,0x1b4cb2b4),LL(0x98687802,0xfbcc3509),L_(0x58f68319)}},
   {{LL(0xb150cf08,0xe16b93aa),LL(0xba41d3c9,0x253eb0dc),LL(0x984df68e,0x87118c04),L_(0xea79efa1)}, {LL(0x480076ed,0x0dcde0c7),LL(0x36503c9e,0x18bdffd8),LL(0x16aa0174,0x1f265d34),L_(0x34f86080)}},
   {{LL(0xee0e75be,0x5617d582),LL(0xab95d71a,0x3e671da8),LL(0x0190d33a,0x3bee1eff),L_(0xa69eb59f)}, {LL(0xe617dbcc,0x09cde5ef),LL(0x2ec74095,0x888eade7),LL(0xfc85f3d6,0x359daa39),L_(0x55fe2fc2)}},
   {{LL(0xb342c648,0x4a00b224),LL(0x153ecb1a,0xb894175c),LL(0x58a94e88,0x70cf2206),L_(0x2b740c25)}, {LL(0xf20b5561,0xdca48b9e),LL(0xcecf5379,0x7707c57d),LL(0xa0a1a133,0xc33e6b03),L_(0x4b326561)}},
   {{LL(0x9c8970a2,0x4c16f18a),LL(0x642f1b1c,0x57b552ca),LL(0xda495e0b,0x1a9ee7da),L_(0x8d81f72e)}, {LL(0x1ad4dac5,0x06f7634a),LL(0xd29378b5,0x1b77c61d),LL(0x110655eb,0xd5bf028c),L_(0x1f8658cf)}},
   {{LL(0x21c59245,0x51b05ff6),LL(0x6f117207,0xaf13b9eb),LL(0x2b5c0017,0x4aa9bdff),L_(0x0df00c88)}, {LL(0x703d8719,0xaf5e4830),LL(0x40500109,0x52cc9369),LL(0xf59d1925,0x013abb12),L_(0xdc8ebfb1)}},
   {{LL(0x1082f991,0x2c9f0b9e),LL(0x02ac62a2,0x666e06e5),LL(0x703ab683,0x6f1b6308),L_(0xf1dd2936)}, {LL(0xc8b722dc,0xd103845d),LL(0x6960a1a5,0x2459c28c),LL(0xd0ce016b,0x60884f40),L_(0xd595cffb)}},
   {{LL(0x7f7ba180,0xe194bda2),LL(0xabd5ba2f,0xbb90a872),LL(0x2977890c,0x3dc3fa11),L_(0xa6cdf4eb)}, {LL(0x1ec1818c,0x2f976a14),LL(0x9509667f,0x6ef670b0),LL(0x547bbc14,0x7a6392b3),L_(0xec34c6c8)}},
   {{LL(0xfa44c4a3,0xd1652cf8),LL(0xd0c64f4a,0x7b98c98c),LL(0xeeaab800,0xb60ccb53),L_(0x41176e83)}, {LL(0x20bc577a,0x3d3d111a),LL(0xc6b3e65e,0xf1c69a56),LL(0x4586a0f5,0x047748d6),L_(0x966336bb)}},
   {{LL(0xbfb2046e,0x1ba40992),LL(0x15e78b76,0x492e1976),LL(0x8fa3d075,0xb62a0315),L_(0x34d22c5e)}, {LL(0x08bcfaa9,0x55ee388d),LL(0x47ca2bb1,0xb50f4e76),LL(0x48d16444,0x947b5775),L_(0x6055efef)}},
   {{LL(0x27f9113a,0x9933e071),LL(0xeeda2fbb,0x16712a59),LL(0xc739c006,0x75d0d886),L_(0x844de074)}, {LL(0x95eed6e3,0x5fd2d047),LL(0x53ef6941,0xfa45dd7f),LL(0x9d54f1fb,0x741b92d0),L_(0xf26829fe)}},
   {{LL(0xe5114e39,0xc2275c0e),LL(0x14f42683,0xb4d5582b),LL(0x3e016601,0x25741fdf),L_(0xb4b6622d)}, {LL(0x4514d2ec,0xc4ccb66a),LL(0xf7783f3c,0xb57016ca),LL(0x59b84d70,0xf83fe7e5),L_(0x1f712454)}},
   {{LL(0x7a6974e4,0xcecae61e),LL(0xddefbfbc,0xa93265ce),LL(0x1f1fb9f1,0xf3c9e95d),L_(0x968d6e21)}, {LL(0x76333191,0x5ef96caf),LL(0x343738c5,0x26204de9),LL(0x66ec9969,0x8ee499c3),L_(0xfa475270)}},
   {{LL(0x1abe8df4,0x0e4c49fd),LL(0x72a09b7d,0x5430fd38),LL(0x5d968105,0xb2efc301),L_(0x82c1807e)}, {LL(0xde3d9217,0x191e5b5e),LL(0xb52c47de,0x32722377),LL(0xa0892090,0xd92f1352),L_(0x8058cafd)}},
   {{LL(0x757c94f0,0x9649e576),LL(0xf15a6d6f,0x79587916),LL(0x8be7ad8d,0xadd01304),L_(0x01368bc3)}, {LL(0x29445ca1,0x285b7eef),LL(0x761da068,0xae55ea08),LL(0x962f7357,0x983ad1c5),L_(0x9e092171)}},
   {{LL(0x40d7b216,0x3897f22b),LL(0x2a0a0ea9,0x0a2968a8),LL(0xae9ce773,0x5eeabf53),L_(0x205e8acc)}, {LL(0xd7707d11,0x3f8692b7),LL(0xc470978d,0x416e3f6b),LL(0x0e77bce9,0xf53b376e),L_(0x4d5b8d32)}},
   {{LL(0x211044a3,0xb9dbcba3),LL(0x174bbf68,0x7e52000c),LL(0x2961ea4a,0x12e1a413),L_(0x1ad46429)}, {LL(0xc87793af,0x36916c16),LL(0x5c151550,0x93f3db96),LL(0xf4f9dbfb,0x2bfda0b4),L_(0x51b088a4)}},
   {{LL(0xbbbbeb9d,0xcc1030ad),LL(0x12a28f87,0x49bedb1c),LL(0x1869975f,0x6a004fbc),L_(0xfb3e5023)}, {LL(0x51c16438,0x71e53119),LL(0x49270eb0,0x5e139c47),LL(0x60f866a3,0x0784d6e0),L_(0xf4f1e774)}},
   {{LL(0x97f340e8,0xa61d043f),LL(0x05c008f3,0x4c35a67a),LL(0x8581b05b,0x44182222),L_(0x3e8c133f)}, {LL(0x45bca0e5,0x8bf96ead),LL(0xf85bee99,0x7c86a090),LL(0x2ad3d962,0x7ea985d4),L_(0x8ce03be2)}},
   {{LL(0x1ef28f3f,0xcbf512f8),LL(0xcf504c88,0xdc8ed43e),LL(0x516f1183,0xec32dc2b),L_(0xa2e2c1fe)}, {LL(0xc45e5adb,0x505156da),LL(0xb7216064,0x983acb8c),LL(0x30958d17,0xe6ec6def),L_(0x28469ede)}},
   {{LL(0x1072cfe0,0xaaf2a64f),LL(0x30962a70,0x33454609),LL(0x8b880e7d,0xa7a46a05),L_(0xa96bc722)}, {LL(0x7354fe08,0xd5821805),LL(0xdd17ff77,0x0fc1cbb0),LL(0x0e58985d,0x9f7a9dad),L_(0xca339489)}},
   {{LL(0x9dc30815,0xe1ebef86),LL(0x51cfd1d3,0x852da87b),LL(0xed5cdd01,0x21bc828d),L_(0x7b53ef92)}, {LL(0x5aadc85a,0xe522baaa),LL(0xb7031c6f,0x608b4b16),LL(0xee466b6e,0x0c33a24a),L_(0x3197bd29)}},
   {{LL(0x4e1fae13,0xf397b191),LL(0x68fba8e4,0x2940f1c8),LL(0x47dd818f,0x68c77c89),L_(0x7dbc2c98)}, {LL(0xba40969f,0x8ffaa799),LL(0xf0b99031,0x712df944),LL(0x997feef1,0x4da4261d),L_(0x1f70d141)}},
   {{LL(0x1b0afa52,0x69fea568),LL(0x3e151e92,0xdfc334a6),LL(0xa2587e32,0xf61b674f),L_(0xbaa4f3f4)}, {LL(0x62eb8d24,0xdd643088),LL(0xa7a5fc88,0x5cde0035),LL(0x53b8bdc0,0x1c5ce488),L_(0xf9fdf735)}},
   {{LL(0xa64a20a6,0xd22b7f88),LL(0xa2eb0438,0x0628283c),LL(0x75129c40,0x6efeb06e),L_(0x4f3e44a6)}, {LL(0x76f8c149,0x181e8b58),LL(0xbb830a80,0xc758648a),LL(0x3d1fbafb,0x069f31f1),L_(0xbfff2e03)}},
   {{LL(0x78f3872d,0xbba94fb7),LL(0x748d4921,0x852ef996),LL(0x48ddef24,0x066ad60c),L_(0xd4ea4021)}, {LL(0xc1ba7360,0x790d395b),LL(0x6901be6e,0x7616ea80),LL(0x61b408b8,0x9278e0eb),L_(0xc7366cae)}},
   {{LL(0x15c9a850,0xae384504),LL(0xd50ba3bf,0xf4814e2a),LL(0x1b8b0e26,0x2b97a1b4),L_(0x3f0e4245)}, {LL(0x260ef3d1,0x5db22d7e),LL(0xbfd1a5ec,0x2e7c6bf8),LL(0x7c1471d1,0x7bdec3b1),L_(0x079a90a5)}},
   {{LL(0xb71fd22f,0x7f17f289),LL(0x5e288b65,0xd4564a47),LL(0x34b284d4,0xb75a3120),L_(0x5d8426b1)}, {LL(0x986c8a29,0x197cc0c1),LL(0x29e64b23,0xcc3ae592),LL(0x7da4599c,0x54ebd08e),L_(0xf69dda7d)}},
   {{LL(0x9addab69,0xbd0c6b5c),LL(0xf7f44578,0x7ec99645),LL(0xaa918115,0x8ee2e6eb),L_(0xd2de3778)}, {LL(0xd9db5cd4,0x2d90dfec),LL(0x5c1253c8,0xf48ec5bb),LL(0x7732d381,0x2daa3783),L_(0x158b18c3)}},
   {{LL(0x4087fb32,0x64d12677),LL(0xa61b3dc0,0xd6d5d34f),LL(0x5aa57c66,0xeebb82c5),L_(0x01983506)}, {LL(0x0b3a8490,0x4bcc334e),LL(0x347044a3,0xafc73b45),LL(0xbdc15e76,0xba3353c1),L_(0xfc066074)}},
   {{LL(0x3f49b08a,0xe4f392d0),LL(0x1fda13b1,0xc0bd57d7),LL(0x4a84d25e,0x8b9c3a47),L_(0x9991791e)}, {LL(0x88262b52,0x5d6233aa),LL(0xd1327af9,0x1da165fe),LL(0xba39a0b7,0x56569118),L_(0x22bb8aef)}},
   {{LL(0x55aff6c4,0xffb26779),LL(0x4e7d53f7,0x581f3f37),LL(0xc259ed17,0x0527ce42),L_(0xb4d8bc18)}, {LL(0xeb967186,0x592962f6),LL(0xf82705f0,0xbc866b81),LL(0x9a5ee9a2,0xb8fcf82b),L_(0x60d27e12)}},
   {{LL(0x7b7d67a2,0x6bed8eca),LL(0x5d9fc1ee,0x74e18695),LL(0x3ac5ac85,0x4777aaae),L_(0xaf2958be)}, {LL(0xaf83cba9,0x114e541f),LL(0x41b4aa09,0x49c90e0c),LL(0x6ffe97ac,0xd22caea2),L_(0x032a68c3)}},
   {{LL(0xad0a7c5d,0xcf68be2d),LL(0xe675d930,0x46042df0),LL(0x44cc34c3,0x83baaad0),L_(0x09136a40)}, {LL(0xe5be5f7c,0x232306d2),LL(0xe3f78e95,0x30e3711a),LL(0x7b102ef8,0x90ca6b87),L_(0x25d20cac)}},
   {{LL(0x24f85c29,0x1c82027d),LL(0x509f2a65,0xc82c4968),LL(0xf294dde1,0xa0dabc8e),L_(0x6fbe2f96)}, {LL(0xeadad60a,0xa2d74908),LL(0x0b27809e,0xea64f015),LL(0x3b0536e3,0x948d4413),L_(0x3ccb3666)}},
   {{LL(0x4cd8e164,0x7e58fb10),LL(0x2cc3a907,0x3a2788f9),LL(0xc860f6b9,0x673a28d4),L_(0x6f03054c)}, {LL(0xc5e46854,0x62c5a4e9),LL(0x02ecf383,0x059005a0),LL(0x5a617a3d,0x06267ff2),L_(0x6a1a7e0c)}},
   {{LL(0x2af744e3,0xae8162d5),LL(0xfa2efdc7,0xbe28dd79),LL(0x950dd089,0xcd76243e),L_(0x1c19727b)}, {LL(0x887a306f,0xc5cf5f6e),LL(0x003d3ccd,0x6076457e),LL(0xd58cb75e,0xa5a9063c),L_(0x95f7ccee)}},
   {{LL(0x6f63cd76,0xa7571852),LL(0x089066c8,0x2ab669d0),LL(0xd42dd087,0xee6f2f6d),L_(0x18dd0fe9)}, {LL(0x0a7d1e21,0xfa5d28f5),LL(0x2b521ac0,0x9441fb8f),LL(0x8466e082,0x4a673507),L_(0xf688bc35)}},
   {{LL(0x980ae611,0xc6222c7e),LL(0xb1d88568,0x3030f358),LL(0x9b8f42e0,0xbbf50262),L_(0xaa3f5497)}, {LL(0x469ee64f,0x3ef5e0ea),LL(0x167a26ce,0x0e1c1519),LL(0xb57bce91,0x806bb655),L_(0x2f545810)}},
},
/* digit=28 base_pwr=2^196 */
{
   {{LL(0x4760716f,0x789ae13c),LL(0x1cb72857,0xf578d1e5),LL(0xc6470ce9,0xd42e2969),L_(0xe716d39d)}, {LL(0x5f1cf231,0x733456c6),LL(0x5fef0d2e,0x592bb199),LL(0x91e21ef9,0x8814ad36),L_(0xb248995f)}},
   {{LL(0xbefb4b7c,0x6eb484b6),LL(0x87780471,0xa01ef7f5),LL(0xdc02e33e,0xf8ac22f1),L_(0x6e5fad2e)}, {LL(0x0cbad015,0xc2d60cca),LL(0x2db33229,0x78a4abdf),LL(0x375f545c,0xbb2518b8),L_(0xe117a7f0)}},
   {{LL(0xe94cf6d0,0x721159cc),LL(0x2528b498,0xd001dacc),LL(0xba328884,0x6ca68824),L_(0x16db431f)}, {LL(0x857c60d1,0x3e1c1a16),LL(0x06be4963,0x4e3a8a4a),LL(0xe5100cc6,0x2fadd498),L_(0x66deb87b)}},
   {{LL(0x96e332c2,0xba35b5c5),LL(0x5c8ec408,0x32a3227d),LL(0x3d572604,0x3287caa4),L_(0xde9663da)}, {LL(0xd3d8aa49,0xd8f05722),LL(0x3d0b0ee6,0x211c6e5e),LL(0x789832b8,0xd7095c24),L_(0x057c1df6)}},
   {{LL(0x8e5e0a09,0xa554b8a7),LL(0xc0ca83b1,0x3cfefa31),LL(0xde9b7d08,0xb712685d),L_(0x890d7a66)}, {LL(0xd942a6e4,0xc242dae3),LL(0x4f659bb9,0x5c2986c0),LL(0xca0a3ad3,0x58984f5a),L_(0xc0f8dce7)}},
   {{LL(0xbb8f3d67,0x75e4f894),LL(0x305ab1c6,0x597d61ec),LL(0xc5900752,0xbb05ff05),L_(0x6f0cfc37)}, {LL(0x78265563,0x8bb8c320),LL(0x928753a6,0x2afed364),LL(0x9604812d,0x2ecd08d1),L_(0x9f7250a9)}},
   {{LL(0x4f8511c5,0xf62b6ad5),LL(0x68ca6d7c,0xe0ab6bd5),LL(0x9849b3b9,0x437c1eb4),L_(0xdab12cd3)}, {LL(0x60c66157,0xe2da9ad4),LL(0x81e8b4b4,0xc72b155a),LL(0xe933cc08,0xad7576ea),L_(0x66098ae9)}},
   {{LL(0x019aa18f,0x06661e71),LL(0x997b0541,0x2a662ce1),LL(0xc9069497,0x5ff8cdb7),L_(0x7c209e40)}, {LL(0x22ef5987,0x62a400b7),LL(0xdcbb6b6a,0xb2915c26),LL(0x1fe38da6,0xffba34b0),L_(0x6f5cedf1)}},
   {{LL(0x6335b064,0x6133838a),LL(0xe7f5ee57,0x76c52676),LL(0xce767a1c,0x5e465185),L_(0x1464e362)}, {LL(0x93f4d930,0xce0d54ed),LL(0x5d714210,0x742e50db),LL(0xabc94edc,0x78bba80f),L_(0xc86bf6f4)}},
   {{LL(0x4b83b6ce,0x81b49586),LL(0xa3c55ff8,0x1fa505f0),LL(0xae4acd7d,0x832817ef),L_(0x77814820)}, {LL(0xd6ef9110,0xd59182a2),LL(0xe98e0fd5,0x764ce5bc),LL(0x323e203f,0x63236d14),L_(0x3a8b8995)}},
   {{LL(0x118cb51a,0x8ea6b928),LL(0x117ab1a0,0x56dd68b6),LL(0x395935d8,0xbd116bcd),L_(0x0a981729)}, {LL(0x5ced4a6c,0xbb4ec3a0),LL(0x569614d1,0x63f4b4af),LL(0x1f24cb4f,0xf9b9d0b1),L_(0x03e53738)}},
   {{LL(0x18a50233,0x25f531db),LL(0x0389f67e,0x8b3f0f16),LL(0xa0701098,0x6a6aef50),L_(0xa69dbc19)}, {LL(0xfb3a537e,0x8c64961c),LL(0x35b24bda,0x83f4d9b6),LL(0x440a2f19,0x7696fcba),L_(0x9f01b04c)}},
   {{LL(0xc0f2402c,0x5360ffc2),LL(0x1debf1ae,0x8d33aed5),LL(0x7da3a904,0x4dde5b44),L_(0x0cb2c453)}, {LL(0x6c75adbc,0x0c9938a7),LL(0x11dd5af7,0xf4ef9f5d),LL(0x30114b89,0xbb08ab61),L_(0xc5ca168c)}},
   {{LL(0xdf042ef8,0x481cae35),LL(0xaab6e2e0,0x6fed557f),LL(0x3de7d914,0xa4e1bcfb),L_(0x7fdaa079)}, {LL(0x85cd2d00,0x4370738c),LL(0xd12da321,0xf95a95e5),LL(0x9814d14c,0xe0fd8dcb),L_(0x32743222)}},
   {{LL(0xc27ed928,0x655d6ffb),LL(0xde08c413,0x43a4fc18),LL(0xde7270a7,0xcba9f29e),L_(0x06e6cdba)}, {LL(0x2e139df9,0x5cc92d84),LL(0x65a9b95c,0xc5a596a0),LL(0x6a5bdae7,0xd5e4697c),L_(0x5db94611)}},
   {{LL(0xa1aad672,0xf4cdb11a),LL(0x0a56c5fd,0x35eb1979),LL(0x06b67d2c,0xd3a50661),L_(0x227904a1)}, {LL(0x47a4312a,0x2c80dfe0),LL(0xdec3bab1,0x1e137540),LL(0x2528a436,0x4a30bc3a),L_(0x306505c1)}},
   {{LL(0xa27429d2,0xac029aa3),LL(0x3ea574fb,0xe555d79c),LL(0xc26117ae,0xeee94c98),L_(0x485edd44)}, {LL(0xb5866e81,0x3b83d3dc),LL(0x6558e1d3,0x497236f4),LL(0xa896a908,0xc1eba30c),L_(0x2a815dd8)}},
   {{LL(0x0b1d2a89,0x6a45073a),LL(0x2ef3d416,0x72b5edce),LL(0xc7a3b1fe,0xac5268b4),L_(0x5c03d364)}, {LL(0x8589b53a,0x25c0f37d),LL(0x247dcc63,0x8cf1d802),LL(0x9d3fe1d2,0x15093e7c),L_(0xfe72abd8)}},
   {{LL(0xac83f07d,0x6eba86a7),LL(0xdaf79a6c,0x7769b4ef),LL(0xce08325a,0x306da9b2),L_(0x72e3aed8)}, {LL(0x9fd7086a,0xbe70ef68),LL(0x8bdda042,0x9649ac38),LL(0x59634c20,0x260c709f),L_(0x169d616d)}},
   {{LL(0xe36b9e36,0x56c9f551),LL(0x8ce0bbd5,0x9a19b344),LL(0x7b1c335b,0xb90484f8),L_(0xcd0b498b)}, {LL(0x0b5ef9ed,0x8c6e220c),LL(0xdabb30f9,0x4ddb2065),LL(0xe5c29249,0x2b898fcc),L_(0xd16ae2eb)}},
   {{LL(0xb7d4dedd,0x3cc11efc),LL(0x87733216,0xc5d55aae),LL(0xebad4304,0x139832e2),L_(0x6f2f9588)}, {LL(0x0089eb00,0xb6ab35e5),LL(0x143ac48b,0x35af18a2),LL(0x363f80f9,0x8c8c5655),L_(0x8c090d66)}},
   {{LL(0x3cad20e3,0xd54cc179),LL(0x56fcbabd,0xc7630400),LL(0x8ece816d,0x4347fa49),L_(0x7ff18955)}, {LL(0xe22f3ed7,0x83d72b28),LL(0xa523c18b,0xa89efb1e),LL(0xc07fe6b1,0xf679cfd6),L_(0xb95c3988)}},
   {{LL(0xdd4159b7,0x90409ff6),LL(0xe0447eff,0x0509d402),LL(0x0d946df5,0x46153d5b),L_(0xe1645594)}, {LL(0x1229cd0f,0x25841f04),LL(0xd79eba33,0x163487eb),LL(0x1f5ac4bb,0x7a894b16),L_(0xd4b696a9)}},
   {{LL(0xd0fd0fdf,0x95e0fb87),LL(0x316c5da1,0x12d4a0db),LL(0xbac4ce3b,0x4aecb41e),L_(0x07cf2ae4)}, {LL(0xf6b866b3,0xc85cee8f),LL(0x4241c703,0x929c773e),LL(0x22cf7705,0xf90855df),L_(0x6b6a6a3d)}},
   {{LL(0xb938eef0,0x054fb615),LL(0x8dd3e916,0x7d1ce422),LL(0x624c46af,0xc8279b91),L_(0xfa11e3a9)}, {LL(0x6fc664fc,0x957fa9ea),LL(0x7faaee34,0x86d71f57),LL(0x8216ab5f,0x4fdcf503),L_(0x8f169cac)}},
   {{LL(0x99acbf60,0xea1c792e),LL(0x995de7af,0x3d4924d1),LL(0x09c0ca0b,0xb94190da),L_(0x95e76a43)}, {LL(0xaccafdf9,0x6c1aa007),LL(0x1f952403,0xe4eccac7),LL(0x07821f33,0x3125c30a),L_(0x56a0a57c)}},
   {{LL(0x76375e86,0xa2da6002),LL(0x0c8b88d7,0x851663d2),LL(0x903300d5,0xe4304412),L_(0x6111b04d)}, {LL(0xf13f93d7,0xac6cc7e1),LL(0x8ce4d931,0xc233fc51),LL(0xfa884feb,0x889b7699),L_(0xe977a15e)}},
   {{LL(0x649b4878,0xb98eb821),LL(0xf89766d9,0xda85057c),LL(0xe8c391ca,0xc53a37ec),L_(0x15f269aa)}, {LL(0x3c241d53,0x1c63bf16),LL(0x40246227,0xe6f1e93f),LL(0x882db63a,0xacf7e8f4),L_(0xa54ec9c8)}},
   {{LL(0xd18b015d,0xf734d021),LL(0x943cb99b,0x62b04fde),LL(0xea71f882,0xe9f4d402),L_(0x4f517e6c)}, {LL(0x3d8f7745,0xad9abc71),LL(0x1fe90081,0x088ad020),LL(0x25dfde15,0x250f4123),L_(0x48badfac)}},
   {{LL(0xe0e96ead,0x46b7097a),LL(0x2937ebe1,0x19ddbba5),LL(0x3ecd0bd0,0x1b6e3a84),L_(0xaf895097)}, {LL(0xfc22adbf,0xbae23185),LL(0x220a75e9,0x09ed4ed6),LL(0x062eb7e7,0xe1ce81e7),L_(0x8b1103c2)}},
   {{LL(0x4649146f,0x91702d2f),LL(0xb9f95742,0xbfdaf3ca),LL(0xcf8f23f1,0xd403a4bb),L_(0xf9164036)}, {LL(0x33bbb505,0x5d48ee37),LL(0x2b84d0f7,0x79ec8716),LL(0x5a9ca7a8,0x76092de3),L_(0xec444f35)}},
   {{LL(0x1f9e6c98,0x17ed7898),LL(0x25820dda,0x668c28ad),LL(0x745aa2ed,0xcf9ce234),L_(0x01a5ca97)}, {LL(0x4a783729,0xbf5955af),LL(0x31ed52ae,0x92891a4a),LL(0xea43a792,0xc67457a9),L_(0x7c0fc898)}},
   {{LL(0xb8dd47fd,0xff42200e),LL(0x253f7438,0xa70a5da7),LL(0x6cbd4092,0x6d058c66),L_(0xbb3089d5)}, {LL(0x110c3a9a,0x67048ffd),LL(0xa8a2fe2a,0x82fc6b0e),LL(0x30242c32,0xa769318b),L_(0xf0614bb4)}},
   {{LL(0xbb708cee,0x7620c902),LL(0x084e78c8,0x75832ff6),LL(0x3a89190f,0x4632df7f),L_(0xdade5c0d)}, {LL(0x48e4bd14,0x62dc813c),LL(0x0d15d331,0x9c55f000),LL(0x36a4457f,0xa330078b),L_(0x3b722d3e)}},
   {{LL(0xc1f3d1aa,0xe37d1ff2),LL(0x5c4d9799,0xed06b0a5),LL(0xb1d3c33d,0x939c8f4c),L_(0x499b0fdd)}, {LL(0xf53390da,0x9e6ebae1),LL(0xb952b38d,0x28276f38),LL(0xd69e1327,0x19f5bac2),L_(0x7c921bee)}},
   {{LL(0xafd90d24,0x0e84e363),LL(0x37d2a8e9,0xabb80eda),LL(0x07e40274,0x95656580),L_(0x877a2cb3)}, {LL(0xa8a767b1,0xc90af857),LL(0x61a42dc7,0x465bb86d),LL(0xbf658fdd,0x7eb455a7),L_(0xf467f4bf)}},
   {{LL(0xda192001,0xad70ec43),LL(0x1aa502d0,0xcb55de32),LL(0x7d66c4af,0xa952a38d),L_(0xb6579e71)}, {LL(0xdceb099f,0xed3e0675),LL(0xaf274aa3,0x635a8629),LL(0xb9f4c7da,0x8d6f1c2a),L_(0xf954b1aa)}},
   {{LL(0x6f166fdd,0x0c46e6e4),LL(0x2f6ebcd8,0x0813e798),LL(0x9066cff4,0x2f759557),L_(0x026ee79d)}, {LL(0xbd44613e,0x06317041),LL(0xc1609f3b,0x6879039a),LL(0x6853d2d8,0x80566bad),L_(0x9a1de27b)}},
   {{LL(0xd83655d4,0x7c1a0bb8),LL(0x40e36289,0xa5228881),LL(0x00c5e925,0xe9c51c0b),L_(0x17fd5193)}, {LL(0x0234b859,0x9a00e742),LL(0xee8b4f08,0xf774358d),LL(0x0eb69043,0xd46eb256),L_(0xf54e46c1)}},
   {{LL(0xe386daac,0x9a305482),LL(0xc37330b6,0x7957d9d7),LL(0xf3f52055,0xd1a9ce7b),L_(0x1e870d5e)}, {LL(0x303b5ae9,0xac2ad67a),LL(0x4d3fc2d7,0xb7b58e09),LL(0x7dc5f5f1,0x6c99d250),L_(0x2d7b151a)}},
   {{LL(0x29507cf5,0x37c15d8e),LL(0x6919a75b,0x738d9ba6),LL(0xeb76b42c,0x0b91ab81),L_(0x3da557ec)}, {LL(0xac1dd9b8,0xb974e300),LL(0x6b8577ca,0xd55b5831),LL(0x9748b154,0xbfcdc886),L_(0xf2e99cba)}},
   {{LL(0xaa791b19,0xb611f73f),LL(0xe44e650a,0x6fd43d10),LL(0xde586616,0xff2a1da0),L_(0x8983db6a)}, {LL(0x7da57055,0x5051af3a),LL(0x73e374c8,0xa7d98205),LL(0x80c04759,0xd4db7bec),L_(0x28c48bc7)}},
   {{LL(0x9051c153,0xd36418fc),LL(0xed854f7d,0x2adcc4eb),LL(0xa2d25b82,0xf9315c6a),L_(0x510b53b5)}, {LL(0x6fec476b,0xbdd2e105),LL(0xbc40ed1e,0xdc2bc2a5),LL(0xd473b242,0x9d87dbe8),L_(0xf9d3eb65)}},
   {{LL(0x4031a20e,0x5001f11c),LL(0x3e83e92e,0x86e3fa8d),LL(0xa7c80719,0x36408819),L_(0x520ab89f)}, {LL(0xaf0e8126,0x430d59f8),LL(0x1479126e,0x5b426c42),LL(0x1c62cb71,0x1a2c3f95),L_(0xba8c53ff)}},
   {{LL(0x42998c5c,0x9b1ab1ad),LL(0xc5bef946,0xe7ad335b),LL(0xd2f18816,0x9b21d6a9),L_(0x7bf5543b)}, {LL(0x09dc1281,0xc3771787),LL(0x8d9a2927,0x69b13c80),LL(0x5419b128,0x472d8eda),L_(0xce12aaf7)}},
   {{LL(0xc26aa121,0xb9665455),LL(0x1990f311,0x3bac4cd2),LL(0x4b96b490,0x15632e30),L_(0xb87e9336)}, {LL(0x03c2ca60,0xeac8678b),LL(0x101c2863,0x5772568c),LL(0xbef62c1e,0x3b53ce81),L_(0x7d135e14)}},
   {{LL(0xbe66a12b,0x753cb0aa),LL(0xd1abe43d,0x38781cdd),LL(0xcf1c41a4,0xab22df0a),L_(0x11b70405)}, {LL(0x7e4c97e5,0x3869cfcb),LL(0x24f71698,0xb43a4e85),LL(0x6ab97c59,0x35c0f24a),L_(0x4df75964)}},
   {{LL(0x292f0d69,0x3e2b7aaa),LL(0xdafad7ad,0xe2ba5f7a),LL(0x77ff29a4,0xe21d90f9),L_(0x67fec4ee)}, {LL(0xbc0b01d0,0xe3677335),LL(0x26e5157e,0x8fee4b54),LL(0x2b1a301a,0x5d1511ed),L_(0x263f34a8)}},
   {{LL(0xe5e9578a,0xd73285ea),LL(0xdf28a719,0x2e506fa1),LL(0xecb1a91e,0x79d54bca),L_(0x5a06ac79)}, {LL(0x7bf48ede,0x9f75d7b8),LL(0xa6637969,0x08014deb),LL(0xaf188a86,0xbfada230),L_(0x423dacf9)}},
   {{LL(0xa63e4dbd,0xe55bd873),LL(0xa80a0320,0xbe01f07e),LL(0x1b09710f,0x6e6cb66a),L_(0x2fee3ce5)}, {LL(0x955bdbc6,0xcecf1493),LL(0x2a3070a3,0x861a0878),LL(0x084d6464,0x0d0e35ed),L_(0xa03afc73)}},
   {{LL(0xcdca5ed5,0xdb335ec9),LL(0x78cc343c,0xa8df4be9),LL(0xcaea8450,0xf564ec58),L_(0xa5438f08)}, {LL(0xffeefeda,0x193c8457),LL(0x0eaf627b,0xddabba77),LL(0xc99c9580,0xa8ec469f),L_(0xc9ea9e43)}},
   {{LL(0xf3c9cc69,0xa888f556),LL(0x8951408c,0xaaa99a76),LL(0x79ee145a,0x4760cde2),L_(0xfbdcc6a6)}, {LL(0xd543c51c,0x4d61119d),LL(0x6d5cde8a,0xb57c70b1),LL(0x6d605846,0x8392010f),L_(0x824097f5)}},
   {{LL(0x79dcc935,0xd988c0b3),LL(0x97a596be,0x66a3883f),LL(0x4cdffd50,0x026d6f1b),L_(0xb093bfc3)}, {LL(0xd01d8fb0,0xc2150428),LL(0x477892be,0x453f0c91),LL(0xff7a816c,0x84b14e80),L_(0xf02eff58)}},
   {{LL(0xa7823319,0xab4b5d45),LL(0x76a3294f,0x1d9dae80),LL(0xce1d1836,0x1f8f6e55),L_(0x7064e78f)}, {LL(0x8d6bd5d7,0xd272957e),LL(0x20068c6a,0xc59f4815),LL(0x15f1375a,0x1dcaf5b0),L_(0xc01d795e)}},
   {{LL(0x6c316cb4,0x5f19c736),LL(0x0082a8d7,0x9bb6aded),LL(0xadea31b9,0x1090e65a),L_(0x27c39f72)}, {LL(0xfec08edb,0xad397892),LL(0x2ae66c84,0xb2c262c0),LL(0xe12a0551,0xcd11a65d),L_(0xc52c046e)}},
   {{LL(0x5db707b6,0xf10d38e7),LL(0x4a3e6c1d,0x5bbaa9f8),LL(0x1692548b,0xa34228d1),L_(0x2ff089a1)}, {LL(0x654e5d42,0xf2342e4d),LL(0xe8e23440,0x53e19ef5),LL(0x254b33f9,0x2609459a),L_(0xd598bc1e)}},
   {{LL(0x7007f546,0xefba2361),LL(0xe1a42d85,0x72c3ffc2),LL(0x600f4b0f,0x68bfc655),L_(0xf5be3dea)}, {LL(0x13f14da8,0x6efd514e),LL(0xaaab23fd,0xcc761583),LL(0x949da3c4,0xd3524443),L_(0xf07ce5ed)}},
   {{LL(0xeca99656,0x98853678),LL(0xa45dd69d,0x959586ec),LL(0x2b4d5496,0x882a160e),L_(0x54e92805)}, {LL(0xdf2a1b89,0x52e13aac),LL(0x8f1d99c7,0x4a5dde80),LL(0xc432a9d8,0xcafd8d25),L_(0x801f070d)}},
   {{LL(0x39455fca,0x0c20053d),LL(0xdaf5e5b7,0x04a4c141),LL(0x14b7b1d1,0x38e2582a),L_(0x51bd8c01)}, {LL(0x20e68184,0xd15dab61),LL(0x1925d2d5,0xba541287),LL(0x62766fdb,0x6619e0a4),L_(0x439e1dd0)}},
   {{LL(0x3bc818e3,0x746af8f1),LL(0x6be8f43b,0xbbe6fd69),LL(0x53589554,0x446a9ab9),L_(0x0ea769ab)}, {LL(0xc30fa666,0xf4d47c43),LL(0xca3dad1b,0x431ea6ad),LL(0x9dafb227,0x4ff395ed),L_(0xbd3937cc)}},
   {{LL(0x2c57d4e1,0xa4cceb3e),LL(0xd6667dff,0xff6cbcb5),LL(0x8918c037,0x268bc00c),L_(0x1c4ef107)}, {LL(0xa92f2fb1,0x76415814),LL(0xd31c6973,0x68597539),LL(0x0a6cac9a,0x399773b8),L_(0x87d12cd2)}},
   {{LL(0x7652379d,0xc59e872e),LL(0x0e493371,0x1fa3657f),LL(0xa77b28fb,0xa7619402),L_(0xeea23b52)}, {LL(0xa85f48bf,0xe180ae18),LL(0x8d4b8437,0x68f2c962),LL(0x3145c6a2,0xe4d30712),L_(0xd9ccc49f)}},
   {{LL(0xb3ff7311,0x00082b09),LL(0x23fb7431,0x0de3deec),LL(0x33ecf9d4,0xab6cc532),L_(0x8eae9253)}, {LL(0xd8a0ae1c,0x2a618d55),LL(0x97045c63,0x3993ca70),LL(0x0f2549ab,0xcc02ada7),L_(0x14884bad)}},
   {{LL(0xb5a7fc93,0xec8a7e38),LL(0xfbda890b,0x786e2a85),LL(0x882cb9d7,0x103bbacc),L_(0x17321d75)}, {LL(0x7395bf16,0x4be2ec80),LL(0x20a02de8,0xb287766f),LL(0x4eb9dbc0,0x54c7f36b),L_(0x399d4a59)}},
},
/* digit=29 base_pwr=2^203 */
{
   {{LL(0x663a8bf6,0xf1bd6f1b),LL(0x6c6b1cae,0x2de2cf35),LL(0x4f49967b,0x49fde755),L_(0xee2050c1)}, {LL(0x9956f1ed,0x69f14061),LL(0x138a98db,0xc5c29863),LL(0x516db87b,0xd4456e50),L_(0xe15ee455)}},
   {{LL(0x78b8a6b9,0x5019a365),LL(0x0782c606,0x025726c3),LL(0xd611e283,0x68f7b67f),L_(0x9292eb6b)}, {LL(0xb4e7c5d0,0x5ed6c698),LL(0x8f074f3c,0x6ea9c71d),LL(0x2c0dd03d,0x6ad6ab61),L_(0x6c985359)}},
   {{LL(0xc1c1b8cd,0xa1ea038f),LL(0xd886b019,0x9d8f1f3a),LL(0x5e44aefb,0xbc0d302a),L_(0x8b9ba536)}, {LL(0x47ec400d,0x9049574e),LL(0x02b6e2ef,0x9778ebf4),LL(0xe0adc015,0xe0348d22),L_(0x2f89c060)}},
   {{LL(0x2b432e4f,0x97be15f2),LL(0x9442eea7,0xfb123347),LL(0x752805f0,0x095f8861),L_(0xbd4f1599)}, {LL(0xe4fd179d,0xbfe10a89),LL(0x8eb30cbd,0xc1a95173),LL(0xb5e94f0f,0x38bffbb7),L_(0xbd72f70b)}},
   {{LL(0x31d5bf11,0x3b628b52),LL(0x6efe2871,0x9f017556),LL(0x50b6b23f,0x2051232c),L_(0x05744158)}, {LL(0x18b1fb8c,0x58660126),LL(0x6683fda3,0x85343658),LL(0xcdefb5cb,0xee7c02af),L_(0x0c98f701)}},
   {{LL(0x842b943a,0x79b770df),LL(0x52f1e8f1,0x2ca8d23c),LL(0x29cf116b,0xd89fd3b6),L_(0xae88c948)}, {LL(0x20362fba,0x17dff9f9),LL(0x7048ea42,0xb8350a9e),LL(0xcbd44b83,0x3adbe1f1),L_(0x5fc81426)}},
   {{LL(0x171138f5,0xd002fd82),LL(0xb9b3d4ff,0xed715f0c),LL(0x14d543b9,0x770fe818),L_(0xc2b33795)}, {LL(0x2c123604,0x3f8f258d),LL(0x22543d13,0xa3564d33),LL(0x9d216804,0x435f77e5),L_(0x97bba91d)}},
   {{LL(0xabe6aead,0x5e84d762),LL(0x84f02bef,0x538499e7),LL(0x1d74cb44,0xbfb3e36a),L_(0x591a2397)}, {LL(0x198dddb8,0xfcbe1984),LL(0x430a25b9,0xb7b8bbcc),LL(0x80cf7321,0x00ac08de),L_(0x44305966)}},
   {{LL(0xe8745380,0xbbb4692e),LL(0x9a66429c,0x02f17d1f),LL(0xe7057c36,0xec132fea),L_(0x87b7dbea)}, {LL(0x89b168f8,0x7f9c9480),LL(0xe2d21c13,0xf9e94167),LL(0xf34e2409,0x13e022d0),L_(0xba4bcae2)}},
   {{LL(0x243d2e97,0x358d9af4),LL(0x518d83d9,0xb47467ae),LL(0x81a7eea1,0x470b2dab),L_(0x37ab475a)}, {LL(0x3d8103c5,0x9a2755c4),LL(0x200b1434,0x9228b07c),LL(0x5b3e4741,0xbc9b37bd),L_(0x8f9c88f6)}},
   {{LL(0x8739f51d,0x3e9880c9),LL(0x183b3d2a,0xe30595ad),LL(0x9d3ff1f0,0x4b286b55),L_(0x2ffa355d)}, {LL(0x89be0142,0xe2576d49),LL(0xf43534f3,0x00a64de7),LL(0x1e63717f,0x0c898bea),L_(0x338d6967)}},
   {{LL(0xd89a7c52,0x489bd20b),LL(0x1cbe5fd9,0xe7ac0a68),LL(0xf9bbb011,0xbe7ead5c),L_(0xfdfdaa41)}, {LL(0x5bf6715e,0x2573c5cb),LL(0x71fec319,0x30483451),LL(0x7ccb3fa6,0x0071ca44),L_(0x02bc8a83)}},
   {{LL(0x83211264,0xd01481a8),LL(0xd292cd9c,0x48c43d0c),LL(0xd5277f78,0x32c5572b),L_(0xb3f946cd)}, {LL(0xb34b30d2,0x780e2016),LL(0x492c55b7,0x43e3806b),LL(0xd4ad90ce,0x97399684),L_(0x2bddb7ec)}},
   {{LL(0xe410e719,0xb76d2e77),LL(0xaa75746e,0x0df8f582),LL(0x6f7bfcdc,0x210d7777),L_(0xce4ca9a2)}, {LL(0x04ef15f2,0x87836a34),LL(0x77b0ffb3,0x44c267d0),LL(0x2fbcd218,0xbb549183),L_(0x8bf3d46f)}},
   {{LL(0xa541189d,0x3029c3b6),LL(0xbeb5a06f,0x277c1d19),LL(0x761b9d9e,0xd09fd429),L_(0x189262a3)}, {LL(0x8fe5cbe0,0xf3cbac6c),LL(0x4d675bde,0xcbbe3e5c),LL(0x4dd89b19,0x75e821e8),L_(0xf4cfe02e)}},
   {{LL(0xf92173f6,0x3b6ebbae),LL(0x66df420a,0x44ad6eda),LL(0x49918d50,0x71e12775),L_(0x6343f5ce)}, {LL(0x7c6c9ea4,0xa76fc6d0),LL(0xde640ea7,0x95bb0f9e),LL(0x0e5cdfb0,0x3505e20d),L_(0x03859db8)}},
   {{LL(0xc0a26f34,0xf36a466b),LL(0x8087482a,0x769bd94b),LL(0x96b0b77f,0xbaa9cde2),L_(0xe98e1c0a)}, {LL(0x961d12a3,0xe383e51b),LL(0x9d59941b,0xb680bdbd),LL(0xfbf75271,0x1f5805cf),L_(0x13cbe6aa)}},
   {{LL(0x205ad8c7,0x84ebf013),LL(0xc207bf8a,0xf24e1950),LL(0x749aace4,0xaaf03f06),L_(0x566ca993)}, {LL(0x33c5f5b5,0x378cf813),LL(0xe91d36bf,0x8ac53e4b),LL(0x9b1441f3,0x0113566b),L_(0xb69efcbb)}},
   {{LL(0x8145a741,0xd10f5321),LL(0x81bb9251,0xe73eb2a7),LL(0x4da9ccbf,0xf35011e6),L_(0xd0a422fb)}, {LL(0x375da80b,0x3b493784),LL(0xac876682,0xe99fa1c1),LL(0xbfcdd019,0xfcd0fffb),L_(0x16d39a14)}},
   {{LL(0x44724691,0x6169a512),LL(0xf345fdba,0xb54ae1f9),LL(0x97426a68,0x84a843c6),L_(0xc87c52d5)}, {LL(0x57ee464a,0x9fa3b852),LL(0x36872038,0xcf4188dc),LL(0xb9fbee93,0xc6f126b9),L_(0xebc64e1f)}},
   {{LL(0xa34571ca,0x48662048),LL(0x54bb6dce,0xfa26591d),LL(0xf6f0b8d2,0xc7f51cd9),L_(0x5cfd4a24)}, {LL(0x5bc438cf,0x66200c18),LL(0xc248c469,0xe6f1c95c),LL(0x6844e3a4,0x6e17cd42),L_(0xa72dc6d8)}},
   {{LL(0xddd1ff7e,0xf244b120),LL(0x431f42e4,0xe6046902),LL(0x40c26687,0x1691fe67),L_(0x30ca7e9b)}, {LL(0x6e105d9d,0x9b50eded),LL(0x961020de,0x00b6a88e),LL(0xe9b08b55,0x4b82631a),L_(0x6cc54fd6)}},
   {{LL(0x04530c1f,0xf4435606),LL(0xf234709d,0x9b79c17a),LL(0x9f045501,0x839380ae),L_(0x5f60c0b0)}, {LL(0x2d608e32,0x9b83bfd2),LL(0xadf75cc7,0x10db4135),LL(0x22cc1949,0x3a2e7962),L_(0xe3436f2d)}},
   {{LL(0xec19a27c,0xe5ad9b86),LL(0xb18ea5cf,0xc7d0db50),LL(0xb0a1794d,0x12340a6f),L_(0x2c1d9f6e)}, {LL(0x8bcb0708,0x54438da3),LL(0x032032f2,0x33597066),LL(0x68e9b910,0x192ee4ac),L_(0xdf1723d3)}},
   {{LL(0x6b3b3e12,0x6076299f),LL(0x49f13a7d,0x210c46f7),LL(0x11bcefd8,0x4c36d81d),L_(0x9b39bf08)}, {LL(0xd49ab219,0xb72e4c99),LL(0x8e53b636,0x0e9348ea),LL(0xa88c581f,0xd67c4ee9),L_(0xa41298b4)}},
   {{LL(0xe899e96d,0xb64bf9de),LL(0x8d198f03,0x43b5e3ae),LL(0xa7cc5c88,0x656b15b3),L_(0xd2995e8b)}, {LL(0xc5c4e24b,0x112512c6),LL(0x409ca870,0x26ff024e),LL(0xccd67ae6,0x4f8c1dc8),L_(0x22878c0c)}},
   {{LL(0x95b13b3e,0x9e553300),LL(0x668c95c0,0x7fafd447),LL(0xf8451257,0x6a113d4a),L_(0x1f574c1c)}, {LL(0xedde7f07,0xaf37550f),LL(0xf5868469,0x0b1d7c54),LL(0x69e02d8f,0xcfd6985a),L_(0x9a97854f)}},
   {{LL(0xecd3e1ab,0xbe482681),LL(0xdc9e0113,0x47798831),LL(0x9fac48a4,0xafbbf20e),L_(0x1edb9dbc)}, {LL(0xd6f1c5bd,0xfe2c6138),LL(0x58a4f161,0xd3d7cf09),LL(0x0fdeccc7,0xa05a2e1d),L_(0xc2e844b9)}},
   {{LL(0x6b3cd8ad,0xa16dcddd),LL(0xe9387af0,0x0fdf0fc3),LL(0x3cb7db3e,0x5144175e),L_(0xc5aef05c)}, {LL(0xdaa284b1,0xb41e11e0),LL(0xc70c6218,0x334a53c4),LL(0xb6f86d54,0xea0526d6),L_(0xd996408c)}},
   {{LL(0xef18bf5d,0x268ce667),LL(0xd54a4821,0x70f67533),LL(0xb9cffb18,0xc3038e8d),L_(0xebd042c2)}, {LL(0x29b2c2c4,0x4d1a3a20),LL(0xd6237f1f,0xe2c18664),LL(0x5ab3157b,0x7a60e7b3),L_(0x6be26cfd)}},
   {{LL(0x1c5c3f1e,0x28e72014),LL(0x5435c35c,0x32068427),LL(0x47c37b5e,0x94c1b6e7),L_(0x0c649b6b)}, {LL(0x5f5b4e72,0x77567891),LL(0x3ebfe7c4,0x55a4f7f1),LL(0xc6f93ff6,0x8f654b8b),L_(0xc6898456)}},
   {{LL(0x8fa02753,0xfcbae4b8),LL(0x9d98e011,0xcff78948),LL(0x22632d48,0x70d86a48),L_(0x247b98b7)}, {LL(0x00132e50,0x3721a7ea),LL(0xad1058dd,0xdcc00609),LL(0x0c2cbce2,0x38fb42c3),L_(0x574791e5)}},
   {{LL(0x5ea4abc8,0x5f3751e3),LL(0xaacf3fc1,0xac364b4a),LL(0x495dd3d3,0xa6c1d838),L_(0x7fd58564)}, {LL(0xba68755c,0x465b3970),LL(0xfc76d6ce,0xfe4e336d),LL(0xfc16f7fe,0x2c759722),L_(0x76296d07)}},
   {{LL(0xdd3a0177,0xcb5a7236),LL(0x9f03a670,0xa2bbecb1),LL(0x1f553745,0x46d5deeb),L_(0xc27cf8e5)}, {LL(0x4c542036,0xa92075fc),LL(0xe9a1424d,0xa99e29eb),LL(0x31176d99,0x7ca41c07),L_(0xd9fd4345)}},
   {{LL(0x39ce1949,0xbff084c9),LL(0x98f4d791,0x43984570),LL(0x05ef0ee6,0x0a861937),L_(0x4abffe54)}, {LL(0x1288a0c4,0x52997092),LL(0xe85a84ab,0x24210cbe),LL(0xec238c31,0x9e2872c4),L_(0x574f26c0)}},
   {{LL(0x40fd474b,0x5b8538a3),LL(0x1c70eab8,0x0d208047),LL(0x1cb60ff4,0x761298e8),L_(0xcfbdc26c)}, {LL(0x1b6d4500,0xbcf74593),LL(0x7f1b2563,0x746b2f75),LL(0x679d4dee,0x32233e28),L_(0xdec42565)}},
   {{LL(0x0c718f1a,0x66cd9b3c),LL(0x18427f69,0x3b3a6f84),LL(0xa9ef5cce,0xca539a7c),L_(0xeb7d5235)}, {LL(0xed44fad8,0x4db83c78),LL(0xc210c701,0x5a9002ed),LL(0x57ef09d7,0x47b485ec),L_(0x27b80c9d)}},
   {{LL(0xe4537556,0x4cc9c9c5),LL(0xea7854b5,0xec59f90a),LL(0x55f05058,0xaa97605c),L_(0x82a4feeb)}, {LL(0x66f84f5e,0x8c26f4dc),LL(0x51a67be7,0xff1f32e7),LL(0xda9cb1ce,0x92d5f2c4),L_(0x3ef1254f)}},
   {{LL(0x5ec4e735,0xffb048ed),LL(0x17a4afa4,0xb9ca6eb1),LL(0x4f8b9e40,0x089b9558),L_(0x6e75ba13)}, {LL(0x43e91e89,0xcca4bf58),LL(0xbbaa89ed,0x21a8592f),LL(0x2180b560,0xa9e4e373),L_(0x2a4fdd34)}},
   {{LL(0x95e834c3,0x85c32c46),LL(0xc68fbf2c,0xad18d468),LL(0xd0c03244,0xf46c7fff),L_(0x8c227e96)}, {LL(0xde2297b6,0x8c32b174),LL(0x6ae73d4a,0x28635d97),LL(0x8b431269,0x47d44429),L_(0x2421faa0)}},
   {{LL(0x28c78e41,0x93181781),LL(0x2ad021ad,0xcb623c4d),LL(0x98d40a79,0xdda4b06c),L_(0x372dd0c7)}, {LL(0x3b688d0e,0xfe3fd141),LL(0xad24bea1,0xedd15e12),LL(0x3a965d08,0xf3097f75),L_(0x27f71cdb)}},
   {{LL(0xbda9cc55,0xa6bfe2fa),LL(0x80a54113,0xf02a2f58),LL(0x1b6a851d,0xcb819b77),L_(0x21f5d0c7)}, {LL(0x30780f8f,0x9f0d56af),LL(0xb5d4e624,0xf8a06fec),LL(0x27c807cb,0xb99605d9),L_(0x75a80279)}},
   {{LL(0x6786d970,0x72255ae1),LL(0xc780fd84,0xc29277a0),LL(0xb9deb14f,0xb01102d2),L_(0xb94bf48a)}, {LL(0x1c8e7e82,0xaf4a6e1a),LL(0x5d7fa22a,0x06c802c5),LL(0x4637b70a,0x21f2e3c1),L_(0x514a3fb5)}},
   {{LL(0xb4a7c609,0x3a6619c7),LL(0xebdd3a32,0x5deb2376),LL(0x24dd90ec,0x7e94da01),L_(0x6003a0d8)}, {LL(0x3788bd2c,0x80b9433f),LL(0x3d74e532,0xd49fdafa),LL(0x47c8e8f1,0x2be1c836),L_(0xce9e83c2)}},
   {{LL(0x7f036147,0xf5f17b3f),LL(0x3b99cbce,0x45a7571b),LL(0x4e68fb20,0x68857bb8),L_(0xa10f6fe3)}, {LL(0xa24d2c8d,0x2d13dee5),LL(0x4f9e2b49,0x88581796),LL(0x118e784d,0xf0493f53),L_(0x9d8cddff)}},
   {{LL(0x58d860e4,0x8971f724),LL(0x316a2d94,0xb4f43e7e),LL(0xb5941f8a,0xb537d4a0),L_(0x5f19914e)}, {LL(0xa327b763,0x0a0d438e),LL(0x5555f01a,0x23118e99),LL(0x2345f880,0xe40f8c30),L_(0x468a1ce3)}},
   {{LL(0xcbd96a39,0x1197dd6b),LL(0xca22160b,0xb47615a0),LL(0x381bf7e1,0x251a8f8f),L_(0x168d73c5)}, {LL(0xad578808,0x34717cff),LL(0x2bd8024e,0x5fbc83f1),LL(0xa4f9d4fb,0x64b6d093),L_(0xff7fe717)}},
   {{LL(0xf183d4df,0xf6acf5ed),LL(0x23f4cacf,0x0cc30970),LL(0x181d2aa6,0x53b37c4b),L_(0x96f6cf89)}, {LL(0x2ce916bb,0x38654a56),LL(0x2803be90,0xaba6d625),LL(0x09ca53b2,0xcdfe1968),L_(0x31bb234b)}},
   {{LL(0x9e6894a9,0xf7c98ba1),LL(0x03ee4c19,0x8e28cba4),LL(0xe28c0250,0xdc432b47),L_(0x6e2b992e)}, {LL(0x52a05294,0xff3120b7),LL(0xbc5f1638,0xd4520461),LL(0x3107ac59,0x92fc422a),L_(0xbc3da988)}},
   {{LL(0x378db33d,0x1b7e14c2),LL(0xafbf4c3d,0x2e5d4aae),LL(0x536bff4a,0x885f68ab),L_(0xbe062464)}, {LL(0xeaff53c4,0x8ea881df),LL(0xd93248b0,0xce5d6df3),LL(0x037450b2,0x6899f186),L_(0x09a8570d)}},
   {{LL(0xe8227714,0xb158373d),LL(0x662aa9d1,0x73e5ad11),LL(0x0d1473e2,0x963a9a91),L_(0x5be0c8ec)}, {LL(0x6796b003,0x32ad74a8),LL(0x5b558e5b,0xcad42d90),LL(0xa157eba3,0x64fb791b),L_(0x3c37606f)}},
   {{LL(0x91b7250c,0xb6fc8bc6),LL(0x88f73c63,0x207504d1),LL(0xa04f258e,0x35b71223),L_(0x9284fb74)}, {LL(0x000189fc,0x8cf221df),LL(0x5e498c73,0xe9692259),LL(0xc7e62afa,0x2482c08d),L_(0x96d697ef)}},
   {{LL(0xa6fa3a45,0xe579dd13),LL(0x2b0e2ed6,0x8643df69),LL(0x57d9dfb8,0x9f01f782),L_(0x4cb4d8ee)}, {LL(0xb5c2c767,0xbb840380),LL(0x358d384f,0x99e8ca50),LL(0x6c698864,0x31b4df34),L_(0xaa8c271b)}},
   {{LL(0xfa87a979,0x6e27995c),LL(0xf59ed52f,0xc806a201),LL(0xc065a04b,0x93cff90b),L_(0x37c928f2)}, {LL(0xbb867037,0x9cdfc841),LL(0x4f37a688,0x0a935f1c),LL(0xa9d96eb8,0xc15bd4d8),L_(0x98cebb91)}},
   {{LL(0xd003f54b,0xa4a49d7a),LL(0xc496b9ae,0xa79d7692),LL(0xa15f7dc6,0xbd4d523d),L_(0x9f46848d)}, {LL(0x6dbfe7b8,0xcc88e7c3),LL(0xcea47678,0x8264318e),LL(0x5e5fd5b2,0x5f2fa1e2),L_(0xe1fd44b0)}},
   {{LL(0xec0ebc0a,0x1936faf8),LL(0xba06a3b9,0xc8c40d08),LL(0x01dd06de,0x54d85f2a),L_(0x18910c39)}, {LL(0x8e9d926d,0xcbcea653),LL(0xc539056f,0x37d0733f),LL(0x0beb45e4,0x3de7fc81),L_(0xd43f2aa7)}},
   {{LL(0x3205880a,0x9850a91e),LL(0x913c877e,0x5464c543),LL(0x1452f874,0xb9637ddd),L_(0xa9b45a5d)}, {LL(0x7298be19,0xdf5911d2),LL(0x01fcff9f,0xf6dfd8a8),LL(0x8bb252ea,0x839f8d56),L_(0x14ad8567)}},
   {{LL(0xc7fe9491,0xa2fc5db9),LL(0x23bc4e41,0x47b2b00a),LL(0x142e5001,0x557178b5),L_(0x7609c04e)}, {LL(0x7281e45d,0x80053d16),LL(0x9af2e131,0x335bfb8a),LL(0x3285f848,0xe9e17269),L_(0xf7e53701)}},
   {{LL(0xd38c0684,0xd98fa793),LL(0x74f33400,0x14117a61),LL(0x0d3b9c08,0x3e3d6538),L_(0x241aea00)}, {LL(0x374f82ab,0x538b9bf1),LL(0x86a9a615,0x1cc16fb7),LL(0xe0c1bced,0xc0de70fa),L_(0x85b224dc)}},
   {{LL(0x96448e8a,0x98c48df5),LL(0x5ba1fe76,0x50f58327),LL(0x4770c98e,0x197ced21),L_(0x2ca4f59f)}, {LL(0x1af1733d,0x2bca7ffe),LL(0x003eeda5,0xeac8a367),LL(0x13278f06,0x5dd0431b),L_(0xa00d2f29)}},
   {{LL(0xa83c536f,0x02379421),LL(0xe322906d,0x218d25f5),LL(0x90caeeec,0xf254a0e4),L_(0x2e6c9704)}, {LL(0x44d824c0,0x0bbdc5a2),LL(0x19e00afb,0x21933c71),LL(0xec2c7cff,0x0a756aae),L_(0x3f239664)}},
   {{LL(0xd8438b58,0x1d02bb13),LL(0xda3809b5,0x25ef520a),LL(0x3dd8667d,0xe3e22c9e),L_(0x68203ac3)}, {LL(0x0d51408f,0x0227f2bc),LL(0x99baf10d,0x090e4a02),LL(0x2fc23bc1,0xf9171e03),L_(0x3a29ec64)}},
   {{LL(0x8d85409c,0xd1184135),LL(0x735db311,0xf357dae0),LL(0x7b364c45,0xb9d690d1),L_(0xee21514f)}, {LL(0x0c5f601d,0xcb872f3a),LL(0x2e8ffdf4,0xbccb4f0f),LL(0xf7bfadef,0x9f017f8b),L_(0xb26ea0cc)}},
   {{LL(0x15442e1d,0x62cd98e6),LL(0x2f59cf2a,0x58dcba94),LL(0x7426c5f8,0x80d0a8cf),L_(0x7c300a9f)}, {LL(0x52648e24,0x3a628b1d),LL(0x1ffce053,0xf0844217),LL(0x9c140c5f,0xb3cc7bfa),L_(0xbb650a75)}},
},
/* digit=30 base_pwr=2^210 */
{
   {{LL(0xf0426be2,0xb4f44241),LL(0xeb69f614,0xefc43949),LL(0x80845fb2,0x90205f0f),L_(0x29198f5f)}, {LL(0x8a6cba71,0x4345224a),LL(0xfb0f1abd,0xfc80d60b),LL(0x9c7703f2,0x3d5a7643),L_(0x663a913a)}},
   {{LL(0xe4b299d3,0x79702991),LL(0xdfeaa3f0,0xad58884a),LL(0xff023a31,0x18b5c6ce),L_(0x8cf14303)}, {LL(0x433c5230,0x3e4907b4),LL(0x13551cac,0x3c1c0282),LL(0xe4454779,0xb4df9ee5),L_(0xae0275c5)}},
   {{LL(0x658675d8,0xdf2f4ac2),LL(0xd8448d41,0xdde660e0),LL(0x50213318,0x64826f9f),L_(0xb76ac409)}, {LL(0xadcc3e46,0xbc26833e),LL(0x508eaf24,0xde547aae),LL(0x619251e2,0xf96f73d1),L_(0x7bcc2806)}},
   {{LL(0xba3e8f2f,0x728cadb5),LL(0x765732f7,0x0feebd09),LL(0xdb740393,0xdcffa1d6),L_(0xab748823)}, {LL(0xa742cd70,0x549a4fca),LL(0xf07ac1f7,0x3d24aeea),LL(0x426c42e9,0xb67d7f28),L_(0x497b724b)}},
   {{LL(0x96cc626f,0xd2b3305b),LL(0xab29e4cb,0x355d47d3),LL(0x33f53993,0x960f4207),L_(0xbf71ccbf)}, {LL(0x329378d4,0x1004c020),LL(0x86ffcabb,0xa9117cf0),LL(0x0e5dbd85,0x73f55c21),L_(0xe9de73fd)}},
   {{LL(0x456357ca,0x1b3ded00),LL(0x47adbb71,0x0f0f2d4f),LL(0x51648f45,0xab4e5da6),L_(0x4659c423)}, {LL(0xe96f0203,0xcb55e2cd),LL(0x74ebdf1a,0x7b3cc4d4),LL(0x0090498b,0xdfb4de74),L_(0xc69c1621)}},
   {{LL(0xba643590,0x757e6fbb),LL(0x65485c65,0x95336360),LL(0x50d4d37c,0xac96f5a7),L_(0x676d85ce)}, {LL(0x53008bb9,0x33869046),LL(0xf21c0791,0xfb1262f1),LL(0x1e3e57ec,0x407af3f6),L_(0x6c3c72ca)}},
   {{LL(0x9d6ad458,0x78e956e9),LL(0x46bcbf36,0xb94af545),LL(0x32977879,0xd20f35a3),L_(0x3ba745fd)}, {LL(0x9c65ff22,0x5c8f99f9),LL(0xac6bb14a,0x4ce9a00d),LL(0x307b3d8d,0x798d2cf9),L_(0x9ddb4cad)}},
   {{LL(0x3b8e6ef6,0xb6ca3273),LL(0x5ade36d2,0xe260b8d0),LL(0x9c66c2a4,0x05eb84cf),L_(0x884d04c5)}, {LL(0xbe1f3ed0,0x1e35adca),LL(0x4cff4e8c,0xa175edc2),LL(0x4681a0ae,0xc6715ff6),L_(0x506a8488)}},
   {{LL(0xe9c88637,0x32143c93),LL(0xdf482c05,0x434818e2),LL(0x25a929c9,0x4c806ec9),L_(0x1179b4ee)}, {LL(0xfe3e80ad,0xa0372526),LL(0xf64ea692,0x5279eb89),LL(0xee97419a,0x98dbe31d),L_(0xdfad8124)}},
   {{LL(0xe8bb79d4,0xa6d3938a),LL(0xbdea4fe1,0x0b2f88a1),LL(0xf279df50,0x59a54b2e),L_(0x343b8869)}, {LL(0x1390a7ad,0xc9c37e25),LL(0x9e03f761,0x0777586e),LL(0x3aa2516a,0x42cece95),L_(0x732cba8a)}},
   {{LL(0xa47e21c2,0x5f9875c7),LL(0x327db722,0x2ecbc86c),LL(0x4ac56066,0x5ea26fc2),L_(0xc69b2d5c)}, {LL(0x31e429d7,0x944c3736),LL(0x7cb051f9,0xcf2fc959),LL(0x02c3927e,0x3bb33cf9),L_(0x5a57d920)}},
   {{LL(0xcf56d754,0x64f98e4d),LL(0xfbe5a38c,0x1c7766cb),LL(0xb20cfe07,0xe351283a),L_(0xd825af7e)}, {LL(0xdc9dc466,0x3bee62a5),LL(0xe5b281f4,0xb2e1f644),LL(0xfe54699d,0x18ee6ea1),L_(0xe71cfdbc)}},
   {{LL(0xd2fe136e,0xe5f658a8),LL(0x39fa85ce,0xd83b5473),LL(0x94a1a0d2,0x586d7529),L_(0x208e5f0f)}, {LL(0x0748fda7,0xe98d9b7d),LL(0x869e47b2,0xf0def9e4),LL(0xc4e12cd5,0xf40402da),L_(0xe2e9c40e)}},
   {{LL(0x2956c763,0x257681d9),LL(0x9408aa0f,0x696e0261),LL(0xfd84ba5b,0x0675debf),L_(0xa7a000fc)}, {LL(0xef9b82a6,0xe38868a1),LL(0x8226a77b,0xcdcdf922),LL(0x292fb9c9,0x269d71d2),L_(0x587c3f9c)}},
   {{LL(0xb5fae58f,0x55622b3a),LL(0xb3258e77,0xff23a08c),LL(0x56c2b54e,0x866ba05b),L_(0xb8835413)}, {LL(0xa737dae6,0xdc5131fd),LL(0xbcb13de4,0xbb1176e5),LL(0x860b7b0d,0xf51c76d2),L_(0xfc88df27)}},
   {{LL(0x6bd647f3,0xe811642e),LL(0xf28850fd,0x0776f556),LL(0x2c9fb90d,0xb8cebf02),L_(0xa7ba32f0)}, {LL(0x643fb866,0xc4902b6c),LL(0x3a4ba84e,0x74aac273),LL(0xfd2878e7,0x1a76b1b7),L_(0x4e9ed0a9)}},
   {{LL(0xf609c0b0,0x0cfea42c),LL(0xc8c86640,0x7a72c3c6),LL(0xb9872dc7,0x1ae5bcf6),L_(0x4967c18a)}, {LL(0x9fe44849,0x2a90f4d7),LL(0x7c9836d4,0xdcdd6bf1),LL(0x32eb5ac2,0x939219ad),L_(0xa85aba1e)}},
   {{LL(0x267059b7,0x2669c8c4),LL(0x92b26067,0xb6ba656c),LL(0xc80a4182,0xa67b63ae),L_(0xdabf1310)}, {LL(0x7acd503c,0x98520b74),LL(0xa4e52b3f,0x0688ae94),LL(0x47fc66da,0xacf9786e),L_(0x53a32432)}},
   {{LL(0xde210bb4,0x74430a09),LL(0x341d2f54,0xf0e810c7),LL(0x85cefd4f,0xba1cac0a),L_(0x649747b6)}, {LL(0x35194ff2,0x6452e158),LL(0x3bf68483,0x8ed85e25),LL(0xd8535bc9,0x3bc32918),L_(0x1618b5a8)}},
   {{LL(0xd3467c73,0x3046a69b),LL(0x11b44c9b,0x996236b5),LL(0xf02f1b90,0xdbce524f),L_(0x2880b5ed)}, {LL(0xdcc369a7,0xa2eda832),LL(0xbc28c7a6,0xcb124909),LL(0x33444c56,0xb4685d00),L_(0xc588ee2f)}},
   {{LL(0x6fe5a182,0xa316d37d),LL(0x7b8d3ccd,0x75ec9a2a),LL(0x8b98fad0,0x0c5e4465),L_(0x96299b8e)}, {LL(0xbd855a8e,0x7661287b),LL(0x21ef677f,0x9b044bfa),LL(0x0f23ddbb,0x7abd77aa),L_(0x217ef151)}},
   {{LL(0xe7d89f59,0x490f55aa),LL(0xf4051055,0x3aa72313),LL(0xdea3a145,0x159c120c),L_(0x3a83cdd8)}, {LL(0x840bb947,0x1516983e),LL(0x3c23bffe,0x527c0f73),LL(0x0d4d39e1,0xe3846304),L_(0xa097f7b6)}},
   {{LL(0x56c53d60,0x8c3176ff),LL(0x95152224,0x26489625),LL(0x94c341e1,0xa5b14d8d),L_(0xe01267c9)}, {LL(0xb4f1f613,0x1a2f793c),LL(0xd1a0049e,0x600ebda3),LL(0x6ba99e2b,0xd1571ee0),L_(0xc34b7e2c)}},
   {{LL(0x122b663a,0x41a70d6d),LL(0x5102d2ac,0x50c3e831),LL(0x37bdeb4c,0x948ab46f),L_(0x69f45329)}, {LL(0x1a55a55b,0x3b20c1e6),LL(0x22ef8bd8,0x81e5763b),LL(0x4ddc462f,0xffa876e2),L_(0x8337e2e2)}},
   {{LL(0x62e4c32d,0x7f945839),LL(0x8947b09f,0xa1e798f7),LL(0xfe755aec,0x5524084e),L_(0x72eaa739)}, {LL(0x02c2a9e3,0xdc627b0e),LL(0xc75c26c5,0x7c827c47),LL(0xbcf0b634,0x677ad978),L_(0x31454aa7)}},
   {{LL(0x74af562d,0x8713ce0f),LL(0xf85d00e9,0x6951ea39),LL(0x23d2ab0f,0x761f3a78),L_(0x1c36b814)}, {LL(0x098fb6b0,0xeb48e791),LL(0x84522286,0x160c9294),LL(0xf1992cc6,0x2ed7be4f),L_(0x48acfd7d)}},
   {{LL(0x02ad20b2,0x30165084),LL(0x162d53f9,0xb8bee43f),LL(0x67757e6c,0x0eedd6d6),L_(0x00fc0927)}, {LL(0x5acf9124,0x8dee8798),LL(0xa5fd0b77,0xa4c99259),LL(0x9970bc21,0x54e339cf),L_(0x9e45d123)}},
   {{LL(0x7f335ce0,0x5019bc77),LL(0xcd9df5c6,0x15ce5f7c),LL(0x4f31c43e,0xfa69d995),L_(0xdaf4bdbc)}, {LL(0x1d57d52d,0x20dc7c66),LL(0x1ccb081b,0x01a82796),LL(0xa8fa2376,0x861ac30c),L_(0x519f6c6e)}},
   {{LL(0x56acbf12,0x7b20a0c7),LL(0x54051bfe,0xbec3d69c),LL(0x5e324e3c,0xd3d97b66),L_(0x5235011b)}, {LL(0xc585e141,0x2b8e18cf),LL(0x3540e46d,0xe0b2513f),LL(0x6e7d6c03,0x7e1a6116),L_(0x9b468e4d)}},
   {{LL(0x23bf2ce9,0x82122cee),LL(0xa723e8e2,0x318ba203),LL(0xa93e44d4,0xd2180b18),L_(0x643b190c)}, {LL(0xd9d85de9,0xd7d06345),LL(0x0751ebbc,0x5bea35e9),LL(0x952d933c,0x72bd41e8),L_(0x2910ff54)}},
   {{LL(0x60e9cb5f,0x107232fb),LL(0xd6c371b4,0x004f8683),LL(0xa23e4299,0xefa3c56b),L_(0x8d86e9ba)}, {LL(0xad436904,0xbb576a94),LL(0x9a66d1e1,0xbd78745f),LL(0x3513e3fe,0x4066ae12),L_(0x69b558b0)}},
   {{LL(0x627d1808,0x7365dc29),LL(0xe9b14b35,0x0dec4459),LL(0x96d5022f,0xf4343e09),L_(0xf65c2fb7)}, {LL(0x00d16dd7,0xeeaa6e35),LL(0xe87ecb3a,0x220f953d),LL(0x91ecc397,0xaf714cd0),L_(0xaea56471)}},
   {{LL(0x6dd11273,0xce83daf3),LL(0x0a835712,0x6ce5dec2),LL(0xce89d045,0xbafd94a5),L_(0x86e671a6)}, {LL(0x577ca0e1,0x1e36fd91),LL(0x6fe72f1f,0x060d396f),LL(0x729ba81d,0x8b7e4fbb),L_(0xea9c3f32)}},
   {{LL(0xdbaa1960,0xb43de333),LL(0x030ad3be,0x2ccf4a4e),LL(0x036ff9cf,0xafc236c4),L_(0xe987e57b)}, {LL(0xd7468cbc,0x9a645200),LL(0x61eddc02,0x5b2deae6),LL(0xb553572e,0xd4efa381),L_(0x9dde1eaa)}},
   {{LL(0xc9536b4b,0x10222aaa),LL(0x6d0e36c2,0x84934c38),LL(0xdf7ca8e3,0x5e5dc148),L_(0x2da86a73)}, {LL(0x4d98778d,0xb7c8140c),LL(0x69fc5568,0x69c09498),LL(0xb0ea1d36,0x7e0cd235),L_(0x6c99f36a)}},
   {{LL(0xe9443a73,0x94747e78),LL(0x377b44d6,0x2415da80),LL(0xf50cdd9d,0x65d2fa97),L_(0x9d194135)}, {LL(0x3ca31050,0xaf39be74),LL(0x06450aa6,0x0441383f),LL(0x393e3175,0x4e5bd1d4),L_(0xb7244573)}},
   {{LL(0x70145556,0xc463ec32),LL(0x867b11cf,0x1e5a98cb),LL(0x59b3adf7,0xeb7c80fc),L_(0x0929efbf)}, {LL(0x5234550c,0x4eb610a4),LL(0x7d606935,0xdc8db9ba),LL(0x52106057,0xbc3a065e),L_(0xad60a058)}},
   {{LL(0x6e360f56,0xee7f7e1c),LL(0x4a514c99,0x7411bd70),LL(0x627a5d54,0xb191903e),L_(0x9d4472f8)}, {LL(0x6bb808ee,0x484c498f),LL(0x110de6ea,0x3cef8b7c),LL(0xcc379eda,0xe1f0ee29),L_(0xcdc2d24b)}},
   {{LL(0x21a05590,0x2ea40699),LL(0x953235f9,0x057a8486),LL(0x5834fcc4,0xb35b630b),L_(0xef9321a5)}, {LL(0x2d1d0e73,0x6560d0d6),LL(0x4083b1e7,0xeb026a8b),LL(0x4d572129,0x720991c7),L_(0x98865061)}},
   {{LL(0x657c8db8,0x5e4f7ccd),LL(0x35de8d7a,0x8351fe6b),LL(0xa43443b2,0x48c1ec10),L_(0xdd11de81)}, {LL(0xb376d6e3,0xc77d6dfd),LL(0x73bed5bf,0x0f953512),LL(0xeae9d0a1,0x91a7a744),L_(0xfeb76333)}},
   {{LL(0x59454f28,0xd2c2004d),LL(0x9f5eebc7,0xacc344aa),LL(0x775a61d7,0x1c186106),L_(0x47a8ab1c)}, {LL(0xdcf00b61,0x4e3f081c),LL(0xbc56bbf8,0xcaa9cad2),LL(0x1e4e523b,0x816a09ca),L_(0x5e9a9445)}},
   {{LL(0xd468c097,0x86aa5ecf),LL(0x244131ce,0x26d08883),LL(0xe43dffa8,0x6501f291),L_(0xf4cfc564)}, {LL(0x88248cda,0x4711c711),LL(0xf7fb2e43,0xa4245343),LL(0xcfde7c18,0x4111cd00),L_(0x5a68a986)}},
   {{LL(0x0df3b4b1,0x9cf9b276),LL(0x71920b8d,0x9dd92d6a),LL(0x6444967a,0x5f3bbd8f),L_(0x82870cd4)}, {LL(0x1a4b4821,0x2d798278),LL(0xd460ee52,0xe9681e54),LL(0x0ce3d37c,0x739d5ad5),L_(0x5645ee72)}},
   {{LL(0xcf20f025,0xf4697953),LL(0x6debfc44,0x789e6af7),LL(0x1050c0c2,0x383c22fa),L_(0x22fd43b3)}, {LL(0x5cc2adcd,0x2ace7063),LL(0x55ea131f,0xf6f6aaf0),LL(0x31c60f20,0x7b350e28),L_(0x9f8df3bf)}},
   {{LL(0xf49e5f62,0x1a92eddc),LL(0x24db3514,0x365a5041),LL(0xc89ba67e,0xa32fc59f),L_(0x33efcf95)}, {LL(0xdb9db26c,0x09322a72),LL(0xf91f9ca4,0x3fda4216),LL(0xc6c63b34,0xddeb9986),L_(0x9a067f25)}},
   {{LL(0x77b280f2,0xc18c7395),LL(0x93723817,0x4cd03829),LL(0x37436f93,0x66528f8e),L_(0xeb73448c)}, {LL(0x2dd6704b,0x83b283eb),LL(0xedc8d2d1,0x54972ce7),LL(0xbca75a4c,0xa9f10506),L_(0x91b481a9)}},
   {{LL(0xee53ff2f,0x5105c866),LL(0x7f1e21b7,0x02b0042d),LL(0x1f9de3bb,0x4f9500fd),L_(0x89fa5ada)}, {LL(0x8cb6dd5d,0xaa04e960),LL(0xb2782218,0xa0112a6f),LL(0xec0e66ff,0xec305363),L_(0x77fc7f14)}},
   {{LL(0xa014524c,0x400fd142),LL(0x17086d36,0xf02c64c2),LL(0xd9684416,0x8da612c7),L_(0xb46c7e04)}, {LL(0x4e54713c,0x7cde46af),LL(0x0546cc9f,0x8380cc14),LL(0x5013084d,0x80963cc8),L_(0xf957f8d9)}},
   {{LL(0x6dcbbe49,0x6b6125b6),LL(0x038efe1a,0x67b840bc),LL(0x16c79367,0x570c02ad),L_(0xcb60af87)}, {LL(0x8552cefe,0xa8167ca5),LL(0x5e88a6fd,0x3636e864),LL(0x9c4305d9,0x43d5af7e),L_(0xc9e5c723)}},
   {{LL(0x413cd65a,0x9c95b00e),LL(0xb0a4ab3f,0x64a28801),LL(0x6ef01bfc,0x858e4b79),L_(0x80657f33)}, {LL(0x113a4fdc,0xd390a92a),LL(0x71a0cef2,0x774c1209),LL(0x63557ff4,0x54e47eaa),L_(0x4ea9a478)}},
   {{LL(0x27760c7e,0x9e3147b9),LL(0x656b24fd,0xb9bb9711),LL(0x89363a50,0xe852edfe),L_(0x6bf2556d)}, {LL(0x43ee10a6,0x2d6acef6),LL(0x3e9d7e57,0x03fc12e9),LL(0x20890385,0x2d45580b),L_(0x6b8bf1da)}},
   {{LL(0x4101d170,0xbebb2e76),LL(0x7fdf91ff,0x533638ee),LL(0x05bdd7f6,0x92a0bd39),L_(0x78bac139)}, {LL(0x4c760097,0x81c6243b),LL(0x8d71803c,0x5119eeba),LL(0x049c6a06,0xc410f246),L_(0xa9b7c2f7)}},
   {{LL(0x51167f9b,0x12be7d2f),LL(0x66cdc913,0xa41ce391),LL(0x0572301a,0x0a761ede),L_(0x2a186fed)}, {LL(0xe5dca065,0x3dad4d6c),LL(0x14733148,0x82d548d7),LL(0x79a33fd1,0x76d2d77b),L_(0x744a660e)}},
   {{LL(0x77f53015,0xa4c2daab),LL(0x54ae5f2a,0x40a7d787),LL(0x604bbbf3,0xcc41bd91),L_(0x7fe1506d)}, {LL(0x4c6757d5,0xe55da08f),LL(0x38212cd6,0xf50459ca),LL(0xf8d3a03d,0x8252e5cb),L_(0x62160548)}},
   {{LL(0xaaca1be2,0xb4f7cd9f),LL(0x3012b6d5,0xfae3835d),LL(0xf685dd47,0x6c5b897b),L_(0x3f52f5a3)}, {LL(0x03e5ac2f,0x71afc215),LL(0xdc13191e,0xce6c64cf),LL(0xa1bdc366,0xe59eacca),L_(0x719ce6ba)}},
   {{LL(0x28e18007,0x428bd7b1),LL(0xba236a3e,0x3bf0eb6c),LL(0x69ce21e0,0xfbe09c69),L_(0x321ac5e5)}, {LL(0xf7c3e583,0x3283bc61),LL(0xaf3a5b06,0x20ead3f8),LL(0x1f935f24,0x3b040b57),L_(0x39549c75)}},
   {{LL(0xa23c1b85,0x41a0017c),LL(0xc677b960,0xc09598e4),LL(0xffd68469,0x1da39b13),L_(0xdfff7bf9)}, {LL(0xb19fcd21,0xe71a0fa0),LL(0xc987c7de,0x83e812da),LL(0xb605b69d,0x0efdfc84),L_(0xdde3a792)}},
   {{LL(0x656d6dd1,0x58a3f7d2),LL(0x9e07158a,0x1396fe6b),LL(0x52fddb17,0x39edad21),L_(0xd739977f)}, {LL(0xfb57f56a,0x1072b4e0),LL(0x80c7df0a,0x94372dad),LL(0x0a26aff2,0x88f38fd7),L_(0x5d33e01b)}},
   {{LL(0x6d4895a4,0xff2438e1),LL(0xa9ba3f6a,0x67f22bee),LL(0x3cfaa48e,0x6bf9fc2b),L_(0x702030cd)}, {LL(0x7b4df30f,0x956e9ec2),LL(0xe0368ebf,0xe6cb0b1b),LL(0xfc98eaec,0x7dbf18a3),L_(0xf5fc22a9)}},
   {{LL(0xf20afcc9,0x9b4028ed),LL(0x2adb3986,0x4e476a55),LL(0x6bf05b58,0x2b0c488d),L_(0x74de42bf)}, {LL(0x01d3ca75,0xc0d6bd28),LL(0x668206d5,0x95f353c8),LL(0x010298ee,0x034e53ca),L_(0x13b81d0b)}},
   {{LL(0x80ff52bc,0x50412d11),LL(0x652fe1b8,0xbe2f9f7b),LL(0xe84a58ad,0x83284261),L_(0x905c06a9)}, {LL(0xc605b12c,0x5da82b44),LL(0x2ca210de,0x837f9c22),LL(0x10d656cc,0x454d1413),L_(0x5a580fa0)}},
   {{LL(0xde3d1dc5,0x6619a39c),LL(0xac6af7d7,0xaaa9723b),LL(0x3910ae67,0x86fdb715),L_(0x4d8f0cf8)}, {LL(0x3782c29b,0x5c35eb01),LL(0xe1483a4f,0x76b4be95),LL(0xdd570ea6,0xf56b5810),L_(0xefb949a7)}},
   {{LL(0x9ec12516,0x3f080f23),LL(0x108296b6,0xc3509a49),LL(0x08bba00b,0x07998613),L_(0xdc853801)}, {LL(0x5c36ecb3,0x75c6840a),LL(0x73946b3d,0x3e8f5265),LL(0x2c678e3c,0xc7f1cc73),L_(0x284107e6)}},
},
/* digit=31 base_pwr=2^217 */
{
   {{LL(0x2409728d,0x2d5fd352),LL(0x84897b53,0x28f3cb13),LL(0xee72bd56,0x21c2f494),L_(0xf66c03f3)}, {LL(0xac73118e,0x283e1b56),LL(0x3b39e360,0xf00c3802),LL(0xaa6c7b0a,0xee304f23),L_(0x0dd1d011)}},
   {{LL(0x5b89b52c,0x433d8d17),LL(0x440ff02f,0xec27bd07),LL(0x49353c25,0xf18b23f5),L_(0xd9e62ddc)}, {LL(0x2c798eed,0x91011438),LL(0x30fcf910,0xfaa77bac),LL(0x6e2db36a,0x2a3b730b),L_(0x3ada8eb1)}},
   {{LL(0xea09f155,0x18c36c89),LL(0x3ba3b853,0xff1cf2a0),LL(0x1d069cd7,0x672fde0f),L_(0x0a42b710)}, {LL(0xf1f4e5ba,0x86238146),LL(0xabca7963,0xcdc7fd6a),LL(0x00bcd404,0x51a105eb),L_(0xf842633a)}},
   {{LL(0x348659fb,0x0afc95de),LL(0x4b20153e,0x478c3432),LL(0x1f2f8d2f,0xaa0e8781),L_(0x27294e88)}, {LL(0xbf3811ff,0x526d1fb9),LL(0x9ad871cc,0xf802b4ad),LL(0xf613e178,0xd3df21bc),L_(0xb619f301)}},
   {{LL(0x40af417a,0x3b133f03),LL(0xb449508f,0x9b5dda1c),LL(0xc2f372b3,0x3948b7d7),L_(0xba404740)}, {LL(0x99028594,0xbcf7e27f),LL(0x96542b58,0x1de8aa5e),LL(0xb2fc7cd3,0xaadbe95a),L_(0xaa07786d)}},
   {{LL(0xff2c8bea,0x26721264),LL(0x5d2747dd,0xbd6c65e6),LL(0x7f2093a5,0x21e5d488),L_(0x2ff2524c)}, {LL(0xa000a3ed,0x68530945),LL(0xef75ea86,0xd1f6fc3b),LL(0x3f15aab2,0x9b5d8cbb),L_(0xfc18b474)}},
   {{LL(0x45f74206,0x97a4da7b),LL(0xcc1484f1,0x6d246546),LL(0x819c4b2d,0x8255bc16),L_(0x4605c7b7)}, {LL(0xf9d396e2,0x5502b719),LL(0xfde1aa97,0x8b9b6ca7),LL(0x94965c57,0x3c7a5ffc),L_(0xea93e0d6)}},
   {{LL(0xc11d6985,0xbad3d4ba),LL(0xb1adc951,0x67c5bea3),LL(0xb82cec9b,0x93b6322c),L_(0x8ae8a415)}, {LL(0x9661c047,0x769c204a),LL(0x31aa2f54,0x35880fdb),LL(0xc215f2cd,0x4f5bf77f),L_(0xe0f608d2)}},
   {{LL(0x8bce7ed0,0x847b9ee6),LL(0xb2892f4e,0x8cb1f02a),LL(0xfffe0edc,0x62c22364),L_(0xb0229884)}, {LL(0xdcd7d51e,0xa9cd4c50),LL(0x4688d647,0x30edb5dc),LL(0x82ea29e2,0xcc4403d4),L_(0xb78d8b6a)}},
   {{LL(0xed02ad3b,0x66e7f904),LL(0xdf2ace65,0x2877668a),LL(0xe7bcfa22,0x873070fa),L_(0x3a2466a0)}, {LL(0xf013615c,0x2fe9dfb1),LL(0xe228049a,0x51ff974b),LL(0x56961e83,0xcd87825f),L_(0x0c9c9408)}},
   {{LL(0x5b989bda,0x2f915297),LL(0xca9c644b,0xcf0387e6),LL(0x80e7dd9f,0x2e7c6dd7),L_(0x4b7f801a)}, {LL(0x861c2ffe,0x1df12d2f),LL(0xe8fd6f24,0x048668bb),LL(0x9b862a1d,0x2b64be2a),L_(0x4dde2a3c)}},
   {{LL(0x644173f8,0xc3611806),LL(0x23a43a5b,0x78a6d7ad),LL(0xe22b26dd,0x1dd89345),L_(0xfd5f2c34)}, {LL(0xd1587983,0x7692c1c8),LL(0xca1f4a95,0xecf6b1c3),LL(0x0a975fb0,0x0d6babd5),L_(0xc6e6c5b9)}},
   {{LL(0x4222b0ed,0xb5c91a5b),LL(0x6d847e8b,0x0ec3e064),LL(0x9d0d2af2,0x660e553e),L_(0x0aaf0e60)}, {LL(0xa6329190,0xff1f387c),LL(0x9970f312,0xd795c508),LL(0x67e40e1b,0xdada8e3d),L_(0xe265b2ea)}},
   {{LL(0x6eb26317,0x4e1593ef),LL(0x12da5305,0x8698f986),LL(0x2d0a715c,0x8e8b3499),L_(0x80e2d2d8)}, {LL(0x0b43672b,0x132c1c9a),LL(0xee8ab3bd,0xfd973660),LL(0xea97e0b3,0x3b811a56),L_(0xbeaaf50d)}},
   {{LL(0x3edb12e0,0xf66872e6),LL(0xb9ab7a65,0xc41cad86),LL(0x0cfa43fd,0x43f4aa49),L_(0x781ca102)}, {LL(0xcfa7ab7b,0xafd31516),LL(0x061e0da2,0xc78d7ba1),LL(0xea786f69,0x2090d8fa),L_(0xdf64f185)}},
   {{LL(0x7ee50090,0x28c86852),LL(0xff58b162,0xd4576859),LL(0xc2987502,0xe596765e),L_(0xf8c4c45c)}, {LL(0xfb631b85,0x42aaadd8),LL(0xb6c6bfcd,0xf7d72923),LL(0x2257f5b6,0x260ae885),L_(0x2f0626ed)}},
   {{LL(0xdb84c9ba,0xdbb13e44),LL(0x7c40add1,0xa7294693),LL(0xc1443c26,0xe02af6c2),L_(0x935cf64d)}, {LL(0x293f667b,0x52bbc8cd),LL(0x5d49c23c,0x7e1de72e),LL(0x6907cdc1,0xf8a9aeab),L_(0x01bb3a0e)}},
   {{LL(0x3fade183,0xd7401dde),LL(0xa78758f9,0xc85449b6),LL(0xc47549c9,0x6ec51217),L_(0x5c22aa19)}, {LL(0xae146939,0x4bbe9886),LL(0xd2682769,0xee34940f),LL(0x9f320ada,0x4fdf9e41),L_(0xaa18dddb)}},
   {{LL(0xf506f84a,0xf58881e9),LL(0xe2a44aae,0xd6feb621),LL(0xedfd3c65,0xb163b10b),L_(0xd0eb9a35)}, {LL(0xb8f8a7bc,0xb10b02be),LL(0x2524cc98,0x1e5816c9),LL(0x0ad84066,0x9cd4eb66),L_(0xecbabab5)}},
   {{LL(0x413431bc,0x9ea4aaec),LL(0x217b55d3,0x135f5f2b),LL(0x9e4519f3,0xd3e6e8f7),L_(0xb072b798)}, {LL(0xfa8179f9,0x318fb330),LL(0xcb83a9f9,0xe0cb106a),LL(0x95e1ca87,0xfb4ce48c),L_(0x87fc571d)}},
   {{LL(0xf3d74e55,0xb7fb663d),LL(0x3bebc7e0,0x6706a48f),LL(0x7f8a7cfb,0x52d7353d),L_(0x13ea8891)}, {LL(0xc42a9ef4,0x5bd192eb),LL(0xdfeb9aa7,0x2b1ff6c0),LL(0xaa53702a,0x5388fa6f),L_(0x7ee98651)}},
   {{LL(0x423e53a4,0x0baae890),LL(0x6536911d,0x08234190),LL(0x9bfec6fd,0xc8f4d373),L_(0xd7ec5802)}, {LL(0x119e3477,0x19587b51),LL(0xfac91771,0x1c276e15),LL(0x951dd485,0x98048932),L_(0x04243a3f)}},
   {{LL(0xbe2d77ea,0x67d742de),LL(0x2df6c417,0x7983fd12),LL(0x7115e508,0xb9406bb8),L_(0xde1f9fde)}, {LL(0xe45eb298,0xdf3a7991),LL(0xa54dc400,0x03ba9fe4),LL(0x25d8f862,0x6ea3053f),L_(0xb145e5fc)}},
   {{LL(0xefbe9106,0x266e3900),LL(0xb0d74420,0xe042e34e),LL(0xbec6b919,0x5cb15a6f),L_(0x6c16d3f4)}, {LL(0x75686740,0xfe7b7595),LL(0x2204cb17,0x3e67572e),LL(0x2d437dfd,0x03299218),L_(0x2be5d786)}},
   {{LL(0xc2d6c314,0x3d460901),LL(0x3d26026c,0x591fe51a),LL(0xeb85aa17,0x44d65538),L_(0x015a9b39)}, {LL(0xea596f91,0x48a2f0df),LL(0x8ce8790a,0xc2cabfcf),LL(0x17f7dae8,0x31a2b7ff),L_(0xba200e74)}},
   {{LL(0xb389e985,0xb19669c8),LL(0xa23e72e7,0x729a4108),LL(0x53062f9c,0x779cbefe),L_(0xde2e49fe)}, {LL(0xa29b29cb,0x6119b4c5),LL(0x0f5a398b,0xeb10107e),LL(0x91c7a000,0xd43d23aa),L_(0xc42c4a03)}},
   {{LL(0xb2465ace,0x954d5e2a),LL(0x361566cb,0xcc9f7e66),LL(0xa5432c5d,0xd4ffbeb9),L_(0x5d478258)}, {LL(0x686dc22e,0xb3aa6ec4),LL(0x29d8cc0b,0xf9a59b06),LL(0xd8d7140f,0x94e1344f),L_(0xfd56ad11)}},
   {{LL(0x7dd1f80e,0x241f38ad),LL(0xc4ce241f,0x72ffb97f),LL(0xff97f61b,0xa3d97bc2),L_(0x9c42f7da)}, {LL(0xd4e6e73b,0xb4b19841),LL(0x1c3d7005,0x5fd346b8),LL(0xa14d6ec3,0x366b72b2),L_(0xd6218402)}},
   {{LL(0xb6fe5b76,0xe3a6e8d3),LL(0x6941f5ec,0x4ed1cc1b),LL(0x9f69a8a4,0xe6f9dbe8),L_(0x8ff4ee20)}, {LL(0xa4d858f0,0x00a65edd),LL(0x30c9c798,0x9733fc57),LL(0xd4ab5744,0x38f910bb),L_(0x08dd1912)}},
   {{LL(0x338c1ad5,0x4a664d78),LL(0x90d8cb53,0xd6ec4c42),LL(0xaee2829b,0xededd826),L_(0x0c5337aa)}, {LL(0x417e0806,0x2c83b6ff),LL(0xd8bc600c,0x65e85629),LL(0x54ffde80,0x8972a5fc),L_(0x465b9ae3)}},
   {{LL(0x49c2561b,0xd2b48b48),LL(0x6937dcf3,0x38241d90),LL(0x7dc5afe2,0xc399df03),L_(0x9c6cdf52)}, {LL(0xc98d8a84,0x666e5386),LL(0xb9102fbb,0x33895284),LL(0xb43f7afa,0xb804c149),L_(0xfe016452)}},
   {{LL(0xd0681519,0x93423c26),LL(0xa105d538,0xc2eabb36),LL(0xea12b99c,0x9297a177),L_(0xb26a7f93)}, {LL(0xddfee3a3,0x4de65e15),LL(0x9a81b364,0x7f3bad80),LL(0xe43ae1fc,0x7af9ba54),L_(0x86e1e8dd)}},
   {{LL(0x13762e74,0x5afe2e21),LL(0x8fe945bd,0xa21b9a1a),LL(0xd03deaed,0xd498e9f2),L_(0xe67e5384)}, {LL(0x6ae69ee9,0x749eed42),LL(0xdd52d330,0x5f4c63de),LL(0x95aa2f5a,0x890e5e3e),L_(0x3772d05c)}},
   {{LL(0xdea7e04a,0xbf3bb703),LL(0x449cf48c,0xc64a209c),LL(0xddb56124,0x6858318b),L_(0x25ad4355)}, {LL(0x9e3a0204,0xf0dbd02b),LL(0xa3421c63,0xa216442b),LL(0x5ebba836,0xc0cbf7a0),L_(0x4f9c0d9f)}},
   {{LL(0x9bdf3b82,0xe5bfa2bd),LL(0x0e12030e,0xef3da843),LL(0x7b3913de,0x107bba5b),L_(0x390841ba)}, {LL(0xda5362ce,0x81eecff8),LL(0x2734c216,0x151060a4),LL(0xd9ea79af,0xe89b5db6),L_(0x2aafd537)}},
   {{LL(0x310d44fd,0x4509feca),LL(0xa4868c10,0x25aa7253),LL(0xd40c4268,0xa958ef77),L_(0xe8930722)}, {LL(0x5dae221c,0x71fc93e9),LL(0x0095e400,0xc3e3cb95),LL(0x0f45640e,0x6b7cb33b),L_(0xb3ea06da)}},
   {{LL(0xc74fe8dd,0x8c9ef406),LL(0xaa86860f,0xb6fe758b),LL(0x04aa9cd6,0x4f8b3877),L_(0xdfcc0451)}, {LL(0x9f568192,0x65aed657),LL(0x44ebcd65,0x85d0ebdf),LL(0x7968220e,0x04b5f3d4),L_(0x44142172)}},
   {{LL(0x275a3076,0x3a2a339d),LL(0x88744d58,0x8a7f68a4),LL(0x5e03cf06,0xd663fea8),L_(0xdffb3ac6)}, {LL(0x83dd1969,0x6151157a),LL(0x18fce4cb,0x620979ba),LL(0x6d82e1aa,0x3ac652d1),L_(0xa3d0c2ce)}},
   {{LL(0x9f8e323e,0xed2fa4ec),LL(0xf62c31ca,0x554cf5e8),LL(0x14820bd1,0xc72e2e81),L_(0xb42dc3c7)}, {LL(0xe15fa1be,0x5f0c938d),LL(0x0920a90b,0xdf2d997a),LL(0x33617c0a,0x3087e812),L_(0x631962db)}},
   {{LL(0x7f81aff6,0xd6ea063d),LL(0x1c54d79e,0x631cf97e),LL(0x46b66ca3,0x3bfc14a4),L_(0xa20e2ff9)}, {LL(0x3a379a19,0xaa5dfd05),LL(0x5f4d0392,0xf3af9e8f),LL(0x95b8ff29,0x928c3c53),L_(0x28ae39a9)}},
   {{LL(0xe65a2e0d,0xa11c784f),LL(0xb5192f99,0xd33e59b1),LL(0x2f4c50c4,0xe15fe3e7),L_(0x30149c86)}, {LL(0xe61d0a03,0xdff24b95),LL(0x2ec9f885,0xd0e84cbb),LL(0xc9c516d5,0xbc6f5dd3),L_(0x71fb6b1e)}},
   {{LL(0x828d848a,0x55c5a959),LL(0x503ffb7b,0xed8f7fcb),LL(0x2be30973,0x0b24dc92),L_(0x90b2a59f)}, {LL(0xc98877f6,0xd599613c),LL(0xe60dd939,0x43a0bc0a),LL(0x34b1b90a,0xa8edd1ca),L_(0x7ca529dd)}},
   {{LL(0x126c6d32,0xb92a7bd7),LL(0xda613cbe,0xd9952b27),LL(0xb5f3accf,0xd21fa7b1),L_(0xda44ff77)}, {LL(0x6ac1dedb,0x7d8ef38b),LL(0xc4ea5e03,0x26d207ab),LL(0xc01bf44b,0x58c8cee8),L_(0xb8f1a2b5)}},
   {{LL(0x5d2a899c,0x94d31b36),LL(0xbc559f35,0x3050256c),LL(0xd6d74149,0xaa4efe0c),L_(0xbc1f929b)}, {LL(0x9b7a733f,0x491dc4d2),LL(0x50e69c58,0x7540af9a),LL(0x358ed8d5,0x17dc58f7),L_(0x5a66b29f)}},
   {{LL(0xb13a4f20,0x4d20428f),LL(0xd768684c,0xf17077d6),LL(0x27509ac6,0x58852489),L_(0xd30f66cb)}, {LL(0x64ad83d0,0xc965f9f3),LL(0xca2684af,0x9a3903b0),LL(0x2677b45a,0x45839bea),L_(0x2155a76d)}},
   {{LL(0x6c1eb10d,0x24b30b08),LL(0x4b90dabd,0x44f85d03),LL(0x688bbe3b,0xc74ccccb),L_(0xe4b5c103)}, {LL(0x94d1ce03,0x14d9a43c),LL(0x5f767150,0xc940698c),LL(0xf3a40155,0xefe2f127),L_(0xbf13e662)}},
   {{LL(0x9378bdc9,0xfbd86f1e),LL(0x50cbdd4e,0x38c7a9cc),LL(0x965226a4,0x434dbb0c),L_(0x83128949)}, {LL(0xb9259729,0xc36bf7c6),LL(0x038090c2,0x0a12f12b),LL(0xb21834ca,0x0134eeee),L_(0xe2621ff2)}},
   {{LL(0xad55e743,0xa30d570d),LL(0xfb77c395,0xdaaa178b),LL(0x525f0059,0xf38af4e6),L_(0xd340b6fe)}, {LL(0xa001bba0,0xa05f60e2),LL(0x855efc1b,0x701418ea),LL(0xfd38f820,0x27e433c5),L_(0x0fbe1529)}},
   {{LL(0xcf1f6418,0x3570870f),LL(0x7c6202e0,0x85875e04),LL(0x96e8ab61,0xefc6f006),L_(0xb0d48b1f)}, {LL(0x0206e9ab,0x6f6540da),LL(0x32b7b29f,0x2a1b80df),LL(0x9caa872e,0x895193a8),L_(0xec63127d)}},
   {{LL(0x40d4de6d,0x288f99a9),LL(0x6835f630,0x74a2cf89),LL(0xad2e6432,0x52ddf73d),L_(0x4185c6dc)}, {LL(0xef978613,0xbb41a387),LL(0x95245fee,0x319f4bf6),LL(0x40e92907,0x9884b224),L_(0xaf8e6385)}},
   {{LL(0xe2fde7bd,0x282179db),LL(0x7db516ce,0x7897f73f),LL(0x14a26c02,0x07ab717c),L_(0x34d1cdd0)}, {LL(0xb3e012de,0xa65b35a3),LL(0xbb8667e0,0xb888ddc9),LL(0xc0879433,0x8819e222),L_(0xf68b3f79)}},
   {{LL(0x32acb173,0x9f902cc1),LL(0xda89f30b,0x2ea4f418),LL(0xec378b71,0xd036aa23),L_(0x56712f20)}, {LL(0x0f8fe475,0x3e16c057),LL(0xe703ddb6,0x134bd529),LL(0xee28b685,0x867d1b08),L_(0x0383ba62)}},
   {{LL(0x49cdbaea,0x5c1f02d0),LL(0xa1775d0a,0xd83d7d9d),LL(0x1b0a0f31,0x2c287cd4),L_(0x80addda6)}, {LL(0xf27c6df8,0x74534873),LL(0x5db066fa,0x5a696a14),LL(0x8118c158,0xfc06292e),L_(0x58371c8a)}},
   {{LL(0x5e909157,0xbd1a69f7),LL(0xb7d52793,0x703cb9bd),LL(0x5a474912,0xcedd645a),L_(0x56b4d076)}, {LL(0xdeeac8ac,0x2ed53772),LL(0x89595a18,0x5e6b86ab),LL(0xc7b731bc,0x5ce1f8a0),L_(0x03aec3d0)}},
   {{LL(0xcc023bb6,0xdf3b4492),LL(0x796e6974,0x87e45b7d),LL(0x0ae46dae,0xe1f7eac7),L_(0x5971c51a)}, {LL(0xb2f3733b,0x72718cd2),LL(0x3b6f5116,0x85e9b624),LL(0x626b0871,0xc20aa08f),L_(0x1ce0f83a)}},
   {{LL(0x44129031,0x8b7e591b),LL(0x5325744a,0xc7753dac),LL(0xf476b49f,0x2fc4d6c3),L_(0xcc7c16ef)}, {LL(0xe26efb19,0x4dd384fc),LL(0xf67b422f,0x478d3760),LL(0xc574295c,0x421851ef),L_(0xbccaa4b1)}},
   {{LL(0x92253e02,0x5a61d65d),LL(0x6f301322,0xd4924563),LL(0x1e712886,0x2dd1532f),L_(0x3a89196d)}, {LL(0xa3f1deda,0x39cdfbfc),LL(0x039de172,0xb32cde92),LL(0x57960c59,0x40ad0b76),L_(0xafd7ab04)}},
   {{LL(0x33392ce0,0x52977c85),LL(0xacb473b0,0x147af936),LL(0x6dd913b5,0x4ec4b1c5),L_(0x838148b6)}, {LL(0x699ab1e9,0x438d69df),LL(0xa416d1fe,0xdf94b69f),LL(0xbc60bb02,0x60dae203),L_(0xc32d87da)}},
   {{LL(0x01cf54f4,0xc3cf28be),LL(0x609058a8,0xfa448ad2),LL(0xf8c1cf70,0x59baadb3),L_(0x67dc813f)}, {LL(0xb0a595fb,0x60b75e6c),LL(0x1cfccbb7,0xb92101cb),LL(0x99833d0b,0x4be8d444),L_(0x91f3954c)}},
   {{LL(0xedd8aca0,0xcb571bbf),LL(0x7257aad6,0x5a13acc6),LL(0x68203714,0x79914deb),L_(0x895f9c20)}, {LL(0x5c3bbaec,0x211e58cd),LL(0xe9a0fbc0,0xdef6ac5d),LL(0x776cb363,0x93cb4804),L_(0x9678da2d)}},
   {{LL(0x7d49d5cc,0x55e82125),LL(0xe0e89e2e,0xa13cee1f),LL(0x7e46156a,0x7d7e5719),L_(0xd361d2e2)}, {LL(0x629c9b30,0x5fd28ac1),LL(0x70a24aea,0xdc393f34),LL(0x8ea6f6e3,0x289c68b1),L_(0xd6c4acd1)}},
   {{LL(0x63596f58,0x31f49f27),LL(0xac40021b,0x7c86680a),LL(0x0b1f95fd,0x02fc0e10),L_(0x0d61fe85)}, {LL(0x8f2843ce,0xca4cab38),LL(0xb53751a2,0xace069ce),LL(0xf56c80ff,0x270d9677),L_(0x9e5b52ac)}},
   {{LL(0x6352c6f6,0x67e09224),LL(0x626c5d2d,0x5be06f9a),LL(0x14efc35d,0x38c6a0a2),L_(0x7169645d)}, {LL(0xbe2d9aa4,0x6d0b11a9),LL(0x74e4964b,0x6c28404e),LL(0x5538c4c6,0x0af532d8),L_(0xfd9cc6b6)}},
   {{LL(0x588f5f01,0xb4277028),LL(0xa788dcdd,0x2477f29d),LL(0x5fe5d31d,0x68c2db2a),L_(0xd8503d86)}, {LL(0x2a5ebb9a,0x7b9c3f49),LL(0xc4b2edb7,0xbf734578),LL(0xbf266e3b,0xfe65a7a9),L_(0xc3945496)}},
},
/* digit=32 base_pwr=2^224 */
{
   {{LL(0xb7478767,0xeaa013ed),LL(0xa2868062,0x1cf3e561),LL(0xe8c99b44,0x142c4251),L_(0xc6f0cb81)}, {LL(0x5ac149fc,0x9c824754),LL(0x8ff9a766,0x6b1233b1),LL(0xb9c09eeb,0xc5afba96),L_(0x95223aa6)}},
   {{LL(0xb7c40cce,0xbf4d6687),LL(0xc187f375,0x88025f51),LL(0x80c953d4,0xe3dc6d83),L_(0x88d2d1bd)}, {LL(0xb7668ae3,0x50faab59),LL(0x4404c1ec,0x235fc5b5),LL(0xac299c46,0x1378e170),L_(0x5d9a4686)}},
   {{LL(0xb43ccb68,0x47792f74),LL(0xf564eef4,0x3dbeda1b),LL(0xb688499b,0x5b9b5ea1),L_(0x1e999e2b)}, {LL(0x6f26a9f7,0xc4515542),LL(0xb3ca4e2f,0x8e4ebca0),LL(0x1f557c40,0xd6441259),L_(0x913b210d)}},
   {{LL(0xea6ea3ca,0xc6c1108f),LL(0x564f45ee,0xcc83bf8e),LL(0x2c30408f,0x7aa17b0c),L_(0x5db1baaa)}, {LL(0x06aed6b0,0xc003650d),LL(0x47b5af33,0x72d853fe),LL(0x66238f2e,0xf46a15a7),L_(0x76de62dc)}},
   {{LL(0xf16a11ad,0xccc9dc6a),LL(0xb7c36e6e,0x71ab7d63),LL(0x54c72750,0x5a001c63),L_(0x0f8a90df)}, {LL(0x226cb518,0xb6f26d84),LL(0xd8c32b73,0x3fc36946),LL(0x9f9ee30a,0xa1549b9d),L_(0x3080e1f3)}},
   {{LL(0x4a2b2247,0xa65e1e3b),LL(0x9b2f8a35,0x671cf2f3),LL(0x6b9d08f0,0x162efd24),L_(0x62ae0f9c)}, {LL(0x3316fc3f,0x84ae1987),LL(0x25516058,0x7d0dc886),LL(0xdbe094f5,0xc7b0730d),L_(0xb05c94c3)}},
   {{LL(0x1bb62b06,0xe8e8411e),LL(0x20b9ca92,0xd192f8c2),LL(0xe45573f0,0xc1f3a5f2),L_(0x4f18bc0e)}, {LL(0x26419175,0x6c1325f6),LL(0xe78ee486,0x2f81e000),LL(0xa91d228e,0x0ff7a9b2),L_(0x694a50a7)}},
   {{LL(0x88e39d86,0x42262241),LL(0xe249830c,0x3ad07add),LL(0x918fdfa4,0xcb771a46),L_(0xe581c444)}, {LL(0x18ba36b8,0xeb29d84d),LL(0x12fb54ee,0x1383d57f),LL(0xf57833f6,0xfda5e179),L_(0x2c61f244)}},
   {{LL(0x47af468a,0x6f69c44b),LL(0xeb264605,0x5e4d4f9f),LL(0xd6821d55,0x46eb8384),L_(0xfdb05fbe)}, {LL(0xcdeb889e,0x871d5809),LL(0xb09553d3,0x1029a1da),LL(0x5cf9245d,0x1bcc2546),L_(0x3a614385)}},
   {{LL(0xfd8a4d1f,0xdb1de26b),LL(0xb08efe49,0x8aa6deaf),LL(0x12fee6da,0x1e010832),L_(0xb6e509a8)}, {LL(0xd9702488,0x9e928c81),LL(0xf87b74dc,0x79014b39),LL(0x844adfba,0xa7623853),L_(0x14e2902c)}},
   {{LL(0x2e0124e7,0x2b6dbd37),LL(0x0ff6ba21,0xcb4fc42b),LL(0xd7cdda0b,0xd5d8f543),L_(0x9ccc5b69)}, {LL(0x485584af,0xdd6eb7f3),LL(0xbc744924,0x2e41e3f7),LL(0x65859cc9,0xeca30c9e),L_(0xe5e2bdd9)}},
   {{LL(0xd5f8805e,0x66979e6c),LL(0xb1900794,0x1b2ea5ec),LL(0xf8ebb830,0x416a9eb2),L_(0x10b371a3)}, {LL(0x6e8ad20d,0x15ff6e9d),LL(0x000e2330,0x0865593f),LL(0xee384b9b,0xe0eeda2c),L_(0x783dd95d)}},
   {{LL(0x642f55b0,0xbfeb8487),LL(0xd5f0f813,0xfa3ad248),LL(0x7145f39b,0xd0f1abf1),L_(0xeb71b58b)}, {LL(0x4defbbd6,0x893a0505),LL(0x52984b68,0xf3e2f26c),LL(0xf54c27a7,0x6f12b38f),L_(0xbfac39e5)}},
   {{LL(0x5d26514b,0x7cad1aa3),LL(0xdd0afc7e,0x74faf86c),LL(0xfeab4314,0x645cca78),L_(0x5e53697c)}, {LL(0x43f00056,0xb72096cb),LL(0xca87208e,0x7a1950ec),LL(0x071d954a,0x5337fbf2),L_(0xc371272c)}},
   {{LL(0xb32000f7,0x5f6dea95),LL(0x5afebd30,0xf4446576),LL(0x19091ccf,0x74118cb1),L_(0xccebf405)}, {LL(0x9210256e,0x9aacef2c),LL(0xb3458ddc,0x1ca9b6d7),LL(0xb1ab02c7,0xa870996c),L_(0x89a1b94b)}},
   {{LL(0x308cbf80,0x552396fc),LL(0xab809f1d,0x7b2fda39),LL(0x4146f800,0xaebf97a3),L_(0xc90e51fe)}, {LL(0x61fb1317,0xf1c9012c),LL(0x3d74201c,0x0142bd07),LL(0x9b191445,0xbab9da1b),L_(0x4bd6ed8b)}},
   {{LL(0x5e508e4b,0xfa8ca29d),LL(0x2d86d22b,0x27a0f674),LL(0x4cfdd6de,0xee0206bb),L_(0x1e0e0bda)}, {LL(0x978ed51f,0xa6bbc41f),LL(0xb73521ec,0xe52c7866),LL(0x006c7072,0x32478f51),L_(0xb4b93178)}},
   {{LL(0x217f1203,0x04851e7d),LL(0x8feb118b,0xd7767c08),LL(0x75b3d632,0x4c08238d),L_(0xac7d86c6)}, {LL(0x3858c44f,0xcd97e8ec),LL(0xb74f0fe2,0x62318c46),LL(0x93355b9b,0xe2405794),L_(0x54c12d79)}},
   {{LL(0xde835c5e,0x92a463c3),LL(0x0228dc25,0x3b10b96a),LL(0x4b4acd48,0x5f5bd93b),L_(0x1c91a329)}, {LL(0xf537b956,0xedc9c8f4),LL(0xc7fcc358,0xf2362c8d),LL(0xe58a0000,0x970cf697),L_(0x2b1ee7ed)}},
   {{LL(0x8e7b172a,0x882e0b3e),LL(0xb8bd5ea9,0xbf2d6013),LL(0x490fbccd,0xc0848551),L_(0x3ffa1cc7)}, {LL(0x73f7a7d9,0x17b850e8),LL(0x3a53ec82,0xfcc68ded),LL(0x9cbe988b,0x8d09bdd2),L_(0xa3b6e95e)}},
   {{LL(0x6266d7aa,0x7f0f6292),LL(0x329716ed,0x47c40355),LL(0x1cfc61e3,0x7a59e7d3),L_(0x0ffa8f3d)}, {LL(0xbddfbabc,0x84b7a613),LL(0x36121ac1,0x89396fd0),LL(0x0f5147ce,0x2b7f355a),L_(0xb3193eff)}},
   {{LL(0x8af99520,0x4d2ceb91),LL(0x15dc69a8,0xcc451e87),LL(0x6d59d3b5,0xc116fdef),L_(0x627d226f)}, {LL(0x79ead74e,0x813a5852),LL(0x5daf904b,0xc12ffcbc),LL(0xcd67731a,0x3fdf17e1),L_(0x2df27ce8)}},
   {{LL(0x764582de,0xefaccc05),LL(0x640388d0,0x63018ab5),LL(0x6f606211,0x6b706f5e),L_(0xa1c500d6)}, {LL(0x26635840,0x2a01814c),LL(0x8ab7ad6f,0xe29eede6),LL(0x756a8f01,0x10f5f095),L_(0x53630671)}},
   {{LL(0xa38d34fa,0x591f8fe0),LL(0xf829c6c6,0xc31b8774),LL(0xae2b4a7d,0x55de9380),L_(0xc9c227e7)}, {LL(0x63877738,0x566cf0c3),LL(0xd9750372,0x1ad0462b),LL(0x5c1a2a83,0xcfd11aad),L_(0x2aff32de)}},
   {{LL(0xbc66bc8f,0x9f39feb3),LL(0x24b329eb,0x21f0dec8),LL(0xd2ff18e6,0x99e3e6c4),L_(0xb398626b)}, {LL(0xcce55b7f,0xf3fabc3c),LL(0xb9a016fd,0x5c869423),LL(0x4f7a9f81,0xc660df04),L_(0xf74d1142)}},
   {{LL(0x302b25d7,0xfbdef76c),LL(0xb2090c70,0x97f6ff80),LL(0xcc0decf0,0x26e1f975),L_(0x2797ff76)}, {LL(0x4f37e255,0x61eee678),LL(0x51e68a0e,0x6a0d8fc5),LL(0xd84014df,0xb7e3a2d4),L_(0x449bf740)}},
   {{LL(0xde3d73e8,0x3178216e),LL(0x90f08891,0x2f89efcd),LL(0xdcbe0246,0xb937b3c2),L_(0x557e9fd5)}, {LL(0xacaa30d0,0x7f06975e),LL(0xd99696a9,0x714b6101),LL(0xd23f9579,0xc1321142),L_(0x8d2afa4b)}},
   {{LL(0x8271dfd1,0x6e2e57bd),LL(0xc1ea81b3,0xec736ab8),LL(0xbcfb4116,0x1cc014d1),L_(0xc3f8311f)}, {LL(0x02178016,0xbcc0c7bd),LL(0x3222341c,0xdf2643e7),LL(0xedc952bb,0x36ab26fe),L_(0xa84ff692)}},
   {{LL(0xc562a325,0xccd2af21),LL(0x1f595cc1,0x8cbd4ea2),LL(0x9f800082,0x7bbad5e0),L_(0x89fbecab)}, {LL(0xbfd65f55,0xfa87a290),LL(0x24ee0ca1,0xd25d631b),LL(0xe063926d,0x0f3d4539),L_(0xac0263f9)}},
   {{LL(0x6bf5b025,0x1a63c568),LL(0xbec29ce6,0x7bad1c42),LL(0x62df4304,0x38b54b11),L_(0x2b705d62)}, {LL(0x94995153,0x7f7b64f5),LL(0x35d2b40a,0xd25d1128),LL(0xbdbbd673,0x4703bb51),L_(0x682e3058)}},
   {{LL(0x2509ad1b,0x9885b7f4),LL(0xc8c1117b,0xf70447a6),LL(0x9bfc053a,0x1b77852c),L_(0xcfb5f714)}, {LL(0x7dc9ba15,0xa43807b7),LL(0xaea633e3,0x02c669a5),LL(0x220e1871,0x991d8093),L_(0x508f55e1)}},
   {{LL(0xc434b864,0x6c74aee9),LL(0x96d76b48,0x072efeda),LL(0x19982c91,0xac91d19f),L_(0x79df6582)}, {LL(0xbc64a3c7,0xcea49a1b),LL(0x6b3b3cac,0xe25c1282),LL(0x37f9ab3c,0x57709b27),L_(0x404ec7c9)}},
   {{LL(0x63a0bab8,0xbdede240),LL(0x702a4776,0x8ec0a030),LL(0xbbb0cddc,0xf6b96225),L_(0xbdd56c73)}, {LL(0x3dff08cb,0xf46b51e6),LL(0xac5d2e41,0xfac7173a),LL(0x3f2c5530,0xe5090eda),L_(0x05b966b5)}},
   {{LL(0x585b98ee,0x4cb0c03a),LL(0x859d7492,0xb6dc9a2c),LL(0x126ee071,0x3331172f),L_(0xa1e73b0d)}, {LL(0xba3aa73a,0x71ac411d),LL(0x6d7dee6b,0x816a4f1a),LL(0xa88c4198,0xe409dbda),L_(0xa924723a)}},
   {{LL(0xccb6c49e,0xa20ba7c3),LL(0x46da89e3,0x6782fea2),LL(0x83b7b5ce,0xf28c836e),L_(0xa02bd753)}, {LL(0x64f82fd2,0x523afba8),LL(0x3203549e,0x024ddda9),LL(0x65d29391,0x4cec16d6),L_(0xb0187244)}},
   {{LL(0xaff84dda,0x1594f62b),LL(0x4bbabfdf,0xa8de453c),LL(0xeb65c1b7,0x30a103ff),L_(0x02fb469c)}, {LL(0xf76afad8,0xa91cd374),LL(0x98a39a41,0xa33359fd),LL(0x0b99c0c0,0x9015867f),L_(0x96fc381a)}},
   {{LL(0xb366fcc8,0xdde98289),LL(0x76693b42,0xe45737f9),LL(0x30c2ee70,0xa6bb364a),L_(0x759a9e66)}, {LL(0xe00c866b,0x8177a004),LL(0x144b5ff3,0x49205953),LL(0x0106e5ac,0x5d4c1327),L_(0x98001111)}},
   {{LL(0xdebcecc8,0xfdc46d5f),LL(0xc9916314,0xa125951b),LL(0x5633f2c4,0x04c2afc1),L_(0xd17c9349)}, {LL(0xacd14df9,0xb5f970f1),LL(0x289012ed,0x0973ccd0),LL(0x236b3580,0x02b2c84a),L_(0x76b898cf)}},
   {{LL(0xcb13811d,0x0d2dd5a3),LL(0x23f6e444,0xd8b5d0f6),LL(0x4f399d9a,0xa46887b9),L_(0x77c6bce8)}, {LL(0x10165161,0xddf54326),LL(0xeb345000,0xf080be9b),LL(0xeee34275,0x047237fe),L_(0xd518ca88)}},
   {{LL(0x11fceb12,0xfaf4fbdf),LL(0x116a3791,0x66bfd60d),LL(0x442c2a46,0x28cf0358),L_(0x85546f76)}, {LL(0x4cebfe25,0x6c01d714),LL(0x1c79b699,0x11d66f5a),LL(0x8cc29c26,0xcf7771ef),L_(0x92ae8875)}},
   {{LL(0xf90d16bf,0xb89933d4),LL(0xa1457028,0xc8f0695e),LL(0x5a6af165,0x82274dbf),L_(0x89528271)}, {LL(0x225b8bb3,0x5e138d61),LL(0x2eed19b9,0x3e2273d0),LL(0x6b42f9f3,0xaa4b331c),L_(0xe7d915a6)}},
   {{LL(0xa9d80f3e,0x6773ac5a),LL(0xc52d7955,0x970b6066),LL(0x78700052,0x2f129047),L_(0xa9a60bf2)}, {LL(0x0393eefb,0x04368569),LL(0xe6519a05,0x5180fe6e),LL(0x3014d2a0,0x6bd99185),L_(0xb28fcd33)}},
   {{LL(0x299011de,0xe9270a23),LL(0x5a66907e,0x54a9528a),LL(0x7af57c36,0x8e613981),L_(0xe6775c8b)}, {LL(0xff5fc082,0xd7dd7f17),LL(0xe0568a5c,0x421cb6c4),LL(0x590b13ce,0x8f9e6852),L_(0x4c77050e)}},
   {{LL(0x6513f9c9,0xd5bf1a5d),LL(0x170fc778,0x1942fc9b),LL(0x6ec5f13f,0x70273957),L_(0x1878fa8e)}, {LL(0x5bb80bbc,0xcededd82),LL(0x83d79890,0xa7c7ee32),LL(0xfa06b05d,0x86a8080f),L_(0x0e66721a)}},
   {{LL(0x9510adb1,0x8a998afb),LL(0x7aa85880,0xfaf28d30),LL(0x0863e4f3,0xdcf64ed3),L_(0x1fe9103e)}, {LL(0x1ea72a10,0x78d5ce7b),LL(0x33edcfdb,0xafe82b0c),LL(0xde25a969,0x1f0a50fe),L_(0x5f4147d4)}},
   {{LL(0x3d5e21a2,0xddb5610d),LL(0x8024cc9b,0x435cd0ff),LL(0xcc34b698,0xf7218db6),L_(0xb6949149)}, {LL(0xc23a0cb8,0x3b787bdf),LL(0x18dd349c,0x71fcc010),LL(0xb0e2a897,0xd6c69c33),L_(0x09bc9f85)}},
   {{LL(0x1cd5d5db,0xe94a402b),LL(0xf30b7314,0x26944f05),LL(0x737d6c5b,0xd0af2018),L_(0xfdbedbb6)}, {LL(0x1b523221,0xa8a984b3),LL(0x8ace709d,0x39663eef),LL(0xbf4b3e0e,0x0297dfd1),L_(0x3e1d52f4)}},
   {{LL(0xc8d20d14,0xf915a4c4),LL(0x0b7dd701,0x2921fb0f),LL(0x6b467b73,0x226421aa),L_(0xf53a0a69)}, {LL(0xf4bdad8d,0xb5512d8c),LL(0xc103b145,0x2df92000),LL(0x4979cf5a,0xd5159856),L_(0x0d121a5d)}},
   {{LL(0xffb25429,0xad4ff433),LL(0xe3c5fc0a,0x22c48ae5),LL(0xdcdd2832,0xee468cb2),L_(0x853d6571)}, {LL(0xad56cbbd,0x1ac2be8d),LL(0x305ea1a5,0x11bdfbf2),LL(0x4c080ea0,0xd326fcc1),L_(0x98961d03)}},
   {{LL(0x3c2a21b3,0x4c900a2e),LL(0x370aac6c,0x540b6e6c),LL(0x40c9b00d,0xf0db08ca),L_(0xd9b48af0)}, {LL(0x75ea8b19,0x23e0167d),LL(0x69f68546,0x3ce76be7),LL(0xc2578bd9,0x274ef866),L_(0xe0f171b1)}},
   {{LL(0x06e96b08,0xa0cc2074),LL(0x658170a1,0x2181d377),LL(0xbc50ac87,0x46b62277),L_(0xd2d87843)}, {LL(0x66a073ef,0xbc771f86),LL(0xef91a0af,0x4727577f),LL(0xa7216ad9,0xd3c1c667),L_(0x3c62c22f)}},
   {{LL(0x37e5b8f5,0xfea02084),LL(0x4baddc8e,0xe9a5f0b7),LL(0xbf587847,0x53933d2b),L_(0xca648fb8)}, {LL(0x492b64e4,0x8257b251),LL(0x0f117e6d,0x8d9ec843),LL(0xfe9686a0,0xeeea3972),L_(0xe386bb0a)}},
   {{LL(0xb93a8f56,0xf5712604),LL(0xa696fd94,0x6981acc8),LL(0xf73a981c,0x86cc6e25),L_(0x53db10f5)}, {LL(0xa6c75879,0x18cf2a94),LL(0xe6e2104d,0x5737dd4f),LL(0x4c1f9f0e,0x3114afeb),L_(0x2e6f8e44)}},
   {{LL(0xbbc2c60f,0x19e92b39),LL(0x06b78031,0xb3de20d1),LL(0x43d40348,0x693597af),L_(0x1396fecd)}, {LL(0xa7c941f2,0x4568fdc8),LL(0xd45e157f,0x92d2afec),LL(0x50493b74,0xa510adc1),L_(0x64794afe)}},
   {{LL(0x60867e2f,0xab4d38b7),LL(0xf04773c1,0x0b8a018e),LL(0x2c774449,0xc11883d8),L_(0xd65b4aa4)}, {LL(0xa50bce8a,0x9bcfd81b),LL(0xfd058e49,0xf5811bc4),LL(0x80f1e9c2,0xe9b876e5),L_(0xc8fc1b3c)}},
   {{LL(0x368a0742,0x7cf8b51b),LL(0x09e82684,0x85f0eb37),LL(0xb5926249,0x08e70a34),L_(0x4decfe48)}, {LL(0xe9a1940e,0x4d61fee9),LL(0x6dcb9563,0x24af91e7),LL(0xa85a5970,0x41f4907d),L_(0x2292e826)}},
   {{LL(0x33182546,0x7b2ef570),LL(0xb4417ce0,0x2a780100),LL(0x8d1180df,0xab954cb9),L_(0x5c74a302)}, {LL(0x88d81acf,0x0d62a965),LL(0x2ec9496d,0xaa16f924),LL(0x2bc7cca4,0x2d99dcc8),L_(0x74958dce)}},
   {{LL(0xb139570a,0x21c56351),LL(0x1f26d583,0x41b20f4e),LL(0x067af3fb,0xbd9ab15f),L_(0xc3a77e33)}, {LL(0x1523f3e0,0x9de5e7e4),LL(0xe1731bb7,0xef0cf37e),LL(0x6c9e48d4,0xc9fc2851),L_(0xd0f91844)}},
   {{LL(0xb81124f1,0xea0ed825),LL(0x788a57ed,0xf4f08526),LL(0x8e19f5c1,0x820a4f79),L_(0x348a1346)}, {LL(0x6eaedecc,0xefe607bc),LL(0x87c20db0,0x75968ada),LL(0x21596aa9,0xcdf9ca14),L_(0x5e2abcad)}},
   {{LL(0xdaef28c3,0x2be4550b),LL(0x29e6daee,0xee673998),LL(0x8162c783,0x4e1fb36c),L_(0xb3249498)}, {LL(0xab7e3386,0xb682150c),LL(0x89285111,0x02a0814e),LL(0x9c692620,0x3bb22aea),L_(0xa759126d)}},
   {{LL(0xa03f91a6,0xd356a2bd),LL(0x7b288f41,0x04625b59),LL(0xf350f847,0x2eacfb01),L_(0x943dbfd9)}, {LL(0x63af93f5,0xba005c26),LL(0xdcc822a7,0x7964a6f8),LL(0xc56edb4d,0x14b146d7),L_(0xb1fb9d62)}},
   {{LL(0xcae5ffde,0x51fbb8f7),LL(0xca0c3f68,0xa961e967),LL(0xbe43f447,0x2a774b0b),L_(0x9b6effff)}, {LL(0x4e174bf0,0xb12fde0a),LL(0xd1976ab8,0x85b59e4e),LL(0x421eac26,0x4c480a51),L_(0x4041c2d8)}},
   {{LL(0xc3a1a863,0xfe8ceef6),LL(0x42e96299,0x0469540a),LL(0xd231a2a4,0x0828f4c0),L_(0x7c0edcf4)}, {LL(0x2c034efb,0x1744d906),LL(0x92393260,0x2a85a8d6),LL(0x28f5775a,0x58717292),L_(0x37693735)}},
   {{LL(0xf12dc792,0xbcc5e464),LL(0xfee1ef55,0xb881c818),LL(0x2bfca9a4,0x7e60531f),L_(0xed54d832)}, {LL(0x28ea4ef0,0x9ef6ada6),LL(0xb9564865,0xf56fbcf0),LL(0x534e2057,0x9857c41f),L_(0x2750bbe8)}},
}
    /* clang-format on */
};
#endif

#endif /* _DISABLE_ECP_224R1_HARDCODED_BP_TBL_ */
#endif /* _IPP_DATA */


IPP_OWN_DEFN(const cpPrecompAP*, gfpec_precom_nistP224r1_fun, (void))
{
    static cpPrecompAP t = { /* w */ 7,
                             /* select function */ p224r1_select_ap_w7,
                             /* precomputed data */ (BNU_CHUNK_T*)ec_p224r1_precomputed };
    return &t;
}
