/*
 * Copyright (C) 2018 Alyssa Rosenzweig
 * Copyright (C) 2019-2020 Collabora, Ltd.
 * SPDX-License-Identifier: MIT
 */

#include "bi_builder.h"
#include "compiler.h"

/* SSA copy propagation */

static bool
bi_reads_fau(bi_instr *ins)
{
   bi_foreach_src(ins, s) {
      if (ins->src[s].type == BI_INDEX_FAU)
         return true;
   }

   return false;
}

void
bi_opt_copy_prop(bi_context *ctx)
{
   /* Chase SPLIT of COLLECT. Instruction selection usually avoids this
    * pattern (due to the split cache), but it is inevitably generated by
    * the UBO pushing pass.
    */
   bi_instr **collects = calloc(sizeof(bi_instr *), ctx->ssa_alloc);
   bi_foreach_instr_global_safe(ctx, I) {
      if (I->op == BI_OPCODE_COLLECT_I32) {
         /* Rewrite trivial collects while we're at it */
         if (I->nr_srcs == 1)
            bi_set_opcode(I, BI_OPCODE_MOV_I32);

         collects[I->dest[0].value] = I;
      } else if (I->op == BI_OPCODE_SPLIT_I32) {
         /* Rewrite trivial splits while we're at it */
         if (I->nr_dests == 1)
            bi_set_opcode(I, BI_OPCODE_MOV_I32);

         bi_instr *collect = collects[I->src[0].value];
         if (!collect)
            continue;

         /* Lower the split to moves, copyprop cleans up */
         bi_builder b = bi_init_builder(ctx, bi_before_instr(I));

         bi_foreach_dest(I, d)
            bi_mov_i32_to(&b, I->dest[d], collect->src[d]);

         bi_remove_instruction(I);
      }
   }

   free(collects);

   bi_index *replacement = calloc(sizeof(bi_index), ctx->ssa_alloc);

   bi_foreach_instr_global_safe(ctx, ins) {
      if (ins->op == BI_OPCODE_MOV_I32 &&
          ins->src[0].type != BI_INDEX_REGISTER) {
         bi_index replace = ins->src[0];

         /* Peek through one layer so copyprop converges in one
          * iteration for chained moves */
         if (bi_is_ssa(replace)) {
            bi_index chained = replacement[replace.value];

            if (!bi_is_null(chained))
               replace = chained;
         }

         assert(ins->nr_dests == 1);
         replacement[ins->dest[0].value] = replace;
      }

      bi_foreach_src(ins, s) {
         bi_index use = ins->src[s];

         if (use.type != BI_INDEX_NORMAL)
            continue;
         if (bi_is_staging_src(ins, s))
            continue;

         bi_index repl = replacement[use.value];

         if (repl.type == BI_INDEX_CONSTANT && bi_reads_fau(ins))
            continue;

         if (!bi_is_null(repl))
            bi_replace_src(ins, s, repl);
      }
   }

   free(replacement);
}
