# -*- coding: utf-8 -*-
# Copyright 2010-2021, Google Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#     * Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above
# copyright notice, this list of conditions and the following disclaimer
# in the documentation and/or other materials provided with the
# distribution.
#     * Neither the name of Google Inc. nor the names of its
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""Tool to update quality regression test result files.

quality_regression_main --test_file query.tsv > result.tsv
quality_regression.py --input result.tsv --input_version 4400
  --output quality_regression.tsv --base quality_regression.tsv
"""

import argparse
from build_tools import mozc_version


def LoadBase(base_file):
  """Load the base file to store the last modified version."""
  base_data = {}
  with open(base_file, encoding='utf-8') as file:
    for line in file:
      if line.startswith('#'):
        continue
      items = line.strip('\n').split('\t')
      result = '\t'.join(items[0:5])  # status, input, output, command, argument
      version = items[5]
      base_data[result] = version
  return base_data


def LoadInput(input_file, input_version, base_data):
  """Load the input file generated by quality_regression_main."""
  output = []
  with open(input_file, encoding='utf-8') as file:
    for line in file:
      line = line.strip('\n')
      if line.startswith('#'):
        output.append(line)
        continue
      items = line.split('\t')
      result = '\t'.join(items[0:5])  # status, input, output, command, argument
      version = base_data.get(result, input_version)
      output.append(result + '\t' + version)
  return output


def SaveOutput(output_file, output_lines):
  """Save the output file."""
  fields = ['status', 'input', 'output', 'command', 'argument', 'version']
  with open(output_file, 'w', encoding='utf-8') as file:
    file.write('# ' + '\t'.join(fields) + '\n')
    file.write('\n'.join(output_lines) + '\n')


def main():
  parser = argparse.ArgumentParser()
  parser.add_argument('--base')
  parser.add_argument('--input')
  parser.add_argument('--version_file')
  parser.add_argument('--output')
  parser.add_argument('--data_type')
  args = parser.parse_args()

  if args.version_file:
    version = mozc_version.MozcVersion(args.version_file)
    use_build_oss = (args.data_type == 'oss')
    version_string = version.GetShortVersionString(use_build_oss=use_build_oss)
  else:
    version_string = ''

  if args.base:
    base_data = LoadBase(args.base)
  else:
    base_data = {}

  output_lines = LoadInput(args.input, version_string, base_data)
  SaveOutput(args.output, output_lines)


if __name__ == '__main__':
  main()
