/* Copyright (c) 2008-2022 the MRtrix3 contributors.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Covered Software is provided under this License on an "as is"
 * basis, without warranty of any kind, either expressed, implied, or
 * statutory, including, without limitation, warranties that the
 * Covered Software is free of defects, merchantable, fit for a
 * particular purpose or non-infringing.
 * See the Mozilla Public License v. 2.0 for more details.
 *
 * For more details, see http://www.mrtrix.org/.
 */

#include "command.h"
#include "mrtrix.h"
#include "debug.h"
#include "math/constrained_least_squares.h"


using namespace MR;
using namespace App;


void usage ()
{
  AUTHOR = "J-Donald Tournier (jdtournier@gmail.com)";

  SYNOPSIS = "Test the constrained least-squares solver";

  REQUIRES_AT_LEAST_ONE_ARGUMENT = false;
}

using matrix_type = Eigen::MatrixXd;
using vector_type = Eigen::VectorXd;;

const size_t num_sig = 100;
const size_t num_coef = 50;
const size_t num_ineq = 290;
const size_t num_eq = 10;

// random test data generated in Matlab:

double problem_matrix_data[] = { 0.814723686393179, 0.905791937075619, 0.126986816293506, 0.913375856139019, 0.63235924622541, 0.0975404049994095, 0.278498218867048, 0.546881519204984, 0.957506835434298, 0.964888535199277, 0.157613081677548, 0.970592781760616, 0.957166948242946, 0.485375648722841, 0.8002804688888, 0.141886338627215, 0.421761282626275, 0.915735525189067, 0.792207329559554, 0.959492426392903, 0.655740699156587, 0.0357116785741896, 0.849129305868777, 0.933993247757551, 0.678735154857773, 0.757740130578333, 0.743132468124916, 0.392227019534168, 0.655477890177557, 0.171186687811562, 0.706046088019609, 0.0318328463774207, 0.27692298496089, 0.0461713906311539, 0.0971317812358475, 0.823457828327293, 0.694828622975817, 0.317099480060861, 0.950222048838355, 0.0344460805029088, 0.438744359656398, 0.381558457093008, 0.765516788149002, 0.795199901137063, 0.186872604554379, 0.489764395788231, 0.445586200710899, 0.646313010111265, 0.709364830858073, 0.754686681982361, 0.276025076998578, 0.679702676853675, 0.655098003973841, 0.162611735194631, 0.118997681558377, 0.498364051982143, 0.959743958516081, 0.340385726666133, 0.585267750979777, 0.223811939491137, 0.751267059305653, 0.255095115459269, 0.505957051665142, 0.699076722656686, 0.890903252535798, 0.959291425205444, 0.547215529963803, 0.138624442828679, 0.149294005559057, 0.257508254123736, 0.840717255983663, 0.254282178971531, 0.814284826068816, 0.243524968724989, 0.929263623187228, 0.349983765984809, 0.196595250431208, 0.251083857976031, 0.616044676146639, 0.473288848902729, 0.351659507062997, 0.830828627896291, 0.585264091152724, 0.54972360829114, 0.91719366382981, 0.285839018820374, 0.757200229110721, 0.753729094278495, 0.380445846975357, 0.567821640725221, 0.0758542895630636, 0.0539501186666071, 0.530797553008973, 0.779167230102011, 0.934010684229183, 0.12990620847373, 0.568823660872193, 0.469390641058206, 0.0119020695012414, 0.337122644398882, 0.162182308193243, 0.794284540683907, 0.311215042044805, 0.528533135506213, 0.165648729499781, 0.601981941401637, 0.262971284540144, 0.654079098476782, 0.689214503140008, 0.748151592823709, 0.450541598502498, 0.0838213779969326, 0.228976968716819, 0.91333736150167, 0.152378018969223, 0.825816977489547, 0.538342435260057, 0.996134716626885, 0.0781755287531837, 0.442678269775446, 0.106652770180584, 0.961898080855054, 0.00463422413406744, 0.774910464711502, 0.817303220653433, 0.86869470536351, 0.0844358455109103, 0.399782649098896, 0.259870402850654, 0.800068480224308, 0.431413827463545, 0.910647594429523, 0.181847028302852, 0.26380291652199, 0.145538980384717, 0.136068558708664, 0.869292207640089, 0.57970458736557, 0.549860201836332, 0.144954798223727, 0.853031117721894, 0.622055131485066, 0.350952380892271, 0.513249539867053, 0.401808033751942, 0.0759666916908419, 0.239916153553658, 0.123318934835166, 0.183907788282417, 0.239952525664903, 0.41726706908437, 0.0496544303257421, 0.902716109915281, 0.944787189721646, 0.49086409246808, 0.489252638400019, 0.337719409821377, 0.900053846417662, 0.369246781120215, 0.111202755293787, 0.780252068321138, 0.389738836961253, 0.241691285913833, 0.403912145588115, 0.0964545251683886, 0.131973292606335, 0.942050590775485, 0.956134540229802, 0.575208595078466, 0.0597795429471558, 0.234779913372406, 0.353158571222071, 0.821194040197959, 0.0154034376515551, 0.0430238016578078, 0.168990029462704, 0.649115474956452, 0.73172238565867, 0.647745963136307, 0.450923706430945, 0.547008892286345, 0.296320805607773, 0.744692807074156, 0.188955015032545, 0.686775433365315, 0.18351115573727, 0.368484596490336, 0.62561856072969, 0.780227435151377, 0.0811257688657853, 0.92938597096873, 0.775712678608402, 0.486791632403172, 0.435858588580919, 0.446783749429806, 0.306349472016557, 0.508508655381127, 0.51077156417211, 0.817627708322262, 0.794831416883453, 0.644318130193692, 0.378609382660268, 0.811580458282477, 0.532825588799455, 0.350727103576883, 0.939001561999887, 0.875942811492984, 0.550156342898422, 0.622475086001227, 0.587044704531417, 0.207742292733028, 0.301246330279491, 0.470923348517591, 0.230488160211558, 0.844308792695389, 0.194764289567049, 0.225921780972399, 0.170708047147859, 0.227664297816554, 0.435698684103899, 0.311102286650413, 0.923379642103244, 0.430207391329584, 0.184816320124136, 0.904880968679893, 0.979748378356085, 0.438869973126103, 0.111119223440599, 0.258064695912067, 0.408719846112552, 0.594896074008614, 0.262211747780845, 0.602843089382083, 0.711215780433683, 0.22174673401724, 0.117417650855806, 0.296675873218327, 0.318778301925882, 0.424166759713807, 0.507858284661118, 0.085515797090044, 0.262482234698333, 0.801014622769739, 0.0292202775621463, 0.928854139478045, 0.730330862855453, 0.488608973803579, 0.578525061023439, 0.237283579771521, 0.458848828179931, 0.963088539286913, 0.546805718738968, 0.521135830804001, 0.231594386708524, 0.488897743920167, 0.62406008817369, 0.679135540865748, 0.395515215668593, 0.367436648544477, 0.987982003161633, 0.0377388662395521, 0.885168008202475, 0.913286827639239, 0.796183873585212, 0.0987122786555743, 0.261871183870716, 0.335356839962797, 0.679727951377338, 0.13655313735537, 0.72122749858174, 0.106761861607241, 0.65375734866856, 0.49417393663927, 0.779051723231275, 0.715037078400694, 0.903720560556316, 0.890922504330789, 0.334163052737496, 0.698745832334794, 0.197809826685929, 0.0305409463046367, 0.744074260367462, 0.500022435590201, 0.47992214114606, 0.904722238067363, 0.609866648422558, 0.617666389588455, 0.859442305646212, 0.805489424529686, 0.576721515614685, 0.182922469414914, 0.239932010568717, 0.886511933076101, 0.0286741524641061, 0.489901388512224, 0.167927145682257, 0.978680649641159, 0.712694471678914, 0.500471624154843, 0.471088374541939, 0.0596188675796392, 0.681971904149063, 0.0424311375007417, 0.0714454646006424, 0.521649842464284, 0.096730025780867, 0.818148553859625, 0.817547092079286, 0.722439592366842, 0.149865442477967, 0.659605252908307, 0.518594942510538, 0.972974554763863, 0.648991492712356, 0.800330575352401, 0.453797708726919, 0.432391503783462, 0.825313795402046, 0.083469814858914, 0.133171007607162, 0.173388613119006, 0.390937802323736, 0.83137974283907, 0.80336439160244, 0.0604711791698936, 0.399257770613576, 0.526875830508296, 0.416799467930787, 0.656859890973707, 0.627973359190104, 0.291984079961715, 0.43165117024872, 0.015487125636019, 0.984063724379154, 0.167168409914656, 0.106216344928664, 0.372409740055537, 0.198118402542975, 0.489687638016024, 0.339493413390758, 0.951630464777727, 0.920332039836564, 0.0526769976807926, 0.737858095516997, 0.269119426398556, 0.422835615008808, 0.547870901214845, 0.942736984276934, 0.417744104316662, 0.983052466469856, 0.301454948712065, 0.701098755900926, 0.666338851584426, 0.539126465042857, 0.698105520180308, 0.666527913402587, 0.178132454400338, 0.128014399720173, 0.999080394761361, 0.171121066356432, 0.032600820530528, 0.56119979270966, 0.88186650045181, 0.669175304534394, 0.190433267179954, 0.368916546063895, 0.460725937260412, 0.98163795097075, 0.156404952226563, 0.855522805845911, 0.644764536870088, 0.376272210278832, 0.190923695236303, 0.428252992979386, 0.482022061031856, 0.120611613297162, 0.589507484695059, 0.226187679752676, 0.384619124369411, 0.582986382747674, 0.251806122472313, 0.290440664276979, 0.617090884393223, 0.265280909810029, 0.824376266688835, 0.98266339972195, 0.730248792267598, 0.343877004114983, 0.584069333278452, 0.107769015243743, 0.906308150649733, 0.879653724481905, 0.817760559370642, 0.260727999055465, 0.594356250664331, 0.0225125927402318, 0.425259320214135, 0.312718886820616, 0.16148474431175, 0.178766186752368, 0.422885689100085, 0.0942293388877347, 0.598523668756741, 0.470924256358334, 0.695949313301608, 0.699887849928292, 0.638530758271838, 0.0336038360664295, 0.0688060991180512, 0.319599735180496, 0.530864280694127, 0.654445707757066, 0.407619197041153, 0.819981222781941, 0.718358943205884, 0.968649330231094, 0.531333906565674, 0.32514568182056, 0.105629203329022, 0.610958658746201, 0.778802241824093, 0.423452918962738, 0.0908232857874395, 0.266471490779072, 0.153656717591307, 0.281005302533871, 0.440085139001721, 0.527142741760652, 0.457424365687674, 0.875371598604185, 0.518052108361104, 0.943622624548388, 0.637709098072174, 0.957693939841583, 0.24070703548016, 0.676122303863752, 0.289064571674477, 0.671808165414215, 0.695140499551737, 0.0679927684700106, 0.254790156597005, 0.224040030824219, 0.667832727013717, 0.844392156527205, 0.344462411301042, 0.780519652731358, 0.675332065747, 0.00671531431847749, 0.602170487581795, 0.386771194520985, 0.915991244131425, 0.00115105712910724, 0.462449159242329, 0.424349039815375, 0.460916366028964, 0.770159728608609, 0.322471807186779, 0.784739294760742, 0.471357153710612, 0.0357627332691179, 0.175874415683531, 0.721758033391102, 0.47348599296532, 0.152721200438232, 0.341124607049109, 0.607389213768347, 0.191745255461798, 0.738426839976942, 0.242849598318169, 0.917424342049382, 0.269061586686018, 0.765500016621438, 0.188661976791491, 0.287498173066131, 0.0911134636865349, 0.576209380663007, 0.683363243294653, 0.546593114590323, 0.425728841871188, 0.644442781431336, 0.647617630172684, 0.679016754093202, 0.635786710514084, 0.945174113109401, 0.208934922426023, 0.709281702710545, 0.236230576993797, 0.119396247797306, 0.607303940685635, 0.450137696965896, 0.458725493648868, 0.661944751905652, 0.77028551480366, 0.350218013441105, 0.662009598359135, 0.416158589969796, 0.841929152691309, 0.832916819075216, 0.256440992229147, 0.613460736812875, 0.582249164527227, 0.54073933712441, 0.869941032358007, 0.26477902647563, 0.318074075481059, 0.119214541054191, 0.939829470344921, 0.645551874972524, 0.479463224948888, 0.639316961040108, 0.544716110526763, 0.647311480293128, 0.543885933999639, 0.721046620579811, 0.522495305777102, 0.993704624120852, 0.218676632399634, 0.105798273250228, 0.109697464523194, 0.0635913709751057, 0.404579995857626, 0.448372912066495, 0.365816176838171, 0.763504640848813, 0.627896379614169, 0.771980385554245, 0.93285357027882, 0.972740854003014, 0.192028349427775, 0.138874202829155, 0.696266337082995, 0.0938200267748656, 0.525404403859336, 0.530344218392863, 0.861139811393332, 0.484853333552102, 0.393456361215266, 0.671431139674026, 0.741257943454206, 0.520052467390387, 0.347712671277525, 0.149997253831683, 0.586092067231462, 0.262145317727807, 0.0444540922782385, 0.754933267231179, 0.242785357820962, 0.442402313001943, 0.687796085120107, 0.359228210401861, 0.736340074301202, 0.394707475278763, 0.683415866967978, 0.704047430334266, 0.442305413383371, 0.0195776235533187, 0.330857880214071, 0.424309496833137, 0.270270423432065, 0.197053798095456, 0.82172118496131, 0.429921409383266, 0.887770954256354, 0.391182995461163, 0.769114387388296, 0.396791517013617, 0.808514095887345, 0.755077099007084, 0.377395544835103, 0.216018915961394, 0.790407217966913, 0.949303911849797, 0.327565434075205, 0.67126437045174, 0.438644982586956, 0.833500595588975, 0.768854252429615, 0.167253545494722, 0.861980478702072, 0.989872153631504, 0.514423456505704, 0.884281023126955, 0.588026055308497, 0.154752348656045, 0.199862822857452, 0.406954837138907, 0.748705718215691, 0.825583815786156, 0.789963029944531, 0.318524245398992, 0.534064127370726, 0.089950678770581, 0.111705744193203, 0.136292548938299, 0.678652304800188, 0.495177019089661, 0.18971040601758, 0.495005824990221, 0.147608221976689, 0.0549741469061882, 0.850712674289007, 0.560559527354885, 0.929608866756663, 0.696667200555228, 0.58279096517584, 0.815397211477421, 0.879013904597178, 0.988911616079589, 0.000522375356944771, 0.865438591013025, 0.612566469483999, 0.989950205708831, 0.527680069338442, 0.479523385210219, 0.801347605521952, 0.227842935706042, 0.49809429119639, 0.900852488532005, 0.574661219130188, 0.845178185054037, 0.738640291995402, 0.585987035826476, 0.246734525985975, 0.666416217319468, 0.0834828136026227, 0.625959785171583, 0.660944557947342, 0.729751855317221, 0.890752116325322, 0.982303222883606, 0.769029085335896, 0.581446487875398, 0.928313062314188, 0.580090365758442, 0.0169829383372613, 0.120859571098558, 0.86271071869967, 0.484296511212102, 0.844855674576263, 0.209405084020935, 0.552291341538775, 0.629883385064421, 0.0319910157625669, 0.614713419117141, 0.362411462273053, 0.0495325790420612, 0.489569989177322, 0.192510396062075, 0.123083747545945, 0.20549417090768, 0.14651491061489, 0.189072174472614, 0.0426524109111434, 0.635197916859882, 0.28186685588043, 0.53859667804534, 0.695163039444332, 0.499116013482589, 0.535801055751113, 0.445183165296042, 0.12393227759807, 0.490357293468018, 0.852998155340816, 0.873927405861733, 0.270294332292698, 0.208461358751314, 0.564979570738201, 0.640311825162758, 0.417028951642886, 0.205975515532243, 0.947933121293169, 0.0820712070977259, 0.105709426581721, 0.142041121903998, 0.166460440876421, 0.620958643935308, 0.573709764841198, 0.0520778902858696, 0.93120138460825, 0.728661681678271, 0.73784165379759, 0.0634045006928182, 0.860440563038232, 0.934405118961213, 0.984398312240972, 0.858938816683866, 0.785558989265031, 0.513377418587575, 0.177602460505865, 0.398589496735843, 0.133931250987971, 0.0308895487449515, 0.939141706069548, 0.301306064586392, 0.295533834475356, 0.332936281836175, 0.467068187028852, 0.648198406466157, 0.0252281814930363, 0.842206612419334, 0.559032544988695, 0.854099949273443, 0.347879194327261, 0.446026648055103, 0.0542394844411296, 0.177107533789109, 0.662808061960974, 0.330828995203305, 0.8984861378343, 0.118155198446711, 0.988417928784981, 0.539982099037929, 0.706917419322763, 0.999491620097705, 0.287849344815137, 0.414522538893108, 0.464839941625137, 0.763957078491957, 0.818204038907671, 0.100221540195492, 0.178116953886766, 0.35963491348208, 0.0567046890682912, 0.521885673661279, 0.335848974676925, 0.175669029675661, 0.208946673993135, 0.905153559004464, 0.675391177336247, 0.468468199903997, 0.912132474239623, 0.104011574779379, 0.745546073701717, 0.736267455596639, 0.561861425281637, 0.184194097515527, 0.597211350337855, 0.299936990089789, 0.134122932828682, 0.212601533358843, 0.894941675440814, 0.0714528127870445, 0.242486558936719, 0.0537543922087138, 0.441722057064424, 0.0132832004672525, 0.897191350973572, 0.196658191367632, 0.093370516755093, 0.30736689958792, 0.456057666843741, 0.101669393624755, 0.995389727655092, 0.332092833452499, 0.297346815887922, 0.0620452213196326, 0.298243971887764, 0.0463512648981808, 0.505428142457703, 0.761425886690113, 0.631069999213594, 0.0898916507760917, 0.0808624231303137, 0.777240536548943, 0.905134744223571, 0.533771951767, 0.109154212042459, 0.825808857855598, 0.338097718802172, 0.293973053026484, 0.746313427703679, 0.0103366183433964, 0.0484473392532221, 0.667916121573624, 0.603467983830769, 0.526102465795561, 0.729709448223228, 0.707253485315422, 0.781377051799277, 0.287976975614171, 0.692531986386519, 0.556669834964013, 0.396520792581593, 0.0615906670539647, 0.780175531491174, 0.337583864052045, 0.607865907262946, 0.741254049502218, 0.1048132419735, 0.127888379782995, 0.549540107015198, 0.485229408584959, 0.890475679184438, 0.798960278812879, 0.73434108369597, 0.0513318861123711, 0.0728852990989761, 0.0885274596747204, 0.798350864113952, 0.943008139570703, 0.683715572408358, 0.132082955713563, 0.722724539656766, 0.110353480642349, 0.117492852151833, 0.640717922965926, 0.328814214756803, 0.653812022595774, 0.749131463103519, 0.583185731454876, 0.740032327987778, 0.234826914747904, 0.734957541696052, 0.970598525086614, 0.866930291751916, 0.0862345298634963, 0.366436616319199, 0.369198804330018, 0.685028472661609, 0.597941635383889, 0.789363943641905, 0.367652918437877, 0.206027859505195, 0.0866665473955323, 0.771933917099107, 0.20567452146476, 0.388271631047802, 0.551778531957227, 0.2289532520231, 0.641940620399187, 0.48448037239818, 0.151845525116267, 0.781931966588002, 0.100606322362422, 0.294066333758628, 0.237373019705579, 0.530872257027928, 0.0914987313394122, 0.405315419880591, 0.104846247115757, 0.112283962156027, 0.784427890743913, 0.291570317906931, 0.603533438750887, 0.964422667215901, 0.432484993970361, 0.69475219461794, 0.758099275289454, 0.432642326147101, 0.655498039803537, 0.109755050723052, 0.933759848385332, 0.187460806421686, 0.266178838907639, 0.797830260211597, 0.487603775433924, 0.768958264058869, 0.396006745217875, 0.272938794123691, 0.037234634070328, 0.673294914108653, 0.429564459251853, 0.451739234904736, 0.609857169290216, 0.0594032968582772, 0.315811438338866, 0.772722130862935, 0.696432989006095, 0.12533218110918, 0.130151450389424, 0.0923523387192016, 0.00782029356933489, 0.423109385164167, 0.655573174937914, 0.722922524692024, 0.531209293582439, 0.108817938273045, 0.631766373528489, 0.126499865329303, 0.134303304313575, 0.0985940927109977, 0.142027248431928, 0.168251298491528, 0.196248922256955, 0.317479775149435, 0.316428999146291, 0.217563309422821, 0.251041846015736, 0.892922405285977, 0.703223224556291, 0.555737942719387, 0.184433667757653, 0.212030842532321, 0.0773468081126768, 0.913800410779568, 0.706715217696931, 0.557788966754876, 0.313428989936591, 0.166203562902151, 0.622497259279895, 0.987934734952495, 0.170432023056883, 0.257792250572013, 0.396799318633144, 0.073994769576938, 0.684096066962009, 0.402388332696162, 0.982835201393951, 0.402183985222485, 0.620671947199578, 0.154369805479272, 0.381345204444472, 0.161133971849361, 0.758112431327419, 0.871111121915389, 0.350776744885893, 0.685535708747537, 0.29414863376785, 0.530629303856886, 0.832423386285184, 0.597490191872579, 0.335311330705246, 0.299225023333107, 0.452592541569324, 0.422645653220462, 0.359606317972236, 0.558319199869297, 0.742545365701939, 0.424334783625691, 0.429355788576205, 0.124872758719813, 0.024434016050374, 0.290185265130727, 0.317520582899226, 0.653690133966475, 0.956935924070684, 0.93573087278488, 0.457886333854367, 0.240478396832085, 0.763897944286478, 0.759327383131096, 0.740648064978614, 0.743688341487326, 0.105920416732765, 0.681560430470316, 0.463260578593719, 0.212163205254934, 0.0985187376881084, 0.823574473927839, 0.17500973738208, 0.163569909784993, 0.665987216411111, 0.894389375354243, 0.51655820835127, 0.702702306950475, 0.1535903766194, 0.953457069886248, 0.540884081241476, 0.679733898210467, 0.0365630180484529, 0.809203851293793, 0.748618871776197, 0.120187017987081, 0.525045164762609, 0.325833628763249, 0.546449439903068, 0.398880752383199, 0.415093386613047, 0.180737760254794, 0.255386740488051, 0.0205357746581846, 0.923675612620407, 0.653699889008253, 0.932613572048564, 0.163512368527526, 0.921097255892197, 0.794657885388753, 0.577394196706649, 0.440035595760254, 0.257613736712438, 0.75194639386745, 0.228669482105501, 0.0641870873918986, 0.767329510776574, 0.671202185356536, 0.71521251478584, 0.642060828433852, 0.419048293624883, 0.390762082204175, 0.816140102875323, 0.31742786365585, 0.814539772900651, 0.789073514938958, 0.852263890343846, 0.505636617571756, 0.635661388861377, 0.950894415378135, 0.44396415501881, 0.060018819779476, 0.866749896999319, 0.631188734269011, 0.355073651878849, 0.997003271606648, 0.224171498983127, 0.652451072968615, 0.604990641908259, 0.387245431483135, 0.142187159290504, 0.0251349857102031, 0.421112253765241, 0.184100289427511, 0.725775267469453, 0.370362686515198, 0.841560087468206, 0.734229691193313, 0.571025872824379, 0.176855057625303, 0.957384022595723, 0.26532203619292, 0.924580895239601, 0.223770404697041, 0.373563807642645, 0.0875003495765857, 0.640116548246715, 0.180616887753108, 0.0450511074735742, 0.723173479183095, 0.34743764558179, 0.660616824502904, 0.383868601071971, 0.627346502443467, 0.0216498146303065, 0.910569988523029, 0.800558656278811, 0.745847484342721, 0.813112813610761, 0.383306318625529, 0.617279232316449, 0.575494859702814, 0.530051704765016, 0.275069755821935, 0.24862895966197, 0.451638770451973, 0.227712826026548, 0.80444958361307, 0.98610424189597, 0.0299919502693899, 0.535664190667238, 0.0870772199008923, 0.802091440555804, 0.98914490970034, 0.0669462583977501, 0.939398361884535, 0.0181775336366957, 0.683838613746355, 0.783736480083219, 0.534137567882728, 0.885359450931142, 0.89900489890614, 0.625937626080496, 0.137868992412558, 0.217801593712125, 0.182141075890434, 0.0418198639729543, 0.106941658550207, 0.616443485085685, 0.939661010161067, 0.354455730967329, 0.410629090059514, 0.984349416984452, 0.945579189035263, 0.676644678433539, 0.988302262313286, 0.766831387216594, 0.336699264391476, 0.662381860399481, 0.244165286790279, 0.295507250831597, 0.680178371230502, 0.527846830418798, 0.411593513407535, 0.602638218036397, 0.750520055923736, 0.583533174261959, 0.551792514935967, 0.583570618757714, 0.511819919958475, 0.0825927269986994, 0.71957013485949, 0.996156111296869, 0.354534304956922, 0.971258815183406, 0.34644876130036, 0.886543861760306, 0.454694864991908, 0.413427289020815, 0.2177320683573, 0.125654587362626, 0.308914593566815, 0.726104431664832, 0.782872072979123, 0.693787614986897, 0.00980225226306208, 0.84321333801051, 0.922331997796276, 0.770954220673924, 0.0426598559350487, 0.378186137050219, 0.704339624483368, 0.729513045504647, 0.224277070664514, 0.269054731773365, 0.673031165004119, 0.477492197726861, 0.623716412667442, 0.23644493264091, 0.177123754498317, 0.829643385697585, 0.766921667935265, 0.93447827311827, 0.107888905084364, 0.182227505601056, 0.0990952824224585, 0.489763801923447, 0.19324533046887, 0.895891573792568, 0.0990896496818146, 0.0441655717626611, 0.557295155804762, 0.772495067187624, 0.311940057496269, 0.178982479314335, 0.338955678247718, 0.210145637043552, 0.51015251976525, 0.906364323265215, 0.628923938652318, 0.101533888812312, 0.390854752726355, 0.0546166152236576, 0.501282913203215, 0.431721171884247, 0.997560349512189, 0.811602580995323, 0.485651669898018, 0.894447755567393, 0.137546594767065, 0.390004914445285, 0.927356224998125, 0.917493832416117, 0.713574011594316, 0.61833738362194, 0.343287890241345, 0.93602732668977, 0.124774040660493, 0.730585361505707, 0.646477432425814, 0.833151985669295, 0.398282228218775, 0.749822209360636, 0.835220510478131, 0.322460397362259, 0.552261616858355, 0.979129132433892, 0.549308533018303, 0.330423609606603, 0.61947155517778, 0.360636571002203, 0.756509543501944, 0.41390074869019, 0.492345104384938, 0.69474323313261, 0.972733885079784, 0.327754960493407, 0.837803183078576, 0.739072227273528, 0.954174456379543, 0.0319226295039784, 0.356868986182542, 0.662653834287214, 0.281501559148491, 0.230383067317464, 0.711128551180325, 0.624572916993309, 0.590608652919636, 0.660437966312602, 0.0475546731138661, 0.348784808510059, 0.451340580355743, 0.240904997120111, 0.715045013296177, 0.856182292006288, 0.281507695118553, 0.731050829723741, 0.137762892519516, 0.836722781749717, 0.13860171574236, 0.588209385389494, 0.366156800454938, 0.806759544661106, 0.503780785776156, 0.489594338723354, 0.877048723385044, 0.353141812938956, 0.449443556571748, 0.963530286843427, 0.0422977979145428, 0.972958334140635, 0.189206843122376, 0.667120300040075, 0.586439614678918, 0.675112416405115, 0.361022049194661, 0.620278427071085, 0.811150885100285, 0.0192574774141414, 0.0838735082828999, 0.97480166718489, 0.651349532415353, 0.231237816164352, 0.40349114312459, 0.12202051825213, 0.268438821397283, 0.257846170112605, 0.331665238742629, 0.152234012862946, 0.348007659716113, 0.121658454307726, 0.88415305774966, 0.0942783900641461, 0.930040626107489, 0.399019969034555, 0.0474014620291507, 0.34237350316074, 0.735966158880164, 0.794682157333805, 0.544905898244672, 0.686223462984041, 0.893632695933602, 0.0547917899197324, 0.303661379717016, 0.0461915563315478, 0.195476763633798, 0.720165802516526, 0.721753273499297, 0.877799071151527, 0.58243296448741, 0.07068433538665, 0.922744570189092, 0.800372092351107, 0.285946856420363, 0.543663232841362, 0.98477623697061, 0.715678067126622, 0.83896959733, 0.433260561007948, 0.470624715606541, 0.560713411044556, 0.269091543530998, 0.749018467641946, 0.503887773460489, 0.64680966627278, 0.307745582030941, 0.138724636078145, 0.475572933797563, 0.362459280818746, 0.788113428398547, 0.78029582084411, 0.668512214253545, 0.133503859661312, 0.021555887203497, 0.55984070587251, 0.300819018069489, 0.939409713873458, 0.980903636046859, 0.286620388894259, 0.800820286951535, 0.896111351432604, 0.59752657681783, 0.88401673572382, 0.943731541195791, 0.549158087419903, 0.728386824594357, 0.57675829785801, 0.0258574710831396, 0.446530978284814, 0.646301957350656, 0.521202952672418, 0.372312660779512, 0.937134666341562, 0.829532824526515, 0.849085479954455, 0.372534239899539, 0.59318457521849, 0.872552564647559, 0.933501608507105, 0.668464274361376, 0.206776457935105, 0.653850592062588, 0.0720515512206499, 0.406726915089364, 0.666931533207499, 0.93372565954593, 0.810950032238265, 0.48454827183499, 0.756749210065512, 0.417047453742767, 0.971785992989294, 0.987974701230832, 0.864147529031203, 0.388883775912429, 0.454741828039112, 0.246687197638079, 0.784423093024177, 0.882837605583052, 0.913711681293018, 0.558284923622275, 0.598868102746205, 0.148876720337549, 0.899713484535739, 0.450393580652116, 0.205672339463963, 0.899650990509081, 0.762585539373227, 0.882486307400298, 0.284950218024821, 0.673225986452258, 0.664279904418051, 0.122814993899526, 0.407318423056813, 0.27528695131569, 0.716669740639142, 0.283384381628936, 0.896198856849495, 0.82657889231356, 0.390026512813998, 0.497902942964187, 0.694805192672494, 0.834369001099938, 0.609629689675732, 0.574737160856408, 0.326042170922791, 0.456424600851747, 0.713795583233135, 0.884405045275995, 0.720855670816931, 0.0186127747263861, 0.674776467128286, 0.438508824236513, 0.43782017916627, 0.117036819505658, 0.814681689518582, 0.324855438630509, 0.246228114234641, 0.34271322112328, 0.375692140009588, 0.546553793082959, 0.561920154265977, 0.395822228624001, 0.398130879531336, 0.515367219211572, 0.657530542095911, 0.950915198264959, 0.722348514327498, 0.400079745362338, 0.83187133932981, 0.134338341728708, 0.0604667719398284, 0.0842470523146596, 0.163898318329708, 0.324219920294049, 0.301726777204647, 0.0116809911303396, 0.539905093849625, 0.0953726926278216, 0.146514856442232, 0.631141207014955, 0.859320411428785, 0.974221631238713, 0.570838427472183, 0.996850214574949, 0.553541573495533, 0.515458454840326, 0.330682065990267, 0.430001798393396, 0.491806253603745, 0.0710370839129527, 0.887739215532716, 0.0646335910508684, 0.436184955731448, 0.826629506618892, 0.394534684233034, 0.613474881705262, 0.818640735082209, 0.886235039062606, 0.931111627487221, 0.190784648009122, 0.258582251418772, 0.897865684840299, 0.593361860386109, 0.503840081846848, 0.612809588835559, 0.819422242231876, 0.531889169492926, 0.202075096548432, 0.453893465680522, 0.427910919820171, 0.966052749611999, 0.620055054133474, 0.695389951918004, 0.720164612017979, 0.346895190287614, 0.51699042052531, 0.556694632012102, 0.156495219114045, 0.562056045561775, 0.694803289108456, 0.426455535710237, 0.836270420676314, 0.731387066161933, 0.360031042085772, 0.454212371853399, 0.386389898592172, 0.775554642250535, 0.734271105685939, 0.430277845074156, 0.693752574133913, 0.94521348716846, 0.784232598231741, 0.705571858131069, 0.109334239524346, 0.389930657156485, 0.590904730405336, 0.459380047966325, 0.0503399867389541, 0.228687584020221, 0.834189061056899, 0.015644692696527, 0.863710865055866, 0.0780690530324343, 0.669042590081555, 0.500211324281102, 0.217993798750154, 0.571615725404089, 0.12218915092342, 0.671166232988653, 0.599585548122814, 0.0559761573877905, 0.0563430185433568, 0.15250063700247, 0.0196210667655841, 0.435175545696952, 0.832221475282818, 0.617390171454084, 0.520129415310742, 0.863868222933611, 0.0976979181614431, 0.908052203186769, 0.108016694136759, 0.516996758096945, 0.143156022083576, 0.559370572403004, 0.00457962394732347, 0.766681998621487, 0.848709226458282, 0.916821270253738, 0.986968274783658, 0.505133101798823, 0.271421624417515, 0.100750511921236, 0.507848830829537, 0.585609125701878, 0.762887095910741, 0.0829626491105439, 0.661596193082714, 0.516979014706213, 0.171048017525447, 0.938557864331842, 0.590483177142572, 0.44063468076084, 0.941918930311283, 0.655913820257141, 0.451945709260386, 0.839697420719669, 0.532623502474346, 0.553887065791275, 0.680065530083361, 0.367189905317367, 0.239290606193545, 0.578923492459094, 0.866887054672508, 0.406776760215226, 0.112615141025047, 0.443845836726957, 0.3001844012139, 0.401386853814493, 0.833363563453134, 0.403628662773607, 0.390175938130607, 0.360448893378945, 0.140255359269444, 0.26013019411244, 0.0868151008658337, 0.429397337085805, 0.25728278476986, 0.297555384151118, 0.424858411704626, 0.119207259421287, 0.495066923800459, 0.70640722753756, 0.243573372680951, 0.785070081934006, 0.0740895768609269, 0.393883426981697, 0.00339412296430741, 0.220676891933024, 0.00130056870059614, 0.189179670994735, 0.142484054820971, 0.268075999123376, 0.174892065335783, 0.138648971592595, 0.598885610367865, 0.901057905682726, 0.939379765884238, 0.221184455939078, 0.482671375396414, 0.376011115924755, 0.523780036965869, 0.26487259354859, 0.0683572204708291, 0.436327077480103, 0.173853037365001, 0.0261071081549049, 0.954678274080449, 0.430596519859417, 0.961558573103663, 0.762414484002993, 0.00734866110284693, 0.680038640460139, 0.705950754248883, 0.645128786607382, 0.55230984450773, 0.21810872978328, 0.772366213613043, 0.22802832360045, 0.370864718973405, 0.890928821025695, 0.856376907728261, 0.402433549813444, 0.318019117496735, 0.60863543063499, 0.910195226694482, 0.909098187800882, 0.591594408907438, 0.332571407335551, 0.853063629209919, 0.442397893041165, 0.904355478217944, 0.0331794059521274, 0.532426482500133, 0.716497346504382, 0.179301843821789, 0.336532925801063, 0.187712948575267, 0.321927183182924, 0.403856711230207, 0.548566299856964, 0.0487385927822667, 0.552732133178943, 0.274811404837501, 0.241501741734112, 0.243145179406551, 0.154159449170074, 0.956416361008024, 0.9356613755166, 0.818714437208244, 0.728261847822834, 0.17581172827553, 0.360370977042454, 0.188789972641605, 0.00119839684004341, 0.316419513732543, 0.699616986397055, 0.62525518017904, 0.543062175343852, 0.439037203387669, 0.287427267482543, 0.50165910674669, 0.761546185669647, 0.762408048788714, 0.576055901496557, 0.747662837641716, 0.645534505981322, 0.123219518324506, 0.504397860092767, 0.347261312721634, 0.0921476848051044, 0.147849468032519, 0.198169701066402, 0.672270237457429, 0.431511182901891, 0.694403909684554, 0.256784563270148, 0.00975864988515951, 0.532283071060889, 0.279391965252965, 0.946230153519927, 0.906443266510002, 0.392684575974112, 0.0248552338448721, 0.67143679657246, 0.837170635377054, 0.971499638311263, 0.056932885435248, 0.450323812733453, 0.582470301736287, 0.686637814496303, 0.719432752595998, 0.650040751925391, 0.726914550912652, 0.373847665839982, 0.581582083212866, 0.1161185127794, 0.0576543612148722, 0.979765223975987, 0.284823726860618, 0.594974298809596, 0.96216103101126, 0.185778263408578, 0.193039815977027, 0.341644104645142, 0.93289789581808, 0.390667536617596, 0.273216707999963, 0.151947079846844, 0.397108842743452, 0.374722466951243, 0.131114707043005, 0.435040717895627, 0.0915131672126108, 0.614626958012946, 0.0109790922908117, 0.573260383263372, 0.789729858025762, 0.235366773150869, 0.448019713464051, 0.569358183284932, 0.061401442290847, 0.496288885639885, 0.642315234585298, 0.221265730127105, 0.837056445531881, 0.971075231442514, 0.846372887693173, 0.505999455890548, 0.278875611145231, 0.746617221828201, 0.23693038411287, 0.957345281619123, 0.620260036077771, 0.600262145507628, 0.172604501647669, 0.0903467381432246, 0.255262202643589, 0.85857053125921, 0.911067053397794, 0.699633767225763, 0.725182355023387, 0.229886078857496, 0.576053456321354, 0.810628105007939, 0.403843368384066, 0.988439267199745, 0.0899988149868883, 0.320941032647761, 0.511408938819178, 0.0606063665682423, 0.725687923545844, 0.556555748561992, 0.529359902481257, 0.829982432033195, 0.858759034071804, 0.789028923313949, 0.317833053726228, 0.452207453762982, 0.752227970049942, 0.109861705750686, 0.109742368593904, 0.269883663704401, 0.524637345396311, 0.972651076977497, 0.710408685278169, 0.311859945147533, 0.291457127647727, 0.850357337374621, 0.911647424007853, 0.639276147276064, 0.255370297944443, 0.0886658400322831, 0.838255587537226, 0.58471861926332, 0.948108735396022, 0.0610289291925092, 0.584641303355111, 0.285108085658642, 0.827732173448263, 0.190986440697398, 0.442529962202884, 0.393411506367576, 0.826573979042765, 0.676871093438419, 0.207603034379981, 0.318104726150263, 0.133810985356126, 0.671462889478031, 0.570991075462406, 0.169767066026489, 0.147655777151737, 0.476079718267456, 0.90810241650695, 0.552175026715835, 0.0329398927498766, 0.0538629264355561, 0.805063228558902, 0.451374854703448, 0.382646229559959, 0.789643703689691, 0.364286869499794, 0.53234993499891, 0.711656705981267, 0.871476517995847, 0.328689611672229, 0.650118025397777, 0.974836148002758, 0.0759673612941356, 0.587019167082772, 0.41388649777336, 0.309136426466267, 0.263834041526795, 0.758766265080204, 0.995215981129752, 0.186571444141369, 0.781145268534765, 0.195797981026732, 0.992358973179926, 0.802261569764289, 0.424226709903306, 0.728863868051374, 0.498353582523552, 0.808990267199045, 0.356508933481046, 0.0732434344898074, 0.590991455274849, 0.910187830728141, 0.19376593616661, 0.432367791534398, 0.749159729068009, 0.039184486647583, 0.946324989805483, 0.763673323663761, 0.55882055050956, 0.183842944465775, 0.497948815018947, 0.517845600234051, 0.99424301064405, 0.854851683090675, 0.962403939711971, 0.678941008977033, 0.403501388804361, 0.934979086178052, 0.479484545632482, 0.231791611663117, 0.396290248509775, 0.705077475112789, 0.55855903456055, 0.756630700894368, 0.995481058555258, 0.962431404366634, 0.535067105256157, 0.963870129971715, 0.115625879181259, 0.0514482932302053, 0.304348945636573, 0.580191833142712, 0.530964452338281, 0.901208092653143, 0.540550425170244, 0.431980610856683, 0.542666987436342, 0.712414805789522, 0.0166747129402325, 0.800920882057848, 0.142509324923791, 0.478474472902863, 0.256835354179634, 0.369091688824861, 0.661764913365414, 0.169608813454314, 0.278784020439019, 0.198221794513839, 0.195071533284261, 0.326839648349976, 0.8803378603792, 0.471101865015748, 0.403969372170941, 0.179231476423447, 0.968924996147292, 0.40745573780262, 0.844487396483448, 0.615325097130849, 0.376611078032737, 0.877181749337097, 0.784852427283024, 0.464954283314534, 0.813976926820934, 0.89844413718012, 0.429238543114827, 0.33432941962222, 0.596647104445246, 0.901990808706562, 0.702066449359676, 0.377455102070865, 0.734955925430014, 0.954102786471214, 0.542813113103551, 0.540105832359366, 0.311109639604002, 0.0712345719245837, 0.181980467169021, 0.0929889268706779, 0.463489247762243, 0.00933251202776941, 0.91502591173579, 0.642741739133104, 0.00141905820240296, 0.0303852724949177, 0.20847022330532, 0.454966145002097, 0.127266033325133, 0.00864768787809034, 0.727079601148527, 0.354116466056586, 0.780445946096025, 0.436656641635291, 0.436554782372268, 0.0492131803564864, 0.0496319015193902, 0.0911001754130539, 0.594037031444121, 0.241084055169022, 0.841369101972144, 0.857212764090606, 0.963612200951355, 0.488899786160604, 0.220310100568633, 0.226208640841286, 0.536787804512826, 0.762109709211147, 0.347567150448759, 0.461231759391499, 0.639323762199356, 0.917336040866845, 0.161572573890331, 0.715635404167474, 0.577738876192409, 0.433298977209119, 0.884242782631094, 0.393051755376913, 0.178975152627732, 0.633333580320122, 0.624000554312344, 0.327941596973249, 0.802965315958237, 0.999477858635892, 0.980978160932146, 0.127036942194857, 0.232240145961793, 0.023632466658225, 0.607432610401855, 0.11080932128715, 0.407459507878399, 0.884076806661962, 0.548132777476588, 0.369003076711617, 0.208345982813347, 0.440943276524324, 0.956196152175878, 0.124025916648043, 0.470763245866957, 0.856896327782193, 0.0433904715669532, 0.691625145201306, 0.978985466675038, 0.283267898492137, 0.133780499994423, 0.685279684412687, 0.909454555749395, 0.610868982383243, 0.899982796432861, 0.193433751939092, 0.75442458148825, 0.346260716664775, 0.418625414302713, 0.155719830354892, 0.819000583926841, 0.624923527846277, 0.738560421610648, 0.805112420706954, 0.0672226575631786, 0.950790316505566, 0.497577011022161, 0.755145904700522, 0.742405070676942, 0.831129575710113, 0.156501846750202, 0.457308720551409, 0.618100496094381, 0.932183355047051, 0.835088235559351, 0.89542351634235, 0.58251852466803, 0.582746806361791, 0.854925938969955, 0.0348657000479308, 0.885420095484803, 0.407730811355982, 0.0363820304478321, 0.74614794273772, 0.154828770751106, 0.143908328674868, 0.605959053323747, 0.254480844051369, 0.324154205005822, 0.401790917846999, 0.406373321401127, 0.386191138560912, 0.609801823837955, 0.166890722666246, 0.188092073195248, 0.0946288698621655, 0.323186272761264, 0.769596929272335, 0.234118002458998, 0.740365469624262, 0.692818413831729, 0.82407821336638, 0.827977549139487, 0.293367934008392, 0.309369242422749, 0.523029798441993, 0.325298992494426, 0.831842510486068, 0.810294790743841, 0.556998261278609, 0.262964033730157, 0.68056620230445, 0.233653145442385, 0.456425364613112, 0.384566894511524, 0.538601438698022, 0.991703720961955, 0.75522040167783, 0.980454849430374, 0.234782640662848, 0.528559088552665, 0.0514360535669078, 0.756875470984622, 0.60197976627551, 0.857168709122209, 0.988277139258114, 0.929484472425617, 0.409514933438815, 0.00034146221813891, 0.540878322763043, 0.207730621996059, 0.21928358818243, 0.325806262734222, 0.0959494166075845, 0.747533661738303, 0.748508957393292, 0.543299433085109, 0.338132316524053, 0.832333661620528, 0.552572178025957, 0.957543116962021, 0.892833319318002, 0.356503617127004, 0.546401892991887, 0.346681751254949, 0.62280279065913, 0.796624685308396, 0.74587490154065, 0.12553623135482, 0.82239400675902, 0.0251505014285022, 0.414428880924032, 0.731407467972937, 0.781374000275963, 0.367285915131369, 0.744867856241673, 0.892267188231107, 0.24260338627967, 0.12959697583754, 0.225067882127913, 0.350013919000766, 0.287084613146176, 0.927488007179428, 0.0513137568167955, 0.592666755206387, 0.162898900217732, 0.83840576967303, 0.167560930211271, 0.502200615409473, 0.999329479025719, 0.355407151267881, 0.0470777095488339, 0.213660585120799, 0.397839124102705, 0.333668180380295, 0.229602531968198, 0.936120177502068, 0.683188784699983, 0.962113795722863, 0.437973177632798, 0.940336654420907, 0.00583432457237087, 0.610307030069875, 0.80107575696612, 0.232981533413648, 0.932468675920979, 0.763262775333742, 0.826449539702682, 0.573463588604866, 0.792581664120226, 0.329041195839248, 0.223461976125853, 0.312386342019254, 0.584523478528556, 0.829914141021523, 0.290462491547268, 0.402554392063122, 0.862057298433146, 0.6147395502164, 0.991187788329793, 0.203698875185622, 0.827209084946655, 0.675861618952774, 0.248949470039808, 0.475785626569533, 0.399075226726792, 0.599438249091412, 0.800522765803851, 0.105068771291171, 0.821442194126392, 0.841086330045423, 0.354506244400439, 0.430069462615833, 0.572239218412685, 0.700824730771237, 0.74246975698728, 0.757884150620785, 0.389128763569227, 0.429302484999837, 0.956344657500659, 0.572971367270484, 0.849722054218674, 0.276345316298223, 0.62232361515245, 0.588361684613539, 0.963468465806992, 0.085902694967251, 0.500498891603951, 0.521589676498055, 0.0901660021689635, 0.904666479690125, 0.88438893850986, 0.43898966680559, 0.781722612917166, 0.148465022943408, 0.619815927260871, 0.260623679394421, 0.445656211439575, 0.843999511474936, 0.196204918731252, 0.303851622777652, 0.483294567734286, 0.337812043871024, 0.798485829521549, 0.987487575189353, 0.159047554722395, 0.236879783226055, 0.70223663266789, 0.375471662697547, 0.973704902674984, 0.972305555688457, 0.643698049421405, 0.860098876854161, 0.401883398008086, 0.631930797977457, 0.98523663941483, 0.559477405987275, 0.933591915842209, 0.720343206000748, 0.484038516812342, 0.63903109268123, 0.887636865889604, 0.198736750098921, 0.39536627395521, 0.992175302495305, 0.402351616935948, 0.658856481072931, 0.901348123034512, 0.995381776517886, 0.65316328109469, 0.108436427855515, 0.036114030795356, 0.618091239046974, 0.567144361897996, 0.961964657805326, 0.746105469418888, 0.662516098435082, 0.52331331841201, 0.259894281484397, 0.961993766916173, 0.54020403751352, 0.0302701634197687, 0.696314463411127, 0.519716157920878, 0.0590305556644841, 0.890036233228213, 0.330202242514021, 0.22970119787112, 0.113948641995189, 0.310922713453606, 0.228432322135737, 0.651997167217003, 0.0661601245983268, 0.275431379860917, 0.281820237560867, 0.880066259726056, 0.444330358699235, 0.755914120907961, 0.603296375815348, 0.783265937404036, 0.113930638678426, 0.978563885159978, 0.848596675479549, 0.0506464976705434, 0.466201833790636, 0.325653274948034, 0.63020512499975, 0.230299163777674, 0.579884973281289, 0.603156320439897, 0.599879095832857, 0.448427880383915, 0.0354234606827885, 0.513814835038277, 0.407730161772637, 0.108046036615559, 0.459875628942314, 0.45088275367583, 0.551140335804419, 0.805404317007998, 0.700850072786061, 0.872235503921598, 0.052192152428901, 0.219681311405346, 0.459642010111367, 0.95853359612901, 0.790045356979438, 0.451874609042501, 0.333428186460866, 0.0590953126800806, 0.740905302893588, 0.506794524055503, 0.199925413139355, 0.427193551101288, 0.168690278712271, 0.751694590854687, 0.3683508416288, 0.941817795239884, 0.0171725409107173, 0.829055908778801, 0.626590974955276, 0.538746515157219, 0.650507635380061, 0.726629532306514, 0.0944885639205785, 0.87757393430766, 0.0143621439459858, 0.294302626846237, 0.17991487850424, 0.92629426819795, 0.0681804368270346, 0.581093227578491, 0.637151222525225, 0.651269264764942, 0.864622012533553, 0.0559525294070209, 0.81685517377385, 0.528922466328429, 0.694350560622442, 0.212404896281834, 0.543279946249851, 0.702520272153464, 0.956434557260469, 0.444542162873471, 0.085397830812645, 0.0573401475412575, 0.629450486099479, 0.796179066498458, 0.691191332858143, 0.345307862215166, 0.946816667161595, 0.520190318211477, 0.953813025197927, 0.0735956358081999, 0.207031948073074, 0.775027813981967, 0.914187821243774, 0.7825506477103, 0.295534196164876, 0.151845722314485, 0.847910522313782, 0.78485459109315, 0.270831502151317, 0.227810704816134, 0.321023217088951, 0.829561804399167, 0.822182194600976, 0.570682850481878, 0.571829636228848, 0.28601827249938, 0.699133559301209, 0.796257943281577, 0.441589056151223, 0.446215611691222, 0.465662408365111, 0.279039187274615, 0.675375318416595, 0.90366452636354, 0.908525898848375, 0.747196944130713, 0.260511507083546, 0.689637840074109, 0.131830665475512, 0.123500832319801, 0.190902853244491, 0.145732099496291, 0.585043615268716, 0.0733616900804668, 0.822326222185306, 0.722902973268465, 0.925858038017486, 0.492638598461781, 0.654882898449669, 0.890123477961687, 0.538525574469021, 0.282205164558222, 0.975957517879964, 0.0364255155254801, 0.326244573060312, 0.97301362389282, 0.365032625305747, 0.309149618895917, 0.120912384580628, 0.91576570404079, 0.135478205681502, 0.332117891828341, 0.897479892399884, 0.499648777847273, 0.615288242215057, 0.583132968088648, 0.698253926627414, 0.029332342754362, 0.527882687738647, 0.0320728464199854, 0.827142322836315, 0.339986206482169, 0.84671095888053, 0.246069567106098, 0.581491252368133, 0.937676879074775, 0.0477872923850662, 0.0539776591702386, 0.0206180341199909, 0.681478514196212, 0.598628526633235, 0.114030034515656, 0.796245350725845, 0.617850587139279, 0.0702135239869583, 0.0692789725680013, 0.13600738862009, 0.788891319657009, 0.0923984579238296, 0.237868792007306, 0.243647912616785, 0.104822212658895, 0.858352760228758, 0.698199766179566, 0.733742133263281, 0.650530624454913, 0.516270565774129, 0.326388135088882, 0.661776196778501, 0.117565484621756, 0.147817414181606, 0.019764663728817, 0.964291730299559, 0.970372902013312, 0.12386050714682, 0.46741002677446, 0.656694003304474, 0.290185566632867, 0.754536639716295, 0.558118051358972, 0.427792595548456, 0.2671940572742, 0.753736074133061, 0.898376325005631, 0.728444033957786, 0.406830134654488, 0.938315840253697, 0.255427454937803, 0.533163240608823, 0.95475487037637, 0.267747536440096, 0.250084620589589, 0.927672720239348, 0.0685823485554556, 0.299400386560865, 0.591583534156367, 0.203299145356336, 0.635883198886887, 0.798370257830259, 0.501701052364218, 0.650812135949142, 0.795954985556127, 0.233373768071567, 0.600838938434792, 0.112462378383286, 0.515765653195914, 0.837840690870234, 0.920790107998705, 0.498227929455356, 0.277611122264388, 0.652519960748294, 0.917298803800668, 0.509839453514735, 0.974191483802598, 0.197278942097314, 0.111184983788934, 0.297354293368622, 0.396418535561727, 0.420755683348492, 0.311475357650453, 0.693843170932226, 0.0918718332714995, 0.402088619037046, 0.295180804029962, 0.306496779026379, 0.105561130735275, 0.593827604564268, 0.282727516244758, 0.1552216209395, 0.000658667586112327, 0.283595417078381, 0.5508108749796, 0.870902191503684, 0.0422534253324928, 0.904721981466327, 0.130974086365166, 0.833728948123066, 0.800468374899062, 0.917880091998639, 0.137303658445877, 0.504732317557449, 0.404958480117297, 0.173572161534717, 0.575183612467853, 0.606217932572783, 0.214445740415978, 0.519932255056438, 0.989185809639387, 0.489915162413395, 0.694873211187083, 0.411421827387765, 0.0347767400997466, 0.292831555968535, 0.801441737777535, 0.346501990563077, 0.0833159448238722, 0.51110637094286, 0.366833407852287, 0.739479504887487, 0.524740163941995, 0.804520877458731, 0.816911179049531, 0.189471481685439, 0.123693456864363, 0.820996478991022, 0.637897620011436, 0.0161198018822152, 0.895954627639733, 0.515375057119979, 0.544521938346247, 0.606442208248527, 0.760436217840179, 0.8553468718769, 0.382868448331762, 0.0846491886172254, 0.733873298781272, 0.331989332624431, 0.839749534874645, 0.371723308016557, 0.828215270033629, 0.17651862896156, 0.129519861034667, 0.879883582417879, 0.0440791284676966, 0.686720018480803, 0.733772904933592, 0.437172437244658, 0.379838916907628, 0.979657439846574, 0.398993422611629, 0.440186914269368, 0.156807887544396, 0.326034284066419, 0.314061928673465, 0.894500663904255, 0.24702417926404, 0.310679015912649, 0.408868908042171, 0.708010894064053, 0.143637595059492, 0.871322066573022, 0.0831558967534599, 0.461737838368131, 0.0303890236611514, 0.753200532708854, 0.700042669529619, 0.214512246163991, 0.679904685523419, 0.557292680689929, 0.850678902590103, 0.558565491802886, 0.901773664047563, 0.419517704013185, 0.358128352789036, 0.488987517379486, 0.255962141002191, 0.929169268992396, 0.466756814210436, 0.254007850556945, 0.431218447278074, 0.702530011252343, 0.402329867986601, 0.181840078665688, 0.856251116146682, 0.584201224536599, 0.373578782679898, 0.221694546768398, 0.218994117711834, 0.5222324430255, 0.433422713487762, 0.741303988298553, 0.0704496323284114, 0.847333436397182, 0.679880234557907, 0.13665152189858, 0.858402188578687, 0.199834124667075, 0.607339921764212, 0.543044983654388, 0.162324692032503, 0.00565306812684552, 0.771485147165409, 0.76478755625534, 0.421069352361882, 0.0568131809736324, 0.585746995234228, 0.174155244434031, 0.72861066056631, 0.534291049941029, 0.253064287751413, 0.917056748285585, 0.758194529715416, 0.887030732228013, 0.0687981711037167, 0.18352819552784, 0.737072654586589, 0.696714852649722, 0.776992905706029, 0.501903419153744, 0.425496828002627, 0.611237130774301, 0.855771891361736, 0.670797108980223, 0.523592044711161, 0.298815410107022, 0.703969078651195, 0.381610875988882, 0.567684997801487, 0.887860955311228, 0.842949288995289, 0.89879855583219, 0.939003091446555, 0.815435155938747, 0.00135782878603108, 0.00309069755746516, 0.0874688163187515, 0.26072735120432, 0.0227986706409334, 0.424084818805879, 0.341064968909229, 0.541353933307704, 0.926169082583073, 0.29849944705924, 0.338084793851282, 0.85948010720679, 0.340478310765433, 0.138120403050086, 0.507798955352603, 0.85665647689169, 0.384313617882614, 0.695690926828769, 0.627904177790549, 0.450388154351925, 0.473617942098884, 0.949706353503532, 0.0834975617893309, 0.279828920000213, 0.447007309334704, 0.587571263995313, 0.877634138414522, 0.469100520229019, 0.437418475701721, 0.746184939974845, 0.467910465807746, 0.860827351057875, 0.466512012381487, 0.498104487122709, 0.487430654415445, 0.229468774748017, 0.0855522324090572, 0.0673833136089066, 0.888390934804578, 0.23316768566966, 0.861595759984367, 0.711735093007919, 0.872812993042854, 0.93800200691963, 0.1396892785475, 0.393900144851003, 0.980562829715054, 0.644794025985376, 0.896409779453934, 0.482230405436799, 0.0140930751890233, 0.622880344434795, 0.231095363111135, 0.527433962776634, 0.724991961357239, 0.607415791144935, 0.588366445795251, 0.433434840033058, 0.244172893507025, 0.428960353778007, 0.0101774555217771, 0.60882144904436, 0.957975188197358, 0.0954465235496006, 0.0355905633801477, 0.886235111325751, 0.246941365057497, 0.00891505625333699, 0.814920271448039, 0.140499436548767, 0.879866440284003, 0.0953767860905247, 0.35256010124864, 0.593420567787491, 0.585181638478593, 0.667681764184503, 0.648027115055705, 0.433369067338592, 0.139758759099258, 0.751929507975084, 0.241786793587219, 0.650459322478062, 0.857374060834467, 0.0843704710451489, 0.972089161712842, 0.0314599355591159, 0.835404550444859, 0.835713348170861, 0.049857501345593, 0.54588620889083, 0.943169839655479, 0.321473069869989, 0.806466803793334, 0.601398755252809, 0.789620465123038, 0.799185035044316, 0.049564765578211, 0.283198633840946, 0.653456823979965, 0.489655345328471, 0.972852237010803, 0.748489909977082, 0.567841149738807, 0.298964163457197, 0.256109781061838, 0.886563794411547, 0.446800862782801, 0.815987253296207, 0.0983373017446377, 0.859593453475272, 0.0276290081611751, 0.899156434256364, 0.899935500526176, 0.524106009617363, 0.120199512647135, 0.177794091822279, 0.706107594508614, 0.831359756387049, 0.034833820567729, 0.757838710385933, 0.957112178001164, 0.342870514996848, 0.638243709424706, 0.343005811970325, 0.216471396182541, 0.786200627030544, 0.723089959545472, 0.278838904787457, 0.58243142870448, 0.42100578136828, 0.0920687067543677, 0.0240273250067485, 0.491145804011365, 0.278267020151879, 0.339757164151065, 0.287349609515076, 0.170903235249478, 0.399263314281429, 0.697649568457898, 0.203676437129015, 0.66632648848951, 0.443066089408273, 0.433295265575696, 0.175239442394595, 0.19320231176039, 0.616421331646761, 0.269010607873387, 0.559677931615908, 0.944783991754306, 0.714471635748948, 0.679219655394007, 0.959380581307086, 0.775334311958986, 0.607726966322105, 0.948002359333351, 0.0596416235853658, 0.268712345242794, 0.986680043980283, 0.772206732094026, 0.475354299957795, 0.680899555553953, 0.416934754931422, 0.380149202893181, 0.213270389075209, 0.382937989134143, 0.0296677515101038, 0.472321065084073, 0.333372686843787, 0.975844804109989, 0.555443538619757, 0.846303637783572, 0.408063494401911, 0.462018163773892, 0.826306720846432, 0.991203199868745, 0.523947649263609, 0.925436889631448, 0.739021861521925, 0.567430056171921, 0.968777662942908, 0.824499419818922, 0.959609278997884, 0.646346065444023, 0.379573485440289, 0.476574680880329, 0.911889812500719, 0.0148538557678735, 0.156693847233962, 0.471567876529848, 0.542991694845257, 0.0596910323858083, 0.658030305628083, 0.889635303379043, 0.109632606115212, 0.437773114218114, 0.280230089607324, 0.985246312451203, 0.608758707334404, 0.253749206918446, 0.132614726848363, 0.545004891218423, 0.827803931470922, 0.837006271022824, 0.833348647034423, 0.203715434293916, 0.544417699248773, 0.874942721542349, 0.121000086169667, 0.856350768537266, 0.899776018962328, 0.21786512393647, 0.0769790849348919, 0.474215160481872, 0.835027465734663, 0.469393502041835, 0.413769939249495, 0.502746263849999, 0.125439959999655, 0.132285325427939, 0.870475365141086, 0.602950181301887, 0.265301726053799, 0.864802580056935, 0.0581094582239756, 0.457753885102086, 0.722209644099518, 0.33899915159608, 0.401219856926313, 0.526979752194827, 0.894235714464514, 0.778360658313311, 0.0693676026690024, 0.278784528621045, 0.379369834686065, 0.864672546383037, 0.419960369410699, 0.239877473491883, 0.597654516413679, 0.479404403601992, 0.898547501835201, 0.93470923791381, 0.817886833811117, 0.708908978901821, 0.743220570404508, 0.899709782516549, 0.0652398048396921, 0.335911171345859, 0.00433507260879828, 0.828095603149386, 0.507435377303176, 0.366162436149062, 0.22663971764437, 0.534831709596897, 0.289484866389565, 0.0683708974452039, 0.0849707455669264, 0.068339468134002, 0.409816499150502, 0.123381382309381, 0.443016592662236, 0.8989397369643, 0.353639001785661, 0.120178119552528, 0.569111025443708, 0.875032726365492, 0.348574977103892, 0.0419210049087696, 0.142339719388392, 0.0765928348764171, 0.740526589300887, 0.456525127319, 0.668249315315529, 0.699246075603127, 0.571357212390758, 0.628693341954442, 0.877763925090051, 0.662351739628039, 0.875413611874801, 0.467517018838227, 0.141336028336665, 0.0681344572325885, 0.714242406919872, 0.307986028182987, 0.671163971711563, 0.652420835422264, 0.531048838345102, 0.715107031762305, 0.504810623023082, 0.487999358505228, 0.497838309030001, 0.935975553324428, 0.389281682796525, 0.117145165727576, 0.240423553971239, 0.684908494629146, 0.839253615059596, 0.970144632898035, 0.215169612804033, 0.760343346418237, 0.584102849858033, 0.402951577571241, 0.510040337757567, 0.495644219085048, 0.651368342531714, 0.743706144622944, 0.301953469127232, 0.0896119352021238, 0.825965482484244, 0.389587008921693, 0.775304853268839, 0.179369095283976, 0.109361209717788, 0.905158338091637, 0.876351003922501, 0.999792564444752, 0.864254686057084, 0.0368775222838726, 0.544682442007247, 0.997616182183573, 0.511008154246046, 0.873512175935722, 0.0702214723365389, 0.987538736827997, 0.922713429546706, 0.564260919599778, 0.431462053674345, 0.337844565256492, 0.720723427847251, 0.0136555666633351, 0.374060192775824, 0.922685220392168, 0.546479648029272, 0.473886869752125, 0.496537318440195, 0.308971081219392, 0.950837558840908, 0.98199328087302, 0.513563103542578, 0.992589818472072, 0.455843702065251, 0.426040880892099, 0.213214778797911, 0.193249480429275, 0.832754245970334, 0.726637099604961, 0.529745693905775, 0.82908073749933, 0.511874431338254, 0.551961846254546, 0.213284894015896, 0.58782062632273, 0.142763225348141, 0.0522235278251505, 0.683307915305935, 0.608559478411928, 0.219655934781161, 0.406279942301345, 0.62990794476248, 0.555307624725453, 0.127579062038263, 0.169198331655614, 0.000999418116753414, 0.418162471197123, 0.488485678876445, 0.15986884315247, 0.666838577223759, 0.0179203690512555, 0.119675356519242, 0.952125308877993, 0.975853970932192, 0.0309145386684058, 0.493881843753437, 0.86272212813883, 0.24287616629311, 0.834270163718799, 0.813591192402868, 0.628973104075631, 0.00223819856083329, 0.379723939726934, 0.904405696492435, 0.680351665064685, 0.37882313072617, 0.631954710702334, 0.24326952247059, 0.571413071603897, 0.981728773776423, 0.849682018905517, 0.283444185689061, 0.682457664450218, 0.358124530982, 0.98692676996185, 0.0839862957561888, 0.2503058863617, 0.811346615315006, 0.0844231038780854, 0.531253963741857, 0.80062297914795, 0.738823241921064, 0.141662980302459, 0.437894257905023, 0.350380372413469, 0.47849749719769, 0.587403400124094, 0.145805695219521, 0.905330607244364, 0.640194248332222, 0.16293957144067, 0.565910591001657, 0.931616136977195, 0.783101883684631, 0.685687136978989, 0.466219284982742, 0.26031809566653, 0.569268171077019, 0.248770662888077, 0.319301591520897, 0.910802232216941, 0.885220040994411, 0.794589424165852, 0.925809807166877, 0.178839950775878, 0.517541148236465, 0.627005316409309, 0.913182345900904, 0.663968253838808, 0.389192822992613, 0.740007581416152, 0.817634859708731, 0.600344793176493, 0.0849970645027028, 0.922358001090054, 0.0535978153165582, 0.527024952677366, 0.118853276619748, 0.380142990241005, 0.812832533015109, 0.24409591557169, 0.884422659093575, 0.71264679475915, 0.378148438082143, 0.248919610995933, 0.252853745239273, 0.767243594526935, 0.0498618850472403, 0.685288567608909, 0.620278122151001, 0.746684629968725, 0.977255648345669, 0.383913512660615, 0.260205602313722, 0.877469536014124, 0.806095983510647, 0.461121079779699, 0.0909616879242212, 0.564268825273198, 0.187382889148607, 0.531689585894733, 0.355033321773383, 0.314783504755494, 0.726741468842694, 0.515772858133538, 0.790644897912478, 0.204492581358956, 0.678106077559796, 0.0524855426380236, 0.801172362362499, 0.678568638819314, 0.946008945084062, 0.0915581428748301, 0.908438370578178, 0.509953037053204, 0.614903565140055, 0.31607120088596, 0.0774874815154217, 0.850614066003707, 0.144526906948512, 0.370485792654012, 0.622391433081251, 0.99755190243137, 0.517344133245693, 0.990511215150315, 0.226534462555839, 0.398005186420214, 0.69656868951164, 0.0646407594535227, 0.747661582378364, 0.420400431232813, 0.811317420238023, 0.379605285770611, 0.319067820216784, 0.986051066726614, 0.718180905209727, 0.413183358569897, 0.0986302372429138, 0.73455912516641, 0.637306290353483, 0.0738418767360661, 0.120508166401978, 0.981596208916383, 0.496799421779149, 0.0224136544401531, 0.0538315489695107, 0.140873797601841, 0.893474311740186, 0.465820073116048, 0.560856738133542, 0.494456347383399, 0.0677854884858151, 0.897646542373691, 0.288565308881261, 0.269046814753517, 0.594194175430363, 0.475879024056308, 0.368311038073403, 0.6556110904965, 0.938200405804263, 0.620425185783994, 0.282840091614325, 0.205181251932987, 0.439134081593786, 0.0272502196608165, 0.876184335960346, 0.610092246274455, 0.203592380287419, 0.519916824790401, 0.0538243021501204, 0.862187431116878, 0.442934666053869, 0.548009157335658, 0.566860831789241, 0.680394991214306, 0.371378578093092, 0.078228709185685, 0.456350723650029, 0.0478438001602993, 0.73825699202679, 0.0380015353959609, 0.954244324359415, 0.742372334463615, 0.937449567040729, 0.513363986157654, 0.240904799077728, 0.259965278716868, 0.758974161342075, 0.993342763841508, 0.356705749924566, 0.75285784110301, 0.110048497112667, 0.597045229614403, 0.430595200320603, 0.73071801732159, 0.261176170571255, 0.0948076815835639, 0.450963489291171, 0.640074510267742, 0.132038817368816, 0.452822429824052, 0.652199598948144, 0.826998331558064, 0.308076908420212, 0.402363579005054, 0.884231316046448, 0.700579510055142, 0.241872295147587, 0.759831815719588, 0.29092573719343, 0.277438718955492, 0.00610780302659897, 0.374711141145558, 0.436932780731988, 0.304298468841678, 0.290860330609039, 0.242516308885001, 0.936683610022334, 0.860190240827037, 0.397228260437835, 0.479419104358615, 0.56499609309486, 0.489619389546545, 0.269811632673816, 0.989740142062215, 0.183675696266124, 0.861656724924023, 0.0326325721064562, 0.331958019586653, 0.748746995782957, 0.644366458181644, 0.169237962651665, 0.952205100722434, 0.543270235392249, 0.251413476155819, 0.578572309853611, 0.915476599033373, 0.895596526144395, 0.482506617380039, 0.442739977763735, 0.311754952248308, 0.0553137719816675, 0.753791773408712, 0.131947479760919, 0.355922119083384, 0.395870964014701, 0.885521227489635, 0.0212395486219491, 0.844080719207746, 0.288070643822235, 0.250342882536472, 0.488379635024966, 0.729036115345013, 0.202616096163795, 0.216276205436509, 0.97633451477942, 0.593235792436551, 0.304403578093467, 0.967702883195329, 0.895969830651651, 0.19002547679228, 0.0017993238791566, 0.711763763315268, 0.867726788909873, 0.118307909837995, 0.0390229767276427, 0.598194525429778, 0.604313505631328, 0.516432410448888, 0.00750740409328399, 0.688933363588118, 0.946020375394758, 0.873536456464217, 0.113276452523982, 0.35456934119252, 0.241936830626701, 0.560333309840618, 0.612727107549884, 0.300809090660725, 0.798141714195157, 0.795641626726704, 0.781092265910387, 0.3510984324146, 0.054297353899977, 0.708704616979225, 0.992928441303996, 0.162476485189887, 0.113566338672047, 0.912875425886543, 0.481656980372555, 0.851805952008354, 0.809913803104205, 0.18675954426984, 0.247199856745663, 0.0541888413672694, 0.608961456453878, 0.77723242923451, 0.511064089112722, 0.0277500884482718, 0.990385281566876, 0.500939923753605, 0.331997486961156, 0.173883139944506, 0.625636394848839, 0.575134832806326, 0.750985384525444, 0.153518799776586, 0.356787196602359, 0.143950800246145, 0.850607655245249, 0.33786974030834, 0.275196492845342, 0.00601097311711618, 0.801911968617498, 0.497408284738571, 0.537842841324121, 0.870912524621325, 0.722843568666448, 0.668085931674845, 0.178827792303708, 0.550495354874599, 0.959875078687972, 0.596021671986299, 0.808570708752387, 0.984532354863445, 0.885924447669938, 0.213837003113112, 0.0346300613124667, 0.451123791403174, 0.0137945567780947, 0.473710749132909, 0.951197090728795, 0.248952278705433, 0.386422800554417, 0.431433376426054, 0.83088648474904, 0.824646819632836, 0.452998612301112, 0.380560169947678, 0.925868542840634, 0.740843627451167, 0.737633073422826, 0.946915999012028, 0.510099705432381, 0.791879184896751, 0.452173502329059, 0.849198707548984, 0.390430829462498, 0.738376511506287, 0.976438557351852, 0.52329905821511, 0.429915301709775, 0.207158082586558, 0.323401379477045, 0.11087021836011, 0.375210054242112, 0.329903979134036, 0.342105172194619, 0.817118851307114, 0.531685314237565, 0.521122048508881, 0.77430970898665, 0.120262770779979, 0.625450151917525, 0.346648685550547, 0.334619385839875, 0.574616563380975, 0.863937641058782, 0.198563072170653, 0.672450963697648, 0.901831370980072, 0.199157181618953, 0.298284149829983, 0.496520270263375, 0.88990423168468, 0.501414589868598, 0.276995239103975, 0.53395817866685, 0.574249553785642, 0.412804401721059, 0.0147636628050427, 0.702764803428255, 0.506747151486991, 0.38126609842696, 0.0649315348738551, 0.358579960661015, 0.234251242532243, 0.20350405288897, 0.81377958316422, 0.393434593593473, 0.0535751708614938, 0.375051578272469, 0.774994177417674, 0.165297620870777, 0.912218897036659, 0.319205781809233, 0.329779668283949, 0.204235670095109, 0.767214736379813, 0.0699731489528251, 0.950033948013614, 0.158209784906886, 0.286445866401004, 0.68712911473695, 0.141149823637791, 0.512086650483433, 0.721326927294165, 0.928845135619855, 0.73210385001626, 0.749848020738123, 0.407319603105505, 0.239491695587883, 0.520850475083039, 0.219077360108904, 0.842387903898747, 0.662932909051294, 0.816235285166455, 0.793877600102571, 0.469105203551762, 0.309524979334574, 0.687579247513742, 0.986852223389968, 0.769933875855542, 0.829580846725193, 0.706084765450312, 0.595336418647749, 0.752874120082542, 0.496722203825239, 0.865132654012061, 0.0680276941294123, 0.968545940212993, 0.0987563119710001, 0.546977086455275, 0.402969908979534, 0.107040155400183, 0.724166170480622, 0.613682347960576, 0.782968192518063, 0.566620582441644, 0.811319158706923, 0.57677586024099, 0.944029193841664, 0.871452148941678, 0.507602207944769, 0.788823165012461, 0.4730306408997, 0.828801698488441, 0.322481588061953, 0.976146535227356, 0.278211040263144, 0.0728308048124289, 0.751223605872008, 0.831188620534169, 0.922338093331131, 0.327024293370015, 0.804069262570712, 0.538250333300567, 0.463294879079307, 0.82075025085467, 0.951906506191454, 0.076272889439772, 0.708671064187825, 0.234926443836988, 0.398895557719012, 0.268124426906609, 0.832513316595536, 0.995373810346204, 0.64975098275464, 0.703952831044269, 0.932303197968757, 0.687652548069517, 0.568353918669942, 0.380847529289527, 0.634579484810422, 0.363228621093965, 0.407619444945905, 0.368699723575791, 0.468398760390358, 0.503413843283962, 0.91053578239871, 0.206431022073808, 0.338603675277383, 0.574125621988357, 0.486932439429646, 0.262218985690441, 0.579593462953119, 0.878327698975432, 0.0609502223844922, 0.440876468930202, 0.0842580013374562, 0.563237615425729, 0.53931058029157, 0.768057900271301, 0.233090272198938, 0.587361742486469, 0.45897386682236, 0.860981521803687, 0.660836094448244, 0.353879166108453, 0.347186297893882, 0.253718102314479, 0.952528145373397, 0.298200989823935, 0.158406077778295, 0.361297056727092, 0.741629092882824, 0.705900010029432, 0.700892172871074, 0.00622578741150148, 0.374345601275841, 0.901495619292659, 0.318344828077252, 0.597082923214809, 0.297795182472319, 0.125014381285728, 0.388355668657057, 0.817687743973866, 0.981175704243327, 0.861989835287155, 0.083820882096006, 0.337711990337124, 0.236129166372143, 0.317805387932171, 0.98444840547435, 0.548250891256417, 0.749251259354427, 0.841851734724089, 0.166889706987028, 0.903097609289584, 0.105124127822219, 0.745093096440724, 0.729371937837905, 0.717469602110285, 0.133431767732035, 0.445788999516268, 0.50878723121457, 0.530490290818691, 0.85971704358549, 0.677724910719689, 0.805838258794752, 0.531242869458582, 0.955895838613192, 0.0666769445993232, 0.54151802744995, 0.281660343494637, 0.480900336365515, 0.684863781348805, 0.208258388796748, 0.608160988594066, 0.326176172376762, 0.880847202361808, 0.133394613360925, 0.102407789887338, 0.959116903133932, 0.152901999674792, 0.152538081095451, 0.155552588056628, 0.089569402926861, 0.454424561403129, 0.668895692810288, 0.831301681950567, 0.790235136913713, 0.712710917791988, 0.472597505041781, 0.708588152395748, 0.958059481020265, 0.505775562297944, 0.305053115317479, 0.789811183801969, 0.236386963587832, 0.234302833273045, 0.464699398608742, 0.619386801215951, 0.615329400167534, 0.12262390155605, 0.123793528327212, 0.284458991295818, 0.735732806826117, 0.411308102983038, 0.828981892712714, 0.935114116049739, 0.399066658162043, 0.0522113801305755, 0.571186102238472, 0.747669975681756, 0.320243558104215, 0.492934318642196, 0.221652754475949, 0.939273742656297, 0.482305289358003, 0.539996086873947, 0.221057099703633, 0.0959450018211512, 0.0601654164594785, 0.819508874942269, 0.771478304726038, 0.195695731019737, 0.895118154553158, 0.684300146224145, 0.656846481247616, 0.990381201487953, 0.0336917931210119, 0.424253270424153, 0.489983672806254, 0.583504470467278, 0.0832697475812942, 0.660154629977784, 0.0523052207703927, 0.556830534131996, 0.712025249134944, 0.487910625260071, 0.617600645905979, 0.213777574762031, 0.645658662061356, 0.380642291284627, 0.103712840186889, 0.377511518762747, 0.262860807508653, 0.241285612653001, 0.622924050374514, 0.52292776215755, 0.413237746494176, 0.217792035525866, 0.858553617120829, 0.861008253299634, 0.283939011564831, 0.615392848945124, 0.779489501843974, 0.954847311419425, 0.919603246226261, 0.384815703553521, 0.162643016406274, 0.796751419775717, 0.113817179634133, 0.158824298675419, 0.355828116252427, 0.847758564395726, 0.582781196988989, 0.586177665240051, 0.925837988949836, 0.575078267707539, 0.00997739496020689, 0.809377285817869, 0.608808359591158, 0.479913680376225, 0.268440158291067, 0.2580963839959, 0.481009535929008, 0.227341483977128, 0.0486022982418921, 0.169241814542259, 0.25844630775063, 0.197910057556161, 0.605693064183021, 0.82370364537992, 0.810614962671604, 0.802237016029933, 0.708113376406747, 0.859377854303028, 0.781084518001382, 0.203796851047344, 0.993343249930919, 0.0936245282198501, 0.650618543907727, 0.215176526117364, 0.24387310214232, 0.339687805074701, 0.197862340505416, 0.506833376635886, 0.95075798274589, 0.394607218303451, 0.584464840318106, 0.606537245952291, 0.714642867087912, 0.401536353789315, 0.858691021937975, 0.920484982161385, 0.750838889493125, 0.285591705194893, 0.796815787888053, 0.142765830129545, 0.504550190608271, 0.610687951481901, 0.70379673535153, 0.383339931019849, 0.728683436587994, 0.887284680027887, 0.0558483212688232, 0.138216342567174, 0.86306497872538, 0.421746482549435, 0.411314174537886, 0.959141340568648, 0.750245292973041, 0.980997114259686, 0.233515805625746, 0.096226761208291, 0.3845821992189, 0.500272550942528, 0.570255868872406, 0.976631596742213, 0.492855181015119, 0.400882866391995, 0.994990261820298, 0.260998008345242, 0.66532471074016, 0.964256616169518, 0.671151036378406, 0.299174856434414, 0.531126526815322, 0.001463175172423, 0.883788809402268, 0.404383857307768, 0.301205586513473, 0.95058159140845, 0.460648193206215, 0.28764751104173, 0.0846266385763814, 0.582181335168515, 0.153068697443093, 0.0730943061694905, 0.580564776830846, 0.287014752397109, 0.361920084401165, 0.724824693820157, 0.858311678125839, 0.347915686667056, 0.961745605325867, 0.953569051566716, 0.206037376830358, 0.768244859070775, 0.615531879840537, 0.918922487452317, 0.602537261209494, 0.702138228304602, 0.743674955204088, 0.385102276784062, 0.25150378917494, 0.036759737425849, 0.472122874247366, 0.645072328435394, 0.278969940402046, 0.517860909872254, 0.245666638599838, 0.297508174424547, 0.650474093241864, 0.891393305520928, 0.861106466709371, 0.209915076762285, 0.399093911604274, 0.887883023404443, 0.25652801190383, 0.966801604367428, 0.619157352979331, 0.165345766530944, 0.826199028826719, 0.655693101208612, 0.546452584058729, 0.251329283428934, 0.0401563845552436, 0.233375412573706, 0.361112936249855, 0.633455035012366, 0.986098096222032, 0.207156646144958, 0.75708394155577, 0.886328109630447, 0.472229859496957, 0.158913526072725, 0.810920785648706, 0.476507752716948, 0.116287532246184, 0.87572061278464, 0.635186379463005, 0.0972813490916307, 0.908439456239631, 0.0350160215102984, 0.0397496941127982, 0.988567891226263, 0.686181598460337, 0.376689192738032, 0.50432425279886, 0.763495679287094, 0.048874796779512, 0.72592765659386, 0.701328058598017, 0.458890825812503, 0.58229480895525, 0.339089801284837, 0.170625413666577, 0.39919340390056, 0.919777194544381, 0.226045437313252, 0.361006696151087, 0.324562012755833, 0.0835821136679928, 0.512665533605464, 0.832864824781385, 0.904613498537146, 0.723595912269745, 0.382995824300504, 0.298017055986365, 0.691711635714817, 0.880456926829212, 0.924548307619519, 0.0812527753704966, 0.482673014572774, 0.128265055730776, 0.252911159241006, 0.883961534752615, 0.19627691321341, 0.121353654736966, 0.543694993460822, 0.314620748221422, 0.382040915068017, 0.791539262851844, 0.83917947954026, 0.680236385926769, 0.416922007301322, 0.642889807506085, 0.214081451540123, 0.617270805784742, 0.67519114584572, 0.601023484608767, 0.346309504003818, 0.364401368261552, 0.171481076831545, 0.795361567918927, 0.492666912300111, 0.3546228545481, 0.775061199974026, 0.236804807064439, 0.844833449014199, 0.816526970819136, 0.846228364063407, 0.370186850863759, 0.383229531559576, 0.861334748060102, 0.463909192882959, 0.570548219905534, 0.695306808701783, 0.960917273175491, 0.546312979252437, 0.636576573431727, 0.570892489230996, 0.927112147047291, 0.863765569817538, 0.169839250866394, 0.178699029167114, 0.243504367623195, 0.751779355061302, 0.199134375117022, 0.982940511503982, 0.70963870247907, 0.175436145519359, 0.858297392360323, 0.909411483043429, 0.961663080193266, 0.57059970408596, 0.562879230350446, 0.176661204694203, 0.513679060375601, 0.548471915929806, 0.165277442620174, 0.493893038532447, 0.535117267476248, 0.198807159832174, 0.623168912334087, 0.0263153936543578, 0.318790961250692, 0.53299953441055, 0.326773737487972, 0.60219044061065, 0.361932727456263, 0.134920636585675, 0.913813503547356, 0.640558837291562, 0.658774142295879, 0.675330331309865, 0.744557714206159, 0.84217757318696, 0.516657167709074, 0.151868701454032, 0.380664274287961, 0.821019401441083, 0.1713643793428, 0.329975281417833, 0.966471986909258, 0.806292596514217, 0.22218792998221, 0.999773122694756, 0.0637386974643527, 0.425483118086481, 0.404338152089687, 0.400292883883605, 0.111922643905894, 0.424310773237634, 0.613545882666124, 0.988061285755543, 0.219900778839496, 0.354081077910303, 0.266241879188127, 0.291498033965174, 0.188389541510827, 0.0228596236256173, 0.449404181948484, 0.243640193394703, 0.868726544739881, 0.528610767442621, 0.914135168141916, 0.973930177058907, 0.58542595866233, 0.118975382673679, 0.926532588931345, 0.593560772481158, 0.883615304924768, 0.424475931087654, 0.60725738588404, 0.0707635824773016, 0.924772431183198, 0.642079416141701, 0.104499667897575, 0.700225073689456, 0.395804328828332, 0.0849047507770724, 0.214478867551072, 0.248798897538397, 0.226653137461408, 0.703004391018124, 0.754152585217827, 0.547286633863654, 0.553483419934252, 0.630572791703042, 0.985456950514749, 0.634278202982618, 0.60045561253082, 0.909186957080075, 0.570836633882231, 0.335419921247063, 0.957139158310899, 0.439923122280153, 0.601542593515676, 0.720262361570523, 0.678778341671463, 0.212759010103203, 0.0816233750456651, 0.274479407885599, 0.867518386466464, 0.559357067397456, 0.46462777102732, 0.430300900780404, 0.773985920020414, 0.653920421280792, 0.657729271666528, 0.161020904584738, 0.432376826943672, 0.505086084355522, 0.375332280929957, 0.480372228085867, 0.342421402767108, 0.777146440140074, 0.383943862389807, 0.711556426991232, 0.480933407391951, 0.729180222121572, 0.937559238963407, 0.517254402550923, 0.903068748626668, 0.218192642130913, 0.873219340172905, 0.0826929319308535, 0.465403416570693, 0.0219298228079787, 0.808280550035548, 0.179209500791232, 0.165392246853951, 0.181603467162106, 0.691436563311207, 0.213763519227956, 0.298104859773207, 0.768335147059746, 0.501150094646019, 0.909464806507279, 0.0578529030879383, 0.436751928242895, 0.572254745571166, 0.565067159777179, 0.823816567226359, 0.126098827482582, 0.30011677753376, 0.00212202483321056, 0.951107357890273, 0.766299363331997, 0.75130416218815, 0.138864837140935, 0.349319542642626, 0.151340560497478, 0.496720678257924, 0.80865225291219, 0.632868793441147, 0.688401497692848, 0.639570217855257, 0.729321726243568, 0.859845708085369, 0.626955379414573, 0.180591372909388, 0.573306546332578, 0.163566156384053, 0.90605173557551, 0.0773427851333207, 0.338534650141676, 0.580617547397133, 0.475235157492277, 0.805320486225532, 0.530778140974806, 0.227310331642288, 0.709484975196481, 0.14863255320485, 0.658116181223384, 0.633982547890093, 0.229306251465772, 0.182228015272702, 0.16635250773645, 0.149607091618741, 0.202747397742598, 0.954958901813773, 0.0159083388094849, 0.957514350171183, 0.0256922853702279, 0.971110663095709, 0.297595676758473, 0.525073079219425, 0.862339211479325, 0.896404903464912, 0.189010555411161, 0.660719557448035, 0.941230937873485, 0.975707125670632, 0.107935253017744, 0.178899228022263, 0.746551427132036, 0.04946852253195, 0.0712848819635339, 0.489125519571328, 0.849893821239637, 0.997041443878, 0.00439261258217538, 0.542607464059079, 0.861348176416565, 0.909139024470062, 0.845350562277612, 0.878872884638524, 0.746182237657744, 0.117489380198907, 0.509022256836494, 0.168831763959386, 0.831111503195807, 0.928010856738739, 0.169484428844022, 0.88373732004521, 0.387862894629115, 0.382569460225602, 0.271452923967019, 0.867883142835154, 0.74150245721089, 0.447872730127646, 0.709638857830719, 0.944331282902017, 0.174117873471634, 0.244596283964325, 0.640929124395263, 0.808612509899044, 0.853371006453131, 0.398117646920762, 0.115493808747709, 0.0802809517729943, 0.360467620079557, 0.828905549640201, 0.214609706276641, 0.791040323906674, 0.654688314176377, 0.0261464480617988, 0.785776285897775, 0.922563045272989, 0.492313130975707, 0.834011655118891, 0.13135368534335, 0.75978347118252, 0.92573618067275, 0.832707574402226, 0.259401160852536, 0.21302176736209, 0.522314781683827, 0.397357100825043, 0.47910978255359, 0.993903639578547, 0.604478457779412, 0.944908646327276, 0.49044221513849, 0.437946559772269, 0.772655572388859, 0.744066567298336, 0.442904067510999, 0.0530000525100436, 0.0878218441574162, 0.797985957677618, 0.655582372961194, 0.0323355534464184, 0.557066829681192, 0.719801554892191, 0.110408048956958, 0.216647376280279, 0.811020215616459, 0.138661882869267, 0.881899200733092, 0.923556061690306, 0.0127555628361417, 0.377159253042995, 0.167811688977599, 0.54022279574486, 0.10166243492783, 0.0392677458897454, 0.933229111773545, 0.971591882864217, 0.360928020047589, 0.644205403766746, 0.0679473020936823, 0.207911996415632, 0.0396038467757357, 0.469359313223853, 0.150096722662633, 0.99130692118419, 0.427062347759695, 0.955372102971021, 0.724247033520084, 0.580891712363774, 0.54025790742078, 0.705441191564081, 0.00502888330112106, 0.782515778836242, 0.926859573385509, 0.00829565739263971, 0.824628342666394, 0.76733586802788, 0.997136895348686, 0.227653072058219, 0.919542206194465, 0.64199930552178, 0.105320182975915, 0.268160913711683, 0.763843764539584, 0.805510155575541, 0.104252997601917, 0.469758776292843, 0.219061890891174, 0.922707906542724, 0.320322632016148, 0.857543808780781, 0.259846667655122, 0.878062942706618, 0.188267616829231, 0.759193837349818, 0.0316893001122069, 0.642339157082965, 0.566870912760646, 0.376409545450067, 0.212548379439868, 0.792156839532589, 0.145443480124611, 0.489142450390656, 0.0128457468737627, 0.186612024812195, 0.48523045694977, 0.838226076700295, 0.14105696859946, 0.73221698741175, 0.691066915983427, 0.0344926197846326, 0.48885726360846, 0.971390410156509, 0.112451379350698, 0.743213774200817, 0.638541371339421, 0.594182912969943, 0.49862251273625, 0.567852659287788, 0.426504251336355, 0.07623944235383, 0.290586653457929, 0.561335047322794, 0.63333346832172, 0.930775859048123, 0.977769157702234, 0.0935965460059537, 0.661734944417943, 0.602772899485439, 0.473817863220927, 0.356256342496277, 0.47557844163322, 0.671022018283348, 0.959645196993795, 0.0890842655064104, 0.797742459847051, 0.590775746140886, 0.912196754252076, 0.101129318887619, 0.293294642967131, 0.0515882053839337, 0.504128180597837, 0.768375751898677, 0.282985376214109, 0.225360245794357, 0.331289642625589, 0.453251351408243, 0.7373846381813, 0.509885654703147, 0.382514191853397, 0.905483216847227, 0.965257748969727, 0.628267352552455, 0.132031170054907, 0.618301778328654, 0.383020104064703, 0.991193747568799, 0.286827284759212, 0.706191429589364, 0.535205897505445, 0.193211038592101, 0.689435100924171, 0.0504550031686729, 0.184434115699354, 0.0456583234773829, 0.885041502041251, 0.839794385798114, 0.118155245153682, 0.410414853697837, 0.12022858999065, 0.572092522260723, 0.949390134629152, 0.25638515979688, 0.989865435151316, 0.349808041044371, 0.20852237389343, 0.665827160044803, 0.97334470672087, 0.622699821538518, 0.0635378925573686, 0.373509533241305, 0.166251535436548, 0.231278078105778, 0.0522086332938522, 0.901756316713514, 0.793291699901326, 0.373014324844645, 0.832054705376542, 0.753834556589773, 0.621863097261786, 0.394093180849151, 0.359277961450212, 0.0888524608922825, 0.341677003639647, 0.548670594067366, 0.460547461088786, 0.64545237343559, 0.513520841267514, 0.814426468563278, 0.0971832059869966, 0.463713560645511, 0.589817489604743, 0.187172070539839, 0.611330146637951, 0.051941981368346, 0.575726948453721, 0.842344701828773, 0.499725973621345, 0.439024887659214, 0.149057083245688, 0.0282794508957929, 0.756669787984348, 0.79610623594284, 0.29355559441744, 0.115206746018031, 0.375091630987986, 0.82889374505677, 0.841776581954761, 0.665238451557799, 0.960139975886254, 0.943117892210703, 0.112699492380728, 0.648287449225586, 0.48080405167944, 0.0665206919491224, 0.897771225532002, 0.497230128936991, 0.771303428858631, 0.0603622294239692, 0.262457464904029, 0.651069368702479, 0.133603910449774, 0.638545736824786, 0.384943094967874, 0.7656981256747, 0.652915682156136, 0.381489065282516, 0.300018582740438, 0.340139555506767, 0.918926536712417, 0.456266813303239, 0.442496952598123, 0.454185819615522, 0.94528200918664, 0.219118952550175, 0.882402795931734, 0.0198754228247094, 0.34176488323731, 0.766027445694415, 0.342803701519393, 0.618806404154405, 0.453021162091896, 0.0101626026279142, 0.599081167470487, 0.601568348358472, 0.649417450961366, 0.342720632582221, 0.493299255574885, 0.70177394525353, 0.88780260142949, 0.0550578461561846, 0.0983619505457304, 0.649782785472762, 0.764070891236327, 0.98795894159784, 0.125324791415593, 0.3644771752026, 0.676230090282253, 0.375758171217492, 0.863458491467109, 0.291976750986029, 0.133474990584297, 0.672650514427429, 0.202584980871927, 0.868515198201092, 0.751157321899609, 0.419379649992969, 0.000231065199416736, 0.149463732367703, 0.273834301544268, 0.87242501896288, 0.601251143911812, 0.321188452831265, 0.284293074456312, 0.435315805784207, 0.903759051181981, 0.925105644987009, 0.505292452205831, 0.62758180927003, 0.71926392682241, 0.0239128813596552, 0.5749325019969, 0.0465344640640383, 0.422531388268483, 0.467734261291993, 0.0226279867559067, 0.0650740424646463, 0.923956335725144, 0.53414286394427, 0.366796468997375, 0.363946179540744, 0.151373665826451, 0.149608598632666, 0.350802091637144, 0.335966095668561, 0.784027846268716, 0.486739211626018, 0.464798012873766, 0.13125255159679, 0.886391470888295, 0.674557425628767, 0.835159638023036, 0.656487934875432, 0.983916976496475, 0.979789871891285, 0.250153013597979, 0.62457070354219, 0.728242423153538, 0.498170315267308, 0.849828294959274, 0.190918203621887, 0.124143787222168, 0.00279042132911189, 0.152952789981843, 0.534160824643418, 0.510633605136478, 0.38521587987977, 0.310596453716413, 0.00355544745321035, 0.815242445313671, 0.638418458166358, 0.448339380701919, 0.244085770261907, 0.803385085996905, 0.823970887228072, 0.852190649398636, 0.467251755433503, 0.97069897317612, 0.841244015022289, 0.0785482063545127, 0.237598887672087, 0.817570881268703, 0.405828714806259, 0.466312220637624, 0.951536218081585, 0.965005305843558, 0.765284866775821, 0.574533590173997, 0.915925025450221, 0.49543240879036, 0.166012377080293, 0.325997669971674, 0.296436106009726, 0.558298491530906, 0.0674768030510901, 0.0689778462108583, 0.166784582313363, 0.947438262628722, 0.811088417621082, 0.710455692980488, 0.970245561677518, 0.998426789341871, 0.987454806749042, 0.150086578018891, 0.958478804453122, 0.53045904923442, 0.0740865173899161, 0.311820903915726, 0.895170779995606, 0.834768311582722, 0.00234875567974879, 0.64020984616491, 0.803182575087435, 0.245130755184197, 0.0641222317658926, 0.263148319353776, 0.102720361262892, 0.483719024833911, 0.41888446265122, 0.381289008027337, 0.886765115012407, 0.420557514837392, 0.283847869017151, 0.0481811440107028, 0.219165982554382, 0.23917601408698, 0.02925848703723, 0.702310918358637, 0.00763555477239597, 0.610920680279354, 0.408089537451172, 0.248946674813356, 0.652459724145901, 0.320277088369962, 0.103672124495835, 0.535564894512648, 0.164870242477034, 0.883438702353724, 0.666457012048093, 0.847742491914359, 0.762661063721127, 0.807013774619851, 0.632952443655251, 0.710427307863708, 0.688663539117781, 0.320947861137131, 0.531648494500774, 0.873193957609839, 0.0545396739097971, 0.500400775515181, 0.432763214528361, 0.904282491633212, 0.630183351426678, 0.983036222076956, 0.585200339522547, 0.840636517595249, 0.468813780579663, 0.545222329693841, 0.17910412094158, 0.634461696770441, 0.96295877536838, 0.534013751451731, 0.479614252288398, 0.793666083025936, 0.0927137188642948, 0.880799542861342, 0.00387353551348724, 0.511529092199426, 0.678463634320796, 0.565747646843948, 0.478458646745197, 0.320512844148039, 0.601579798458235, 0.913152742372502, 0.68251781671711, 0.946733471115376, 0.0990890086376651, 0.511028924158354, 0.11012834018281, 0.545261877330557, 0.688785898972834, 0.147416988338171, 0.77756103760802, 0.39905123310678, 0.898299948178677, 0.30704387392557, 0.0610511462357189, 0.219471144449904, 0.0828329370324061, 0.950395335131906, 0.0163706608566853, 0.114668433182841, 0.0124099459694967, 0.216227329276232, 0.011427528814081, 0.64244926558009, 0.516990939531608, 0.245542144387298, 0.193747340053374, 0.0908705781480208, 0.368441665384023, 0.00779291063747434, 0.602701319215812, 0.478854744051061, 0.308117688829788, 0.744444447627629, 0.839349277568775, 0.262431920229879, 0.514238061322073, 0.4467741275926, 0.341211145567738, 0.839141805139886, 0.982492501505978, 0.626464501073332, 0.181275170605328, 0.123017016857453, 0.579968709639967, 0.328535803427897, 0.26818637266786, 0.550239103808507, 0.180516896172902, 0.678491869414882, 0.0556884902568199, 0.0340571844812154, 0.286518454768013, 0.0773900470329255, 0.900572290255528, 0.846605947069211, 0.395698472582128, 0.169215265314684, 0.43045231333246, 0.41621813688281, 0.728763483255897, 0.406473678914791, 0.95180745705255, 0.911985020914832, 0.95141431651076, 0.346001202788865, 0.290244208783556, 0.886700762839575, 0.210030796707028, 0.130876547760118, 0.520515887325233, 0.905462749730804, 0.402530235597016, 0.215761252172513, 0.0787394832466347, 0.933059861765135, 0.602871560972297, 0.377492173215029, 0.664931441200008, 0.792190264444163, 0.333492129876999, 0.6926588643338, 0.203815677165533, 0.958713595889891, 0.711832456405164, 0.166906592124375, 0.442777173122209, 0.632993568362347, 0.929967171592209, 0.529330969397888, 0.626474423948033, 0.680818922727965, 0.923197523158096, 0.152834142252458, 0.405720793714528, 0.312475998842778, 0.693898958369993, 0.890687873398548, 0.4906709662866, 0.805823509416861, 0.326438909311377, 0.549879145125724, 0.388784334890571, 0.896829007264381, 0.676120081005331, 0.828396948330795, 0.110088847733172, 0.279227334556401, 0.767636165697467, 0.216056596066409, 0.0340621281717747, 0.436551897742306, 0.936863710408633, 0.262088962808782, 0.569745499842289, 0.359553044905957, 0.0268386289622559, 0.500419192257721, 0.827014907580009, 0.258982420521299, 0.0458850066372709, 0.246471604097961, 0.660732144629048, 0.329409691116965, 0.659496247567225, 0.0130041242672279, 0.718067856677326, 0.391118812134764, 0.0335052596625038, 0.406015402500669, 0.716307636801179, 0.921336414781853, 0.984015743254487, 0.983420872823396, 0.896306092986061, 0.865703209612899, 0.800960972217418, 0.554978851487549, 0.418872147516562, 0.127122506592021, 0.654620743083901, 0.863964912060124, 0.274598434868982, 0.840163937389387, 0.0707552442693478, 0.378792075061979, 0.26816943383197, 0.152921178784748, 0.63099957992261, 0.316374816239219, 0.959111852529902, 0.498673989242012, 0.73860851672542, 0.0127556688922719, 0.605353188516464, 0.576451321456248, 0.807378820903335, 0.654968061518584, 0.878228395135494, 0.902372925442161, 0.152232856672578, 0.192580086806526, 0.790975648641478, 0.0607051042239655, 0.389827192453456, 0.299965854261437, 0.734179906575914, 0.104209253757311, 0.792575487554153, 0.782728942751883, 0.532397693218747, 0.253352341821251, 0.070954585011654, 0.62580333117104, 0.0246814701289989, 0.0620424925095289, 0.129611982278315, 0.450614072287436, 0.672335843922579, 0.856110833478116, 0.498445424254591, 0.048784504036128, 0.313832359872792, 0.641631167720519, 0.786386643614576, 0.289150493176199, 0.497868010687049, 0.818434615976952, 0.595128292529225, 0.5364246089182, 0.330873140619941, 0.411690226251413, 0.794006240687227, 0.343208220509353, 0.462606735967318, 0.367823667799665, 0.679570619604256, 0.567779031870686, 0.651776846130543, 0.491113748307781, 0.398458234911806, 0.477484208997986, 0.0665876229516056, 0.411028487183606, 0.969091958454476, 0.780721030890361, 0.729017888682995, 0.765650616008666, 0.756580803862821, 0.843270317045169, 0.770156849951834, 0.978660928731569, 0.111359442553858, 0.396076941209457, 0.492060650839531, 0.258094618284834, 0.0369662213086762, 0.974381323033703, 0.72642597902203, 0.14796483038576, 0.147894557371599, 0.704824099106688, 0.380993522402518, 0.0764114441659032, 0.410840343032633, 0.142990554089145, 0.798920120530614, 0.930248163138044, 0.00471947787680982, 0.650040581422713, 0.67853246105249, 0.253622841789289, 0.843180912870887, 0.293958774213842, 0.0268603841526058, 0.0933071628967971, 0.797884972292497, 0.711402626220883, 0.783406783842256, 0.623922644020696, 0.825415460073693, 0.0350225850889124, 0.405474972293269, 0.249668755283506, 0.480896841642134, 0.880839127459738, 0.280684817595266, 0.599142509308593, 0.02622613864761, 0.155198666268626, 0.833910870716356, 0.194893367249896, 0.829788035104075, 0.338074859029462, 0.67111654953098, 0.0523663190627388, 0.734309066385156, 0.499481777820457, 0.943299937460375, 0.289772416367565, 0.376560426992884, 0.113755361650924, 0.964850707268855, 0.43251156721242, 0.0845632009931329, 0.716676412592981, 0.506787379534883, 0.328087741248748, 0.753524141094147, 0.836005824603354, 0.253716352276902, 0.53442501679032, 0.435165232964158, 0.157704145695579, 0.600481203671839, 0.937450760032403, 0.107758578775102, 0.899981434728875, 0.550464991190469, 0.427356686821267, 0.152380830362768, 0.247546002513464, 0.447368793703639, 0.532783437696949, 0.354651134298573, 0.773114812272379, 0.881681142450407, 0.734092093476126, 0.406432206360085, 0.604178745602068, 0.641105051660087, 0.127466836015375, 0.496191996945931, 0.310469355507002, 0.57857446388477, 0.943608937038841, 0.42693960927636, 0.0331284411700892, 0.929433878785966, 0.924982251332918, 0.358306919554313, 0.259989116322472, 0.786859645603551, 0.511582243380335, 0.562527153389906, 0.684793764143528, 0.0923965953476921, 0.872578967039445, 0.942937880219704, 0.0965942366019155, 0.845882505221427, 0.909395614856024, 0.0113411405597612, 0.523682703745454, 0.650345273658944, 0.385145350705933, 0.64930204945739, 0.762853479752345, 0.575685981383182, 0.631919148712233, 0.278204003887099, 0.839836422799811, 0.426834956239074, 0.631622378349663, 0.833466981520768, 0.27018546361456, 0.400800736172277, 0.554254592413654, 0.44386535683262, 0.090384122028611, 0.744381309987074, 0.0326154493029029, 0.429742543433021, 0.0372622794740213, 0.97579775303611, 0.522339908013089, 0.909629978597814, 0.383247921340015, 0.884452236904782, 0.255017609331153, 0.909049841942801, 0.894560444751255, 0.398517160990578, 0.625020001836008, 0.567597286185841, 0.894511885444997, 0.214165803867894, 0.00385915886072064, 0.88058146103171, 0.235121635040512, 0.244864622347078, 0.640917035139739, 0.304524481208468, 0.825620811888813, 0.883688710947261, 0.945372957985459, 0.39079533840462, 0.801320064711665, 0.157113287977468, 0.625165396779904, 0.698985352759655, 0.0858689331204349, 0.53117957278844, 0.888566522140567, 0.263666008189626, 0.23478552530433, 0.839657227724237, 0.495540029725937, 0.152366366412905, 0.230768155809496, 0.657953957084698, 0.562948381240076, 0.291828737648848, 0.622304984715124, 0.715905408292378, 0.280731034080199, 0.412273158549075, 0.362205713414129, 0.781392022002529, 0.135487151301981, 0.90206951454967, 0.28963608587814, 0.499557926109802, 0.783590388178023, 0.677062062697606, 0.1498117300189, 0.696619479617911, 0.129013043122745, 0.945946287415715, 0.886411849242866, 0.51500461293742, 0.679408745931636, 0.976793357833632, 0.12546035664003, 0.752243051246404, 0.827051909456158, 0.781430326910244, 0.190882666758224, 0.428641340603978, 0.0144567471162916, 0.325285488719696, 0.13470202862637, 0.450517462084021, 0.572274930178577, 0.792023281983178, 0.419736609978054, 0.532536544340956, 0.925704392028016, 0.899081837742182, 0.544833398468742, 0.901123950170368, 0.0518267089040074, 0.808602518680107, 0.334904579377686, 0.228683117248675, 0.822401807645737, 0.348234578993427, 0.165470595368233, 0.0281335680082572, 0.955369765277556, 0.6802907117056, 0.860562114724664, 0.939093515352434, 0.680193964448052, 0.917423748447434, 0.256691670901158, 0.885618134429019, 0.920042790118182, 0.300063440036683, 0.0733909466937926, 0.767398734960137, 0.084952215865014, 0.728764079592672, 0.44788964358771, 0.651248663216932, 0.169501870614945, 0.53144924135958, 0.63380085819396, 0.0140960041826924, 0.470371335231002, 0.886325971168106, 0.114028435678071, 0.442541151650179, 0.659547888161553, 0.294772970267685, 0.9503684015149, 0.694286247565555, 0.206806527020916, 0.554762325956739, 0.879278484511919, 0.557857690719796, 0.752334060151411, 0.894900792229508, 0.84184417306489, 0.130856682645948, 0.189153933851989, 0.153639830892556, 0.0289020656650996, 0.00908489849181371, 0.596451741972037, 0.609049324383808, 0.918923432010552, 0.733574443373122, 0.301146952100848, 0.49557447903043, 0.258163023498381, 0.732854363876553, 0.116760751041515, 0.746041563257602, 0.809789107094191, 0.745233681211662, 0.337143314113377, 0.584324980436338, 0.468952104955939, 0.0872650344600689, 0.828717426160072, 0.685944708438845, 0.267325055095755, 0.969483564592453, 0.183774202665835, 0.299940922944898, 0.411184890688863, 0.236489187115421, 0.195055322681085, 0.705379279645915, 0.180548253665843, 0.522332917921855, 0.296171823286006, 0.462781888284344, 0.925232416679281, 0.215889142296026, 0.00100887850701248, 0.906606243116842, 0.680041945628298, 0.514951805346954, 0.522068881119071, 0.102918579221459, 0.996880488188375, 0.358968819374506, 0.625242598780211, 0.393366521225294, 0.00766244550040407, 0.545284934032101, 0.509110374446584, 0.246778920321399, 0.0453952746616395, 0.84172974476101, 0.048249917605632, 0.316320497497147, 0.783419430439279, 0.972400059712246, 0.586464080739028, 0.778043849281618, 0.727697443692381, 0.650986797126378, 0.664615020149917, 0.938779800520264, 0.53508126571778, 0.398440118410103, 0.670457661455949, 0.440534471776437, 0.13287394477806, 0.439203747941782, 0.547642742273276, 0.395136375845411, 0.39827199452513, 0.751348651125886, 0.522350346054575, 0.490433033651247, 0.0886789345424768, 0.250851633092814, 0.447559022780024, 0.637960896081718, 0.709445291503946, 0.992619809281419, 0.9321948700926, 0.0922294963888484, 0.953539972829461, 0.162795658817501, 0.970455087271826, 0.597006665267872, 0.240227119123713, 0.0702951614777546, 0.300041215108974, 0.813544823896987, 0.0767098742008153, 0.354473095439048, 0.132010874408976, 0.158179630334593, 0.0621470516173679, 0.701843478892324, 0.0864816970794666, 0.616787239562239, 0.173771583224603, 0.651401064738548, 0.498696405983114, 0.284510688567093, 0.830559738735304, 0.81835988248334, 0.938169403753504, 0.000326096340118931, 0.640389224633762, 0.00735569277648584, 0.106421106692614, 0.106794413254953, 0.3671090438542, 0.239607700586711, 0.346140201697452, 0.249619835400228, 0.387064314143913, 0.421038370433554, 0.640077289145426, 0.787552972956609, 0.269994423620667, 0.84398236072591, 0.740468380221871, 0.826101948913986, 0.182192247374924, 0.0654361827556491, 0.610350262112943, 0.701552809275043, 0.111617712753208, 0.0958237324700979, 0.597833772320779, 0.812232610355035, 0.81457817403632, 0.0894369651433987, 0.731278705325746, 0.903856576224039, 0.452231787215498, 0.0706876763586191, 0.241278155067504, 0.731865059823696, 0.0404916156591709, 0.424522762856597, 0.540215087042906, 0.953827837894086, 0.208905741989429, 0.116332096224402, 0.646220067892793, 0.108411089785024, 0.983496924216579, 0.248344038446042, 0.606355720132135, 0.816695272780671, 0.830054212606881, 0.489039332451187, 0.76072825611808, 0.915107935348786, 0.900974956264358, 0.214237510073561, 0.547061105978206, 0.78470948099561, 0.194441094252232, 0.746889151997501, 0.475558224441359, 0.583258581227704, 0.260548081851841, 0.0848224596308363, 0.298132851184017, 0.91712935887688, 0.470518192745995, 0.269467553234544, 0.76297047694478, 0.772172105182967, 0.0212996241096677, 0.879986192497539, 0.798170184137338, 0.324165238549624, 0.669044416639185, 0.29629386335108, 0.92995204367677, 0.281960434914488 };
double inequality_constraint_matrix_data[] = { 0.130651597357995, 0.0513557326803022, 0.627506476971335, 0.0290875518506981, 0.136194609644932, 0.694555598689595, 0.515678957454232, 0.542582420314772, 0.808483508969732, 0.793686029689862, 0.501867087038036, 0.276632057868203, 0.119664941102955, 0.886597760157759, 0.970288991112123, 0.942526312878199, 0.638128132512361, 0.0905901520413662, 0.07470811489881, 0.18245093620079, 0.0316876620019078, 0.724936308045872, 0.144158809946439, 0.635926816608026, 0.789847558684815, 0.566253126608988, 0.377413923888473, 0.821586115511533, 0.304880829506504, 0.319373097536832, 0.78497989138293, 0.503736437057276, 0.260996731933566, 0.732475524156582, 0.162949965018604, 0.921075222102221, 0.222223462665215, 0.0836087173480035, 0.0737015537421595, 0.769551657662049, 0.817677026694343, 0.7404255352805, 0.758248787177962, 0.961194980110539, 0.466424962099562, 0.786996488000829, 0.422568083677206, 0.943729958429594, 0.00131439798824418, 0.981267907154657, 0.570194182308453, 0.346470554408469, 0.557503053109067, 0.299784702501575, 0.159072822525737, 0.665256285872595, 0.684202783274851, 0.792409362348291, 0.348624437659894, 0.250074733353068, 0.345003385293862, 0.328632592865502, 0.927485532892117, 0.756072432453704, 0.288238633424684, 0.606179876148598, 0.766029364237188, 0.846152344274838, 0.901981191878953, 0.595707147578878, 0.0685361446160678, 0.218043374944113, 0.869408491537521, 0.414217854149326, 0.661209152001318, 0.783484840322146, 0.247886007622878, 0.554425288853058, 0.229585395135581, 0.00691276397036733, 0.766639397234634, 0.0218332495144079, 0.393103434605689, 0.252540155757166, 0.204216921394176, 0.66227946721661, 0.914742080029946, 0.0069025477433895, 0.746423200362149, 0.799670794169673, 0.907798941071005, 0.97457393538529, 0.119863491132223, 0.518969763416861, 0.821974811688185, 0.637022798742926, 0.953910376899607, 0.946926373226851, 0.966605245331986, 0.0673441884719516, 0.437534793660125, 0.320839942810518, 0.13405205840044, 0.1345851502257, 0.805947660318698, 0.524764264394895, 0.944256970581438, 0.988340076638724, 0.409894179847122, 0.371187877142478, 0.226854092445029, 0.446030958823852, 0.266221437567159, 0.459096393967459, 0.432905328118769, 0.259623434291208, 0.133714508366169, 0.419228418590875, 0.506860511242541, 0.3243245550001, 0.684688800419701, 0.443093511026668, 0.435663079837528, 0.793023515561676, 0.815556369684112, 0.752112653709533, 0.78925594132128, 0.501267453084442, 0.555181922697123, 0.630749738384446, 0.0979891648463104, 0.245676181936889, 0.615725142289619, 0.304959957607017, 0.766967602751279, 0.267229092367089, 0.0395063042516288, 0.296561777532987, 0.556364975093691, 0.969074471088058, 0.689124711134179, 0.717881398387593, 0.559029119569936, 0.533354207803659, 0.87572400256463, 0.393098592724503, 0.458066938000918, 0.208245204973289, 0.757273162983819, 0.546692028178502, 0.357388192967744, 0.700997789695237, 0.109222496688655, 0.00661076596420229, 0.597302079801353, 0.65917606662187, 0.580006430551879, 0.909951610123007, 0.636006152111755, 0.525561251235387, 0.259623080110499, 0.0511708673569459, 0.73195012915998, 0.164291390918458, 0.280397005689543, 0.25942744797746, 0.547095959676547, 0.541268467485969, 0.788113184048992, 0.869605555315031, 0.787540530159212, 0.969427855895745, 0.180468109688552, 0.930598962817579, 0.0451518760404848, 0.240645885287937, 0.00885503009860023, 0.671593534312873, 0.904813230017362, 0.57241631161166, 0.155461290328972, 0.502367433727028, 0.567732878272974, 0.188271775739867, 0.32419646346704, 0.716037342838811, 0.552928708571216, 0.142281739817257, 0.380363559412667, 0.396569245211042, 0.576741695694105, 0.0194015027111201, 0.577580061269445, 0.932196092808549, 0.106872780238886, 0.732149758734352, 0.970522786956106, 0.608885538697176, 0.719665612178919, 0.302752325729782, 0.459022394078519, 0.0480286131823641, 0.385352105175205, 0.361716380248139, 0.287584055249314, 0.816712735890955, 0.450528094868533, 0.806632520363243, 0.790172935024334, 0.282957873186578, 0.0683092386250433, 0.0549307282223953, 0.637522639339998, 0.4242855812171, 0.905531348068602, 0.417320883626846, 0.154058327516862, 0.539999263770361, 0.937091367960309, 0.660955064915145, 0.394657441563815, 0.258990542158967, 0.847912159970545, 0.945056074939906, 0.376999837283115, 0.0672801845433559, 0.181581768675897, 0.575747615215569, 0.185885031598486, 0.291448355951266, 0.461664106934666, 0.346977441011452, 0.318165716539783, 0.459914579612446, 0.235894920096525, 0.0277520375895161, 0.658453455956023, 0.158803562723388, 0.802662600545296, 0.408579771834726, 0.32739187888055, 0.746014728567984, 0.74635192034512, 0.173954964110564, 0.117544801822912, 0.174037524385108, 0.627409163987095, 0.841890255550603, 0.510082854851545, 0.165763743909395, 0.71430528242953, 0.907037296307911, 0.218537741114588, 0.87096144790796, 0.211835943615888, 0.83666494027777, 0.859301689176415, 0.523377472974036, 0.47736279426484, 0.889883132710091, 0.0650762004173646, 0.509456125157452, 0.620797947105788, 0.733567439827716, 0.22999623088413, 0.021871875699117, 0.13898888969974, 0.769507446909155, 0.969765180908154, 0.386794416028229, 0.993427789797032, 0.32639346768068, 0.137161036188198, 0.384753630015334, 0.562635832519233, 0.633835426779715, 0.541606384905119, 0.314993424436676, 0.15932672340166, 0.152616233605404, 0.136950773617248, 0.709803116359699, 0.464864466073703, 0.113271183329894, 0.700878852298407, 0.179986137481548, 0.803668080429507, 0.513956104801505, 0.548438361778913, 0.207844759450221, 0.78458887459606, 0.526489632720483, 0.571023363557369, 0.42204075873948, 0.721156309917713, 0.0731383111712196, 0.594861918268641, 0.861977364476974, 0.44880106057929, 0.65257739045701, 0.303473118594698, 0.607439950952842, 0.278902309933453, 0.799562340196158, 0.796169117931073, 0.954086903392649, 0.444338165332343, 0.456882812653057, 0.599817138290485, 0.842621831470619, 0.0312004529517733, 0.187270141616205, 0.943594399351999, 0.947922138757693, 0.452984391203147, 0.810832551157257, 0.928879200710096, 0.672717280988528, 0.372332279639825, 0.405695690981349, 0.438823921792014, 0.678649200703148, 0.465070328743592, 0.95326416812625, 0.354697982136033, 0.339020861851316, 0.895860046717408, 0.545434258258452, 0.74928371733147, 0.124877199137868, 0.453234818121504, 0.0747494384557664, 0.663340613593339, 0.703651641928068, 0.918950568308254, 0.660071908400953, 0.690104919281635, 0.853723609811573, 0.467901722684299, 0.458478681919285, 0.80610418744917, 0.824767236231277, 0.190436103187897, 0.025673208318273, 0.0568149604955239, 0.142935580292919, 0.17141873755445, 0.625844781330649, 0.0295152077239043, 0.472332265248782, 0.678442770367227, 0.114788969165259, 0.236063923996315, 0.289080602151233, 0.172756294298157, 0.3237061227164, 0.801110098991791, 0.299633699216943, 0.775625723147884, 0.552770830568696, 0.55469172162718, 0.73064728664199, 0.773624822489763, 0.900847104626894, 0.138167156355332, 0.794076816385522, 0.189413374222062, 0.0289743825154317, 0.127387192476748, 0.133735686293017, 0.12831095970352, 0.935259793027556, 0.273230055155759, 0.942679255988563, 0.638150501471665, 0.872530820881277, 0.367073238198904, 0.236201904992405, 0.187318238113239, 0.54564926492706, 0.255116455050383, 0.305823607807962, 0.0155486185338325, 0.587488093829029, 0.962558310905168, 0.849847674190795, 0.00794029708478894, 0.634025539141077, 0.35929113517307, 0.114059615278847, 0.540817926388617, 0.416421383244014, 0.517098497408891, 0.886122876372267, 0.149395908274354, 0.434676522661601, 0.0590388082425261, 0.381030056945427, 0.722369468378458, 0.0951192289399846, 0.66719039744132, 0.296400554372516, 0.598562764039922, 0.151851989802075, 0.436377106681521, 0.0126823005250837, 0.229022946792988, 0.263679653547978, 0.511385232060082, 0.215084888111855, 0.346109790896044, 0.747817091128099, 0.413619453185279, 0.0557525615493182, 0.390028141360061, 0.474483940516422, 0.825320716477175, 0.303643571137111, 0.821790139319044, 0.565679763291859, 0.0543816168663496, 0.260014180767495, 0.589103551202266, 0.479733615438452, 0.198650616984483, 0.239012662248282, 0.780193479231517, 0.6173099191591, 0.144128799607613, 0.716132020170876, 0.401506428648898, 0.462385367587806, 0.707282006505421, 0.401210546056948, 0.0143854005690653, 0.0746404415832269, 0.59106817494651, 0.446001520274253, 0.92662060248374, 0.0948933557867648, 0.375418783646653, 0.545998864026788, 0.111678374014769, 0.904463234412108, 0.633276998600678, 0.905411643486748, 0.630552939496556, 0.0142282768725254, 0.316477786767082, 0.11187073833634, 0.629463131041725, 0.0607094478975195, 0.673994922037838, 0.477439086586489, 0.305545802598978, 0.516332982266915, 0.70702824985761, 0.81361098330337, 0.315808340008749, 0.31130218122975, 0.34497687110554, 0.6663048930945, 0.861071629859334, 0.761779254255429, 0.875837576645011, 0.871235497380345, 0.172790403055309, 0.850223982229123, 0.959588376293602, 0.770206989138259, 0.875015350637667, 0.0674134763683253, 0.646792215945972, 0.324096590428357, 0.640346912484586, 0.879751030938417, 0.373637675523219, 0.766737557593821, 0.168086658651728, 0.519724683593696, 0.627455609342921, 0.713913685799429, 0.306396105344489, 0.263687109426303, 0.916003507324704, 0.615034263753935, 0.0931750600170528, 0.627697202078366, 0.192028022891077, 0.776964690064336, 0.864500505417078, 0.333581611828889, 0.135413999328541, 0.765467158330414, 0.318613733518638, 0.252379511039797, 0.200074611449477, 0.0690098523576854, 0.551910735123011, 0.403811623172016, 0.750115800816356, 0.487183502440431, 0.384786122713223, 0.0614085102251287, 0.213691490822888, 0.543873364464748, 0.410644612260082, 0.900963206050373, 0.0562895369863406, 0.443509247434072, 0.537812527177546, 0.134060600345818, 0.540947224212142, 0.857368192475905, 0.198017363329941, 0.155609101290637, 0.0613777678138677, 0.661073988932246, 0.0186026419313314, 0.291102936564624, 0.973823826601782, 0.76464045426978, 0.243683930209342, 0.68211606104951, 0.137854510225166, 0.629811667025549, 0.857014692659853, 0.89979833327036, 0.348368402231864, 0.486310193286963, 0.679517492334616, 0.704121891203037, 0.46088405674605, 0.364272297195358, 0.280265224232864, 0.0762044335388465, 0.444624971046257, 0.165706296551124, 0.398748973477181, 0.920584670910461, 0.511334385523834, 0.914142330394849, 0.091933938842531, 0.993037174325257, 0.0964336228904868, 0.313147493421857, 0.785375111038118, 0.602400257965727, 0.465908353449366, 0.298131346189006, 0.133172784517682, 0.295008954572581, 0.166627082935154, 0.31710394116872, 0.109839680140689, 0.83208796140696, 0.971592754167428, 0.218271272437479, 0.706077628393318, 0.0390136924156214, 0.616295123827849, 0.669364170554407, 0.0372015327510951, 0.00333456174371782, 0.142464354163468, 0.862405458890795, 0.276037416106544, 0.531691536998367, 0.522221302087482, 0.567617714955188, 0.333032553637148, 0.413416552284401, 0.414348152562633, 0.983921784285924, 0.0577392977967374, 0.39654288589012, 0.791320582163363, 0.594185462784967, 0.309563092541107, 0.901802906347484, 0.0931269898117809, 0.319027136248953, 0.886961614706448, 0.657431673442441, 0.684516890301582, 0.473948624996901, 0.14124415843264, 0.950924881573767, 0.882626123493787, 0.437438546090718, 0.834956800914038, 0.325143400597059, 0.367639299837922, 0.794839466004007, 0.0993087162421777, 0.95178155766075, 0.00147392291052195, 0.295395849171415, 0.0484545523824337, 0.442740383821396, 0.789846368465099, 0.913520615552384, 0.533254359985603, 0.804076444490502, 0.562660356611037, 0.750876451961769, 0.00923035437930009, 0.476784851857698, 0.250329526525303, 0.307918258418837, 0.96694910442555, 0.208804360594363, 0.520476252773875, 0.22554612939767, 0.567197873830803, 0.998163526074098, 0.131865336731198, 0.954676584902944, 0.123877511497801, 0.186238205758118, 0.646502710583242, 0.128169077499018, 0.0813205028568441, 0.65922705422111, 0.0273988510815916, 0.985179800828888, 0.539330947126751, 0.373834916709097, 0.706734886141076, 0.94741137440689, 0.382277348843862, 0.692908554948395, 0.602071312401121, 0.77525771213823, 0.591834205678923, 0.376179807647948, 0.850649199595792, 0.225735190784356, 0.796958284325984, 0.996881006288545, 0.281310820370757, 0.710380975024432, 0.664642067152854, 0.414835533181356, 0.498269902274275, 0.949123025507723, 0.953181675899852, 0.732923311686192, 0.38468000462379, 0.0400827288350992, 0.582933585930967, 0.56471264171437, 0.355182617448626, 0.88020082100168, 0.624530132995142, 0.624015546339929, 0.295741411286576, 0.0746799268927503, 0.29370599439237, 0.234739808545134, 0.345897380916814, 0.848490359055661, 0.16036438186983, 0.157864376280118, 0.508659269642265, 0.603301000059333, 0.161396968499513, 0.635454324849744, 0.843940575853591, 0.782269211946215, 0.264566354354165, 0.314698962300904, 0.183195571521365, 0.447473272545392, 0.326678616614663, 0.279816289960718, 0.931763908416897, 0.399692412962547, 0.379423156844245, 0.592846449490407, 0.0685066859472768, 0.205237662103164, 0.723618278102201, 0.575147612830649, 0.200232503636967, 0.843480345277107, 0.423730001010525, 0.544750962597235, 0.527967383839007, 0.185107820794947, 0.0816897139884339, 0.464093622980913, 0.0305547757004935, 0.434984157549178, 0.557864961322982, 0.638782129306526, 0.0342174132925895, 0.709900447291196, 0.169323948454534, 0.593382473210353, 0.608064919326455, 0.772359586986895, 0.0562736157423926, 0.854725146948443, 0.384284325544405, 0.399618961028858, 0.325417485398251, 0.555390110486142, 0.295420103219611, 0.366120002034356, 0.349048328671612, 0.630224055538105, 0.664442080806904, 0.992096336616896, 0.944361262162069, 0.350347861866455, 0.192997944876553, 0.919576138372647, 0.288692050332415, 0.550859427511333, 0.919324153814327, 0.0900485221385604, 0.257696302203106, 0.427047957707649, 0.577723290040737, 0.899521394889066, 0.218225616992869, 0.967031884473414, 0.43398137803461, 0.784822423174501, 0.525233844620492, 0.331276852824455, 0.431593646762121, 0.717901601019375, 0.916212078117228, 0.890027987368058, 0.134706058376893, 0.119912755397263, 0.893452421083844, 0.653085541752716, 0.0402780766775586, 0.504715963554567, 0.894453465413225, 0.385728464443744, 0.292051877858027, 0.234036569972519, 0.200948357365792, 0.380306063271272, 0.594790142248649, 0.268385974964346, 0.622442641967573, 0.804567309882283, 0.10397125852275, 0.729240307495485, 0.648590264415262, 0.474654623354786, 0.932912110039707, 0.0964436793421524, 0.599106918365426, 0.233564709215544, 0.0323283382069697, 0.579859263775277, 0.84219815234061, 0.556896988122294, 0.839882989981967, 0.204952273668009, 0.621254886812813, 0.174020887296907, 0.289539818779723, 0.0185029353190032, 0.701496390153979, 0.9520913717603, 0.749009335045582, 0.756727389482938, 0.542089075444797, 0.282062872879027, 0.244874846890166, 0.286318217824419, 0.963134410230015, 0.230695345634932, 0.537321413436084, 0.205008252217627, 0.434041294551689, 0.142211972469126, 0.375583774965978, 0.793560626787436, 0.812815650855027, 0.903790906459165, 0.540375734165828, 0.817863312231253, 0.708416645332428, 0.0432088518866429, 0.145945980419716, 0.233335662736813, 0.246701081135556, 0.170282742970061, 0.235056835045954, 0.275461075952424, 0.951606626170317, 0.346670983224879, 0.2973219378091, 0.404426208195778, 0.302237425794988, 0.757311083635501, 0.359743894064038, 0.12489957161308, 0.61718255947227, 0.355519453713233, 0.36293818661831, 0.068488286374946, 0.867161871778362, 0.457896198017245, 0.0776153852163201, 0.904921195882039, 0.2816958807944, 0.613877842567314, 0.661857942093214, 0.200024595896795, 0.959986227572448, 0.665105984350354, 0.541299839388014, 0.868968871664997, 0.557047467058915, 0.0213981169649879, 0.482680551891677, 0.807985579141521, 0.736009366155118, 0.572304395272699, 0.00898485336788568, 0.718280463661178, 0.449426591726197, 0.659616768761805, 0.753208345950567, 0.804739013812769, 0.0291555821799541, 0.779817267477441, 0.567350498584087, 0.0761166268515728, 0.251621824309164, 0.13345452861896, 0.564473921782696, 0.540981282819544, 0.0689214476617044, 0.988430867357282, 0.251095440428348, 0.315472991527951, 0.300656948802795, 0.0420390381653749, 0.527901300708439, 0.25602233376832, 0.408709992443545, 0.947512642788601, 0.919280969984633, 0.121243677375576, 0.591944834715854, 0.359652173164073, 0.719306932475474, 0.523551091500771, 0.260843351982082, 0.493079306041446, 0.855835900703399, 0.724406283688824, 0.199110301818659, 0.157285874413643, 0.370475519779316, 0.862272004954744, 0.684762628110137, 0.634197879289704, 0.141323291186943, 0.0793022450665238, 0.876145347529076, 0.42042870124015, 0.487663607210372, 0.460325494339181, 0.515677249607505, 0.271994390193763, 0.231579838051303, 0.899534076872039, 0.908697326644413, 0.603642852234244, 0.365236483082946, 0.598590500684565, 0.668487146264776, 0.894564090918275, 0.0873355239070054, 0.539010349313424, 0.428449936007297, 0.617153494459035, 0.558875876956837, 0.225850688199737, 0.104519529950634, 0.00997765604969036, 0.0591533040753343, 0.322652370507717, 0.779476506848221, 0.335478586307353, 0.61956861250297, 0.992885717464124, 0.648005694952403, 0.539775820318468, 0.23225475305078, 0.739832227063707, 0.88899234515285, 0.859807625912606, 0.597060788610351, 0.654752143480193, 0.915013993756726, 0.433183080639157, 0.28976075584224, 0.631882300545926, 0.295421839485955, 0.622028790367341, 0.0475341941914539, 0.994609895249091, 0.206764149130446, 0.607363956107084, 0.347630325692841, 0.717737067921327, 0.0279932951713195, 0.0668417611778476, 0.927064140870894, 0.0877717606376327, 0.332398561915436, 0.526180895421029, 0.246642124058313, 0.542899456586946, 0.780867850604077, 0.521882381409282, 0.931949933886006, 0.147111980322787, 0.416765701751856, 0.280292866619113, 0.598100317141488, 0.0364748567011317, 0.0636865030513627, 0.322882505170559, 0.0983785303848749, 0.170044626268754, 0.371163798766144, 0.0397616226212385, 0.709240953533567, 0.641341360210975, 0.174060860147872, 0.0621519426336853, 0.406659420297776, 0.463062925909165, 0.202674299437306, 0.869549890049143, 0.597940369107138, 0.0230135370914409, 0.899426437101584, 0.452946115671495, 0.0580261764965854, 0.106269376654518, 0.998425291506397, 0.86632740366717, 0.615205796593277, 0.0269444844838657, 0.322522635620462, 0.463776233943276, 0.0990172329213107, 0.570988144269953, 0.325875094193265, 0.450492336647337, 0.577843795117458, 0.0748443075998355, 0.0573426693754819, 0.300956435005654, 0.521721248782275, 0.561880235062444, 0.241554770085929, 0.912720162652781, 0.825734269699093, 0.444545883918906, 0.982062563214574, 0.578267561462169, 0.23442349639393, 0.810590442462569, 0.451274164510932, 0.249969169410276, 0.955437531793689, 0.142650346985794, 0.512563384780399, 0.971925137586938, 0.648320621265659, 0.614671003416509, 0.469650371915958, 0.577780697646748, 0.911313227169849, 0.376220283460707, 0.228764772549632, 0.423524871222756, 0.273596220894793, 0.444565843866667, 0.627515046901424, 0.534641253536576, 0.385442159974304, 0.873450099080692, 0.300348734897046, 0.400029715619638, 0.517723514394718, 0.0618247144123362, 0.231367220579307, 0.118485991561029, 0.0987800358735031, 0.890275455546976, 0.0333890773864002, 0.839034991901461, 0.507266724816296, 0.113716441044558, 0.49044565696202, 0.599441720175534, 0.0902161331201864, 0.978224034127668, 0.653018772578069, 0.4611321265498, 0.863758110483555, 0.262829072110785, 0.82396294284602, 0.328975305064346, 0.941300531148415, 0.244108071013869, 0.95711016517866, 0.510824964684785, 0.564579859307507, 0.993716131600963, 0.770983940011353, 0.313801920872552, 0.0578726248164741, 0.0440737918191335, 0.812943969672552, 0.412073277714361, 0.384163168822831, 0.523126532390386, 0.892148966891949, 0.405920640599473, 0.604388896137601, 0.0948035564584498, 0.335981277769252, 0.144841838461304, 0.244880529743163, 0.37897165848807, 0.270319612046424, 0.215599299506781, 0.633731111043642, 0.799715570149051, 0.208514026718345, 0.896256192163473, 0.745709470805585, 0.536385807038486, 0.969965194019504, 0.564883170177399, 0.217689898943781, 0.857763286845635, 0.861999317827353, 0.313302194836426, 0.326984211497351, 0.840013404018371, 0.492532786265149, 0.0519024840881931, 0.778018368946308, 0.426672385823044, 0.279961881386735, 0.333508011813132, 0.366939952046648, 0.794792410212092, 0.0386907855675434, 0.726669335495522, 0.872684459300564, 0.285814782532667, 0.656849259385463, 0.231886625099597, 0.621953941544503, 0.0751237673533113, 0.966755875499731, 0.609969956623824, 0.383715179754413, 0.0305697733414173, 0.85756558681509, 0.603543185381736, 0.847840405091208, 0.504616246876344, 0.00791083757342614, 0.919073177464621, 0.410704795653201, 0.732365747785734, 0.151105278860338, 0.815233276687081, 0.909602531861235, 0.777942461183173, 0.757123136425643, 0.405197523540783, 0.701901080171244, 0.565460937121375, 0.584736351923449, 0.344914599267407, 0.704233239500645, 0.16086367902273, 0.00151393336675809, 0.562000852140945, 0.546484124944015, 0.550827761821269, 0.69453676079433, 0.927494092054399, 0.944210308868036, 0.903263329325526, 0.604106286391469, 0.926209756983518, 0.116946258411827, 0.340801507895534, 0.0367304435835989, 0.538369850492198, 0.737228801225072, 0.181222502654283, 0.426246661556396, 0.0981679570288845, 0.303221566678817, 0.780122611965039, 0.536960416719289, 0.769071432974493, 0.638943645628397, 0.893148245658072, 0.0606903874333861, 0.175771545768642, 0.416331859021737, 0.739791840981316, 0.892954792490445, 0.0258533065890132, 0.137563028376576, 0.424087432629228, 0.764566781879008, 0.524437187651802, 0.754481358949251, 0.169798443378302, 0.672727005911346, 0.618652742745988, 0.00684608946096921, 0.740225524137642, 0.991735477952075, 0.128181357237449, 0.360582760275341, 0.191073265372555, 0.714397660814069, 0.177867011268348, 0.986848749985563, 0.0169991299016039, 0.0397964695696392, 0.804330531934784, 0.859792285576099, 0.566252527997595, 0.754112680957328, 0.51951476638776, 0.585904472449875, 0.190733026170859, 0.503441463478443, 0.0509146326559937, 0.0560957431770505, 0.335197399918532, 0.630904756752298, 0.891977815013119, 0.673396626434058, 0.685267040499651, 0.695744793197217, 0.799831433153626, 0.660605167199519, 0.519936595351645, 0.331736410800664, 0.934172712819677, 0.246476948149498, 0.511206371142388, 0.740317886475583, 0.254168532971016, 0.84558475858476, 0.538154421989849, 0.910136134062759, 0.35070641014647, 0.935324882715596, 0.927719459132508, 0.587700360395952, 0.101762134380532, 0.36687593972201, 0.275845534967993, 0.266097501342967, 0.226538649427265, 0.0013207800733116, 0.893007371636192, 0.453497507245686, 0.578421005504199, 0.315605538850825, 0.994016875066189, 0.983892883553004, 0.964547891083534, 0.666265239052169, 0.726158176374528, 0.334042554571979, 0.522772766352871, 0.273354365608348, 0.718379028435567, 0.778017556357603, 0.0810642195290494, 0.221580140407825, 0.203959494037159, 0.624099305895655, 0.725205115537093, 0.834364187698236, 0.01889538192665, 0.202117631572217, 0.469092961868098, 0.378395181508715, 0.340393733719399, 0.063851184823046, 0.761335934073557, 0.402662085784262, 0.674289716276666, 0.551075659074815, 0.0514571665686354, 0.30749593756604, 0.965444766947439, 0.931475149923169, 0.378026726540899, 0.617670191946775, 0.56253339120956, 0.830560709643071, 0.957698738551541, 0.075505653290734, 0.888024396725501, 0.562447291599302, 0.194355125458527, 0.221452790874618, 0.705763049479018, 0.596761435885847, 0.586296073113408, 0.968494148140419, 0.58174342527109, 0.0998002912774532, 0.166602821775987, 0.102093794787091, 0.146247104764678, 0.671564766879643, 0.63985318181102, 0.372435451684637, 0.162872966132474, 0.389518808304106, 0.799976566417261, 0.399793699859006, 0.755119084952242, 0.295210411660445, 0.640005910590266, 0.885117154387046, 0.209643322721594, 0.480356487911089, 0.112987559416233, 0.132369640030042, 0.0640022573461611, 0.0792973954603232, 0.626954163968362, 0.411333966195378, 0.63858968599787, 0.856556962747254, 0.76357503291757, 0.976160296189226, 0.781537245089357, 0.935583203546324, 0.739152167184324, 0.253619767514479, 0.719250284659865, 0.693512615600087, 0.773318950848084, 0.747217085683957, 0.318383666829911, 0.51043095721181, 0.773066476433341, 0.572815526042185, 0.953850335084432, 0.171572304395605, 0.90723509510185, 0.752219422620808, 0.286175231686453, 0.627410342635826, 0.46282792316263, 0.129439245288947, 0.549384662580998, 0.96950932318167, 0.44214048973971, 0.590963929826328, 0.110970247102797, 0.199852291472223, 0.16296410308384, 0.0368774272976408, 0.272724663967449, 0.230133589900284, 0.358544688773658, 0.134342698970895, 0.998594597293282, 0.51352085473234, 0.38784630473503, 0.249852757945931, 0.364752426466412, 0.399057541172143, 0.926300759649667, 0.495472930639889, 0.609187004749775, 0.00480846355691233, 0.587728764344903, 0.77746242251411, 0.657183183790527, 0.528314693572702, 0.825773161442162, 0.962433769377391, 0.313736456711858, 0.797085057474744, 0.285175798017812, 0.0148948755305561, 0.094149754391039, 0.328723020879438, 0.305566322109223, 0.0180221330550887, 0.162207408733534, 0.443989463403324, 0.766770029537118, 0.6812559351515, 0.714536728018305, 0.459780878009065, 0.919268407378886, 0.9889434196639, 0.932601115420277, 0.461523347376351, 0.904922169553713, 0.386451653685206, 0.603016552280489, 0.560319008710356, 0.845839755610487, 0.284804954762948, 0.663318497680869, 0.602258176526865, 0.656462946712791, 0.309934086607524, 0.331643807037803, 0.18819732885014, 0.100683407202839, 0.286563954039492, 0.354826221665979, 0.535761698075949, 0.990832978041241, 0.0281780829153787, 0.709532015618129, 0.905157017594289, 0.865829381113203, 0.119169322975101, 0.955275201170228, 0.440900790942236, 0.87576021747857, 0.864951642952837, 0.355150575228126, 0.631146447463298, 0.864549250021238, 0.0209917345100399, 0.0768202549829806, 0.376714427294387, 0.149193663545822, 0.034096895707185, 0.782249707503029, 0.327245076240868, 0.817476329698851, 0.173619050225113, 0.676117732631756, 0.875600480808911, 0.757155472370129, 0.229625714585824, 0.358943619285115, 0.363164094081499, 0.267443111282436, 0.337302762280291, 0.0869606489015061, 0.451559652004615, 0.454664027740148, 0.0290355097235441, 0.637101872656819, 0.0594587869745378, 0.169196114038212, 0.684679742407259, 0.554567312566934, 0.00603583365666827, 0.288085152626356, 0.376611564759353, 0.146493033365658, 0.0745336154710834, 0.460742876227146, 0.370394379387662, 0.824557518411133, 0.537535656063027, 0.817516082947517, 0.460380826593087, 0.0140338143822113, 0.00498320514324968, 0.166226783877916, 0.365938838897355, 0.718210476449713, 0.159423636777771, 0.278771470720201, 0.644938447686513, 0.287204149170803, 0.321930892683723, 0.155416794977341, 0.387663140685647, 0.895733134019957, 0.88776686793174, 0.393613327737208, 0.675457742148991, 0.252214194594822, 0.949987708485075, 0.624682173299944, 0.206947941462873, 0.109655161101898, 0.565849610384352, 0.274333396208018, 0.0711652600846836, 0.158449300200466, 0.0495202961955035, 0.680883747990123, 0.781755812401584, 0.80736519625422, 0.265053593193212, 0.895899397129799, 0.607986088024156, 0.0144008386388609, 0.344028748760077, 0.533664735315603, 0.627762577051544, 0.446737300620639, 0.81110564384679, 0.144827989332741, 0.974829375111397, 0.833760065956362, 0.340101160883444, 0.616239065867128, 0.303756168655075, 0.0892187674766183, 0.521525804911955, 0.825291532919773, 0.763926622410418, 0.947639307901116, 0.333547273199825, 0.38973838244907, 0.150412879875541, 0.333704420658865, 0.553678828448656, 0.5502939448828, 0.160421044946278, 0.117089469573958, 0.398550434016254, 0.831383004688908, 0.185001205373067, 0.500785757513212, 0.126314479899365, 0.864628453057318, 0.766645688295246, 0.564353535554258, 0.389569371944222, 0.490137663316258, 0.886904095824422, 0.905048281577648, 0.498388955348408, 0.529226950242885, 0.909700719119941, 0.578598446989567, 0.775785103656777, 0.661394630545105, 0.46965723894088, 0.219767619845275, 0.602589738655931, 0.18425899678177, 0.197509715511001, 0.861992146483479, 0.125649252286031, 0.645580255723501, 0.43731821030233, 0.612605067498194, 0.737482414756424, 0.303279447163244, 0.0429677132819266, 0.83550753064121, 0.36930817955419, 0.661339306433348, 0.895719918011541, 0.274099065906944, 0.997898477049783, 0.83435338138818, 0.791343687458312, 0.656612708164405, 0.543673019266521, 0.386681329635985, 0.822232087530085, 0.595334513926938, 0.781641320694625, 0.987674556243708, 0.788253800442339, 0.85292088665501, 0.485612829189145, 0.873756072506857, 0.33384708318725, 0.205316778886351, 0.494357067828648, 0.0312089596566836, 0.827574414768107, 0.264257437743154, 0.677499332357288, 0.7938643359396, 0.681843864737366, 0.650759237451362, 0.237341728511186, 0.477403479181025, 0.936419265220215, 0.241103511012773, 0.20911387941948, 0.272472027662029, 0.775817420377806, 0.331411414741901, 0.603074701436595, 0.184014863849, 0.0874588502239265, 0.30891538156382, 0.230878404239435, 0.90918705194888, 0.93688434585519, 0.0318788407443867, 0.5936470432608, 0.0438175218533404, 0.424902843333343, 0.521581278940114, 0.840339421490275, 0.625029679129942, 0.255204680450144, 0.904691235804131, 0.767300602484737, 0.562665501616448, 0.897153533637285, 0.368079409452616, 0.337731006946369, 0.618855356023971, 0.0868954399451265, 0.375497531631136, 0.18738106742477, 0.810526689988458, 0.72324819510584, 0.674674046120504, 0.913981417236972, 0.82096089736944, 0.548724636499035, 0.0155415941457361, 0.0946863574453204, 0.514758291470818, 0.952482638232214, 0.843061196139862, 0.935012245967505, 0.973413085109433, 0.372526390066493, 0.320900690746152, 0.783247900846885, 0.21222171433272, 0.959152205838633, 0.930232869221638, 0.365729487415287, 0.344912389725989, 0.692681628396483, 0.716603864448832, 0.80034349643769, 0.593670477334659, 0.0971750120720246, 0.559509166370969, 0.0430899603777365, 0.757631721044845, 0.659647529115392, 0.599459320988681, 0.22699474981533, 0.713714360951096, 0.724502413091687, 0.344217969775603, 0.0327508258606198, 0.843615848232873, 0.770300114624033, 0.509646636811985, 0.706963278762882, 0.791173907397134, 0.330669018224895, 0.913675141954088, 0.576704962817838, 0.287909274145398, 0.266864860260348, 0.890178179046525, 0.366101189573211, 0.150015387587698, 0.946327569769858, 0.65083531459943, 0.982214568873265, 0.193833694009784, 0.828742594901994, 0.493802089353527, 0.590778536105395, 0.0448905722328005, 0.564458151981422, 0.633544303156179, 0.498099762823955, 0.0596236289852767, 0.515163184212802, 0.0531949390179804, 0.843631920552928, 0.594672781587329, 0.691018830555377, 0.647226771882314, 0.623439835644007, 0.953858319757518, 0.176943992133502, 0.474905196360106, 0.842403609497674, 0.496963455325755, 0.762404727340236, 0.179877258121768, 0.691622057235644, 0.914424854263935, 0.162657424701267, 0.428096119735541, 0.0418518662382871, 0.225746256720798, 0.666150366031971, 0.871344882700827, 0.344728343726005, 0.0272515642671105, 0.905282667459542, 0.471062246494722, 0.910464018771302, 0.327979774810905, 0.691103543176427, 0.0174479807938583, 0.512346849920018, 0.33181367758602, 0.336112447449826, 0.953345604983632, 0.750505410107031, 0.333924229403516, 0.632832748728656, 0.415545708937644, 0.576499797766923, 0.95950052865169, 0.615848229131544, 0.1505778963445, 0.838270697505464, 0.726165419489224, 0.108846200368082, 0.161351540475342, 0.297010289521935, 0.423582438892364, 0.528414418289072, 0.275097372620348, 0.827956080955766, 0.792784736413836, 0.273747993077394, 0.615694400284643, 0.101550723794203, 0.669963217389328, 0.347396597850599, 0.669495603356068, 0.295302241154495, 0.680427967038852, 0.393773066700295, 0.391818329756063, 0.206963300889426, 0.724520306856971, 0.819305359946909, 0.926471637376764, 0.514119991814143, 0.193143288141933, 0.602900907521622, 0.198180187967336, 0.410392575176234, 0.85307481422537, 0.429081790557031, 0.925658579314925, 0.373077718311826, 0.698246369162013, 0.588625421185045, 0.100866143550775, 0.412395000208901, 0.924708227033502, 0.0555504735405636, 0.384047546003501, 0.240640116346963, 0.227808606940135, 0.357120387132325, 0.924102561054751, 0.502498346328144, 0.803237639785716, 0.350272034640837, 0.458682510872841, 0.951096662625773, 0.132983376216182, 0.0703072409946732, 0.706081441736291, 0.931702775674971, 0.108900913968859, 0.651696841769168, 0.146062140347686, 0.693962928564284, 0.62561362265368, 0.0437233988696382, 0.797923826673112, 0.986210699083828, 0.436744819062915, 0.344748854345864, 0.0488061184240225, 0.225202909375981, 0.581012918730146, 0.601375338264139, 0.25251840447237, 0.654341659587006, 0.0600955956355986, 0.940506090100758, 0.0925305777674283, 0.333977931176734, 0.651049642981125, 0.503910776363358, 0.530925140628448, 0.379123730790155, 0.768907412838519, 0.282511265779884, 0.649272738826965, 0.922183822379736, 0.330671885619597, 0.8134391296668, 0.787635595388448, 0.124322479444047, 0.157387547750491, 0.414998271586948, 0.687332798473363, 0.657703905050132, 0.679977496766688, 0.00938667700924689, 0.128609573532336, 0.763906381329514, 0.841915497973284, 0.746948593323088, 0.944523465722156, 0.919714225946387, 0.223074290279877, 0.509767188085986, 0.787740194021004, 0.522451914333082, 0.61739194496612, 0.558240701061404, 0.730992025590584, 0.0634021161483509, 0.49278652681837, 0.229924038235982, 0.337391081769921, 0.551638107123105, 0.743685672036025, 0.412305545422802, 0.507618332638558, 0.128972372590063, 0.214835656563848, 0.292829641356233, 0.449965831563606, 0.240282116532755, 0.263263485549617, 0.891582571758429, 0.586139828756669, 0.938674617699067, 0.61807987726774, 0.250719039558465, 0.102537452785395, 0.546363901182181, 0.442645614818116, 0.691506135803519, 0.759439458112306, 0.399421186724809, 0.567413399000053, 0.205583924928062, 0.94352861041327, 0.512806644659089, 0.276246602073697, 0.22913955249854, 0.952928414501858, 0.689420829610687, 0.707872749897411, 0.656921910028513, 0.778967448467013, 0.580469555874256, 0.503230133613125, 0.878731309131335, 0.392949270164128, 0.242635507549908, 0.281626628145212, 0.904771010998606, 0.702675221153008, 0.853055727233087, 0.378683694796009, 0.633021630675909, 0.0315568425574113, 0.920033538805044, 0.469746884935762, 0.519903186132389, 0.766312112602389, 0.336216599694675, 0.625985120212825, 0.326902007816966, 0.311106848888362, 0.646078397864106, 0.433691632243578, 0.102886548069359, 0.236150966651974, 0.97178279590953, 0.698897416325079, 0.659450255872462, 0.326097341551261, 0.0198219171909205, 0.570472591978458, 0.0363349019473779, 0.264742744870145, 0.190658434187946, 0.44929962442202, 0.768655178962958, 0.627936197397814, 0.314700292825058, 0.292043134550917, 0.0782379885024695, 0.171943833266124, 0.721280345216873, 0.0894485867323322, 0.917812202791595, 0.429676950914959, 0.0797228712324417, 0.572144113781267, 0.831772314298265, 0.347367804853563, 0.779889004323627, 0.708339230111793, 0.590942358853515, 0.376993696231509, 0.393825024763218, 0.428308845834898, 0.423025608994076, 0.218924283366909, 0.496764652079592, 0.0398697134798933, 0.564816498716393, 0.0491429593922389, 0.872281243332778, 0.860024759716473, 0.338257942897051, 0.974030463434996, 0.954775008053706, 0.988620997542051, 0.157717674744754, 0.00291835029377641, 0.24817996036152, 0.346290120726571, 0.233338712802426, 0.755891184137008, 0.225927682042708, 0.448443245432694, 0.25944456659852, 0.345794513857056, 0.902954947885711, 0.0127008682661497, 0.132222804167349, 0.388062654293169, 0.840621010357475, 0.213266302910019, 0.935234397675956, 0.969800379015831, 0.375258555862433, 0.984862842316508, 0.122085694580906, 0.760957470100977, 0.579361683136027, 0.621156542467873, 0.489904240446515, 0.220450353873685, 0.144134244539961, 0.635175705304298, 0.802415693517919, 0.669643258984183, 0.471602957589594, 0.944478114463053, 0.591480867588899, 0.226747759363656, 0.683843687213689, 0.320642855770221, 0.748918543809053, 0.998143207374415, 0.269733351684676, 0.931869635780535, 0.696387052982075, 0.0952592833431032, 0.245872172228089, 0.934657551550311, 0.479784218135594, 0.882527743636948, 0.779297316773493, 0.018283884759971, 0.770761745335744, 0.685798403928687, 0.867667063085114, 0.580509670362295, 0.626104781171799, 0.111147193228052, 0.213694223133563, 0.0363697673560918, 0.44454492912699, 0.326580903156127, 0.287294231129642, 0.496451989255175, 0.18170519452472, 0.933791871777743, 0.94018128442575, 0.591463082475282, 0.000808618205962341, 0.90300644629573, 0.682892941698706, 0.0738510179919515, 0.996384162337222, 0.314456966367478, 0.809219837628967, 0.461450605980961, 0.587141467074297, 0.477595513339762, 0.55245363212413, 0.175316539083581, 0.587244869016264, 0.154801531119142, 0.445452550042128, 0.537621422723437, 0.46190011497577, 0.834109708891139, 0.99617230952962, 0.972244273128648, 0.745929372627335, 0.199956952995643, 0.0363292360197636, 0.188604065746608, 0.577547348622185, 0.628272342585065, 0.364774751422131, 0.636558009588955, 0.891293458108173, 0.25351820745369, 0.286359337940548, 0.125419176245427, 0.528573772748323, 0.820766361991123, 0.299775460104113, 0.787678917413547, 0.437012573040754, 0.550539716147499, 0.986114473782701, 0.689780984124998, 0.970136776630278, 0.0959658384301467, 0.898092237410388, 0.0584430934125807, 0.123318151074745, 0.630725743368396, 0.370104345713866, 0.319572273729759, 0.795694331193399, 0.301704811674572, 0.607461688382013, 0.461972243165118, 0.29259269105507, 0.832418956424072, 0.570525957373189, 0.0705081138567619, 0.266311690876462, 0.170874830762596, 0.560307422031177, 0.202090504074273, 0.492368811109359, 0.729367419628283, 0.954996382288605, 0.762368339977811, 0.0800681389349021, 0.938246224630109, 0.974473843479017, 0.607110428021082, 0.957494958695934, 0.33065054115953, 0.901215142217029, 0.000365623763305734, 0.483280019453646, 0.147474747267827, 0.314992086852311, 0.171375458208079, 0.592906936782863, 0.6576080119545, 0.348962692898616, 0.053693447435528, 0.0471680310748598, 0.0558578091685061, 0.071511995620654, 0.0156755449546512, 0.973679007254639, 0.73479672093121, 0.752385614736353, 0.0333136081311348, 0.117606117130326, 0.261303269881319, 0.893358001913081, 0.972061137678921, 0.994073886211684, 0.346786263254747, 0.292302135568305, 0.187247407159404, 0.74314833631864, 0.718018722684615, 0.188894725512381, 0.43393780460202, 0.0455931354331871, 0.655473475980963, 0.781042208904641, 0.862900425221566, 0.785352871833154, 0.433492302723436, 0.347821246965429, 0.0553885502592265, 0.819441442588722, 0.106040338209226, 0.00607404653196786, 0.275042352548372, 0.942771304822948, 0.674259328016191, 0.385658394829595, 0.856946289582242, 0.674824887935709, 0.608096918089583, 0.439425837023674, 0.961285202895325, 0.822906498290483, 0.0104994319560711, 0.141616734678351, 0.482809835482827, 0.580618433453006, 0.292988963991134, 0.438866965577127, 0.398265186954992, 0.137305480611584, 0.952761708644887, 0.360044666230198, 0.836977407273554, 0.0413357256847293, 0.423830588052359, 0.0756379701383665, 0.313814877315829, 0.945982651440048, 0.854503454518775, 0.173531117920384, 0.535883588923296, 0.294889205964141, 0.902872154222318, 0.0813874092956911, 0.787474899998168, 0.174408494450077, 0.719987951742843, 0.232006661228245, 0.207137482800106, 0.216132109524524, 0.976670544960923, 0.381849006378176, 0.635288896447045, 0.0465640390505186, 0.232068379506165, 0.874840378958323, 0.460142548060862, 0.00306357415717695, 0.46494127608993, 0.82407189316069, 0.871885571701645, 0.301074526079261, 0.236005206878626, 0.831578262729024, 0.437752605868436, 0.507087306378726, 0.185474577732799, 0.15149857398175, 0.906805627355109, 0.992749903551154, 0.390081439337444, 0.826810550607002, 0.954057682185195, 0.893842059352079, 0.841006255307916, 0.68328415745026, 0.171796261442336, 0.237551305568045, 0.43428515858871, 0.0250621847512497, 0.268646592413321, 0.980159345094034, 0.312692454258049, 0.57175164244331, 0.611663217389069, 0.901538065407817, 0.396039920386533, 0.692766655813401, 0.20448582449894, 0.0628527864844988, 0.622385892134218, 0.688019576799395, 0.296785133197485, 0.571561840931681, 0.316222560852251, 0.809497998396411, 0.365497581066373, 0.698789902488688, 0.63342595377055, 0.964579101664028, 0.200796668743141, 0.598177740313402, 0.0837034416710265, 0.544414709761594, 0.846149148541502, 0.0406983292302818, 0.329360440753283, 0.709349661789857, 0.86031088116752, 0.500898616536949, 0.994759585555902, 0.428640136534168, 0.267345254738738, 0.472951879363933, 0.160845331178376, 0.158897811993886, 0.268341968826029, 0.757120064889279, 0.609174029586122, 0.966933032936248, 0.773320765204118, 0.572864576286178, 0.450223504237578, 0.287057077700738, 0.752386592401845, 0.0943581612914803, 0.106785749855255, 0.734893629307165, 0.354592025606987, 0.58063359968943, 0.298855434035719, 0.71368094189454, 0.360486200417661, 0.718459525105082, 0.995323733027603, 0.0837001049906664, 0.468205896953011, 0.0938030575753772, 0.725170456137609, 0.912045444279009, 0.0864144334599773, 0.519950888705065, 0.55263100106469, 0.685149037652955, 0.299654783124567, 0.601877715939767, 0.198503905239169, 0.656757415222109, 0.698357060238472, 0.459680380401339, 0.157659169447911, 0.421785817063031, 0.596054366496088, 0.32233149631154, 0.830728820703439, 0.122685008120538, 0.251180701963122, 0.937607516890418, 0.655208209015462, 0.752910869021947, 0.810858531249274, 0.0484267181861879, 0.414709137697206, 0.725489055407353, 0.139044011035605, 0.632557259152003, 0.246833457405577, 0.839972273220783, 0.836670823291279, 0.453072418339775, 0.394758736326736, 0.961595301654415, 0.00817830248062723, 0.398438689641481, 0.487772273911447, 0.618136762922027, 0.0700099940684277, 0.142687919676604, 0.65422663236224, 0.218319594919908, 0.119131928791299, 0.0431371009641738, 0.165558181078708, 0.511105903601362, 0.867268018847985, 0.766260738181077, 0.134950924298905, 0.0649206776199452, 0.6724449515439, 0.482206924740791, 0.495417913107209, 0.313483664719508, 0.3235009696048, 0.909101313751816, 0.450336623974149, 0.748635800290558, 0.18525332948793, 0.0849220204256829, 0.207029203928117, 0.558915662241761, 0.661700436914934, 0.939213156370957, 0.0402453742649577, 0.163658499050043, 0.438123344172232, 0.325667604046026, 0.0899323623104628, 0.81031963599662, 0.399558974701933, 0.655314300069037, 0.566843498428166, 0.00312186435870165, 0.651351480966231, 0.317097344109857, 0.265933837214148, 0.460448203298888, 0.623160939909453, 0.801853430705716, 0.561164119281572, 0.0248040903417872, 0.793540436257167, 0.273088822573628, 0.8229113441199, 0.688554782752196, 0.603902451528603, 0.386950094136329, 0.0655353381042242, 0.998593388275938, 0.661043898011053, 0.229172299721171, 0.0922545755501529, 0.240978662136254, 0.90256966246009, 0.217347988702561, 0.479522753134842, 0.909228263005315, 0.193782442199078, 0.447987606452409, 0.617800086454913, 0.594213032930688, 0.665877218675335, 0.267680191381798, 0.42386718305213, 0.79932379675241, 0.932312830215984, 0.469719097366558, 0.306303387408322, 0.439371832428688, 0.306860040681821, 0.275012973937648, 0.138747386182163, 0.166547468691594, 0.493947393301626, 0.0865835108558467, 0.926838908385081, 0.148476249908329, 0.160479198406076, 0.717723587756788, 0.674894462408198, 0.358519543033036, 0.0904579037306417, 0.00824804511109911, 0.372779031664365, 0.788307138672737, 0.447286056531946, 0.099624682921879, 0.805831138132599, 0.659454985328742, 0.452364092600565, 0.533362361213085, 0.526664736926102, 0.62968105880478, 0.646564430916446, 0.191830732290044, 0.801732062713705, 0.196570990132495, 0.411293638131001, 0.235693361516778, 0.272205438124067, 0.00866849408275416, 0.928280818433486, 0.52021949503067, 0.464125608109992, 0.52352972891717, 0.336812107309553, 0.332378948126979, 0.988287882218548, 0.473716902300036, 0.746860937890996, 0.669958827683787, 0.795591999661478, 0.272543425036734, 0.283915129328217, 0.717255749860709, 0.184637117932269, 0.833703358649899, 0.169665012879374, 0.953265676702766, 0.722548045183909, 0.134248402878824, 0.85309249591138, 0.931692742214252, 0.564184666841829, 0.398823330482776, 0.441602502251028, 0.701680740801675, 0.260042990659034, 0.028529450771738, 0.947965234456929, 0.996168960152439, 0.779483271476543, 0.989401408957501, 0.487552957293551, 0.127982660072096, 0.464367277418625, 0.742590490862924, 0.673391623130582, 0.87397950733707, 0.758250399432241, 0.617175241638266, 0.274817223251346, 0.33637834442203, 0.835723542423446, 0.385886376816239, 0.0803465622400926, 0.865584371478723, 0.435374060410848, 0.172080873523429, 0.0042964132012645, 0.652496830492259, 0.369689625970097, 0.376099195620655, 0.441342460352586, 0.516123848264546, 0.30011461607604, 0.478321805972902, 0.00624688582989652, 0.171435895803303, 0.303557434684266, 0.744464561687132, 0.58089334220114, 0.701207183127599, 0.804664425130936, 0.851985238947533, 0.117720955924319, 0.657263655856109, 0.752951472185895, 0.70212878991997, 0.60511845372825, 0.0382666254153522, 0.195150090626943, 0.3306816983353, 0.642833401655145, 0.217403927613111, 0.0769098078480039, 0.980589355553048, 0.206758961862981, 0.983253339051778, 0.610371013940751, 0.247978552543921, 0.517957929512896, 0.712831054074753, 0.259869027399158, 0.114686005971122, 0.549669496591051, 0.218901765834715, 0.312175584209972, 0.973133082827508, 0.309190490494506, 0.946247503422748, 0.410561866618572, 0.0103336015361754, 0.31006899727994, 0.155206837637815, 0.994082540226609, 0.450239204612996, 0.545218287587308, 0.262331778031573, 0.882541234582713, 0.103061071907459, 0.823562689525412, 0.427175730436468, 0.920344682595652, 0.528012158818769, 0.870834639780891, 0.352810810608377, 0.400152411523643, 0.597922890497427, 0.911517928166822, 0.13298701913312, 0.0421674174049691, 0.401038960967571, 0.59570768744932, 0.919679296384555, 0.293775812244145, 0.545039342418705, 0.781681778277845, 0.327399541722703, 0.0952627683681123, 0.441682612344088, 0.0290768197651279, 0.675390380934978, 0.11159037338325, 0.199640280886106, 0.530161538610553, 0.62735684723012, 0.373596632230918, 0.606453634019215, 0.846388684543292, 0.701945689210205, 0.392873914420265, 0.571868663025823, 0.249773735183737, 0.727346733111401, 0.343329145359396, 0.0786346748632595, 0.302230928274436, 0.300402481923997, 0.0232211975966347, 0.176114251657706, 0.860918796611026, 0.655948169533555, 0.856537601321621, 0.270940465812086, 0.895947779622096, 0.208790395872381, 0.964168180246058, 0.414360088039167, 0.229565213863562, 0.818370198964757, 0.360443493262076, 0.501524068149432, 0.747674118436213, 0.831849880664806, 0.394470982041155, 0.0966363468817153, 0.265391614825356, 0.791945012552107, 0.936860742600308, 0.568258588342543, 0.93802155723225, 0.597248394550004, 0.987951967656024, 0.762348502525266, 0.185615868606856, 0.516770005620535, 0.640152095210696, 0.868950900520154, 0.600118369540063, 0.999669379464957, 0.129176300403452, 0.293370644224903, 0.476879373687157, 0.129582432240684, 0.228123465100651, 0.181072044279788, 0.672015577816978, 0.325784840279706, 0.790587527491233, 0.546013905045363, 0.23055656089269, 0.68676246918501, 0.0490826014713077, 0.572598535076368, 0.993762320965993, 0.362969029951081, 0.558720660200442, 0.92728035439936, 0.259876495471737, 0.832475026591379, 0.175125904798296, 0.879842193769806, 0.279632764390449, 0.84947600134364, 0.806716015291715, 0.189125152331529, 0.978558955309026, 0.2537276599274, 0.778546128041703, 0.810784541101316, 0.497952947745918, 0.915723151614534, 0.100983542995495, 0.135824589221803, 0.858316796023346, 0.0186464848449687, 0.728327979461209, 0.816657869237454, 0.116220449710662, 0.488556380805457, 0.347448406526296, 0.773890998889309, 0.285286735469464, 0.824396709548242, 0.173061082408738, 0.863643463462434, 0.926054760985941, 0.191103739299451, 0.0503524940495049, 0.54844947195608, 0.0095178987256791, 0.828363667546585, 0.90854440365025, 0.705296743968039, 0.243330313461752, 0.589231590840864, 0.938899963410325, 0.895502012450452, 0.60622620517904, 0.792378324816737, 0.836167527246825, 0.560141090539235, 0.000695103972078281, 0.251275130098045, 0.431948595536914, 0.43065286957651, 0.875233039277568, 0.0676409487514369, 0.33990679000736, 0.177062066596412, 0.246035200800865, 0.727819490287377, 0.288204888870018, 0.237341669493572, 0.139659015042569, 0.220730281527239, 0.817546704707282, 0.378229055354728, 0.66551521926226, 0.935813288497315, 0.42648111768064, 0.605160035122139, 0.969584427590807, 0.364650952267355, 0.664447578762588, 0.314755083048005, 0.885715838134897, 0.295544730059373, 0.295366080158357, 0.90708206682137, 0.492884587443052, 0.207139650632745, 0.0258192482641476, 0.170208936858787, 0.302340547746726, 0.961439610517837, 0.783657434008827, 0.673256460743765, 0.840722266443398, 0.420854280434659, 0.703737349590271, 0.481286619331981, 0.820575982939573, 0.437390204257175, 0.859973843692663, 0.00889723397954534, 0.996031119442512, 0.275201060262919, 0.0335164097792511, 0.152099956273812, 0.479596154427735, 0.888842341112199, 0.283419108447771, 0.7841270274749, 0.170392500862118, 0.731293549857125, 0.246621834068679, 0.0818409262479641, 0.852908745231369, 0.686552160189959, 0.958110618863192, 0.985055604524747, 0.534722618542974, 0.905980449711056, 0.476648620409696, 0.889671251769614, 0.0238497596433673, 0.128724832540048, 0.740640204617028, 0.391020205236447, 0.393628460946228, 0.601848346624112, 0.0238854731844742, 0.387670982908419, 0.0567461275568246, 0.689570563501308, 0.520390906147981, 0.0760864589193043, 0.406259745646239, 0.430212874307518, 0.65931227475277, 0.218391174494733, 0.0810886587320904, 0.467671672157908, 0.728671716514413, 0.613983836213877, 0.900597744618627, 0.252779362261954, 0.220095317728444, 0.235804992489581, 0.810129509832692, 0.702409051954609, 0.878504857620729, 0.711011230032179, 0.661080239559753, 0.439577433804283, 0.562793600207276, 0.990311323399718, 0.957315411567651, 0.336818265944374, 0.155455797421629, 0.417079119883688, 0.554477016050884, 0.0297521187908203, 0.866706271697933, 0.894414243427353, 0.49719038432326, 0.209287003220931, 0.721597132004682, 0.69971610349797, 0.460747231191468, 0.170593046636282, 0.794371448631307, 0.888499638996542, 0.142514517556934, 0.939671596272712, 0.000260855170530649, 0.438805016030051, 0.0111677697481697, 0.234515506923353, 0.706984202646605, 0.694164365209663, 0.265120839031383, 0.326573618038929, 0.093233937019789, 0.0415993525173467, 0.759280151237716, 0.324874493506119, 0.521821812384203, 0.951887882170099, 0.668978416949832, 0.473121620923222, 0.649372159529437, 0.665635914258534, 0.230147442427046, 0.804889673779446, 0.502785045522651, 0.434029713819417, 0.709033583511777, 0.555263985885739, 0.255003789510392, 0.0752005683697703, 0.589102379922289, 0.674242070575373, 0.588013412658126, 0.533702257256621, 0.472616371195739, 0.32381119585237, 0.430986087686128, 0.764897800510776, 0.996462675076854, 0.265018285144099, 0.792067001328556, 0.134609260492325, 0.0355361484202162, 0.453946488913241, 0.562977064850486, 0.230882795329875, 0.144762424745718, 0.677095829206699, 0.354070510682812, 0.510472306979145, 0.694706652140736, 0.595292948344732, 0.506110755115878, 0.776399255277813, 0.685202743605319, 0.319493640474063, 0.832142882335474, 0.660821562164636, 0.58706822058993, 0.0190784299669665, 0.836950030872697, 0.113997864648988, 0.84042044696273, 0.337496725855252, 0.429435629479966, 0.793778670348512, 0.14156748123215, 0.608101843262703, 0.930628822268718, 0.63644376192001, 0.0477706574946866, 0.698346125142574, 0.257109528349725, 0.69090150378119, 0.363179771180166, 0.541884227343648, 0.222141917491138, 0.905487329016586, 0.373541166741193, 0.0788164973630092, 0.0309299369336496, 0.626900804567717, 0.936996636793824, 0.319935295747151, 0.74517472806789, 0.288928404611631, 0.124354043910244, 0.514080993552528, 0.38751186915977, 0.978609763754846, 0.291817737857962, 0.592342563565038, 0.571356443653253, 0.615855992133189, 0.560914262096457, 0.279859395181155, 0.151849633345938, 0.733012505524816, 0.0396985204005581, 0.80162230631508, 0.406416889727256, 0.8300114375863, 0.971035485409373, 0.422483055378165, 0.0539048468490123, 0.787036774387481, 0.630077021670898, 0.508819384742858, 0.567686157113902, 0.309190867175215, 0.747135972885189, 0.336660850217914, 0.883354376608119, 0.333655919554046, 0.796014568867704, 0.804783828780175, 0.0502386763099912, 0.402505130841642, 0.965191642894027, 0.414375943507169, 0.330523570628246, 0.636372354340482, 0.858025370209453, 0.788283984026919, 0.538409743181206, 0.741670446331037, 0.739720607113103, 0.44479460682372, 0.850409858251714, 0.494534859776397, 0.948157534812833, 0.797788189494096, 0.133044583459622, 0.23776975438524, 0.127723467094002, 0.534120398947354, 0.0566966899581766, 0.468574998383849, 0.982687802974098, 0.682221161596553, 0.870900718064827, 0.496210565449876, 0.287662210628729, 0.183019317387167, 0.961219012410983, 0.756261475709856, 0.950353490513984, 0.67126664903198, 0.186054610125211, 0.289841958143097, 0.0582078244969975, 0.0440662224101041, 0.279558831325007, 0.170469620820808, 0.677100559769676, 0.902417928326231, 0.351987321656847, 0.100158516082145, 0.79300223573447, 0.468432429034969, 0.687341206893558, 0.683078528747416, 0.421435885869654, 0.659342526120964, 0.633427129209305, 0.52514600876635, 0.191281449822153, 0.511906009740829, 0.748330262803477, 0.419826916666033, 0.737479419477784, 0.68646773390447, 0.955726012542669, 0.307906461330165, 0.3423008625495, 0.387033572005669, 0.0261759053584362, 0.633546405117132, 0.565251059045207, 0.365511412831231, 0.421458698859179, 0.462200272029309, 0.314438776057226, 0.225661691348409, 0.115936884070751, 0.133966105678184, 0.946523443660567, 0.492432971268393, 0.748213626946422, 0.310935159548769, 0.0077052847671536, 0.615621671090482, 0.357573339280257, 0.648829071056901, 0.845977228606973, 0.101353212019747, 0.58830183432178, 0.582382188606586, 0.521803760752112, 0.454599829876523, 0.966702005063991, 0.18230695765641, 0.559392128803714, 0.626474489391048, 0.0301299972742376, 0.64227305727191, 0.838161131726778, 0.395136016606464, 0.43734220305824, 0.0848827780712836, 0.581377474019122, 0.871681976441323, 0.20690707709101, 0.0946100322363238, 0.18795219943332, 0.667836576742059, 0.73242090550251, 0.63101044889175, 0.468206154981374, 0.314362331018771, 0.641466662312382, 0.562026895384302, 0.663726669250216, 0.124877044414676, 0.67402717520716, 0.680311278220873, 0.930120144061947, 0.828819691141441, 0.178100354657261, 0.949240627233084, 0.199622848941959, 0.217680661784721, 0.00852193725918504, 0.300626578445116, 0.274470822730138, 0.586448450735631, 0.396552847483153, 0.186402773848738, 0.392816398668311, 0.154593839817764, 0.832304454369848, 0.149752074201497, 0.679545951702308, 0.750089983488543, 0.0148900082131522, 0.142246498113071, 0.0306556042449693, 0.808695512737925, 0.0471459553099438, 0.666181478102293, 0.456032435583132, 0.802996379286108, 0.358438905820941, 0.0253340451044256, 0.218588156293116, 0.710914900034793, 0.54892078628425, 0.869693032693991, 0.738835655794053, 0.103230132429703, 0.971433214284749, 0.738712607549068, 0.0138575768250171, 0.67912166606694, 0.485502731044505, 0.0347183504662847, 0.114439620520798, 0.076625110649079, 0.607824678326836, 0.233465851995181, 0.142011443549023, 0.530173198263413, 0.930301338228933, 0.616902770299402, 0.689891917613728, 0.234425866671782, 0.206843859923497, 0.0373662659169114, 0.663364965978368, 0.507254570626257, 0.485800449327184, 0.861695008718736, 0.965514904446286, 0.759687935323131, 0.246679448091837, 0.676267461003008, 0.759305444740498, 0.586189814048359, 0.822439267056292, 0.926835145406699, 0.637566877942551, 0.0189378674782367, 0.184340631152521, 0.026740709202532, 0.934998671026996, 0.484467358343172, 0.794785559035756, 0.456835340091208, 0.494209260917299, 0.4257139524986, 0.891772155131654, 0.874285119942708, 0.633150369522766, 0.292279312795091, 0.291678340387292, 0.245500043671933, 0.70434792021815, 0.774611872928455, 0.822653561158606, 0.806189645733725, 0.253770953483364, 0.80049986076873, 0.62970872246262, 0.231501567172398, 0.381076761554469, 0.453201640402185, 0.500559558109715, 0.540571545174246, 0.345145158635824, 0.291430028036247, 0.703528590277232, 0.93407841453068, 0.594174757294573, 0.593090772744867, 0.738731033860174, 0.372643183529079, 0.453270375606459, 0.509375564521365, 0.928710835304755, 0.450787806648045, 0.637434868053735, 0.459363857824156, 0.553774459414757, 0.0412326589245212, 0.802202178906401, 0.671511634916904, 0.200327594790571, 0.0145863734915734, 0.76316602510702, 0.56751175986235, 0.654213800806848, 0.63474038850615, 0.734330200742573, 0.887326579451826, 0.665909010473654, 0.120674601432719, 0.71373559183833, 0.447227179879924, 0.640982352772434, 0.648896128616056, 0.678375394497192, 0.0588276390680549, 0.223953263596582, 0.167981576494888, 0.928529406713592, 0.515230990757997, 0.673363516961032, 0.667800225131382, 0.0814614251978483, 0.225073469916587, 0.284189209478045, 0.913763725215688, 0.229101565004004, 0.248620409342287, 0.676290754129885, 0.648490661588466, 0.703676852593494, 0.837103279031751, 0.727295703022793, 0.0194343898051922, 0.583746478987147, 0.92087682353205, 0.0916780067464865, 0.554560610172389, 0.930802794741394, 0.199452943385522, 0.525319584672584, 0.126739107842297, 0.748727280124328, 0.314668692442591, 0.187174131320456, 0.980395126252835, 0.71282713314025, 0.825649989994876, 0.683219819258806, 0.689401802033395, 0.53087588184484, 0.188853701862766, 0.493015706762197, 0.559539866509234, 0.828280270881282, 0.202868606875368, 0.527746219640815, 0.807811207675693, 0.355104459020389, 0.547755034672239, 0.717437841139437, 0.202512364860898, 0.643756898766352, 8.90159834356918e-05, 0.513176980195556, 0.569829013093084, 0.00703669992094036, 0.892472453049291, 0.833193605858737, 0.775165459613869, 0.788135636370071, 0.37362958037254, 0.15231197775643, 0.352260302965304, 0.645097923322399, 0.931936746975513, 0.0933210352216629, 0.738777771583748, 0.0552673585389177, 0.75715846949648, 0.463489355112863, 0.045004448249091, 0.842029504434635, 0.164711562125754, 0.115073935394135, 0.271560871557272, 0.314386287381797, 0.606089623287801, 0.67746670667879, 0.987642256866403, 0.993260751994598, 0.757536341832873, 0.275059606551679, 0.954013744692784, 0.411082374012313, 0.216623284180289, 0.62913007873505, 0.0148690366825931, 0.043299786169118, 0.180262133968697, 0.200465093728464, 0.719357562918045, 0.442867561020517, 0.845492629483514, 0.389790101861989, 0.837552198504066, 0.748418652975973, 0.583861044484139, 0.160573597526479, 0.528840185856415, 0.462670904301894, 0.379543847380014, 0.0932299868200384, 0.259145447629789, 0.335836642789911, 0.37511221313212, 0.146259655090884, 0.31676988022765, 0.28122540614529, 0.820123624171821, 0.342089658039549, 0.872002145466778, 0.268029488340617, 0.775431767034123, 0.609208994494494, 0.0179688396310653, 0.701511447274707, 0.0157523069350503, 0.683893123014725, 0.877901635820724, 0.43180606747593, 0.631342915218935, 0.586040742683134, 0.227411240912157, 0.784654680221485, 0.287636296053338, 0.924375602736177, 0.298475138054529, 0.536303186318446, 0.333647483151576, 0.237293962150814, 0.545239226453039, 0.107685623177518, 0.137155315532942, 0.0944274786091438, 0.31088361729522, 0.812636385775984, 0.513557684885926, 0.866476420992838, 0.980487206685855, 0.326652641684787, 0.219616874157853, 0.518223740905415, 0.353005851033885, 0.859477144406444, 0.260020654604617, 0.8437210310039, 0.582801554325491, 0.719405533572899, 0.343523776841576, 0.00927391809591305, 0.19244647774415, 0.507387782950318, 0.0242418385771282, 0.550041439015746, 0.276184101406823, 0.696241209175294, 0.881352814657079, 0.0246814022448006, 0.341206460019793, 0.421405677531719, 0.0800111586678662, 0.0793538091733186, 0.0723762491986621, 0.900341006081089, 0.969308298866411, 0.391306958811561, 0.313585104867255, 0.553313579937445, 0.792048062497391, 0.79831408308021, 0.863258540559901, 0.797962204447309, 0.00637641023078572, 0.919901363134868, 0.0179622413055945, 0.029416479877065, 0.711451916368299, 0.538414186540141, 0.524588580518976, 0.502239930416174, 0.0656724982252748, 0.230040623758099, 0.117040319812544, 0.989771061420198, 0.0705164910828937, 0.0600236559152354, 0.66157710152913, 0.344057769477769, 0.128513348233833, 0.637135663847885, 0.746488169954789, 0.805267297635433, 0.269027131586068, 0.433979778983359, 0.401806487264135, 0.608142339434952, 0.770087452045645, 0.941123967428511, 0.131464787784357, 0.255722726526904, 0.378310807716707, 0.993735695560811, 0.34095124405104, 0.899552101502665, 0.23748161210208, 0.220001082833657, 0.991076399886198, 0.951105052081144, 0.638216769384198, 0.504130841148177, 0.358118346208246, 0.768452113958302, 0.784362887821808, 0.0289180925420336, 0.0524412345274602, 0.323418983975215, 0.799686624804644, 0.630229767256509, 0.982797647953506, 0.158249736798077, 0.616676231899861, 0.308717217656907, 0.0850785064114937, 0.876829557243819, 0.590701438503017, 0.303123137416162, 0.168070102311015, 0.339876753088708, 0.0674461137917849, 0.653489854830751, 0.244285574183342, 0.757918233246777, 0.299014203915666, 0.418815094624534, 0.0558321937549995, 0.0308563198786532, 0.186580384046202, 0.270905649220341, 0.0662432918006509, 0.333493692462697, 0.0499055727055313, 0.101308500836591, 0.0718697755970464, 0.904513016878465, 0.311797811962142, 0.591768922699387, 0.377625134145656, 0.951563836368381, 0.742395178298399, 0.887093362945226, 0.598896155566852, 0.689629863433133, 0.391116504445376, 0.223273683828833, 0.131260028898856, 0.172478324001365, 0.938597900727205, 0.0344819532151366, 0.879556991693605, 0.522327449139686, 0.808247366979281, 0.447579732479795, 0.3465389545978, 0.757150764154326, 0.360212924085881, 0.556606678601427, 0.888721254875391, 0.840163857925737, 0.244027206925632, 0.435618212770492, 0.828552855541459, 0.152703284564792, 0.757953934801518, 0.876508487446845, 0.502123593781908, 0.570266734272872, 0.851481954780221, 0.98464273563072, 0.790213551835248, 0.350682941737272, 0.501904841268809, 0.255212280298396, 0.509871647061478, 0.0786198523470859, 0.055899271874001, 0.634987866627395, 0.616903377733328, 0.0814734662398937, 0.609520695044004, 0.544670640165517, 0.855921636947752, 0.642689549026361, 0.271632826064597, 0.106750661663727, 0.0605134250619745, 0.621883006319941, 0.699070049717114, 0.274940030094985, 0.421969292814989, 0.365822700833342, 0.0821587214201359, 0.0697697882311092, 0.120889619303585, 0.677852996893229, 0.218936238263835, 0.23727993554721, 0.00863462136557691, 0.727111239870687, 0.561034186465586, 0.55197260834084, 0.742253575774765, 0.824633236038294, 0.393394708353574, 0.135943650190657, 0.195743209655017, 0.720299189967333, 0.669634926413627, 0.0598496163161879, 0.384276759913808, 0.491235941590901, 0.774161726265183, 0.14781115536374, 0.48553390640095, 0.071834908309533, 0.787443605814668, 0.393732114592501, 0.236161400683381, 0.0805676568832787, 0.079004760725217, 0.927428647748805, 0.204715342861412, 0.956580583362327, 0.339742016416888, 0.959966482630091, 0.343146068525392, 0.0634228372595661, 0.612473253316259, 0.555613386929755, 0.906450019303969, 0.201225333897885, 0.787693546613555, 0.347719087049813, 0.972150840935148, 0.836052426658696, 0.711805508140425, 0.214899515126572, 0.0154476582463164, 0.210043700170322, 0.536468405966757, 0.108458077330826, 0.853761688524587, 0.910851126324184, 0.333725254456889, 0.4691432331734, 0.00289686184822668, 0.310908261503356, 0.485222225839798, 0.262165409852364, 0.68311184607433, 0.911185720328523, 0.57050162228114, 0.265278004496186, 0.928471808831839, 0.0642447538087859, 0.626301671675264, 0.948278710955466, 0.516393381454792, 0.956703606596671, 0.272021260863624, 0.526990035653281, 0.91119242884022, 0.214685175702248, 0.826785301465952, 0.13832771691239, 0.437751439672082, 0.87528268019388, 0.704312297607399, 0.497022182813452, 0.609596865551495, 0.0778284015888121, 0.241432696279325, 0.0257841558368984, 0.846476935922661, 0.0148627961322116, 0.550415888412853, 0.119671157504084, 0.587949126612204, 0.622210944171715, 0.747017696574521, 0.309928090864745, 0.916504111010662, 0.782265972162975, 0.10104677456651, 0.281104551992675, 0.907505274069729, 0.111087158527045, 0.789819513768107, 0.102279555242489, 0.767943890319528, 0.66205542403941, 0.773097970039387, 0.931981429492658, 0.176884429007977, 0.0204217614951548, 0.833638920620638, 0.473193592719211, 0.108646270227394, 0.0890543781818622, 0.115129587656534, 0.167972858419593, 0.972510338313947, 0.286003635094458, 0.59902390228632, 0.85220948812203, 0.244784514355965, 0.727168400379461, 0.400790707528956, 0.310600564673083, 0.653799103461046, 0.391825199976143, 0.119336032120018, 0.978538160243678, 0.518251794413838, 0.507768656462804, 0.325021879978474, 0.197271748490752, 0.862892878085443, 0.267868293585116, 0.538680832527775, 0.551465556329815, 0.963256972470723, 0.269575514209685, 0.265717741956869, 0.309989783697098, 0.557297634365454, 0.378189288877086, 0.245765059883093, 0.676179118184178, 0.261294314285037, 0.96148625715627, 0.497576728263512, 0.427772303103763, 0.56779207333315, 0.613038459346904, 0.806993254654286, 0.880783416271628, 0.159978307447974, 0.0891769254299106, 0.313134600651602, 0.60735274434765, 0.378738421986569, 0.92554276486772, 0.0570134857183929, 0.994015911858732, 0.666958072114279, 0.617149959674567, 0.147885774203755, 0.0289961774769988, 0.572470324243455, 0.0299815618859215, 0.711533976371191, 0.214038344147739, 0.255186971581086, 0.958593003689882, 0.181890557882741, 0.798456590982073, 0.040042650394728, 0.309159366037418, 0.967092453004442, 0.266990197120926, 0.12187556219067, 0.607377800028162, 0.64535244047793, 0.874424628909035, 0.897329193515009, 0.053177558225919, 0.759513216837703, 0.858903964834273, 0.481723192247121, 0.0750964951102085, 0.090721227330344, 0.200215829922983, 0.778286689704801, 0.407279566291529, 0.827041644876579, 0.147967967728047, 0.732914767362224, 0.149891745797385, 0.838430343763336, 0.545366948317161, 0.961168135399345, 0.0754952622953388, 0.933833391937951, 0.359424082810482, 0.0669303956066851, 0.965933497190346, 0.649452905398687, 0.420616821916022, 0.596440727235821, 0.347088437706129, 0.217626622893063, 0.0378412965054163, 0.153425431927021, 0.373382238064904, 0.876815022150563, 0.441148312011565, 0.637769803608918, 0.387806681892762, 0.837196029917218, 0.766317274370885, 0.125584268559209, 0.0926611386914012, 0.660318282163125, 0.377670384389884, 0.10779234730228, 0.62381096330808, 0.621600567453236, 0.911627435200025, 0.606131759389423, 0.485345571751595, 0.576740988471584, 0.98210116282324, 0.518568287261806, 0.348701725027223, 0.982874314821169, 0.992010282672365, 0.896157434562775, 0.741580342770005, 0.664736632915057, 0.733362696825624, 0.435916314688945, 0.399632174139783, 0.217891212061694, 0.881629994790013, 0.942069664070742, 0.948954859871267, 0.106442832162301, 0.0579753249674162, 0.163723955923687, 0.527473920741545, 0.0697480922626674, 0.0101457500922325, 0.0916800250418189, 0.607435016165034, 0.0560854959898156, 0.658100429987502, 0.575819894288036, 0.0738353604633349, 0.241569248284151, 0.351195035228651, 0.126738675949934, 0.739408639260099, 0.445679823858157, 0.0176116164539244, 0.681708279346507, 0.880833084532577, 0.139821021807629, 0.810853784598876, 0.252859672695871, 0.0243856516307096, 0.530723581584766, 0.934854674058909, 0.769328503579249, 0.404752312321499, 0.717810925286434, 0.481322623694167, 0.441175792767957, 0.94236082852963, 0.578778389737531, 0.783609700108995, 0.714807050474105, 0.142166183546158, 0.692918534836627, 0.393648952904216, 0.127767835277924, 0.640904397431641, 0.497508484654534, 0.0639725182542102, 0.523537173118404, 0.314539848571178, 0.854891947878158, 0.00901510974662856, 0.202585480488372, 0.880604292511579, 0.218049700543528, 0.317689684176363, 0.343963688281966, 0.800077397173464, 0.571084463055376, 0.813238457687556, 0.0861796053600588, 0.192137486939703, 0.629501671560865, 0.0638634547512725, 0.742962315665952, 0.603397395502138, 0.798444254766518, 0.160594753333092, 0.737930806430807, 0.75703951847586, 0.881311092352293, 0.429242036405712, 0.187482780290635, 0.303031561193805, 0.320757064660418, 0.442006699341834, 0.361884766096166, 0.625600056392859, 0.938826545120144, 0.286904020014333, 0.244599449361359, 0.920853948389995, 0.926216579675972, 0.30994122999392, 0.0764753582490928, 0.780877963798827, 0.130745166372061, 0.461481818561445, 0.440169171435722, 0.60500184816658, 0.127020527575341, 0.599044138961853, 0.132125676004141, 0.115336587352134, 0.441073345438728, 0.271099417235113, 0.944106928864162, 0.51664302112176, 0.445291982243803, 0.308588132649913, 0.469597538616804, 0.779443165720922, 0.168812869305858, 0.0224509620376667, 0.708729139571642, 0.605149722286496, 0.413680063638153, 0.914770234776412, 0.477180777576018, 0.91622174734147, 0.496906772495132, 0.927553137958054, 0.703650986777986, 0.694642421434371, 0.00459360434395373, 0.540219302150664, 0.412970393183922, 0.628129917955273, 0.543535701836705, 0.215998008108597, 0.552104837644301, 0.862748824288905, 0.00493932243497441, 0.125084158007945, 0.775070711440501, 0.0203863534703393, 0.142632778936698, 0.320410715042542, 0.904750037966952, 0.614155588162246, 0.0158979285907918, 0.311394411727353, 0.192772007847912, 0.275467987331362, 0.0267102475463046, 0.830640161087832, 0.764278223471135, 0.72115718346831, 0.822717372361506, 0.764543882483295, 0.120566995706688, 0.38433159750701, 0.0279232074371637, 0.0013044828529255, 0.00699797569458094, 0.167784239524609, 0.150435728448108, 0.515494000947287, 0.418882510439806, 0.593812613595044, 0.276980912513087, 0.592827064229147, 0.585140260127604, 0.121590329739679, 0.0785936180850281, 0.0320662475929582, 0.168168404658411, 0.524171185536537, 0.841368646141704, 0.306412010720835, 0.191160545320537, 0.0651194074691477, 0.946010021399962, 0.244766216743813, 0.625404182641001, 0.20667350246779, 0.241743103980946, 0.743412596909233, 0.156003815116099, 0.823172279483205, 0.0939854193486277, 0.979797721442637, 0.653220746976476, 0.284114301258602, 0.867232798681723, 0.0301398455804008, 0.760938377788029, 0.896176970225063, 0.618874649449497, 0.720745500458894, 0.463850005809108, 0.758886423634387, 0.00714197495850233, 0.193485853582188, 0.0914517743944392, 0.167932589381953, 0.942248112969339, 0.84943222175199, 0.13728307287999, 0.660725417770123, 0.0384896892325469, 0.14816305935392, 0.184764102296017, 0.645361939256566, 0.856553283205376, 0.853375704153819, 0.494038071987485, 0.780877447257763, 0.274653814871974, 0.96737974195428, 0.990529334302469, 0.0844490485070604, 0.329793416868381, 0.268298765895231, 0.215922817165247, 0.767021757886534, 0.365054440466852, 0.154017276232375, 0.165021996898072, 0.893363970022434, 0.945789807188418, 0.517467386897568, 0.0183733967109805, 0.419305823607178, 0.122085149179939, 0.951847854754233, 0.270180347872545, 0.893653230674012, 0.895191697751348, 0.775532080849597, 0.908820927880488, 0.0166803775926252, 0.959288723057054, 0.0752526336204783, 0.285133462638043, 0.488117136617045, 0.970571734826081, 0.568113287842601, 0.273970340414811, 0.1290833606079, 0.155192060803672, 0.268811830307459, 0.433889439762103, 0.395903461208209, 0.425246491332236, 0.614389269174566, 0.446805397974927, 0.0120233375739399, 0.0342708948659173, 0.398770447090456, 0.979450081266238, 0.542768911276629, 0.912387045873848, 0.324159020475475, 0.290516522346404, 0.878973846599332, 0.779144192111784, 0.394412107425938, 0.882325966968808, 0.425668474518982, 0.46637798245207, 0.424931183524876, 0.455210042041102, 0.695424557384927, 0.0955782281253956, 0.733460316740176, 0.130017518881015, 0.496528714068463, 0.259658484595181, 0.591666675318971, 0.227618177760208, 0.955117711267026, 0.0630095013700224, 0.183247159943892, 0.624241987240634, 0.291405775713711, 0.137187421974728, 0.188933463031362, 0.956617051122946, 0.296613092041325, 0.237202875317307, 0.660157527673311, 0.00429596790761244, 0.07816846795587, 0.309584364488481, 0.223970707035416, 0.695289398523166, 0.700231895352869, 0.851815285589269, 0.559059596010345, 0.570332384497938, 0.530920326518782, 0.858886344721014, 0.446345838109449, 0.383381236368961, 0.595358732788148, 0.305039973652132, 0.101107780387475, 0.371939367447167, 0.222155311725768, 0.643226627695753, 0.776656226674668, 0.430034052361366, 0.621061527594932, 0.279074659517896, 0.478047779409584, 0.314741590045131, 0.827169904331262, 0.74316126578952, 0.144311190410344, 0.0682650940508772, 0.532136538431755, 0.743955283414338, 0.835733619084368, 0.720662984500743, 0.887622612352668, 0.162853378124297, 0.201239844993438, 0.881961532336157, 0.461958935455464, 0.187049833613823, 0.724882256383909, 0.883552638018478, 0.798011250941417, 0.842972323418388, 0.538046282182582, 0.209589073928846, 0.530445456579364, 0.84330239392576, 0.215340306900658, 0.716725877028947, 0.86266831762266, 0.471264328646535, 0.460504028586852, 0.942528214362927, 0.628342588725861, 0.80638662045252, 0.401540205625168, 0.321394345466324, 0.378806067071634, 0.718969487975335, 0.193064044961898, 0.489986781012221, 0.120770798507873, 0.139919889520453, 0.947195378354738, 0.630943387631282, 0.438655678779185, 0.707761622544099, 0.529646377418489, 0.652426924602975, 0.778184273108272, 0.618017877263182, 0.502087361950392, 0.485372468920494, 0.110149645563402, 0.91654985716382, 0.432618594297892, 0.426998257680196, 0.0292861332860498, 0.51042636095327, 0.963702869001799, 0.275350279314674, 0.743411440154786, 0.116802548593782, 0.769330517932145, 0.765719495409825, 0.540766257734286, 0.169318248456869, 0.93012047363673, 0.208208513575198, 0.0197606465104744, 0.601611188055368, 0.607371066617602, 0.4688548991388, 0.596313583634214, 0.723458708734656, 0.408110649599894, 0.597613851687145, 0.425709626784973, 0.121841610054854, 0.349953328021012, 0.546745606752016, 0.127099129507106, 0.0814965823838198, 0.289457172204747, 0.67681622658634, 0.457042216286926, 0.499215085156717, 0.0584782442632846, 0.0736061307506023, 0.337876779402104, 0.336071153323714, 0.984959665409757, 0.751663921406205, 0.646531020199834, 0.882865343487551, 0.511889934480955, 0.376202813132731, 0.883762145497225, 0.24576182202781, 0.164003277453222, 0.658603851474983, 0.255618231731193, 0.157603230265043, 0.276605782073797, 0.298708748274398, 0.00590432741202496, 0.228651995951985, 0.670117791792113, 0.981665602559872, 0.256148780172368, 0.619172654310474, 0.971829486521239, 0.492171259272353, 0.301026941658217, 0.597096973961872, 0.543382815076756, 0.129770765120927, 0.55221937777743, 0.14906418420329, 0.258548884361818, 0.528711898367735, 0.813376698188229, 0.718534241606903, 0.500819312276497, 0.893981323762822, 0.95071669894304, 0.462355395252696, 0.28759735155742, 0.926367097537817, 0.496328199447024, 0.877330049068642, 0.213237925120459, 0.114401278721662, 0.124315525786611, 0.574103013949968, 0.466372230199766, 0.374038623079248, 0.754615155332284, 0.455602106460509, 0.706028329831995, 0.929507923793932, 0.820465703644312, 0.0398475008924675, 0.966188638437053, 0.887968259555916, 0.638773792432629, 0.0109540987208894, 0.977258694419572, 0.385528281541337, 0.399658821860232, 0.510288092757317, 0.551630339117072, 0.958893447208648, 0.85540143918067, 0.476705428977696, 0.877815542966812, 0.389877844848677, 0.0491625166931662, 0.307434799472219, 0.320209671426157, 0.461295578948575, 0.488133586643543, 0.706741368286084, 0.367297037907044, 0.941439399625806, 0.268187523408796, 0.745395550292589, 0.909073314307629, 0.584552542229158, 0.906503297816296, 0.0566830717484283, 0.556736497238052, 0.00296136075400477, 0.153156257004478, 0.407491290302073, 0.233445870697115, 0.535313772787948, 0.311606182586189, 0.708634397084473, 0.587060544537307, 0.239191240738768, 0.0640589209508431, 0.0033691934264205, 0.189072843491816, 0.491203688530959, 0.616468850967163, 0.122776013145099, 0.133031538749759, 0.314180831498018, 0.0878551510562618, 0.408793332949964, 0.282319105614927, 0.630590383638054, 0.219755413178204, 0.65425375851299, 0.193468416822936, 0.0946000464074753, 0.918360127253356, 0.558239594779947, 0.417641271492223, 0.110447306861044, 0.552744418257712, 0.111280571103007, 0.236022456116171, 0.138654121679806, 0.328239997535495, 0.346442101496025, 0.928756689344785, 0.888316654616343, 0.790044485454323, 0.0744703335924647, 0.41979729089607, 0.396724358856348, 0.412930642589426, 0.973954145692078, 0.237895162364785, 0.108576388783756, 0.800029106714939, 0.0448125779585653, 0.959956315365957, 0.801644262492697, 0.960595897630585, 0.765748673841807, 0.608232074767387, 0.9890415917446, 0.562556097213718, 0.495888053374059, 0.253041287481471, 0.195961209739914, 0.492734528399564, 0.710884276910267, 0.987618686278734, 0.334619654178442, 0.220165223866697, 0.345280378717792, 0.0916062630297847, 0.631601067289842, 0.378188210113517, 0.887362939134, 0.426527447710288, 0.890774006089237, 0.224229567976273, 0.796929561338388, 0.858281244347888, 0.427096257688259, 0.588859679687696, 0.0916994883398988, 0.250582897455173, 0.593771407251358, 0.520579367380017, 0.914054034210805, 0.663292744921091, 0.499652548608032, 0.6348026447787, 0.0863604881892903, 0.040807195260917, 0.75156377398684, 0.32375671765599, 0.0664906497451696, 0.923256904886424, 0.0543605498699473, 0.0679425854551938, 0.573534391507379, 0.439198573941158, 0.0273687476054979, 0.049699303638169, 0.238552855684971, 0.127301305267451, 0.630328814655416, 0.47641417036902, 0.109997194842048, 0.435431700157832, 0.901878505901566, 0.565834019978347, 0.730643616149948, 0.980960067917655, 0.588208867682695, 0.503161441532323, 0.531593697337533, 0.606287214737918, 0.764317085947368, 0.600623976248048, 0.7225858265307, 0.435214945171143, 0.646524630120114, 0.176695444543474, 0.876476103556867, 0.25169011105544, 0.356484156279608, 0.303345779720641, 0.967759636815928, 0.192291561959909, 0.0978644229734044, 0.693789929723245, 0.0960244795324849, 0.199386377758388, 0.607054969735293, 0.709710956149306, 0.890527513633816, 0.602641520415634, 0.862453391628356, 0.675957201702169, 0.78499850597659, 0.536648173793507, 0.00444743616771903, 0.198611134560469, 0.438724687393658, 0.558291253088291, 0.38512949459595, 0.646858622490599, 0.585674110311991, 0.766331024429985, 0.174634715334084, 0.503114672761298, 0.382674936246296, 0.977198260848376, 0.936083615254719, 0.892736950625253, 0.443529651997787, 0.751526591561565, 0.909403150345774, 0.373317829808321, 0.116290623635322, 0.52512101396906, 0.34112513661995, 0.849402827210523, 0.186231546266546, 0.208744985543195, 0.530430923250666, 0.0842852158345175, 0.240769579655034, 0.634552631256282, 0.111075886244578, 0.607344676728676, 0.889757592057214, 0.490685670749715, 0.31350772523628, 0.762140386259043, 0.954526366192397, 0.645883779507662, 0.485319557851113, 0.438181870023572, 0.186382059337146, 0.443134295142254, 0.488119800375148, 0.861131460429389, 0.716455874206007, 0.620332674320607, 0.0897736340267153, 0.111761470396403, 0.57558928468062, 0.461657464107991, 0.545449335627208, 0.0380122580363678, 0.579286180434096, 0.628977874707396, 0.481927379227512, 0.160526983813516, 0.754378879728026, 0.793495953195373, 0.291369256387895, 0.507608302715694, 0.237992440581033, 0.192684623156822, 0.808697088764272, 0.0666732847580229, 0.195862462661189, 0.974781870456444, 0.575090438566798, 0.58321359654909, 0.950021650522013, 0.100039205551311, 0.87873381266647, 0.89145061664955, 0.214036775065934, 0.166952498390275, 0.646125491529177, 0.180343548889192, 0.116549999463432, 0.365334046675389, 0.402375080438805, 0.0898098879354711, 0.683768075343256, 0.546894246347622, 0.985958765691952, 0.212749573352599, 0.918990532146302, 0.665161658481339, 0.743478278603814, 0.378457412222976, 0.474768329821097, 0.106261719679747, 0.883784126443769, 0.445103232625583, 0.668228337858386, 0.728215802857534, 0.285643294824282, 0.593686745676459, 0.370530244558346, 0.795637627901999, 0.0735344904180074, 0.992737631294711, 0.776029375831996, 0.00737045747201659, 0.888948417525175, 0.864124974221574, 0.69759374178396, 0.884992083036526, 0.465644546914755, 0.371558450597141, 0.0830216037806415, 0.997218557152221, 0.869829858547889, 7.02606509638271e-05, 0.679976722440047, 0.654880065173935, 0.411813042019929, 0.476368370112675, 0.388271005969114, 0.452057047359726, 0.0267598392921459, 0.425286386334643, 0.105805291248756, 0.0431112647160214, 0.556122207555288, 0.611287045828178, 0.738832254015234, 0.256919275512097, 0.726258937222326, 0.713252933317905, 0.503764521794355, 0.892769581275294, 0.779385204668588, 0.494860164608991, 0.761507163829187, 0.667044704502124, 0.97811221049485, 0.117070843668811, 0.869323255064457, 0.567855212686656, 0.12015126122469, 0.970329418796307, 0.839966643717207, 0.0623166463528964, 0.278072556412672, 0.409629985963109, 0.626808766569199, 0.919103094307026, 0.415151690528069, 0.711500160423874, 0.970874751988099, 0.435472023774651, 0.635381943696805, 0.901555048064467, 0.871583689351562, 0.142715930387127, 0.200597552541351, 0.620927999134314, 0.775029187621826, 0.246811014604651, 0.369452210189405, 0.341097455013041, 0.456389502473899, 0.170823708735669, 0.745605964240426, 0.0979779191677121, 0.890269412978509, 0.0820941668349969, 0.781522651309255, 0.848677833156697, 0.450877380288705, 0.605325267733671, 0.35447004066918, 0.75909545635784, 0.589440857159745, 0.611569957557999, 0.843170652783054, 0.456987903973026, 0.882789712518761, 0.493572649725507, 0.62285064869631, 0.562165104273614, 0.721660876884469, 0.546769907378579, 0.784441413111423, 0.446352615947538, 0.911907966816347, 0.0865870491891041, 0.99893735558537, 0.727581093112644, 0.558406673201837, 0.142824132227928, 0.837636930910624, 0.725247343498373, 0.417379689589011, 0.273170685145915, 0.314707578771364, 0.58169603986928, 0.689358266199527, 0.863867634289102, 0.210858726345845, 0.506751619087609, 0.425049739023589, 0.239700081174437, 0.757523250196315, 0.79191190792567, 0.385879892626725, 0.318054372419322, 0.480925150906785, 0.27850591777576, 0.158036175111175, 0.395408012059265, 0.285456047665088, 0.243050642632938, 0.64251965669292, 0.956923735889548, 0.186335529100702, 0.909516914498303, 0.568740443803014, 0.193516472368835, 0.704238441073574, 0.651579457195294, 0.331896927501464, 0.521568670597584, 0.0247634387149843, 0.0222729215994639, 0.7985243352727, 0.107812258665461, 0.688807044774572, 0.775025689962241, 0.791493941945778, 0.709469039005541, 0.946537898012442, 0.298019804632059, 0.995821109677417, 0.554447262640606, 0.656088488910554, 0.945348787791639, 0.224637230410263, 0.981518097047326, 0.17062158632833, 0.941060182055896, 0.956960018391622, 0.175014728412052, 0.0543098138217288, 0.268208449506612, 0.546009709026143, 0.775186480064704, 0.603759426552689, 0.316096196697364, 0.433352559966111, 0.842779084444054, 0.626132723631131, 0.27254804988843, 0.514534421735357, 0.888400504895605, 0.174295939734701, 0.82006832483906, 0.18119187904963, 0.0972636725529937, 0.0313148061504409, 0.010175858945182, 0.967446859878319, 0.88803754399546, 0.0496853671951873, 0.670206930528232, 0.97662132555341, 0.220266122324996, 0.0840585998594031, 0.358864936659553, 0.733183181091036, 0.602144222103707, 0.451370809684955, 0.912803645297478, 0.00317726473438762, 0.510079725910721, 0.144460919968261, 0.504649898029852, 0.568214714082411, 0.642170222496799, 0.206933822779016, 0.340187157318059, 0.327474259513805, 0.614964858624015, 0.591189735572906, 0.687924102304308, 0.166437806630021, 0.65222747374801, 0.768637662519672, 0.395766371329785, 0.396264162417368, 0.692120226998751, 0.237672995940921, 0.948882711741866, 0.135752351893535, 0.51151548464367, 0.0414307574522093, 0.31240958945995, 0.321770064687923, 0.948322298178369, 0.207341231997868, 0.489342551555371, 0.444675270111867, 0.820848365714427, 0.0174802152143164, 0.788515375474457, 0.540790186663523, 0.0454764331563404, 0.513850444711364, 0.445623927363925, 0.491364929977122, 0.659889487202613, 0.505370240046695, 0.741859707505951, 0.943849643391484, 0.799325103200989, 0.437726417087494, 0.0903127352856981, 0.933568036446572, 0.378376301508962, 0.758922329171813, 0.803793411121403, 0.804236279837128, 0.595793580639453, 0.324986983453654, 0.748347919486055, 0.476471788327166, 0.515590608565341, 0.480094642335568, 0.424016840504236, 0.176017579899769, 0.280556598142782, 0.877671245907041, 0.873855478538618, 0.489586701259362, 0.606281630434558, 0.717687310260396, 0.140023498577083, 0.583053531245655, 0.195000055978706, 0.0633377183538213, 0.0639016711463452, 0.939866138223327, 0.520184381565328, 0.150116015220049, 0.646592273445157, 0.58378215852233, 0.312333771010346, 0.672144148864592, 0.397056210106296, 0.219515762004761, 0.240820779500018, 0.386375844873445, 0.0335012541173872, 0.346722778657214, 0.552042015144926, 0.558440039511648, 0.0668147552421045, 0.581939747175456, 0.475857710144297, 0.66193515607039, 0.78609758223483, 0.190027809046715, 0.961508557779012, 0.53506134323565, 0.914517510824429, 0.645663947350771, 0.153469407799863, 0.0521820026799924, 0.0150507779820401, 0.196362582832902, 0.527712285118239, 0.147671943137404, 0.0731336962951644, 0.554839047469919, 0.981282723902736, 0.162861126313644, 0.156598164551616, 0.163670621744391, 0.0784896107396063, 0.718400734123366, 0.446265291731416, 0.51011925366128, 0.605143599074304, 0.883071442682195, 0.121915548351526, 0.290255257586168, 0.0623420486313578, 0.143533097033189, 0.0679653582467269, 0.185086628538735, 0.768677360027649, 0.897677117128261, 0.518271728575502, 0.84463194173241, 0.228756576056597, 0.428373048036282, 0.368291142783557, 0.696047825343944, 0.944380682488048, 0.217886250517214, 0.753003796889367, 0.519838500271786, 0.760074517796761, 0.880869503011166, 0.237478171531178, 0.939693628311005, 0.0271520129645229, 0.561514248059542, 0.890452286211106, 0.16699116553042, 0.425242772902521, 0.326896110625452, 0.567590961568867, 0.826987155910331, 0.391681517419391, 0.621350792395628, 0.96515194219731, 0.828470631417097, 0.813402264899086, 0.955094546482904, 0.183119730988461, 0.67042831513269, 0.735596496186232, 0.819551375089305, 0.681018008104165, 0.330058787959299, 0.0346922085721765, 0.120188971138821, 0.283713492318524, 0.838235975027426, 0.901352664134446, 0.449947224375982, 0.988536344286592, 0.0337887880948867, 0.859450419795405, 0.528556841528848, 0.975458617246541, 0.877221510803783, 0.161335082195768, 0.353375270856092, 0.0580068150130946, 0.444116035212502, 0.441093588935784, 0.326700532971002, 0.914398314442574, 0.494556317190109, 0.710237402894326, 0.664641356836543, 0.831781417268648, 0.258035344311037, 0.769241439044804, 0.410567312449012, 0.0556567426954229, 0.870649079743115, 0.972300312775366, 0.116976195722055, 0.508313009040244, 0.755566126683496, 0.109986654949089, 0.0338635352558601, 0.441097675068658, 0.129783563321146, 0.971602452337381, 0.940153258292462, 0.93311964719537, 0.983758270635912, 0.69303580316046, 0.70247548464431, 0.371312184808707, 0.064105941014906, 0.443451755787831, 0.542987352374312, 0.540162073918995, 0.786284036629558, 0.601409398329654, 0.94701355426604, 0.631218455441575, 0.878837482885143, 0.122686301726737, 0.813548224731922, 0.977604352516636, 0.231243180933921, 0.912736296851293, 0.676367762339147, 0.725065986205818, 0.685789575526426, 0.796718908286193, 0.529626983369735, 0.75463911319108, 0.50355419131824, 0.907040337537649, 0.123610236573079, 0.852225217767729, 0.175045733740469, 0.764165979451581, 0.459332366225958, 0.606063682234264, 0.364039527915513, 0.401945069912581, 0.730074652107921, 0.955128305970023, 0.878798807305791, 0.60973179740691, 0.967765758815587, 0.634138147662946, 0.777264519129162, 0.898503838981923, 0.22656559049047, 0.101266633190795, 0.828259130516178, 0.424469986579628, 0.727096156947871, 0.936748077262738, 0.296532426169076, 0.312200158206679, 0.549696342345876, 0.55258162794108, 0.640400459430984, 0.596649124610099, 0.161174638425761, 0.417908447427095, 0.369055406474318, 0.0802646275537059, 0.236307873552953, 0.994339787125827, 0.909303861089337, 0.790786872622263, 0.135146256407473, 0.550339191242501, 0.0538383478382951, 0.67655349616172, 0.822280588758888, 0.911402101903129, 0.147205047718601, 0.304862058908776, 0.779871787267305, 0.487228420517551, 0.549174167593667, 0.878216598539943, 0.988290925907741, 0.0191522922847513, 0.351323772673551, 0.534646818915062, 0.74781776511188, 0.970013174675087, 0.695143092070966, 0.273268477416153, 0.951137954398432, 0.907670714576379, 0.457424511704018, 0.733171033450958, 0.558240068015207, 0.993894380815152, 0.389300333470762, 0.881683175016832, 0.893269564817609, 0.385999834968771, 0.727550770896358, 0.912649678011347, 0.904127074398526, 0.0814523436441722, 0.0333415578305712, 0.572409445459027, 0.00929021743182679, 0.198328389991404, 0.0206801420908972, 0.792271258713652, 0.190829412717135, 0.952082633878, 0.770226280990266, 0.816573533075648, 0.426050105979764, 0.300218158272779, 0.680290474244691, 0.858515669110961, 0.87179294021128, 0.897062484963166, 0.843586761732537, 0.485151224776803, 0.488540509868193, 0.0653339565150842, 0.456981036899934, 0.595555303695487, 0.110083465534542, 0.343784260490864, 0.264065147470231, 0.580180339875577, 0.435391082649911, 0.415100184764592, 0.0542130057394555, 0.259695618433796, 0.244178041093522, 0.70909508384964, 0.658850732432459, 0.11918134159428, 0.129219133315405, 0.0779686199278742, 0.735240088989146, 0.556149994390713, 0.83437118942725, 0.106124465172072, 0.256561490822235, 0.0541240236566644, 0.656553901761324, 0.277436581686551, 0.709130299501955, 0.130295521332358, 0.523997273889528, 0.0458901142277052, 0.101305687973029, 0.468935089858187, 0.710646362418379, 0.601346407503426, 0.204916976156494, 0.618786526718974, 0.699150113317535, 0.314828670017582, 0.804788645898101, 0.178022430412338, 0.241305720471971, 0.59472996886471, 0.868232914442517, 0.818668373595483, 0.590084132290161, 0.0022878412100279, 0.944532043448367, 0.792476901440052, 0.399328429265853, 0.87344949814295, 0.67439632589643, 0.242365605924339, 0.168444723953063, 0.861124801370978, 0.326641002443284, 0.325151694004907, 0.400173420848008, 0.763506119119905, 0.164067787517012, 0.688290348420654, 0.643316618717173, 0.821353525764952, 0.378546930330226, 0.11190066076972, 0.859483274373394, 0.713183106337416, 0.575010315474108, 0.741137892103085, 0.753670730566362, 0.5550881101582, 0.800912259027531, 0.67034598226316, 0.53510748096641, 0.685926116437637, 0.955183839598853, 0.627752290609928, 0.819131996520413, 0.188965332654366, 0.751475429847612, 0.221580499489749, 0.416449813116223, 0.790873596754484, 0.399961864586141, 0.0451092487106278, 0.201045341699827, 0.0428628361556463, 0.639163201462111, 0.251758621650634, 0.419883949824652, 0.171836238765551, 0.884455866252361, 0.34101411030416, 0.205123583989287, 0.795386865892225, 0.0244699833220734, 0.0397795442038889, 0.529818635947425, 0.933367035373425, 0.602214607998047, 0.601176727619887, 0.643866218140665, 0.921930576293345, 0.219257236865375, 0.884904384794935, 0.0488219833472131, 0.812039741124752, 0.0325917922804437, 0.415042468753787, 0.594666557339971, 0.196795280383026, 0.796091212333281, 0.210988217213697, 0.17441351850841, 0.613841779284192, 0.941056448782904, 0.661564946484972, 0.214898375165428, 0.913589800435506, 0.380196750008249, 0.0349822849409034, 0.227345098973147, 0.90688653803507, 0.868988066508274, 0.582468668870504, 0.560786270349853, 0.00108863140115678, 0.126862113264826, 0.214640967396051, 0.826819630530735, 0.387614281902223, 0.318075469094124, 0.261778535315275, 0.296523982892583, 0.428178043205905, 0.192406316320093, 0.567478340153144, 0.657665715702006, 0.685579440173825, 0.35374039113385, 0.956790526289497, 0.243738577256566, 0.748271712947556, 0.390227532169999, 0.844756632255404, 0.371983980195628, 0.707605986488196, 0.345655294339593, 0.877157906709213, 0.750527916904124, 0.434425334008016, 0.400660697960175, 0.372937849313976, 0.443250295360651, 0.38030740354687, 0.0727124423860978, 0.440280771542163, 0.606526772668605, 0.304784161642443, 0.406643755660731, 0.101011805991535, 0.476880141641489, 0.292410341839211, 0.0162825536223754, 0.884415714000928, 0.210310646742894, 0.683644298178902, 0.154390302217875, 0.974315872573114, 0.502662350320489, 0.891408193642843, 0.72649108314172, 0.497179645779558, 0.607214245653674, 0.349288812007961, 0.526214560852746, 0.796915184290517, 0.273675928230473, 0.292683596938, 0.348151885523119, 0.875972868527746, 0.0933191501421066, 0.206680406912014, 0.302759096720275, 0.351522394642866, 0.726747501555589, 0.886325273029742, 0.0519261904007149, 0.078559181348574, 0.0144860561675039, 0.193340128031517, 0.0262892797272799, 0.468765850520912, 0.243382618677917, 0.919916865953037, 0.715833638570424, 0.488645734443052, 0.0780742575248091, 0.5300661096032, 0.666050368713066, 0.607712617246006, 0.419551483098544, 0.110797646282591, 0.816614440783048, 0.249304579566938, 0.50917839903936, 0.804132672189402, 0.509830030671977, 0.302700003346685, 0.861926621438791, 0.0573943181115859, 0.730756828265569, 0.900879693637962, 0.464043368948885, 0.469363970061087, 0.153814130637761, 0.961794919704818, 0.876254676284256, 0.48862927749971, 0.407076533143994, 0.126576313737181, 0.925425280986515, 0.00558112226984153, 0.186388406230158, 0.324061031117453, 0.0501866065856974, 0.144536857301886, 0.729369435485953, 0.482279604115223, 0.338064748642705, 0.236767046375269, 0.450863206704084, 0.185450989625189, 0.324254811212058, 0.263971482620191, 0.830077173704864, 0.696360046650443, 0.333537393949211, 0.58023249180469, 0.287838203769745, 0.263970660843424, 0.259901265487771, 0.677080035574851, 0.519826549216854, 0.0767537345862533, 0.0558452550345419, 0.258728319443232, 0.439927418828113, 0.284289116400108, 0.678767849925557, 0.949579547672378, 0.773955089314808, 0.636139008645793, 0.753581470197562, 0.746816604380458, 0.586031153658634, 0.77305633570769, 0.392488258270265, 0.605290119072969, 0.247432873807587, 0.290202196580577, 0.0193325967231857, 0.347309670570022, 0.141795751740657, 0.411534708672259, 0.153068350318529, 0.829008913512915, 0.739199528478463, 0.0988703062858602, 0.82063308844874, 0.227111609190605, 0.106941347858146, 0.662775960208951, 0.954576107771929, 0.814564459224572, 0.623252783402418, 0.32826875039811, 0.2773680017205, 0.434376858879976, 0.350295544428917, 0.87775918980391, 0.00615449331121032, 0.696449233172726, 0.338063107415899, 0.304983670089253, 0.648100699635131, 0.921146277146987, 0.893073781095652, 0.997179327755608, 0.0728838912507408, 0.129618623816683, 0.981568230272936, 0.0902422890159004, 0.686158658288716, 0.928964801293577, 0.141816370950941, 0.884415653763051, 0.0197619612901836, 0.342685896626743, 0.238306087550869, 0.984564000963032, 0.846597902117332, 0.794511596394945, 0.900278066682986, 0.780104718722294, 0.836510771646748, 0.321093264444702, 0.74274557590167, 0.664494284996551, 0.289183690602033, 0.337372232977803, 0.90863705917544, 0.0322509804412977, 0.696357727955666, 0.208769815888889, 0.824149356327936, 0.218232263733975, 0.0996423029616137, 0.619505816445823, 0.103814782651106, 0.799061803589637, 0.902925119282939, 0.312512845986667, 0.281589166159224, 0.00678194126805909, 0.495871665666786, 0.988482243012287, 0.737940751495519, 0.31071981739157, 0.600407500351151, 0.781679826661115, 0.111533124684943, 0.579329289388906, 0.87037122031172, 0.689776249611124, 0.242970250921629, 0.342722251869759, 0.545439787975426, 0.0675726926893163, 0.410448399698545, 0.237511397229394, 0.488973556424727, 0.806066588301787, 0.377847951671179, 0.517978144507388, 0.0945978893449516, 0.909095192849836, 0.207630593997342, 0.38206452354053, 0.660280265178685, 0.758373302277572, 0.173069412392096, 0.517379761506326, 0.995338112303466, 0.707609406590709, 0.0805673009239308, 0.0433080963751025, 0.49115556298426, 0.446596440169203, 0.486798565845346, 0.165891122085342, 0.360656504813112, 0.880721941763713, 0.7443521799962, 0.416771050919451, 0.907354808475253, 0.0943068737715302, 0.181326261018145, 0.946587232183202, 0.100849056860381, 0.388038069974207, 0.289224862024986, 0.073086860797593, 0.194608005437083, 0.417485231103283, 0.29292655914394, 0.702138511791249, 0.239713177935322, 0.959482620708005, 0.305462202307119, 0.154915458213084, 0.555508419323977, 0.790543689112232, 0.443872293587844, 0.995818721727493, 0.436586698162556, 0.304441172973089, 0.246510340916772, 0.960825251654563, 0.222880958438666, 0.39560918146623, 0.224524553336017, 0.270024669015495, 0.418441479002589, 0.997735979773324, 0.911030403930781, 0.550426888716037, 0.59633581577638, 0.0791452393029073, 0.576636037390549, 0.898172311971638, 0.463310567259345, 0.398396426873099, 0.104461114798391, 0.65223638867871, 0.991700005353602, 0.678072815973294, 0.428477593633851, 0.654801321575878, 0.588745528605038, 0.745054910247788, 0.640870288169766, 0.503676186207024, 0.938041475805194, 0.605343580621484, 0.638923748179702, 0.702657414229967, 0.860949089103837, 0.379662417344194, 0.712122883405377, 0.523520435518561, 0.363489438793966, 0.434675873117433, 0.687597621532754, 0.22685114591404, 0.979003604971064, 0.97567093914987, 0.289513801080148, 0.338433319612924, 0.996444665038234, 0.788970526945116, 0.794921547968916, 0.632368873728348, 0.811450785576298, 0.448093250950899, 0.830613552616427, 0.126663314260811, 0.513268127828168, 0.715931252353248, 0.248146352936429, 0.531900325168972, 0.382209682936579, 0.801761242973036, 0.670860771332138, 0.982859184594668, 0.93682132862859, 0.576272354062037, 0.0801861870300866, 0.413824842108441, 0.180845024472233, 0.995631187232624, 0.520386194349656, 0.885282695498373, 0.648251521430719, 0.46625359036798, 0.0952808770093054, 0.967820799552412, 0.620111409375626, 0.156003156891034, 0.398367064139535, 0.882504945784612, 0.539013920213969, 0.543676069037676, 0.442452519844676, 0.183745286455007, 0.249230797808014, 0.285111552065582, 0.529509509421184, 0.559837373797065, 0.415145682077322, 0.905680322300139, 0.30514318607763, 0.649769679507266, 0.288781946588738, 0.255251450381098, 0.358183109875883, 0.805917511796414, 0.538926555930357, 0.590063613392035, 0.231141741381785, 0.101946250640533, 0.644644489342671, 0.980138726709384, 0.101722418921063, 0.187866127394795, 0.00938535849366362, 0.838303405638847, 0.481342368519864, 0.468530475415428, 0.908549525020189, 0.417147487867677, 0.544138005959048, 0.699008861851566, 0.0790502555821841, 0.509415865757464, 0.486898298871691, 0.855880283157922, 0.614404770599109, 0.117415253437209, 0.606872938986649, 0.164179812891284, 0.399025526799344, 0.532357346670746, 0.875211025381479, 0.659038502381112, 0.787625789822402, 0.12602500389977, 0.524974173459941, 0.899479765143176, 0.109084411966843, 0.634571295971392, 0.0808460113494329, 0.41123220026496, 0.712633901603284, 0.100422395686608, 0.809946815070828, 0.637809292155302, 0.898068922063116, 0.621821192936434, 0.414607487023785, 0.647558117291859, 0.48926891512772, 0.0937553111713212, 0.63732307644509, 0.950261518468529, 0.476376435669646, 0.602804994315556, 0.591482684426665, 0.225261740402303, 0.66843306699474, 0.15655171869925, 0.774251408512456, 0.213052708957754, 0.169123576924318, 0.725752760954544, 0.256189214315397, 0.162766631440149, 0.625777035254523, 0.25074747975054, 0.262978105723081, 0.843928120083632, 0.397420827099293, 0.104578630545849, 0.193854300807646, 0.363070415684153, 0.87453427738401, 0.599770225428786, 0.258113255999517, 0.358427636951333, 0.887540074096455, 0.900530589472715, 0.447952153747646, 0.26885145625313, 0.553815498449399, 0.178847600694052, 0.859702254226447, 0.23197620827449, 0.168105268247641, 0.026722869131562, 0.322430878368402, 0.55516455216144, 0.824528624620703, 0.804247540654593, 0.0244082181687771, 0.371505539208431, 0.491869874567966, 0.466052241640922, 0.0417138004048614, 0.617001825304197, 0.578032920363319, 0.298808845557799, 0.435666044620737, 0.13663303238639, 0.299720768591529, 0.761374768859731, 0.0353025299471756, 0.269500010509823, 0.99626851072718, 0.446945900333483, 0.152784873488119, 0.88622818237563, 0.0313803798336819, 0.116034938528325, 0.250855042901307, 0.759674945665036, 0.898309625182559, 0.223420487495824, 0.673299132809055, 0.818785808270458, 0.948929804519657, 0.874315327405466, 0.39373630249072, 0.936962316174899, 0.436895718503059, 0.162522500868464, 0.309820869331437, 0.681109422354129, 0.934054984035697, 0.947389231624043, 0.599083105821127, 0.94887414013056, 0.404018260019854, 0.0410324442136993, 0.293825541244185, 0.0319394867240307, 0.864512967993011, 0.432546386123055, 0.092808540567106, 0.137791693256157, 0.241971198905531, 0.22300904277175, 0.867740047794131, 0.764176703620248, 0.344714491619246, 0.384780180070239, 0.594865201594655, 0.535120729712505, 0.333555724446053, 0.854682575498013, 0.265644013654779, 0.933940179367925, 0.389834124949842, 0.683065408830429, 0.275009792425157, 0.0279577317797921, 0.940613153230652, 0.533970326515391, 0.671173297416918, 0.607534990065537, 0.750873263268339, 0.981322114529533, 0.72766965221564, 0.857321181770915, 0.991842838647439, 0.759460401600067, 0.146041024840356, 0.32630525604813, 0.0287861049296658, 0.694550964799332, 0.958756921956043, 0.729047089733469, 0.736754206669912, 0.174610632976036, 0.355409546993102, 0.574649448009948, 0.459945273105568, 0.833724792018791, 0.815444005808074, 0.323951744171924, 0.461684496488332, 0.673959217802466, 0.595226473440957, 0.134387298274905, 0.0194917927014507, 0.125108452164199, 0.223276033037744, 0.449137491249103, 0.534531553775199, 0.990389983584254, 0.717347186430348, 0.980057480277258, 0.0537377978224084, 0.636874186192189, 0.960381900016186, 0.269923101528854, 0.949407459012793, 0.902234260190305, 0.194648419265109, 0.733951518092096, 0.174861225612536, 0.105141051493097, 0.314112668019192, 0.348765660574067, 0.398962270410862, 0.283917454698583, 0.313911911733671, 0.718279795899535, 0.944646728658566, 0.0877887731563458, 0.27953439993778, 0.59615917974003, 0.828434587203008, 0.782152537870952, 0.557238729851673, 0.0362763902580856, 0.669371774231321, 0.848988498397931, 0.0654645715563107, 0.360765630552285, 0.258051248691284, 0.432596536841419, 0.306070907987052, 0.966588596591512, 0.129932951614421, 0.217436157710838, 0.893393781630662, 0.621706102231186, 0.39816237711045, 0.356089379392303, 0.646573091835647, 0.733091225545276, 0.731701094715264, 0.958175537661443, 0.0459544528666211, 0.424361171553098, 0.00901745827702971, 0.703805187352889, 0.780892367414027, 0.564840235265457, 0.0232952358321538, 0.00756391715486804, 0.988968831862988, 0.201647153298904, 0.823338228814864, 0.36095026922802, 0.46125298147071, 0.110045046402197, 0.781261654234013, 0.453443561373799, 0.297102450750003, 0.358400376522914, 0.482436038757475, 0.431224816912617, 0.698822552413114, 0.675066488042303, 0.00687099180618944, 0.0789897231088653, 0.460626620408994, 0.777536281453543, 0.816763969058827, 0.631388044045389, 0.364876109535912, 0.887498137898593, 0.250926826504142, 0.066119034487799, 0.727151179078646, 0.766797532489449, 0.898325706370439, 0.766825438969897, 0.946883448881301, 0.535733229060206, 0.959780779424305, 0.978188331630221, 0.522100193605972, 0.845360094810423, 0.898026506709126, 0.93122872421728, 0.457639630998734, 0.759161568301128, 0.93879585214302, 0.810700710998895, 0.930391955487759, 0.446957299146988, 0.833886065830377, 0.987781124263612, 0.369639430845853, 0.170766984773304, 0.823165410822415, 0.587057272178772, 0.961611628748891, 0.489129320184897, 0.253522780521137, 0.678224126939535, 0.914491782054582, 0.617074790070565, 0.322498739172047, 0.490511929940789, 0.407455342435367, 0.0839396235940845, 0.591959524469289, 0.8790154786391, 0.553307861404819, 0.299471068556099, 0.897008315680652, 0.794306306226872, 0.775812455441121, 0.5989682964747, 0.557221023059154, 0.299707632069582, 0.547657254432423, 0.991829829468103, 0.48341297047181, 0.684773696180477, 0.480016831195868, 0.746520774837809, 0.211422430258003, 0.248486726468514, 0.0978390795725171, 0.70867043451161, 0.855745919278928, 0.789034249923306, 0.742842549759275, 0.104522115223072, 0.520874909265143, 0.846632465315532, 0.843613150651208, 0.377747297699522, 0.272095361461968, 0.125303924302938, 0.69135253930652, 0.555131164040551, 0.00847423194155961, 0.416031807168918, 0.43911820541147, 0.784360645016731, 0.829997491112413, 0.589606438454759, 0.142074038271687, 0.593313383711502, 0.726192799270104, 0.428380358133696, 0.210792055986132, 0.265384268404268, 0.993183755212665, 0.480756369506705, 0.82775013186447, 0.603238265822881, 0.817841681068066, 0.955547170535556, 0.650378193390669, 0.627647346270777, 0.646563331304311, 0.0621331286917197, 0.93819664587878, 0.898716437594415, 0.694859162934643, 0.310406171229421, 0.343545605630366, 0.0762294431350499, 0.519836940596257, 0.608777321469076, 0.727159960434458, 7.39335770538752e-05, 0.169766268796397, 0.463291023631989, 0.361852151271784, 0.952065614552366, 0.932193359196872, 0.958068660798924, 0.206538420408243, 0.159700734052653, 0.571760567030289, 0.592919191548301, 0.698610749870688, 0.305813464081035, 0.393851341078336, 0.336885826085107, 0.128993041260436, 0.0869363561072062, 0.54894334880604, 0.31773324514083, 0.992747576055697, 0.723565253441234, 0.57210139010579, 0.717227171632155, 0.976559227688336, 0.42699055823081, 0.913013771059023, 0.897311254281181, 0.835228567475913, 0.0479581714127454, 0.359280569413193, 0.295753896108793, 0.629125618231216, 0.136183801688937, 0.374045801279154, 0.864781698209823, 0.250273591941488, 0.0295055785950556, 0.999448971390378, 0.605142675082281, 0.244195121925386, 0.43819527155361, 0.735093567145817, 0.557989092238218, 0.438339389533556, 0.603541763621437, 0.180998814378212, 0.808786139096593, 0.819058429818549, 0.487235718311636, 0.755050397326748, 0.404372532737707, 0.725676304367456, 0.154008580097915, 0.50907327902992, 0.668035593540186, 0.152575652904519, 0.0626960142135211, 0.337114882844155, 0.445943980473642, 0.757648467463006, 0.130533578951629, 0.90151183576853, 0.2203382404478, 0.378550737410665, 0.42275588837334, 0.972686200089076, 0.177016043504957, 0.882276510650048, 0.551257350492901, 0.321491907721351, 0.899845601217554, 0.542111720374395, 0.462889256184769, 0.807974927922483, 0.360818241502539, 0.971002979809595, 0.555517324431048, 0.769819702776687, 0.680121019128845, 0.205199551020355, 0.834622933076673, 0.865467029729352, 0.0415207826942647, 0.614193025281795, 0.27328023124576, 0.746706917123192, 0.487871946746158, 0.79237549386471, 0.52446289591534, 0.597511525477123, 0.220030197105833, 0.364019256716953, 0.297242030695802, 0.851616918974204, 0.973689594593274, 0.623621604130313, 0.892279193407138, 0.980902708287649, 0.64697910479048, 0.798818508157606, 0.404874443888309, 0.482364431286915, 0.0638923592909068, 0.802928530818637, 0.351070998819267, 0.849272450184556, 0.99180721638263, 0.48432395446319, 0.356364595357585, 0.0543710850399006, 0.914077060021409, 0.704968741364115, 0.438083604851226, 0.968109025296514, 0.723820838156331, 0.656803307680569, 0.514771588938691, 0.877056257493346, 0.0601741606944168, 0.471477646995295, 0.311016573938192, 0.194693071729154, 0.151049538868845, 0.90142040496197, 0.0706582953484205, 0.958387716954149, 0.699889739577651, 0.0120096638883701, 0.00488039931910367, 0.853291397884026, 0.467287774456014, 0.836718658877817, 0.220624806883894, 0.805960501451909, 0.24857944074091, 0.998272611507824, 0.937897628438129, 0.553956353352095, 0.912697424500699, 0.635489936226613, 0.780095732356123, 0.567218177239772, 0.191761666926058, 0.906248331513915, 0.514856417815248, 0.998976311034164, 0.377328665446907, 0.206111978117406, 0.729946702724716, 0.313435673302934, 0.117580925647527, 0.888605592971567, 0.693122898023161, 0.939900141522557, 0.417167317074028, 0.815770850199081, 0.804842686588035, 0.3604042770197, 0.913814142909148, 0.288315832504754, 0.165001861922716, 0.455862510160389, 0.242817315442618, 0.00192205255343114, 0.615342876447461, 0.661227860666023, 0.660336428092937, 0.680475172422518, 0.850649672812977, 0.0373144684309216, 0.680775196000537, 0.571137601206777, 0.690214775346011, 0.895566443250527, 0.266862292427761, 0.0685546083682096, 0.313601462605439, 0.789179515820702, 0.498339823066703, 0.437032740896536, 0.168405698428294, 0.264562676667796, 0.683881583354393, 0.816637406531762, 0.705520989919261, 0.576538892782125, 0.994062492960322, 0.461193269888687, 0.493353250001433, 0.273959687386109, 0.655102886737909, 0.407057092409188, 0.299963138302547, 0.605435572011172, 0.791173272035512, 0.970893567879101, 0.773383827316607, 0.59998170968759, 0.449068632784391, 0.153472770161465, 0.211233003135498, 0.70034904129039, 0.956182186144888, 0.635468664889003, 0.891271161381996, 0.216285013698052, 0.683543705465102, 0.315058619404266, 0.0253865802689583, 0.820249038054779, 0.432668699121688, 0.088706612584648, 0.935083991453488, 0.390776816002005, 0.587931646455049, 0.239835941356027, 0.653565004702233, 0.366532483627925, 0.481249932607441, 0.401685564357083, 0.333797544152539, 0.267585368411324, 0.767842963639545, 0.240381918946828, 0.961277161583316, 0.923551261388231, 0.274246913005953, 0.223278980867889, 0.252123562882164, 0.246993420297568, 0.23957637699895, 0.428250960804787, 0.825925886908257, 0.617186241365241, 0.193874402273731, 0.808081447291849, 0.985634194254939, 0.148602448431187, 0.853030808041202, 0.238601950434641, 0.416940550773971, 0.32239990234706, 0.434038639259104, 0.32742371163469, 0.098968782637155, 0.671174399831591, 0.0493586958059395, 0.200995885470055, 0.499136927267249, 0.275419317978239, 0.419603922629909, 0.596992881985111, 0.0491449094333751, 0.886679430596533, 0.858299166946022, 0.0850615500465453, 0.177416773225097, 0.0133985208033961, 0.589985436037871, 0.707861993500709, 0.758922559365245, 0.213971293699592, 0.966167268836941, 0.484388972423233, 0.062468101076842, 0.178459197784273, 0.338263383061555, 0.880482195056471, 0.042720547860268, 0.355407416417742, 0.64073932471876, 0.827883707476496, 0.0446230145768678, 0.790825634168463, 0.0511506610896971, 0.296425847729493, 0.272142381229007, 0.913453811472951, 0.605467095370523, 0.502217451311879, 0.736600448720886, 0.112123169605383, 0.884489884652626, 0.529550675374489, 0.00549392160547302, 0.162867914013353, 0.262326052549151, 0.0388698303626328, 0.682064471031633, 0.028196464109196, 0.499949207290394, 0.480694934559476, 0.27852885138082, 0.590491712486988, 0.287415948241235, 0.919946318974198, 0.00606384495963352, 0.564530378271599, 0.368692321283141, 0.371360757342727, 0.0211943212626503, 0.687250642698453, 0.554563743845085, 0.765435790737544, 0.79133895957124, 0.24656086055619, 0.199806651127325, 0.366939058177183, 0.71957202475292, 0.125364821133519, 0.429894235847257, 0.0163534867198939, 0.0710201582109448, 0.71030610251391, 0.80294782212672, 0.910196621634692, 0.91212369527331, 0.629536177330463, 0.131493794759506, 0.0621365072779505, 0.77754382520324, 0.742149511397706, 0.625929233638159, 0.729096676676907, 0.203014359562667, 0.794994806452332, 0.974571169963958, 0.420144160439961, 0.245720589980802, 0.480122010015046, 0.668972423132799, 0.852375787151127, 0.945322619785618, 0.128732263344334, 0.69664525313009, 0.080084145011743, 0.103562651160616, 0.131198363580771, 0.51505539908652, 0.481781084274185, 0.445883402722486, 0.798176904106939, 0.528938246875424, 0.668843730692296, 0.483337587418811, 0.332687004945907, 0.258801653161318, 0.973702708957246, 0.264360294841675, 0.789940849883973, 0.497068921156867, 0.32078526507041, 0.616282788890271, 0.00689414989878112, 0.0629336536785003, 0.92002496779444, 0.906163694464487, 0.831537125309564, 0.913674673545519, 0.679157649618103, 0.777058598287295, 0.693667946391772, 0.198856469805609, 0.4523882665424, 0.788886748898314, 0.0019245466318607, 0.555577597221659, 0.433248293229115, 0.0584159458814885, 0.441959849248699, 0.839593678799227, 0.618256676442999, 0.741594056980671, 0.935637401517796, 0.325310014605541, 0.745991443954401, 0.872898224076021, 0.433285909093465, 0.521559662455926, 0.652659259481555, 0.46901145750537, 0.183299320019268, 0.226002187762493, 0.411955423601656, 0.0152917076662148, 0.750670409873601, 0.0621234168692766, 0.431684367895932, 0.542779359875112, 0.610148550333959, 0.436234173008208, 0.0810685578990358, 0.283957238018756, 0.0732268043497053, 0.729977551654322, 0.346375102931938, 0.652205269883068, 0.481757451678895, 0.382593638230219, 0.326885840206537, 0.103039917466465, 0.370412379306934, 0.377441539597471, 0.481798320776792, 0.795619562877393, 0.691393696170815, 0.849790098652774, 0.512282525896526, 0.855033408437334, 0.465441546759527, 0.264848597763578, 0.629484213729975, 0.56273411894192, 0.541156467644886, 0.393311439318474, 0.420462371266418, 0.137121977734418, 0.292177878495224, 0.00732321318462159, 0.932701224776507, 0.60925854108403, 0.891845693826279, 0.420824840285832, 0.400181344013625, 0.553934117968891, 0.596259438250773, 0.506278821104989, 0.58720962744222, 0.132821686163433, 0.761597982782591, 0.651143496901798, 0.351089868130725, 0.312596377842862, 0.274250525946727, 0.149530737649424, 0.334345676729058, 0.833034050755781, 0.233575761034425, 0.0692409630655364, 0.494727170771375, 0.122151342280053, 0.40262126523765, 0.0912142110090802, 0.720548079831209, 0.342342908687307, 0.566304886072485, 0.654512371086194, 0.652501153776739, 0.700856710274513, 0.301104486681769, 0.0854638141905937, 0.63343570510963, 0.26643461534502, 0.894251779262979, 0.300268970334753, 0.715056925025827, 0.330352313276514, 0.364873259363617, 0.592348112171321, 0.905185468183319, 0.264528838835246, 0.949825880926556, 0.738728979330615, 0.00446114804216502, 0.850736759615423, 0.695572187869537, 0.562432122462881, 0.33771611338345, 0.332799347980861, 0.0773227876617238, 0.458300931052364, 0.657190066458505, 0.571953845497912, 0.402108643857949, 0.93470116941582, 0.36053423159937, 0.947714008953816, 0.0301235789352898, 0.441096667616585, 0.700709149870727, 0.702992816494182, 0.510166553347364, 0.612243886999097, 0.746368943947868, 0.801435610394844, 0.336724978708169, 0.564093006094713, 0.85526607041321, 0.589158603314194, 0.508241196713054, 0.853430500168208, 0.68379834206274, 0.106338256052922, 0.501956829307045, 0.0191666654402798, 0.442609383187404, 0.907175431054674, 0.0446696238732692, 0.945190804272722, 0.180380359690636, 0.369908146037757, 0.205239082352517, 0.095585844841468, 0.432740934798816, 0.277560941256448, 0.100858209884632, 0.498920454459566, 0.239034073525407, 0.349601331184063, 0.558262290294083, 0.548066911064257, 0.421273930541683, 0.385499666100542, 0.380663779179727, 0.280728362871376, 0.179912384250702, 0.127284024791542, 0.0156902244781708, 0.75308796929176, 0.504955187751938, 0.722110239780998, 0.761100805058562, 0.268948879578651, 0.193621982657256, 0.064148789564475, 0.425374317376553, 0.702146989068384, 0.977320645148091, 0.55158273200396, 0.0438067766504862, 0.582148770885677, 0.278375734918771, 0.542194249003883, 0.443320893375701, 0.117205430154714, 0.379941220729444, 0.764832885122452, 0.918368291779864, 0.692699055629129, 0.97062424392806, 0.126164966572996, 0.646413885521923, 0.386849955406608, 0.562344590529849, 0.198350583276079, 0.393926315230782, 0.34024226063819, 0.35647776993323, 0.373039889631078, 0.115364231245223, 0.501209758180346, 0.816912820607621, 0.343506994178503, 0.49853619674529, 0.118023982623188, 0.575516673330638, 0.515464245678848, 0.355276680178066, 0.863348931328343, 0.34401498111718, 0.456881381420472, 0.238088911770812, 0.771163080213046, 0.273404521881393, 0.525002112967555, 0.531113458408943, 0.873384186491755, 0.937613033728797, 0.0487174800041844, 0.978844913769709, 0.63232586385316, 0.0726338122521066, 0.211824011789988, 0.359887223354005, 0.426681280080052, 0.758145374626805, 0.915750766694828, 0.629740149655975, 0.955909112580599, 0.755118437130233, 0.526111172511993, 0.850110997930701, 0.816030924819638, 0.132130470911246, 0.760724168556332, 0.617232468110452, 0.397647218664938, 0.841271983988238, 0.038319707137773, 0.0962059280401967, 0.948102236245469, 0.396015614532877, 0.532262097804393, 0.903642478832919, 0.340035241143331, 0.663895644921266, 0.727361065028516, 0.223013600554722, 0.555454031860841, 0.455232670087629, 0.610264537979666, 0.543990038083104, 0.518356338980037, 0.286047746993634, 0.879016163320059, 0.936543036972422, 0.806209281954369, 0.969084927113422, 0.52423492220366, 0.11286893511401, 0.650234438862078, 0.190885967758888, 0.856935835270173, 0.942853072462009, 0.103446899719393, 0.296438947188895, 0.799034030595288, 0.4417147592252, 0.492520561454617, 0.0295161723505364, 0.572643385381941, 0.572462166315962, 0.741691368586382, 0.0766055629506109, 0.787166466045777, 0.193494994981051, 0.53092770419017, 0.92412978138448, 0.98605646492184, 0.975293398022476, 0.30476885524585, 0.753638407392678, 0.206355115194266, 0.861380713220579, 0.657844163185054, 0.102205755991926, 0.580464803713664, 0.963488280571831, 0.844872649140198, 0.753106898724697, 0.703049812286035, 0.246623649018418, 0.0399414953913875, 0.462113180980447, 0.117373812039105, 0.16161997461293, 0.419755839846402, 0.666115353714377, 0.973154394839134, 0.279650436496267, 0.531392965039603, 0.930692657616977, 0.166662406259431, 0.789249733235986, 0.102965347545769, 0.88362269550079, 0.522237613066693, 0.974126416343325, 0.756104194477763, 0.722570073546167, 0.00260118827601852, 0.205613727342686, 0.260369188673319, 0.0799547034648465, 0.73532773338309, 0.822650675052788, 0.343390636299772, 0.136138099416879, 0.665889426918717, 0.830441139206246, 0.673620870312758, 0.139459097922449, 0.378826883947296, 0.678362941175648, 0.51592509238978, 0.978055374587865, 0.26772469865577, 0.866910500650046, 0.373754868456556, 0.136630342828806, 0.082691545968604, 0.738823686826319, 0.372067804408789, 0.380516300174175, 0.302173764842392, 0.294525406542816, 0.259813565495272, 0.463475015412254, 0.98785227404697, 0.385075878767233, 0.943234206408962, 0.167001198424498, 0.925004059172159, 0.754337367579319, 0.856599436714716, 0.602386516706994, 0.107366821569831, 0.455472257215352, 0.654589710487762, 0.0575331207818541, 0.0212628768462449, 0.289801740163327, 0.931478831273109, 0.514019357456591, 0.875298430368223, 0.378099511665477, 0.884511519542972, 0.534773870148156, 0.892602735301347, 0.102763426671829, 0.212045832814161, 0.701605739512544, 0.361350829585055, 0.00139571470804567, 0.384308339853684, 0.32014291239325, 0.393345382592014, 0.181251223276283, 0.209376721050467, 0.593724632960721, 0.594466355677077, 0.172470281408444, 0.933001566648586, 0.0562167692335156, 0.201724141742876, 0.508252064976506, 0.759214859586613, 0.146052091797417, 0.931072986188042, 0.421948065424291, 0.0564987373903816, 0.674200438845229, 0.279808732271639, 0.78008704325976, 0.962841216037584, 0.688244836652805, 0.498422560204492, 0.0990324600505317, 0.287751247312469, 0.538112282800573, 0.0113300513836901, 0.28284348457517, 0.589571290675808, 0.172048111874952, 0.170098165329199, 0.676342793813034, 0.569594430906669, 0.551670501187363, 0.967440984560659, 0.628421902063027, 0.32454971751524, 0.52101569086757, 0.864936676087545, 0.994108234786551, 0.838488230140948, 0.347397147933049, 0.749409879786991, 0.939412832737864, 0.668081838726903, 0.481202717380885, 0.286163077001452, 0.595155805599182, 0.336407144805892, 0.959550567460059, 0.442075021719014, 0.961996208463649, 0.676383620364058, 0.706077836757738, 0.957692753154191, 0.939876937811068, 0.833757534873689, 0.155241320818542, 0.830432174693319, 0.464140322969328, 0.298688427677999, 0.523305319741285, 0.831630435143942, 0.539132352809421, 0.259831859670961, 0.354038426137652, 0.186738136073245, 0.350656734108598, 0.189718535133722, 0.835690938223247, 0.384150955598582, 0.00949972238134222, 0.30902469522195, 0.836747234202809, 0.281336821162533, 0.781343312099506, 0.531514249243941, 0.153914515712914, 0.525253684159556, 0.28323978774568, 0.714866114508951, 0.734389284920324, 0.449635843063718, 0.18207743398426, 0.135540426407628, 0.272771613665478, 0.635643507130864, 0.806187918425097, 0.66787732267887, 0.286589147299096, 0.142441562751007, 0.245875235268404, 0.691897013511217, 0.0717468435078286, 0.498660875135198, 0.985997777331523, 0.904900645763287, 0.575207697512688, 0.766517273968399, 0.746767448472435, 0.695331406974203, 0.677317250641443, 0.282399827751482, 0.581867980228999, 0.618872290886971, 0.81423991683717, 0.0159870604397865, 0.214352603814297, 0.987867358551446, 0.817337253784204, 0.632081710733128, 0.682147446573711, 0.808216822906653, 0.279662138183623, 0.664969267909487, 0.0293241123476947, 0.544380154247748, 0.528970245787236, 0.873164278154414, 0.18760830382017, 0.931460574723358, 0.976352275920366, 0.331204427585016, 0.623709234135018, 0.527060718506005, 0.459081521527663, 0.358565625709114, 0.334269551453081, 0.0831952498878953, 0.390401868396455, 0.23934394594483, 0.486271989196401, 0.70009360460331, 0.137270331515517, 0.826467338719534, 0.930053219530656, 0.828730506887895, 0.484952762748445, 0.659769263952863, 0.456401081913508, 0.418462054184599, 0.0342962619999633, 0.926847546326002, 0.0433800556813319, 0.47606009927654, 0.385747734733797, 0.534058334246592, 0.998168271284668, 0.902474188299729, 0.0147554983520319, 0.78051492878948, 0.485289450415378, 0.390219764554317, 0.823090104861606, 0.874417544562118, 0.99835602642921, 0.19566058928756, 0.869968918189405, 0.30749161863362, 0.411236453370934, 0.816340198541331, 0.368254747431217, 0.426215425372259, 0.0912168103482063, 0.667498181249811, 0.731308161123567, 0.300818409726586, 0.337997209328381, 0.734322359806766, 0.985456620328616, 0.698771013868857, 0.0899318126996229, 0.470737526480896, 0.635362887995652, 0.525269355195397, 0.141339436867213, 0.275515481819713, 0.987102734492889, 0.794998307857391, 0.933334910317266, 0.348372558719299, 0.209133482238212, 0.680365514351822, 0.355423792948098, 0.438373146306208, 0.908778179399217, 0.833589959460029, 0.391383815464098, 0.733979099285805, 0.290660035067534, 0.292136644463826, 0.29679627738495, 0.0248424216656624, 0.917841813209207, 0.686030122584367, 0.234277855152853, 0.497352640141022, 0.790581550930947, 0.380232617925932, 0.2185834125899, 0.722665277239192, 0.772832557068947, 0.693282655112069, 0.619674794711139, 0.762303040962266, 0.511480295707378, 0.329405466987943, 0.00134406153403044, 0.58064405022573, 0.17183806896399, 0.194907220851385, 0.150321508288129, 0.505419725200383, 0.0173869249304102, 0.471207952661443, 0.0460774941275637, 0.442265248700239, 0.163559193899743, 0.572099955415187, 0.818320322737847, 0.0853419806648935, 0.232836093164044, 0.246046032482717, 0.351934213272797, 0.409084090828811, 0.114243114777677, 0.407214771262226, 0.675930708273143, 0.543023940750119, 0.357129615646619, 0.661592730676232, 0.678702790220863, 0.131560183156139, 0.439619611546473, 0.937195575523914, 0.570223860573477, 0.57763929116183, 0.764602370308821, 0.246951668483173, 0.693855425808869, 0.294862911829712, 0.570700530837074, 0.634804165571924, 0.398891197227157, 0.24888368951307, 0.935332380914663, 0.448901901434686, 0.531784017364865, 0.63483854991718, 0.668120361357177, 0.439013188721266, 0.172989364509112, 0.694384241456172, 0.924926210208933, 0.370605944633525, 0.638787885336972, 0.889679086573347, 0.482942333755868, 0.629817868379091, 0.465017493979997, 0.109172860502643, 0.113116377713374, 0.10789230851418, 0.445730958177544, 0.938845036625008, 0.633813978300454, 0.929669348074345, 0.0628762558931983, 0.306938911659444, 0.949764495374922, 0.0103643790888722, 0.167827708425684, 0.755272448052324, 0.833821732275786, 0.605860436383901, 0.517917168369214, 0.942857603277382, 0.741206395549934, 0.701078867005602, 0.744718149025705, 0.873825773621753, 0.964120595947437, 0.981916907295625, 0.918632827736361, 0.283480366492016, 0.0586255751084055, 0.820994546935882, 0.179522235053377, 0.0349312717679734, 0.771326713861638, 0.540526119338446, 0.331539955860905, 0.578216606212224, 0.854507421818445, 0.64237508408152, 0.420103532406922, 0.545912886195662, 0.182297728364877, 0.23627422713053, 0.205413666735986, 0.259686312588506, 0.942632406182967, 0.641570357548248, 0.829063366929745, 0.388762193616527, 0.168412217880984, 0.740117740402888, 0.872795850353736, 0.355524975509097, 0.178828671174118, 0.996423607919638, 0.959127558877631, 0.15599623700921, 0.928935665275031, 0.424574186767475, 0.223083945072798, 0.136395092586256, 0.260175654111998, 0.544465382989762, 0.0822083920194249, 0.598127313326546, 0.0291861783993683, 0.454908897687904, 0.33249868418861, 0.94367283155584, 0.303101820009693, 0.428224336498126, 0.661033759829754, 0.381088012036445, 0.136170611625815, 0.0966266230433772, 0.305161641571482, 0.823883649864703, 0.627252866033001, 0.205932764156298, 0.980185458037584, 0.437774280133287, 0.835195774239894, 0.539180447961038, 0.0251269331883593, 0.00682427912703176, 0.786086238518133, 0.447392855916459, 0.288737449408737, 0.863180264924441, 0.656228495115062, 0.284693632908978, 0.644566224735289, 0.263581178572265, 0.174778664281482, 0.368619500621701, 0.537542223191153, 0.926407157098554, 0.749120729449102, 0.147735486618356, 0.00288774509561229, 0.127013893617675, 0.916921352125118, 0.6721217072533, 0.113789807451064, 0.960227492817496, 0.119383120967445, 0.492415695453145, 0.361313401952203, 0.470000490547994, 0.0587797688209679, 0.330011701173118, 0.264520477657796, 0.245917666364951, 0.947676359396337, 0.032441994505736, 0.906150634115906, 0.368497704140453, 0.662540373112819, 0.953624630494782, 0.0941284570557659, 0.685315621928818, 0.412440482165339, 0.111788722708257, 0.491835063078308, 0.461552383778851, 0.877058141059825, 0.660239177477185, 0.277964838412501, 0.657168216036714, 0.645414950162192, 0.897223744826558, 0.882336561925662, 0.335172425736869, 0.427001928450755, 0.803764988658286, 0.923094204600217, 0.866723840146016, 0.499650272539607, 0.584483044848704, 0.924757001456557, 0.166116897602024, 0.374360138093039, 0.259408031116373, 0.830480916680207, 0.0701766867331509, 0.281030277160919, 0.555044837600402, 0.618966880542704, 0.409787087237061, 0.258428445296645, 0.416746159880173, 0.382055538955754, 0.0963341329941303, 0.131284219393697, 0.609898270417548, 0.656193731879345, 0.840995312544397, 0.499817878099585, 0.307198081516692, 0.169763921941861, 0.0487653930379749, 0.225817131001496, 0.366971084916278, 0.567846384923709, 0.569715406944438, 0.778776477723553, 0.590241549067763, 0.110663463494512, 0.640677943414038, 0.0720193262477142, 0.365845796787181, 0.287303064933183, 0.741438100853781, 0.0669874307501631, 0.75716595545508, 0.240713033541195, 0.230909836018301, 0.0119883821994998, 0.132474625822457, 0.20938654847904, 0.21418271638405, 0.905794370282143, 0.277772501400827, 0.0971019894267973, 0.782821526670524, 0.66752094460457, 0.612672548528441, 0.343792421289708, 0.690338891464082, 0.233558970402658, 0.308029426654619, 0.357300705035804, 0.216697932367009, 0.677280004283045, 0.736014713740831, 0.0100185705081633, 0.355914314193773, 0.239954583001563, 0.927608731363223, 0.762986816223549, 0.379387102683397, 0.685956275663823, 0.698872673219376, 0.590398235040816, 0.962055535472913, 0.781345653172047, 0.538427479061901, 0.094138251016693, 0.886008685110991, 0.257495133440266, 0.463506865876196, 0.33079827009126, 0.756773387686601, 0.80284933532799, 0.803117690893612, 0.876636860328713, 0.0874740675440413, 0.667913624903703, 0.502421972529354, 0.335562040666086, 0.588561469425868, 0.35193258697291, 0.206717653448786, 0.262154546296404, 0.0986493029057204, 0.150063189520438, 0.321019344134903, 0.594389455161452, 0.153738062315018, 0.0626255991383137, 0.0077822300572673, 0.774260127945251, 0.682045019145825, 0.120503154992744, 0.719253399296351, 0.161854103749308, 0.120160149952966, 0.681152106103643, 0.144826655994988, 0.0578277521777769, 0.134942002869956, 0.0147804385042802, 0.189117151280089, 0.0708621182879046, 0.0378597506004695, 0.307870208130694, 0.51573554602372, 0.3935107401546, 0.995195514105615, 0.439599979444392, 0.151439647032156, 0.778050529499016, 0.741173463627264, 0.0876437222952857, 0.470425708481648, 0.292829610083521, 0.406850582025529, 0.430237247543503, 0.871262140219956, 0.707250137217728, 0.0142973312746552, 0.0245837084462989, 0.236994247170202, 0.598265436268775, 0.926198153290609, 0.334413488822878, 0.443032939162917, 0.0991380567194883, 0.715151205267871, 0.721638406610134, 0.782470952751407, 0.942045100936627, 0.0475684765606691, 0.800350387947629, 0.466788405374054, 0.279139849677783, 0.0447210821128327, 0.787550968567015, 0.899123861423961, 0.775683899677438, 0.83745023558209, 0.344420705880329, 0.191192603795677, 0.378438923860061, 0.256333841550158, 0.0214632398092577, 0.293278784780504, 0.240422777514371, 0.202589008910586, 0.348220780592835, 0.537149596656765, 0.124314272622256, 0.1927383096706, 0.488619539455234, 0.407010404102642, 0.734918843743705, 0.299904918967185, 0.807136141451077, 0.913664517168352, 0.765108041299724, 0.334211696126535, 0.719337934780301, 0.0335931645744745, 0.615623896929065, 0.11218486575756, 0.0506021581783623, 0.0552102492479777, 0.0262195967692186, 0.95543558180011, 0.525213803497993, 0.756254015153826, 0.589818318147174, 0.738905633959864, 0.319416999363896, 0.848129833708729, 0.493646821428674, 0.114835278208844, 0.812272382178224, 0.666829160039589, 0.381783083914011, 0.718808964344897, 0.418627894303215, 0.0559327094009257, 0.584999660582087, 0.29882772319867, 0.40956257879111, 0.586089706274874, 0.315621564978314, 0.147358447504408, 0.99152927537782, 0.107493243817194, 0.334256834168659, 0.0379589602640042, 0.922406597999313, 0.373141767132931, 0.989160720830219, 0.543456031093595, 0.563316516800312, 0.18789861860945, 0.21549562874861, 0.485306895364261, 0.219500737305357, 0.936872548362282, 0.277624636235479, 0.953103538581981, 0.81493251079315, 0.349598804621487, 0.0216060447638035, 0.341013043092474, 0.384397811924298, 0.0932896254234417, 0.969252776709913, 0.190044957173184, 0.164620463933557, 0.50703144302524, 0.508238273215872, 0.0514781512552508, 0.367011700837063, 0.584797236625955, 0.994750591834159, 0.435131772858032, 0.695997788483381, 0.0164754955139705, 0.152540095503124, 0.899538852278359, 0.046376301382325, 0.0293269895896349, 0.574444491295931, 0.340404165658471, 0.219300767245549, 0.903991054375012, 0.683087841522802, 0.371648147433041, 0.401117836492822, 0.204970269546629, 0.23769919379998, 0.884360645947479, 0.921243055684344, 0.781648862269082, 0.241915788011649, 0.583596302269313, 0.201855690473658, 0.127067768373594, 0.68766932132812, 0.910739011175164, 0.750940069868364, 0.822908904336123, 0.225120852448046, 0.386746304449196, 0.418688452045263, 0.935785662295632, 0.105554849257096, 0.531289272107127, 0.621191657295655, 0.280341753274966, 0.197762634604298, 0.408386910747043, 0.362604572782088, 0.161901392903929, 0.893525483138456, 0.440476366658922, 0.670310019520092, 0.210426739318263, 0.471146230276458, 0.049913841549351, 0.495931517705249, 0.26972119441696, 0.684174307762781, 0.747995619291006, 0.391641368887366, 0.558827045705254, 0.732245614140509, 0.611621744657112, 0.581229664436586, 0.0706039272518529, 0.374246311460359, 0.0335896223196844, 0.406143089668485, 0.65795811226944, 0.575240544824983, 0.975956238113695, 0.441639338168845, 0.777836973559024, 0.569458425859217, 0.0210997848788582, 0.512273552595416, 0.407552145238053, 0.219837073696457, 0.230460668047538, 0.438549783462451, 0.919893830864455, 0.31903756749757, 0.284334536926044, 0.363779007864189, 0.499317543098154, 0.813524280935686, 0.234818248522346, 0.0222639360560148, 0.301121413722734, 0.298062236145094, 0.82592797647296, 0.136395439397702, 0.81458040564826, 0.861088354928281, 0.552941825691261, 0.0185298677779496, 0.507821480549943, 0.713645133185084, 0.697747973936715, 0.893455595941863, 0.701184970536691, 0.937290380549169, 0.0108927735593852, 0.819740403776599, 0.0342676541566569, 0.516160320975267, 0.653330320369338, 0.754127262526012, 0.27999697494331, 0.603051060044045, 0.888780732453205, 0.643234106940034, 0.786063353505465, 0.491141242071711, 0.403460405589341, 0.877440097783825, 0.70818292686665, 0.826491606032295, 0.0106704162584034, 0.0955571422893711, 0.543284252749573, 0.856217447370269, 0.264355933909156, 0.775547260141591, 0.637920749171789, 0.725059755063973, 0.894647612386255, 0.767987004548132, 0.206846838462889, 0.0433041575402289, 0.153563556887829, 0.718313564120591, 0.128643505891081, 0.220788923652274, 0.453559441489318, 0.465260851971691, 0.0897862581146435, 0.506729645061984, 0.0765090983525664, 0.166524695872434, 0.747710039127429, 0.334567260318324, 0.305054021165106, 0.818878552761969, 0.982048945374202, 0.968970008395478, 0.576622335269014, 0.233482991102818, 0.292098451838874, 0.67376433187999, 0.573738701774179, 0.199702249129722, 0.95819839711472, 0.336154890392076, 0.149664105303348, 0.328383218060923, 0.113096676567634, 0.771914476030847, 0.792562519354107, 0.587068781068013, 0.0791916209259588, 0.192901989423697, 0.775405135198076, 0.353391437287061, 0.838947784834775, 0.133887257475644, 0.698147940585025, 0.190344628030756, 0.396806895101165, 0.33894732428959, 0.92269275683973, 0.838402608756176, 0.629524052851101, 0.86549064981276, 0.216289591681076, 0.0648990849056403, 0.678000534448261, 0.535120190236234, 0.517349233632873, 0.255718668457606, 0.852827587736811, 0.823201061074928, 0.0165722102089281, 0.474710003612981, 0.944118569767781, 0.458620542739682, 0.849972299911911, 0.535552391452369, 0.434101281159365, 0.0244337051299548, 0.504359780762734, 0.87648395531354, 0.908248818284871, 0.285203484332663, 0.132703661976258, 0.000132543545445563, 0.317960710505006, 0.530107999980754, 0.829176120704962, 0.673583532306214, 0.913014498645068, 0.756141926441929, 0.168512427287545, 0.964015937706506, 0.348265950752917, 0.59848756193256, 0.731391705452838, 0.870816363254405, 0.33339943713535, 0.463042712596264, 0.166886711732574, 0.466389296693143, 0.456370455255218, 0.127822299283932, 0.639930370392182, 0.896488227238531, 0.427427957846508, 0.17584603336588, 0.294898346991532, 0.0587383128869224, 0.0601728810384173, 0.0266598923540069, 0.550467405320711, 0.563050714329366, 0.304595613761088, 0.754510784702296, 0.671103187580155, 0.939838254337713, 0.485154691181415, 0.518087840973396, 0.789203213435278, 0.571168243892834, 0.766494844432297, 0.92016610016362, 0.210306333287111, 0.608114181349688, 0.465725204435141, 0.616699894336789, 0.203092123754255, 0.677600524047734, 0.0173155583220036, 0.226607511496155, 0.914466037759236, 0.768434032614168, 0.53331264270668, 0.0683802164192485, 0.199584232173589, 0.0120176306233901, 0.966571912566926, 0.832200690295632, 0.445309060248767, 0.990495465712949, 0.7942843836536, 0.488611168068191, 0.130589254209538, 0.212242519268368, 0.831110154723853, 0.32378736967358, 0.738767713808585, 0.15987275885464, 0.73147071548382, 0.0536941171904529, 0.714389527083785, 0.121735642299081, 0.984044737989997, 0.593098196266203, 0.269543205364821, 0.432003590924972, 0.930512342342939, 0.507885308540296, 0.0640980052808117, 0.659111312806854, 0.449172017844814, 0.147168497595702, 0.775228089906687, 0.531261756843832, 0.0878941964863205, 0.900460695951332, 0.86815118402594, 0.0562998612769433, 0.640056253376155, 0.889264629794461, 0.347723192695842, 0.666719506046494, 0.454460742891694, 0.446878937928708, 0.0170310231720651, 0.861417656196086, 0.937822118715544, 0.478774086979043, 0.56983149379317, 0.992795582450183, 0.404134625950054, 0.219841318752584, 0.866496585605024, 0.689023069077314, 0.171469383296141, 0.00902372872092816, 0.169077127538321, 0.206387770951504, 0.901622188806186, 0.709139897416604, 0.131930049727708, 0.425204689342426, 0.661446714250263, 0.322383080983399, 0.597436790067751, 0.210647153317548, 0.189618227177975, 0.208315555445146, 0.713345381914946, 0.252696410897019, 0.270015009688202, 0.481757359283581, 0.220673729581596, 0.354664189148112, 0.637734017278605, 0.749754307661468, 0.492421401561606, 0.617388763954745, 0.319453111261942, 0.302499087994426, 0.141789496281462, 0.844342381495237, 0.288154319299721, 0.908832329658068, 0.113209175875818, 0.61386059505054, 0.28390410895503, 0.508132007325918, 0.538875114256029, 0.0521505093155784, 0.437082481116893, 0.218729639950497, 0.91833899065768, 0.691453098813723, 0.11592525990435, 0.359622792241797, 0.306055720710478, 0.302248075913999, 0.561445487419883, 0.796098974386462, 0.469948705129241, 0.79117052588079, 0.418768954706078, 0.739529750330297, 0.35752315456492, 0.949822083180672, 0.750146571002327, 0.0195774978398223, 0.755167140259959, 0.911801131562645, 0.347753871933268, 0.0712852111294682, 0.855088619240342, 0.0943533481055111, 0.549567876186815, 0.915525126570028, 0.522011578805557, 0.0519654842517967, 0.440908959881184, 0.691189229271193, 0.26137753921271, 0.895907539440173, 0.459712428749353, 0.232019041822638, 0.19594288957914, 0.849693844920558, 0.741608560642791, 0.662215360447538, 0.10708839955975, 0.836172920709126, 0.454915385323753, 0.650629948128032, 0.904422585429451, 0.110705625079734, 0.603829508749812, 0.0163824662953167, 0.0912104015125412, 0.905671652617877, 0.547618978580024, 0.644206311361358, 0.235876900650091, 0.620573635836441, 0.0789822725201043, 0.645751100531398, 0.176299805784857, 0.618924778123305, 0.970324158361631, 0.669193468169111, 0.331811326855709, 0.457887015850102, 0.0249493459820481, 0.523104636865495, 0.989191958605462, 0.723363696905421, 0.396944731290823, 0.590729815872983, 0.573179891510593, 0.507674291681839, 0.145546385625009, 0.290948730772037, 0.501144071620637, 0.141869850833385, 0.406383401453808, 0.593103184955249, 0.809379944787499, 0.470598959727016, 0.674121679009063, 0.757179557379411, 0.182099790582197, 0.619639176924652, 0.769927078227743, 0.421086262944373, 0.438988765380035, 0.48878020153305, 0.216832960985771, 0.832912384105407, 0.50154387632697, 0.102731676251023, 0.364359910045486, 0.820802024303507, 0.122843525294514, 0.649358746529001, 0.736624715037985, 0.710664940067149, 0.105334934681613, 0.958604467785189, 0.423692880085245, 0.560113444003007, 0.0987827711110292, 0.483117658218534, 0.153624763543921, 0.308469833264301, 0.130916366315163, 0.343518600929229, 0.663060376190738, 0.516202663397852, 0.796666886923888, 0.576595567775278, 0.0668739600414979, 0.248482422458477, 0.908679436157011, 0.88950847377688, 0.989806158451988, 0.323655454647909, 0.987420465083663, 0.613864765292363, 0.652081525715332, 0.601346094254829, 0.797794327789756, 0.610446566439828, 0.377238680708853, 0.174646608490432, 0.0599142046576271, 0.15235264468071, 0.383418588028833, 0.0130838670143263, 0.965373531078622, 0.318186884514696, 0.955603653434116, 0.0289597177906955, 0.397178806733152, 0.277810016828821, 0.361853097379357, 0.780453305213959, 0.445664016649587, 0.0573122580666343, 0.0479943403138646, 0.530814875588139, 0.766768060920667, 0.614629194324132, 0.93243288261142, 0.781576747911696, 0.754902740907117, 0.326217156572103, 0.521981259694099, 0.645579304666025, 0.511238454761319, 0.66210009328207, 0.231309821394203, 0.982569292670047, 0.0859194848713005, 0.412678414690997, 0.985284229676402, 0.485258809822885, 0.611872176798615, 0.0112204544432973, 0.768551517768436, 0.741559542612891, 0.217346748370756, 0.397321397395443, 0.262563873189216, 0.0786386197013217, 0.220119743646035, 0.484985012265265, 0.515296236249035, 0.839009167320221, 0.413152066815883, 0.402175572445444, 0.952686925216141, 0.335098175840949, 0.133750202136955, 0.695436443082378, 0.922773421317134, 0.487524639007083, 0.505692448683919, 0.00462576058235187, 0.188468748383506, 0.779639443833603, 0.610825949687488, 0.792456860497627, 0.837900449095441, 0.687358399403696, 0.0814341000567961, 0.806140569013718, 0.134776995255468, 0.569914349014358, 0.574691506274377, 0.0161027507918959, 0.934828315776462, 0.614780026596316, 0.668574890264944, 0.42177280894813, 0.702767173608989, 0.470696635874201, 0.408892933050312, 0.988636018671481, 0.485250739926732, 0.117262739217962, 0.858085916286096, 0.341975792395652, 0.316379134170064, 0.377527146252593, 0.869881663339071, 0.531433268122008, 0.0251555091550247, 0.248355247547984, 0.723680484135885, 0.432381178248802, 0.261450582796841, 0.189409062205332, 0.214478014236663, 0.494894581780918, 0.0330008321070456, 0.372624260801018, 0.356676672691004, 0.274706538681145, 0.264398923042338, 0.744515378801041, 0.131683750791789, 0.80308400320993, 0.304403059930111, 0.342846208359973, 0.659341184259834, 0.0389644876131549, 0.77214283878312, 0.690084579256376, 0.927245445494015, 0.0620185847052662, 0.687466619742027, 0.725694886449768, 0.846529403285658, 0.553800074015867, 0.842396744786328, 0.0489188877887211, 0.758804444178936, 0.675999874115613, 0.874429094310285, 0.330943235702591, 0.518499669460245, 0.453945205687372, 0.179503067177272, 0.84739945723103, 0.0464259746177923, 0.523596071921231, 0.232703629297068, 0.903273904045203, 0.54125696534778, 0.0986849735350523, 0.857083631354171, 0.0427379691775538, 0.142041748351223, 0.21749764979384, 0.777462074482948, 0.357605951488887, 0.0253237781112557, 0.833779609211363, 0.363197813340829, 0.788561803113904, 0.879659663407538, 0.67650784079487, 0.545244062198411, 0.898888853484247, 0.0994934710882014, 0.090939291677099, 0.034239332447879, 0.324174228191344, 0.166065758635189, 0.998078594941955, 0.415485201282559, 0.75433338439618, 0.69331349740241, 0.770644011008085, 0.912260083517071, 0.310679768102696, 0.204717068840118, 0.0728788179522709, 0.587200686173216, 0.542486621082793, 0.644085818423362, 0.598005259559691, 0.450019731881968, 0.162550180480761, 0.764920733957713, 0.983300332487826, 0.143146444633522, 0.827703375396012, 0.881132801236475, 0.0704502775610348, 0.477865346178079, 0.738754794315637, 0.733736908889774, 0.816245277273153, 0.533480060855813, 0.560833396451542, 0.346526090989694, 0.550755647407727, 0.0211027090790837, 0.736771212947088, 0.833971738754734, 0.836319659472614, 0.0331203679365166, 0.912172343754512, 0.0949251782848999, 0.85090800856685, 0.522199819493933, 0.0163664557099252, 0.129177156602223, 0.593174705169208, 0.721787369280583, 0.0370252216352661, 0.176169340389678, 0.232436637070023, 0.715012793230278, 0.770586055243288, 0.589379482134796, 0.87293878729415, 0.453991393466129, 0.983929045970749, 0.259961416761581, 0.690669142380331, 0.596743802050488, 0.279328899198392, 0.279209912723788, 0.683767710931583, 0.811281073479239, 0.0762412206020542, 0.369372693937438, 0.806237085379592, 0.958330989384955, 0.993734842237702, 0.425572267672731, 0.433152991325934, 0.338445567047592, 0.454667841051699, 0.769975273230104, 0.755700280867838, 0.992393933214707, 0.203073406455532, 0.375416971500507, 0.708670176429508, 0.163497946989236, 0.323962052812977, 0.809012249269781, 0.218956897703679, 0.328388088762296, 0.46199820413596, 0.352330166721764, 0.0735395609286982, 0.505193068431356, 0.274199968621932, 0.567635438025136, 0.0275111118604425, 0.613717258420128, 0.423611963323757, 0.645295719313487, 0.306118066630253, 0.290784581891318, 0.430515440480443, 0.550274236761594, 0.122461045504512, 0.130425046892518, 0.228788197941979, 0.106923399897953, 0.0930371793283046, 0.188572905742624, 0.683495507692552, 0.541482588423314, 0.521406921805295, 0.941816575794057, 0.637501236487765, 0.71855755485241, 0.424441585982622, 0.248357586426725, 0.935987282396234, 0.802128402172876, 0.761633821857509, 0.0617528397800685, 0.906653800098843, 0.655628124184556, 0.947949684175439, 0.545241930189528, 0.305318784952586, 0.088515984049663, 0.42153451375263, 0.367533367605065, 0.852528590674442, 0.983923978038975, 0.0538531199528909, 0.357135169210054, 0.99906704216243, 0.415017360884189, 0.380647980788953, 0.104579559223633, 0.589612224609026, 0.146320634207263, 0.376575715029282, 0.936352471950852, 0.700369541067379, 0.242934140773795, 0.742119399006556, 0.326637754080371, 0.369959383224723, 0.492794046122532, 0.200618413004165, 0.287567318795898, 0.292593071831062, 0.42984361010847, 0.478112808942906, 0.93192001313755, 0.836229736161115, 0.289975682550938, 0.958075003433249, 0.723048334319942, 0.864807861773361, 0.523249843961214, 0.0527687033779026, 0.960015357080117, 0.672208035216727, 0.065466756187273, 0.979136144491202, 0.294845030689413, 0.949908846358517, 0.173419866776642, 0.000904598853058602, 0.878455437823958, 0.850536754283391, 0.336408996005985, 0.0983301033742745, 0.628119103521096, 0.400416696744513, 0.928052430831459, 0.880605659457452, 0.619080856675795, 0.893940761512517, 0.209053301909822, 0.258297753380052, 0.00412487501082448, 0.0904526895660369, 0.595820023228063, 0.299312727255091, 0.775433796692887, 0.642389925104526, 0.179546536444424, 0.283340135314643, 0.135854835001849, 0.563495506068162, 0.144256211878118, 0.856053421517863, 0.0304084344485077, 0.850323862718259, 0.232800489509599, 0.754936468225829, 0.0563904328316007, 0.707646403670149, 0.184121151982, 0.994602559989464, 0.516699703376943, 0.538151013594407, 0.271513995624893, 0.378256980450822, 0.512146572773542, 0.515418545179978, 0.468933678773109, 0.154167158186831, 0.215126080518626, 0.645172822995348, 0.606139166231606, 0.182345124911953, 0.867691456244425, 0.527718104582934, 0.974537629655931, 0.2854848172305, 0.949081000950822, 0.607605310564352, 0.776487807153511, 0.0634819601111457, 0.591157019335551, 0.899757533231989, 0.50978799400869, 0.53283478564956, 0.798575636163701, 0.655095978368141, 0.463057825671424, 0.557317047715037, 0.743033643725929, 0.514321677398306, 0.120782358492523, 0.268086670460766, 0.924859012550818, 0.589397306346545, 0.53922341873659, 0.937265264691391, 0.255628958293643, 0.911900091597237, 0.557116139480797, 0.738684886893027, 0.684320844074304, 0.13970589503849, 0.973488582318938, 0.424539768316478, 0.0762116456896338, 0.107926188231594, 0.972241315159768, 0.570353339236255, 0.483824883183887, 0.644989753034534, 0.218204376162376, 0.694902920496989, 0.245756858601072, 0.633619064008014, 0.960137140827273, 0.726421985150142, 0.96965829636015, 0.762140898634666, 0.525095840809228, 0.446790244714577, 0.638361974896685, 0.307137787125137, 0.443405928293768, 0.665269945243287, 0.822277174423012, 0.319021311251583, 0.657045037216022, 0.505403407519289, 0.157446031156873, 0.305076553398929, 0.105101287671345, 0.870621818612893, 0.678950665498254, 0.826875222837622, 0.946069323977357, 0.630603912133686, 0.513402218344765, 0.555994970075604, 0.987430970242978, 0.626901509093195, 0.852016973881539, 0.490250621804489, 0.0199587554199986, 0.243831396532202, 0.550075064418023, 0.101567016003368, 0.474312083106096, 0.17453116810711, 0.325383132003368, 0.110386313590483, 0.428236510815134, 0.71899060326803, 0.767616586603275, 0.927185202538995, 0.756954049487111, 0.332498210386162, 0.88055950957798, 0.525609736535591, 0.267340285980459, 0.168216820292786, 0.0450111867713388, 0.170774934576798, 0.925775412396874, 0.108623586670381, 0.834149341938526, 0.151815012788866, 0.855876116022027, 0.598567527704985, 0.197768259893043, 0.797530519850749, 0.787385651645916, 0.0955850832602574, 0.171447782118478, 0.602812097009879, 0.325098482832322, 0.267284872097206, 0.789299751611353, 0.616613442810327, 0.686464205029079, 0.697433301028023, 0.932020594067502, 0.120188087418612, 0.606886216020793, 0.489867649801154, 0.529701380634013, 0.221869131323914, 0.748180899172756, 0.258153903293609, 0.199113976799482, 0.871523372745114, 0.0271855015178478, 0.0363402135001056, 0.437993666997028, 0.0656876281407486, 0.979386424179718, 0.628900063956544, 0.656328083176834, 0.327906051944788, 0.475499863167847, 0.0162418237144878, 0.20002814566854, 0.858494732522432, 0.0675401047899903, 0.31074856041879, 0.826622732777694, 0.611868414525087, 0.914600373605335, 0.884110438157532, 0.245313033938969, 0.958201906958456, 0.452296699863362, 0.888558143908744, 0.46485039369031, 0.58522136130226, 0.0116790120991487, 0.728435390588811, 0.41530035149341, 0.118637485115407, 0.596256415905622, 0.00468785953265283, 0.939454779184277, 0.691412038568923, 0.271798077177966, 0.753737331398782, 0.658331924693264, 0.283710125861441, 0.535614507734284, 0.0602160721163846, 0.67742727479711, 0.92288310288109, 0.466638676984603, 0.368810167484628, 0.917393929268497, 0.0115730362262235, 0.929235201900928, 0.693417540536448, 0.821237024807033, 0.857736890453325, 0.110576890289008, 0.704999404231229, 0.551666431230606, 0.520627790433433, 0.313731750190138, 0.161414920332861, 0.319727777916472, 0.671066598933686, 0.523950795320327, 0.607800460893577, 0.760314937511405, 0.163862530614109, 0.494361693992048, 0.764688031197058, 0.300811475261198, 0.838865613935359, 0.679807617396712, 0.328461179516421, 0.538263665925179, 0.837816925078426, 0.242976884143918, 0.0194840559449759, 0.607478653900332, 0.538793036021842, 0.22634947908528, 0.919941012173987, 0.935995221534033, 0.488629533539644, 0.697627838276884, 0.516585647824518, 0.285233545526516, 0.00945048493516643, 0.29294971536843, 0.666484506596781, 0.155092369293142, 0.225577605295406, 0.877622027971485, 0.605727560686071, 0.646813809044758, 0.0691828794492394, 0.218545646898892, 0.779994277680491, 0.629077444732318, 0.0706164231144557, 0.657218575100363, 0.519659436030759, 0.947529638503565, 0.020850670754902, 0.728717839188495, 0.398517860314893, 0.687246295036282, 0.0730870653482387, 0.758045563208249, 0.050722591029533, 0.64241007908695, 0.500774905032007, 0.530089774954143, 0.738251527013513, 0.274568454562043, 0.238035649865866, 0.759943112834004, 0.960094440618583, 0.166206360459239, 0.378593891588754, 0.0968023809607836, 0.934562390018701, 0.802915865891375, 0.730205328452648, 0.794713406885929, 0.573569621911423, 0.361690147915376, 0.161333658106953, 0.886370585396897, 0.901998690405762, 0.758582140498749, 0.361792126113733, 0.496224881215363, 0.658685485864018, 0.288966782967414, 0.143057151152454, 0.308133123131642, 0.347441299611992, 0.139538463016827, 0.216238529731889, 0.555821687231147, 0.0511837396110727, 0.417352629067102, 0.711580162983245, 0.879273035145493, 0.601479439532303, 0.464972047659108, 0.815420643331783, 0.322124911346677, 0.376389436668927, 0.138015717117946, 0.6535923869703, 0.0254919736308038, 0.23368634603103, 0.978782120362435, 0.943021444117846, 0.981063854302025, 0.066963965723849, 0.195257465046639, 0.423405961364762, 0.537076428867559, 0.463277149042941, 0.62590939531874, 0.502559399774613, 0.791777049227957, 0.710351922743513, 0.569404875609126, 0.157098778152825, 0.625603360322974, 0.925833323780883, 0.983197125982487, 0.723173576203451, 0.187558434130952, 0.860131725000804, 0.129059896593257, 0.74347161088726, 0.0478784442107644, 0.649699439806262, 0.0595371200358907, 0.919762447188517, 0.171724525327324, 0.0880464176538843, 0.304542795853931, 0.67159514985602, 0.204443303050323, 0.974809670601372, 0.0395596684082392, 0.49179106642333, 0.640932962398653, 0.40501613579226, 0.0570819583651246, 0.630336994925782, 0.214581225644755, 0.629709414429345, 0.394560935064163, 0.796107697524695, 0.879036941926672, 0.780077122353383, 0.702482596425827, 0.420672226718274, 0.643855831622151, 0.0783089938170077, 0.380749971057497, 0.43768922365533, 0.406951441197482, 0.273786188357266, 0.231351027027652, 0.737897949419493, 0.879400035000465, 0.772488018509588, 0.0358328925739689, 0.554316329289458, 0.588174490828259, 0.246188510868016, 0.398386263151892, 0.806779803711287, 0.301835603145397, 0.489634390843651, 0.781287145262056, 0.449976974673924, 0.833240408663494, 0.676674584803263, 0.0866290637763408, 0.980718474806977, 0.566428418729578, 0.207679389907336, 0.0956086679195497, 0.062521467236323, 0.0257820180502023, 0.334126941923027, 0.556806599520539, 0.845939420795139, 0.626754221819756, 0.599863609913649, 0.210636149188983, 0.253914831802292, 0.651973711260126, 0.328916770934103, 0.523173950113028, 0.124716558330939, 0.452696285403787, 0.992595860667378, 0.399228825940178, 0.0619117167566564, 0.29296125597282, 0.552271825080349, 0.740127279264061, 0.318404496836786, 0.493858175306489, 0.628668641699644, 0.753146300574143, 0.693887330240748, 0.136880534264345, 0.906962104678691, 0.378876588287616, 0.529622209285288, 0.988615001590479, 0.95820931345633, 0.81764607651304, 0.423895470686491, 0.817976533449552, 0.444495886655101, 0.380260347416991, 0.707235169146731, 0.238916119195147, 0.175765112638488, 0.23814676751737, 0.283591618719028, 0.0406487137340067, 0.551416683951155, 0.942498982581975, 0.787836608192952, 0.241432222328273, 0.695975547553569, 0.0152575092526921, 0.768547506744655, 0.331892690873412, 0.322330093998334, 0.765437295434549, 0.384164971274996, 0.218626720990041, 0.563322082264663, 0.220633325251943, 0.162637054472193, 0.399640050093037, 0.0203949410630104, 0.423270130284549, 0.00327329991158387, 0.0448600286941586, 0.189124115146121, 0.20133633199692, 0.600349620129511, 0.231722457925969, 0.566547879788075, 0.550749671427242, 0.338006293843717, 0.989923506936783, 0.148359250361407, 0.851287667320353, 0.787180010527188, 0.524300977638559, 0.0188389622432148, 0.263451715194578, 0.998589723414933, 0.710540617746818, 0.218090334789492, 0.641612687350768, 0.12106824838672, 0.556681340271155, 0.787968206543568, 0.414501925417731, 0.0836644060611021, 0.587106720646176, 0.639266464024492, 0.125154731347819, 0.865261678105272, 0.48778428565265, 0.276017818678229, 0.311471635950869, 0.666566706759921, 0.500098078791296, 0.444138521478666, 0.325582509471333, 0.565923689643975, 0.0152227150873103, 0.552833547476891, 0.935824215383956, 0.481772301596827, 0.328475651526597, 0.803081686629851, 0.143465565201252, 0.07862275160931, 0.445407925813256, 0.465510336176384, 0.736678637754769, 0.699037204178715, 0.533090326961467, 0.206323180349956, 0.156967245259197, 0.402239519762668, 0.437575868338577, 0.975063468894014, 0.926744192452944, 0.308064151740705, 0.862582601622939, 0.560227281225362, 0.5441622817794, 0.842279598560883, 0.787970869649794, 0.536371208102865, 0.0018303838721293, 0.370152466633967, 0.436120802744759, 0.63320250489121, 0.14677739654492, 0.565367033193397, 0.403116857984051, 0.0539818085177951, 0.135984719774865, 0.390509652996561, 0.457555564498429, 0.331911062224268, 0.857332663054989, 0.607155815474691, 0.19711833280533, 0.143521090561309, 0.767379950824231, 0.652723109508908, 0.34227988053656, 0.0566110406990284, 0.420361597358419, 0.460863802875603, 0.535502643550151, 0.828422805979749, 0.99837009190541, 0.382405458794742, 0.190079981771239, 0.992391333780454, 0.679936803126806, 0.934896023577981, 0.636410175625084, 0.648704621905513, 0.382611717672659, 0.0278736060452789, 0.593066375584962, 0.731305229694702, 0.653124293686903, 0.0619615956486654, 0.153040789539826, 0.755653792294532, 0.495833218424375, 0.384369079163957, 0.288317822217499, 0.186366464429075, 0.947337577515949, 0.398056390262167, 0.139151681124447, 0.541372661862117, 0.0305163950854274, 0.548450043884377, 0.396588617462299, 0.760945615012275, 0.16278458202546, 0.468100120595734, 0.593644658893117, 0.0853343767891372, 0.497140759324717, 0.776898758063257, 0.133817922442649, 0.700194935136202, 0.830731114447627, 0.513147833932878, 0.00481381008987647, 0.356711919340374, 0.973021728005723, 0.144657628427456, 0.774765742405549, 0.360979298860082, 0.281578150850313, 0.543409504461026, 0.407238592459026, 0.920303860124535, 0.356899308198899, 0.112509233229014, 0.255083897211059, 0.15567712194058, 0.813110879775434, 0.629638827774502, 0.659374308712731, 0.664117604363941, 0.547277026175674, 0.285949044987479, 0.0973131576039695, 0.831218419657925, 0.315020436471353, 0.598783566224333, 0.468767695860642, 0.658211338813577, 0.260999895824086, 0.284530442375231, 0.691056636256199, 0.973478589083218, 0.282558150297223, 0.853993012569155, 0.901327981524271, 0.837754162053049, 0.752279741369618, 0.259131280125865, 0.947138489197696, 0.847132318058908, 0.842374695070539, 0.777156343575887, 0.473755329949355, 0.851165526567264, 0.929184422884571, 0.0242436465594476, 0.729087305104052, 0.466319730923973, 0.591193765502943, 0.984497712415632, 0.847007491320461, 0.976697224260452, 0.344855209713522, 0.277363250387578, 0.316407242248477, 0.33648882842007, 0.548099909662181, 0.285205740939857, 0.915859244348825, 0.670683297015381, 0.785491175591435, 0.183005379238528, 0.140885847649733, 0.0557446501296444, 0.618373489821286, 0.589941389374279, 0.375036038352001, 0.0226182249039055, 0.159251181302361, 0.506772879136637, 0.787172059365427, 0.762037590058288, 0.934324779439803, 0.795179958221634, 0.454261390379564, 0.344398683625314, 0.132859803204658, 0.56123375128913, 0.207615247634502, 0.11800666145835, 0.767654663544934, 0.457255282782492, 0.725843102869637, 0.106725569491779, 0.439141166369223, 0.557388009554416, 0.855583682509651, 0.911929354923008, 0.351330301722152, 0.372362436968764, 0.102235461961869, 0.657593190601165, 0.218809858555459, 0.866431315646603, 0.393404889143131, 0.453884991158896, 0.181753759052123, 0.067438529210575, 0.784553384142327, 0.801866982799001, 0.46341895470128, 0.852422629064397, 0.62564799356442, 0.180490998845987, 0.0300702094583002, 0.129778491130344, 0.315606096332486, 0.731534578056076, 0.772980317541286, 0.148568765553056, 0.795799583470678, 0.165345277647948, 0.717102915689721, 0.0785568343415051, 0.393865980885231, 0.815829943432681, 0.529975271796378, 0.947356451382742, 0.316680767252843, 0.578165406939369, 0.190229096159925, 0.392261112726517, 0.998756283443465, 0.343688229748088, 0.908570819571801, 0.697666340991445, 0.0297702368308634, 0.481277122200334, 0.676905434158697, 0.402482970825134, 0.154091033772534, 0.561167878148708, 0.949325495276995, 0.959295423619886, 0.989940799059916, 0.836013835411808, 0.654080037125545, 0.869172938937459, 0.863771403304308, 0.761924626541841, 0.165267646441439, 0.0794741568677688, 0.508894579290508, 0.0930402625569668, 0.437859128847775, 0.311971481102577, 0.379150875535018, 0.995592367220847, 0.869492229395656, 0.947906559807061, 0.254304842122024, 0.0484916324220905, 0.831838695092381, 0.688725578355128, 0.596021891944303, 0.21473979076633, 0.535169282982425, 0.055755557082901, 0.431049305282942, 0.190738233247638, 0.32777545973882, 0.937657078181142, 0.847367010662909, 0.256983639677106, 0.37720553423657, 0.733514334816772, 0.782224136821224, 0.0116370584518722, 0.221706147649821, 0.0670717982470392, 0.705654138788337, 0.87593417313855, 0.277482257266782, 0.692804116921939, 0.139013530408076, 0.37476900695331, 0.835877833468623, 0.695748329396229, 0.694899405806817, 0.853361787858345, 0.294588222482631, 0.65096798500137, 0.298627873361377, 0.944677488887656, 0.780789014157975, 0.825018173215279, 0.736184777058494, 0.793672564356399, 0.405157520190284, 0.130611902000957, 0.159556404462258, 0.81664390299325, 0.0305541308855796, 0.543991457088426, 0.823497135695437, 0.948397030081815, 0.460268498841646, 0.0695003047793414, 0.522399521736054, 0.195857237275154, 0.298711468445708, 0.975715106193772, 0.165316044856419, 0.777349978774401, 0.341378302949708, 0.474102851800009, 0.100236675311602, 0.060181696514428, 0.861720314705014, 0.0708716121029944, 0.882795629868792, 0.393839988588658, 0.440050570288512, 0.350732038346523, 0.0323509215747284, 0.725566010997692, 0.353489736338562, 0.641060414858034, 0.771589045418736, 0.677080580459944, 0.485872486616551, 0.812129130758608, 0.0759967149124641, 0.0283328534410648, 0.433507020834965, 0.143478050198143, 0.808037066329201, 0.0017547536443514, 0.0962215381066635, 0.421675024973116, 0.235113980185245, 0.95630117360797, 0.371487236516994, 0.491320074854713, 0.267509216346759, 0.576280334840543, 0.520045327247974, 0.978869080627559, 0.66074062672636, 0.846242347596213, 0.0871435958533396, 0.422875820319271, 0.0414064878758635, 0.602131935626858, 0.731446443100109, 0.682628693120168, 0.061031946367778, 0.676502886792259, 0.307075078822024, 0.348293550588033, 0.726499574168278, 0.107091998815553, 0.252499056193571, 0.438930682808095, 0.587179731499977, 0.307006568091182, 0.435469011349774, 0.310336376334416, 0.915968714765546, 0.658419640389902, 0.176906544568258, 0.645103934721718, 0.035292728592597, 0.782606590840516, 0.00180398525284597, 0.384515328479833, 0.672467057454, 0.0394906003262301, 0.434759300755122, 0.174380740094313, 0.583211469644893, 0.19103579405284, 0.383639378075521, 0.882179780555006, 0.00184136063443696, 0.680203428109989, 0.282514608313118, 0.821408632195107, 0.294109431454648, 0.390100290285044, 0.951080809713353, 0.835254610211636, 0.890216671253305, 0.932588997352431, 0.129360617046314, 0.348286226703185, 0.713610359777742, 0.6251491990316, 0.833618740770845, 0.220503888860836, 0.31932052897071, 0.219386069688385, 0.028035339401949, 0.680209782371017, 0.286148937183355, 0.0307907779697146, 0.284450926073586, 0.113193778907647, 0.0716234254370061, 0.581015353687781, 0.010961023186938, 0.843501298207087, 0.0660617909315602, 0.484097038272404, 0.16240271355884, 0.421958202441673, 0.863059982999069, 0.656410120407126, 0.962473544255085, 0.395760035175164, 0.411929571699202, 0.779264949423058, 0.902051342746027, 0.66931158251522, 0.171375631528944, 0.319782456406807, 0.271205893830029, 0.230797235793655, 0.702491951192397, 0.309470439876305, 0.973869424941119, 0.862436071698607, 0.619773122559444, 0.879446841360147, 0.307114125828345, 0.626195784314279, 0.633983223451217, 0.301197443496774, 0.58295388716435, 0.107676235177141, 0.798408497912839, 0.87513680684476, 0.647669682070392, 0.0550691998651349, 0.938561914809799, 0.057961392756911, 0.249144978116015, 0.375117890459934, 0.413529288103275, 0.165184220433118, 0.941973857509164, 0.128043356055901, 0.755388830654624, 0.640155426290878, 0.306615462983474, 0.115718888510759, 0.955548365898243, 0.824147963955575, 0.152197387254032, 0.392890269796926, 0.462762761766449, 0.737818251451129, 0.875653560182455, 0.973842391577327, 0.723877205001363, 0.622914554998592, 0.842092788873228, 0.378041997485894, 0.898681751773086, 0.788760857952852, 0.908798066026294, 0.42141020612364, 0.0705867719123744, 0.907522626731964, 0.612738786508213, 0.750338175801376, 0.580403345441455, 0.99452242313054, 0.165867413446024, 0.70882291571104, 0.16670287609451, 0.0793309029041546, 0.634077151094271, 0.551480744595699, 0.772273631096565, 0.054882515753559, 0.0466424456658761, 0.582509892505412, 0.761744751885447, 0.185260368095339, 0.535964441708085, 0.573696348603338, 0.604938867660859, 0.0407915796933144, 0.270122793532389, 0.1789488206375, 0.5277281949498, 0.735075274240604, 0.917633085656947, 0.0305984436758793, 0.824218068228895, 0.847394209195945, 0.469635662188504, 0.370318725663683, 0.075636685022303, 0.16995707178022, 0.671964612452888, 0.590129244280405, 0.66778831773947, 0.450259656561841, 0.79194430412096, 0.455798150978936, 0.894465072748311, 0.0725565114773917, 0.635435252910313, 0.808621719424292, 0.321098324279368, 0.536071659552693, 0.866079957955654, 0.534216138527274, 0.793612134065617, 0.586233173298685, 0.881507006587805, 0.443176311942374, 0.0931638984832585, 0.929496207556362, 0.418572454279479, 0.526702554124962, 0.690943595510226, 0.0546573236981224, 0.340409571393862, 0.792788715672613, 0.00654511040386185, 0.0542019109115516, 0.274073754592981, 0.357478809294579, 0.781056681981743, 0.188844984713199, 0.406479597643488, 0.693954680519029, 0.82948044748936, 0.546175271849303, 0.912339173791593, 0.917628651012621, 0.618542000924163, 0.910471645956192, 0.893861143599632, 0.345292825334657, 0.996385344429187, 0.890264966428751, 0.363815795211376, 0.782896572618745, 0.328323519313295, 0.156707479627937, 0.739483572866722, 0.631736699977235, 0.599903637131454, 0.240561982020034, 0.282346756420681, 0.798736718316478, 0.586944946761127, 0.301752776117387, 0.296111713525832, 0.225205429403963, 0.505441762061689, 0.565036631059077, 0.0688958474954688, 0.163903075968651, 0.510518780134503, 0.541539257161068, 0.51003592156681, 0.121916607907219, 0.155749536131136, 0.409146414495001, 0.693653336369023, 0.780771694941078, 0.636363369882475, 0.581805167787555, 0.974733516778378, 0.529638235684598, 0.288370830577306, 0.196208702050257, 0.215372962839834, 0.437663408267884, 0.566949720723725, 0.915583485397488, 0.316097410451932, 0.281023091147718, 0.618898841784714, 0.220444812720516, 0.172017444134905, 0.0164175672011078, 0.516333711494217, 0.002416171544668, 0.556002444329214, 0.555734261592874, 0.073732221838767, 0.396909231442454, 0.77220409585751, 0.572535518845722, 0.403069134364186, 0.523322004017897, 0.392486014299919, 0.0670251061505053, 0.778518886564804, 0.764309893976533, 0.568529722597221, 0.614462640111386, 0.588870389932447, 0.616628568319131, 0.239777320188078, 0.791083701519205, 0.705646703447523, 0.737876332989012, 0.13841777189992, 0.00549026210557846, 0.738517570539967, 0.810786684258385, 0.564721516396457, 0.0556820212912313, 0.323096406037047, 0.933614703164219, 0.852757845924034, 0.686772701820273, 0.167237727486248, 0.679664293958122, 0.0795688618013785, 0.56100376448481, 0.459041822382362, 0.665067515123777, 0.434114442662507, 0.0307028759373024, 0.442408672873009, 0.467066509078016, 0.0103843695834612, 0.593550306259397, 0.30163095013132, 0.825934109059496, 0.915917389447592, 0.267353022363131, 0.296017020496327, 0.0924736318186949, 0.725831812093107, 0.717460274478425, 0.620868096152535, 0.376115038002208, 0.630671595427571, 0.517286751502089, 0.650206239200601, 0.87125422695433, 0.331019185865252, 0.557544888630403, 0.0286139199897849, 0.220865513066866, 0.785588440622026, 0.290904144706537, 0.0571833459536408, 0.698900822013167, 0.638333087953258, 0.647709210354812, 0.88260306597814, 0.164070518293558, 0.666668424288991, 0.383200669584051, 0.693227732690363, 0.304173140434863, 0.649632186815609, 0.359535240315204, 0.912867761114875, 0.388772495378266, 0.678021857848942, 0.752663777457531, 0.199393847033604, 0.678316995397072, 0.791000739337585, 0.726212306605526, 0.175196358724433, 0.103216090863034, 0.81754542241579, 0.980096005521268, 0.751684027310974, 0.90974084513227, 0.608514994465511, 0.45920136851199, 0.045913512100114, 0.599667817530127, 0.781423935309801, 0.818052364315487, 0.22308238126626, 0.732903459782046, 0.677559090973786, 0.972098144006049, 0.411600352045823, 0.0512268864142016, 0.315270545084878, 0.31475858234969, 0.492736572862743, 0.0556306292651011, 0.0949827963051177, 0.239078277853282, 0.782435934216185, 0.868830407975705, 0.62398405210987, 0.332523373668957, 0.271247940839793, 0.0319739703791037, 0.869332459866006, 0.247646763874657, 0.985421807575469, 0.127586383350604, 0.532953449447278, 0.378596783996379, 0.783771696788792, 0.914090332273794, 0.361124621035295, 0.91104408015182, 0.119121720497745, 0.0348276901679443, 0.904506728914641, 0.930740878431636, 0.589996197292355, 0.0564480697228618, 0.882159502849003, 0.849515591686605, 0.568739721598088, 0.460585525664272, 0.0676049985569037, 0.38321314125404, 0.740402324494855, 0.52232751037915, 0.496759262274967, 0.830198788161747, 0.0167762123929304, 0.264058568502416, 0.141901125095577, 0.0329705520921767, 0.289313823959527, 0.89386616837622, 0.163104474956854, 0.988978670224991, 0.131302381093545, 0.467299577319128, 0.370159349987236, 0.657561633275093, 0.873564922067754, 0.294875564962543, 0.27282504504105, 0.0808910056328515, 0.186345829527341, 0.760009352669009, 0.909609018931668, 0.644622902814197, 0.517449306261014, 0.701595048422459, 0.0450741995339383, 0.890251371223444, 0.672619760310943, 0.164873006449157, 0.502564483126831, 0.531819449418421, 0.624002538761107, 0.96456849733222, 0.569174194009442, 0.577671953284539, 0.937487785255635, 0.0615602129472785, 0.658996636625157, 0.0993981036500629, 0.608668865394413, 0.804352282814979, 0.0323153708239287, 0.521932951398231, 0.297647950773702, 0.6606481665126, 0.597717356281389, 0.0615488767878051, 0.891163811880989, 0.953334131435845, 0.180876206379909, 0.073545247878916, 0.988031740772864, 0.427905968422523, 0.758982402109534, 0.923166041275947, 0.872115748653451, 0.963364612210939, 0.296760292740772, 0.71003869762877, 0.901097787084415, 0.874200675667437, 0.828601315190104, 0.0352094222612055, 0.906135806383649, 0.627932504498251, 0.572596077361343, 0.738186962051856, 0.508221529990653, 0.864826598284171, 0.140238369386772, 0.657135956310468, 0.592993758427694, 0.513366628422572, 0.244619914485927, 0.973505625311446, 0.916214299891215, 0.6520095504672, 0.110501844372499, 0.514626474410192, 0.481052599317722, 0.0681718734048383, 0.388843144123239, 0.805203215911553, 0.786655637028361, 0.119977227811961, 0.411976393369852, 0.798066760618606, 0.252720060568718, 0.587479338035421, 0.0367334280747598, 0.681630126531708, 0.826819924987296, 0.699910631416603, 0.351771995822958, 0.48914740808751, 0.284127022809931, 0.265970544306739, 0.764926458085946, 0.58437933050771, 0.538556461892072, 0.99067477991206, 0.769157985149629, 0.58609807774805, 0.790853258887403, 0.548924856413875, 0.0764199308439892, 0.474316534660652, 0.849727496009202, 0.403079362915694, 0.236797569892631, 0.541488542065278, 0.368907202722624, 0.758521516279212, 0.769036947395334, 0.346308372955164, 0.967452915231563, 0.355807120256821, 0.193194442421816, 0.0910558747253021, 0.313234823307591, 0.958343792715846, 0.638464280992852, 0.208140010076906, 0.333122153950807, 0.343015689310365, 0.894842939476145, 0.504160043115009, 0.562223438049246, 0.789344549208544, 0.350928314133683, 0.220180026909679, 0.822832180422898, 0.0760419419301047, 0.0821856780218047, 0.309716152331043, 0.855495581104989, 0.474297337651695, 0.324237138742249, 0.460043815587336, 0.67417460873382, 0.574061870053921, 0.842825235947153, 0.075821527460479, 0.259849889012651, 0.687559066821638, 0.432499273480823, 0.323726473240111, 0.0460720460262015, 0.310353092407763, 0.426622732841337, 0.615571878010216, 0.243014654438024, 0.84471659973377, 0.882797388089925, 0.928256135074519, 0.56263375375444, 0.302985684869484, 0.770146814816446, 0.507220197392055, 0.612585159631817, 0.686576418239678, 0.0839054419965696, 0.714107751355513, 0.298378969306807, 0.233191378279156, 0.317444153824896, 0.282018317157157, 0.25056373043785, 0.123544643414438, 0.668144126672398, 0.27839076206333, 0.24756833366693, 0.568767474783378, 0.0513477287859975, 0.823165874217522, 0.0369454469815478, 0.267170720746628, 0.302189867878529, 0.662304541532659, 0.625757474800833, 0.712340670978156, 0.275834584707883, 0.996084814349018, 0.300446289033103, 0.501265364878921, 0.132322364847183, 0.89755990010348, 0.237941500948258, 0.0670901057522987, 0.365623863649467, 0.989005563991414, 0.212024554796666, 0.287534636449249, 0.504589313925345, 0.630568334005913, 0.393183301418052, 0.679960521952362, 0.130033528591493, 0.0702381847159612, 0.045383669407882, 0.553927958731331, 0.890816186871401, 0.585793736675153, 0.531545942369249, 0.644040181941977, 0.407506672042408, 0.261374692222157, 0.609210277533585, 0.632786303309675, 0.759272640614699, 0.233894078246413, 0.288725339203068, 0.0226978341164412, 0.291437156684378, 0.354082613588034, 0.649630311408537, 0.607601700563822, 0.756167923596291, 0.884363551511168, 0.89550269153701, 0.0780761008444534, 0.525871135885568, 0.0451454253907682, 0.796612607887417, 0.314831111996124, 0.235507341550144, 0.267437572186468, 0.142383823921751, 0.193852082297026, 0.833385561052224, 0.745104264516339, 0.743854537613791, 0.265942319270568, 0.113418325852813, 0.0489807128052342, 0.276119578046616, 0.65735966159897, 0.626700986549613, 0.574108017616169, 0.200358343371018, 0.409374043617391, 0.284313720327795, 0.834082705105683, 0.757351681107466, 0.865907887445865, 0.416934669908416, 0.742513517841662, 0.0699354873125952, 0.958453942044462, 0.68719074698325, 0.260030464204147, 0.735625776527832, 0.478618395085393, 0.871374928352472, 0.322552663998023, 0.561023285528273, 0.245850036918813, 0.134488408708818, 0.237611512578303, 0.363672931742862, 0.688088512012782, 0.0390195122460559, 0.688733984283037, 0.260817682013955, 0.856589328262363, 0.546114267194284, 0.460498710407753, 0.152665939974693, 0.255788548706678, 0.391645805590429, 0.964606511590979, 0.519423883254554, 0.120883855081262, 0.914894686910449, 0.456607321713078, 0.9046827646939, 0.291900257612421, 0.782188832835848, 0.0559069255372359, 0.213043797351518, 0.776544044444511, 0.696810151591465, 0.907882716205727, 0.608882731596351, 0.544556787196266, 0.753117743468018, 0.877259107150336, 0.893773611198366, 0.855831450381452, 0.683244278757409, 0.407614455577635, 0.047855291222163, 0.238427738425595, 0.0139591852522315, 0.381699933629399, 0.907122966953268, 0.303595631262235, 0.867329904127074, 0.14717268033619, 0.309912600447444, 0.418244026857659, 0.210175701878343, 0.767454867737524, 0.242545724394995, 0.886141259508753, 0.297543142455649, 0.190677953527808, 0.72411948735155, 0.956810368364428, 0.901467029531018, 0.700584448355976, 0.813472056506204, 0.866424693493208, 0.88821315487955, 0.230451301108749, 0.626475535657307, 0.243942257793792, 0.442680123313793, 0.782785582693547, 0.715829336459532, 0.244633404403314, 0.737832292346297, 0.907811867621156, 0.842232477650557, 0.971466220988457, 0.142859491474701, 0.856911997459438, 0.762051784915656, 0.76026155224039, 0.459418118775865, 0.384084020041047, 0.107428681932874, 0.455471809246363, 0.147246518434498, 0.293591548700414, 0.380417492091752, 0.924546480405133, 0.193467883962102, 0.655050538856556, 0.394228693004462, 0.193486034492081, 0.0317353525689889, 0.801822278408664, 0.313325325320564, 0.746942359108157, 0.590634275144069, 0.296428660980551, 0.815503242315164, 0.892512509373892, 0.589582772188245, 0.815188486290697, 0.555127769649246, 0.814574005385126, 0.0383539150350227, 0.848485428857009, 0.114485858633026, 0.151174262775075, 0.728034491713809, 0.201974027625232, 0.738008455846378, 0.867171221263074, 0.209038974682237, 0.728759777502934, 0.613299975483702, 0.512972180342107, 0.959335269074201, 0.184443548361911, 0.200737704237217, 0.659120691561232, 0.73910510913024, 0.374367847879008, 0.976079698595778, 0.035211259909129, 0.328246488415635, 0.650959339968381, 0.0393503446873331, 0.13562201450484, 0.238739364541859, 0.859359462229851, 0.914478276499064, 0.00760761002230126, 0.246181557902457, 0.609729696858451, 0.984513582283362, 0.979311190044503, 0.619194637447203, 0.948317968491582, 0.373760756735783, 0.227565375346174, 0.568183289371269, 0.560081271920557, 0.281102520697554, 0.872199114562592, 0.845636604435765, 0.647978711094238, 0.723668530402102, 0.266740249623012, 0.121991028115945, 0.150852744780748, 0.667905929007622, 0.336550587635755, 0.617685248614141, 0.775325616592217, 0.732936280103416, 0.736283558244648, 0.0599006466906205, 0.227050611254484, 0.466279469338536, 0.898957920892012, 0.544240637065518, 0.0571369650366444, 0.794492661558489, 0.774849110506541, 0.509021208740931, 0.0257517995825106, 0.975527527517364, 0.242368419154235, 0.901011772014337, 0.448383782974622, 0.0627939272988536, 0.0992591275526461, 0.368498156218048, 0.513530624194578, 0.380071368278669, 0.00169660405914329, 0.717576837477861, 0.345075963786048, 0.656386819847764, 0.169334994904026, 0.621150838370814, 0.424611275058102, 0.677083467239989, 0.178810185796471, 0.887455637212483, 0.159934583953068, 0.0978455485899274, 0.761438918911578, 0.894076152223764, 0.247727253500115, 0.597233260020138, 0.0766172962453304, 0.266547538274611, 0.846328076626995, 0.684056406858091, 0.487419392207132, 0.356783439560155, 0.213334544013397, 0.228838341026877, 0.764782048395018, 0.396143550272663, 0.822659247713372, 0.525027913483901, 0.102738943558564, 0.048382540701817, 0.938880609732373, 0.61473538118606, 0.991897359182668, 0.476777866350487, 0.982657567200865, 0.572066651345122, 0.232496979156012, 0.117017285771908, 0.13037640152561, 0.741824103710231, 0.138313264430914, 0.187152091446844, 0.396475725840333, 0.335995501021902, 0.458718027215108, 0.102915435010245, 0.170467627427738, 0.641398638511119, 0.78768248797398, 0.372359685392546, 0.306597006114821, 0.897637887690481, 0.724393456837973, 0.406703414775156, 0.186363016824633, 0.0515842380028517, 0.943964003732852, 0.420620040014145, 0.596144224822929, 0.51354418869711, 0.895827497800479, 0.185832659305623, 0.0935017105822687, 0.0565558653007302, 0.0476367591715683, 0.682719581513607, 0.475505148161666, 0.927014113630185, 0.411362997872264, 0.870005757027722, 0.710000176993582, 0.942366732692211, 0.480171261790828, 0.434611189544999, 0.512072548223904, 0.524489965199801, 0.469051032847975, 0.958542844315621, 0.856098365868987, 0.305534127202869, 0.720322397957868, 0.85461826338332, 0.30769053013517, 0.156802762402933, 0.0771065877119569, 0.748091265752701, 0.534212891875811, 0.453919233848035, 0.792426884258441, 0.218776684921725, 0.306509724541186, 0.731537491800562, 0.459925963659995, 0.405042885330904, 0.251685269100308, 0.581144730433948, 0.808327436867258, 0.0504336916413133, 0.148746386214487, 0.491410020650077, 0.891694644230395, 0.313020443381926, 0.282826166325368, 0.0647121084466005, 0.0605151038488364, 0.579086631659611, 0.836962990640345, 0.347393430210407, 0.348428889936716, 0.0415269237204632, 0.391318374563894, 0.866306502020659, 0.0252594423804325, 0.171378239916805, 0.83092754719348, 0.0128308541561479, 0.466689640137522, 0.831623506789702, 0.900265758421862, 0.507053187458296, 0.838729531957502, 0.567656773449172, 0.773961150177168, 0.312720978207796, 0.107592461277769, 0.694582420993042, 0.741751248827562, 0.396930610524544, 0.346585733222825, 0.0543589633141243, 0.37486640209912, 0.366856082017091, 0.284172863820933, 0.914388669534335, 0.660146342601309, 0.425533741218118, 0.625100294479835, 0.600122179491422, 0.980508493704767, 0.764330942615799, 0.718900321707892, 0.880662068750788, 0.0106503008422745, 0.571535370546996, 0.971603380632146, 0.533629837422273, 0.0627168017233694, 0.564695085378318, 0.228126560211354, 0.440578498394002, 0.098905675505357, 0.127115742599267, 0.801628699007852, 0.254883088972119, 0.49992846387618, 0.522995609955883, 0.0196049993426315, 0.877013597667148, 0.46010315914106, 0.979938806059452, 0.56947491107823, 0.874304268392361, 0.241524416499248, 0.117559394450335, 0.151404575196897, 0.059493900012201, 0.200205950054691, 0.868241887576531, 0.770874095022765, 0.931870752410629, 0.0807529971218045, 0.0698914508974635, 0.977305036427196, 0.521701720928797, 0.78769907471917, 0.309754290835704, 0.118515907454858, 0.485428085193575, 0.415277708452992, 0.915841035413253, 0.15945766559324, 0.203607917298397, 0.150089535165601, 0.957696443803654, 0.748681324956193, 0.206451980537833, 0.665701308737751, 0.2195634606961, 0.449300229401053, 0.940374486077781, 0.58193973276751, 0.147726755535676, 0.629530818460087, 0.184302726638946, 0.0828925618287876, 0.213927712657482, 0.526436081645721, 0.666704281073031, 0.45899921020417, 0.215458622864084, 0.22027565835521, 0.890088497396006, 0.301552910833664, 0.250060596747603, 0.896616649062539, 0.119727750238206, 0.665198525729057, 0.327472361482963, 0.326094275499449, 0.825488198307434, 0.550211589193962, 0.644777627836345, 0.845129221893249, 0.594998942858514, 0.330986506516012, 0.211408575923907, 0.625687545326982, 0.263061503131858, 0.975649063469879, 0.161356114000083, 0.79454246957069, 0.924433095877258, 0.466307715142174, 0.61356510606666, 0.410488860573768, 0.969224539040448, 0.653199934403763, 0.771409919806905, 0.415174840274179, 0.149070708808015, 0.494318775311428, 0.848694388607203, 0.677785931430129, 0.445928229153763, 0.49707736099998, 0.156128141242073, 0.697865423779033, 0.204691800495836, 0.677374610069244, 0.124508324888258, 0.595471508982274, 0.772918276174151, 0.579581470041055, 0.683934154021532, 0.427302931829224, 0.276805534590329, 0.978040385175789, 0.96491728377686, 0.728053972877367, 0.724777304632142, 0.644964585196439, 0.0858497215766604, 0.527904226389608, 0.851542742669723, 0.209986036918477, 0.826739561415363, 0.2064095695891, 0.475495627297075, 0.299928693486744, 0.344969885273051, 0.92794993715419, 0.0301212458382484, 0.72003619511874, 0.376215712094936, 0.588028157609236, 0.329946684253459, 0.864926871458687, 0.854954785066285, 0.54809927987423, 0.0434136672547442, 0.533580393824481, 0.825267805030646, 0.566274570651202, 0.219237477073995, 0.323337659287901, 0.0650433077580499, 0.373822578367492, 0.258527369669353, 0.260603249969576, 0.472485253765861, 0.094833110251867, 0.0631454337532877, 0.816480682565122, 0.0684481793352679, 0.297635229729985, 0.787570709887446, 0.456372395119526, 0.910177298905722, 0.550671099530174, 0.429217538247723, 0.399053573935824, 0.063296603169351, 0.716191719759761, 0.719923681383164, 0.74117270442551, 0.689373196607866, 0.260415736246741, 0.00987002688579819, 0.350101348729123, 0.896160755443216, 0.74323603767074, 0.29398629983106, 0.184342604119778, 0.162956226687924, 0.075790019563777, 0.436978740562955, 0.271619738143893, 0.237105380754701, 0.146549362796243, 0.470824735615893, 0.728587851124837, 0.391399518583142, 0.156016490842941, 0.141592359877246, 0.904515542351303, 0.3524178439736, 0.159461502428387, 0.871876954911026, 0.803712432445542, 0.73415489035814, 0.492171222201712, 0.401841879546834, 0.862601071294337, 0.1808363977808, 0.92556784444155, 0.822036222510055, 0.686577957520219, 0.469160798375915, 0.887939019898208, 0.982358897226306, 0.0323355015709063, 0.82477279774451, 0.659317930451542, 0.44827919140364, 0.573719504094903, 0.797915762729323, 0.632758044688588, 0.789703853144417, 0.0595592872386542, 0.38940555431291, 0.420234118136111, 0.66407218905249, 0.634470704931963, 0.885303246347037, 0.611507512145832, 0.615348707277027, 0.877161705233405, 0.766863255961453, 0.750090983119669, 0.115621892999915, 0.204089525841316, 0.116260192541944, 0.841459795418762, 0.381573844203647, 0.454045781177121, 0.0738874433403629, 0.0251649701599331, 0.961114519930276, 0.158394534288605, 0.933642422738259, 0.236765101181137, 0.0755539644790679, 0.672555516137938, 0.084327464233755, 0.0621675919688018, 0.695168900104093, 0.00205895532405598, 0.15685728792107, 0.274550320652175, 0.315229098268104, 0.714036002448029, 0.571786584196653, 0.515193161916588, 0.577329950748616, 0.00494271342247377, 0.352234711999676, 0.544734709612511, 0.092064053002642, 0.272323872914894, 0.590728015148575, 0.00462732395415677, 0.0307507353083792, 0.376707175168631, 0.165515310532669, 0.306224819500855, 0.269974486871267, 0.307209649310437, 0.584021930201778, 0.538518534306093, 0.790670725774906, 0.372718938149102, 0.752009256170531, 0.652579120789334, 0.551939075971854, 0.850919790140008, 0.425248660642612, 0.574400626027995, 0.768253952267114, 0.539514467528075, 0.98427032230492, 0.105585169484348, 0.301004788535869, 0.379853810070236, 0.472715928249649, 0.222568720638446, 0.320742483459245, 0.226850908342202, 0.0217578003871021, 0.0154733849753969, 0.783849853309317, 0.01864708960649, 0.902727159582464, 0.196604736084927, 0.609034727698757, 0.522871110360676, 0.706730330442121, 0.353196240912083, 0.758894376648529, 0.50557570207133, 0.765774402799521, 0.362531943717043, 0.925017556626161, 0.901099215301894, 0.551714879609583, 0.521025047230434, 0.17902461784173, 0.608923256007423, 0.727960171573794, 0.875941509519679, 0.473321817886645, 0.752213780912454, 0.181048297722995, 0.0213357549505021, 0.129271428321, 0.860052226194233, 0.796709315815591, 0.187592221318804, 0.171215005324823, 0.675568403155122, 0.399894501156497, 0.935526882464649, 0.4762351231542, 0.402235787745651, 0.190023460637405, 0.337065796463878, 0.44098071896043, 0.137983753920739, 0.980338805574874, 0.439650061821641, 0.803841880969555, 0.817665022881234, 0.231527841123482, 0.64125836441644, 0.25780983133295, 0.124106931736225, 0.40705777488399, 0.0676924853563038, 0.109760272899246, 0.529556221508633, 0.21150196962379, 0.0259674234512381, 0.994119952916807, 0.744640810683006, 0.982235113456661, 0.951381996639724, 0.471607769507925, 0.987535598807767, 0.852921112353773, 0.309076975220888, 0.951736091303245, 0.11364868133621, 0.117273783464776, 0.000695414597978616, 0.303156433903953, 0.369181311324344, 0.120341122710259, 0.711912231934842, 0.957513757835886, 0.322288205734023, 0.788983788799339, 0.671485891572147, 0.109614340006912, 0.72396254812821, 0.882602098767862, 0.000493125684086104, 0.101943588785697, 0.370384028485165, 0.914689192960946, 0.998988798380455, 0.170493888056515, 0.172617331419261, 0.103070408242586, 0.606147698295146, 0.411890152500204, 0.255782757875282, 0.201684845813074, 0.187689244221382, 0.883772446839634, 0.631413791465568, 0.622426430546763, 0.658268684389098, 0.17708447071651, 0.625848708962837, 0.876455653660146, 0.178202978563138, 0.645573879265516, 0.884122118374896, 0.250516236163517, 0.58622115564751, 0.48687308295075, 0.839086287084837, 0.0516755530489257, 0.739472577134459, 0.554516066883534, 0.42947243827954, 0.779722542628934, 0.396667315899833, 0.357229065656165, 0.65947244638581, 0.394171884355014, 0.178477539693034, 0.0542950305495224, 0.247121241003842, 0.901813342063004, 0.982443093275665, 0.484108990274859, 0.144749572943835, 0.511666573591686, 0.993527617668642, 0.762568901256859, 0.96602377253342, 0.929880550051944, 0.236490563286212, 0.0975221579704002, 0.409774325328297, 0.131172608578969, 0.754566728464401, 0.348103138925641, 0.449783724777544, 0.199335481599013, 0.736517998687994, 0.203367460588069, 0.768160368454895, 0.241280143752167, 0.613733487704976, 0.345771808311846, 0.722631801419899, 0.117223115089934, 0.180788384892, 0.193864461463192, 0.360548444281175, 0.404393724513204, 0.225780108613863, 0.669057078251312, 0.957212846191017, 0.431665129293364, 0.803814209845457, 0.798189196221954, 0.995241013277042, 0.211804401990861, 0.329018849592939, 0.762341846792248, 0.756759497241506, 0.961276586752869, 0.569892331102988, 0.608506117574247, 0.365092220127133, 0.343547564883229, 0.599987953772717, 0.525629021187786, 0.484966518767135, 0.655650299360392, 0.0382519894768205, 0.189922768991427, 0.928745278001823, 0.582979705920789, 0.0154104132279962, 0.293675293386791, 0.0452601821369605, 0.794731529681172, 0.711365969870893, 0.536361268010108, 0.586794651653269, 0.210947366659256, 0.413915007879009, 0.515820663495435, 0.838334877242271, 0.659910638628692, 0.481176081688008, 0.852764707269578, 0.481528120748781, 0.272295872592021, 0.731195708518752, 0.969469264220633, 0.375622592534595, 0.238600103405568, 0.242704747514913, 0.0189393989637048, 0.982653179528008, 0.80261952576789, 0.563485264948973, 0.388829023491005, 0.652528413085922, 0.993708159044625, 0.713765827005093, 0.102366984422743, 0.447288867951519, 0.692794787966611, 0.332955913744586, 0.921748849619465, 0.632490987889895, 0.932185452930741, 0.425920199903581, 0.300489635796437, 0.889028705015034, 0.017373796943928, 0.148055123035974, 0.959162355654271, 0.714469447506153, 0.306466720767665, 0.828096168708363, 0.807891863332686, 0.909255061671775, 0.642812974261832, 0.628685522156717, 0.118471080264435, 0.918972361900732, 0.623981073246435, 0.257584904861149, 0.951303564634363, 0.0462057599233064, 0.0209271416797508, 0.0880501273123243, 0.517842854729946, 0.878663782129359, 0.402179094135287, 0.561323323830618, 0.598155845918433, 0.075050856214161, 0.391490000468759, 0.35498512741472, 0.244788992597819, 0.1080283112025, 0.11228920099477, 0.267204546105472, 0.266392375643823, 0.936081279623042, 0.186176927923841, 0.507435095452657, 0.147594213749106, 0.920700033646083, 0.929461267454393, 0.136751414122753, 0.871569210195413, 0.0123551045927317, 0.722036106605153, 0.878585300998519, 0.387446705382358, 0.246430445482503, 0.111723553462704, 0.846781138792694, 0.659659010424896, 0.7796622028982, 0.358562686466743, 0.184135410449119, 0.0101795572933461, 0.0859147851847423, 0.329467204596988, 0.30083015428414, 0.506571160327654, 0.386605295004411, 0.937296385592923, 0.159541227572663, 0.779709931012603, 0.177462276185731, 0.805293492811886, 0.562383811081336, 0.470074765555492, 0.35961164119398, 0.371774411004676, 0.382872535051112, 0.256306730868091, 0.972037906953203, 0.660564523740152, 0.505286560269733, 0.198020730719715, 0.104384167443236, 0.888413346244056, 0.362959966336052, 0.453713606901237, 0.0647936415783419, 0.850762129905981, 0.27943298041249, 0.911919617937373, 0.0597701738011598, 0.615966324872765, 0.563943232899573, 0.668269816580461, 0.335442969200535, 0.0377942764613848, 0.285785219292139, 0.931249052120084, 0.799851727135939, 0.331074735997235, 0.737419509953164, 0.567439883879176, 0.0160636671491574, 0.425514554937692, 0.0293784350191676, 0.78417666074833, 0.980814838363703, 0.689944192745695, 0.779685378040724, 0.597978510089888, 0.284609461712204, 0.746254817052245, 0.693874247720603, 0.442523147513293, 0.419128308148907, 0.150771375419779, 0.540292024958021, 0.196360266362043, 0.749182671596323, 0.715660247509676, 0.704497043452544, 0.736287353134735, 0.656687144816026, 0.891096146885089, 0.491079509911995, 0.341777207792887, 0.96270873757886, 0.124944320968716, 0.208612260908165, 0.154970171141313, 0.807604865577157, 0.0673665741623887, 0.490603177705487, 0.598418175498478, 0.365020173235539, 0.688558911068573, 0.856907145310387, 0.900638468315614, 0.951495543147469, 0.900046823983949, 0.00653692660244798, 0.0419322032361951, 0.870892177330941, 0.444400581676349, 0.442540532509408, 0.330598620853358, 0.939673106189928, 0.712841262309695, 0.938907021258909, 0.530955940353526, 0.0218029817795918, 0.820391081183784, 0.287496057774423, 0.867379666127282, 0.860210812236999, 0.550632368521707, 0.457238498233938, 0.850024433507351, 0.347778352084427, 0.23846099828593, 0.050937808261975, 0.134958498237535, 0.578133375861206, 0.904369433805407, 0.771166371913549, 0.704143113694573, 0.422411057454891, 0.957856554459203, 0.755589244112969, 0.36600347875956, 0.550614142181343, 0.221429444255205, 0.255944793127236, 0.735253439081163, 0.315545222285068, 0.422405305235033, 0.74844197103062, 0.35308903354651, 0.446458527644457, 0.0339350721194552, 0.7280144508162, 0.621375857610279, 0.125794699349698, 0.21545592313338, 0.308377269255781, 0.0309519093633485, 0.201455092334539, 0.0674297895882162, 0.000152249575564478, 0.636429450040266, 0.802753941415439, 0.394338065680218, 0.262735850162332, 0.283828355754023, 0.71364432283962, 0.194144688547385, 0.580080438912243, 0.247472456999387, 0.935906797581537, 0.564556827383569, 0.879609091101584, 0.992676138431266, 0.946252915393803, 0.478376283649921, 0.124994240255344, 0.0641559855840964, 0.386664880532907, 0.558245691279696, 0.922429224841605, 0.93430846141486, 0.824811938202454, 0.183582125058882, 0.393108093448545, 0.912782147048597, 0.534170292387813, 0.451461598784967, 0.285596533472468, 0.622774097441083, 0.181772486372029, 0.927145928405068, 0.713973625944434, 0.832367769827503, 0.496482602288669, 0.963216885397842, 0.843314200065084, 0.11389755643808, 0.936179270569304, 0.688364344521786, 0.307111791289143, 0.860654552453241, 0.948458494460394, 0.681841228529534, 0.56319510284956, 0.149974089105093, 0.986210263457366, 0.815023860040167, 0.91221997710983, 0.683965901899248, 0.303030790522416, 0.731797418841154, 0.276210352592125, 0.615238724145232, 0.294852119850955, 0.518275470487495, 0.738923783707589, 0.744021085639173, 0.403459442077624, 0.160928420877656, 0.0224009672093312, 0.105878566780213, 0.931999026784266, 0.61554945634954, 0.63328400827756, 0.00668715205199222, 0.214865677674565, 0.0810032254105648, 0.161464824260389, 0.0918035632789459, 0.134133301078541, 0.0717961754764216, 0.268847330928543, 0.0400667051753275, 0.744080103196827, 0.944833135835646, 0.221360537640585, 0.600565369809429, 0.436093243649487, 0.1735605496506, 0.194893982854189, 0.189120836332638, 0.929891654919019, 0.490543801761666, 0.796324414387007, 0.184799114616105, 0.22778451515102, 0.849808373296731, 0.494287070467505, 0.985738675782726, 0.17222661804526, 0.950103063603143, 0.382335092322349, 0.447633358936399, 0.359891306905956, 0.144467011842652, 0.816638916199155, 0.181711489977495, 0.832972792588356, 0.406295642965692, 0.421450808813947, 0.533254182478529, 0.430317476657299, 0.00242130741918201, 0.66465941141942, 0.25678792415858, 0.59192337739218, 0.357849621712194, 0.548053679601372, 0.780688571225943, 0.55888033908713, 0.301692338655292, 0.460614824500235, 0.920918319706505, 0.886181035202805, 0.879473814580399, 0.970343091941273, 0.752305177514364, 0.0155432903193451, 0.992341920032693, 0.455763165741066, 0.520498254930125, 0.0888930235728137, 0.785104577952912, 0.71769383079086, 0.54540157157536, 0.744135687805869, 0.687323293461848, 0.611293550892093, 0.0823415217677902, 0.377002455455355, 0.404969648595625, 0.687377998840093, 0.40501468273963, 0.554466675052666, 0.618265479941905, 0.417461754597146, 0.84487875692714, 0.551509917695376, 0.81314977572484, 0.3153950016363, 0.859363018117095, 0.217365052425399, 0.115355670551105, 0.0858715075213294, 0.154632577071583, 0.802180065044321, 0.96447223448495, 0.196792137039398, 0.778049589387886, 0.0264828584948619, 0.113878875522794, 0.186622583409079, 0.925405285188395, 0.549709937000636, 0.0311732574099408, 0.876555436656059, 0.950212328175207, 0.482800836202897, 0.432901384373766, 0.619807689681551, 0.230256203562491, 0.700548478254102, 0.980992502935693, 0.163515197965189, 0.630784608459855, 0.641791242815605, 0.6365272071573, 0.98721476759212, 0.386913963868104, 0.261724513150424, 0.457808636855341, 0.114374241970301, 0.358104855591226, 0.785349003959282, 0.305115509380061, 0.73067399945616, 0.645077851674213, 0.839365564675445, 0.520610263165607, 0.381409152890155, 0.619245249772216, 0.604760217036584, 0.535554668956034, 0.730111646594, 0.376313284606446, 0.121741879892213, 0.36692231862091, 0.81077971515182, 0.897703963322814, 0.346714263903556, 0.0159029917137622, 0.0287833455846664, 0.84964227275214, 0.939559029286177, 0.76270576933756, 0.96580264890206, 0.492717577056568, 0.554943155189496, 0.652237175979007, 0.973129057873863, 0.903729654703445, 0.0250447334129356, 0.574323021634687, 0.72937575122949, 0.151670303860512, 0.0494824244228268, 0.534938684616384, 0.0398680277683846, 0.499979746941666, 0.210334424527367, 0.475990733255014, 0.65404937423048, 0.0489727921781804, 0.858313346216151, 0.0395082669558927, 0.640554824221822, 0.506434541958911, 0.577115911982249, 0.879054663776528, 0.184353447681558, 0.201457338713147, 0.131960453296543, 0.822206764594548, 0.735343001439119, 0.893487566440367, 0.159684880497028, 0.422291978982678, 0.628907730951406, 0.596781311000286, 0.295739468563511, 0.782024018012094, 0.381772234271775, 0.984771192204898, 0.205127979839682, 0.77975029495413, 0.84501980178482, 0.461285128551582, 0.679125456508723, 0.79715488074814, 0.235574809183403, 0.752229718246138, 0.92006216822474, 0.994638399943021, 0.882963072986794, 0.503401905307184, 0.00588020711096782, 0.0150863855566561, 0.0959368360316104, 0.78696019161028, 0.553592593049065, 0.719519136124452, 0.228478481601982, 0.0126918976983572, 0.12022384960201, 0.632978247387977, 0.0152107986023149, 0.940265801274802, 0.999565802166347, 0.378218577952513, 0.895205829016823, 0.590050046123053, 0.835055668009556, 0.482345893753727, 0.0594410432308502, 0.460336271903498, 0.318710525919819, 0.91016044841047, 0.897861138138131, 0.65116808861297, 0.246053122675982, 0.234024543907626, 0.54843499481427, 0.392823359037477, 0.754794783427574, 0.392572750112804, 0.265420538102781, 0.346891225012417, 0.934262643233912, 0.373853220283368, 0.482526458351767, 0.238214604202081, 0.531249059103078, 0.210121743903408, 0.295207062722193, 0.531985952222649, 0.507584300893617, 0.499120506977046, 0.911497518362777, 0.730343407679337, 0.899712180977677, 0.804924830340285, 0.447934202414055, 0.235005737553724, 0.0671237238004009, 0.284198759745919, 0.431737927905247, 0.455124342543594, 0.979351878330393, 0.926905645123287, 0.597573739079201, 0.583679856087246, 0.113072056351238, 0.399199706004797, 0.942879257500447, 0.214324660868373, 0.170452381814166, 0.346884624122856, 0.620189260101124, 0.435735285074457, 0.415828070838776, 0.285196267926401, 0.905089493578955, 0.682777843244689, 0.104263061128232, 0.0834784966143705, 0.0133939880345498, 0.918761576653231, 0.367572926123221, 0.422024984000635, 0.575709576138501, 0.186023054801502, 0.0222263573579385, 0.658000558975091, 0.458649359616453, 0.199988812121715, 0.196519526469822, 0.693574123660097, 0.747417150478947, 0.543334284518511, 0.525856285931273, 0.406714416404093, 0.556972314937607, 0.730630899175558, 0.293511843781448, 0.784555119832447, 0.0758188793319802, 0.471680463239192, 0.399156110579927, 0.911111718012147, 0.428022249125834, 0.945084139664324, 0.40452928424998, 0.242496664977483, 0.872619211499858, 0.493988294187869, 0.423573848480531, 0.552366461018615, 0.248568987407723, 0.257158897598938, 0.209735425253366, 0.580623328857634, 0.835150872155204, 0.85265999738682, 0.643897083290837, 0.985007765368291, 0.978981934079659, 0.682626155323271, 0.785477746971993, 0.348797903953764, 0.780514693239935, 0.0463973383023351, 0.352832639734651, 0.356997824976063, 0.129735038436151, 0.982246278024004, 0.415753843769049, 0.406485265840144, 0.782690731191032, 0.433204076200704, 0.576278133584309, 0.650468332085168, 0.0330107895199593, 0.404011686703009, 0.958944237104952, 0.595519789208454, 0.933078447206818, 0.937040801805417, 0.755739132102865, 0.538914185396016, 0.824291331591074, 0.0107468467941946, 0.7465189612793, 0.103034771475569, 0.0484107839062488, 0.185691118816932, 0.472839709594364, 0.00963562412951458, 0.858180487504108, 0.473489526284177, 0.14993475024806, 0.245454645633844, 0.0567911434729612, 0.205244797760049, 0.465489139189281, 0.460975178172688, 0.35105607458574, 0.441002771923841, 0.329407034078215, 0.572097266663225, 0.992353294257249, 0.71785082472587, 0.360995549494396, 0.572728531086695, 0.380334616231601, 0.372083265954268, 0.424274644079213, 0.743126890424134, 0.360080485980249, 0.62648925259987, 0.852869697994059, 0.120024751155449, 0.174342661023487, 0.950627797394873, 0.977501474464618, 0.252004875020007, 0.284620466338052, 0.838286322237171, 0.80210074064665, 0.369961325273633, 0.362810980898383, 0.27174128036864, 0.259469306820017, 0.994394376478291, 0.0462633215190076, 0.676333122761173, 0.501684647694468, 0.113288004524716, 0.461314661008764, 0.718630508731383, 0.960602806974003, 0.528900107168757, 0.935416699007273, 0.989586946718148, 0.465715874257281, 0.363796463511712, 0.0903358580397309, 0.844203757712776, 0.413706063980533, 0.757358446322995, 0.554033391718794, 0.476240269518994, 0.266590463475962, 0.135277473048135, 0.0777007824223893, 0.84659344810183, 0.33704296767451, 0.673993808562364, 0.104432752720544, 0.587365871172312, 0.114802921597286, 0.249345347125335, 0.414172719482726, 0.715466741055474, 0.869670580513205, 0.815427904750593, 0.232449226704971, 0.346335232103068, 0.960612364004653, 0.301312799865969, 0.154196271404013, 0.744436488973226, 0.282894819213423, 0.328839434106386, 0.00133189932426603, 0.563913502706923, 0.444133666472318, 0.161362972675659, 0.703462073076674, 0.544031082091745, 0.254619721043888, 0.393110846323697, 0.24461908835165, 0.5826889888498, 0.0527434728531885, 0.602292674646767, 0.0399708089586055, 0.862410181147088, 0.343109861284126, 0.112233466853019, 0.593426579335789, 0.630076743626349, 0.781380428621867, 0.736514128965626, 0.791597149593318, 0.648711337590653, 0.911908320418669, 0.606076094051409, 0.736445592397589, 0.713251525249648, 0.68998719351215, 0.919869871404748, 0.969545932645271, 0.328926071221176, 0.0948157350606309, 0.806532724290416, 0.682734801821412, 0.173246974946626, 0.481009947957728, 0.939338496208182, 0.138072353146462, 0.65965397071256, 0.19403925416876, 0.626483977408436, 0.620735066746049, 0.440977182290603, 0.658947333435875, 0.447152214920952, 0.851604399005066, 0.707325167402407, 0.428546091239153, 0.620197007379653, 0.641192310276468, 0.887750113999928, 0.993604468875572, 0.728353802970809, 0.505822776802076, 0.0489289775130024, 0.108218636820373, 0.858296316129758, 0.915944485928171, 0.671659842949523, 0.462335572822978, 0.961389729456933, 0.0799172426769383, 0.385913478373219, 0.0164362100358881, 0.591907149991785, 0.568236306327787, 0.0122706905972162, 0.589805309946718, 0.360216531887583, 0.0254346707847339, 0.978579761238208, 0.132194080589215, 0.550576564942783, 0.863438087457926, 0.682540688819793, 0.0349691968296445, 0.440197876154164, 0.439827126578504, 0.766111721674601, 0.999839285991336, 0.257465988333992, 0.885814320863779, 0.0497215710338997, 0.331655770309879, 0.729553684632621, 0.498701773778971, 0.546659594797732, 0.224639361335514, 0.122980666409248, 0.485211990253694, 0.874042291689743, 0.64142525070139, 0.448237264729561, 0.137143753746977, 0.14115926264733, 0.982595787707881, 0.971884259280109, 0.428892547533141, 0.756038510062081, 0.558590649492675, 0.756654394054833, 0.975920093113044, 0.041874048910745, 0.351003499456694, 0.476647962674611, 0.174935331391048, 0.355904543428072, 0.888469466735866, 0.0691530491899658, 0.0927474343382961, 0.205804181169517, 0.615198979773537, 0.273432050517857, 0.747785372301515, 0.305726330316065, 0.76544436082854, 0.25288573138143, 0.67101597540585, 0.131914107685128, 0.942448397517158, 0.444583183003789, 0.357995338956069, 0.169800958046254, 0.499074065297007, 0.452166200378947, 0.0662159801850896, 0.887881674640202, 0.907784125901346, 0.782654328509083, 0.471943559212895, 0.796142303750967, 0.87564738845527, 0.672251562170698, 0.070382735692499, 0.12447190516367, 0.425366183349375, 0.782665768727162, 0.589146545218194, 0.472135906386569, 0.0337884713120837, 0.180073535625725, 0.931819658789012, 0.590764009202279, 0.506335507400454, 0.218438856708936, 0.952051035940071, 0.362259513867018, 0.45854675470317, 0.830810137981569, 0.790665029293851, 0.441022733276423, 0.624033793781809, 0.955508256317789, 0.663478658469131, 0.266490943364355, 0.723154840215161, 0.561077886344455, 0.972980667353596, 0.300872644671042, 0.497020498287351, 0.283954108418013, 0.0742080047906911, 0.111397187358409, 0.0753069359705839, 0.996192027605659, 0.599597679942704, 0.788968013382453, 0.861268639611926, 0.827068266282738, 0.936923731937818, 0.0358534605987286, 0.139710786828577, 0.725442080874176, 0.872094169044431, 0.374103271527148, 0.216187517237589, 0.57601009917298, 0.025646755078704, 0.0435049245994346, 0.933449284045686, 0.401651928267034, 0.784920768198348, 0.242628568324623, 0.608542724961418, 0.893183909504416, 0.629419861274986, 0.924146871351501, 0.99580457170105, 0.15008987270811, 0.0247968997642114, 0.911264097869455, 0.14799745847627, 0.0899430572891483, 0.715693599151247, 0.1750144087214, 0.159721590176509, 0.478932143833692, 0.948731651592942, 0.7410701414584, 0.672333462130722, 0.0703452477692121, 0.306105485335777, 0.936833055516247, 0.856937003671443, 0.890847380737143, 0.588586918837613, 0.162586327547285, 0.176045119807099, 0.747563614665687, 0.945285815976482, 0.657979789002505, 0.543831976949375, 0.597653088285055, 0.234602892008367, 0.819767731385293, 0.788495533288682, 0.589059613176341, 0.436259778173497, 0.549739427769825, 0.352968705935504, 0.99579119352887, 0.364963600015756, 0.966174491517511, 0.0247408729527182, 0.777839114133281, 0.440182119351492, 0.630499928301014, 0.31402469620271, 0.921815284507743, 0.223324620519386, 0.37448627185821, 0.308144382373381, 0.0378224741229389, 0.605169674511657, 0.835864369359658, 0.588252232813568, 0.363454795408988, 0.139885183619558, 0.116533721537474, 0.981217434069437, 0.541692349030371, 0.0865099084385226, 0.256096227788964, 0.209775010836282, 0.314609803604705, 0.903751596102294, 0.520696307127195, 0.162236010012221, 0.14361220751327, 0.0640530754780911, 0.854645458623604, 0.974456785593373, 0.929025103554582, 0.409520221274419, 0.901611220034849, 0.484497228782305, 0.469673840237391, 0.16782819888636, 0.082858473102399, 0.633161255034096, 0.310978917444454, 0.466354034749258, 0.211069421812716, 0.0014457196440969, 0.797611265022929, 0.368047426089151, 0.0151375722100053, 0.714978466470137, 0.218578762078392, 0.688717269625175, 0.246007221266387, 0.266084854588675, 0.132070823774348, 0.344210335433149, 0.754389983788448, 0.889238646634628, 0.710436376647841, 0.246475025240484, 0.451389348495065, 0.931224274284887, 0.615667580201693, 0.258289980493455, 0.631103174057245, 0.695532194169074, 0.403659878091701, 0.193481857413117, 0.307074224713182, 0.692954513388275, 0.630359565476236, 0.249823988840664, 0.0434220177335936, 0.53145145361482, 0.176201447978594, 0.521902037473046, 0.871282135692825, 0.613612826084653, 0.271432121466628, 0.854539610220498, 0.157460139605902, 0.259418638129986, 0.670580480612924, 0.225957392866863, 0.825966529550158, 0.32409628231535, 0.488386575826106, 0.763723753369187, 0.737675502332654, 0.636025946217411, 0.249140939135425, 0.734316827783819, 0.339505137924502, 0.30276783839018, 0.127141988983118, 0.0101402987261833, 0.924928777054715, 0.910006195626544, 0.382147912218909, 0.701392065887907, 0.672341929094608, 0.127674130158185, 0.200969038593963, 0.902826543553463, 0.891836308297259, 0.377681734866925, 0.250192901317445, 0.602701480231355, 0.625450545463954, 0.0196828894788748, 0.819732602404028, 0.517944571318759, 0.909434244169669, 0.118624967385146, 0.946149041256063, 0.927044817717209, 0.0948050112133592, 0.41525738914511, 0.322862316746869, 0.846253559373736, 0.947950319602612, 0.729220577434225, 0.6838497247755, 0.641411905643839, 0.315882600559583, 0.732146773481898, 0.157648409491942, 0.760070691228337, 0.0919099031730741, 0.96765399195703, 0.608966348346455, 0.894952440275417, 0.194146530723079, 0.988057942048401, 0.24131486157395, 0.223420955768555, 0.40199794178472, 0.780906365500356, 0.83921201293691, 0.409499774557134, 0.402400532579752, 0.771750033223037, 0.249124686312797, 0.865741064645646, 0.682186066810687, 0.97739298040856, 0.185523905543996, 0.216182760513138, 0.312921858193491, 0.75201203772573, 0.367736606860203, 0.604456003539142, 0.383434891407544, 0.550487829259962, 0.385442421714884, 0.665435033273495, 0.014714163546339, 0.813124468759885, 0.349855974829203, 0.715515525264852, 0.958893803878619, 0.507417120718267, 0.666168220267888, 0.429465434362195, 0.849498543568243, 0.654233377284434, 0.504257747592193, 0.368516059530713, 0.685249429019409, 0.0992075151757565, 0.0476468369667534, 0.86933591497735, 0.651867365863888, 0.610458701205885, 0.421945705248746, 0.503707374957771, 0.164140923375798, 0.0931721295240463, 0.685324864521948, 0.142671298086907, 0.528711653269026, 0.00357277332427541, 0.425057790934981, 0.22434090517641, 0.659942132313487, 0.503101896379671, 0.600057239807837, 0.200173658860432, 0.137401018724244, 0.0237890286343844, 0.939542698220884, 0.695501323790041, 0.554233183009153, 0.295414289119064, 0.0426317475056841, 0.853760208334462, 0.596682100841419, 0.0185267796429373, 0.93319096187148, 0.00978832426687193, 0.640955539628156, 0.788685383549152, 0.52650294423104, 0.291248417281234, 0.628328540883956, 0.425468671551764, 0.546370363272546, 0.190797083318664, 0.744488837009526, 0.212856219278171, 0.100122455877196, 0.896384986297513, 0.822336975387737, 0.882783750050209, 0.0881242954121649, 0.700187647740891, 0.170224223729873, 0.83304427105305, 0.728057084721778, 0.775893015135978, 0.763495251449153, 0.0570994655747198, 0.609847721618856, 0.898471800448384, 0.516130724509215, 0.506171538067419, 0.255616556410658, 0.971603134958536, 0.581123922450753, 0.968386414336657, 0.0414383729208978, 0.188628229362153, 0.519760059068036, 0.640853919662848, 0.340731322525365, 0.648329819207684, 0.355437965546251, 0.850430024041377, 0.542804681241221, 0.67308008746076, 0.256572579313377, 0.121338276707935, 0.0406064458603407, 0.887943133089476, 0.753187315831733, 0.802079185026265, 0.441405626647267, 0.0751725857162204, 0.154229697287694, 0.247244148394022, 0.0792259868541022, 0.853159563134165, 0.15717685529079, 0.616662683688958, 0.528449944458942, 0.191219823355632, 0.759750760227389, 0.614998425492876, 0.152063606777082, 0.454637446858156, 0.682097599808961, 0.281259121575909, 0.624863120814537, 0.941730122601418, 0.566803336047469, 0.693115599105178, 0.978586411034322, 0.899117378675088, 0.64847989601533, 0.982920814944961, 0.163814968179811, 0.509786857227575, 0.627348429656104, 0.800000906333289, 0.417769086107256, 0.0548344318937808, 0.519144193650956, 0.134532135688309, 0.36697443464504, 0.69662088740343, 0.129583524717075, 0.785849529987449, 0.0503089735963517, 0.815615519700691, 0.866938862459353, 0.196896453653625, 0.607393801126777, 0.996679751886203, 0.0423743274600825, 0.138823003129407, 0.222953974091225, 0.58301047796428, 0.671106486855616, 0.723195985118222, 0.50443851298613, 0.644630262831149, 0.0363693406760421, 0.193851825334961, 0.749545712102824, 0.308178375844892, 0.749052151824695, 0.85203732135201, 0.0993554763827961, 0.597778198787964, 0.720602255764323, 0.568713832115472, 0.23511009108531, 0.910636226408683, 0.950619757095401, 0.516315294819808, 0.707088130257172, 0.798795192940303, 0.199689366652775, 0.106840515449705, 0.618574843315811, 0.0222239407823022, 0.0375620436883189, 0.115364121588021, 0.818697240446272, 0.376899225803119, 0.22915631213616, 0.3681399130369, 0.174540423871336, 0.774779638832738, 0.330030614354014, 0.31578013992846, 0.724586900626482, 0.584954983442022, 0.777755524267169, 0.359576387650632, 0.184733001916479, 0.573689108128662, 0.483094584564781, 0.421626940454074, 0.317325192381402, 0.972571507193181, 0.258222640836992, 0.567099899646433, 0.352066904740983, 0.312029608919655, 0.00939760275739143, 0.668127492035812, 0.885687680497191, 0.421317726471668, 0.858580052734657, 0.947992491692748, 0.376106774797058, 0.203293248489839, 0.179169495241401, 0.30629408334121, 0.850311935666138, 0.525602531044741, 0.828442251880212, 0.106876458873914, 0.617142031700498, 0.566826518566431, 0.260936156906943, 0.00230811043293844, 0.287255062879815, 0.821270082456773, 0.895115404680466, 0.99580790960623, 0.00843840180703381, 0.732843238440102, 0.0846172679425323, 0.633348568099204, 0.80508642560858, 0.829964886435069, 0.567221449723939, 0.260043390493052, 0.784532049401665, 0.169205334021096, 0.274258345175233, 0.00490319428856278, 0.397561240528039, 0.0265515532308013, 0.854699049809875, 0.204891689163361, 0.860350581097684, 0.0561247088055489, 0.223511772366391, 0.0452805178083929, 0.750991773112615, 0.698056379930209, 0.918787889121138, 0.908922604332158, 0.805523856544559, 0.999677445374649, 0.800766275754545, 0.32291533264413, 0.0251445841444535, 0.732823471283791, 0.472786704548189, 0.392588402165001, 0.860014027813345, 0.0574616491206149, 0.384545138525541, 0.141840420366856, 0.0456813977750177, 0.697994732962103, 0.913511502068192, 0.383545632290609, 0.243110817299358, 0.907097732647764, 0.702017912868084, 0.895637886893663, 0.947973559242691, 0.936691958432421, 0.996786631011984, 0.396942471901925, 0.659936449375974, 0.822950297949471, 0.622366730006282, 0.700506318144961, 0.489515104713947, 0.508816801240717, 0.779713295559863, 0.0404857761628601, 0.901007402566114, 0.732045721121022, 0.615783068479236, 0.755188044469967, 0.284467185594173, 0.206249114576936, 0.652186212909185, 0.928119937980734, 0.80457351484991, 0.350436063102228, 0.270188405995597, 0.744617713123743, 0.72299211004354, 0.856720355874117, 0.918819323643678, 0.269008369165018, 0.688151779362003, 0.617997051690114, 0.0715534729394275, 0.188398234988831, 0.335569468860859, 0.311045820959137, 0.420204017960576, 0.0877387990616133, 0.55986848290701, 0.667625249862375, 0.115301713639157, 0.31566038919899, 0.980796342397318, 0.885869537711305, 0.0227583294274912, 0.863285505241343, 0.244993620883412, 0.43360172923796, 0.135376201663307, 0.215599998231807, 0.774728915108001, 0.0871862609352515, 0.400377825330332, 0.590110967553579, 0.196523569072947, 0.696812464244864, 0.592969314926699, 0.464189975726267, 0.24115494785945, 0.412770454503875, 0.770494470747454, 0.0353388916446334, 0.607550299133461, 0.079473648898011, 0.415376455043999, 0.985477576209605, 0.842648435276967, 0.0524026021080585, 0.72751791083111, 0.114958029745548, 0.485380961589241, 0.868594367822937, 0.0167038791136971, 0.818908378037653, 0.783873672409838, 0.193707732877488, 0.912844291315192, 0.683689941661077, 0.888393508527682, 0.712869704398642, 0.484892021541747, 0.833218042660154, 0.0907464207730675, 0.0254917565053651, 0.335727870856154, 0.486922626333375, 0.225362827294348, 0.554708799489516, 0.681148392757953, 0.834943208419999, 0.895872813222938, 0.437597918328548, 0.399655288850365, 0.440119954134542, 0.309626357593902, 0.446953372036611, 0.964390123247796, 0.462348232342065, 0.495992012385964, 0.0215398927902066, 0.192285131466523, 0.898134260529596, 0.389292223420651, 0.10204615407611, 0.349698581677999, 0.178546091108624, 0.249091062376147, 0.640984449884642, 0.975754160166402, 0.920663851687385, 0.0538069285001792, 0.903084679011044, 0.189999378694629, 0.992615197724179, 0.326499595591164, 0.22754349577409, 0.943710729255865, 0.810846874136766, 0.924444160009448, 0.718502000394367, 0.879256561257223, 0.865061767470056, 0.357425088447911, 0.136276468350636, 0.80578040932757, 0.681124846687269, 0.431387116373183, 0.947115419019467, 0.951172736997583, 0.0320806073181996, 0.276166759932479, 0.0677530131328091, 0.826949301723875, 0.396323445233256, 0.581176711612284, 0.343533779992188, 0.971209743582416, 0.603405805647498, 0.667641149962883, 0.863325161148393, 0.420087048424167, 0.0734110022083088, 0.871397058169332, 0.694654243500936, 0.841424689426169, 0.676245834592188, 0.701401708733773, 0.89541721719206, 0.587413929825187, 0.325195362589217, 0.994850645679252, 0.207538426637717, 0.286252616218108, 0.833108338605399, 0.77024774743615, 0.200579107720145, 0.764766296571413, 0.615923392714844, 0.964267415983719, 0.503691258942202, 0.478701822787988, 0.585696853852641, 0.979591726133051, 0.349099088670411, 0.261002756873785, 0.589942415365823, 0.745384843673345, 0.0819839042722866, 0.374955838803374, 0.0870084443419643, 0.318151405481005, 0.453229945577407, 0.161080905246452, 0.059868148696068, 0.444865217813094, 0.28527376756492, 0.199178223249176, 0.509811895523361, 0.0194784330073147, 0.830102595087511, 0.200385563170072, 0.426563226284877, 0.733088493416766, 0.526796772931903, 0.269298511266955, 0.76783522775038, 0.954412473841177, 0.117687371272035, 0.0248882200910384, 0.422040114555194, 0.0643839604461038, 0.125602655854972, 0.0924177538405936, 0.23268927348069, 0.211193061163408, 0.609810980500931, 0.394527052707534, 0.726698391885944, 0.961298153849644, 0.750217204340688, 0.0787397728796151, 0.867953051118049, 0.362770863871237, 0.797007127192494, 0.786387187901702, 0.33135474760017, 0.949991455318108, 0.828489643488458, 0.629129747005007, 0.29562363193007, 0.617004524184042, 0.988503430641176, 0.344008892061286, 0.239644666247618, 0.000397105906740247, 0.909289852299736, 0.627540457253528, 0.786229298280286, 0.748716336003762, 0.616897634263607, 0.183401581196979, 0.843593247070369, 0.715992939449676, 0.651801335577001, 0.612807071592845, 0.873786137226238, 0.792677271821998, 0.72442497596767, 0.409154429262935, 0.997653402297484, 0.62380850361555, 0.353165253585684, 0.182723141204326, 0.320904351338262, 0.255606766451051, 0.941084747291136, 0.892906675560223, 0.988052107960338, 0.952471975063841, 0.87420691763809, 0.162243500732745, 0.146688413160472, 0.649803100368284, 0.790085941142052, 0.0723016092073964, 0.244519440368239, 0.0107071520515589, 0.232541408581099, 0.25485487900282, 0.912187961859587, 0.367546572759846, 0.0380340235252129, 0.65083231658774, 0.816366863596328, 0.0837504080334276, 0.0114810473794458, 0.365615322319801, 0.412765442193972, 0.416020989958074, 0.463376014974758, 0.46600651254965, 0.548158532351233, 0.532690921787102, 0.507692554074212, 0.78109132802197, 0.289965147981413, 0.550501881245131, 0.574665330623441, 0.203891508109562, 0.555236972412542, 0.214116023895196, 0.549156722038551, 0.0697698299188489, 0.860447514482123, 0.208382707321565, 0.238632095670492, 0.380585215278611, 0.287366391786694, 0.750523129421709, 0.318938971250355, 0.107369675336952, 0.806135868336565, 0.252058454470901, 0.503904852055424, 0.0398508387910902, 0.624193754211487, 0.38744160768852, 0.308273849685748, 0.332049962380999, 0.931593246376367, 0.467989560310096, 7.09931934543828e-05, 0.725780504048086, 0.399482919273069, 0.0229236339733638, 0.637742389070658, 0.439352012713857, 0.59176913638089, 0.617984610357016, 0.454096672485187, 0.0490852758965976, 0.719694172744749, 0.740092862384778, 0.404798794595577, 0.0514125253940096, 0.948004653343517, 0.877386850235409, 0.827649947688969, 0.448636593326791, 0.884474714194616, 0.853718337359078, 0.647736744115042, 0.506783267055629, 0.0760229085454986, 0.362406884500106, 0.231091155805152, 0.860180503723948, 0.634390055837083, 0.731457114777647, 0.156355283373281, 0.949646937912652, 0.400422953772862, 0.611386675351746, 0.798489232869671, 0.686369819520236, 4.8344557509572e-05, 0.502657654353642, 0.98561999575393, 0.00700372094719215, 0.0409293243773717, 0.103422911335195, 0.603406558356747, 0.046190286326236, 0.421871944821509, 0.355863103245393, 0.697670457057168, 0.362977725714622, 0.324555047630042, 0.677475743153821, 0.583252369854356, 0.24045479726547, 0.374866566332158, 0.20473461270147, 0.258642084677665, 0.847314492491916, 0.427040942379921, 0.656982963005712, 0.531271748039302, 0.937019040745027, 0.333422420094046, 0.360911832253805, 0.746493540455523, 0.863929468768405, 0.0220537699344427, 0.849523927097227, 0.807625427184444, 0.542782945458753, 0.433783074730281, 0.602436919119053, 0.335105203033884, 0.778107204749409, 0.0967197534128158, 0.451954974000803, 0.761949065427891, 0.625833233228555, 0.00149152325565227, 0.0171724051507521, 0.456854072168927, 0.221564056395325, 0.371420685802539, 0.441232164258044, 0.0606895470210004, 0.364042374137165, 0.376869074010979, 0.301240237258343, 0.918389180148255, 0.477350675771568, 0.180392438029658, 0.297037639462726, 0.82929947365432, 0.618123992904246, 0.663599541511285, 0.616203411816305, 0.26126672437812, 0.872498716650101, 0.00536812740368875, 0.365255291862225, 0.591971838400334, 0.924839059321826, 0.458004897587719, 0.424208224297733, 0.654961612063972, 0.764236308413968, 0.379643566413889, 0.454068967068252, 0.643802012293806, 0.735154526338052, 0.633821562872391, 0.782944271864228, 0.0344870193209459, 0.100862339158675, 0.0417407787581285, 0.160518423407093, 0.835002947100983, 0.559434082461648, 0.172021699825831, 0.938799009118024, 0.542159328902428, 0.830257856136279, 0.317443770333258, 0.232739356450114, 0.70199381249533, 0.519783648059401, 0.286146291582106, 0.0276729267053365, 0.152802902265912, 0.783922609710904, 0.518751889795213, 0.416822862672221, 0.108547556984511, 0.631427675090401, 0.126733568056866, 0.580643663513365, 0.184500375545941, 0.494911090203233, 0.335440618316172, 0.2068487698259, 0.105165521104586, 0.153632517968856, 0.873329549947576, 0.724371498334733, 0.00358417601612571, 0.810415238012061, 0.245622407967823, 0.885936548815393, 0.839280649475202, 0.355567385189081, 0.315247116832202, 0.154186308339116, 0.388423229604318, 0.305439848198947, 0.82117581307302, 0.279882346584207, 0.851453947760294, 0.195285155964464, 0.355002466794182, 0.345975581098889, 0.585631351305979, 0.456250795406776, 0.0384487116383723, 0.380015594346897, 0.546487045006288, 0.0682580666253524, 0.213226465005379, 0.771897567107188, 0.630379369335341, 0.808721360720695, 0.0214900082320982, 0.0117568878880779, 0.398796665192688, 0.209695674834458, 0.587914740177718, 0.822721895086327, 0.749075982358463, 0.929144914732334, 0.524545534321685, 0.538645416634196, 0.667294499380659, 0.871261196741571, 0.181324929181593, 0.699106624590959, 0.188910203261924, 0.199816212424685, 0.770965949106791, 0.529896188205266, 0.00471463719527176, 0.955361050072058, 0.877279000485839, 0.872308896383523, 0.494025234903589, 0.531101692692875, 0.0322607179908051, 0.750156708319586, 0.758352012919249, 0.583562739006172, 0.370117008266112, 0.491999937510986, 0.527182458093841, 0.448724465375686, 0.142400905148274, 0.652743732329403, 0.674670288382137, 0.749762774891718, 0.3059258561246, 0.804347592128608, 0.299966798532055, 0.274086675032318, 0.270484982062664, 0.763490929279078, 0.31857521544177, 0.357004448273479, 0.675481877125386, 0.0335382250580746, 0.150022544057828, 0.421550122095638, 0.212894144789799, 0.760072124538572, 0.643306136555478, 0.839646713945293, 0.207795764772368, 0.43834759272771, 0.732119751641718, 0.104751718583339, 0.444564714157358, 0.490552705674478, 0.694426353284342, 0.930807985481853, 0.155150775370797, 0.0526812206681669, 0.608676453040973, 0.242940784548139, 0.558302196431607, 0.678500674432211, 0.466504768523564, 0.658405065829952, 0.808349338923948, 0.856439235517049, 0.710877914024559, 0.263333615762483, 0.296107240124455, 0.185845249420964, 0.599376706569911, 0.778297442119277, 0.0957662280920685, 0.409976448842909, 0.631245711201028, 0.765313364891196, 0.515606699960211, 0.0467893532856348, 0.440639621831668, 0.781269083828604, 0.99114597391613, 0.849279003114081, 0.0599863054427622, 0.766725198143708, 0.724906562569402, 0.754682907768219, 0.201431457169383, 0.323838572852246, 0.597078842366368, 0.977113562894548, 0.451428339736858, 0.496305402839853, 0.971656138539336, 0.76300459210049, 0.979765333348377, 0.20283133878472, 0.031668613261674, 0.158148028695235, 0.719703401884927, 0.77728236171796, 0.644957778367517, 0.338064658578324, 0.654652328121952, 0.0456563493426857, 0.867484780088114, 0.102961749488117, 0.946969335700421, 0.379497748848172, 0.13452089600363, 0.312879832834512, 0.898684091136074, 0.780790397082025, 0.193041323864976, 0.173940827932019, 0.893316219824227, 0.77076450757436, 0.792116942050586, 0.213557852512077, 0.193372676209104, 0.646739082572099, 0.111286936957793, 0.505912520725804, 0.684921572433945, 0.15702756821061, 0.373374636300056, 0.808372197161161, 0.909321931243242, 0.160689536484079, 0.737947882076883, 0.555695633321277, 0.187856596891467, 0.690014354052756, 0.299816313121061, 0.632268077566992, 0.235783501434267, 0.489343100836104, 0.758264216642305, 0.620207419584395, 0.247775280433747, 0.653207917679531, 0.330785884081767, 0.119203161145104, 0.656856203096871, 0.947319888498822, 0.841154337357591, 0.146388071778344, 0.687492784344319, 0.664213083872843, 0.510489183734226, 0.434721893205686, 0.383588263811121, 0.842294234465626, 0.250835018159834, 0.291479801013163, 0.56096884927098, 0.626663690335705, 0.449719456150245, 0.864425322492111, 0.953957119592282, 0.816527341866468, 0.72006311209402, 0.12204180665446, 0.0829542278320125, 0.83912333598996, 0.350551199871815, 0.90563695785518, 0.577891399433266, 0.0510865325536103, 0.215654074024877, 0.146684565459999, 0.416301340868488, 0.877558291488561, 0.0929587056682385, 0.929844942371153, 0.935837717518479, 0.265357934875375, 0.0356788697102947, 0.466467218379429, 0.584379997191826, 0.726742793899485, 0.333494369098737, 0.0636822667271657, 0.891637497750506, 0.828962582350315, 0.372949582640135, 0.76546316699376, 0.437400961066013, 0.5182717138718, 0.56018892690641, 0.245505881565662, 0.128305288765861, 0.610483289226905, 0.613567706978691, 0.23493130398307, 0.857609951030176, 0.115304673120862, 0.882515016894715, 0.388909110536336, 0.299822438021288, 0.752914143117466, 0.265169501451006, 0.525548324539743, 0.229234717720593, 0.0563855511862896, 0.827781051935489, 0.115814113693472, 0.541923001030613, 0.624683965809759, 0.000375822189393915, 0.163615120180612, 0.0095165645827977, 0.753001454748553, 0.703709754708172, 0.356341060780136, 0.670412114754383, 0.0909950542389956, 0.822654990430608, 0.423219974167238, 0.101588188651332, 0.0975893558289755, 0.106357329408205, 0.61869887641475, 0.20098735192962, 0.423198035803532, 0.809022153547126, 0.328916472471804, 0.273349826435365, 0.818828811199619, 0.206466392760056, 0.654566847241784, 0.576013077936972, 0.563712767112517, 0.314677486132306, 0.145663820743122, 0.130216847110491, 0.84266670699527, 0.474611689556818, 0.231697349362769, 0.781042052467342, 0.682424363146733, 0.161218564857837, 0.141193716485023, 0.668742830514843, 0.658732884775842, 0.643805936700005, 0.186787962837116, 0.822098464780699, 0.61374607513041, 0.93222198934305, 0.661416159059786, 0.249203164862746, 0.0466922013939649, 0.0371405370267769, 0.718269243778726, 0.243418098449157, 0.145676316831861, 0.0439551517780215, 0.80883045311085, 0.840723748108901, 0.229517030136812, 0.747839046310075, 0.281946937648841, 0.338251513307904, 0.329206178196116, 0.066634405260419, 0.0771534533856184, 0.494831629552557, 0.376096669400761, 0.416586648781951, 0.619685177791168, 0.66352952873711, 0.635509114573593, 0.340201638270653, 0.744755242786133, 0.695505600258642, 0.586577046055119, 0.882742270506663, 0.957209191810798, 0.860079167276577, 0.320362344215398, 0.663587983731134, 0.905632646992071, 0.68541120640391, 0.999124758436585, 0.499394031705946, 0.272899717104218, 0.693410334231831, 0.995012528480035, 0.210377607215077, 0.415730733172781, 0.162007507128176, 0.249721977107832, 0.393707682257794, 0.340166660641936, 0.928524883667688, 0.452723853743156, 0.526566576652555, 0.501154125125536, 0.810326381468528, 0.804838200476692, 0.117820832310906, 0.618350228920607, 0.772727522655687, 0.284838454859818, 0.269120237835473, 0.466116253270852, 0.371250324835367, 0.479240666788674, 0.438207686734609, 0.255226195863612, 0.141137643640155, 0.266171959539122, 0.348483722205877, 0.25252196501588, 0.017718028432933, 0.450747489082489, 0.904581695652367, 0.657064984368632, 0.143432380984605, 0.258496836585302, 0.791264715699055, 0.727379890178898, 0.823118525431626, 0.936302772522354, 0.191171592008495, 0.52694366070842, 0.280609960608241, 0.408387426794459, 0.239852907545559, 0.35755558177561, 0.164320991306434, 0.00252969447787887, 0.406658027318829, 0.279617668833347, 0.177275520262081, 0.585657977062575, 0.721518188832355, 0.72126091470881, 0.508586119635723, 0.325925829418884, 0.454837897255576, 0.296880598335777, 0.683026854068294, 0.924728143195075, 0.291745652672231, 0.475994235069974, 0.391544341110887, 0.318895884309893, 0.400185827121515, 0.877592539852425, 0.953160377075742, 0.00438343825445786, 0.601428139592099, 0.785577616211084, 0.638506626625535, 0.117014790611095, 0.710512764054586, 0.0977497385813179, 0.584765608453912, 0.847564771084339, 0.230725369552222, 0.195987912666101, 0.0378281575337927, 0.827546700974101, 0.0282616737720139, 0.367238611045286, 0.174811466364454, 0.504073009156875, 0.495986198002481, 0.864881433700093, 0.571139583114852, 0.171015581462748, 0.908353001332888, 0.663637726947259, 0.502688991892321, 0.822084970197406, 0.287696837856296, 0.880965097265547, 0.531438138592796, 0.0925936437936093, 0.483544266284841, 0.321797152353726, 0.543080646573422, 0.95670387506052, 0.367524177600231, 0.0264606839837801, 0.739510538613349, 0.324884968076628, 0.665286534428497, 0.497846294918713, 0.857685172576209, 0.748631965670332, 0.397915820333789, 0.514114355258412, 0.583756327905349, 0.508243785545476, 0.733870809705105, 0.728590066738009, 0.194129736213402, 0.454892682079344, 0.51861630533123, 0.149085761067097, 0.781939941512405, 0.165432979536329, 0.495223961095246, 0.259175029738592, 0.621758736974696, 0.457111774492219, 0.255455791118278, 0.534400622572249, 0.0916214964714942, 0.561532984356566, 0.852199876924213, 0.727490602774095, 0.538697511160083, 0.671617075904606, 0.860682517484911, 0.55192771301879, 0.910683115282866, 0.556749508305831, 0.858337826159205, 0.452762805574304, 0.160615906297805, 0.237215093794534, 0.611743375274072, 0.433721706699308, 0.607155314894792, 0.236655835135741, 0.933524933977363, 0.703408954785415, 0.67563525774412, 0.393635814325175, 0.647503736778851, 0.449255150534191, 0.989971796830069, 0.0936148127858498, 0.468374052168575, 0.693682230951478, 0.941818199036467, 0.631511658819372, 0.617115987180577, 0.643528901806731, 0.151532736805843, 0.244668226240985, 0.550142288677071, 0.241307617888434, 0.504063842083072, 0.801251837979967, 0.66351503821258, 0.0367733645148635, 0.208029770187126, 0.403123150965031, 0.918238226617969, 0.607991386824477, 0.21829997476751, 0.348915500369601, 0.267919380367911, 0.385385927547361, 0.87827511374199, 0.555024776237483, 0.324908066104846, 0.64853574184727, 0.610403619390566, 0.0889626291430494, 0.366805487372507, 0.939952424962599, 0.92281351256635, 0.26085728854189, 0.264573291552954, 0.780330797204811, 0.401907791609731, 0.88645591662546, 0.452187063976591, 0.496988526732737, 0.372961866068886, 0.381709541908587, 0.458085650913121, 0.717784945764994, 0.729951268495423, 0.336132063162444, 0.464297946037283, 0.377705712616505, 0.0639498248076031, 0.982701953767416, 0.596293182189395, 0.450136834000043, 0.972010588597495, 0.688654393677232, 0.487983368736516, 0.357346460841731, 0.43148794518131, 0.206194770704721, 0.544849909479693, 0.957502087527355, 0.187005169668815, 0.77020479533809, 0.837380501044341, 0.908313740355502, 0.630663070676491, 0.593594448925006, 0.772602644287646, 0.273068648621246, 0.00823761504309184, 0.268418804699654, 0.53385463919365, 0.225141131470043, 0.912491011918331, 0.112738541032868, 0.26634204321073, 0.479796067595697, 0.660934322904922, 0.835947091311739, 0.634712730305659, 0.806840616944945, 0.572420336409728, 0.491711306913113, 0.801513667317031, 0.44105140049689, 0.0488697271856416, 0.987460290905757, 0.749249478345187, 0.578644771837704, 0.33416658827205, 0.593351992917835, 0.901644291983509, 0.842945140317962, 0.868416133921232, 0.103637198398485, 0.787310609496386, 0.405960894652325, 0.306966845680945, 0.0265203245765036, 0.710455875645696, 0.843627569565131, 0.0498878052822397, 0.245631403055111, 0.950156813805664, 0.110080384100564, 0.957974988272248, 0.345386033657317, 0.880448196157124, 0.276029507490185, 0.714448536660363, 0.0497157447490537, 0.107253562155779, 0.858990753529828, 0.424279468111771, 0.562525685005673, 0.732724594304781, 0.960882982461081, 0.0591349838898697, 0.119566088014961, 0.13080015163655, 0.802720515516512, 0.160170857168564, 0.525530254482209, 0.708592152917387, 0.990396200524505, 0.610435169301708, 0.291582297318709, 0.550880163600681, 0.540710589991237, 0.900516270860716, 0.0709286455369771, 0.983751606032539, 0.905694005063952, 0.938992607052662, 0.671860398254504, 0.726553520816944, 0.0933698254443055, 0.135473401595675, 0.810643081607458, 0.845292595807665, 0.102971861627645, 0.755896661112373, 0.321173753759313, 0.238237567057305, 0.839599928898844, 0.421444094809798, 0.659958771768037, 0.0357786992124387, 0.740803062954404, 0.997566586687326, 0.384147276391293, 0.529256640953674, 0.784164248411179, 0.162200802414689, 0.901419515298764, 0.310288035465942, 0.679309314225763, 0.459185379484139, 0.664184088060521, 0.682898317059236, 0.242328684688291, 0.543729965078926, 0.164229632434299, 0.906588660712387, 0.572722018042565, 0.845280788979512, 0.97570218980676, 0.796367685536646, 0.554124034923461, 0.873106287419445, 0.267553020689022, 0.941065004890873, 0.990263399616074, 0.917370386181948, 0.0372020722220396, 0.585980306873088, 0.42100697573961, 0.0455294782477368, 0.925214309937598, 0.751346074587538, 0.0603985924489863, 0.722134035331643, 0.28633200286264, 0.964473572105753, 0.149330881720764, 0.597203422815951, 0.465168696523358, 0.4113363793156, 0.880488962719551, 0.224888187619255, 0.542962313880256, 0.947602464385372, 0.653340666289388, 0.0340970832670981, 0.388855595651783, 0.873919422743349, 0.388312005234588, 0.746906921092039, 0.741106014749551, 0.229777947378772, 0.92380203511933, 0.172040543259882, 0.830252720218142, 0.410844509671483, 0.770336837149796, 0.340021883022988, 0.89133524160372, 0.855426429970068, 0.131903067629001, 0.682366170566363, 0.149579526894791, 0.410367959408876, 0.549947684939901, 0.551802658800882, 0.772998152902337, 0.740750088787858, 0.907675372637339, 0.463715278605034, 0.400803157178557, 0.94390109594086, 0.0956147848207154, 0.558747559680857, 0.317395317012076, 0.94477065109592, 0.618901958552541, 0.464385849180094, 0.879031535184321, 0.732685206664228, 0.589215431100345, 0.100644356979212, 0.021878280558034, 0.0842969746775835, 0.455152772646203, 0.18831863458545, 0.0751257907582271, 0.329066710409041, 0.227286979231096, 0.337171026888294, 0.552447374053178, 0.968311344447144, 0.227229683748033, 0.400145665597332, 0.990771453184438, 0.462018504124883, 0.984530452312384, 0.498440319587565, 0.988003909679557, 0.538099402281207, 0.132896493392918, 0.339203198158711, 0.729008316104168, 0.415643877703603, 0.924721431759455, 0.0702916464374613, 0.379395562877676, 0.565159017573399, 0.756591786234123, 0.73366556234191, 0.381244895745333, 0.904656414888254, 0.19302065764694, 0.890348095003441, 0.258175695079439, 0.413657129776186, 0.00327934192190638, 0.545836334246207, 0.345007071100708, 0.645588023619165, 0.932185481864626, 0.587708283878026, 0.260603011691282, 0.899945852728827, 0.0172462356909631, 0.227921137348044, 0.716858192704059, 0.0996278537257523, 0.717017521803656, 0.239580553677171, 0.41283890879748, 0.289439868054638, 0.151618258803281, 0.309801640534927, 0.633439302040717, 0.0225924003937511, 0.624159799163395, 0.200188822080674, 0.397056404257779, 0.465830567404375, 0.798357544490119, 0.221004357443454, 0.382385956179222, 0.906787435726044, 0.123876076420517, 0.0450573171495012, 0.689603793919938, 0.298120940307296, 0.996560332046831, 0.183931533594716, 0.555146746375634, 0.591298466506725, 0.499036806057356, 0.462559690946915, 0.152769114952622, 0.258558826701801, 0.722380433394533, 0.996347414340064, 0.555310311717323, 0.643678816727025, 0.68080671506056, 0.952809615586593, 0.366399040866368, 0.469221788901091, 0.128539707924391, 0.0798067021343954, 0.544008124821951, 0.672306958059288, 0.843506947991935, 0.598324752118874, 0.0624011177335974, 0.737645613791579, 0.153585449474272, 0.0851073713166624, 0.0490069903861494, 0.0804031963487781, 0.213290497535409, 0.388538275065089, 0.287982380489148, 0.149711341131578, 0.633817003627647, 0.113690917502882, 0.734481428050498, 0.890022406662325, 0.442910302494869, 0.408086391050608, 0.413853497783082, 0.878701888866586, 0.527216252103076, 0.12774379705434, 0.211536127402121, 0.168247588674638, 0.354724107416145, 0.364558685951674, 0.551676753449916, 0.131341868059563, 0.710387727276987, 0.354987809184588, 0.647080208023738, 0.425283021804849, 0.757335017355644, 0.242964184989073, 0.188444196337688, 0.152735920449503, 0.991213812253641, 0.974719163955551, 0.92210120062995, 0.451302690084197, 0.201801107156956, 0.375567162041209, 0.743166312812861, 0.901100782137483, 0.860453293948196, 0.973288979289841, 0.151505139294492, 0.123157717024781, 0.193804249534416, 0.257629088405677, 0.728177093254306, 0.223366229935471, 0.597042186791992, 0.174088560782455, 0.575221320773925, 0.458982778696048, 0.882937972621323, 0.00650198951103231, 0.367411161350731, 0.162736921017905, 0.0762916483789469, 0.901150625634785, 0.427891618442538, 0.546476648041144, 0.0870373480211519, 0.58889574263249, 0.407885337026936, 0.612577938018737, 0.0403466761149135, 0.047968059894246, 0.457422464301943, 0.819527155586555, 0.698851985384549, 0.549944778668122, 0.200925703694596, 0.637987234719788, 0.0453295100982313, 0.720745666531769, 0.42430446049819, 0.515791060957365, 0.91128073571204, 0.749633768749228, 0.962074885669124, 0.757375036882871, 0.438088029272259, 0.455497175954669, 0.511582849213175, 0.233610094154093, 0.439686691027036, 0.144217788598874, 0.523388239374593, 0.403795235679158, 0.0549636457111743, 0.364322176409065, 0.0985759762700023, 0.609968849395632, 0.223400836386824, 0.580666414744323, 0.330510147112465, 0.746494719412165, 0.781606758802573, 0.119134526308894, 0.430988738781785, 0.145344323460251, 0.0995088163553153, 0.513675451399273, 0.185054526935119, 0.725277406636172, 0.0386088369531696, 0.175246858698871, 0.848634144764767, 0.829908835067938, 0.203961588322897, 0.487527623521674, 0.53543582114243, 0.195502353530067, 0.914186767112513, 0.2827732453398, 0.951048404969743, 0.683168328899385, 0.617910093228535, 0.548731838670029, 0.777954095668881, 0.233193059959588, 0.116576197602689, 0.353428164494115, 0.0996200662581027, 0.380836945553942, 0.977993018254794, 0.614111076914419, 0.575922494252192, 0.449712513017651, 0.445774079720046, 0.407442294041293, 0.898586240438299, 0.416341549236607, 0.680155814774734, 0.0359558849342771, 0.178071270927471, 0.0939088173022644, 0.12519619721918, 0.188528968939945, 0.465440773578037, 0.171773402572559, 0.521053744200958, 0.6752911763543, 0.868889720530305, 0.756936934744046, 0.984231752489205, 0.219840023488755, 0.17299855767028, 0.922727967771878, 0.898937259860698, 0.377197441842611, 0.137198561191727, 0.142253734790729, 0.00677180636250008, 0.807929724586047, 0.82480553691314, 0.914956174385026, 0.34894271425431, 0.456637506760217, 0.645614634041325, 0.673504174641959, 0.975907220472087, 0.613358665077668, 0.9277498563074, 0.928171911918171, 0.997717793800683, 0.268293617555448, 0.946144138307184, 0.112127648840552, 0.869964351034623, 0.109511207608333, 0.898298449748917, 0.635193146125505, 0.600993799383576, 0.468421734258991, 0.513904987868295, 0.544773090403598, 0.690247289902548, 0.563029474973477, 0.203872655612215, 0.375176843801099, 0.630855870321767, 0.51516151823398, 0.128801860189977, 0.829810121932202, 0.839440967003994, 0.32347648716515, 0.158390361445082, 0.857177804945972, 0.317448139780466, 0.269635143579181, 0.860606296988356, 0.801219989661243, 0.262469758827791, 0.898853121754428, 0.125889240430579, 0.856530667272244, 0.824722301542086, 0.453872896195531, 0.128094187393847, 0.0821987100067491, 0.174167656402849, 0.727454799655225, 0.486973021604021, 0.506924766371539, 0.919943155922417, 0.215466762135267, 0.135851467108968, 0.441895185979926, 0.22208361235684, 0.236069491691537, 0.275397412769343, 0.138118805191854, 0.176380197535307, 0.372319538003299, 0.986729222296987, 0.858851072283221, 0.733991228297696, 0.589774618306049, 0.238835099760399, 0.242429701481582, 0.0189239686720866, 0.816261333411063, 0.712925008125233, 0.774855080991709, 0.295749412191594, 0.798321914430839, 0.50861496383945, 0.117903235970571, 0.876169975353899, 0.205440232737884, 0.291949804401226, 0.290843512338408, 0.875444065825031, 0.160987002024723, 0.722809725079101, 0.805565543512764, 0.7418323116947, 0.792085395829871, 0.697821407865205, 0.778400664689576, 0.365185234185173, 0.909672419513999, 0.55315718211792, 0.85643221143914, 0.957510757954678, 0.805097061931735, 0.417961342401037, 0.602931412462276, 0.52306085413977, 0.608297520867164, 0.175605238949945, 0.518054536040671, 0.496825805050741, 0.846212824527999, 0.572806361986874, 0.17604623683167, 0.129696073471382, 0.657252387118059, 0.213118283117643, 0.445418312891947, 0.745188920438825, 0.66550995909755, 0.299395622213487, 0.772555257354774, 0.355395084332697, 0.090119945056847, 0.727179262913629, 0.716518445644336, 0.0810665699739642, 0.736509297139809, 0.0837457453522166, 0.89169061949682, 0.88808894000696, 0.569339413599545, 0.847043241022158, 0.126624391351658, 0.204711202811273, 0.0494695798458733, 0.692381559649098, 0.293286012592438, 0.747849573969439, 0.534738953770585, 0.327923014672404, 0.921022014335099, 0.82846538037923, 0.52299047704593, 0.122678937476802, 0.0923400487874675, 0.672960646182439, 0.710431044271399, 0.980868077449441, 0.962349007263144, 0.402067669433401, 0.186915009477471, 0.389354144935206, 0.885119176776565, 0.595220050541887, 0.628195826689187, 0.960661542821547, 0.239762072080574, 0.318027163843958, 0.286204218392689, 0.91801192941001, 0.80854657263268, 0.977464625887192, 0.173098927425456, 0.805187853570242, 0.364399713274211, 0.561876494869237, 0.288411106077789, 0.718679664450527, 0.328518911872259, 0.676988760043015, 0.8335452742398, 0.0425012503857446, 0.964167106604212, 0.13219719857119, 0.136863616352688, 0.652430004863217, 0.330119977126648, 0.977634889132602, 0.663341124332282, 0.915636802368722, 0.144173311992227, 0.104043976412811, 0.939679264052967, 0.734526167581914, 0.0232597404707208, 0.108073314286133, 0.107345326275519, 0.281817324825198, 0.862780034494015, 0.109755239339176, 0.574089155908057, 0.921953095784068, 0.978371413105389, 0.90440141362574, 0.0506720447047913, 0.412456318896103, 0.311647926529913, 0.423227146204769, 0.914799425827318, 0.284657920925413, 0.574950603356869, 0.80654393674224, 0.718628207656864, 0.901144382537628, 0.0553910514672971, 0.860098342190033, 0.00775546206598576, 0.717889795410453, 0.710900993582111, 0.175537161953199, 0.869655230072547, 0.15656218671881, 0.911963558374017, 0.233808358194806, 0.68605929821869, 0.639898507381536, 0.157594329622638, 0.289685921839263, 0.1985220085135, 0.931812414193321, 0.540734493604897, 0.952563527123107, 0.217463483050788, 0.645032882513212, 0.786224432446053, 0.400258109751021, 0.836795378998138, 0.163049800935092, 0.203313076259454, 0.343979607240712, 0.24913003766186, 0.345701393032118, 0.986380292911815, 0.62395075589284, 0.596264469699329, 0.577291960511246, 0.403034019418634, 0.545446219273245, 0.777425284122461, 0.470403491137414, 0.088387449412932, 0.933904328109049, 0.0435984770570527, 0.648696341682491, 0.733022827726627, 0.0932238105655576, 0.785547167975576, 0.92482414658983, 0.564250343611759, 0.454247239263726, 0.793535581116191, 0.283778812096222, 0.992101845027678, 0.985979279263242, 0.536179557244393, 0.894398466519281, 0.235551273005255, 0.478506294638978, 0.90756723873527, 0.663744525326212, 0.43575429373567, 0.853988579848298, 0.39603528949207, 0.468605072723976, 0.449999153642314, 0.323984194380817, 0.87081941787461, 0.883295224342292, 0.3430104732742, 0.953920849572316, 0.309728267551436, 0.931513844460595, 0.161897867804766, 0.952939391325021, 0.878518890278084, 0.50717054954104, 0.0182444328293786, 0.778154779832036, 0.359457237006372, 0.848549181477326, 0.433062044566629, 0.605781897535805, 0.377579987786796, 0.721410011851881, 0.316173177651865, 0.949351481547407, 0.849745859636266, 0.7755979107193, 0.299516570122497, 0.872699031386631, 0.375082793554892, 0.79841957959226, 0.816787431317946, 0.999012957622323, 0.0450887746726181, 0.236802667460946, 0.430905871018584, 0.847239986152552, 0.386435238255589, 0.12280690798893, 0.91526185373976, 0.182327654385822, 0.59077686343556, 0.81235032472824, 0.261161147561688, 0.985808374756683, 0.441256874106545, 0.225260738980286, 0.184718539437457, 0.698240328571743, 0.875195458525484, 0.768964420326345, 0.261434712558202, 0.985705873359139, 0.0441839926335158, 0.344146335294759, 0.673184900946409, 0.0567341661691324, 0.0545707441021237, 0.564865286478818, 0.65094642655382, 0.237027799041954, 0.626916372282016, 0.530038348345737, 0.514283809073736, 0.701627648058301, 0.373690660242227, 0.614033077192303, 0.307286921470342, 0.0928969635412105, 0.662921453362801, 0.0285752042937565, 0.38454483107454, 0.0720519119322381, 0.291918443617576, 0.213159570281903, 0.721997396398347, 0.705025025531248, 0.715119919964416, 0.361222644967133, 0.546469979718963, 0.357832010260271, 0.940956411697522, 0.444791535285581, 0.110818939538868, 0.865121765621827, 0.0355549879738798, 0.223263653284556, 0.137962865534412, 0.302575123031492, 0.254191780954881, 0.731183834192674, 0.6376858890977, 0.97524511022823, 0.331082019914313, 0.420691006107648, 0.480425530991636, 0.909834239400194, 0.572124678740605, 0.424835845937944, 0.454417851654793, 0.951207834978692, 0.582945882750176, 0.391435297912972, 0.55807778382827, 0.98655410739113, 0.783623039201407, 0.0882224442327111, 0.378497662262159, 0.76950461550327, 0.199407543819982, 0.353075833113245, 0.217678500085411, 0.00330514971862206, 0.191625111374249, 0.450238535576727, 0.681828919403825, 0.468066479908331, 0.128375685669463, 0.803684288310312, 0.0759729679531037, 0.123078097146991, 0.871057395886045, 0.685439641349015, 0.511907546264598, 0.616499119138186, 0.832547000869831, 0.0105681571288471, 0.522006821143108, 0.383858754482607, 0.0587555667151998, 0.714304520985707, 0.778115616615674, 0.716649735218837, 0.119850377589458, 0.323629574725132, 0.866183742926756, 0.675168652148196, 0.239315830812321, 0.0284746419549312, 0.263856453806861, 0.74855758268174, 0.430818299961055, 0.950183090616478, 0.0844840243524803, 0.0164284667183011, 0.6146099219645, 0.213516159682177, 0.753660319785068, 0.712053280195641, 0.4065295252711, 0.242931074698522, 0.529311677653324, 0.792372772902561, 0.77154231865989, 0.372952032540412, 0.436001468009836, 0.986685085003935, 0.896192359562873, 0.735465312704808, 0.220711245187146, 0.0691363362134851, 0.0885895881254305, 0.262430405067953, 0.151457652948413, 0.364139689805619, 0.681044024183814, 0.102128499255563, 0.887078149607136, 0.467241865843286, 0.534877964268878, 0.0811182697830365, 0.138553837715714, 0.032083250050429, 0.273419873131307, 0.760616797028773, 0.938730474728435, 0.0153162895140798, 0.064294493741232, 0.667291479388443, 0.562792355244017, 0.236514945879954, 0.892676056225764, 0.461608697243822, 0.605183777911081, 0.314501339667149, 0.134871134831657, 0.928122678875817, 0.957911113688375, 0.989271457946254, 0.222673759761081, 0.367960361285373, 0.713168316457693, 0.0292660300446663, 0.596403775878687, 0.347654729158748, 0.444995207122685, 0.146102873510154, 0.176703072842682, 0.582891473319988, 0.419112322396977, 0.302614694429095, 0.454971173926771, 0.40388954940394, 0.892283756762325, 0.42059034794595, 0.740936599009069, 0.267468359718688, 0.24424743351122, 0.165360308785821, 0.607166086764022, 0.615071904149315, 0.927509938751301, 0.958221837170664, 0.801808259085617, 0.612654987495126, 0.888404424291398, 0.518880284913073, 0.463330914467006, 0.246202172908744, 0.757982987324507, 0.594862636837158, 0.00487275609632309, 0.653197986307119, 0.955331170803472, 0.498737202346259, 0.604601085572133, 0.713503377183815, 0.0629767711788746, 0.182439895641131, 0.878017967176024, 0.785639052293713, 0.938705350530863, 0.3002112698076, 0.857206504521371, 0.817726168598085, 0.830411384832491, 0.435450566350761, 0.540584485536151, 0.913597136050362, 0.597722790410454, 0.00687106331842979, 0.272338509662135, 0.13746677291111, 0.708699521577712, 0.0415070366439134, 0.436744169228878, 0.68035681027707, 0.55174141349666, 0.177940203422543, 0.0855453247199688, 0.186771978036625, 0.368764391025356, 0.93108970551754, 0.696638458117239, 0.166342815880045, 0.773500720784425, 0.413202328587861, 0.708613409045023, 0.332137459967926, 0.551058337499245, 0.99646818888946, 0.667182111672852, 0.888950285438331, 0.35825786795814, 0.434015037311128, 0.733049222501752, 0.994528889771893, 0.476645211977044, 0.331558468275499, 0.605344501119172, 0.773157784497556, 0.385647040304875, 0.203863304668464, 0.386674044332298, 0.0650316030550663, 0.432319187349009, 0.689734061017163, 0.506562454509307, 0.716917088744948, 0.301216644720693, 0.706008465959635, 0.915207000068682, 0.705798062713409, 0.13312345160555, 0.665466334995061, 0.375564946686223, 0.50244187112309, 0.399008690517975, 0.960435153662486, 0.115673645231039, 0.561219129812773, 0.467225691937315, 0.158936914098509, 0.525197758459624, 0.319739835240835, 0.781342407145475, 0.0629339055954374, 0.622343385510496, 0.63705379308988, 0.081105224083995, 0.846314395229993, 0.861935992743714, 0.758723769025251, 0.9377342090538, 0.647618067534789, 0.996026818364396, 0.736215676324277, 0.0574790450398441, 0.00934525757421878, 0.830913139567107, 0.335007981383175, 0.268971598623451, 0.10830071521189, 0.181663689696502, 0.383804544576176, 0.83710370317383, 0.761701898995251, 0.987244990403161, 0.711000501161428, 0.344842507617651, 0.537591944248493, 0.744182557654133, 0.929392725683407, 0.128399994648619, 0.220987024773803, 0.328011924210891, 0.338135086066146, 0.867304972389185, 0.675717065142281, 0.0169834659827949, 0.51743925346358, 0.512295530472657, 0.396039091604275, 0.92070974102436, 0.991756824712822, 0.554696003725512, 0.57080138289976, 0.232129974945879, 0.798207198118094, 0.237841356885289, 0.573310729822184, 0.619431751108203, 0.453728165940017, 0.754412771599239, 0.565870046902584, 0.77786522358599, 0.73419313864594, 0.965407094795998, 0.763000515668061, 0.56686029679736, 0.599694968703176, 0.591279069006801, 0.961648152739608, 0.441617004025253, 0.692480512898259, 0.235857378579646, 0.770613012361659, 0.160000757045431, 0.0786548146581735, 0.31065558709325, 0.324245128628953, 0.17690008416177, 0.246012391699788, 0.991838790960318, 0.611304451430512, 0.483004024535443, 0.244265593402648, 0.973911393381789, 0.715415776035476, 0.698390213436035, 0.174839262775748, 0.470388389578201, 0.805089545881816, 0.109709355562925, 0.795011431642314, 0.868027924142454, 0.0973489593348447, 0.0658448030659305, 0.0602826762585706, 0.45503033091711, 0.879421641426555, 0.347852507579447, 0.436686341436296, 0.131199567391182, 0.6533989030209, 0.537490872197965, 0.729113008620568, 0.132126735006215, 0.894200002244513, 0.457946975882648, 0.100627182258511, 0.401037530776682, 0.868276705519292, 0.957446761708148, 0.0213950998605059, 0.944959176591185, 0.921909258298927, 0.610961586342133, 0.490169170391811, 0.444750798158141, 0.396818677314087, 0.151911567521883, 0.870169255999238, 0.485674194902842, 0.859897318606698, 0.728214624538208, 0.542567725535353, 0.630757626740935, 0.269076315415997, 0.591812006788566, 0.866305096716458, 0.925544898560559, 0.951119572011534, 0.0667044792590026, 0.946137972467446, 0.496633305608366, 0.068027386603233, 0.15364170976211, 0.296073114887304, 0.168748938145153, 0.735194614493405, 0.87969706522077, 0.851423388162803, 0.672029350928386, 0.798496214678467, 0.1004802426334, 0.522668514129408, 0.058688252904172, 0.813646068464573, 0.345685331495202, 0.82892407862342, 0.724963349991644, 0.337293776914633, 0.211463983876867, 0.33984012293254, 0.459291792737912, 0.528747296798372, 0.401205825760823, 0.751922206181604, 0.0269389098681395, 0.374649450407488, 0.186681263476482, 0.371981919355179, 0.330847753005635, 0.0185118842303693, 0.685230702362444, 0.672337943143978, 0.833036334892142, 0.436388160437168, 0.635741381924272, 0.665115120528064, 0.355772565787277, 0.318534891072671, 0.689773087674081, 0.0263139745535609, 0.939920912539126, 0.626542984816351, 0.135101404432941, 0.938512808493749, 0.467858070436307, 0.933204497037136, 0.973246864792899, 0.284361627399017, 0.538808753434114, 0.0125754473833061, 0.157151076976654, 0.351646793648724, 0.773126733465265, 0.320200863959387, 0.553478215245487, 0.380122290620936, 0.750209669999736, 0.985073387208069, 0.447116741444502, 0.0746274309721684, 0.100438073829211, 0.336225543569876, 0.292208779403678, 0.916975020679447, 0.504762222831395, 0.579223901535662, 0.0516839562233129, 0.266766885591999, 0.782876822238966, 0.538027668514762, 0.939762250834365, 0.53760966123797, 0.412949107445452, 0.96408934566066, 0.599124253395707, 0.941265880855016, 0.761007180696557, 0.110870232257325, 0.285132571487838, 0.212596800745744, 0.414766474283189, 0.501671176426372, 0.841203631735745, 0.779606230011813, 0.583654264333153, 0.518982490652975, 0.299080325275482, 0.690151572334161, 0.627909058655756, 0.776411683100987, 0.867993821997256, 0.0264463820208864, 0.615516430681084, 0.198291874302869, 0.864763882791447, 0.928199536915847, 0.847255166003145, 0.0388062552421621, 0.527219861259948, 0.3638228287232, 0.0701277965208416, 0.248257559988378, 0.0590644093000539, 0.231342735179724, 0.93543981266209, 0.806716233914255, 0.0380683834158489, 0.968145559172742, 0.93004686549121, 0.0459226249529765, 0.329520134442665, 0.71652085428429, 0.0314579415705909, 0.242842985509325, 0.306535683071097, 0.833998919280889, 0.521828661555905, 0.115485341315688, 0.139302436658084, 0.404541139687733, 0.159313689260253, 0.944453176222054, 0.129839531533709, 0.227778466630938, 0.0267461645525247, 0.890789147999104, 0.720900144658359, 0.204047817263531, 0.340586076516964, 0.596289879584673, 0.612485440395129, 0.0657474769568257, 0.176573358288304, 0.173449524554605, 0.856474701789446, 0.579538464123781, 0.201830831261811, 0.675428067009403, 0.0994686494641299, 0.00411217881750736, 0.561701693321334, 0.356470595925915, 0.431978250656565, 0.274073010702791, 0.505806670866248, 0.262959737312549, 0.00582414304133683, 0.925009669830771, 0.494492867617938, 0.2210625323429, 0.885052685543261, 0.120334604510345, 0.531992692176155, 0.170326214976644, 0.965588120954121, 0.945125869028441, 0.491663308960789, 0.167258830671543, 0.710888220809289, 0.75677064864037, 0.479871330557751, 0.836508220988943, 0.837079954474257, 0.570325362144909, 0.783084386870491, 0.475747518314256, 0.77766406596977, 0.895995290630358, 0.956420707260906, 0.131301718165707, 0.626414507805459, 0.749587732494248, 0.190878881374426, 0.443756648790902, 0.291997727143744, 0.436021393837465, 0.679946475427728, 0.252995263439065, 0.872141780323535, 0.536203300972268, 0.912387495358426, 0.46591464613971, 0.3731350097078, 0.693488918603341, 0.756916472661015, 0.908452446818315, 0.0623008409547539, 0.783456960206153, 0.167869975757398, 0.832849221103354, 0.129459966086082, 0.492047854427907, 0.839369929104788, 0.578569354163389, 0.275630328655568, 0.362342825298542, 0.534341590080874, 0.938206926388583, 0.28331238478735, 0.639807660199308, 0.0377423316751517, 0.757197832101347, 0.187402786828509, 0.50962909684916, 0.592036183598158, 0.239064418156521, 0.48581593323602, 0.920619876593511, 0.362533154252935, 0.06353952645009, 0.606513864914999, 0.174102683296972, 0.631263259596786, 0.356817884005109, 0.691071852614466, 0.0550318701373792, 0.326877919639481, 0.612816226154716, 0.982823469309885, 0.454661377456279, 0.25332787042119, 0.0372223506451244, 0.564769992784299, 0.0457093746939572, 0.937787018133484, 0.453920095626456, 0.86965939474544, 0.927039690891646, 0.68152187648325, 0.3572433186077, 0.801644600581428, 0.57625821630316, 0.680263795321122, 0.582427200034119, 0.909061807509198, 0.898366559178676, 0.456334960565971, 0.182168274949183, 0.327845550436993, 0.0237893660331053, 0.00717053149584468, 0.803531826080285, 0.242548872580686, 0.792568585647509, 0.798999750784777, 0.205029489556223, 0.491708106334057, 0.850046808896721, 0.625678071475631, 0.214782165915429, 0.972773504408982, 0.536885216694958, 0.834640614161082, 0.815553692681559, 0.554828209010308, 0.428528097925352, 0.292305565180024, 0.802653274387073, 0.98876453505998, 0.508386223085942, 0.190799341764655, 0.541160855896932, 0.248694914976465, 0.391545747451545, 0.206596620617521, 0.785773389066648, 0.47317256510391, 0.47298592518744, 0.718192062445971, 0.128004918082133, 0.950802582263996, 0.734839930218717, 0.618638060108836, 0.561629980027797, 0.227762630975718, 0.885068365914219, 0.514031908099006, 0.418641067393353, 0.775304498411853, 0.697213773572511, 0.990275373994243, 0.592910507638677, 0.156334190447356, 0.130790291601893, 0.170356979171251, 0.0660196279012246, 0.375278501325524, 0.726358873972178, 0.389103635700203, 0.0360160105651978, 0.00565422740816024, 0.65616128150177, 0.752658419337526, 0.0676140295318269, 0.339271174658234, 0.497391907174856, 0.311823877318729, 0.672018629452127, 0.808391496477726, 0.676905068160315, 0.930723208035337, 0.638917637311985, 0.572912112423737, 0.990677385311778, 0.0545516736108403, 0.0850607905607035, 0.961250636466643, 0.405431593774134, 0.685279901674016, 0.406040726970767, 0.994991603914192, 0.837892177512324, 0.24599228967387, 0.4164766934677, 0.369977011301307, 0.791801126902813, 0.0321086712428561, 0.796476226098879, 0.0340560200474812, 0.0184653089218791, 0.447450823524414, 0.837515940560554, 0.795998297279493, 0.936868969471949, 0.201312257874401, 0.862680586798852, 0.255804533168903, 0.0596814152258668, 0.968679951943002, 0.0655055295440604, 0.22954103987623, 0.0330783997679384, 0.127822034227125, 0.680034786406752, 0.0282963178771346, 0.851903728563756, 0.28222186366775, 0.42198995047167, 0.653312836228923, 0.921595131567574, 0.29235042450842, 0.966800955890742, 0.779414513544575, 0.0863146974159165, 0.178891732209681, 0.190303722817113, 0.0991795740395214, 0.909900373676491, 0.048424725033624, 0.910023681779888, 0.122042492002593, 0.353663311690865, 0.231771447377546, 0.560724502943158, 0.573308989115434, 0.302768810346661, 0.773448542515788, 0.013864648544228, 0.267161396605947, 0.932077546550194, 0.838305707338285, 0.0228831339075435, 0.965606835175219, 0.397912615154625, 0.51522887807056, 0.731655159693645, 0.553230617534227, 0.983242421384206, 0.534353503575923, 0.641311072034266, 0.336625457904485, 0.982333853805934, 0.886031251919307, 0.106709626719116, 0.62031881913977, 0.87112467062374, 0.588828183532762, 0.765590096170102, 0.747787764290155, 0.126251326637296, 0.636739922727361, 0.903567411729373, 0.333755943050631, 0.490888003113312, 0.225383933376492, 0.764011331585692, 0.404791851629027, 0.960108032623739, 0.612394524860949, 0.353583471617713, 0.208462896408394, 0.0767153649061801, 0.585291075471344, 0.263759520401117, 0.58371728221952, 0.0369064774350222, 0.896514238355332, 0.175270976222151, 0.720939846042773, 0.118598842624757, 0.476790729776519, 0.742696671898213, 0.533550897294353, 0.635104226348431, 0.957403059315655, 0.243035675138141, 0.628838596017375, 0.746147914251123, 0.852528285579114, 0.670857196082542, 0.968776404517905, 0.00286565235432312, 0.0799642071199025, 0.567704959227614, 0.337405312185187, 0.607000066693442, 0.124433850190054, 0.581614352968005, 0.86800359400858, 0.130699533222642, 0.363704341154399, 0.0890565282853079, 0.182736898729036, 0.612701918143585, 0.926601614703096, 0.738207540827723, 0.00524670652940706, 0.453466476933293, 0.552925339929297, 0.360648332549356, 0.0250715276881056, 0.672116457280775, 0.677362957838427, 0.3784750018235, 0.434442338924472, 0.532948957717022, 0.0519278879095435, 0.231413508531899, 0.426072207857013, 0.14278291994304, 0.988124589852255, 0.848090700864974, 0.928070501021586, 0.872468016740066, 0.39460440440744, 0.377014795719064, 0.0257393599359305, 0.774744414784401, 0.352194685944672, 0.198064106187228, 0.689376090482463, 0.212043217359797, 0.11148980995513, 0.942469564750566, 0.855164130643152, 0.862778642512992, 0.125874466374742, 0.229491137685251, 0.844390787077373, 0.806825335194664, 0.276473764625476, 0.311237755920822, 0.380603778994493, 0.418476631335865, 0.0659325962209559, 0.947149379733973, 0.790256877085953, 0.652142698611373, 0.975280378309018, 0.130744698276658, 0.515836026426119, 0.531822230955089, 0.131869163596896, 0.644132680133535, 0.407307395585411, 0.69172142435076, 0.695268568653786, 0.30176005846827, 0.832079260048842, 0.770458137945529, 0.986868308239219, 0.612174890925808, 0.365637961305712, 0.150739496787713, 0.975700356675666, 0.706263637480424, 0.766585653189381, 0.177362043774218, 0.240518333498242, 0.904488326584785, 0.733798381059945, 0.874674307332286, 0.267734756181228, 0.851824725372474, 0.82833835522631, 0.879549541898559, 0.0537511553050095, 0.251030412718603, 0.853769837363927, 0.952942336508878, 0.131588197055904, 0.300878842686364, 0.0519092747961227, 0.546441612934141, 0.43669515976829, 0.196771005371458, 0.204636450685895, 0.0901499183790865, 0.00452093509188833, 0.187125972205646, 0.651350077499493, 0.595724107875004, 0.942288013820397, 0.640202862579832, 0.449175040208123, 0.376823100185069, 0.00879007885083871, 0.216685967587839, 0.443192321750266, 0.339368240977801, 0.140140637041523, 0.0884111464949593, 0.00373111696646455, 0.797665353495706, 0.293663016898885, 0.71135235660542, 0.443497241116738, 0.296089033698722, 0.2620622278389, 0.2752419101841, 0.0737013140896307, 0.792192249423989, 0.300424791025954, 0.0503365765810105, 0.336091550352029, 0.862364084341725, 0.151861223131031, 0.309545770935439, 0.530871149133982, 0.539322839842854, 0.794497837875105, 0.0607676848405296, 0.842682724072832, 0.931039330247171, 0.75925920324808, 0.118355596851141, 0.125964228433316, 0.300001601195746, 0.376687963417636, 0.656217546503779, 0.84668373606852, 0.778508563377216, 0.698830074886309, 0.478728899793064, 0.392181687636536, 0.549187604491825, 0.595028056701018, 0.0596209114253973, 0.919210360308339, 0.482580859126777, 0.0735834521861951, 0.342805226989209, 0.540715814912947, 0.605088046002887, 0.776784526255746, 0.661482105035203, 0.655713497763003, 0.577820051095709, 0.287445829709382, 0.777319801566022, 0.521644117571933, 0.470604318353448, 0.123602721867068, 0.841852704365975, 0.155351633244972, 0.766639579429069, 0.0480175572098605, 0.76359016464548, 0.350508156315623, 0.459881106458728, 0.0485355742638932, 0.293303827881567, 0.745346387590054, 0.926028616255638, 0.752651195263199, 0.14521326809408, 0.346983421546869, 0.893750444340903, 0.141769783070216, 0.851188331580125, 0.477534513094796, 0.926000211250801, 0.541409440262695, 0.528100664614302, 0.842075578348273, 0.179645737113151, 0.276169145514343, 0.156308530576256, 0.784108735845911, 0.613586244650307, 0.631053132943444, 0.73130294564249, 0.118357834097237, 0.973976377272467, 0.124467282660163, 0.901712976940705, 0.0792069274493014, 0.874109404098772, 0.604860188754782, 0.847550262561697, 0.71517806887821, 0.959884475323722, 0.193622739507481, 0.0751440053888787, 0.0841323276998058, 0.961059472063457, 0.965241337687973, 0.932281942265587, 0.686906840088052, 0.13829324688242, 0.647749744535459, 0.407492506834105, 0.70350281341506, 0.224708886548267, 0.995599827896049, 0.101385179999969, 0.818112331041775, 0.685841118013878, 0.395169886846344, 0.891401767727372, 0.382262868208066, 0.445986295483579, 0.872385076890101, 0.885029374156899, 0.207787915387991, 0.795842622844165, 0.691429967090675, 0.620176269381209, 0.404717363139943, 0.146061045487498, 0.046497698730936, 0.245682916413389, 0.172979087885868, 0.26013860024354, 0.956673482447943, 0.608924532442043, 0.608558797959612, 0.250579386083627, 0.785001302865601, 0.0112850212392475, 0.123473488811845, 0.643596938016399, 0.901491767886539, 0.385266949577157, 0.767315890991487, 0.888129591635809, 0.0759655243970787, 0.352615079800291, 0.963996051967761, 0.560734891789643, 0.849555295342507, 0.851781062950678, 0.723875899421331, 0.945934154782245, 0.696686239958025, 0.0672111882455044, 0.290057304074911, 0.197185271200558, 0.157380672937732, 0.296112803352921, 0.257477799251388, 0.216348236706125, 0.526025944469512, 0.320889888847558, 0.284681429168109, 0.0764900349590381, 0.885090055095909, 0.1375960104857, 0.666251822361021, 0.00127016266772084, 0.890023596617398, 0.498327621584295, 0.937376648855562, 0.0169571419860958, 0.697271729986029, 0.778319519627564, 0.65248845586072, 0.551583890047413, 0.517397941174482, 0.404962763800286, 0.540615184166759, 0.273651719932771, 0.155228450161989, 0.882462504637313, 0.187970483410596, 0.43580371217463, 0.363053679870465, 0.881023847514958, 0.801242551920121, 0.00595395132734355, 0.589546729191954, 0.406781441011526, 0.0309439111773996, 0.432382828935691, 0.41321211430675, 0.885450489523656, 0.226749680414479, 0.974679967417549, 0.509848702128493, 0.821521366827744, 0.203635186600278, 0.191930916872748, 0.88982056178693, 0.732503858090024, 0.752261062376373, 0.403901266500546, 0.721783048118954, 0.416836465168648, 0.986590582068056, 0.539052916092479, 0.423707399907368, 0.0709782222286096, 0.845701746531009, 0.657470538031124, 0.367304470505196, 0.699711106979227, 0.273576953112091, 0.144949511770621, 0.304749807567675, 0.111915906818522, 0.983205671842697, 0.810815709954073, 0.467765301250505, 0.380410325056971, 0.692455686023322, 0.104776731564931, 0.0283932609070279, 0.199128876884601, 0.263502730078326, 0.987383495592245, 0.206641759355501, 0.0535224132221298, 0.0861432080337774, 0.358083759345446, 0.834597708432639, 0.0290313135093165, 0.132812426113984, 0.186357500549406, 0.950512136201215, 0.996887741758114, 0.950536363226135, 0.95941154000962, 0.226753840499592, 0.260474806978176, 0.383671559487495, 0.875622184575691, 0.959102279527919, 0.630670020080555, 0.278622357685988, 0.0930163137812554, 0.202281115098587, 0.743576715689091, 0.681692728586875, 0.529430321601231, 0.837446392217739, 0.466297324095343, 0.322373758912214, 0.58725536539387, 0.639543054838828, 0.01748728925039, 0.0367376126398826, 0.0551374561532074, 0.0639835847659775, 0.07137338240094, 0.964775332992617, 0.0831562516059666, 0.455264600561466, 0.956195548004749, 0.508635911209233, 0.210308422130961, 0.998452982888296, 0.969910605736518, 0.673148170571132, 0.076505598359817, 0.211693475950667, 0.0457577485728925, 0.505710446973576, 0.950159515338856, 0.577561126673339, 0.439446311752574, 0.602004090396649, 0.214546440401506, 0.672566870578187, 0.323857975250956, 0.634142806056069, 0.355580712275571, 0.152582022288085, 0.0844857004743876, 0.738799482656323, 0.837413006096115, 0.367474025579957, 0.12704808754955, 0.0733284011407932, 0.892217333373334, 0.234806257611924, 0.767200090273578, 0.487061538227434, 0.796427805570129, 0.308604616235857, 0.508593427200324, 0.807122498375477, 0.750313121542552, 0.861750773876817, 0.553295186261435, 0.795642519653479, 0.949860508160626, 0.745837500200977, 0.0619829910361802, 0.190351590417353, 0.746450627549246, 0.109224731051392, 0.976738503776063, 0.773325541597088, 0.994299010058766, 0.317271949702558, 0.733974140224717, 0.815520440478536, 0.415411901711349, 0.446648478916874, 0.576176952232469, 0.762675484599402, 0.618939926969947, 0.314782712039813, 0.801594343214002, 0.763227172463709, 0.245734652304898, 0.507473444761333, 0.147019300893965, 0.129698665458025, 0.660654616874952, 0.0732312040468013, 0.201564139678115, 0.850747255518307, 0.595795473603532, 0.677312192739587, 0.964954542102979, 0.805441876939971, 0.75105704406253, 0.798439669844153, 0.273873273820798, 0.412873920749214, 0.273422480286923, 0.214854339538904, 0.284610862379391, 0.684535163002128, 0.434760642240072, 0.559762070756985, 0.200731538750809, 0.609905824110137, 0.788695294925569, 0.0614295506428291, 0.988984267520285, 0.621533832402367, 0.974383409903632, 0.854679699791956, 0.0393484636694887, 0.445475789118009, 0.323215770563546, 0.154959669482769, 0.873993216384763, 0.352135742418362, 0.399798611557375, 0.91112540858241, 0.398178123440111, 0.474933124377296, 0.894120954340671, 0.687510487618711, 0.659992612221552, 0.774742888183922, 0.8085497647028, 0.317968631143905, 0.00954553215680054, 0.914243391675322, 0.0975259847458207, 0.864317899953104, 0.957797352860632, 0.865014056306828, 0.673897420778921, 0.556163993749389, 0.451354389417382, 0.329942772872638, 0.0879311718179626, 0.331576500929679, 0.847955971093492, 0.0816064543213875, 0.139915630274235, 0.287279137737404, 0.828023861478086, 0.554792164815677, 0.3203728492042, 0.779676473504885, 0.877420416311865, 0.570817684712713, 0.559481212464933, 0.710702912170038, 0.332126389365687, 0.937676456037682, 0.117470338049465, 0.794218144577591, 0.329918942159395, 0.114577034368532, 0.549509402757334, 0.650743881377073, 0.613114147476359, 0.75187709411039, 0.162624513587545, 0.773011679611731, 0.771451095772245, 0.85464798805587, 0.542695423776974, 0.594334747837864, 0.263133985077178, 0.495746585477788, 0.991420985155086, 0.134565388812973, 0.330673309296106, 0.0151319757274821, 0.810350014262261, 0.727601365751976, 0.801017282892293, 0.400156786112105, 0.819352069083793, 0.692175064376482, 0.915341253027676, 0.641535104393421, 0.661277462222028, 0.0134599549993701, 0.86360530989591, 0.188147724518046, 0.264671254519114, 0.181176639489982, 0.539711976990391, 0.51863454114763, 0.700866968492315, 0.784574838468495, 0.255523555405238, 0.915122273254157, 0.327590989833603, 0.176565056117245, 0.228905469922716, 0.139538699249807, 0.574746670071348, 0.642184328155172, 0.249030894279087, 0.59021985196694, 0.360686284770389, 0.719582429266234, 0.335574051913089, 0.478018497872361, 0.223835022456644, 0.424421359564127, 0.621795488877918, 0.0912716549264273, 0.149159392165337, 0.149530212733051, 0.0887736272517701, 0.677968832711808, 0.552866821773887, 0.805898884739537, 0.400018595136627, 0.573945834172421, 0.27166351185441, 0.606500804087977, 0.913091458961742, 0.571814420758783, 0.985231062250882, 0.0180786866751088, 0.460841968214704, 0.503893085736357, 0.71416146324495, 0.690597256412212, 0.992692473902224, 0.268803718997518, 0.551104780566647, 0.059094681156226, 0.251510880772859, 0.538184133308532, 0.487699874871502, 0.304409880457336, 0.140910855009897, 0.342090005469764, 0.839038733726417, 0.629967632945586, 0.246026467974475, 0.478514039347688, 0.99311283414522, 0.726587530111616, 0.029170989097183, 0.552704847056484, 0.638715408409708, 0.378150679685213, 0.255088499630543, 0.558286535589857, 0.754724309868761, 0.0964286703668721, 0.417146643217975, 0.496841420196132, 0.858455576432876, 0.595265915092083, 0.639509460629693, 0.0583901331581728, 0.48509088861042, 0.128934147111779, 0.672737850669774, 0.767017515968682, 0.123054901522888, 0.718137288549845, 0.683378715885202, 0.0336640036036588, 0.929969523431792, 0.531312703194273, 0.203364967265012, 0.570212176606183, 0.689304455908066, 0.264519695217322, 0.434131009038638, 0.781390330783624, 0.923349873465747, 0.394039945943927, 0.390018609846232, 0.666083949578015, 0.954650335293111, 0.918578900494594, 0.368403918562729, 0.562901959263156, 0.800391488751479, 0.898317394114518, 0.924289856351876, 0.413089745677058, 0.203264922091345, 0.200031511770755, 0.0027593989950867, 0.983550668959425, 0.920471845864549, 0.776118269033223, 0.794333204446284, 0.551649789567046, 0.71141735947434, 0.0626128443050599, 0.0616486479001858, 0.606379241221464, 0.246812124985851, 0.00183632614033569, 0.766192055663874, 0.425430904670411, 0.989563772276567, 0.0702524785724785, 0.124507807279658, 0.968414819851344, 0.87722323716295, 0.24036862306885, 0.515920400782305, 0.0787726784337847, 0.635600698586545, 0.616206355518481, 0.679456250585846, 0.839602632284021, 0.768684932836651, 0.0546404311537196, 0.0819871031421398, 0.54301716978799, 0.322119702595809, 0.244383399683201, 0.022326453475462, 0.170627007320958, 0.950952702570456, 0.338382190067313, 0.497872385070424, 0.370201191956042, 0.661228649901267, 0.465686416114736, 0.892727945098785, 0.259551322684372, 0.680289918210551, 0.530949241549428, 0.0462061964343286, 0.678195447828788, 0.298320464194589, 0.105341423973374, 0.467517721103171, 0.682294599588768, 0.779982158177895, 0.827625377811253, 0.428071766800133, 0.504308440606464, 0.634221856977471, 0.726703127568162, 0.412961184958554, 0.356316107824608, 0.321548126678338, 0.462440976072155, 0.664047570327654, 0.645998205262341, 0.343888046624969, 0.589383179694379, 0.84668904049533, 0.177941491253013, 0.296553279093561, 0.0458101698715523, 0.799978344049936, 0.796095308571083, 0.395709565757626, 0.319934095424658, 0.991886219489774, 0.550913532987462, 0.0145785999173758, 0.859176796669086, 0.59380934266731, 0.861370716925761, 0.777049808689663, 0.104902639723293, 0.39419617729856, 0.975986219203375, 0.510489838989448, 0.393469780083594, 0.555266745953814, 0.975435960336468, 0.44788883712014, 0.86407813279255, 0.14292724626443, 0.0956774618056918, 0.347729778892109, 0.0727525880133207, 0.544439560601022, 0.965395776697798, 0.281172975675314, 0.080135820884417, 0.145801508955477, 0.163045621569183, 0.538577935176382, 0.468961205104059, 0.265314338431152, 0.674538743110966, 0.124544528832055, 0.251765698456251, 0.589169729268855, 0.351098128310759, 0.522773859255701, 0.335170711407786, 0.0235351253784355, 0.401679638189028, 0.50468348389528, 0.129589379578932, 0.612969398568137, 0.9634558272788, 0.0896571347769251, 0.308662096713197, 0.529213856593459, 0.447415069320088, 0.939962907017726, 0.721104329446225, 0.737882832406458, 0.639131985033605, 0.0353144947958958, 0.641615464196834, 0.318590642040134, 0.10110174781973, 0.0809821628826388, 0.902145357163891, 0.252735692022642, 0.49490775804124, 0.388113025418348, 0.263626328209528, 0.263116205244998, 0.699527429206452, 0.0567314468664598, 0.788716525101522, 0.292302368652778, 0.83943762172432, 0.484634284794606, 0.23347852383755, 0.220535297788423, 0.314285129094029, 0.995427364636573, 0.189131666587351, 0.850490968361452, 0.843904796932081, 0.528316509324555, 0.664871331282169, 0.369616370544268, 0.317437525377534, 0.810323914672184, 0.102623298673856, 0.212762922674614, 0.446422187898148, 0.38655117148252, 0.774647846087153, 0.474231132606704, 0.796791219344211, 0.0248594733248674, 0.999140061069325, 0.283053431145168, 0.767824918394698, 0.0350637132123328, 0.870359462315582, 0.742944959351178, 0.109293044868281, 0.866616736044428, 0.821525570124464, 0.00379177436567935, 0.66026328336546, 0.632364930658099, 0.210481488409767, 0.434946649333644, 0.933109401117101, 0.457282727874737, 0.110028603363793, 0.679891322407547, 0.97173224576134, 0.435377314850813, 0.237958923131825, 0.887420776605914, 0.952771191845256, 0.391654925226844, 0.763328041175447, 0.10005275059817, 0.223862121608506, 0.102477499262966, 0.0214209962288959, 0.516928272679177, 0.258686167093603, 0.033903026960707, 0.749681881226105, 0.541712263533479, 0.881948228728863, 0.561390734403322, 0.968690162359166, 0.752475152295962, 0.949282944871547, 0.738813912269678, 0.263525014112104, 0.0874845709328383, 0.685884616972962, 0.810464869829464, 0.150807285958347, 0.313813626801227, 0.487028420282263, 0.70363102432328, 0.596085202961814, 0.0543511551754016, 0.971161441540908, 0.574372180284333, 0.574443015319354, 0.577154105355525, 0.600919956008999, 0.136016346601626, 0.0349329951144906, 0.413836818905206, 0.27294932425643, 0.689648425647617, 0.985590792390625, 0.246915388180145, 0.76634835098271, 0.666659834668567, 0.536316437937782, 0.900524617508558, 0.666920523562836, 0.543384778363458, 0.0892718174801181, 0.0277882525673147, 0.959792781479445, 0.732243132649956, 0.890659928728319, 0.895253073981383, 0.794324316826594, 0.535217674434914, 0.342400738418947, 0.28078448984121, 0.47446145497677, 0.00857963783463589, 0.42480024467276, 0.717986747474848, 0.410535999967197, 0.604571157183583, 0.590882364524143, 0.978606088806717, 0.275977088932444, 0.0427861397876032, 0.870653354651164, 0.88784437050419, 0.37195058007886, 0.148873935384885, 0.169315241325282, 0.162822889015554, 0.595220624814682, 0.851465538187106, 0.0441128313172059, 0.590111010332239, 0.24779252496655, 0.218488867424226, 0.72844499372862, 0.183081501658225, 0.106937643819702, 0.453107556570544, 0.193715261777417, 0.371959049914611, 0.183760567956111, 0.886769244664499, 0.21348661197217, 0.542774791896146, 0.165275071871229, 0.448272641688966, 0.822608156007469, 0.472541134637184, 0.879506494609336, 0.656411272643043, 0.547909242381855, 0.518560834422865, 0.425357630061231, 0.812118817242681, 0.642029938812838, 0.184272622909147, 0.997703838502922, 0.488387615191985, 0.509235468594366, 0.764175999223294, 0.674589495770682, 0.725291646609568, 0.222260338025177, 0.383084938800701, 0.346163214747737, 0.841847895082889, 0.975266291839263, 0.282174238000458, 0.975571215353995, 0.602170430559249, 0.186944915542537, 0.463329408803362, 0.124167154181252, 0.699965061845979, 0.758414596317139, 0.479747817831244, 0.683120937856991, 0.718049584621942, 0.212783117284337, 0.140420234393711, 0.480692410229626, 0.99940712796102, 0.880797576664817, 0.0389256489911894, 0.948633435946305, 0.892860439819072, 0.212325031230301, 0.981948743627835, 0.254546991984541, 0.626856435030855, 0.518767107409676, 0.494841046049729, 0.246603751586934, 0.0637559093146444, 0.168443523538876, 0.664876218884106, 0.706078172516633, 0.858070212968541, 0.269825752108275, 0.0224692909948204, 0.558130327502484, 0.19897523591331, 0.920241156708524, 0.275098100162149, 0.228960321681769, 0.518149673243104, 0.256060080844484, 0.0161486059952399, 0.976485753531106, 0.898779900812044, 0.959210169820479, 0.844525897262394, 0.380965269011838, 0.0896018520718789, 0.732737407850429, 0.682573594018766, 0.435168828529941, 0.143203771582761, 0.307105332011133, 0.680316594528726, 0.503945045580246, 0.734657144040491, 0.648020233562405, 0.0678309914104192 };
double equality_constraint_matrix_data[] = { 0.0722903243316734, 0.814575295907754, 0.884313548710428, 0.370244622006705, 0.80034477029127, 0.810674714668735, 0.178550443583786, 0.694293686193437, 0.660977176617282, 0.990062543598768, 0.383514752132963, 0.730505181481838, 0.0761992894283375, 0.0746261574044129, 0.534884979602282, 0.384844333502836, 0.375226018030324, 0.972919674204315, 0.713074912746308, 0.34589110859154, 0.243755037201071, 0.755657925520444, 0.884884700711321, 0.34657302878818, 0.0734143740645788, 0.552793575975769, 0.416689418107301, 0.224473772469534, 0.478228091242001, 0.499305184989684, 0.602913685377184, 0.285802621185552, 0.539570602664134, 0.519512291039389, 0.478191269597923, 0.580958470996604, 0.390835632106098, 0.920560855623545, 0.676390428139623, 0.291465622935452, 0.963629726793733, 0.016388466179857, 0.222066999494708, 0.828726912435869, 0.778836771694497, 0.302716497233626, 0.864857747829728, 0.927323249348823, 0.751463611996291, 0.442291487089583, 0.542609244978437, 0.875105707929658, 0.977469477438062, 0.595547561074026, 0.586490491370179, 0.883002040909933, 0.791177036383089, 0.512307686032664, 0.549912182051919, 0.639516306766381, 0.331145916949888, 0.328813249787199, 0.148411268391866, 0.727147065861518, 0.369138956631123, 0.606424635842114, 0.00683860720017471, 0.13891527955502, 0.964326718600994, 0.720468670709073, 0.915345338167616, 0.727895042700198, 0.497689138681011, 0.637315181637286, 0.484723514908553, 0.411636573065993, 0.273815876133912, 0.884744995265374, 0.905237832260349, 0.457724563131211, 0.419483856433989, 0.551574757434618, 0.79039647751428, 0.0587274986605592, 0.465779785484124, 0.774542631215436, 0.704232651065078, 0.676345339646446, 0.775616534373164, 0.454739615219013, 0.225752049846746, 0.936337944072215, 0.62282138266756, 0.733323914927756, 0.836393876119943, 0.594530009286183, 0.394787963199587, 0.618225121558663, 0.677166973296173, 0.0173207868555989, 0.738938105477091, 0.0547025776535461, 0.24325557799514, 0.729087564124854, 0.436224435569103, 0.00149287817728583, 0.219477841874477, 0.388089818893286, 0.724353985897651, 0.767101828106371, 0.240688559954776, 0.0354787515261803, 0.952299352583891, 0.530484209985167, 0.16041974842808, 0.362753110206593, 0.277613680208874, 0.580703710120059, 0.640865920975084, 0.118886968499292, 0.0312923176313687, 0.480834218023856, 0.154103301809375, 0.926600445516158, 0.412228892317898, 0.403843980043636, 0.153172787145426, 0.875066375124695, 0.877335566747687, 0.0928446949722125, 0.332592422147533, 0.693491866368946, 0.170895767582281, 0.345922495819317, 0.365070967053021, 0.341660328311553, 0.915387712213923, 0.171213743884432, 0.193373411062736, 0.966211629043158, 0.461039454334853, 0.734878282109131, 0.086795347696784, 0.215335461216343, 0.570569912238872, 0.323566081683997, 0.957257841426681, 0.590393498319034, 0.120141812048495, 0.276718396567295, 0.145665094562066, 0.719139751387948, 0.755145537055251, 0.847926149368648, 0.160876569003879, 0.42693028532815, 0.801657242185846, 0.20638453143481, 0.656937438648436, 0.290158896778862, 0.685802182830369, 0.30879215248787, 0.942700205261103, 0.238910324973161, 0.698779840616269, 0.13991983617283, 0.445181891292248, 0.646045858422411, 0.0851175634041994, 0.512013861067964, 0.249177377290097, 0.375284616942874, 0.58969059036408, 0.995182511804146, 0.0383515459496157, 0.265685104434068, 0.302191148972522, 0.20614320861523, 0.633699799504344, 0.715582397847856, 0.535502890361271, 0.0746155030500456, 0.153359504290199, 0.683657659638263, 0.333780892277469, 0.841984853401868, 0.947992207392236, 0.75168388642918, 0.0271897132794073, 0.74409974821869, 0.336755823604231, 0.801352241554071, 0.166903131955043, 0.636688145240555, 0.143241723221395, 0.820112026324193, 0.178481100823752, 0.179376325350282, 0.151675898483359, 0.434621922778401, 0.88000025801773, 0.499519924895774, 0.568013278463339, 0.0875586815063482, 0.443955443636316, 0.704571197322385, 0.981805703472202, 0.358108399458237, 0.115318408331771, 0.260124943167344, 0.995279833529812, 0.837499321725222, 0.750067727239643, 0.764551235231723, 0.975703249931249, 0.883578855145905, 0.640913644757537, 0.904587339505662, 0.810720654445894, 0.606039067197211, 0.857351842467162, 0.156899350931274, 0.323412834299391, 0.506473330585984, 0.393267637172327, 0.280706603454583, 0.888469848496961, 0.305931525115947, 0.827062884425789, 0.183115094482735, 0.851052390822192, 0.296070973712287, 0.0311654946032668, 0.370988524492556, 0.385176146649839, 0.856519478911259, 0.922538973616711, 0.256756247895122, 0.578292714785851, 0.327274680810468, 0.474590334289429, 0.74309371856119, 0.670820867625192, 0.0638907077548951, 0.901822086633396, 0.720924830693746, 0.443294727569869, 0.172351475073116, 0.415939869995554, 0.000715548317527626, 0.546541412298883, 0.564302836009085, 0.0650672211791405, 0.38669377306048, 0.50496490332552, 0.86857274268002, 0.748512601183002, 0.745714045254549, 0.8479373075143, 0.237099525107628, 0.337529562273883, 0.964130933918967, 0.143513409540289, 0.103534133848662, 0.908645908280158, 0.115186033859867, 0.744369309377401, 0.724664694576168, 0.12222063882862, 0.984905877254281, 0.208144306904609, 0.24974367298058, 0.486239628358987, 0.292744018798444, 0.716498673821078, 0.574191996709191, 0.71102525861162, 0.100811693552636, 0.31387604307437, 0.92828747304973, 0.892930254694621, 0.598705361456502, 0.4966466994201, 0.825701742049605, 0.81578107914187, 0.14120287153134, 0.604663858435241, 0.790571390123187, 0.748585664767025, 0.00982236072546616, 0.193401719156516, 0.384792839964191, 0.438757209224073, 0.892295834016662, 0.930285661397258, 0.602364650801703, 0.728975800795655, 0.894364927082967, 0.361839180211903, 0.203689130268915, 0.530281080664476, 0.92720320120777, 0.209948828518869, 0.982018434403266, 0.416363845353053, 0.268663446255055, 0.730739264353682, 0.534110028726556, 0.375113415580673, 0.382405999357521, 0.217159465390273, 0.845147382112809, 0.071739493080019, 0.3625193954232, 0.109586863933308, 0.913580509167226, 0.0405085123627152, 0.227929387166302, 0.351237721633424, 0.890982608386968, 0.483962302945169, 0.209413394448571, 0.318153214072262, 0.899285735385992, 0.00698699378511347, 0.17388603991545, 0.767738873239282, 0.175570807603494, 0.507339792110628, 0.293802360292034, 0.873250401444983, 0.823484306189087, 0.998822035155532, 0.374543086830501, 0.508094513107832, 0.748498589366762, 0.39991514651251, 0.931655153576901, 0.427685926765233, 0.0422192947160035, 0.157553040529377, 0.361351717681785, 0.0588538918369136, 0.84336036456866, 0.889983755622639, 0.0588942471833853, 0.202492541136726, 0.0873696666108087, 0.573072916480887, 0.43699023283442, 0.0888124173215159, 0.775723198136281, 0.733927379031315, 0.145227386857227, 0.615828514724771, 0.798206778933779, 0.820231043902685, 0.368625331951348, 0.809936891925674, 0.060680020423148, 0.0907508274678313, 0.805585641902073, 0.546980919566268, 0.895124242734885, 0.186858761589189, 0.832460573908841, 0.0626397734212979, 0.600029557848887, 0.0600721266080804, 0.233532910627902, 0.219706353145267, 0.259250380460217, 0.599285161743039, 0.694915398370391, 0.922127964308747, 0.655575466925251, 0.633643357559537, 0.40179813929328, 0.357841996705088, 0.453987599494908, 0.838933580444433, 0.390013418605796, 0.889675973309932, 0.811448379649953, 0.822251657461199, 0.864895217749837, 0.33874464732438, 0.0632638852819618, 0.266862545953493, 0.0925699514987907, 0.67130999140951, 0.944915448571485, 0.656308816159899, 0.416360101765498, 0.241606360052552, 0.350991432832415, 0.699740414476989, 0.488717014809074, 0.820372499255613, 0.329963318321199, 0.0940674616885745, 0.831761747601365, 0.426677551738374, 0.256949691340334, 0.605892642557118, 0.577125304829104, 0.961446714134265, 0.0633784441506314, 0.719434069219439, 0.902009465824431, 0.195184184542984, 0.897667932373422, 0.821605771707393, 0.665945381323852, 0.656676024099073, 0.53554177982516, 0.148039406542768, 0.615372649538667, 0.373914880657112, 0.438732996175382, 0.599164947022138, 0.877968834495135, 0.115913419416761, 0.985739540883822, 0.857254469293949, 0.441606147265157, 0.831619780151143, 0.292644842379715, 0.510935901808601, 0.75123716604151, 0.380326632265418, 0.266266942314067, 0.26260392121776, 0.582624380387433, 0.44312053108434, 0.446478632441557, 0.978698027425513, 0.79136094554561, 0.211472445653468, 0.948570817980897, 0.0585511461756274, 0.0239480487550577, 0.208484420420066, 0.294285371021574, 0.366028343025171, 0.85006418311479, 0.941759228807027, 0.0637994243281149, 0.409576227270505, 0.947448788352265, 0.222860614992551, 0.0475658698086285, 0.00722300874743897, 0.219968027004846, 0.673440894019482, 0.131854989652384, 0.704347394731617, 0.384885483429272, 0.362163503410741, 0.962599013218848, 0.0875689210675951, 0.255660876251761, 0.412617762743869, 0.703928253296993, 0.74797848497825, 0.0374367222841506, 0.781343841536334, 0.316967910740484, 0.414421711994423, 0.0879032229549459, 0.349375779722459, 0.12942356100625, 0.304327202838602, 0.946217354716618, 0.770128099543164, 0.82130728753809, 0.11478253101169, 0.203310651903552, 0.592598263314838, 0.0106912757564572, 0.27181317658423, 0.660901376287224, 0.200749943817415, 0.6857224493742, 0.107285913105537, 0.446994972411219, 0.127039953137372, 0.519774582833318, 0.256774164943503, 0.159930772426874, 0.421714971604656, 0.558692031556859, 0.300660109775817, 0.735579040615798, 0.340174651164455, 0.0446923134250257, 0.408185331322768, 0.32218135106595, 0.358314252710203, 0.0542195484063197 };

double problem_vector_data[] = { 0.168879971344252, 0.745166604030062, 0.477134351507709, 0.653444541099485, 0.966575933053896, 0.313027488227721, 0.0764394578120781, 0.791415146963651, 0.365383979290258, 0.585099389225412, 0.183336706183362, 0.0769189422460057, 0.153662766530705, 0.826875990170966, 0.300956742633309, 0.383884490005655, 0.650748947230908, 0.817370524039485, 0.766266219369973, 0.374175557582085, 0.18986247780247, 0.646501755218381, 0.00360530450734453, 0.282887905349612, 0.638600594751137, 0.592074579603645, 0.325289204685754, 0.988951251221728, 0.123235846277873, 0.73587698636635, 0.156617020109506, 0.434647411424697, 0.832243218102738, 0.359909561454824, 0.0762380475474331, 0.556925545740794, 0.273930224222276, 0.13205467930969, 0.699654848968811, 0.485904907517832, 0.182716250813704, 0.101214538189656, 0.201584327245249, 0.13474601073799, 0.323789153997764, 0.950545048096336, 0.532130560615834, 0.24768570993469, 0.437276253172887, 0.669087783169736, 0.547695710952879, 0.609054494316718, 0.863134760252027, 0.380695949821835, 0.748955615477151, 0.156696797557896, 0.0581238178949717, 0.339706962070075, 0.817176415224338, 0.377547669768167, 0.972604932176721, 0.605318590650978, 0.338236165549055, 0.927983722291575, 0.898424694301282, 0.850706460713442, 0.256791917068097, 0.285495500891502, 0.779946910428948, 0.701395228244418, 0.492511776265171, 0.967667748479233, 0.476161494330651, 0.994905517152424, 0.490578621847789, 0.503453396492993, 0.768772875919031, 0.388084482070887, 0.453259259537158, 0.132853968150371, 0.758506328245576, 0.56523693408585, 0.648636599114185, 0.798058063124077, 0.220448242881705, 0.857890771063757, 0.904728919137344, 0.29201988504298, 0.725869758147902, 0.339445804371305, 0.272681226854142, 0.170274589896562, 0.664018225811431, 0.535851657583605, 0.829109418219232, 0.26736208064657, 0.176155780622799, 0.431186366407187, 0.475655037798496, 0.78522090790695 };
double inequality_constraint_vector_data[] = { 1.252928412204, -1.25147139522053, -0.0824761930753967, 0.131336920725672, 0.420950869682912, 1.44300974811023, 1.047174283506, -0.987304116732754, -0.960288626601099, 1.02221051237135, 0.628662270317025, 0.180160276648956, -0.0441304226145965, 0.592770298774949, -0.467291664235938, 2.14997656423022, -0.0438514763820471, -0.0318399277570167, -0.371524367943901, 0.233495594251812, 0.142262857051161, -0.906012692241345, 0.930680391371775, -0.679548688419307, -0.885054989996388, 0.472821797926402, -0.470574474018815, -0.293260272276593, -1.45780533927725, -0.637806417267604, 0.62448248404038, -0.221872047882692, -0.0491718424212241, -0.999886526470525, -1.68555279122428, -0.834138631464721, -0.53694049737483, -0.438519283190944, 1.06258559132559, -0.302294123317399, 1.29763192116671, 1.51610297662712, 0.148565090958743, -0.53086162266333, 0.107475853457677, -0.421000348291611, -0.225211526136277, 0.71290934883175, 0.778703231131448, -1.28670486180094, -1.12258533857174, 1.25711392986795, -0.496783058387503, -0.554185066786352, 1.51962886401109, 0.77966127114595, 1.23931564946546, -0.515100703194017, 1.63340825018287, -0.445370746412585, 0.118948398402246, -0.855526878723119, 0.156921682521673, -0.290837538691499, 0.464383764285696, -1.14836257511985, -1.1514658176179, -0.925596232180019, -0.757507406568412, -0.855923102188286, -0.557742271395218, -1.60549148692289, -0.89414049236902, -2.0822769972295, -1.63312527355335, 0.425500386999832, -1.15058357341333, -0.134857120998555, -0.739463191388369, 0.804852100993994, -2.34828185927977, 0.0134954859706867, 1.54901194028571, 0.904742525935518, 0.669560412976613, -0.409452280062335, 3.38488195719628, -0.0973849978038014, 1.03182639451065, 1.71642673303363, -0.265457569518798, 0.0271744279605466, -0.359886006458418, -1.01285207341278, 0.730186249501654, -0.0818721089613458, -1.89324170442753, -0.0463173527664355, 1.10219013565341, -0.196313208152673, 0.435156102786045, -0.170028782987589, -2.04049090047598, -0.527051600063072, -0.20597178276059, -1.51310156781713, -1.90527610009854, 0.751622000655511, 0.38767285722701, -0.782775420572307, -1.70242431307364, 1.07711560629106, 0.959586761686633, -0.597041409719408, -0.0719811615343258, 0.48076773869679, -0.260799673956048, 0.159828019312723, 1.87624202150408, -0.00370271435426153, 1.26311967946653, 0.904984108933125, -0.749300285989674, 1.26372444539365, -0.614774492162787, 0.441163974588469, -0.0297946834036647, -0.356444277257048, 1.1271863519862, 0.508790271452601, -0.15788022549061, 0.845072541390849, 0.639190912808983, 0.739136547804015, 0.00274368931479601, -1.8568965934688, 0.350106183006019, 1.17377683150027, -0.448486546809128, -0.628679515874492, -0.0974178684256868, 0.524311724551763, 0.338867183774623, -0.418156490556024, 1.52261319582141, -1.32444882132573, -1.3906417378574, 2.48762432453425, 0.343792666564358, 1.05106471460986, -0.849052630824456, -0.152043242485238, 1.75056647060533, -0.0779942926905294, -0.522082400789671, 0.062981830871717, -2.2227545469037, 0.898260198917533, -0.569020370694908, -1.00564110769562, -0.488240472130906, -0.324495316856492, -0.695323926049477, -2.00672858542293, -0.805373674520454, -0.0720252112570498, -0.977598275616295, -0.577288126559509, -0.346292878031393, 0.149849412946617, 0.0720276445630019, -0.613611436092609, -1.18209482753559, -1.2308484277203, 0.389896262474429, -0.477893718467461, 0.605879114509728, 0.540679016153091, 0.933670043027275, 0.625574317134019, -0.293320195411722, 0.0634401030481035, 0.292503944236043, -0.0854239970978128, -1.00681184244266, -0.47065363733607, 1.80721498146898, -0.4326427012483, -1.16256240410332, 0.130223527365399, -0.386369509952482, 1.60065297773492, 0.591273376749004, 0.491505313295882, -0.647570852929537, 0.992770927195342, -0.790452750807849, 1.22723883358347, -2.20138782545931, 1.48751069681045, -0.703724126450506, -0.184618630003257, -0.143768351922331, 0.311851530254381, 0.291563145008077, 1.61750294741629, -0.0822444970920521, 0.252172248624306, 0.561619941897011, 0.261244685382425, -0.244456807279551, 0.600111363059314, -0.924359041258171, -0.385429049136948, 0.506631500142572, 0.0302905919757953, 0.642185465899772, -0.507129028351255, -0.822027373452498, 0.0330438720017641, -0.206170957517299, -1.44826543255999, -0.0281932850259887, -0.0785708326113705, -1.54588999156833, -2.08379590183473, -0.525883491994684, 1.6303773186893, 0.422367036362029, 2.00055178609009, -0.98517369554286, 0.079900279255275, -1.1228663416537, 1.70361526176809, -1.21846299649439, 0.177846852323407, 0.486342812339919, -1.05433416850518, 1.16192113378031, -0.662995679441098, -1.32468718297024, 0.120637921776227, -0.223401220887076, 0.045062947267608, -0.468342807663716, 1.09375366381878, 0.0764612951616215, 0.606234635847042, -1.46821367784706, 1.16804970386968, -0.545031471131559, 1.2254904158197, -1.0837327635999, 0.107920209866619, -2.46875068880758, -0.628729248802342, 0.780516250885651, 0.262101825993813, 1.15721552573564, 0.479408498483621, -1.2423514838703, 0.430176119115143, -0.0421242111871756, 1.80064460025449, 0.529771471385742, 0.98547462910855, 0.488327510492237, -0.443086765983, 0.71707187631599, 0.792181187243773, 0.933261782406432, -0.232294234159124, -1.42227357911124, 0.636814537459286, -0.520587068325067, 1.22319299868122, -0.49643606582539, 1.39752374677927, 1.78979005837436, -1.00097170655804, 0.565648363778986, 0.989216878601791, 0.340109855342457, -0.924325043255378, -1.3363390293427, 1.22564671102823, 0.371191271151444, -1.57898493764431, -1.12456780781909, 1.02336977822904 };
double equality_constraint_vector_data[] = { 0.748507645380094, 0.0797376003902225, 0.292543038806435, 0.416568365953175, 0.69978170374644, 0.229094331408131, 0.79898259841103, 0.760793310432207, 0.340752818464625, 0.23103188813534 };

// solutions obtained using Matlab's quadprog():

double solution_data[] = { -1.54947234325176, 0.259545183049741, 0.0374022489950132, 0.77750547958476, -1.25801450662183, -1.88154867343434, 1.70994092666898, 0.625506965990297, 0.931598738873324, -1.19672348208534, -0.288791641313397, 0.522679073901326, 0.441398936352697, 0.122831357359872, 0.429275688092712, 0.0176644499002628, 0.339830789050105, 0.0293209657947787, -0.158787798388674, -0.796153721818046, 2.4967728949651, -1.63762997687845, 1.18080718917784, -1.05738069340969, 0.0585476233545824, 0.141682784117771, 1.2767931989414, 1.33402395090252, -0.348082019097237, -0.215560588105247, 0.0316207413277165, 0.572094599141034, -0.290435979472242, 0.188297485451063, 0.578593102676131, 0.259308649374783, 0.399062481995456, 0.410063542023687, -0.0657370575432304, -1.26854315037399, -0.893961712559284, 0.445506586037936, -0.866067527224422, -0.421977537544534, 0.113013912336643, 0.0582328718205902, 0.612082712005026, 0.428693628945845, 1.16944645740358, 1.65533762500153 };
double solution_no_eq_data[] = { 0.100557011294565, 0.0774609022426853, 0.0511825083173843, -0.166122333116954, 0.577774848029813, 0.167095709499533, -0.140736741762911, 1.05265237884383, 0.12834718765109, 0.365896008127372, 0.111271087791453, 0.0153667809055645, -0.246807732095477, -0.139250571454491, 0.236124574799001, 0.440706860263127, 0.0267377677542438, -0.0526273190040818, -0.127163568297604, 0.00891028293537102, -0.311643286712343, 0.43226631819491, -0.27330761421088, -0.162801497991826, -0.392143423452724, -0.178441846341096, 0.363246926285958, 0.720077504204944, 0.25824605726061, 0.111663453363391, 0.814763535049907, -0.0339642537053691, -0.090647615150187, -0.316651142501127, 0.426403561316129, 0.0209427776415292, 0.175058502358357, 0.235111094823552, 0.0365209820095678, -0.40125271484762, -0.12199466629301, 0.43519716876162, -0.482182232688658, 0.163890165749847, 0.123599044673034, -0.0856983578508494, -0.507111670952678, -0.586611374955362, 0.145341396041547, 0.0243483540569064 };
double solution_zero_data[] = { -0.0909876170384021, 0.0905047474180401, 0.0912146643146082, -0.00500809444016724, -0.125019239792976, -0.112066412721407, 0.0554789408971033, -0.0833658981061983, -0.0238835978767787, -0.105887031589131, 0.0526796937321079, 0.201284689770002, 0.0721764896728619, -0.0379460881753516, 0.0977677810157743, -0.180028575525486, 0.0417179726682485, 0.0931166766325753, 0.00202015958293831, 0.0567422530340581, 0.257207216814311, -0.396613444192145, 0.029171931261875, 0.00366949227676262, 0.102678584640742, 0.0306280784954446, 0.113229368658903, 0.037969242714925, -0.052488725145084, -0.0598411959324047, -0.0563573428596585, 0.0887038150156723, 0.0657076180846572, -0.00960314758180444, 0.113054183104475, 0.0858754522332255, -0.0199266124840442, 0.0975342988386541, -0.0351693406381735, 0.0705218609687235, -0.212481180391147, 0.0566695604659621, -0.0632125543048537, -0.115779248671975, -0.0521777434064297, 0.207261330706659, 0.0500968164510367, 0.0858132816924573, 0.177238850106903, 0.0818369263626538 };
double solution_no_eq_zero_data[] = { -0.00837671700845143, -0.0367253309698206, 0.0104643387030714, 0.0147362006591086, 0.0632112096198801, 0.117805099714431, -0.129446802361635, 0.122637558342626, -0.0601966399354118, 0.087670298742532, -0.00856905269483837, 0.0838674363561341, 0.0262661237496275, -0.0490305030097515, 0.12607905876157, -0.204143795330225, -0.0388869575566157, 0.04199106670216, 0.0827599184173558, 0.0682556268630987, 0.144471387029796, -0.0433512874225062, -0.014567864260988, 0.0440705100077805, -0.00139469146236212, 0.150205929743892, 0.165068704142435, 0.155155878157374, -0.0531636996018179, 0.0240351922956988, 0.0809643514211241, 0.0450743994143347, 0.061090621394066, -0.00945927149864192, -0.0335777005208452, -0.0168225055197412, -0.273859161216313, 0.00821215362299372, -0.0296222639219962, 0.214658905224127, 0.0240887321065743, -0.097566176239216, 0.172403189514966, 0.0101764577015725, -0.0283474768894243, 0.188454428558924, 0.0209083311843356, -0.0866098351808266, -0.0835625666816506, -0.104878298319644 };





void run ()
{

  matrix_type problem_matrix                = Eigen::Map<matrix_type> (problem_matrix_data, num_sig, num_coef);
  matrix_type inequality_constraint_matrix  = Eigen::Map<matrix_type> (inequality_constraint_matrix_data, num_ineq, num_coef);
  matrix_type equality_constraint_matrix    = Eigen::Map<matrix_type> (equality_constraint_matrix_data, num_eq, num_coef);
  vector_type problem_vector                = Eigen::Map<vector_type> (problem_vector_data, num_sig);
  vector_type inequality_constraint_vector  = Eigen::Map<vector_type> (inequality_constraint_vector_data, num_ineq);
  vector_type equality_constraint_vector    = Eigen::Map<vector_type> (equality_constraint_vector_data, num_eq);
  vector_type solution                      = Eigen::Map<vector_type> (solution_data, num_coef);
  vector_type solution_no_eq                = Eigen::Map<vector_type> (solution_no_eq_data, num_coef);
  vector_type solution_zero                 = Eigen::Map<vector_type> (solution_zero_data, num_coef);
  vector_type solution_no_eq_zero           = Eigen::Map<vector_type> (solution_no_eq_zero_data, num_coef);


  {
    vector_type x;
    Math::ICLS::Problem<double> problem (problem_matrix, inequality_constraint_matrix, equality_constraint_matrix, inequality_constraint_vector, equality_constraint_vector);
    Math::ICLS::Solver<double> solve (problem);
    solve (x, problem_vector);
    if (!x.isApprox (solution, 1.0e-6))
      throw Exception ("ICLS solver test failed at test 1");
  }

  {
    vector_type x;
    Math::ICLS::Problem<double> problem (problem_matrix, inequality_constraint_matrix, equality_constraint_matrix);
    Math::ICLS::Solver<double> solve (problem);
    solve (x, problem_vector);
    if (!x.isApprox (solution_zero, 1.0e-6))
      throw Exception ("ICLS solver test failed at test 2");
  }

  {
    vector_type x;
    Math::ICLS::Problem<double> problem (problem_matrix, inequality_constraint_matrix, inequality_constraint_vector);
    Math::ICLS::Solver<double> solve (problem);
    solve (x, problem_vector);
    if (!x.isApprox (solution_no_eq, 1.0e-6))
      throw Exception ("ICLS solver test failed at test 3");
  }

  {
    vector_type x;
    Math::ICLS::Problem<double> problem (problem_matrix, inequality_constraint_matrix);
    Math::ICLS::Solver<double> solve (problem);
    solve (x, problem_vector);
    if (!x.isApprox (solution_no_eq_zero, 1.0e-6))
      throw Exception ("ICLS solver test failed at test 4");
  }

  matrix_type constraint_matrix (inequality_constraint_matrix.rows()+equality_constraint_matrix.rows(), inequality_constraint_matrix.cols());
  constraint_matrix.topRows (inequality_constraint_matrix.rows()) = inequality_constraint_matrix;
  constraint_matrix.bottomRows (equality_constraint_matrix.rows()) = equality_constraint_matrix;

  vector_type constraint_vector (inequality_constraint_vector.rows()+equality_constraint_vector.rows());
  constraint_vector.head (inequality_constraint_vector.rows()) = inequality_constraint_vector;
  constraint_vector.tail (equality_constraint_vector.rows()) = equality_constraint_vector;

  {
    vector_type x;
    Math::ICLS::Problem<double> problem (problem_matrix, constraint_matrix, constraint_vector, equality_constraint_vector.size());
    Math::ICLS::Solver<double> solve (problem);
    solve (x, problem_vector);
    if (!x.isApprox (solution, 1.0e-6))
      throw Exception ("ICLS solver test failed at test 4");
  }




  CONSOLE ("All tests passed OK");

}

