/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* ## Notice
*
* The original C code, long comment, copyright, license, and constants are from [Cephes]{@link http://www.netlib.org/cephes}. The implementation follows the original, but has been modified for JavaScript.
*
* ```text
* Copyright 1985, 1987, 1989, 2000 by Stephen L. Moshier
*
* Some software in this archive may be from the book _Methods and Programs for Mathematical Functions_ (Prentice-Hall or Simon & Schuster International, 1989) or from the Cephes Mathematical Library, a commercial product. In either event, it is copyrighted by the author. What you see here may be used freely but it comes with no support or guarantee.
*
* Stephen L. Moshier
* moshier@na-net.ornl.gov
* ```
*/

'use strict';

// MODULES //

var isnan = require( './../../../../base/assert/is-nan' );
var ln = require( './../../../../base/special/ln' );
var PI_SQUARED = require( '@stdlib/constants/float64/pi-squared' );
var polyvalA = require( './polyval_a.js' );
var polyvalB = require( './polyval_b.js' );


// MAIN //

/**
* Evaluates Spence’s function, which is also known as the dilogarithm.
*
* ## Method
*
* -   A rational approximation gives the integral in the interval (0.5, 1.5).
* -   Transformation formulas for \\( \tfrac{1}{x} \\) and \\( 1 - x \\) are employed outside the basic expansion range.
*
* ## Notes
*
* -   Relative error:
*
*     | arithmetic | domain      | # trials | peak    | rms     |
*     |:----------:|:-----------:|:--------:|:-------:|:-------:|
*     | IEEE       | 0,4         | 30000    | 3.9e-15 | 5.4e-16 |
*
*
* @param {NonNegativeNumber} x - input value
* @returns {number} function value
*
* @example
* var v = spence( 3.0 );
* // returns ~-1.437
*
* @example
* var v = spence( 0.0 );
* // returns ~1.645
*
* @example
* var v = spence( -9.0 );
* // returns NaN
*
* @example
* var v = spence( NaN );
* // returns NaN
*/
function spence( x ) {
	var flg;
	var w;
	var y;
	var z;

	if ( isnan( x ) || x < 0.0 ) {
		return NaN;
	}
	if ( x === 1.0 ) {
		return 0.0;
	}
	if ( x === 0.0 ) {
		return ( PI_SQUARED / 6.0 );
	}
	flg = 0;
	if ( x > 2.0 ) {
		x = 1.0 / x;
		flg |= 2;
	}
	if ( x > 1.5 ) {
		w = (1.0 / x) - 1.0;
		flg |= 2;
	}
	else if ( x < 0.5 ) {
		w = -x;
		flg |= 1;
	}
	else {
		w = x - 1.0;
	}
	y = -w * polyvalA( w ) / polyvalB( w );
	if ( flg & 1 ) {
		y = ( PI_SQUARED/6.0 ) - ( ln( x ) * ln( 1.0-x ) ) - y;
	}
	if ( flg & 2 ) {
		z = ln( x );
		y = -( 0.5 * z * z ) - y;
	}
	return y;
}


// EXPORTS //

module.exports = spence;
