% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/memoise.R
\name{memoise}
\alias{memoise}
\alias{memoize}
\title{Memoise a function}
\usage{
memoise(
  f,
  ...,
  envir = environment(f),
  cache = cachem::cache_mem(max_size = 1024 * 1024^2),
  omit_args = c(),
  hash = function(x) rlang::hash(x)
)
}
\arguments{
\item{f}{Function of which to create a memoised copy.}

\item{...}{optional variables to use as additional restrictions on
caching, specified as one-sided formulas (no LHS). See Examples for usage.}

\item{envir}{Environment of the returned function.}

\item{cache}{Cache object. The default is a [cachem::cache_mem()] with a max
size of 1024 MB.}

\item{omit_args}{Names of arguments to ignore when calculating hash.}

\item{hash}{A function which takes an R object as input and returns a string
which is used as a cache key.}
}
\description{
\code{mf <- memoise(f)} creates \code{mf}, a memoised copy of
\code{f}. A memoised copy is basically a
lazier version of the same function: it saves the answers of
new invocations, and re-uses the answers of old ones. Under the right
circumstances, this can provide a very nice speedup indeed.
}
\details{
There are two main ways to use the \code{memoise} function. Say that
you wish to memoise \code{glm}, which is in the \code{stats}
package; then you could use \cr
  \code{  mem_glm <- memoise(glm)}, or you could use\cr
  \code{  glm <- memoise(stats::glm)}. \cr
The first form has the advantage that you still have easy access to
both the memoised and the original function. The latter is especially
useful to bring the benefits of memoisation to an existing block
of R code.

Two example situations where \code{memoise} could be of use:
\itemize{
  \item You're evaluating a function repeatedly over the rows (or
    larger chunks) of a dataset, and expect to regularly get the same
    input.
  \item You're debugging or developing something, which involves
    a lot of re-running the code.  If there are a few expensive calls
    in there, memoising them can make life a lot more pleasant.
    If the code is in a script file that you're \code{source()}ing,
    take care that you don't just put \cr
      \code{  glm <- memoise(stats::glm)} \cr
    at the top of your file: that would reinitialise the memoised
    function every time the file was sourced. Wrap it in \cr
      \code{  if (!is.memoised(glm)) }, or do the memoisation call
    once at the R prompt, or put it somewhere else where it won't get
    repeated.
}

It is recommended that functions in a package are not memoised at build-time,
but when the package is loaded. The simplest way to do this is within
\code{.onLoad()} with, for example


\preformatted{
# file.R
fun <- function() {
 some_expensive_process()
}

# zzz.R
.onLoad <- function(libname, pkgname) {
 fun <<- memoise::memoise(fun)
}
}
}
\examples{
# a() is evaluated anew each time. memA() is only re-evaluated
# when you call it with a new set of parameters.
a <- function(n) { runif(n) }
memA <- memoise(a)
replicate(5, a(2))
replicate(5, memA(2))

# Caching is done based on parameters' value, so same-name-but-
# changed-value correctly produces two different outcomes...
N <- 4; memA(N)
N <- 5; memA(N)
# ... and same-value-but-different-name correctly produces
#     the same cached outcome.
N <- 4; memA(N)
N2 <- 4; memA(N2)

# memoise() knows about default parameters.
b <- function(n, dummy="a") { runif(n) }
memB <- memoise(b)
memB(2)
memB(2, dummy="a")
# This works, because the interface of the memoised function is the same as
# that of the original function.
formals(b)
formals(memB)
# However, it doesn't know about parameter relevance.
# Different call means different caching, no matter
# that the outcome is the same.
memB(2, dummy="b")

# You can create multiple memoisations of the same function,
# and they'll be independent.
memA(2)
memA2 <- memoise(a)
memA(2)  # Still the same outcome
memA2(2) # Different cache, different outcome

# Multiple memoized functions can share a cache.
cm <- cachem::cache_mem(max_size = 50 * 1024^2)
memA <- memoise(a, cache = cm)
memB <- memoise(b, cache = cm)

# Don't do the same memoisation assignment twice: a brand-new
# memoised function also means a brand-new cache, and *that*
# you could as easily and more legibly achieve using forget().
# (If you're not sure whether you already memoised something,
#  use is.memoised() to check.)
memA(2)
memA <- memoise(a)
memA(2)

# Make a memoized result automatically time out after 10 seconds.
memA3 <- memoise(a, cache = cachem::cache_mem(max_age = 10))
memA3(2)
}
\seealso{
\code{\link{forget}}, \code{\link{is.memoised}},
  \code{\link{timeout}}, \url{https://en.wikipedia.org/wiki/Memoization},
  \code{\link{drop_cache}}
}
