use ui;
use layout;
use geometry;

/**
 * A slide in a presentation.
 *
 * Contains a number of elements laid out according to the current layout.
 *
 * A slide is scaled to occupy 1024x768 virtual pixels regardless of the actual size of the output.
 */
class Slide on Render {
	// Intro animation.
	SlideIntro? intro;

	// The layout for this slide.
	private Layout layout;

	// All elements in this slide.
	private Element[] elements;

	// Length of animations in this slide.
	private Duration[] animationTimes;

	// Create, providing a layout. Traverses the layout to find all elements.
	init(Layout layout) {
		init() {
			layout = layout;
		}

		// The intro always exists, even if we don't have a specific animation for it.
		animationTimes << 0 ms;
		layout.findAll(&this.addElement);
	}

	// Cleanup. Called before the presentation is done being displayed.
	void cleanup() {
		for (element in elements)
			element.cleanup();
	}

	// Called when we're added to the presentation so that we get a chance to perform layout.
	package void added(Presentation to) {
		layout.pos(Rect(to.size));

		if (intro) {
			if (intro.duration == 0 ms) {
				intro.duration = to.animationTime;
			}
		}

		for (e in elements) {
			e.setupAnimations(to);
			for (a in e.animations) {
				addAnimation(a);
			}
		}
	}

	// Called when a new repaint callback is known. Only called when shown interactively.
	package void setupRepaint(Fn<void> callback) {
		for (e in elements)
			e.setupRepaint(callback);
	}

	// Draw this slide. 'step' is the animation number currently being played and 'time' determines
	// the length of the current animation.
	void draw(Graphics g, Nat step, Duration time) {
		for (element in elements) {
			element.draw(g, step, time);
		}
	}

	// Get the total number of animations in this slide.
	Nat stepCount() {
		animationTimes.count;
	}

	// Get the total animation duration for a certain animation step.
	Duration duration(Nat step) {
		if (step < animationTimes.count)
			animationTimes[step];
		else
			0 ms;
	}

	// Get visible element at a particular coordinate.
	Element? elementAt(Point pos) {
		for (Nat i = elements.count; i > 0; i--) {
			Element elem = elements[i - 1];
			if (elem.visible)
				if (elem.pos.contains(pos))
					return elem;
		}

		return null;
	}

private:
	// Add an element if we know about it.
	void addElement(Component c) {
		if (c as ElementWrap) {
			elements << c.control;
		}
	}

	// Add an animation.
	void addAnimation(Animation a) {
		Nat step = a.step;
		while (animationTimes.count <= step)
			animationTimes << 0 ms;

		animationTimes[step] = max(animationTimes[step], a.duration + a.offset);
	}
}
