/* SPDX-License-Identifier: BSD-2-Clause */
/*******************************************************************************
 * Copyright 2018-2019, Fraunhofer SIT sponsored by Infineon Technologies AG
 * All rights reserved.
 ******************************************************************************/
#ifndef FAPI_CERTIFICATES_H
#define FAPI_CERTIFICATES_H

static char * root_cert_list[] = {
    /* IFX RSA root certificate */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIFqzCCA5OgAwIBAgIBAzANBgkqhkiG9w0BAQsFADB3MQswCQYDVQQGEwJERTEh\n"
    "MB8GA1UECgwYSW5maW5lb24gVGVjaG5vbG9naWVzIEFHMRswGQYDVQQLDBJPUFRJ\n"
    "R0EoVE0pIERldmljZXMxKDAmBgNVBAMMH0luZmluZW9uIE9QVElHQShUTSkgUlNB\n"
    "IFJvb3QgQ0EwHhcNMTMwNzI2MDAwMDAwWhcNNDMwNzI1MjM1OTU5WjB3MQswCQYD\n"
    "VQQGEwJERTEhMB8GA1UECgwYSW5maW5lb24gVGVjaG5vbG9naWVzIEFHMRswGQYD\n"
    "VQQLDBJPUFRJR0EoVE0pIERldmljZXMxKDAmBgNVBAMMH0luZmluZW9uIE9QVElH\n"
    "QShUTSkgUlNBIFJvb3QgQ0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC\n"
    "AQC7E+gc0B5T7awzux66zMMZMTtCkPqGv6a3NVx73ICg2DSwnipFwBiUl9soEodn\n"
    "25SVVN7pqmvKA2gMTR5QexuYS9PPerfRZrBY00xyFx84V+mIRPg4YqUMLtZBcAwr\n"
    "R3GO6cffHp20SBH5ITpuqKciwb0v5ueLdtZHYRPq1+jgy58IFY/vACyF/ccWZxUS\n"
    "JRNSe4ruwBgI7NMWicxiiWQmz1fE3e0mUGQ1tu4M6MpZPxTZxWzN0mMz9noj1oIT\n"
    "ZUnq/drN54LHzX45l+2b14f5FkvtcXxJ7OCkI7lmWIt8s5fE4HhixEgsR2RX5hzl\n"
    "8XiHiS7uD3pQhBYSBN5IBbVWREex1IUat5eAOb9AXjnZ7ivxJKiY/BkOmrNgN8k2\n"
    "7vOS4P81ix1GnXsjyHJ6mOtWRC9UHfvJcvM3U9tuU+3dRfib03NGxSPnKteL4SP1\n"
    "bdHfiGjV3LIxzFHOfdjM2cvFJ6jXg5hwXCFSdsQm5e2BfT3dWDBSfR4h3Prpkl6d\n"
    "cAyb3nNtMK3HR5yl6QBuJybw8afHT3KRbwvOHOCR0ZVJTszclEPcM3NQdwFlhqLS\n"
    "ghIflaKSPv9yHTKeg2AB5q9JSG2nwSTrjDKRab225+zJ0yylH5NwxIBLaVHDyAEu\n"
    "81af+wnm99oqgvJuDKSQGyLf6sCeuy81wQYO46yNa+xJwQIDAQABo0IwQDAdBgNV\n"
    "HQ4EFgQU3LtWq/EY/KaadREQZYQSntVBkrkwDgYDVR0PAQH/BAQDAgAGMA8GA1Ud\n"
    "EwEB/wQFMAMBAf8wDQYJKoZIhvcNAQELBQADggIBAGHTBUx3ETIXYJsaAgb2pyyN\n"
    "UltVL2bKzGMVSsnTCrXUU8hKrDQh3jNIMrS0d6dU/fGaGJvehxmmJfjaN/IFWA4M\n"
    "BdZEnpAe2fJEP8vbLa/QHVfsAVuotLD6QWAqeaC2txpxkerveoV2JAwj1jrprT4y\n"
    "rkS8SxZuKS05rYdlG30GjOKTq81amQtGf2NlNiM0lBB/SKTt0Uv5TK0jIWbz2WoZ\n"
    "gGut7mF0md1rHRauWRcoHQdxWSQTCTtgoQzeBj4IS6N3QxQBKV9LL9UWm+CMIT7Y\n"
    "np8bSJ8oW4UdpSuYWe1ZwSjZyzDiSzpuc4gTS6aHfMmEfoVwC8HN03/HD6B1Lwo2\n"
    "DvEaqAxkya9IYWrDqkMrEErJO6cqx/vfIcfY/8JYmUJGTmvVlaODJTwYwov/2rjr\n"
    "la5gR+xrTM7dq8bZimSQTO8h6cdL6u+3c8mGriCQkNZIZEac/Gdn+KwydaOZIcnf\n"
    "Rdp3SalxsSp6cWwJGE4wpYKB2ClM2QF3yNQoTGNwMlpsxnU72ihDi/RxyaRTz9OR\n"
    "pubNq8Wuq7jQUs5U00ryrMCZog1cxLzyfZwwCYh6O2CmbvMoydHNy5CU3ygxaLWv\n"
    "JpgZVHN103npVMR3mLNa3QE+5MFlBlP3Mmystu8iVAKJas39VO5y5jad4dRLkwtM\n"
    "6sJa8iBpdRjZrBp5sJBI\n"
    "-----END CERTIFICATE-----\n",

    /* IFX ECC root certificate */
    "-----BEGIN CERTIFICATE-----\n"
    "MIICWzCCAeKgAwIBAgIBBDAKBggqhkjOPQQDAzB3MQswCQYDVQQGEwJERTEhMB8G\n"
    "A1UECgwYSW5maW5lb24gVGVjaG5vbG9naWVzIEFHMRswGQYDVQQLDBJPUFRJR0Eo\n"
    "VE0pIERldmljZXMxKDAmBgNVBAMMH0luZmluZW9uIE9QVElHQShUTSkgRUNDIFJv\n"
    "b3QgQ0EwHhcNMTMwNzI2MDAwMDAwWhcNNDMwNzI1MjM1OTU5WjB3MQswCQYDVQQG\n"
    "EwJERTEhMB8GA1UECgwYSW5maW5lb24gVGVjaG5vbG9naWVzIEFHMRswGQYDVQQL\n"
    "DBJPUFRJR0EoVE0pIERldmljZXMxKDAmBgNVBAMMH0luZmluZW9uIE9QVElHQShU\n"
    "TSkgRUNDIFJvb3QgQ0EwdjAQBgcqhkjOPQIBBgUrgQQAIgNiAAQm1HxLVgvAu1q2\n"
    "GM+ymTz12zdTEu0JBVG9CdsVEJv/pE7pSWOlsG3YwU792YAvjSy7zL+WtDK40KGe\n"
    "Om8bSWt46QJ00MQUkYxz6YqXbb14BBr06hWD6u6IMBupNkPd9pKjQjBAMB0GA1Ud\n"
    "DgQWBBS0GIXISkrFEnryQDnexPWLHn5K0TAOBgNVHQ8BAf8EBAMCAAYwDwYDVR0T\n"
    "AQH/BAUwAwEB/zAKBggqhkjOPQQDAwNnADBkAjA6QZcV8DjjbPuKjKDZQmTRywZk\n"
    "MAn8wE6kuW3EouVvBt+/2O+szxMe4vxj8R6TDCYCMG7c9ov86ll/jDlJb/q0L4G+\n"
    "+O3Bdel9P5+cOgzIGANkOPEzBQM3VfJegfnriT/kaA==\n"
    "-----END CERTIFICATE-----\n",

    /* Intel root certificate */
    "-----BEGIN CERTIFICATE-----\n"
    "MIICdzCCAh6gAwIBAgIUB+dPf7a3IyJGO923z34oQLRP7pwwCgYIKoZIzj0EAwIw\n"
    "gYcxCzAJBgNVBAYMAlVTMQswCQYDVQQIDAJDQTEUMBIGA1UEBwwLU2FudGEgQ2xh\n"
    "cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMSEwHwYDVQQLDBhUUE0gRUsg\n"
    "cm9vdCBjZXJ0IHNpZ25pbmcxFjAUBgNVBAMMDXd3dy5pbnRlbC5jb20wHhcNMTQw\n"
    "MTE1MDAwMDAwWhcNNDkxMjMxMjM1OTU5WjCBhzELMAkGA1UEBgwCVVMxCzAJBgNV\n"
    "BAgMAkNBMRQwEgYDVQQHDAtTYW50YSBDbGFyYTEaMBgGA1UECgwRSW50ZWwgQ29y\n"
    "cG9yYXRpb24xITAfBgNVBAsMGFRQTSBFSyByb290IGNlcnQgc2lnbmluZzEWMBQG\n"
    "A1UEAwwNd3d3LmludGVsLmNvbTBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABJR9\n"
    "gVEsjUrMb+E/dl19ywJsKZDnghmwVyG16dAfQ0Pftp1bjhtPEGEguvbLGRRopKWH\n"
    "VscAOlTFnvCHq+6/9/SjZjBkMB8GA1UdIwQYMBaAFOhSBcJP2NLVpSFHFrbODHtb\n"
    "uncPMB0GA1UdDgQWBBToUgXCT9jS1aUhRxa2zgx7W7p3DzASBgNVHRMBAf8ECDAG\n"
    "AQH/AgEBMA4GA1UdDwEB/wQEAwIBBjAKBggqhkjOPQQDAgNHADBEAiAldFScWQ6L\n"
    "PQgW/YT+2GILcATEA2TgzASaCrG+AzL6FgIgLH8ABRzm028hRYR/JZVGkHiomzYX\n"
    "VILmTjHwSL7uZBU=\n"
    "-----END CERTIFICATE-----",

    /* STM RSA root certificate */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIEDDCCAvSgAwIBAgILBAAAAAABIsFs834wDQYJKoZIhvcNAQELBQAwgYcxOzA5\n"
    "BgNVBAsTMkdsb2JhbFNpZ24gVHJ1c3RlZCBDb21wdXRpbmcgQ2VydGlmaWNhdGUg\n"
    "QXV0aG9yaXR5MRMwEQYDVQQKEwpHbG9iYWxTaWduMTMwMQYDVQQDEypHbG9iYWxT\n"
    "aWduIFRydXN0ZWQgUGxhdGZvcm0gTW9kdWxlIFJvb3QgQ0EwHhcNMDkwNzI4MTIw\n"
    "MDAwWhcNMzkxMjMxMjM1OTU5WjBKMQswCQYDVQQGEwJDSDEeMBwGA1UEChMVU1RN\n"
    "aWNyb2VsZWN0cm9uaWNzIE5WMRswGQYDVQQDExJTVE0gVFBNIEVLIFJvb3QgQ0Ew\n"
    "ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDxBLG5wcB9J0MsiJMreoWQ\n"
    "l21bBN12SSGZPJ3HoPjzcrzAz6SPy+TrFmZ6eUVspsFL/23wdPprqTUtDHi+C2pw\n"
    "k/3dF3/Rb2t/yHgiPlbCshYpi5f/rJ7nzbQ1ca2LzX3saBe53VfNQQV0zd5uM0DT\n"
    "SrmAKU1RIAj2WlZFWXoN4NWTyRtqT5suPHa2y8FlCWMZKlS0FiY4pfM20b5YQ+EL\n"
    "4zqb9zN53u/TdYZegrfSlc30Nl9G13Mgi+8rtPFKwsxx05EBbhVroH7aKVI1djsf\n"
    "E1MVrUzw62PHik3xlzznXML8OjY//xKeiCWcsApuGCaIAf7TsTRi2l8DNB3rCr1X\n"
    "AgMBAAGjgbQwgbEwDgYDVR0PAQH/BAQDAgIEMBIGA1UdEwEB/wQIMAYBAf8CAQEw\n"
    "HQYDVR0OBBYEFG/mxWwHt2yLCoGSg1zLQR72jtEnMEsGA1UdIAREMEIwQAYJKwYB\n"
    "BAGgMgFaMDMwMQYIKwYBBQUHAgEWJWh0dHA6Ly93d3cuZ2xvYmFsc2lnbi5uZXQv\n"
    "cmVwb3NpdG9yeS8wHwYDVR0jBBgwFoAUHiNj8IW19iVO7RrAUL5lfMfUFXowDQYJ\n"
    "KoZIhvcNAQELBQADggEBAFrKpwFmRh7BGdpPZWc1Y6wIbdTAF6T+q1KwDJcyAjgJ\n"
    "qThFp3xTAt3tvyVrCRf7T/YARYE24DNa0iFaXsIXeQASDYHJjAZ6LQTslYBeRYLb\n"
    "C9v8ZE2ocKSCiC8ALYlJWk39Wob0H1Lk6l2zcUo3oKczGiAcRrlmwV496wvGyted\n"
    "2RBcLZro7yhOOGr9KMabV14fNl0lG+31J1nWI2hgTqh53GXg1QH2YpggD3b7UbVm\n"
    "c6GZaX37N3z15XfQafuAfHt10kYCNdePzC9tOwirHIsO8lrxoNlzOSxX8SqQGbBI\n"
    "+kWoe5+SY3gdOGGDQKIdw3W1poMN8bQ5x7XFcgVMwVU=\n"
    "-----END CERTIFICATE-----\n",

    /* STM ECC root certificate */
    "-----BEGIN CERTIFICATE-----\n"
    "MIICyDCCAk+gAwIBAgIORyzLp/OdsAvb9r+66LowCgYIKoZIzj0EAwMwgYsxOzA5\n"
    "BgNVBAsTMkdsb2JhbFNpZ24gVHJ1c3RlZCBDb21wdXRpbmcgQ2VydGlmaWNhdGUg\n"
    "QXV0aG9yaXR5MRMwEQYDVQQKEwpHbG9iYWxTaWduMTcwNQYDVQQDEy5HbG9iYWxT\n"
    "aWduIFRydXN0ZWQgUGxhdGZvcm0gTW9kdWxlIEVDQyBSb290IENBMB4XDTE1MTAy\n"
    "ODAwMDAwMFoXDTM4MDExOTAzMTQwN1owTjELMAkGA1UEBhMCQ0gxHjAcBgNVBAoT\n"
    "FVNUTWljcm9lbGVjdHJvbmljcyBOVjEfMB0GA1UEAxMWU1RNIFRQTSBFQ0MgUm9v\n"
    "dCBDQSAwMTB2MBAGByqGSM49AgEGBSuBBAAiA2IABG7/OLXMiprQQHwNnkpT6aqG\n"
    "zOGLcbbAgUtyjlXOZtuv0GB0ttJ6fwMwgFtt8RKlko8Bwn89/BoZOUcI4ne8ddRS\n"
    "oqE6StnU3I13qqjalToq3Rnz61Omn6NErK1pxUe3j6OBtTCBsjAOBgNVHQ8BAf8E\n"
    "BAMCAgQwEgYDVR0TAQH/BAgwBgEB/wIBATAdBgNVHQ4EFgQUIJJWPAtDqAVyUwMp\n"
    "BxwH4OvsAwQwHwYDVR0jBBgwFoAUYT78EZkKf7CpW5CgJl4pYUe3MAMwTAYDVR0g\n"
    "BEUwQzBBBgkrBgEEAaAyAVowNDAyBggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xv\n"
    "YmFsc2lnbi5jb20vcmVwb3NpdG9yeS8wCgYIKoZIzj0EAwMDZwAwZAIwWnuUAzwy\n"
    "vHUhHehymKTZ2QcPUwHX0LdcVTac4ohyEL3zcuv/dM0BN62kFxHgBOhWAjAIxt9i\n"
    "50yAxy0Z/MeV2NTXqKpLwdhWNuzOSFZnzRKsh9MxY3zj8nebDNlHTDGSMR0=\n"
    "-----END CERTIFICATE-----",

    /* NTC_TPM_EK_Root_CA_01 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDSjCCAjKgAwIBAgIGAK3jXfbVMA0GCSqGSIb3DQEBBQUAMFIxUDAcBgNVBAMT\n"
    "FU5UQyBUUE0gRUsgUm9vdCBDQSAwMTAlBgNVBAoTHk51dm90b24gVGVjaG5vbG9n\n"
    "eSBDb3Jwb3JhdGlvbjAJBgNVBAYTAlRXMB4XDTEyMDcxMTE2MjkzMFoXDTMyMDcx\n"
    "MTE2MjkzMFowUjFQMBwGA1UEAxMVTlRDIFRQTSBFSyBSb290IENBIDAxMCUGA1UE\n"
    "ChMeTnV2b3RvbiBUZWNobm9sb2d5IENvcnBvcmF0aW9uMAkGA1UEBhMCVFcwggEi\n"
    "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDoNqxhtD4yUtXhqKQGGZemoKJy\n"
    "uj1RnWvmNgzItLeejNU8B6fOnpMQyoS4K72tMhhFRK2jV9RYzyJMSjEwyX0ASTO1\n"
    "2yMti2UJQS60d36eGwk8WLgrFnnITlemshi01h9t1MOmay3TO1LLH/3/VDKJ+jbd\n"
    "cbfIO2bBquN8r3/ojYUaNSPj6pK1mmsMoJXF4dGRSEwb/4ozBIw5dugm1MEq4Zj3\n"
    "GZ0YPg5wyLRugQbt7DkUOX4FGuK5p/C0u5zX8u33EGTrDrRz3ye3zO+aAY1xXF/m\n"
    "qwEqgxX5M8f0/DXTTO/CfeIksuPeOzujFtXfi5Cy64eeIZ0nAUG3jbtnGjoFAgMB\n"
    "AAGjJjAkMA4GA1UdDwEB/wQEAwICBDASBgNVHRMBAf8ECDAGAQH/AgEAMA0GCSqG\n"
    "SIb3DQEBBQUAA4IBAQBBQznOPJAsD4Yvyt/hXtVJSgBX/+rRfoaqbdt3UMbUPJYi\n"
    "pUoTUgaTx02DVRwommO+hLx7CS++1F2zorWC8qQyvNbg7iffQbbjWitt8NPE6kCr\n"
    "q0Y5g7M/LkQDd5N3cFfC15uFJOtlj+A2DGzir8dlXU/0qNq9dBFbi+y+Y3rAT+wK\n"
    "fktmN82UT861wTUzDvnXO+v7H5DYXjUU8kejPW6q+GgsccIbVTOdHNNWbMrcD9yf\n"
    "oS91nMZ/+/n7IfFWXNN82qERsrvOFCDsbIzUOR30N0IP++oqGfwAbKFfCOCFUz6j\n"
    "jpXUdJlh22tp12UMsreibmi5bsWYBgybwSbRgvzE\n"
    "-----END CERTIFICATE-----\n",

    /* NTC_TPM_EK_Root_CA_02 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDSjCCAjKgAwIBAgIGAPadBmPZMA0GCSqGSIb3DQEBBQUAMFIxUDAcBgNVBAMT\n"
    "FU5UQyBUUE0gRUsgUm9vdCBDQSAwMjAlBgNVBAoTHk51dm90b24gVGVjaG5vbG9n\n"
    "eSBDb3Jwb3JhdGlvbjAJBgNVBAYTAlRXMB4XDTEyMDcxMTE2MzMyNFoXDTMyMDcx\n"
    "MTE2MzMyNFowUjFQMBwGA1UEAxMVTlRDIFRQTSBFSyBSb290IENBIDAyMCUGA1UE\n"
    "ChMeTnV2b3RvbiBUZWNobm9sb2d5IENvcnBvcmF0aW9uMAkGA1UEBhMCVFcwggEi\n"
    "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDSagWxaANT1YA2YUSN7sq7yzOT\n"
    "1ymbIM+WijhE5AGcLwLFoJ9fmaQrYL6fAW2EW/Q3yu97Q9Ysr8yYZ2XCCfxfseEr\n"
    "Vs80an8Nk6LkTDz8+0Hm0Cct0klvNUAZEIvWpmgHZMvGijXyOcp4z494d8B28Ynb\n"
    "I7x0JMXZZQQKQi+WfuHtntF+2osYScweocipPrGeONLKU9sngWZ2vnnvw1SBneTa\n"
    "irxq0Q0SD6Bx9jtxvdf87euk8JzfPhX8jp8GEeAjmLwGR+tnOQrDmczGNmp7YYNN\n"
    "R+Q7NZVoYWHw5jaoZnNxbouWUXZZxFqDsB/ndCKWtsIzRYPuWcqrFcmUN4SVAgMB\n"
    "AAGjJjAkMA4GA1UdDwEB/wQEAwICBDASBgNVHRMBAf8ECDAGAQH/AgEAMA0GCSqG\n"
    "SIb3DQEBBQUAA4IBAQAIkdDSErzPLPYrVthw4lKjW4tRYelUicMPEHKjQeVUAAS5\n"
    "y9XTzB4DWISDAFsgtQjqHJj0xCG+vpY0Rmn2FCO/0YpP+YBQkdbJOsiyXCdFy9e4\n"
    "gGjQ24gw1B+rr84+pkI51y952NYBdoQDeb7diPe+24U94f//DYt/JQ8cJua4alr3\n"
    "2Pohhh5TxCXXfU2EHt67KyqBSxCSy9m4OkCOGLHL2X5nQIdXVj178mw6DSAwyhwR\n"
    "n3uJo5MvUEoQTFZJKGSXfab619mIgzEr+YHsIQToqf44VfDMDdM+MFiXQ3a5fLii\n"
    "hEKQ9DhBPtpHAbhFA4jhCiG9HA8FdEplJ+M4uxNz\n"
    "-----END CERTIFICATE-----\n",

    /* NTC_TPM_EK_Root_CA_ARSUF_01 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDWTCCAkGgAwIBAgIJAMklAEG4bgQ6MA0GCSqGSIb3DQEBBQUAMFgxVjAiBgNV\n"
    "BAMTG05UQyBUUE0gRUsgUm9vdCBDQSBBUlNVRiAwMTAlBgNVBAoTHk51dm90b24g\n"
    "VGVjaG5vbG9neSBDb3Jwb3JhdGlvbjAJBgNVBAYTAlRXMB4XDTE0MDQwMTE4MzQz\n"
    "OFoXDTM0MDMyODE4MzQzOFowWDFWMCIGA1UEAxMbTlRDIFRQTSBFSyBSb290IENB\n"
    "IEFSU1VGIDAxMCUGA1UEChMeTnV2b3RvbiBUZWNobm9sb2d5IENvcnBvcmF0aW9u\n"
    "MAkGA1UEBhMCVFcwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDCxcfP\n"
    "yaNsGhaR28qqisqkrb4Z2OPul7BRNlIEYP8jSFORygyfp4j7bKRyVTTONCUbPq+J\n"
    "/a4yRcdbEs8dzvzXypQbVUjuC4sOKjPiWLfOhj1Z1yvOn19Xe3Ei4UzMKJm+xpb1\n"
    "BYR4YfrnuVzL4do/B/lCr2AYs4Fmtn1uzXBp1St8TRJz9HTW1yKJ2ZOqTgW3DX80\n"
    "6DP//3kIatTuLCZ6Zsdl6fsgMPxJGwrI35ThKBtaUMT93abb/KB/dugvoIgtEi9D\n"
    "GEC2C0UWsvJEfu0Qi8zoxtYvd9Y2tRlMxMhK75uShXHxRcG+WOGEnm6uVpGphLKg\n"
    "qxAl1tuFcb94vi7dAgMBAAGjJjAkMA4GA1UdDwEB/wQEAwICBDASBgNVHRMBAf8E\n"
    "CDAGAQH/AgEAMA0GCSqGSIb3DQEBBQUAA4IBAQB7epeKy2Sa+9huMzK4PnIpjiaX\n"
    "QrxPx+E8BVGw6VuQqTcTPQRvPhNpc4VF/6/7MA9qb6vDxWEf40tqNi8e/RPNlRFY\n"
    "Dh4tQ1Hhl69NrZVYZeXl1cU/ometoAAbz79ugq78iFndJ5rHMQ85GRwtW9i/q0p1\n"
    "VjJ8dLYJ7aRBDTP3hndc35GmZg3q1UX93WD6mM5KuE+mOdv7MXKMtYSrV+dE/iGM\n"
    "ASrratJf57P6N8BpegPQaSb6UnElwBpwhRxzW7N9qgjQWIqrxe97CfJk41RvtnKu\n"
    "SePqlm1PtWkygt9bYaInLZYkcknXTD/7BtzAyyS25HtG/YTvuMtKItCp7Z4n\n"
    "-----END CERTIFICATE-----\n",

    /* Nuvoton TPM Root CA 1013 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDkjCCAnqgAwIBAgIISN0JfIK6vE0wDQYJKoZIhvcNAQEFBQAwVTFTMB8GA1UE\n"
    "AxMYTnV2b3RvbiBUUE0gUm9vdCBDQSAxMDEzMCUGA1UEChMeTnV2b3RvbiBUZWNo\n"
    "bm9sb2d5IENvcnBvcmF0aW9uMAkGA1UEBhMCVFcwHhcNMTUwNTExMDg0MzI1WhcN\n"
    "MzUwNTA3MDg0MzI1WjBVMVMwHwYDVQQDExhOdXZvdG9uIFRQTSBSb290IENBIDEw\n"
    "MTMwJQYDVQQKEx5OdXZvdG9uIFRlY2hub2xvZ3kgQ29ycG9yYXRpb24wCQYDVQQG\n"
    "EwJUVzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALDAta6EZBlhF1MC\n"
    "Z9GeMXqw8puwZEDI3qR/rwGhEUj2oqhFY/K9zUk2YQCkC6X5lrr/lbWfvZtUGMFC\n"
    "P4VQlt+bGPTOladGg6zJ/7a6yCd9MqkZbw92niDNhWcXsiB7SRyHYdr/He8tNOoD\n"
    "mVdNFXxknP8QH3soBPahxckqtrhhk+24Iran04jOAc0959VnP8H0Jyg4BjehIQjj\n"
    "BGGK+bJWZXHYRFlDj4dRW+epChdOqTpWOulf5GOvwNm3sv4ojU2fJ8cA5TznX81z\n"
    "+Se6hmw/RF8rUGjf1uiKbsxnbIf3An01mZYgD98FXEHAWAW92vAJUuEQJVBlTest\n"
    "1YmsaT0CAwEAAaNmMGQwDgYDVR0PAQH/BAQDAgIEMBIGA1UdEwEB/wQIMAYBAf8C\n"
    "AQAwHwYDVR0jBBgwFoAUoNc3KQ4WzyrivucQDPVrLwTF8EMwHQYDVR0OBBYEFKDX\n"
    "NykOFs8q4r7nEAz1ay8ExfBDMA0GCSqGSIb3DQEBBQUAA4IBAQCOXMzQYz3vr9tg\n"
    "SiFr6qha2+Jay+EK0iLjSKvNzcTv5yaO8I6pb7tdocvze8394PtM42d2OTOM99lJ\n"
    "bZogquaJ6fLHMwzO7UEGndtm6YMp6APXk4ecRqUDLqofIWL6PQUVwSEYlAC6RM9k\n"
    "n4MJqckIxsc6iC38lsjyn4ut8o/E3fIo8UzYDl2P+KK1VkjDcmmgNf6seHmBsOYC\n"
    "vOc4xYpq0yWuZFfxeyC4wC4mOAKLZX2yLMYrYBmnDd60nc0hgI1/TKb1H/Ew2P7R\n"
    "UxEDMGe8e3A9YR4M/09FLn8cTTjq7hflRlcqiarpPo6+9Z3dqzmqTQxvVQ/DIVqE\n"
    "3r3WOnnr\n"
    "-----END CERTIFICATE-----\n",

    /* Nuvoton TPM Root CA 1014 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDkjCCAnqgAwIBAgIIbXSeZ/22esUwDQYJKoZIhvcNAQEFBQAwVTFTMB8GA1UE\n"
    "AxMYTnV2b3RvbiBUUE0gUm9vdCBDQSAxMDE0MCUGA1UEChMeTnV2b3RvbiBUZWNo\n"
    "bm9sb2d5IENvcnBvcmF0aW9uMAkGA1UEBhMCVFcwHhcNMTcwODIxMTEwMzM2WhcN\n"
    "MzcwODE3MTEwMzM2WjBVMVMwHwYDVQQDExhOdXZvdG9uIFRQTSBSb290IENBIDEw\n"
    "MTQwJQYDVQQKEx5OdXZvdG9uIFRlY2hub2xvZ3kgQ29ycG9yYXRpb24wCQYDVQQG\n"
    "EwJUVzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMd3U7eh2gF2h7LE\n"
    "b6GZTP/TtWq3cK8QMFuGI6PrwXjwGm1JfG6vF+1u/z4tJbc8znK8Kc/XTZ1yqoZW\n"
    "sz7dXP20eltU0fuB7qwa2yJOqbl6hZ6nAD+eOuteSUQ8AnEuFH2XqqN++pj9DJXS\n"
    "zDpvHqP1J6PnpNi6gxVas6PBFrRZuYGOxJwG2miyObHK1/kKrkuE7/yh+ju1j0Rw\n"
    "9DXhvmEelrztPOkm4TkLxkTQSL7jj+EWimpmGP8Wo+B/2FdVpJZdoFUa2v5GldK8\n"
    "VBdkacDaSqQ0iX1tP2oI93Jyx1xu9JDqjThrtu+FFIlHFJLw2N91fSCmsJukUiCI\n"
    "9PPakQUCAwEAAaNmMGQwDgYDVR0PAQH/BAQDAgIEMBIGA1UdEwEB/wQIMAYBAf8C\n"
    "AQAwHwYDVR0jBBgwFoAUqdlH8/mBcu623MXtYJoA3nwGnLAwHQYDVR0OBBYEFKnZ\n"
    "R/P5gXLuttzF7WCaAN58BpywMA0GCSqGSIb3DQEBBQUAA4IBAQBCVMox72wTII5D\n"
    "fjznST8qr61I6ctKouW401l/x5EA2gBctsx5gl3tuq9EArM2TuA9F+leyJ8r/Tdu\n"
    "do8TgM4iCVl+FthXlrkPVemd1le63v0Emaxd/8Vca9ZZbrLlC6WNiUj/nGyYThPK\n"
    "3XIHQN0+OoNdXQM5IxUYylgiA3udzcLCS0oL8uWhpa5YX2ad2UB3xKiz3maI7sXF\n"
    "Dk5ds45MimR0p/dd+4y2tnAB1f/neR9/CqE58GQcFDY10mpNUa3rBqcUnHcJCfvb\n"
    "i5TX/clYY/wsVDPwjA5u3k7KSOip7FR33EfIPubE/nMeVqDTiZt+RBJ81xskBO0+\n"
    "x7bKAs+x\n"
    "-----END CERTIFICATE-----\n",

    /* Nuvoton TPM Root CA 1110 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIICBjCCAaygAwIBAgIIEDiqn2SaqGMwCgYIKoZIzj0EAwIwVTFTMB8GA1UEAxMY\n"
    "TnV2b3RvbiBUUE0gUm9vdCBDQSAxMTEwMCUGA1UEChMeTnV2b3RvbiBUZWNobm9s\n"
    "b2d5IENvcnBvcmF0aW9uMAkGA1UEBhMCVFcwHhcNMTUwNTExMDg0MzMzWhcNMzUw\n"
    "NTA3MDg0MzMzWjBVMVMwHwYDVQQDExhOdXZvdG9uIFRQTSBSb290IENBIDExMTAw\n"
    "JQYDVQQKEx5OdXZvdG9uIFRlY2hub2xvZ3kgQ29ycG9yYXRpb24wCQYDVQQGEwJU\n"
    "VzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABDVkEOpuyhuviaDH6xQj3faaV2Z4\n"
    "FvXSdwUkTiB1JjPDgv1PU0SFYtEE1W9VmI1GcOn5FAUi2/QM36DPhmPTd+qjZjBk\n"
    "MA4GA1UdDwEB/wQEAwICBDASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBQV\n"
    "kdS26vmNAQSGS2kDpI3QAmB30zAfBgNVHSMEGDAWgBQVkdS26vmNAQSGS2kDpI3Q\n"
    "AmB30zAKBggqhkjOPQQDAgNIADBFAiEAlfxysfHDcxYDed5dmRbvHPKHLEEq9Y9P\n"
    "wAxoKqH7Q5kCIGfsxiLr2j9nJ9jELwXz0/VWN9PhUNdM3qmsx2JEne6p\n"
    "-----END CERTIFICATE-----\n",

    /* Nuvoton TPM Root CA 1111 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIICBzCCAa2gAwIBAgIJAN/u3L0l8osZMAoGCCqGSM49BAMCMFUxUzAfBgNVBAMT\n"
    "GE51dm90b24gVFBNIFJvb3QgQ0EgMTExMTAlBgNVBAoTHk51dm90b24gVGVjaG5v\n"
    "bG9neSBDb3Jwb3JhdGlvbjAJBgNVBAYTAlRXMB4XDTE3MDgyMTAzMzI1M1oXDTM3\n"
    "MDgxNzAzMzI1M1owVTFTMB8GA1UEAxMYTnV2b3RvbiBUUE0gUm9vdCBDQSAxMTEx\n"
    "MCUGA1UEChMeTnV2b3RvbiBUZWNobm9sb2d5IENvcnBvcmF0aW9uMAkGA1UEBhMC\n"
    "VFcwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAATRcgnHcQk99FFMQ+9UVVFzf64J\n"
    "1jnhxpGAdZEHRleZfkfTYGHsdr3L9FLdGVUb+BA2cim30ovuPDZOGOUKPraLo2Yw\n"
    "ZDAOBgNVHQ8BAf8EBAMCAgQwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU\n"
    "iC8Ee4cSHPmIXzEWC8e7VYavRxswHwYDVR0jBBgwFoAUiC8Ee4cSHPmIXzEWC8e7\n"
    "VYavRxswCgYIKoZIzj0EAwIDSAAwRQIhANrKhBfLXyHCznBRtWRBmTbfYJR9mCl1\n"
    "2N/xZkcohRZIAiBDufnmhZWCbX4pibbzAOgcxjsc5+qmpAovxW9ipWJEsA==\n"
    "-----END CERTIFICATE-----\n",

    /* Nuvoton TPM Root CA 2010 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDkjCCAnqgAwIBAgIIWAnP9p2CIZcwDQYJKoZIhvcNAQEFBQAwVTFTMB8GA1UE\n"
    "AxMYTnV2b3RvbiBUUE0gUm9vdCBDQSAyMDEwMCUGA1UEChMeTnV2b3RvbiBUZWNo\n"
    "bm9sb2d5IENvcnBvcmF0aW9uMAkGA1UEBhMCVFcwHhcNMTUwNDIzMDY1OTE5WhcN\n"
    "MzUwNDE5MDY1OTE5WjBVMVMwHwYDVQQDExhOdXZvdG9uIFRQTSBSb290IENBIDIw\n"
    "MTAwJQYDVQQKEx5OdXZvdG9uIFRlY2hub2xvZ3kgQ29ycG9yYXRpb24wCQYDVQQG\n"
    "EwJUVzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKcE9saVURE582ny\n"
    "dHsZO7+3xmdMFbOPCdplBda/EJg9cg7n6bZ79Qv7hyymN5qE23SOPNFvm8SAdmCJ\n"
    "ybmTnk1y+SyiDw5gUpckbXsRYAetTwqtdfBkF4TkFoRJDIraQC8miTdYqXMXfWTo\n"
    "bhHXf/oV953laOCO/SRlqXzAWzm5d8PwixUBLZTnvcgxM+pXwv6JY6wgXpv55fY1\n"
    "D3M1hyiNALib+rg0LwazalU0DOryAAIqFzMgkR2IaefkAmpmQ1xpfMJsK+BMixcI\n"
    "XUCzSGGKKdkc3WUDye/vsyXYQ5zoYuLt3xb7BEZxes31lqbs1gniNz4oD5ptmrS4\n"
    "8V7Rz/kCAwEAAaNmMGQwDgYDVR0PAQH/BAQDAgIEMBIGA1UdEwEB/wQIMAYBAf8C\n"
    "AQAwHwYDVR0jBBgwFoAUCDAPQ6j0uMjmJKT3Bgz1nnRQFecwHQYDVR0OBBYEFAgw\n"
    "D0Oo9LjI5iSk9wYM9Z50UBXnMA0GCSqGSIb3DQEBBQUAA4IBAQAE0pMnjz5o3QUd\n"
    "S3lLQn3+vXkS2xc1EmPxcVFxjPbrJDtnNRMWwglC8zo70VgWu/+ulwzy783zJSiT\n"
    "nkWPeuszqp3xOtCPWDE4D2sxVbWH3pvel2tgZJv0KJsJH93QE53WbHUwSn2JjHNH\n"
    "UJiBpq0genUxGD+zBI3NGDGB1iti66aJfCdjn8C0G0gTmQ8jFpZ6AsX1GSvPYeU6\n"
    "EqN9ynIEYUVcRKwoHQaSmqDd7HVp97fwD+mkOfFYByLVUqC09rNFW81Va4Ze2gw2\n"
    "HiKz/SVSA5mA/91wfEZSZ6azOgDZNQlbgBo27mZFJ5mR7iJbWgtD+vO4+wRZK8Bc\n"
    "8yWxV8ri\n"
    "-----END CERTIFICATE-----\n",

    /* Nuvoton TPM Root CA 2011 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDkjCCAnqgAwIBAgIIGCgneR7Ow/EwDQYJKoZIhvcNAQEFBQAwVTFTMB8GA1UE\n"
    "AxMYTnV2b3RvbiBUUE0gUm9vdCBDQSAyMDExMCUGA1UEChMeTnV2b3RvbiBUZWNo\n"
    "bm9sb2d5IENvcnBvcmF0aW9uMAkGA1UEBhMCVFcwHhcNMTcwOTExMDcxMDEyWhcN\n"
    "MzcwOTA3MDcxMDEyWjBVMVMwHwYDVQQDExhOdXZvdG9uIFRQTSBSb290IENBIDIw\n"
    "MTEwJQYDVQQKEx5OdXZvdG9uIFRlY2hub2xvZ3kgQ29ycG9yYXRpb24wCQYDVQQG\n"
    "EwJUVzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJAfy7Iu3eTXYdXe\n"
    "WWc8ZPHb1ekofnHWY4Y6QD+z256j2rXTGmk5bA1Y6nCsYtBqUV4B2gfPAaZXunMD\n"
    "k3KDEBaI8Fkd7BTXbWHrfgxZstGHlWzVqi1r2Jj31MdDMz0NRovWXJwgG5N9tgs5\n"
    "KEeQ0Kk2ock1Vdn/jIqx3xY9vBMZm97N11m/WtZ5U2NyxbfqIIloIhxuvoPZRzx8\n"
    "c6MmI77MtM2ylm+2nbXf1DtbEKXQYuGnvM3Ib9K5f2+PTjNi/nWGAcCgLo89BNSB\n"
    "LdG9GxK1hO9QXYcCkAWsEWUD+8DM1v+zAl2cnrtj4iu48sU5i6hVO0CTtCP0yOkf\n"
    "VdVfNVsCAwEAAaNmMGQwDgYDVR0PAQH/BAQDAgIEMBIGA1UdEwEB/wQIMAYBAf8C\n"
    "AQAwHwYDVR0jBBgwFoAUpSy2R+CQm9ouf32RP2LYixOJ4sYwHQYDVR0OBBYEFKUs\n"
    "tkfgkJvaLn99kT9i2IsTieLGMA0GCSqGSIb3DQEBBQUAA4IBAQAUwrhniUs4CLgb\n"
    "UIFPslx3aly8M2P8Vn0GR+EcE0GklFa26WOT+YZxacUP3rMGteZGCtbFggYcM94n\n"
    "pog7FBnnDR1ySHbHOY0kWBO75ePVBmytqpPeZX8E5202sRgagnojl59bPj1tqhlH\n"
    "vwj6Xj6/XAFZDo0Or2GhCC5NzRbXmc6wB0zIjTlhPwVXQ+ikKBqdt3/HKCB4i3Sv\n"
    "7Dxyy/N4aPg/s6MJxgmUjtGxRyjU6RVgrzqVN/0Csv8L75g2SvzgXnqrqcpk82XT\n"
    "CnJgflpaCfV2SC5MGWHEjmj1tghC133JcEMQlsodWRm+0x7E/fsHvJNj4A0nh9K3\n"
    "H3TryFJ7\n"
    "-----END CERTIFICATE-----\n",

    /* Nuvoton TPM Root CA 2110 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIICBjCCAaygAwIBAgIIP5MvnZk8FrswCgYIKoZIzj0EAwIwVTFTMB8GA1UEAxMY\n"
    "TnV2b3RvbiBUUE0gUm9vdCBDQSAyMTEwMCUGA1UEChMeTnV2b3RvbiBUZWNobm9s\n"
    "b2d5IENvcnBvcmF0aW9uMAkGA1UEBhMCVFcwHhcNMTUxMDE5MDQzMjAwWhcNMzUx\n"
    "MDE1MDQzMjAwWjBVMVMwHwYDVQQDExhOdXZvdG9uIFRQTSBSb290IENBIDIxMTAw\n"
    "JQYDVQQKEx5OdXZvdG9uIFRlY2hub2xvZ3kgQ29ycG9yYXRpb24wCQYDVQQGEwJU\n"
    "VzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABPv9uK2BNm8/nmIyNsc2/aKHV0WR\n"
    "ptzge3jKAIgUMosQIokl4LE3iopXWD3Hruxjf9vkLMDJrTeK3hWh2ySS4ySjZjBk\n"
    "MA4GA1UdDwEB/wQEAwICBDASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBSf\n"
    "u3mqD1JieL7RUJKacXHpajW+9zAfBgNVHSMEGDAWgBSfu3mqD1JieL7RUJKacXHp\n"
    "ajW+9zAKBggqhkjOPQQDAgNIADBFAiEA/jiywhOKpiMOUnTfDmXsXfDFokhKVNTX\n"
    "B6Xtqm7J8L4CICjT3/Y+rrSnf8zrBXqWeHDh8Wi41+w2ppq6Ev9orZFI\n"
    "-----END CERTIFICATE-----\n",

    /* Nuvoton TPM Root CA 2111 */
    "-----BEGIN CERTIFICATE-----\n"
    "MIICBzCCAa2gAwIBAgIJAOCIUWvKMfuGMAoGCCqGSM49BAMCMFUxUzAfBgNVBAMT\n"
    "GE51dm90b24gVFBNIFJvb3QgQ0EgMjExMTAlBgNVBAoTHk51dm90b24gVGVjaG5v\n"
    "bG9neSBDb3Jwb3JhdGlvbjAJBgNVBAYTAlRXMB4XDTE3MDkxMTA3MDk0OVoXDTM3\n"
    "MDkwNzA3MDk0OVowVTFTMB8GA1UEAxMYTnV2b3RvbiBUUE0gUm9vdCBDQSAyMTEx\n"
    "MCUGA1UEChMeTnV2b3RvbiBUZWNobm9sb2d5IENvcnBvcmF0aW9uMAkGA1UEBhMC\n"
    "VFcwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAATanNywPkE/RICBZ4VjwCymLDts\n"
    "B7Unv52Oj0HyEsAfc7+vjOl2DP8JbrcozY05s7GFfQqR3vhv4ZdkJMalgOr2o2Yw\n"
    "ZDAOBgNVHQ8BAf8EBAMCAgQwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU\n"
    "I/TiKtO+N0pEl3KVSqKDrtdSVy4wHwYDVR0jBBgwFoAUI/TiKtO+N0pEl3KVSqKD\n"
    "rtdSVy4wCgYIKoZIzj0EAwIDSAAwRQIgXgXs2eVt2U4sCoRf1GLoUTf2ZzYTSsyg\n"
    "6I5w6hPTLigCIQDLLFlXK1xw1a1D1/idVhdd3a8gkE0FnTJO890WwiQbpg==\n"
    "-----END CERTIFICATE-----"
};

#endif /* FAPI_CERTIFICATES_H */
