/*
 * Copyright (C) Research In Motion Limited 2010. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include "SVGResourcesCache.h"

#include "ElementInlines.h"
#include "LegacyRenderSVGResourceContainer.h"
#include "RenderElementInlines.h"
#include "SVGResources.h"
#include "SVGResourcesCycleSolver.h"
#include "Settings.h"
#include <wtf/TZoneMallocInlines.h>

namespace WebCore {

WTF_MAKE_TZONE_ALLOCATED_IMPL(SVGResourcesCache);

SVGResourcesCache::SVGResourcesCache() = default;

SVGResourcesCache::~SVGResourcesCache() = default;

void SVGResourcesCache::addResourcesFromRenderer(RenderElement& renderer, const RenderStyle& style)
{
    // Verify that LBSE does not make use of SVGResourcesCache.
    if (renderer.document().settings().layerBasedSVGEngineEnabled())
        RELEASE_ASSERT_NOT_REACHED();

    ASSERT(!renderer.hasCachedSVGResource());
    ASSERT(!m_cache.contains(renderer));

    // Build a list of all resources associated with the passed RenderObject
    auto newResources = SVGResources::buildCachedResources(renderer, style);
    if (!newResources)
        return;

    // Put object in cache.
    SVGResources& resources = *m_cache.add(renderer, WTF::move(newResources)).iterator->value;
    renderer.setHasCachedSVGResource(true);

    // Run cycle-detection _afterwards_, so self-references can be caught as well.
    SVGResourcesCycleSolver::resolveCycles(renderer, resources);

    // Walk resources and register the render object at each resources.
    SingleThreadWeakHashSet<LegacyRenderSVGResourceContainer> resourceSet;
    resources.buildSetOfResources(resourceSet);

    for (auto& resourceContainer : resourceSet)
        resourceContainer.addClient(renderer);
}

void SVGResourcesCache::removeResourcesFromRenderer(RenderElement& renderer)
{
    if (!renderer.hasCachedSVGResource())
        return;

    // Verify that LBSE does not make use of SVGResourcesCache.
    if (renderer.document().settings().layerBasedSVGEngineEnabled())
        RELEASE_ASSERT_NOT_REACHED();

    auto resources = m_cache.take(renderer);
    renderer.setHasCachedSVGResource(false);
    if (!resources)
        return;

    // Walk resources and register the render object at each resources.
    SingleThreadWeakHashSet<LegacyRenderSVGResourceContainer> resourceSet;
    resources->buildSetOfResources(resourceSet);

    for (auto& resourceContainer : resourceSet)
        resourceContainer.removeClient(renderer);
}

static inline SVGResourcesCache& resourcesCacheFromRenderer(const RenderElement& renderer)
{
    // Verify that LBSE does not make use of SVGResourcesCache.
    if (renderer.document().settings().layerBasedSVGEngineEnabled())
        RELEASE_ASSERT_NOT_REACHED();

    return renderer.document().svgExtensions().resourcesCache();
}

SVGResources* SVGResourcesCache::cachedResourcesForRenderer(const RenderElement& renderer)
{
    if (!renderer.hasCachedSVGResource())
        return nullptr;
    return resourcesCacheFromRenderer(renderer).m_cache.get(renderer);
}

static bool hasPaintResourceRequiringRemovalOnClientLayoutChange(LegacyRenderSVGResource* resource)
{
    return resource && resource->resourceType() == PatternResourceType;
}

static bool hasResourcesRequiringRemovalOnClientLayoutChange(SVGResources& resources)
{
    return resources.masker() || resources.filter() || hasPaintResourceRequiringRemovalOnClientLayoutChange(resources.fill()) || hasPaintResourceRequiringRemovalOnClientLayoutChange(resources.stroke());
}

void SVGResourcesCache::clientLayoutChanged(RenderElement& renderer)
{
    if (!renderer.hasCachedSVGResource())
        return;

    // Verify that LBSE does not make use of SVGResourcesCache.
    if (renderer.document().settings().layerBasedSVGEngineEnabled())
        RELEASE_ASSERT_NOT_REACHED();

    auto* resources = SVGResourcesCache::cachedResourcesForRenderer(renderer);
    if (!resources)
        return;

    // Invalidate the resources if either the RenderElement itself changed,
    // or we have filter resources, which could depend on the layout of children.
    if ((renderer.selfNeedsLayout() || resources->filter()) && hasResourcesRequiringRemovalOnClientLayoutChange(*resources))
        resources->removeClientFromCacheAndMarkForInvalidation(renderer, false);
}

static inline bool rendererCanHaveResources(RenderObject& renderer)
{
    return renderer.node() && renderer.node()->isSVGElement() && !renderer.isRenderSVGInlineText();
}

void SVGResourcesCache::clientStyleChanged(RenderElement& renderer, Style::Difference diff, const RenderStyle* oldStyle, const RenderStyle& newStyle)
{
    // Verify that LBSE does not make use of SVGResourcesCache.
    if (renderer.document().settings().layerBasedSVGEngineEnabled())
        RELEASE_ASSERT_NOT_REACHED();

    ASSERT(!renderer.element() || renderer.element()->isSVGElement());

    if (!renderer.parent())
        return;

    // For filter primitives, when diff is Repaint or RepaintIsText, the
    // SVGFE*Element will decide whether the modified CSS properties require a
    // relayout or repaint.
    //
    // Since diff can be Equal even if we have have a filter property change
    // (due to how RenderElement::adjustStyleDifference works), in general we
    // want to continue to the comparison of oldStyle and newStyle below, and
    // so we don't return early just when diff == Style::DifferenceResult::Equal. But
    // this isn't necessary for filter primitives, to which the filter property
    // doesn't apply, so we check for it here too.
    if (renderer.isLegacyRenderSVGResourceFilterPrimitive() && (diff == Style::DifferenceResult::Equal || diff == Style::DifferenceResult::Repaint || diff == Style::DifferenceResult::RepaintIfText))
        return;

    auto hasStyleDifferencesAffectingResources = [&] {
        if (!rendererCanHaveResources(renderer))
            return false;

        if (!oldStyle)
            return true;

        if (oldStyle->clipPath() != newStyle.clipPath())
            return true;

        // RenderSVGResourceMarker only supports SVG <mask> references.
        if (oldStyle->maskLayers().usedFirst().image() != newStyle.maskLayers().usedFirst().image())
            return true;

        if (oldStyle->filter() != newStyle.filter())
            return true;

        // -apple-color-filter affects gradients.
        if (oldStyle->appleColorFilter() != newStyle.appleColorFilter())
            return true;

        if (oldStyle->fill().urlDisregardingType() != newStyle.fill().urlDisregardingType())
            return true;

        if (oldStyle->stroke().urlDisregardingType() != newStyle.stroke().urlDisregardingType())
            return true;

        if (oldStyle->markerStart() != newStyle.markerStart())
            return true;

        if (oldStyle->markerMid() != newStyle.markerMid())
            return true;

        if (oldStyle->markerEnd() != newStyle.markerEnd())
            return true;

        return false;
    };

    if (hasStyleDifferencesAffectingResources()) {
        auto& cache = resourcesCacheFromRenderer(renderer);
        cache.removeResourcesFromRenderer(renderer);
        cache.addResourcesFromRenderer(renderer, newStyle);
    }

    LegacyRenderSVGResource::markForLayoutAndParentResourceInvalidation(renderer, false);
}

void SVGResourcesCache::clientWasAddedToTree(RenderObject& renderer)
{
    // Verify that LBSE does not make use of SVGResourcesCache.
    if (renderer.document().settings().layerBasedSVGEngineEnabled())
        RELEASE_ASSERT_NOT_REACHED();

    if (renderer.isAnonymous())
        return;

    LegacyRenderSVGResource::markForLayoutAndParentResourceInvalidation(renderer, false);

    if (!rendererCanHaveResources(renderer))
        return;
    RenderElement& elementRenderer = downcast<RenderElement>(renderer);
    resourcesCacheFromRenderer(elementRenderer).addResourcesFromRenderer(elementRenderer, elementRenderer.style());
}

void SVGResourcesCache::clientWillBeRemovedFromTree(RenderObject& renderer)
{
    if (!rendererCanHaveResources(renderer))
        return;
    RenderElement& elementRenderer = downcast<RenderElement>(renderer);

    if (!elementRenderer.hasCachedSVGResource())
        return;

    // While LBSE does not make use of SVGResourcesCache, we might get here after switching from legacy to LBSE
    // and destructing the legacy tree -- when LBSE is already activated - don't assert here that this is not reached.
    if (renderer.document().settings().layerBasedSVGEngineEnabled())
        return;

    if (renderer.isAnonymous())
        return;

    LegacyRenderSVGResource::markForLayoutAndParentResourceInvalidation(renderer, false);
    resourcesCacheFromRenderer(elementRenderer).removeResourcesFromRenderer(elementRenderer);
}

void SVGResourcesCache::clientDestroyed(RenderElement& renderer)
{
    if (!renderer.hasCachedSVGResource())
        return;

    // While LBSE does not make use of SVGResourcesCache, we might get here after switching from legacy to LBSE
    // and destructing the legacy tree -- when LBSE is already activated - don't assert here that this is not reached.
    if (renderer.document().settings().layerBasedSVGEngineEnabled())
        return;

    if (auto* resources = SVGResourcesCache::cachedResourcesForRenderer(renderer)) {
        resources->removeClientFromCacheAndMarkForInvalidation(renderer);
        resourcesCacheFromRenderer(renderer).removeResourcesFromRenderer(renderer);
    }
}

void SVGResourcesCache::resourceDestroyed(LegacyRenderSVGResourceContainer& resource)
{
    // While LBSE does not make use of SVGResourcesCache, we might get here after switching from legacy to LBSE
    // and destructing the legacy tree -- when LBSE is already activated - don't assert here that this is not reached.
    if (resource.document().settings().layerBasedSVGEngineEnabled())
        return;

    auto& cache = resourcesCacheFromRenderer(resource);

    // The resource itself may have clients, that need to be notified.
    cache.removeResourcesFromRenderer(resource);

    for (auto& it : cache.m_cache) {
        if (it.value->resourceDestroyed(resource)) {
            // Mark users of destroyed resources as pending resolution based on the id of the old resource.
            Ref clientElement = *it.key->element();
            clientElement->treeScopeForSVGReferences().addPendingSVGResource(resource.element().getIdAttribute(), downcast<SVGElement>(clientElement.get()));
        }
    }
}

SVGResourcesCache::SetStyleForScope::SetStyleForScope(RenderElement& renderer, const RenderStyle& scopedStyle, const RenderStyle& newStyle)
    : m_renderer(renderer)
    , m_scopedStyle(scopedStyle)
    , m_needsNewStyle(scopedStyle != newStyle && rendererCanHaveResources(renderer))
{
    setStyle(newStyle);
}

SVGResourcesCache::SetStyleForScope::~SetStyleForScope()
{
    setStyle(m_scopedStyle);
}

void SVGResourcesCache::SetStyleForScope::setStyle(const RenderStyle& style)
{
    if (!m_needsNewStyle)
        return;

    // FIXME: Check if a similar mechanism is needed for LBSE + text rendering.
    if (m_renderer.document().settings().layerBasedSVGEngineEnabled())
        return;

    auto& cache = resourcesCacheFromRenderer(m_renderer);
    cache.removeResourcesFromRenderer(m_renderer);
    cache.addResourcesFromRenderer(m_renderer, style);
}

}
