/* SOGoSource.h - this file is part of SOGo
 *
 * Copyright (C) 2009-2022 Inverse inc.
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef SOGOSOURCE_H
#define SOGOSOURCE_H


#import "SOGoConstants.h"

@class EOQualifier;
@class EOSortOrdering;
@class NSDictionary;
@class NSException;
@class NSString;

@protocol SOGoSource <NSObject>

+ (id) sourceFromUDSource: (NSDictionary *) udSource
                 inDomain: (NSString *) domain;

- (id) initFromUDSource: (NSDictionary *) udSource
               inDomain: (NSString *) domain;

- (NSString *) domain;
- (NSArray *) searchFields;
- (NSArray *) userPasswordPolicy;

- (id) connection;
- (void) releaseConnection: (id) connection;
- (BOOL) isConnected;

/* requires a "." to obtain the full list of contacts */
- (void) setListRequiresDot: (BOOL) aBool;
- (BOOL) listRequiresDot;

- (BOOL) checkLogin: (NSString *) _login
	   password: (NSString *) _pwd
	       perr: (SOGoPasswordPolicyError *) _perr
	     expire: (int *) _expire
	      grace: (int *) _grace;

- (BOOL) changePasswordForLogin: (NSString *) login
                    oldPassword: (NSString *) oldPassword
                    newPassword: (NSString *) newPassword
               passwordRecovery: (BOOL) passwordRecovery
                           perr: (SOGoPasswordPolicyError *) perr;

- (NSDictionary *) lookupContactEntry: (NSString *) theID
                             inDomain: (NSString *) domain;
- (NSDictionary *) lookupContactEntry: (NSString *) theID
                             inDomain: (NSString *) domain
		      usingConnection: (id) connection;
- (NSDictionary *) lookupContactEntryWithUIDorEmail: (NSString *) entryID
                                           inDomain: (NSString *) domain;

- (NSArray *) allEntryIDs;
- (NSArray *) allEntryIDsVisibleFromDomain: (NSString *) domain;
- (NSArray *) fetchContactsMatching: (NSString *) filter
                       withCriteria: (NSArray *) criteria
                           inDomain: (NSString *) domain;
- (NSArray *) lookupContactsWithQualifier: (EOQualifier *) qualifier
                          andSortOrdering: (EOSortOrdering *) ordering
                                 inDomain: (NSString *) domain;

- (void) setSourceID: (NSString *) newSourceID;
- (NSString *) sourceID;

- (void) setDisplayName: (NSString *) newDisplayName;
- (NSString *) displayName;

- (void) setModifiers: (NSArray *) newModifiers;
- (NSArray *) modifiers;

- (BOOL) hasUserAddressBooks;
- (NSArray *) addressBookSourcesForUser: (NSString *) user;

- (NSException *) addContactEntry: (NSDictionary *) roLdifRecord
                           withID: (NSString *) aId;
- (NSException *) updateContactEntry: (NSDictionary *) ldifRecord;
- (NSException *) removeContactEntryWithID: (NSString *) aId;

- (void) addVCardProperty: (NSString *) property
               toCriteria: (NSMutableArray *) criteria;

/* user address books */
- (NSArray *) addressBookSourcesForUser: (NSString *) user;

- (NSException *) addAddressBookSource: (NSString *) newId
                       withDisplayName: (NSString *) newDisplayName
                               forUser: (NSString *) user;
- (NSException *) renameAddressBookSource: (NSString *) newId
                          withDisplayName: (NSString *) newDisplayName
                                  forUser: (NSString *) user;
- (NSException *) removeAddressBookSource: (NSString *) newId
                                  forUser: (NSString *) user;

@end

@protocol SOGoDNSource <SOGoSource>

- (void) setBindDN: (NSString *) theDN;
- (NSString *) bindDN;

- (void) setBindPassword: (NSString *) thePassword;
- (NSString *) bindPassword;
- (BOOL) bindAsCurrentUser;

- (NSString *) lookupLoginByDN: (NSString *) theDN;
- (NSString *) lookupDNByLogin: (NSString *) theLogin;

- (NSString *) baseDN;
- (NSString *) MSExchangeHostname;

- (NSArray *) groupObjectClasses;

- (void) updateBaseDNFromLogin: (NSString *) theLogin;
@end

@protocol SOGoMembershipSource <SOGoSource>
- (NSArray *) membersForGroupWithUID: (NSString *) uid;
- (BOOL) groupWithUIDHasMemberWithUID: (NSString *) uid
                            memberUid: (NSString *) memberUid;
@end

#endif /* SOGOSOURCE_H */
