/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.translog.transfer;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.opensearch.action.ActionListener;
import org.opensearch.action.LatchedActionListener;
import org.opensearch.common.blobstore.BlobPath;
import org.opensearch.common.lucene.store.ByteArrayIndexInput;
import org.opensearch.index.translog.Translog;
import org.opensearch.index.translog.transfer.FileSnapshot;
import org.opensearch.index.translog.transfer.FileTransferException;
import org.opensearch.index.translog.transfer.FileTransferTracker;
import org.opensearch.index.translog.transfer.TransferService;
import org.opensearch.index.translog.transfer.TransferSnapshot;
import org.opensearch.index.translog.transfer.TranslogTransferMetadata;
import org.opensearch.index.translog.transfer.listener.TranslogTransferListener;

public class TranslogTransferManager {
    private final TransferService transferService;
    private final BlobPath remoteBaseTransferPath;
    private final BlobPath remoteMetadataTransferPath;
    private final FileTransferTracker fileTransferTracker;
    private static final long TRANSFER_TIMEOUT_IN_MILLIS = 30000L;
    private static final Logger logger = LogManager.getLogger(TranslogTransferManager.class);
    private static final String METADATA_DIR = "metadata";

    public TranslogTransferManager(TransferService transferService, BlobPath remoteBaseTransferPath, FileTransferTracker fileTransferTracker) {
        this.transferService = transferService;
        this.remoteBaseTransferPath = remoteBaseTransferPath;
        this.remoteMetadataTransferPath = remoteBaseTransferPath.add(METADATA_DIR);
        this.fileTransferTracker = fileTransferTracker;
    }

    public boolean transferSnapshot(TransferSnapshot transferSnapshot, TranslogTransferListener translogTransferListener) throws IOException {
        ArrayList exceptionList = new ArrayList(transferSnapshot.getTranslogTransferMetadata().getCount());
        HashSet<FileSnapshot.TransferFileSnapshot> toUpload = new HashSet<FileSnapshot.TransferFileSnapshot>(transferSnapshot.getTranslogTransferMetadata().getCount());
        try {
            toUpload.addAll(this.fileTransferTracker.exclusionFilter(transferSnapshot.getTranslogFileSnapshots()));
            toUpload.addAll(this.fileTransferTracker.exclusionFilter(transferSnapshot.getCheckpointFileSnapshots()));
            if (toUpload.isEmpty()) {
                logger.trace("Nothing to upload for transfer");
                translogTransferListener.onUploadComplete(transferSnapshot);
                return true;
            }
            CountDownLatch latch = new CountDownLatch(toUpload.size());
            LatchedActionListener latchedActionListener = new LatchedActionListener(ActionListener.wrap(this.fileTransferTracker::onSuccess, ex -> {
                assert (ex instanceof FileTransferException);
                logger.error(() -> new ParameterizedMessage("Exception during transfer for file {}", (Object)((FileTransferException)ex).getFileSnapshot().getName()), (Throwable)ex);
                FileTransferException e = (FileTransferException)ex;
                this.fileTransferTracker.onFailure(e.getFileSnapshot(), (Exception)ex);
                exceptionList.add(ex);
            }), latch);
            toUpload.forEach(fileSnapshot -> this.transferService.uploadBlobAsync("translog_transfer", (FileSnapshot.TransferFileSnapshot)fileSnapshot, this.remoteBaseTransferPath.add(String.valueOf(fileSnapshot.getPrimaryTerm())), latchedActionListener));
            try {
                if (!latch.await(30000L, TimeUnit.MILLISECONDS)) {
                    TimeoutException ex2 = new TimeoutException("Timed out waiting for transfer of snapshot " + transferSnapshot + " to complete");
                    exceptionList.forEach(ex2::addSuppressed);
                    throw ex2;
                }
            }
            catch (InterruptedException ex3) {
                exceptionList.forEach(ex3::addSuppressed);
                Thread.currentThread().interrupt();
                throw ex3;
            }
            if (exceptionList.isEmpty()) {
                this.transferService.uploadBlob(this.prepareMetadata(transferSnapshot), this.remoteMetadataTransferPath);
                translogTransferListener.onUploadComplete(transferSnapshot);
                return true;
            }
            IOException ex4 = new IOException("Failed to upload " + exceptionList.size() + " files during transfer");
            exceptionList.forEach(ex4::addSuppressed);
            throw ex4;
        }
        catch (Exception ex5) {
            logger.error(() -> new ParameterizedMessage("Transfer failed for snapshot {}", (Object)transferSnapshot), (Throwable)ex5);
            translogTransferListener.onUploadFailed(transferSnapshot, ex5);
            return false;
        }
    }

    public boolean downloadTranslog(String primaryTerm, String generation, Path location) throws IOException {
        logger.info("Downloading translog files with: Primary Term = {}, Generation = {}, Location = {}", (Object)primaryTerm, (Object)generation, (Object)location);
        String ckpFileName = Translog.getCommitCheckpointFileName(Long.parseLong(generation));
        this.downloadToFS(ckpFileName, location, primaryTerm);
        String translogFilename = Translog.getFilename(Long.parseLong(generation));
        this.downloadToFS(translogFilename, location, primaryTerm);
        return true;
    }

    private void downloadToFS(String fileName, Path location, String primaryTerm) throws IOException {
        Path filePath = location.resolve(fileName);
        if (Files.exists(filePath, new LinkOption[0])) {
            Files.delete(filePath);
        }
        try (InputStream inputStream = this.transferService.downloadBlob(this.remoteBaseTransferPath.add(primaryTerm), fileName);){
            Files.copy(inputStream, filePath, new CopyOption[0]);
        }
        this.fileTransferTracker.add(fileName, true);
    }

    public TranslogTransferMetadata readMetadata() throws IOException {
        return this.transferService.listAll(this.remoteMetadataTransferPath).stream().max(TranslogTransferMetadata.METADATA_FILENAME_COMPARATOR).map(filename -> {
            TranslogTransferMetadata translogTransferMetadata;
            block8: {
                InputStream inputStream = this.transferService.downloadBlob(this.remoteMetadataTransferPath, (String)filename);
                try {
                    ByteArrayIndexInput indexInput = new ByteArrayIndexInput("metadata file", inputStream.readAllBytes());
                    translogTransferMetadata = new TranslogTransferMetadata(indexInput);
                    if (inputStream == null) break block8;
                }
                catch (Throwable throwable) {
                    try {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        logger.error(() -> new ParameterizedMessage("Exception while reading metadata file: {}", filename), (Throwable)e);
                        return null;
                    }
                }
                inputStream.close();
            }
            return translogTransferMetadata;
        }).orElse(null);
    }

    private FileSnapshot.TransferFileSnapshot prepareMetadata(TransferSnapshot transferSnapshot) throws IOException {
        Map<String, String> generationPrimaryTermMap = transferSnapshot.getTranslogFileSnapshots().stream().map(s -> {
            assert (s instanceof FileSnapshot.TranslogFileSnapshot);
            return (FileSnapshot.TranslogFileSnapshot)s;
        }).collect(Collectors.toMap(snapshot -> String.valueOf(snapshot.getGeneration()), snapshot -> String.valueOf(snapshot.getPrimaryTerm())));
        TranslogTransferMetadata translogTransferMetadata = transferSnapshot.getTranslogTransferMetadata();
        translogTransferMetadata.setGenerationToPrimaryTermMapper(new HashMap<String, String>(generationPrimaryTermMap));
        return new FileSnapshot.TransferFileSnapshot(TranslogTransferMetadata.getFileName(translogTransferMetadata.getPrimaryTerm(), translogTransferMetadata.getGeneration()), translogTransferMetadata.createMetadataBytes(), translogTransferMetadata.getPrimaryTerm());
    }

    public void deleteGenerationAsync(long primaryTerm, Set<Long> generations, Runnable onCompletion) {
        ArrayList<String> translogFiles = new ArrayList<String>();
        ArrayList<String> metadataFiles = new ArrayList<String>();
        generations.forEach(generation -> {
            String ckpFileName = Translog.getCommitCheckpointFileName(generation);
            String translogFileName = Translog.getFilename(generation);
            translogFiles.addAll(List.of(ckpFileName, translogFileName));
            String metadataFileName = TranslogTransferMetadata.getFileName(primaryTerm, generation);
            metadataFiles.add(metadataFileName);
        });
        this.deleteTranslogFilesAsync(primaryTerm, translogFiles, onCompletion);
        this.deleteMetadataFilesAsync(metadataFiles, onCompletion);
    }

    public void deletePrimaryTermsAsync(final long minPrimaryTermToKeep) {
        logger.info("Deleting primary terms from remote store lesser than {}", (Object)minPrimaryTermToKeep);
        this.transferService.listFoldersAsync("remote_purge", this.remoteBaseTransferPath, new ActionListener<Set<String>>(){

            @Override
            public void onResponse(Set<String> folders) {
                Set primaryTermsInRemote = folders.stream().filter(folderName -> {
                    try {
                        Long.parseLong(folderName);
                        return true;
                    }
                    catch (Exception exception) {
                        return false;
                    }
                }).map(Long::parseLong).collect(Collectors.toSet());
                Set<Long> primaryTermsToDelete = primaryTermsInRemote.stream().filter(term -> term < minPrimaryTermToKeep).collect(Collectors.toSet());
                primaryTermsToDelete.forEach(term -> TranslogTransferManager.this.deletePrimaryTermAsync((long)term));
            }

            @Override
            public void onFailure(Exception e) {
                logger.error("Exception occurred while getting primary terms from remote store", (Throwable)e);
            }
        });
    }

    private void deletePrimaryTermAsync(final long primaryTerm) {
        this.transferService.deleteAsync("remote_purge", this.remoteBaseTransferPath.add(String.valueOf(primaryTerm)), new ActionListener<Void>(){

            @Override
            public void onResponse(Void unused) {
                logger.info("Deleted primary term {}", (Object)primaryTerm);
            }

            @Override
            public void onFailure(Exception e) {
                logger.error((Message)new ParameterizedMessage("Exception occurred while deleting primary term {}", (Object)primaryTerm), (Throwable)e);
            }
        });
    }

    public void deleteStaleTranslogMetadataFilesAsync() {
        this.transferService.listAllAsync("remote_purge", this.remoteMetadataTransferPath, new ActionListener<Set<String>>(){

            @Override
            public void onResponse(Set<String> metadataFiles) {
                List sortedMetadataFiles = metadataFiles.stream().sorted(TranslogTransferMetadata.METADATA_FILENAME_COMPARATOR).collect(Collectors.toList());
                if (sortedMetadataFiles.size() <= 1) {
                    logger.trace("Remote Metadata file count is {}, so skipping deletion", (Object)sortedMetadataFiles.size());
                    return;
                }
                List<String> metadataFilesToDelete = sortedMetadataFiles.subList(0, sortedMetadataFiles.size() - 1);
                TranslogTransferManager.this.deleteMetadataFilesAsync(metadataFilesToDelete);
            }

            @Override
            public void onFailure(Exception e) {
                logger.error("Exception occurred while listing translog metadata files from remote store", (Throwable)e);
            }
        });
    }

    private void deleteTranslogFilesAsync(final long primaryTerm, final List<String> files, final Runnable onCompletion) {
        try {
            this.transferService.deleteBlobsAsync("remote_purge", this.remoteBaseTransferPath.add(String.valueOf(primaryTerm)), files, new ActionListener<Void>(){

                @Override
                public void onResponse(Void unused) {
                    TranslogTransferManager.this.fileTransferTracker.delete(files);
                    logger.trace("Deleted translogs for primaryTerm={} files={}", (Object)primaryTerm, (Object)files);
                    onCompletion.run();
                }

                @Override
                public void onFailure(Exception e) {
                    onCompletion.run();
                    logger.error(() -> new ParameterizedMessage("Exception occurred while deleting translog for primaryTerm={} files={}", (Object)primaryTerm, (Object)files), (Throwable)e);
                }
            });
        }
        catch (Exception e) {
            onCompletion.run();
            throw e;
        }
    }

    private void deleteMetadataFilesAsync(List<String> metadataFilesToDelete) {
        this.deleteMetadataFilesAsync(metadataFilesToDelete, () -> {});
    }

    private void deleteMetadataFilesAsync(final List<String> files, final Runnable onCompletion) {
        try {
            this.transferService.deleteBlobsAsync("remote_purge", this.remoteMetadataTransferPath, files, new ActionListener<Void>(){

                @Override
                public void onResponse(Void unused) {
                    onCompletion.run();
                    logger.trace("Deleted remote translog metadata files {}", (Object)files);
                }

                @Override
                public void onFailure(Exception e) {
                    onCompletion.run();
                    logger.error((Message)new ParameterizedMessage("Exception occurred while deleting remote translog metadata files {}", (Object)files), (Throwable)e);
                }
            });
        }
        catch (Exception e) {
            onCompletion.run();
            throw e;
        }
    }
}

