"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _next_url = require("../../../utils/next_url");

var _common = require("../../../../common");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SamlAuthRoutes {
  constructor(router, // @ts-ignore: unused variable
  config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }

  setupRoutes() {
    this.router.get({
      path: _common.SAML_AUTH_LOGIN,
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          redirectHash: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (request.auth.isAuthenticated) {
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      }

      try {
        const samlHeader = await this.securityClient.getSamlHeader(request); // const { nextUrl = '/' } = request.query;

        const cookie = {
          saml: {
            nextUrl: request.query.nextUrl,
            requestId: samlHeader.requestId,
            redirectHash: request.query.redirectHash === 'true'
          }
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: samlHeader.location
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed to get saml header: ${error}`);
        return response.internalError(); // TODO: redirect to error page?
      }
    });
    this.router.post({
      path: `/_opendistro/_security/saml/acs`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      let requestId = '';
      let nextUrl = '/';
      let redirectHash = false;

      try {
        const cookie = await this.sessionStorageFactory.asScoped(request).get();

        if (cookie) {
          var _cookie$saml, _cookie$saml2, _cookie$saml3;

          requestId = ((_cookie$saml = cookie.saml) === null || _cookie$saml === void 0 ? void 0 : _cookie$saml.requestId) || '';
          nextUrl = ((_cookie$saml2 = cookie.saml) === null || _cookie$saml2 === void 0 ? void 0 : _cookie$saml2.nextUrl) || `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`;
          redirectHash = ((_cookie$saml3 = cookie.saml) === null || _cookie$saml3 === void 0 ? void 0 : _cookie$saml3.redirectHash) || false;
        }

        if (!requestId) {
          return response.badRequest({
            body: 'Invalid requestId'
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`Failed to parse cookie: ${error}`);
        return response.badRequest();
      }

      try {
        const credentials = await this.securityClient.authToken(requestId, request.body.SAMLResponse, undefined);
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');

        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }

        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());

        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }

        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValue: credentials.authorization
          },
          authType: _common.AuthType.SAML,
          // TODO: create constant
          expiryTime
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);

        if (redirectHash) {
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/redirectUrlFragment?nextUrl=${escape(nextUrl)}`
            }
          });
        } else {
          return response.redirected({
            headers: {
              location: nextUrl
            }
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`SAML SP initiated authentication workflow failed: ${error}`);
      }

      return response.internalError();
    });
    this.router.post({
      path: `/_opendistro/_security/saml/acs/idpinitiated`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      const acsEndpoint = `${this.coreSetup.http.basePath.serverBasePath}/_opendistro/_security/saml/acs/idpinitiated`;

      try {
        const credentials = await this.securityClient.authToken(undefined, request.body.SAMLResponse, acsEndpoint);
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');

        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }

        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());

        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }

        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValue: credentials.authorization
          },
          authType: _common.AuthType.SAML,
          // TODO: create constant
          expiryTime
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML IDP initiated authentication workflow failed: ${error}`);
      }

      return response.internalError();
    }); // captureUrlFragment is the first route that will be invoked in the SP initiated login.
    // This route will execute the captureUrlFragment.js script.

    this.coreSetup.http.resources.register({
      path: '/auth/saml/captureUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const serverBasePath = this.coreSetup.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
            <!DOCTYPE html>
            <title>OSD SAML Capture</title>
            <link rel="icon" href="data:,">
            <script src="${serverBasePath}/auth/saml/captureUrlFragment.js"></script>
          `
      });
    }); // This script will store the URL Hash in browser's local storage.

    this.coreSetup.http.resources.register({
      path: '/auth/saml/captureUrlFragment.js',
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.renderJs({
        body: `let samlHash=window.location.hash.toString();
                 let redirectHash = false;
                 if (samlHash !== "") {
                    window.localStorage.removeItem('samlHash');
                    window.localStorage.setItem('samlHash', samlHash);
                     redirectHash = true;
                  }
                 let params = new URLSearchParams(window.location.search);
                 let nextUrl = params.get("nextUrl");
                 finalUrl = "login?nextUrl=" + encodeURIComponent(nextUrl);
                 finalUrl += "&redirectHash=" + encodeURIComponent(redirectHash);
                 window.location.replace(finalUrl);
                `
      });
    }); //  Once the User is authenticated via the '_opendistro/_security/saml/acs' route,
    //  the browser will be redirected to '/auth/saml/redirectUrlFragment' route,
    //  which will execute the redirectUrlFragment.js.

    this.coreSetup.http.resources.register({
      path: '/auth/saml/redirectUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.any()
        })
      },
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      const serverBasePath = this.coreSetup.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
            <!DOCTYPE html>
            <title>OSD SAML Success</title>
            <link rel="icon" href="data:,">
            <script src="${serverBasePath}/auth/saml/redirectUrlFragment.js"></script>
          `
      });
    }); // This script will pop the Hash from local storage if it exists.
    // And forward the browser to the next url.

    this.coreSetup.http.resources.register({
      path: '/auth/saml/redirectUrlFragment.js',
      validate: false,
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      return response.renderJs({
        body: `let samlHash=window.localStorage.getItem('samlHash');
                 window.localStorage.removeItem('samlHash');
                 let params = new URLSearchParams(window.location.search);
                 let nextUrl = params.get("nextUrl");
                 finalUrl = nextUrl + samlHash;
                 window.location.replace(finalUrl);
                `
      });
    });
    this.router.get({
      path: _common.SAML_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      try {
        const authInfo = await this.securityClient.authinfo(request);
        this.sessionStorageFactory.asScoped(request).clear(); // TODO: need a default logout page

        const redirectUrl = authInfo.sso_logout_url || this.coreSetup.http.basePath.serverBasePath || '/';
        return response.redirected({
          headers: {
            location: redirectUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML logout failed: ${error}`);
        return response.badRequest();
      }
    });
  }

}

exports.SamlAuthRoutes = SamlAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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