#!/usr/bin/perl -w
# $File: //automated_pp_test.pl $ $Author: mnooning $
# $Revision: #005 $ $Change: 20040412_01 $ $DateTime: 2004/04/12 11:56:
########################################################################
# Copyright 2004 by Malcolm Nooning
# This program does not impose any
# licensing restrictions on files generated by their execution, in
# accordance with the 8th article of the Artistic License:
#
#    "Aggregation of this Package with a commercial distribution is
#    always permitted provided that the use of this Package is embedded;
#    that is, when no overt attempt is made to make this Package's
#    interfaces visible to the end user of the commercial distribution.
#    Such use shall not be construed as a distribution of this Package."
#
# Therefore, you are absolutely free to place any license on the resulting
# executable(s), as long as the packed 3rd-party libraries are also available
# under the Artistic License.
#
# This program is free software; you can redistribute it and/or modify it
# under the same terms as Perl itself.
#
# See L<http://www.perl.com/perl/misc/Artistic.html>
#
#
#
########################################################################
our $VERSION = 0.01;

########################################################################
# Prior to each test
#   . Remove any possible files that could exist from a previous
#     invocation of the test.  If a file cannot be removed, report
#     the failure and move on to the next test.
#
# During each test
#   . After any file is created, check for it's existence to make
#     sure it is there.
#   . If a file cannot be created report the failure and move on
#     to the next test.
#
# After each test
#   . Do not do anything.  If a test should cause a fault we
#     want to leave any work product (files) for postmortem analysis.
#   . Maybe we want to print result verbiage?
#
# Windows versus Unix
#   . For each test, the command to "system" or back tick, path, etc.,
#     will be determined by the type of OS.
#     For example,
#     if Unix, use "./tmp1/foo1".  For windows, use "temp\\foo1".
#
# Note when the expected result is just "hello":
#            "if ($result =~ m/hello/) {...
#     versus
#            chomp($result);
#            "if ($result eq "hello") {...
#    The $result can have the string "hello" in it and
#    also contain extraneous or other error strings, so
#    don't match for hello. Chomp and do an "eq".
#
########################################################################
use Test::More tests => 32;
use Cwd qw(chdir cwd);

use Config;
use File::Copy;
use File::Path;
use File::Spec;
use File::Basename;

use POSIX qw(uname);
use POSIX qw(EXIT_SUCCESS EXIT_FAILURE);

use Getopt::Long;


use strict;

########################################################################
# Home grown perl modules go here
use prior_to_test;
use back_tick_a_command;
use test_in_further_subdir;
use remove_file_and_try_executable_again;
########################################################################


########################################################################
# The module prior_to_test creates these four subdirs underneath
# whatever test subdir it is to work with.  They are created for
# general use whether the current test uses them or not.
########################################################################
our  $SUBDIR1 = "subdir1";
our  $SUBDIR2 = "subdir2";
our  $SUBDIR3 = "subdir3";
our  $SUBDIR4 = "subdir4";
########################################################################


########################################################################
our $os = (uname())[0];
if ($os =~ m/^Win/i) { eval {
    require Win32::Exe;
    Win32::Exe->import();
} }
########################################################################

our $TRUE = 1;
our $FALSE = 0;

########################################################################
sub okay_response {
  my ($we_top) = @_;

  $we_top->destroy;
}

########################################################################
sub after_test {
  my ($test_number, $error, $message, $verbose) = @_;

  if ($error == EXIT_SUCCESS) {
    print ("Test $test_number PASSED\n") if $verbose;
  } else {
    print ("Test $test_number FAILED: \n$message\n") if $verbose;
  }
}

#########################################################################
sub create_file {
  my ($test_file, $verbiage, $verbose, $message_ref, $top_of_file_text) = @_;

  $$message_ref = "";

  if (!(open(OUTFILE, ">$test_file"))) {
      $$message_ref = "\n\[CF01\]Cannot open file $test_file:$!:\n";
      return (EXIT_FAILURE);
  }

  if (defined($top_of_file_text) && ($top_of_file_text ne "") ) {
    print OUTFILE $top_of_file_text;
  }

  if ($verbiage ne "") {
    print OUTFILE ("print \"${verbiage}\";");
  }

  if (!(close(OUTFILE))) {
      $$message_ref = "\n\[CF02\]Cannot close file $test_file:$!:\n";
      return (EXIT_FAILURE);
  }

  print ("\n\[CF03\]Created file $test_file\n") if $verbose;
  return (EXIT_SUCCESS);
}

#########################################################################
sub pp_hello_1 {
  my ($test_name_string,
      $os,
      $test_number,
      $test_dir,
      $hello_pl_file,
      $a_default_executable,
      $verbose,
      $message_ref,
      ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp hello'
  # The command should: # Pack 'hello' into executable 'a.exe'
  #
  #  . Create the file "hello" with the code that will
  #    print out the word "hello".
  #  . system pp hello
  #    a.exe will be created on windows
  #  . Back tick 'a' and collect the results.
  #
  #  Success if the result is "hello", failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $test_file = $test_dir . "/$hello_pl_file";
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";


  #.................................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }
  #.................................................................

  $error = create_file($test_file, "hello", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #.................................................................
  $cmd = 'pp ' . "\"$hello_pl_file\" ";
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }

  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_o_hello_hello_dot_pl {
  my ($test_name_string,
      $os,
      $test_number,
      $test_dir,
      $hello_pl_file,
      $hello_executable,
      $verbose,
      $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp -o hello hello.pl'
  # The command should: # Pack 'hello.pl' into executable 'hello.exe'
  # (The .exe assumes windows)
  # . Create hello.pl with the code that will print out the word "hello".
  # . system pp -o hello hello.pl
  # . Back tick the hello executable and collect the results.
  #
  #  Success if the result is "hello", failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $test_file = $test_dir . "/$hello_pl_file";
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";
  #.................................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string: cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($test_file, "hello", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: $$message_ref";
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }
  #.................................................................
  $cmd = "pp -o " . "\"$hello_executable\" \"$hello_pl_file\" ";
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string: cannot system $cmd\n";
    return (EXIT_FAILURE);
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $hello_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );
  if ($error == EXIT_FAILURE) {
   $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_executable?\n";
  }

  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_o_foo_foo_dot_pl_bar_dot_pl {
  my (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $foo_pl_file,
        $bar_pl_file,
        $foo_executable,
        $bar_executable,
        $verbose,
        $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  #Test
  #----
  #  Goal: # Test of 'pp -o foo foo.pl bar.pl'
  #  ----
  #  The command should: Pack 'foo.pl' and 'bar.pl' into 'foo'
  #
  #  Outline
  #  -------
  #  . Create foo.pl with the code that will print out the word "hello foo".
  #  . Create bar.pl with the code that will print out the word "hello bar".
  #  . system pp -o foo foo.pl bar.pl
  #  . Back tick ./foo and collect the results.  It should be "hello foo".
  #  . Copy foo to bar
  #  . Back tick ./bar and collect the results.  It should be "hello bar".
  #
  #Success if both "hello foo" and "hello bar" were appropriately collected.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string: " .
                      "cannot chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($foo_pl_file, "hello foo", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $error = create_file($bar_pl_file, "hello bar", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[015\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $cmd = "pp -o $foo_executable " . "\"$foo_pl_file\" \"$bar_pl_file\" ";
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string: cannot system $cmd)\n";
    return (EXIT_FAILURE);
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $foo_executable,
                           "hello foo",
                           $os,
                           $verbose,
                           $message_ref,
                         );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $foo_executable?\n";
    return ($error);
  }

  #.................................................................

  if(!(copy("$foo_executable", "$bar_executable"))) {
      $$message_ref = "\n\[030\]sub $test_name_string: cannot " .
                       "copy $foo_executable to $bar_executable\n";
      return (EXIT_FAILURE);
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $bar_executable,
                           "hello bar",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nIs there a $bar_executable?\n";
  }

  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_p_hello {

  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_dot_par,
       $verbose,
       $message_ref,
       $perl,
       $par,
     ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp -p hello'
  # The command should: Create a PAR hello, 'a.par'
  #
  # . Create file "hello" with the code that will print out the word "hello".
  # . system pp -p hello
  # . Back tick './par a' and collect the results.  It should be "hello".
  #
  #  Success if  "hello" was collected.  Failure otherwise
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $test_file = $hello_pl_file;
  my $back_tick_command_string = "$perl $par ";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot chdir " .
                      "$test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($test_file, "hello", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $cmd = "pp -p \"$test_file\"";
  # This should produce $a_default_dot_par
  if (system("$cmd")) {
    $$message_ref = "\n\[010\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_dot_par,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                         );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_dot_par?\n";
  }
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_p_minus_o_hello_dot_par_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $hello_par_file_with_dot_par,
       $verbose,
       $message_ref,
       $perl,
       $par,
     ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp -p -o file.par file'
  # The command should: Create a PAR file, 'file.par'
  #
  # . Create file "hello" with the code that will print out the word "hello".
  # . system pp -p -o hello.par hello
  # . Back tick './par hello.par' and collect the results.  It should
  #   be hello.
  # . Back tick './par hello' and collect the results.  It should
  #   once again be "hello".
  #  Success if "hello" was collected both times.  Failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $test_file = $hello_pl_file;
  my $back_tick_command_string = "$perl $par ";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($test_file, "hello", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $cmd = 'pp -p -o ' . " \"$hello_par_file_with_dot_par\" \"$test_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[010\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $hello_par_file_with_dot_par,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                         );
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_par_file_with_dot_par?\n";
    return ($error);
  }

  #.................................................................
  $back_tick_command_string = "$perl $par hello";
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "", # We don't want the sub to try
                               # to chmod +x anything.
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                         );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_par_file_with_dot_par?\n";
  }
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_o_hello_file_dot_par {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $hello_par_file_with_dot_par,
       $hello_par_file_no_dot_par,
       $hello_executable,
       $verbose,
       $message_ref,
       $perl,
       $par,
     ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp -o hello file.par'
  # The command should: Pack 'file.par' to executable 'hello'
  #
  # . Create file file.pl with the code that will print out the word "hello".
  # . system pp -p -o file.par file.pl
  #   This will create the par file file.par
  # . Back tick './par file.par' and collect the results.  It should
  #   be hello.
  # . Back tick './par file' and collect the results.  It should
  #   once again be "hello".
  # . system pp -o file file.par
  #   This will pack file.par into file.exe (Assuming windows)
  # . Back tick 'file' and collect the results.  It should again be "hello"
  #
  # Success if "hello" was collected all three times.  Failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $test_file = $hello_pl_file;
  my $back_tick_command_string = "$perl $par ";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($test_file, "hello", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  # Create a par file
  $cmd = 'pp -p -o ' . "\"$hello_par_file_with_dot_par\" \"$hello_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[010\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $hello_par_file_with_dot_par\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $hello_par_file_with_dot_par,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                         );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_par_file_with_dot_par?\n";
    return ($error);
  }

  #.................................................................

  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string . 'hello',
                           "", # We don't want the sub to try
                               # to chmod +x anything.
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                         );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_par_file_with_dot_par?\n";
    return ($error);
  }

  #.................................................................

  $cmd = 'pp -o ' . "\"$hello_executable\" \"$hello_pl_file\"  ";
  if (system("$cmd")) {
    $$message_ref = "\n\[010\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $hello_executable\n");
    }
  }
  #.................................................................
  $back_tick_command_string = "";
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $hello_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                         );
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_executable?\n";
  }

  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_S_minus_o_hello_file {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $hello_par_file_with_dot_par,
       $hello_executable,
       $verbose,
       $message_ref,
       $perl,
       $par,
     ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp -S -o hello hello.pl'
  # The command should: Create a PAR file, 'hello.par'
  #                     Pack 'hello.par' to executable 'hello'
  #
  # . Create file "hello.pl" with the code that will print out the
  #   word "hello".
  # . system pp -S -o hello hello.pl
  #   This will create the par file hello.par, and also pack hello.par
  #   into the executable "hello.exe". (Assuming windows)
  # . Back tick './par hello.par' and collect the results.  It should
  #   be "hello".
  # . Back tick './par hello' and collect the results.  It should be "hello".
  # . Back tick the created executable and collect the results.  It
  #   should again be "hello"
  #
  # Success if "hello" was collected all three times.  Failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "$perl $par ";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($hello_pl_file, "hello", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $cmd = 'pp -S -o ' . "\"$hello_executable\" \"$hello_pl_file\"  ";
  if (system("$cmd")) {
    $$message_ref = "\n\[010\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $hello_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $hello_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_executable?\n";
    return ($error);
  }

  #.................................................................

  $back_tick_command_string = "$perl $par hello";
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "", # We don't want the sub to try
                               # to chmod +x anything.
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_executable " .
                                                "and hello.par?\n";
    return ($error);
  }

  #.................................................................

  $back_tick_command_string = "";
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $hello_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_executable?\n";
  }
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_p_minus_o_out_dot_par_file {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $verbose,
       $message_ref,
       $perl,
       $par,
     ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp -p -o out.par file'
  #  The command should: Create 'out.par' from 'file'
  #                      Same as the test for 'pp -p -o file.par file'
  #                      except here we have renaming.
  #
  #  . Create file "file" with the code that will print out the word "hello".
  #  . system pp -p -o out.par file
  #  . Back tick './par out.par' and collect the results.  It should
  #    be "hello".
  #  . Back tick './par out' and collect the results.  It should be "hello".
  #
  # Success if "hello" was collected both times.  Failure otherwise
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "$perl $par ";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot chdir " .
                      "$test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($hello_pl_file, "hello", $verbose, $message_ref);
  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $cmd = 'pp -p -o out.par ' . "\"$hello_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[010\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created out.par\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'out.par',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce out.par?\n";
    return ($error);
  }

  #.................................................................

  $back_tick_command_string = "$perl $par out";
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "", # Don't let sub try to chmod  +x anything.
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce out.par?\n";
  }
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_B_with_small_minus_p_tests {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $verbose,
       $message_ref,
       $perl,
       $par,
     ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp -B -p -o out.par hello.pl'
  #  The command should: Create 'out.par' from 'file'
  #                      Same as the test for 'pp -p -o file.par file'
  #                      except here we bundle core modules.
  #
  #
  #  Since -B is the default except with -p or -P, the only way it
  #  seemed relevent was to test was by testing -B with -p, and by
  #  testing -B with -P.  I did.  -B or it's absense seems to mean
  #  nothing when creating either a .par file
  #  The file sizes with and without the -B within mere bytes
  #  of each other.
  #
  # Anyone know a way to really test -B?
  #
  # The four tests were:
  #      pp -p -o out.par hello.pl
  #      pp -B -p -o out.par hello.pl
  #
  # Again, the "-B" does not seem to have relevence.
  #
  # What I will do for now is to include the four tests and execute
  # the generated .par and just check for "hello" being printed out.
  #  I will do this even though it is a do-nothing test.  At least it
  # shows that the -B does not harm anything.
  #
  #
  # WARNING:  This tests only tests that the generated files produces
  #           are okay.  It does not check anything else.
  #
  #
  #  . Create the file hello.pl with the code that will print out the word
  #    "hello" and use strict.
  #  . system pp -B -p -o out_par_B.par hello.pl
  #    This creates out.par and bundles the core modules.
  #  . system pp -p -o out_par.par hello.pl
  #    This creates out.par
  #  . Back tick  './par out_par.par', './par out_par_B.par'
  #               './par out_par',     './par out_par_B'
  #
  # After all of the above, success if "hello" was collected each time.
  # Failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;

  my $back_tick_command_string = "$perl $par ";
  my $cmd = "";
  my $sub_test = 0;

  my $top_of_created_file_text = "use strict;\n";

  $$message_ref = "";

  print ("\n\nI will do test $test_name_string even though it DOES NOT \n");
  print ("REALLY TEST ANYTHING.  At least it may show that the -B  \n");
  print ("switch does not harm anything.\n\n");

  #.................................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #.................................................................
  $error = create_file($hello_pl_file,
                       "hello",
                       $verbose,
                       $message_ref,
                       $top_of_created_file_text);

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #.................................................................
  $cmd = 'pp -p -o out_par.par ' . "\"$hello_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created out_par.par\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'out_par.par',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce out_par.par?\n";
    return ($error);
  }
  #.................................................................

  $back_tick_command_string = "$perl $par out_par";
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "",  # Don't let sub try to chmod  +x anything.
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce out_par.par?\n";
    return ($error);
  }

  #.................................................................
  $cmd = 'pp -B -p -o out_par_B.par ' . "\"$hello_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created out_par_B.par\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'out_par_B.par',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce out_par_B.par?\n";
    return ($error);
  }

  #.................................................................

  $back_tick_command_string = "$perl $par out_par_B";
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "",  # Don't let sub try to chmod  +x anything.
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce out_par_B.par?\n";
  }

  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_B_with_large_minus_P_tests {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $verbose,
       $message_ref,
       $perl,
     ) = @_;

  #--------------------------------------------------------------------
  # Test of 'pp -B -P -o out.pl hello.pl'
  #  The command should: Create 'out.pl' from 'file'
  #                      Same as the test for 'pp -P -o file.pl file'
  #                      except here we bundle core modules.
  #
  #
  #  Since -B is the default except with -p or -P, the only way it
  #  seemed relevent was to test was by testing -B with -p, and by
  #  testing -B with -P.  I did.  -B or it's absense seems to mean
  #  nothing when creating either a .pl file
  #  The file sizes with and without the -B within mere bytes
  #  of each other.
  #
  # Anyone know a way to really test -B?
  #
  # The four tests I tried were:
  #      pp -P -o out.pl hello.pl
  #      pp -B -P -o out.pl hello.pl
  #
  # Again, the "-B" does not seem to have relevence.
  #
  # What I will do for now is to include the four tests and execute
  # the generated .pl and just check for "hello" being printed out.
  #  I will do this even though it is a do-nothing test.  At least it
  # shows that the -B does not harm anything.
  #
  #
  # WARNING:  This tests only tests that the generated files produces
  #           are okay.  It does not check anything else.
  #
  #
  #  . Create the file hello.pl with the code that will print out the word
  #    "hello" and use strict.
  #  . system pp -B -P -o out_pl_B.pl hello.pl
  #    This creates out_pl_B.pl and bundles the core modules.
  #  . system pp -P -o out_pl.pl hello.pl
  #    This creates out.pl
  #  . Back tick  'perl out_pl.pl', 'perl out_pl_B.pl'
  #
  # After all of the above, success if "hello" was collected each time.
  # Failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "$perl ";
  my $cmd = "";
  my $sub_test = 0;

  my $top_of_created_file_text = "use strict;\n";

  $$message_ref = "";

  print ("\n\nI will do test $test_name_string even though it DOES NOT \n");
  print ("REALLY TEST ANYTHING.  At least it may show that the -B  \n");
  print ("switch does not harm anything.\n\n");

  #.................................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #.................................................................
  $error = create_file($hello_pl_file,
                       "hello",
                       $verbose,
                       $message_ref,
                       $top_of_created_file_text);

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }
  #.................................................................

  $cmd = 'pp -P -o out_pl.pl ' . "\"$hello_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created out_pl.pl\n");
    }
  }

  #.................................................................

  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'out_pl.pl',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce out_pl.pl?\n";
    return ($error);
  }

  #.................................................................

  $cmd = 'pp -B -P -o out_pl_B.pl ' . "\"$hello_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created out_pl_B.pl\n");
    }
  }
  #.................................................................

  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'out_pl_B.pl',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce out_pl_B.pl?\n";
  }

  #.................................................................
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_e_print_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -e "print \"hello\n\";"
  # ----
  # The command should: Create 'a.exe' if windows
  #
  # Outline
  # -------
  # . system pp -e "print \"hello\n\";"
  # . back tick 'a' and collect the results
  # Success if "hello" was collected.  Failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $cmd = 'pp -e "print \"hello\n\";" ';
  if (system("$cmd")) {
    $$message_ref = "\n\[010\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }


  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_p_minus_e_print_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $verbose,
       $message_ref,
       $perl,
       $par,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -p -e "print \"hello\n\";"
  # ----
  # The command should: Create 'a.par'
  #
  # Outline
  # -------
  # system pp -p -e "print \"hello\n\";"
  # back tick 'par a.par' and collect the results
  # back tick 'par a' and collect the results
  #
  # Success if "hello" was collected each time.  Failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;

  my $back_tick_command_string = "$perl $par ";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot" .
                      " chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $cmd = 'pp -p -e "print \"hello\n\"";';
  if (system(" $cmd ")) {
    $$message_ref = "\n\[015\]sub $test_name_string Cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created a.par\n");
    }
  }


  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "a.par",
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce a.par?\n";
    return ($error);
  }

  #.................................................................

  $back_tick_command_string = "$perl $par a";
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "",
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce a.par?\n";
  }
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_P_minus_e_print_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $verbose,
       $message_ref,
       $perl,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -P -e "print \"hello\n\";"
  # ----
  # The command should: Create perl script 'a.pl'
  #
  # Outline
  # -------
  # system pp -P -e "print \"hello\n\";"
  # back tick 'perl a.pl' and collect the results
  #
  # Success if "hello" was collected.  Failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "$perl ";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";


  #.................................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #.................................................................
  $cmd = 'pp -P -e "print \"hello\n\";" ';
  if (system(" $cmd ")) {
    $$message_ref = "\n\[010\]sub $test_name_string Cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created a.par\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'a.pl',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce a.pl?\n";
  }
  return ($error);
  #.................................................................
 
}

#########################################################################
sub pp_minus_c_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -c hello
  # ----
  # The command should: Create executable 'a.exe'
  #
  # WARNING:  This tests only tests that the executable produced
  #           is okay.  It does not check anything else.
  #
  # Outline
  # -------
  # Create a file that will print "hello".
  # system pp -c hello
  # back tick 'a' and collect the results
  #
  # Success if "hello" was collected.  Failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($hello_pl_file,
                       "hello",
                       $verbose,
                       $message_ref);

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $cmd = 'pp -c ' . " \"$hello_pl_file\" ";
  if (system(" $cmd ")) {
    $$message_ref = "\n\[015\]sub $test_name_string Cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_x_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -x hello
  # ----
  # The command should: Create executable 'a.exe'
  #                     Also it will check dependencies
  #                     from "perl hello" during execution
  #
  # WARNING:  This tests only tests that the executable produced
  #           is okay.  It does not check anything else.
  #
  # Outline
  # -------
  # Create a file that will print "hello".
  # system pp -x hello
  # back tick 'a' and collect the results
  #
  # Success if "hello" was collected.  Failure otherwise.
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($hello_pl_file,
                       "hello",
                       $verbose,
                       $message_ref);

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $cmd = 'pp -x  ' . "\"$hello_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string Cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }

  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_n_minus_x_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -n -x hello
  # ----
  # The command should: Create executable 'a.exe'
  #                     Also it will check dependencies
  #                     from "perl hello" during execution
  #
  # WARNING:  This tests only tests that the executable produced
  #           is okay.  It does not check anything else.
  #
  # Outline
  # -------
  # Create a file that will print "hello".
  # system pp -n -x hello
  # back tick 'a' and collect the results
  #
  # Success if "hello" was collected.  Failure otherwise.
  #--------------------------------------------------------------------
  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  $error = create_file($hello_pl_file,
                       "hello",
                       $verbose,
                       $message_ref);

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  $cmd = 'pp -n -x  ' . "\"$hello_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }


  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_I_foo_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -I /foo hello.pl
  # ----
  # The command should:
  #    Add the given directory to the perl library file search path.
  #
  # Outline
  # -------
  # . Create a temp dir "hidden_dir" that Perl and PP would not know about.
  # . Create a module in it called hidden_print.pm that has a
  #   subroutine called "hidden_print", that takes in a string
  #   to print.
  # . Put the new module in the new temp dir.
  # . Create a file foo in the current dir with code that will
  #   invoke hidden_print
  # . system 'pp foo.pl'
  #   The file a.exe is created on windows.
  # . Back tick 'a'
  #   The result should be something like: "Can't locate hidden_print"
  # . system pp -I "hidden_dir" foo.pl
  #   Once again, a.exe is created on windows
  # . Back tick 'a' and collect the results.
  # . The result should be "hello"
  # . Copy the a.exe to a different subdirectory
  # . chdir to the new subdirectory
  # . Back tick a.exe
  # . The result should be "hello"
  # . Remove the hidden_print file.
  # . Back tick 'a' again and collect the results.
  #   It should still pass.
  #
  # Success if as described above.  Failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $hidden_dir = $test_dir . "/$SUBDIR1";
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

#..............................................
  my $foo_top_of_file_text = '
use hidden_print;
hidden_print("hello");
';
#..............................................

#..............................................
  my $hidden_top_of_file_text = '
package hidden_print;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("hidden_print");

sub hidden_print {
  my ($text_to_print) = shift;
  print ("$text_to_print\n");
}
1;
';
#..............................................

  $$message_ref = "";

  #..........................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $hidden_dir . "/hidden_print\.pm",
                         "",
                         $verbose,
                         $message_ref,
                         $hidden_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file( $test_dir . "/foo\.pl",
                        "",
                        $verbose,
                        $message_ref,
                        $foo_top_of_file_text,
                      );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[015\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }
  #..........................................................
  $cmd = 'pp foo.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[010\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }
  print ("\nThe Line Below SHOULD BE  \"Can\'t locate \.\.\. ");
  print (" along with a \"BEGIN failed \.\.\. \" line\n");

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );
  ####################################################
  ##### This SHOULD fail, so don't return
  ##### return ($error) if ($error == EXIT_FAILURE);
  ###################################################
  $$message_ref = ""; # Wipe out the nasty messages from the
                      # last back tick command.

  print ("\n"); # To add a line after the above expected error messages.

  #.................................................................
  $cmd = 'pp -I  ' . "\"$hidden_dir\" foo.pl";
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #..........................................................
  print ("About to test in a different subdir\n") if ($verbose);
  $error = test_in_further_subdir (
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $SUBDIR1,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to remove a file and try executable again\n") if ($verbose);
  $error = remove_file_and_try_executable_again
                                  (
                                    "$test_dir/foo.pl", # File to remove
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  return ($error);
}

#########################################################################
sub pp_minus_lib_foo_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp --lib /foo hello.pl
  # ----
  # The command should:
  #    Add the given directory to the perl library file search path.
  #
  # Outline
  # -------
  #   First, to give an outline of the directories and files this
  #   test will create.
  #    -----------------------------------------------------
  #   | current working test dir/foo.pl                     |
  #   |   foo.pl has "use hidden_print;"                    |
  #   |-----------------------------------------------------|
  #   | current working test dir/$SUBDIR1/hidden_print.pm   |
  #   |   hidden_print.pm prints the string passed in.      |
  #    -----------------------------------------------------
  #
  # . In a dir $SUBDIR1 that PP would not know about, create
  #   a module called hidden_print.pm that has a subroutine
  #   called "hidden_print", that takes in a string to
  #   print, and prints it.
  # . In the current directory, create a file foo.pl that invokes
  #   hidden_print with the text "hello".
  # . system pp --lib $SUBDIR1 foo.pl
  #   An  a.exe is created on windows
  # . Back tick 'a' and collect the results.
  # . The result should be "hello"
  #
  # Success if as described above.  Failure otherwise.
  #
  #--------------------------------------------------------------------


  my $error = EXIT_FAILURE;
  my $foo_dir = $test_dir . "/$SUBDIR1";
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;
#..............................................
  my $foo_top_of_file_text = '
use hidden_print;
hidden_print("hello");
';

#..............................................
  my $hidden_print_top_of_file_text = '
package hidden_print;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("hidden_print");

sub hidden_print {
  my ($text_to_print) = shift;
  print ("$text_to_print");
}
1;
';
#..............................................

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $foo_dir . "/hidden_print\.pm",
                         "",
                         $verbose,
                         $message_ref,
                         $hidden_print_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file( $test_dir . "/foo\.pl",
                        "",
                        $verbose,
                        $message_ref,
                        $foo_top_of_file_text,
                      );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[015\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }
  #..........................................................
  $cmd = 'pp foo.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }
  print ("\n\[025\]The Line Below SHOULD BE  \"Can\'t locate \.\.\. ");
  print (" along with a \"BEGIN failed \.\.\. \" line\n");

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  ########################################################
  ##### This SHOULD fail, so don't do the usual return
  ##### return ($error) if ($error == EXIT_FAILURE);
  ########################################################
  $$message_ref = ""; # Wipe out the nasty messages from the
                      # last back tick command.
  print ("\n"); # To add a line after the above expected error messages.

  #.................................................................
  $cmd = 'pp --lib  ' . "$foo_dir foo.pl";
  if (system("$cmd")) {
    $$message_ref = "\n\[035\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #..........................................................
  print ("About to test in a different subdir\n") if ($verbose);
  $error = test_in_further_subdir (
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $SUBDIR1,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to remove a file and try executable again\n") if ($verbose);
  $error = remove_file_and_try_executable_again
                                  (
                                    "$foo_dir/hidden_print\.pm",
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  return ($error);

}

#########################################################################
sub pp_minus_I_foo_minus_I_bar_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -I /foo -I bar hello.pl
  # ----
  # The command should:
  #    Add the given directory to the perl library file search path.
  #
  # Outline
  # -------
  #   First, to give an outline of the directories and files this
  #   test will create.
  #    ----------------------------------------------------------
  #   | current working test dir/foo.pl                          |
  #   |   foo.pl has "use hidden_print_caller;     "             |
  #   |----------------------------------------------------------|
  #   | current working test dir/$SUBDIR1/hidden_print_caller.pm |
  #   |   hidden_print_caller.pm has "use hidden_print;"         |
  #   |----------------------------------------------------------|
  #   | current working test dir/$SUBDIR2/hidden_print.pm        |
  #   |   hidden_print.pm prints the string passed in.           |
  #    ----------------------------------------------------------
  #
  # . In subdir $SUBDIR1, create
  # . Create a module in $SUBDIR2 called hidden_print.pm that
  #   has a subroutine called "hidden_print", that takes in a
  #   string to print, and prints it.
  # . Create a module in $SUBDIR1 called hidden_print_caller.pm
  #   that has a routine called hidden_print_caller that
  #   takes in a string to print, and invokes hidden_print to print it.
  # . In the current directory, create a file foo.pl that invokes
  #   hidden_print_caller with the text "hello".
  # . system 'pp foo.pl'
  #   The file a.exe is created on windows.
  # . Back tick 'a'
  #   The result should be: Nothing.
  # . system pp -I foo -I bar foo.pl
  #   Once again, a.exe is created on windows
  # . Back tick 'a' and collect the results.
  # . The result should be "hello"
  # . Copy a.exe to a different directory
  # . chdir to the directory.
  # . Back tick 'a.exe' and collect the results.
  # . The result should be "hello"
  #
  # Success if as described above.  Failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $foo_dir = $test_dir . "/$SUBDIR1";
  my $bar_dir = $test_dir . "/$SUBDIR2";
  my $foo_dir_file = $foo_dir . "/hidden_print_caller\.pm";
  my $bar_dir_file = $bar_dir . "/hidden_print\.pm";
  my $foo_file = $test_dir . "/foo\.pl";

  my $further_subdir = "";
  my $further_file = "";

  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

#..............................................
  my $foo_top_of_file_text = '
use hidden_print_caller;
hidden_print_caller("hello");
';
#..............................................

#..............................................
  my $hidden_print_caller_top_of_file_text = '
package hidden_print_caller;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("hidden_print_caller");

use hidden_print;

sub hidden_print_caller {
  my ($text_to_print) = shift;
  hidden_print ("$text_to_print");
}
1;
';
#..............................................

#..............................................
  my $hidden_print_top_of_file_text = '
package hidden_print;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("hidden_print");

sub hidden_print {
  my ($text_to_print) = shift;
  print ("$text_to_print");
}
1;
';
#..............................................

  $$message_ref = "";
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $bar_dir_file,
                         "",
                         $verbose,
                         $message_ref,
                         $hidden_print_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................

  $error = create_file(  $foo_dir_file,
                         "",
                         $verbose,
                         $message_ref,
                         $hidden_print_caller_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[020\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file( $foo_file,
                        "",
                        $verbose,
                        $message_ref,
                        $foo_top_of_file_text,
                      );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[030\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $cmd = 'pp ' . "\"$foo_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[040\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("\n\[045\]sub $test_name_string created $a_default_executable\n");
    }
  }
  #.................................................................
  print ("\nThe Line Below SHOULD BE  \"Can\'t locate \.\.\. ");
  print (" along with a \"BEGIN failed \.\.\. \" line\n");

  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  ########################################################
  ##### This SHOULD fail, so don't do the usual return
  ##### return ($error) if ($error == EXIT_FAILURE);
  ########################################################
  $$message_ref = ""; # Wipe out the nasty messages from the
                      # last back tick command.
  print ("\n"); # To add a line after the above expected error messages.

  #.................................................................
  $cmd = 'pp -I ' . "\"$foo_dir\" -I \"$bar_dir\" \"$foo_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[055\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }

 #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to test in a different subdir\n") if ($verbose);
  $error = test_in_further_subdir (
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $SUBDIR1,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to remove a file and try executable again\n") if ($verbose);
  $error = remove_file_and_try_executable_again
                                  (
                                    $bar_dir_file,
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_lib_foo_minus_lib_bar_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;


  #--------------------------------------------------------------------
  # Goal: Test of pp --lib /foo --lib bar hello.pl
  # ----
  # The command should:
  #    Add the given directory to the perl library file search path.
  #
  # Outline
  # -------
  #   First, to give an outline of the directories and files this
  #   test will create.
  #    ----------------------------------------------------------
  #   | current working test dir/foo.pl                          |
  #   |   foo.pl has "use hidden_print_caller;"                  |
  #   |----------------------------------------------------------|
  #   | current working test dir/$SUBDIR1/hidden_print_caller.pm |
  #   |   hidden_print_caller.pm has "use hidden_print;"         |
  #   |----------------------------------------------------------|
  #   | current working test dir/$SUBDIR2/hidden_print.pm        |
  #   |   hidden_print.pm prints the string passed in.           |
  #    ----------------------------------------------------------
  #
  # . Create a module in $SUBDIR2 called hidden_print.pm that
  #   has a subroutine called "hidden_print", that takes in a
  #   string to print, and prints it.
  # . Create a module in $SUBDIR1 called hidden_print_caller.pm
  #   that has a routine called hidden_print_caller that
  #   takes in a string to print, and invokes hidden_print to print it.
  # . In the current directory, create a file foo.pl that invokes
  #   hidden_print_caller with the text "hello".
  # . system pp --lib foo --lib bar foo.pl
  #   Once again, a.exe is created on windows
  # . Back tick 'a' and collect the results.
  # . The result should be "hello"
  #
  # Success if as described above.  Failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $foo_dir = $test_dir . "/$SUBDIR1";
  my $bar_dir = $test_dir . "/$SUBDIR2";
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

#..............................................
  my $foo_top_of_file_text = '
use hidden_print_caller;
hidden_print_caller("hello");
';
#..............................................

#..............................................
  my $hidden_print_caller_top_of_file_text = '
package hidden_print_caller;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("hidden_print_caller");

use hidden_print;

sub hidden_print_caller {
  my ($text_to_print) = shift;
  hidden_print ("$text_to_print");
}
1;
';
#..............................................

#..............................................
  my $hidden_print_top_of_file_text = '
package hidden_print;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("hidden_print");

sub hidden_print {
  my ($text_to_print) = shift;
  print ("$text_to_print");
}
1;
';
#..............................................
  my $further_subdir = "";

  $$message_ref = "";
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $bar_dir . "/hidden_print\.pm",
                         "",
                         $verbose,
                         $message_ref,
                         $hidden_print_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $foo_dir . "/hidden_print_caller\.pm",
                         "",
                         $verbose,
                         $message_ref,
                         $hidden_print_caller_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[015\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file( "$test_dir/foo\.pl",
                        "",
                        $verbose,
                        $message_ref,
                        $foo_top_of_file_text,
                      );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[020\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }
  #..........................................................
  $cmd = 'pp foo.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }

  #.................................................................
  print ("\nThe Line Below SHOULD BE  \"Can\'t locate \.\.\. ");
  print (" along with a \"BEGIN failed \.\.\. \" line\n");

  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  ########################################################
  ##### This SHOULD fail, so don't do the usual return
  ##### return ($error) if ($error == EXIT_FAILURE);
  ########################################################
  $$message_ref = ""; # Wipe out the nasty messages from the
                      # last back tick command.
  print ("\n"); # To add a line after the above expected error messages.

  #.................................................................
  $cmd = 'pp --lib ' . "\"$foo_dir\"" .
         ' --lib ' . "\"$bar_dir\"" . ' foo.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[030\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string created $a_default_executable\n");
    }
  }

 #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to test in a different subdir\n") if ($verbose);
  $error = test_in_further_subdir (
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $SUBDIR1,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to remove a file and try executable again\n") if ($verbose);
  $error = remove_file_and_try_executable_again
                                  (
                                    "$test_dir/foo\.pl",
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  return ($error);
  #.................................................................

}
#########################################################################
sub pp_minus_M_foo_hidden_print_foo {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -M module foo.pl
  # The command should:  Add the given module
  #
  # Outline
  # -------
  #    -----------------------------------------------------
  #   | current working test dir/foo.pl                     |
  #   |   foo.pl has "use hidden_print;"                    |
  #   |-----------------------------------------------------|
  #   | current working test dir/$SUBDIR1/hidden_print.pm   |
  #   |   hidden_print.pm prints the string passed in.      |
  #   |   It is "package hidden_print;"                     |
  #    -----------------------------------------------------
  #
  # . Create $SUBDIR1/hidden_print.pm that has a subroutine
  #   called "hidden_print", that takes in a  string to
  #   print, and prints it.
  # . In the current directory, create a file foo.pl that invokes
  #   hidden_print with the text "hello".
  # . system pp foo.pl
  #   An  a.exe is created on windows
  # . Back tick the created executable and collect the results.
  #   There will be error
  #   messages on the screen, and the results will be: nothing.
  # . system pp -M $SUBDIR1::hidden_print foo.pl
  #   An  a.exe is created on windows
  # . Back tick the created executable and collect the results.
  # . The result should be "hello"
  # . Remove the included module
  # . Once again, back the created executable
  #   The result should still be hello.
  # Success if as described above.  Failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $foo_dir = $test_dir . "/$SUBDIR1";
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;
  my $hidden_print_file = "$foo_dir/hidden_print\.pm";

#..............................................
  my $foo_top_of_file_text = '
use ' . $SUBDIR1 . '::hidden_print;
hidden_print("hello");
';

#..............................................
  my $hidden_print_top_of_file_text = '
package ' . $SUBDIR1 . '::hidden_print;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("hidden_print");

sub hidden_print {
  my ($text_to_print) = shift;
  print ("$text_to_print");
}
1;
';
#..............................................
  my $further_subdir = "";

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $hidden_print_file,
                         "",
                         $verbose,
                         $message_ref,
                         $hidden_print_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file( $test_dir . "/foo\.pl",
                        "",
                        $verbose,
                        $message_ref,
                        $foo_top_of_file_text,
                      );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[015\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }
  #..........................................................
  $cmd = 'pp foo.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #..........................................................

   $cmd = "pp -M  ${SUBDIR1}::hidden_print foo.pl";
  if (system("$cmd")) {
    $$message_ref = "\n\[025\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to test in a different subdir\n") if ($verbose);
  $error = test_in_further_subdir (
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $SUBDIR1,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to remove a file and try executable again\n") if ($verbose);
  $error = remove_file_and_try_executable_again
                                  (
                                    $hidden_print_file,
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_M_foo_minus_M_bar_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of pp -M hidden1 -M hidden2 foo.pl
  # The command should:  Add the given modules
  #
  # Outline
  # -------
  #    -----------------------------------------------------
  #   | current working test dir/foo.pl                     |
  #   |   foo.pl has "use foo_1;"                           |
  #   |              "use bar_1;"                           |
  #   |              foo_1;                                 |
  #   |              bar_1;                                 |
  #   |-----------------------------------------------------|
  #   | current working test dir/$SUBDIR1/foo_1.pm          |
  #   |   foo_1.pm will print "hello_foo"                   |
  #   |-----------------------------------------------------|
  #   | current working test dir/$SUBDIR2/bar_1.pm          |
  #   |   bar_1.pm will print "hello_bar"                   |
  #    -----------------------------------------------------
  #
  # . Create $SUBDIR1/foo_1.pm that has a subroutine called "foo_1",
  #   that prints hello_foo.
  # . Create $SUBDIR2/bar_1.pm that has a subroutine called "bar_1",
  #   that prints hello_bar.
  # . In the current directory, create a file foo.pl that invokes
  #   foo_1 and bar_1.
  # . system pp foo.pl
  #   An  a.exe is created on windows
  # . Back tick 'a' and collect the results.  There will be error
  #   messages on the screen, and the results will be: nothing.
  # . system pp -M $SUBDIR1::foo_1 -M $SUBDIR2::bar_1 foo.pl
  #   An  a.exe is created on windows
  # . Back tick 'a' and collect the results.
  # . The result contain "hello_foo" and "hello_bar".
  #
  # Success if as described above.  Failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;
  my $foo_dir = $test_dir . "/$SUBDIR1";
  my $bar_dir = $test_dir . "/$SUBDIR2";
  my $subdir_foo_file =  "$foo_dir/foo_1\.pm";

#..............................................
  my $foo_top_of_file_text = '
use ' . $SUBDIR1 . '::foo_1;
use ' . $SUBDIR2 . '::bar_1;
foo_1;
bar_1;
';

#..............................................
  my $foo_1_top_of_file_text = '
package ' . $SUBDIR1 . '::foo_1;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("foo_1");

sub foo_1 {
  print ("hello_foo");
}
1;
';

#..............................................
  my $bar_1_top_of_file_text = '
package ' . $SUBDIR2 . '::bar_1;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = ("bar_1");

sub bar_1 {
  print ("hello_bar");
}
1;
';
#..............................................
  my $further_subdir = "";

  $$message_ref = "";

  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot" .
                      " chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $subdir_foo_file,
                         "",
                         $verbose,
                         $message_ref,
                         $foo_1_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $bar_dir . "/bar_1\.pm",
                         "",
                         $verbose,
                         $message_ref,
                         $bar_1_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[015\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file( $test_dir . "/foo\.pl",
                        "",
                        $verbose,
                        $message_ref,
                        $foo_top_of_file_text,
                      );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[020\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }
  #..........................................................
  $cmd = 'pp foo.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[025\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello_foohello_bar",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  $cmd = "pp -M ${SUBDIR1}::foo_1 -M ${SUBDIR2}::bar_1 foo.pl";

  if (system("$cmd")) {
    $$message_ref = "\n\[040\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello_foohello_bar",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to test in a different subdir\n") if ($verbose);
  $error = test_in_further_subdir (
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $SUBDIR1,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello_foohello_bar",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to remove a file and try executable again\n") if ($verbose);
  $error = remove_file_and_try_executable_again
                                  (
                                    $subdir_foo_file,
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "hello_foohello_bar",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  return ($error);
  #.................................................................

}


#########################################################################
sub pp_minus_M_abbrev_dot_pl_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of 'pp -M abbrev.pl hello.pl'
  # ----
  # Extra files under @INC
  # pp will search for abbrev.pl in @INC
  #
  # Outline
  # -------
  # . Create $SUBDIR1\abbrev.pl to print out "abbrev_dot_pl".
  # . Create the perl file hello.pl in the current directory that
  #   will "do "invoked.pl";
  # Note: Back tick or exec does not work.  Only 'do'.
  # . Build the executable like this:
  #       pp -M $SUBDIR1\abbrev.pl hello.pl
  #   The file a.exe will be produced
  # . Back tick a.exe and collect the results.
  #
  # Success if the result are abbrev_dot_pl, failure otherwise
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;
  my $subdir_abbrev_file = $SUBDIR1 . "/abbrev.pl";

  $test_dir =~ s!\\!\/!g;
  #..............................................
  my $hello_top_of_file_text = '
my $result = do '  .   "\"$subdir_abbrev_file\";" . '
print $result;
';

#..............................................
  my $abbrev_top_of_file_text = '
print "abbrev_dot_pl";
';

#..............................................
  my $further_subdir = "";

  $$message_ref = "";


  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $test_dir . "/$hello_pl_file",
                         "",
                         $verbose,
                         $message_ref,
                         $hello_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $subdir_abbrev_file,
                         "",
                         $verbose,
                         $message_ref,
                         $abbrev_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[015\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $cmd = 'pp ' . "$hello_pl_file";
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }


  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "abbrev_dot_pl",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
    return($error);
  }

  #.................................................................
  $cmd = "pp -M $test_dir/$SUBDIR1/abbrev.pl $hello_pl_file";
  if (system("$cmd")) {
    $$message_ref = "\n\[025\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "abbrev_dot_pl",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
    return($error);
  }

  #.................................................................
  print ("About to test in a different subdir\n") if ($verbose);
  $error = test_in_further_subdir (
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $SUBDIR1,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "abbrev_dot_pl",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................
  print ("About to remove a file and try executable again\n") if ($verbose);
  $error = remove_file_and_try_executable_again
                                  (
                                    $subdir_abbrev_file,
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $back_tick_command_string,
                                    $a_default_executable,
                                    "abbrev_dot_pl",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_M_abbrev_dot_pl_minus_o_hello_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $hello_executable,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of 'pp -M dir/abbrev.pl -o hello.exe hello.pl'
  # ----
  # Extra files under @INC
  # pp will search for abbrev.pl in @INC, so if you have a
  #        -M dir/abbrev.pl
  # it will be included.
  #
  # Outline
  # -------
  # . Create dir/abbrev.pl to print out "abbrev_dot_pl".
  # . Create the perl file hello.pl in the current directory that
  #   will system "abgrev.pl".
  # . Build the executable like this:
  #       pp -M dir/abbrev.pl -o hello.exe hello.pl
  #   The file hello.exe will be produced
  # . Back tick hello.exe and collect the results.
  #
  # Success if the result are abbrev_dot_pl, failure otherwise
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;
  my $subdir_abbrev_file = $SUBDIR1 . "/abbrev.pl";

  $test_dir =~ s!\\!\/!g;
  #..............................................
  my $hello_top_of_file_text = '
my $result = do ' . "\"$subdir_abbrev_file\";" . '
print $result;
';

#..............................................
  my $abbrev_top_of_file_text = '
print "abbrev_dot_pl";
';

#..............................................
  my $further_subdir = "";

  $$message_ref = "";


  #..........................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $test_dir . "/$hello_pl_file",
                         "",
                         $verbose,
                         $message_ref,
                         $hello_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[030\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $subdir_abbrev_file,
                         "",
                         $verbose,
                         $message_ref,
                         $abbrev_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[035\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $cmd = 'pp ' . "$hello_pl_file";
  if (system("$cmd")) {
    $$message_ref = "\n\[040\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "abbrev_dot_pl",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
    return ($error);
  }

  #.................................................................

  $cmd = "pp -M $test_dir/$SUBDIR1/abbrev.pl -o $hello_executable $hello_pl_file";
  if (system("$cmd")) {
    $$message_ref = "\n\[050\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$hello_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $hello_executable,
                           "abbrev_dot_pl",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_executable?\n";
    return ($error);
  }

  #.................................................................
  print ("About to test in a different subdir\n") if ($verbose);
  $error = test_in_further_subdir (
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $SUBDIR1,
                                    $back_tick_command_string,
                                    $hello_executable,
                                    "abbrev_dot_pl",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $hello_executable?\n";
    return ($error);
  }

  #.................................................................
  print ("About to remove a file and try executable again\n") if ($verbose);
  $error = remove_file_and_try_executable_again
                                  (
                                    $subdir_abbrev_file,
                                    $test_number,
                                    $sub_test++,
                                    $test_name_string,
                                    $test_dir,
                                    $back_tick_command_string,
                                    $hello_executable,
                                    "abbrev_dot_pl",
                                    $os,
                                    $verbose,
                                    $message_ref,
                                  );

  #.................................................................
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_minus_X_module_foo {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $foo_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of 'pp -X Foo::Bar foo'
  # ----
  # The command should: Exclude a module (notice space after -X)
  #
  # Outline
  # -------
  # . Create the perl file test_X_foo_bar with code that will utilize
  #   the module "basename".  That is, use File::Basename;
  # . Have the line "print basename($^X)" in the perl file
  #   to invoke basename.
  # . system "pp -X File::Basename test_X_foo_bar".
  # . Back tick the created 'a' and collect the results.
  # .
  # Success if the first result is "perl.exe" on Windows, and success
  # if it fails to give "perl.exe" the second time.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  #..............................................
  my $foo_top_of_file_text = '
use File::Basename;
my $basename = basename($^X);
print $basename;
';

#..............................................
  my $further_subdir = "";

#..............................................

  $$message_ref = "";

  print ("\n\nI will do test $test_name_string even though it DOES NOT \n");
  print ("REALLY TEST ANYTHING.  At least it may show that the -X  \n");
  print ("switch does not harm anything.\n\n");

#..............................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $test_dir . "/$foo_pl_file",
                         "",
                         $verbose,
                         $message_ref,
                         $foo_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $cmd = 'pp ' . "\"$test_dir/$foo_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           'perl',
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
     return ($error);
  }

  #.................................................................

  $cmd = 'pp -X File::Basename ' . "\"$test_dir/$foo_pl_file\"";
  if (system("$cmd")) {
    $$message_ref = "\n\[020\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #.................................................................
  # Note: If Basename were really excluded this would fail.
  #          But it doesn't!!!
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           'perl',
                           $os,
                           $verbose,
                           $message_ref,
                        );

  #.................................................................
  return ($error);

}


#########################################################################
sub pp_minus_r_hello {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of 'pp -r hello.pl'
  # ----
  # The command should: Pack hello.pl into a.exe, and then run a.exe
  #                     after packaging it.
  #
  # Outline
  # -------
  # . Create the perl file hello.pl with code that will print "hello".
  # . Back tick "pp -r hello.pl" and collect the results.
  # 
  # Success if "hello", failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;

  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  #..............................................
  my $hello_top_of_file_text = '
print "hello";
';
#..............................................
  $$message_ref = "";

  #..............................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $hello_pl_file,
                         "",
                         $verbose,
                         $message_ref,
                         $hello_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $back_tick_command_string = "pp -r $hello_pl_file";
  $cmd = $back_tick_command_string; # Just to keep our code 
                                    # template here consistent.
  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "", # No separate executable name this time
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }

  #.................................................................
  return ($error);

}

#########################################################################
sub pp_minus_r_hello_a_b_c {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of 'pp -r hello.pl a b c'
  # ----
  # The command should: Pack hello.pl into a.exe, and then run a.exe
  #                     after packaging it.  The a b c are parameters.
  #
  # Outline
  # -------
  # . Create the perl file hello.pl with code that will print "hello".
  # . Back tick "pp -r hello.pl" and collect the results.
  # 
  # Success if "hello", failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;

  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  #..............................................
  my $hello_top_of_file_text = '
print "hello $ARGV[0] $ARGV[1] $ARGV[2]";
';
#..............................................
  $$message_ref = "";

#..............................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $hello_pl_file,
                         "",
                         $verbose,
                         $message_ref,
                         $hello_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $back_tick_command_string = 
                "pp -r $hello_pl_file \"one\" \"two\" \"three\"";
  $cmd = $back_tick_command_string; # Just to keep our code 
                                    # template here consistent.
  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "", # No separate executable name this time
                           "hello one two three",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }

  #.................................................................
  return ($error);

}

#########################################################################
sub pp_hello_to_log_file {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of 'pp hello.pl --log=c' and 'pp -L c hello.pl'
  # ----
  # The command should: Pack hello.pl into a.exe, and then run a.exe
  #                     after packaging it.  The a b c are parameters.
  #
  # Outline
  # -------
  # . Create the perl file hello.pl with code that will print "hello".
  # . Back tick "pp hello.pl --log=c" and collect the results.
  # 
  # THIS IS A DO-NOTHING TEST ... SO FAR ...
  # At least it will show that --log=c does no harm
  # 
  # Success if "hello", failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;

  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;
  my $log_file = 'c.txt';

  #..............................................
  my $hello_top_of_file_text = '
print "hello";
';
#..............................................
  $$message_ref = "";

  print ("\n\nI will do test $test_name_string even though it DOES NOT \n");
  print ("REALLY TEST ANYTHING.  At least it may show that the  --log=c \n");
  print ("switch does not harm anything.\n\n");

#..............................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $hello_pl_file,
                         "",
                         $verbose,
                         $message_ref,
                         $hello_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $cmd = 'pp hello.pl -v --log=' . "$log_file";
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #..........................................................
  if (-e($log_file) && (-s($log_file) > 10)) {
    if ($verbose) {
      print ("The log file $log_file has lines in it\n");
    }
  } else {
    $$message_ref = 
        "sub ${test_name_string}_$sub_test command $cmd \n"  .
        "did not produce file $log_file or $log_file does not have " .
        "more than 10 bytes in it\n";
    return (EXIT_FAILURE);
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
    return ($error);
  }

  #..........................................................
  if (!(unlink($a_default_executable))) {
    $$message_ref = 
        "Test ${test_name_string}_$sub_test "   .
        "cannot remove file $a_default_executable\n";
    return(EXIT_FAILURE);
  }
  #..........................................................
  $log_file = 'd.txt';
  $cmd = 'pp -L ' . $log_file .  ' -v hello.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #..........................................................
  if (-e($log_file) && (-s($log_file) > 10)) {
    if ($verbose) {
      print ("The log file $log_file has lines in it\n");
    }
  } else {
    $$message_ref = 
        "sub ${test_name_string}_$sub_test command $cmd \n"  .
        "did not produce file $log_file or $log_file does not have " .
        "more than 10 bytes in it\n";
    return (EXIT_FAILURE);
  }


  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }

  #.................................................................
  return ($error);

}

#########################################################################
sub pp_name_four_ways {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test of four ways to name the created executable
  # ----
  # 
  #     % pp hello.pl (produces default a.exe on windows)
  #     % pp -o output1.exe hello.pl
  #     % pp --output output2.exe hello.pl
  #     % pp --output=output3.exe hello.pl
  # 
  # . Create the file hello.pl that will print "hello".
  # . system "pp hello.pl"
  # . system "pp -o output1.exe hello.pl"
  # . system "--output output2.exe hello.pl"
  # . system "--output=output3.exe hello.pl"
  # . Back tick each of the three executables, one at a time,
  #   and collect the results.  Each should be "hello".
  # . Get the size of the executables.
  # . Compare the sizes.  They should all be the same size.
  # 
  # Success if "hello" in each case, failure otherwise.
  #
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;

  my $back_tick_command_string = "";
  my $cmd = "";
  my $sub_test = 0;

  #..............................................
  my $hello_top_of_file_text = '
print "hello";
';
#..............................................
  $$message_ref = "";

#..............................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $hello_pl_file,
                         "",
                         $verbose,
                         $message_ref,
                         $hello_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $cmd = 'pp hello.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/$a_default_executable\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           $a_default_executable,
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\n\[010\]sub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $cmd = 'pp -o output1.exe hello.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/output1.exe\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'output1.exe',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }
  #..........................................................
  $cmd = 'pp --output output2.exe hello.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/output2.exe\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'output2.exe',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce output2.exe?\n";
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  $cmd = 'pp --output=output3.exe hello.pl';
  if (system("$cmd")) {
    $$message_ref = "\n\[015\]sub $test_name_string cannot system $cmd\n";
    return (EXIT_FAILURE);
  } else {
    if ($verbose) {
      print ("sub $test_name_string ");
      print ("Hopefully, \"$cmd\" created $test_dir/output3.exe\n");
    }
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           'output3.exe',
                           "hello",
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $a_default_executable?\n";
  }

  #..........................................................
  return ($error);

}

#########################################################################
sub pp_minus_v_tests {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $hello_pl_file,
       $hello_executable,
       $a_default_executable,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test -v with no arguements, with some other parameter, too,
  #       in many different ways.\
  #
  #       pp -v 1 hello.pl > v_1.txt
  #       pp -v 2 hello.pl > v_2.txt
  #       pp -v 3 hello.pl > v_3.txt
  #
  #       pp -v    hello.pl > v.txt
  #       pp -vv   hello.pl > vv.txt
  #       pp -vvv  hello.pl > vvv.txt
  #
  #       pp -o hello.exe -v hello.pl   > o_v.txt
  #       pp -o hello.exe -vv hello.pl  > o_vv.txt
  #       pp -o hello.exe -vvv hello.pl > o_vvv.txt
  # 
  #       pp -o hello.exe -v 1 hello.pl > o_v_1.txt
  #       pp -o hello.exe -v 2 hello.pl > o_v_2.txt
  #       pp -o hello.exe -v 3 hello.pl > o_v_3.txt
  # 
  #       pp -v 1 hello.pl -o hello.exe > v_1_h_o.txt
  #       pp -v 2 hello.pl -o hello.exe > v_2_h_o.txt
  #       pp -v 3 hello.pl -o hello.exe > v_3_h_o.txt
  #
  #  . Create the file  hello.pl with the code that will print out "hello".
  #  . For each of the above shown five sets of three commands:
  #    . "system" the commands, which capture the outputs in the
  #      shown .txt files.
  #    . Examine the three created text files for each set of five,
  #      for the patterns shown below.
  # 
  #      For v 1
  #           pp:\s+Packing\s+hello.pl
  #           pp:\s+ Running.*parl.exe
  #      For v 2
  #           pp:\s+Packing\s+hello.pl
  #           pp:\s+Writing\s+PAR\s+on
  #           pp:\s+ Running.*parl.exe
  #      For v 3
  #           pp:\s+Packing\s+hello.pl
  #           pp:\s+Writing\s+PAR\s+on
  #           pp:.* making\s+MANIFEST
  #           pp:\s+ Running.*parl.exe
  # 
  # 
  #  . Back tick the created executable and collect the results.
  #    . If the created text file has an "o" in it,
  #      Back tick hello.exe on Windows.
  #      Otherwise back tick just a.exe on windows.
  # 
  #    Hello should be the result in each case.
  # 
  #--------------------------------------------------------------------

  my $error = EXIT_FAILURE;
  my $back_tick_command_string = "";
  my $cmd = "";
  my $at_least_one_line_not_found = $FALSE;

  my $MODULUS = 3;
  my $max_command_strings = "";
  my $i;
  my $command_string = "";
  my $text_file_to_examine = "";
  my $modulus_result = "";
  my @expected_lines = ();
  my $line = "";
  my $all_lines = ();
  my $test_and_sub_test = "00_00";
  my $file_to_back_tick = "";
  my @converted_array = ();

  #..............................................
  my $hello_top_of_file_text = '
print "hello";
';
#..............................................
  my @command_strings = (

        'pp -v 1 hello.pl > v_1.txt',
        'pp -v 2 hello.pl > v_2.txt',
        'pp -v 3 hello.pl > v_3.txt',
        'pp -v    hello.pl > v.txt',
        'pp -vv   hello.pl > vv.txt',
        'pp -vvv  hello.pl > vvv.txt',
        'pp -o hello.exe -v hello.pl   > o_v.txt',
        'pp -o hello.exe -vv hello.pl  > o_vv.txt',
        'pp -o hello.exe -vvv hello.pl > o_vvv.txt',
        'pp -o hello.exe -v 1 hello.pl > o_v_1.txt',
        'pp -o hello.exe -v 2 hello.pl > o_v_2.txt',
        'pp -o hello.exe -v 3 hello.pl > o_v_3.txt',
        'pp -v 1 hello.pl -o hello.exe > v_1_h_o.txt',
        'pp -v 2 hello.pl -o hello.exe > v_2_h_o.txt',
        'pp -v 3 hello.pl -o hello.exe > v_3_h_o.txt',
  );

  if ($os !~ m/^Win/i) {
    @converted_array = ();
    foreach $command_string (@command_strings) {
        $command_string =~ s/hello.exe/hello.out/g;
        push(@converted_array, ($command_string));
    }
    @command_strings = ();
    push(@command_strings, @converted_array);

  }


  my @text_files_to_examine = (
        'v_1.txt',
        'v_2.txt',
        'v_3.txt',
        'v.txt',
        'vv.txt',
        'vvv.txt',
        'o_v.txt',
        'o_vv.txt',
        'o_vvv.txt',
        'o_v_1.txt',
        'o_v_2.txt',
        'o_v_3.txt',
        'v_1_h_o.txt',
        'v_2_h_o.txt',
        'v_3_h_o.txt',
  );


  my @results_to_expect_v = (
          'pp:\s+Packing\s+hello.pl',
          'pp:\s+Running.*parl.exe',
  );
  
  my @results_to_expect_vv = (
          'pp:\s+Packing\s+hello.pl',
          'pp:\s+Writing\s+PAR\s+on',
          'pp:\s+Running.*parl.exe',
  );
  
  my @results_to_expect_vvv = (
          'pp:\s+Packing\s+hello.pl',
          'pp:\s+Writing\s+PAR\s+on',
          'pp:.*making\s+MANIFEST',
          'pp:\s+Running.*parl.exe',
  );


  #............. Remove the ".exe" parts if not Windows
  if ($os !~ m/^Win/i) {
    @converted_array = ();
    foreach $line (@results_to_expect_v) {
        $line =~ s/parl.exe/\/parl /g;
        push(@converted_array, ($line));
    }
    @results_to_expect_v = ();
    push(@results_to_expect_v, @converted_array);
  }
    
  if ($os !~ m/^Win/i) {
    @converted_array = ();
    foreach $line (@results_to_expect_vv) {
        $line =~ s/parl.exe/\/parl /g;
        push(@converted_array, ($line));
    }
    @results_to_expect_vv = ();
    push(@results_to_expect_vv, @converted_array);
  }
    
  if ($os !~ m/^Win/i) {
    @converted_array = ();
    foreach $line (@results_to_expect_vvv) {
        $line =~ s/parl.exe/\/parl /g;
        push(@converted_array, ($line));
    }
    @results_to_expect_vvv = ();
    push(@results_to_expect_vvv, @converted_array);
  }

  #..........................................................
  $max_command_strings = @command_strings;

  $$message_ref = "";

  #..........................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #..........................................................
  $error = create_file(  $test_dir . "/$hello_pl_file",
                         "",
                         $verbose,
                         $message_ref,
                         $hello_top_of_file_text,
                       );

  if ($error == EXIT_FAILURE) {
    $$message_ref = "\nsub $test_name_string: " . $$message_ref;
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #..........................................................
  for ($i = 0; $i < $max_command_strings; $i++) {
    @expected_lines = ();
    $all_lines = ();
    $at_least_one_line_not_found = $FALSE;
    $test_and_sub_test = $test_number . '_' . $i;


    $text_file_to_examine = $text_files_to_examine[$i];
    if ($verbose) {
      print ("\n\nAbout to test $test_and_sub_test: ");
      print ("$test_name_string\n\n");
      print ("Text file to examine is $text_file_to_examine\n");
    }

    #..........................................................
    # Remove any executables from prior iterations
    if ($text_file_to_examine =~  m/o/) {
      $file_to_back_tick = $hello_executable;
      if (-e($hello_executable)) {
        # Remove any executables from prior sub tests
        if (!(unlink($hello_executable))) {
          $$message_ref = 
              "Test $test_and_sub_test: $test_name_string "   .
              "cannot remove file $hello_executable\n";
          return(EXIT_FAILURE);
        }
      } # exists
    } else {
      if (-e($a_default_executable)) {
        # Remove any executables from prior sub tests
        if (!(unlink($a_default_executable))) {
          $$message_ref = 
              "Test $test_and_sub_test: $test_name_string "   .
              "cannot remove file $a_default_executable\n";
          return(EXIT_FAILURE);
        }
      } # exists
      $file_to_back_tick = $a_default_executable;
    }

    #..........................................................
    $cmd = $command_strings[$i];
    if (system("$cmd")) {
      $$message_ref = "\n\[015\]sub ${test_name_string}_$test_and_sub_test" .
                      " cannot system $cmd\n";
      return (EXIT_FAILURE);
    } else {
      if ($verbose) {
        print ("sub $test_name_string ");
        print ("Hopefully, \"$cmd\" created $test_dir/");
        if ($text_file_to_examine =~  m/o/) {
          print ("$hello_executable\n");
        } else {
          print ("$a_default_executable\n");
        }
      }
    }
    #..........................................................
    if ( ($i % $MODULUS) == 0) {
       push(@expected_lines, (@results_to_expect_v));
    } elsif ( ($i % $MODULUS) == 1) {
       push(@expected_lines, (@results_to_expect_vv));
    } else {
       push(@expected_lines, (@results_to_expect_vvv));
    }

    #..........................................................
    # Get the results from the created text file.

    $text_file_to_examine = $text_files_to_examine[$i];
    if (-e($text_file_to_examine)) {

      if (open (FH, "$text_file_to_examine")) {

        # Slurp in all the lines of the file at once
        $all_lines = do { local( @ARGV, $/ ) = $text_file_to_examine ; <> };

        if (!(close(FH))) {
          $$message_ref = 
             "Something is wrong with test $test_name_string "            .
             "in directory $test_dir\n"                                   .
             "File $text_file_to_examine exists, and I opened it, "       .
             "but now I cannot close it.\n"                               .
             "Cannot continue with test $test_name_string\n";
          return (EXIT_FAILURE);
        }

      } else {
        $$message_ref = 
           "Something is wrong with test $test_name_string "            .
           "in directory $test_dir\n"                                   .
           "File $text_file_to_examine exists but I cannot open it.\n"  .
           "Cannot continue with test $test_name_string\n";
        return (EXIT_FAILURE);
      }

    } else {
      $$message_ref = 
         "Something is wrong with test $test_name_string "            .
         "in directory $test_dir\n"                                   .
         "Command $cmd did not create file $text_file_to_examine\n"   .
         "Cannot continue with test $test_name_string\n";
      return (EXIT_FAILURE);
    }

    #..........................................................
    # By this time, I have extracted opened the text file,
    # extracted all of the lines into $all_lines, and closed the
    # file.
    #..........................................................
    foreach $line (@expected_lines) {
      if ($all_lines !~ m!$line!gm) {
        $at_least_one_line_not_found = $TRUE;
        print ("Line $line does not match\n") if ($verbose);
      }
    }
    #..........................................................
    if ($at_least_one_line_not_found) {
      $$message_ref = 
         "Something is wrong with test $test_name_string "            .
         "in directory $test_dir\n"                                   .
         "Command $cmd did provide the expected results in file "     .
         "$text_file_to_examine.\n  I expected matches to regexp \n"  .
         "@expected_lines"                                            .
         "\nbut instead got \n$all_lines\n"                           .
         "Cannot continue with test $test_name_string\n";
      return (EXIT_FAILURE);
    } else {
      if ($verbose) {
        print ("Test $test_name_string, command $cmd, \n");
        print ("file $text_file_to_examine ");
        print ("had the expected results. \.\.\. passed so far ");
        print ("\.\.\.\n");
      }
    }

    #..........................................................
    # Now to see if the created executable works
    $error = back_tick_a_command
                           (
                             $test_number,
                             $i,
                             $test_name_string,
                             $test_dir,
                             $back_tick_command_string,
                             $file_to_back_tick,
                             "hello",
                             $os,
                             $verbose,
                             $message_ref,
                          );
    if ($error == EXIT_FAILURE) {
      $$message_ref = 
        " Test $test_and_sub_test \n" . $$message_ref . 
        "\nDid $cmd produce $file_to_back_tick?\n";
      return ($error);
    }

    #.................................................................


  } # for $i
  #..........................................................
  return (EXIT_SUCCESS);

}


#########################################################################
sub pp_minus_V {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test -V and of --version
  # ----
  #
  # Outline
  # -------
  # . Back tick "pp -V" and collect the results.
  # . The string
  #     "Such use shall not be construed as a distribution"
  #   should be part of what was collected.
  # 
  #--------------------------------------------------------------------
  my $error = EXIT_FAILURE;
  my $back_tick_command_string_big_V = 'pp -V';
  my $back_tick_command_string_minus_minus = 'pp --version';
  my $sub_test = 0;
  my $expected_string = 
         "Such use shall not be construed as a distribution";
  my $cmd = "";

  #..........................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #.................................................................
  $cmd = $back_tick_command_string_big_V; # Keeps template the same
                                          # as possible.
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string_big_V,
                           "",
                           $expected_string,
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $expected_string?\n";
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }

  #.................................................................
  $cmd = $back_tick_command_string_minus_minus; # Keeps template the
                                                # same as possible.
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string_minus_minus,
                           "",
                           $expected_string,
                           $os,
                           $verbose,
                           $message_ref,
                        );
  if ($error == EXIT_FAILURE) {
    $$message_ref =
      $$message_ref . "\nDid $cmd produce $expected_string?\n";
    return (EXIT_FAILURE) if ($error == EXIT_FAILURE);
  }


  #.................................................................
  return ($error);
  #.................................................................

}

#########################################################################
sub pp_help_tests {
  my (
       $test_name_string,
       $os,
       $test_number,
       $test_dir,
       $verbose,
       $message_ref,
     ) = @_;

  #--------------------------------------------------------------------
  # Goal: Test  of 'pp -h' and of 'pp --help'
  # ----
  #            The help screen should be shown.
  #
  # Outline
  # -------
  # . Back tick "pp -h" and collect the results
  # . The string "Perl Packager" should have been collected.
  # 
  #--------------------------------------------------------------------
  my $error = EXIT_FAILURE;
  my $back_tick_command_string = 'pp -h';
  my $sub_test = 0;
  my $expected_string = 'Perl Packager';


  #..........................................................
  if (!(chdir("$test_dir"))) {
      $$message_ref = "\n\[005\]sub $test_name_string cannot " .
                      "chdir $test_dir\n:$!:\n";
      return (EXIT_FAILURE);
  }

  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "",
                           $expected_string,
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
      $$message_ref = 
        "\nTest ${test_number}_$sub_test \n" . $$message_ref;
     return ($error);
  }

  #.................................................................
  $back_tick_command_string = 'pp --help';
  #.................................................................
  $error = back_tick_a_command
                         (
                           $test_number,
                           $sub_test++,
                           $test_name_string,
                           $test_dir,
                           $back_tick_command_string,
                           "",
                           $expected_string,
                           $os,
                           $verbose,
                           $message_ref,
                        );

  if ($error == EXIT_FAILURE) {
      $$message_ref = 
        "\nTest ${test_number}_$sub_test \n" . $$message_ref;
  }

  return ($error);
  #.................................................................

}

#########################################################################


#########################################################################
##################### Beginning - Start of Main #########################
#########################################################################

my $startdir = "";

my $answer = "";

my $test_name_string = "";
my $test_number = 1;
my $error = EXIT_SUCCESS;
my $message = "";
my $test_dir = "";

my $hello_pl_file = "hello\.pl";
my $foo_pl_file = "foo\.pl";
my $bar_pl_file = "bar\.pl";
my $hello_par_file_with_dot_par = "hello\.par";
my $hello_par_file_no_dot_par = "hello";

my $hello_executable = "hello\.exe";
my $foo_executable = "foo\.exe";
my $bar_executable = "bar\.exe";
my $a_default_executable = "a\.exe";
my $a_default_dot_par = "a\.par";

my $verbose = "";
my $debug_log = "";
my $debug = $FALSE;
my $perl = "";
my $par = "";

GetOptions(  "verbose"         => \$verbose,
             "debug"           => \$debug,
             "startdir=s"      => \$startdir,
             "perl_location=s" => \$perl,
             "par_location=s"  => \$par,
          );


$verbose = 0 if (!defined($verbose) or ($verbose eq ""));

$perl = $^X if ($perl eq "");
if (!(-e($perl))) {
  print ("The perl executable \"$perl\" does not exist\n");
  exit(EXIT_FAILURE);
}

  ###############################################################
  # Examples for Posix os, hostname, release, version, hardware
  #
  #      Example from Unix:
  #                   os        FreeBSD,
  #                   hostname  my_machine_name
  #                   release   4.3-RELEASE
  #                   version   FreeBSD 4.3-RELEASE #0: Sat Apr
  #                   hardware  i386
  #      Example from windows 2000:
  #                   os        Windows NT
  #                   hostname  my_machine_name
  #                   release   5.0
  #                   version   Build 2195 (Service Pack 2)
  #                   hardware  x86
  #
  # os examples: could match Win, CYGWIN_NT, FreeBSD, SunOS, Linux
  #
  ###############################################################

if (!$par) {
  foreach my $dir ( split(/\Q$Config{path_sep}\E/, $ENV{PATH}) ) {
    $par = File::Spec->catfile($dir, 'par.pl');
    last if -f $par;
  }
}

if (!(-f($par))) {
  print ("The par executable \"$par\" does not exist\n");
  exit(EXIT_FAILURE);
}

my $_out = $Config{_exe} || '.out';

$hello_pl_file = "hello.pl";
$foo_pl_file = "foo.pl";
$bar_pl_file = "bar.pl";
$hello_par_file_with_dot_par = "hello.par";
$hello_par_file_no_dot_par = "hello";
$a_default_executable = "a$_out";
$a_default_dot_par = "a.par";
$hello_executable = "hello$_out";
$foo_executable = "foo$_out";
$bar_executable = "bar$_out";

$startdir ||= File::Spec->catdir(File::Spec->tmpdir, 'pp_switch_tests');

if ($debug) {
  # Open up a debug log to log the tests that passed
  $debug_log = File::Spec->catfile($startdir, "debug.log");
  if(!(open (DEBUG, ">$debug_log"))) {
    die ("Cannot open debug log $debug_log:$!:\n");
  }
}

#SKIP: { 
#  $test_number = 25;
#  skip("Skipping  tests for brevity "  . "$test_number \n", 24);

########################### Next Test 001 ##################################
$test_name_string = "pp_hello_1";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n" .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error = pp_hello_1(  $test_name_string,
                      $os,
                      $test_number,
                      $test_dir,
                      $hello_pl_file,
                      $a_default_executable,
                      $verbose,
                      \$message,
                   );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 002 ##################################
$test_name_string = "pp_minus_o_hello_hello_dot_pl";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}


if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_o_hello_hello_dot_pl
    ($test_name_string,
     $os,
     $test_number,
     $test_dir,
     $hello_pl_file,
     $hello_executable,
     $verbose,
     \$message);

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 003 ##################################
$test_name_string = "pp_minus_o_foo_foo_dot_pl_bar_dot_pl";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_o_foo_foo_dot_pl_bar_dot_pl
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $foo_pl_file,
        $bar_pl_file,
        $foo_executable,
        $bar_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 004 ##################################
$test_name_string = "pp_minus_p_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_p_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_dot_par,
        $verbose,
        \$message,
        $perl,
        $par,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 005 ##################################
$test_name_string = "pp_minus_p_minus_o_hello_dot_par_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_p_minus_o_hello_dot_par_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $hello_par_file_with_dot_par,
        $verbose,
        \$message,
        $perl,
        $par,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 006 ##################################
$test_name_string = "pp_minus_o_hello_file_dot_par";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_o_hello_file_dot_par
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $hello_par_file_with_dot_par,
        $hello_par_file_no_dot_par,
        $hello_executable,
        $verbose,
        \$message,
        $perl,
        $par,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 007 ##################################
$test_name_string = "pp_minus_S_minus_o_hello_file";
SKIP: {
  skip("Skipping " . $test_number++ . " $test_name_string because we know it does not work\n", 1);

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_S_minus_o_hello_file
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $hello_par_file_with_dot_par,
        $hello_executable,
        $verbose,
        \$message,
        $perl,
        $par,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);
       } # SKIP 1
########################### Next Test 008 ##################################
$test_name_string = "pp_minus_p_minus_o_out_dot_par_file";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_p_minus_o_out_dot_par_file
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $verbose,
        \$message,
        $perl,
        $par,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 009 ##################################
$test_name_string = "pp_minus_B_with_small_minus_p_tests";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_B_with_small_minus_p_tests
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $verbose,
        \$message,
        $perl,
        $par,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 010 ##################################
$test_name_string = "pp_minus_B_with_large_minus_P_tests";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_B_with_large_minus_P_tests
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $verbose,
        \$message,
        $perl,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 011 ##################################
$test_name_string = "pp_minus_e_print_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_e_print_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 012 ##################################
$test_name_string = "pp_minus_p_minus_e_print_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_p_minus_e_print_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $verbose,
        \$message,
        $perl,
        $par,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 013 ##################################
$test_name_string = "pp_minus_P_minus_e_print_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_P_minus_e_print_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $verbose,
        \$message,
        $perl,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 014 ##################################
$test_name_string = "pp_minus_c_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_c_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 015 ##################################
$test_name_string = "pp_minus_x_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_x_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
print ("\n\n"); # To get by some "hello" print outs that interfere

ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 016 ##################################
$test_name_string = "pp_minus_n_minus_x_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_n_minus_x_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
print ("\n\n"); # To get by some "hello" print outs that interfere

ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 017 ##################################
$test_name_string = "pp_minus_I_foo_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_I_foo_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 018 ##################################
$test_name_string = "pp_minus_I_foo_minus_I_bar_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_I_foo_minus_I_bar_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 019 ##################################
$test_name_string = "pp_minus_lib_foo_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_lib_foo_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 020 ##################################
$test_name_string = "pp_minus_lib_foo_minus_lib_bar_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_lib_foo_minus_lib_bar_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 021 ##################################
$test_name_string = "pp_minus_M_foo_hidden_print_foo";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
   pp_minus_M_foo_hidden_print_foo
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 022 ##################################
$test_name_string = "pp_minus_M_foo_minus_M_bar_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}
$error =
   pp_minus_M_foo_minus_M_bar_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 023 ##################################
$test_name_string = "pp_minus_M_abbrev_dot_pl_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_minus_M_abbrev_dot_pl_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 024 ##################################
$test_name_string = "pp_minus_M_abbrev_dot_pl_minus_o_hello_hello";

$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"    .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_minus_M_abbrev_dot_pl_minus_o_hello_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $hello_executable,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);
#         } # SKIP 24
########################### Next Test 025 ##################################
$test_name_string = "pp_minus_X_module_foo";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"     .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_minus_X_module_foo
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $foo_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 026 ##################################
$test_name_string = "pp_minus_r_hello";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"     .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_minus_r_hello
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 027 ##################################
$test_name_string = "pp_minus_r_hello_a_b_c";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"     .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_minus_r_hello_a_b_c
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 028 ##################################
$test_name_string = "pp_hello_to_log_file";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"     .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_hello_to_log_file
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);


########################### Next Test 029 ##################################
$test_name_string = "pp_name_four_ways";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"     .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_name_four_ways
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);



########################### Next Test 030 ##################################
$test_name_string = "pp_minus_v_tests";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"     .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_minus_v_tests
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $hello_pl_file,
        $hello_executable,
        $a_default_executable,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);

########################### Next Test 031 ##################################
$test_name_string = "pp_minus_V";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"     .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_minus_V
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);


########################### Next Test 032 ##################################
$test_name_string = "pp_help_tests";
$error = prior_to_test($test_number,
                       $startdir,
                       $os,
                       \$test_dir,
                       $verbose,
                       \$message);
if ($error == EXIT_FAILURE) {
  $message = "\nCannot run test $test_name_string due to\n"     .
             "prior_to_test: Test $test_number : $message\n";
  die($message);
}

if ($verbose) {
  print ("About to run test $test_number: $test_name_string ");
  print ("in directory $test_dir\n");
}

$error =
  pp_help_tests
     (
        $test_name_string,
        $os,
        $test_number,
        $test_dir,
        $verbose,
        \$message,
     );

if ($debug) {
  if ($error) {
    print DEBUG ("\n\nTest $test_number: $test_name_string FAILED\n");
    print DEBUG ("$message\n");
  } else {
    print DEBUG ("\n\nTest $test_number: $test_name_string PASSED\n");
  }
}

after_test($test_number++, $error, $message, $verbose);
ok ($error == EXIT_SUCCESS, "$test_name_string" . " $message");
print ("\n\n\n") if ($error == EXIT_FAILURE);


########################################################################
if ($debug) {
  close(DEBUG) or die ("At end of test: Cannot close file $debug_log:$!:\n");
}

