package Path::Dispatcher::Match;
use Any::Moose;

use Path::Dispatcher::Path;
use Path::Dispatcher::Rule;

has path => (
    is       => 'ro',
    isa      => 'Path::Dispatcher::Path',
    required => 1,
);

has leftover => (
    is  => 'ro',
    isa => 'Str',
);

has rule => (
    is       => 'ro',
    isa      => 'Path::Dispatcher::Rule',
    required => 1,
);

has positional_captures => (
    is      => 'ro',
    isa     => 'ArrayRef[Str|Undef]',
    default => sub { [] },
);

has named_captures => (
    is      => 'ro',
    isa     => 'HashRef[Str|Undef]',
    default => sub { {} },
);

sub run {
    my $self = shift;

    local $_ = $self->path;
    return scalar $self->rule->run($self, @_);
}

sub pos {
    my $self  = shift;
    my $index = shift;

    return undef if $index == 0;

    $index-- if $index > 0;

    return $self->positional_captures->[$index];
}

sub named {
    my $self = shift;
    my $key  = shift;
    return $self->named_captures->{$key};
}

__PACKAGE__->meta->make_immutable;
no Any::Moose;

1;

__END__

=head1 NAME

Path::Dispatcher::Match - the result of a successful rule match

=head1 SYNOPSIS

    my $rule = Path::Dispatcher::Rule::Tokens->new(
        tokens => [ 'attack', qr/^\w+$/ ],
        block  => sub {
            my $match = shift;
            attack($match->pos(2))
        },
    );

    my $match = $rule->match("attack dragon");

    # introspection
    $match->path                # "attack dragon"
    $match->leftover            # empty string (populated with prefix rules)
    $match->rule                # $rule
    $match->positional_captures # ["attack", "dragon"] (decided by the rule)
    $match->pos(1)              # "attack"
    $match->pos(2)              # "dragon"

    $match->run                 # attack("dragon")

=head1 DESCRIPTION

If a L<Path::Dispatcher::Rule> successfully matches a path, it creates one or
more C<Path::Dispatcher::Match> objects.

=head1 ATTRIBUTES

=head2 rule

The L<Path::Dispatcher::Rule> that created this match.

=head2 path

The path that the rule matched.

=head2 leftover

The rest of the path. This is populated when the rule matches a prefix of the
path.

=head2 positional_captures

Any positional captures generated by the rule. For example,
L<Path::Dispatcher::Rule::Regex> populates this with the capture variables.

=head1 METHODS

=head2 run

Executes the rule's codeblock with the same arguments.

=head2 pos($i)

Returns the C<$i>th positional capture, 1-indexed.

=cut

