package Lingua::PT::Ords2Nums;

use 5.006;
use strict;
use warnings;

use Lingua::PT::Words2Nums qw/word2num/;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
	ord2num
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
);

our $VERSION = '0.08';

my %values = (
    'bilionsimo'	=> 1000000000,
    'milionsimo'	=> 1000000,
    'milsimo'		=> 1000,

    'nongentsimo'	=> 900,
    'octigentsimo'	=> 800,
    'septigentsimo'	=> 700,
    'seiscentsimo'	=> 600,
    'quingentsimo'	=> 500,
    'quadrigentsimo'	=> 400,
    'tricentsimo'	=> 300,
    'ducentsimo'	=> 200,
    'centsimo' 	=> 100,

    'nonagsimo'	=> 90,
    'octogsimo' 	=> 80,
    'septuagsimo' 	=> 70,
    'sexagsimo' 	=> 60,
    'quinquagsimo' 	=> 50,
    'quadragsimo' 	=> 40,
    'trigsimo' 	=> 30,
    'vigsimo' 		=> 20,
    'dcimo' 		=> 10,

    'nono'		=> 9,
    'oitavo'		=> 8,
    'stimo'		=> 7,
    'sexto'		=> 6,
    'quinto'		=> 5,
    'quarto'		=> 4,
    'terceiro'		=> 3,
    'segundo'		=> 2,
    'primeiro'		=> 1,
);

# Order of application is important, but hashes are unsorted.
# So we need to store the order of the keys in an array.
my @key_sorted_order = (
    'bilionsimo',#	=> 1000000000,
    'milionsimo',#	=> 1000000,
    'milsimo',#		=> 1000,

    'nongentsimo',#	=> 900,
    'octigentsimo',#	=> 800,
    'septigentsimo',#	=> 700,
    'seiscentsimo',#	=> 600,
    'quingentsimo',#	=> 500,
    'quadrigentsimo',#	=> 400,
    'tricentsimo',#	=> 300,
    'ducentsimo',#	=> 200,
    'centsimo',# 	=> 100,

    'nonagsimo',#	=> 90,
    'octogsimo',# 	=> 80,
    'septuagsimo',# 	=> 70,
    'sexagsimo',# 	=> 60,
    'quinquagsimo',# 	=> 50,
    'quadragsimo',# 	=> 40,
    'trigsimo',# 	=> 30,
    'vigsimo',# 		=> 20,
    'dcimo',# 		=> 10,

    'nono',#		=> 9,
    'oitavo',#		=> 8,
    'stimo',#		=> 7,
    'sexto',#		=> 6,
    'quinto',#		=> 5,
    'quarto',#		=> 4,
    'terceiro',#		=> 3,
    'segundo',#		=> 2,
    'primeiro',#		=> 1,
);


=head1 NAME

=encoding latin1

Lingua::PT::Ords2Nums - Converts Portuguese ordinals to numbers

=head1 SYNOPSIS

  use Lingua::PT::Ords2Nums qw/ord2num/;

  $num = word2num('dcimo primeiro')   # 11

=head1 DESCRIPTION

Converts Portuguese ordinals to numbers. Works up to 999.999.999.999
('novecentos e noventa e nove bilionsimos novecentos e noventa e nove
milionsimos novecentos e noventa e nove milsimos nongentsimo nonagsimo
nono').

=cut

=head2 ord2num

Turns an ordinal number into a regular number (decimal).

  $num = word2num('segundo')
  # $num now holds 2

=cut

sub ord2num {
  $_ = shift || return undef;
  my $result = 0;

  s/(.*)bilionsimos/$result += (word2num($1) * 1000000000)/e;
  s/(.*)milionsimos/$result += (word2num($1) * 1000000)/e;
  s/(.*)milsimos/$result += (word2num($1) * 1000)/e;

  for my $value ( @key_sorted_order ) {
    s/$value/$result += $values{$value}/e;
  }

  $result;
}

1;
__END__

=head1 DEPENDENCIES

Lingua::PT::Words2Nums

=head1 TO DO

=over 6

=item * Implement function isord()

=back

=head1 SEE ALSO

More tools for the Portuguese language processing can be found at the
Natura project: L<http://natura.di.uminho.pt>

=head1 AUTHOR

Jose Castro, C<< <cog@cpan.org> >>

=head1 COPYRIGHT & LICENSE

Copyright 2004 Jose Castro, All Rights Reserved.

This program is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
