package JobAd::Job;

use strict;

use base 'JobAd::GEN::Job';

use Gantry::Plugins::CRUD qw( select_multiple_closure );
use Gantry::Utils::FormMunger;

use JobAd::Model;
use JobAd::Model::job       qw( $JOB );
use JobAd::Model::skill     qw( $SKILL );
use JobAd::Model::job_skill qw( $JOB_SKILL );

sub schema_base_class { return 'JobAd::Model'; }
use Gantry::Plugins::DBIxClassConn qw( get_schema );

#-----------------------------------------------------------------
# $self->do_main(  )
#-----------------------------------------------------------------
# This method supplied by JobAd::GEN::Job

my $job = Gantry::Plugins::CRUD->new(
    add_action      => \&job_add,
    edit_action     => \&job_edit,
    delete_action   => \&job_delete,
    form            => \&job_form,
    text_descr      => 'job',
);

#-------------------------------------------------
# $self->do_add( )
#-------------------------------------------------
sub do_add {
    my $self = shift;

    $job->add( $self );
}

#-------------------------------------------------
# $self->job_add( $params, $data )
#-------------------------------------------------
sub job_add {
    my ( $self, $params, $data ) = @_;

    # first pull skills out of params
    my $r_new_skills = delete $params->{ skill };
    my @skills = ( ref $r_new_skills eq 'ARRAY' )
               ? @{ $r_new_skills }
               : ( $r_new_skills );

    my $schema = $self->get_schema();

    $schema->txn_begin();

        # add the job row
        my $new_job = $JOB->gcreate( $self, $params );

        # add the job_skill rows
        foreach my $skill ( @skills ) {
            $JOB_SKILL->gcreate( $self,
                {
                    job   => $new_job->id,
                    skill => $skill,
                }
            );
        }

    $schema->txn_commit();
}

#-------------------------------------------------
# $self->do_delete( $doomed_id, $confirm )
#-------------------------------------------------
sub do_delete {
    my ( $self, $doomed_id, $confirm ) = @_;

    $job->delete( $self, $confirm, { id => $doomed_id } );
}

#-------------------------------------------------
# $self->job_delete( $data )
#-------------------------------------------------
sub job_delete {
    my ( $self, $data ) = @_;

    my $schema          = $self->get_schema();
    my $doomed_row      = $JOB->gfind( $self, $data->{ id } );

    $schema->txn_begin();

        $JOB_SKILL->gsearch( $self, { job => $doomed_row->id } )->delete_all;
        $doomed_row->delete();

    $schema->txn_commit();
}

#-------------------------------------------------
# $self->do_edit( $id )
#-------------------------------------------------
sub do_edit {
    my ( $self, $id ) = @_;

    my $row = $JOB->gfind( $self, $id );

    $job->edit( $self, { row => $row } );
}

#-------------------------------------------------
# $self->job_edit( $param, $data )
#-------------------------------------------------
sub job_edit {
    my( $self, $params, $data ) = @_;

    my $row = $data->{ row };

    my @new_skills;
    if ( defined $params->{ skill } ) {
        my $r_new_skills = delete $params->{ skill };
        @new_skills = ( ref $r_new_skills eq 'ARRAY' )
                   ? @{ $r_new_skills }
                   : ( $r_new_skills );

    }
    my %new_skills = map { $_ => 1 } @new_skills;

    my $schema = $self->get_schema();

    $schema->txn_begin();

        # remove deleted skills
        my @old_skills = $row->job_skills;

        foreach my $old_skill ( @old_skills ) {

            my $skill_id = $old_skill->skill->id;

            if ( $new_skills{ $skill_id } ) {

                delete $new_skills{ $skill_id };

            }
            else {
                $old_skill->delete;
            }
        }

        # create new skills
        foreach my $new_skill_id ( keys %new_skills ) {
            $JOB_SKILL->gcreate(
                $self,
                {
                    job   => $row->id,
                    skill => $new_skill_id,
                },
            );
        }

        # update job row
        $row->update( $params );

    $schema->txn_commit();
}

#-----------------------------------------------------------------
# $self->job_form( $data )
#-----------------------------------------------------------------
sub job_form {
    my ( $self, $data ) = @_;

    # skills (three way join with skill table via job_skill)
    my @skill_options;
    my @all_skills = $SKILL->gsearch( $self, undef, { order_by => 'ident' } );

    foreach my $skill ( @all_skills ) {
        push @skill_options, {
            value => $skill->id,
            label => $skill->ident,
        };
    }

    # then, find the ones associated with our row (if there are any)
    my %selected_skill;

    if ( defined $data->{ row } ) {
        my @skills = $data->{ row }->skills;

        foreach my $skill ( @skills ) {
            $selected_skill{ $skill->id }++;
        }
    }

    my $form = $self->SUPER::job_form( $data );

    my $munger = Gantry::Utils::FormMunger->new( $form );

    $munger->append_field(
        {
            name     => 'skill',
            type     => 'select_multiple',
            label    => 'Skills',
            options  => \@skill_options,
            selected => select_multiple_closure(
                    'skill',
                    \%selected_skill,
            ),
        },
    );

    return $form;
} # END job_form

1;

=head1 NAME

JobAd::Job - A controller in the JobAd application

=head1 SYNOPSIS

This package is meant to be used in a stand alone server/CGI script or the
Perl block of an httpd.conf file.

Stand Alone Server or CGI script:

    use JobAd::Job;

    my $cgi = Gantry::Engine::CGI->new( {
        config => {
            #...
        },
        locations => {
            '/someurl' => 'JobAd::Job',
            #...
        },
    } );

httpd.conf:

    <Perl>
        # ...
        use JobAd::Job;
    </Perl>

    <Location /someurl>
        SetHandler  perl-script
        PerlHandler JobAd::Job
    </Location>

If all went well, one of these was correctly written during app generation.

=head1 DESCRIPTION

This module was originally generated by Bigtop.  But feel free to edit it.
You might even want to describe the table this module controls here.

=head1 METHODS

=over 4

=item get_model_name

=item text_descr

=item schema_base_class

=item get_orm_helper


=back


=head1 METHODS INHERITED FROM JobAd::GEN::Job

=over 4

=item do_main


=back


=head1 DEPENDENCIES

    JobAd
    JobAd::GEN::Job
    JobAd::Model::job
    Gantry::Plugins::CRUD

=head1 AUTHOR

Phil Crow, E<lt>pcrow@localdomainE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 Phil Crow

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.

=cut
